"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupPlan = void 0;
const core_1 = require("@aws-cdk/core");
const backup_generated_1 = require("./backup.generated");
const rule_1 = require("./rule");
const selection_1 = require("./selection");
const vault_1 = require("./vault");
/**
 * A backup plan
 */
class BackupPlan extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.rules = [];
        const plan = new backup_generated_1.CfnBackupPlan(this, 'Resource', {
            backupPlan: {
                backupPlanName: props.backupPlanName || id,
                backupPlanRule: core_1.Lazy.anyValue({ produce: () => this.rules }, { omitEmptyArray: true }),
            },
        });
        this.backupPlanId = plan.attrBackupPlanId;
        this.backupPlanArn = plan.attrBackupPlanArn;
        this.versionId = plan.attrVersionId;
        this._backupVault = props.backupVault;
        for (const rule of props.backupPlanRules || []) {
            this.addRule(rule);
        }
    }
    /**
     * Import an existing backup plan
     */
    static fromBackupPlanId(scope, id, backupPlanId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.backupPlanId = backupPlanId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Daily with 35 day retention
     */
    static daily35DayRetention(scope, id, backupVault) {
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        return plan;
    }
    /**
     * Daily and monthly with 1 year retention
     */
    static dailyMonthly1YearRetention(scope, id, backupVault) {
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        plan.addRule(rule_1.BackupPlanRule.monthly1Year());
        return plan;
    }
    /**
     * Daily, weekly and monthly with 5 year retention
     */
    static dailyWeeklyMonthly5YearRetention(scope, id, backupVault) {
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        plan.addRule(rule_1.BackupPlanRule.weekly());
        plan.addRule(rule_1.BackupPlanRule.monthly5Year());
        return plan;
    }
    /**
     * Daily, weekly and monthly with 7 year retention
     */
    static dailyWeeklyMonthly7YearRetention(scope, id, backupVault) {
        const plan = new BackupPlan(scope, id, { backupVault });
        plan.addRule(rule_1.BackupPlanRule.daily());
        plan.addRule(rule_1.BackupPlanRule.weekly());
        plan.addRule(rule_1.BackupPlanRule.monthly7Year());
        return plan;
    }
    /**
     * Adds a rule to a plan
     *
     * @param rule the rule to add
     */
    addRule(rule) {
        var _a, _b, _c, _d, _e, _f;
        let vault;
        if (rule.props.backupVault) {
            vault = rule.props.backupVault;
        }
        else if (this._backupVault) {
            vault = this._backupVault;
        }
        else {
            this._backupVault = new vault_1.BackupVault(this, 'Vault');
            vault = this._backupVault;
        }
        this.rules.push({
            completionWindowMinutes: (_a = rule.props.completionWindow) === null || _a === void 0 ? void 0 : _a.toMinutes(),
            lifecycle: (rule.props.deleteAfter || rule.props.moveToColdStorageAfter) && {
                deleteAfterDays: (_b = rule.props.deleteAfter) === null || _b === void 0 ? void 0 : _b.toDays(),
                moveToColdStorageAfterDays: (_c = rule.props.moveToColdStorageAfter) === null || _c === void 0 ? void 0 : _c.toDays(),
            },
            ruleName: (_d = rule.props.ruleName) !== null && _d !== void 0 ? _d : `${this.node.id}Rule${this.rules.length}`,
            scheduleExpression: (_e = rule.props.scheduleExpression) === null || _e === void 0 ? void 0 : _e.expressionString,
            startWindowMinutes: (_f = rule.props.startWindow) === null || _f === void 0 ? void 0 : _f.toMinutes(),
            targetBackupVault: vault.backupVaultName,
        });
    }
    /**
     * The backup vault where backups are stored if not defined at
     * the rule level
     */
    get backupVault() {
        if (!this._backupVault) {
            // This cannot happen but is here to make TypeScript happy
            throw new Error('No backup vault!');
        }
        return this._backupVault;
    }
    /**
     * Adds a selection to this plan
     */
    addSelection(id, options) {
        return new selection_1.BackupSelection(this, id, {
            backupPlan: this,
            ...options,
        });
    }
    validate() {
        if (this.rules.length === 0) {
            return ['A backup plan must have at least 1 rule.'];
        }
        return [];
    }
}
exports.BackupPlan = BackupPlan;
//# sourceMappingURL=data:application/json;base64,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