"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupResource = exports.TagOperation = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * An operation that is applied to a key-value pair
 */
var TagOperation;
(function (TagOperation) {
    /**
     * StringEquals
     */
    TagOperation["STRING_EQUALS"] = "STRINGEQUALS";
    /**
     * Dummy member
     */
    TagOperation["DUMMY"] = "dummy";
})(TagOperation = exports.TagOperation || (exports.TagOperation = {}));
/**
 * A resource to backup
 */
class BackupResource {
    constructor(resource, tagCondition, construct) {
        this.resource = resource;
        this.tagCondition = tagCondition;
        this.construct = construct;
    }
    /**
     * Adds all supported resources in a construct
     *
     * @param construct The construct containing resources to backup
     */
    static fromConstruct(construct) {
        return new BackupResource(undefined, undefined, construct);
    }
    /**
     * A DynamoDB table
     */
    static fromDynamoDbTable(table) {
        return BackupResource.fromArn(table.tableArn);
    }
    /**
     * An EC2 instance
     */
    static fromEc2Instance(instance) {
        return BackupResource.fromArn(core_1.Stack.of(instance).formatArn({
            service: 'ec2',
            resource: 'instance',
            resourceName: instance.instanceId,
        }));
    }
    /**
     * An EFS file system
     */
    static fromEfsFileSystem(fileSystem) {
        return BackupResource.fromArn(core_1.Stack.of(fileSystem).formatArn({
            service: 'elasticfilesystem',
            resource: 'file-system',
            resourceName: fileSystem.fileSystemId,
        }));
    }
    /**
     * A RDS database instance
     */
    static fromRdsDatabaseInstance(instance) {
        return BackupResource.fromArn(instance.instanceArn);
    }
    /**
     * A list of ARNs or match patterns such as
     * `arn:aws:ec2:us-east-1:123456789012:volume/*`
     */
    static fromArn(arn) {
        return new BackupResource(arn);
    }
    /**
     * A tag condition
     */
    static fromTag(key, value, operation) {
        return new BackupResource(undefined, {
            key,
            value,
            operation,
        });
    }
}
exports.BackupResource = BackupResource;
//# sourceMappingURL=data:application/json;base64,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