"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupPlanRule = void 0;
const events = require("@aws-cdk/aws-events");
const core_1 = require("@aws-cdk/core");
/**
 * A backup plan rule
 */
class BackupPlanRule {
    /** @param props Rule properties */
    constructor(props) {
        this.props = props;
        if (props.deleteAfter && props.moveToColdStorageAfter &&
            props.deleteAfter.toSeconds() < props.moveToColdStorageAfter.toSeconds()) {
            throw new Error('`deleteAfter` must be greater than `moveToColdStorageAfter`');
        }
        if (props.scheduleExpression && !/^cron/.test(props.scheduleExpression.expressionString)) {
            throw new Error('`scheduleExpression` must be of type `cron`');
        }
    }
    /**
     * Daily with 35 days retention
     */
    static daily(backupVault) {
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Daily',
            scheduleExpression: events.Schedule.cron({
                hour: '5',
                minute: '0',
            }),
            deleteAfter: core_1.Duration.days(35),
        });
    }
    /**
     * Weekly with 3 months retention
     */
    static weekly(backupVault) {
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Weekly',
            scheduleExpression: events.Schedule.cron({
                hour: '5',
                minute: '0',
                weekDay: 'SAT',
            }),
            deleteAfter: core_1.Duration.days(30 * 3),
        });
    }
    /**
     * Monthly 1 year retention, move to cold storage after 1 month
     */
    static monthly1Year(backupVault) {
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly1Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30),
            deleteAfter: core_1.Duration.days(365),
        });
    }
    /**
     * Monthly 5 year retention, move to cold storage after 3 months
     */
    static monthly5Year(backupVault) {
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly5Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30 * 3),
            deleteAfter: core_1.Duration.days(365 * 5),
        });
    }
    /**
     * Monthly 7 year retention, move to cold storage after 3 months
     */
    static monthly7Year(backupVault) {
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly7Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30 * 3),
            deleteAfter: core_1.Duration.days(365 * 7),
        });
    }
}
exports.BackupPlanRule = BackupPlanRule;
//# sourceMappingURL=data:application/json;base64,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