"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupSelection = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const backup_generated_1 = require("./backup.generated");
const backupable_resources_collector_1 = require("./backupable-resources-collector");
const resource_1 = require("./resource");
/**
 * A backup selection
 */
class BackupSelection extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.listOfTags = [];
        this.resources = [];
        this.backupableResourcesCollector = new backupable_resources_collector_1.BackupableResourcesCollector();
        const role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('backup.amazonaws.com'),
        });
        role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBackupServiceRolePolicyForBackup'));
        if (props.allowRestores) {
            role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBackupServiceRolePolicyForRestores'));
        }
        this.grantPrincipal = role;
        const selection = new backup_generated_1.CfnBackupSelection(this, 'Resource', {
            backupPlanId: props.backupPlan.backupPlanId,
            backupSelection: {
                iamRoleArn: role.roleArn,
                selectionName: props.backupSelectionName || this.node.id,
                listOfTags: core_1.Lazy.anyValue({
                    produce: () => this.listOfTags,
                }, { omitEmptyArray: true }),
                resources: core_1.Lazy.listValue({
                    produce: () => [...this.resources, ...this.backupableResourcesCollector.resources],
                }, { omitEmpty: true }),
            },
        });
        this.backupPlanId = selection.attrBackupPlanId;
        this.selectionId = selection.attrSelectionId;
        for (const resource of props.resources) {
            this.addResource(resource);
        }
    }
    addResource(resource) {
        if (resource.tagCondition) {
            this.listOfTags.push({
                conditionKey: resource.tagCondition.key,
                conditionType: resource.tagCondition.operation || resource_1.TagOperation.STRING_EQUALS,
                conditionValue: resource.tagCondition.value,
            });
        }
        if (resource.resource) {
            this.resources.push(resource.resource);
        }
        if (resource.construct) {
            core_1.Aspects.of(resource.construct).add(this.backupableResourcesCollector);
            // Cannot push `this.backupableResourcesCollector.resources` to
            // `this.resources` here because it has not been evaluated yet.
            // Will be concatenated to `this.resources` in a `Lazy.listValue`
            // in the constructor instead.
        }
    }
}
exports.BackupSelection = BackupSelection;
//# sourceMappingURL=data:application/json;base64,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