"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupVault = exports.BackupVaultEvents = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const backup_generated_1 = require("./backup.generated");
/**
 * Backup vault events
 */
var BackupVaultEvents;
(function (BackupVaultEvents) {
    /** BACKUP_JOB_STARTED */
    BackupVaultEvents["BACKUP_JOB_STARTED"] = "BACKUP_JOB_STARTED";
    /** BACKUP_JOB_COMPLETED */
    BackupVaultEvents["BACKUP_JOB_COMPLETED"] = "BACKUP_JOB_COMPLETED";
    /** BACKUP_JOB_SUCCESSFUL */
    BackupVaultEvents["BACKUP_JOB_SUCCESSFUL"] = "BACKUP_JOB_SUCCESSFUL";
    /** BACKUP_JOB_FAILED */
    BackupVaultEvents["BACKUP_JOB_FAILED"] = "BACKUP_JOB_FAILED";
    /** BACKUP_JOB_EXPIRED */
    BackupVaultEvents["BACKUP_JOB_EXPIRED"] = "BACKUP_JOB_EXPIRED";
    /** RESTORE_JOB_STARTED */
    BackupVaultEvents["RESTORE_JOB_STARTED"] = "RESTORE_JOB_STARTED";
    /** RESTORE_JOB_COMPLETED */
    BackupVaultEvents["RESTORE_JOB_COMPLETED"] = "RESTORE_JOB_COMPLETED";
    /** RESTORE_JOB_SUCCESSFUL */
    BackupVaultEvents["RESTORE_JOB_SUCCESSFUL"] = "RESTORE_JOB_SUCCESSFUL";
    /** RESTORE_JOB_FAILED */
    BackupVaultEvents["RESTORE_JOB_FAILED"] = "RESTORE_JOB_FAILED";
    /** COPY_JOB_STARTED */
    BackupVaultEvents["COPY_JOB_STARTED"] = "COPY_JOB_STARTED";
    /** COPY_JOB_SUCCESSFUL */
    BackupVaultEvents["COPY_JOB_SUCCESSFUL"] = "COPY_JOB_SUCCESSFUL";
    /** COPY_JOB_FAILED */
    BackupVaultEvents["COPY_JOB_FAILED"] = "COPY_JOB_FAILED";
    /** RECOVERY_POINT_MODIFIED */
    BackupVaultEvents["RECOVERY_POINT_MODIFIED"] = "RECOVERY_POINT_MODIFIED";
    /** BACKUP_PLAN_CREATED */
    BackupVaultEvents["BACKUP_PLAN_CREATED"] = "BACKUP_PLAN_CREATED";
    /** BACKUP_PLAN_MODIFIED */
    BackupVaultEvents["BACKUP_PLAN_MODIFIED"] = "BACKUP_PLAN_MODIFIED";
})(BackupVaultEvents = exports.BackupVaultEvents || (exports.BackupVaultEvents = {}));
/**
 * A backup vault
 */
class BackupVault extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        if (props.backupVaultName && !/^[a-zA-Z0-9\-_]{2,50}$/.test(props.backupVaultName)) {
            throw new Error('Expected vault name to match pattern `^[a-zA-Z0-9\-_]{2,50}$`');
        }
        let notifications;
        if (props.notificationTopic) {
            notifications = {
                backupVaultEvents: props.notificationEvents || Object.values(BackupVaultEvents),
                snsTopicArn: props.notificationTopic.topicArn,
            };
            props.notificationTopic.grantPublish(new iam.ServicePrincipal('backup.amazonaws.com'));
        }
        const vault = new backup_generated_1.CfnBackupVault(this, 'Resource', {
            backupVaultName: props.backupVaultName || this.uniqueVaultName(),
            accessPolicy: props.accessPolicy && props.accessPolicy.toJSON(),
            encryptionKeyArn: props.encryptionKey && props.encryptionKey.keyArn,
            notifications,
        });
        vault.applyRemovalPolicy(props.removalPolicy);
        this.backupVaultName = vault.attrBackupVaultName;
        this.backupVaultArn = vault.attrBackupVaultArn;
    }
    /**
     * Import an existing backup vault
     */
    static fromBackupVaultName(scope, id, backupVaultName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.backupVaultName = backupVaultName;
            }
        }
        return new Import(scope, id);
    }
    uniqueVaultName() {
        // Max length of 50 chars, get the last 50 chars
        const id = this.node.uniqueId;
        return id.substring(Math.max(id.length - 50, 0), id.length);
    }
}
exports.BackupVault = BackupVault;
//# sourceMappingURL=data:application/json;base64,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