"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const events = require("@aws-cdk/aws-events");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
let stack;
beforeEach(() => {
    stack = new core_1.Stack();
});
test('create a plan and add rules', () => {
    // GIVEN
    const vault = new lib_1.BackupVault(stack, 'Vault');
    const otherVault = new lib_1.BackupVault(stack, 'OtherVault');
    // WHEN
    const plan = new lib_1.BackupPlan(stack, 'Plan', {
        backupVault: vault,
        backupPlanRules: [
            new lib_1.BackupPlanRule({
                completionWindow: core_1.Duration.hours(2),
                startWindow: core_1.Duration.hours(1),
                scheduleExpression: events.Schedule.cron({
                    day: '15',
                    hour: '3',
                    minute: '30',
                }),
                moveToColdStorageAfter: core_1.Duration.days(30),
            }),
        ],
    });
    plan.addRule(lib_1.BackupPlanRule.monthly5Year(otherVault));
    // THEN
    expect(stack).toHaveResource('AWS::Backup::BackupPlan', {
        BackupPlan: {
            BackupPlanName: 'Plan',
            BackupPlanRule: [
                {
                    CompletionWindowMinutes: 120,
                    Lifecycle: {
                        MoveToColdStorageAfterDays: 30,
                    },
                    RuleName: 'PlanRule0',
                    ScheduleExpression: 'cron(30 3 15 * ? *)',
                    StartWindowMinutes: 60,
                    TargetBackupVault: {
                        'Fn::GetAtt': [
                            'Vault23237E5B',
                            'BackupVaultName',
                        ],
                    },
                },
                {
                    Lifecycle: {
                        DeleteAfterDays: 1825,
                        MoveToColdStorageAfterDays: 90,
                    },
                    RuleName: 'Monthly5Year',
                    ScheduleExpression: 'cron(0 5 1 * ? *)',
                    TargetBackupVault: {
                        'Fn::GetAtt': [
                            'OtherVault3C99BCE2',
                            'BackupVaultName',
                        ],
                    },
                },
            ],
        },
    });
});
test('daily35DayRetention', () => {
    // WHEN
    lib_1.BackupPlan.daily35DayRetention(stack, 'D35');
    // THEN
    expect(stack).toHaveResource('AWS::Backup::BackupPlan', {
        BackupPlan: {
            BackupPlanName: 'D35',
            BackupPlanRule: [
                {
                    Lifecycle: {
                        DeleteAfterDays: 35,
                    },
                    RuleName: 'Daily',
                    ScheduleExpression: 'cron(0 5 * * ? *)',
                    TargetBackupVault: {
                        'Fn::GetAtt': [
                            'D35Vault2A9EB06F',
                            'BackupVaultName',
                        ],
                    },
                },
            ],
        },
    });
});
test('dailyWeeklyMonthly7YearRetention', () => {
    // WHEN
    lib_1.BackupPlan.dailyWeeklyMonthly7YearRetention(stack, 'DWM7');
    // THEN
    expect(stack).toHaveResource('AWS::Backup::BackupPlan', {
        BackupPlan: {
            BackupPlanName: 'DWM7',
            BackupPlanRule: [
                {
                    Lifecycle: {
                        DeleteAfterDays: 35,
                    },
                    RuleName: 'Daily',
                    ScheduleExpression: 'cron(0 5 * * ? *)',
                    TargetBackupVault: {
                        'Fn::GetAtt': [
                            'DWM7Vault21F17E61',
                            'BackupVaultName',
                        ],
                    },
                },
                {
                    Lifecycle: {
                        DeleteAfterDays: 90,
                    },
                    RuleName: 'Weekly',
                    ScheduleExpression: 'cron(0 5 ? * SAT *)',
                    TargetBackupVault: {
                        'Fn::GetAtt': [
                            'DWM7Vault21F17E61',
                            'BackupVaultName',
                        ],
                    },
                },
                {
                    Lifecycle: {
                        DeleteAfterDays: 2555,
                        MoveToColdStorageAfterDays: 90,
                    },
                    RuleName: 'Monthly7Year',
                    ScheduleExpression: 'cron(0 5 1 * ? *)',
                    TargetBackupVault: {
                        'Fn::GetAtt': [
                            'DWM7Vault21F17E61',
                            'BackupVaultName',
                        ],
                    },
                },
            ],
        },
    });
});
test('automatically creates a new vault', () => {
    // GIVEN
    const plan = new lib_1.BackupPlan(stack, 'Plan');
    // WHEN
    plan.addRule(lib_1.BackupPlanRule.daily());
    // THEN
    expect(stack).toHaveResource('AWS::Backup::BackupPlan', {
        BackupPlan: {
            BackupPlanName: 'Plan',
            BackupPlanRule: [
                {
                    Lifecycle: {
                        DeleteAfterDays: 35,
                    },
                    RuleName: 'Daily',
                    ScheduleExpression: 'cron(0 5 * * ? *)',
                    TargetBackupVault: {
                        'Fn::GetAtt': [
                            'PlanVault0284B0C2',
                            'BackupVaultName',
                        ],
                    },
                },
            ],
        },
    });
});
test('throws when deleteAfter is not greater than moveToColdStorageAfter', () => {
    expect(() => new lib_1.BackupPlanRule({
        deleteAfter: core_1.Duration.days(5),
        moveToColdStorageAfter: core_1.Duration.days(6),
    })).toThrow(/`deleteAfter` must be greater than `moveToColdStorageAfter`/);
});
test('throws when scheduleExpression is not of type cron', () => {
    expect(() => new lib_1.BackupPlanRule({
        scheduleExpression: events.Schedule.rate(core_1.Duration.hours(5)),
    })).toThrow(/`scheduleExpression` must be of type `cron`/);
});
test('synth fails when plan has no rules', () => {
    // GIVEN
    const app = new core_1.App();
    const myStack = new core_1.Stack(app, 'Stack');
    // WHEN
    new lib_1.BackupPlan(myStack, 'Plan');
    expect(() => app.synth()).toThrow(/A backup plan must have at least 1 rule/);
});
//# sourceMappingURL=data:application/json;base64,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