"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const ec2 = require("@aws-cdk/aws-ec2");
const efs = require("@aws-cdk/aws-efs");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
let stack;
let plan;
beforeEach(() => {
    stack = new core_1.Stack();
    plan = lib_1.BackupPlan.dailyWeeklyMonthly5YearRetention(stack, 'Plan');
});
test('create a selection', () => {
    // WHEN
    new lib_1.BackupSelection(stack, 'Selection', {
        backupPlan: plan,
        resources: [
            lib_1.BackupResource.fromArn('arn1'),
            lib_1.BackupResource.fromArn('arn2'),
            lib_1.BackupResource.fromTag('stage', 'prod'),
            lib_1.BackupResource.fromTag('cost center', 'cloud'),
        ],
    });
    // THEN
    expect(stack).toHaveResource('AWS::Backup::BackupSelection', {
        BackupPlanId: {
            'Fn::GetAtt': [
                'PlanDAF4E53A',
                'BackupPlanId',
            ],
        },
        BackupSelection: {
            IamRoleArn: {
                'Fn::GetAtt': [
                    'SelectionRoleD0EAEC83',
                    'Arn',
                ],
            },
            ListOfTags: [
                {
                    ConditionKey: 'stage',
                    ConditionType: 'STRINGEQUALS',
                    ConditionValue: 'prod',
                },
                {
                    ConditionKey: 'cost center',
                    ConditionType: 'STRINGEQUALS',
                    ConditionValue: 'cloud',
                },
            ],
            Resources: [
                'arn1',
                'arn2',
            ],
            SelectionName: 'Selection',
        },
    });
    expect(stack).toHaveResource('AWS::IAM::Role', {
        ManagedPolicyArns: [
            {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':iam::aws:policy/service-role/AWSBackupServiceRolePolicyForBackup',
                    ],
                ],
            },
        ],
    });
});
test('allow restores', () => {
    // WHEN
    new lib_1.BackupSelection(stack, 'Selection', {
        backupPlan: plan,
        resources: [
            lib_1.BackupResource.fromArn('arn1'),
        ],
        allowRestores: true,
    });
    // THEN
    expect(stack).toHaveResource('AWS::IAM::Role', {
        ManagedPolicyArns: [
            {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':iam::aws:policy/service-role/AWSBackupServiceRolePolicyForBackup',
                    ],
                ],
            },
            {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':iam::aws:policy/service-role/AWSBackupServiceRolePolicyForRestores',
                    ],
                ],
            },
        ],
    });
});
test('fromConstruct', () => {
    // GIVEN
    class EfsConstruct extends core_1.Construct {
        constructor(scope, id) {
            super(scope, id);
            new efs.CfnFileSystem(this, 'FileSystem');
        }
    }
    class MyConstruct extends core_1.Construct {
        constructor(scope, id) {
            super(scope, id);
            new dynamodb.Table(this, 'Table', {
                partitionKey: {
                    name: 'id',
                    type: dynamodb.AttributeType.STRING,
                },
            });
            new EfsConstruct(this, 'EFS');
        }
    }
    const myConstruct = new MyConstruct(stack, 'MyConstruct');
    const efsConstruct = new EfsConstruct(stack, 'EfsConstruct');
    // WHEN
    plan.addSelection('Selection', {
        resources: [
            lib_1.BackupResource.fromConstruct(myConstruct),
            lib_1.BackupResource.fromConstruct(efsConstruct),
        ],
    });
    // THEN
    expect(stack).toHaveResource('AWS::Backup::BackupSelection', {
        BackupSelection: {
            IamRoleArn: {
                'Fn::GetAtt': [
                    'PlanSelectionRole6D10F4B7',
                    'Arn',
                ],
            },
            Resources: [
                {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':dynamodb:',
                            {
                                Ref: 'AWS::Region',
                            },
                            ':',
                            {
                                Ref: 'AWS::AccountId',
                            },
                            ':table/',
                            {
                                Ref: 'MyConstructTable25959456',
                            },
                        ],
                    ],
                },
                {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':elasticfilesystem:',
                            {
                                Ref: 'AWS::Region',
                            },
                            ':',
                            {
                                Ref: 'AWS::AccountId',
                            },
                            ':file-system/',
                            {
                                Ref: 'MyConstructEFSFileSystemC68B6B78',
                            },
                        ],
                    ],
                },
                {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':elasticfilesystem:',
                            {
                                Ref: 'AWS::Region',
                            },
                            ':',
                            {
                                Ref: 'AWS::AccountId',
                            },
                            ':file-system/',
                            {
                                Ref: 'EfsConstructFileSystemFBE43F88',
                            },
                        ],
                    ],
                },
            ],
            SelectionName: 'Selection',
        },
    });
});
test('fromEc2Instance', () => {
    // GIVEN
    const vpc = new ec2.Vpc(stack, 'Vpc');
    const instance = new ec2.Instance(stack, 'Instance', {
        vpc,
        instanceType: ec2.InstanceType.of(ec2.InstanceClass.T3, ec2.InstanceSize.NANO),
        machineImage: new ec2.AmazonLinuxImage({ generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    // WHEN
    plan.addSelection('Selection', {
        resources: [
            lib_1.BackupResource.fromEc2Instance(instance),
        ],
    });
    // THEN
    expect(stack).toHaveResource('AWS::Backup::BackupSelection', {
        BackupSelection: {
            IamRoleArn: {
                'Fn::GetAtt': [
                    'PlanSelectionRole6D10F4B7',
                    'Arn',
                ],
            },
            Resources: [
                {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':ec2:',
                            {
                                Ref: 'AWS::Region',
                            },
                            ':',
                            {
                                Ref: 'AWS::AccountId',
                            },
                            ':instance/',
                            {
                                Ref: 'InstanceC1063A87',
                            },
                        ],
                    ],
                },
            ],
            SelectionName: 'Selection',
        },
    });
});
test('fromDynamoDbTable', () => {
    // GIVEN
    const newTable = new dynamodb.Table(stack, 'New', {
        partitionKey: {
            name: 'id',
            type: dynamodb.AttributeType.STRING,
        },
    });
    const existingTable = dynamodb.Table.fromTableArn(stack, 'Existing', 'arn:aws:dynamodb:eu-west-1:123456789012:table/existing');
    // WHEN
    plan.addSelection('Selection', {
        resources: [
            lib_1.BackupResource.fromDynamoDbTable(newTable),
            lib_1.BackupResource.fromDynamoDbTable(existingTable),
        ],
    });
    // THEN
    expect(stack).toHaveResource('AWS::Backup::BackupSelection', {
        BackupSelection: {
            IamRoleArn: {
                'Fn::GetAtt': [
                    'PlanSelectionRole6D10F4B7',
                    'Arn',
                ],
            },
            Resources: [
                {
                    'Fn::GetAtt': [
                        'New8A81B073',
                        'Arn',
                    ],
                },
                'arn:aws:dynamodb:eu-west-1:123456789012:table/existing',
            ],
            SelectionName: 'Selection',
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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