"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
let stack;
beforeEach(() => {
    stack = new core_1.Stack();
});
test('create a vault', () => {
    // WHEN
    new lib_1.BackupVault(stack, 'Vault');
    // THEN
    expect(stack).toHaveResource('AWS::Backup::BackupVault', {
        BackupVaultName: 'Vault',
    });
});
test('with access policy', () => {
    // GIVEN
    const accessPolicy = new iam.PolicyDocument({
        statements: [
            new iam.PolicyStatement({
                effect: iam.Effect.DENY,
                principals: [new iam.AnyPrincipal()],
                actions: ['backup:DeleteRecoveryPoint'],
                resources: ['*'],
                conditions: {
                    StringNotLike: {
                        'aws:userId': [
                            'user-arn',
                        ],
                    },
                },
            }),
        ],
    });
    // WHEN
    new lib_1.BackupVault(stack, 'Vault', {
        accessPolicy,
    });
    // THEN
    expect(stack).toHaveResource('AWS::Backup::BackupVault', {
        AccessPolicy: {
            Version: '2012-10-17',
            Statement: [
                {
                    Effect: 'Deny',
                    Principal: '*',
                    Action: 'backup:DeleteRecoveryPoint',
                    Resource: '*',
                    Condition: {
                        StringNotLike: {
                            'aws:userId': [
                                'user-arn',
                            ],
                        },
                    },
                },
            ],
        },
    });
});
test('with encryption key', () => {
    // GIVEN
    const encryptionKey = new kms.Key(stack, 'Key');
    // WHEN
    new lib_1.BackupVault(stack, 'Vault', {
        encryptionKey,
    });
    // THEN
    expect(stack).toHaveResource('AWS::Backup::BackupVault', {
        EncryptionKeyArn: {
            'Fn::GetAtt': [
                'Key961B73FD',
                'Arn',
            ],
        },
    });
});
test('with notifications', () => {
    // GIVEN
    const topic = new sns.Topic(stack, 'Topic');
    // WHEN
    new lib_1.BackupVault(stack, 'Vault', {
        notificationTopic: topic,
        notificationEvents: [
            lib_1.BackupVaultEvents.BACKUP_JOB_COMPLETED,
            lib_1.BackupVaultEvents.COPY_JOB_FAILED,
        ],
    });
    // THEN
    expect(stack).toHaveResource('AWS::Backup::BackupVault', {
        Notifications: {
            BackupVaultEvents: [
                'BACKUP_JOB_COMPLETED',
                'COPY_JOB_FAILED',
            ],
            SNSTopicArn: {
                Ref: 'TopicBFC7AF6E',
            },
        },
    });
});
test('defaults to all notifications', () => {
    // GIVEN
    const topic = new sns.Topic(stack, 'Topic');
    // WHEN
    new lib_1.BackupVault(stack, 'Vault', {
        notificationTopic: topic,
    });
    // THEN
    expect(stack).toHaveResource('AWS::Backup::BackupVault', {
        Notifications: {
            BackupVaultEvents: Object.values(lib_1.BackupVaultEvents),
            SNSTopicArn: {
                Ref: 'TopicBFC7AF6E',
            },
        },
    });
});
test('throws with invalid name', () => {
    expect(() => new lib_1.BackupVault(stack, 'Vault', {
        backupVaultName: 'Hello!Inv@lid',
    })).toThrow(/Expected vault name to match pattern/);
});
test('throws with whitespace in name', () => {
    expect(() => new lib_1.BackupVault(stack, 'Vault', {
        backupVaultName: 'Hello Invalid',
    })).toThrow(/Expected vault name to match pattern/);
});
test('throws with too short name', () => {
    expect(() => new lib_1.BackupVault(stack, 'Vault', {
        backupVaultName: 'x',
    })).toThrow(/Expected vault name to match pattern/);
});
//# sourceMappingURL=data:application/json;base64,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