import { AccessLevelList } from '../shared/access-level';
import { PolicyStatement, Operator } from '../shared';
/**
 * Statement provider for service [elasticache](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticache.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Elasticache extends PolicyStatement {
    servicePrefix: string;
    /**
     * Statement provider for service [elasticache](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticache.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid?: string);
    /**
     * Grants permission to add tags to an ElastiCache resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource(): this;
    /**
     * Grants permission to authorize an EC2 security group on a ElastiCache security group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupIngress
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_AuthorizeCacheSecurityGroupIngress.html
     */
    toAuthorizeCacheSecurityGroupIngress(): this;
    /**
     * Grants permission to apply ElastiCache service updates to sets of clusters and replication groups
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_BatchApplyUpdateAction.html
     */
    toBatchApplyUpdateAction(): this;
    /**
     * Grants permission to stop ElastiCache service updates from being executed on a set of clusters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_BatchStopUpdateAction.html
     */
    toBatchStopUpdateAction(): this;
    /**
     * Grants permission to complete an online migration of data from hosted Redis on Amazon EC2 to ElastiCache
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CompleteMigration.html
     */
    toCompleteMigration(): this;
    /**
     * Allows an IAM user or role to connect as a specified ElastiCache user to a node in a replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/auth-iam.html
     */
    toConnect(): this;
    /**
     * Grants permission to make a copy of an existing snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifKmsKeyId()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     * - s3:DeleteObject
     * - s3:GetBucketAcl
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CopySnapshot.html
     */
    toCopySnapshot(): this;
    /**
     * Grants permission to create a cache cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - elasticache:AddTagsToResource
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheCluster.html
     */
    toCreateCacheCluster(): this;
    /**
     * Grants permission to create a parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCacheParameterGroupName()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheParameterGroup.html
     */
    toCreateCacheParameterGroup(): this;
    /**
     * Grants permission to create a cache security group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheSecurityGroup.html
     */
    toCreateCacheSecurityGroup(): this;
    /**
     * Grants permission to create a cache subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheSubnetGroup.html
     */
    toCreateCacheSubnetGroup(): this;
    /**
     * Grants permission to create a global replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateGlobalReplicationGroup.html
     */
    toCreateGlobalReplicationGroup(): this;
    /**
     * Grants permission to create a replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - elasticache:AddTagsToResource
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateReplicationGroup.html
     */
    toCreateReplicationGroup(): this;
    /**
     * Grants permission to create a copy of an entire Redis cluster at a specific moment in time
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     * - s3:DeleteObject
     * - s3:GetBucketAcl
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateSnapshot.html
     */
    toCreateSnapshot(): this;
    /**
     * Grants permission to create a user for Redis. Users are supported from Redis 6.0 onwards
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifUserAuthenticationMode()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateUser.html
     */
    toCreateUser(): this;
    /**
     * Grants permission to create a user group for Redis. Groups are supported from Redis 6.0 onwards
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateUserGroup.html
     */
    toCreateUserGroup(): this;
    /**
     * Grants permission to decrease the number of node groups in global replication groups
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNumNodeGroups()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DecreaseNodeGroupsInGlobalReplicationGroup.html
     */
    toDecreaseNodeGroupsInGlobalReplicationGroup(): this;
    /**
     * Grants permission to decrease the number of replicas in a Redis (cluster mode disabled) replication group or the number of replica nodes in one or more node groups (shards) of a Redis (cluster mode enabled) replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifReplicasPerNodeGroup()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DecreaseReplicaCount.html
     */
    toDecreaseReplicaCount(): this;
    /**
     * Grants permission to delete a previously provisioned cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheCluster.html
     */
    toDeleteCacheCluster(): this;
    /**
     * Grants permission to delete the specified cache parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifCacheParameterGroupName()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheParameterGroup.html
     */
    toDeleteCacheParameterGroup(): this;
    /**
     * Grants permission to delete a cache security group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheSecurityGroup.html
     */
    toDeleteCacheSecurityGroup(): this;
    /**
     * Grants permission to delete a cache subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheSubnetGroup.html
     */
    toDeleteCacheSubnetGroup(): this;
    /**
     * Grants permission to delete an existing global replication group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteGlobalReplicationGroup.html
     */
    toDeleteGlobalReplicationGroup(): this;
    /**
     * Grants permission to delete an existing replication group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteReplicationGroup.html
     */
    toDeleteReplicationGroup(): this;
    /**
     * Grants permission to delete an existing snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteSnapshot.html
     */
    toDeleteSnapshot(): this;
    /**
     * Grants permission to delete an existing user and thus remove it from all user groups and replication groups where it was assigned
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser(): this;
    /**
     * Grants permission to delete an existing user group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteUserGroup.html
     */
    toDeleteUserGroup(): this;
    /**
     * Grants permission to list information about provisioned cache clusters
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheClusters.html
     */
    toDescribeCacheClusters(): this;
    /**
     * Grants permission to list available cache engines and their versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheEngineVersions.html
     */
    toDescribeCacheEngineVersions(): this;
    /**
     * Grants permission to list cache parameter group descriptions
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheParameterGroups.html
     */
    toDescribeCacheParameterGroups(): this;
    /**
     * Grants permission to retrieve the detailed parameter list for a particular cache parameter group
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheParameters.html
     */
    toDescribeCacheParameters(): this;
    /**
     * Grants permission to list cache security group descriptions
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheSecurityGroups.html
     */
    toDescribeCacheSecurityGroups(): this;
    /**
     * Grants permission to list cache subnet group descriptions
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheSubnetGroups.html
     */
    toDescribeCacheSubnetGroups(): this;
    /**
     * Grants permission to retrieve the default engine and system parameter information for the specified cache engine
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeEngineDefaultParameters.html
     */
    toDescribeEngineDefaultParameters(): this;
    /**
     * Grants permission to list events related to clusters, cache security groups, and cache parameter groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents(): this;
    /**
     * Grants permission to list information about global replication groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeGlobalReplicationGroups.html
     */
    toDescribeGlobalReplicationGroups(): this;
    /**
     * Grants permission to list information about provisioned replication groups
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeReplicationGroups.html
     */
    toDescribeReplicationGroups(): this;
    /**
     * Grants permission to list information about purchased reserved cache nodes
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeReservedCacheNodes.html
     */
    toDescribeReservedCacheNodes(): this;
    /**
     * Grants permission to list available reserved cache node offerings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeReservedCacheNodesOfferings.html
     */
    toDescribeReservedCacheNodesOfferings(): this;
    /**
     * Grants permission to list details of the service updates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeServiceUpdates.html
     */
    toDescribeServiceUpdates(): this;
    /**
     * Grants permission to list information about cluster or replication group snapshots
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeSnapshots.html
     */
    toDescribeSnapshots(): this;
    /**
     * Grants permission to list details of the update actions for a set of clusters or replication groups
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeUpdateActions.html
     */
    toDescribeUpdateActions(): this;
    /**
     * Grants permission to list information about Redis user groups
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeUserGroups.html
     */
    toDescribeUserGroups(): this;
    /**
     * Grants permission to list information about Redis users
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeUsers.html
     */
    toDescribeUsers(): this;
    /**
     * Grants permission to remove a secondary replication group from the global replication group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DisassociateGlobalReplicationGroup.html
     */
    toDisassociateGlobalReplicationGroup(): this;
    /**
     * Grants permission to failover the primary region to a selected secondary region of a global replication group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_FailoverGlobalReplicationGroup.html
     */
    toFailoverGlobalReplicationGroup(): this;
    /**
     * Grants permission to increase the number of node groups in a global replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNumNodeGroups()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_IncreaseNodeGroupsInGlobalReplicationGroup.html
     */
    toIncreaseNodeGroupsInGlobalReplicationGroup(): this;
    /**
     * Grants permission to increase the number of replicas in a Redis (cluster mode disabled) replication group or the number of replica nodes in one or more node groups (shards) of a Redis (cluster mode enabled) replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifReplicasPerNodeGroup()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_IncreaseReplicaCount.html
     */
    toIncreaseReplicaCount(): this;
    /**
     * Grants permission to list available node type that can be used to scale a particular Redis cluster or replication group
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ListAllowedNodeTypeModifications.html
     */
    toListAllowedNodeTypeModifications(): this;
    /**
     * Grants permission to list tags for an ElastiCache resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to modify settings for a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheCluster.html
     */
    toModifyCacheCluster(): this;
    /**
     * Grants permission to modify parameters of a cache parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifCacheParameterGroupName()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheParameterGroup.html
     */
    toModifyCacheParameterGroup(): this;
    /**
     * Grants permission to modify an existing cache subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheSubnetGroup.html
     */
    toModifyCacheSubnetGroup(): this;
    /**
     * Grants permission to modify settings for a global replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCacheNodeType()
     * - .ifEngineVersion()
     * - .ifAutomaticFailoverEnabled()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyGlobalReplicationGroup.html
     */
    toModifyGlobalReplicationGroup(): this;
    /**
     * Grants permission to modify the settings for a replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyReplicationGroup.html
     */
    toModifyReplicationGroup(): this;
    /**
     * Grants permission to add shards, remove shards, or rebalance the keyspaces among existing shards of a replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifNumNodeGroups()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyReplicationGroupShardConfiguration.html
     */
    toModifyReplicationGroupShardConfiguration(): this;
    /**
     * Grants permission to change Redis user password(s) and/or access string
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifUserAuthenticationMode()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyUser.html
     */
    toModifyUser(): this;
    /**
     * Grants permission to change list of users that belong to the user group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyUserGroup.html
     */
    toModifyUserGroup(): this;
    /**
     * Grants permission to purchase a reserved cache node offering
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_PurchaseReservedCacheNodesOffering.html
     */
    toPurchaseReservedCacheNodesOffering(): this;
    /**
     * Grants permission to perform a key space rebalance operation to redistribute slots and ensure uniform key distribution across existing shards in a global replication group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RebalanceSlotsInGlobalReplicationGroup.html
     */
    toRebalanceSlotsInGlobalReplicationGroup(): this;
    /**
     * Grants permission to reboot some, or all, of the cache nodes within a provisioned cache cluster or replication group (cluster mode disabled)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RebootCacheCluster.html
     */
    toRebootCacheCluster(): this;
    /**
     * Grants permission to remove tags from a ElastiCache resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource(): this;
    /**
     * Grants permission to modify parameters of a cache parameter group back to their default values
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifCacheParameterGroupName()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ResetCacheParameterGroup.html
     */
    toResetCacheParameterGroup(): this;
    /**
     * Grants permission to remove an EC2 security group ingress from a ElastiCache security group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RevokeCacheSecurityGroupIngress.html
     */
    toRevokeCacheSecurityGroupIngress(): this;
    /**
     * Grants permission to start a migration of data from hosted Redis on Amazon EC2 to ElastiCache for Redis
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_StartMigration.html
     */
    toStartMigration(): this;
    /**
     * Grants permission to test automatic failover on a specified node group in a replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_TestFailover.html
     */
    toTestFailover(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type parametergroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html#WhatIs.Components.ParameterGroups
     *
     * @param cacheParameterGroupName - Identifier for the cacheParameterGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifCacheParameterGroupName()
     */
    onParametergroup(cacheParameterGroupName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type securitygroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html#WhatIs.Components.SecurityGroups
     *
     * @param cacheSecurityGroupName - Identifier for the cacheSecurityGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onSecuritygroup(cacheSecurityGroupName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type subnetgroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html#WhatIs.Components.SubnetGroups
     *
     * @param cacheSubnetGroupName - Identifier for the cacheSubnetGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onSubnetgroup(cacheSubnetGroupName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type replicationgroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html#WhatIs.Components.ReplicationGroups
     *
     * @param replicationGroupId - Identifier for the replicationGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAtRestEncryptionEnabled()
     * - .ifAuthTokenEnabled()
     * - .ifAutomaticFailoverEnabled()
     * - .ifCacheNodeType()
     * - .ifCacheParameterGroupName()
     * - .ifClusterModeEnabled()
     * - .ifEngineType()
     * - .ifEngineVersion()
     * - .ifKmsKeyId()
     * - .ifMultiAZEnabled()
     * - .ifNumNodeGroups()
     * - .ifReplicasPerNodeGroup()
     * - .ifSnapshotRetentionLimit()
     * - .ifTransitEncryptionEnabled()
     */
    onReplicationgroup(replicationGroupId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html#WhatIs.Components.Clusters
     *
     * @param cacheClusterId - Identifier for the cacheClusterId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAuthTokenEnabled()
     * - .ifCacheNodeType()
     * - .ifCacheParameterGroupName()
     * - .ifEngineType()
     * - .ifEngineVersion()
     * - .ifMultiAZEnabled()
     * - .ifSnapshotRetentionLimit()
     */
    onCluster(cacheClusterId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type reserved-instance to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/reserved-nodes.html
     *
     * @param reservedCacheNodeId - Identifier for the reservedCacheNodeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onReservedInstance(reservedCacheNodeId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html#WhatIs.Components.Snapshots
     *
     * @param snapshotName - Identifier for the snapshotName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifKmsKeyId()
     */
    onSnapshot(snapshotName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type globalreplicationgroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Redis-Global-Datastore.html
     *
     * @param globalReplicationGroupId - Identifier for the globalReplicationGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAtRestEncryptionEnabled()
     * - .ifAuthTokenEnabled()
     * - .ifAutomaticFailoverEnabled()
     * - .ifCacheNodeType()
     * - .ifCacheParameterGroupName()
     * - .ifClusterModeEnabled()
     * - .ifEngineType()
     * - .ifEngineVersion()
     * - .ifKmsKeyId()
     * - .ifMultiAZEnabled()
     * - .ifNumNodeGroups()
     * - .ifReplicasPerNodeGroup()
     * - .ifSnapshotRetentionLimit()
     * - .ifTransitEncryptionEnabled()
     */
    onGlobalreplicationgroup(globalReplicationGroupId: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Clusters.RBAC.html
     *
     * @param userId - Identifier for the userId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifUserAuthenticationMode()
     */
    onUser(userId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type usergroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Clusters.RBAC.html
     *
     * @param userGroupId - Identifier for the userGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onUsergroup(userGroupId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the AtRestEncryptionEnabled parameter present in the request or default false value if parameter is not present
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - globalreplicationgroup
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAtRestEncryptionEnabled(value?: boolean): this;
    /**
     * Filters access by the presence of non empty AuthToken parameter in the request
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateCacheCluster()
     * - .toCreateReplicationGroup()
     * - .toModifyCacheCluster()
     * - .toModifyReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - cluster
     * - globalreplicationgroup
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAuthTokenEnabled(value?: boolean): this;
    /**
     * Filters access by the AutomaticFailoverEnabled parameter in the request
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateReplicationGroup()
     * - .toModifyGlobalReplicationGroup()
     * - .toModifyReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - globalreplicationgroup
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAutomaticFailoverEnabled(value?: boolean): this;
    /**
     * Filters access by the cacheNodeType parameter present in the request. This key can be used to restrict which cache node types can be used on cluster creation or scaling operations
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateCacheCluster()
     * - .toCreateReplicationGroup()
     * - .toModifyCacheCluster()
     * - .toModifyGlobalReplicationGroup()
     * - .toModifyReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - cluster
     * - globalreplicationgroup
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCacheNodeType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the CacheParameterGroupName parameter in the request
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateCacheCluster()
     * - .toCreateCacheParameterGroup()
     * - .toCreateReplicationGroup()
     * - .toDeleteCacheParameterGroup()
     * - .toModifyCacheCluster()
     * - .toModifyCacheParameterGroup()
     * - .toModifyReplicationGroup()
     * - .toResetCacheParameterGroup()
     *
     * Applies to resource types:
     * - parametergroup
     * - replicationgroup
     * - cluster
     * - globalreplicationgroup
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCacheParameterGroupName(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the cluster mode parameter present in the request. Default value for single node group (shard) creations is false
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - globalreplicationgroup
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifClusterModeEnabled(value?: boolean): this;
    /**
     * Filters access by the engine type present in creation requests. For replication group creations, default engine 'redis' is used as key if parameter is not present
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateCacheCluster()
     * - .toCreateReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - cluster
     * - globalreplicationgroup
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEngineType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the engineVersion parameter present in creation or cluster modification requests
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateCacheCluster()
     * - .toCreateReplicationGroup()
     * - .toModifyCacheCluster()
     * - .toModifyGlobalReplicationGroup()
     * - .toModifyReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - cluster
     * - globalreplicationgroup
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEngineVersion(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the KmsKeyId parameter in the request
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCopySnapshot()
     * - .toCreateReplicationGroup()
     * - .toCreateSnapshot()
     *
     * Applies to resource types:
     * - replicationgroup
     * - snapshot
     * - globalreplicationgroup
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKmsKeyId(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the AZMode parameter, MultiAZEnabled parameter or the number of availability zones that the cluster or replication group can be placed in
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateCacheCluster()
     * - .toCreateReplicationGroup()
     * - .toModifyCacheCluster()
     * - .toModifyReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - cluster
     * - globalreplicationgroup
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifMultiAZEnabled(value?: boolean): this;
    /**
     * Filters access by the NumNodeGroups or NodeGroupCount parameter specified in the request. This key can be used to restrict the number of node groups (shards) clusters can have after creation or scaling operations
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateReplicationGroup()
     * - .toDecreaseNodeGroupsInGlobalReplicationGroup()
     * - .toIncreaseNodeGroupsInGlobalReplicationGroup()
     * - .toModifyReplicationGroupShardConfiguration()
     *
     * Applies to resource types:
     * - replicationgroup
     * - globalreplicationgroup
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifNumNodeGroups(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by the number of replicas per node group (shards) specified in creations or scaling requests
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateReplicationGroup()
     * - .toDecreaseReplicaCount()
     * - .toIncreaseReplicaCount()
     *
     * Applies to resource types:
     * - replicationgroup
     * - globalreplicationgroup
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifReplicasPerNodeGroup(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by the SnapshotRetentionLimit parameter in the request
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateCacheCluster()
     * - .toCreateReplicationGroup()
     * - .toModifyCacheCluster()
     * - .toModifyReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - cluster
     * - globalreplicationgroup
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSnapshotRetentionLimit(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by the TransitEncryptionEnabled parameter present in the request. For replication group creations, default value 'false' is used as key if parameter is not present
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateReplicationGroup()
     * - .toModifyReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - globalreplicationgroup
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifTransitEncryptionEnabled(value?: boolean): this;
    /**
     * Filters access by the UserAuthenticationMode parameter in the request
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateUser()
     * - .toModifyUser()
     *
     * Applies to resource types:
     * - user
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserAuthenticationMode(value: string | string[], operator?: Operator | string): this;
}
