import { AccessLevelList } from '../shared/access-level';
import { PolicyStatement, Operator } from '../shared';
/**
 * Statement provider for service [rds](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonrds.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Rds extends PolicyStatement {
    servicePrefix: string;
    /**
     * Statement provider for service [rds](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonrds.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid?: string);
    /**
     * Grants permission to associate an Identity and Access Management (IAM) role from an Aurora DB cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddRoleToDBCluster.html
     */
    toAddRoleToDBCluster(): this;
    /**
     * Grants permission to associate an AWS Identity and Access Management (IAM) role with a DB instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddRoleToDBInstance.html
     */
    toAddRoleToDBInstance(): this;
    /**
     * Grants permission to add a source identifier to an existing RDS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddSourceIdentifierToSubscription.html
     */
    toAddSourceIdentifierToSubscription(): this;
    /**
     * Grants permission to add metadata tags to an Amazon RDS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource(): this;
    /**
     * Grants permission to apply a pending maintenance action to a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ApplyPendingMaintenanceAction.html
     */
    toApplyPendingMaintenanceAction(): this;
    /**
     * Grants permission to enable ingress to a DBSecurityGroup using one of two forms of authorization
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AuthorizeDBSecurityGroupIngress.html
     */
    toAuthorizeDBSecurityGroupIngress(): this;
    /**
     * Grants permission to backtrack a DB cluster to a specific time, without creating a new DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_BacktrackDBCluster.html
     */
    toBacktrackDBCluster(): this;
    /**
     * Grants permission to cancel an export task in progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CancelExportTask.html
     */
    toCancelExportTask(): this;
    /**
     * Grants permission to copy the specified DB cluster parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBClusterParameterGroup.html
     */
    toCopyDBClusterParameterGroup(): this;
    /**
     * Grants permission to create a snapshot of a DB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBClusterSnapshot.html
     */
    toCopyDBClusterSnapshot(): this;
    /**
     * Grants permission to copy the specified DB parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBParameterGroup.html
     */
    toCopyDBParameterGroup(): this;
    /**
     * Grants permission to copy the specified DB snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCopyOptionGroup()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyDBSnapshot.html
     */
    toCopyDBSnapshot(): this;
    /**
     * Grants permission to copy the specified option group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CopyOptionGroup.html
     */
    toCopyOptionGroup(): this;
    /**
     * Grants permission to create a blue-green deployment for a given source cluster or instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifClusterTag()
     * - .ifClusterPgTag()
     * - .ifDbTag()
     * - .ifPgTag()
     * - .ifReqTag()
     * - .ifDatabaseEngine()
     * - .ifDatabaseName()
     * - .ifStorageEncrypted()
     * - .ifDatabaseClass()
     * - .ifStorageSize()
     * - .ifMultiAz()
     * - .ifPiops()
     * - .ifVpc()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     * - rds:CreateDBCluster
     * - rds:CreateDBClusterEndpoint
     * - rds:CreateDBInstance
     * - rds:CreateDBInstanceReadReplica
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateBlueGreenDeployment.html
     */
    toCreateBlueGreenDeployment(): this;
    /**
     * Grants permission to create a custom engine version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - mediaimport:CreateDatabaseBinarySnapshot
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateCustomDBEngineVersion.html
     */
    toCreateCustomDBEngineVersion(): this;
    /**
     * Grants permission to create a new Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     * - .ifDatabaseEngine()
     * - .ifDatabaseName()
     * - .ifStorageEncrypted()
     * - .ifDatabaseClass()
     * - .ifStorageSize()
     * - .ifPiops()
     * - .ifManageMasterUserPassword()
     *
     * Dependent actions:
     * - iam:PassRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - rds:AddTagsToResource
     * - rds:CreateDBInstance
     * - secretsmanager:CreateSecret
     * - secretsmanager:TagResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBCluster.html
     */
    toCreateDBCluster(): this;
    /**
     * Grants permission to create a new custom endpoint and associates it with an Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEndpointType()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBClusterEndpoint.html
     */
    toCreateDBClusterEndpoint(): this;
    /**
     * Grants permission to create a new DB cluster parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBClusterParameterGroup.html
     */
    toCreateDBClusterParameterGroup(): this;
    /**
     * Grants permission to create a snapshot of a DB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBClusterSnapshot.html
     */
    toCreateDBClusterSnapshot(): this;
    /**
     * Grants permission to create a new DB instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBackupTarget()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     * - .ifManageMasterUserPassword()
     *
     * Dependent actions:
     * - iam:PassRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - rds:AddTagsToResource
     * - secretsmanager:CreateSecret
     * - secretsmanager:TagResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html
     */
    toCreateDBInstance(): this;
    /**
     * Grants permission to create a DB instance that acts as a Read Replica of a source DB instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstanceReadReplica.html
     */
    toCreateDBInstanceReadReplica(): this;
    /**
     * Grants permission to create a new DB parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBParameterGroup.html
     */
    toCreateDBParameterGroup(): this;
    /**
     * Grants permission to create a database proxy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBProxy.html
     */
    toCreateDBProxy(): this;
    /**
     * Grants permission to create a database proxy endpoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBProxyEndpoint.html
     */
    toCreateDBProxyEndpoint(): this;
    /**
     * Grants permission to create a new DB security group. DB security groups control access to a DB instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBSecurityGroup.html
     */
    toCreateDBSecurityGroup(): this;
    /**
     * Grants permission to create a DBSnapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBackupTarget()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBSnapshot.html
     */
    toCreateDBSnapshot(): this;
    /**
     * Grants permission to create a new DB subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBSubnetGroup.html
     */
    toCreateDBSubnetGroup(): this;
    /**
     * Grants permission to create an RDS event notification subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateEventSubscription.html
     */
    toCreateEventSubscription(): this;
    /**
     * Grants permission to create an Aurora global database spread across multiple regions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateGlobalCluster.html
     */
    toCreateGlobalCluster(): this;
    /**
     * Grants permission to create a new option group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateOptionGroup.html
     */
    toCreateOptionGroup(): this;
    /**
     * Grants permission to access a resource in the remote Region when executing cross-Region operations, such as cross-Region snapshot copy or cross-Region read replica creation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     */
    toCrossRegionCommunication(): this;
    /**
     * Grants permission to delete blue green deployments
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:DeleteDBCluster
     * - rds:DeleteDBClusterEndpoint
     * - rds:DeleteDBInstance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteBlueGreenDeployment.html
     */
    toDeleteBlueGreenDeployment(): this;
    /**
     * Grants permission to delete an existing custom engine version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteCustomDBEngineVersion.html
     */
    toDeleteCustomDBEngineVersion(): this;
    /**
     * Grants permission to delete a previously provisioned DB cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - rds:DeleteDBInstance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBCluster.html
     */
    toDeleteDBCluster(): this;
    /**
     * Grants permission to delete a custom endpoint and removes it from an Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBClusterEndpoint.html
     */
    toDeleteDBClusterEndpoint(): this;
    /**
     * Grants permission to delete a specified DB cluster parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBClusterParameterGroup.html
     */
    toDeleteDBClusterParameterGroup(): this;
    /**
     * Grants permission to delete a DB cluster snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBClusterSnapshot.html
     */
    toDeleteDBClusterSnapshot(): this;
    /**
     * Grants permission to delete a previously provisioned DB instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBInstance.html
     */
    toDeleteDBInstance(): this;
    /**
     * Grants permission to deletes automated backups based on the source instance's DbiResourceId value or the restorable instance's resource ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBInstanceAutomatedBackup.html
     */
    toDeleteDBInstanceAutomatedBackup(): this;
    /**
     * Grants permission to delete a specified DBParameterGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBParameterGroup.html
     */
    toDeleteDBParameterGroup(): this;
    /**
     * Grants permission to delete a database proxy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBProxy.html
     */
    toDeleteDBProxy(): this;
    /**
     * Grants permission to delete a database proxy endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBProxyEndpoint.html
     */
    toDeleteDBProxyEndpoint(): this;
    /**
     * Grants permission to delete a DB security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBSecurityGroup.html
     */
    toDeleteDBSecurityGroup(): this;
    /**
     * Grants permission to delete a DBSnapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBSnapshot.html
     */
    toDeleteDBSnapshot(): this;
    /**
     * Grants permission to delete a DB subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBSubnetGroup.html
     */
    toDeleteDBSubnetGroup(): this;
    /**
     * Grants permission to delete an RDS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteEventSubscription.html
     */
    toDeleteEventSubscription(): this;
    /**
     * Grants permission to delete a global database cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteGlobalCluster.html
     */
    toDeleteGlobalCluster(): this;
    /**
     * Grants permission to delete an existing option group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteOptionGroup.html
     */
    toDeleteOptionGroup(): this;
    /**
     * Grants permission to remove targets from a database proxy target group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeregisterDBProxyTargets.html
     */
    toDeregisterDBProxyTargets(): this;
    /**
     * Grants permission to list all of the attributes for a customer account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeAccountAttributes.html
     */
    toDescribeAccountAttributes(): this;
    /**
     * Grants permission to describe blue green deployments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeBlueGreenDeployments.html
     */
    toDescribeBlueGreenDeployments(): this;
    /**
     * Grants permission to list the set of CA certificates provided by Amazon RDS for this AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeCertificates.html
     */
    toDescribeCertificates(): this;
    /**
     * Grants permission to return information about backtracks for a DB cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterBacktracks.html
     */
    toDescribeDBClusterBacktracks(): this;
    /**
     * Grants permission to return information about endpoints for an Amazon Aurora DB cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterEndpoints.html
     */
    toDescribeDBClusterEndpoints(): this;
    /**
     * Grants permission to return a list of DBClusterParameterGroup descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterParameterGroups.html
     */
    toDescribeDBClusterParameterGroups(): this;
    /**
     * Grants permission to return the detailed parameter list for a particular DB cluster parameter group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterParameters.html
     */
    toDescribeDBClusterParameters(): this;
    /**
     * Grants permission to return a list of DB cluster snapshot attribute names and values for a manual DB cluster snapshot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterSnapshotAttributes.html
     */
    toDescribeDBClusterSnapshotAttributes(): this;
    /**
     * Grants permission to return information about DB cluster snapshots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusterSnapshots.html
     */
    toDescribeDBClusterSnapshots(): this;
    /**
     * Grants permission to return information about provisioned Aurora DB clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusters.html
     */
    toDescribeDBClusters(): this;
    /**
     * Grants permission to return a list of the available DB engines
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBEngineVersions.html
     */
    toDescribeDBEngineVersions(): this;
    /**
     * Grants permission to return a list of automated backups for both current and deleted instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstanceAutomatedBackups.html
     */
    toDescribeDBInstanceAutomatedBackups(): this;
    /**
     * Grants permission to return information about provisioned RDS instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstances.html
     */
    toDescribeDBInstances(): this;
    /**
     * Grants permission to return a list of DB log files for the DB instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBLogFiles.html
     */
    toDescribeDBLogFiles(): this;
    /**
     * Grants permission to return a list of DBParameterGroup descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBParameterGroups.html
     */
    toDescribeDBParameterGroups(): this;
    /**
     * Grants permission to return the detailed parameter list for a particular DB parameter group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBParameters.html
     */
    toDescribeDBParameters(): this;
    /**
     * Grants permission to view proxies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxies.html
     */
    toDescribeDBProxies(): this;
    /**
     * Grants permission to view proxy endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxyEndpoints.html
     */
    toDescribeDBProxyEndpoints(): this;
    /**
     * Grants permission to view database proxy target group details
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxyTargetGroups.html
     */
    toDescribeDBProxyTargetGroups(): this;
    /**
     * Grants permission to view database proxy target details
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBProxyTargets.html
     */
    toDescribeDBProxyTargets(): this;
    /**
     * Grants permission to return a list of DBSecurityGroup descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSecurityGroups.html
     */
    toDescribeDBSecurityGroups(): this;
    /**
     * Grants permission to return a list of DB snapshot attribute names and values for a manual DB snapshot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSnapshotAttributes.html
     */
    toDescribeDBSnapshotAttributes(): this;
    /**
     * Grants permission to return information about DB snapshots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSnapshots.html
     */
    toDescribeDBSnapshots(): this;
    /**
     * Grants permission to return a list of DBSubnetGroup descriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSubnetGroups.html
     */
    toDescribeDBSubnetGroups(): this;
    /**
     * Grants permission to return the default engine and system parameter information for the cluster database engine
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEngineDefaultClusterParameters.html
     */
    toDescribeEngineDefaultClusterParameters(): this;
    /**
     * Grants permission to return the default engine and system parameter information for the specified database engine
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEngineDefaultParameters.html
     */
    toDescribeEngineDefaultParameters(): this;
    /**
     * Grants permission to display a list of categories for all event source types, or, if specified, for a specified source type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEventCategories.html
     */
    toDescribeEventCategories(): this;
    /**
     * Grants permission to list all the subscription descriptions for a customer account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEventSubscriptions.html
     */
    toDescribeEventSubscriptions(): this;
    /**
     * Grants permission to return events related to DB instances, DB security groups, DB snapshots, and DB parameter groups for the past 14 days
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents(): this;
    /**
     * Grants permission to return information about the export tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeExportTasks.html
     */
    toDescribeExportTasks(): this;
    /**
     * Grants permission to return information about Aurora global database clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeGlobalClusters.html
     */
    toDescribeGlobalClusters(): this;
    /**
     * Grants permission to describe all available options
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeOptionGroupOptions.html
     */
    toDescribeOptionGroupOptions(): this;
    /**
     * Grants permission to describe the available option groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeOptionGroups.html
     */
    toDescribeOptionGroups(): this;
    /**
     * Grants permission to return a list of orderable DB instance options for the specified engine
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeOrderableDBInstanceOptions.html
     */
    toDescribeOrderableDBInstanceOptions(): this;
    /**
     * Grants permission to return a list of resources (for example, DB instances) that have at least one pending maintenance action
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribePendingMaintenanceActions.html
     */
    toDescribePendingMaintenanceActions(): this;
    /**
     * Grants permission to return information about recommendation groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/USER_Recommendations.html
     */
    toDescribeRecommendationGroups(): this;
    /**
     * Grants permission to return information about recommendations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/USER_Recommendations.html
     */
    toDescribeRecommendations(): this;
    /**
     * Grants permission to return information about reserved DB instances for this account, or about a specified reserved DB instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeReservedDBInstances.html
     */
    toDescribeReservedDBInstances(): this;
    /**
     * Grants permission to list available reserved DB instance offerings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeReservedDBInstancesOfferings.html
     */
    toDescribeReservedDBInstancesOfferings(): this;
    /**
     * Grants permission to return a list of the source AWS Regions where the current AWS Region can create a Read Replica or copy a DB snapshot from
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeSourceRegions.html
     */
    toDescribeSourceRegions(): this;
    /**
     * Grants permission to list available modifications you can make to your DB instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeValidDBInstanceModifications.html
     */
    toDescribeValidDBInstanceModifications(): this;
    /**
     * Grants permission to download specified log file
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/USER_LogAccess.html
     */
    toDownloadCompleteDBLogFile(): this;
    /**
     * Grants permission to download all or a portion of the specified log file, up to 1 MB in size
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DownloadDBLogFilePortion.html
     */
    toDownloadDBLogFilePortion(): this;
    /**
     * Grants permission to force a failover for a DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_FailoverDBCluster.html
     */
    toFailoverDBCluster(): this;
    /**
     * Grants permission to failover a global cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_FailoverGlobalCluster.html
     */
    toFailoverGlobalCluster(): this;
    /**
     * Grants permission to list all tags on an Amazon RDS resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to modify a database activity stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyActivityStream.html
     */
    toModifyActivityStream(): this;
    /**
     * Grants permission to modify the system-default Secure Sockets Layer/Transport Layer Security (SSL/TLS) certificate for Amazon RDS for new DB instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyCertificates.html
     */
    toModifyCertificates(): this;
    /**
     * Grants permission to modify current cluster capacity for an Amazon Aurora Severless DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyCurrentDBClusterCapacity.html
     */
    toModifyCurrentDBClusterCapacity(): this;
    /**
     * Grants permission to modify an existing custom engine version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyCustomDBEngineVersion.html
     */
    toModifyCustomDBEngineVersion(): this;
    /**
     * Grants permission to modify a setting for an Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDatabaseClass()
     * - .ifStorageSize()
     * - .ifPiops()
     * - .ifManageMasterUserPassword()
     *
     * Dependent actions:
     * - iam:PassRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - rds:ModifyDBInstance
     * - secretsmanager:CreateSecret
     * - secretsmanager:RotateSecret
     * - secretsmanager:TagResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBCluster.html
     */
    toModifyDBCluster(): this;
    /**
     * Grants permission to modify the properties of an endpoint in an Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBClusterEndpoint.html
     */
    toModifyDBClusterEndpoint(): this;
    /**
     * Grants permission to modify the parameters of a DB cluster parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBClusterParameterGroup.html
     */
    toModifyDBClusterParameterGroup(): this;
    /**
     * Grants permission to add an attribute and values to, or removes an attribute and values from, a manual DB cluster snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBClusterSnapshotAttribute.html
     */
    toModifyDBClusterSnapshotAttribute(): this;
    /**
     * Grants permission to modify settings for a DB instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifManageMasterUserPassword()
     *
     * Dependent actions:
     * - iam:PassRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - secretsmanager:CreateSecret
     * - secretsmanager:RotateSecret
     * - secretsmanager:TagResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBInstance.html
     */
    toModifyDBInstance(): this;
    /**
     * Grants permission to modify the parameters of a DB parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBParameterGroup.html
     */
    toModifyDBParameterGroup(): this;
    /**
     * Grants permission to modify database proxy
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBProxy.html
     */
    toModifyDBProxy(): this;
    /**
     * Grants permission to modify database proxy endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBProxyEndpoint.html
     */
    toModifyDBProxyEndpoint(): this;
    /**
     * Grants permission to modify target group for a database proxy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBProxyTargetGroup.html
     */
    toModifyDBProxyTargetGroup(): this;
    /**
     * Grants permission to update a manual DB snapshot, which can be encrypted or not encrypted, with a new engine version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSnapshot.html
     */
    toModifyDBSnapshot(): this;
    /**
     * Grants permission to add an attribute and values to, or removes an attribute and values from, a manual DB snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSnapshotAttribute.html
     */
    toModifyDBSnapshotAttribute(): this;
    /**
     * Grants permission to modify an existing DB subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSubnetGroup.html
     */
    toModifyDBSubnetGroup(): this;
    /**
     * Grants permission to modify an existing RDS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyEventSubscription.html
     */
    toModifyEventSubscription(): this;
    /**
     * Grants permission to modify a setting for an Amazon Aurora global cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyGlobalCluster.html
     */
    toModifyGlobalCluster(): this;
    /**
     * Grants permission to modify an existing option group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyOptionGroup.html
     */
    toModifyOptionGroup(): this;
    /**
     * Grants permission to modify recommendation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/USER_Recommendations.html
     */
    toModifyRecommendation(): this;
    /**
     * Grants permission to promote a Read Replica DB instance to a standalone DB instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_PromoteReadReplica.html
     */
    toPromoteReadReplica(): this;
    /**
     * Grants permission to promote a Read Replica DB cluster to a standalone DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_PromoteReadReplicaDBCluster.html
     */
    toPromoteReadReplicaDBCluster(): this;
    /**
     * Grants permission to purchase a reserved DB instance offering
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_PurchaseReservedDBInstancesOffering.html
     */
    toPurchaseReservedDBInstancesOffering(): this;
    /**
     * Grants permission to reboot a previously provisioned DB cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - rds:RebootDBInstance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RebootDBCluster.html
     */
    toRebootDBCluster(): this;
    /**
     * Grants permission to restart the database engine service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RebootDBInstance.html
     */
    toRebootDBInstance(): this;
    /**
     * Grants permission to add targets to a database proxy target group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RegisterDBProxyTargets.html
     */
    toRegisterDBProxyTargets(): this;
    /**
     * Grants permission to detach an Aurora secondary cluster from an Aurora global database cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveFromGlobalCluster.html
     */
    toRemoveFromGlobalCluster(): this;
    /**
     * Grants permission to disassociate an AWS Identity and Access Management (IAM) role from an Amazon Aurora DB cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveRoleFromDBCluster.html
     */
    toRemoveRoleFromDBCluster(): this;
    /**
     * Grants permission to disassociate an AWS Identity and Access Management (IAM) role from a DB instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveRoleFromDBInstance.html
     */
    toRemoveRoleFromDBInstance(): this;
    /**
     * Grants permission to remove a source identifier from an existing RDS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveSourceIdentifierFromSubscription.html
     */
    toRemoveSourceIdentifierFromSubscription(): this;
    /**
     * Grants permission to remove metadata tags from an Amazon RDS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource(): this;
    /**
     * Grants permission to modify the parameters of a DB cluster parameter group to the default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ResetDBClusterParameterGroup.html
     */
    toResetDBClusterParameterGroup(): this;
    /**
     * Grants permission to modify the parameters of a DB parameter group to the engine/system default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ResetDBParameterGroup.html
     */
    toResetDBParameterGroup(): this;
    /**
     * Grants permission to create an Amazon Aurora DB cluster from data stored in an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     * - .ifDatabaseEngine()
     * - .ifDatabaseName()
     * - .ifStorageEncrypted()
     * - .ifManageMasterUserPassword()
     *
     * Dependent actions:
     * - iam:PassRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - rds:AddTagsToResource
     * - secretsmanager:CreateSecret
     * - secretsmanager:TagResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBClusterFromS3.html
     */
    toRestoreDBClusterFromS3(): this;
    /**
     * Grants permission to create a new DB cluster from a DB cluster snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     * - .ifDatabaseClass()
     * - .ifStorageSize()
     * - .ifPiops()
     *
     * Dependent actions:
     * - iam:PassRole
     * - rds:AddTagsToResource
     * - rds:CreateDBInstance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBClusterFromSnapshot.html
     */
    toRestoreDBClusterFromSnapshot(): this;
    /**
     * Grants permission to restore a DB cluster to an arbitrary point in time
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     * - .ifDatabaseClass()
     * - .ifStorageSize()
     * - .ifPiops()
     *
     * Dependent actions:
     * - iam:PassRole
     * - rds:AddTagsToResource
     * - rds:CreateDBInstance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBClusterToPointInTime.html
     */
    toRestoreDBClusterToPointInTime(): this;
    /**
     * Grants permission to create a new DB instance from a DB snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBackupTarget()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceFromDBSnapshot.html
     */
    toRestoreDBInstanceFromDBSnapshot(): this;
    /**
     * Grants permission to create a new DB instance from an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     * - .ifManageMasterUserPassword()
     *
     * Dependent actions:
     * - iam:PassRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - rds:AddTagsToResource
     * - secretsmanager:CreateSecret
     * - secretsmanager:TagResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceFromS3.html
     */
    toRestoreDBInstanceFromS3(): this;
    /**
     * Grants permission to restore a DB instance to an arbitrary point in time
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBackupTarget()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     * - rds:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceToPointInTime.html
     */
    toRestoreDBInstanceToPointInTime(): this;
    /**
     * Grants permission to revoke ingress from a DBSecurityGroup for previously authorized IP ranges or EC2 or VPC Security Groups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RevokeDBSecurityGroupIngress.html
     */
    toRevokeDBSecurityGroupIngress(): this;
    /**
     * Grants permission to start Activity Stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartActivityStream.html
     */
    toStartActivityStream(): this;
    /**
     * Grants permission to start the DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartDBCluster.html
     */
    toStartDBCluster(): this;
    /**
     * Grants permission to start the DB instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartDBInstance.html
     */
    toStartDBInstance(): this;
    /**
     * Grants permission to start replication of automated backups to a different AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartDBInstanceAutomatedBackupsReplication.html
     */
    toStartDBInstanceAutomatedBackupsReplication(): this;
    /**
     * Grants permission to start a new Export task for a DB snapshot
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartExportTask.html
     */
    toStartExportTask(): this;
    /**
     * Grants permission to stop Activity Stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopActivityStream.html
     */
    toStopActivityStream(): this;
    /**
     * Grants permission to stop the DB cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopDBCluster.html
     */
    toStopDBCluster(): this;
    /**
     * Grants permission to stop the DB instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopDBInstance.html
     */
    toStopDBInstance(): this;
    /**
     * Grants permission to stop automated backup replication for a DB instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopDBInstanceAutomatedBackupsReplication.html
     */
    toStopDBInstanceAutomatedBackupsReplication(): this;
    /**
     * Grants permission to switch a blue-green deployment from source instance or cluster to target
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - rds:ModifyDBCluster
     * - rds:ModifyDBInstance
     * - rds:PromoteReadReplica
     * - rds:PromoteReadReplicaDBCluster
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_SwitchoverBlueGreenDeployment.html
     */
    toSwitchoverBlueGreenDeployment(): this;
    /**
     * Grants permission to switch over a read replica, making it the new primary database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_SwitchoverReadReplica.html
     */
    toSwitchoverReadReplica(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/CHAP_Aurora.html
     *
     * @param dbClusterInstanceName - Identifier for the dbClusterInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifClusterTag()
     */
    onCluster(dbClusterInstanceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type cluster-endpoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Overview.Endpoints.html
     *
     * @param dbClusterEndpoint - Identifier for the dbClusterEndpoint.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onClusterEndpoint(dbClusterEndpoint: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type cluster-pg to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_WorkingWithParamGroups.html
     *
     * @param clusterParameterGroupName - Identifier for the clusterParameterGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifClusterPgTag()
     */
    onClusterPg(clusterParameterGroupName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type cluster-snapshot to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Managing.Backups.html
     *
     * @param clusterSnapshotName - Identifier for the clusterSnapshotName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifClusterSnapshotTag()
     */
    onClusterSnapshot(clusterSnapshotName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type db to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.html
     *
     * @param dbInstanceName - Identifier for the dbInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifDatabaseClass()
     * - .ifDatabaseEngine()
     * - .ifDatabaseName()
     * - .ifMultiAz()
     * - .ifPiops()
     * - .ifStorageEncrypted()
     * - .ifStorageSize()
     * - .ifVpc()
     * - .ifDbTag()
     */
    onDb(dbInstanceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type es to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Events.html
     *
     * @param subscriptionName - Identifier for the subscriptionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifEsTag()
     */
    onEs(subscriptionName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type global-cluster to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html
     *
     * @param globalCluster - Identifier for the globalCluster.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGlobalCluster(globalCluster: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type og to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithOptionGroups.html
     *
     * @param optionGroupName - Identifier for the optionGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifOgTag()
     */
    onOg(optionGroupName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type pg to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html
     *
     * @param parameterGroupName - Identifier for the parameterGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifPgTag()
     */
    onPg(parameterGroupName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type proxy to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html
     *
     * @param dbProxyId - Identifier for the dbProxyId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProxy(dbProxyId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type proxy-endpoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html
     *
     * @param dbProxyEndpointId - Identifier for the dbProxyEndpointId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProxyEndpoint(dbProxyEndpointId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type ri to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithReservedDBInstances.html
     *
     * @param reservedDbInstanceName - Identifier for the reservedDbInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifRiTag()
     */
    onRi(reservedDbInstanceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type secgrp to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.RDSSecurityGroups.html
     *
     * @param securityGroupName - Identifier for the securityGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSecgrpTag()
     */
    onSecgrp(securityGroupName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html
     *
     * @param snapshotName - Identifier for the snapshotName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSnapshotTag()
     */
    onSnapshot(snapshotName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type subgrp to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.Scenarios.html#USER_VPC.Scenario1
     *
     * @param subnetGroupName - Identifier for the subnetGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSubgrpTag()
     */
    onSubgrp(subnetGroupName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type target to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html
     *
     * @param targetId - Identifier for the targetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTarget(targetId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type target-group to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html
     *
     * @param targetGroupId - Identifier for the targetGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTargetGroup(targetGroupId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type cev to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/custom-cev.html
     *
     * @param engine - Identifier for the engine.
     * @param engineVersion - Identifier for the engineVersion.
     * @param customDbEngineVersionId - Identifier for the customDbEngineVersionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCev(engine: string, engineVersion: string, customDbEngineVersionId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type deployment to the statement
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/blue-green-deployments.html
     *
     * @param blueGreenDeploymentIdentifier - Identifier for the blueGreenDeploymentIdentifier.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeployment(blueGreenDeploymentIdentifier: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the type of backup target. One of: REGION, OUTPOSTS
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateDBInstance()
     * - .toCreateDBSnapshot()
     * - .toRestoreDBInstanceFromDBSnapshot()
     * - .toRestoreDBInstanceToPointInTime()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBackupTarget(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the value that specifies whether the CopyDBSnapshot action requires copying the DB option group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCopyDBSnapshot()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifCopyOptionGroup(value?: boolean): this;
    /**
     * Filters access by the type of DB instance class
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     * - .toCreateDBCluster()
     * - .toModifyDBCluster()
     * - .toRestoreDBClusterFromSnapshot()
     * - .toRestoreDBClusterToPointInTime()
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDatabaseClass(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the database engine. For possible values refer to the engine parameter in CreateDBInstance API
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     * - .toCreateDBCluster()
     * - .toRestoreDBClusterFromS3()
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDatabaseEngine(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the user-defined name of the database on the DB instance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     * - .toCreateDBCluster()
     * - .toRestoreDBClusterFromS3()
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDatabaseName(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the type of the endpoint. One of: READER, WRITER, CUSTOM
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateDBClusterEndpoint()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEndpointType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the value that specifies whether RDS manages master user password in AWS Secrets Manager for the DB instance or cluster
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateDBCluster()
     * - .toCreateDBInstance()
     * - .toModifyDBCluster()
     * - .toModifyDBInstance()
     * - .toRestoreDBClusterFromS3()
     * - .toRestoreDBInstanceFromS3()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifManageMasterUserPassword(value?: boolean): this;
    /**
     * Filters access by the value that specifies whether the DB instance runs in multiple Availability Zones. To indicate that the DB instance is using Multi-AZ, specify true
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     *
     * Applies to resource types:
     * - db
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifMultiAz(value?: boolean): this;
    /**
     * Filters access by the value that contains the number of Provisioned IOPS (PIOPS) that the instance supports. To indicate a DB instance that does not have PIOPS enabled, specify 0
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     * - .toCreateDBCluster()
     * - .toModifyDBCluster()
     * - .toRestoreDBClusterFromSnapshot()
     * - .toRestoreDBClusterToPointInTime()
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPiops(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by the value that specifies whether the DB instance storage should be encrypted. To enforce storage encryption, specify true
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     * - .toCreateDBCluster()
     * - .toRestoreDBClusterFromS3()
     *
     * Applies to resource types:
     * - db
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifStorageEncrypted(value?: boolean): this;
    /**
     * Filters access by the storage volume size (in GB)
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     * - .toCreateDBCluster()
     * - .toModifyDBCluster()
     * - .toRestoreDBClusterFromSnapshot()
     * - .toRestoreDBClusterToPointInTime()
     *
     * Applies to resource types:
     * - db
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifStorageSize(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by the value that specifies whether the DB instance runs in an Amazon Virtual Private Cloud (Amazon VPC). To indicate that the DB instance runs in an Amazon VPC, specify true
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     *
     * Applies to resource types:
     * - db
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifVpc(value?: boolean): this;
    /**
     * Filters access by the tag attached to a DB cluster parameter group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     *
     * Applies to resource types:
     * - cluster-pg
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifClusterPgTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag attached to a DB cluster snapshot
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - cluster-snapshot
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifClusterSnapshotTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag attached to a DB cluster
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     *
     * Applies to resource types:
     * - cluster
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifClusterTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag attached to a DB instance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     *
     * Applies to resource types:
     * - db
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDbTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag attached to an event subscription
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - es
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEsTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag attached to a DB option group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - og
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOgTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag attached to a DB parameter group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toCreateBlueGreenDeployment()
     *
     * Applies to resource types:
     * - pg
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPgTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the set of tag keys and values that can be used to tag a resource
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCreateBlueGreenDeployment()
     * - .toCreateDBCluster()
     * - .toCreateDBClusterParameterGroup()
     * - .toCreateDBClusterSnapshot()
     * - .toCreateDBInstance()
     * - .toCreateDBInstanceReadReplica()
     * - .toCreateDBParameterGroup()
     * - .toCreateDBSecurityGroup()
     * - .toCreateDBSnapshot()
     * - .toCreateDBSubnetGroup()
     * - .toCreateEventSubscription()
     * - .toCreateOptionGroup()
     * - .toDeleteBlueGreenDeployment()
     * - .toRemoveTagsFromResource()
     * - .toRestoreDBClusterFromS3()
     * - .toRestoreDBClusterFromSnapshot()
     * - .toRestoreDBClusterToPointInTime()
     * - .toRestoreDBInstanceFromDBSnapshot()
     * - .toRestoreDBInstanceFromS3()
     * - .toRestoreDBInstanceToPointInTime()
     * - .toSwitchoverBlueGreenDeployment()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReqTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag attached to a reserved DB instance
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - ri
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRiTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag attached to a DB security group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - secgrp
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecgrpTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag attached to a DB snapshot
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - snapshot
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSnapshotTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag attached to a DB subnet group
     *
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/security_iam_service-with-iam.html#UsingWithRDS.IAM.Conditions
     *
     * Applies to resource types:
     * - subgrp
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubgrpTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
}
