import { AccessLevelList } from '../shared/access-level';
import { PolicyStatement, Operator } from '../shared';
/**
 * Statement provider for service [s3](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class S3 extends PolicyStatement {
    servicePrefix: string;
    /**
     * Statement provider for service [s3](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid?: string);
    /**
     * Grants permission to abort a multipart upload
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_AbortMultipartUpload.html
     */
    toAbortMultipartUpload(): this;
    /**
     * Grants permission to allow circumvention of governance-mode object retention settings
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzCopySource()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzMetadataDirective()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifXAmzServerSideEncryptionCustomerAlgorithm()
     * - .ifXAmzStorageClass()
     * - .ifXAmzWebsiteRedirectLocation()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     * - .ifObjectLockLegalHold()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lock-managing.html#object-lock-managing-bypass
     */
    toBypassGovernanceRetention(): this;
    /**
     * Grants permission to create a new access point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifLocationconstraint()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html
     */
    toCreateAccessPoint(): this;
    /**
     * Grants permission to create an object lambda enabled accesspoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPointForObjectLambda.html
     */
    toCreateAccessPointForObjectLambda(): this;
    /**
     * Grants permission to create a new bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifLocationconstraint()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzObjectOwnership()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateBucket.html
     */
    toCreateBucket(): this;
    /**
     * Grants permission to create a new Amazon S3 Batch Operations job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifRequestJobPriority()
     * - .ifRequestJobOperation()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html
     */
    toCreateJob(): this;
    /**
     * Grants permission to create a new Multi-Region Access Point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateMultiRegionAccessPoint.html
     */
    toCreateMultiRegionAccessPoint(): this;
    /**
     * Grants permission to delete the access point named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html
     */
    toDeleteAccessPoint(): this;
    /**
     * Grants permission to delete the object lambda enabled access point named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointForObjectLambda.html
     */
    toDeleteAccessPointForObjectLambda(): this;
    /**
     * Grants permission to delete the policy on a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html
     */
    toDeleteAccessPointPolicy(): this;
    /**
     * Grants permission to delete the policy on a specified object lambda enabled access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicyForObjectLambda.html
     */
    toDeleteAccessPointPolicyForObjectLambda(): this;
    /**
     * Grants permission to delete the bucket named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucket.html
     */
    toDeleteBucket(): this;
    /**
     * Grants permission to delete the policy on a specified bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketPolicy.html
     */
    toDeleteBucketPolicy(): this;
    /**
     * Grants permission to remove the website configuration for a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketWebsite.html
     */
    toDeleteBucketWebsite(): this;
    /**
     * Grants permission to remove tags from an existing Amazon S3 Batch Operations job
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteJobTagging.html
     */
    toDeleteJobTagging(): this;
    /**
     * Grants permission to delete the Multi-Region Access Point named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteMultiRegionAccessPoint.html
     */
    toDeleteMultiRegionAccessPoint(): this;
    /**
     * Grants permission to remove the null version of an object and insert a delete marker, which becomes the current version of the object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    toDeleteObject(): this;
    /**
     * Grants permission to use the tagging subresource to remove the entire tag set from the specified object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    toDeleteObjectTagging(): this;
    /**
     * Grants permission to remove a specific version of an object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    toDeleteObjectVersion(): this;
    /**
     * Grants permission to remove the entire tag set for a specific version of the object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    toDeleteObjectVersionTagging(): this;
    /**
     * Grants permission to delete an existing Amazon S3 Storage Lens configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteStorageLensConfiguration.html
     */
    toDeleteStorageLensConfiguration(): this;
    /**
     * Grants permission to remove tags from an existing Amazon S3 Storage Lens configuration
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteStorageLensConfigurationTagging.html
     */
    toDeleteStorageLensConfigurationTagging(): this;
    /**
     * Grants permission to retrieve the configuration parameters and status for a batch operations job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeJob.html
     */
    toDescribeJob(): this;
    /**
     * Grants permission to retrieve the configurations for a Multi-Region Access Point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeMultiRegionAccessPointOperation.html
     */
    toDescribeMultiRegionAccessPointOperation(): this;
    /**
     * Grants permission to uses the accelerate subresource to return the Transfer Acceleration state of a bucket, which is either Enabled or Suspended
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAccelerateConfiguration.html
     */
    toGetAccelerateConfiguration(): this;
    /**
     * Grants permission to return configuration information about the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html
     */
    toGetAccessPoint(): this;
    /**
     * Grants permission to retrieve the configuration of the object lambda enabled access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointConfigurationForObjectLambda.html
     */
    toGetAccessPointConfigurationForObjectLambda(): this;
    /**
     * Grants permission to create an object lambda enabled accesspoint
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointForObjectLambda.html
     */
    toGetAccessPointForObjectLambda(): this;
    /**
     * Grants permission to returns the access point policy associated with the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicy.html
     */
    toGetAccessPointPolicy(): this;
    /**
     * Grants permission to returns the access point policy associated with the specified object lambda enabled access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicyForObjectLambda.html
     */
    toGetAccessPointPolicyForObjectLambda(): this;
    /**
     * Grants permission to return the policy status for a specific access point policy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicyStatus.html
     */
    toGetAccessPointPolicyStatus(): this;
    /**
     * Grants permission to return the policy status for a specific object lambda access point policy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicyStatusForObjectLambda.html
     */
    toGetAccessPointPolicyStatusForObjectLambda(): this;
    /**
     * Grants permission to retrieve the PublicAccessBlock configuration for an AWS account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetPublicAccessBlock.html
     */
    toGetAccountPublicAccessBlock(): this;
    /**
     * Grants permission to get an analytics configuration from an Amazon S3 bucket, identified by the analytics configuration ID
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAnalyticsConfiguration.html
     */
    toGetAnalyticsConfiguration(): this;
    /**
     * Grants permission to use the acl subresource to return the access control list (ACL) of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAcl.html
     */
    toGetBucketAcl(): this;
    /**
     * Grants permission to return the CORS configuration information set for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketCors.html
     */
    toGetBucketCORS(): this;
    /**
     * Grants permission to return the Region that an Amazon S3 bucket resides in
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLocation.html
     */
    toGetBucketLocation(): this;
    /**
     * Grants permission to return the logging status of an Amazon S3 bucket and the permissions users have to view or modify that status
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLogging.html
     */
    toGetBucketLogging(): this;
    /**
     * Grants permission to get the notification configuration of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketNotification.html
     */
    toGetBucketNotification(): this;
    /**
     * Grants permission to get the Object Lock configuration of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifSignatureversion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLockConfiguration.html
     */
    toGetBucketObjectLockConfiguration(): this;
    /**
     * Grants permission to retrieve ownership controls on a bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketOwnershipControls.html
     */
    toGetBucketOwnershipControls(): this;
    /**
     * Grants permission to return the policy of the specified bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicy.html
     */
    toGetBucketPolicy(): this;
    /**
     * Grants permission to retrieve the policy status for a specific Amazon S3 bucket, which indicates whether the bucket is public
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicyStatus.html
     */
    toGetBucketPolicyStatus(): this;
    /**
     * Grants permission to retrieve the PublicAccessBlock configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetPublicAccessBlock.html
     */
    toGetBucketPublicAccessBlock(): this;
    /**
     * Grants permission to return the request payment configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketRequestPayment.html
     */
    toGetBucketRequestPayment(): this;
    /**
     * Grants permission to return the tag set associated with an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketTagging.html
     */
    toGetBucketTagging(): this;
    /**
     * Grants permission to return the versioning state of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketVersioning.html
     */
    toGetBucketVersioning(): this;
    /**
     * Grants permission to return the website configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketWebsite.html
     */
    toGetBucketWebsite(): this;
    /**
     * Grants permission to return the default encryption configuration an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketEncryption.html
     */
    toGetEncryptionConfiguration(): this;
    /**
     * Grants permission to get an or list all Amazon S3 Intelligent Tiering configuration in a S3 Bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketIntelligentTieringConfiguration.html
     */
    toGetIntelligentTieringConfiguration(): this;
    /**
     * Grants permission to return an inventory configuration from an Amazon S3 bucket, identified by the inventory configuration ID
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketInventoryConfiguration.html
     */
    toGetInventoryConfiguration(): this;
    /**
     * Grants permission to return the tag set of an existing Amazon S3 Batch Operations job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetJobTagging.html
     */
    toGetJobTagging(): this;
    /**
     * Grants permission to return the lifecycle configuration information set on an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLifecycleConfiguration.html
     */
    toGetLifecycleConfiguration(): this;
    /**
     * Grants permission to get a metrics configuration from an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketMetricsConfiguration.html
     */
    toGetMetricsConfiguration(): this;
    /**
     * Grants permission to return configuration information about the specified Multi-Region Access Point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetMultiRegionAccessPoint.html
     */
    toGetMultiRegionAccessPoint(): this;
    /**
     * Grants permission to returns the access point policy associated with the specified Multi-Region Access Point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetMultiRegionAccessPointPolicy.html
     */
    toGetMultiRegionAccessPointPolicy(): this;
    /**
     * Grants permission to return the policy status for a specific Multi-Region Access Point policy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetMultiRegionAccessPointPolicyStatus.html
     */
    toGetMultiRegionAccessPointPolicyStatus(): this;
    /**
     * Grants permission to return the route configuration for a Multi-Region Access Point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetMultiRegionAccessPointRoutes.html
     */
    toGetMultiRegionAccessPointRoutes(): this;
    /**
     * Grants permission to retrieve objects from Amazon S3
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    toGetObject(): this;
    /**
     * Grants permission to return the access control list (ACL) of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html
     */
    toGetObjectAcl(): this;
    /**
     * Grants permission to retrieve attributes related to a specific object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAttributes.html
     */
    toGetObjectAttributes(): this;
    /**
     * Grants permission to get an object's current Legal Hold status
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLegalHold.html
     */
    toGetObjectLegalHold(): this;
    /**
     * Grants permission to retrieve the retention settings for an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectRetention.html
     */
    toGetObjectRetention(): this;
    /**
     * Grants permission to return the tag set of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTagging.html
     */
    toGetObjectTagging(): this;
    /**
     * Grants permission to return torrent files from an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html
     */
    toGetObjectTorrent(): this;
    /**
     * Grants permission to retrieve a specific version of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    toGetObjectVersion(): this;
    /**
     * Grants permission to return the access control list (ACL) of a specific object version
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html
     */
    toGetObjectVersionAcl(): this;
    /**
     * Grants permission to retrieve attributes related to a specific version of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAttributes.html
     */
    toGetObjectVersionAttributes(): this;
    /**
     * Grants permission to replicate both unencrypted objects and objects encrypted with SSE-S3 or SSE-KMS
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/replication-config-for-kms-objects.html
     */
    toGetObjectVersionForReplication(): this;
    /**
     * Grants permission to return the tag set for a specific version of the object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/setting-repl-config-perm-overview.html
     */
    toGetObjectVersionTagging(): this;
    /**
     * Grants permission to get Torrent files about a different version using the versionId subresource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html
     */
    toGetObjectVersionTorrent(): this;
    /**
     * Grants permission to get the replication configuration information set on an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketReplication.html
     */
    toGetReplicationConfiguration(): this;
    /**
     * Grants permission to get an Amazon S3 Storage Lens configuration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetStorageLensConfiguration.html
     */
    toGetStorageLensConfiguration(): this;
    /**
     * Grants permission to get the tag set of an existing Amazon S3 Storage Lens configuration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetStorageLensConfigurationTagging.html
     */
    toGetStorageLensConfigurationTagging(): this;
    /**
     * Grants permission to get an Amazon S3 Storage Lens dashboard
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/storage_lens_dashboard.html
     */
    toGetStorageLensDashboard(): this;
    /**
     * Grants permission to initiate the replication process by setting replication status of an object to pending
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceAccount()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/setting-repl-config-perm-overview.html
     */
    toInitiateReplication(): this;
    /**
     * Grants permission to list access points
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPoints.html
     */
    toListAccessPoints(): this;
    /**
     * Grants permission to list object lambda enabled accesspoints
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPointsForObjectLambda.html
     */
    toListAccessPointsForObjectLambda(): this;
    /**
     * Grants permission to list all buckets owned by the authenticated sender of the request
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListBuckets.html
     */
    toListAllMyBuckets(): this;
    /**
     * Grants permission to list some or all of the objects in an Amazon S3 bucket (up to 1000)
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifDelimiter()
     * - .ifMaxKeys()
     * - .ifPrefix()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectsV2.html
     */
    toListBucket(): this;
    /**
     * Grants permission to list in-progress multipart uploads
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListMultipartUploads.html
     */
    toListBucketMultipartUploads(): this;
    /**
     * Grants permission to list metadata about all the versions of objects in an Amazon S3 bucket
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifDelimiter()
     * - .ifMaxKeys()
     * - .ifPrefix()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectVersions.html
     */
    toListBucketVersions(): this;
    /**
     * Grants permission to list current jobs and jobs that have ended recently
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListJobs.html
     */
    toListJobs(): this;
    /**
     * Grants permission to list Multi-Region Access Points
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListMultiRegionAccessPoints.html
     */
    toListMultiRegionAccessPoints(): this;
    /**
     * Grants permission to list the parts that have been uploaded for a specific multipart upload
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListParts.html
     */
    toListMultipartUploadParts(): this;
    /**
     * Grants permission to list Amazon S3 Storage Lens configurations
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListStorageLensConfigurations.html
     */
    toListStorageLensConfigurations(): this;
    /**
     * Grants permission to change replica ownership
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/replication-change-owner.html#repl-ownership-add-role-permission
     */
    toObjectOwnerOverrideToBucketOwner(): this;
    /**
     * Grants permission to use the accelerate subresource to set the Transfer Acceleration state of an existing S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAccelerateConfiguration.html
     */
    toPutAccelerateConfiguration(): this;
    /**
     * Grants permission to set the configuration of the object lambda enabled access point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointConfigurationForObjectLambda.html
     */
    toPutAccessPointConfigurationForObjectLambda(): this;
    /**
     * Grants permission to associate an access policy with a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html
     */
    toPutAccessPointPolicy(): this;
    /**
     * Grants permission to associate an access policy with a specified object lambda enabled access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicyForObjectLambda.html
     */
    toPutAccessPointPolicyForObjectLambda(): this;
    /**
     * Grants permission to associate public access block configurations with a specified access point, while creating a access point
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-control-block-public-access.html#access-control-block-public-access-examples-access-point
     */
    toPutAccessPointPublicAccessBlock(): this;
    /**
     * Grants permission to create or modify the PublicAccessBlock configuration for an AWS account
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutPublicAccessBlock.html
     */
    toPutAccountPublicAccessBlock(): this;
    /**
     * Grants permission to set an analytics configuration for the bucket, specified by the analytics configuration ID
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAnalyticsConfiguration.html
     */
    toPutAnalyticsConfiguration(): this;
    /**
     * Grants permission to set the permissions on an existing bucket using access control lists (ACLs)
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAcl.html
     */
    toPutBucketAcl(): this;
    /**
     * Grants permission to set the CORS configuration for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketCors.html
     */
    toPutBucketCORS(): this;
    /**
     * Grants permission to set the logging parameters for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLogging.html
     */
    toPutBucketLogging(): this;
    /**
     * Grants permission to receive notifications when certain events happen in an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketNotification.html
     */
    toPutBucketNotification(): this;
    /**
     * Grants permission to put Object Lock configuration on a specific bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     * - .ifSignatureversion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLockConfiguration.html
     */
    toPutBucketObjectLockConfiguration(): this;
    /**
     * Grants permission to add, replace or delete ownership controls on a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketOwnershipControls.html
     */
    toPutBucketOwnershipControls(): this;
    /**
     * Grants permission to add or replace a bucket policy on a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketPolicy.html
     */
    toPutBucketPolicy(): this;
    /**
     * Grants permission to create or modify the PublicAccessBlock configuration for a specific Amazon S3 bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutPublicAccessBlock.html
     */
    toPutBucketPublicAccessBlock(): this;
    /**
     * Grants permission to set the request payment configuration of a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketRequestPayment.html
     */
    toPutBucketRequestPayment(): this;
    /**
     * Grants permission to add a set of tags to an existing Amazon S3 bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketTagging.html
     */
    toPutBucketTagging(): this;
    /**
     * Grants permission to set the versioning state of an existing Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketVersioning.html
     */
    toPutBucketVersioning(): this;
    /**
     * Grants permission to set the configuration of the website that is specified in the website subresource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketWebsite.html
     */
    toPutBucketWebsite(): this;
    /**
     * Grants permission to set the encryption configuration for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketEncryption.html
     */
    toPutEncryptionConfiguration(): this;
    /**
     * Grants permission to create new or update or delete an existing Amazon S3 Intelligent Tiering configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketIntelligentTieringConfiguration.html
     */
    toPutIntelligentTieringConfiguration(): this;
    /**
     * Grants permission to add an inventory configuration to the bucket, identified by the inventory ID
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketInventoryConfiguration.html
     */
    toPutInventoryConfiguration(): this;
    /**
     * Grants permission to replace tags on an existing Amazon S3 Batch Operations job
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutJobTagging.html
     */
    toPutJobTagging(): this;
    /**
     * Grants permission to create a new lifecycle configuration for the bucket or replace an existing lifecycle configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLifecycleConfiguration.html
     */
    toPutLifecycleConfiguration(): this;
    /**
     * Grants permission to set or update a metrics configuration for the CloudWatch request metrics from an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketMetricsConfiguration.html
     */
    toPutMetricsConfiguration(): this;
    /**
     * Grants permission to associate an access policy with a specified Multi-Region Access Point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutMultiRegionAccessPointPolicy.html
     */
    toPutMultiRegionAccessPointPolicy(): this;
    /**
     * Grants permission to add an object to a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzCopySource()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzMetadataDirective()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifXAmzServerSideEncryptionCustomerAlgorithm()
     * - .ifXAmzStorageClass()
     * - .ifXAmzWebsiteRedirectLocation()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     * - .ifObjectLockLegalHold()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObject.html
     */
    toPutObject(): this;
    /**
     * Grants permission to set the access control list (ACL) permissions for new or existing objects in an S3 bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzStorageClass()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    toPutObjectAcl(): this;
    /**
     * Grants permission to apply a Legal Hold configuration to the specified object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifObjectLockLegalHold()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLegalHold.html
     */
    toPutObjectLegalHold(): this;
    /**
     * Grants permission to place an Object Retention configuration on an object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectRetention.html
     */
    toPutObjectRetention(): this;
    /**
     * Grants permission to set the supplied tag-set to an object that already exists in a bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    toPutObjectTagging(): this;
    /**
     * Grants permission to use the acl subresource to set the access control list (ACL) permissions for an object that already exists in a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzStorageClass()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    toPutObjectVersionAcl(): this;
    /**
     * Grants permission to set the supplied tag-set for a specific version of an object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    toPutObjectVersionTagging(): this;
    /**
     * Grants permission to create a new replication configuration or replace an existing one
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketReplication.html
     */
    toPutReplicationConfiguration(): this;
    /**
     * Grants permission to create or update an Amazon S3 Storage Lens configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutStorageLensConfiguration.html
     */
    toPutStorageLensConfiguration(): this;
    /**
     * Grants permission to put or replace tags on an existing Amazon S3 Storage Lens configuration
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutStorageLensConfigurationTagging.html
     */
    toPutStorageLensConfigurationTagging(): this;
    /**
     * Grants permission to replicate delete markers to the destination bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/setting-repl-config-perm-overview.html
     */
    toReplicateDelete(): this;
    /**
     * Grants permission to replicate objects and object tags to the destination bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifXAmzServerSideEncryptionCustomerAlgorithm()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/setting-repl-config-perm-overview.html
     */
    toReplicateObject(): this;
    /**
     * Grants permission to replicate object tags to the destination bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/setting-repl-config-perm-overview.html
     */
    toReplicateTags(): this;
    /**
     * Grants permission to restore an archived copy of an object back into Amazon S3
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_RestoreObject.html
     */
    toRestoreObject(): this;
    /**
     * Grants permission to submit a route configuration update for a Multi-Region Access Point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_SubmitMultiRegionAccessPointRoutes.html
     */
    toSubmitMultiRegionAccessPointRoutes(): this;
    /**
     * Grants permission to update the priority of an existing job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifRequestJobPriority()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobPriority.html
     */
    toUpdateJobPriority(): this;
    /**
     * Grants permission to update the status for the specified job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     * - .ifJobSuspendedCause()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobStatus.html
     */
    toUpdateJobStatus(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type accesspoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points.html
     *
     * @param accessPointName - Identifier for the accessPointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAccesspoint(accessPointName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type bucket to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/UsingBucket.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBucket(bucketName: string, partition?: string): this;
    /**
     * Adds a resource of type object to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/UsingObjects.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param objectName - Identifier for the objectName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onObject(bucketName: string, objectName: string, partition?: string): this;
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/batch-ops-managing-jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onJob(jobId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type storagelensconfiguration to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/storage_lens.html
     *
     * @param configId - Identifier for the configId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStoragelensconfiguration(configId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type objectlambdaaccesspoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/transforming-objects.html
     *
     * @param accessPointName - Identifier for the accessPointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onObjectlambdaaccesspoint(accessPointName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type multiregionaccesspoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/MultiRegionAccessPointRequests.html
     *
     * @param accessPointAlias - Identifier for the accessPointAlias.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMultiregionaccesspoint(accessPointAlias: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type multiregionaccesspointrequestarn to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/MultiRegionAccessPointRequests.html
     *
     * @param operation - Identifier for the operation.
     * @param token - Identifier for the token.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMultiregionaccesspointrequestarn(operation: string, token: string, account?: string, partition?: string): this;
    /**
     * Filters access by the network origin (Internet or VPC)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateMultiRegionAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteMultiRegionAccessPoint()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetMultiRegionAccessPoint()
     * - .toGetMultiRegionAccessPointPolicy()
     * - .toGetMultiRegionAccessPointPolicyStatus()
     * - .toGetMultiRegionAccessPointRoutes()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListMultipartUploadParts()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutMultiRegionAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toRestoreObject()
     * - .toSubmitMultiRegionAccessPointRoutes()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessPointNetworkOrigin(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the AWS Account ID that owns the access point
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateMultiRegionAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteMultiRegionAccessPoint()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetMultiRegionAccessPoint()
     * - .toGetMultiRegionAccessPointPolicy()
     * - .toGetMultiRegionAccessPointPolicyStatus()
     * - .toGetMultiRegionAccessPointRoutes()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListMultipartUploadParts()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutMultiRegionAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toRestoreObject()
     * - .toSubmitMultiRegionAccessPointRoutes()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointAccount(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by an access point Amazon Resource Name (ARN)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateMultiRegionAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteMultiRegionAccessPoint()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetMultiRegionAccessPoint()
     * - .toGetMultiRegionAccessPointPolicy()
     * - .toGetMultiRegionAccessPointPolicyStatus()
     * - .toGetMultiRegionAccessPointRoutes()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListMultipartUploadParts()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutMultiRegionAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toRestoreObject()
     * - .toSubmitMultiRegionAccessPointRoutes()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointArn(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by operation to updating the job priority
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toDeleteJobTagging()
     * - .toPutJobTagging()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExistingJobOperation(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by priority range to cancelling existing jobs
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toDeleteJobTagging()
     * - .toPutJobTagging()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifExistingJobPriority(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by existing object tag key and value
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersionTagging()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionTagging()
     * - .toPutObjectAcl()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExistingObjectTag(key: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by a specific job suspended cause (for example, AWAITING_CONFIRMATION) to cancelling suspended jobs
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifJobSuspendedCause(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by operation to creating jobs
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toCreateJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestJobOperation(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by priority range to creating new jobs
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toCreateJob()
     * - .toUpdateJobPriority()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRequestJobPriority(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by the tag keys and values to be added to objects
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionTagging()
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestObjectTag(key: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag keys to be added to objects
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestObjectTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the resource owner AWS account ID
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-policy-keys.html#example-object-resource-account
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateBucket()
     * - .toCreateJob()
     * - .toCreateMultiRegionAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteMultiRegionAccessPoint()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDeleteStorageLensConfiguration()
     * - .toDeleteStorageLensConfigurationTagging()
     * - .toDescribeJob()
     * - .toDescribeMultiRegionAccessPointOperation()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLocation()
     * - .toGetBucketLogging()
     * - .toGetBucketNotification()
     * - .toGetBucketObjectLockConfiguration()
     * - .toGetBucketOwnershipControls()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetEncryptionConfiguration()
     * - .toGetIntelligentTieringConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetMultiRegionAccessPoint()
     * - .toGetMultiRegionAccessPointPolicy()
     * - .toGetMultiRegionAccessPointPolicyStatus()
     * - .toGetMultiRegionAccessPointRoutes()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toGetStorageLensConfiguration()
     * - .toGetStorageLensConfigurationTagging()
     * - .toGetStorageLensDashboard()
     * - .toInitiateReplication()
     * - .toListAccessPoints()
     * - .toListAccessPointsForObjectLambda()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListJobs()
     * - .toListMultiRegionAccessPoints()
     * - .toListMultipartUploadParts()
     * - .toListStorageLensConfigurations()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketObjectLockConfiguration()
     * - .toPutBucketOwnershipControls()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutIntelligentTieringConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutMultiRegionAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toPutStorageLensConfiguration()
     * - .toPutStorageLensConfigurationTagging()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toSubmitMultiRegionAccessPointRoutes()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceAccount(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the TLS version used by the client
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateBucket()
     * - .toCreateJob()
     * - .toCreateMultiRegionAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteMultiRegionAccessPoint()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDeleteStorageLensConfiguration()
     * - .toDeleteStorageLensConfigurationTagging()
     * - .toDescribeJob()
     * - .toDescribeMultiRegionAccessPointOperation()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLocation()
     * - .toGetBucketLogging()
     * - .toGetBucketNotification()
     * - .toGetBucketObjectLockConfiguration()
     * - .toGetBucketOwnershipControls()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetEncryptionConfiguration()
     * - .toGetIntelligentTieringConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetMultiRegionAccessPoint()
     * - .toGetMultiRegionAccessPointPolicy()
     * - .toGetMultiRegionAccessPointPolicyStatus()
     * - .toGetMultiRegionAccessPointRoutes()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toGetStorageLensConfiguration()
     * - .toGetStorageLensConfigurationTagging()
     * - .toGetStorageLensDashboard()
     * - .toListAccessPoints()
     * - .toListAccessPointsForObjectLambda()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListJobs()
     * - .toListMultiRegionAccessPoints()
     * - .toListMultipartUploadParts()
     * - .toListStorageLensConfigurations()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketObjectLockConfiguration()
     * - .toPutBucketOwnershipControls()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutIntelligentTieringConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutMultiRegionAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toPutStorageLensConfiguration()
     * - .toPutStorageLensConfigurationTagging()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toSubmitMultiRegionAccessPointRoutes()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifTlsVersion(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by authentication method
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateBucket()
     * - .toCreateJob()
     * - .toCreateMultiRegionAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteMultiRegionAccessPoint()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDeleteStorageLensConfiguration()
     * - .toDeleteStorageLensConfigurationTagging()
     * - .toDescribeJob()
     * - .toDescribeMultiRegionAccessPointOperation()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLocation()
     * - .toGetBucketLogging()
     * - .toGetBucketNotification()
     * - .toGetBucketObjectLockConfiguration()
     * - .toGetBucketOwnershipControls()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetEncryptionConfiguration()
     * - .toGetIntelligentTieringConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetMultiRegionAccessPoint()
     * - .toGetMultiRegionAccessPointPolicy()
     * - .toGetMultiRegionAccessPointPolicyStatus()
     * - .toGetMultiRegionAccessPointRoutes()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toGetStorageLensConfiguration()
     * - .toGetStorageLensConfigurationTagging()
     * - .toGetStorageLensDashboard()
     * - .toListAccessPoints()
     * - .toListAccessPointsForObjectLambda()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListJobs()
     * - .toListMultiRegionAccessPoints()
     * - .toListMultipartUploadParts()
     * - .toListStorageLensConfigurations()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketObjectLockConfiguration()
     * - .toPutBucketOwnershipControls()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutIntelligentTieringConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutMultiRegionAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toPutStorageLensConfiguration()
     * - .toPutStorageLensConfigurationTagging()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toSubmitMultiRegionAccessPointRoutes()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by delimiter parameter
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/walkthrough1.html
     *
     * Applies to actions:
     * - .toListBucket()
     * - .toListBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDelimiter(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by a specific Region
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-policy-keys.html#condition-key-bucket-ops-1
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLocationconstraint(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by maximum number of keys returned in a ListBucket request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-policy-keys.html#example-numeric-condition-operators
     *
     * Applies to actions:
     * - .toListBucket()
     * - .toListBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaxKeys(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by object legal hold status
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lock-overview.html#object-lock-legal-holds
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectLegalHold()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockLegalHold(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by object retention mode (COMPLIANCE or GOVERNANCE)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lock-overview.html#object-lock-retention-modes
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockMode(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by remaining object retention days
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lock-managing.html#object-lock-managing-retention-limits
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifObjectLockRemainingRetentionDays(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by object retain-until date
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lock-overview.html#object-lock-retention-periods
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateEquals`
     */
    ifObjectLockRetainUntilDate(value: Date | string | (Date | string)[], operator?: Operator | string): this;
    /**
     * Filters access by key name prefix
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-policy-keys.html#condition-key-bucket-ops-2
     *
     * Applies to actions:
     * - .toListBucket()
     * - .toListBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrefix(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the age in milliseconds of the request signature
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateBucket()
     * - .toCreateJob()
     * - .toCreateMultiRegionAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteMultiRegionAccessPoint()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDeleteStorageLensConfiguration()
     * - .toDeleteStorageLensConfigurationTagging()
     * - .toDescribeJob()
     * - .toDescribeMultiRegionAccessPointOperation()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLocation()
     * - .toGetBucketLogging()
     * - .toGetBucketNotification()
     * - .toGetBucketObjectLockConfiguration()
     * - .toGetBucketOwnershipControls()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetEncryptionConfiguration()
     * - .toGetIntelligentTieringConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetMultiRegionAccessPoint()
     * - .toGetMultiRegionAccessPointPolicy()
     * - .toGetMultiRegionAccessPointPolicyStatus()
     * - .toGetMultiRegionAccessPointRoutes()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toGetStorageLensConfiguration()
     * - .toGetStorageLensConfigurationTagging()
     * - .toGetStorageLensDashboard()
     * - .toListAccessPoints()
     * - .toListAccessPointsForObjectLambda()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListJobs()
     * - .toListMultiRegionAccessPoints()
     * - .toListMultipartUploadParts()
     * - .toListStorageLensConfigurations()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketObjectLockConfiguration()
     * - .toPutBucketOwnershipControls()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutIntelligentTieringConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutMultiRegionAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toPutStorageLensConfiguration()
     * - .toPutStorageLensConfigurationTagging()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toSubmitMultiRegionAccessPointRoutes()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSignatureAge(value: number | number[], operator?: Operator | string): this;
    /**
     * Filters access by the version of AWS Signature used on the request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateBucket()
     * - .toCreateJob()
     * - .toCreateMultiRegionAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteMultiRegionAccessPoint()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDeleteStorageLensConfiguration()
     * - .toDeleteStorageLensConfigurationTagging()
     * - .toDescribeJob()
     * - .toDescribeMultiRegionAccessPointOperation()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLocation()
     * - .toGetBucketLogging()
     * - .toGetBucketNotification()
     * - .toGetBucketObjectLockConfiguration()
     * - .toGetBucketOwnershipControls()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetEncryptionConfiguration()
     * - .toGetIntelligentTieringConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetMultiRegionAccessPoint()
     * - .toGetMultiRegionAccessPointPolicy()
     * - .toGetMultiRegionAccessPointPolicyStatus()
     * - .toGetMultiRegionAccessPointRoutes()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toGetStorageLensConfiguration()
     * - .toGetStorageLensConfigurationTagging()
     * - .toGetStorageLensDashboard()
     * - .toListAccessPoints()
     * - .toListAccessPointsForObjectLambda()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListJobs()
     * - .toListMultiRegionAccessPoints()
     * - .toListMultipartUploadParts()
     * - .toListStorageLensConfigurations()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketObjectLockConfiguration()
     * - .toPutBucketOwnershipControls()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutIntelligentTieringConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutMultiRegionAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toPutStorageLensConfiguration()
     * - .toPutStorageLensConfigurationTagging()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toSubmitMultiRegionAccessPointRoutes()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSignatureversion(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by a specific object version
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-policy-keys.html#getobjectversion-limit-access-to-specific-version-3
     *
     * Applies to actions:
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionid(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by canned ACL in the request's x-amz-acl header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzAcl(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by unsigned content in your bucket
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateBucket()
     * - .toCreateJob()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDeleteStorageLensConfiguration()
     * - .toDeleteStorageLensConfigurationTagging()
     * - .toDescribeJob()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLocation()
     * - .toGetBucketLogging()
     * - .toGetBucketNotification()
     * - .toGetBucketOwnershipControls()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetEncryptionConfiguration()
     * - .toGetIntelligentTieringConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toGetStorageLensConfiguration()
     * - .toGetStorageLensConfigurationTagging()
     * - .toGetStorageLensDashboard()
     * - .toListAccessPoints()
     * - .toListAccessPointsForObjectLambda()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListJobs()
     * - .toListMultipartUploadParts()
     * - .toListStorageLensConfigurations()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketOwnershipControls()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutIntelligentTieringConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toPutStorageLensConfiguration()
     * - .toPutStorageLensConfigurationTagging()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzContentSha256(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by copy source bucket, prefix, or object in the copy object requests
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-policy-keys.html#putobject-limit-copy-source-3
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzCopySource(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by x-amz-grant-full-control (full control) header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantFullControl(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by x-amz-grant-read (read access) header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantRead(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the x-amz-grant-read-acp (read permissions for the ACL) header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantReadAcp(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the x-amz-grant-write (write access) header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantWrite(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the x-amz-grant-write-acp (write permissions for the ACL) header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantWriteAcp(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by object metadata behavior (COPY or REPLACE) when objects are copied
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CopyObject.html
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzMetadataDirective(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by Object Ownership
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/ensure-object-ownership.html#object-ownership-requiring-bucket-owner-enforced
     *
     * Applies to actions:
     * - .toCreateBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzObjectOwnership(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/UsingServerSideEncryption.html
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toReplicateObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryption(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by AWS KMS customer managed CMK for server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/UsingKMSEncryption.html#require-sse-kms
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toReplicateObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryptionAwsKmsKeyId(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by customer specified algorithm for server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/ServerSideEncryptionCustomerKeys.html
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toReplicateObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryptionCustomerAlgorithm(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by storage class
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/storage-class-intro.html#sc-howtoset
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzStorageClass(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by a specific website redirect location for buckets that are configured as static websites
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/how-to-page-redirect.html#page-redirect-using-rest-api
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzWebsiteRedirectLocation(value: string | string[], operator?: Operator | string): this;
}
