# Story 3.5: Implement CCXT Data Adapter (Priority: MVP - Crypto)

## Status
Completed


## Story

**As a** quantitative trader,
**I want** CCXT adapter for 100+ crypto exchanges,
**so that** I can backtest crypto strategies with data from Binance, Coinbase, Kraken, etc.

## Acceptance Criteria

1. CCXT library integrated (v4.x+) with dependency added to requirements
2. CCXTAdapter implements BaseDataAdapter interface
3. Exchange selection supported (Binance, Coinbase, Kraken, etc. via CCXT unified API)
4. OHLCV data fetched via CCXT `fetch_ohlcv()` method
5. Multiple resolutions supported (1m, 5m, 15m, 1h, 4h, 1d)
6. Rate limiting configured per exchange (respect CCXT rate limit metadata)
7. Data standardization converts CCXT format to unified schema with Decimal precision
8. Error handling covers exchange-specific issues (maintenance, delisted pairs)
9. Integration tests fetch live data from 3+ exchanges and validate schema
10. Example notebook demonstrates crypto backtest using CCXT data

## Tasks / Subtasks

- [x] Add CCXT dependency to project (AC: 1)
  - [x] Add ccxt >= 4.0.0 to pyproject.toml dependencies
  - [x] Update requirements.txt with ccxt and version constraint
  - [x] Verify installation with `uv pip install ccxt`
  - [x] Document CCXT version requirements in docs

- [x] Implement CCXTAdapter class (AC: 2, 3)
  - [x] Create rustybt/data/adapters/ccxt_adapter.py
  - [x] Implement CCXTAdapter class extending BaseDataAdapter
  - [x] Add exchange_id parameter for exchange selection (e.g., 'binance', 'coinbase')
  - [x] Initialize CCXT exchange instance in __init__()
  - [x] Add testnet/sandbox mode support for development
  - [x] Register CCXTAdapter with AdapterRegistry

- [x] Implement OHLCV data fetching (AC: 4, 5)
  - [x] Implement fetch() method using ccxt.fetch_ohlcv()
  - [x] Add resolution mapping (rustybt format → CCXT timeframe: "1d" → "1d", "1h" → "1h")
  - [x] Handle pagination for large date ranges (CCXT limit typically 500-1000 bars)
  - [x] Implement batch fetching for multiple symbols
  - [x] Add since parameter conversion (pd.Timestamp → Unix milliseconds)
  - [x] Handle exchange-specific resolution support (not all exchanges support all timeframes)

- [x] Configure rate limiting per exchange (AC: 6)
  - [x] Extract rate limit metadata from CCXT exchange.rateLimit property
  - [x] Initialize RateLimiter with exchange-specific limits
  - [x] Enable CCXT built-in rate limiting: exchange.enableRateLimit = True
  - [x] Add additional throttling for conservative usage
  - [x] Implement request tracking and statistics

- [x] Implement data standardization (AC: 7)
  - [x] Convert CCXT OHLCV format [[timestamp, open, high, low, close, volume]] to Polars DataFrame
  - [x] Convert Unix millisecond timestamps to pd.Timestamp (microsecond precision)
  - [x] Convert float prices to Decimal(18, 8) for all OHLCV columns
  - [x] Add symbol column with normalized format (e.g., "BTC/USDT")
  - [x] Implement standardize() method to apply unified schema
  - [x] Validate Decimal precision preservation in roundtrip

- [x] Implement exchange-specific error handling (AC: 8)
  - [x] Handle ccxt.NetworkError (connection failures, timeouts)
  - [x] Handle ccxt.ExchangeNotAvailable (maintenance, downtime)
  - [x] Handle ccxt.BadSymbol (delisted pairs, invalid symbols)
  - [x] Handle ccxt.RateLimitExceeded (despite rate limiting)
  - [x] Add exchange status checking before fetch
  - [x] Implement graceful degradation for temporary issues

- [x] Add symbol normalization and validation (AC: 3, 8)
  - [x] Normalize symbol format: "BTC-USD" → "BTC/USD", "BTCUSDT" → "BTC/USDT"
  - [x] Validate symbol exists on exchange using exchange.markets
  - [x] Handle exchange-specific symbol formats (e.g., Binance uses "BTC/USDT", some use "BTCUSD")
  - [x] Implement symbol mapping for consistency across exchanges
  - [x] Add symbol availability checking (trading_enabled, active status)

- [x] Write comprehensive tests (AC: 9)
  - [x] Unit test: CCXT data format converts to unified schema correctly
  - [x] Unit test: Decimal conversion preserves precision
  - [x] Unit test: Resolution mapping works for all supported timeframes
  - [x] Unit test: Pagination handles large date ranges correctly
  - [x] Integration test: Fetch live data from Binance (with @pytest.mark.live flag)
  - [x] Integration test: Fetch live data from Coinbase (with @pytest.mark.live flag)
  - [x] Integration test: Fetch live data from Kraken (with @pytest.mark.live flag)
  - [x] Integration test: Validate schema compliance for all exchanges
  - [x] Integration test: Error handling for delisted pairs and maintenance

- [x] Create example notebook (AC: 10)
  - [x] Create examples/crypto_backtest_ccxt.ipynb
  - [x] Demonstrate fetching BTC/USDT data from Binance
  - [x] Show multi-exchange comparison (Binance, Coinbase, Kraken)
  - [x] Implement simple moving average crossover strategy
  - [x] Display backtest results with performance metrics
  - [x] Document CCXT adapter configuration options

## Dev Notes

**Dependencies**: This story requires Story 3.4 (Implement Base Data Adapter Framework) to be completed first.

### Architecture Context

**Source Tree References:** [Source: architecture/source-tree.md]
- Implementation location:
  - `rustybt/data/adapters/ccxt_adapter.py` - CCXTDataAdapter implementation
- Test location:
  - `tests/data/adapters/test_ccxt_adapter.py`
- Example location:
  - `examples/crypto_backtest_ccxt.ipynb`

**Tech Stack Requirements:** [Source: architecture/tech-stack.md]
- **ccxt >= 4.x+**: Unified crypto exchange API for 100+ exchanges
- **Polars 1.x**: DataFrame library for data processing
- **Python Decimal**: For financial-grade arithmetic

**Component Architecture:** [Source: architecture/component-architecture.md#ccxtdataadapter]
- **CCXTDataAdapter**: Fetch crypto OHLCV data via CCXT
  - Support 100+ exchanges for historical data
  - Resolutions: 1m, 5m, 15m, 30m, 1h, 2h, 4h, 6h, 8h, 12h, 1d, 3d, 1w, 1M
  - History: Varies by exchange (typically 500-1000 bars per request)
  - Rate limiting per exchange using CCXT metadata

**External API Integration:** [Source: architecture/external-api-integration.md#ccxt-generic-multi-exchange]
- **Supported Exchanges**: Binance, Coinbase, Kraken, FTX, Huobi, OKX, Bitfinex, etc.
- **Rate Limits**: Automatic per-exchange via `enableRateLimit: True`
- **Error Handling**:
  - `ccxt.NetworkError`: Connection issues, retry
  - `ccxt.ExchangeError`: Exchange-specific errors
  - `ccxt.InsufficientFunds`: Insufficient balance (not applicable for data fetching)
  - `ccxt.InvalidOrder`: Order validation failed (not applicable for data fetching)

**CCXT OHLCV Format:**
```python
# CCXT returns list of lists: [[timestamp_ms, open, high, low, close, volume], ...]
ccxt_data = [
    [1609459200000, 29000.5, 29500.0, 28800.0, 29200.0, 1500.5],
    [1609545600000, 29200.0, 29800.0, 29100.0, 29500.0, 1800.2],
]
```

### Key Implementation Requirements

**CCXTAdapter Implementation:**
```python
import ccxt
import polars as pl
from decimal import Decimal
from typing import List, Optional
import pandas as pd
from rustybt.data.adapters.base import BaseDataAdapter, NetworkError, RateLimitError, InvalidDataError

class CCXTAdapter(BaseDataAdapter):
    """CCXT adapter for fetching crypto OHLCV data from 100+ exchanges."""

    RESOLUTION_MAPPING = {
        '1m': '1m',
        '5m': '5m',
        '15m': '15m',
        '30m': '30m',
        '1h': '1h',
        '4h': '4h',
        '1d': '1d',
        '1w': '1w',
    }

    def __init__(
        self,
        exchange_id: str = 'binance',
        testnet: bool = False,
        api_key: Optional[str] = None,
        api_secret: Optional[str] = None
    ):
        """Initialize CCXT adapter.

        Args:
            exchange_id: Exchange identifier (e.g., 'binance', 'coinbase', 'kraken')
            testnet: Use testnet/sandbox mode if available
            api_key: API key (optional, only needed for private endpoints)
            api_secret: API secret (optional)
        """
        # Get exchange class from CCXT
        exchange_class = getattr(ccxt, exchange_id)

        # Initialize exchange
        self.exchange = exchange_class({
            'enableRateLimit': True,  # Enable built-in rate limiting
            'options': {'defaultType': 'spot'}  # Use spot market by default
        })

        if api_key and api_secret:
            self.exchange.apiKey = api_key
            self.exchange.secret = api_secret

        if testnet and self.exchange.has['sandbox']:
            self.exchange.set_sandbox_mode(True)

        # Extract rate limit from exchange metadata
        rate_limit_ms = self.exchange.rateLimit  # Milliseconds between requests
        requests_per_second = 1000 / rate_limit_ms if rate_limit_ms > 0 else 10

        # Initialize base adapter with exchange-specific rate limit
        super().__init__(
            name=f"CCXTAdapter({exchange_id})",
            rate_limit_per_second=int(requests_per_second * 0.8)  # 80% of max for safety
        )

        self.exchange_id = exchange_id

    async def fetch(
        self,
        symbols: List[str],
        start_date: pd.Timestamp,
        end_date: pd.Timestamp,
        resolution: str
    ) -> pl.DataFrame:
        """Fetch OHLCV data from CCXT exchange.

        Args:
            symbols: List of trading pairs (e.g., ["BTC/USDT", "ETH/USDT"])
            start_date: Start date for data range
            end_date: End date for data range
            resolution: Time resolution (e.g., "1d", "1h", "1m")

        Returns:
            Polars DataFrame with standardized OHLCV schema

        Raises:
            NetworkError: If API request fails
            InvalidDataError: If symbol is invalid or delisted
        """
        # Map resolution to CCXT timeframe
        if resolution not in self.RESOLUTION_MAPPING:
            raise ValueError(
                f"Unsupported resolution: {resolution}. "
                f"Supported: {list(self.RESOLUTION_MAPPING.keys())}"
            )

        ccxt_timeframe = self.RESOLUTION_MAPPING[resolution]

        # Convert timestamps to Unix milliseconds
        since = int(start_date.timestamp() * 1000)
        until = int(end_date.timestamp() * 1000)

        all_data = []

        for symbol in symbols:
            # Normalize symbol format
            normalized_symbol = self._normalize_symbol(symbol)

            # Validate symbol exists on exchange
            if normalized_symbol not in self.exchange.markets:
                raise InvalidDataError(
                    f"Symbol {normalized_symbol} not found on {self.exchange_id}. "
                    f"Available markets: {len(self.exchange.markets)}"
                )

            # Fetch data with pagination (CCXT typically limits to 500-1000 bars)
            symbol_data = await self._fetch_with_pagination(
                symbol=normalized_symbol,
                timeframe=ccxt_timeframe,
                since=since,
                until=until
            )

            all_data.extend(symbol_data)

        # Convert to Polars DataFrame
        if not all_data:
            return pl.DataFrame(schema={
                'timestamp': pl.Datetime('us'),
                'symbol': pl.Utf8,
                'open': pl.Decimal(18, 8),
                'high': pl.Decimal(18, 8),
                'low': pl.Decimal(18, 8),
                'close': pl.Decimal(18, 8),
                'volume': pl.Decimal(18, 8),
            })

        df = pl.DataFrame({
            'timestamp': [pd.Timestamp(row[0], unit='ms') for row in all_data],
            'symbol': [row[6] for row in all_data],  # Symbol added in _fetch_with_pagination
            'open': [Decimal(str(row[1])) for row in all_data],
            'high': [Decimal(str(row[2])) for row in all_data],
            'low': [Decimal(str(row[3])) for row in all_data],
            'close': [Decimal(str(row[4])) for row in all_data],
            'volume': [Decimal(str(row[5])) for row in all_data],
        })

        # Standardize and validate
        df = self.standardize(df)
        self.validate(df)

        return df

    async def _fetch_with_pagination(
        self,
        symbol: str,
        timeframe: str,
        since: int,
        until: int
    ) -> List:
        """Fetch OHLCV data with pagination for large date ranges."""
        all_ohlcv = []
        current_since = since

        while current_since < until:
            # Apply rate limiting
            await self.rate_limiter.acquire()

            try:
                # Fetch batch
                ohlcv = await self.exchange.fetch_ohlcv(
                    symbol=symbol,
                    timeframe=timeframe,
                    since=current_since,
                    limit=1000  # Max bars per request
                )

                if not ohlcv:
                    break

                # Add symbol to each row
                ohlcv_with_symbol = [row + [symbol] for row in ohlcv]
                all_ohlcv.extend(ohlcv_with_symbol)

                # Update since for next iteration
                current_since = ohlcv[-1][0] + 1  # Last timestamp + 1ms

                # Break if we've reached the end
                if ohlcv[-1][0] >= until:
                    break

            except ccxt.NetworkError as e:
                raise NetworkError(f"CCXT network error: {e}") from e
            except ccxt.ExchangeNotAvailable as e:
                raise NetworkError(f"Exchange {self.exchange_id} unavailable: {e}") from e
            except ccxt.BadSymbol as e:
                raise InvalidDataError(f"Invalid symbol {symbol}: {e}") from e
            except ccxt.RateLimitExceeded as e:
                raise RateLimitError(f"Rate limit exceeded on {self.exchange_id}: {e}") from e

        return all_ohlcv

    def _normalize_symbol(self, symbol: str) -> str:
        """Normalize symbol format to CCXT standard (e.g., BTC/USDT)."""
        # Handle common formats
        symbol = symbol.upper().strip()

        # Already in correct format
        if '/' in symbol:
            return symbol

        # Handle dash format: BTC-USDT → BTC/USDT
        if '-' in symbol:
            return symbol.replace('-', '/')

        # Handle concatenated format: BTCUSDT → BTC/USDT (heuristic)
        # This is exchange-specific and may need refinement
        for quote in ['USDT', 'USD', 'BTC', 'ETH', 'EUR']:
            if symbol.endswith(quote):
                base = symbol[:-len(quote)]
                return f"{base}/{quote}"

        return symbol

    def validate(self, df: pl.DataFrame) -> bool:
        """Validate CCXT data (uses base class validation)."""
        return validate_ohlcv_relationships(df)

    def standardize(self, df: pl.DataFrame) -> pl.DataFrame:
        """Data already standardized in fetch() method."""
        return df
```

**Error Handling for Exchange-Specific Issues:**
```python
async def fetch_with_error_handling(self, symbols, start_date, end_date, resolution):
    """Fetch with comprehensive error handling."""

    try:
        return await self.fetch(symbols, start_date, end_date, resolution)

    except ccxt.ExchangeNotAvailable as e:
        logger.warning(
            "exchange_unavailable",
            exchange=self.exchange_id,
            error=str(e),
            retry_recommended=True
        )
        raise NetworkError(f"Exchange {self.exchange_id} is under maintenance") from e

    except ccxt.BadSymbol as e:
        logger.error(
            "invalid_symbol",
            exchange=self.exchange_id,
            symbols=symbols,
            error=str(e)
        )
        raise InvalidDataError(f"Symbol not found on {self.exchange_id}") from e

    except ccxt.RateLimitExceeded as e:
        logger.warning(
            "rate_limit_exceeded",
            exchange=self.exchange_id,
            error=str(e)
        )
        raise RateLimitError(f"Rate limit exceeded despite throttling") from e

    except ccxt.NetworkError as e:
        logger.error(
            "network_error",
            exchange=self.exchange_id,
            error=str(e),
            retryable=True
        )
        raise NetworkError(f"Network error connecting to {self.exchange_id}") from e
```

### Coding Standards

**Type Hints:** [Source: architecture/coding-standards.md#python-coding-standards]
- 100% type hint coverage for public APIs
- Example:
```python
async def fetch(
    self,
    symbols: List[str],
    start_date: pd.Timestamp,
    end_date: pd.Timestamp,
    resolution: str
) -> pl.DataFrame:
    """Fetch OHLCV data from CCXT exchange."""
```

**Docstrings:** [Source: architecture/coding-standards.md#python-coding-standards]
- Google-style docstrings for all public methods
- Include Args, Returns, Raises sections

**Decimal Usage:** [Source: architecture/coding-standards.md#decimal-precision]
- String construction: `Decimal(str(float_value))` for CCXT float conversion
- Polars Decimal type: `pl.Decimal(precision=18, scale=8)`

**Logging:** [Source: architecture/coding-standards.md#logging]
```python
import structlog
logger = structlog.get_logger()

logger.info(
    "ccxt_fetch_complete",
    exchange=self.exchange_id,
    symbols=symbols,
    rows=len(df),
    resolution=resolution,
    start_date=str(start_date),
    end_date=str(end_date)
)
```

### Testing

#### Testing Standards [Source: architecture/testing-strategy.md]

**Test File Location:**
- `tests/data/adapters/test_ccxt_adapter.py`

**Test Frameworks:**
- **pytest >= 7.2.0**: Primary test framework
- **pytest-asyncio**: For async adapter testing
- **pytest-cov >= 3.0.0**: Coverage reporting (target: ≥90%)

**Test Types:**

**Unit Tests:**
```python
import pytest
from decimal import Decimal
import polars as pl
from rustybt.data.adapters.ccxt_adapter import CCXTAdapter

def test_ccxt_data_format_conversion():
    """CCXT format converts to unified schema correctly."""

    ccxt_data = [
        [1609459200000, 29000.5, 29500.0, 28800.0, 29200.0, 1500.5, 'BTC/USDT'],
        [1609545600000, 29200.0, 29800.0, 29100.0, 29500.0, 1800.2, 'BTC/USDT'],
    ]

    adapter = CCXTAdapter(exchange_id='binance')

    df = pl.DataFrame({
        'timestamp': [pd.Timestamp(row[0], unit='ms') for row in ccxt_data],
        'symbol': [row[6] for row in ccxt_data],
        'open': [Decimal(str(row[1])) for row in ccxt_data],
        'high': [Decimal(str(row[2])) for row in ccxt_data],
        'low': [Decimal(str(row[3])) for row in ccxt_data],
        'close': [Decimal(str(row[4])) for row in ccxt_data],
        'volume': [Decimal(str(row[5])) for row in ccxt_data],
    })

    # Verify schema
    assert df.schema['open'] == pl.Decimal(18, 8)
    assert df.schema['high'] == pl.Decimal(18, 8)
    assert df.schema['timestamp'] == pl.Datetime('us')

def test_symbol_normalization():
    """Symbol normalization handles various formats."""

    adapter = CCXTAdapter(exchange_id='binance')

    assert adapter._normalize_symbol('BTC/USDT') == 'BTC/USDT'
    assert adapter._normalize_symbol('BTC-USDT') == 'BTC/USDT'
    assert adapter._normalize_symbol('BTCUSDT') == 'BTC/USDT'
    assert adapter._normalize_symbol('btc/usdt') == 'BTC/USDT'

def test_resolution_mapping():
    """Resolution mapping works for all supported timeframes."""

    adapter = CCXTAdapter(exchange_id='binance')

    assert adapter.RESOLUTION_MAPPING['1m'] == '1m'
    assert adapter.RESOLUTION_MAPPING['1h'] == '1h'
    assert adapter.RESOLUTION_MAPPING['1d'] == '1d'
    assert adapter.RESOLUTION_MAPPING['1w'] == '1w'
```

**Integration Tests (Live Data):**
```python
@pytest.mark.live
@pytest.mark.asyncio
async def test_fetch_live_data_binance():
    """Fetch live BTC/USDT data from Binance."""

    adapter = CCXTAdapter(exchange_id='binance')

    df = await adapter.fetch(
        symbols=['BTC/USDT'],
        start_date=pd.Timestamp('2024-01-01'),
        end_date=pd.Timestamp('2024-01-02'),
        resolution='1h'
    )

    # Verify data fetched
    assert len(df) > 0
    assert 'BTC/USDT' in df['symbol'].unique().to_list()

    # Verify schema compliance
    assert df.schema['open'] == pl.Decimal(18, 8)
    assert df.schema['timestamp'] == pl.Datetime('us')

    # Validate OHLCV relationships
    adapter.validate(df)

@pytest.mark.live
@pytest.mark.asyncio
async def test_fetch_live_data_coinbase():
    """Fetch live ETH/USD data from Coinbase."""

    adapter = CCXTAdapter(exchange_id='coinbase')

    df = await adapter.fetch(
        symbols=['ETH/USD'],
        start_date=pd.Timestamp('2024-01-01'),
        end_date=pd.Timestamp('2024-01-02'),
        resolution='1d'
    )

    assert len(df) > 0
    assert 'ETH/USD' in df['symbol'].unique().to_list()
    adapter.validate(df)

@pytest.mark.live
@pytest.mark.asyncio
async def test_multi_exchange_comparison():
    """Fetch same symbol from multiple exchanges."""

    exchanges = ['binance', 'coinbase', 'kraken']
    results = {}

    for exchange_id in exchanges:
        adapter = CCXTAdapter(exchange_id=exchange_id)

        df = await adapter.fetch(
            symbols=['BTC/USD'],
            start_date=pd.Timestamp('2024-01-01'),
            end_date=pd.Timestamp('2024-01-02'),
            resolution='1d'
        )

        results[exchange_id] = df

    # Verify all exchanges returned data
    for exchange_id, df in results.items():
        assert len(df) > 0, f"No data from {exchange_id}"
        assert df.schema['close'] == pl.Decimal(18, 8)
```

**Error Handling Tests:**
```python
@pytest.mark.asyncio
async def test_invalid_symbol_error():
    """Invalid symbol raises InvalidDataError."""

    adapter = CCXTAdapter(exchange_id='binance')

    with pytest.raises(InvalidDataError, match="not found"):
        await adapter.fetch(
            symbols=['INVALID/SYMBOL'],
            start_date=pd.Timestamp('2024-01-01'),
            end_date=pd.Timestamp('2024-01-02'),
            resolution='1d'
        )

@pytest.mark.asyncio
async def test_unsupported_resolution_error():
    """Unsupported resolution raises ValueError."""

    adapter = CCXTAdapter(exchange_id='binance')

    with pytest.raises(ValueError, match="Unsupported resolution"):
        await adapter.fetch(
            symbols=['BTC/USDT'],
            start_date=pd.Timestamp('2024-01-01'),
            end_date=pd.Timestamp('2024-01-02'),
            resolution='2h'  # Not in RESOLUTION_MAPPING
        )
```

**Zero-Mock Enforcement:** [Source: architecture/coding-standards.md#zero-mock-enforcement-mandatory]
- NO mock CCXT exchange objects (use real CCXT or testnet)
- Use real network requests for integration tests (with @pytest.mark.live flag)
- All validation must perform real OHLCV checks
- Use real Decimal conversion, not mocked

**Coverage Target:** [Source: architecture/testing-strategy.md#test-coverage-targets]
- Overall: ≥90%
- CCXT adapter: ≥90%

## Change Log

| Date | Version | Description | Author |
|------|---------|-------------|--------|
| 2025-10-01 | 1.0 | Initial story creation | SM (Bob) |

## Dev Agent Record

### Agent Model Used

claude-sonnet-4-5-20250929

### Debug Log References

None - Implementation completed without blocking issues

### Completion Notes List

- CCXT v4.5.6 successfully integrated with uv package manager
- CCXTAdapter implements full BaseDataAdapter interface with async support
- Exchange auto-discovery via CCXT - supports 100+ exchanges out of the box
- Rate limiting automatically configured per exchange (80% of max for safety margin)
- Symbol normalization handles slash, dash, and concatenated formats
- Pagination automatically handles large date ranges (1000 bars per request)
- Full error handling for NetworkError, ExchangeNotAvailable, BadSymbol, RateLimitExceeded
- Data standardization converts CCXT floats to Decimal(18, 8) with precision preservation
- Comprehensive test suite: 14 unit tests, 6 integration tests (marked with @pytest.mark.live)
- All unit tests passing (14/14) - integration tests require network access
- Example Jupyter notebook demonstrates multi-exchange comparison and SMA crossover strategy
- Code passes ruff linting with zero violations
- Type hints: 100% coverage with modern Python 3.12+ syntax (ClassVar, | union types)

### File List

**Source Files (Modified):**
- [pyproject.toml](../../pyproject.toml) - Added ccxt>=4.0.0 dependency and @pytest.mark.live marker

**Source Files (Created):**
- [rustybt/data/adapters/ccxt_adapter.py](../../rustybt/data/adapters/ccxt_adapter.py) - CCXTAdapter implementation (425 lines)
- [rustybt/data/adapters/__init__.py](../../rustybt/data/adapters/__init__.py) - Updated exports

**Test Files (Created):**
- [tests/data/adapters/test_ccxt_adapter.py](../../tests/data/adapters/test_ccxt_adapter.py) - Comprehensive test suite (378 lines)

**Example Files (Created):**
- [examples/crypto_backtest_ccxt.ipynb](../../examples/crypto_backtest_ccxt.ipynb) - Interactive example notebook

## QA Results

### Review Date: 2025-10-01

### Reviewed By: Quinn (Test Architect)

### Code Quality Assessment

**Overall Assessment: EXCELLENT**

The CCXTAdapter implementation demonstrates production-grade quality with comprehensive error handling, proper async patterns, and excellent adherence to project standards. The code is well-structured, thoroughly documented, and follows financial software best practices including Decimal precision and zero-mock enforcement.

**Key Strengths:**
- Clean architecture with proper separation of concerns (fetch, pagination, normalization, validation)
- Comprehensive CCXT exception mapping to domain errors
- Efficient pagination handling for large date ranges (1000 bars per request)
- Proper async/await with executor fallback for synchronous exchanges
- 100% type hint coverage using modern Python 3.12+ syntax
- Excellent structured logging with contextual information
- Token bucket rate limiting at 80% of exchange max for safety
- Symbol normalization handles multiple input formats (slash, dash, concatenated)

### Refactoring Performed

No refactoring required. The implementation is clean, efficient, and follows all coding standards without any code quality issues.

### Compliance Check

- Coding Standards: ✓ Full compliance
  - Modern Python 3.12+ features (ClassVar, union types with `|`)
  - 100% type hint coverage on public APIs
  - Google-style docstrings with complete Args/Returns/Raises sections
  - Ruff linting: All checks passed!
  - Decimal precision: Proper `Decimal(str(float))` construction
  - Structured logging with context throughout

- Project Structure: ✓ Full compliance
  - Source location: `rustybt/data/adapters/ccxt_adapter.py`
  - Test location: `tests/data/adapters/test_ccxt_adapter.py`
  - Proper module exports in `__init__.py`

- Testing Strategy: ✓ Full compliance
  - 14 unit tests covering all core functionality
  - 6 integration tests with `@pytest.mark.live` flag
  - Property-based Decimal precision validation
  - Test naming: `test_<function>_<scenario>` convention
  - All unit tests passing (14/14) ✓

- Zero-Mock Enforcement: ✓ Full compliance
  - No hardcoded return values - all methods perform real calculations
  - Real validation in `validate_ohlcv_relationships()`
  - Real symbol normalization with heuristic logic
  - No mocks in tests - uses real CCXT exchanges or explicit market setup
  - Integration tests validate live API behavior
  - All error handlers raise exceptions with context

- All ACs Met: ✓ 10/10 acceptance criteria fully implemented

### Requirements Traceability

All 10 acceptance criteria have comprehensive test coverage:

1. **AC1 - CCXT Integration**: ✓ ccxt>=4.0.0 in pyproject.toml, `test_ccxt_adapter_initialization`
2. **AC2 - BaseDataAdapter Interface**: ✓ Full implementation with fetch/validate/standardize methods
3. **AC3 - Exchange Selection**: ✓ `test_multi_exchange_comparison` validates Binance/Coinbase/Kraken
4. **AC4 - OHLCV Fetching**: ✓ `_fetch_with_pagination` uses CCXT fetch_ohlcv()
5. **AC5 - Multiple Resolutions**: ✓ `test_resolution_mapping` validates 9 supported timeframes
6. **AC6 - Rate Limiting**: ✓ Token bucket at 80% of exchange max, `test_ccxt_adapter_initialization` verifies
7. **AC7 - Decimal Precision**: ✓ `test_decimal_conversion_preserves_precision` validates Decimal(18,8)
8. **AC8 - Error Handling**: ✓ `test_fetch_invalid_symbol_error` + comprehensive exception mapping
9. **AC9 - Integration Tests**: ✓ 6 live tests for Binance/Coinbase/Kraken with schema validation
10. **AC10 - Example Notebook**: ✓ `crypto_backtest_ccxt.ipynb` demonstrates multi-exchange comparison

**Coverage Gap Identified:**
- AC10 notebook exists but lacks automated execution test (manual verification needed)

### Security Review

**Status: PASS**

- ✓ API credentials accepted as parameters, never hardcoded
- ✓ No secrets in logs - structured logging excludes sensitive data
- ✓ Input sanitization via symbol normalization before API calls
- ✓ Exception messages don't leak sensitive information
- ⚠️ Minor: No explicit validation for symbol injection characters (low risk - CCXT validates)

**Recommendation:** Consider adding explicit symbol character validation in future iteration

### Performance Considerations

**Status: PASS**

- ✓ Token bucket rate limiting prevents API throttling (80% safety margin)
- ✓ Efficient pagination handles 1000 bars per request
- ✓ Proper async/await with executor fallback for sync exchanges
- ✓ Batch symbol fetching supported
- ✓ Decimal string construction minimizes conversion overhead
- ⚠️ No caching layer (intentional - deferred to Story 3.3 cache_manager.py)

**Measured Performance:**
- Unit tests execute in <6s for 14 tests
- Testnet initialization: ~5.5s (includes market loading)

### NFR Validation Summary

**Security:** PASS - Proper credential handling, no sensitive data leakage
**Performance:** PASS - Efficient API usage, proper rate limiting
**Reliability:** PASS - Comprehensive error handling, graceful degradation
**Maintainability:** PASS - Excellent documentation, clean code structure

### Gate Status

Gate: **PASS** → [docs/qa/gates/3.5-implement-ccxt-data-adapter.yml](../../docs/qa/gates/3.5-implement-ccxt-data-adapter.yml)

### Recommended Status

**✓ Ready for Done**

All acceptance criteria met, tests passing, code quality excellent, standards compliant.

### Additional Notes

**Outstanding Quality Indicators:**
- Async/await executor fallback pattern for CCXT exchanges without native async support
- Comprehensive error taxonomy with proper exception chaining
- Symbol normalization supports 3 input formats (slash, dash, concatenated)
- Empty DataFrame with correct schema returned for edge cases
- Market loading fallback on first fetch if initialization fails

**Future Enhancement Opportunities (Non-Blocking):**
- Add retry decorator to `_fetch_with_pagination` for transient NetworkErrors
- Consider symbol character validation beyond format normalization
- Add notebook execution test to CI/CD pipeline
