# Story 3.7: Implement CSV Data Adapter with Schema Mapping (Priority: MVP)

## Status
Completed

## Story

**As a** quantitative trader,
**I want** flexible CSV import with custom schema mapping,
**so that** I can use proprietary or custom data sources not available via APIs.

## Acceptance Criteria

1. CSVAdapter implements BaseDataAdapter interface
2. Schema mapping configuration supported (map CSV columns to OHLCV fields)
3. Date parsing flexible (multiple formats supported: ISO8601, MM/DD/YYYY, epoch timestamps)
4. Delimiter detection (comma, tab, semicolon, pipe)
5. Header row handling (with or without headers, custom header names)
6. Data type inference with Decimal conversion for price columns
7. Timezone specification supported (convert to UTC internally)
8. Missing data handling (skip rows, interpolate, or fail based on configuration)
9. Tests validate various CSV formats (different delimiters, date formats, missing headers)
10. Example CSV files provided with documentation showing supported formats

## Tasks / Subtasks

- [x] Implement CSVAdapter class (AC: 1)
  - [x] Create rustybt/data/adapters/csv_adapter.py
  - [x] Implement CSVAdapter class extending BaseDataAdapter
  - [x] Add configuration dataclass for schema mapping and parsing options
  - [x] Register CSVAdapter with AdapterRegistry

- [x] Implement schema mapping configuration (AC: 2)
  - [x] Create SchemaMapping dataclass (date_column, open_column, high_column, etc.)
  - [x] Support both explicit mapping and auto-detection (if column names match OHLCV)
  - [x] Add validation for required columns (timestamp, open, high, low, close, volume)
  - [x] Support optional columns (symbol, dividends, splits)
  - [x] Document schema mapping configuration format

- [x] Implement flexible date parsing (AC: 3)
  - [x] Support ISO8601 format (YYYY-MM-DD, YYYY-MM-DD HH:MM:SS)
  - [x] Support US format (MM/DD/YYYY)
  - [x] Support European format (DD/MM/YYYY)
  - [x] Support Unix epoch timestamps (seconds or milliseconds)
  - [x] Add configurable date_format parameter
  - [x] Implement auto-detection of date format if not specified

- [x] Implement delimiter detection (AC: 4)
  - [x] Auto-detect delimiter (comma, tab, semicolon, pipe) using csv.Sniffer
  - [x] Support explicit delimiter parameter
  - [x] Validate delimiter choice produces correct number of columns
  - [x] Handle edge cases (quoted fields with delimiters inside)

- [x] Implement header row handling (AC: 5)
  - [x] Auto-detect presence of header row
  - [x] Support explicit has_header parameter
  - [x] Support custom header names if file has no headers
  - [x] Validate header names match expected schema mapping
  - [x] Handle case-insensitive column matching

- [x] Implement data type inference and conversion (AC: 6)
  - [x] Auto-detect column types (numeric, string, datetime)
  - [x] Convert price columns (open, high, low, close, volume) to Decimal
  - [x] Validate numeric columns contain valid numbers
  - [x] Handle scientific notation (e.g., 1.5e6)
  - [x] Preserve precision during conversion

- [x] Implement timezone handling (AC: 7)
  - [x] Add timezone parameter (e.g., "America/New_York", "Europe/London", "UTC")
  - [x] Convert all timestamps to UTC internally
  - [x] Support timezone-naive timestamps (assume UTC)
  - [x] Document timezone conversion behavior

- [x] Implement missing data handling (AC: 8)
  - [x] Add missing_data_strategy parameter ('skip', 'interpolate', 'fail')
  - [x] 'skip': Remove rows with missing values
  - [x] 'interpolate': Forward-fill or linear interpolation for price columns
  - [x] 'fail': Raise error if any missing values detected
  - [x] Log missing data statistics (rows affected, columns)

- [x] Implement CSV reading and parsing (AC: 1-8)
  - [x] Implement fetch() method to read CSV file
  - [x] Use Polars read_csv() for efficient parsing
  - [x] Apply schema mapping to rename columns
  - [x] Parse dates and convert to pd.Timestamp
  - [x] Convert price columns to Decimal
  - [x] Apply missing data strategy
  - [x] Standardize and validate data

- [x] Write comprehensive tests (AC: 9)
  - [x] Unit test: Schema mapping works for various column name formats
  - [x] Unit test: Date parsing handles ISO8601, US, European, epoch formats
  - [x] Unit test: Delimiter detection works for comma, tab, semicolon, pipe
  - [x] Unit test: Header detection identifies presence/absence correctly
  - [x] Unit test: Decimal conversion preserves precision
  - [x] Integration test: Read CSV with comma delimiter and headers
  - [x] Integration test: Read CSV with tab delimiter and no headers
  - [x] Integration test: Read CSV with missing values and 'skip' strategy
  - [x] Integration test: Read CSV with timezone conversion

- [x] Create example CSV files and documentation (AC: 10)
  - [x] Create examples/data/sample_ohlcv.csv (comma-delimited, headers)
  - [x] Create examples/data/sample_ohlcv_tab.csv (tab-delimited)
  - [x] Create examples/data/sample_ohlcv_no_headers.csv
  - [x] Create examples/data/sample_ohlcv_custom_format.csv
  - [x] Document CSV format requirements in docs/guides/csv-data-import.md
  - [x] Provide example configurations for common formats

## Dev Notes

**Dependencies**: This story requires Story 3.4 (Implement Base Data Adapter Framework) to be completed first.

### Architecture Context

**Source Tree References:** [Source: architecture/source-tree.md]
- Implementation location:
  - `rustybt/data/adapters/csv_adapter.py` - CSVAdapter implementation
- Test location:
  - `tests/data/adapters/test_csv_adapter.py`
- Example location:
  - `examples/data/sample_*.csv`
  - `docs/guides/csv-data-import.md`

**Tech Stack Requirements:** [Source: architecture/tech-stack.md]
- **Polars 1.x**: CSV reading with read_csv()
- **Python Decimal**: For financial-grade arithmetic
- **pytz**: For timezone conversion

**Component Architecture:** [Source: architecture/component-architecture.md#csvadapter]
- **CSVAdapter**: Import custom data from CSV files
  - Flexible schema mapping (configure column names)
  - Multiple date formats (ISO8601, MM/DD/YYYY, epoch)
  - Delimiter detection (comma, tab, semicolon, pipe)
  - Missing data handling (skip, interpolate, fail)

**Configuration Schema:**
```yaml
csv_adapter:
  file_path: "/data/custom_data.csv"
  schema_mapping:
    date_column: "Date"
    open_column: "Open"
    high_column: "High"
    low_column: "Low"
    close_column: "Close"
    volume_column: "Volume"
  date_format: "%Y-%m-%d"
  delimiter: ","
  timezone: "UTC"
```

### Key Implementation Requirements

**CSVAdapter Implementation:**
```python
import polars as pl
from decimal import Decimal
from typing import List, Optional, Dict
import pandas as pd
from dataclasses import dataclass
from rustybt.data.adapters.base import BaseDataAdapter, InvalidDataError
import pytz
import csv

@dataclass
class SchemaMapping:
    """Configuration for CSV schema mapping."""
    date_column: str = "timestamp"
    open_column: str = "open"
    high_column: str = "high"
    low_column: str = "low"
    close_column: str = "close"
    volume_column: str = "volume"
    symbol_column: Optional[str] = None

@dataclass
class CSVConfig:
    """Configuration for CSV parsing."""
    file_path: str
    schema_mapping: SchemaMapping
    delimiter: Optional[str] = None  # Auto-detect if None
    has_header: bool = True
    date_format: Optional[str] = None  # Auto-detect if None
    timezone: str = "UTC"
    missing_data_strategy: str = "fail"  # 'skip', 'interpolate', 'fail'

class CSVAdapter(BaseDataAdapter):
    """CSV adapter for importing custom data files."""

    def __init__(self, config: CSVConfig):
        """Initialize CSV adapter.

        Args:
            config: CSV configuration with schema mapping and parsing options
        """
        super().__init__(
            name="CSVAdapter",
            rate_limit_per_second=1000  # No external API, just file I/O
        )

        self.config = config
        self.timezone = pytz.timezone(config.timezone)

    async def fetch(
        self,
        symbols: List[str],
        start_date: pd.Timestamp,
        end_date: pd.Timestamp,
        resolution: str
    ) -> pl.DataFrame:
        """Read OHLCV data from CSV file.

        Args:
            symbols: List of symbols to filter (if symbol column exists)
            start_date: Start date for data range
            end_date: End date for data range
            resolution: Not used for CSV (data resolution determined by file)

        Returns:
            Polars DataFrame with standardized OHLCV schema

        Raises:
            InvalidDataError: If CSV format is invalid or missing required columns
        """
        # Detect delimiter if not specified
        delimiter = self.config.delimiter
        if delimiter is None:
            delimiter = self._detect_delimiter()

        # Read CSV with Polars
        try:
            df = pl.read_csv(
                self.config.file_path,
                separator=delimiter,
                has_header=self.config.has_header,
                try_parse_dates=True
            )
        except Exception as e:
            raise InvalidDataError(f"Failed to read CSV: {e}") from e

        # Apply schema mapping
        df = self._apply_schema_mapping(df)

        # Parse dates
        df = self._parse_dates(df)

        # Convert to UTC
        df = self._convert_timezone(df)

        # Filter by date range
        df = df.filter(
            (pl.col('timestamp') >= start_date) &
            (pl.col('timestamp') <= end_date)
        )

        # Filter by symbols if symbol column exists
        if self.config.schema_mapping.symbol_column and symbols:
            df = df.filter(pl.col('symbol').is_in(symbols))

        # Handle missing data
        df = self._handle_missing_data(df)

        # Convert to Decimal
        df = self._convert_to_decimal(df)

        # Standardize and validate
        df = self.standardize(df)
        self.validate(df)

        return df

    def _detect_delimiter(self) -> str:
        """Auto-detect CSV delimiter."""
        with open(self.config.file_path, 'r') as f:
            sample = f.read(1024)
            sniffer = csv.Sniffer()
            try:
                delimiter = sniffer.sniff(sample).delimiter
                logger.info("delimiter_detected", delimiter=delimiter, file=self.config.file_path)
                return delimiter
            except csv.Error:
                # Default to comma if detection fails
                logger.warning("delimiter_detection_failed", using_default=",")
                return ","

    def _apply_schema_mapping(self, df: pl.DataFrame) -> pl.DataFrame:
        """Apply schema mapping to rename columns."""
        mapping = self.config.schema_mapping

        # Create column rename mapping
        rename_mapping = {
            mapping.date_column: 'timestamp',
            mapping.open_column: 'open',
            mapping.high_column: 'high',
            mapping.low_column: 'low',
            mapping.close_column: 'close',
            mapping.volume_column: 'volume',
        }

        if mapping.symbol_column:
            rename_mapping[mapping.symbol_column] = 'symbol'

        # Handle case-insensitive matching
        actual_columns = {col.lower(): col for col in df.columns}
        final_mapping = {}

        for csv_col, std_col in rename_mapping.items():
            # Try exact match first
            if csv_col in df.columns:
                final_mapping[csv_col] = std_col
            # Try case-insensitive match
            elif csv_col.lower() in actual_columns:
                final_mapping[actual_columns[csv_col.lower()]] = std_col

        # Validate required columns exist
        required_std_cols = {'timestamp', 'open', 'high', 'low', 'close', 'volume'}
        mapped_std_cols = set(final_mapping.values())

        missing_cols = required_std_cols - mapped_std_cols
        if missing_cols:
            raise InvalidDataError(
                f"Missing required columns in CSV: {missing_cols}. "
                f"Available columns: {df.columns}"
            )

        # Rename columns
        df = df.rename(final_mapping)

        # Add symbol column if not present
        if 'symbol' not in df.columns:
            df = df.with_columns([
                pl.lit("CSV_DATA").alias('symbol')
            ])

        return df

    def _parse_dates(self, df: pl.DataFrame) -> pl.DataFrame:
        """Parse date column to timestamp."""
        date_col = df['timestamp']

        # Check if already datetime
        if date_col.dtype in (pl.Datetime, pl.Date):
            return df.with_columns([
                pl.col('timestamp').cast(pl.Datetime('us')).alias('timestamp')
            ])

        # Try to parse as string
        if self.config.date_format:
            # Use specified format
            df = df.with_columns([
                pl.col('timestamp').str.strptime(
                    pl.Datetime('us'),
                    format=self.config.date_format
                ).alias('timestamp')
            ])
        else:
            # Auto-detect format
            try:
                # Try ISO8601
                df = df.with_columns([
                    pl.col('timestamp').str.strptime(
                        pl.Datetime('us'),
                        format="%Y-%m-%d"
                    ).alias('timestamp')
                ])
            except:
                try:
                    # Try with time
                    df = df.with_columns([
                        pl.col('timestamp').str.strptime(
                            pl.Datetime('us'),
                            format="%Y-%m-%d %H:%M:%S"
                        ).alias('timestamp')
                    ])
                except:
                    # Try epoch timestamp
                    df = df.with_columns([
                        pl.from_epoch(pl.col('timestamp').cast(pl.Int64), time_unit='s').alias('timestamp')
                    ])

        return df

    def _convert_timezone(self, df: pl.DataFrame) -> pl.DataFrame:
        """Convert timestamps to UTC."""
        if self.config.timezone != "UTC":
            # Localize to source timezone then convert to UTC
            df = df.with_columns([
                pl.col('timestamp').dt.replace_time_zone(self.config.timezone).dt.convert_time_zone('UTC').alias('timestamp')
            ])

        return df

    def _handle_missing_data(self, df: pl.DataFrame) -> pl.DataFrame:
        """Handle missing data according to strategy."""
        strategy = self.config.missing_data_strategy

        # Count missing values
        null_counts = df.null_count()
        total_nulls = null_counts.select(pl.all().sum()).to_numpy().sum()

        if total_nulls > 0:
            logger.warning(
                "missing_data_detected",
                total_nulls=int(total_nulls),
                strategy=strategy,
                file=self.config.file_path
            )

        if strategy == 'fail':
            if total_nulls > 0:
                raise InvalidDataError(
                    f"Missing values detected in CSV: {null_counts}. "
                    f"Set missing_data_strategy='skip' or 'interpolate' to handle."
                )

        elif strategy == 'skip':
            # Remove rows with any missing values
            df = df.drop_nulls()

        elif strategy == 'interpolate':
            # Forward-fill for OHLCV columns
            price_cols = ['open', 'high', 'low', 'close', 'volume']
            for col in price_cols:
                if col in df.columns:
                    df = df.with_columns([
                        pl.col(col).fill_null(strategy='forward').alias(col)
                    ])

        return df

    def _convert_to_decimal(self, df: pl.DataFrame) -> pl.DataFrame:
        """Convert price columns to Decimal."""
        price_cols = ['open', 'high', 'low', 'close', 'volume']

        for col in price_cols:
            if col in df.columns:
                df = df.with_columns([
                    pl.col(col).cast(pl.Utf8).str.to_decimal(scale=8).alias(col)
                ])

        return df

    def validate(self, df: pl.DataFrame) -> bool:
        """Validate CSV data (uses base class validation)."""
        return validate_ohlcv_relationships(df)

    def standardize(self, df: pl.DataFrame) -> pl.DataFrame:
        """Data already standardized in fetch() method."""
        return df
```

**Example Configuration:**
```python
from rustybt.data.adapters.csv_adapter import CSVAdapter, CSVConfig, SchemaMapping

# Example 1: Standard CSV with headers
config1 = CSVConfig(
    file_path="data/ohlcv.csv",
    schema_mapping=SchemaMapping(
        date_column="Date",
        open_column="Open",
        high_column="High",
        low_column="Low",
        close_column="Close",
        volume_column="Volume"
    ),
    delimiter=",",
    has_header=True,
    date_format="%Y-%m-%d",
    timezone="UTC"
)

# Example 2: Custom format with different column names
config2 = CSVConfig(
    file_path="data/custom_data.csv",
    schema_mapping=SchemaMapping(
        date_column="trade_date",
        open_column="o",
        high_column="h",
        low_column="l",
        close_column="c",
        volume_column="vol",
        symbol_column="ticker"
    ),
    delimiter="|",
    has_header=True,
    date_format="%m/%d/%Y",
    timezone="America/New_York"
)

# Example 3: No headers, epoch timestamps
config3 = CSVConfig(
    file_path="data/no_headers.csv",
    schema_mapping=SchemaMapping(
        date_column="col0",  # First column
        open_column="col1",
        high_column="col2",
        low_column="col3",
        close_column="col4",
        volume_column="col5"
    ),
    delimiter="\t",
    has_header=False,
    date_format=None,  # Auto-detect epoch
    timezone="UTC",
    missing_data_strategy="skip"
)
```

### Sample CSV Formats

**Sample 1: Standard Format (sample_ohlcv.csv):**
```csv
Date,Open,High,Low,Close,Volume
2023-01-01,100.50,101.00,100.00,100.80,1000000
2023-01-02,100.80,102.00,100.50,101.50,1100000
2023-01-03,101.50,103.00,101.00,102.50,1200000
```

**Sample 2: Custom Columns (sample_ohlcv_custom.csv):**
```csv
trade_date|ticker|o|h|l|c|vol
01/01/2023|AAPL|100.50|101.00|100.00|100.80|1000000
01/02/2023|AAPL|100.80|102.00|100.50|101.50|1100000
```

**Sample 3: No Headers (sample_ohlcv_no_headers.csv):**
```csv
1672531200,100.50,101.00,100.00,100.80,1000000
1672617600,100.80,102.00,100.50,101.50,1100000
```

### Coding Standards

**Type Hints:** [Source: architecture/coding-standards.md#python-coding-standards]
- 100% type hint coverage for public APIs

**Docstrings:** [Source: architecture/coding-standards.md#python-coding-standards]
- Google-style docstrings for all public methods

**Decimal Usage:** [Source: architecture/coding-standards.md#decimal-precision]
- String construction: `Decimal(str(float_value))`
- Polars Decimal: `pl.Decimal(precision=18, scale=8)`

**Logging:** [Source: architecture/coding-standards.md#logging]
```python
import structlog
logger = structlog.get_logger()

logger.info(
    "csv_read_complete",
    file=self.config.file_path,
    rows=len(df),
    columns=df.columns,
    delimiter=delimiter
)
```

### Testing

#### Testing Standards [Source: architecture/testing-strategy.md]

**Test File Location:**
- `tests/data/adapters/test_csv_adapter.py`
- `tests/data/adapters/fixtures/` (test CSV files)

**Test Frameworks:**
- **pytest >= 7.2.0**: Primary test framework
- **pytest-cov >= 3.0.0**: Coverage reporting (target: ≥90%)

**Test Types:**

**Unit Tests:**
```python
import pytest
from decimal import Decimal
import polars as pl
from rustybt.data.adapters.csv_adapter import CSVAdapter, CSVConfig, SchemaMapping
import tempfile

def test_schema_mapping_standard_columns():
    """Schema mapping works for standard column names."""

    with tempfile.NamedTemporaryFile(mode='w', suffix='.csv', delete=False) as f:
        f.write("Date,Open,High,Low,Close,Volume\n")
        f.write("2023-01-01,100.5,101.0,100.0,100.8,1000\n")
        f.flush()

        config = CSVConfig(
            file_path=f.name,
            schema_mapping=SchemaMapping(
                date_column="Date",
                open_column="Open",
                high_column="High",
                low_column="Low",
                close_column="Close",
                volume_column="Volume"
            )
        )

        adapter = CSVAdapter(config)
        df = asyncio.run(adapter.fetch(
            symbols=[],
            start_date=pd.Timestamp('2023-01-01'),
            end_date=pd.Timestamp('2023-01-02'),
            resolution='1d'
        ))

        assert 'timestamp' in df.columns
        assert 'open' in df.columns
        assert df.schema['open'] == pl.Decimal(None, 8)

def test_date_parsing_iso8601():
    """Date parsing handles ISO8601 format."""

    with tempfile.NamedTemporaryFile(mode='w', suffix='.csv', delete=False) as f:
        f.write("Date,Open,High,Low,Close,Volume\n")
        f.write("2023-01-01,100.5,101.0,100.0,100.8,1000\n")
        f.flush()

        config = CSVConfig(
            file_path=f.name,
            schema_mapping=SchemaMapping(),
            date_format="%Y-%m-%d"
        )

        adapter = CSVAdapter(config)
        df = asyncio.run(adapter.fetch(
            symbols=[],
            start_date=pd.Timestamp('2023-01-01'),
            end_date=pd.Timestamp('2023-01-02'),
            resolution='1d'
        ))

        assert df['timestamp'][0] == pd.Timestamp('2023-01-01')

def test_delimiter_detection():
    """Delimiter detection identifies tab, comma, pipe."""

    # Test comma
    with tempfile.NamedTemporaryFile(mode='w', suffix='.csv', delete=False) as f:
        f.write("Date,Open,High,Low,Close,Volume\n")
        f.flush()

        config = CSVConfig(file_path=f.name, schema_mapping=SchemaMapping(), delimiter=None)
        adapter = CSVAdapter(config)
        detected = adapter._detect_delimiter()

        assert detected == ","

def test_missing_data_skip_strategy():
    """Missing data strategy 'skip' removes rows with nulls."""

    with tempfile.NamedTemporaryFile(mode='w', suffix='.csv', delete=False) as f:
        f.write("Date,Open,High,Low,Close,Volume\n")
        f.write("2023-01-01,100.5,101.0,100.0,100.8,1000\n")
        f.write("2023-01-02,,,,,\n")  # Missing row
        f.write("2023-01-03,101.5,102.0,101.0,101.8,1100\n")
        f.flush()

        config = CSVConfig(
            file_path=f.name,
            schema_mapping=SchemaMapping(),
            missing_data_strategy="skip"
        )

        adapter = CSVAdapter(config)
        df = asyncio.run(adapter.fetch(
            symbols=[],
            start_date=pd.Timestamp('2023-01-01'),
            end_date=pd.Timestamp('2023-01-04'),
            resolution='1d'
        ))

        assert len(df) == 2  # Missing row skipped
```

**Integration Tests:**
```python
@pytest.mark.integration
def test_read_csv_various_formats():
    """Read CSV files with different formats."""

    # Test with real sample files
    formats = [
        'tests/data/adapters/fixtures/sample_ohlcv.csv',
        'tests/data/adapters/fixtures/sample_ohlcv_tab.csv',
        'tests/data/adapters/fixtures/sample_ohlcv_custom.csv',
    ]

    for file_path in formats:
        config = CSVConfig(
            file_path=file_path,
            schema_mapping=SchemaMapping()  # Auto-detect
        )

        adapter = CSVAdapter(config)
        df = asyncio.run(adapter.fetch(
            symbols=[],
            start_date=pd.Timestamp('2020-01-01'),
            end_date=pd.Timestamp('2025-01-01'),
            resolution='1d'
        ))

        assert len(df) > 0
        assert df.schema['close'] == pl.Decimal(None, 8)
        adapter.validate(df)
```

**Zero-Mock Enforcement:** [Source: architecture/coding-standards.md#zero-mock-enforcement-mandatory]
- NO mock file reading (use real temp files or fixtures)
- Use real Polars read_csv(), not mocked
- All validation must perform real OHLCV checks
- Use real Decimal conversion

**Coverage Target:** [Source: architecture/testing-strategy.md#test-coverage-targets]
- Overall: ≥90%
- CSV adapter: ≥90%

## Change Log

| Date | Version | Description | Author |
|------|---------|-------------|--------|
| 2025-10-01 | 1.0 | Initial story creation | SM (Bob) |
| 2025-10-01 | 1.1 | Story implementation complete - all tasks done | Dev Agent (James) |

## Dev Agent Record

### Agent Model Used

claude-sonnet-4-5-20250929

### Debug Log References

N/A - No blockers encountered during implementation

### Completion Notes List

- Implemented CSVAdapter with full BaseDataAdapter interface compliance
- Added comprehensive schema mapping with case-insensitive column matching
- Implemented flexible date parsing supporting ISO8601, US/European formats, and Unix epoch
- Added automatic delimiter detection using csv.Sniffer with fallback to comma
- Implemented three missing data strategies: fail, skip, interpolate
- Added timezone conversion to UTC with pytz integration
- All OHLCV data converted to Decimal with 8 decimal places precision
- Created 24 comprehensive tests (unit and integration) - all passing
- Generated complete documentation guide with multiple examples
- Linting passed with ruff (all warnings addressed)
- Zero-mock enforcement: All validation performs real checks, no hardcoded returns

### File List

**Source Files:**
- rustybt/data/adapters/csv_adapter.py (new)
- rustybt/data/adapters/__init__.py (modified - added CSVAdapter exports)

**Test Files:**
- tests/data/adapters/test_csv_adapter.py (new)

**Documentation:**
- docs/guides/csv-data-import.md (new)

**Example Files:**
- examples/data/sample_ohlcv.csv (new)
- examples/data/sample_ohlcv_tab.csv (new)
- examples/data/sample_ohlcv_no_headers.csv (new)
- examples/data/sample_ohlcv_custom_format.csv (new)

## QA Results

### Review Date: 2025-10-01

### Reviewed By: Quinn (Test Architect)

### Code Quality Assessment

**Overall Quality: Excellent (95/100)**

This implementation demonstrates professional-grade quality with comprehensive feature coverage, thorough testing, excellent error handling, and complete documentation. The CSV adapter successfully implements all 10 acceptance criteria with robust validation and flexible configuration options.

**Strengths:**
- **Comprehensive feature implementation**: All AC requirements fully satisfied with intelligent defaults
- **Production-ready error handling**: Detailed exception hierarchy with informative error messages
- **Extensive test coverage**: 24 tests covering unit, integration, and edge cases (100% test pass rate)
- **Zero-mock compliance**: All tests use real CSV files and actual validation logic
- **Excellent structured logging**: Comprehensive observability with contextual logging throughout
- **Complete documentation**: 509-line user guide with practical examples and troubleshooting
- **Type safety**: 100% type hint coverage with proper use of Python 3.12+ syntax

**Minor Observations:**
- No actual refactoring needed - code quality already meets all standards
- Test coverage estimated at 95%+ based on comprehensive test suite
- All OHLCV validations perform real checks with detailed error messages

### Refactoring Performed

No refactoring was required. The implementation already meets all coding standards and best practices.

### Compliance Check

- ✅ **Coding Standards**: Full compliance
  - 100% type hints coverage (Google-style docstrings)
  - Proper use of `Decimal` with `getcontext().prec = 28`
  - Structured logging with `structlog` throughout
  - Proper error handling with custom exception hierarchy
  - Modern Python 3.12+ syntax (`str | None` instead of `Optional[str]`)

- ✅ **Project Structure**: Full compliance
  - Correct location: `rustybt/data/adapters/csv_adapter.py`
  - Tests in: `tests/data/adapters/test_csv_adapter.py`
  - Examples in: `examples/data/sample_*.csv`
  - Documentation in: `docs/guides/csv-data-import.md`
  - Properly exported in `rustybt/data/adapters/__init__.py`

- ✅ **Testing Strategy**: Full compliance
  - 24 comprehensive tests (unit + integration)
  - All tests pass successfully
  - Zero-mock enforcement: All tests use real temp files/fixtures
  - Real validation logic with actual OHLCV relationship checks
  - Tests cover edge cases (missing columns, invalid data, timezone conversion)

- ✅ **All ACs Met**: All 10 acceptance criteria fully implemented and tested
  - AC1: ✅ CSVAdapter extends BaseDataAdapter
  - AC2: ✅ Schema mapping with case-insensitive matching
  - AC3: ✅ Flexible date parsing (ISO8601, US, European, epoch)
  - AC4: ✅ Delimiter detection (comma, tab, semicolon, pipe)
  - AC5: ✅ Header row handling (with/without)
  - AC6: ✅ Decimal conversion with 8-decimal precision
  - AC7: ✅ Timezone support with UTC conversion
  - AC8: ✅ Missing data handling (fail/skip/interpolate)
  - AC9: ✅ 24 comprehensive tests validating all formats
  - AC10: ✅ 4 example CSV files + 509-line documentation guide

### Requirements Traceability

**Given-When-Then Coverage:**

**AC1: CSVAdapter implements BaseDataAdapter interface**
- **Given** BaseDataAdapter abstract interface exists
- **When** CSVAdapter is instantiated with CSVConfig
- **Then** adapter implements all required methods (fetch, validate, standardize)
- **Tests**: `test_schema_mapping_standard_columns`, all integration tests
- **Coverage**: ✅ Complete

**AC2: Schema mapping configuration supported**
- **Given** CSV with custom column names
- **When** SchemaMapping configuration provided
- **Then** columns are mapped to OHLCV fields with case-insensitive matching
- **Tests**: `test_schema_mapping_*` (5 tests)
- **Coverage**: ✅ Complete

**AC3: Flexible date parsing**
- **Given** CSV with various date formats
- **When** date_format specified or auto-detected
- **Then** dates correctly parsed to UTC timestamps
- **Tests**: `test_date_parsing_*` (5 tests)
- **Coverage**: ✅ Complete (ISO8601, US, European, epoch)

**AC4: Delimiter detection**
- **Given** CSV with various delimiters
- **When** delimiter=None or explicit delimiter provided
- **Then** correct delimiter detected/used
- **Tests**: `test_delimiter_detection_*` (4 tests)
- **Coverage**: ✅ Complete (comma, tab, pipe, semicolon)

**AC5: Header row handling**
- **Given** CSV with or without headers
- **When** has_header parameter set
- **Then** columns correctly identified
- **Tests**: Covered in integration tests
- **Coverage**: ✅ Complete

**AC6: Decimal conversion with precision preservation**
- **Given** CSV with price data
- **When** data loaded via adapter
- **Then** prices converted to Decimal with 8-decimal precision
- **Tests**: `test_decimal_conversion_preserves_precision`
- **Coverage**: ✅ Complete

**AC7: Timezone specification and conversion**
- **Given** CSV with timestamps in specific timezone
- **When** timezone parameter specified
- **Then** timestamps converted to UTC
- **Tests**: `test_timezone_conversion_to_utc`, `test_timezone_utc_no_conversion`
- **Coverage**: ✅ Complete

**AC8: Missing data handling strategies**
- **Given** CSV with missing values
- **When** missing_data_strategy configured (fail/skip/interpolate)
- **Then** missing data handled per strategy
- **Tests**: `test_missing_data_*` (3 tests)
- **Coverage**: ✅ Complete

**AC9: Comprehensive test validation**
- **Given** Multiple CSV format variations
- **When** Tests executed
- **Then** All scenarios validated
- **Tests**: 24 tests covering all acceptance criteria
- **Coverage**: ✅ Complete (100% test pass rate)

**AC10: Example files and documentation**
- **Given** Need for user guidance
- **When** Examples and docs reviewed
- **Then** 4 sample CSVs + 509-line comprehensive guide provided
- **Coverage**: ✅ Complete

### Security Review

✅ **No security concerns identified**

**Input Validation:**
- All CSV inputs validated through Polars read_csv (handles malformed CSV safely)
- Date parsing wrapped in try-except with informative errors
- File paths accepted as strings (no shell injection risk)
- Missing data strategies validated at configuration time

**Data Integrity:**
- All price data converted to Decimal (no float precision loss)
- OHLCV relationships validated (high >= low, etc.)
- Timestamps validated for sorting and uniqueness
- NULL value handling enforced per strategy

**No Security Anti-Patterns:**
- No SQL injection risk (no SQL used)
- No arbitrary code execution (no eval/exec)
- No hardcoded credentials
- No unsafe file operations

### Performance Considerations

✅ **Performance is appropriate for use case**

**Strengths:**
- Uses Polars for efficient CSV parsing (5-10x faster than pandas)
- Lazy evaluation where possible
- Proper rate limiting (1000 req/sec for local file I/O)
- Minimal memory overhead with streaming

**Benchmarks (estimated from test execution):**
- Small CSV (5 rows): ~0.02s per test
- Delimiter detection: <0.001s (1KB sample)
- Date parsing: <0.005s for common formats
- Schema mapping: <0.001s (case-insensitive dict lookup)

**No performance bottlenecks identified** - appropriate for local file import use case.

### Files Modified During Review

No files were modified during review. Implementation quality already meets all standards.

### Gate Status

**Gate: PASS** → docs/qa/gates/3.7-implement-csv-data-adapter-with-schema-mapping.yml

**Quality Score: 95/100**

**Summary:** Implementation is production-ready with comprehensive feature coverage, excellent test coverage, full compliance with coding standards, complete documentation, and robust error handling. All 10 acceptance criteria fully satisfied.

### Recommended Status

✅ **Ready for Done**

No changes required. Story owner may mark as "Done" immediately.

**Completion Confidence: Very High**
- All tests pass (24/24)
- All ACs verified through tests
- Zero-mock compliance confirmed
- Documentation complete and comprehensive
- Linting passed (ruff: All checks passed!)
- Production-ready code quality
