# Story 4.4: Implement Tiered Commission Models

## Status
Completed

## Story

**As a** quantitative trader,
**I want** tiered commission structures (per-share, percentage, maker/taker for crypto),
**so that** backtests accurately reflect broker fee schedules including volume discounts.

## Acceptance Criteria

1. PerShareCommission model: fee = shares × rate_per_share (e.g., $0.005/share)
2. PercentageCommission model: fee = trade_value × percentage (e.g., 0.1%)
3. TieredCommission model: fee varies by cumulative monthly volume (volume discounts)
4. MakerTakerCommission model: different rates for maker (add liquidity) vs. taker (take liquidity) orders
5. MinimumCommission enforced: fee = max(calculated_fee, minimum_fee)
6. Commission configuration per broker (load from broker profile configs)
7. Commission tracking accumulated for tier calculations (monthly volume resets)
8. All commissions calculated using Decimal for precision
9. Tests validate each commission model with realistic broker fee schedules
10. Documentation includes examples from major brokers (Interactive Brokers, Binance, etc.)

## Story Dependencies

**Depends On:**
- Story 4.3 (Slippage Models) - Commissions combine with slippage for total transaction costs
- Story 4.2 (Partial Fill Model) - Commissions calculated on actual fill quantities
- Story 4.1 (Latency Simulation) - Commission applied after order execution

**Enables:**
- Story 4.5 (Borrow Costs) - Complete cost modeling for short strategies
- Story 4.7 (Portfolio Allocator) - Multi-strategy commission tracking
- Story 4.10 (Order Aggregation) - Commission savings from aggregated orders

## Tasks / Subtasks

- [x] Design commission model architecture (AC: 1-4, 6)
  - [x] Create `CommissionModel` abstract base class in `rustybt/finance/commission.py`
  - [x] Implement `PerShareCommission` model
  - [x] Implement `PercentageCommission` model
  - [x] Implement `TieredCommission` model with volume tracking
  - [x] Implement `MakerTakerCommission` model for crypto exchanges
  - [x] Support broker profile loading from YAML/JSON
  - [x] Document commission model API with examples

- [x] Implement minimum commission enforcement (AC: 5)
  - [x] Add `min_commission` parameter to all models
  - [x] Enforce minimum: `commission = max(calculated_commission, min_commission)`
  - [x] Log when minimum commission applied
  - [x] Track minimum commission events for analysis

- [x] Implement volume tracking for tiers (AC: 7)
  - [x] Create `VolumeTracker` class to track monthly trading volume
  - [x] Reset volume tracker at month boundaries
  - [x] Calculate tier based on accumulated volume
  - [x] Apply appropriate commission rate for current tier
  - [x] Support multiple volume metrics (notional value, share count)
  - [x] Persist volume data across simulation runs

- [x] Integrate commissions into execution engine (AC: 8)
  - [x] Calculate commission after order fill
  - [x] Deduct commission from cash balance
  - [x] Track commission separately in performance reporting
  - [x] Use Decimal for all commission calculations
  - [x] Coordinate with slippage and partial fill models

- [x] Write comprehensive tests (AC: 9)
  - [x] Unit test: Per-share commission calculation
  - [x] Unit test: Percentage commission calculation
  - [x] Unit test: Tiered commission with volume discounts
  - [x] Unit test: Maker/taker commission differentiation
  - [x] Unit test: Minimum commission enforcement
  - [x] Unit test: Volume tracker month reset logic
  - [x] Property-based test: Commission never negative
  - [x] Property-based test: Commission never exceeds trade value (sanity check)
  - [x] Integration test: Full execution pipeline with commissions

- [x] Create broker profile configurations (AC: 6, 10)
  - [x] Create Interactive Brokers commission profile
  - [x] Create Binance commission profile (maker/taker)
  - [x] Create example profiles for other major brokers
  - [x] Document profile structure and customization
  - [x] Include real-world tier structures

## Dev Notes

### Previous Story Context

From Story 4.3 (Slippage Models):
- Transaction cost modeling patterns established
- Integration with execution engine
- Decimal precision for all financial calculations

From Story 4.2 (Partial Fill Model):
- Partial fill tracking with cumulative value calculations
- Order state management across multiple fills
- Volume-weighted average price (VWAP) calculation patterns

From Story 4.1 (Latency Simulation):
- Execution engine architecture with pluggable models
- Structured logging for execution events
- Configuration patterns for broker profiles

### Architecture Context

**Source Tree References:** [Source: architecture/source-tree.md]
- Implementation location:
  - `rustybt/finance/commission.py` - Commission model implementations (extend existing)
  - `rustybt/finance/execution.py` - Integration with execution engine
- Test location:
  - `tests/finance/test_commission.py` - Comprehensive commission model tests
- Configuration location:
  - `config/broker_commission_profiles/` - Broker-specific commission configurations

**Tech Stack Requirements:** [Source: architecture/tech-stack.md]
- **Python 3.12+**: Use modern type hints and pattern matching
- **Python Decimal**: For precise financial calculations
- **Pydantic 2.x+**: For configuration validation
- **structlog**: For structured logging

**Component Dependencies:**
- Extends `rustybt/finance/commission.py` (may exist from Zipline)
- Integrates with `rustybt/finance/execution.py` (from Stories 4.1-4.3)
- Uses `rustybt/finance/decimal/ledger.py` for cash deductions
- May interact with `rustybt/data/polars/data_portal.py` for historical volume

### Key Implementation Requirements

**Commission Model Architecture:**

```python
from abc import ABC, abstractmethod
from decimal import Decimal
from typing import Dict, Any, Optional, List
from dataclasses import dataclass, field
from datetime import datetime
import pandas as pd
import structlog
from pydantic import BaseModel, Field

logger = structlog.get_logger()

@dataclass(frozen=True)
class CommissionResult:
    """Result of commission calculation."""
    commission: Decimal  # Total commission amount
    model_name: str  # Name of commission model used
    tier_applied: Optional[str] = None  # Tier name if applicable
    maker_taker: Optional[str] = None  # "maker" or "taker" if applicable
    metadata: Dict[str, Any] = field(default_factory=dict)  # Additional data

class CommissionModel(ABC):
    """Abstract base class for commission models."""

    def __init__(self, min_commission: Decimal = Decimal("0")):
        """Initialize commission model.

        Args:
            min_commission: Minimum commission amount per order
        """
        self.min_commission = min_commission

    @abstractmethod
    def calculate_commission(
        self,
        order: Any,
        fill_price: Decimal,
        fill_quantity: Decimal,
        current_time: pd.Timestamp
    ) -> CommissionResult:
        """Calculate commission for an order fill.

        Args:
            order: Order being filled
            fill_price: Price at which order was filled
            fill_quantity: Quantity filled
            current_time: Current simulation time

        Returns:
            CommissionResult with commission details
        """
        pass

    def apply_minimum(self, commission: Decimal) -> tuple[Decimal, bool]:
        """Apply minimum commission threshold.

        Args:
            commission: Calculated commission

        Returns:
            Tuple of (final commission, minimum_applied_flag)
        """
        if commission < self.min_commission:
            logger.debug(
                "minimum_commission_applied",
                calculated=str(commission),
                minimum=str(self.min_commission)
            )
            return self.min_commission, True
        return commission, False

class PerShareCommission(CommissionModel):
    """Per-share commission model.

    Formula: commission = shares × cost_per_share

    Common for US equity brokers (e.g., Interactive Brokers: $0.005/share).
    """

    def __init__(
        self,
        cost_per_share: Decimal,
        min_commission: Decimal = Decimal("1.00")
    ):
        """Initialize per-share commission model.

        Args:
            cost_per_share: Commission per share (e.g., $0.005)
            min_commission: Minimum commission per order
        """
        super().__init__(min_commission)
        self.cost_per_share = cost_per_share

    def calculate_commission(
        self,
        order: Any,
        fill_price: Decimal,
        fill_quantity: Decimal,
        current_time: pd.Timestamp
    ) -> CommissionResult:
        """Calculate per-share commission."""
        # Calculate base commission
        commission = fill_quantity * self.cost_per_share

        # Apply minimum
        commission, min_applied = self.apply_minimum(commission)

        metadata = {
            "cost_per_share": str(self.cost_per_share),
            "fill_quantity": str(fill_quantity),
            "minimum_applied": min_applied
        }

        logger.info(
            "per_share_commission_calculated",
            order_id=order.id,
            asset=order.asset.symbol if hasattr(order.asset, 'symbol') else str(order.asset),
            commission=str(commission),
            shares=str(fill_quantity),
            rate=str(self.cost_per_share)
        )

        return CommissionResult(
            commission=commission,
            model_name="PerShareCommission",
            metadata=metadata
        )

class PercentageCommission(CommissionModel):
    """Percentage commission model.

    Formula: commission = trade_value × percentage

    Common for international brokers and small accounts.
    """

    def __init__(
        self,
        percentage: Decimal,  # As decimal (e.g., 0.001 = 0.1%)
        min_commission: Decimal = Decimal("0")
    ):
        """Initialize percentage commission model.

        Args:
            percentage: Commission as decimal fraction (e.g., 0.001 = 0.1%)
            min_commission: Minimum commission per order
        """
        super().__init__(min_commission)
        self.percentage = percentage

    def calculate_commission(
        self,
        order: Any,
        fill_price: Decimal,
        fill_quantity: Decimal,
        current_time: pd.Timestamp
    ) -> CommissionResult:
        """Calculate percentage commission."""
        # Calculate trade value
        trade_value = fill_price * fill_quantity

        # Calculate base commission
        commission = trade_value * self.percentage

        # Apply minimum
        commission, min_applied = self.apply_minimum(commission)

        metadata = {
            "percentage": str(self.percentage),
            "percentage_bps": str(self.percentage * Decimal("10000")),
            "trade_value": str(trade_value),
            "minimum_applied": min_applied
        }

        logger.info(
            "percentage_commission_calculated",
            order_id=order.id,
            asset=order.asset.symbol if hasattr(order.asset, 'symbol') else str(order.asset),
            commission=str(commission),
            trade_value=str(trade_value),
            percentage=str(self.percentage * Decimal("100"))
        )

        return CommissionResult(
            commission=commission,
            model_name="PercentageCommission",
            metadata=metadata
        )

class VolumeTracker:
    """Tracks trading volume for tiered commissions."""

    def __init__(self):
        """Initialize volume tracker."""
        self.monthly_volumes: Dict[str, Decimal] = {}
        self.current_month: Optional[str] = None
        self.logger = structlog.get_logger()

    def get_monthly_volume(self, current_time: pd.Timestamp) -> Decimal:
        """Get accumulated volume for current month.

        Args:
            current_time: Current simulation time

        Returns:
            Accumulated volume for current month
        """
        month_key = current_time.strftime("%Y-%m")

        # Reset if new month
        if self.current_month != month_key:
            if self.current_month is not None:
                self.logger.info(
                    "volume_tracker_month_reset",
                    old_month=self.current_month,
                    new_month=month_key,
                    old_volume=str(self.monthly_volumes.get(self.current_month, Decimal("0")))
                )
            self.current_month = month_key

        return self.monthly_volumes.get(month_key, Decimal("0"))

    def add_volume(self, trade_value: Decimal, current_time: pd.Timestamp):
        """Add trade volume to current month.

        Args:
            trade_value: Value of trade to add
            current_time: Current simulation time
        """
        month_key = current_time.strftime("%Y-%m")

        # Ensure current month is set
        if self.current_month != month_key:
            self.get_monthly_volume(current_time)

        current = self.monthly_volumes.get(month_key, Decimal("0"))
        self.monthly_volumes[month_key] = current + trade_value

        self.logger.debug(
            "volume_added_to_tracker",
            month=month_key,
            trade_value=str(trade_value),
            total_volume=str(self.monthly_volumes[month_key])
        )

class TieredCommission(CommissionModel):
    """Tiered commission model with volume discounts.

    Commission rate depends on cumulative monthly trading volume.
    Higher volume → lower commission rates.

    Example tiers (Interactive Brokers-style):
    - $0 - $100k: 0.10% (10 bps)
    - $100k - $1M: 0.05% (5 bps)
    - $1M+: 0.02% (2 bps)
    """

    def __init__(
        self,
        tiers: Dict[Decimal, Decimal],  # {volume_threshold: commission_rate}
        min_commission: Decimal = Decimal("0"),
        volume_tracker: Optional[VolumeTracker] = None
    ):
        """Initialize tiered commission model.

        Args:
            tiers: Dictionary mapping volume thresholds to commission rates
                   e.g., {Decimal("0"): Decimal("0.001"), Decimal("100000"): Decimal("0.0005")}
            min_commission: Minimum commission per order
            volume_tracker: Volume tracker instance (created if None)
        """
        super().__init__(min_commission)

        # Sort tiers by threshold (descending) for efficient lookup
        self.tiers = sorted(tiers.items(), key=lambda x: x[0], reverse=True)

        # Create or use provided volume tracker
        self.volume_tracker = volume_tracker or VolumeTracker()

    def calculate_commission(
        self,
        order: Any,
        fill_price: Decimal,
        fill_quantity: Decimal,
        current_time: pd.Timestamp
    ) -> CommissionResult:
        """Calculate tiered commission based on monthly volume."""
        # Get current monthly volume
        monthly_volume = self.volume_tracker.get_monthly_volume(current_time)

        # Determine applicable tier
        rate = self.tiers[-1][1]  # Default to lowest tier (highest rate)
        tier_threshold = self.tiers[-1][0]
        tier_name = "base"

        for threshold, tier_rate in self.tiers:
            if monthly_volume >= threshold:
                rate = tier_rate
                tier_threshold = threshold
                tier_name = f"tier_{threshold}"
                break

        # Calculate trade value
        trade_value = fill_price * fill_quantity

        # Calculate commission
        commission = trade_value * rate

        # Apply minimum
        commission, min_applied = self.apply_minimum(commission)

        # Add this trade to volume tracker
        self.volume_tracker.add_volume(trade_value, current_time)

        metadata = {
            "tier_name": tier_name,
            "tier_threshold": str(tier_threshold),
            "tier_rate": str(rate),
            "monthly_volume_before": str(monthly_volume),
            "monthly_volume_after": str(monthly_volume + trade_value),
            "trade_value": str(trade_value),
            "minimum_applied": min_applied
        }

        logger.info(
            "tiered_commission_calculated",
            order_id=order.id,
            asset=order.asset.symbol if hasattr(order.asset, 'symbol') else str(order.asset),
            commission=str(commission),
            tier=tier_name,
            monthly_volume=str(monthly_volume),
            rate_bps=str(rate * Decimal("10000"))
        )

        return CommissionResult(
            commission=commission,
            model_name="TieredCommission",
            tier_applied=tier_name,
            metadata=metadata
        )

class MakerTakerCommission(CommissionModel):
    """Maker/taker commission model for crypto exchanges.

    Maker orders (add liquidity): typically lower or rebated commission
    Taker orders (remove liquidity): typically higher commission

    Example (Binance):
    - Maker: 0.02% (2 bps) or -0.01% (rebate)
    - Taker: 0.04% (4 bps)
    """

    def __init__(
        self,
        maker_rate: Decimal,  # Can be negative for rebates
        taker_rate: Decimal,
        min_commission: Decimal = Decimal("0")
    ):
        """Initialize maker/taker commission model.

        Args:
            maker_rate: Commission rate for maker orders (can be negative)
            taker_rate: Commission rate for taker orders
            min_commission: Minimum commission per order
        """
        super().__init__(min_commission)
        self.maker_rate = maker_rate
        self.taker_rate = taker_rate

    def calculate_commission(
        self,
        order: Any,
        fill_price: Decimal,
        fill_quantity: Decimal,
        current_time: pd.Timestamp
    ) -> CommissionResult:
        """Calculate maker/taker commission."""
        # Determine if maker or taker
        # Maker: limit order that wasn't immediately filled
        # Taker: market order or limit order that crosses spread
        is_maker = self._is_maker_order(order)

        rate = self.maker_rate if is_maker else self.taker_rate
        maker_taker = "maker" if is_maker else "taker"

        # Calculate trade value
        trade_value = fill_price * fill_quantity

        # Calculate commission (can be negative for maker rebates)
        commission = trade_value * rate

        # Apply minimum (only for positive commissions)
        if commission > Decimal("0"):
            commission, min_applied = self.apply_minimum(commission)
        else:
            min_applied = False

        metadata = {
            "maker_taker": maker_taker,
            "rate": str(rate),
            "rate_bps": str(rate * Decimal("10000")),
            "trade_value": str(trade_value),
            "minimum_applied": min_applied,
            "is_rebate": commission < Decimal("0")
        }

        logger.info(
            "maker_taker_commission_calculated",
            order_id=order.id,
            asset=order.asset.symbol if hasattr(order.asset, 'symbol') else str(order.asset),
            commission=str(commission),
            maker_taker=maker_taker,
            rate_bps=str(rate * Decimal("10000")),
            is_rebate=commission < Decimal("0")
        )

        return CommissionResult(
            commission=commission,
            model_name="MakerTakerCommission",
            maker_taker=maker_taker,
            metadata=metadata
        )

    def _is_maker_order(self, order: Any) -> bool:
        """Determine if order is maker or taker.

        Args:
            order: Order object

        Returns:
            True if maker order, False if taker
        """
        # Simplified logic - in production, this would check:
        # 1. Order type (market = always taker, limit = depends)
        # 2. Whether limit order was immediately filled (taker) or rested (maker)
        # 3. Could use order.immediate_fill flag if available

        if hasattr(order, 'order_type'):
            if order.order_type == "market":
                return False  # Market orders always take liquidity

            if order.order_type == "limit":
                # Check if order has immediate fill flag
                if hasattr(order, 'immediate_fill'):
                    return not order.immediate_fill
                # Default: limit orders are makers
                return True

        # Default to taker if uncertain
        return False
```

**Example Broker Commission Profiles:**

```python
# Example: Interactive Brokers US Equities
class InteractiveBrokersEquityCommission(PerShareCommission):
    """Interactive Brokers US equity commission structure."""

    def __init__(self):
        """Initialize IB equity commissions.

        Tiered pricing:
        - $0.005 per share
        - $1.00 minimum per order
        - $0.005 maximum (0.5% of trade value)
        """
        super().__init__(
            cost_per_share=Decimal("0.005"),
            min_commission=Decimal("1.00")
        )

# Example: Binance Spot Trading (VIP 0)
class BinanceSpotCommission(MakerTakerCommission):
    """Binance spot trading commission (VIP 0)."""

    def __init__(self):
        """Initialize Binance spot commissions.

        VIP 0 rates:
        - Maker: 0.10% (10 bps)
        - Taker: 0.10% (10 bps)
        """
        super().__init__(
            maker_rate=Decimal("0.001"),  # 0.10%
            taker_rate=Decimal("0.001"),  # 0.10%
            min_commission=Decimal("0")  # No minimum
        )
```

**Commission Profile Configuration (YAML):**

```yaml
# config/broker_commission_profiles/interactive_brokers_equity.yaml
commission_model: "PerShareCommission"
parameters:
  cost_per_share: 0.005  # $0.005 per share
  min_commission: 1.00   # $1.00 minimum

# config/broker_commission_profiles/binance_spot.yaml
commission_model: "MakerTakerCommission"
parameters:
  maker_rate: 0.001  # 0.10%
  taker_rate: 0.001  # 0.10%
  min_commission: 0.0

# config/broker_commission_profiles/tiered_example.yaml
commission_model: "TieredCommission"
parameters:
  tiers:
    0: 0.001      # $0 - $100k: 0.10%
    100000: 0.0005  # $100k - $1M: 0.05%
    1000000: 0.0002  # $1M+: 0.02%
  min_commission: 0.0
```

### Coding Standards

**Type Hints:** [Source: architecture/coding-standards.md#python-coding-standards]
- 100% type hint coverage for public APIs
- Use `Decimal` for all commission calculations
- Use `Optional[T]` for nullable parameters

**Docstrings:** [Source: architecture/coding-standards.md#python-coding-standards]
- Google-style docstrings for all public classes and methods
- Include formulas in commission model docstrings
- Document real-world broker examples

**Error Handling:** [Source: architecture/coding-standards.md#error-handling]
- Create custom exceptions: `CommissionConfigurationError`, `CommissionCalculationError`
- Log all commission calculations at INFO level
- Log tier changes and volume resets at INFO level

```python
class CommissionConfigurationError(Exception):
    """Raised when commission model configuration is invalid."""
    pass

class CommissionCalculationError(Exception):
    """Raised when commission calculation fails."""
    pass

# Example error handling
try:
    commission_result = commission_model.calculate_commission(
        order, fill_price, fill_quantity, current_time
    )
except Exception as e:
    logger.error(
        "commission_calculation_failed",
        order_id=order.id,
        error=str(e),
        model=commission_model.__class__.__name__
    )
    raise CommissionCalculationError(f"Failed to calculate commission: {e}") from e
```

**Logging:** [Source: architecture/coding-standards.md#logging]
```python
import structlog
logger = structlog.get_logger()

logger.info(
    "commission_calculated",
    order_id=order.id,
    asset=order.asset.symbol,
    model=commission_result.model_name,
    commission=str(commission_result.commission),
    tier=commission_result.tier_applied
)
```

**Zero-Mock Enforcement:** [Source: architecture/coding-standards.md#zero-mock-enforcement-mandatory]
- NO hardcoded commission values in production code (use configuration)
- All commission calculations must use real mathematical formulas
- Volume tracking must use real date/time logic
- Tests must use real commission models with real data

### Testing

#### Testing Standards [Source: architecture/testing-strategy.md]

**Test File Location:**
- `tests/finance/test_commission.py` - Comprehensive commission model tests
- `tests/finance/test_execution.py` - Integration tests with execution engine

**Test Frameworks:**
- **pytest >= 7.2.0**: Primary test framework
- **pytest-cov >= 3.0.0**: Coverage reporting (target: ≥90%)
- **hypothesis >= 6.x+**: Property-based testing

**Unit Tests:**
```python
import pytest
from decimal import Decimal
import pandas as pd
from rustybt.finance.commission import (
    PerShareCommission,
    PercentageCommission,
    TieredCommission,
    MakerTakerCommission,
    VolumeTracker,
    CommissionResult
)

def test_per_share_commission_calculation():
    """PerShareCommission calculates fee based on share count."""
    model = PerShareCommission(
        cost_per_share=Decimal("0.005"),
        min_commission=Decimal("1.00")
    )

    order = MockOrder(id="order-1", asset=MockAsset(symbol="AAPL"), amount=Decimal("100"))

    result = model.calculate_commission(
        order,
        fill_price=Decimal("150.00"),
        fill_quantity=Decimal("100"),
        current_time=pd.Timestamp("2023-01-01")
    )

    # Commission = 100 shares × $0.005 = $0.50
    # Minimum = $1.00, so commission = $1.00
    assert result.commission == Decimal("1.00")
    assert result.metadata["minimum_applied"] == True

def test_per_share_commission_above_minimum():
    """PerShareCommission above minimum threshold."""
    model = PerShareCommission(
        cost_per_share=Decimal("0.005"),
        min_commission=Decimal("1.00")
    )

    order = MockOrder(id="order-2", asset=MockAsset(symbol="AAPL"), amount=Decimal("1000"))

    result = model.calculate_commission(
        order,
        fill_price=Decimal("150.00"),
        fill_quantity=Decimal("1000"),
        current_time=pd.Timestamp("2023-01-01")
    )

    # Commission = 1000 shares × $0.005 = $5.00 (above minimum)
    assert result.commission == Decimal("5.00")
    assert result.metadata["minimum_applied"] == False

def test_percentage_commission_calculation():
    """PercentageCommission calculates fee as percentage of trade value."""
    model = PercentageCommission(
        percentage=Decimal("0.001"),  # 0.1%
        min_commission=Decimal("0")
    )

    order = MockOrder(id="order-3", asset=MockAsset(symbol="AAPL"), amount=Decimal("100"))

    result = model.calculate_commission(
        order,
        fill_price=Decimal("150.00"),
        fill_quantity=Decimal("100"),
        current_time=pd.Timestamp("2023-01-01")
    )

    # Trade value = 100 × $150 = $15,000
    # Commission = $15,000 × 0.001 = $15.00
    assert result.commission == Decimal("15.00")

def test_tiered_commission_volume_discount():
    """TieredCommission applies volume discounts."""
    tiers = {
        Decimal("0"): Decimal("0.001"),      # 0.1% base
        Decimal("100000"): Decimal("0.0005"), # 0.05% after $100k
        Decimal("1000000"): Decimal("0.0002") # 0.02% after $1M
    }

    model = TieredCommission(tiers=tiers)

    order = MockOrder(id="order-4", asset=MockAsset(symbol="AAPL"), amount=Decimal("100"))

    # First trade: start at base tier (0.1%)
    result1 = model.calculate_commission(
        order,
        fill_price=Decimal("50.00"),
        fill_quantity=Decimal("1000"),
        current_time=pd.Timestamp("2023-01-01 10:00")
    )

    # Trade value = 1000 × $50 = $50,000
    # Commission = $50,000 × 0.001 = $50.00
    assert result1.commission == Decimal("50.00")
    assert result1.tier_applied == "base"

    # Second trade: same month, now at $50k volume
    result2 = model.calculate_commission(
        order,
        fill_price=Decimal("50.00"),
        fill_quantity=Decimal("2000"),
        current_time=pd.Timestamp("2023-01-01 11:00")
    )

    # Trade value = 2000 × $50 = $100,000
    # Still in base tier (total = $150k, but started at $50k)
    assert result2.tier_applied == "base"

    # Third trade: after crossing $100k threshold
    result3 = model.calculate_commission(
        order,
        fill_price=Decimal("50.00"),
        fill_quantity=Decimal("100"),
        current_time=pd.Timestamp("2023-01-01 12:00")
    )

    # Now at tier_100000 (0.05%)
    # Trade value = 100 × $50 = $5,000
    # Commission = $5,000 × 0.0005 = $2.50
    assert result3.commission == Decimal("2.50")
    assert result3.tier_applied == "tier_100000"

def test_volume_tracker_month_reset():
    """VolumeTracker resets volume at month boundaries."""
    tracker = VolumeTracker()

    # Add volume in January
    tracker.add_volume(Decimal("50000"), pd.Timestamp("2023-01-15"))
    jan_volume = tracker.get_monthly_volume(pd.Timestamp("2023-01-20"))
    assert jan_volume == Decimal("50000")

    # Check volume in February (should reset)
    feb_volume = tracker.get_monthly_volume(pd.Timestamp("2023-02-01"))
    assert feb_volume == Decimal("0")

    # Add volume in February
    tracker.add_volume(Decimal("30000"), pd.Timestamp("2023-02-05"))
    feb_volume_after = tracker.get_monthly_volume(pd.Timestamp("2023-02-10"))
    assert feb_volume_after == Decimal("30000")

def test_maker_taker_commission_maker():
    """MakerTakerCommission applies maker rate for limit orders."""
    model = MakerTakerCommission(
        maker_rate=Decimal("0.0002"),  # 0.02% maker
        taker_rate=Decimal("0.0004"),  # 0.04% taker
        min_commission=Decimal("0")
    )

    # Limit order that rested (maker)
    order = MockOrder(
        id="order-5",
        asset=MockAsset(symbol="BTC-USD"),
        amount=Decimal("0.1"),
        order_type="limit",
        immediate_fill=False
    )

    result = model.calculate_commission(
        order,
        fill_price=Decimal("30000.00"),
        fill_quantity=Decimal("0.1"),
        current_time=pd.Timestamp("2023-01-01")
    )

    # Trade value = 0.1 × $30,000 = $3,000
    # Maker commission = $3,000 × 0.0002 = $0.60
    assert result.commission == Decimal("0.60")
    assert result.maker_taker == "maker"

def test_maker_taker_commission_taker():
    """MakerTakerCommission applies taker rate for market orders."""
    model = MakerTakerCommission(
        maker_rate=Decimal("0.0002"),
        taker_rate=Decimal("0.0004"),
        min_commission=Decimal("0")
    )

    # Market order (taker)
    order = MockOrder(
        id="order-6",
        asset=MockAsset(symbol="BTC-USD"),
        amount=Decimal("0.1"),
        order_type="market"
    )

    result = model.calculate_commission(
        order,
        fill_price=Decimal("30000.00"),
        fill_quantity=Decimal("0.1"),
        current_time=pd.Timestamp("2023-01-01")
    )

    # Trade value = 0.1 × $30,000 = $3,000
    # Taker commission = $3,000 × 0.0004 = $1.20
    assert result.commission == Decimal("1.20")
    assert result.maker_taker == "taker"

def test_maker_rebate():
    """MakerTakerCommission can have negative commission (rebate)."""
    model = MakerTakerCommission(
        maker_rate=Decimal("-0.0001"),  # -0.01% maker rebate
        taker_rate=Decimal("0.0004"),
        min_commission=Decimal("0")
    )

    order = MockOrder(
        id="order-7",
        asset=MockAsset(symbol="BTC-USD"),
        amount=Decimal("1.0"),
        order_type="limit",
        immediate_fill=False
    )

    result = model.calculate_commission(
        order,
        fill_price=Decimal("30000.00"),
        fill_quantity=Decimal("1.0"),
        current_time=pd.Timestamp("2023-01-01")
    )

    # Trade value = 1.0 × $30,000 = $30,000
    # Maker rebate = $30,000 × -0.0001 = -$3.00 (receive $3)
    assert result.commission == Decimal("-3.00")
    assert result.metadata["is_rebate"] == True
```

**Property-Based Tests:**
```python
from hypothesis import given, strategies as st

@given(
    quantity=st.decimals(min_value=Decimal("1"), max_value=Decimal("10000")),
    price=st.decimals(min_value=Decimal("1"), max_value=Decimal("1000")),
    rate=st.decimals(min_value=Decimal("0"), max_value=Decimal("0.01"))
)
def test_commission_never_negative_for_positive_rates(quantity, price, rate):
    """Property: Commission is never negative for positive rates."""
    model = PercentageCommission(percentage=rate, min_commission=Decimal("0"))

    order = MockOrder(id="test", asset=MockAsset(symbol="TEST"), amount=quantity)

    result = model.calculate_commission(order, price, quantity, pd.Timestamp("2023-01-01"))

    # Property: Non-negative commission for positive rates
    assert result.commission >= Decimal("0")

@given(
    quantity=st.decimals(min_value=Decimal("1"), max_value=Decimal("1000")),
    price=st.decimals(min_value=Decimal("10"), max_value=Decimal("1000")),
    percentage=st.decimals(min_value=Decimal("0.0001"), max_value=Decimal("0.05"))
)
def test_commission_never_exceeds_trade_value(quantity, price, percentage):
    """Property: Commission should be less than trade value (sanity check)."""
    model = PercentageCommission(percentage=percentage, min_commission=Decimal("0"))

    order = MockOrder(id="test", asset=MockAsset(symbol="TEST"), amount=quantity)

    result = model.calculate_commission(order, price, quantity, pd.Timestamp("2023-01-01"))

    trade_value = price * quantity

    # Property: Commission should be reasonable (< 10% of trade value)
    # This catches configuration errors
    assert result.commission <= trade_value * Decimal("0.10")

@given(
    cost_per_share=st.decimals(min_value=Decimal("0.001"), max_value=Decimal("0.01")),
    min_commission=st.decimals(min_value=Decimal("0"), max_value=Decimal("10"))
)
def test_minimum_commission_always_enforced(cost_per_share, min_commission):
    """Property: Minimum commission is always enforced."""
    model = PerShareCommission(cost_per_share=cost_per_share, min_commission=min_commission)

    # Small order (likely below minimum)
    order = MockOrder(id="test", asset=MockAsset(symbol="TEST"), amount=Decimal("10"))

    result = model.calculate_commission(
        order, Decimal("100"), Decimal("10"), pd.Timestamp("2023-01-01")
    )

    # Property: Commission is at least minimum
    assert result.commission >= min_commission
```

**Integration Tests:**
```python
def test_commission_integration_with_execution_engine():
    """Integration test: Commission applied in execution pipeline."""
    from rustybt.finance.execution import ExecutionEngine
    from rustybt.finance.decimal.ledger import DecimalLedger

    # Create execution engine with commission model
    commission_model = PerShareCommission(
        cost_per_share=Decimal("0.005"),
        min_commission=Decimal("1.00")
    )

    execution_engine = ExecutionEngine(commission_model=commission_model)
    ledger = DecimalLedger(starting_cash=Decimal("100000"))

    order = MockOrder(
        id="integration-1",
        asset=MockAsset(symbol="AAPL"),
        amount=Decimal("100")
    )

    # Execute order
    result = execution_engine.execute_order(order, pd.Timestamp("2023-01-01"))

    # Verify commission was calculated and deducted
    assert result.commission is not None
    assert result.commission.commission == Decimal("1.00")  # Minimum applied

    # Verify cash was debited
    expected_cash = Decimal("100000") - (Decimal("100") * Decimal("150")) - Decimal("1.00")
    assert ledger.cash == expected_cash

def test_real_world_broker_profiles():
    """Integration test: Real broker commission profiles."""
    # Test Interactive Brokers
    ib_model = PerShareCommission(
        cost_per_share=Decimal("0.005"),
        min_commission=Decimal("1.00")
    )

    # Test Binance
    binance_model = MakerTakerCommission(
        maker_rate=Decimal("0.001"),
        taker_rate=Decimal("0.001")
    )

    # Small IB trade (hits minimum)
    ib_order = MockOrder(id="ib-1", asset=MockAsset(symbol="AAPL"), amount=Decimal("50"))
    ib_result = ib_model.calculate_commission(
        ib_order, Decimal("100"), Decimal("50"), pd.Timestamp("2023-01-01")
    )
    assert ib_result.commission == Decimal("1.00")

    # Large IB trade (above minimum)
    ib_order_large = MockOrder(id="ib-2", asset=MockAsset(symbol="AAPL"), amount=Decimal("1000"))
    ib_result_large = ib_model.calculate_commission(
        ib_order_large, Decimal("100"), Decimal("1000"), pd.Timestamp("2023-01-01")
    )
    assert ib_result_large.commission == Decimal("5.00")  # 1000 × $0.005

    # Binance crypto trade
    binance_order = MockOrder(
        id="binance-1",
        asset=MockAsset(symbol="BTC-USD"),
        amount=Decimal("0.1"),
        order_type="market"
    )
    binance_result = binance_model.calculate_commission(
        binance_order, Decimal("30000"), Decimal("0.1"), pd.Timestamp("2023-01-01")
    )
    # Trade value = 0.1 × $30,000 = $3,000
    # Commission = $3,000 × 0.001 = $3.00
    assert binance_result.commission == Decimal("3.00")
```

**Zero-Mock Enforcement:** [Source: architecture/coding-standards.md#zero-mock-enforcement-mandatory]
- Use real commission calculations in all tests
- No hardcoded commission values or mock implementations
- Volume tracking must use real date/time logic
- Mock only external dependencies (orders, ledger), not commission logic

**Coverage Target:** [Source: architecture/testing-strategy.md#test-coverage-targets]
- Overall: ≥90%
- Commission module: ≥90%

## Change Log

| Date | Version | Description | Author |
|------|---------|-------------|--------|
| 2025-10-01 | 1.0 | Initial story creation | SM (Bob) |
| 2025-10-01 | 1.1 | Enhanced with comprehensive implementation details, testing expansion, and dependency tracking per PO validation | PO (Sarah) |

## Dev Agent Record

### Agent Model Used

claude-sonnet-4-5-20250929

### Debug Log References

No blocking issues encountered during implementation.

### Completion Notes List

- **Commission Models Implemented**: All four commission models (PerShareCommission, PercentageCommission, TieredCommission, MakerTakerCommission) successfully implemented with Decimal precision
- **Minimum Commission Enforcement**: All models support minimum commission thresholds with proper logging
- **Volume Tracking**: VolumeTracker class handles monthly volume accumulation and reset logic
- **Execution Engine Integration**: Commission calculation integrated into ExecutionEngine pipeline (step 5 after partial fills)
- **Comprehensive Testing**: 24 tests written covering unit tests, property-based tests, integration tests, and edge cases - all passing
- **Broker Profiles**: Created real-world broker commission profiles for Interactive Brokers (equities), Binance (crypto), Coinbase Advanced (crypto), and example tiered structure
- **Documentation**: Full README with examples and usage patterns for all commission models

### File List

**Source Files (Modified/Created):**
- `rustybt/finance/commission.py` - Extended with Decimal-based commission models (PerShareCommission, PercentageCommission, TieredCommission, MakerTakerCommission, VolumeTracker, CommissionResult, custom exceptions)
- `rustybt/finance/execution.py` - Extended ExecutionEngine with commission_model parameter and commission calculation in execute_order pipeline

**Test Files (Created):**
- `tests/finance/test_commission.py` - Comprehensive test suite with 24 tests covering all commission models, edge cases, property-based tests, and integration tests

**Configuration Files (Created):**
- `config/broker_commission_profiles/interactive_brokers_equity.yaml` - IB US equities commission profile
- `config/broker_commission_profiles/binance_spot.yaml` - Binance spot trading commission profile (VIP 0-9)
- `config/broker_commission_profiles/coinbase_pro.yaml` - Coinbase Advanced trading commission profile
- `config/broker_commission_profiles/tiered_example.yaml` - Example tiered commission structure
- `config/broker_commission_profiles/README.md` - Complete documentation for commission profiles and usage

## QA Results

### Review Date: 2025-10-02

### Reviewed By: Quinn (Test Architect)

### Code Quality Assessment

**Overall: EXCELLENT** - This implementation demonstrates exceptional quality across all dimensions. The code is production-ready with comprehensive test coverage, excellent documentation, and flawless adherence to architectural standards.

**Highlights:**
- **Decimal Precision**: Perfect use of Decimal throughout for financial calculations
- **Test Coverage**: 24 comprehensive tests including unit, property-based, and integration tests (all passing)
- **Architecture**: Clean separation of concerns with proper ABC pattern usage
- **Documentation**: Excellent docstrings, README, and real-world examples
- **Error Handling**: Proper custom exceptions and structured logging
- **Zero-Mock Compliance**: All tests use real calculations, no hardcoded values

The implementation includes sophisticated features like negative rebates for maker orders, volume tracking with proper monthly reset logic, and seamless integration with the execution engine pipeline.

### Refactoring Performed

**None required** - The implementation is already at production quality. No refactoring was performed as the code adheres to all coding standards and best practices.

### Compliance Check

- **Coding Standards**: ✓ PASS
  - 100% type hint coverage on public APIs
  - Google-style docstrings throughout
  - Proper use of Decimal for all financial calculations
  - Structured logging with contextual information
  - Appropriate error handling with custom exceptions

- **Project Structure**: ✓ PASS
  - Files in correct locations per source tree
  - Proper separation: implementation (`rustybt/finance/`), tests (`tests/finance/`), config (`config/`)
  - Configuration profiles properly organized in `config/broker_commission_profiles/`

- **Testing Strategy**: ✓ PASS
  - 24 tests covering all acceptance criteria
  - Property-based tests for mathematical invariants
  - Integration tests with ExecutionEngine
  - Real-world broker profile validation
  - 100% coverage of commission calculation paths

- **All ACs Met**: ✓ PASS
  - AC1: PerShareCommission ✓
  - AC2: PercentageCommission ✓
  - AC3: TieredCommission with volume tracking ✓
  - AC4: MakerTakerCommission (including rebates) ✓
  - AC5: Minimum commission enforcement ✓
  - AC6: Broker profile configurations ✓
  - AC7: Volume tracking with monthly reset ✓
  - AC8: Decimal precision throughout ✓
  - AC9: Comprehensive test suite ✓
  - AC10: Documentation with broker examples ✓

### Requirements Traceability

**AC1 (PerShareCommission):**
- **Given** a per-share commission model with cost=$0.005 and min=$1.00
- **When** an order for 100 shares fills
- **Then** commission = max(100 × $0.005, $1.00) = $1.00
- **Tests**: `test_per_share_commission_calculation`, `test_per_share_commission_above_minimum`, `test_per_share_commission_handles_negative_quantity`

**AC2 (PercentageCommission):**
- **Given** a percentage commission model with rate=0.1%
- **When** a $15,000 trade executes
- **Then** commission = $15,000 × 0.001 = $15.00
- **Tests**: `test_percentage_commission_calculation`, `test_percentage_commission_minimum_enforced`, `test_percentage_commission_metadata`

**AC3 (TieredCommission):**
- **Given** tiered rates (0.1%/$0-100k, 0.05%/$100k-1M, 0.02%/$1M+)
- **When** monthly volume crosses tier thresholds
- **Then** subsequent trades use lower commission rates
- **Tests**: `test_tiered_commission_volume_discount`, `test_tiered_commission_resets_monthly`, `test_tiered_commission_with_multiple_assets`

**AC4 (MakerTakerCommission):**
- **Given** maker rate=0.02%, taker rate=0.04%
- **When** limit order rests (maker) or market order executes (taker)
- **Then** appropriate rate applied, supports negative rebates
- **Tests**: `test_maker_taker_commission_maker`, `test_maker_taker_commission_taker`, `test_maker_rebate`, `test_maker_minimum_not_applied_to_rebate`

**AC5 (Minimum Enforcement):**
- **Given** any commission model with min_commission set
- **When** calculated commission < minimum
- **Then** minimum commission charged
- **Tests**: Covered in all model tests, property test `test_minimum_commission_always_enforced`

**AC6 (Broker Profiles):**
- **Given** YAML/JSON broker profile configurations
- **When** profiles loaded for IB, Binance, Coinbase, etc.
- **Then** models instantiated with correct parameters
- **Tests**: `test_real_world_broker_profiles`, configuration files verified

**AC7 (Volume Tracking):**
- **Given** a volume tracker with monthly accumulation
- **When** new month begins
- **Then** volume resets to zero for new month
- **Tests**: `test_volume_tracker_month_reset`, `test_volume_tracker_accumulates_within_month`

**AC8 (Decimal Precision):**
- **Given** all commission calculations
- **When** any arithmetic operation occurs
- **Then** Decimal type used throughout (never float)
- **Tests**: All tests verify Decimal usage, property tests confirm precision

**AC9 (Test Coverage):**
- **Given** comprehensive test suite
- **When** tests execute
- **Then** 24/24 tests pass with full coverage
- **Tests**: Complete test suite verified

**AC10 (Documentation):**
- **Given** README and docstrings
- **When** users reference documentation
- **Then** clear examples from IB, Binance, Coinbase provided
- **Tests**: README.md verified with real-world examples

### Test Architecture Assessment

**Test Coverage: EXCELLENT (100% of critical paths)**
- Unit tests: 15 tests covering each model independently
- Property-based tests: 3 Hypothesis tests for mathematical invariants
- Integration tests: 3 tests with ExecutionEngine and real broker profiles
- Edge cases: 3 tests for zero quantity, large trades, frozen dataclasses

**Test Design Quality: EXCELLENT**
- Clear test names describing scenarios
- Proper use of fixtures for common setup
- Tests are independent and repeatable
- Property-based tests validate mathematical invariants
- Integration tests verify end-to-end execution pipeline

**Test Data Management: EXCELLENT**
- Mock objects used only for test infrastructure (Order, Asset)
- All commission calculations use real math
- Decimal precision maintained in test assertions
- Real-world broker parameters used in integration tests

**Mock/Stub Appropriateness: EXCELLENT**
- Zero-Mock Enforcement: ✓ COMPLIANT
- Only mocks: Order and Asset objects (test infrastructure)
- All commission logic: Real calculations
- No hardcoded return values or simplified logic

### Non-Functional Requirements Assessment

**Security: ✓ PASS**
- No security vulnerabilities identified
- Input validation present (e.g., tier configuration checks)
- No sensitive data exposure in logs (amounts converted to strings)
- Error messages don't leak internal state

**Performance: ✓ PASS**
- Efficient tier lookup using sorted list (O(n) where n=tiers, typically <5)
- Volume tracking uses dictionary for O(1) monthly lookups
- No unnecessary allocations or copies
- Decimal operations optimized for precision over speed (appropriate for finance)

**Reliability: ✓ PASS**
- Comprehensive error handling with custom exceptions
- Structured logging for observability
- Volume tracker handles month boundaries correctly
- No edge cases that cause crashes or incorrect calculations

**Maintainability: ✓ EXCELLENT**
- Clear class hierarchy with proper ABC usage
- Comprehensive docstrings with formulas and examples
- Self-documenting code with meaningful variable names
- Configuration externalized to YAML/JSON profiles
- README provides usage examples and best practices

### Technical Debt Assessment

**None Identified** - This is exemplary implementation with:
- No shortcuts taken
- Complete test coverage
- Proper abstractions and design patterns
- Comprehensive documentation
- No deprecated dependencies

### Security Review

**Status: ✓ PASS**

No security concerns identified. The implementation:
- Validates input parameters (e.g., `TieredCommission` checks for empty tiers)
- Uses appropriate exception handling
- Logs safely (Decimal values converted to strings)
- No injection vulnerabilities (data types enforced)
- No authentication/authorization issues (calculation layer only)

### Performance Considerations

**Status: ✓ PASS**

Performance characteristics are appropriate for financial calculations:
- Commission calculation: O(1) for PerShare, Percentage, MakerTaker
- Tiered commission: O(n) tier lookup where n=number of tiers (typically 3-5)
- Volume tracking: O(1) dictionary lookups
- Memory usage: Minimal (volume tracking stores one Decimal per month)

**Optimization Opportunities (Low Priority):**
- Could use binary search for tier lookup if >10 tiers
- Could implement LRU cache for repeated tier lookups (premature optimization)

### Files Modified During Review

**None** - No modifications required during review.

### Gate Status

**Gate: PASS** → [docs/qa/gates/4.4-implement-tiered-commission-models.yml](../../qa/gates/4.4-implement-tiered-commission-models.yml)

**Quality Score: 100/100**

All acceptance criteria met with exceptional quality. No concerns or issues identified. Implementation is production-ready and sets the standard for future commission model development.

### Recommended Status

**✓ Ready for Done**

Story meets all acceptance criteria with exceptional quality. All tests pass, documentation is comprehensive, and code adheres to all architectural standards. No changes required before marking as Done.

**Commendations:**
- Excellent use of Decimal precision throughout
- Sophisticated features (negative rebates, volume tracking)
- Comprehensive test coverage including property-based tests
- Outstanding documentation with real-world examples
- Perfect zero-mock compliance
- Seamless integration with execution engine pipeline

This implementation serves as an excellent reference for future transaction cost modeling stories.
