# Story 4.7: Implement Portfolio Allocator for Multi-Strategy Management

## Status
Ready for Review

## Story

**As a** quantitative trader,
**I want** portfolio allocator supporting multiple concurrent strategies,
**so that** I can run diversified strategy portfolios with sophisticated capital allocation.

## Acceptance Criteria

1. PortfolioAllocator class manages multiple Strategy instances concurrently
2. Capital allocation to each strategy tracked and enforced
3. Strategy isolation: each strategy operates on its allocated capital independently
4. Portfolio-level cash management: aggregate cash across strategies
5. Strategy performance tracked individually (per-strategy returns, drawdowns, metrics)
6. Portfolio-level performance aggregated (combined returns, diversification benefit)
7. Rebalancing support: reallocate capital between strategies based on performance
8. Strategy start/stop control (add/remove strategies dynamically during live trading)
9. Tests validate multi-strategy execution with capital allocation enforcement
10. Example demonstrates 3-strategy portfolio (long equity, short equity, market-neutral)

## Story Dependencies

**Depends On:**
- Story 4.1 (Latency Simulation) - Execution timing for synchronized bar processing
- Story 4.2 (Partial Fill Model) - Position tracking framework per strategy
- Story 4.3 (Multiple Slippage Models) - Transaction cost modeling per strategy
- Story 4.4 (Tiered Commission Models) - DecimalLedger per-strategy cost tracking
- Story 4.5 (Borrow Cost Model) - Cost accrual for short strategies
- Story 4.6 (Overnight Financing) - Financing costs for leveraged strategies

**Enables:**
- Story 4.8 (Capital Allocation Algorithms) - Sophisticated allocation across strategies
- Story 4.9 (Risk Manager) - Portfolio-level risk monitoring
- Story 4.10 (Performance Attribution) - Multi-strategy performance analysis
- Epic 5 (Live Trading) - Multi-strategy execution in production

## Tasks / Subtasks

- [x] Design PortfolioAllocator architecture (AC: 1, 2, 3)
  - [x] Create `PortfolioAllocator` class in `rustybt/portfolio/allocator.py` (new module)
  - [x] Create `StrategyAllocation` dataclass with complete fields
  - [x] Create `StrategyPerformance` class for metrics tracking
  - [x] Manage multiple Strategy instances with independent ledgers
  - [x] Track capital allocation per strategy
  - [x] Enforce allocation limits and constraints
  - [x] Document multi-strategy execution flow

- [x] Implement strategy isolation (AC: 3, 4)
  - [x] Each strategy has independent DecimalLedger instance
  - [x] Strategies cannot interfere with each other's positions
  - [x] Aggregate cash management across all strategy ledgers
  - [x] Prevent capital leakage between strategies
  - [x] Track per-strategy vs. portfolio-level cash
  - [x] Validate capital conservation (sum = total)

- [x] Implement synchronized execution (AC: 1)
  - [x] All strategies process same bar simultaneously
  - [x] Event synchronization across strategies
  - [x] Sequential strategy execution with logging
  - [x] Handle strategy execution failures gracefully
  - [x] Track execution order and timing
  - [x] Document execution sequence with diagrams

- [x] Implement performance tracking (AC: 5, 6)
  - [x] Track per-strategy metrics (returns, Sharpe, drawdown)
  - [x] Calculate portfolio-level metrics (aggregated returns)
  - [x] Measure diversification benefit (portfolio Sharpe vs. weighted avg)
  - [x] Track correlation between strategies
  - [x] Historical performance windows
  - [x] Performance attribution per strategy

- [x] Implement rebalancing (AC: 7)
  - [x] Reallocate capital between strategies
  - [x] Transfer capital with cash adjustments
  - [x] Support rebalancing rules and triggers
  - [x] Close positions if capital reduced
  - [x] Log rebalancing events with before/after state
  - [x] Validate capital conservation during rebalancing

- [x] Implement dynamic strategy management (AC: 8)
  - [x] Add strategies during runtime with validation
  - [x] Remove/pause strategies without affecting others
  - [x] Handle strategy lifecycle (init, running, paused, stopped)
  - [x] Liquidate positions when removing strategy
  - [x] Return capital to portfolio on strategy removal
  - [x] Track strategy state transitions

- [x] Write comprehensive tests (AC: 9, 10)
  - [x] Unit test: Add strategy with capital allocation
  - [x] Unit test: Execute bar across all strategies
  - [x] Unit test: Rebalance capital between strategies
  - [x] Unit test: Aggregate metrics calculation
  - [x] Unit test: Remove strategy and return capital
  - [x] Unit test: Strategy isolation verification
  - [x] Unit test: Cash conservation validation
  - [x] Unit test: Execution synchronization
  - [x] Property test: Capital always sums to total
  - [x] Property test: No strategy interference
  - [x] Property test: Portfolio metrics consistency
  - [x] Property test: Rebalancing preserves capital
  - [x] Integration test: 3-strategy portfolio over time
  - [x] Integration test: Dynamic add/remove strategies
  - [x] Example: Long equity + short equity + market-neutral

## Dev Notes

### Previous Story Context

From Story 4.6 (Overnight Financing):
- Complete financing cost framework for leveraged positions
- Position-level cost tracking with accumulated_financing field
- DecimalPosition extensions for cost accumulation
- Integration patterns with DecimalLedger for cash operations

From Story 4.5 (Borrow Cost Model):
- Daily cost accrual patterns for short positions
- Position-level accumulated_borrow_cost tracking
- Rate provider abstraction for extensibility
- Decimal precision for all financial calculations

From Story 4.4 (Tiered Commission Models):
- DecimalLedger integration patterns for cost debits
- Cost tracking and reporting framework
- Structured logging for financial events

From Story 4.3 (Multiple Slippage Models):
- Transaction cost modeling patterns
- Configuration-driven model selection
- Property-based testing for cost invariants

From Story 4.2 (Partial Fill Model):
- Position state tracking with cumulative values
- Order lifecycle management patterns

From Story 4.1 (Latency Simulation):
- Execution timing framework for event processing
- Structured logging with structlog

### Architecture Context

**Source Tree References:** [Source: architecture/source-tree.md]
- Implementation location:
  - `rustybt/portfolio/allocator.py` - Primary portfolio allocator implementation (new module)
  - `rustybt/portfolio/strategy_allocation.py` - Strategy allocation dataclass (new module)
  - `rustybt/portfolio/performance.py` - Performance tracking classes (new module)
  - `rustybt/finance/decimal/ledger.py` - Integration with DecimalLedger
- Test location:
  - `tests/portfolio/test_allocator.py` - Comprehensive allocator tests (new file)
  - `tests/portfolio/test_performance.py` - Performance tracking tests (new file)
- Configuration location:
  - `config/portfolio/` - Portfolio configuration examples (new directory)

**Tech Stack Requirements:** [Source: architecture/tech-stack.md]
- **Python 3.12+**: Use modern type hints and pattern matching
- **Python Decimal**: For precise capital allocation and performance calculations
- **NumPy**: For efficient performance metric calculations (returns, volatility, correlations)
- **Polars**: For performance data storage and analysis
- **structlog**: For structured logging of portfolio events
- **pydantic 2.x+**: For configuration validation

**Component Dependencies:**
- Integrates with `rustybt/finance/decimal/ledger.py` (DecimalLedger per strategy)
- Uses `rustybt/finance/decimal/position.py` (position tracking per strategy)
- Coordinates with `rustybt/algorithm.py` (TradingAlgorithm instances)
- Extends `rustybt/finance/costs.py` (cost models per strategy)

### Key Implementation Requirements

**Portfolio Allocator Architecture:**

```python
from decimal import Decimal
from typing import Dict, Any, Optional, List, Protocol
from dataclasses import dataclass, field
from enum import Enum
from datetime import datetime
import pandas as pd
import numpy as np
import structlog

logger = structlog.get_logger()

class StrategyState(Enum):
    """Strategy lifecycle states."""
    INITIALIZING = "initializing"
    RUNNING = "running"
    PAUSED = "paused"
    LIQUIDATING = "liquidating"
    STOPPED = "stopped"

@dataclass
class StrategyAllocation:
    """Allocation details for a single strategy.

    Each strategy has:
    - Independent ledger for isolated capital
    - Allocated capital amount
    - Performance tracker
    - State management
    """
    strategy_id: str
    strategy: Any  # TradingAlgorithm instance
    allocated_capital: Decimal
    ledger: Any  # DecimalLedger instance
    performance: 'StrategyPerformance'
    state: StrategyState = field(default=StrategyState.INITIALIZING)
    metadata: Dict[str, Any] = field(default_factory=dict)
    created_at: pd.Timestamp = field(default_factory=pd.Timestamp.now)

    @property
    def current_value(self) -> Decimal:
        """Current portfolio value for this strategy."""
        return self.ledger.portfolio_value

    @property
    def return_pct(self) -> Decimal:
        """Return percentage since inception."""
        if self.allocated_capital > Decimal("0"):
            return (self.current_value - self.allocated_capital) / self.allocated_capital
        return Decimal("0")

    @property
    def is_active(self) -> bool:
        """Check if strategy is actively trading."""
        return self.state == StrategyState.RUNNING

    def __str__(self) -> str:
        """String representation for logging."""
        return (
            f"StrategyAllocation(id={self.strategy_id}, "
            f"capital={self.allocated_capital}, "
            f"value={self.current_value}, "
            f"return={self.return_pct:.2%}, "
            f"state={self.state.value})"
        )

class StrategyPerformance:
    """Performance tracker for individual strategy.

    Tracks:
    - Returns over time
    - Volatility (rolling and cumulative)
    - Sharpe ratio
    - Maximum drawdown
    - Win rate and profit factor
    """

    def __init__(
        self,
        strategy_id: str,
        lookback_window: int = 252  # Trading days for rolling metrics
    ):
        """Initialize performance tracker.

        Args:
            strategy_id: Unique strategy identifier
            lookback_window: Number of periods for rolling metrics (252 = ~1 year daily)
        """
        self.strategy_id = strategy_id
        self.lookback_window = lookback_window

        # Time series data
        self.timestamps: List[pd.Timestamp] = []
        self.portfolio_values: List[Decimal] = []
        self.returns: List[Decimal] = []

        # Cumulative metrics
        self.peak_value = Decimal("0")
        self.current_drawdown = Decimal("0")
        self.max_drawdown = Decimal("0")

        # Trade statistics
        self.winning_periods = 0
        self.losing_periods = 0
        self.total_profit = Decimal("0")
        self.total_loss = Decimal("0")

        logger.info(
            "strategy_performance_initialized",
            strategy_id=strategy_id,
            lookback_window=lookback_window
        )

    def update(
        self,
        timestamp: pd.Timestamp,
        portfolio_value: Decimal
    ):
        """Update performance metrics with new observation.

        Args:
            timestamp: Current timestamp
            portfolio_value: Current portfolio value
        """
        # Store values
        self.timestamps.append(timestamp)
        self.portfolio_values.append(portfolio_value)

        # Calculate return (if we have previous value)
        if len(self.portfolio_values) > 1:
            prev_value = self.portfolio_values[-2]
            if prev_value > Decimal("0"):
                period_return = (portfolio_value - prev_value) / prev_value
                self.returns.append(period_return)

                # Update trade statistics
                if period_return > Decimal("0"):
                    self.winning_periods += 1
                    self.total_profit += period_return
                elif period_return < Decimal("0"):
                    self.losing_periods += 1
                    self.total_loss += abs(period_return)

        # Update drawdown metrics
        if portfolio_value > self.peak_value:
            self.peak_value = portfolio_value
            self.current_drawdown = Decimal("0")
        else:
            if self.peak_value > Decimal("0"):
                self.current_drawdown = (portfolio_value - self.peak_value) / self.peak_value

                # Update max drawdown (more negative = worse)
                if self.current_drawdown < self.max_drawdown:
                    self.max_drawdown = self.current_drawdown

        logger.debug(
            "performance_updated",
            strategy_id=self.strategy_id,
            timestamp=str(timestamp),
            portfolio_value=str(portfolio_value),
            current_drawdown=f"{float(self.current_drawdown):.2%}",
            max_drawdown=f"{float(self.max_drawdown):.2%}"
        )

    @property
    def volatility(self) -> Decimal:
        """Calculate annualized volatility.

        Returns:
            Annualized volatility (standard deviation of returns × sqrt(252))
        """
        if len(self.returns) < 2:
            return Decimal("0")

        # Use recent returns (lookback window)
        recent_returns = self.returns[-self.lookback_window:]

        # Convert to numpy for efficient calculation
        returns_array = np.array([float(r) for r in recent_returns])

        # Calculate standard deviation
        std = np.std(returns_array, ddof=1)  # Sample std deviation

        # Annualize (assume daily data, 252 trading days)
        annualized_std = std * np.sqrt(252)

        return Decimal(str(annualized_std))

    @property
    def mean_return(self) -> Decimal:
        """Calculate mean period return (annualized).

        Returns:
            Annualized mean return
        """
        if len(self.returns) < 1:
            return Decimal("0")

        # Use recent returns
        recent_returns = self.returns[-self.lookback_window:]

        # Calculate mean
        returns_array = np.array([float(r) for r in recent_returns])
        mean = np.mean(returns_array)

        # Annualize (assume daily data, 252 trading days)
        annualized_mean = mean * Decimal("252")

        return Decimal(str(annualized_mean))

    @property
    def sharpe_ratio(self) -> Decimal:
        """Calculate Sharpe ratio.

        Formula: (mean_return - risk_free_rate) / volatility
        Assumes risk-free rate = 0 for simplicity

        Returns:
            Sharpe ratio (annualized)
        """
        vol = self.volatility
        if vol <= Decimal("0"):
            return Decimal("0")

        mean_ret = self.mean_return

        # Sharpe = mean / std (assuming risk-free = 0)
        sharpe = mean_ret / vol

        return sharpe

    @property
    def win_rate(self) -> Decimal:
        """Calculate win rate (% of winning periods).

        Returns:
            Win rate as decimal (0.6 = 60%)
        """
        total_periods = self.winning_periods + self.losing_periods
        if total_periods == 0:
            return Decimal("0")

        return Decimal(str(self.winning_periods)) / Decimal(str(total_periods))

    @property
    def profit_factor(self) -> Decimal:
        """Calculate profit factor (total profit / total loss).

        Returns:
            Profit factor (>1 = profitable, <1 = unprofitable)
        """
        if self.total_loss == Decimal("0"):
            return Decimal("0") if self.total_profit == Decimal("0") else Decimal("999")

        return self.total_profit / self.total_loss

    def get_metrics(self) -> Dict[str, Any]:
        """Get all performance metrics as dictionary.

        Returns:
            Dictionary with all performance metrics
        """
        return {
            "strategy_id": self.strategy_id,
            "num_observations": len(self.portfolio_values),
            "current_value": str(self.portfolio_values[-1]) if self.portfolio_values else "0",
            "peak_value": str(self.peak_value),
            "current_drawdown": f"{float(self.current_drawdown):.2%}",
            "max_drawdown": f"{float(self.max_drawdown):.2%}",
            "mean_return": f"{float(self.mean_return):.2%}",
            "volatility": f"{float(self.volatility):.2%}",
            "sharpe_ratio": f"{float(self.sharpe_ratio):.2f}",
            "win_rate": f"{float(self.win_rate):.2%}",
            "profit_factor": f"{float(self.profit_factor):.2f}",
            "winning_periods": self.winning_periods,
            "losing_periods": self.losing_periods
        }

class PortfolioAllocator:
    """Portfolio allocator for multi-strategy management.

    Multi-Strategy Execution Flow:
    ==============================

    1. Initialization:
       - Create PortfolioAllocator with total capital
       - Add strategies with allocation percentages
       - Each strategy gets independent DecimalLedger
       - Validate: sum(allocations) <= 100%

    2. Bar-by-Bar Execution (Synchronized):
       - For each bar (timestamp, data):
         a. Update all strategy ledgers with latest prices
         b. Execute all strategies sequentially:
            - Strategy 1: handle_data(ledger_1, data)
            - Strategy 2: handle_data(ledger_2, data)
            - Strategy 3: handle_data(ledger_3, data)
         c. Update performance metrics for each strategy
         d. Aggregate portfolio-level metrics
         e. Log execution summary

    3. Rebalancing (Periodic or Event-Driven):
       - Calculate new allocations (manual or algorithmic)
       - For each strategy with allocation change:
         a. Calculate capital delta (new - old)
         b. If delta > 0: transfer cash to strategy ledger
         c. If delta < 0: reduce positions and return cash
       - Validate: total capital conserved
       - Log rebalancing events

    4. Strategy Management:
       - Add: Create ledger, allocate capital, set to RUNNING
       - Pause: Set state to PAUSED, skip execution
       - Remove: Liquidate positions, return capital, set to STOPPED

    Strategy Isolation Mechanism:
    ============================
    - Each strategy has separate DecimalLedger instance
    - Strategies cannot access other ledgers' positions
    - Cash transfers only through PortfolioAllocator
    - Position interference prevented by ledger isolation

    Cash Aggregation:
    ================
    - Portfolio cash = sum(strategy.ledger.cash for all strategies)
    - Portfolio value = sum(strategy.ledger.portfolio_value for all strategies)
    - Validates: portfolio_value == total_capital + cumulative_returns
    """

    def __init__(
        self,
        total_capital: Decimal,
        name: str = "Portfolio"
    ):
        """Initialize portfolio allocator.

        Args:
            total_capital: Total capital to allocate across strategies
            name: Portfolio name for logging
        """
        self.total_capital = total_capital
        self.name = name
        self.strategies: Dict[str, StrategyAllocation] = {}

        # Track allocated capital to prevent over-allocation
        self.allocated_capital = Decimal("0")

        # Portfolio-level tracking
        self.current_timestamp: Optional[pd.Timestamp] = None
        self.execution_count = 0

        logger.info(
            "portfolio_allocator_initialized",
            name=name,
            total_capital=str(total_capital)
        )

    def add_strategy(
        self,
        strategy_id: str,
        strategy: Any,  # TradingAlgorithm
        allocation_pct: Decimal,
        metadata: Optional[Dict[str, Any]] = None
    ) -> StrategyAllocation:
        """Add strategy to portfolio with capital allocation.

        Args:
            strategy_id: Unique identifier for strategy
            strategy: TradingAlgorithm instance
            allocation_pct: Allocation percentage (0.3 = 30%)
            metadata: Optional metadata for strategy

        Returns:
            StrategyAllocation instance

        Raises:
            ValueError: If allocation would exceed 100% or strategy_id exists
        """
        # Validation
        if strategy_id in self.strategies:
            raise ValueError(f"Strategy {strategy_id} already exists")

        # Check allocation doesn't exceed 100%
        new_allocated_pct = (self.allocated_capital / self.total_capital) + allocation_pct
        if new_allocated_pct > Decimal("1"):
            raise ValueError(
                f"Allocation would exceed 100%: "
                f"current={float(self.allocated_capital / self.total_capital):.1%}, "
                f"new={float(allocation_pct):.1%}, "
                f"total={float(new_allocated_pct):.1%}"
            )

        # Calculate capital amount
        allocated_capital = self.total_capital * allocation_pct

        # Create independent ledger for strategy
        from rustybt.finance.decimal.ledger import DecimalLedger
        ledger = DecimalLedger(starting_cash=allocated_capital)

        # Create performance tracker
        performance = StrategyPerformance(strategy_id)

        # Create allocation
        allocation = StrategyAllocation(
            strategy_id=strategy_id,
            strategy=strategy,
            allocated_capital=allocated_capital,
            ledger=ledger,
            performance=performance,
            state=StrategyState.RUNNING,
            metadata=metadata or {}
        )

        # Store allocation
        self.strategies[strategy_id] = allocation
        self.allocated_capital += allocated_capital

        logger.info(
            "strategy_added",
            portfolio=self.name,
            strategy_id=strategy_id,
            allocation_pct=f"{float(allocation_pct):.1%}",
            allocated_capital=str(allocated_capital),
            total_allocated=str(self.allocated_capital),
            remaining_capital=str(self.total_capital - self.allocated_capital)
        )

        return allocation

    def remove_strategy(
        self,
        strategy_id: str,
        liquidate: bool = True
    ) -> Decimal:
        """Remove strategy from portfolio.

        Args:
            strategy_id: Strategy to remove
            liquidate: If True, liquidate all positions before removing

        Returns:
            Capital returned to portfolio

        Raises:
            KeyError: If strategy_id not found
        """
        if strategy_id not in self.strategies:
            raise KeyError(f"Strategy {strategy_id} not found")

        allocation = self.strategies[strategy_id]

        # Change state to liquidating
        allocation.state = StrategyState.LIQUIDATING

        if liquidate:
            # Liquidate all positions (simplified - just close everything)
            # In real implementation, would iterate positions and close
            logger.info(
                "liquidating_strategy_positions",
                strategy_id=strategy_id,
                num_positions=len(allocation.ledger.positions)
            )
            # allocation.ledger.liquidate_all_positions()  # Would call this

        # Get final value
        final_value = allocation.ledger.portfolio_value

        # Update allocated capital
        self.allocated_capital -= allocation.allocated_capital

        # Mark as stopped
        allocation.state = StrategyState.STOPPED

        # Remove from active strategies
        del self.strategies[strategy_id]

        logger.info(
            "strategy_removed",
            portfolio=self.name,
            strategy_id=strategy_id,
            initial_capital=str(allocation.allocated_capital),
            final_value=str(final_value),
            return_pct=f"{float(allocation.return_pct):.2%}",
            capital_returned=str(final_value)
        )

        return final_value

    def pause_strategy(self, strategy_id: str):
        """Pause strategy execution (keeps positions, stops trading).

        Args:
            strategy_id: Strategy to pause
        """
        if strategy_id not in self.strategies:
            raise KeyError(f"Strategy {strategy_id} not found")

        allocation = self.strategies[strategy_id]
        allocation.state = StrategyState.PAUSED

        logger.info(
            "strategy_paused",
            portfolio=self.name,
            strategy_id=strategy_id
        )

    def resume_strategy(self, strategy_id: str):
        """Resume paused strategy.

        Args:
            strategy_id: Strategy to resume
        """
        if strategy_id not in self.strategies:
            raise KeyError(f"Strategy {strategy_id} not found")

        allocation = self.strategies[strategy_id]
        allocation.state = StrategyState.RUNNING

        logger.info(
            "strategy_resumed",
            portfolio=self.name,
            strategy_id=strategy_id
        )

    def execute_bar(
        self,
        timestamp: pd.Timestamp,
        data: Dict[str, Any]
    ):
        """Execute all active strategies for current bar (synchronized).

        All strategies process the same bar simultaneously (sequentially in code,
        but conceptually at the same timestamp).

        Args:
            timestamp: Current bar timestamp
            data: Market data for all assets
        """
        self.current_timestamp = timestamp
        self.execution_count += 1

        logger.info(
            "executing_bar",
            portfolio=self.name,
            timestamp=str(timestamp),
            num_strategies=len(self.strategies),
            execution_count=self.execution_count
        )

        # Execute each strategy
        for strategy_id, allocation in self.strategies.items():
            # Skip if not running
            if not allocation.is_active:
                logger.debug(
                    "skipping_inactive_strategy",
                    strategy_id=strategy_id,
                    state=allocation.state.value
                )
                continue

            try:
                # Execute strategy with its isolated ledger
                # Strategy can only modify its own ledger
                allocation.strategy.handle_data(allocation.ledger, data)

                # Update performance metrics
                portfolio_value = allocation.ledger.portfolio_value
                allocation.performance.update(timestamp, portfolio_value)

                logger.debug(
                    "strategy_executed",
                    portfolio=self.name,
                    strategy_id=strategy_id,
                    portfolio_value=str(portfolio_value),
                    return_pct=f"{float(allocation.return_pct):.2%}",
                    num_positions=len(allocation.ledger.positions)
                )

            except Exception as e:
                logger.error(
                    "strategy_execution_failed",
                    portfolio=self.name,
                    strategy_id=strategy_id,
                    error=str(e),
                    exc_info=True
                )
                # Optionally pause failed strategy
                # allocation.state = StrategyState.PAUSED

        # Log portfolio summary
        portfolio_metrics = self.get_portfolio_metrics()
        logger.info(
            "bar_execution_complete",
            portfolio=self.name,
            timestamp=str(timestamp),
            **portfolio_metrics
        )

    def rebalance(
        self,
        new_allocations: Dict[str, Decimal],
        reason: str = "Manual rebalancing"
    ):
        """Rebalance capital between strategies.

        Capital Transfer Logic:
        - If new_allocation > old_allocation: transfer cash to strategy
        - If new_allocation < old_allocation: reduce positions, return cash

        Args:
            new_allocations: Dict of {strategy_id: new_allocation_pct}
            reason: Reason for rebalancing (for logging)

        Raises:
            ValueError: If allocations don't sum to valid amount or strategy not found
        """
        # Validate allocations
        total_allocation = sum(new_allocations.values())
        if total_allocation > Decimal("1"):
            raise ValueError(
                f"New allocations exceed 100%: {float(total_allocation):.1%}"
            )

        logger.info(
            "rebalancing_portfolio",
            portfolio=self.name,
            reason=reason,
            num_strategies=len(new_allocations),
            total_allocation=f"{float(total_allocation):.1%}"
        )

        # Track capital movements
        capital_changes: Dict[str, Decimal] = {}

        for strategy_id, new_pct in new_allocations.items():
            if strategy_id not in self.strategies:
                raise ValueError(f"Strategy {strategy_id} not found")

            allocation = self.strategies[strategy_id]

            # Calculate new capital amount
            new_capital = self.total_capital * new_pct
            old_capital = allocation.allocated_capital
            capital_delta = new_capital - old_capital

            # Update allocated capital
            allocation.allocated_capital = new_capital

            # Transfer cash
            # Positive delta: add cash to strategy
            # Negative delta: remove cash from strategy (may require liquidating positions)
            allocation.ledger.cash += capital_delta

            capital_changes[strategy_id] = capital_delta

            logger.info(
                "strategy_rebalanced",
                portfolio=self.name,
                strategy_id=strategy_id,
                old_allocation=f"{float(old_capital / self.total_capital):.1%}",
                new_allocation=f"{float(new_pct):.1%}",
                old_capital=str(old_capital),
                new_capital=str(new_capital),
                capital_delta=str(capital_delta),
                new_cash=str(allocation.ledger.cash)
            )

        # Validate capital conservation
        total_capital_after = sum(
            alloc.allocated_capital
            for alloc in self.strategies.values()
        )

        if abs(total_capital_after - self.allocated_capital) > Decimal("0.01"):
            logger.error(
                "capital_conservation_violated",
                portfolio=self.name,
                expected=str(self.allocated_capital),
                actual=str(total_capital_after),
                difference=str(total_capital_after - self.allocated_capital)
            )

        # Update allocated capital
        self.allocated_capital = total_capital_after

        logger.info(
            "rebalancing_complete",
            portfolio=self.name,
            total_allocated=str(self.allocated_capital),
            capital_changes={k: str(v) for k, v in capital_changes.items()}
        )

    def get_portfolio_metrics(self) -> Dict[str, Any]:
        """Calculate portfolio-level performance metrics.

        Aggregates across all strategies to compute:
        - Total portfolio value
        - Portfolio return
        - Diversification benefit

        Returns:
            Dictionary with portfolio metrics
        """
        # Aggregate portfolio value
        total_value = sum(
            alloc.ledger.portfolio_value
            for alloc in self.strategies.values()
        )

        # Calculate total cash
        total_cash = sum(
            alloc.ledger.cash
            for alloc in self.strategies.values()
        )

        # Calculate portfolio return
        if self.total_capital > Decimal("0"):
            portfolio_return = (total_value - self.total_capital) / self.total_capital
        else:
            portfolio_return = Decimal("0")

        # Calculate weighted average Sharpe ratio
        total_sharpe_weighted = Decimal("0")
        for alloc in self.strategies.values():
            weight = alloc.allocated_capital / self.allocated_capital if self.allocated_capital > 0 else Decimal("0")
            total_sharpe_weighted += weight * alloc.performance.sharpe_ratio

        # Calculate diversification benefit (simplified)
        # Diversification benefit = portfolio Sharpe - weighted avg Sharpe
        # Would need portfolio-level returns history for true calculation

        return {
            "total_value": str(total_value),
            "total_cash": str(total_cash),
            "total_capital": str(self.total_capital),
            "allocated_capital": str(self.allocated_capital),
            "unallocated_capital": str(self.total_capital - self.allocated_capital),
            "portfolio_return": f"{float(portfolio_return):.2%}",
            "num_strategies": len(self.strategies),
            "active_strategies": sum(1 for a in self.strategies.values() if a.is_active),
            "weighted_avg_sharpe": f"{float(total_sharpe_weighted):.2f}"
        }

    def get_strategy_metrics(self) -> Dict[str, Dict[str, Any]]:
        """Get metrics for all strategies.

        Returns:
            Dict mapping strategy_id to metrics dict
        """
        return {
            strategy_id: {
                **allocation.performance.get_metrics(),
                "allocated_capital": str(allocation.allocated_capital),
                "current_value": str(allocation.current_value),
                "return_pct": f"{float(allocation.return_pct):.2%}",
                "state": allocation.state.value
            }
            for strategy_id, allocation in self.strategies.items()
        }

    def get_correlation_matrix(self) -> Optional[pd.DataFrame]:
        """Calculate correlation matrix between strategies.

        Returns:
            DataFrame with correlation matrix or None if insufficient data
        """
        # Need at least 2 strategies with returns
        strategy_returns = {}

        for strategy_id, allocation in self.strategies.items():
            if len(allocation.performance.returns) > 0:
                strategy_returns[strategy_id] = allocation.performance.returns

        if len(strategy_returns) < 2:
            return None

        # Find minimum length
        min_length = min(len(returns) for returns in strategy_returns.values())

        if min_length < 2:
            return None

        # Create DataFrame with aligned returns
        returns_dict = {
            strategy_id: [float(r) for r in returns[-min_length:]]
            for strategy_id, returns in strategy_returns.items()
        }

        returns_df = pd.DataFrame(returns_dict)

        # Calculate correlation matrix
        corr_matrix = returns_df.corr()

        return corr_matrix


# Example: 3-Strategy Portfolio
class LongEquityStrategy:
    """Example long-only equity strategy."""

    def handle_data(self, ledger, data):
        """Buy and hold large cap stocks."""
        # Simplified example implementation
        pass

class ShortEquityStrategy:
    """Example short-only equity strategy."""

    def handle_data(self, ledger, data):
        """Short overvalued stocks."""
        # Simplified example implementation
        pass

class MarketNeutralStrategy:
    """Example market-neutral equity strategy."""

    def handle_data(self, ledger, data):
        """Long undervalued, short overvalued (beta-neutral)."""
        # Simplified example implementation
        pass

def create_diversified_portfolio_example():
    """Example: Create 3-strategy diversified portfolio.

    Strategies:
    1. Long Equity (40%) - Buy and hold quality stocks
    2. Short Equity (30%) - Short overvalued stocks
    3. Market Neutral (30%) - Long/short beta-neutral

    Returns:
        Configured PortfolioAllocator instance
    """
    # Initialize portfolio with $1M
    portfolio = PortfolioAllocator(
        total_capital=Decimal("1000000.00"),
        name="Diversified Equity Portfolio"
    )

    # Add long equity strategy (40%)
    long_strategy = LongEquityStrategy()
    portfolio.add_strategy(
        strategy_id="long_equity",
        strategy=long_strategy,
        allocation_pct=Decimal("0.40"),
        metadata={"description": "Long-only large cap growth"}
    )

    # Add short equity strategy (30%)
    short_strategy = ShortEquityStrategy()
    portfolio.add_strategy(
        strategy_id="short_equity",
        strategy=short_strategy,
        allocation_pct=Decimal("0.30"),
        metadata={"description": "Short overvalued momentum stocks"}
    )

    # Add market neutral strategy (30%)
    neutral_strategy = MarketNeutralStrategy()
    portfolio.add_strategy(
        strategy_id="market_neutral",
        strategy=neutral_strategy,
        allocation_pct=Decimal("0.30"),
        metadata={"description": "Beta-neutral long/short value"}
    )

    logger.info(
        "diversified_portfolio_created",
        total_capital="$1,000,000",
        strategies=["long_equity (40%)", "short_equity (30%)", "market_neutral (30%)"],
        allocated_capital=str(portfolio.allocated_capital),
        unallocated=str(portfolio.total_capital - portfolio.allocated_capital)
    )

    return portfolio
```

### Coding Standards

**Type Hints:** [Source: architecture/coding-standards.md#python-coding-standards]
- 100% type hint coverage for public APIs
- Use `Decimal` for all financial calculations (capital, returns, allocations)
- Use `Protocol` for extensible interfaces (strategy interface)
- Use `Enum` for strategy states

**Docstrings:** [Source: architecture/coding-standards.md#python-coding-standards]
- Google-style docstrings for all public classes and methods
- Include execution flow diagrams in class docstrings
- Document capital transfer mechanics
- Provide examples for common use cases

**Error Handling:** [Source: architecture/coding-standards.md#error-handling]
- Validate allocation constraints (sum <= 100%)
- Handle strategy execution failures gracefully
- Validate capital conservation during rebalancing
- Log all state transitions

```python
# Validation example
if strategy_id in self.strategies:
    raise ValueError(f"Strategy {strategy_id} already exists")

# Check allocation doesn't exceed 100%
new_allocated_pct = (self.allocated_capital / self.total_capital) + allocation_pct
if new_allocated_pct > Decimal("1"):
    raise ValueError(
        f"Allocation would exceed 100%: "
        f"current={float(self.allocated_capital / self.total_capital):.1%}, "
        f"new={float(allocation_pct):.1%}"
    )

# Capital conservation check
if abs(total_capital_after - self.allocated_capital) > Decimal("0.01"):
    logger.error(
        "capital_conservation_violated",
        expected=str(self.allocated_capital),
        actual=str(total_capital_after)
    )
```

**Logging:** [Source: architecture/coding-standards.md#logging]
```python
import structlog
logger = structlog.get_logger()

logger.info(
    "strategy_added",
    portfolio=self.name,
    strategy_id=strategy_id,
    allocation_pct=f"{float(allocation_pct):.1%}",
    allocated_capital=str(allocated_capital)
)

logger.info(
    "bar_execution_complete",
    portfolio=self.name,
    timestamp=str(timestamp),
    total_value=str(total_value),
    portfolio_return=f"{float(portfolio_return):.2%}"
)
```

**Zero-Mock Enforcement:** [Source: architecture/coding-standards.md#zero-mock-enforcement-mandatory]
- NO hardcoded allocations in production code (use configuration)
- All capital calculations must use real Decimal arithmetic
- All validation functions must perform real checks
- Tests must use real PortfolioAllocator instances with real ledgers

### Testing

#### Testing Standards [Source: architecture/testing-strategy.md]

**Test File Location:**
- `tests/portfolio/test_allocator.py` - Comprehensive allocator tests (new file)
- `tests/portfolio/test_performance.py` - Performance tracking tests (new file)

**Test Frameworks:**
- **pytest >= 7.2.0**: Primary test framework
- **pytest-cov >= 3.0.0**: Coverage reporting (target: ≥90%)
- **hypothesis >= 6.x+**: Property-based testing

**Unit Tests:**

```python
import pytest
from decimal import Decimal
import pandas as pd
from rustybt.portfolio.allocator import (
    PortfolioAllocator,
    StrategyAllocation,
    StrategyPerformance,
    StrategyState
)

def test_add_strategy_with_capital_allocation():
    """Add strategy with capital allocation."""
    portfolio = PortfolioAllocator(total_capital=Decimal("100000.00"))

    # Create mock strategy
    strategy = MockStrategy()

    # Add strategy with 30% allocation
    allocation = portfolio.add_strategy(
        strategy_id="test_strategy",
        strategy=strategy,
        allocation_pct=Decimal("0.30")
    )

    # Verify allocation
    assert allocation.strategy_id == "test_strategy"
    assert allocation.allocated_capital == Decimal("30000.00")
    assert allocation.state == StrategyState.RUNNING
    assert allocation.ledger.cash == Decimal("30000.00")

    # Verify portfolio state
    assert portfolio.allocated_capital == Decimal("30000.00")
    assert len(portfolio.strategies) == 1

def test_add_strategy_exceeding_100_percent():
    """Adding strategy that exceeds 100% allocation raises error."""
    portfolio = PortfolioAllocator(total_capital=Decimal("100000.00"))

    # Add first strategy (70%)
    portfolio.add_strategy(
        strategy_id="strategy1",
        strategy=MockStrategy(),
        allocation_pct=Decimal("0.70")
    )

    # Try to add second strategy (50%) - should fail
    with pytest.raises(ValueError, match="exceed 100%"):
        portfolio.add_strategy(
            strategy_id="strategy2",
            strategy=MockStrategy(),
            allocation_pct=Decimal("0.50")
        )

def test_execute_bar_across_all_strategies():
    """Execute bar processes all active strategies."""
    portfolio = PortfolioAllocator(total_capital=Decimal("100000.00"))

    # Add multiple strategies
    strategy1 = MockStrategy()
    strategy2 = MockStrategy()

    portfolio.add_strategy("strategy1", strategy1, Decimal("0.50"))
    portfolio.add_strategy("strategy2", strategy2, Decimal("0.30"))

    # Execute bar
    timestamp = pd.Timestamp("2023-01-01")
    data = {"AAPL": {"price": Decimal("150.00")}}

    portfolio.execute_bar(timestamp, data)

    # Verify both strategies executed
    assert strategy1.handle_data_called
    assert strategy2.handle_data_called

    # Verify performance updated
    alloc1 = portfolio.strategies["strategy1"]
    alloc2 = portfolio.strategies["strategy2"]

    assert len(alloc1.performance.timestamps) == 1
    assert len(alloc2.performance.timestamps) == 1

def test_rebalance_capital_between_strategies():
    """Rebalance transfers capital between strategies."""
    portfolio = PortfolioAllocator(total_capital=Decimal("100000.00"))

    # Add two strategies
    portfolio.add_strategy("strategy1", MockStrategy(), Decimal("0.50"))
    portfolio.add_strategy("strategy2", MockStrategy(), Decimal("0.30"))

    # Initial state
    alloc1 = portfolio.strategies["strategy1"]
    alloc2 = portfolio.strategies["strategy2"]

    assert alloc1.allocated_capital == Decimal("50000.00")
    assert alloc2.allocated_capital == Decimal("30000.00")

    # Rebalance: strategy1 40%, strategy2 40%
    portfolio.rebalance({
        "strategy1": Decimal("0.40"),
        "strategy2": Decimal("0.40")
    })

    # Verify new allocations
    assert alloc1.allocated_capital == Decimal("40000.00")
    assert alloc2.allocated_capital == Decimal("40000.00")

    # Verify cash transferred
    assert alloc1.ledger.cash == Decimal("40000.00")  # Reduced from 50k
    assert alloc2.ledger.cash == Decimal("40000.00")  # Increased from 30k

def test_aggregate_portfolio_metrics():
    """Aggregate metrics calculation across strategies."""
    portfolio = PortfolioAllocator(total_capital=Decimal("100000.00"))

    # Add strategies
    portfolio.add_strategy("strategy1", MockStrategy(), Decimal("0.50"))
    portfolio.add_strategy("strategy2", MockStrategy(), Decimal("0.30"))

    # Simulate some returns
    alloc1 = portfolio.strategies["strategy1"]
    alloc2 = portfolio.strategies["strategy2"]

    # Update values (10% gain for strategy1, -5% for strategy2)
    alloc1.ledger._portfolio_value = Decimal("55000.00")
    alloc2.ledger._portfolio_value = Decimal("28500.00")

    # Get metrics
    metrics = portfolio.get_portfolio_metrics()

    # Total value = 55k + 28.5k = 83.5k
    assert Decimal(metrics["total_value"]) == Decimal("83500.00")

    # Return = (83.5k - 80k) / 80k = 4.375%
    # (initial allocated was 80k total)
    portfolio_return = (Decimal("83500.00") - Decimal("80000.00")) / Decimal("80000.00")
    assert "4.3" in metrics["portfolio_return"] or "4.4" in metrics["portfolio_return"]

def test_remove_strategy_and_return_capital():
    """Remove strategy returns capital to portfolio."""
    portfolio = PortfolioAllocator(total_capital=Decimal("100000.00"))

    # Add strategy
    portfolio.add_strategy("test_strategy", MockStrategy(), Decimal("0.30"))

    assert portfolio.allocated_capital == Decimal("30000.00")

    # Remove strategy
    returned_capital = portfolio.remove_strategy("test_strategy", liquidate=True)

    # Verify removal
    assert "test_strategy" not in portfolio.strategies
    assert portfolio.allocated_capital == Decimal("0")

    # Capital returned should equal allocated (no P&L in this test)
    assert returned_capital == Decimal("30000.00")

def test_strategy_isolation_verification():
    """Verify strategies cannot interfere with each other."""
    portfolio = PortfolioAllocator(total_capital=Decimal("100000.00"))

    # Add two strategies
    portfolio.add_strategy("strategy1", MockStrategy(), Decimal("0.50"))
    portfolio.add_strategy("strategy2", MockStrategy(), Decimal("0.30"))

    alloc1 = portfolio.strategies["strategy1"]
    alloc2 = portfolio.strategies["strategy2"]

    # Verify separate ledgers
    assert alloc1.ledger is not alloc2.ledger

    # Modify strategy1's ledger
    alloc1.ledger.cash = Decimal("25000.00")

    # Verify strategy2's ledger unchanged
    assert alloc2.ledger.cash == Decimal("30000.00")

    # Add position to strategy1
    alloc1.ledger.positions = {"AAPL": MockPosition()}

    # Verify strategy2 has no positions
    assert len(alloc2.ledger.positions) == 0

def test_cash_conservation_validation():
    """Validate total cash is conserved across strategies."""
    portfolio = PortfolioAllocator(total_capital=Decimal("100000.00"))

    # Add strategies
    portfolio.add_strategy("strategy1", MockStrategy(), Decimal("0.40"))
    portfolio.add_strategy("strategy2", MockStrategy(), Decimal("0.30"))
    portfolio.add_strategy("strategy3", MockStrategy(), Decimal("0.20"))

    # Total allocated cash should equal sum of strategy cash
    total_strategy_cash = sum(
        alloc.ledger.cash
        for alloc in portfolio.strategies.values()
    )

    assert total_strategy_cash == portfolio.allocated_capital
    assert portfolio.allocated_capital == Decimal("90000.00")

def test_execution_synchronization():
    """All strategies process same bar at same timestamp."""
    portfolio = PortfolioAllocator(total_capital=Decimal("100000.00"))

    # Add strategies with timestamp tracking
    strategy1 = TimestampTrackingStrategy()
    strategy2 = TimestampTrackingStrategy()

    portfolio.add_strategy("strategy1", strategy1, Decimal("0.50"))
    portfolio.add_strategy("strategy2", strategy2, Decimal("0.30"))

    # Execute bar
    timestamp = pd.Timestamp("2023-01-01 09:30:00")
    data = {}

    portfolio.execute_bar(timestamp, data)

    # Verify both strategies saw same timestamp
    assert strategy1.last_timestamp == timestamp
    assert strategy2.last_timestamp == timestamp
```

**Property-Based Tests:**

```python
from hypothesis import given, strategies as st

@given(
    num_strategies=st.integers(min_value=1, max_value=10),
    total_capital=st.decimals(min_value=Decimal("1000"), max_value=Decimal("10000000"))
)
def test_capital_always_sums_to_total(num_strategies, total_capital):
    """Property: Sum of strategy allocations always equals total allocated capital."""
    portfolio = PortfolioAllocator(total_capital=total_capital)

    # Add strategies with random allocations (ensuring they sum < 100%)
    allocation_pct = Decimal("0.80") / Decimal(str(num_strategies))

    for i in range(num_strategies):
        portfolio.add_strategy(
            strategy_id=f"strategy_{i}",
            strategy=MockStrategy(),
            allocation_pct=allocation_pct
        )

    # Property: Sum of allocated capital equals total allocated
    total_allocated = sum(
        alloc.allocated_capital
        for alloc in portfolio.strategies.values()
    )

    assert total_allocated == portfolio.allocated_capital

    # Property: Total allocated <= total capital
    assert portfolio.allocated_capital <= total_capital

@given(
    initial_alloc=st.decimals(min_value=Decimal("0.1"), max_value=Decimal("0.5")),
    new_alloc=st.decimals(min_value=Decimal("0.1"), max_value=Decimal("0.5"))
)
def test_rebalancing_preserves_capital(initial_alloc, new_alloc):
    """Property: Rebalancing preserves total capital."""
    total_capital = Decimal("100000.00")
    portfolio = PortfolioAllocator(total_capital=total_capital)

    # Add strategy
    portfolio.add_strategy("test", MockStrategy(), initial_alloc)

    initial_total = portfolio.allocated_capital

    # Rebalance
    portfolio.rebalance({"test": new_alloc})

    # Property: Total allocated capital changes but is still valid
    assert portfolio.allocated_capital == total_capital * new_alloc

    # Property: Ledger cash matches allocated capital
    alloc = portfolio.strategies["test"]
    assert alloc.ledger.cash == alloc.allocated_capital

@given(
    num_strategies=st.integers(min_value=2, max_value=5)
)
def test_no_strategy_interference(num_strategies):
    """Property: Strategies cannot interfere with each other's state."""
    portfolio = PortfolioAllocator(total_capital=Decimal("100000.00"))

    # Add multiple strategies
    allocation_pct = Decimal("0.15")
    for i in range(num_strategies):
        portfolio.add_strategy(
            f"strategy_{i}",
            MockStrategy(),
            allocation_pct
        )

    # Modify one strategy's ledger
    first_strategy = list(portfolio.strategies.values())[0]
    first_strategy.ledger.cash = Decimal("999.99")

    # Property: Other strategies unchanged
    for i, (sid, alloc) in enumerate(portfolio.strategies.items()):
        if i == 0:
            assert alloc.ledger.cash == Decimal("999.99")
        else:
            expected_cash = portfolio.total_capital * allocation_pct
            assert alloc.ledger.cash == expected_cash

@given(
    num_periods=st.integers(min_value=10, max_value=100)
)
def test_portfolio_metrics_consistency(num_periods):
    """Property: Portfolio metrics are consistent with strategy metrics."""
    portfolio = PortfolioAllocator(total_capital=Decimal("100000.00"))

    portfolio.add_strategy("strategy1", MockStrategy(), Decimal("0.50"))
    portfolio.add_strategy("strategy2", MockStrategy(), Decimal("0.30"))

    # Simulate multiple periods
    for i in range(num_periods):
        timestamp = pd.Timestamp("2023-01-01") + pd.Timedelta(days=i)
        portfolio.execute_bar(timestamp, {})

    # Property: Portfolio value = sum of strategy values
    portfolio_metrics = portfolio.get_portfolio_metrics()
    total_value = sum(
        alloc.ledger.portfolio_value
        for alloc in portfolio.strategies.values()
    )

    assert Decimal(portfolio_metrics["total_value"]) == total_value
```

**Integration Tests:**

```python
def test_three_strategy_portfolio_over_time():
    """Integration test: 3-strategy portfolio over 100 days."""
    # Create portfolio
    portfolio = create_diversified_portfolio_example()

    # Verify initial state
    assert len(portfolio.strategies) == 3
    assert portfolio.allocated_capital == Decimal("1000000.00")

    # Simulate 100 trading days
    start_date = pd.Timestamp("2023-01-01")

    for day in range(100):
        timestamp = start_date + pd.Timedelta(days=day)

        # Mock market data
        data = {
            "AAPL": {"price": Decimal("150.00") + Decimal(str(day * 0.5))},
            "GOOGL": {"price": Decimal("100.00") + Decimal(str(day * 0.3))}
        }

        # Execute bar
        portfolio.execute_bar(timestamp, data)

    # Verify all strategies executed
    for strategy_id, allocation in portfolio.strategies.items():
        assert len(allocation.performance.timestamps) == 100
        assert len(allocation.performance.portfolio_values) == 100

    # Get final metrics
    portfolio_metrics = portfolio.get_portfolio_metrics()
    strategy_metrics = portfolio.get_strategy_metrics()

    # Verify metrics computed
    assert "total_value" in portfolio_metrics
    assert len(strategy_metrics) == 3

    # Each strategy should have performance data
    for strategy_id, metrics in strategy_metrics.items():
        assert "sharpe_ratio" in metrics
        assert "max_drawdown" in metrics
        assert "return_pct" in metrics

def test_dynamic_add_remove_strategies():
    """Integration test: Dynamically add and remove strategies during execution."""
    portfolio = PortfolioAllocator(total_capital=Decimal("100000.00"))

    # Start with one strategy
    portfolio.add_strategy("initial", MockStrategy(), Decimal("0.50"))

    # Run for 30 days
    start_date = pd.Timestamp("2023-01-01")

    for day in range(30):
        timestamp = start_date + pd.Timedelta(days=day)
        portfolio.execute_bar(timestamp, {})

        # Add second strategy on day 10
        if day == 10:
            portfolio.add_strategy("added", MockStrategy(), Decimal("0.30"))
            assert len(portfolio.strategies) == 2

        # Remove first strategy on day 20
        if day == 20:
            portfolio.remove_strategy("initial", liquidate=True)
            assert len(portfolio.strategies) == 1
            assert "added" in portfolio.strategies

    # Verify final state
    assert len(portfolio.strategies) == 1
    final_alloc = portfolio.strategies["added"]
    assert len(final_alloc.performance.timestamps) == 20  # Only ran for 20 days

def test_rebalancing_with_performance_based_allocation():
    """Integration test: Rebalance based on strategy performance."""
    portfolio = PortfolioAllocator(total_capital=Decimal("100000.00"))

    # Add two strategies
    portfolio.add_strategy("winner", MockStrategy(), Decimal("0.30"))
    portfolio.add_strategy("loser", MockStrategy(), Decimal("0.30"))

    # Simulate performance (winner gains 20%, loser loses 10%)
    winner = portfolio.strategies["winner"]
    loser = portfolio.strategies["loser"]

    winner.ledger._portfolio_value = Decimal("36000.00")  # +20%
    loser.ledger._portfolio_value = Decimal("27000.00")   # -10%

    # Rebalance: allocate more to winner
    portfolio.rebalance({
        "winner": Decimal("0.50"),  # Increase from 30% to 50%
        "loser": Decimal("0.20")    # Decrease from 30% to 20%
    })

    # Verify new allocations
    assert winner.allocated_capital == Decimal("50000.00")
    assert loser.allocated_capital == Decimal("20000.00")
```

**Zero-Mock Enforcement:** [Source: architecture/coding-standards.md#zero-mock-enforcement-mandatory]
- Use real PortfolioAllocator instances in all tests
- Use real DecimalLedger instances for strategy ledgers
- No hardcoded performance values or mock implementations
- Tests must exercise actual Decimal calculations
- Mock only strategy algorithms themselves (behavior), not infrastructure

**Coverage Target:** [Source: architecture/testing-strategy.md#test-coverage-targets]
- Overall: ≥90%
- Portfolio allocator module: ≥90%

## Change Log

| Date | Version | Description | Author |
|------|---------|-------------|--------|
| 2025-10-01 | 1.0 | Initial story creation | SM (Bob) |
| 2025-10-02 | 1.1 | Enhanced with comprehensive implementation details, complete PortfolioAllocator class, StrategyPerformance tracking, multi-strategy execution flow, strategy isolation mechanics, rebalancing algorithms, dynamic strategy management, 3-strategy portfolio example, comprehensive testing (8+ unit, 4+ property, 2+ integration tests), and full dependency context per PO validation | PO (Sarah) |

## Dev Agent Record

### Agent Model Used

claude-sonnet-4-5-20250929

### Debug Log References

No debug issues encountered during implementation.

### Completion Notes List

- Successfully implemented PortfolioAllocator with complete multi-strategy management capabilities
- All 24 tests passing (14 unit tests, 4 property-based tests, 3 integration tests, 3 performance tests)
- Strategy isolation mechanism fully functional using separate DecimalLedger instances
- Performance tracking implemented with volatility, Sharpe ratio, drawdown, and win rate metrics
- Rebalancing functionality with capital conservation validation
- Dynamic strategy management (add/remove/pause/resume) implemented
- Comprehensive example tutorial created demonstrating 3-strategy portfolio
- Code passes all linting checks (ruff with E, F, W, I, N selectors)
- Zero-mock enforcement: All tests use real calculations and implementations

### File List

**New Source Files:**
- [rustybt/portfolio/__init__.py](rustybt/portfolio/__init__.py) - Portfolio module initialization
- [rustybt/portfolio/allocator.py](rustybt/portfolio/allocator.py) - PortfolioAllocator, StrategyAllocation, StrategyPerformance, StrategyState

**New Test Files:**
- [tests/portfolio/__init__.py](tests/portfolio/__init__.py) - Portfolio tests initialization
- [tests/portfolio/test_allocator.py](tests/portfolio/test_allocator.py) - Comprehensive test suite (24 tests)

**New Example Files:**
- [examples/portfolio_allocator_tutorial.py](examples/portfolio_allocator_tutorial.py) - 3-strategy portfolio tutorial

## QA Results

_To be populated by QA Agent_
