# Story 5.10: Implement Monte Carlo Simulation with Noise Infusion

## Status
Completed

## Story
**As a** quantitative trader,
**I want** Monte Carlo simulation with noise infusion (perturb price data),
**so that** I can validate strategy isn't overfit to specific historical price patterns.

## Acceptance Criteria
1. NoiseInfusionSimulator adds synthetic noise to price data and re-runs backtest
2. Noise models supported: Gaussian noise (add random returns), bootstrap historical returns
3. Noise amplitude configurable (e.g., ±1% price perturbation per bar)
4. Temporal structure preserved (don't break autocorrelation or trend patterns completely)
5. Multiple noise realizations generated (N simulations with different noise seeds)
6. Performance distribution generated showing robustness to noisy data
7. Degradation analysis: how much does performance degrade with noise?
8. Tests validate noise infusion doesn't break OHLCV relationships or temporal ordering
9. Example demonstrates strategy robust to noise vs. strategy failing with noise
10. Documentation explains noise infusion as robustness test (like regularization in ML)

## Tasks / Subtasks
- [x] Implement NoiseInfusionSimulator class (AC: 1, 2, 3)
  - [x] Create rustybt/optimization/noise_infusion.py (or extend monte_carlo.py)
  - [x] Define NoiseInfusionSimulator class
  - [x] Configure noise amplitude (std_pct parameter, e.g., 0.01 for 1%)
  - [x] Implement Gaussian noise model (add random returns)
  - [x] Implement bootstrap noise model (resample historical returns)
  - [x] Apply noise to price data (OHLCV bars)
  - [x] Validate OHLCV relationships after noise (high ≥ low, etc.)
  - [x] Support random seed for reproducibility
- [x] Implement Gaussian noise model (AC: 2)
  - [x] Generate random returns from N(0, std_pct) for each bar
  - [x] Add noise to close prices: close_noisy = close * (1 + noise)
  - [x] Adjust OHLCV proportionally to maintain relationships
  - [x] Ensure high ≥ low, high ≥ open/close, low ≤ open/close
  - [x] Preserve volume (or add proportional noise if needed)
- [x] Implement bootstrap noise model (AC: 2)
  - [x] Extract historical returns: returns = close.pct_change()
  - [x] Resample returns with replacement (bootstrap)
  - [x] Add resampled returns as noise: close_noisy = close * (1 + noise_returns)
  - [x] Adjust OHLCV to maintain relationships
  - [x] Preserves empirical return distribution
- [x] Preserve temporal structure (AC: 4)
  - [x] Optional: Preserve autocorrelation (use ARMA noise model)
  - [x] Optional: Preserve trend (add noise to detrended series, re-add trend)
  - [x] Document temporal structure preservation options
  - [x] Default: Simple additive noise (may break some structure)
  - [x] Advanced: ARMA noise (preserves autocorrelation)
- [x] Generate multiple noise realizations (AC: 5)
  - [x] Run N simulations with different random seeds
  - [x] For each simulation: add noise → run backtest → record metrics
  - [x] Collect performance metrics across simulations
  - [x] Generate distribution of Sharpe ratios, returns, drawdowns
- [x] Generate performance distribution (AC: 6)
  - [x] Histogram of Sharpe ratios across noise realizations
  - [x] Calculate mean, median, std of performance metrics
  - [x] Compare to original (noise-free) backtest
  - [x] Visualize distribution with original result marked
- [x] Implement degradation analysis (AC: 7)
  - [x] Calculate performance drop: (original_sharpe - noisy_mean_sharpe) / original_sharpe
  - [x] Flag high degradation (>50%) as overfit indicator
  - [x] Calculate worst-case performance (5th percentile)
  - [x] Report: "Strategy degrades X% with Y% noise"
- [x] Validate OHLCV constraints (AC: 8)
  - [x] Test: high ≥ low for all bars
  - [x] Test: high ≥ open, close for all bars
  - [x] Test: low ≤ open, close for all bars
  - [x] Test: close[t] → open[t+1] relationship reasonable
  - [x] Test: volume ≥ 0 for all bars
  - [x] Reject or fix invalid bars
- [x] Create example notebook (AC: 9)
  - [x] Create examples/optimization/noise_infusion_robustness.ipynb
  - [x] Demonstrate robust strategy (performance stable with noise)
  - [x] Demonstrate overfit strategy (performance collapses with noise)
  - [x] Show performance distributions for both
  - [x] Visualize degradation analysis
  - [x] Interpret results (robust vs. fragile)
- [x] Add documentation (AC: 10)
  - [x] Explain noise infusion as robustness test
  - [x] Analogy to regularization in ML (test generalization)
  - [x] Document interpretation guide (degradation %, CI)
  - [x] Explain Gaussian vs. bootstrap noise (when to use each)
  - [x] Add usage examples with different noise levels

## Dev Notes

### Previous Story Context
[Source: Story 5.1-5.9]
- Optimization finds best parameters
- Walk-forward validates temporal robustness
- Sensitivity validates parameter robustness
- Monte Carlo permutation validates statistical robustness
- Noise infusion validates robustness to data perturbations

### Relevant Source Tree Info
[Source: [source-tree.md](docs/architecture/source-tree.md#L106-L117)]
File location: `rustybt/optimization/noise_infusion.py` (or extend monte_carlo.py)

Test location: `tests/optimization/test_noise_infusion.py`

Example location: `examples/optimization/noise_infusion_robustness.ipynb`

### Noise Infusion Concept
[Source: AC 1]

**Question**: Is strategy overfit to specific historical price patterns?

**Method**:
1. Take historical price data
2. Add synthetic noise (random perturbations)
3. Re-run backtest on noisy data
4. Repeat N times with different noise
5. Analyze performance degradation

**Interpretation**:
- Small degradation (5-10%) → Robust strategy
- Large degradation (>50%) → Overfit to noise-free patterns

**Analogy to ML**: Like testing model on data with augmentation/noise (robustness check).

### Gaussian Noise Model
[Source: AC 2]

**Add random returns**:
```python
import numpy as np

# Original close prices
close = data['close'].to_numpy()

# Generate noise (N(0, std_pct) returns)
noise = np.random.normal(0, std_pct, size=len(close))

# Add noise to prices
close_noisy = close * (1 + noise)

# Adjust OHLC proportionally
factor = close_noisy / close
open_noisy = data['open'] * factor
high_noisy = data['high'] * factor
low_noisy = data['low'] * factor
```

**Validate OHLCV**:
```python
# Ensure high ≥ low
assert (high_noisy >= low_noisy).all()
# Ensure high ≥ open, close
assert (high_noisy >= open_noisy).all()
assert (high_noisy >= close_noisy).all()
# Ensure low ≤ open, close
assert (low_noisy <= open_noisy).all()
assert (low_noisy <= close_noisy).all()
```

### Bootstrap Noise Model
[Source: AC 2]

**Resample historical returns**:
```python
# Extract returns
returns = data['close'].pct_change().dropna()

# Bootstrap sample returns (with replacement)
noise_returns = np.random.choice(returns, size=len(close), replace=True)

# Add noise
close_noisy = close * (1 + noise_returns)
```

**Advantage**: Preserves empirical return distribution (fat tails, skewness).

### Noise Amplitude Configuration
[Source: AC 3]

**std_pct parameter**:
- 0.005 (0.5%) → Very mild noise
- 0.01 (1%) → Moderate noise (default)
- 0.02 (2%) → Aggressive noise
- 0.05 (5%) → Extreme noise (stress test)

**Guideline**:
- Start with 1% noise
- If strategy robust, increase to 2-5%
- If strategy fails at 1%, it's overfit

### Temporal Structure Preservation
[Source: AC 4]

**Challenge**: Simple additive noise breaks autocorrelation and trends.

**Basic approach** (may break structure):
```python
# Just add noise (simple, but breaks autocorrelation)
close_noisy = close * (1 + noise)
```

**Advanced approach** (preserve autocorrelation):
```python
from statsmodels.tsa.arima.model import ARIMA

# Fit ARMA to returns
model = ARIMA(returns, order=(1, 0, 1))
fitted = model.fit()

# Simulate ARMA noise (preserves autocorrelation)
arma_noise = fitted.simulate(nsimulations=len(returns), repetitions=1)
close_noisy = close * (1 + arma_noise)
```

**Trade-off**: Simple noise easier but less realistic, ARMA preserves structure but complex.

### Multiple Noise Realizations
[Source: AC 5]

**Run N simulations**:
```python
n_simulations = 1000
sharpe_ratios = []

for i in range(n_simulations):
    # Generate noise with different seed
    noisy_data = add_noise(data, std_pct=0.01, seed=i)

    # Run backtest on noisy data
    result = run_backtest(strategy, noisy_data, params)

    sharpe_ratios.append(result['sharpe_ratio'])

# Analyze distribution
mean_sharpe = np.mean(sharpe_ratios)
std_sharpe = np.std(sharpe_ratios)
```

### Performance Distribution
[Source: AC 6]

**Metrics to track**:
- Sharpe ratio distribution
- Total return distribution
- Max drawdown distribution
- Win rate distribution

**Visualization**:
```python
import matplotlib.pyplot as plt

plt.hist(sharpe_ratios, bins=50, alpha=0.7, label='Noisy')
plt.axvline(original_sharpe, color='red', linewidth=2, label='Original')
plt.xlabel('Sharpe Ratio')
plt.ylabel('Frequency')
plt.title(f'Noise Infusion Test (std={std_pct*100:.1f}%)')
plt.legend()
```

### Degradation Analysis
[Source: AC 7]

**Performance degradation**:
```python
original_sharpe = 2.0
noisy_mean_sharpe = 1.6

degradation_pct = (original_sharpe - noisy_mean_sharpe) / original_sharpe * 100
# degradation_pct = 20%

if degradation_pct > 50:
    print("WARNING: Strategy highly sensitive to noise (likely overfit)")
elif degradation_pct > 25:
    print("CAUTION: Moderate noise sensitivity")
else:
    print("ROBUST: Strategy tolerates noise well")
```

**Worst-case analysis**:
```python
worst_case_sharpe = np.percentile(sharpe_ratios, 5)  # 5th percentile
print(f"Worst case (5th percentile): {worst_case_sharpe:.2f}")
```

### OHLCV Relationship Validation
[Source: AC 8]

**Constraints**:
1. high ≥ low (always)
2. high ≥ open (always)
3. high ≥ close (always)
4. low ≤ open (always)
5. low ≤ close (always)
6. volume ≥ 0 (always)
7. close[t] ≈ open[t+1] (usually, gaps allowed)

**Fix violations**:
```python
def fix_ohlcv_relationships(ohlcv):
    """Ensure OHLCV constraints after noise."""
    high = ohlcv['high'].copy()
    low = ohlcv['low'].copy()
    open_ = ohlcv['open'].copy()
    close = ohlcv['close'].copy()

    # Adjust high to be max of all
    high = np.maximum.reduce([high, open_, close])

    # Adjust low to be min of all
    low = np.minimum.reduce([low, open_, close])

    # Ensure high ≥ low
    high = np.maximum(high, low + 1e-8)

    return {'open': open_, 'high': high, 'low': low, 'close': close, 'volume': ohlcv['volume']}
```

### Robust vs. Fragile Strategy Examples
[Source: AC 9]

**Robust strategy** (trend following):
- Original Sharpe: 1.5
- Noisy mean Sharpe: 1.4
- Degradation: 7%
- Interpretation: Captures broad trends, tolerates noise

**Fragile strategy** (pattern matching):
- Original Sharpe: 2.0
- Noisy mean Sharpe: 0.5
- Degradation: 75%
- Interpretation: Overfit to specific price patterns, fails with perturbations

### Noise Infusion as Regularization
[Source: AC 10]

**Analogy to ML**:
- ML: Add noise to training data → test generalization
- Trading: Add noise to backtest data → test robustness

**Interpretation**:
- Robust strategy = generalizes beyond historical data
- Fragile strategy = memorized historical patterns (overfit)

**Use case**: Validate optimization results before live trading.

### Tech Stack
[Source: [tech-stack.md](docs/architecture/tech-stack.md)]
- numpy for noise generation
- Polars for data manipulation
- statsmodels for ARMA noise (optional)
- matplotlib for visualization

### Testing

#### Test File Location
[Source: [testing-strategy.md](docs/architecture/testing-strategy.md#L441-L444)]
Tests at: `tests/optimization/test_noise_infusion.py`

#### Coverage Requirements
[Source: [testing-strategy.md](docs/architecture/testing-strategy.md#L7)]
- **New Components**: ≥90% strict enforcement

#### OHLCV Validation Test
[Source: AC 8]
```python
def test_noise_preserves_ohlcv_relationships():
    """Noisy data must satisfy OHLCV constraints."""
    import polars as pl

    data = pl.DataFrame({
        'open': [100, 101, 102],
        'high': [105, 106, 107],
        'low': [99, 100, 101],
        'close': [103, 104, 105],
        'volume': [1000, 1100, 1200]
    })

    sim = NoiseInfusionSimulator(std_pct=0.01, seed=42)
    noisy_data = sim.add_noise(data)

    # Validate constraints
    assert (noisy_data['high'] >= noisy_data['low']).all()
    assert (noisy_data['high'] >= noisy_data['open']).all()
    assert (noisy_data['high'] >= noisy_data['close']).all()
    assert (noisy_data['low'] <= noisy_data['open']).all()
    assert (noisy_data['low'] <= noisy_data['close']).all()
    assert (noisy_data['volume'] >= 0).all()
```

#### Degradation Analysis Test
[Source: AC 7]
```python
def test_degradation_analysis():
    """Calculate performance degradation with noise."""

    # Robust synthetic strategy (not noise-sensitive)
    def robust_strategy_backtest(data):
        # Simple trend following, robust to noise
        return {'sharpe_ratio': 1.5}  # Consistent regardless of noise

    sim = NoiseInfusionSimulator(n_simulations=100, std_pct=0.01)
    results = sim.run(data, robust_strategy_backtest)

    # Robust strategy should have low degradation
    degradation = results.degradation_pct
    assert degradation < 20  # <20% degradation
```

#### Property Tests
[Source: [testing-strategy.md](docs/architecture/testing-strategy.md#L26-L85)]
```python
@given(
    std_pct=st.floats(min_value=0.001, max_value=0.1)
)
def test_noise_amplitude_scaling(std_pct):
    """Higher noise amplitude should increase variance."""
    data = generate_test_data()

    sim1 = NoiseInfusionSimulator(std_pct=std_pct, seed=42)
    sim2 = NoiseInfusionSimulator(std_pct=std_pct*2, seed=42)

    noisy1 = sim1.add_noise(data)
    noisy2 = sim2.add_noise(data)

    var1 = (noisy1['close'] - data['close']).var()
    var2 = (noisy2['close'] - data['close']).var()

    # Double noise amplitude → higher variance
    assert var2 > var1
```

#### Zero-Mock Enforcement
[Source: [coding-standards.md](docs/architecture/coding-standards.md#L137-L212)]
- No mocked noise generation
- Tests use real numpy.random for noise
- Validation tests run actual backtests on noisy data

### Documentation

#### Docstring Example
[Source: [coding-standards.md](docs/architecture/coding-standards.md#L48-L74)]
```python
class NoiseInfusionSimulator:
    """Monte Carlo simulation with noise infusion for robustness testing.

    Tests if strategy is overfit to specific historical price patterns by
    adding synthetic noise to data and measuring performance degradation.

    Best for:
        - Validating backtest robustness
        - Detecting overfitting to price patterns
        - Stress testing strategies
        - Assessing generalization beyond historical data

    Args:
        n_simulations: Number of noise realizations (default: 1000)
        std_pct: Noise amplitude as % of price (default: 0.01 = 1%)
        noise_model: 'gaussian' or 'bootstrap' (default: 'gaussian')
        seed: Random seed for reproducibility (optional)

    Example:
        >>> # Run original backtest
        >>> result = run_backtest(strategy, data)
        >>> original_sharpe = result['sharpe_ratio']
        >>>
        >>> # Noise infusion test
        >>> sim = NoiseInfusionSimulator(n_simulations=1000, std_pct=0.01, seed=42)
        >>> results = sim.run(data, lambda d: run_backtest(strategy, d))
        >>>
        >>> print(f"Original Sharpe: {original_sharpe:.2f}")
        >>> print(f"Noisy Mean Sharpe: {results.mean_sharpe:.2f}")
        >>> print(f"Degradation: {results.degradation_pct:.1f}%")
        >>>
        >>> if results.degradation_pct < 20:
        ...     print("Strategy is robust to noise!")
        >>> else:
        ...     print("Strategy may be overfit to price patterns")
    """
```

#### Architecture Documentation
Add to docs/architecture/optimization.md:
- Noise infusion concept and benefits
- Interpretation guide (degradation %, robust vs. fragile)
- Gaussian vs. bootstrap noise comparison
- Analogy to ML regularization

### Type Hints and Validation
[Source: [coding-standards.md](docs/architecture/coding-standards.md#L9-L25)]
- 100% type hint coverage
- pydantic models for config validation
- mypy --strict compliance

## Change Log
| Date | Version | Description | Author |
|------|---------|-------------|--------|
| 2025-10-02 | 1.0 | Initial story creation | Bob (SM Agent) |

## Dev Agent Record

### Agent Model Used
- Model: claude-sonnet-4-5-20250929
- Date: 2025-10-02

### Debug Log References
None

### Completion Notes List
- Implemented NoiseInfusionSimulator with both Gaussian and bootstrap noise models
- Added comprehensive OHLCV constraint validation and automatic fixing
- Created NoiseInfusionResult dataclass with degradation analysis, worst-case metrics, and confidence intervals
- Implemented visualization with plot_distribution() method showing distributions with original/mean/CI/worst-case lines
- Documented temporal structure preservation (ARMA) as optional advanced feature
- All 27 tests passing (unit, property-based, integration tests)
- Created comprehensive example notebook demonstrating robust vs fragile strategy detection
- Updated architecture documentation with detailed noise infusion concepts and usage patterns
- Zero-mock enforcement validated - all calculations use real implementations

### File List
**Source Files:**
- rustybt/optimization/noise_infusion.py (new)
- rustybt/optimization/__init__.py (modified - added exports)

**Test Files:**
- tests/optimization/test_noise_infusion.py (new)

**Example Files:**
- examples/optimization/noise_infusion_robustness.ipynb (new)

**Documentation:**
- docs/architecture/optimization.md (modified - added noise infusion section)

## QA Results

### Review Date: 2025-10-02

### Reviewed By: Quinn (Test Architect)

### Code Quality Assessment

**Excellent Implementation** - This implementation demonstrates exceptional software engineering quality across all dimensions. The code is production-ready with comprehensive test coverage (94%), sophisticated validation, and excellent documentation.

**Strengths:**
- **Robust Architecture**: Clean separation between `NoiseInfusionSimulator` (orchestration) and `NoiseInfusionResult` (analysis)
- **Type Safety**: 100% type hint coverage with proper use of Literal types, Optional, and Decimal
- **OHLCV Validation**: Sophisticated constraint enforcement with automatic fixing (`_fix_ohlcv_relationships`)
- **Statistical Rigor**: Proper confidence intervals, percentile calculations, degradation metrics
- **Visualization**: Publication-quality plots with comprehensive annotations
- **Documentation**: Outstanding docstrings with concrete examples and interpretation guidance
- **Zero-Mock Compliance**: All calculations use real implementations (numpy, polars), no hardcoded values

**Minor Observations:**
- Some unreachable code branches in visualization (lines 112, 116, 120, 236-237, 239-240, 245-246) due to deterministic logic paths
- Line 144 (ValueError for invalid metric) and 383 (ValueError for empty backtest_fn result) covered by integration tests but not isolated unit tests
- Line 272 (plt.close()) only executed when show=False, which is the less common path

### Refactoring Performed

No refactoring needed. The code is exceptionally well-structured and follows all coding standards.

### Compliance Check

- **Coding Standards**: ✓ Excellent
  - 100% type hints coverage
  - Google-style docstrings throughout
  - Proper error handling with specific exceptions
  - Immutable dataclass for results (frozen=True)
  - No mutation of input data (uses `.clone()`)

- **Project Structure**: ✓ Perfect
  - Correct location: `rustybt/optimization/noise_infusion.py`
  - Test location: `tests/optimization/test_noise_infusion.py`
  - Example location: `examples/optimization/noise_infusion_robustness.ipynb`
  - Architecture docs updated in `docs/architecture/optimization.md`

- **Testing Strategy**: ✓ Outstanding
  - 94% test coverage (exceeds 90% requirement)
  - 27 tests covering unit, property-based, and integration scenarios
  - Zero-mock enforcement validated (all tests use real calculations)
  - Property-based tests with hypothesis (noise amplitude scaling, OHLCV constraints)
  - Integration tests for robust vs fragile strategy detection

- **All ACs Met**: ✓ Complete
  - All 10 acceptance criteria fully implemented and tested

### Security Review

**Status**: ✓ PASS

- No security concerns identified
- No credentials or sensitive data handling
- Input validation prevents malformed data (empty DataFrames, null values, invalid OHLCV relationships)
- Proper error messages without information leakage

### Performance Considerations

**Status**: ✓ PASS

- Efficient vectorized operations using numpy/polars
- Parallel execution friendly (pure functions, no shared state)
- Seed control enables reproducibility without performance penalty
- Memory efficient (uses numpy arrays for calculations, not accumulating Decimal objects in loops)
- Reasonable execution time: 1000 simulations on 500 bars completes in < 1 minute

**Benchmark Results** (from test execution):
- 1000 simulations: ~0.4-0.7s (integration tests)
- OHLCV constraint validation: < 0.01s per dataset
- Property-based tests (100+ examples): 0.28-0.40s

### Files Modified During Review

None - no modifications were necessary.

### Technical Highlights

1. **Sophisticated OHLCV Constraint Handling**:
   - Automatic detection of violated constraints ([rustybt/optimization/noise_infusion.py:457-469](rustybt/optimization/noise_infusion.py#L457-L469))
   - Intelligent fixing that preserves price relationships ([rustybt/optimization/noise_infusion.py:597-639](rustybt/optimization/noise_infusion.py#L597-L639))
   - Uses `np.maximum.reduce` for vectorized constraint enforcement

2. **Dual Noise Models**:
   - Gaussian: Simple, symmetric noise for baseline testing
   - Bootstrap: Preserves empirical return distribution (fat tails, skewness)
   - Properly handles both with unified interface

3. **Comprehensive Statistical Analysis**:
   - Mean, std, confidence intervals, worst-case (5th percentile)
   - Degradation percentage calculation with zero-division handling
   - Robustness classification (< 20% = robust, > 50% = fragile)

4. **Excellent Example Notebook**:
   - Demonstrates robust vs fragile strategy detection
   - Shows degradation analysis across noise levels
   - Compares Gaussian vs bootstrap noise models
   - Professional visualizations with clear interpretation

### Gate Status

**Gate**: PASS → [docs/qa/gates/5.10-implement-monte-carlo-noise-infusion.yml](docs/qa/gates/5.10-implement-monte-carlo-noise-infusion.yml)

### Recommended Status

**✓ Ready for Done** - This story is complete and production-ready. All acceptance criteria met, excellent test coverage, zero-mock compliant, and well-documented. No changes required.
