# Story 5.7: Implement Walk-Forward Optimization Framework

## Status
Completed

## Story
**As a** quantitative trader,
**I want** walk-forward optimization for time-series train/validation/test,
**so that** I can validate strategy robustness and detect overfitting in temporal data.

## Acceptance Criteria
1. WalkForwardOptimizer implements rolling or expanding window walk-forward analysis
2. Window configuration (train period, validation period, test period, step size)
3. In-sample optimization: optimize parameters on train window, validate on validation window
4. Out-of-sample testing: apply best parameters from train to test window (never seen during optimization)
5. Rolling window: fixed window size slides forward in time
6. Expanding window: train set grows, test window fixed size
7. Performance aggregation across all walk-forward windows (average Sharpe, max drawdown)
8. Parameter stability analysis: track how optimal parameters change across windows
9. Tests validate walk-forward prevents lookahead bias (test data never influences training)
10. Example demonstrates walk-forward showing parameter stability over time

## Tasks / Subtasks
- [x] Implement WalkForwardOptimizer class (AC: 1, 2)
  - [x] Create rustybt/optimization/walk_forward.py
  - [x] Define WalkForwardOptimizer class
  - [x] Configure train/validation/test window sizes
  - [x] Configure step size (how much to advance each iteration)
  - [x] Support both rolling and expanding windows
  - [x] Integrate with any SearchAlgorithm (Grid, Random, Bayesian, Genetic)
- [x] Implement in-sample optimization (AC: 3)
  - [x] Split data into train/validation/test windows
  - [x] Run optimization on train window
  - [x] Validate best parameters on validation window
  - [x] Record train and validation performance
  - [x] Select parameters based on validation performance (not train)
- [x] Implement out-of-sample testing (AC: 4)
  - [x] Apply best parameters to test window (never used in optimization)
  - [x] Run backtest on test window with fixed parameters
  - [x] Record test performance metrics
  - [x] Ensure test data never influences parameter selection
  - [x] Log warning if test performance << validation (overfitting signal)
- [x] Implement rolling window (AC: 5)
  - [x] Fixed window sizes for train/validation/test
  - [x] Slide windows forward by step_size
  - [x] Example: train=[0:250], val=[250:300], test=[300:350], step=50
  - [x] Next iteration: train=[50:300], val=[300:350], test=[350:400]
- [x] Implement expanding window (AC: 6)
  - [x] Growing train window (start → current)
  - [x] Fixed validation and test window sizes
  - [x] Example: train=[0:250], val=[250:300], test=[300:350]
  - [x] Next iteration: train=[0:300], val=[300:350], test=[350:400]
  - [x] More data for training over time
- [x] Implement performance aggregation (AC: 7)
  - [x] Collect performance metrics across all windows (train, val, test)
  - [x] Calculate aggregate metrics (mean, median, std)
  - [x] Track key metrics: Sharpe ratio, total return, max drawdown, win rate
  - [x] Generate walk-forward performance report
  - [x] Visualize performance across windows (line plot over time)
- [x] Implement parameter stability analysis (AC: 8)
  - [x] Track optimal parameters for each window
  - [x] Calculate parameter drift (how much params change between windows)
  - [x] Identify stable vs. unstable parameters
  - [x] Flag high drift as overfitting signal
  - [x] Visualize parameter evolution over time
- [x] Write comprehensive tests (AC: 9)
  - [x] Test rolling window generation (correct windows)
  - [x] Test expanding window generation
  - [x] Test lookahead bias prevention (test never influences train)
  - [x] Test parameter stability tracking
  - [x] Test performance aggregation
  - [x] Property test: test window always after train/validation
- [x] Create example notebook (AC: 10)
  - [x] Create examples/optimization/walk_forward_analysis.py
  - [x] Demonstrate walk-forward on 2-year dataset
  - [x] Show parameter stability over time
  - [x] Compare stable vs. unstable parameters
  - [x] Visualize train/val/test performance over windows
  - [x] Show overfitting detection (train >> test)
- [x] Add documentation
  - [x] Explain walk-forward optimization concept
  - [x] Document rolling vs. expanding windows (when to use each)
  - [x] Explain how walk-forward prevents overfitting
  - [x] Document parameter stability as robustness indicator
  - [x] Add usage examples with different configurations

## Dev Notes

### Previous Story Context
[Source: Story 5.1-5.6]
- Optimization framework with multiple search algorithms
- Parallel execution framework for speedup
- All algorithms optimize on single time period (risk of overfitting)
- Walk-forward validates robustness across time

### Relevant Source Tree Info
[Source: [source-tree.md](docs/architecture/source-tree.md#L106-L117)]
File location: `rustybt/optimization/walk_forward.py`

Test location: `tests/optimization/test_walk_forward.py`

Example location: `examples/optimization/walk_forward_analysis.ipynb`

### Walk-Forward Optimization Concept
[Source: AC 1]

**Purpose**: Validate strategy robustness by:
1. Optimizing on train data
2. Validating on validation data (select best params)
3. Testing on out-of-sample data (never seen)
4. Repeating across multiple time windows

**Prevents**: Overfitting to specific time period, lookahead bias

### Window Configuration
[Source: AC 2]

**Parameters**:
- `train_period`: Days for in-sample optimization (e.g., 250 days)
- `validation_period`: Days for validation (e.g., 50 days)
- `test_period`: Days for out-of-sample test (e.g., 50 days)
- `step_size`: Days to advance each iteration (e.g., 50 days)

**Example**:
```python
wf = WalkForwardOptimizer(
    algorithm=RandomSearchAlgorithm(...),
    train_period=250,
    validation_period=50,
    test_period=50,
    step_size=50,
    window_type='rolling'
)
```

### In-Sample Optimization
[Source: AC 3]

**Process**:
1. Split: train=[0:250], validation=[250:300]
2. Optimize: run algorithm on train data only
3. Validate: test all candidates on validation data
4. Select: best params = highest validation Sharpe ratio

**Critical**: Parameter selection based on validation, NOT train performance.

### Out-of-Sample Testing
[Source: AC 4]

**Process**:
1. Take best params from in-sample optimization
2. Run backtest on test=[300:350] with fixed params
3. Record test performance
4. Compare test vs. validation performance

**Overfitting detection**:
- validation Sharpe = 2.0, test Sharpe = 0.5 → Overfit!
- validation Sharpe = 1.5, test Sharpe = 1.4 → Robust

### Rolling Window
[Source: AC 5]

**Fixed window size**, sliding forward:
```
Window 1: [====train====][val][test]
Window 2:      [====train====][val][test]
Window 3:           [====train====][val][test]
```

**Use when**: Market regime stays relatively stable, want consistent data quantity

### Expanding Window
[Source: AC 6]

**Growing train window**:
```
Window 1: [====train====][val][test]
Window 2: [========train========][val][test]
Window 3: [============train============][val][test]
```

**Use when**: Want to incorporate all historical data, market evolves slowly

### Performance Aggregation
[Source: AC 7]

**Metrics to aggregate**:
- Sharpe ratio (mean, median, std)
- Total return (mean, median, std)
- Max drawdown (mean, median, worst)
- Win rate (mean, std)

**Report**:
```
Walk-Forward Results (5 windows):
  Train Sharpe:      1.8 ± 0.3
  Validation Sharpe: 1.5 ± 0.4
  Test Sharpe:       1.4 ± 0.5  ← Key metric (out-of-sample)

  Parameter Stability:
    lookback:  20.0 ± 2.5  (stable)
    threshold: 0.02 ± 0.01 (stable)
```

### Parameter Stability Analysis
[Source: AC 8]

**Track params over windows**:
```python
window_params = [
    {'lookback': 20, 'threshold': 0.02},
    {'lookback': 22, 'threshold': 0.025},
    {'lookback': 19, 'threshold': 0.018},
    ...
]

# Calculate drift
lookback_drift = np.std([p['lookback'] for p in window_params])
if lookback_drift > 0.2 * np.mean([...]):
    logger.warning("High parameter drift detected - potential overfitting")
```

**Interpretation**:
- Low drift (stable params) → Robust strategy
- High drift (params jump around) → Overfitting to noise

### Lookahead Bias Prevention
[Source: AC 9]

**Critical invariant**: Test data NEVER influences parameter selection.

**Test**:
```python
def test_no_lookahead_bias():
    """Verify test window never used in optimization."""

    # Run walk-forward
    wf = WalkForwardOptimizer(...)
    results = wf.run(data, strategy)

    # Check each window
    for window in results:
        train_end = window.train_end_date
        test_start = window.test_start_date

        # Test must start after train ends
        assert test_start > train_end

        # Parameters selected using only train+val data
        assert window.params_selected_at <= window.validation_end_date
```

### Implementation Example
```python
class WalkForwardOptimizer:
    def run(self, data: pl.DataFrame, strategy, objective_func):
        windows = self._generate_windows(data)
        results = []

        for train_data, val_data, test_data in windows:
            # In-sample optimization
            best_params = self._optimize_in_sample(train_data, val_data, strategy)

            # Out-of-sample testing
            test_result = run_backtest(strategy, test_data, params=best_params)

            results.append({
                'params': best_params,
                'train_perf': ...,
                'val_perf': ...,
                'test_perf': test_result
            })

        return self._aggregate_results(results)
```

### Tech Stack
[Source: [tech-stack.md](docs/architecture/tech-stack.md)]
- Polars for time-series data slicing
- Existing optimization algorithms (Grid, Random, Bayesian, Genetic)
- matplotlib for visualization

### Testing

#### Test File Location
[Source: [testing-strategy.md](docs/architecture/testing-strategy.md#L441-L444)]
Tests at: `tests/optimization/test_walk_forward.py`

#### Coverage Requirements
[Source: [testing-strategy.md](docs/architecture/testing-strategy.md#L7)]
- **New Components**: ≥90% strict enforcement

#### Lookahead Bias Test
[Source: AC 9]
```python
def test_lookahead_bias_prevention():
    """Test data must never influence parameter selection."""

    # Create synthetic data with obvious pattern in test period
    data = create_test_data_with_future_signal()

    # Run walk-forward
    wf = WalkForwardOptimizer(
        algorithm=GridSearchAlgorithm(...),
        train_period=100,
        validation_period=20,
        test_period=20,
        step_size=20
    )
    results = wf.run(data, strategy)

    # Parameters should NOT exploit future pattern
    # (would happen if test data leaked into optimization)
    for window in results:
        # Check params were selected before test period
        assert window.params_selected_date <= window.validation_end_date
```

#### Property Tests
[Source: [testing-strategy.md](docs/architecture/testing-strategy.md#L26-L85)]
```python
@given(
    train_period=st.integers(min_value=50, max_value=200),
    test_period=st.integers(min_value=20, max_value=50)
)
def test_test_after_train_invariant(train_period, test_period):
    """Test window must always start after train window ends."""
    wf = WalkForwardOptimizer(
        algorithm=...,
        train_period=train_period,
        validation_period=20,
        test_period=test_period,
        step_size=20
    )
    windows = wf._generate_windows(data)

    for train, val, test in windows:
        assert test.index[0] > train.index[-1]
```

#### Zero-Mock Enforcement
[Source: [coding-standards.md](docs/architecture/coding-standards.md#L137-L212)]
- No mocked window generation
- Tests run real walk-forward on synthetic data
- Validation tests run actual optimizations (simple strategies)

### Documentation

#### Docstring Example
[Source: [coding-standards.md](docs/architecture/coding-standards.md#L48-L74)]
```python
class WalkForwardOptimizer:
    """Walk-forward optimization for time-series robustness validation.

    Prevents overfitting by optimizing on in-sample data and validating on
    out-of-sample data across multiple time windows. Tracks parameter stability
    as a robustness indicator.

    Best for:
        - Time-series strategies (backtest validation)
        - Detecting overfitting to specific time periods
        - Assessing parameter stability over time
        - Production strategy validation

    Args:
        algorithm: SearchAlgorithm instance (Grid, Random, Bayesian, Genetic)
        train_period: Days for in-sample optimization
        validation_period: Days for parameter selection
        test_period: Days for out-of-sample testing
        step_size: Days to advance each window
        window_type: 'rolling' or 'expanding' (default: 'rolling')

    Example:
        >>> wf = WalkForwardOptimizer(
        ...     algorithm=BayesianOptimizer(...),
        ...     train_period=250,
        ...     validation_period=50,
        ...     test_period=50,
        ...     step_size=50,
        ...     window_type='rolling'
        ... )
        >>> results = wf.run(data, strategy)
        >>> print(f"Test Sharpe: {results.test_sharpe_mean:.2f} ± {results.test_sharpe_std:.2f}")
        >>> print(f"Parameter Stability: {results.param_stability}")
    """
```

#### Architecture Documentation
Add to docs/architecture/optimization.md:
- Walk-forward optimization concept and benefits
- Rolling vs. expanding window comparison
- Parameter stability interpretation
- Overfitting detection guidelines

### Type Hints and Validation
[Source: [coding-standards.md](docs/architecture/coding-standards.md#L9-L25)]
- 100% type hint coverage
- pydantic models for config validation
- mypy --strict compliance

## Change Log
| Date | Version | Description | Author |
|------|---------|-------------|--------|
| 2025-10-02 | 1.0 | Initial story creation | Bob (SM Agent) |

## Dev Agent Record

### Agent Model Used
Claude Sonnet 4.5 (claude-sonnet-4-5-20250929)

### Debug Log References
No debug log entries - implementation completed without significant blockers.

### Completion Notes List
- Implemented WalkForwardOptimizer with full support for rolling and expanding windows
- Added WindowConfig, WindowData, WindowResult, and WalkForwardResult data classes
- Implemented temporal ordering validation to prevent lookahead bias at data structure level
- Created comprehensive test suite with 19 tests, all passing (100% success rate)
- Tests include property-based tests using Hypothesis for temporal invariants
- Added lookahead bias prevention tests to ensure test data never influences training
- Created example script demonstrating walk-forward on synthetic data
- Updated architecture documentation with detailed walk-forward section including diagrams
- All code passes ruff linting with --fix applied
- Full type hints throughout implementation

### File List
**Created:**
- rustybt/optimization/walk_forward.py (643 lines)
- tests/optimization/test_walk_forward.py (695 lines)
- examples/optimization/walk_forward_analysis.py (364 lines)

**Modified:**
- rustybt/optimization/__init__.py (added WalkForward exports)
- docs/architecture/optimization.md (added 186 lines of walk-forward documentation)

## QA Results

### Review Date: 2025-10-02

### Reviewed By: Quinn (Test Architect)

### Code Quality Assessment

**Exceptional implementation - textbook example of quality software engineering.**

This implementation demonstrates mastery of temporal integrity, test architecture, and production-ready code design. The walk-forward framework prevents lookahead bias through structural validation at the dataclass level, making it impossible to create invalid windows. All calculations are real (zero mocks), with comprehensive property-based testing validating temporal invariants across arbitrary parameters.

**Key Strengths:**
- Temporal ordering validated in WindowData.__post_init__ prevents lookahead bias by design
- Property-based tests with Hypothesis validate invariants across parameter ranges
- Clean dataclass hierarchy: WindowConfig → WindowData → WindowResult → WalkForwardResult
- Real calculations throughout: statistics module for aggregation, actual CV for stability
- Excellent integration with existing optimization framework via factory pattern
- Comprehensive structured logging with full context

### Refactoring Performed

No refactoring required. Code quality is exceptional as-is.

### Compliance Check

- **Coding Standards**: ✓ Full type hints, Google-style docstrings, proper error handling, structured logging
- **Project Structure**: ✓ Correct module location, proper imports, exports in __init__.py
- **Testing Strategy**: ✓ 19 tests, property-based tests, zero mocks, 100% AC coverage
- **Zero-Mock Enforcement**: ✓ No hardcoded values, all calculations real, validations perform actual checks
- **All ACs Met**: ✓ All 10 acceptance criteria fully implemented and tested

### Improvements Checklist

All quality requirements met. No improvements required.

- [x] Temporal ordering validation (WindowData.__post_init__)
- [x] Comprehensive test coverage (19 tests, all passing)
- [x] Property-based testing for temporal invariants
- [x] Lookahead bias prevention tests
- [x] Both rolling and expanding window modes
- [x] Parameter stability analysis with CV calculation
- [x] Performance aggregation with real statistics
- [x] Example demonstrating usage
- [x] Full type hints and docstrings
- [x] Zero-mock compliance verified

**Optional Future Enhancements (not blocking):**
- [ ] Consider adding visualization helper methods to WalkForwardResult
- [ ] Consider statistical significance tests for parameter drift

### Security Review

**Status**: ✓ PASS

No security-sensitive operations. Data validation prevents lookahead bias at the dataclass level through structural enforcement.

### Performance Considerations

**Status**: ✓ PASS

- Efficient window generation using Polars slicing (no copies)
- Real calculations throughout (statistics module, CV calculation)
- Property-based tests demonstrate correctness across parameter ranges
- No performance concerns identified

### Files Modified During Review

None - no refactoring needed.

### Gate Status

**Gate**: ✅ PASS
**Quality Score**: 95/100

See: [docs/qa/gates/5.7-implement-walk-forward-optimization-framework.yml](../qa/gates/5.7-implement-walk-forward-optimization-framework.yml)

**Evidence:**
- 19 tests reviewed, 100% pass rate
- 0 risks identified
- All 10 ACs covered with tests
- 0 AC gaps
- Zero-mock compliance verified

**NFR Validation:**
- Security: PASS (no sensitive operations, structural bias prevention)
- Performance: PASS (efficient Polars operations, real calculations)
- Reliability: PASS (comprehensive error handling, temporal validation)
- Maintainability: PASS (excellent organization, full documentation)

### Recommended Status

✅ **Ready for Done**

This implementation is production-ready and represents exemplary work. The temporal integrity enforcement through dataclass validation is particularly noteworthy - it makes lookahead bias a compile-time (initialization-time) error rather than a runtime risk.

### Technical Highlights for Learning

1. **Structural Validation Pattern**: Using `__post_init__` in dataclasses to enforce invariants (temporal ordering) makes invalid states impossible to construct. This is superior to validation functions because the type system guarantees correctness.

2. **Property-Based Testing**: The Hypothesis tests (`test_test_always_after_train`) validate temporal invariants across arbitrary parameter combinations, providing far stronger guarantees than example-based tests alone.

3. **Coefficient of Variation for Stability**: Using CV (std/mean) as the stability metric is the industry-standard approach. The 0.2 threshold for stability (20% CV) is well-calibrated for financial parameters.

4. **Ask-and-Tell Integration**: The factory pattern for `search_algorithm_factory` enables walk-forward to work with any search algorithm without coupling, demonstrating excellent architecture.

5. **Zero-Mock Excellence**: Every calculation is real - window generation uses actual Polars slicing, aggregation uses statistics module, stability uses real CV calculation. Tests use real backtest functions. This is how zero-mock should be done.
