# Story 6.4: Implement Position Reconciliation with Broker

## Status
Ready for Done

## Story
**As a** quantitative trader,
**I want** automatic position reconciliation comparing local state vs. broker positions,
**so that** I can detect and resolve discrepancies before they cause trading errors.

## Acceptance Criteria
1. Reconciliation runs on engine startup and periodically during operation (e.g., every 5 minutes)
2. Fetch positions from broker via API
3. Compare local positions vs. broker positions (symbol, quantity, side)
4. Discrepancy detection (differences flagged with severity: minor vs. critical)
5. Reconciliation strategies configurable (sync_to_broker, sync_to_local, halt_and_alert)
6. Cash balance reconciliation (compare local cash vs. broker account balance)
7. Order reconciliation (compare local pending orders vs. broker open orders)
8. Reconciliation report generated (summary of discrepancies and actions taken)
9. Tests validate reconciliation with simulated discrepancies
10. Documentation explains reconciliation logic and configuration options

## Tasks / Subtasks
- [x] Implement PositionReconciler core (AC: 1, 2, 3)
  - [x] Create PositionReconciler class in rustybt/live/reconciler.py
  - [x] Implement async reconcile_positions() method
  - [x] Fetch broker positions via BrokerAdapter.get_positions()
  - [x] **CRITICAL:** Convert broker position amounts to Decimal (broker APIs return float/string)
  - [x] Fetch local positions from DecimalLedger (already Decimal)
  - [x] Compare positions by asset (sid matching)
  - [x] **CRITICAL:** Use Decimal arithmetic for quantity comparison (never convert to float)
  - [x] Detect quantity discrepancies (local.amount != broker.amount with Decimal precision)
  - [x] Detect missing positions (exists in local but not broker, or vice versa)
- [x] Implement discrepancy detection and severity classification (AC: 4)
  - [x] Define PositionDiscrepancy dataclass (asset, local_amount, broker_amount, severity, discrepancy_type)
  - [x] Classify severity: MINOR (<1% difference), MODERATE (1-5% difference), CRITICAL (>5% difference or missing position)
  - [x] Detect discrepancy types: QUANTITY_MISMATCH, MISSING_LOCAL, MISSING_BROKER, SIDE_MISMATCH
  - [x] Calculate discrepancy percentage: abs(local - broker) / max(local, broker)
  - [x] Flag critical discrepancies for immediate attention
- [x] Implement reconciliation strategies (AC: 5)
  - [x] Define ReconciliationStrategy enum (SYNC_TO_BROKER, SYNC_TO_LOCAL, HALT_AND_ALERT, WARN_ONLY)
  - [x] Implement SYNC_TO_BROKER: Update local positions to match broker
  - [x] Implement SYNC_TO_LOCAL: Submit orders to broker to match local positions (use with caution!)
  - [x] Implement HALT_AND_ALERT: Stop engine, log critical alert, require manual intervention
  - [x] Implement WARN_ONLY: Log discrepancies, continue operation (default for minor discrepancies)
  - [x] Apply strategy based on discrepancy severity and configuration
  - [x] Log all reconciliation actions with before/after state
- [x] Implement cash balance reconciliation (AC: 6)
  - [x] Fetch broker cash balance via BrokerAdapter.get_account_info()
  - [x] Fetch local cash balance from DecimalLedger
  - [x] Compare cash balances with tolerance (allow 1% difference for rounding)
  - [x] Detect cash discrepancies exceeding tolerance
  - [x] Apply reconciliation strategy for cash discrepancies
  - [x] Log cash reconciliation results
- [x] Implement order reconciliation (AC: 7)
  - [x] Implement get_open_orders() method for BrokerAdapter (extends interface defined in Story 6.1, update base.py)
  - [x] Fetch local pending orders from OrderManager
  - [x] Compare orders by broker_order_id
  - [x] Detect orphaned orders (in OrderManager but not in broker, or vice versa)
  - [x] Detect status mismatches (local=pending, broker=filled)
  - [x] Apply reconciliation: update local order status or cancel orphaned broker orders
  - [x] Log order reconciliation results
- [x] Implement periodic reconciliation scheduling (AC: 1)
  - [x] Integrate with LiveTradingEngine and APScheduler
  - [x] Schedule reconciliation job (default: every 5 minutes, configurable)
  - [x] Run reconciliation on engine startup (after state restoration)
  - [x] Run reconciliation after significant events (large order fills, strategy changes)
  - [x] Add configurable reconciliation_interval parameter
- [x] Generate reconciliation reports (AC: 8)
  - [x] Create ReconciliationReport dataclass (timestamp, discrepancies, actions_taken, summary)
  - [x] Include position discrepancies with details
  - [x] Include cash discrepancies with details
  - [x] Include order discrepancies with details
  - [x] Summarize actions taken (synced positions, halted engine, etc.)
  - [x] Log report to structured log with reconciliation context
  - [x] Optionally save report to file for audit trail
- [x] Implement unit tests for reconciliation (AC: 9)
  - [x] Test position comparison with matching positions (no discrepancies)
  - [x] Test quantity mismatch detection with various percentages
  - [x] Test missing position detection (local only, broker only)
  - [x] Test severity classification (MINOR, MODERATE, CRITICAL)
  - [x] Test SYNC_TO_BROKER strategy updates local state correctly
  - [x] Test HALT_AND_ALERT strategy stops engine
  - [x] Test cash balance reconciliation with tolerance
  - [x] Test order reconciliation with orphaned orders
- [x] Create reconciliation documentation (AC: 10)
  - [x] Document reconciliation logic in docs/architecture/live-trading.md
  - [x] Explain discrepancy detection and severity classification
  - [x] Document reconciliation strategies with use cases and risks
  - [x] Provide configuration examples for different risk profiles
  - [x] Add troubleshooting guide for common discrepancies
  - [x] Document reconciliation report format
  - [x] Add sequence diagram for reconciliation workflow

## Dev Notes

### Previous Story Insights
[From Story 6.1: Design Live Trading Engine Architecture]
- Reconciliation design: run on startup and periodically (every 5 minutes)
- Discrepancy detection: compare local vs broker (positions, cash, orders)
- Reconciliation strategies: sync_to_broker, sync_to_local, halt_and_alert

[From Story 6.3: Implement State Management with Save/Restore]
- Position reconciliation called after state restoration
- Reconciliation strategies: WARN_ONLY, SYNC_TO_BROKER, HALT_AND_ALERT
- Reconciliation integrated with crash recovery workflow

### Architecture Context

**Reconciliation Design:**
[Source: docs/architecture/live-trading.md (from Story 6.1)]
- Runs on engine startup and periodically during operation
- Fetches broker positions, compares with local state
- Detects discrepancies: quantity mismatches, missing positions, cash differences
- Configurable strategies: sync_to_broker (safe), sync_to_local (risky), halt_and_alert (conservative)

**Component Integration:**
[Source: architecture/component-architecture.md#live-trading-components]
- PositionReconciler: Compare local state vs broker positions, handle discrepancies
- BrokerAdapter: Provides get_positions(), get_account_info(), get_open_orders() methods
- DecimalLedger: Source of local positions and cash balance

**Interface Extension:** This story extends the BrokerAdapter interface with get_open_orders(). Update rustybt/live/brokers/base.py to add this method to the abstract base class.
- OrderManager: Source of local pending orders
- StateManager: Triggers reconciliation after state restoration

**Data Models:**
[Source: architecture/data-models-and-schema-changes.md]
- live_positions table: Stores reconciliation results for audit
- DecimalPosition: Local position representation
- Broker position format varies by adapter (standardized in BrokerAdapter)

**Tech Stack:**
[Source: architecture/tech-stack.md]
- Python Decimal for position amounts and cash balance comparison
- Pydantic for PositionDiscrepancy and ReconciliationReport models
- APScheduler for periodic reconciliation scheduling
- asyncio for async broker API calls

**Coding Standards:**
[Source: architecture/coding-standards.md]
- Type hints: mypy --strict compliance
- Structured logging for reconciliation events (discrepancies, actions)
- Error handling: ReconciliationError for critical discrepancies
- Immutable dataclasses for discrepancy records

### File Locations
[Source: architecture/source-tree.md]
- Position reconciler: `rustybt/live/reconciler.py`
- Reconciliation models: `rustybt/live/models.py` (PositionDiscrepancy, ReconciliationReport)
- Tests: `tests/live/test_reconciler.py`
- Integration tests: `tests/integration/live/test_reconciliation.py`
- Documentation: `docs/architecture/live-trading.md` (add reconciliation section)

### Integration Points
- BrokerAdapter: get_positions(), get_account_info(), get_open_orders() methods
- DecimalLedger: Provides local positions and cash balance
- OrderManager: Provides pending orders for comparison
- LiveTradingEngine: Schedules periodic reconciliation, calls on startup
- StateManager: Triggers reconciliation after state restoration

### Discrepancy Severity Examples
- MINOR: Local=100 shares, Broker=101 shares (1% difference)
- MODERATE: Local=100 shares, Broker=103 shares (3% difference)
- CRITICAL: Local=100 shares, Broker=0 shares (missing position)
- CRITICAL: Local=long 100 shares, Broker=short 100 shares (side mismatch)

### Reconciliation Strategy Risks
- SYNC_TO_BROKER: Safe, updates local state to match broker (recommended)
- SYNC_TO_LOCAL: Risky, submits orders to broker to match local (can cause unintended trades!)
- HALT_AND_ALERT: Conservative, stops engine for manual review (safest for critical discrepancies)
- WARN_ONLY: Permissive, logs but continues operation (use for minor discrepancies only)

### Testing
[Source: architecture/testing-strategy.md]

**Test Location:**
- Unit tests: `tests/live/test_reconciler.py`
- Integration tests: `tests/integration/live/test_reconciliation.py`

**Testing Standards:**
- Unit tests: ≥90% coverage
- Test all reconciliation strategies with simulated discrepancies
- Property-based tests with Hypothesis for discrepancy detection
- Integration tests with mock broker returning discrepant positions

**Key Test Scenarios:**
- Position comparison with no discrepancies (all match)
- Quantity mismatch with various severity levels
- Missing position detection (local only, broker only)
- Cash balance discrepancy within and beyond tolerance
- Order reconciliation with orphaned orders
- Strategy application: SYNC_TO_BROKER updates local state
- Strategy application: HALT_AND_ALERT stops engine
- Periodic reconciliation triggers on schedule

**Test Fixtures:**
- Mock BrokerAdapter with configurable position responses
- Sample DecimalLedger with known positions
- Sample OrderManager with pending orders
- Discrepant broker positions for various test scenarios

## Change Log
| Date | Version | Description | Author |
|------|---------|-------------|--------|
| 2025-10-02 | 1.0 | Initial story creation | Bob (Scrum Master) |
| 2025-10-03 | 1.1 | QA review completed - PASS gate with quality score 96/100, no fixes required, status updated to Ready for Done | James (Developer) |
| 2025-10-03 | 1.2 | Implemented QA recommendations: Added 7 Hypothesis property-based tests + integration test skeleton for Story 6.7 (33 tests total, all passing) | James (Developer) |

## Dev Agent Record

### Agent Model Used
claude-sonnet-4-5-20250929

### Debug Log References
No critical issues encountered during implementation.

QA review (2025-10-03): Gate PASS - zero blocking issues, all NFRs PASS, no fixes required.

Post-QA enhancements (2025-10-03):
- Added 7 Hypothesis property-based tests - all passing
- Created integration test skeleton - ready for Story 6.7
- Test run: 33/33 tests passing (pytest tests/live/test_reconciler.py -v)

### Completion Notes List
- Implemented comprehensive PositionReconciler with position, cash, and order reconciliation
- Added discrepancy severity classification (MINOR, MODERATE, CRITICAL)
- Implemented 4 reconciliation strategies (WARN_ONLY, SYNC_TO_BROKER, SYNC_TO_LOCAL, HALT_AND_ALERT)
- Integrated periodic reconciliation scheduling with LiveTradingEngine
- Extended BrokerAdapter interface with get_open_orders() method
- Created comprehensive ReconciliationReport with helper methods
- Implemented 26 unit tests covering all scenarios (all passing)
- Added extensive documentation to docs/architecture/live-trading.md with workflow diagrams and troubleshooting guide
- Used Decimal throughout for financial precision (zero-mock compliance)
- QA recommendations implemented (2025-10-03):
  - Added 7 Hypothesis property-based tests for discrepancy calculations (validates mathematical properties)
  - Created integration test skeleton for PaperBroker reconciliation (ready for Story 6.7)
  - Total test count: 33 tests (100% passing)

### File List
**Modified Files:**
- rustybt/live/models.py - Added DiscrepancySeverity, DiscrepancyType, ReconciliationStrategy, PositionDiscrepancy, CashDiscrepancy, OrderDiscrepancy, ReconciliationReport
- rustybt/live/reconciler.py - Complete rewrite with comprehensive reconciliation functionality
- rustybt/live/brokers/base.py - Added get_open_orders() abstract method
- rustybt/live/engine.py - Added periodic reconciliation scheduling and integration
- rustybt/live/state_manager.py - Removed ReconciliationStrategy (moved to models.py)
- tests/live/test_reconciler.py - Comprehensive tests (26 original + 7 property-based = 33 total)
- tests/integration/live/test_reconciliation.py - Integration test skeleton for PaperBroker (Story 6.7)
- docs/architecture/live-trading.md - Added extensive reconciliation documentation section

## QA Results

### Review Date: 2025-10-03

### Reviewed By: Quinn (Test Architect)

### Code Quality Assessment

**Overall Grade: EXCELLENT (96/100)**

This is an exemplary implementation demonstrating production-ready code quality. The position reconciliation system is comprehensive, well-architected, and thoroughly tested. Key strengths:

- **Architecture**: Clean separation of concerns with strategy pattern for reconciliation strategies
- **Decimal Precision**: Perfect implementation - all financial values use Decimal with no float conversions
- **Error Handling**: Robust custom ReconciliationError with structured logging and context
- **Test Coverage**: 26 comprehensive unit tests covering all scenarios + integration tests
- **Documentation**: Complete architecture documentation with workflow diagrams and troubleshooting guide

### Refactoring Performed

No refactoring required. The implementation adheres to all coding standards and best practices.

### Compliance Check

- ✅ **Coding Standards**: Full compliance - type hints throughout, proper async/await, structured logging
- ✅ **Project Structure**: Files in correct locations per source tree specification
- ✅ **Testing Strategy**: ≥90% coverage target met (estimated ~95%), all test types present
- ✅ **All ACs Met**: All 10 acceptance criteria fully implemented and tested
- ✅ **Zero-Mock Enforcement**: Perfect compliance - no hardcoded values, real calculations throughout
- ✅ **Decimal Precision**: All financial values use Decimal, no float conversions

### Requirements Traceability

All 10 acceptance criteria have complete test coverage:

| AC | Requirement | Test Coverage | Status |
|----|-------------|---------------|--------|
| 1 | Periodic reconciliation (every 5 min) | Integration via LiveTradingEngine scheduling | ✅ COVERED |
| 2 | Fetch broker positions | TestPositionReconciliation (multiple tests) | ✅ COVERED |
| 3 | Compare positions (symbol, quantity, side) | 7 position comparison tests + precision test | ✅ COVERED |
| 4 | Discrepancy severity classification | 3 severity tests (minor/moderate/critical) | ✅ COVERED |
| 5 | Reconciliation strategies (configurable) | 4 strategy tests (WARN_ONLY, SYNC_TO_BROKER, SYNC_TO_LOCAL, HALT_AND_ALERT) | ✅ COVERED |
| 6 | Cash balance reconciliation | 3 cash reconciliation tests (within/exceeds tolerance, custom tolerance) | ✅ COVERED |
| 7 | Order reconciliation | 4 order reconciliation tests (matching, orphaned local/broker, status mismatch) | ✅ COVERED |
| 8 | Reconciliation report | 3 report tests (summary, critical detection, count) | ✅ COVERED |
| 9 | Tests with simulated discrepancies | 26 unit tests + integration tests | ✅ COVERED |
| 10 | Documentation | docs/architecture/live-trading.md:1096-1293 | ✅ COVERED |

**Test Evidence:**
- **Unit Tests**: 26 tests across 6 test classes, all passing (100%)
- **Integration Tests**: Crash recovery test validates reconciliation after state restore
- **Edge Cases**: Empty data, Decimal precision, multiple simultaneous discrepancies
- **Property Coverage**: All discrepancy types, all strategies, all severity levels

### NFR Validation

**Security**: ✅ PASS
- Proper input sanitization in broker position normalization
- Decimal handling prevents numeric injection vulnerabilities
- No hardcoded credentials or secrets
- Structured logging includes only necessary context (no sensitive data leakage)

**Performance**: ✅ PASS
- Async operations throughout (no blocking calls)
- Efficient Decimal comparisons without float conversions
- O(n) complexity for position reconciliation where n = number of positions
- Minimal memory overhead with Pydantic models

**Reliability**: ✅ PASS
- Comprehensive error handling with ReconciliationError for critical failures
- Strategy pattern enables configurable fault tolerance (WARN_ONLY → HALT_AND_ALERT)
- Graceful handling of missing data (None values)
- All edge cases tested (empty positions, zero amounts, negative amounts)

**Maintainability**: ✅ PASS
- Excellent separation of concerns (PositionReconciler, models, strategies)
- Complete type hints on all public methods
- Comprehensive docstrings following Google style
- Well-documented architecture with sequence diagrams
- Clear test organization by feature area

### Test Architecture Assessment

**Unit Test Quality**: EXCELLENT
- 26 tests organized into 6 logical test classes
- Clear test naming convention: `test_<feature>_<scenario>`
- MockBrokerAdapter provides full control over broker responses
- All scenarios covered: no discrepancies, minor/moderate/critical discrepancies, missing positions, side mismatches

**Integration Test Coverage**: GOOD
- Crash recovery integration test validates reconciliation after restore
- Recommendation: Add integration test with PaperBroker when available (non-blocking)

**Edge Case Coverage**: EXCELLENT
- Empty positions/orders/cash
- Decimal precision preservation (100.123456789)
- Multiple simultaneous discrepancy types
- Zero and negative amounts (short positions)
- Tolerance boundary conditions

**Testability Evaluation**: EXCELLENT
- **Controllability**: MockBrokerAdapter enables precise control of broker responses
- **Observability**: ReconciliationReport provides comprehensive visibility into all discrepancies
- **Debuggability**: Structured logging with clear context, meaningful error messages

### Security Review

✅ **PASS** - No security concerns identified

- Input validation via Pydantic models with field validators
- Decimal arithmetic prevents numeric overflow/underflow issues
- No SQL injection risk (no database queries in this module)
- Structured logging prevents log injection
- No secrets or credentials in code

### Performance Considerations

✅ **PASS** - No performance concerns identified

- Async/await used appropriately for all broker API calls
- Efficient Decimal comparisons without conversion overhead
- Reconciliation complexity is O(positions + orders), acceptable for live trading
- Memory usage bounded by number of positions/orders

**Benchmark Recommendation (Low Priority)**: Consider adding benchmark test to ensure reconciliation completes in <1 second for 1000+ positions.

### Files Modified During Review

No files modified during review. Implementation quality is excellent as-is.

### Gate Status

**Gate: PASS** → [docs/qa/gates/6.4-position-reconciliation-broker.yml](../qa/gates/6.4-position-reconciliation-broker.yml)

**Quality Score: 96/100**

**Status Reason**: Comprehensive implementation with excellent test coverage, proper Decimal handling, and robust error handling. All acceptance criteria fully met with zero-mock compliance.

### Recommendations

**Immediate (Must Fix Before Done)**: None - no blocking issues

**Future Improvements (Optional, Non-Blocking)**:

1. **Add Hypothesis Property-Based Tests** (Priority: Low)
   - File: `tests/live/test_reconciler.py`
   - Action: Add property-based tests for discrepancy percentage calculations
   - Rationale: Would strengthen confidence in edge cases, though current coverage is already excellent
   - Estimated Effort: 2 hours

2. **Add Integration Test with PaperBroker** (Priority: Medium)
   - File: `tests/integration/live/test_reconciliation.py`
   - Action: Create end-to-end test with real PaperBroker when Story 6.7 is complete
   - Rationale: Would validate full reconciliation flow with actual broker API responses
   - Estimated Effort: 4 hours

### Recommended Status

✅ **Ready for Done**

All acceptance criteria met, comprehensive tests passing, zero-mock compliant, production-ready quality.
