# Story 11.6: API Export Gap Analysis

## Artifact Metadata
- **Story**: 11.6 User-Facing Documentation Quality Validation
- **Phase**: Phase 0 - API Export Bug Fix
- **Task**: Task 0.1 - Analyze API Export Structure
- **Created**: 2025-10-17
- **Author**: James (Dev Agent)
- **Status**: Complete

## Executive Summary

**Critical Bug Confirmed**: User-facing documentation shows import patterns that **FAIL AT RUNTIME**:

```python
from rustybt.api import order_target, record, symbol
# ImportError: cannot import name 'order_target' from 'rustybt.api'
```

**Root Cause**: Circular dependency issue between `rustybt.api` module load order and `@api_method` decorator registration.

**Impact**: ALL new users attempting Quick Start tutorial encounter ImportError on first line of code.

---

## Current State Analysis

### rustybt/api.py Static Exports

The `rustybt/api.py` module defines a static `__all__` list (lines 35-54) containing **ONLY 14 items**:

```python
__all__ = [
    "RESTRICTION_STATES",
    "EODCancel",
    "FixedBasisPointsSlippage",
    "FixedSlippage",
    "HistoricalRestrictions",
    "NeverCancel",
    "Restriction",
    "StaticRestrictions",
    "VolumeShareSlippage",
    "calendars",
    "cancel_policy",
    "commission",
    "date_rules",
    "events",
    "execution",
    "math_utils",
    "slippage",
    "time_rules",
]
```

**Missing from `__all__`**:
- `order_target` ❌
- `order` ❌
- `record` ❌
- `symbol` ❌
- `schedule_function` ❌
- `order_percent` ❌
- `order_value` ❌
- `order_target_percent` ❌
- `order_target_value` ❌
- `get_open_orders` ❌
- `get_order` ❌
- `cancel_order` ❌
- `sid` ❌
- `symbols` ❌
- `future_symbol` ❌
- `continuous_future` ❌
- `get_datetime` ❌
- `get_environment` ❌
- `fetch_csv` ❌
- `attach_pipeline` ❌
- `pipeline_output` ❌
- `set_benchmark` ❌
- `set_slippage` ❌
- `set_commission` ❌
- `set_cancel_policy` ❌
- `set_symbol_lookup_date` ❌
- `set_max_leverage` ❌
- `set_min_leverage` ❌
- `set_max_position_size` ❌
- `set_max_order_size` ❌
- `set_max_order_count` ❌
- `set_do_not_order_list` ❌
- `set_asset_restrictions` ❌
- `set_long_only` ❌
- `batch_market_order` ❌

### rustybt/api.pyi Type Stubs

The `rustybt/api.pyi` file (829 lines) contains type stub definitions for **ALL** the missing functions above. This creates a **disconnect**:
- Type checkers (mypy, pyright) see the functions as available
- Runtime imports **FAIL**
- Users get confusing error: "mypy says it's fine, but Python says ImportError"

### Dynamic Registration Mechanism

The `rustybt/utils/api_support.py` defines the `@api_method` decorator (lines 44-61):

```python
def api_method(f):
    # Decorator that adds the decorated class method as a callable
    # function (wrapped) to rustybt.api
    @wraps(f)
    def wrapped(*args, **kwargs):
        # Get the instance and call the method
        algo_instance = get_algo_instance()
        if algo_instance is None:
            raise RuntimeError(
                "rustybt api method %s must be called during a simulation." % f.__name__
            )
        return getattr(algo_instance, f.__name__)(*args, **kwargs)

    # Add functor to rustybt.api
    setattr(rustybt.api, f.__name__, wrapped)
    rustybt.api.__all__.append(f.__name__)  # ← Dynamically adds to __all__
    f.is_api_method = True
    return f
```

**Key Line 59**: `rustybt.api.__all__.append(f.__name__)` - Functions are added **DYNAMICALLY** when the decorator runs.

### Functions Using @api_method Decorator

Grep of `rustybt/algorithm.py` shows 38 methods decorated with `@api_method`:

1. `get_environment` (line 895)
2. `fetch_csv` (line 954)
3. `schedule_function` (line 1059)
4. `record` (line 1132)
5. `set_benchmark` (line 1158)
6. `continuous_future` (line 1177)
7. `symbol` (line 1209)
8. `symbols` (line 1253)
9. `sid` (line 1282)
10. `future_symbol` (line 1303)
11. `order` (line 1388)
12. `order_value` (line 1523)
13. `get_datetime` (line 1623)
14. `set_slippage` (line 1653)
15. `set_commission` (line 1694)
16. `set_cancel_policy` (line 1737)
17. `set_symbol_lookup_date` (line 1759)
18. `order_percent` (line 1786)
19. `order_target` (line 1838)
20. `order_target_value` (line 1909)
21. `order_target_percent` (line 1975)
22. `batch_market_order` (line 2044)
23. `get_open_orders` (line 2068)
24. `get_order` (line 2097)
25. `cancel_order` (line 2115)
26. `set_max_leverage` (line 2151)
27. `set_min_leverage` (line 2164)
28. `set_max_position_size` (line 2191)
29. `set_max_order_size` (line 2224)
30. `set_max_order_count` (line 2251)
31. `set_do_not_order_list` (line 2264)
32. `set_asset_restrictions` (line 2296)
33. `set_long_only` (line 2317)
34. `attach_pipeline` (line 2327)
35. `pipeline_output` (line 2378)

---

## Documentation Usage Analysis

### Home Page (docs/index.md)

**Lines 48-64** - Quick Start example:

```python
from rustybt.api import order_target, record, symbol  # ← FAILS

def initialize(context):
    context.asset = symbol('AAPL')

def handle_data(context, data):
    short_mavg = data.history(context.asset, 'price',
                              bar_count=100, frequency="1d").mean()
    long_mavg = data.history(context.asset, 'price',
                             bar_count=300, frequency="1d").mean()

    if short_mavg > long_mavg:
        order_target(context.asset, 100)
    elif short_mavg < long_mavg:
        order_target(context.asset, 0)
```

**Functions Used**:
- `symbol` - ❌ Not in static `__all__`
- `order_target` - ❌ Not in static `__all__`
- Implicit: `record` (shown in import but not used in snippet) - ❌ Not in static `__all__`

### Quick Start (docs/getting-started/quickstart.md)

**Line 20** - Tutorial example:

```python
from rustybt.api import order_target, record, symbol  # ← FAILS
```

**Functions Used**: Same as Home page.

---

## Root Cause Analysis

### The Circular Dependency Problem

**Module Load Order**:

1. **User code executes**: `from rustybt.api import order_target`
2. **Python imports rustybt.api**: Runs `rustybt/api.py`
3. **rustybt/api.py loads**: Defines static `__all__` list (14 items), does NOT import `algorithm`
4. **Python checks `rustybt.api.__all__`**: Does not find `order_target`
5. **ImportError raised**: `cannot import name 'order_target' from 'rustybt.api'`

**Why `@api_method` Doesn't Run**:

- The `@api_method` decorator only runs when `rustybt.algorithm` module is loaded
- `rustybt.algorithm` is NOT imported by `rustybt.api` (to avoid circular imports)
- Therefore, decorated methods are never registered in `rustybt.api.__all__`
- Direct imports like `from rustybt.api import order_target` fail

### Evidence: Import Test

```bash
$ python3 -c "from rustybt.api import order_target"
ImportError: cannot import name 'order_target' from 'rustybt.api' (/path/to/rustybt/api.py)
```

### Why This Worked in Zipline

Original Zipline likely:
1. Always imported algorithm module during package initialization
2. OR used a different import pattern in documentation
3. OR had explicit exports in `__all__`

---

## Gap Summary

### Functions Documented But Not Exported

**Critical (used in Quick Start)**:
- `order_target` - Core trading function
- `symbol` - Asset lookup
- `record` - Metrics tracking

**High Priority (commonly used)**:
- `order` - Basic order placement
- `order_percent` - Percentage-based ordering
- `order_value` - Value-based ordering
- `schedule_function` - Scheduling handlers
- `get_open_orders` - Order management
- `cancel_order` - Order cancellation

**Medium Priority (advanced features)**:
- `sid` - Asset lookup by ID
- `symbols` - Batch asset lookup
- `future_symbol` - Futures lookup
- `continuous_future` - Continuous futures
- `attach_pipeline` - Pipeline registration
- `pipeline_output` - Pipeline results
- `get_datetime` - Simulation time
- `get_environment` - Environment info
- `fetch_csv` - CSV data loading

**Lower Priority (configuration functions)**:
- `set_*` functions (15 total) - Usually called in initialize()
- `get_order` - Order lookup
- `batch_market_order` - Batch ordering

### Total Export Gap

- **Static `__all__`**: 14 items
- **Functions with `@api_method`**: 35 functions
- **Total expected exports**: ~49 items
- **Current exports**: 14 items
- **Missing exports**: ~35 functions

---

## Backward Compatibility Analysis

### Current Usage Patterns

**Pattern 1: Direct Import (BROKEN)**:
```python
from rustybt.api import order_target, symbol  # ← FAILS
```

**Pattern 2: Module Import (WORKS)**:
```python
import rustybt.api as api
# Later in strategy:
api.order_target(asset, 100)  # ← Works if algorithm loaded first
```

**Pattern 3: Algorithm Method (WORKS)**:
```python
class MyStrategy(TradingAlgorithm):
    def handle_data(self, context, data):
        self.order_target(asset, 100)  # ← Always works
```

### Breaking Change Risk

**LOW RISK**: The fix will make previously-broken imports work. No existing working code will break.

**Scenarios**:
1. **User has Pattern 1**: Currently broken, will be fixed ✅
2. **User has Pattern 2**: Currently works, will continue to work ✅
3. **User has Pattern 3**: Currently works, will continue to work ✅

---

## Recommended Solution

### Option A: Lazy Import Trigger (RECOMMENDED)

**Implementation**: Add import of `algorithm` at end of `rustybt/api.py`:

```python
# At end of rustybt/api.py (after __all__ definition):

# Trigger @api_method decorator registration by importing algorithm module
# This ensures all API methods are available for direct import
from rustybt import algorithm as _algorithm  # noqa: F401
```

**Pros**:
- ✅ Minimal code change (1 line)
- ✅ No breaking changes
- ✅ Preserves dynamic registration mechanism
- ✅ Works with all usage patterns
- ✅ Maintains separation of concerns

**Cons**:
- ❌ Adds slight import overhead (algorithm module always loads)
- ❌ Doesn't make dependencies explicit

### Option B: Explicit Static Exports

**Implementation**: Manually add all 35 functions to `__all__` in `rustybt/api.py`:

```python
__all__ = [
    # Existing static exports
    "RESTRICTION_STATES",
    "EODCancel",
    # ... (14 existing)

    # API methods (to be dynamically registered)
    "order",
    "order_target",
    "symbol",
    # ... (35 more)
]
```

**Pros**:
- ✅ Explicit and clear
- ✅ No hidden module loading

**Cons**:
- ❌ Duplicates information (list maintained in 2 places)
- ❌ Easy to get out of sync
- ❌ Still requires algorithm module to load for functions to work

### Option C: Update Documentation Import Pattern

**Implementation**: Change docs to use Algorithm method pattern:

```python
# OLD (broken):
from rustybt.api import order_target, symbol

# NEW:
from rustybt.algorithm import TradingAlgorithm

class MyStrategy(TradingAlgorithm):
    def handle_data(self, context, data):
        self.order_target(...)
```

**Pros**:
- ✅ Works without code changes
- ✅ More explicit about algorithm usage

**Cons**:
- ❌ Breaks Quick Start simplicity
- ❌ Doesn't match typical Zipline usage
- ❌ Inconsistent with API documentation

---

## Recommendation

**Implement Option A**: Add lazy import trigger in `rustybt/api.py`.

**Rationale**:
1. Fixes bug with minimal code change
2. Makes documentation imports work as expected
3. No breaking changes to existing code
4. Preserves the elegant `@api_method` pattern
5. Aligns with user expectations from Zipline

---

## Next Steps

1. **Implement fix** (Task 0.3): Add algorithm import to `rustybt/api.py`
2. **Create test** (Task 0.3): Verify all documented imports work
3. **Verify backward compat** (Task 0.4): Run existing test suite
4. **Update docs** (Phase 1): No changes needed if fix works
5. **Document pattern** (Phase 1): Add note about import patterns to Getting Started

---

## References

- [Source: rustybt/api.py](../../rustybt/api.py)
- [Source: rustybt/api.pyi](../../rustybt/api.pyi)
- [Source: rustybt/utils/api_support.py](../../rustybt/utils/api_support.py)
- [Source: rustybt/algorithm.py](../../rustybt/algorithm.py)
- [Source: docs/index.md](../../docs/index.md#your-first-backtest)
- [Source: docs/getting-started/quickstart.md](../../docs/getting-started/quickstart.md#your-first-strategy)
