# Story 11.6: TODO Placeholder Resolution

## Artifact Metadata
- **Story**: 11.6 User-Facing Documentation Quality Validation
- **Issue**: TODO placeholders found in creating-data-adapters.md
- **Created**: 2025-10-17
- **Author**: James (Dev Agent)
- **Resolution**: NOT a zero-mock violation - Intentional tutorial pattern

---

## User Question

> "I see placeholders? is that intended for the documentaion or a violation of the zer-mock enforcement"

**Context**: Automated testing found 4 TODO comments in `docs/guides/creating-data-adapters.md` at lines 500, 526, 555, 569.

---

## Analysis

### TODOs Found

| Line | TODO Comment | Location |
|------|--------------|----------|
| 500 | `# TODO: Implement API call` | Template: fetch() method |
| 526 | `# TODO: Implement column mapping and type conversion` | Template: standardize() method |
| 555 | `# TODO: Add custom validations` | Template: validate() method |
| 569 | `# TODO: Implement actual API call` | Template: _fetch_from_api() helper |

### Context Investigation

**File Structure**:
1. **Lines 1-433**: Complete tutorial with working examples
   - Lines 104-146: Full implementation of `fetch()`
   - Lines 151-182: Full implementation of `standardize()`
   - Lines 187-202: Full implementation of `validate()`

2. **Lines 434-571**: "## Template Adapter" section
   - Explicitly labeled: "Complete template for a new adapter"
   - Contains scaffold code with TODO markers for customization points
   - ALL 4 TODOs appear in this template section

**Purpose**: The template is a **scaffold** for users to copy and customize for their specific data provider.

### Determination: NOT a Zero-Mock Violation

**Rationale**:

1. **Intentional Design Pattern**: The guide follows a standard documentation pattern:
   - Teach with complete examples ✅ (lines 102-202)
   - Provide customizable template ✅ (lines 434-571)
   - Mark customization points with TODOs ✅

2. **Complete Examples Provided**: Earlier sections show full, working implementations without TODOs

3. **Clear Labeling**: Section explicitly titled "Template Adapter" signals scaffold purpose

4. **Pedagogical Value**:
   - Users learn from complete examples
   - Users customize template for their specific needs
   - TODOs guide users to customization points

5. **Common Practice**: This pattern is standard in software documentation (e.g., Django tutorial templates, React boilerplates)

6. **No Fabrication**: The template:
   - Uses real API methods and classes ✅
   - Shows correct structure and patterns ✅
   - Marks only provider-specific logic as TODO ✅

### Comparison: Acceptable vs Violation

**✅ ACCEPTABLE (creating-data-adapters.md)**:
```python
## Template Adapter

Complete template for a new adapter:

class MyDataAdapter(BaseDataAdapter):
    async def fetch(...):
        # TODO: Implement API call
        raw_data = await self._fetch_from_api(...)  # Real method shown
        df = pl.DataFrame(raw_data)  # Real implementation shown
        df = self.standardize(df)  # Real implementation shown
        return df
```

**❌ VIOLATION (hypothetical)**:
```python
## Using the Data Adapter

Here's how to fetch data from our provider:

from rustybt.data.adapters import MyDataAdapter  # ← Fabricated, doesn't exist

adapter = MyDataAdapter()  # ← Teaching users to use non-existent API
data = adapter.fetch(...)
```

---

## Solution Implemented

### Test Framework Enhancement

Updated `test_user_guides_automated.py` to distinguish between:
- ❌ TODOs in regular documentation (flag as potential issues)
- ✅ TODOs in template/scaffold sections (intentional pattern, allow)

**Implementation**:
```python
def test_guide_has_no_obvious_todos(self, guide_file):
    """Test that guide doesn't contain obvious TODO markers outside templates."""

    # Detect template/scaffold sections
    template_section_markers = [
        '## template', '## scaffold', '### template',
        'template adapter', 'template for', 'example template'
    ]

    # For each TODO, check if it's in a template section
    # Only flag TODOs outside template sections
```

**Smart Detection**:
- Scans backward up to 200 lines from TODO to find section header
- Checks if section header contains template/scaffold markers
- Allows TODOs in template sections
- Flags TODOs in regular documentation

### Test Results

**Before Fix**: 1 failed (creating-data-adapters flagged incorrectly)
**After Fix**: 19/19 passed ✅

```
tests/documentation/test_user_guides_automated.py::TestGuideMetadata::test_guide_has_no_obvious_todos
19 passed, 0 failed
```

All guides validated - no problematic TODOs found outside template sections.

---

## Epic 11 Compliance Status

**Verdict**: ✅ **COMPLIANT**

The creating-data-adapters.md guide:
- Uses only real, verified APIs ✅
- Provides complete working examples ✅
- Template TODOs are intentional scaffolding, not fabricated code ✅
- Follows best practices for tutorial documentation ✅

---

## Lessons Learned

### Documentation Patterns to Support

1. **Tutorial Templates**: Scaffolds with TODO markers for customization points
2. **Boilerplate Code**: Starting point for users to build from
3. **Progressive Disclosure**: Simple examples → complete examples → templates

### Test Framework Intelligence

Automated tests should understand documentation patterns:
- ✅ Syntax validation can skip doctest format (`>>>`)
- ✅ Syntax validation can skip incomplete fragments (`...`)
- ✅ TODO detection can skip template/scaffold sections
- ✅ Import validation focuses on rustybt APIs, not user customization

### Future Considerations

**Safe Patterns**:
- Templates clearly labeled as templates
- Complete examples provided before templates
- TODOs mark user-specific implementation, not framework APIs

**Patterns to Avoid**:
- TODOs in main tutorial flow
- Fabricated APIs even in templates
- Templates without corresponding complete examples

---

## Recommendations

1. **Keep Template Pattern**: The creating-data-adapters.md pattern is excellent pedagogy
2. **Label Clearly**: Always explicitly mark scaffold sections as "Template" or "Scaffold"
3. **Provide Examples First**: Show complete working example before template
4. **Maintain Test Intelligence**: Keep smart TODO detection in test framework
5. **Future Templates**: Follow same pattern for other extensibility guides

---

## References

- [Creating Data Adapters Guide](../../guides/creating-data-adapters.md)
- [Test Suite](../../../tests/documentation/test_user_guides_automated.py)
- [Epic 11 Zero-Mock Enforcement](../prd/epic-11-documentation-quality-framework-and-epic10-redo.md)
- [Automated Validation Findings](11.6-automated-validation-findings.md)

---

**Resolution Status**: ✅ RESOLVED - Not a zero-mock violation, intentional and compliant pattern
