"""Unit tests for BacktestArtifactManager.

Tests cover:
- ID generation format and uniqueness
- Thread-safe concurrent ID generation
- Directory structure creation
- Write permission validation
- Configuration handling
- Error conditions
"""

import os
import re
import sys
import threading
import time
from concurrent.futures import ThreadPoolExecutor
from pathlib import Path
from unittest.mock import MagicMock, Mock, patch

import pytest

from rustybt.backtest.artifact_manager import (
    BacktestArtifactError,
    BacktestArtifactManager,
)


class TestBacktestIDGeneration:
    """Test backtest ID generation."""

    def test_generate_backtest_id_format(self, tmp_path):
        """Verify backtest ID matches YYYYMMDD_HHMMSS_mmm format."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        backtest_id = manager.generate_backtest_id()

        # Verify format: YYYYMMDD_HHMMSS_mmm (19 characters)
        assert len(backtest_id) == 19
        assert re.match(r"^\d{8}_\d{6}_\d{3}$", backtest_id)

        # Verify components are valid
        date_part = backtest_id[:8]
        time_part = backtest_id[9:15]
        millis_part = backtest_id[16:]

        # Date: YYYYMMDD
        year = int(date_part[:4])
        month = int(date_part[4:6])
        day = int(date_part[6:8])
        assert 2020 <= year <= 2100
        assert 1 <= month <= 12
        assert 1 <= day <= 31

        # Time: HHMMSS
        hour = int(time_part[:2])
        minute = int(time_part[2:4])
        second = int(time_part[4:6])
        assert 0 <= hour <= 23
        assert 0 <= minute <= 59
        assert 0 <= second <= 59

        # Milliseconds: mmm
        millis = int(millis_part)
        assert 0 <= millis <= 999

    def test_generate_backtest_id_uniqueness(self, tmp_path):
        """Verify consecutive ID generations produce unique IDs."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        ids = set()

        # Generate 100 IDs rapidly
        for _ in range(100):
            backtest_id = manager.generate_backtest_id()
            ids.add(backtest_id)
            # Small delay to ensure time progression
            time.sleep(0.001)

        # All IDs should be unique
        assert len(ids) == 100

    def test_backtest_id_property(self, tmp_path):
        """Verify backtest_id property returns generated ID."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))

        # Initially None
        assert manager.backtest_id is None

        # After generation
        generated_id = manager.generate_backtest_id()
        assert manager.backtest_id == generated_id

    def test_thread_safe_id_generation(self, tmp_path):
        """Verify concurrent ID generation produces unique IDs without race conditions."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        ids = []
        ids_lock = threading.Lock()

        def generate_id():
            """Generate ID and store in shared list."""
            # Create new manager instance for each thread to avoid sharing state
            thread_manager = BacktestArtifactManager(base_dir=str(tmp_path))
            backtest_id = thread_manager.generate_backtest_id()
            with ids_lock:
                ids.append(backtest_id)

        # Spawn 10 threads generating IDs simultaneously
        threads = []
        for _ in range(10):
            thread = threading.Thread(target=generate_id)
            threads.append(thread)
            thread.start()

        # Wait for all threads to complete
        for thread in threads:
            thread.join()

        # Verify all 10 IDs are unique
        assert len(ids) == 10
        assert len(set(ids)) == 10


class TestDirectoryStructureCreation:
    """Test directory structure creation."""

    def test_create_directory_structure_success(self, tmp_path):
        """Verify directory structure created with all subdirectories."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        output_dir = manager.create_directory_structure()

        # Verify output directory exists
        assert output_dir.exists()
        assert output_dir.is_dir()

        # Verify subdirectories exist
        assert (output_dir / "results").exists()
        assert (output_dir / "results").is_dir()
        assert (output_dir / "code").exists()
        assert (output_dir / "code").is_dir()
        assert (output_dir / "metadata").exists()
        assert (output_dir / "metadata").is_dir()

    def test_create_directory_structure_generates_id(self, tmp_path):
        """Verify create_directory_structure generates ID if not already generated."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))

        # ID not generated yet
        assert manager.backtest_id is None

        # Create directory structure
        output_dir = manager.create_directory_structure()

        # ID should now be generated
        assert manager.backtest_id is not None
        assert output_dir.name == manager.backtest_id

    def test_create_directory_structure_uses_existing_id(self, tmp_path):
        """Verify create_directory_structure uses existing ID if already generated."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))

        # Generate ID first
        generated_id = manager.generate_backtest_id()

        # Create directory structure
        output_dir = manager.create_directory_structure()

        # Should use the same ID
        assert output_dir.name == generated_id
        assert manager.backtest_id == generated_id

    def test_output_dir_property(self, tmp_path):
        """Verify output_dir property returns created directory path."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))

        # Initially None
        assert manager.output_dir is None

        # After creation
        output_dir = manager.create_directory_structure()
        assert manager.output_dir == output_dir


class TestDirectoryValidation:
    """Test directory validation and permissions."""

    def test_validate_base_directory_creates_if_missing(self, tmp_path):
        """Verify base directory created if it doesn't exist."""
        base_dir = tmp_path / "new_backtests"
        assert not base_dir.exists()

        manager = BacktestArtifactManager(base_dir=str(base_dir))

        # Base directory should be created
        assert base_dir.exists()
        assert base_dir.is_dir()

    def test_validate_base_directory_readonly_fails(self, tmp_path):
        """Verify initialization fails if base directory is not writable."""
        base_dir = tmp_path / "readonly_backtests"
        base_dir.mkdir()

        # Make directory read-only
        os.chmod(base_dir, 0o444)

        try:
            # Should raise BacktestArtifactError
            with pytest.raises(BacktestArtifactError, match="not writable"):
                BacktestArtifactManager(base_dir=str(base_dir))
        finally:
            # Restore permissions for cleanup
            os.chmod(base_dir, 0o755)

    def test_validate_base_directory_creation_fails(self, tmp_path):
        """Verify initialization fails if base directory cannot be created."""
        # Create a file where we want a directory
        base_dir = tmp_path / "file_not_dir"
        base_dir.write_text("blocking file")

        # Should raise BacktestArtifactError
        with pytest.raises(BacktestArtifactError, match="Failed to create base directory"):
            BacktestArtifactManager(base_dir=str(base_dir))


class TestDisabledArtifactManagement:
    """Test behavior when artifact management is disabled."""

    def test_disabled_artifact_manager(self, tmp_path):
        """Verify disabled manager doesn't create directories."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path), enabled=False)

        # Should raise error when trying to create directory structure
        with pytest.raises(BacktestArtifactError, match="disabled"):
            manager.create_directory_structure()

    def test_disabled_manager_skips_validation(self, tmp_path):
        """Verify disabled manager skips directory validation."""
        # Use non-existent path
        base_dir = tmp_path / "nonexistent"

        # Should not raise error since validation is skipped when disabled
        manager = BacktestArtifactManager(base_dir=str(base_dir), enabled=False)

        # Base directory should NOT be created
        assert not base_dir.exists()


class TestSubdirectoryAccessors:
    """Test subdirectory accessor methods."""

    def test_get_results_dir(self, tmp_path):
        """Verify get_results_dir returns correct path."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        output_dir = manager.create_directory_structure()

        results_dir = manager.get_results_dir()
        assert results_dir == output_dir / "results"
        assert results_dir.exists()

    def test_get_code_dir(self, tmp_path):
        """Verify get_code_dir returns correct path."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        output_dir = manager.create_directory_structure()

        code_dir = manager.get_code_dir()
        assert code_dir == output_dir / "code"
        assert code_dir.exists()

    def test_get_metadata_dir(self, tmp_path):
        """Verify get_metadata_dir returns correct path."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        output_dir = manager.create_directory_structure()

        metadata_dir = manager.get_metadata_dir()
        assert metadata_dir == output_dir / "metadata"
        assert metadata_dir.exists()

    def test_get_dirs_before_creation_fails(self, tmp_path):
        """Verify accessor methods fail if directory structure not created."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))

        with pytest.raises(BacktestArtifactError, match="not created"):
            manager.get_results_dir()

        with pytest.raises(BacktestArtifactError, match="not created"):
            manager.get_code_dir()

        with pytest.raises(BacktestArtifactError, match="not created"):
            manager.get_metadata_dir()


class TestConcurrentBacktestExecution:
    """Test concurrent backtest execution scenarios."""

    def test_concurrent_directory_creation(self, tmp_path):
        """Verify multiple backtests can create directories concurrently."""
        output_dirs = []
        dirs_lock = threading.Lock()

        def create_backtest_dir():
            """Create backtest directory and store path."""
            manager = BacktestArtifactManager(base_dir=str(tmp_path))
            output_dir = manager.create_directory_structure()
            with dirs_lock:
                output_dirs.append(output_dir)

        # Run 5 concurrent backtest directory creations
        with ThreadPoolExecutor(max_workers=5) as executor:
            futures = [executor.submit(create_backtest_dir) for _ in range(5)]
            for future in futures:
                future.result()

        # Verify all 5 directories were created with unique names
        assert len(output_dirs) == 5
        assert len(set(str(d) for d in output_dirs)) == 5

        # Verify all directories exist
        for output_dir in output_dirs:
            assert output_dir.exists()
            assert (output_dir / "results").exists()
            assert (output_dir / "code").exists()
            assert (output_dir / "metadata").exists()


class TestConfiguration:
    """Test configuration handling."""

    def test_custom_base_dir(self, tmp_path):
        """Verify custom base directory is used."""
        custom_dir = tmp_path / "custom_backtests"
        manager = BacktestArtifactManager(base_dir=str(custom_dir))

        output_dir = manager.create_directory_structure()

        # Verify output directory is under custom base directory
        assert output_dir.parent == custom_dir
        assert custom_dir.exists()

    def test_absolute_path_base_dir(self, tmp_path):
        """Verify absolute path for base directory works."""
        abs_dir = tmp_path / "absolute_backtests"
        manager = BacktestArtifactManager(base_dir=str(abs_dir.absolute()))

        output_dir = manager.create_directory_structure()

        assert output_dir.exists()
        assert output_dir.parent == abs_dir

    def test_relative_path_base_dir(self, tmp_path, monkeypatch):
        """Verify relative path for base directory resolves to ~/.zipline/."""
        from rustybt.utils.paths import zipline_root

        manager = BacktestArtifactManager(base_dir="relative_backtests")
        output_dir = manager.create_directory_structure()

        # Verify directory created under ~/.zipline/ (not current directory)
        assert output_dir.exists()
        zipline_dir = Path(zipline_root())
        expected_base = zipline_dir / "relative_backtests"
        assert str(manager.base_dir) == str(expected_base)
        assert manager.base_dir.exists()


class TestGetOutputPath:
    """Test get_output_path method."""

    def test_get_output_path_simple_filename(self, tmp_path):
        """Verify get_output_path with simple filename returns correct path."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.create_directory_structure()

        output_path = manager.get_output_path("backtest_results.csv")

        # Should be in results subdirectory by default
        assert output_path.name == "backtest_results.csv"
        assert output_path.parent.name == "results"
        assert output_path.parent.parent.name == manager.backtest_id

    def test_get_output_path_nested_filename(self, tmp_path):
        """Verify get_output_path with nested path creates parent directories."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.create_directory_structure()

        output_path = manager.get_output_path("reports/basic_report.html")

        # Should create nested directory structure
        assert output_path.name == "basic_report.html"
        assert output_path.parent.name == "reports"
        assert output_path.parent.parent.name == "results"
        assert output_path.parent.exists()

    def test_get_output_path_custom_subdir(self, tmp_path):
        """Verify get_output_path with custom subdir parameter."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.create_directory_structure()

        output_path = manager.get_output_path("strategy.py", subdir="code")

        # Should be in code subdirectory
        assert output_path.name == "strategy.py"
        assert output_path.parent.name == "code"
        assert output_path.parent.parent.name == manager.backtest_id

    def test_get_output_path_nested_subdir(self, tmp_path):
        """Verify get_output_path with nested subdir parameter."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.create_directory_structure()

        output_path = manager.get_output_path("report.html", subdir="results/reports")

        # Should handle nested subdir
        assert output_path.name == "report.html"
        assert "reports" in str(output_path)
        assert output_path.parent.exists()

    def test_get_output_path_creates_parent_dirs(self, tmp_path):
        """Verify get_output_path creates parent directories if they don't exist."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.create_directory_structure()

        # Path with multiple nested levels
        output_path = manager.get_output_path("deep/nested/structure/file.txt", subdir="results")

        # All parent directories should be created
        assert output_path.parent.exists()
        assert output_path.parent.name == "structure"
        assert output_path.parent.parent.name == "nested"

    def test_get_output_path_before_directory_creation_fails(self, tmp_path):
        """Verify get_output_path fails if directory structure not created."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))

        with pytest.raises(BacktestArtifactError, match="not created"):
            manager.get_output_path("file.csv")

    def test_get_output_path_returns_absolute_path(self, tmp_path):
        """Verify get_output_path returns absolute path."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.create_directory_structure()

        output_path = manager.get_output_path("results.csv")

        # Should be absolute path
        assert output_path.is_absolute()

    def test_get_output_path_multiple_calls_same_file(self, tmp_path):
        """Verify multiple calls with same filename return same path."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.create_directory_structure()

        path1 = manager.get_output_path("results.csv")
        path2 = manager.get_output_path("results.csv")

        assert path1 == path2

    def test_get_output_path_different_files(self, tmp_path):
        """Verify different filenames return different paths."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.create_directory_structure()

        path1 = manager.get_output_path("results.csv")
        path2 = manager.get_output_path("summary.csv")

        assert path1 != path2
        assert path1.parent == path2.parent  # Same parent directory


class TestMetadataGeneration:
    """Test metadata generation functionality."""

    def test_generate_metadata_all_fields(self, tmp_path):
        """Verify metadata contains all required fields."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        strategy_entry = Path("/path/to/strategy.py")
        captured_files = [Path("strategy.py"), Path("utils/indicators.py")]
        data_bundle_info = {"bundle_name": "test", "dataset_ids": ["ds-1"]}

        metadata = manager.generate_metadata(
            strategy_entry_point=strategy_entry,
            captured_files=captured_files,
            data_bundle_info=data_bundle_info,
        )

        # Verify all required fields present
        assert "backtest_id" in metadata
        assert "timestamp" in metadata
        assert "framework_version" in metadata
        assert "python_version" in metadata
        assert "strategy_entry_point" in metadata
        assert "captured_files" in metadata
        assert "data_bundle_info" in metadata

    def test_generate_metadata_backtest_id(self, tmp_path):
        """Verify metadata includes correct backtest_id."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        backtest_id = manager.generate_backtest_id()

        metadata = manager.generate_metadata(
            strategy_entry_point=Path("/path/to/strategy.py"), captured_files=[]
        )

        assert metadata["backtest_id"] == backtest_id

    def test_generate_metadata_timestamp_iso8601(self, tmp_path):
        """Verify timestamp is in ISO8601 format."""
        import re
        from datetime import datetime

        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        metadata = manager.generate_metadata(
            strategy_entry_point=Path("/path/to/strategy.py"), captured_files=[]
        )

        timestamp = metadata["timestamp"]

        # Verify ISO8601 format (with timezone)
        # Example: 2025-10-18T14:35:27.123000+00:00
        iso8601_pattern = r"^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\.\d{6}[+-]\d{2}:\d{2}$"
        assert re.match(iso8601_pattern, timestamp)

        # Verify timestamp can be parsed
        parsed = datetime.fromisoformat(timestamp)
        assert parsed is not None

    def test_generate_metadata_framework_version(self, tmp_path):
        """Verify framework version is extracted correctly."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        metadata = manager.generate_metadata(
            strategy_entry_point=Path("/path/to/strategy.py"), captured_files=[]
        )

        # Should have a version (from rustybt.__version__)
        assert metadata["framework_version"] is not None
        assert isinstance(metadata["framework_version"], str)
        # Should not be empty
        assert len(metadata["framework_version"]) > 0

    def test_generate_metadata_python_version(self, tmp_path):
        """Verify Python version is extracted correctly."""
        import sys

        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        metadata = manager.generate_metadata(
            strategy_entry_point=Path("/path/to/strategy.py"), captured_files=[]
        )

        python_version = metadata["python_version"]

        # Should match system Python version
        expected_version = sys.version.split()[0]
        assert python_version == expected_version

        # Should be in format X.Y.Z
        parts = python_version.split(".")
        assert len(parts) >= 2  # At least major.minor

    def test_generate_metadata_strategy_entry_point(self, tmp_path):
        """Verify strategy entry point is recorded correctly."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        strategy_entry = Path("/absolute/path/to/my_strategy.py")

        metadata = manager.generate_metadata(strategy_entry_point=strategy_entry, captured_files=[])

        assert metadata["strategy_entry_point"] == str(strategy_entry)

    def test_generate_metadata_captured_files_relative(self, tmp_path):
        """Verify captured files are converted to relative paths."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        # Create test directory structure
        project_dir = tmp_path / "project"
        project_dir.mkdir()
        strategy_file = project_dir / "strategy.py"
        strategy_file.write_text("# strategy")
        utils_dir = project_dir / "utils"
        utils_dir.mkdir()
        indicator_file = utils_dir / "indicators.py"
        indicator_file.write_text("# indicators")

        captured_files = [strategy_file, indicator_file]

        metadata = manager.generate_metadata(
            strategy_entry_point=strategy_file, captured_files=captured_files
        )

        captured_rel = metadata["captured_files"]

        # Should be relative to strategy entry point parent
        assert "strategy.py" in captured_rel
        assert "utils/indicators.py" in captured_rel or "utils\\indicators.py" in captured_rel

    def test_generate_metadata_data_bundle_info_present(self, tmp_path):
        """Verify data bundle info is included when provided."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        bundle_info = {"bundle_name": "quandl", "dataset_ids": ["uuid-1", "uuid-2"]}

        metadata = manager.generate_metadata(
            strategy_entry_point=Path("/path/to/strategy.py"),
            captured_files=[],
            data_bundle_info=bundle_info,
        )

        assert metadata["data_bundle_info"] == bundle_info
        assert metadata["data_bundle_info"]["bundle_name"] == "quandl"
        assert len(metadata["data_bundle_info"]["dataset_ids"]) == 2

    def test_generate_metadata_data_bundle_info_none(self, tmp_path):
        """Verify data bundle info is null when not provided."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        metadata = manager.generate_metadata(
            strategy_entry_point=Path("/path/to/strategy.py"),
            captured_files=[],
            data_bundle_info=None,
        )

        assert metadata["data_bundle_info"] is None

    def test_generate_metadata_without_backtest_id_fails(self, tmp_path):
        """Verify generate_metadata fails if backtest_id not generated."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))

        # Don't generate backtest_id
        with pytest.raises(BacktestArtifactError, match="Backtest ID not generated"):
            manager.generate_metadata(
                strategy_entry_point=Path("/path/to/strategy.py"), captured_files=[]
            )

    def test_generate_metadata_empty_captured_files(self, tmp_path):
        """Verify generate_metadata works with empty captured files list."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        metadata = manager.generate_metadata(
            strategy_entry_point=Path("/path/to/strategy.py"), captured_files=[]
        )

        assert metadata["captured_files"] == []


class TestMetadataWriting:
    """Test metadata writing functionality."""

    def test_write_metadata_success(self, tmp_path):
        """Verify metadata is written to correct location."""
        import json

        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.create_directory_structure()

        metadata = {
            "backtest_id": "20251018_143527_123",
            "timestamp": "2025-10-18T14:35:27.123000+00:00",
            "framework_version": "0.5.0",
            "python_version": "3.12.1",
            "strategy_entry_point": "/path/to/strategy.py",
            "captured_files": ["strategy.py"],
            "data_bundle_info": None,
        }

        manager.write_metadata(metadata)

        # Verify file exists
        metadata_path = manager.output_dir / "metadata" / "backtest_metadata.json"
        assert metadata_path.exists()

        # Verify content
        with open(metadata_path) as f:
            loaded = json.load(f)

        assert loaded == metadata

    def test_write_metadata_pretty_printed(self, tmp_path):
        """Verify metadata JSON is pretty-printed with indent=2."""
        import json

        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.create_directory_structure()

        metadata = {
            "backtest_id": "test",
            "timestamp": "2025-10-18T14:35:27.123000+00:00",
            "framework_version": "0.5.0",
            "python_version": "3.12.1",
            "strategy_entry_point": "/path/to/strategy.py",
            "captured_files": ["file1.py", "file2.py"],
            "data_bundle_info": {"bundle_name": "test", "dataset_ids": ["1", "2"]},
        }

        manager.write_metadata(metadata)

        metadata_path = manager.output_dir / "metadata" / "backtest_metadata.json"

        # Read raw content
        content = metadata_path.read_text()

        # Verify indentation (should have 2-space indents)
        assert "  " in content  # Has indentation
        assert "    " in content  # Has nested indentation

        # Verify it's valid JSON
        loaded = json.loads(content)
        assert loaded == metadata

    def test_write_metadata_no_output_dir(self, tmp_path):
        """Verify write_metadata logs error but doesn't raise if no output dir."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        # Don't create directory structure

        metadata = {"backtest_id": "test"}

        # Should not raise exception (logs error instead)
        manager.write_metadata(metadata)

    def test_write_metadata_permission_error(self, tmp_path):
        """Verify write_metadata handles permission errors gracefully."""
        import os

        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.create_directory_structure()

        metadata = {"backtest_id": "test"}

        # Make metadata directory read-only
        metadata_dir = manager.output_dir / "metadata"
        os.chmod(metadata_dir, 0o444)

        try:
            # Should not raise exception (logs error instead)
            manager.write_metadata(metadata)
        finally:
            # Restore permissions
            os.chmod(metadata_dir, 0o755)


class TestDataCatalogIntegration:
    """Test DataCatalog integration."""

    def test_get_data_bundle_info_datacatalog_unavailable(self, tmp_path):
        """Verify get_data_bundle_info returns None when DataCatalog unavailable."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        # DataCatalog doesn't exist yet in codebase
        bundle_info = manager.get_data_bundle_info()

        # Should return None gracefully
        assert bundle_info is None

    # Note: The following tests will be enabled when DataCatalog is implemented in Epic X3.7
    # For now, we verify that the method handles missing DataCatalog gracefully (test above)


class TestMetadataPerformance:
    """Test metadata generation performance requirements."""

    @pytest.mark.benchmark
    def test_metadata_generation_performance(self, tmp_path, benchmark):
        """Verify metadata generation completes in <1 second."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        strategy_entry = Path("/path/to/strategy.py")
        captured_files = [Path(f"file{i}.py") for i in range(10)]
        data_bundle_info = {"bundle_name": "test", "dataset_ids": [f"ds-{i}" for i in range(5)]}

        # Benchmark metadata generation
        result = benchmark(
            manager.generate_metadata,
            strategy_entry_point=strategy_entry,
            captured_files=captured_files,
            data_bundle_info=data_bundle_info,
        )

        # Verify result is valid
        assert "backtest_id" in result

        # Performance requirement: <1 second
        # Benchmark provides stats in seconds
        assert benchmark.stats["mean"] < 1.0, "Metadata generation exceeded 1 second threshold"

    @pytest.mark.benchmark
    def test_metadata_write_performance(self, tmp_path, benchmark):
        """Verify metadata writing completes quickly."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.create_directory_structure()

        metadata = {
            "backtest_id": "20251018_143527_123",
            "timestamp": "2025-10-18T14:35:27.123000+00:00",
            "framework_version": "0.5.0",
            "python_version": "3.12.1",
            "strategy_entry_point": "/path/to/strategy.py",
            "captured_files": [f"file{i}.py" for i in range(20)],
            "data_bundle_info": {
                "bundle_name": "test",
                "dataset_ids": [f"ds-{i}" for i in range(10)],
            },
        }

        # Benchmark metadata writing
        benchmark(manager.write_metadata, metadata)

        # Performance requirement: <1 second
        assert benchmark.stats["mean"] < 1.0, "Metadata writing exceeded 1 second threshold"

    def test_end_to_end_metadata_workflow_performance(self, tmp_path):
        """Verify complete metadata workflow (generate + write) completes in <1 second."""
        import time

        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.create_directory_structure()

        strategy_entry = Path("/path/to/strategy.py")
        captured_files = [Path(f"file{i}.py") for i in range(10)]
        data_bundle_info = {"bundle_name": "test", "dataset_ids": [f"ds-{i}" for i in range(5)]}

        # Measure complete workflow time
        start_time = time.perf_counter()

        metadata = manager.generate_metadata(
            strategy_entry_point=strategy_entry,
            captured_files=captured_files,
            data_bundle_info=data_bundle_info,
        )
        manager.write_metadata(metadata)

        end_time = time.perf_counter()
        elapsed = end_time - start_time

        # Performance requirement: <1 second for complete workflow
        assert elapsed < 1.0, f"Complete metadata workflow took {elapsed:.3f}s, exceeds 1 second"


# Fixtures
@pytest.fixture
def temp_backtest_dir(tmp_path):
    """Create temporary directory for backtest outputs."""
    backtest_dir = tmp_path / "backtests"
    backtest_dir.mkdir()
    return backtest_dir


@pytest.fixture
def artifact_manager(temp_backtest_dir):
    """Create BacktestArtifactManager with temp directory."""
    return BacktestArtifactManager(base_dir=str(temp_backtest_dir))


class TestErrorPathCoverage:
    """Tests specifically targeting error path coverage."""

    def test_generate_backtest_id_failure_edge_case(self, tmp_path):
        """Test edge case where backtest_id generation might fail."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))

        # This should work normally, but tests the safety check
        manager.generate_backtest_id()
        assert manager.backtest_id is not None

        # Now try to create directory - should work
        output_dir = manager.create_directory_structure()
        assert output_dir.exists()

    def test_get_output_path_with_pathlib_edge_cases(self, tmp_path):
        """Test get_output_path with various Path edge cases."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.create_directory_structure()

        # Test with Path object vs string
        from pathlib import Path

        # Simple filename
        path1 = manager.get_output_path("test.csv")
        assert path1.exists() or path1.parent.exists()

        # With forward slashes
        path2 = manager.get_output_path("subdir/file.txt")
        assert path2.parent.exists()

        # Multiple levels
        path3 = manager.get_output_path("a/b/c.txt", subdir="metadata")
        assert path3.parent.exists()

    def test_capture_strategy_code_lazy_import(self, tmp_path):
        """Test that code capture handles lazy imports correctly."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path), code_capture_enabled=True)
        manager.create_directory_structure()

        # Create a valid project structure
        project = tmp_path / "project"
        project.mkdir()
        (project / ".git").mkdir()

        strategy = project / "strat.py"
        strategy.write_text("def initialize(c): pass")

        # This exercises the lazy import path
        captured = manager.capture_strategy_code(strategy, project)
        assert isinstance(captured, list)

    def test_get_data_bundle_info_import_error_path(self, tmp_path):
        """Test get_data_bundle_info when import fails."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        # DataCatalog doesn't exist, should return None
        result = manager.get_data_bundle_info()
        assert result is None

    def test_generate_metadata_with_path_edge_cases(self, tmp_path):
        """Test generate_metadata with various path configurations."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        # Test with absolute paths
        strategy = tmp_path / "strat.py"
        strategy.write_text("pass")

        captured = [tmp_path / "file1.py", tmp_path / "file2.py"]

        metadata = manager.generate_metadata(strategy_entry_point=strategy, captured_files=captured)

        assert "captured_files" in metadata
        assert len(metadata["captured_files"]) == 2

    def test_write_metadata_with_json_serialization(self, tmp_path):
        """Test write_metadata with complex nested structures."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.create_directory_structure()

        complex_metadata = {
            "backtest_id": "test123",
            "timestamp": "2025-01-01T00:00:00+00:00",
            "nested": {"level1": {"level2": ["item1", "item2"]}},
            "lists": [1, 2, 3],
            "framework_version": "1.0.0",
            "python_version": "3.12.0",
            "strategy_entry_point": "/path/to/strat.py",
            "captured_files": [],
            "data_bundle_info": None,
        }

        manager.write_metadata(complex_metadata)

        # Verify it was written and is readable
        import json

        metadata_path = manager.output_dir / "metadata" / "backtest_metadata.json"
        assert metadata_path.exists()

        with open(metadata_path) as f:
            loaded = json.load(f)
        assert loaded["nested"]["level1"]["level2"] == ["item1", "item2"]


class TestDataCatalogIntegration:
    """Test DataCatalog integration for backtest-bundle linkage (Story X3.7)."""

    def test_link_backtest_to_bundles_success(self, tmp_path):
        """Test successful backtest-bundle linkage."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        # Create mock DataCatalog module and class
        mock_catalog_instance = Mock()
        mock_catalog_instance.get_bundle_name.return_value = "test_bundle"
        mock_catalog_instance.link_backtest_to_bundles = Mock()

        mock_catalog_class = Mock(return_value=mock_catalog_instance)
        mock_module = MagicMock()
        mock_module.DataCatalog = mock_catalog_class

        with patch.dict(sys.modules, {"rustybt.data.catalog": mock_module}):
            # Link backtest to bundles
            bundle_names = manager.link_backtest_to_bundles()

        assert bundle_names == ["test_bundle"]
        mock_catalog_instance.link_backtest_to_bundles.assert_called_once_with(
            manager.backtest_id, ["test_bundle"]
        )

    def test_link_backtest_to_bundles_no_backtest_id(self, tmp_path):
        """Test link fails gracefully when backtest_id not generated."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))

        # Don't generate backtest_id
        bundle_names = manager.link_backtest_to_bundles()

        assert bundle_names is None

    def test_link_backtest_to_bundles_no_bundles_found(self, tmp_path):
        """Test link handles case when no bundles found."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        # Mock DataCatalog to return "unknown"
        mock_catalog_class = Mock()
        mock_catalog = Mock()
        mock_catalog.get_bundle_name.return_value = "unknown"
        mock_catalog_class.return_value = mock_catalog

        # Patch in sys.modules to handle local imports
        with patch.dict(
            "sys.modules", {"rustybt.data.catalog": Mock(DataCatalog=mock_catalog_class)}
        ):
            bundle_names = manager.link_backtest_to_bundles()

        assert bundle_names is None

    def test_link_backtest_to_bundles_import_error(self, tmp_path):
        """Test link handles DataCatalog import failure."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        # Mock import to raise ImportError
        mock_module = Mock()
        mock_module.DataCatalog.side_effect = ImportError("DataCatalog not available")

        with patch.dict("sys.modules", {"rustybt.data.catalog": mock_module}):
            bundle_names = manager.link_backtest_to_bundles()

        assert bundle_names is None

    def test_link_backtest_to_bundles_database_error(self, tmp_path):
        """Test link handles database errors gracefully."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        # Mock DataCatalog to raise database error
        mock_catalog_class = Mock()
        mock_catalog = Mock()
        mock_catalog.get_bundle_name.return_value = "test_bundle"
        mock_catalog.link_backtest_to_bundles.side_effect = OSError("Database connection failed")
        mock_catalog_class.return_value = mock_catalog

        with patch.dict(
            "sys.modules", {"rustybt.data.catalog": Mock(DataCatalog=mock_catalog_class)}
        ):
            bundle_names = manager.link_backtest_to_bundles()

        assert bundle_names is None

    def test_get_data_bundle_info_success(self, tmp_path):
        """Test successful data bundle info retrieval."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        # Mock DataCatalog
        mock_catalog_class = Mock()
        mock_catalog = Mock()
        mock_catalog.get_bundle_name.return_value = "test_bundle"
        mock_catalog.get_bundles_for_backtest.return_value = ["test_bundle", "test_bundle_2"]
        mock_catalog_class.return_value = mock_catalog

        with patch.dict(
            "sys.modules", {"rustybt.data.catalog": Mock(DataCatalog=mock_catalog_class)}
        ):
            bundle_info = manager.get_data_bundle_info()

        assert bundle_info is not None
        assert bundle_info["bundle_name"] == "test_bundle"
        assert bundle_info["bundle_names"] == ["test_bundle", "test_bundle_2"]

    def test_get_data_bundle_info_no_linked_bundles(self, tmp_path):
        """Test bundle info when no bundles linked yet."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        # Mock DataCatalog
        mock_catalog_class = Mock()
        mock_catalog = Mock()
        mock_catalog.get_bundle_name.return_value = "test_bundle"
        mock_catalog.get_bundles_for_backtest.return_value = []  # No bundles linked yet
        mock_catalog_class.return_value = mock_catalog

        with patch.dict(
            "sys.modules", {"rustybt.data.catalog": Mock(DataCatalog=mock_catalog_class)}
        ):
            bundle_info = manager.get_data_bundle_info()

        assert bundle_info is not None
        assert bundle_info["bundle_name"] == "test_bundle"
        assert bundle_info["bundle_names"] == ["test_bundle"]

    def test_get_data_bundle_info_unknown_bundle(self, tmp_path):
        """Test bundle info when bundle name is unknown."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        # Mock DataCatalog to return "unknown"
        mock_catalog_class = Mock()
        mock_catalog = Mock()
        mock_catalog.get_bundle_name.return_value = "unknown"
        mock_catalog_class.return_value = mock_catalog

        with patch.dict(
            "sys.modules", {"rustybt.data.catalog": Mock(DataCatalog=mock_catalog_class)}
        ):
            bundle_info = manager.get_data_bundle_info()

        assert bundle_info is None

    def test_get_data_bundle_info_import_error(self, tmp_path):
        """Test bundle info handles import error."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))

        # Mock import to raise ImportError
        mock_module = Mock()
        mock_module.DataCatalog.side_effect = ImportError("DataCatalog not available")

        with patch.dict("sys.modules", {"rustybt.data.catalog": mock_module}):
            bundle_info = manager.get_data_bundle_info()

        assert bundle_info is None

    def test_get_data_bundle_info_query_error(self, tmp_path):
        """Test bundle info handles query errors."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        # Mock DataCatalog to raise error
        mock_catalog_class = Mock()
        mock_catalog = Mock()
        mock_catalog.get_bundle_name.side_effect = RuntimeError("Query failed")
        mock_catalog_class.return_value = mock_catalog

        with patch.dict(
            "sys.modules", {"rustybt.data.catalog": Mock(DataCatalog=mock_catalog_class)}
        ):
            bundle_info = manager.get_data_bundle_info()

        assert bundle_info is None

    def test_generate_metadata_includes_bundle_info(self, tmp_path):
        """Test metadata generation includes bundle info from DataCatalog."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        strategy = tmp_path / "strat.py"
        strategy.write_text("pass")

        # Mock bundle info
        bundle_info = {
            "bundle_name": "test_bundle",
            "bundle_names": ["test_bundle"],
        }

        metadata = manager.generate_metadata(
            strategy_entry_point=strategy,
            captured_files=[],
            data_bundle_info=bundle_info,
        )

        assert metadata["data_bundle_info"] == bundle_info
        assert metadata["data_bundle_info"]["bundle_name"] == "test_bundle"

    def test_generate_metadata_handles_none_bundle_info(self, tmp_path):
        """Test metadata generation handles None bundle info."""
        manager = BacktestArtifactManager(base_dir=str(tmp_path))
        manager.generate_backtest_id()

        strategy = tmp_path / "strat.py"
        strategy.write_text("pass")

        metadata = manager.generate_metadata(
            strategy_entry_point=strategy,
            captured_files=[],
            data_bundle_info=None,
        )

        assert metadata["data_bundle_info"] is None
