"""Unit tests for Interactive Brokers adapter.

Tests IB adapter functionality with mocked ib_async library.
"""

from decimal import Decimal
from unittest.mock import AsyncMock, Mock, patch

import pandas as pd
import pytest
from ib_insync import Contract, Order, Stock, Trade

from rustybt.assets import Equity
from rustybt.live.brokers.ib_adapter import (
    IBBrokerAdapter,
    IBConnectionError,
    IBOrderRejectError,
)


@pytest.fixture
def mock_ib():
    """Create mock IB object."""
    with patch("rustybt.live.brokers.ib_adapter.IB") as mock_ib_class:
        mock_ib_instance = Mock()
        mock_ib_class.return_value = mock_ib_instance

        # Mock async methods
        mock_ib_instance.connectAsync = AsyncMock()
        mock_ib_instance.disconnect = Mock()
        mock_ib_instance.isConnected = Mock(return_value=True)

        # Mock event handlers
        mock_ib_instance.orderStatusEvent = Mock()
        mock_ib_instance.orderStatusEvent.__iadd__ = Mock()
        mock_ib_instance.execDetailsEvent = Mock()
        mock_ib_instance.execDetailsEvent.__iadd__ = Mock()
        mock_ib_instance.errorEvent = Mock()
        mock_ib_instance.errorEvent.__iadd__ = Mock()
        mock_ib_instance.disconnectedEvent = Mock()
        mock_ib_instance.disconnectedEvent.__iadd__ = Mock()

        # Mock order methods
        mock_ib_instance.placeOrder = Mock()
        mock_ib_instance.cancelOrder = Mock()

        # Mock account/position methods
        mock_ib_instance.accountSummary = Mock(return_value=[])
        mock_ib_instance.positions = Mock(return_value=[])
        mock_ib_instance.openTrades = Mock(return_value=[])

        # Mock market data methods
        mock_ib_instance.reqMktData = Mock()
        mock_ib_instance.cancelMktData = Mock()

        # Mock client
        mock_ib_instance.client = Mock()
        mock_ib_instance.client.getReqId = Mock(return_value=1)

        yield mock_ib_instance


@pytest.fixture
def sample_asset():
    """Create sample equity asset."""
    from rustybt.assets import ExchangeInfo

    # Create exchange info (simplified)
    nasdaq = ExchangeInfo(
        name="NASDAQ",
        canonical_name="NASDAQ",
        country_code="US",
    )

    return Equity(
        sid=1,
        exchange_info=nasdaq,
        symbol="AAPL",
    )


@pytest.mark.asyncio
async def test_connect_success(mock_ib):
    """Test successful connection to IB."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)

    await adapter.connect()

    assert adapter.is_connected()
    mock_ib.connectAsync.assert_called_once_with(
        host="127.0.0.1",
        port=7496,
        clientId=1,
        timeout=30,
    )


@pytest.mark.asyncio
async def test_connect_failure(mock_ib):
    """Test connection failure."""
    mock_ib.connectAsync.side_effect = Exception("Connection refused")

    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)

    with pytest.raises(IBConnectionError, match="Failed to connect to IB"):
        await adapter.connect()

    assert not adapter.is_connected()


@pytest.mark.asyncio
async def test_disconnect(mock_ib):
    """Test disconnect from IB."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)

    await adapter.connect()
    await adapter.disconnect()

    mock_ib.disconnect.assert_called_once()
    assert not adapter.is_connected()


@pytest.mark.asyncio
async def test_submit_market_order(mock_ib, sample_asset):
    """Test market order submission."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    # Mock trade response (order ID will be generated by adapter)
    mock_trade = Mock(spec=Trade)
    mock_trade.order = Mock(spec=Order)
    mock_ib.placeOrder.return_value = mock_trade

    # Submit order
    order_id = await adapter.submit_order(
        asset=sample_asset,
        amount=Decimal("100"),
        order_type="market",
    )

    # Order ID should be string
    assert order_id is not None
    assert order_id.isdigit()
    mock_ib.placeOrder.assert_called_once()

    # Verify contract
    call_args = mock_ib.placeOrder.call_args
    contract = call_args[0][0]
    assert isinstance(contract, Stock)
    assert contract.symbol == "AAPL"

    # Verify order
    order = call_args[0][1]
    assert order.orderType == "MKT"
    assert order.action == "BUY"
    assert order.totalQuantity == 100


@pytest.mark.asyncio
async def test_submit_limit_order(mock_ib, sample_asset):
    """Test limit order submission."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    # Mock trade response
    mock_trade = Mock(spec=Trade)
    mock_trade.order = Mock(spec=Order)
    mock_ib.placeOrder.return_value = mock_trade

    # Submit limit order
    order_id = await adapter.submit_order(
        asset=sample_asset,
        amount=Decimal("-50"),  # Sell
        order_type="limit",
        limit_price=Decimal("150.50"),
    )

    assert order_id is not None
    assert order_id.isdigit()

    # Verify order
    call_args = mock_ib.placeOrder.call_args
    order = call_args[0][1]
    assert order.orderType == "LMT"
    assert order.action == "SELL"
    assert order.totalQuantity == 50
    assert order.lmtPrice == 150.50


@pytest.mark.asyncio
async def test_submit_stop_order(mock_ib, sample_asset):
    """Test stop order submission."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    # Mock trade response
    mock_trade = Mock(spec=Trade)
    mock_trade.order = Mock(spec=Order)
    mock_ib.placeOrder.return_value = mock_trade

    # Submit stop order
    order_id = await adapter.submit_order(
        asset=sample_asset,
        amount=Decimal("100"),
        order_type="stop",
        stop_price=Decimal("140.00"),
    )

    assert order_id is not None
    assert order_id.isdigit()

    # Verify order
    call_args = mock_ib.placeOrder.call_args
    order = call_args[0][1]
    assert order.orderType == "STP"
    assert order.auxPrice == 140.00


@pytest.mark.asyncio
async def test_submit_stop_limit_order(mock_ib, sample_asset):
    """Test stop-limit order submission."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    # Mock trade response
    mock_trade = Mock(spec=Trade)
    mock_trade.order = Mock(spec=Order)
    mock_ib.placeOrder.return_value = mock_trade

    # Submit stop-limit order
    order_id = await adapter.submit_order(
        asset=sample_asset,
        amount=Decimal("100"),
        order_type="stop-limit",
        limit_price=Decimal("145.00"),
        stop_price=Decimal("140.00"),
    )

    assert order_id is not None
    assert order_id.isdigit()

    # Verify order
    call_args = mock_ib.placeOrder.call_args
    order = call_args[0][1]
    assert order.orderType == "STP LMT"
    assert order.lmtPrice == 145.00
    assert order.auxPrice == 140.00


@pytest.mark.asyncio
async def test_submit_trailing_stop_order(mock_ib, sample_asset):
    """Test trailing stop order submission."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    # Mock trade response
    mock_trade = Mock(spec=Trade)
    mock_trade.order = Mock(spec=Order)
    mock_ib.placeOrder.return_value = mock_trade

    # Submit trailing stop order
    order_id = await adapter.submit_order(
        asset=sample_asset,
        amount=Decimal("100"),
        order_type="trailing-stop",
        stop_price=Decimal("5.00"),  # Trailing amount
    )

    assert order_id is not None
    assert order_id.isdigit()

    # Verify order
    call_args = mock_ib.placeOrder.call_args
    order = call_args[0][1]
    assert order.orderType == "TRAIL"
    assert order.auxPrice == 5.00


@pytest.mark.asyncio
async def test_submit_order_not_connected(sample_asset):
    """Test order submission when not connected."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)

    with pytest.raises(IBConnectionError, match="Not connected to IB"):
        await adapter.submit_order(
            asset=sample_asset,
            amount=Decimal("100"),
            order_type="market",
        )


@pytest.mark.asyncio
async def test_submit_order_invalid_type(mock_ib, sample_asset):
    """Test order submission with invalid order type."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    with pytest.raises(IBOrderRejectError, match="Unsupported order type"):
        await adapter.submit_order(
            asset=sample_asset,
            amount=Decimal("100"),
            order_type="invalid",
        )


@pytest.mark.asyncio
async def test_cancel_order(mock_ib):
    """Test order cancellation."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    # Add pending order
    mock_trade = Mock(spec=Trade)
    mock_trade.order = Mock(spec=Order)
    mock_trade.order.orderId = 123
    adapter._pending_orders[123] = mock_trade

    # Cancel order
    await adapter.cancel_order("123")

    mock_ib.cancelOrder.assert_called_once_with(mock_trade.order)


@pytest.mark.asyncio
async def test_cancel_order_not_found(mock_ib):
    """Test cancelling non-existent order."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    # Should not raise error
    await adapter.cancel_order("999")


@pytest.mark.asyncio
async def test_get_account_info(mock_ib):
    """Test retrieving account information."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    # Mock account summary
    mock_summary_item = Mock()
    mock_summary_item.tag = "TotalCashValue"
    mock_summary_item.value = "100000.50"

    mock_summary_item2 = Mock()
    mock_summary_item2.tag = "NetLiquidation"
    mock_summary_item2.value = "150000.75"

    mock_summary_item3 = Mock()
    mock_summary_item3.tag = "BuyingPower"
    mock_summary_item3.value = "400000.00"

    mock_ib.accountSummary.return_value = [
        mock_summary_item,
        mock_summary_item2,
        mock_summary_item3,
    ]

    # Get account info
    account_info = await adapter.get_account_info()

    assert account_info["cash"] == Decimal("100000.50")
    assert account_info["equity"] == Decimal("150000.75")
    assert account_info["buying_power"] == Decimal("400000.00")


@pytest.mark.asyncio
async def test_get_positions(mock_ib):
    """Test retrieving positions."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    # Mock position
    mock_position = Mock()
    mock_position.contract = Mock(spec=Contract)
    mock_position.contract.symbol = "AAPL"
    mock_position.position = 100
    mock_position.avgCost = 150.50

    mock_ib.positions.return_value = [mock_position]

    # Get positions
    positions = await adapter.get_positions()

    assert len(positions) == 1
    assert positions[0]["symbol"] == "AAPL"
    assert positions[0]["amount"] == Decimal("100")
    assert positions[0]["cost_basis"] == Decimal("150.50")


@pytest.mark.asyncio
async def test_get_open_orders(mock_ib):
    """Test retrieving open orders."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    # Mock trade
    mock_trade = Mock(spec=Trade)
    mock_trade.contract = Mock(spec=Contract)
    mock_trade.contract.symbol = "AAPL"
    mock_trade.order = Mock(spec=Order)
    mock_trade.order.orderId = 123
    mock_trade.order.totalQuantity = 100
    mock_trade.order.orderType = "LMT"
    mock_trade.orderStatus = Mock()
    mock_trade.orderStatus.status = "Submitted"

    mock_ib.openTrades.return_value = [mock_trade]

    # Get open orders
    orders = await adapter.get_open_orders()

    assert len(orders) == 1
    assert orders[0]["order_id"] == "123"
    assert orders[0]["symbol"] == "AAPL"
    assert orders[0]["amount"] == Decimal("100")
    assert orders[0]["status"] == "Submitted"


@pytest.mark.asyncio
async def test_subscribe_market_data(mock_ib, sample_asset):
    """Test subscribing to market data."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    # Mock ticker
    mock_ticker = Mock()
    mock_ticker.updateEvent = Mock()
    mock_ticker.updateEvent.__iadd__ = Mock()
    mock_ib.reqMktData.return_value = mock_ticker

    # Subscribe
    await adapter.subscribe_market_data([sample_asset])

    mock_ib.reqMktData.assert_called_once()
    assert sample_asset in adapter._subscribed_contracts


@pytest.mark.asyncio
async def test_unsubscribe_market_data(mock_ib, sample_asset):
    """Test unsubscribing from market data."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    # Mock ticker
    mock_ticker = Mock()
    mock_ticker.updateEvent = Mock()
    mock_ticker.updateEvent.__iadd__ = Mock()
    mock_ib.reqMktData.return_value = mock_ticker

    # Subscribe first
    await adapter.subscribe_market_data([sample_asset])

    # Unsubscribe
    await adapter.unsubscribe_market_data([sample_asset])

    mock_ib.cancelMktData.assert_called_once()
    assert sample_asset not in adapter._subscribed_contracts


@pytest.mark.asyncio
async def test_get_current_price(mock_ib, sample_asset):
    """Test getting current price."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    # Mock ticker
    mock_ticker = Mock()
    mock_ticker.last = 150.75
    mock_ticker.close = 149.50
    mock_ib.reqMktData.return_value = mock_ticker

    # Get price
    price = await adapter.get_current_price(sample_asset)

    assert price == Decimal("150.75")
    mock_ib.cancelMktData.assert_called_once()


@pytest.mark.asyncio
async def test_get_current_price_fallback_to_close(mock_ib, sample_asset):
    """Test getting current price falls back to close if no last price."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    # Mock ticker with no last price
    mock_ticker = Mock()
    mock_ticker.last = 0
    mock_ticker.close = 149.50
    mock_ib.reqMktData.return_value = mock_ticker

    # Get price
    price = await adapter.get_current_price(sample_asset)

    assert price == Decimal("149.50")


@pytest.mark.asyncio
async def test_get_current_price_no_data(mock_ib, sample_asset):
    """Test getting current price with no data available."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    # Mock ticker with no data
    mock_ticker = Mock()
    mock_ticker.last = 0
    mock_ticker.close = 0
    mock_ib.reqMktData.return_value = mock_ticker

    # Should raise error
    with pytest.raises(IBConnectionError, match="No price available"):
        await adapter.get_current_price(sample_asset)


@pytest.mark.asyncio
async def test_is_connected(mock_ib):
    """Test connection status check."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)

    assert not adapter.is_connected()

    await adapter.connect()

    assert adapter.is_connected()

    await adapter.disconnect()

    assert not adapter.is_connected()


@pytest.mark.asyncio
async def test_get_next_market_data_empty():
    """Test getting next market data when queue is empty."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)

    data = await adapter.get_next_market_data()

    assert data is None


@pytest.mark.asyncio
async def test_market_data_update_callback(mock_ib, sample_asset):
    """Test market data update callback."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)
    await adapter.connect()

    # Create mock ticker
    mock_ticker = Mock()
    mock_ticker.last = 150.75
    mock_ticker.bid = 150.70
    mock_ticker.ask = 150.80
    mock_ticker.volume = 1000000
    mock_ticker.time = pd.Timestamp("2025-01-15 10:30:00")

    # Call update handler
    adapter._on_market_data_update(mock_ticker, sample_asset)

    # Get data from queue
    data = await adapter.get_next_market_data()

    assert data is not None
    assert data["asset"] == sample_asset
    assert data["price"] == Decimal("150.75")
    assert data["bid"] == Decimal("150.70")
    assert data["ask"] == Decimal("150.80")
    assert data["volume"] == 1000000


def test_create_contract_equity(sample_asset):
    """Test creating IB contract from equity asset."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)

    contract = adapter._create_contract(sample_asset)

    assert isinstance(contract, Stock)
    assert contract.symbol == "AAPL"
    assert contract.exchange == "SMART"
    assert contract.currency == "USD"


def test_get_next_order_id():
    """Test order ID generation."""
    adapter = IBBrokerAdapter(host="127.0.0.1", port=7496, client_id=1)

    # Before connection, should use counter
    order_id1 = adapter._get_next_order_id()
    order_id2 = adapter._get_next_order_id()

    assert order_id2 > order_id1
