from itertools import combinations

from compas.geometry import Point
from compas.geometry import distance_point_line
from compas_model.interactions import Interaction

from compas_timber.errors import BeamJoiningError

from .solver import JointTopology


class Joint(Interaction):
    """Base class for a joint connecting two beams.

    This is a base class and should not be instantiated directly.
    Use the `create()` class method of the respective implementation of `Joint` instead.

    Parameters
    ----------
    topology : literal, one of :class:`JointTopology`
        The topology by which the two elements connected with this joint interact.
    location : :class:`~compas.geometry.Point`
        The estimated location of the interaction point of the two elements connected with this joint.

    Attributes
    ----------
    name : str
        The name of the joint. Corresponds to the class name.
    elements : tuple(:class:`~compas_model.elements.Element`)
        The elements joined by this joint.
    generated_elements : list(:class:`~compas_model.elements.Element`)
        A list of elements that were generated by this joint.
    ends : dict(str, str)
        Maps the GUID of each element to ``start`` or ``end``, depending on which end of the element is connected by this joint.
    interactions : list(tuple(:class:`~compas_model.elements.Element`, :class:`~compas_model.elements.Element`))
        A list of tuples containing the elements that are interacting with each other through this joint.
    features : list(:class:`~compas_timber.fabrication.BTLxProcessing`)
        A list of features that were added to the elements by this joint.
    topology : literal, one of :class:`JointTopology`
        The topology by which the two elements connected with this joint interact.
    location : :class:`~compas.geometry.Point`
        The estimated location of the interaction point of the two elements connected with this joint.
    """

    SUPPORTED_TOPOLOGY = JointTopology.TOPO_UNKNOWN
    MIN_ELEMENT_COUNT = 2
    MAX_ELEMENT_COUNT = 2

    def __init__(self, topology=None, location=None, **kwargs):
        super(Joint, self).__init__(name=self.__class__.__name__)
        self._topology = topology if topology is not None else JointTopology.TOPO_UNKNOWN
        self._location = location or Point(0, 0, 0)

    @property
    def topology(self):
        return self._topology

    @topology.setter
    def topology(self, value):
        """Set the topology of the joint."""
        self._topology = value

    @property
    def location(self):
        return self._location

    @location.setter
    def location(self, value):
        """Set the location of the joint."""
        if not isinstance(value, Point):
            raise TypeError("Location must be a Point.")
        self._location = value

    @property
    def elements(self):
        raise NotImplementedError

    @property
    def generated_elements(self):
        return []

    @property
    def ends(self):
        # TODO: this may be obsolete, don't see it used anywhere, consider removing
        self._ends = {}
        for index, beam in enumerate(self.elements):
            if distance_point_line(beam.centerline.start, self.elements[index - 1].centerline) < distance_point_line(beam.centerline.end, self.elements[index - 1].centerline):
                self._ends[str(beam.guid)] = "start"
            else:
                self._ends[str(beam.guid)] = "end"
        return self._ends

    @property
    def interactions(self):
        interactions = []
        for pair in combinations(self.elements, 2):
            interactions.append((pair[0], pair[1]))
        return interactions

    @classmethod
    def element_count_complies(cls, elements):
        """Checks if the number of elements complies with the joint's requirements.

        Parameters
        ----------
        elements : list(:class:`~compas_model.elements.Element`)
            The elements to be checked.

        Returns
        -------
        bool

        """
        if cls.MAX_ELEMENT_COUNT:
            return len(elements) >= cls.MIN_ELEMENT_COUNT and len(elements) <= cls.MAX_ELEMENT_COUNT
        else:
            return len(elements) >= cls.MIN_ELEMENT_COUNT

    def add_features(self):
        """Adds the features defined by this joint to affected beam(s).

        Raises
        ------
        :class:`~compas_timber.connections.BeamJoiningError`
            Should be raised whenever the joint was not able to calculate the features to be applied to the beams.

        """
        raise NotImplementedError

    def add_extensions(self):
        """Adds the extensions defined by this joint to affected beam(s).
        This is optional and should only be implemented by joints that require it.

        Notes
        -----
        Extensions are added to all beams before the features are added.

        Raises
        ------
        :class:`~compas_timber.connections.BeamJoiningError`
            Should be raised whenever the joint was not able to calculate the extensions to be applied to the beams.

        """
        pass

    def restore_beams_from_keys(self, model):
        """Restores the reference to the elements associated with this joint.

        During serialization, :class:`compas_timber.parts.Beam` objects
        are serialized by :class:`compas_timber.model.Model`. To avoid circular references, Joint only stores the keys
        of the respective elements.

        This method is called by :class:`compas_timber.model.TimberModel` during de-serialization to restore the references.
        Since the roles of the elements are joint specific (e.g. main/cross beam) this method should be implemented by
        the concrete implementation.

        Examples
        --------
        See :class:`compas_timber.connections.TButtJoint`.

        """
        raise NotImplementedError

    @classmethod
    def create(cls, model, *elements, **kwargs):
        """Creates an instance of this joint and creates the new connection in `model`.

        `elements` are expected to have been added to `model` before calling this method.

        This code does not verify that the given elements are adjacent and/or lie in a topology which allows connecting
        them. This is the responsibility of the calling code.

        Parameters
        ----------
        model : :class:`~compas_timber.model.TimberModel`
            The model to which the elements and this joint belong.
        *elements : :class:`~compas_model.elements.Element`
            The elements to be connected by this joint. The number of elements must comply with the `Joint` class's
            `MIN_ELEMENT_COUNT` and `MAX_ELEMENT_COUNT` attributes.
        **kwargs : dict
            Additional keyword arguments that are passed to the joint's constructor.

        Returns
        -------
        :class:`compas_timber.connections.Joint`
            The instance of the created joint.

        """

        joint = cls(*elements, **kwargs)
        model.add_joint(joint)
        return joint

    @classmethod
    def promote_cluster(cls, model, cluster, reordered_elements=None, **kwargs):
        """Creates an instance of this joint from a cluster of elements.

        Parameters
        ----------
        model : :class:`~compas_timber.model.TimberModel`
            The model to which the elements and this joint belong.
        cluster : :class:`~compas_model.clusters.Cluster`
            The cluster containing the elements to be connected by this joint.
        reordered_elements : list(:class:`~compas_model.elements.Element`), optional
            The elements to be connected by this joint. If not provided, the elements of the cluster will be used.
            This is used to explicitly define the element order.
        **kwargs : dict
            Additional keyword arguments that are passed to the joint's constructor.

        Returns
        -------
        :class:`compas_timber.connections.Joint`
            The instance of the created joint.

        """
        if reordered_elements:
            if set(reordered_elements) != cluster.elements:
                raise BeamJoiningError(cls, "Elements of the joint candidate must match the provided elements.", [e.blank for e in reordered_elements])
        if len(cluster.joints) == 1:
            elements = reordered_elements or cluster.joints[0].elements
            return cls.promote_joint_candidate(model, cluster.joints[0], reordered_elements=elements, **kwargs)
        else:
            elements = reordered_elements or list(cluster.elements)
        return cls.create(model, *elements, **kwargs)

    @classmethod
    def promote_joint_candidate(cls, model, candidate, reordered_elements=None, **kwargs):
        """Creates an instance of this joint from a joint candidate.

        Parameters
        ----------
        model : :class:`~compas_timber.model.TimberModel`
            The model to which the elements and this joint belong.
        candidate : :class:`~compas_timber.connections.JointCandidate`
            The joint candidate to be converted.
        reordered_elements : list(:class:`~compas_model.elements.Element`), optional
            The elements to be connected by this joint. If not provided, the elements of the generic joint will be used.
            This is used to explicitly define the element order.
        **kwargs : dict
            Additional keyword arguments that are passed to the joint's constructor.

        Returns
        -------
        :class:`compas_timber.connections.Joint`
            The instance of the created joint.

        """
        if reordered_elements:
            assert set(reordered_elements) == set(candidate.elements), "Elements of the generic joint must match the provided elements."
            elements = reordered_elements
        else:
            elements = candidate.elements
        kwargs.update({"topology": candidate.topology, "location": candidate.location})  # pass topology, distance and location from candidate
        joint = cls.create(model, *elements, **kwargs)
        return joint

    @classmethod
    def check_elements_compatibility(cls, elements, raise_error=False):
        """Checks if the cluster of beams complies with the requirements for the Joint.

        Parameters
        ----------
        elements : list of :class:`~compas_timber.parts.Beam`
            The beams to check.
        raise_error : bool, optional
            If True, raises a `BeamJoiningError` if the requirements are not met.

        Returns
        -------
        bool
            True if the cluster complies with the requirements, False otherwise.

        """
        return True
