"""Tests for SWOT configuration module."""

from unique_swot.config import TOOL_DESCRIPTION, SwotAnalysisToolConfig
from unique_swot.services.generation import ReportGenerationConfig
from unique_swot.services.report import ReportRendererConfig


class TestSwotConfig:
    """Test cases for SwotConfig class."""

    def test_swot_config_default_values(self):
        """Test that SwotConfig initializes with correct default values."""
        config = SwotAnalysisToolConfig()

        assert config.cache_scope_id == ""
        assert isinstance(config.report_generation_config, ReportGenerationConfig)
        assert config.tool_description == TOOL_DESCRIPTION
        assert config.tool_description_for_system_prompt == TOOL_DESCRIPTION
        assert config.tool_format_information_for_system_prompt == ""
        assert config.tool_description_for_user_prompt == ""
        assert config.tool_format_information_for_user_prompt == ""
        assert config.tool_format_reminder_for_user_prompt == ""

    def test_swot_config_custom_cache_scope_id(self):
        """Test SwotConfig with custom cache_scope_id."""
        custom_scope = "custom_swot_scope"
        config = SwotAnalysisToolConfig(cache_scope_id=custom_scope)

        assert config.cache_scope_id == custom_scope

    def test_swot_config_custom_report_generation_config(self):
        """Test SwotConfig with custom report generation config."""
        custom_report_config = ReportGenerationConfig(
            extraction_batch_size=5,
            max_tokens_per_extraction_batch=2000,
        )
        config = SwotAnalysisToolConfig(report_generation_config=custom_report_config)

        assert config.report_generation_config.extraction_batch_size == 5
        assert config.report_generation_config.max_tokens_per_extraction_batch == 2000

    def test_swot_config_custom_descriptions(self):
        """Test SwotConfig with custom description fields."""
        custom_desc = "Custom description"
        config = SwotAnalysisToolConfig(
            tool_description=custom_desc,
            tool_description_for_system_prompt=custom_desc,
        )

        assert config.tool_description == custom_desc
        assert config.tool_description_for_system_prompt == custom_desc

    def test_tool_description_contains_key_information(self):
        """Test that TOOL_DESCRIPTION contains expected key information."""
        assert "SWOT" in TOOL_DESCRIPTION
        assert "analysis" in TOOL_DESCRIPTION.lower()
        assert (
            "strengths" in TOOL_DESCRIPTION.lower()
            or "weaknesses" in TOOL_DESCRIPTION.lower()
        )

    def test_swot_config_serialization(self):
        """Test that SwotConfig can be serialized and deserialized."""
        config = SwotAnalysisToolConfig(
            cache_scope_id="test_scope",
            report_generation_config=ReportGenerationConfig(),
            report_renderer_config=ReportRendererConfig(),
        )
        config_dict = config.model_dump()

        assert config_dict["cache_scope_id"] == "test_scope"
        assert "report_generation_config" in config_dict
        assert "report_renderer_config" in config_dict

        # Recreate from dict
        config_restored = SwotAnalysisToolConfig.model_validate(config_dict)
        assert config_restored.cache_scope_id == "test_scope"
