//! プロジェクト全体の定数定義
//!
//! QuantForgeプロジェクトで使用される共通定数を定義
//! 精度レベルは用途別に階層化されており、適切なレベルを選択して使用してください。

/// 実務精度: 価格計算・金融実務用（0.1% = 1bp×10）
///
/// 金融実務で標準的な精度。機関投資家の取引でも0.1セント単位で十分。
/// この精度は浮動小数点演算で確実に達成可能。
/// 価格計算、バッチ処理、実務アプリケーションで使用。
pub const PRACTICAL_TOLERANCE: f64 = 1e-3;

/// 理論精度: アルゴリズムの精度制約下での検証用
///
/// norm_cdf実装の現在の精度限界（約1e-5レベル）を考慮した精度。
/// 理論的な価格モデルの検証や、数学的性質のテストで使用。
pub const THEORETICAL_TOLERANCE: f64 = 1e-5;

/// 数値精度: 基本的な数値計算の検証用
///
/// 高精度が必要な数学的性質の検証（対称性、単調性など）で使用。
/// 浮動小数点演算の限界内での最高精度レベル。
pub const NUMERICAL_TOLERANCE: f64 = 1e-7;

/// デフォルト精度（実務用途）
///
/// 後方互換性のため、EPSILONは実務精度を指す。
/// 新規コードではPRACTICAL_TOLERANCEの使用を推奨。
pub const EPSILON: f64 = PRACTICAL_TOLERANCE;

/// 数学定数: 1/√(2π)
///
/// 標準正規分布の確率密度関数で使用。
/// 値は高精度計算により導出。
pub const INV_SQRT_2PI: f64 = 0.3989422804014327;

/// グリークス計算用: 年間日数
///
/// Theta計算で1日あたりの時間価値減少を計算する際に使用。
/// 金融実務では365日を標準とする。
pub const DAYS_PER_YEAR: f64 = 365.0;

/// グリークス計算用: ベーシスポイント
///
/// VegaとRhoの計算で1%変化に対する感応度を表現する際に使用。
/// 1% = 100ベーシスポイント
pub const BASIS_POINT_MULTIPLIER: f64 = 100.0;

/// グリークス計算用: プットデルタ調整係数
///
/// プット・コール・パリティより、プットデルタ = コールデルタ - 1
/// この関係式はオプション理論の基本的な性質から導かれる。
pub const PUT_DELTA_ADJUSTMENT: f64 = 1.0;

/// グリークス計算用: Theta計算の分母係数
///
/// Black-Scholes式でのtheta計算における標準的な分母係数。
/// -(S * φ(d1) * σ) / (2 * √T) の式における係数。
pub const THETA_DENOMINATOR_FACTOR: f64 = 2.0;

// ============================================================================
// 数学関数係数定数
// ============================================================================

/// Abramowitz & Stegun (1964)の誤差関数近似係数 a1
///
/// Handbook of Mathematical Functions, Formula 7.1.26
/// 高速誤差関数近似で使用される標準係数。
pub const ABRAMOWITZ_A1: f64 = 0.254829592;

/// Abramowitz & Stegun (1964)の誤差関数近似係数 a2
pub const ABRAMOWITZ_A2: f64 = -0.284496736;

/// Abramowitz & Stegun (1964)の誤差関数近似係数 a3
pub const ABRAMOWITZ_A3: f64 = 1.421413741;

/// Abramowitz & Stegun (1964)の誤差関数近似係数 a4
pub const ABRAMOWITZ_A4: f64 = -1.453152027;

/// Abramowitz & Stegun (1964)の誤差関数近似係数 a5
pub const ABRAMOWITZ_A5: f64 = 1.061405429;

/// Abramowitz & Stegun (1964)の誤差関数近似係数 p
///
/// 近似式で使用される定数係数。
pub const ABRAMOWITZ_P: f64 = 0.3275911;

// ============================================================================
// 入力パラメータ制約定数
// ============================================================================

/// 最小価格（スポット/ストライク）
///
/// 実務的に意味のある最小価格。
/// これ以下の価格は計算精度が保証できない。
pub const MIN_PRICE: f64 = 0.01;

/// 最大価格（スポット/ストライク）
///
/// 実務的に意味のある最大価格（2^31）。
/// これ以上の価格は計算精度が保証できない。
pub const MAX_PRICE: f64 = 2147483648.0;

/// 最小時間（年単位）
///
/// 約8.76時間（0.001年）。
/// これ以下の短期オプションは実務的でない。
pub const MIN_TIME: f64 = 0.001;

/// 最大時間（年単位）
///
/// 100年。
/// これ以上の長期オプションは実務的でない。
pub const MAX_TIME: f64 = 100.0;

/// 最小ボラティリティ（実務下限）
///
/// 市場で観測される実質的な最小ボラティリティ（0.5%）。
/// これ以下の値は非現実的とみなす。
pub const MIN_VOLATILITY_PRACTICAL: f64 = 0.005;

/// 最小金利
///
/// 極端な負の金利の下限（-100%）。
/// これ以下の金利は非現実的。
pub const MIN_RATE: f64 = -1.0;

/// 最大金利
///
/// 極端な高金利の上限（100%）。
/// これ以上の金利は非現実的。
pub const MAX_RATE: f64 = 1.0;

// ============================================================================
// インプライドボラティリティ計算定数
// ============================================================================

/// IV計算: 最大反復回数
///
/// Newton-Raphson法およびBrent法での最大反復回数。
/// 通常3-5回で収束するが、エッジケースを考慮して20回に設定。
pub const IV_MAX_ITERATIONS: usize = 20;

/// IV計算: 価格収束判定閾値
///
/// 計算された価格と市場価格の差がこの値以下なら収束とみなす。
/// 金融実務での価格精度を考慮した値。
pub const IV_TOLERANCE_PRICE: f64 = 1e-9;

/// IV計算: ボラティリティ収束判定閾値
///
/// 連続する反復でのボラティリティ変化がこの値以下なら収束とみなす。
pub const IV_TOLERANCE_VOL: f64 = 1e-6;

/// IV計算: 最小ボラティリティ（0.1%）
///
/// 市場で観測される実質的な最小ボラティリティ。
/// これ以下の値は非現実的とみなす。
pub const IV_MIN_VOL: f64 = 0.001;

/// IV計算: 最大ボラティリティ（500%）
///
/// 市場で観測される実質的な最大ボラティリティ。
/// 暗号通貨等の極端なケースを考慮。
pub const IV_MAX_VOL: f64 = 5.0;

// エイリアス（後方互換性のため）
pub const MIN_VOLATILITY: f64 = IV_MIN_VOL;
pub const MAX_VOLATILITY: f64 = IV_MAX_VOL;
pub const IV_TOLERANCE: f64 = IV_TOLERANCE_PRICE;
pub const MAX_IV_ITERATIONS: usize = IV_MAX_ITERATIONS;

/// IV計算: Vega最小閾値（数値安定性）
///
/// Newton-Raphson法でVegaがこの値以下の場合、数値的に不安定とみなす。
/// ゼロ除算を避けるための安全閾値。
pub const VEGA_MIN_THRESHOLD: f64 = 1e-10;

/// IV計算: 初期ブラケット幅
///
/// Brent法での初期探索範囲の幅。
/// 初期推定値の周辺をこの幅で探索。
pub const IV_INITIAL_BRACKET_WIDTH: f64 = 0.5;

/// IV計算: Newton-Raphson法の初期ボラティリティ推定値
///
/// 市場で一般的な20%のボラティリティを初期値として使用。
/// Newton-Raphson法の収束性能に影響する重要なパラメータ。
pub const IV_INITIAL_SIGMA: f64 = 0.2;

/// IV計算: Newton-Raphson法の最大反復回数
///
/// 通常3-5回で収束するが、エッジケースを考慮して100回に設定。
/// この回数を超えた場合は収束失敗とみなす。
pub const IV_NEWTON_MAX_ITERATIONS: i32 = 100;

/// IV計算: Newton-Raphson法の収束判定閾値
///
/// 計算価格と市場価格の差がこの値以下なら収束とみなす。
/// 金融実務での価格精度を考慮した高精度閾値。
pub const IV_NEWTON_TOLERANCE: f64 = 1e-8;

/// Newton法: 最大試行回数
///
/// Newton-Raphson法での最大試行回数。
/// これを超えたらBrent法にフォールバック。
pub const NEWTON_MAX_ATTEMPTS: usize = 7;

// ============================================================================
// テスト用精度定数
// ============================================================================

/// Vega計算のテスト許容値
///
/// Vegaは他のGreeksより絶対値が大きいため、専用の許容値を設定。
/// 0.01 = 1セント（オプション価格の実務的な最小単位）
pub const TEST_TOLERANCE_VEGA: f64 = 0.01;

/// 浮動小数点演算の理論的最高精度
///
/// f64のマシンイプシロン近傍での比較に使用。
/// 数学関数の理論値との一致確認で使用。
pub const PRECISION_HIGHEST: f64 = 1e-15;

/// Arrow演算の精度検証用閾値
///
/// FFI境界での精度保証に使用。
pub const ARROW_PRECISION_THRESHOLD: f64 = 1e-10;

/// Black-Scholesテストケースの期待価格範囲（下限）
///
/// S=100, K=100, T=1, r=0.05, σ=0.2の標準ケースでの価格範囲
pub const TEST_BS_PRICE_LOWER: f64 = 8.0;

/// Black-Scholesテストケースの期待価格範囲（上限）
///
/// S=100, K=100, T=1, r=0.05, σ=0.2の標準ケースでの価格範囲
pub const TEST_BS_PRICE_UPPER: f64 = 12.0;

// ============================================================================
// 数値計算用定数
// ============================================================================

/// ボラティリティ二乗の半分の係数（1/2）
///
/// Black-Scholesフォーミュラでのd1計算で使用
pub const HALF: f64 = 0.5;

/// ボラティリティ二乗項の係数（2.0）
///
/// Black-Scholesフォーミュラでのd1計算で使用
pub const VOL_SQUARED_HALF: f64 = 2.0;

/// 正規分布CDF下限（実質的にゼロ）
///
/// x < -8.0 の場合、Φ(x) ≈ 0 として扱う
pub const NORM_CDF_LOWER_BOUND: f64 = -8.0;

/// 正規分布CDF上限（実質的に1）
///
/// x > 8.0 の場合、Φ(x) ≈ 1 として扱う  
pub const NORM_CDF_UPPER_BOUND: f64 = 8.0;

/// 正規分布PDFゼロ判定閾値
///
/// |x| > 10.0 の場合、φ(x) ≈ 0 として扱う
pub const NORM_PDF_ZERO_BOUND: f64 = 10.0;

/// Brent法: 最大反復回数
///
/// Brent法での最大反復回数。
/// 収束が保証されているが、念のため上限を設定。
pub const BRENT_MAX_ITERATIONS: usize = 50;

// ============================================================================
// American Option定数 (BAW and BS2002)
// ============================================================================

/// BAW (Barone-Adesi-Whaley) 早期行使プレミアムの経験的調整係数
///
/// BENCHOP参照値との比較により経験的に導出された調整係数。
/// BAW近似が早期行使プレミアムを過大評価する傾向を補正。
/// この係数により、American put価格の誤差が5.7%から0.98%に改善。
///
/// **制限事項**:
/// - 最適化範囲: ATM（S/K = 0.9-1.1）、T = 0.5-1.5年
/// - 検証済みパラメータ: S=100, K=100, T=1, r=0.05, q=0, σ=0.2
/// - BENCHOP期待値: 6.248, 実装値: 6.309（誤差0.98%）
///
/// **警告**: この係数は単一のパラメータ点で最適化されており、
/// 以下の領域では精度が保証されません：
/// - Deep ITM/OTM（S/K < 0.8 または S/K > 1.2）
/// - 短期オプション（T < 0.1年）
/// - 高ボラティリティ（σ > 0.5）
///
/// 将来的にはmonenessとtime to maturityに応じた
/// 適応的な調整係数の実装が推奨されます。
pub const BAW_DAMPENING_FACTOR: f64 = 0.695;

/// BS2002: βパラメータ最小値
///
/// Bjerksund-Stensland 2002モデルでのβパラメータの下限値。
/// モデルの数値的安定性を保証するための制約。
pub const BS2002_BETA_MIN: f64 = 0.5;

/// BS2002: 収束判定閾値
///
/// Bjerksund-Stensland 2002モデルでの反復計算の収束判定に使用。
/// 早期行使境界の計算等で使用。
pub const BS2002_CONVERGENCE_TOL: f64 = 1e-9;

/// 早期行使境界: 最大反復回数
///
/// 早期行使境界を求める反復法での最大反復回数。
/// 通常5-10回で収束するが、余裕を持たせて20回に設定。
pub const EXERCISE_BOUNDARY_MAX_ITER: usize = 20;

/// BS2002: h(T)計算係数
///
/// Bjerksund-Stensland 2002モデルでのh(T)関数の係数。
/// 早期行使境界の時間依存性を表現する重要なパラメータ。
pub const BS2002_H_FACTOR: f64 = 2.0;

// ============================================================================
// パフォーマンス最適化定数
// ============================================================================

/// キャッシュライン・サイズ（バイト）
///
/// 現代のx86_64プロセッサの標準的なキャッシュラインサイズ。
/// データアラインメントとパディングに使用。
pub const CACHE_LINE_SIZE: usize = 64;

/// L1キャッシュサイズ（バイト）
///
/// 典型的なプロセッサのL1データキャッシュサイズ。
/// 最高速のアクセスが可能なデータサイズ。
pub const L1_CACHE_SIZE: usize = 32 * 1024; // 32KB

/// L2キャッシュサイズ（バイト）
///
/// 典型的なプロセッサのL2キャッシュサイズ。
/// L1より大きいが、依然として高速アクセス可能。
pub const L2_CACHE_SIZE: usize = 256 * 1024; // 256KB

/// L3キャッシュサイズ（バイト）
///
/// 典型的なプロセッサのL3キャッシュサイズ。
/// 複数コア間で共有される最終レベルキャッシュ。
pub const L3_CACHE_SIZE: usize = 8 * 1024 * 1024; // 8MB

/// チャンクサイズ: L1キャッシュ最適化
///
/// L1キャッシュに収まるf64要素数。
/// 最高速の処理が可能なチャンクサイズ。
pub const CHUNK_SIZE_L1: usize = L1_CACHE_SIZE / 8 / 4; // 1024要素 (4-way split)

/// チャンクサイズ: L2キャッシュ最適化
///
/// L2キャッシュに収まるf64要素数。
/// 中規模データの効率的な処理に使用。
pub const CHUNK_SIZE_L2: usize = L2_CACHE_SIZE / 8 / 4; // 8192要素

/// チャンクサイズ: L3キャッシュ最適化
///
/// L3キャッシュに収まるf64要素数。
/// 大規模データの並列処理時のチャンクサイズ。
pub const CHUNK_SIZE_L3: usize = L3_CACHE_SIZE / 8 / 4; // 262144要素

/// 並列化閾値: 小規模
///
/// この要素数以下ではシーケンシャル処理が高速。
/// FFIオーバーヘッドと並列化コストを考慮。
///
/// 2025-09-02: 実測により現在の実装では10,000が最適値と判明。
/// - 実験: 閾値を10,000から50,000に変更してベンチマーク実施
/// - 結果: 10,000要素のデータで処理時間が97%増加（性能が約半分に劣化）
/// - 結論: 現在の実装構造では10,000要素から並列化が有効
/// - 注: プロトタイプ実装では50,000が最適だったが、実装が異なれば最適値も異なる
///
/// 環境変数QUANTFORGE_PARALLEL_THRESHOLDで個別環境向けに調整可能。
pub const PARALLEL_THRESHOLD_SMALL: usize = 10_000;

/// 並列化閾値を環境変数から取得（オーバーライド可能）
///
/// QUANTFORGE_PARALLEL_THRESHOLD環境変数で調整可能。
/// 設定されていない場合はPARALLEL_THRESHOLD_SMALLを使用。
pub fn get_parallel_threshold() -> usize {
    std::env::var("QUANTFORGE_PARALLEL_THRESHOLD")
        .ok()
        .and_then(|s| s.parse().ok())
        .unwrap_or(PARALLEL_THRESHOLD_SMALL)
}

/// マイクロバッチ閾値: 極小規模
///
/// この要素数以下では専用の最適化処理を使用。
/// ループアンローリングとインデックスベースアクセスで高速化。
/// 2025-08-31: 100件バッチの性能改善のため追加。
pub const MICRO_BATCH_THRESHOLD: usize = 200;

/// 並列化閾値: 中規模
///
/// この要素数以上で並列化が有効になり始める。
/// スレッドプール起動のオーバーヘッドを考慮。
pub const PARALLEL_THRESHOLD_MEDIUM: usize = 200_000;

/// 並列化閾値: 大規模
///
/// この要素数以上で積極的な並列化が有効。
/// 全CPUコアを活用した処理を行う。
pub const PARALLEL_THRESHOLD_LARGE: usize = 1_000_000;

/// スレッドあたり最小ワークロード
///
/// 並列化時の各スレッドが処理する最小要素数。
/// これ以下では並列化のオーバーヘッドが大きすぎる。
pub const MIN_WORK_PER_THREAD: usize = 1000;

/// 最大並列度
///
/// 同時に使用する最大スレッド数。
/// システムのCPUコア数に基づいて動的に調整。
pub const MAX_PARALLELISM: usize = 32;

/// プリフェッチ距離（要素数）
///
/// CPUプリフェッチのための先読み距離。
/// 現在処理中の要素からこの数だけ先のデータをプリフェッチ。
pub const PREFETCH_DISTANCE: usize = 8;

// ============================================================================
// Greeks計算用定数
// ============================================================================

/// Greeksの有限差分計算用: 価格変化率
/// Delta, Gammaの計算で使用する価格の変化割合
pub const GREEK_PRICE_CHANGE_RATIO: f64 = 0.01; // 1%

/// Greeksの有限差分計算用: ボラティリティ変化幅
/// Vegaの計算で使用するボラティリティの変化幅
pub const GREEK_VOL_CHANGE: f64 = 0.01; // 1%

/// Greeksの有限差分計算用: 金利変化幅（ベーシスポイント）
/// Rhoの計算で使用する金利の変化幅
pub const GREEK_RATE_CHANGE: f64 = 0.0001; // 1 basis point

/// 満期直前判定閾値
/// この時間以下の場合、満期直前として特別処理
pub const TIME_NEAR_EXPIRY_THRESHOLD: f64 = 1e-10;

/// 配当利回りゼロ判定閾値
/// アメリカンオプションで早期行使判定に使用
pub const DIVIDEND_ZERO_THRESHOLD: f64 = 1e-10;

/// ボラティリティゼロ判定閾値
/// 決定論的価格計算への切り替え判定
pub const VOLATILITY_ZERO_THRESHOLD: f64 = 1e-10;

// ============================================================================
// 標準テストケース定数
// ============================================================================

/// 標準テストケース: スポット価格
pub const TEST_SPOT: f64 = 100.0;

/// 標準テストケース: 権利行使価格
pub const TEST_STRIKE: f64 = 100.0;

/// 標準テストケース: 満期（年）
pub const TEST_TIME: f64 = 1.0;

/// 標準テストケース: 無リスク金利
pub const TEST_RATE: f64 = 0.05;

/// 標準テストケース: ボラティリティ
pub const TEST_VOLATILITY: f64 = 0.2;

/// 標準テストケース: 配当利回り
pub const TEST_DIVIDEND_YIELD: f64 = 0.02;

/// Black-Scholesテスト期待値範囲（formulas.rs用）
pub const TEST_BS_FORMULAS_PRICE_LOWER: f64 = 6.0;
pub const TEST_BS_FORMULAS_PRICE_UPPER: f64 = 10.0;

/// Black76テスト期待値範囲（formulas.rs用）
pub const TEST_BLACK76_FORMULAS_PRICE_LOWER: f64 = 7.0;
pub const TEST_BLACK76_FORMULAS_PRICE_UPPER: f64 = 11.0;

// ============================================================================
// 一般的な数値変換定数
// ============================================================================

/// パーセンテージ変換乗数
///
/// 小数値（0.05）をパーセンテージ表示（5%）に変換する際の乗数。
/// エラーメッセージやレポート生成で使用。
pub const PERCENTAGE_MULTIPLIER: f64 = 100.0;

// ============================================================================
// Market Data定数
// ============================================================================

/// Market utilities用定数モジュール
pub mod market {
    /// デフォルトの異常スプレッド閾値（仲値に対する比率）
    /// オプション市場を考慮して50%に設定
    pub const DEFAULT_ABNORMAL_SPREAD_THRESHOLD_PCT: f64 = 0.50; // 50%

    /// クロススプレッドの許容誤差（数値誤差を考慮）
    pub const CROSS_SPREAD_TOLERANCE: f64 = 1e-10;

    /// 最小有効価格（負値チェック用、ゼロは有効）
    pub const MIN_VALID_PRICE: f64 = 0.0;

    /// 最小有効数量（負値チェック用、ゼロは有効）
    pub const MIN_VALID_QUANTITY: f64 = 0.0;
}
