#!/usr/bin/env python3
"""
简化的工具类测试

测试核心工具类功能，避免复杂依赖
"""

import sys
from datetime import datetime, date
from pathlib import Path

# 添加项目根目录到Python路径
project_root = Path(__file__).parent
sys.path.insert(0, str(project_root))


def test_basic_imports():
    """测试基本导入"""
    print("📦 测试基本导入")
    print("-" * 40)
    
    try:
        # 测试日期工具导入
        from pyadvincekit.utils.date_utils import DateUtils, now, utc_now, format_duration
        print("✅ 日期工具类导入成功")
        
        # 测试金额工具导入
        from pyadvincekit.utils.money_utils import Money, MoneyUtils, Currency, money, cny, usd
        print("✅ 金额工具类导入成功")
        
        # 测试HTTP工具导入（可能失败但不影响核心功能）
        try:
            from pyadvincekit.utils.http_utils import HTTPUtils
            print("✅ HTTP工具类导入成功")
        except ImportError as e:
            print(f"⚠️ HTTP工具类导入失败（缺少依赖）: {e}")
        
        print("✅ 基本导入测试通过")
        return True
        
    except Exception as e:
        print(f"❌ 基本导入测试失败: {e}")
        return False


def test_date_utils_basic():
    """测试日期工具类基本功能"""
    print("\n📅 测试日期工具类基本功能")
    print("-" * 40)
    
    try:
        from pyadvincekit.utils.date_utils import DateUtils, format_duration
        
        # 测试当前时间（使用简单的timezone）
        current_time = datetime.now()
        print(f"✅ 当前时间: {current_time}")
        
        # 测试日期格式化
        formatted = current_time.strftime(DateUtils.FORMAT_DATETIME)
        print(f"✅ 格式化时间: {formatted}")
        
        # 测试时间计算
        tomorrow = DateUtils.add_days(current_time, 1)
        print(f"✅ 明天: {tomorrow.strftime('%Y-%m-%d %H:%M:%S')}")
        
        # 测试时长格式化
        duration = format_duration(3661.5)  # 1小时1分1.5秒
        print(f"✅ 时长格式化: {duration}")
        
        # 测试工作日判断
        is_workday = DateUtils.is_workday(current_time)
        weekday_name = DateUtils.get_weekday_name(current_time, "cn")
        print(f"✅ 今天是{weekday_name}, {'工作日' if is_workday else '休息日'}")
        
        # 测试年龄计算
        birthday = date(1990, 5, 15)
        age = DateUtils.age_from_birthday(birthday)
        print(f"✅ 1990-05-15出生的年龄: {age}岁")
        
        print("✅ 日期工具类基本功能测试通过")
        return True
        
    except Exception as e:
        print(f"❌ 日期工具类基本功能测试失败: {e}")
        return False


def test_money_utils_basic():
    """测试金额工具类基本功能"""
    print("\n💰 测试金额工具类基本功能")
    print("-" * 40)
    
    try:
        from pyadvincekit.utils.money_utils import Money, MoneyUtils, Currency, money, cny, usd
        
        # 测试基本金额操作
        amount1 = cny("100.50")
        amount2 = cny(50.25)
        total = amount1 + amount2
        print(f"✅ 金额计算: {amount1} + {amount2} = {total}")
        
        # 测试金额格式化
        formatted = total.format(show_symbol=True, thousands_separator=",")
        print(f"✅ 格式化显示: {formatted}")
        
        # 测试金额比较
        print(f"✅ 金额比较: {amount1} > {amount2} = {amount1 > amount2}")
        
        # 测试百分比计算
        discount = MoneyUtils.calculate_percentage(total, 10)  # 10%折扣
        final_price = total - discount
        print(f"✅ 10%折扣: 原价{total}, 折扣{discount}, 实付{final_price}")
        
        # 测试税额计算
        tax_amount, total_with_tax = MoneyUtils.calculate_tax(cny(100), 13)  # 13%增值税
        print(f"✅ 税额计算: 不含税价100元, 税额{tax_amount}, 含税总价{total_with_tax}")
        
        # 测试金额分配
        total_money = cny(100)
        ratios = [3, 2, 5]  # 按3:2:5分配
        allocated = MoneyUtils.allocate_money(total_money, ratios)
        print(f"✅ 金额分配: 100元按3:2:5分配 = {[str(a) for a in allocated]}")
        
        # 测试中文大写
        chinese_amount = MoneyUtils.format_chinese_amount(cny(12345.67))
        print(f"✅ 中文大写: 12345.67元 = {chinese_amount}")
        
        # 测试货币转换
        usd_amount = usd(100)
        cny_converted = MoneyUtils.convert_currency(usd_amount, Currency.CNY, 7.2)
        print(f"✅ 货币转换: {usd_amount} × 7.2 = {cny_converted}")
        
        print("✅ 金额工具类基本功能测试通过")
        return True
        
    except Exception as e:
        print(f"❌ 金额工具类基本功能测试失败: {e}")
        return False


def test_http_utils_basic():
    """测试HTTP工具类基本功能（不依赖网络）"""
    print("\n🌐 测试HTTP工具类基本功能")
    print("-" * 40)
    
    try:
        # 只测试不需要额外依赖的功能
        try:
            from pyadvincekit.utils.http_utils import HTTPUtils
            
            # 测试URL解析
            url = "https://api.example.com:8080/v1/users?page=1&size=10#section"
            parsed = HTTPUtils.parse_url(url)
            print(f"✅ URL解析: {url}")
            print(f"   - 域名: {parsed['hostname']}")
            print(f"   - 端口: {parsed['port']}")
            print(f"   - 路径: {parsed['path']}")
            print(f"   - 查询参数: {parsed['query_dict']}")
            
            # 测试URL验证
            valid_urls = ["https://www.example.com", "http://localhost:8080"]
            invalid_urls = ["not-a-url", "just-text"]
            
            print("✅ URL验证:")
            for url in valid_urls:
                is_valid = HTTPUtils.is_valid_url(url)
                print(f"   - {url}: {'有效' if is_valid else '无效'}")
            
            for url in invalid_urls:
                is_valid = HTTPUtils.is_valid_url(url)
                print(f"   - {url}: {'有效' if is_valid else '无效'}")
            
            # 测试查询字符串构建
            params = {"page": 1, "size": 10, "status": "active"}
            query_string = HTTPUtils.build_query_string(params)
            print(f"✅ 查询字符串构建: {params} -> {query_string}")
            
            # 测试JSON解析
            json_text = '{"name": "test", "value": 123}'
            parsed_json = HTTPUtils.safe_json_decode(json_text)
            print(f"✅ JSON解析: {json_text} -> {parsed_json}")
            
            invalid_json = 'not-json-text'
            parsed_invalid = HTTPUtils.safe_json_decode(invalid_json)
            print(f"✅ 无效JSON解析: {invalid_json} -> {parsed_invalid}")
            
            print("✅ HTTP工具类基本功能测试通过")
            return True
            
        except ImportError as e:
            print(f"⚠️ HTTP工具类导入失败（缺少依赖）: {e}")
            print("💡 请安装依赖: pip install aiohttp requests")
            return True  # 不算失败，只是缺少依赖
        
    except Exception as e:
        print(f"❌ HTTP工具类基本功能测试失败: {e}")
        return False


def test_package_integration():
    """测试包集成"""
    print("\n🔗 测试包集成")
    print("-" * 40)
    
    try:
        # 测试从主包导入
        from pyadvincekit import (
            now, utc_now, format_duration,
            money, cny, usd,
            Money, Currency
        )
        
        print("✅ 从主包导入成功")
        
        # 测试简单集成使用
        current_time = datetime.now()  # 使用简单的datetime
        amount = cny(100.50)
        duration = format_duration(120.5)
        
        print(f"✅ 当前时间: {current_time}")
        print(f"✅ 金额: {amount}")
        print(f"✅ 时长: {duration}")
        
        print("✅ 包集成测试通过")
        return True
        
    except Exception as e:
        print(f"❌ 包集成测试失败: {e}")
        return False


def main():
    """主测试函数"""
    print("🚀 PyAdvanceKit 第二阶段工具类简化测试")
    print("=" * 60)
    
    # 运行所有测试
    tests = [
        ("基本导入", test_basic_imports),
        ("日期工具类基本功能", test_date_utils_basic),
        ("金额工具类基本功能", test_money_utils_basic),
        ("HTTP工具类基本功能", test_http_utils_basic),
        ("包集成", test_package_integration)
    ]
    
    results = []
    for test_name, test_func in tests:
        try:
            result = test_func()
            results.append((test_name, result))
        except Exception as e:
            print(f"❌ {test_name}测试异常: {e}")
            results.append((test_name, False))
    
    # 显示测试结果
    print("\n" + "=" * 60)
    print("📊 测试结果汇总")
    print("=" * 60)
    
    passed = 0
    total = len(results)
    
    for test_name, result in results:
        status = "✅ 通过" if result else "❌ 失败"
        print(f"{test_name}: {status}")
        if result:
            passed += 1
    
    print("-" * 60)
    print(f"总计: {passed}/{total} 通过 ({passed/total*100:.1f}%)")
    
    if passed == total:
        print("🎉 所有测试通过！第二阶段工具类核心功能正常")
    else:
        print("⚠️ 部分测试失败，请检查相关功能")
    
    print("\n💡 注意事项:")
    print("- 如需完整HTTP功能，请安装: pip install aiohttp requests")
    print("- 日期时区功能需要: pip install pytz (Python < 3.9)")
    
    return passed == total


if __name__ == "__main__":
    success = main()
    sys.exit(0 if success else 1)
