"""
数据库迁移管理模块

提供Alembic数据库迁移的封装和管理功能。
"""

import os
from pathlib import Path
from typing import Optional

from alembic import command
from alembic.config import Config as AlembicConfig
from alembic.runtime.migration import MigrationContext
from alembic.script import ScriptDirectory
from sqlalchemy import Engine, text
from sqlalchemy.ext.asyncio import AsyncEngine

from pyadvincekit.core.config import Settings, get_settings
from pyadvincekit.core.database import get_database_manager
from pyadvincekit.core.exceptions import DatabaseError
from pyadvincekit.logging import get_logger

logger = get_logger(__name__)


class MigrationManager:
    """数据库迁移管理器"""

    def __init__(
        self, 
        settings: Optional[Settings] = None,
        alembic_ini_path: Optional[str] = None
    ) -> None:
        self.settings = settings or get_settings()
        self.alembic_ini_path = alembic_ini_path or self.settings.database_alembic_ini_path
        self._alembic_config: Optional[AlembicConfig] = None

    @property
    def alembic_config(self) -> AlembicConfig:
        """获取Alembic配置"""
        if self._alembic_config is None:
            self._alembic_config = self._create_alembic_config()
        return self._alembic_config

    def _create_alembic_config(self) -> AlembicConfig:
        """创建Alembic配置"""
        # 检查alembic.ini文件是否存在
        if not os.path.exists(self.alembic_ini_path):
            self._create_alembic_ini()

        config = AlembicConfig(self.alembic_ini_path)
        
        # 设置数据库URL（使用同步驱动）
        database_url = self.settings.get_database_url(async_driver=False)
        config.set_main_option("sqlalchemy.url", database_url)
        
        return config

    def _create_alembic_ini(self) -> None:
        """创建默认的alembic.ini文件"""
        alembic_ini_content = f"""# Alembic config file - Generated by PyAdvanceKit

[alembic]
# Alembic version storage location
script_location = alembic

# Template file directory
file_template = %%(year)d%%(month).2d%%(day).2d_%%(hour).2d%%(minute).2d_%%(rev)s_%%(slug)s

# sys.path path (for importing app modules)
prepend_sys_path = .

# Timezone setting
timezone = 

# SQLAlchemy URL (set dynamically at runtime)
sqlalchemy.url = 

# SQLAlchemy engine configuration
sqlalchemy.echo = false
sqlalchemy.echo_pool = false
sqlalchemy.pool_size = 10
sqlalchemy.max_overflow = 20

# Logging configuration
[loggers]
keys = root,sqlalchemy,alembic

[handlers]
keys = console

[formatters]
keys = generic

[logger_root]
level = WARN
handlers = console
qualname =

[logger_sqlalchemy]
level = WARN
handlers =
qualname = sqlalchemy.engine

[logger_alembic]
level = INFO
handlers =
qualname = alembic

[handler_console]
class = StreamHandler
args = (sys.stderr,)
level = NOTSET
formatter = generic

[formatter_generic]
format = %(levelname)-5.5s [%(name)s] %(message)s
datefmt = %H:%M:%S
"""
        
        with open(self.alembic_ini_path, "w", encoding="utf-8") as f:
            f.write(alembic_ini_content)
        
        logger.info(f"Created alembic.ini at {self.alembic_ini_path}")

    def init_alembic(self, directory: str = "alembic") -> None:
        """初始化Alembic目录结构"""
        try:
            if os.path.exists(directory):
                logger.warning(f"Alembic directory {directory} already exists")
                return
            
            # 创建Alembic目录结构
            command.init(self.alembic_config, directory)
            
            # 修改env.py文件以支持异步和自动模型导入
            self._update_env_py(directory)
            
            logger.info(f"Initialized Alembic in directory: {directory}")
            
        except Exception as e:
            logger.error(f"Failed to initialize Alembic: {e}")
            raise DatabaseError(f"Failed to initialize migrations: {e}")

    def _update_env_py(self, directory: str) -> None:
        """更新env.py文件以支持PyAdvanceKit"""
        env_py_path = os.path.join(directory, "env.py")
        
        env_py_content = '''"""
Alembic Environment Configuration - Generated by PyAdvanceKit
"""

import asyncio
import logging
from logging.config import fileConfig

from sqlalchemy import pool
from sqlalchemy.engine import Connection
from sqlalchemy.ext.asyncio import async_engine_from_config

from alembic import context

# Import PyAdvanceKit models
from pyadvincekit.models.base import Base

# Get Alembic config object
config = context.config

# Setup logging
if config.config_file_name is not None:
    fileConfig(config.config_file_name)

# Set MetaData for auto-generating migrations
target_metadata = Base.metadata

# Other configuration values
def get_exclude_tables():
    """Get tables to exclude from migrations"""
    return []

def run_migrations_offline() -> None:
    """Run migrations in offline mode"""
    url = config.get_main_option("sqlalchemy.url")
    context.configure(
        url=url,
        target_metadata=target_metadata,
        literal_binds=True,
        dialect_opts={"paramstyle": "named"},
        compare_type=True,
        compare_server_default=True,
        exclude_object=lambda obj, name, type_, reflected, compare_to: (
            type_ == "table" and name in get_exclude_tables()
        ),
    )

    with context.begin_transaction():
        context.run_migrations()


def do_run_migrations(connection: Connection) -> None:
    """Execute migrations"""
    context.configure(
        connection=connection,
        target_metadata=target_metadata,
        compare_type=True,
        compare_server_default=True,
        exclude_object=lambda obj, name, type_, reflected, compare_to: (
            type_ == "table" and name in get_exclude_tables()
        ),
    )

    with context.begin_transaction():
        context.run_migrations()


async def run_async_migrations() -> None:
    """Run migrations in async mode"""
    connectable = async_engine_from_config(
        config.get_section(config.config_ini_section, {}),
        prefix="sqlalchemy.",
        poolclass=pool.NullPool,
    )

    async with connectable.connect() as connection:
        await connection.run_sync(do_run_migrations)

    await connectable.dispose()


def run_migrations_online() -> None:
    """Run migrations in online mode"""
    asyncio.run(run_async_migrations())


if context.is_offline_mode():
    run_migrations_offline()
else:
    run_migrations_online()
'''
        
        with open(env_py_path, "w", encoding="utf-8") as f:
            f.write(env_py_content)
        
        logger.info(f"Updated env.py at {env_py_path}")

    def create_migration(
        self, 
        message: str,
        autogenerate: bool = True,
        head: str = "head"
    ) -> str:
        """
        创建新的迁移文件
        
        Args:
            message: 迁移描述信息
            autogenerate: 是否自动生成迁移内容
            head: 基于的版本
            
        Returns:
            新迁移的版本ID
        """
        try:
            revision = command.revision(
                self.alembic_config,
                message=message,
                autogenerate=autogenerate,
                head=head
            )
            
            logger.info(f"Created migration: {revision}")
            return revision
            
        except Exception as e:
            logger.error(f"Failed to create migration: {e}")
            raise DatabaseError(f"Failed to create migration: {e}")

    def upgrade(self, revision: str = "head") -> None:
        """
        升级数据库到指定版本
        
        Args:
            revision: 目标版本，默认为最新版本
        """
        try:
            command.upgrade(self.alembic_config, revision)
            logger.info(f"Upgraded database to revision: {revision}")
            
        except Exception as e:
            logger.error(f"Failed to upgrade database: {e}")
            raise DatabaseError(f"Failed to upgrade database: {e}")
    
    def generate_sql(self, revision: str = "head", output_file: Optional[str] = None) -> str:
        """
        生成升级SQL语句（不执行）
        
        Args:
            revision: 目标版本，默认为最新版本
            output_file: SQL输出文件路径，如果不指定则返回字符串
            
        Returns:
            生成的SQL语句字符串
        """
        try:
            import io
            import sys
            import os
            import subprocess
            import tempfile
            from contextlib import redirect_stdout
            
            # 设置环境变量确保UTF-8编码
            env = os.environ.copy()
            env['PYTHONIOENCODING'] = 'utf-8'
            
            # 使用subprocess调用alembic命令来避免编码问题
            try:
                # 构建alembic命令
                alembic_cmd = [
                    sys.executable, '-m', 'alembic',
                    '-c', self.alembic_ini_path,
                    'upgrade', revision, '--sql'
                ]
                
                # 执行命令并捕获输出
                result = subprocess.run(
                    alembic_cmd,
                    capture_output=True,
                    text=True,
                    encoding='utf-8',
                    env=env,
                    cwd=os.path.dirname(self.alembic_ini_path) or '.'
                )
                
                if result.returncode == 0:
                    sql_content = result.stdout
                else:
                    # 如果subprocess失败，尝试原来的方法
                    raise subprocess.CalledProcessError(result.returncode, alembic_cmd, result.stderr)
                    
            except (subprocess.CalledProcessError, FileNotFoundError):
                # 回退到原来的方法，但设置正确的编码
                logger.warning("Subprocess method failed, falling back to direct call")
                
                # 临时设置标准输出编码
                old_stdout = sys.stdout
                sql_buffer = io.StringIO()
                
                try:
                    # 重定向stdout并设置编码
                    sys.stdout = sql_buffer
                    
                    # 临时修改Alembic配置以避免编码问题
                    config = self.alembic_config
                    
                    command.upgrade(config, revision, sql=True)
                    sql_content = sql_buffer.getvalue()
                    
                finally:
                    sys.stdout = old_stdout
            
            # 如果指定了输出文件，写入文件
            if output_file:
                with open(output_file, 'w', encoding='utf-8') as f:
                    f.write(sql_content)
                logger.info(f"SQL statements written to: {output_file}")
            
            logger.info("Generated SQL statements for database upgrade")
            return sql_content
            
        except Exception as e:
            logger.error(f"Failed to generate SQL: {e}")
            raise DatabaseError(f"Failed to generate SQL: {e}")

    def downgrade(self, revision: str) -> None:
        """
        降级数据库到指定版本
        
        Args:
            revision: 目标版本
        """
        try:
            command.downgrade(self.alembic_config, revision)
            logger.info(f"Downgraded database to revision: {revision}")
            
        except Exception as e:
            logger.error(f"Failed to downgrade database: {e}")
            raise DatabaseError(f"Failed to downgrade database: {e}")

    def get_current_revision(self) -> Optional[str]:
        """获取当前数据库版本"""
        try:
            # 需要同步连接来获取当前版本
            db_manager = get_database_manager()
            sync_url = self.settings.get_database_url(async_driver=False)
            
            from sqlalchemy import create_engine
            engine = create_engine(sync_url)
            
            with engine.connect() as connection:
                context = MigrationContext.configure(connection)
                current_rev = context.get_current_revision()
            
            return current_rev
            
        except Exception as e:
            logger.error(f"Failed to get current revision: {e}")
            return None

    def get_migration_history(self) -> list:
        """获取迁移历史"""
        try:
            script_dir = ScriptDirectory.from_config(self.alembic_config)
            revisions = []
            
            for revision in script_dir.walk_revisions():
                revisions.append({
                    "revision": revision.revision,
                    "down_revision": revision.down_revision,
                    "branch_labels": revision.branch_labels,
                    "depends_on": revision.depends_on,
                    "doc": revision.doc,
                })
            
            return revisions
            
        except Exception as e:
            logger.error(f"Failed to get migration history: {e}")
            return []

    def check_pending_migrations(self) -> bool:
        """检查是否有待执行的迁移"""
        try:
            current_rev = self.get_current_revision()
            script_dir = ScriptDirectory.from_config(self.alembic_config)
            head_rev = script_dir.get_current_head()
            
            return current_rev != head_rev
            
        except Exception as e:
            logger.error(f"Failed to check pending migrations: {e}")
            return False


# 全局迁移管理器实例
_migration_manager: Optional[MigrationManager] = None


def get_migration_manager() -> MigrationManager:
    """获取迁移管理器实例"""
    global _migration_manager
    if _migration_manager is None:
        _migration_manager = MigrationManager()
    return _migration_manager


def init_migrations(directory: str = "alembic") -> None:
    """初始化数据库迁移"""
    manager = get_migration_manager()
    manager.init_alembic(directory)


def create_migration(message: str, autogenerate: bool = True) -> str:
    """创建新迁移"""
    manager = get_migration_manager()
    return manager.create_migration(message, autogenerate)


def upgrade_database(revision: str = "head") -> None:
    """升级数据库"""
    manager = get_migration_manager()
    manager.upgrade(revision)


def generate_upgrade_sql(revision: str = "head", output_file: Optional[str] = None) -> str:
    """
    生成数据库升级SQL语句（不执行）
    
    Args:
        revision: 目标版本，默认为最新版本
        output_file: SQL输出文件路径，如果不指定则返回字符串
        
    Returns:
        生成的SQL语句字符串
    """
    manager = get_migration_manager()
    return manager.generate_sql(revision, output_file)


def downgrade_database(revision: str) -> None:
    """降级数据库"""
    manager = get_migration_manager()
    manager.downgrade(revision)
