import logging
import os

from sandroid.core.toolbox import Toolbox

from .datagather import DataGather

try:
    from dexray_intercept import AppProfiler, setup_frida_device
except ImportError:
    logger = logging.getLogger(__name__)
    logger.warning(
        "dexray-intercept package not installed. Malware monitoring will be disabled."
    )
    AppProfiler = None
    setup_frida_device = None

logger = logging.getLogger(__name__)

# Set up dedicated dexray log file
def _setup_dexray_logging():
    """Set up dedicated file logging for dexray-intercept in the Sandroid results folder."""
    dexray_logger = logging.getLogger('dexray_intercept')

    # Check if we already have a file handler to avoid duplicates
    has_file_handler = any(
        isinstance(handler, logging.FileHandler)
        for handler in dexray_logger.handlers
    )

    if not has_file_handler and os.getenv('RAW_RESULTS_PATH'):
        file_handler = logging.FileHandler(
            f"{os.getenv('RAW_RESULTS_PATH')}dexray.log"
        )
        file_handler.setLevel(logging.DEBUG)
        file_formatter = logging.Formatter(
            "%(asctime)s~%(levelname)s~%(message)s~module:%(module)s~function:%(funcName)s"
        )
        file_handler.setFormatter(file_formatter)
        dexray_logger.addHandler(file_handler)
        dexray_logger.setLevel(logging.DEBUG)

        logger.info(f"Dexray logs will be saved to {os.getenv('RAW_RESULTS_PATH')}dexray.log")


class MalwareMonitor(DataGather):
    """This class monitors provides a wrapper for the Dexray Intercept package to implement it into Sandroid.

    .. note::
        Inherits from :class:`DataGather`.

    **Attributes:**
        - **logger** (*Logger*): Logger instance for the class.
        - **last_results** (*dict*): Stores the last results of the monitoring.
        - **job_manager** (*JobManager*): Manages Frida jobs.
        - **process_session** (*Session*): Current Frida process session.
        - **app_package** (*str*): Package name of the application being monitored.
        - **profiler** (*AppProfiler*): Instance of the AppProfiler class.
        - **frida_script_path** (*str*): Path to the Frida script.
        - **running** (*bool*): Indicates if the monitoring is currently running.
        - **last_output** (*str*): Stores the last output of the profiling.
        - **_new_results_available** (*bool*): Indicates if there are new results available.
    """

    def __init__(self, path_filters=None):
        """Initializes the MalwareMonitor with dexray-intercept API.

        :param path_filters: List of file paths to filter for monitoring.
        :type path_filters: list or None
        """
        super().__init__()

        if AppProfiler is None:
            logger.error(
                "dexray-intercept not available. MalwareMonitor will not function."
            )
            self.available = False
            return

        self.available = True
        if path_filters == []:
            self.path_filters = None
        else:
            self.path_filters = path_filters

        self.last_results = {}
        self.job_manager = Toolbox.get_frida_job_manager()
        self.process_session = None  # Will be set up with new API
        self.app_package = None
        self.frida_device = None
        self.profiler = None

        self.running = False
        self.last_output = None
        self._new_results_available = False

        # Hook configuration for dexray-intercept
        self.hook_config = {
            "aes_hooks": True,
            "web_hooks": True,
            "file_system_hooks": True,
            "socket_hooks": True,
            "process_hooks": True,
            "ipc_hooks": True,
        }

        # Set up dedicated dexray logging
        _setup_dexray_logging()

        logger.info(
            "MalwareMonitor initialized with dexray-intercept API and path filters: "
            + str(self.path_filters)
        )

    def gather(self):
        """Gather data from the monitored application using dexray-intercept.

        .. warning::
            Context dependent behavior: Calling this method acts as a toggle, it starts or stops the monitoring process based on the current state.
        """
        if not self.available:
            logger.error(
                "MalwareMonitor not available due to missing dexray-intercept package"
            )
            return

        if self.running:
            # Stop monitoring and collect results
            try:
                if self.profiler:
                    self.profiler.stop_profiling()
                    self.last_output = self.profiler.get_profile_data()
                    logger.info("MalwareMonitor stopped and data collected")
                self.running = False
                Toolbox.malware_monitor_running = False
                self._new_results_available = True
            except Exception as e:
                logger.error(f"Error stopping MalwareMonitor: {e}")

        elif not self.running:
            # Start monitoring
            try:
                self.app_package, _ = Toolbox.get_spotlight_application()

                # Set up Frida device and attach to app
                self.frida_device = setup_frida_device()
                self.process_session = self.frida_device.attach(self.app_package)

                # Create profiler with new API
                self.profiler = AppProfiler(
                    self.process_session,
                    verbose_mode=True,
                    output_format="JSON",
                    hook_config=self.hook_config,
                )

                # Start profiling
                script = self.profiler.start_profiling()
                self.running = True
                Toolbox.malware_monitor_running = True

                logger.info(f"MalwareMonitor started for package: {self.app_package}")

            except Exception as e:
                logger.error(f"Error starting MalwareMonitor: {e}")
                self.running = False
                Toolbox.malware_monitor_running = False

    def has_new_results(self):
        """Check if there are new results available.

        :returns: True if there are new results, False otherwise.
        :rtype: bool
        """
        if self.running:
            return False
        return self._new_results_available  # Updated variable name

    def return_data(self):
        """Return the last profiling data.

        This method returns the last profiling data and resets the new results flag.

        :returns: The last profiling data in JSON format.
        :rtype: str
        """
        self._new_results_available = False  # Updated variable name
        return self.last_output

    def pretty_print(self):
        """Not implemented"""
