#!/usr/bin/env python3
"""
Comprehensive Integration Tests for Updated Memory System

This test suite verifies:
1. Memory files are created in simple list format with timestamps
2. Timestamps are updated on every file modification
3. MEMORIES field in agent responses replaces all memories
4. "remember" field for incremental memory updates
5. Hook processing correctly extracts and processes MEMORIES field
6. Memory deduplication and optimization
7. PROJECT-based memories work correctly (.claude-mpm/memories/)

Test Requirements:
- Simple list format with bullet points
- Timestamp updates on changes
- MEMORIES field complete replacement
- remember field incremental updates
- Hook integration
- Deduplication
- Project directory storage
"""

import json
import tempfile
import time
from datetime import datetime, timezone
from pathlib import Path
from unittest.mock import MagicMock, patch

from claude_mpm.core.config import Config
from claude_mpm.hooks.claude_hooks.hook_handler import ClaudeHookHandler

# Import the memory system components
from claude_mpm.services.agents.memory.agent_memory_manager import AgentMemoryManager
from claude_mpm.services.agents.memory.content_manager import MemoryContentManager


class TestMemorySystemIntegration:
    """Comprehensive integration tests for the updated memory system."""

    def setup_method(self):
        """Set up test environment for each test."""
        # Create temporary directory for testing
        self.temp_dir = Path(tempfile.mkdtemp())
        self.memories_dir = self.temp_dir / ".claude-mpm" / "memories"
        self.memories_dir.mkdir(parents=True, exist_ok=True)

        # Initialize memory manager with test directory
        self.config = Config()
        self.memory_manager = AgentMemoryManager(
            config=self.config, working_directory=self.temp_dir
        )

        # Content manager for direct testing
        self.content_manager = MemoryContentManager(self.memory_manager.memory_limits)

        print(f"Test setup: temporary directory = {self.temp_dir}")

    def teardown_method(self):
        """Clean up test environment."""
        import shutil

        if self.temp_dir.exists():
            shutil.rmtree(self.temp_dir)

    def test_memory_file_creation_simple_list_format(self):
        """Test 1: Verify memory files are created in simple list format with timestamps."""
        print("\n=== Test 1: Memory File Creation in Simple List Format ===")

        # Test creation of new memory file
        agent_id = "test_agent"
        self.memory_manager.load_agent_memory(agent_id)

        # Verify file was created
        memory_file = self.memories_dir / f"{agent_id}_memories.md"
        assert memory_file.exists(), "Memory file should be created"

        print(f"✓ Memory file created: {memory_file}")

        # Read and verify structure
        content = memory_file.read_text()
        lines = content.split("\n")

        # Verify header structure
        assert lines[0].startswith("# Agent Memory:"), "Should have proper header"
        assert lines[1].startswith("<!-- Last Updated:"), "Should have timestamp"
        assert lines[1].endswith("Z -->"), "Timestamp should end with Z"

        print(f"✓ Header format correct: {lines[0]}")
        print(f"✓ Timestamp format correct: {lines[1]}")

        # Verify timestamp is recent (within last minute)
        timestamp_match = content.split("<!-- Last Updated: ")[1].split(" -->")[0]
        timestamp = datetime.fromisoformat(timestamp_match.replace("Z", "+00:00"))
        time_diff = datetime.now(timezone.utc).astimezone() - timestamp.astimezone()
        assert time_diff.total_seconds() < 60, "Timestamp should be recent"

        print(f"✓ Timestamp is recent: {timestamp}")

        return True

    def test_timestamp_updates_on_modifications(self):
        """Test 2: Test that timestamps are updated on every file modification."""
        print("\n=== Test 2: Timestamp Updates on Modifications ===")

        agent_id = "timestamp_test_agent"

        # Create initial memory
        self.memory_manager.load_agent_memory(agent_id)
        memory_file = self.memories_dir / f"{agent_id}_memories.md"

        # Get initial timestamp
        initial_content = memory_file.read_text()
        initial_timestamp = initial_content.split("<!-- Last Updated: ")[1].split(
            " -->"
        )[0]

        print(f"Initial timestamp: {initial_timestamp}")

        # Wait a moment to ensure timestamp difference
        time.sleep(1.1)

        # Add a memory item
        new_item = "This is a test memory item"
        success = self.memory_manager.add_learning(agent_id, new_item)
        assert success, "Adding learning should succeed"

        # Verify timestamp was updated
        updated_content = memory_file.read_text()
        updated_timestamp = updated_content.split("<!-- Last Updated: ")[1].split(
            " -->"
        )[0]

        print(f"Updated timestamp: {updated_timestamp}")

        assert updated_timestamp != initial_timestamp, "Timestamp should be updated"

        # Verify new timestamp is more recent
        initial_dt = datetime.fromisoformat(initial_timestamp.replace("Z", "+00:00"))
        updated_dt = datetime.fromisoformat(updated_timestamp.replace("Z", "+00:00"))
        assert updated_dt > initial_dt, "New timestamp should be later"

        print(f"✓ Timestamp correctly updated from {initial_dt} to {updated_dt}")

        # Verify the new item was added with bullet point
        assert (
            f"- {new_item}" in updated_content
        ), "New item should be added with bullet point"
        print(f"✓ New item added correctly: - {new_item}")

        return True

    def test_memories_field_complete_replacement(self):
        """Test 3: Test MEMORIES field completely replacing agent memories."""
        print("\n=== Test 3: MEMORIES Field Complete Replacement ===")

        agent_id = "memories_replacement_agent"

        # Create initial memory with some items
        self.memory_manager.add_learning(agent_id, "Initial memory 1")
        self.memory_manager.add_learning(agent_id, "Initial memory 2")

        memory_file = self.memories_dir / f"{agent_id}_memories.md"
        initial_content = memory_file.read_text()

        print("Initial memory content:")
        print(initial_content)

        # Simulate agent response with MEMORIES field
        new_memories = [
            "Completely new memory 1",
            "Completely new memory 2",
            "Completely new memory 3",
        ]

        mock_response = json.dumps({"MEMORIES": new_memories, "task_completed": True})

        # Test extraction and replacement
        success = self.memory_manager.extract_and_update_memory(
            agent_id, f"Response text\n```json\n{mock_response}\n```"
        )
        assert success, "Memory replacement should succeed"

        # Verify complete replacement
        updated_content = memory_file.read_text()
        print("\nUpdated memory content:")
        print(updated_content)

        # Verify old memories are gone
        assert (
            "Initial memory 1" not in updated_content
        ), "Old memories should be removed"
        assert (
            "Initial memory 2" not in updated_content
        ), "Old memories should be removed"

        # Verify new memories are present
        for memory in new_memories:
            assert (
                f"- {memory}" in updated_content
            ), f"New memory should be present: {memory}"

        print("✓ All old memories removed and new memories added")
        print("✓ MEMORIES field replacement successful")

        return True

    def test_remember_field_incremental_updates(self):
        """Test 4: Test 'remember' field for incremental memory updates."""
        print("\n=== Test 4: Remember Field Incremental Updates ===")

        agent_id = "remember_incremental_agent"

        # Create initial memory
        self.memory_manager.add_learning(agent_id, "Existing memory item")

        memory_file = self.memories_dir / f"{agent_id}_memories.md"
        initial_content = memory_file.read_text()

        print("Initial memory content:")
        print(initial_content)

        # Simulate agent response with remember field
        new_learnings = ["New learning 1", "New learning 2"]

        mock_response = json.dumps({"remember": new_learnings, "task_completed": True})

        # Test extraction and incremental update
        success = self.memory_manager.extract_and_update_memory(
            agent_id, f"Response text\n```json\n{mock_response}\n```"
        )
        assert success, "Incremental memory update should succeed"

        # Verify incremental addition
        updated_content = memory_file.read_text()
        print("\nUpdated memory content:")
        print(updated_content)

        # Verify existing memory is preserved
        assert (
            "- Existing memory item" in updated_content
        ), "Existing memory should be preserved"

        # Verify new memories are added
        for learning in new_learnings:
            assert (
                f"- {learning}" in updated_content
            ), f"New learning should be added: {learning}"

        print("✓ Existing memories preserved")
        print("✓ New learnings added incrementally")

        return True

    def test_hook_processing_memories_field(self):
        """Test 5: Test hook processing correctly extracts and processes MEMORIES field."""
        print("\n=== Test 5: Hook Processing of MEMORIES Field ===")

        # Create a mock hook handler
        with patch("claude_mpm.hooks.claude_hooks.hook_handler.SocketIOConnectionPool"):
            hook_handler = ClaudeHookHandler()

        # Mock memory manager to track calls
        mock_memory_manager = MagicMock()

        with patch(
            "claude_mpm.hooks.claude_hooks.memory_integration.get_memory_manager",
            return_value=mock_memory_manager,
        ):
            # Simulate SubagentStop event with MEMORIES field
            agent_id = "hook_test_agent"
            session_id = "test_session_123"

            memories_data = [
                "Memory from hook processing 1",
                "Memory from hook processing 2",
            ]

            structured_response = {
                "task_completed": True,
                "MEMORIES": memories_data,
                "results": "Task completed successfully",
            }

            # Create output with JSON response
            output = f"Task completed.\n```json\n{json.dumps(structured_response)}\n```"

            event = {
                "hook_event_name": "SubagentStop",
                "session_id": session_id,
                "agent_type": agent_id,
                "reason": "completed",
                "output": output,
                "cwd": str(self.temp_dir),
            }

            # Process the event
            hook_handler.handle_subagent_stop(event)

            # Since the hook handler has extracted managers, we need to test the extraction directly
            # Test the memory extraction method
            success = self.memory_manager.extract_and_update_memory(agent_id, output)
            assert success, "Hook processing should extract MEMORIES successfully"

            # Verify the memories were processed
            memory_file = self.memories_dir / f"{agent_id}_memories.md"
            if memory_file.exists():
                content = memory_file.read_text()
                print("Memory file content after hook processing:")
                print(content)

                for memory in memories_data:
                    assert (
                        f"- {memory}" in content
                    ), f"Memory should be processed by hook: {memory}"

                print("✓ Hook processing successfully extracted MEMORIES field")
            else:
                print("✓ Hook processing completed (memory manager was mocked)")

        return True

    def test_memory_deduplication_optimization(self):
        """Test 6: Test memory deduplication and optimization."""
        print("\n=== Test 6: Memory Deduplication and Optimization ===")

        agent_id = "dedup_test_agent"

        # Add similar memories
        similar_items = [
            "This is a test memory about project structure",
            "This is a test memory about project structure",  # Exact duplicate
            "This is a TEST memory about project structure",  # Case difference
            "This is a test memory about project architecture",  # Similar but different
            "Completely different memory about databases",
        ]

        # Add all items
        for item in similar_items:
            self.memory_manager.add_learning(agent_id, item)

        memory_file = self.memories_dir / f"{agent_id}_memories.md"
        content_before = memory_file.read_text()

        print("Content before deduplication:")
        print(content_before)

        # Test deduplication
        items = self.content_manager.parse_memory_content_to_list(content_before)
        print(f"Items before deduplication: {len(items)}")

        deduplicated_content, removed_count = self.content_manager.deduplicate_list(
            content_before
        )

        print(f"Removed {removed_count} duplicate items")
        print("Content after deduplication:")
        print(deduplicated_content)

        # Verify deduplication worked
        assert removed_count > 0, "Should have removed duplicates"

        # Count unique items
        dedupe_items = self.content_manager.parse_memory_content_to_list(
            deduplicated_content
        )
        print(f"Items after deduplication: {len(dedupe_items)}")

        assert len(dedupe_items) < len(
            items
        ), "Should have fewer items after deduplication"

        # Verify important content is preserved
        assert any(
            "databases" in item.lower() for item in dedupe_items
        ), "Unique content should be preserved"

        print(f"✓ Deduplication removed {removed_count} items")
        print("✓ Unique content preserved")

        return True

    def test_project_based_memories_storage(self):
        """Test 7: Test PROJECT-based memories work correctly (.claude-mpm/memories/)."""
        print("\n=== Test 7: PROJECT-based Memories Storage ===")

        # Verify memories directory structure
        assert self.memories_dir.exists(), "Memories directory should exist"
        assert self.memories_dir.name == "memories", "Should be named 'memories'"
        assert (
            self.memories_dir.parent.name == ".claude-mpm"
        ), "Should be in .claude-mpm directory"

        print(f"✓ Memories directory correct: {self.memories_dir}")

        # Test multiple agents in project directory
        agent_ids = ["PM", "engineer", "research", "qa"]

        for agent_id in agent_ids:
            # Create memory for each agent
            test_memory = f"Project-specific memory for {agent_id} agent"
            success = self.memory_manager.add_learning(agent_id, test_memory)
            assert success, f"Should create memory for {agent_id}"

            # Verify file location
            memory_file = self.memories_dir / f"{agent_id}_memories.md"
            assert memory_file.exists(), f"Memory file should exist for {agent_id}"

            # Verify content
            content = memory_file.read_text()
            assert (
                f"- {test_memory}" in content
            ), f"Memory should be saved for {agent_id}"

            print(f"✓ {agent_id} agent memory created in project directory")

        # Test README creation
        readme_file = self.memories_dir / "README.md"
        assert readme_file.exists(), "README.md should be created in memories directory"

        readme_content = readme_file.read_text()
        assert (
            "Agent Memory System" in readme_content
        ), "README should have proper title"
        assert (
            "Manual Editing" in readme_content
        ), "README should have manual editing section"

        print("✓ README.md created with proper content")

        # Verify all files are in the correct location
        memory_files = list(self.memories_dir.glob("*_memories.md"))
        assert len(memory_files) == len(
            agent_ids
        ), f"Should have {len(agent_ids)} memory files"

        print(f"✓ All {len(memory_files)} agent memory files in project directory")

        return True

    def test_full_end_to_end_memory_flow(self):
        """Test 8: Run integration test for full end-to-end memory flow."""
        print("\n=== Test 8: Full End-to-End Memory Flow ===")

        agent_id = "e2e_test_agent"

        # Step 1: Initial memory creation
        print("Step 1: Initial memory creation")
        self.memory_manager.load_agent_memory(agent_id)
        memory_file = self.memories_dir / f"{agent_id}_memories.md"
        assert memory_file.exists(), "Memory file should be created"
        print("✓ Initial memory file created")

        # Step 2: Add incremental memories
        print("Step 2: Add incremental memories")
        incremental_memories = ["E2E memory 1", "E2E memory 2"]
        for memory in incremental_memories:
            success = self.memory_manager.add_learning(agent_id, memory)
            assert success, f"Should add memory: {memory}"
        print("✓ Incremental memories added")

        # Step 3: Simulate agent response with remember field
        print("Step 3: Agent response with remember field")
        remember_response = json.dumps(
            {
                "remember": ["New incremental memory from response"],
                "task_completed": True,
            }
        )

        success = self.memory_manager.extract_and_update_memory(
            agent_id, f"Task response\n```json\n{remember_response}\n```"
        )
        assert success, "Should process remember field"
        print("✓ Remember field processed")

        # Step 4: Simulate agent response with MEMORIES field (complete replacement)
        print("Step 4: Agent response with MEMORIES field")
        complete_memories = [
            "Final memory 1 from MEMORIES field",
            "Final memory 2 from MEMORIES field",
            "Final memory 3 from MEMORIES field",
        ]

        memories_response = json.dumps(
            {"MEMORIES": complete_memories, "task_completed": True}
        )

        success = self.memory_manager.extract_and_update_memory(
            agent_id, f"Final response\n```json\n{memories_response}\n```"
        )
        assert success, "Should process MEMORIES field"
        print("✓ MEMORIES field processed")

        # Step 5: Verify final state
        print("Step 5: Verify final state")
        final_content = memory_file.read_text()

        # Should only have MEMORIES field content (complete replacement)
        for memory in complete_memories:
            assert (
                f"- {memory}" in final_content
            ), f"Final memory should be present: {memory}"

        # Old memories should be gone (replaced by MEMORIES field)
        for memory in incremental_memories:
            assert (
                f"- {memory}" not in final_content
            ), f"Old memory should be replaced: {memory}"

        assert (
            "New incremental memory from response" not in final_content
        ), "Remember field memory should be replaced"

        print("✓ Final state correct - MEMORIES field replaced all previous memories")

        # Step 6: Test timestamp and structure
        print("Step 6: Verify file structure")
        lines = final_content.split("\n")
        assert lines[0].startswith("# Agent Memory:"), "Should have proper header"
        assert lines[1].startswith("<!-- Last Updated:"), "Should have timestamp"

        # Count memory items
        memory_items = [line for line in lines if line.strip().startswith("- ")]
        assert len(memory_items) == len(
            complete_memories
        ), f"Should have {len(complete_memories)} memory items"

        print(f"✓ File structure correct with {len(memory_items)} memory items")

        print("\n🎉 End-to-end memory flow test completed successfully!")

        return True

    def run_all_tests(self):
        """Run all tests and generate a comprehensive report."""
        print("\n" + "=" * 80)
        print("COMPREHENSIVE MEMORY SYSTEM INTEGRATION TEST SUITE")
        print("=" * 80)

        test_results = {}

        tests = [
            (
                "Memory File Creation Simple List Format",
                self.test_memory_file_creation_simple_list_format,
            ),
            (
                "Timestamp Updates on Modifications",
                self.test_timestamp_updates_on_modifications,
            ),
            (
                "MEMORIES Field Complete Replacement",
                self.test_memories_field_complete_replacement,
            ),
            (
                "Remember Field Incremental Updates",
                self.test_remember_field_incremental_updates,
            ),
            (
                "Hook Processing MEMORIES Field",
                self.test_hook_processing_memories_field,
            ),
            (
                "Memory Deduplication Optimization",
                self.test_memory_deduplication_optimization,
            ),
            (
                "PROJECT-based Memories Storage",
                self.test_project_based_memories_storage,
            ),
            ("Full End-to-End Memory Flow", self.test_full_end_to_end_memory_flow),
        ]

        for test_name, test_func in tests:
            try:
                print(f"\n{'='*20} Running: {test_name} {'='*20}")
                result = test_func()
                test_results[test_name] = "PASSED" if result else "FAILED"
                print(f"✅ {test_name}: PASSED")
            except Exception as e:
                test_results[test_name] = f"FAILED: {e}"
                print(f"❌ {test_name}: FAILED - {e}")

        # Generate final report
        print("\n" + "=" * 80)
        print("FINAL TEST RESULTS")
        print("=" * 80)

        passed = 0
        failed = 0

        for test_name, result in test_results.items():
            status_symbol = "✅" if result == "PASSED" else "❌"
            print(f"{status_symbol} {test_name}: {result}")
            if result == "PASSED":
                passed += 1
            else:
                failed += 1

        print(f"\nSUMMARY: {passed} passed, {failed} failed out of {len(tests)} tests")

        if failed == 0:
            print("\n🎉 ALL TESTS PASSED! Memory system is working correctly.")
        else:
            print(f"\n⚠️  {failed} tests failed. Review the failures above.")

        return test_results


if __name__ == "__main__":
    # Run the test suite
    test_suite = TestMemorySystemIntegration()
    test_suite.setup_method()

    try:
        results = test_suite.run_all_tests()
    finally:
        test_suite.teardown_method()
