# SPDX-License-Identifier: AGPL-3.0-or-later
# Copyright (C) 2018-2020 Andrew Rechnitzer
# Copyright (C) 2020 Dryden Wiebe
# Copyright (C) 2020 Vala Vakilian
# Copyright (C) 2021-2025 Colin B. Macdonald

import csv
from importlib import resources

import plom
from plom.aliceBob import (
    make_numbered_user_list,
    make_random_user_list,
    simple_password,
)

list_of_required_users = ["manager", "scanner", "reviewer"]
minimum_number_of_required_users = 4


def build_canned_users(number_of_users, *, numbered=False, manager=False):
    """Creates a list of fake users.

    Arguments:
        number_of_users (int): number of fake users to create.

    Keyword Arguments:
        numbered (bool): if True, make numbered users such as "user017"
            otherwise (default) make real-ish names like "gwen" or
            "talia07".
        manager (bool): if True, also make a manager password.  Default
            is False.

    Returns:
        list: pairs of autogenerated usernames and passwords.
    """
    user_list = []
    for user in list_of_required_users:
        if not manager and user == "manager":
            continue
        user_list.append([user, simple_password(n=4)])

    # append list of standard users
    if numbered:
        user_list.extend(make_numbered_user_list(number_of_users))
    else:
        user_list.extend(make_random_user_list(number_of_users))

    return user_list


def get_raw_user_dict_from_csv(user_file_path):
    """Gets the user/password dictionary from a csv file.

    Arguments:
        user_file_path (str/pathlib.Path): a csv file of proposed
            usernames and passwords.

    Returns:
        dict: A dict(str:str) which represents (username: password).

    Raises:
        ValueError: malformed csv file.
    """
    with open(user_file_path, "r") as f:
        return _get_raw_user_dict(f)


def _get_raw_user_list(f):
    lst = []
    reader = csv.reader(f, skipinitialspace=True)
    csv_headers = next(reader, None)
    if csv_headers != ["user", "password"]:
        raise ValueError('csv file must have two columns "user" and "password".')
    for row in reader:
        lst.append(row)
    return lst


def _get_raw_user_dict(f):
    return {r[0]: r[1] for r in _get_raw_user_list(f)}


def get_template_user_list():
    """Gets the user/password list for some fixed demo values."""
    with (resources.files(plom) / "templateUserList.csv").open("r") as f:
        return _get_raw_user_list(f)


def get_template_user_dict():
    """Gets the user/password dictionary for some fixed demo values."""
    with (resources.files(plom) / "templateUserList.csv").open("r") as f:
        return _get_raw_user_dict(f)


def write_csv_user_list(lst, filename):
    """Write the user/password pairs into a CSV file."""
    with open(filename, "w") as fh:
        writer = csv.writer(fh, quoting=csv.QUOTE_NONNUMERIC)
        writer.writerow(["user", "password"])
        for row in lst:
            writer.writerow(row)
