# SPDX-License-Identifier: AGPL-3.0-or-later
# Copyright (C) 2025 Bryan Tanady
# Copyright (C) 2025 Colin B. Macdonald

"""This is an abstracted inference for https://github.com/BryanTanady/plom_ml_clustering."""

import yaml
from abc import abstractmethod
from importlib import resources
from pathlib import Path
from typing import Mapping

import numpy as np
from sklearn.metrics import silhouette_score, davies_bouldin_score
from sklearn.cluster import AgglomerativeClustering
from huggingface_hub import hf_hub_download

import plom_ml.clustering.model
from plom_ml.clustering.embedding.embedder import (
    Embedder,
    SymbolicEmbedder,
    TrOCREmbedder,
    MCQEmbedder,
)
from plom_ml.clustering.exceptions import MissingEmbedderException, NoThresholdFound


def get_best_clustering(
    X: np.ndarray,
    thresholds: np.ndarray,
    distance_metric: str,
    metric="silhouette",
) -> np.ndarray:
    """Get the best clustering of X by searching for optimal threshold that maximizes the metric.

    This function defaults with AgglomerativeClustering clustering algorithm. Note that to get
    more fine-grained clustering one can provide smaller thresholds range. Furthermore, it would
    be even better if we provide another argument where we force the threshold at a specific value
    and if that value is None then we do the search.

    Args:
        X: the feature matrix.
        thresholds: the choices of distance thresholds.
        distance_metric: What is the distance metric in this context, eg: "euclidean", "cosine"
        metric: which clustering metric to optimize. Currently supports: "silhouette" and "davies".
            From my experiment, optimizing silhouette tends to provide less duplicated clusterings but
            each grouping may have less purity than davies optimized clustering.

    Returns:
        A numpy array of clusterId where the order matches with the
        inputs (index 0 provides Id for row 0 of X)
    """
    best_score = -np.inf if metric == "silhouette" else np.inf
    best_labels = np.array([])
    best_thresh = -1  # for debug

    # Be careful with linkage, I have tuned my thresholds for distance_metric = ward,
    # and if we change that to "complete" we need to retune again.
    linkage = "average" if distance_metric == "cosine" else "ward"

    for t in thresholds:
        clustering = AgglomerativeClustering(
            n_clusters=None,
            metric=distance_metric,
            distance_threshold=t,
            linkage=linkage,
        )
        labels = clustering.fit_predict(X)
        # need at least 2 clusters to score
        if len(set(labels)) < 2:
            continue

        if metric == "silhouette":
            score = silhouette_score(X, labels)
            # silhouette: higher -> better
            if score > best_score:
                best_score = score
                best_labels = labels
                best_thresh = t

        elif metric == "davies":
            score = davies_bouldin_score(X, labels)
            # DB index: lower -> better
            if score < best_score:
                best_score = score
                best_labels = labels
                best_thresh = t

    assert isinstance(best_labels, np.ndarray)
    if not best_labels.size > 0:
        raise NoThresholdFound(
            f"Can't find any threshold within {thresholds} to produce any clustering"
        )

    print(f"Chosen thresh: {best_thresh:.3f}")
    return best_labels


class ClusteringStrategy:
    """Abstract base class for clustering strategy.

    A clustering strategy defines:
        a. Embedders used to generate features for PREPROCESSED inputs
        b. clustering algorithm used to cluster the generated features.
        c. Any task-specific logic eg: PCA decomposition before clustering

    This class enforces every subclass to implement cluster_papers that outputs the
    paper_number to their clusterId. Furthermore the class has get_embeddings method
    that generate a feature vector for an image. The feature vector is a concatenation of
    embeddings generated by the embedderes.
    """

    embedders: list[Embedder]

    def get_embeddings(self, image: np.ndarray) -> np.ndarray:
        """Generate an array of embeddings generated by embedders for the inputted image.

        Args:
            image: the image whose feature vector will be generated for clustering.

        Returns:
            a 1D array of feature vector with shape of shape (D,). D is the sum of output
            dimensions from all embedders.

        Raises:
            MissingEmbedderException: if ClusteringStrategy has not initialized embedders property.
        """
        if not hasattr(self, "embedders") or not self.embedders:
            raise MissingEmbedderException(
                f"Missing self.embedders in {self.__class__.__name__} ClusteringStrategy"
            )

        return np.concatenate([embedder.embed(image) for embedder in self.embedders])

    @abstractmethod
    def cluster_papers(
        self, paper_to_image: Mapping[int, np.ndarray]
    ) -> dict[int, int]:
        """Cluster the given papers into a mapping of paper_num to clusterId.

        This method directly calls inference models on the provided images. Therefore, if
        there are expected preprocessing steps, the images must be preprocessed before
        feeding them into this function.

        Args:
            paper_to_image: a dictionary mapping paper number to a (processed) image that
            represents that paper.

        Returns:
            A dictionary mapping the paper number to their cluster id.
        """
        pass


class HMEClusteringStrategy(ClusteringStrategy):
    """Handwritten math expression model.

    An ongoing clustering system that I can't guarantee to work in all cases.

    So far, I noticed that it fails in data with minimum information eg: single character
    '1', '2', 'a' and very small fonts. Furthermore, this system may be somewhat size sensitive
    so it may cluster based on font size instead of purely semantic. One main reason is that
    during training, the vocabulary built for the model differentiates small and capital letter.

    Future work can try to :
        - Resolve capital & small letters confusion
            train the model where similar small & capital letters
            are merged eg: ('x', 'X') ('c', 'C') etc.
        - Self identify bounding rect of math expression.
            The training involves minimum padding (mainly handwritten stuff at the middle).
            Current system feeds the entire selected regions to the system (with diffing), so
            there are potentially a lot of empty spaced that may confuse the model.
        - Make clustering system location agnostic
            Sort of related to above. Since we are picking a large region with large empty space,
            there is bound to be large variance in the handwritten stuff position. There is no escape
            that this will create some distortion into the generated features, potentially weaken
            intracluster cohesion.

    If things go wrong with current system what to do?:
        - Clustering granularity
            This is the lowest hanging fruit, can be found when the system calls
            get_best_clustering. If you expect the result to be more fine-grained
            move the distance thresholds lower, I would say the range is (3 - 7).
            If you want more "merged" result then do the contrary.

        - Noise prevails
            If it fails on noisy paper eg: gaussian noise, scan noise, or dirty paper. Then it may be
            in get_diff. Notice how get_diff has BLOB_MIN_AREA. If you want the system to denoise
            stronger increase that constant.

        - Feature engineering
            This is the gray area of the system. Currently, the system merges two embeddings produced
            by two completely different models then directly cluster on that. That feels odd, naturally
            I want to do some normalization such that they have somewhat the same scale. However doing
            that seems to perform worse on "some" of my dataset than existing system.
            BUT, it seems to perform better on the autogenerated dataset from demo. Details on
            how I did that:
                * Scale each feature with sklearn.preprocessing.StandardScaler
                * merge with weight, the best result I got was (1 : 2) ratio for (Symbolic: TrOCR)
                * normalize feature to unit vector
                * cluster with AgglomerativeClustering with distance 1.5 to 3.
    """

    def __init__(self):
        config_path = resources.files(plom_ml.clustering.model) / "model_config.yaml"
        with config_path.open("r") as f:
            config = yaml.safe_load(f)

        # load weights path
        symbolic_model_filename = config["models"]["hme_symbolic"]["filename"]
        trocr_model_filename = config["models"]["hme_trocr"]["filename"]

        symbolic_model_path = Path(f"model_cache/{symbolic_model_filename}")
        trocr_model_path = Path(f"model_cache/{trocr_model_filename}")

        if not symbolic_model_path.exists():
            print("Downloading HME symbolic weight")
            hf_hub_download(
                repo_id=config["models"]["hme_symbolic"]["repo_id"],
                filename=symbolic_model_filename,
                local_dir=symbolic_model_path.parent,
            )
            print(
                "HME symbolic model has been downloaded, saved at: ",
                symbolic_model_path,
            )

        if not trocr_model_path.exists():
            print("Downloading trOCR weight")
            hf_hub_download(
                repo_id=config["models"]["hme_trocr"]["repo_id"],
                filename=trocr_model_filename,
                local_dir=trocr_model_path.parent,
            )
            print(
                "HME symbolic model has been downloaded, saved at: ", trocr_model_path
            )

        # Init embedders
        self.embedders = [
            SymbolicEmbedder(symbolic_model_path),
            TrOCREmbedder(trocr_model_path),
        ]

    def cluster_papers(
        self, paper_to_image: Mapping[int, np.ndarray]
    ) -> dict[int, int]:
        """Cluster the given papers.

        Args:
            paper_to_image: a dictionary mapping paper number to a (processed) image.

        Returns:
            A dictionary mapping the paper number to their cluster id.
        """
        # NOTE: Ideally, I want to normalize probability from self-trained model
        # and embedding from TrOCR before merging and cluster with cosine metric.
        # However, that has worse performance than raw merging of features and
        # cluster with euclidean distance. So I think this implies the embeddings
        # generated by trOCR provides some separability power and for now, I tune those
        # distance thresholds range, such that they have consistent enough behavior
        # for differing datasets.

        # Build feature matrix
        X = np.vstack(
            [self.get_embeddings(image) for pn, image in paper_to_image.items()]
        )

        # set up distance threshold search space.
        # Make range to smaller value if intends for more fine-grained clustering eg: (3.5, 5).
        # From my experimentation: don't go under 3, otherwise it will be too fine-grained.
        min_thresh = 4
        max_thresh = 7
        thresh_counts = (
            40  # bumping this up has little impact, 10 - 40 is a reasonable choice
        )
        thresholds = np.linspace(min_thresh, max_thresh, thresh_counts)

        clusterIDs = get_best_clustering(X, thresholds, "euclidean", "davies")
        return dict(zip(list(paper_to_image.keys()), clusterIDs))


class MCQClusteringStrategy(ClusteringStrategy):
    """Handwritten MCQ clustering model.

    This clustering system tends to be more robust compared to HME. So perhaps
    threshold (found in .cluster_papers), hopefully, doesn't need to be tuned as much.

    The critical component in the pipeline is in the computer vision part. The details
    are in the MCQEmbedder, where we filter to components of certain area then run the inference
    on the blobs. We then pick the inference with highest confidence of that blob being a letter.

    So if things go wrong, here are my suspects:
        - Broken handwriting (not a clean connected component) eg: 'B' written as '13'
            Handwritten stuff may be disconnected such that the algorithm only sees parts of
            the letter. This is somewhat addressed in get_diff, where we do MORPH_CLOSE, which connects
            components that are almost connected. However, perhaps this may not be enough, BUT BEWARE
            that increasing that kernel size may lead to stronger noises.

            Another way is to add some paddings to the bounding rect i.e in
            crop = img[y : y + h, x : x + w], we can add some extra padding like y + h + pad_y, so
            we give some wiggle room for broken component to be seen in the crop.

        - Letters are too small
            Currently there are some dilations applied to the scanned image, i.e in get_diff where we
            do cv2.dilate on scanned. So, if letters are too small, this dilation may be too strong,
            such that their structure is disoriented. However, note that in my experiment, for reasonable
            font size, the dilation is reasonable enough such that it doesn't mess up the structure.
    """

    def __init__(self):
        config_path = resources.files(plom_ml.clustering.model) / "model_config.yaml"
        with config_path.open("r") as f:
            config = yaml.safe_load(f)

        # check if weight has been downloaded
        weight_filename = config["models"]["mcq"]["filename"]
        weight_path = Path(f"model_cache/{weight_filename}")

        # download weight if the weight is not present
        if not Path(weight_path).exists():
            print("Downloading MCQ clusterer weight")
            hf_hub_download(
                repo_id=config["models"]["mcq"]["repo_id"],
                filename=weight_filename,
                local_dir=weight_path.parent,
            )
            print("Model has been downloaded, saved at: ", weight_path)

        # init embedder
        out_features = 11
        self.embedders = [
            MCQEmbedder(weight_path=weight_path, out_features=out_features)
        ]

    def cluster_papers(
        self, paper_to_image: Mapping[int, np.ndarray]
    ) -> dict[int, int]:
        """Cluster papers based on handwritten MCQ.

        Args:
            paper_to_image: a dictionary mapping paper number to the
                cropped region used for clustering.

        Returns:
            A dictionary mapping the paper number to their cluster id
        """
        # Build feature matrix
        X = np.vstack(
            [self.get_embeddings(image) for _, image in paper_to_image.items()]
        )

        # NOTE: this threshold space is empirically tuned with custom dataset
        # to enforce more fine-grained cluster move the threshold to smaller value range.
        # Unlike HME, this range tends to be pretty robust, but that could be because I have less
        # varying datasets. But if it turns out that we need more fine grain, I would say move
        # min to 0.05
        thresholds = np.linspace(0.1, 0.5, 50)
        clusterIDs = get_best_clustering(
            X, thresholds=thresholds, distance_metric="cosine", metric="silhouette"
        )
        return dict(zip(list(paper_to_image.keys()), clusterIDs))
