# SPDX-License-Identifier: AGPL-3.0-or-later
# Copyright (C) 2023 Edith Coates
# Copyright (C) 2024 Colin B. Macdonald
# Copyright (C) 2024 Aidan Murphy

from django.db import models
from django.contrib.auth.models import User
from . import MarkingTask


class AnnotationImage(models.Model):
    """A raster representation of an annotated question.

    Attributes:
        image: a jpeg or png file.
        hash: a string of a hash of the file, currently the md5sum.

    Because of the OneToOneField in Annotation, there will also be an
    autogenerated field called ``annotation`` (note lowercase).
    """

    image = models.FileField(upload_to="annotation_images/", null=False)
    hash = models.TextField(null=False, default="")


class Annotation(models.Model):
    """Represents a marker's annotation of a particular paper's question.

    Attributes:
        edition: The edition of the annotation for the specified task.
        score: The score given to the student's work.
        image: The image of the annotated question.
        annotation_data: A Json blob of annotation data containing image path info,
            rubric info, svg annotation info.
        marking_time: The total cumulative time in seconds spent by all
            markers on this question of this paper.  As a question gets
            remarked or revisited this number will increase.
        marking_delta_time: the time in seconds spent making this
            particular annotation (which could be a revision to a
            previous annotation).  The total ``marking_time`` is thus
            the sum of all the ``marking_delta_time`` over all revisions.
        task: The marking task.
        user: The user who made the annotation.
        time_of_last_update: The time of the last update.
    """

    edition = models.IntegerField(null=True)
    score = models.FloatField(null=True)
    image = models.OneToOneField(AnnotationImage, on_delete=models.CASCADE)
    annotation_data = models.JSONField(null=True)
    marking_time = models.FloatField(null=True)
    marking_delta_time = models.FloatField(null=True)
    task = models.ForeignKey(MarkingTask, null=True, on_delete=models.SET_NULL)
    user = models.ForeignKey(User, null=True, on_delete=models.SET_NULL)
    time_of_last_update = models.DateTimeField(auto_now=True)
