# SPDX-License-Identifier: AGPL-3.0-or-later
# Copyright (C) 2022 Edith Coates
# Copyright (C) 2022 Brennen Chiu
# Copyright (C) 2023-2024 Andrew Rechnitzer
# Copyright (C) 2024-2025 Colin B. Macdonald
# Copyright (C) 2025 Aidan Murphy

from django.test import TestCase
from model_bakery import baker

from plom_server.TestingSupport.utils import config_test
from plom_server.Papers.models import Image, FixedPage, MobilePage, Bundle, Paper
from ..services import ManageScanService


class ManageScanServiceTests(TestCase):
    """Tests for Progress.services.ManageScanService."""

    @config_test({"test_spec": "demo"})
    def setUp(self):
        self.bundle = baker.make(
            Bundle,
            pdf_hash="qwerty",
        )
        # make some papers
        # * 1 has all fixed-page images  (6 scanned pages) and 1 mobile page for Q1.
        # * 1,2,3,4 with fixed-page images  (6*4 scanned pages)
        # * 6,7 = 2 scanned fixed pages, 4 unscanned = incomplete  (2*2 scanned pages)
        # * 8,9 = completely unscanned = unused
        # * 10,11 = 2 scanned fixed pages, 4 unscanned, 2 mobile pages = incomplete  (2*2 scanned, 2*2 mobile)
        # * 12,13,14,15 = three mobile pages each (questions 1, 2, 3). (3*2 mobile)
        ord = 0
        # make the 5 complete papers
        for paper_number in [1]:
            paper = baker.make(Paper, paper_number=paper_number)
            for pg in range(1, 7):
                ord += 1
                img = baker.make(Image, bundle=self.bundle, bundle_order=ord)
                baker.make(FixedPage, paper=paper, image=img, version=1, page_number=pg)
            for qn in [1]:
                ord += 1
                img = baker.make(Image, bundle=self.bundle, bundle_order=ord)
                baker.make(MobilePage, paper=paper, question_index=qn, image=img)
        # now make the other 4 papers, all fixed, but no mobile pages
        for paper_number in [2, 3, 4, 5]:
            paper = baker.make(Paper, paper_number=paper_number)
            for pg in range(1, 7):
                ord += 1
                img = baker.make(Image, bundle=self.bundle, bundle_order=ord)
                baker.make(FixedPage, paper=paper, image=img, version=1, page_number=pg)

        # make 2 papers with 2 pages with images and 4 without (ie an incomplete paper)
        for paper_number in [6, 7]:
            paper = baker.make(Paper, paper_number=paper_number)
            for pg in range(1, 3):
                ord += 1
                img = baker.make(Image, bundle=self.bundle, bundle_order=ord)
                baker.make(FixedPage, paper=paper, image=img, version=1, page_number=pg)
            for pg in range(3, 7):
                baker.make(
                    FixedPage, paper=paper, image=None, version=1, page_number=pg
                )

        # make another 2 unused papers - no images at all
        for paper_number in [8, 9]:
            paper = baker.make(Paper, paper_number=paper_number)
            for pg in range(1, 7):
                baker.make(
                    FixedPage, paper=paper, image=None, version=1, page_number=pg
                )

        # make another 2 papers with 2 pages with images and 4 without (ie an incomplete paper), but 2 mobile pages (for q 1,2)
        for paper_number in [10, 11]:
            paper = baker.make(Paper, paper_number=paper_number)
            for pg in range(1, 3):
                ord += 1
                img = baker.make(Image, bundle=self.bundle, bundle_order=ord)
                baker.make(FixedPage, paper=paper, image=img, version=1, page_number=pg)
            for pg in range(3, 7):
                baker.make(
                    FixedPage, paper=paper, image=None, version=1, page_number=pg
                )
            for qn in range(1, 3):
                ord += 1
                img = baker.make(Image, bundle=self.bundle, bundle_order=ord)
                baker.make(MobilePage, paper=paper, question_index=qn, image=img)

        # make 3 papers with 3 mobile pages and all fixed pages unscanned ("complete")
        for paper_number in [12, 13, 14]:
            paper = baker.make(Paper, paper_number=paper_number)
            for pg in range(1, 7):
                baker.make(
                    FixedPage, paper=paper, image=None, version=1, page_number=pg
                )
            for qn in range(1, 4):
                ord += 1
                img = baker.make(Image, bundle=self.bundle, bundle_order=ord)
                baker.make(MobilePage, paper=paper, question_index=qn, image=img)

        # make 1 paper with 4 mobile pages and all fixed pages unscanned ("complete")
        paper_number = 15
        paper = baker.make(Paper, paper_number=paper_number)
        for pg in range(1, 7):
            baker.make(FixedPage, paper=paper, image=None, version=1, page_number=pg)
        for qn in range(1, 4):
            ord += 1
            img = baker.make(Image, bundle=self.bundle, bundle_order=ord)
            baker.make(MobilePage, paper=paper, question_index=qn, image=img)
        # The DNM page shouldn't change anything about this paper's 'completeness'
        img = baker.make(Image, bundle=self.bundle, bundle_order=ord)
        baker.make(
            MobilePage, paper=paper, question_index=MobilePage.DNM_qidx, image=img
        )

        # make 2 papers with 2 mobile pages and all fixed pages unscanned ("incomplete")
        for paper_number in [16, 17]:
            paper = baker.make(Paper, paper_number=paper_number)
            for pg in range(1, 7):
                baker.make(
                    FixedPage, paper=paper, image=None, version=1, page_number=pg
                )
            for qn in range(1, 3):
                ord += 1
                img = baker.make(Image, bundle=self.bundle, bundle_order=ord)
                baker.make(MobilePage, paper=paper, question_index=qn, image=img)

        # make 1 paper with 3 mobile pages and all fixed pages unscanned ("incomplete")
        paper_number = 18
        paper = baker.make(Paper, paper_number=paper_number)
        for pg in range(1, 7):
            baker.make(FixedPage, paper=paper, image=None, version=1, page_number=pg)
        for qn in range(1, 3):
            ord += 1
            img = baker.make(Image, bundle=self.bundle, bundle_order=ord)
            baker.make(MobilePage, paper=paper, question_index=qn, image=img)
        # The DNM page shouldn't change anything about this paper's 'completeness'
        img = baker.make(Image, bundle=self.bundle, bundle_order=ord)
        baker.make(
            MobilePage, paper=paper, question_index=MobilePage.DNM_qidx, image=img
        )

        return super().setUp()

    def test_counts(self) -> None:
        # we have made some papers
        # * 1 has all fixed-page images  (6 scanned pages) and 1 mobile page for Q1.
        # * 1,2,3,4 with fixed-page images  (6*4 scanned pages)
        # * 6,7 = 2 scanned fixed pages, 4 unscanned = incomplete  (2*2 scanned pages)
        # * 8,9 = completely unscanned = unused
        # * 10,11 = 2 scanned fixed pages, 4 unscanned, 2 mobile pages = incomplete  (2*2 scanned, 2*2 mobile)
        # * 12,13,14 = three mobile pages each (questions 1, 2, 3). (3*3 mobile)
        # * 15 = four mobile pages (questions 1, 2, 3, DNM). (1*4 mobile)
        # * 16,17 = two mobile pages each (questions 1, 2). (2*2 mobile)
        # * 18 = two mobile pages each (questions 1, 2, DNM). (1*3 mobile)
        mss = ManageScanService()
        assert ManageScanService.get_total_papers() == 18
        assert mss.get_total_fixed_pages() == 18 * 6
        assert mss.get_total_mobile_pages() == 1 + 2 * 2 + 3 * 3 + 1 * 4 + 2 * 2 + 3 * 1
        assert (
            mss.get_number_of_scanned_pages()
            == 5 * 6 + 1 + 2 * 2 + 2 * 2 + 2 * 2 + 3 * 3 + 4 * 1 + 2 * 2 + 1 * 3
        )
        assert mss.get_number_unused_papers() == 2
        assert ManageScanService.get_number_completed_papers() == 5 + 4
        assert ManageScanService.get_number_incomplete_papers() == 2 + 2 + 3

    def test_get_all_used_and_unused_papers(self) -> None:
        unused = [8, 9]
        self.assertEqual(ManageScanService.get_all_unused_papers(), unused)
        used = [x for x in range(1, 19) if x not in unused]
        self.assertEqual(ManageScanService.get_all_used_papers(), used)

    def test_get_all_incomplete_papers(self) -> None:
        mss_incomplete = ManageScanService.get_all_incomplete_papers()
        # papers 6,7,10,11 are incomplete - should return dict of the form
        #
        # 6: {'fixed': [{'status': 'present', 'page_number': 1,
        # 'page_pk': 211, 'img_pk': 142}, {'status': 'present',
        # 'page_number': 2, 'page_pk': 212, 'img_pk': 143},
        # {'status': 'missing', 'page_number': 3, 'page_pk': 213,
        # 'kind': 'QuestionPage'}, {'status': 'missing',
        # 'page_number': 4, 'page_pk': 214, 'kind': 'QuestionPage'},
        # {'status': 'missing', 'page_number': 5, 'page_pk': 215,
        # 'kind': 'QuestionPage'}, {'status': 'missing',
        # 'page_number': 6, 'page_pk': 216, 'kind': 'QuestionPage'}],
        # 'mobile': []},
        #
        # papers 16, 17, 18 also incomplete - should return dict of the form
        #
        # 16: {'fixed': [], 'mobile': [{'question_idx': 1,'img_pk': 345,
        # 'page_pk': 678, 'page_label': 'qi.1'}, {'question_idx': 2,
        # 'img_pk': 346, 'page_pk': 679, 'page_label': 'qi.2'}]}

        assert len(mss_incomplete) == 7
        for pn in [6, 7]:
            assert 6 in mss_incomplete
            # it is missing pages 3,4,5,6, but has fixed pages 1,2 - the img_pk of those we can ignore.
            f_pg_data = mss_incomplete[pn]["fixed"]
            assert len(f_pg_data) == 6
            for pg in [1, 2]:
                assert f_pg_data[pg - 1]["status"] == "present"
                assert f_pg_data[pg - 1]["page_number"] == pg
                assert (
                    "img_pk" in f_pg_data[pg - 1]
                )  # not testing the actual value of image_pk
            for pg in range(3, 7):
                assert f_pg_data[pg - 1]["status"] == "missing"
                assert f_pg_data[pg - 1]["page_number"] == pg
                assert f_pg_data[pg - 1]["kind"] == "QuestionPage"
                # not testing value of page_pk
                assert "page_pk" in f_pg_data[pg - 1]
                # the image should be missing
                assert "img_pk" not in f_pg_data[pg - 1]
        for pn in [10, 11]:
            assert pn in mss_incomplete
            # it is missing pages 3,4,5,6, but has fixed pages 1,2 and mobile for q1,2- the img_pk of those we can ignore.
            f_pg_data = mss_incomplete[pn]["fixed"]
            m_pg_data = mss_incomplete[pn]["mobile"]
            assert len(f_pg_data) == 6
            assert len(m_pg_data) == 2
            for pg in [1, 2]:
                assert f_pg_data[pg - 1]["status"] == "present"
                assert f_pg_data[pg - 1]["page_number"] == pg
                assert (
                    "img_pk" in f_pg_data[pg - 1]
                )  # not testing the actual value of image_pk
            for pg in range(3, 7):
                assert f_pg_data[pg - 1]["status"] == "missing"
                assert f_pg_data[pg - 1]["page_number"] == pg
                assert f_pg_data[pg - 1]["kind"] == "QuestionPage"
                # not testing value of page_pk
                assert "page_pk" in f_pg_data[pg - 1]
                # the image should be missing
                assert "img_pk" not in f_pg_data[pg - 1]
            for pg in range(1, 3):
                assert (
                    m_pg_data[pg - 1]["question_idx"] == pg
                )  # 7th, 8th entries are q's 1,2.
                assert (
                    "img_pk" in m_pg_data[pg - 1]
                )  # not testing the actual value of image_pk

        for pn in [16, 17]:
            assert pn in mss_incomplete
            # the papers have 1 mobile page for each of Q1 and Q2.
            f_pg_data = mss_incomplete[pn]["fixed"]
            m_pg_data = mss_incomplete[pn]["mobile"]
            assert len(f_pg_data) == 0
            assert len(m_pg_data) == 2
            for pg in range(1, 3):
                assert (
                    m_pg_data[pg - 1]["question_idx"] == pg
                )  # 7th, 8th entries are q's 1,2.
                assert (
                    "img_pk" in m_pg_data[pg - 1]
                )  # not testing the actual value of image_pk

        for pn in [18]:
            assert pn in mss_incomplete
            # it is missing pages 3,4,5,6, but has fixed pages 1,2 and mobile for q1,2- the img_pk of those we can ignore.
            f_pg_data = mss_incomplete[pn]["fixed"]
            m_pg_data = mss_incomplete[pn]["mobile"]
            assert len(f_pg_data) == 0
            assert len(m_pg_data) == 3
            assert m_pg_data[0]["question_idx"] == MobilePage.DNM_qidx
            assert m_pg_data[1]["question_idx"] == 1
            assert m_pg_data[2]["question_idx"] == 2
            for pg in range(1, 3):
                assert (
                    "img_pk" in m_pg_data[pg - 1]
                )  # not testing the actual value of image_pk

    def test_get_all_complete_papers(self) -> None:
        mss_complete = ManageScanService.get_all_complete_papers()
        # should return a dict of papers and their pages
        # paper 1 has 6 fixed and 1 mobile.
        # papers 2,3,4,5 = should have all 6 fixed pages - returned in page-number order
        # papers 12,13,14,15 should have 3 mobile pages (one each for q 1,2,3) - returned in question order
        assert len(mss_complete) == 9, f"mss_complete ({len(mss_complete)}) should be 9"

        for pn in [1]:
            assert pn in mss_complete
            assert len(mss_complete[pn]["fixed"]) == 6
            assert len(mss_complete[pn]["mobile"]) == 1
            f_page_data = mss_complete[pn]["fixed"]
            m_page_data = mss_complete[pn]["mobile"]
            for pg in range(1, 7):
                self.assertEqual(f_page_data[pg - 1]["page_number"], pg)
                assert (
                    "img_pk" in f_page_data[pg - 1]
                )  # not testing the actual value of image_pk

            for qn in [1]:  # is the 7th page of the test
                self.assertEqual(m_page_data[0]["question_idx"], qn)
                assert (
                    "img_pk" in m_page_data[qn - 1]
                )  # not testing the actual value of image_pk
        for pn in [2, 3, 4, 5]:
            assert pn in mss_complete
            assert len(mss_complete[pn]["fixed"]) == 6
            f_page_data = mss_complete[pn]["fixed"]
            for pg in range(1, 7):
                self.assertEqual(f_page_data[pg - 1]["page_number"], pg)
                assert (
                    "img_pk" in f_page_data[pg - 1]
                )  # not testing the actual value of image_pk

        for pn in [12, 13, 14]:
            assert pn in mss_complete
            m_page_data = mss_complete[pn]["mobile"]
            for qn in range(1, 4):
                self.assertEqual(m_page_data[qn - 1]["question_idx"], qn)
                assert (
                    "img_pk" in m_page_data[qn - 1]
                )  # not testing the actual value of image_pk

        for pn in [15]:
            assert pn in mss_complete
            m_page_data = mss_complete[pn]["mobile"]
            self.assertEqual(m_page_data[0]["question_idx"], MobilePage.DNM_qidx)
            self.assertEqual(m_page_data[1]["question_idx"], 1)
            self.assertEqual(m_page_data[2]["question_idx"], 2)
            self.assertEqual(m_page_data[3]["question_idx"], 3)
            for qn in range(1, 5):
                assert (
                    "img_pk" in m_page_data[qn - 1]
                )  # not testing the actual value of image_pk

    def test_is_paper_completely_scanned(self) -> None:
        """Test whether we can tell if a paper is scanned."""
        # papers 6, 7, 10, 11, 16, 17, 18 are incomplete

        assert ManageScanService.is_paper_completely_scanned(1)
        assert ManageScanService.is_paper_completely_scanned(12)
        assert not ManageScanService.is_paper_completely_scanned(6)
        assert not ManageScanService.is_paper_completely_scanned(16)
