"""Search Catalog MCP Tool - Query local catalog artifacts."""

from __future__ import annotations

from typing import Any, Dict, Iterable, List, Optional, Sequence

from igloo_mcp.catalog import CatalogIndex

from .base import MCPTool
from .schema_utils import (
    integer_schema,
    snowflake_identifier_schema,
    string_schema,
)


class SearchCatalogTool(MCPTool):
    """MCP tool for searching catalog artifacts generated by build_catalog."""

    def __init__(self) -> None:
        # No shared state required; catalog files are read per invocation
        pass

    @property
    def name(self) -> str:
        return "search_catalog"

    @property
    def description(self) -> str:
        return (
            "Search the locally cached Snowflake catalog built via build_catalog. "
            "Supports filtering by object type, database/schema, name, and column names."
        )

    @property
    def category(self) -> str:
        return "metadata"

    @property
    def tags(self) -> list[str]:
        return ["catalog", "metadata", "search", "discovery"]

    @property
    def usage_examples(self) -> list[Dict[str, Any]]:
        return [
            {
                "description": "Find tables containing the word 'customers'",
                "parameters": {
                    "name_contains": "customers",
                    "object_types": ["table"],
                },
            },
            {
                "description": "Search for columns mentioning 'revenue' in ANALYTICS database",
                "parameters": {
                    "database": "ANALYTICS",
                    "column_contains": "revenue",
                },
            },
        ]

    async def execute(
        self,
        catalog_dir: str = "./data_catalogue",
        object_types: Optional[List[str]] = None,
        database: Optional[str] = None,
        schema: Optional[str] = None,
        name_contains: Optional[str] = None,
        column_contains: Optional[str] = None,
        limit: int = 20,
        **kwargs: Any,
    ) -> Dict[str, Any]:
        try:
            index = CatalogIndex(catalog_dir)
            results, total_matches, metadata = index.search(
                object_types=_normalize_types(object_types),
                database=database,
                schema=schema,
                name_contains=name_contains,
                column_contains=column_contains,
                limit=max(1, limit),
            )

            return {
                "status": "success",
                "catalog_dir": catalog_dir,
                "metadata": metadata,
                "total_matches": total_matches,
                "limit": limit,
                "results": [
                    {
                        "object_type": obj.object_type,
                        "database": obj.database,
                        "schema": obj.schema,
                        "name": obj.name,
                        "comment": obj.comment,
                        "columns": obj.columns,
                        "raw": obj.raw,
                    }
                    for obj in results
                ],
            }
        except FileNotFoundError as exc:
            return {
                "status": "error",
                "error": str(exc),
                "catalog_dir": catalog_dir,
            }
        except Exception as exc:  # pragma: no cover - defensive
            return {
                "status": "error",
                "error": f"Catalog search failed: {exc}",
                "catalog_dir": catalog_dir,
            }

    def get_parameter_schema(self) -> Dict[str, Any]:
        return {
            "title": "Search Catalog Parameters",
            "type": "object",
            "additionalProperties": False,
            "properties": {
                "catalog_dir": string_schema(
                    "Directory containing catalog artifacts (catalog.json or catalog.jsonl).",
                    title="Catalog Directory",
                    default="./data_catalogue",
                    examples=["./data_catalogue", "./artifacts/catalog"],
                ),
                "object_types": {
                    "type": "array",
                    "title": "Object Types",
                    "items": {
                        "type": "string",
                        "enum": list(_SUPPORTED_OBJECT_TYPES),
                    },
                    "description": "Optional list of object types to include (e.g., table, view, function).",
                },
                "database": snowflake_identifier_schema(
                    "Filter results to a specific database.",
                    title="Database",
                    examples=["ANALYTICS", "PROD_DB"],
                ),
                "schema": snowflake_identifier_schema(
                    "Filter results to a specific schema.",
                    title="Schema",
                    examples=["PUBLIC", "REPORTING"],
                ),
                "name_contains": string_schema(
                    "Substring match on object name (case-insensitive).",
                    title="Name Contains",
                    examples=["customer", "sales"],
                ),
                "column_contains": string_schema(
                    "Substring match on column name (case-insensitive).",
                    title="Column Contains",
                    examples=["revenue", "id"],
                ),
                "limit": integer_schema(
                    "Maximum number of results to return.",
                    minimum=1,
                    maximum=500,
                    default=20,
                    examples=[10, 20, 50],
                ),
            },
        }


_SUPPORTED_OBJECT_TYPES: Sequence[str] = (
    "database",
    "schema",
    "table",
    "view",
    "materialized_view",
    "dynamic_table",
    "task",
    "function",
    "procedure",
)


def _normalize_types(types: Optional[Iterable[str]]) -> Optional[List[str]]:
    if not types:
        return None
    normalized: List[str] = []
    for value in types:
        if not value:
            continue
        lower = value.lower()
        if lower in _SUPPORTED_OBJECT_TYPES:
            normalized.append(lower)
    return normalized or None
