# Cite-Agent: AI Research Assistant

[![Version](https://img.shields.io/badge/version-1.4.1-blue.svg)](https://pypi.org/project/cite-agent/)
[![Python](https://img.shields.io/badge/python-3.8+-green.svg)](https://python.org)
[![License](https://img.shields.io/badge/license-MIT-orange.svg)](LICENSE)

**Cite-Agent** is a sophisticated AI research assistant that combines academic research, financial data, and truth-seeking capabilities in one powerful tool. Built for researchers, academics, and professionals who need accurate, cited information.

## 🌟 Features

### 🔬 **Academic Research**
- Search academic papers across multiple databases
- **Full PDF reading and summarization** (NEW!)
- Citation verification and quality scoring
- DOI resolution and metadata extraction
- Multi-source verification (Semantic Scholar, OpenAlex, PubMed)
- Automatic access to paywalled papers via Unpaywall

### 💰 **Financial Data**
- Real-time stock market data via FinSight API
- SEC filings and financial reports
- Company metrics and KPIs
- Historical financial analysis

### 🎯 **Truth-Seeking AI**
- Fact-checking with source verification
- Confidence scoring for responses
- Multi-language support (English, Chinese)
- Temperature-controlled responses (0.2 for accuracy)

### 📊 **Analytics & Tracking**
- User activity tracking
- Download analytics
- Usage statistics
- Citation quality metrics

### 🔄 **Workflow Integration** (NEW!)
- Local paper library management
- BibTeX export for citation managers
- Clipboard integration for instant citations
- Markdown export for Obsidian/Notion
- Session history and query replay
- **Zero context switching** - stay in your flow

## 🚀 Quick Start

### Installation

**Option 1: pipx (Recommended - handles PATH automatically)**
```bash
# Install pipx if you don't have it
pip install --user pipx
python3 -m pipx ensurepath

# Install cite-agent
pipx install cite-agent

# Ready to use (no PATH setup needed)
cite-agent --version
```

**Option 2: pip (requires PATH setup)**
```bash
# Install
pip install --user cite-agent

# Add to PATH (one-time setup)
echo 'export PATH="$HOME/.local/bin:$PATH"' >> ~/.bashrc
source ~/.bashrc

# Ready to use
cite-agent --version
```

**If cite-agent command not found:** Run `python3 -m cite_agent.cli` instead.

### Basic Usage

```bash
# Interactive mode
cite-agent

# Single query
cite-agent "Find research papers on machine learning in healthcare"

# Workflow integration (NEW!)
cite-agent "Find BERT paper" --save --format bibtex --copy
cite-agent --library              # View saved papers
cite-agent --export-bibtex        # Export to .bib file
cite-agent --history              # See recent queries

# Get help
cite-agent --help
```

### Python API

```python
import asyncio
from cite_agent import EnhancedNocturnalAgent, ChatRequest

async def main():
    agent = EnhancedNocturnalAgent()
    await agent.initialize()
    
    request = ChatRequest(
        question="What is the current state of AI in healthcare?",
        user_id="user123",
        conversation_id="conv456"
    )
    
    response = await agent.process_request(request)
    print(response.response)
    print(f"Confidence: {response.confidence_score}")
    print(f"Tools used: {response.tools_used}")
    
    await agent.close()

asyncio.run(main())
```

## 📖 Documentation

### Command Line Interface

#### Basic Commands

```bash
# Show version
cite-agent --version

# Interactive setup
cite-agent --setup

# Show tips
cite-agent --tips

# Check for updates
cite-agent --check-updates
```

#### Query Examples

```bash
# Academic research
cite-agent "Find papers on transformer architecture"
cite-agent "Verify this citation: Smith, J. (2023). AI in Medicine. Nature, 45(2), 123-145."

# Financial data
cite-agent "What is Apple's current revenue?"
cite-agent "Get Tesla's financial metrics for Q3 2024"

# Fact-checking
cite-agent "Is water's boiling point 100°C at standard pressure?"
cite-agent "Did Shakespeare write Harry Potter?"

# Multi-language
cite-agent "我的p值是0.05，這顯著嗎？"
cite-agent "天空是藍色的嗎？"
```

#### Runtime Controls

- Responses render immediately—there’s no artificial typing delay.
- Press `Ctrl+C` while the agent is thinking or streaming to interrupt and ask a different question on the spot.

### Python API Reference

#### EnhancedNocturnalAgent

The main agent class for programmatic access.

```python
class EnhancedNocturnalAgent:
    async def initialize(self, force_reload: bool = False)
    async def process_request(self, request: ChatRequest) -> ChatResponse
    async def process_request_streaming(self, request: ChatRequest)
    async def search_academic_papers(self, query: str, limit: int = 10) -> Dict[str, Any]
    async def get_financial_data(self, ticker: str, metric: str, limit: int = 12) -> Dict[str, Any]
    async def synthesize_research(self, paper_ids: List[str], max_words: int = 500) -> Dict[str, Any]
    async def close(self)
```

#### Data Models

```python
@dataclass
class ChatRequest:
    question: str
    user_id: str = "default"
    conversation_id: str = "default"
    context: Dict[str, Any] = field(default_factory=dict)

@dataclass
class ChatResponse:
    response: str
    tools_used: List[str] = field(default_factory=list)
    reasoning_steps: List[str] = field(default_factory=list)
    model: str = "enhanced-nocturnal-agent"
    timestamp: str = field(default_factory=lambda: datetime.now().isoformat())
    tokens_used: int = 0
    confidence_score: float = 0.0
    execution_results: Dict[str, Any] = field(default_factory=dict)
    api_results: Dict[str, Any] = field(default_factory=dict)
    error_message: Optional[str] = None
```

### API Endpoints

#### Authentication
- `POST /api/auth/register` - Register new user
- `POST /api/auth/login` - User login
- `POST /api/auth/refresh` - Refresh token
- `GET /api/auth/me` - Get current user info
- `POST /api/auth/logout` - Logout user

#### Research
- `POST /api/search` - Search academic papers
- `POST /api/synthesize` - Synthesize research papers
- `POST /api/format` - Format citations

#### Financial Data
- `GET /v1/finance/calc/{ticker}/{metric}` - Get financial metrics
- `GET /v1/finance/kpis/{ticker}` - Get company KPIs
- `GET /v1/finance/reports/{ticker}` - Get financial reports

#### Analytics
- `GET /api/download/stats/summary` - Download statistics
- `GET /api/analytics/overview` - Usage overview
- `GET /api/analytics/users` - User statistics

#### Download Tracking
- `GET /api/download/windows` - Track Windows downloads
- `GET /api/download/macos` - Track macOS downloads
- `GET /api/download/linux` - Track Linux downloads

## 🔧 Configuration

### Environment Variables

```bash
# Authentication
NOCTURNAL_ACCOUNT_EMAIL=your@email.edu
NOCTURNAL_ACCOUNT_PASSWORD=your_password

# API Configuration
NOCTURNAL_API_URL=https://cite-agent-api-720dfadd602c.herokuapp.com
ARCHIVE_API_URL=https://cite-agent-api-720dfadd602c.herokuapp.com/api
FINSIGHT_API_URL=https://cite-agent-api-720dfadd602c.herokuapp.com/v1/finance

# Optional
NOCTURNAL_DEBUG=1  # Enable debug logging
NOCTURNAL_QUERY_LIMIT=25  # Default query limit
```

### Session Management

Sessions are automatically managed and stored in:
- **Linux/macOS**: `~/.nocturnal_archive/session.json`
- **Windows**: `%USERPROFILE%\.nocturnal_archive\session.json`

## 📊 Analytics & Monitoring

### User Tracking

The system automatically tracks:
- User registrations and logins
- Query history and usage patterns
- Token consumption and costs
- Response quality and citation accuracy

### Download Analytics

Track installer downloads across platforms:
- Windows, macOS, Linux downloads
- Geographic distribution (IP-based)
- Referrer tracking
- Download trends and patterns

### Dashboard Access

Access the analytics dashboard at:
```
https://cite-agent-api-720dfadd602c.herokuapp.com/dashboard
```

## 💰 Monetization & Pricing

### Current Pricing Tiers

| Tier | Price | Queries/Month | Rate Limit | Features |
|------|-------|---------------|------------|----------|
| **Free** | $0 | 100 | 100/hour | Basic research, limited finance |
| **Pro** | $9/month | 1,000 | 1,000/hour | Full features, priority support |
| **Academic** | $5/month | 500 | 500/hour | Student discount, same features |
| **Enterprise** | $99/month | Unlimited | 5,000/hour | API access, custom integrations |

### Revenue Model

- **Subscription-based**: Monthly recurring revenue
- **Usage-based**: Pay-per-query options available
- **API licensing**: Enterprise customers
- **White-label**: Custom deployments

## 🛠️ Development

### Local Development

```bash
# Clone repository
git clone https://github.com/yourusername/cite-agent.git
cd cite-agent

# Install dependencies
pip install -r requirements.txt
pip install -r requirements-dev.txt

# Run tests
pytest

# Start development server
python -m cite_agent.dashboard
```

### Building from Source

```bash
# Build wheel
python setup.py bdist_wheel

# Install locally
pip install dist/cite_agent-1.0.5-py3-none-any.whl
```

### Contributing

1. Fork the repository
2. Create a feature branch
3. Make your changes
4. Add tests
5. Submit a pull request

## 🔒 Security & Privacy

### Data Protection
- All user data encrypted in transit and at rest
- JWT-based authentication with 30-day expiration
- No storage of sensitive personal information
- GDPR compliant data handling

### API Security
- Rate limiting per user tier
- Input validation and sanitization
- SQL injection prevention
- CORS protection

## 📈 Performance

### Benchmarks
- **Average response time**: 2-5 seconds
- **Citation verification**: 95%+ accuracy
- **Uptime**: 99.9% SLA
- **Concurrent users**: 1000+ supported

### Optimization
- Async/await architecture
- Connection pooling
- Response caching
- CDN distribution

## 🐛 Troubleshooting

### Common Issues

#### CLI Hangs on Startup
```bash
# Clear session and reconfigure
rm -rf ~/.nocturnal_archive
cite-agent --setup
```

#### Authentication Errors
```bash
# Check credentials
cite-agent --setup

# Verify email format (must be academic)
# Valid: user@university.edu, student@ac.uk
# Invalid: user@gmail.com, user@company.com
```

#### API Connection Issues
```bash
# Check network connectivity
curl https://cite-agent-api-720dfadd602c.herokuapp.com/api/health

# Verify API keys
echo $NOCTURNAL_ACCOUNT_EMAIL
```

### Debug Mode

```bash
# Enable debug logging
export NOCTURNAL_DEBUG=1
cite-agent "your query"
```

### Support

- **Documentation**: [Full docs](https://docs.cite-agent.com)
- **Issues**: [GitHub Issues](https://github.com/yourusername/cite-agent/issues)
- **Email**: support@cite-agent.com
- **Discord**: [Community Server](https://discord.gg/cite-agent)

## 📄 License

This project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details.

## 🙏 Acknowledgments

- **OpenAlex** for academic data
- **Semantic Scholar** for research papers
- **FinSight** for financial data
- **Groq** for LLM processing
- **FastAPI** for the backend framework

## 📞 Contact

- **Website**: https://cite-agent.com
- **Email**: contact@cite-agent.com
- **Twitter**: [@cite_agent](https://twitter.com/cite_agent)
- **LinkedIn**: [Cite-Agent](https://linkedin.com/company/cite-agent)

---

**Made with ❤️ for the research community**
