from numpy import exp
from arsenal import colors
from abc import ABC, abstractmethod
from arsenal.maths import sample_dict

from ..util import escape


class StatefulTokenizedLM:
    """A stateful tokenized language model that maintains context and generates next token logprobs.

    Args:
        model (genlm.backend.AsyncLM): The underlying language model
        context (list): List of token IDs representing the current context
        n_calls (int): Number of times the model has been called
        max_context_length (int, optional): Maximum length of context to maintain
    """

    def __init__(self, model, context, n_calls=0, max_context_length=None):
        self.model = model
        self.context = context
        self._n_calls = n_calls
        self.max_context_length = max_context_length

    @classmethod
    def initial(cls, model, initial_context=None, max_context_length=None):
        """Creates an initial state for the language model.

        Args:
            model (genlm.backend.AsyncLM): The language model to use
            initial_context (list, optional): Initial context of token IDs. Defaults to [tokenizer.bos_token_id]
            max_context_length (int, optional): Maximum context length to maintain

        Returns:
            (StatefulTokenizedLM): A new instance with initial state
        """
        if initial_context is None:
            initial_context = [model.tokenizer.bos_token_id]
        return cls(model, initial_context, max_context_length=max_context_length)

    def __lshift__(self, token):
        """Adds a new token to the context and returns a new state.

        Args:
            token (int): Token ID to add to context

        Returns:
            (StatefulTokenizedLM): New state with updated context
        """
        assert isinstance(token, int)
        if (
            self.max_context_length is not None
            and len(self.context) >= self.max_context_length
        ):
            self.context = self.context[-(self.max_context_length - 1) :]
        return StatefulTokenizedLM(
            self.model, self.context + [token], n_calls=self._n_calls
        )

    async def logp_next(self):
        """Computes log probabilities for the next token given the current context.

        Returns:
            (torch.Tensor): Log probabilities for next tokens
        """
        self._n_calls += 1
        return await self.model.next_token_logprobs(self.context)

    def __repr__(self):
        return colors.purple % (
            "|".join([escape(self.model.byte_vocab[x]) for x in self.context])
        )


class StatefulByteLM(ABC):
    """Abstract base class for byte-level language models with state."""

    @abstractmethod
    async def __lshift__(self, b: int):
        """Adds a byte to the state and returns new state.

        Args:
            b (int): Byte to add

        Returns:
            (StatefulByteLM): New state with added byte
        """
        pass  # pragma: no cover

    def prune(self):
        """Prunes the current state if needed.

        Override in subclasses.

        Returns:
            (StatefulByteLM): Pruned state
        """
        raise NotImplementedError(
            "Subclasses must implement this method"
        )  # pragma: no cover

    @abstractmethod
    async def logp_next(self):
        """Computes the log probability distribution for the next byte.

        Returns:
            (LazyByteProbs): Log probabilities for next possible bytes
        """
        pass  # pragma: no cover

    async def greedy(self, context, steps):
        """Performs greedy decoding for given number of steps.

        Args:
            context (bytes): Initial byte context
            steps (int): Number of generation steps

        Returns:
            (bytes): Generated byte sequence
        """
        context = list(context)
        state = await self.prefill(context)
        for _ in range(steps):
            Q = (await state.logp_next()).materialize()
            b = Q.argmax()
            state = await (state.prune() << b)
            context.append(b)
        return bytes(context)

    async def sample(self, context, steps, draw=sample_dict):
        """Samples from the model for given number of steps.

        Args:
            context (bytes): Initial byte context
            steps (int): Number of generation steps
            draw: Sampling function to use (default: sample_dict)

        Returns:
            (bytes): Generated byte sequence
        """
        context = list(context)
        state = await self.prefill(context)
        for _ in range(steps):
            Q = (await state.logp_next()).materialize()
            b = draw(Q.map_values(exp))
            state = await (state.prune() << b)
            context.append(b)
        return bytes(context)

    async def cleanup(self):
        """Performs any necessary cleanup of the model state."""
        pass  # pragma: no cover
