#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
# GAM7
#
# Copyright 2025, All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
GAM is a command line tool which allows Administrators to control their Google Workspace domain and accounts.

For more information, see:
https://github.com/GAM-team/GAM
https://github.com/GAM-team/GAM/wiki
"""

__author__ = 'GAM Team <google-apps-manager@googlegroups.com>'
__version__ = '7.15.00'
__license__ = 'Apache License 2.0 (http://www.apache.org/licenses/LICENSE-2.0)'

#pylint: disable=wrong-import-position
import base64
import calendar as calendarlib
import codecs
import collections
import configparser
import csv
import datetime
from email.charset import add_charset, QP
from email.generator import Generator
from email.header import decode_header, Header
from email import message_from_string
from email.mime.application import MIMEApplication
from email.mime.audio import MIMEAudio
from email.mime.base import MIMEBase
from email.mime.image import MIMEImage
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from email.utils import formatdate
from email.policy import SMTP as policySMTP
import hashlib
from html.entities import name2codepoint
from html.parser import HTMLParser
import http.client as http_client
import importlib
from importlib.metadata import version as lib_version
import io
import ipaddress
import json
import logging
from logging.handlers import RotatingFileHandler
import mimetypes
import multiprocessing
import os
import platform
import queue
import random
import re
from secrets import SystemRandom
import shlex
import signal
import smtplib
import socket
import sqlite3
import ssl
import string
import struct
import subprocess
import sys
from tempfile import TemporaryFile
try:
  import termios
except ImportError:
  # termios does not exist for Windows
  pass
import threading
import time
from traceback import print_exc
import types
from urllib.parse import quote, quote_plus, unquote, urlencode, urlparse, parse_qs
import uuid
import warnings
import webbrowser
import wsgiref.simple_server
import wsgiref.util
import zipfile

# disable legacy stuff we don't use and isn't secure
os.environ['CRYPTOGRAPHY_OPENSSL_NO_LEGACY'] = "1"
from cryptography import x509
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives import hashes, serialization
from cryptography.hazmat.primitives.asymmetric import rsa
from cryptography.x509.oid import NameOID

# 10/2024 - I don't recall why we did this but PyInstaller
# 6.10.0+ does not like it. Only run this when we're not
# Frozen.
if not getattr(sys, 'frozen', False):
  sys.path.insert(0, os.path.dirname(os.path.realpath(__file__)))

from dateutil.relativedelta import relativedelta

from pathvalidate import sanitize_filename, sanitize_filepath

import google.oauth2.credentials
import google.oauth2.id_token
import google.auth
from google.auth.jwt import Credentials as JWTCredentials
import google.oauth2.service_account
import google_auth_oauthlib.flow
import google_auth_httplib2
import httplib2

httplib2.RETRIES = 5

from passlib.hash import sha512_crypt
from filelock import FileLock

if platform.system() == 'Linux':
  import distro

from gamlib import glaction
from gamlib import glapi as API
from gamlib import glcfg as GC
from gamlib import glclargs
from gamlib import glentity
from gamlib import glgapi as GAPI
from gamlib import glgdata as GDATA
from gamlib import glglobals as GM
from gamlib import glindent
from gamlib import glmsgs as Msg
from gamlib import glskus as SKU
from gamlib import gluprop as UProp
from gamlib import glverlibs

import gdata.apps.service
import gdata.apps.audit
import gdata.apps.audit.service
import gdata.apps.contacts
import gdata.apps.contacts.service
# Import local library, does not include discovery documents
import googleapiclient
import googleapiclient.discovery
import googleapiclient.errors
import googleapiclient.http
from iso8601 import iso8601

IS08601_TIME_FORMAT = '%Y-%m-%dT%H:%M:%S%:z'
RFC2822_TIME_FORMAT = '%a, %d %b %Y %H:%M:%S %z'

def ISOformatTimeStamp(timestamp):
  return timestamp.isoformat('T', 'seconds')

def currentISOformatTimeStamp(timespec='milliseconds'):
  return datetime.datetime.now(GC.Values[GC.TIMEZONE]).isoformat('T', timespec)

Act = glaction.GamAction()
Cmd = glclargs.GamCLArgs()
Ent = glentity.GamEntity()
Ind = glindent.GamIndent()

# Finding path method varies between Python source, PyInstaller and StaticX
if os.environ.get('STATICX_PROG_PATH', False):
  # StaticX static executable
  GM.Globals[GM.GAM_PATH] = os.path.dirname(os.environ['STATICX_PROG_PATH'])
  GM.Globals[GM.GAM_TYPE] = 'staticx'
elif getattr(sys, 'frozen', False):
  # Pyinstaller executable
  GM.Globals[GM.GAM_PATH] = os.path.dirname(sys.executable)
  GM.Globals[GM.GAM_TYPE] = 'pyinstaller'
else:
  # Source code
  GM.Globals[GM.GAM_PATH] = os.path.dirname(os.path.realpath(__file__))
  GM.Globals[GM.GAM_TYPE] = 'pythonsource'

GIT_USER = 'GAM-team'
GAM = 'GAM'
GAM_URL = f'https://github.com/{GIT_USER}/{GAM}'
GAM_USER_AGENT = (f'{GAM} {__version__} - {GAM_URL} / '
                  f'{__author__} / '
                  f'Python {sys.version_info[0]}.{sys.version_info[1]}.{sys.version_info[2]} {sys.version_info[3]} / '
                  f'{platform.platform()} {platform.machine()} /'
                  )
GAM_RELEASES = f'https://github.com/{GIT_USER}/{GAM}/releases'
GAM_WIKI = f'https://github.com/{GIT_USER}/{GAM}/wiki'
GAM_LATEST_RELEASE = f'https://api.github.com/repos/{GIT_USER}/{GAM}/releases/latest'
GAM_PROJECT_CREATION = 'GAM Project Creation'
GAM_PROJECT_CREATION_CLIENT_ID = '297408095146-fug707qsjv4ikron0hugpevbrjhkmsk7.apps.googleusercontent.com'

TRUE = 'true'
FALSE = 'false'
TRUE_VALUES = [TRUE, 'on', 'yes', 'enabled', '1']
FALSE_VALUES = [FALSE, 'off', 'no', 'disabled', '0']
TRUE_FALSE = [TRUE, FALSE]
ERROR = 'ERROR'
ERROR_PREFIX = ERROR+': '
WARNING = 'WARNING'
WARNING_PREFIX = WARNING+': '
ONE_KILO_10_BYTES = 1000
ONE_MEGA_10_BYTES = 1000000
ONE_GIGA_10_BYTES = 1000000000
ONE_KILO_BYTES = 1024
ONE_MEGA_BYTES = 1048576
ONE_GIGA_BYTES = 1073741824
SECONDS_PER_MINUTE = 60
SECONDS_PER_HOUR = 3600
SECONDS_PER_DAY = 86400
SECONDS_PER_WEEK = 604800
MAX_GOOGLE_SHEET_CELLS = 10000000 # See https://support.google.com/drive/answer/37603
MAX_LOCAL_GOOGLE_TIME_OFFSET = 30
SHARED_DRIVE_MAX_FILES_FOLDERS = 500000
UTF8 = 'utf-8'
UTF8_SIG = 'utf-8-sig'
EV_GAMCFGDIR = 'GAMCFGDIR'
EV_GAMCFGSECTION = 'GAMCFGSECTION'
EV_OLDGAMPATH = 'OLDGAMPATH'
FN_GAM_CFG = 'gam.cfg'
FN_LAST_UPDATE_CHECK_TXT = 'lastupdatecheck.txt'
FN_GAMCOMMANDS_TXT = 'GamCommands.txt'
MY_DRIVE = 'My Drive'
TEAM_DRIVE = 'Drive'
ROOT = 'root'
ROOTID = 'rootid'
ORPHANS = 'Orphans'
SHARED_WITHME = 'SharedWithMe'
SHARED_DRIVES = 'SharedDrives'

LOWERNUMERIC_CHARS = string.ascii_lowercase+string.digits
ALPHANUMERIC_CHARS = LOWERNUMERIC_CHARS+string.ascii_uppercase
URL_SAFE_CHARS = ALPHANUMERIC_CHARS+'-._~'
PASSWORD_SAFE_CHARS = ALPHANUMERIC_CHARS+'!#$%&()*-./:;<=>?@[\\]^_{|}~'
FILENAME_SAFE_CHARS = ALPHANUMERIC_CHARS+'-_.() '
CHAT_MESSAGEID_CHARS = string.ascii_lowercase+string.digits+'-'

ADMIN_ACCESS_OPTIONS = {'adminaccess', 'asadmin'}
OWNER_ACCESS_OPTIONS = {'owneraccess', 'asowner'}

# Python 3 values
DEFAULT_CSV_READ_MODE = 'r'
DEFAULT_FILE_APPEND_MODE = 'a'
DEFAULT_FILE_READ_MODE = 'r'
DEFAULT_FILE_WRITE_MODE = 'w'

# Google API constants
APPLICATION_VND_GOOGLE_APPS = 'application/vnd.google-apps.'
MIMETYPE_GA_DOCUMENT = f'{APPLICATION_VND_GOOGLE_APPS}document'
MIMETYPE_GA_DRAWING = f'{APPLICATION_VND_GOOGLE_APPS}drawing'
MIMETYPE_GA_FILE = f'{APPLICATION_VND_GOOGLE_APPS}file'
MIMETYPE_GA_FOLDER = f'{APPLICATION_VND_GOOGLE_APPS}folder'
MIMETYPE_GA_FORM = f'{APPLICATION_VND_GOOGLE_APPS}form'
MIMETYPE_GA_FUSIONTABLE = f'{APPLICATION_VND_GOOGLE_APPS}fusiontable'
MIMETYPE_GA_JAM = f'{APPLICATION_VND_GOOGLE_APPS}jam'
MIMETYPE_GA_MAP = f'{APPLICATION_VND_GOOGLE_APPS}map'
MIMETYPE_GA_PRESENTATION = f'{APPLICATION_VND_GOOGLE_APPS}presentation'
MIMETYPE_GA_SCRIPT = f'{APPLICATION_VND_GOOGLE_APPS}script'
MIMETYPE_GA_SCRIPT_JSON = f'{APPLICATION_VND_GOOGLE_APPS}script+json'
MIMETYPE_GA_SHORTCUT = f'{APPLICATION_VND_GOOGLE_APPS}shortcut'
MIMETYPE_GA_3P_SHORTCUT = f'{APPLICATION_VND_GOOGLE_APPS}drive-sdk'
MIMETYPE_GA_SITE = f'{APPLICATION_VND_GOOGLE_APPS}site'
MIMETYPE_GA_SPREADSHEET = f'{APPLICATION_VND_GOOGLE_APPS}spreadsheet'
MIMETYPE_TEXT_CSV = 'text/csv'
MIMETYPE_TEXT_HTML = 'text/html'
MIMETYPE_TEXT_PLAIN = 'text/plain'

GOOGLE_NAMESERVERS = ['8.8.8.8', '8.8.4.4']
GOOGLE_TIMECHECK_LOCATION = 'admin.googleapis.com'
NEVER_DATE = '1970-01-01'
NEVER_DATETIME = '1970-01-01 00:00'
NEVER_TIME = '1970-01-01T00:00:00.000Z'
NEVER_TIME_NOMS = '1970-01-01T00:00:00Z'
NEVER_END_DATE = '1969-12-31'
NEVER_START_DATE = NEVER_DATE
PROJECTION_CHOICE_MAP = {'basic': 'BASIC', 'full': 'FULL'}
REFRESH_EXPIRY = '1970-01-01T00:00:01Z'
REPLACE_GROUP_PATTERN = re.compile(r'\\(\d+)')
UNKNOWN = 'Unknown'

# Queries
ME_IN_OWNERS = "'me' in owners"
ME_IN_OWNERS_AND = ME_IN_OWNERS+" and "
AND_ME_IN_OWNERS = " and "+ME_IN_OWNERS
NOT_ME_IN_OWNERS = "not "+ME_IN_OWNERS
NOT_ME_IN_OWNERS_AND = NOT_ME_IN_OWNERS+" and "
AND_NOT_ME_IN_OWNERS = " and "+NOT_ME_IN_OWNERS
ANY_FOLDERS = "mimeType = '"+MIMETYPE_GA_FOLDER+"'"
MY_FOLDERS = ME_IN_OWNERS_AND+ANY_FOLDERS
NON_TRASHED = "trashed = false"
WITH_PARENTS = "'{0}' in parents"
ANY_NON_TRASHED_WITH_PARENTS = "trashed = false and '{0}' in parents"
ANY_NON_TRASHED_FOLDER_NAME = "mimeType = '"+MIMETYPE_GA_FOLDER+"' and name = '{0}' and trashed = false"
MY_NON_TRASHED_FOLDER_NAME = ME_IN_OWNERS_AND+ANY_NON_TRASHED_FOLDER_NAME
MY_NON_TRASHED_FOLDER_NAME_WITH_PARENTS = ME_IN_OWNERS_AND+"mimeType = '"+MIMETYPE_GA_FOLDER+"' and name = '{0}' and trashed = false and '{1}' in parents"
ANY_NON_TRASHED_FOLDER_NAME_WITH_PARENTS = "mimeType = '"+MIMETYPE_GA_FOLDER+"' and name = '{0}' and trashed = false and '{1}' in parents"
WITH_ANY_FILE_NAME = "name = '{0}'"
WITH_MY_FILE_NAME = ME_IN_OWNERS_AND+WITH_ANY_FILE_NAME
WITH_OTHER_FILE_NAME = NOT_ME_IN_OWNERS_AND+WITH_ANY_FILE_NAME
AND_NOT_SHORTCUT = " and mimeType != '"+MIMETYPE_GA_SHORTCUT+"'"

# Program return codes
UNKNOWN_ERROR_RC = 1
USAGE_ERROR_RC = 2
SOCKET_ERROR_RC = 3
GOOGLE_API_ERROR_RC = 4
NETWORK_ERROR_RC = 5
FILE_ERROR_RC = 6
MEMORY_ERROR_RC = 7
KEYBOARD_INTERRUPT_RC = 8
HTTP_ERROR_RC = 9
SCOPES_NOT_AUTHORIZED_RC = 10
DATA_ERROR_RC = 11
API_ACCESS_DENIED_RC = 12
CONFIG_ERROR_RC = 13
SYSTEM_ERROR_RC = 14
NO_SCOPES_FOR_API_RC = 15
CLIENT_SECRETS_JSON_REQUIRED_RC = 16
OAUTH2SERVICE_JSON_REQUIRED_RC = 16
OAUTH2_TXT_REQUIRED_RC = 16
INVALID_JSON_RC = 17
JSON_ALREADY_EXISTS_RC = 17
AUTHENTICATION_TOKEN_REFRESH_ERROR_RC = 18
HARD_ERROR_RC = 19
# Information
ENTITY_IS_A_USER_RC = 20
ENTITY_IS_A_USER_ALIAS_RC = 21
ENTITY_IS_A_GROUP_RC = 22
ENTITY_IS_A_GROUP_ALIAS_RC = 23
ENTITY_IS_AN_UNMANAGED_ACCOUNT_RC = 24
ORGUNIT_NOT_EMPTY_RC = 25
USER_SUSPENDED_RC = 26
CHECK_USER_GROUPS_ERROR_RC = 29
ORPHANS_COLLECTED_RC = 30
# Warnings/Errors
ACTION_FAILED_RC = 50
ACTION_NOT_PERFORMED_RC = 51
INVALID_ENTITY_RC = 52
BAD_REQUEST_RC = 53
ENTITY_IS_NOT_UNIQUE_RC = 54
DATA_NOT_AVALIABLE_RC = 55
ENTITY_DOES_NOT_EXIST_RC = 56
ENTITY_DUPLICATE_RC = 57
ENTITY_IS_NOT_AN_ALIAS_RC = 58
ENTITY_IS_UKNOWN_RC = 59
NO_ENTITIES_FOUND_RC = 60
INVALID_DOMAIN_RC = 61
INVALID_DOMAIN_VALUE_RC = 62
INVALID_TOKEN_RC = 63
JSON_LOADS_ERROR_RC = 64
MULTIPLE_DELETED_USERS_FOUND_RC = 65
MULTIPLE_PROJECT_FOLDERS_FOUND_RC = 65
STDOUT_STDERR_ERROR_RC = 66
INSUFFICIENT_PERMISSIONS_RC = 67
REQUEST_COMPLETED_NO_RESULTS_RC = 71
REQUEST_NOT_COMPLETED_RC = 72
SERVICE_NOT_APPLICABLE_RC = 73
TARGET_DRIVE_SPACE_ERROR_RC = 74
USER_REQUIRED_TO_CHANGE_PASSWORD_ERROR_RC = 75
USER_SUSPENDED_ERROR_RC = 76
NO_CSV_DATA_TO_UPLOAD_RC = 77
NO_SA_ACCESS_CONTEXT_MANAGER_EDITOR_ROLE_RC = 78
ACCESS_POLICY_ERROR_RC = 79
YUBIKEY_CONNECTION_ERROR_RC = 80
YUBIKEY_INVALID_KEY_TYPE_RC = 81
YUBIKEY_INVALID_SLOT_RC = 82
YUBIKEY_INVALID_PIN_RC = 83
YUBIKEY_APDU_ERROR_RC = 84
YUBIKEY_VALUE_ERROR_RC = 85
YUBIKEY_MULTIPLE_CONNECTED_RC = 86
YUBIKEY_NOT_FOUND_RC = 87

# Multiprocessing lock
mplock = None

# stdin/stdout/stderr
def readStdin(prompt):
  return input(prompt)

def stdErrorExit(e):
  try:
    sys.stderr.write(f'\n{ERROR_PREFIX}{str(e)}\n')
  except IOError:
    pass
  sys.exit(STDOUT_STDERR_ERROR_RC)

def writeStdout(data):
  try:
    GM.Globals[GM.STDOUT].get(GM.REDIRECT_MULTI_FD, sys.stdout).write(data)
  except IOError as e:
    stdErrorExit(e)

def flushStdout():
  try:
    GM.Globals[GM.STDOUT].get(GM.REDIRECT_MULTI_FD, sys.stdout).flush()
  except IOError as e:
    stdErrorExit(e)

def writeStderr(data):
  flushStdout()
  try:
    GM.Globals[GM.STDERR].get(GM.REDIRECT_MULTI_FD, sys.stderr).write(data)
  except IOError as e:
    stdErrorExit(e)

def flushStderr():
  try:
    GM.Globals[GM.STDERR].get(GM.REDIRECT_MULTI_FD, sys.stderr).flush()
  except IOError as e:
    stdErrorExit(e)

# Error messages
def setSysExitRC(sysRC):
  GM.Globals[GM.SYSEXITRC] = sysRC

def stderrErrorMsg(message):
  writeStderr(f'\n{ERROR_PREFIX}{message}\n')

def stderrWarningMsg(message):
  writeStderr(f'\n{WARNING_PREFIX}{message}\n')

def systemErrorExit(sysRC, message):
  if message:
    stderrErrorMsg(message)
  sys.exit(sysRC)

def printErrorMessage(sysRC, message):
  setSysExitRC(sysRC)
  writeStderr(f'\n{Ind.Spaces()}{ERROR_PREFIX}{message}\n')

def printWarningMessage(sysRC, message):
  setSysExitRC(sysRC)
  writeStderr(f'\n{Ind.Spaces()}{WARNING_PREFIX}{message}\n')

def supportsColoredText():
  """Determines if the current terminal environment supports colored text.

  Returns:
    Bool, True if the current terminal environment supports colored text via
    ANSI escape characters.
  """
  # Make a rudimentary check for Windows. Though Windows does seem to support
  # colorization with VT100 emulation, it is disabled by default. Therefore,
  # we'll simply disable it in GAM on Windows for now.
  return not sys.platform.startswith('win')

def createColoredText(text, color):
  """Uses ANSI escape characters to create colored text in supported terminals.

  See more at https://en.wikipedia.org/wiki/ANSI_escape_code#Colors

  Args:
    text: String, The text to colorize using ANSI escape characters.
    color: String, An ANSI escape sequence denoting the color of the text to be
      created. See more at https://en.wikipedia.org/wiki/ANSI_escape_code#Colors

  Returns:
    The input text with appropriate ANSI escape characters to create
    colorization in a supported terminal environment.
  """
  END_COLOR_SEQUENCE = '\033[0m'  # Ends the applied color formatting
  if supportsColoredText():
    return color + text + END_COLOR_SEQUENCE
  return text  # Hand back the plain text, uncolorized.

def createRedText(text):
  """Uses ANSI encoding to create red colored text if supported."""
  return createColoredText(text, '\033[91m')

def createGreenText(text):
  """Uses ANSI encoding to create green colored text if supported."""
  return createColoredText(text, '\u001b[32m')

def createYellowText(text):
  """Uses ANSI encoding to create yellow text if supported."""
  return createColoredText(text, '\u001b[33m')

def executeBatch(dbatch):
  dbatch.execute()
  if GC.Values[GC.INTER_BATCH_WAIT] > 0:
    time.sleep(GC.Values[GC.INTER_BATCH_WAIT])

def _stripControlCharsFromName(name):
  for cc in ['\x00', '\r', '\n']:
    name = name.replace(cc, '')
  return name

class LazyLoader(types.ModuleType):
  """Lazily import a module, mainly to avoid pulling in large dependencies.

  `contrib`, and `ffmpeg` are examples of modules that are large and not always
  needed, and this allows them to only be loaded when they are used.
  """

  # The lint error here is incorrect.
  def __init__(self, local_name, parent_module_globals, name):
    self._local_name = local_name
    self._parent_module_globals = parent_module_globals

    super().__init__(name)

  def _load(self):
    # Import the target module and insert it into the parent's namespace
    module = importlib.import_module(self.__name__)
    self._parent_module_globals[self._local_name] = module

    # Update this object's dict so that if someone keeps a reference to the
    #   LazyLoader, lookups are efficient (__getattr__ is only called on lookups
    #   that fail).
    self.__dict__.update(module.__dict__)

    return module

  def __getattr__(self, item):
    module = self._load()
    return getattr(module, item)

  def __dir__(self):
    module = self._load()
    return dir(module)

yubikey = LazyLoader('yubikey', globals(), 'gam.gamlib.yubikey')

# gam yubikey resetpvi [yubikey_serialnumber <String>]
def doResetYubiKeyPIV():
  new_data = {}
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'yubikeyserialnumber':
      new_data['yubikey_serial_number'] =  getInteger()
    else:
      unknownArgumentExit()
  yk = yubikey.YubiKey(new_data)
  yk.serial_number = yk.get_serial_number()
  yk.reset_piv()

class _DeHTMLParser(HTMLParser): #pylint: disable=abstract-method
  def __init__(self):
    HTMLParser.__init__(self)
    self.__text = []

  def handle_data(self, data):
    self.__text.append(data)

  def handle_charref(self, name):
    self.__text.append(chr(int(name[1:], 16)) if name.startswith('x') else chr(int(name)))

  def handle_entityref(self, name):
    cp = name2codepoint.get(name)
    if cp:
      self.__text.append(chr(cp))
    else:
      self.__text.append('&'+name)

  def handle_starttag(self, tag, attrs):
    if tag == 'p':
      self.__text.append('\n\n')
    elif tag == 'br':
      self.__text.append('\n')
    elif tag == 'a':
      for attr in attrs:
        if attr[0] == 'href':
          self.__text.append(f'({attr[1]}) ')
          break
    elif tag == 'div':
      if not attrs:
        self.__text.append('\n')
    elif tag in {'http:', 'https'}:
      self.__text.append(f' ({tag}//{attrs[0][0]}) ')

  def handle_startendtag(self, tag, attrs):
    if tag == 'br':
      self.__text.append('\n\n')

  def text(self):
    return re.sub(r'\n{2}\n+', '\n\n', re.sub(r'\n +', '\n', ''.join(self.__text))).strip()

def dehtml(text):
  parser = _DeHTMLParser()
  parser.feed(str(text))
  parser.close()
  return parser.text()

def currentCount(i, count):
  return f' ({i}/{count})' if (count > GC.Values[GC.SHOW_COUNTS_MIN]) else ''

def currentCountNL(i, count):
  return f' ({i}/{count})\n' if (count > GC.Values[GC.SHOW_COUNTS_MIN]) else '\n'

# Format a key value list
#   key, value	-> "key: value" + ", " if not last item
#   key, ''	-> "key:" + ", " if not last item
#   key, None	-> "key" + " " if not last item
def formatKeyValueList(prefixStr, kvList, suffixStr):
  msg = prefixStr
  i = 0
  l = len(kvList)
  while i < l:
    if isinstance(kvList[i], (bool, float, int)):
      msg += str(kvList[i])
    else:
      msg += kvList[i]
    i += 1
    if i < l:
      val = kvList[i]
      if (val is not None) or (i == l-1):
        msg += ':'
        if (val is not None) and (not isinstance(val, str) or val):
          msg += ' '
          if isinstance(val, (bool, float, int)):
            msg += str(val)
          else:
            msg += val
        i += 1
        if i < l:
          msg += ', '
      else:
        i += 1
        if i < l:
          msg += ' '
  msg += suffixStr
  return msg

# Something's wrong with CustomerID??
def accessErrorMessage(cd, errMsg=None):
  if cd is None:
    cd = buildGAPIObject(API.DIRECTORY)
  try:
    callGAPI(cd.customers(), 'get',
             throwReasons=[GAPI.BAD_REQUEST, GAPI.INVALID_INPUT, GAPI.RESOURCE_NOT_FOUND,
                           GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
             customerKey=GC.Values[GC.CUSTOMER_ID], fields='id')
  except (GAPI.badRequest, GAPI.invalidInput):
    return formatKeyValueList('',
                              [Ent.Singular(Ent.CUSTOMER_ID), GC.Values[GC.CUSTOMER_ID],
                               Msg.INVALID],
                              '')
  except GAPI.resourceNotFound:
    return formatKeyValueList('',
                              [Ent.Singular(Ent.CUSTOMER_ID), GC.Values[GC.CUSTOMER_ID],
                               Msg.DOES_NOT_EXIST],
                              '')
  except GAPI.forbidden:
    return formatKeyValueList('',
                              Ent.FormatEntityValueList([Ent.CUSTOMER_ID, GC.Values[GC.CUSTOMER_ID],
                                                         Ent.DOMAIN, GC.Values[GC.DOMAIN],
                                                         Ent.USER, GM.Globals[GM.ADMIN]])+[Msg.ACCESS_FORBIDDEN],
                              '')
  if errMsg:
    return formatKeyValueList('',
                              [Ent.Singular(Ent.CUSTOMER_ID), GC.Values[GC.CUSTOMER_ID],
                               errMsg],
                              '')
  return None

def accessErrorExit(cd, errMsg=None):
  systemErrorExit(INVALID_DOMAIN_RC, accessErrorMessage(cd or buildGAPIObject(API.DIRECTORY), errMsg))

def accessErrorExitNonDirectory(api, errMsg):
  systemErrorExit(API_ACCESS_DENIED_RC,
                  formatKeyValueList('',
                                     Ent.FormatEntityValueList([Ent.CUSTOMER_ID, GC.Values[GC.CUSTOMER_ID],
                                                                Ent.DOMAIN, GC.Values[GC.DOMAIN],
                                                                Ent.API, api])+[errMsg],
                                     ''))

def ClientAPIAccessDeniedExit(errMsg=None):
  if errMsg is None:
    stderrErrorMsg(Msg.API_ACCESS_DENIED)
    missingScopes = API.getClientScopesSet(GM.Globals[GM.CURRENT_CLIENT_API])-GM.Globals[GM.CURRENT_CLIENT_API_SCOPES]
    if missingScopes:
      writeStderr(Msg.API_CHECK_CLIENT_AUTHORIZATION.format(GM.Globals[GM.OAUTH2_CLIENT_ID],
                                                            ','.join(sorted(missingScopes))))
    systemErrorExit(API_ACCESS_DENIED_RC, None)
  else:
    stderrErrorMsg(errMsg)
    systemErrorExit(API_ACCESS_DENIED_RC, Msg.REAUTHENTICATION_IS_NEEDED)

def SvcAcctAPIAccessDenied():
  _getSvcAcctData()
  if (GM.Globals[GM.CURRENT_SVCACCT_API] == API.GMAIL and
      GM.Globals[GM.CURRENT_SVCACCT_API_SCOPES] and
      GM.Globals[GM.CURRENT_SVCACCT_API_SCOPES][0] == API.GMAIL_SEND_SCOPE):
    systemErrorExit(OAUTH2SERVICE_JSON_REQUIRED_RC, Msg.NO_SVCACCT_ACCESS_ALLOWED)
  stderrErrorMsg(Msg.API_ACCESS_DENIED)
  apiOrScopes = API.getAPIName(GM.Globals[GM.CURRENT_SVCACCT_API]) if GM.Globals[GM.CURRENT_SVCACCT_API] else ','.join(sorted(GM.Globals[GM.CURRENT_SVCACCT_API_SCOPES]))
  writeStderr(Msg.API_CHECK_SVCACCT_AUTHORIZATION.format(GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['client_id'],
                                                         apiOrScopes,
                                                         GM.Globals[GM.CURRENT_SVCACCT_USER] or _getAdminEmail()))
def SvcAcctAPIAccessDeniedExit():
  SvcAcctAPIAccessDenied()
  systemErrorExit(API_ACCESS_DENIED_RC, None)

def SvcAcctAPIDisabledExit():
  if not GM.Globals[GM.CURRENT_SVCACCT_USER] and GM.Globals[GM.CURRENT_CLIENT_API]:
    ClientAPIAccessDeniedExit()
  if GM.Globals[GM.CURRENT_SVCACCT_API]:
    stderrErrorMsg(Msg.SERVICE_ACCOUNT_API_DISABLED.format(API.getAPIName(GM.Globals[GM.CURRENT_SVCACCT_API])))
    systemErrorExit(API_ACCESS_DENIED_RC, None)
  systemErrorExit(API_ACCESS_DENIED_RC, Msg.API_ACCESS_DENIED)

def APIAccessDeniedExit():
  if not GM.Globals[GM.CURRENT_SVCACCT_USER] and GM.Globals[GM.CURRENT_CLIENT_API]:
    ClientAPIAccessDeniedExit()
  if GM.Globals[GM.CURRENT_SVCACCT_API]:
    SvcAcctAPIAccessDeniedExit()
  systemErrorExit(API_ACCESS_DENIED_RC, Msg.API_ACCESS_DENIED)

def checkEntityDNEorAccessErrorExit(cd, entityType, entityName, i=0, count=0):
  message = accessErrorMessage(cd)
  if message:
    systemErrorExit(INVALID_DOMAIN_RC, message)
  entityDoesNotExistWarning(entityType, entityName, i, count)

def checkEntityAFDNEorAccessErrorExit(cd, entityType, entityName, i=0, count=0):
  message = accessErrorMessage(cd)
  if message:
    systemErrorExit(INVALID_DOMAIN_RC, message)
  entityActionFailedWarning([entityType, entityName], Msg.DOES_NOT_EXIST, i, count)

def checkEntityItemValueAFDNEorAccessErrorExit(cd, entityType, entityName, itemType, itemValue, i=0, count=0):
  message = accessErrorMessage(cd)
  if message:
    systemErrorExit(INVALID_DOMAIN_RC, message)
  entityActionFailedWarning([entityType, entityName, itemType, itemValue], Msg.DOES_NOT_EXIST, i, count)

def invalidClientSecretsJsonExit(errMsg):
  stderrErrorMsg(Msg.DOES_NOT_EXIST_OR_HAS_INVALID_FORMAT.format(Ent.Singular(Ent.CLIENT_SECRETS_JSON_FILE), GC.Values[GC.CLIENT_SECRETS_JSON], errMsg))
  writeStderr(Msg.INSTRUCTIONS_CLIENT_SECRETS_JSON)
  systemErrorExit(CLIENT_SECRETS_JSON_REQUIRED_RC, None)

def invalidOauth2serviceJsonExit(errMsg):
  stderrErrorMsg(Msg.DOES_NOT_EXIST_OR_HAS_INVALID_FORMAT.format(Ent.Singular(Ent.OAUTH2SERVICE_JSON_FILE), GC.Values[GC.OAUTH2SERVICE_JSON], errMsg))
  writeStderr(Msg.INSTRUCTIONS_OAUTH2SERVICE_JSON)
  systemErrorExit(OAUTH2SERVICE_JSON_REQUIRED_RC, None)

def invalidOauth2TxtExit(errMsg):
  stderrErrorMsg(Msg.DOES_NOT_EXIST_OR_HAS_INVALID_FORMAT.format(Ent.Singular(Ent.OAUTH2_TXT_FILE), GC.Values[GC.OAUTH2_TXT], errMsg))
  writeStderr(Msg.EXECUTE_GAM_OAUTH_CREATE)
  systemErrorExit(OAUTH2_TXT_REQUIRED_RC, None)

def expiredRevokedOauth2TxtExit():
  stderrErrorMsg(Msg.IS_EXPIRED_OR_REVOKED.format(Ent.Singular(Ent.OAUTH2_TXT_FILE), GC.Values[GC.OAUTH2_TXT]))
  writeStderr(Msg.EXECUTE_GAM_OAUTH_CREATE)
  systemErrorExit(OAUTH2_TXT_REQUIRED_RC, None)

def invalidDiscoveryJsonExit(fileName, errMsg):
  stderrErrorMsg(Msg.DOES_NOT_EXIST_OR_HAS_INVALID_FORMAT.format(Ent.Singular(Ent.DISCOVERY_JSON_FILE), fileName, errMsg))
  systemErrorExit(INVALID_JSON_RC, None)

def entityActionFailedExit(entityValueList, errMsg, i=0, count=0):
  systemErrorExit(ACTION_FAILED_RC, formatKeyValueList(Ind.Spaces(),
                                                       Ent.FormatEntityValueList(entityValueList)+[Act.Failed(), errMsg],
                                                       currentCountNL(i, count)))

def entityDoesNotExistExit(entityType, entityName, i=0, count=0, errMsg=None):
  Cmd.Backup()
  writeStderr(Cmd.CommandLineWithBadArgumentMarked(False))
  systemErrorExit(ENTITY_DOES_NOT_EXIST_RC, formatKeyValueList(Ind.Spaces(),
                                                               [Ent.Singular(entityType), entityName, errMsg or Msg.DOES_NOT_EXIST],
                                                               currentCountNL(i, count)))

def entityDoesNotHaveItemExit(entityValueList, i=0, count=0):
  Cmd.Backup()
  writeStderr(Cmd.CommandLineWithBadArgumentMarked(False))
  systemErrorExit(ENTITY_DOES_NOT_EXIST_RC, formatKeyValueList(Ind.Spaces(),
                                                               Ent.FormatEntityValueList(entityValueList)+[Msg.DOES_NOT_EXIST],
                                                               currentCountNL(i, count)))

def entityIsNotUniqueExit(entityType, entityName, valueType, valueList, i=0, count=0):
  Cmd.Backup()
  writeStderr(Cmd.CommandLineWithBadArgumentMarked(False))
  systemErrorExit(ENTITY_IS_NOT_UNIQUE_RC, formatKeyValueList(Ind.Spaces(),
                                                              [Ent.Singular(entityType), entityName, Msg.IS_NOT_UNIQUE.format(Ent.Plural(valueType), ','.join(valueList))],
                                                              currentCountNL(i, count)))

def usageErrorExit(message, extraneous=False):
  writeStderr(Cmd.CommandLineWithBadArgumentMarked(extraneous))
  stderrErrorMsg(message)
  writeStderr(Msg.HELP_SYNTAX.format(os.path.join(GM.Globals[GM.GAM_PATH], FN_GAMCOMMANDS_TXT)))
  writeStderr(Msg.HELP_WIKI.format(GAM_WIKI))
  sys.exit(USAGE_ERROR_RC)

def csvFieldErrorExit(fieldName, fieldNames, backupArg=False, checkForCharset=False):
  if backupArg:
    Cmd.Backup()
    if checkForCharset and Cmd.Previous() == 'charset':
      Cmd.Backup()
      Cmd.Backup()
  usageErrorExit(Msg.HEADER_NOT_FOUND_IN_CSV_HEADERS.format(fieldName, ','.join(fieldNames)))

def csvDataAlreadySavedErrorExit():
  Cmd.Backup()
  usageErrorExit(Msg.CSV_DATA_ALREADY_SAVED)

# The last thing shown is unknown
def unknownArgumentExit():
  Cmd.Backup()
  usageErrorExit(Cmd.ARGUMENT_ERROR_NAMES[Cmd.ARGUMENT_INVALID][1])

# Argument describes what's expected
def expectedArgumentExit(problem, argument):
  usageErrorExit(f'{problem}: {Msg.EXPECTED} <{argument}>')

def blankArgumentExit(argument):
  expectedArgumentExit(Cmd.ARGUMENT_ERROR_NAMES[Cmd.ARGUMENT_BLANK][1], f'{Msg.NON_BLANK} {argument}')

def emptyArgumentExit(argument):
  expectedArgumentExit(Cmd.ARGUMENT_ERROR_NAMES[Cmd.ARGUMENT_EMPTY][1], f'{Msg.NON_EMPTY} {argument}')

def invalidArgumentExit(argument):
  expectedArgumentExit(Cmd.ARGUMENT_ERROR_NAMES[Cmd.ARGUMENT_INVALID][1], argument)

def missingArgumentExit(argument):
  expectedArgumentExit(Cmd.ARGUMENT_ERROR_NAMES[Cmd.ARGUMENT_MISSING][1], argument)

def deprecatedArgument(argument):
  Cmd.Backup()
  writeStderr(Cmd.CommandLineWithBadArgumentMarked(False))
  Cmd.Advance()
  stderrWarningMsg(f'{Cmd.ARGUMENT_ERROR_NAMES[Cmd.ARGUMENT_DEPRECATED][1]}: {Msg.IGNORED} <{argument}>')

def deprecatedArgumentExit(argument):
  usageErrorExit(f'{Cmd.ARGUMENT_ERROR_NAMES[Cmd.ARGUMENT_DEPRECATED][1]}: <{argument}>')

def deprecatedCommandExit():
  systemErrorExit(USAGE_ERROR_RC, Msg.SITES_COMMAND_DEPRECATED.format(Cmd.CommandDeprecated()))

# Choices is the valid set of choices that was expected
def formatChoiceList(choices):
  choiceList = [c if c else "''" for c in choices]
  if len(choiceList) <= 5:
    return '|'.join(choiceList)
  return '|'.join(sorted(choiceList))

def invalidChoiceExit(choice, choices, backupArg):
  if backupArg:
    Cmd.Backup()
  expectedArgumentExit(Cmd.ARGUMENT_ERROR_NAMES[Cmd.ARGUMENT_INVALID_CHOICE][1].format(choice), formatChoiceList(choices))

def missingChoiceExit(choices):
  expectedArgumentExit(Cmd.ARGUMENT_ERROR_NAMES[Cmd.ARGUMENT_MISSING][1], formatChoiceList(choices))

# Check if argument present
def checkArgumentPresent(choices, required=False):
  choiceList = choices if isinstance(choices, (list, set)) else [choices]
  if Cmd.ArgumentsRemaining():
    choice = Cmd.Current().strip().lower().replace('_', '')
    if choice:
      if choice in choiceList:
        Cmd.Advance()
        return True
    if not required:
      return False
    invalidChoiceExit(choice, choiceList, False)
  elif not required:
    return False
  missingChoiceExit(choiceList)

# Check that there are no extraneous arguments at the end of the command line
def checkForExtraneousArguments():
  if Cmd.ArgumentsRemaining():
    usageErrorExit(Cmd.ARGUMENT_ERROR_NAMES[Cmd.ARGUMENT_EXTRANEOUS][[1, 0][Cmd.MultipleArgumentsRemaining()]], extraneous=True)

# Check that an argument remains, get an argument, downshift, delete underscores
def checkGetArgument():
  if Cmd.ArgumentsRemaining():
    argument = Cmd.Current().lower()
    if argument:
      Cmd.Advance()
      return argument.replace('_', '')
  missingArgumentExit(Cmd.OB_ARGUMENT)

# Get an argument, downshift, delete underscores
def getArgument():
  argument = Cmd.Current().lower()
  if argument:
    Cmd.Advance()
    return argument.replace('_', '')
  missingArgumentExit(Cmd.OB_ARGUMENT)

# Get an argument, downshift, delete underscores
# An empty argument is allowed
def getArgumentEmptyAllowed():
  argument = Cmd.Current().lower()
  Cmd.Advance()
  return argument.replace('_', '')

def getACLRoles(aclRolesMap):
  roles = []
  for role in getString(Cmd.OB_ROLE_LIST, minLen=0).strip().lower().replace(',', ' ').split():
    if role == 'all':
      for arole in aclRolesMap:
        roles.append(aclRolesMap[arole])
    elif role in aclRolesMap:
      roles.append(aclRolesMap[role])
    else:
      invalidChoiceExit(role, aclRolesMap, True)
  return set(roles)

def getBoolean(defaultValue=True):
  if Cmd.ArgumentsRemaining():
    boolean = Cmd.Current().strip().lower()
    if boolean in TRUE_VALUES:
      Cmd.Advance()
      return True
    if boolean in FALSE_VALUES:
      Cmd.Advance()
      return False
    if defaultValue is not None:
      if not Cmd.Current().strip(): # If current argument is empty, skip over it
        Cmd.Advance()
      return defaultValue
    invalidChoiceExit(boolean, TRUE_FALSE, False)
  if defaultValue is not None:
    return defaultValue
  missingChoiceExit(TRUE_FALSE)

def getCharSet():
  if checkArgumentPresent('charset'):
    return getString(Cmd.OB_CHAR_SET)
  return GC.Values[GC.CHARSET]

DEFAULT_CHOICE = 'defaultChoice'
CHOICE_ALIASES = 'choiceAliases'
MAP_CHOICE = 'mapChoice'
NO_DEFAULT = 'NoDefault'

def getChoice(choices, **opts):
  if Cmd.ArgumentsRemaining():
    choice = Cmd.Current().strip().lower()
    if choice or '' in choices:
      if choice in opts.get(CHOICE_ALIASES, []):
        choice = opts[CHOICE_ALIASES][choice]
      if choice not in choices:
        choice = choice.replace('_', '').replace('-', '')
        if choice in opts.get(CHOICE_ALIASES, []):
          choice = opts[CHOICE_ALIASES][choice]
      if choice in choices:
        Cmd.Advance()
        return choice if not opts.get(MAP_CHOICE, False) else choices[choice]
    if opts.get(DEFAULT_CHOICE, NO_DEFAULT) != NO_DEFAULT:
      return opts[DEFAULT_CHOICE]
    invalidChoiceExit(choice, choices, False)
  elif opts.get(DEFAULT_CHOICE, NO_DEFAULT) != NO_DEFAULT:
    return opts[DEFAULT_CHOICE]
  missingChoiceExit(choices)

def getChoiceAndValue(item, choices, delimiter):
  if not Cmd.ArgumentsRemaining() or Cmd.Current().find(delimiter) == -1:
    return (None, None)
  choice, value = Cmd.Current().strip().split(delimiter, 1)
  choice = choice.strip().lower()
  value = value.strip()
  if choice in choices:
    if value:
      Cmd.Advance()
      return (choice, value)
    missingArgumentExit(item)
  invalidChoiceExit(choice, choices, False)

SUSPENDED_ARGUMENTS = {'notsuspended', 'suspended', 'issuspended'}
SUSPENDED_CHOICE_MAP = {'notsuspended': False, 'suspended': True}
def _getIsSuspended(myarg):
  if myarg in SUSPENDED_CHOICE_MAP:
    return SUSPENDED_CHOICE_MAP[myarg]
  return getBoolean()

ARCHIVED_ARGUMENTS = {'notarchived', 'archived', 'isarchived'}
ARCHIVED_CHOICE_MAP = {'notarchived': False, 'archived': True}
def _getIsArchived(myarg):
  if myarg in ARCHIVED_CHOICE_MAP:
    return ARCHIVED_CHOICE_MAP[myarg]
  return getBoolean()

def _getOptionalIsSuspendedIsArchived():
  isSuspended = isArchived = None
  while True:
    if Cmd.PeekArgumentPresent(SUSPENDED_ARGUMENTS):
      isSuspended = getChoice(SUSPENDED_CHOICE_MAP, defaultChoice=None, mapChoice=True)
      if isSuspended is None:
        isSuspended = getBoolean()
    elif Cmd.PeekArgumentPresent(ARCHIVED_ARGUMENTS):
      isArchived = getChoice(ARCHIVED_CHOICE_MAP, defaultChoice=None, mapChoice=True)
      if isArchived is None:
        isArchived = getBoolean()
    else:
      break
  return isSuspended, isArchived

CALENDAR_COLOR_MAP = {
  'amethyst': 24, 'avocado': 10, 'banana': 12, 'basil': 8, 'birch': 20, 'blueberry': 16,
  'cherryblossom': 22, 'citron': 11, 'cobalt': 15, 'cocoa': 1, 'eucalyptus': 7, 'flamingo': 2,
  'grape': 23, 'graphite': 19, 'lavender': 17, 'mango': 6, 'peacock': 14, 'pistachio': 9,
  'pumpkin': 5, 'radicchio': 21, 'sage': 13, 'tangerine': 4, 'tomato': 3, 'wisteria': 18,
  }

CALENDAR_EVENT_COLOR_MAP = {
  'banana': 5, 'basil': 10, 'blueberry': 9, 'flamingo': 4, 'graphite': 8, 'grape': 3,
  'lavender': 1, 'peacock': 7, 'sage': 2, 'tangerine': 6, 'tomato': 11,
  }

GOOGLE_COLOR_MAP = {
  'asparagus': '#7bd148', 'bluevelvet': '#9a9cff', 'bubblegum': '#f691b2', 'cardinal': '#f83a22',
  'chocolateicecream': '#ac725e', 'denim': '#9fc6e7', 'desertsand': '#fbe983', 'earthworm': '#cca6ac',
  'macaroni': '#fad165', 'marsorange': '#ff7537', 'mountaingray': '#cabdbf', 'mountaingrey': '#cabdbf',
  'mouse': '#8f8f8f', 'oldbrickred': '#d06b64', 'pool': '#9fe1e7', 'purpledino': '#b99aff',
  'purplerain': '#cd74e6', 'rainysky': '#4986e7', 'seafoam': '#92e1c0', 'slimegreen': '#b3dc6c',
  'spearmint': '#42d692', 'toyeggplant': '#a47ae2', 'vernfern': '#16a765', 'wildstrawberries': '#fa573c',
  'yellowcab': '#ffad46',
  }

WEB_COLOR_MAP = {
  'aliceblue': '#f0f8ff', 'antiquewhite': '#faebd7', 'aqua': '#00ffff', 'aquamarine': '#7fffd4',
  'azure': '#f0ffff', 'beige': '#f5f5dc', 'bisque': '#ffe4c4', 'black': '#000000',
  'blanchedalmond': '#ffebcd', 'blue': '#0000ff', 'blueviolet': '#8a2be2', 'brown': '#a52a2a',
  'burlywood': '#deb887', 'cadetblue': '#5f9ea0', 'chartreuse': '#7fff00', 'chocolate': '#d2691e',
  'coral': '#ff7f50', 'cornflowerblue': '#6495ed', 'cornsilk': '#fff8dc', 'crimson': '#dc143c',
  'cyan': '#00ffff', 'darkblue': '#00008b', 'darkcyan': '#008b8b', 'darkgoldenrod': '#b8860b',
  'darkgray': '#a9a9a9', 'darkgrey': '#a9a9a9', 'darkgreen': '#006400', 'darkkhaki': '#bdb76b',
  'darkmagenta': '#8b008b', 'darkolivegreen': '#556b2f', 'darkorange': '#ff8c00', 'darkorchid': '#9932cc',
  'darkred': '#8b0000', 'darksalmon': '#e9967a', 'darkseagreen': '#8fbc8f', 'darkslateblue': '#483d8b',
  'darkslategray': '#2f4f4f', 'darkslategrey': '#2f4f4f', 'darkturquoise': '#00ced1', 'darkviolet': '#9400d3',
  'deeppink': '#ff1493', 'deepskyblue': '#00bfff', 'dimgray': '#696969', 'dimgrey': '#696969',
  'dodgerblue': '#1e90ff', 'firebrick': '#b22222', 'floralwhite': '#fffaf0', 'forestgreen': '#228b22',
  'fuchsia': '#ff00ff', 'gainsboro': '#dcdcdc', 'ghostwhite': '#f8f8ff', 'gold': '#ffd700',
  'goldenrod': '#daa520', 'gray': '#808080', 'grey': '#808080', 'green': '#008000',
  'greenyellow': '#adff2f', 'honeydew': '#f0fff0', 'hotpink': '#ff69b4', 'indianred': '#cd5c5c',
  'indigo': '#4b0082', 'ivory': '#fffff0', 'khaki': '#f0e68c', 'lavender': '#e6e6fa',
  'lavenderblush': '#fff0f5', 'lawngreen': '#7cfc00', 'lemonchiffon': '#fffacd', 'lightblue': '#add8e6',
  'lightcoral': '#f08080', 'lightcyan': '#e0ffff', 'lightgoldenrodyellow': '#fafad2', 'lightgray': '#d3d3d3',
  'lightgrey': '#d3d3d3', 'lightgreen': '#90ee90', 'lightpink': '#ffb6c1', 'lightsalmon': '#ffa07a',
  'lightseagreen': '#20b2aa', 'lightskyblue': '#87cefa', 'lightslategray': '#778899', 'lightslategrey': '#778899',
  'lightsteelblue': '#b0c4de', 'lightyellow': '#ffffe0', 'lime': '#00ff00', 'limegreen': '#32cd32',
  'linen': '#faf0e6', 'magenta': '#ff00ff', 'maroon': '#800000', 'mediumaquamarine': '#66cdaa',
  'mediumblue': '#0000cd', 'mediumorchid': '#ba55d3', 'mediumpurple': '#9370db', 'mediumseagreen': '#3cb371',
  'mediumslateblue': '#7b68ee', 'mediumspringgreen': '#00fa9a', 'mediumturquoise': '#48d1cc', 'mediumvioletred': '#c71585',
  'midnightblue': '#191970', 'mintcream': '#f5fffa', 'mistyrose': '#ffe4e1', 'moccasin': '#ffe4b5',
  'navajowhite': '#ffdead', 'navy': '#000080', 'oldlace': '#fdf5e6', 'olive': '#808000',
  'olivedrab': '#6b8e23', 'orange': '#ffa500', 'orangered': '#ff4500', 'orchid': '#da70d6',
  'palegoldenrod': '#eee8aa', 'palegreen': '#98fb98', 'paleturquoise': '#afeeee', 'palevioletred': '#db7093',
  'papayawhip': '#ffefd5', 'peachpuff': '#ffdab9', 'peru': '#cd853f', 'pink': '#ffc0cb',
  'plum': '#dda0dd', 'powderblue': '#b0e0e6', 'purple': '#800080', 'red': '#ff0000',
  'rosybrown': '#bc8f8f', 'royalblue': '#4169e1', 'saddlebrown': '#8b4513', 'salmon': '#fa8072',
  'sandybrown': '#f4a460', 'seagreen': '#2e8b57', 'seashell': '#fff5ee', 'sienna': '#a0522d',
  'silver': '#c0c0c0', 'skyblue': '#87ceeb', 'slateblue': '#6a5acd', 'slategray': '#708090',
  'slategrey': '#708090', 'snow': '#fffafa', 'springgreen': '#00ff7f', 'steelblue': '#4682b4',
  'tan': '#d2b48c', 'teal': '#008080', 'thistle': '#d8bfd8', 'tomato': '#ff6347',
  'turquoise': '#40e0d0', 'violet': '#ee82ee', 'wheat': '#f5deb3', 'white': '#ffffff',
  'whitesmoke': '#f5f5f5', 'yellow': '#ffff00', 'yellowgreen': '#9acd32',
  }

COLORHEX_PATTERN = re.compile(r'^#[0-9a-fA-F]{6}$')
COLORHEX_FORMAT_REQUIRED = 'ColorName|ColorHex'

def getColor():
  if Cmd.ArgumentsRemaining():
    color = Cmd.Current().strip().lower()
    if color in GOOGLE_COLOR_MAP:
      Cmd.Advance()
      return GOOGLE_COLOR_MAP[color]
    if color in WEB_COLOR_MAP:
      Cmd.Advance()
      return WEB_COLOR_MAP[color]
    tg = COLORHEX_PATTERN.match(color)
    if tg:
      Cmd.Advance()
      return tg.group(0)
    invalidArgumentExit(COLORHEX_FORMAT_REQUIRED)
  missingArgumentExit(COLORHEX_FORMAT_REQUIRED)

LABEL_COLORS = [
  '#000000', '#076239', '#0b804b', '#149e60', '#16a766', '#1a764d', '#1c4587', '#285bac',
  '#2a9c68', '#3c78d8', '#3dc789', '#41236d', '#434343', '#43d692', '#44b984', '#4a86e8',
  '#653e9b', '#666666', '#68dfa9', '#6d9eeb', '#822111', '#83334c', '#89d3b2', '#8e63ce',
  '#999999', '#a0eac9', '#a46a21', '#a479e2', '#a4c2f4', '#aa8831', '#ac2b16', '#b65775',
  '#b694e8', '#b9e4d0', '#c6f3de', '#c9daf8', '#cc3a21', '#cccccc', '#cf8933', '#d0bcf1',
  '#d5ae49', '#e07798', '#e4d7f5', '#e66550', '#eaa041', '#efa093', '#efefef', '#f2c960',
  '#f3f3f3', '#f691b3', '#f6c5be', '#f7a7c0', '#fad165', '#fb4c2f', '#fbc8d9', '#fcda83',
  '#fcdee8', '#fce8b3', '#fef1d1', '#ffad47', '#ffbc6b', '#ffd6a2', '#ffe6c7', '#ffffff',
  ]
LABEL_BACKGROUND_COLORS = [
  '#16a765', '#2da2bb', '#42d692', '#4986e7', '#98d7e4', '#a2dcc1',
  '#b3efd3', '#b6cff5', '#b99aff', '#c2c2c2', '#cca6ac', '#e3d7ff',
  '#e7e7e7', '#ebdbde', '#f2b2a8', '#f691b2', '#fb4c2f', '#fbd3e0',
  '#fbe983', '#fdedc1', '#ff7537', '#ffad46', '#ffc8af', '#ffdeb5',
  ]
LABEL_TEXT_COLORS = [
  '#04502e', '#094228', '#0b4f30', '#0d3472', '#0d3b44', '#3d188e',
  '#464646', '#594c05', '#662e37', '#684e07', '#711a36', '#7a2e0b',
  '#7a4706', '#8a1c0a', '#994a64', '#ffffff',
  ]

def getLabelColor(colorType):
  if Cmd.ArgumentsRemaining():
    color = Cmd.Current().strip().lower()
    tg = COLORHEX_PATTERN.match(color)
    if tg:
      color = tg.group(0)
      if color in colorType or color in LABEL_COLORS:
        Cmd.Advance()
        return color
    elif color.startswith('custom:'):
      tg = COLORHEX_PATTERN.match(color[7:])
      if tg:
        Cmd.Advance()
        return tg.group(0)
    invalidArgumentExit('|'.join(colorType))
  missingArgumentExit(Cmd.OB_LABEL_COLOR_HEX)

# Language codes used in Drive Labels/Youtube
BCP47_LANGUAGE_CODES_MAP = {
  'ar-sa': 'ar-SA', 'cs-cz': 'cs-CZ', 'da-dk': 'da-DK', 'de-de': 'de-DE', #Arabic Saudi Arabia, Czech Czech Republic, Danish Denmark, German Germany
  'el-gr': 'el-GR', 'en-au': 'en-AU', 'en-gb': 'en-GB', 'en-ie': 'en-IE', #Modern Greek Greece, English Australia, English United Kingdom, English Ireland
  'en-us': 'en-US', 'en-za': 'en-ZA', 'es-es': 'es-ES', 'es-mx': 'es-MX', #English United States, English South Africa, Spanish Spain, Spanish Mexico
  'fi-fi': 'fi-FI', 'fr-ca': 'fr-CA', 'fr-fr': 'fr-FR', 'he-il': 'he-IL', #Finnish Finland, French Canada, French France, Hebrew Israel
  'hi-in': 'hi-IN', 'hu-hu': 'hu-HU', 'id-id': 'id-ID', 'it-it': 'it-IT', #Hindi India, Hungarian Hungary, Indonesian Indonesia, Italian Italy
  'ja-jp': 'ja-JP', 'ko-kr': 'ko-KR', 'nl-be': 'nl-BE', 'nl-nl': 'nl-NL', #Japanese Japan, Korean Republic of Korea, Dutch Belgium, Dutch Netherlands
  'no-no': 'no-NO', 'pl-pl': 'pl-PL', 'pt-br': 'pt-BR', 'pt-pt': 'pt-PT', #Norwegian Norway, Polish Poland, Portuguese Brazil, Portuguese Portugal
  'ro-ro': 'ro-RO', 'ru-ru': 'ru-RU', 'sk-sk': 'sk-SK', 'sv-se': 'sv-SE', #Romanian Romania, Russian Russian Federation, Slovak Slovakia, Swedish Sweden
  'th-th': 'th-TH', 'tr-tr': 'tr-TR', 'zh-cn': 'zh-CN', 'zh-hk': 'zh-HK', #Thai Thailand, Turkish Turkey, Chinese China, Chinese Hong Kong
  'zh-tw': 'zh-TW' #Chinese Taiwan
  }

# Valid language codes
LANGUAGE_CODES_MAP = {
  'ach': 'ach', 'af': 'af', 'ag': 'ga', 'ak': 'ak', 'am': 'am', 'ar': 'ar', 'az': 'az', #Luo, Afrikaans, Irish, Akan, Amharic, Arabica, Azerbaijani
  'be': 'be', 'bem': 'bem', 'bg': 'bg', 'bn': 'bn', 'br': 'br', 'bs': 'bs', 'ca': 'ca', #Belarusian, Bemba, Bulgarian, Bengali, Breton, Bosnian, Catalan
  'chr': 'chr', 'ckb': 'ckb', 'co': 'co', 'crs': 'crs', 'cs': 'cs', 'cy': 'cy', 'da': 'da', #Cherokee, Kurdish (Sorani), Corsican, Seychellois Creole, Czech, Welsh, Danish
  'de': 'de', 'ee': 'ee', 'el': 'el', 'en': 'en', 'en-ca': 'en-CA', 'en-gb': 'en-GB', 'en-us': 'en-US', 'eo': 'eo', #German, Ewe, Greek, English, English (CA), English (UK), English (US), Esperanto
  'es': 'es', 'es-419': 'es-419', 'et': 'et', 'eu': 'eu', 'fa': 'fa', 'fi': 'fi', 'fil': 'fil', 'fo': 'fo', #Spanish, Spanish (Latin American), Estonian, Basque, Persian, Finnish, Filipino, Faroese
  'fr': 'fr', 'fr-ca': 'fr-CA', 'fy': 'fy', 'ga': 'ga', 'gaa': 'gaa', 'gd': 'gd', 'gl': 'gl', #French, French (Canada), Frisian, Irish, Ga, Scots Gaelic, Galician
  'gn': 'gn', 'gu': 'gu', 'ha': 'ha', 'haw': 'haw', 'he': 'he', 'hi': 'hi', 'hr': 'hr', #Guarani, Gujarati, Hausa, Hawaiian, Hebrew, Hindi, Croatian
  'ht': 'ht', 'hu': 'hu', 'hy': 'hy', 'ia': 'ia', 'id': 'id', 'ig': 'ig', 'in': 'in', #Haitian Creole, Hungarian, Armenian, Interlingua, Indonesian, Igbo, in
  'is': 'is', 'it': 'it', 'iw': 'iw', 'ja': 'ja', 'jw': 'jw', 'ka': 'ka', 'kg': 'kg', #Icelandic, Italian, Hebrew, Japanese, Javanese, Georgian, Kongo
  'kk': 'kk', 'km': 'km', 'kn': 'kn', 'ko': 'ko', 'kri': 'kri', 'k': 'k', 'ky': 'ky', #Kazakh, Khmer, Kannada, Korean, Krio (Sierra Leone), Kurdish, Kyrgyz
  'la': 'la', 'lg': 'lg', 'ln': 'ln', 'lo': 'lo', 'loz': 'loz', 'lt': 'lt', 'lua': 'lua', #Latin, Luganda, Lingala, Laothian, Lozi, Lithuanian, Tshiluba
  'lv': 'lv', 'mfe': 'mfe', 'mg': 'mg', 'mi': 'mi', 'mk': 'mk', 'ml': 'ml', 'mn': 'mn', #Latvian, Mauritian Creole, Malagasy, Maori, Macedonian, Malayalam, Mongolian
  'mo': 'mo', 'mr': 'mr', 'ms': 'ms', 'mt': 'mt', 'my': 'my', 'ne': 'ne', 'nl': 'nl', #Moldavian, Marathi, Malay, Maltese, Burmese, Nepali, Dutch
  'nn': 'nn', 'no': 'no', 'nso': 'nso', 'ny': 'ny', 'nyn': 'nyn', 'oc': 'oc', 'om': 'om', #Norwegian (Nynorsk), Norwegian, Northern Sotho, Chichewa, Runyakitara, Occitan, Oromo
  'or': 'or', 'pa': 'pa', 'pcm': 'pcm', 'pl': 'pl', 'ps': 'ps', 'pt-br': 'pt-BR', 'pt-pt': 'pt-PT', #Oriya, Punjabi, Nigerian Pidgin, Polish, Pashto, Portuguese (Brazil), Portuguese (Portugal)
  'q': 'q', 'rm': 'rm', 'rn': 'rn', 'ro': 'ro', 'ru': 'ru', 'rw': 'rw', 'sd': 'sd', #Quechua, Romansh, Kirundi, Romanian, Russian, Kinyarwanda, Sindhi
  'sh': 'sh', 'si': 'si', 'sk': 'sk', 'sl': 'sl', 'sn': 'sn', 'so': 'so', 'sq': 'sq', #Serbo-Croatian, Sinhalese, Slovak, Slovenian, Shona, Somali, Albanian
  'sr': 'sr', 'sr-me': 'sr-ME', 'st': 'st', 'su': 'su', 'sv': 'sv', 'sw': 'sw', 'ta': 'ta', #Serbian, Montenegrin, Sesotho, Sundanese, Swedish, Swahili, Tamil
  'te': 'te', 'tg': 'tg', 'th': 'th', 'ti': 'ti', 'tk': 'tk', 'tl': 'tl', 'tn': 'tn', #Telugu, Tajik, Thai, Tigrinya, Turkmen, Tagalog, Setswana
  'to': 'to', 'tr': 'tr', 'tt': 'tt', 'tum': 'tum', 'tw': 'tw', 'ug': 'ug', 'uk': 'uk', #Tonga, Turkish, Tatar, Tumbuka, Twi, Uighur, Ukrainian
  'ur': 'ur', 'uz': 'uz', 'vi': 'vi', 'wo': 'wo', 'xh': 'xh', 'yi': 'yi', 'yo': 'yo', #Urdu, Uzbek, Vietnamese, Wolof, Xhosa, Yiddish, Yoruba
  'zh-cn': 'zh-CN', 'zh-hk': 'zh-HK', 'zh-tw': 'zh-TW', 'zu': 'zu', #Chinese (Simplified), Chinese (Hong Kong/Traditional), Chinese (Taiwan/Traditional), Zulu
  }

LOCALE_CODES_MAP = {
  '': '',
  'ar-eg': 'ar_EG', #Arabic, Egypt
  'az-az': 'az_AZ', #Azerbaijani, Azerbaijan
  'be-by': 'be_BY', #Belarusian, Belarus
  'bg-bg': 'bg_BG', #Bulgarian, Bulgaria
  'bn-in': 'bn_IN', #Bengali, India
  'ca-es': 'ca_ES', #Catalan, Spain
  'cs-cz': 'cs_CZ', #Czech, Czech Republic
  'cy-gb': 'cy_GB', #Welsh, United Kingdom
  'da-dk': 'da_DK', #Danish, Denmark
  'de-ch': 'de_CH', #German, Switzerland
  'de-de': 'de_DE', #German, Germany
  'el-gr': 'el_GR', #Greek, Greece
  'en-au': 'en_AU', #English, Australia
  'en-ca': 'en_CA', #English, Canada
  'en-gb': 'en_GB', #English, United Kingdom
  'en-ie': 'en_IE', #English, Ireland
  'en-us': 'en_US', #English, U.S.A.
  'es-ar': 'es_AR', #Spanish, Argentina
  'es-bo': 'es_BO', #Spanish, Bolivia
  'es-cl': 'es_CL', #Spanish, Chile
  'es-co': 'es_CO', #Spanish, Colombia
  'es-ec': 'es_EC', #Spanish, Ecuador
  'es-es': 'es_ES', #Spanish, Spain
  'es-mx': 'es_MX', #Spanish, Mexico
  'es-py': 'es_PY', #Spanish, Paraguay
  'es-uy': 'es_UY', #Spanish, Uruguay
  'es-ve': 'es_VE', #Spanish, Venezuela
  'fi-fi': 'fi_FI', #Finnish, Finland
  'fil-ph': 'fil_PH', #Filipino, Philippines
  'fr-ca': 'fr_CA', #French, Canada
  'fr-fr': 'fr_FR', #French, France
  'gu-in': 'gu_IN', #Gujarati, India
  'hi-in': 'hi_IN', #Hindi, India
  'hr-hr': 'hr_HR', #Croatian, Croatia
  'hu-hu': 'hu_HU', #Hungarian, Hungary
  'hy-am': 'hy_AM', #Armenian, Armenia
  'in-id': 'in_ID', #Indonesian, Indonesia
  'it-it': 'it_IT', #Italian, Italy
  'iw-il': 'iw_IL', #Hebrew, Israel
  'ja-jp': 'ja_JP', #Japanese, Japan
  'ka-ge': 'ka_GE', #Georgian, Georgia
  'kk-kz': 'kk_KZ', #Kazakh, Kazakhstan
  'kn-in': 'kn_IN', #Kannada, India
  'ko-kr': 'ko_KR', #Korean, Korea
  'lt-lt': 'lt_LT', #Lithuanian, Lithuania
  'lv-lv': 'lv_LV', #Latvian, Latvia
  'ml-in': 'ml_IN', #Malayalam, India
  'mn-mn': 'mn_MN', #Mongolian, Mongolia
  'mr-in': 'mr_IN', #Marathi, India
  'my-mn': 'my_MN', #Burmese, Myanmar
  'nl-nl': 'nl_NL', #Dutch, Netherlands
  'nn-no': 'nn_NO', #Nynorsk, Norway
  'no-no': 'no_NO', #Bokmal, Norway
  'pa-in': 'pa_IN', #Punjabi, India
  'pl-pl': 'pl_PL', #Polish, Poland
  'pt-br': 'pt_BR', #Portuguese, Brazil
  'pt-pt': 'pt_PT', #Portuguese, Portugal
  'ro-ro': 'ro_RO', #Romanian, Romania
  'ru-ru': 'ru_RU', #Russian, Russia
  'sk-sk': 'sk_SK', #Slovak, Slovakia
  'sl-si': 'sl_SI', #Slovenian, Slovenia
  'sr-rs': 'sr_RS', #Serbian, Serbia
  'sv-se': 'sv_SE', #Swedish, Sweden
  'ta-in': 'ta_IN', #Tamil, India
  'te-in': 'te_IN', #Telugu, India
  'th-th': 'th_TH', #Thai, Thailand
  'tr-tr': 'tr_TR', #Turkish, Turkey
  'uk-ua': 'uk_UA', #Ukrainian, Ukraine
  'vi-vn': 'vi_VN', #Vietnamese, Vietnam
  'zh-cn': 'zh_CN', #Simplified Chinese, China
  'zh-hk': 'zh_HK', #Traditional Chinese, Hong Kong SAR China
  'zh-tw': 'zh_TW', #Traditional Chinese, Taiwan
  }

def getLanguageCode(languageCodeMap):
  if Cmd.ArgumentsRemaining():
    choice = Cmd.Current().strip().lower().replace('_', '-')
    if choice in languageCodeMap:
      Cmd.Advance()
      return languageCodeMap[choice]
    invalidChoiceExit(choice, languageCodeMap, False)
  missingChoiceExit(languageCodeMap)

def addCourseIdScope(courseId):
  if not courseId.isdigit() and courseId[:2] not in {'d:', 'p:'}:
    return f'd:{courseId}'
  return courseId

def removeCourseIdScope(courseId):
  if courseId.startswith('d:'):
    return courseId[2:]
  return courseId

def addCourseAliasScope(alias):
  if alias[:2] not in {'d:', 'p:'}:
    return f'd:{alias}'
  return alias

def removeCourseAliasScope(alias):
  if alias.startswith('d:'):
    return alias[2:]
  return alias

def getCourseAlias():
  if Cmd.ArgumentsRemaining():
    courseAlias = Cmd.Current()
    if courseAlias:
      Cmd.Advance()
      return addCourseAliasScope(courseAlias)
  missingArgumentExit(Cmd.OB_COURSE_ALIAS)

DELIVERY_SETTINGS_UNDEFINED = 'DSU'
GROUP_DELIVERY_SETTINGS_MAP = {
  'allmail': 'ALL_MAIL',
  'abridged': 'DAILY',
  'daily': 'DAILY',
  'digest': 'DIGEST',
  'disabled': 'DISABLED',
  'none': 'NONE',
  'nomail': 'NONE',
  }

def getDeliverySettings():
  if checkArgumentPresent(['delivery', 'deliverysettings']):
    return getChoice(GROUP_DELIVERY_SETTINGS_MAP, mapChoice=True)
  return getChoice(GROUP_DELIVERY_SETTINGS_MAP, defaultChoice=DELIVERY_SETTINGS_UNDEFINED, mapChoice=True)

UID_PATTERN = re.compile(r'u?id: ?(.+)', re.IGNORECASE)
PEOPLE_PATTERN = re.compile(r'people/([0-9]+)$', re.IGNORECASE)

def validateEmailAddressOrUID(emailAddressOrUID, checkPeople=True, ciGroupsAPI=False):
  cg = UID_PATTERN.match(emailAddressOrUID)
  if cg:
    return cg.group(1)
  if checkPeople:
    cg = PEOPLE_PATTERN.match(emailAddressOrUID)
    if cg:
      return cg.group(1)
  if ciGroupsAPI and emailAddressOrUID.startswith('groups/'):
    return emailAddressOrUID
  return emailAddressOrUID.find('@') != 0 and emailAddressOrUID.count('@') <= 1

# Normalize user/group email address/uid
# uid:12345abc -> 12345abc
# foo -> foo@domain
# foo@ -> foo@domain
# foo@bar.com -> foo@bar.com
# @domain -> domain
def normalizeEmailAddressOrUID(emailAddressOrUID, noUid=False, checkForCustomerId=False, noLower=False, ciGroupsAPI=False):
  if checkForCustomerId and (emailAddressOrUID == GC.Values[GC.CUSTOMER_ID]):
    return emailAddressOrUID
  if not noUid:
    cg = UID_PATTERN.match(emailAddressOrUID)
    if cg:
      return cg.group(1)
    cg = PEOPLE_PATTERN.match(emailAddressOrUID)
    if cg:
      return cg.group(1)
  if ciGroupsAPI and emailAddressOrUID.startswith('groups/'):
    return emailAddressOrUID
  atLoc = emailAddressOrUID.find('@')
  if atLoc == 0:
    return emailAddressOrUID[1:].lower() if not noLower else emailAddressOrUID[1:]
  if (atLoc == -1) or (atLoc == len(emailAddressOrUID)-1) and GC.Values[GC.DOMAIN]:
    if atLoc == -1:
      emailAddressOrUID = f'{emailAddressOrUID}@{GC.Values[GC.DOMAIN]}'
    else:
      emailAddressOrUID = f'{emailAddressOrUID}{GC.Values[GC.DOMAIN]}'
  return emailAddressOrUID.lower() if not noLower else emailAddressOrUID

# Normalize student/guardian email address/uid
# 12345678 -> 12345678
# - -> -
# Otherwise, same results as normalizeEmailAddressOrUID
def normalizeStudentGuardianEmailAddressOrUID(emailAddressOrUID, allowDash=False):
  if emailAddressOrUID.isdigit() or (allowDash and emailAddressOrUID == '-'):
    return emailAddressOrUID
  return normalizeEmailAddressOrUID(emailAddressOrUID)

def getEmailAddress(noUid=False, minLen=1, optional=False, returnUIDprefix=''):
  if Cmd.ArgumentsRemaining():
    emailAddress = Cmd.Current().strip().lower()
    if emailAddress:
      cg = UID_PATTERN.match(emailAddress)
      if cg:
        if not noUid:
          if cg.group(1):
            Cmd.Advance()
            return f'{returnUIDprefix}{cg.group(1)}'
        else:
          invalidArgumentExit('name@domain')
      else:
        atLoc = emailAddress.find('@')
        if atLoc == -1:
          if GC.Values[GC.DOMAIN]:
            emailAddress = f'{emailAddress}@{GC.Values[GC.DOMAIN]}'
          Cmd.Advance()
          return emailAddress
        if atLoc != 0:
          if (atLoc == len(emailAddress)-1) and GC.Values[GC.DOMAIN]:
            emailAddress = f'{emailAddress}{GC.Values[GC.DOMAIN]}'
          Cmd.Advance()
          return emailAddress
        invalidArgumentExit('name@domain')
    if optional:
      Cmd.Advance()
      return None
    if minLen == 0:
      Cmd.Advance()
      return ''
  elif optional:
    return None
  missingArgumentExit([Cmd.OB_EMAIL_ADDRESS_OR_UID, Cmd.OB_EMAIL_ADDRESS][noUid])

def getFilename():
  filename = os.path.expanduser(getString(Cmd.OB_FILE_NAME))
  if os.path.isfile(filename):
    return filename
  entityDoesNotExistExit(Ent.FILE, filename)

def getPermissionId():
  if Cmd.ArgumentsRemaining():
    emailAddress = Cmd.Current().strip()
    if emailAddress:
      cg = UID_PATTERN.match(emailAddress)
      if cg:
        Cmd.Advance()
        return (False, cg.group(1))
      emailAddress = emailAddress.lower()
      atLoc = emailAddress.find('@')
      if atLoc == -1:
        if emailAddress == 'anyone':
          Cmd.Advance()
          return (False, emailAddress)
        if emailAddress == 'anyonewithlink':
          Cmd.Advance()
          return (False, 'anyoneWithLink')
        if GC.Values[GC.DOMAIN]:
          emailAddress = f'{emailAddress}@{GC.Values[GC.DOMAIN]}'
        Cmd.Advance()
        return (True, emailAddress)
      if atLoc != 0:
        if (atLoc == len(emailAddress)-1) and GC.Values[GC.DOMAIN]:
          emailAddress = f'{emailAddress}{GC.Values[GC.DOMAIN]}'
        Cmd.Advance()
        return (True, emailAddress)
      invalidArgumentExit('name@domain')
  missingArgumentExit(Cmd.OB_DRIVE_FILE_PERMISSION_ID)

def getGoogleProduct():
  if Cmd.ArgumentsRemaining():
    product = Cmd.Current().strip()
    if product:
      status, productId = SKU.normalizeProductId(product)
      if not status:
        invalidChoiceExit(productId, SKU.getSortedProductList(), False)
      Cmd.Advance()
      return productId
  missingArgumentExit(Cmd.OB_PRODUCT_ID)

def getGoogleProductList():
  if Cmd.ArgumentsRemaining():
    productsList = []
    for product in Cmd.Current().split(','):
      status, productId = SKU.normalizeProductId(product)
      if not status:
        invalidChoiceExit(productId, SKU.getSortedProductList(), False)
      if productId not in productsList:
        productsList.append(productId)
    Cmd.Advance()
    return productsList
  missingArgumentExit(Cmd.OB_PRODUCT_ID_LIST)

def getGoogleSKU():
  if Cmd.ArgumentsRemaining():
    sku = Cmd.Current().strip()
    if sku:
      Cmd.Advance()
      return SKU.getProductAndSKU(sku)
  missingArgumentExit(Cmd.OB_SKU_ID)

def getGoogleSKUList(allowUnknownProduct=False):
  if Cmd.ArgumentsRemaining():
    skusList = []
    for sku in Cmd.Current().split(','):
      productId, sku = SKU.getProductAndSKU(sku)
      if not productId and not allowUnknownProduct:
        invalidChoiceExit(sku, SKU.getSortedSKUList(), False)
      if (productId, sku) not in skusList:
        skusList.append((productId, sku))
    Cmd.Advance()
    return skusList
  missingArgumentExit(Cmd.OB_SKU_ID_LIST)

def floatLimits(minVal, maxVal, item='float'):
  if (minVal is not None) and (maxVal is not None):
    return f'{item} {minVal:.3f}<=x<={maxVal:.3f}'
  if minVal is not None:
    return f'{item} x>={minVal:.3f}'
  if maxVal is not None:
    return f'{item} x<={maxVal:.3f}'
  return f'{item} x'

def getFloat(minVal=None, maxVal=None):
  if Cmd.ArgumentsRemaining():
    try:
      number = float(Cmd.Current().strip())
      if ((minVal is None) or (number >= minVal)) and ((maxVal is None) or (number <= maxVal)):
        Cmd.Advance()
        return number
    except ValueError:
      pass
    invalidArgumentExit(floatLimits(minVal, maxVal))
  missingArgumentExit(floatLimits(minVal, maxVal))

def integerLimits(minVal, maxVal, item='integer'):
  if (minVal is not None) and (maxVal is not None):
    return f'{item} {minVal}<=x<={maxVal}'
  if minVal is not None:
    return f'{item} x>={minVal}'
  if maxVal is not None:
    return f'{item} x<={maxVal}'
  return f'{item} x'

def getInteger(minVal=None, maxVal=None, default=None):
  if Cmd.ArgumentsRemaining():
    try:
      number = int(Cmd.Current().strip())
      if ((minVal is None) or (number >= minVal)) and ((maxVal is None) or (number <= maxVal)):
        Cmd.Advance()
        return number
    except ValueError:
      if default is not None:
        if not Cmd.Current().strip(): # If current argument is empty, skip over it
          Cmd.Advance()
        return default
    invalidArgumentExit(integerLimits(minVal, maxVal))
  elif default is not None:
    return default
  missingArgumentExit(integerLimits(minVal, maxVal))

def getIntegerEmptyAllowed(minVal=None, maxVal=None, default=0):
  if Cmd.ArgumentsRemaining():
    number = Cmd.Current().strip()
    if not number:
      Cmd.Advance()
      return default
    try:
      number = int(number)
      if ((minVal is None) or (number >= minVal)) and ((maxVal is None) or (number <= maxVal)):
        Cmd.Advance()
        return number
    except ValueError:
      pass
    invalidArgumentExit(integerLimits(minVal, maxVal))
  return default

SORTORDER_CHOICE_MAP = {'ascending': 'ASCENDING', 'descending': 'DESCENDING'}

class OrderBy():
  def __init__(self, choiceMap, ascendingKeyword='', descendingKeyword='desc'):
    self.choiceMap = choiceMap
    self.ascendingKeyword = ascendingKeyword
    self.descendingKeyword = descendingKeyword
    self.items = []

  def GetChoice(self):
    fieldName = getChoice(self.choiceMap, mapChoice=True)
    fieldNameAscending = fieldName
    if self.ascendingKeyword:
      fieldNameAscending += f' {self.ascendingKeyword}'
    if fieldNameAscending in self.items:
      self.items.remove(fieldNameAscending)
    fieldNameDescending = fieldName
    if self.descendingKeyword:
      fieldNameDescending += f' {self.descendingKeyword}'
    if fieldNameDescending in self.items:
      self.items.remove(fieldNameDescending)
    if getChoice(SORTORDER_CHOICE_MAP, defaultChoice=None, mapChoice=True) != 'DESCENDING':
      self.items.append(fieldNameAscending)
    else:
      self.items.append(fieldNameDescending)

  def SetItems(self, itemList):
    self.items = itemList.split(',')

  @property
  def orderBy(self):
    return ','.join(self.items)

def getOrderBySortOrder(choiceMap, defaultSortOrderChoice='ASCENDING', mapSortOrderChoice=True):
  return (getChoice(choiceMap, mapChoice=True),
          getChoice(SORTORDER_CHOICE_MAP, defaultChoice=defaultSortOrderChoice, mapChoice=mapSortOrderChoice))

def orgUnitPathQuery(path, isSuspended):
  query = "orgUnitPath='{0}'".format(path.replace("'", "\\'")) if path != '/' else ''
  if isSuspended is not None:
    query += f' isSuspended={isSuspended}'
  return query

def makeOrgUnitPathAbsolute(path):
  if path == '/':
    return path
  if path.startswith('/'):
    if not path.endswith('/'):
      return path
    return path[:-1]
  if path.startswith('id:'):
    return path
  if path.startswith('uid:'):
    return path[1:]
  if not path.endswith('/'):
    return '/'+path
  return '/'+path[:-1]

def makeOrgUnitPathRelative(path):
  if path == '/':
    return path
  if path.startswith('/'):
    if not path.endswith('/'):
      return path[1:]
    return path[1:-1]
  if path.startswith('id:'):
    return path
  if path.startswith('uid:'):
    return path[1:]
  if not path.endswith('/'):
    return path
  return path[:-1]

def encodeOrgUnitPath(path):
# 6.22.19 - Encoding doesn't work
# % no longer needs encoding and + is handled incorrectly in API with or without encoding
  return path
#  if path.find('+') == -1 and path.find('%') == -1:
#    return path
#  encpath = ''
#  for c in path:
#    if c == '+':
#      encpath += '%2B'
#    elif c == '%':
#      encpath += '%25'
#    else:
#      encpath += c
#  return encpath

def getOrgUnitItem(pathOnly=False, absolutePath=True, cd=None):
  if Cmd.ArgumentsRemaining():
    path = Cmd.Current().strip()
    if path:
      if pathOnly and (path.startswith('id:') or path.startswith('uid:')) and cd is not None:
        try:
          result = callGAPI(cd.orgunits(), 'get',
                            throwReasons=GAPI.ORGUNIT_GET_THROW_REASONS,
                            customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath=path,
                            fields='orgUnitPath')
          Cmd.Advance()
          if absolutePath:
            return makeOrgUnitPathAbsolute(result['orgUnitPath'])
          return makeOrgUnitPathRelative(result['orgUnitPath'])
        except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError,
                GAPI.badRequest, GAPI.invalidCustomerId, GAPI.loginRequired):
          checkEntityAFDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, path)
        invalidArgumentExit(Cmd.OB_ORGUNIT_PATH)
      Cmd.Advance()
      if absolutePath:
        return makeOrgUnitPathAbsolute(path)
      return makeOrgUnitPathRelative(path)
  missingArgumentExit([Cmd.OB_ORGUNIT_ITEM, Cmd.OB_ORGUNIT_PATH][pathOnly])

def getTopLevelOrgId(cd, parentOrgUnitPath):
  if parentOrgUnitPath != '/':
    try:
      result = callGAPI(cd.orgunits(), 'get',
                        throwReasons=GAPI.ORGUNIT_GET_THROW_REASONS,
                        customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath=encodeOrgUnitPath(makeOrgUnitPathRelative(parentOrgUnitPath)),
                        fields='orgUnitId')
      return result['orgUnitId']
    except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError):
      return None
    except (GAPI.badRequest, GAPI.invalidCustomerId, GAPI.loginRequired):
      checkEntityAFDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, parentOrgUnitPath)
      return None
  try:
    result = callGAPI(cd.orgunits(), 'list',
                      throwReasons=GAPI.ORGUNIT_GET_THROW_REASONS,
                      customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath='/', type='allIncludingParent',
                      fields='organizationUnits(orgUnitId,orgUnitPath)')
    for orgUnit in result.get('organizationUnits', []):
      if orgUnit['orgUnitPath'] == '/':
        return orgUnit['orgUnitId']
    return None
  except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError):
    return None
  except (GAPI.badRequest, GAPI.invalidCustomerId, GAPI.loginRequired):
    checkEntityAFDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, parentOrgUnitPath)
    return None

def getOrgUnitId(cd=None, orgUnit=None):
  if cd is None:
    cd = buildGAPIObject(API.DIRECTORY)
  if orgUnit is None:
    orgUnit = getOrgUnitItem()
  try:
    if orgUnit == '/':
      result = callGAPI(cd.orgunits(), 'list',
                        throwReasons=GAPI.ORGUNIT_GET_THROW_REASONS,
                        customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath='/', type='children',
                        fields='organizationUnits(parentOrgUnitId,parentOrgUnitPath)')
      if result.get('organizationUnits', []):
        return (result['organizationUnits'][0]['parentOrgUnitPath'], result['organizationUnits'][0]['parentOrgUnitId'])
      topLevelOrgId = getTopLevelOrgId(cd, '/')
      if topLevelOrgId:
        return (orgUnit, topLevelOrgId)
      return (orgUnit, '/') #Bogus but should never happen
    result = callGAPI(cd.orgunits(), 'get',
                      throwReasons=GAPI.ORGUNIT_GET_THROW_REASONS,
                      customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath=encodeOrgUnitPath(makeOrgUnitPathRelative(orgUnit)),
                      fields='orgUnitId,orgUnitPath')
    return (result['orgUnitPath'], result['orgUnitId'])
  except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError):
    entityDoesNotExistExit(Ent.ORGANIZATIONAL_UNIT, orgUnit)
  except (GAPI.badRequest, GAPI.invalidCustomerId, GAPI.loginRequired):
    accessErrorExit(cd)

def getAllParentOrgUnitsForUser(cd, user):
  try:
    result = callGAPI(cd.users(), 'get',
                      throwReasons=GAPI.USER_GET_THROW_REASONS,
                      userKey=user, fields='orgUnitPath', projection='basic')
  except (GAPI.userNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden):
    entityDoesNotExistExit(Ent.USER, user)
  except (GAPI.badRequest, GAPI.invalidCustomerId, GAPI.loginRequired):
    accessErrorExit(cd)
  parentPath = result['orgUnitPath']
  if parentPath == '/':
    orgUnitPath, orgUnitId = getOrgUnitId(cd, '/')
    return {orgUnitId: orgUnitPath}
  parentPath = encodeOrgUnitPath(makeOrgUnitPathRelative(parentPath))
  orgUnits = {}
  while True:
    try:
      result = callGAPI(cd.orgunits(), 'get',
                        throwReasons=GAPI.ORGUNIT_GET_THROW_REASONS,
                        customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath=parentPath,
                        fields='orgUnitId,orgUnitPath,parentOrgUnitId')
      orgUnits[result['orgUnitId']] = result['orgUnitPath']
      if 'parentOrgUnitId' not in result:
        break
      parentPath = result['parentOrgUnitId']
    except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError):
      entityDoesNotExistExit(Ent.ORGANIZATIONAL_UNIT, parentPath)
    except (GAPI.badRequest, GAPI.invalidCustomerId, GAPI.loginRequired):
      accessErrorExit(cd)
  return orgUnits

def validateREPattern(patstr, flags=0):
  try:
    return re.compile(patstr, flags)
  except re.error as e:
    Cmd.Backup()
    usageErrorExit(f'{Cmd.OB_RE_PATTERN} {Msg.ERROR}: {e}')

def getREPattern(flags=0):
  if Cmd.ArgumentsRemaining():
    patstr = Cmd.Current()
    if patstr:
      Cmd.Advance()
      return validateREPattern(patstr, flags)
  missingArgumentExit(Cmd.OB_RE_PATTERN)

def validateREPatternSubstitution(pattern, replacement):
  try:
    re.sub(pattern, replacement, '')
    return (pattern, replacement)
  except re.error as e:
    Cmd.Backup()
    usageErrorExit(f'{Cmd.OB_RE_SUBSTITUTION} {Msg.ERROR}: {e}')

def getREPatternSubstitution(flags=0):
  pattern = getREPattern(flags)
  replacement = getString(Cmd.OB_RE_SUBSTITUTION, minLen=0)
  return validateREPatternSubstitution(pattern, replacement)

def getSheetEntity(allowBlankSheet):
  if Cmd.ArgumentsRemaining():
    sheet = Cmd.Current()
    if sheet or allowBlankSheet:
      cg = UID_PATTERN.match(sheet)
      if cg:
        if cg.group(1).isdigit():
          Cmd.Advance()
          return {'sheetType': Ent.SHEET_ID, 'sheetValue': int(cg.group(1)), 'sheetId': int(cg.group(1)), 'sheetTitle': ''}
      else:
        Cmd.Advance()
        return {'sheetType': Ent.SHEET, 'sheetValue': sheet, 'sheetId': None, 'sheetTitle': sheet}
  missingArgumentExit(Cmd.OB_SHEET_ENTITY)

def getSheetIdFromSheetEntity(spreadsheet, sheetEntity):
  if sheetEntity['sheetType'] == Ent.SHEET_ID:
    for sheet in spreadsheet['sheets']:
      if sheetEntity['sheetId'] == sheet['properties']['sheetId']:
        return sheet['properties']['sheetId']
  else:
    sheetTitleLower = sheetEntity['sheetTitle'].lower()
    for sheet in spreadsheet['sheets']:
      if sheetTitleLower == sheet['properties']['title'].lower():
        return sheet['properties']['sheetId']
  return None

def protectedSheetId(spreadsheet, sheetId):
  for sheet in spreadsheet['sheets']:
    for protectedRange in sheet.get('protectedRanges', []):
      if protectedRange.get('range', {}).get('sheetId', -1) == sheetId and not protectedRange.get('requestingUserCanEdit', False):
        return True
  return False

def getString(item, checkBlank=False, optional=False, minLen=1, maxLen=None):
  if Cmd.ArgumentsRemaining():
    argstr = Cmd.Current()
    if argstr:
      if checkBlank:
        if argstr.isspace():
          blankArgumentExit(item)
      if (len(argstr) >= minLen) and ((maxLen is None) or (len(argstr) <= maxLen)):
        Cmd.Advance()
        return argstr
      invalidArgumentExit(f'{integerLimits(minLen, maxLen, Msg.STRING_LENGTH)} for {item}')
    if optional or (minLen == 0):
      Cmd.Advance()
      return ''
    emptyArgumentExit(item)
  elif optional:
    return ''
  missingArgumentExit(item)

def escapeCRsNLs(value):
  return value.replace('\r', '\\r').replace('\n', '\\n')

def unescapeCRsNLs(value):
  return value.replace('\\r', '\r').replace('\\n', '\n')

def getStringWithCRsNLs():
  return unescapeCRsNLs(getString(Cmd.OB_STRING, minLen=0))

def getStringReturnInList(item):
  argstr = getString(item, minLen=0).strip()
  if argstr:
    return [argstr]
  return []

SORF_SIG_ARGUMENTS = {'signature', 'sig', 'textsig', 'htmlsig'}
SORF_MSG_ARGUMENTS = {'message', 'textmessage', 'htmlmessage'}
SORF_FILE_ARGUMENTS = {'file', 'textfile', 'htmlfile', 'gdoc', 'ghtml', 'gcsdoc', 'gcshtml'}
SORF_HTML_ARGUMENTS = {'htmlsig', 'htmlmessage', 'htmlfile', 'ghtml', 'gcshtml'}
SORF_TEXT_ARGUMENTS = {'text', 'textfile', 'gdoc', 'gcsdoc'}
SORF_SIG_FILE_ARGUMENTS = SORF_SIG_ARGUMENTS.union(SORF_FILE_ARGUMENTS)
SORF_MSG_FILE_ARGUMENTS = SORF_MSG_ARGUMENTS.union(SORF_FILE_ARGUMENTS)

def getStringOrFile(myarg, minLen=0, unescapeCRLF=False):
  if myarg in SORF_SIG_ARGUMENTS:
    if checkArgumentPresent(SORF_FILE_ARGUMENTS):
      myarg = Cmd.Previous().strip().lower().replace('_', '')
  html = myarg in SORF_HTML_ARGUMENTS
  if myarg in SORF_FILE_ARGUMENTS:
    if myarg in {'file', 'textfile', 'htmlfile'}:
      filename = getString(Cmd.OB_FILE_NAME)
      encoding = getCharSet()
      return (readFile(filename, encoding=encoding), encoding, html)
    if myarg in {'gdoc', 'ghtml'}:
      f = getGDocData(myarg)
      data = f.read()
      f.close()
      return (data, UTF8, html)
    return (getStorageFileData(myarg), UTF8, html)
  if not unescapeCRLF:
    return (getString(Cmd.OB_STRING, minLen=minLen), UTF8, html)
  return (unescapeCRsNLs(getString(Cmd.OB_STRING, minLen=minLen)), UTF8, html)

def getStringWithCRsNLsOrFile():
  if checkArgumentPresent(SORF_FILE_ARGUMENTS):
    return getStringOrFile(Cmd.Previous().strip().lower().replace('_', ''), minLen=0)
  return (unescapeCRsNLs(getString(Cmd.OB_STRING, minLen=0)), UTF8, False)

def todaysDate():
  return datetime.datetime(GM.Globals[GM.DATETIME_NOW].year, GM.Globals[GM.DATETIME_NOW].month, GM.Globals[GM.DATETIME_NOW].day,
                           tzinfo=GC.Values[GC.TIMEZONE])

def todaysTime():
  return datetime.datetime(GM.Globals[GM.DATETIME_NOW].year, GM.Globals[GM.DATETIME_NOW].month, GM.Globals[GM.DATETIME_NOW].day,
                           GM.Globals[GM.DATETIME_NOW].hour, GM.Globals[GM.DATETIME_NOW].minute,
                           tzinfo=GC.Values[GC.TIMEZONE])

def getDelta(argstr, pattern):
  if argstr == 'NOW':
    return todaysTime()
  if argstr == 'TODAY':
    return todaysDate()
  tg = pattern.match(argstr.lower())
  if tg is None:
    return None
  sign = tg.group(1)
  delta = int(tg.group(2))
  unit = tg.group(3)
  if unit == 'y':
    deltaTime = datetime.timedelta(days=delta*365)
  elif unit == 'w':
    deltaTime = datetime.timedelta(weeks=delta)
  elif unit == 'd':
    deltaTime = datetime.timedelta(days=delta)
  elif unit == 'h':
    deltaTime = datetime.timedelta(hours=delta)
  elif unit == 'm':
    deltaTime = datetime.timedelta(minutes=delta)
  baseTime = todaysDate()
  if unit in {'h', 'm'}:
    baseTime = baseTime+datetime.timedelta(hours=GM.Globals[GM.DATETIME_NOW].hour, minutes=GM.Globals[GM.DATETIME_NOW].minute)
  if sign == '-':
    return baseTime-deltaTime
  return baseTime+deltaTime

DELTA_DATE_PATTERN = re.compile(r'^([+-])(\d+)([dwy])$')
DELTA_DATE_FORMAT_REQUIRED = '(+|-)<Number>(d|w|y)'
def getDeltaDate(argstr):
  deltaDate = getDelta(argstr, DELTA_DATE_PATTERN)
  if deltaDate is None:
    invalidArgumentExit(DELTA_DATE_FORMAT_REQUIRED)
  return deltaDate

DELTA_TIME_PATTERN = re.compile(r'^([+-])(\d+)([mhdwy])$')
DELTA_TIME_FORMAT_REQUIRED = '(+|-)<Number>(m|h|d|w|y)'

def getDeltaTime(argstr):
  deltaTime = getDelta(argstr, DELTA_TIME_PATTERN)
  if deltaTime is None:
    invalidArgumentExit(DELTA_TIME_FORMAT_REQUIRED)
  return deltaTime

YYYYMMDD_FORMAT = '%Y-%m-%d'
YYYYMMDD_FORMAT_REQUIRED = 'yyyy-mm-dd'

TODAY_NOW = {'TODAY', 'NOW'}
PLUS_MINUS = {'+', '-'}

def getYYYYMMDD(minLen=1, returnTimeStamp=False, returnDateTime=False, alternateValue=None):
  if Cmd.ArgumentsRemaining():
    argstr = Cmd.Current().strip().upper()
    if argstr:
      if alternateValue is not None and argstr == alternateValue.upper():
        Cmd.Advance()
        return None
      if argstr in TODAY_NOW or argstr[0] in PLUS_MINUS:
        if argstr == 'NOW':
          argstr = 'TODAY'
        argstr = getDeltaDate(argstr).strftime(YYYYMMDD_FORMAT)
      elif argstr == 'NEVER':
        argstr = NEVER_DATE
      try:
        dateTime = datetime.datetime.strptime(argstr, YYYYMMDD_FORMAT)
        Cmd.Advance()
        if returnTimeStamp:
          return time.mktime(dateTime.timetuple())*1000
        if returnDateTime:
          return dateTime
        return argstr
      except ValueError:
        invalidArgumentExit(YYYYMMDD_FORMAT_REQUIRED)
    elif minLen == 0:
      Cmd.Advance()
      return ''
  missingArgumentExit(YYYYMMDD_FORMAT_REQUIRED)

HHMM_FORMAT = '%H:%M'
HHMM_FORMAT_REQUIRED = 'hh:mm'

def getHHMM():
  if Cmd.ArgumentsRemaining():
    argstr = Cmd.Current().strip().upper()
    if argstr:
      try:
        datetime.datetime.strptime(argstr, HHMM_FORMAT)
        Cmd.Advance()
        return argstr
      except ValueError:
        invalidArgumentExit(HHMM_FORMAT_REQUIRED)
  missingArgumentExit(HHMM_FORMAT_REQUIRED)

YYYYMMDD_HHMM_FORMAT = '%Y-%m-%d %H:%M'
YYYYMMDD_HHMM_FORMAT_REQUIRED = 'yyyy-mm-dd hh:mm'

def getYYYYMMDD_HHMM():
  if Cmd.ArgumentsRemaining():
    argstr = Cmd.Current().strip().upper()
    if argstr:
      if argstr in TODAY_NOW or argstr[0] in PLUS_MINUS:
        argstr = getDeltaTime(argstr).strftime(YYYYMMDD_HHMM_FORMAT)
      elif argstr == 'NEVER':
        argstr = NEVER_DATETIME
      argstr = argstr.replace('T', ' ')
      try:
        datetime.datetime.strptime(argstr, YYYYMMDD_HHMM_FORMAT)
        Cmd.Advance()
        return argstr
      except ValueError:
        invalidArgumentExit(YYYYMMDD_HHMM_FORMAT_REQUIRED)
  missingArgumentExit(YYYYMMDD_HHMM_FORMAT_REQUIRED)

YYYYMMDDTHHMMSSZ_FORMAT = '%Y-%m-%dT%H:%M:%SZ'
YYYYMMDD_PATTERN = re.compile(r'^[0-9]{4}-[0-9]{2}-[0-9]{2}$')

def getDateOrDeltaFromNow(returnDateTime=False):
  if Cmd.ArgumentsRemaining():
    argstr = Cmd.Current().strip().upper()
    if argstr:
      if argstr in TODAY_NOW or argstr[0] in PLUS_MINUS:
        if argstr == 'NOW':
          argstr = 'TODAY'
        argDate = getDeltaDate(argstr)
      elif argstr == 'NEVER':
        argDate = datetime.datetime.strptime(NEVER_DATE, YYYYMMDD_FORMAT)
      elif YYYYMMDD_PATTERN.match(argstr):
        try:
          argDate = datetime.datetime.strptime(argstr, YYYYMMDD_FORMAT)
        except ValueError:
          invalidArgumentExit(YYYYMMDD_FORMAT_REQUIRED)
      else:
        invalidArgumentExit(YYYYMMDD_FORMAT_REQUIRED)
      Cmd.Advance()
      if not returnDateTime:
        return argDate.strftime(YYYYMMDD_FORMAT)
      return (datetime.datetime(argDate.year, argDate.month, argDate.day, tzinfo=GC.Values[GC.TIMEZONE]),
              GC.Values[GC.TIMEZONE], argDate.strftime(YYYYMMDD_FORMAT))
  missingArgumentExit(YYYYMMDD_FORMAT_REQUIRED)

YYYYMMDDTHHMMSS_FORMAT_REQUIRED = 'yyyy-mm-ddThh:mm:ss[.fff](Z|(+|-(hh:mm)))'
TIMEZONE_FORMAT_REQUIRED = 'Z|(+|-(hh:mm))'

def getTimeOrDeltaFromNow(returnDateTime=False):
  if Cmd.ArgumentsRemaining():
    argstr = Cmd.Current().strip().upper()
    if argstr:
      if argstr in TODAY_NOW or argstr[0] in PLUS_MINUS:
        argstr = ISOformatTimeStamp(getDeltaTime(argstr))
      elif argstr == 'NEVER':
        argstr = NEVER_TIME
      elif YYYYMMDD_PATTERN.match(argstr):
        try:
          dateTime = datetime.datetime.strptime(argstr, YYYYMMDD_FORMAT)
        except ValueError:
          invalidArgumentExit(YYYYMMDD_FORMAT_REQUIRED)
        try:
          argstr = ISOformatTimeStamp(dateTime.replace(tzinfo=GC.Values[GC.TIMEZONE]))
        except OverflowError:
          pass
      try:
        fullDateTime, tz = iso8601.parse_date(argstr)
        Cmd.Advance()
        if not returnDateTime:
          return argstr.replace(' ', 'T')
        return (fullDateTime, tz, argstr.replace(' ', 'T'))
      except (iso8601.ParseError, OverflowError):
        pass
      invalidArgumentExit(YYYYMMDDTHHMMSS_FORMAT_REQUIRED)
  missingArgumentExit(YYYYMMDDTHHMMSS_FORMAT_REQUIRED)

def getRowFilterDateOrDeltaFromNow(argstr):
  argstr = argstr.strip().upper()
  if argstr in TODAY_NOW or argstr[0] in PLUS_MINUS:
    if argstr == 'NOW':
      argstr = 'TODAY'
    deltaDate = getDelta(argstr, DELTA_DATE_PATTERN)
    if deltaDate is None:
      return (False, DELTA_DATE_FORMAT_REQUIRED)
    argstr = ISOformatTimeStamp(deltaDate.replace(tzinfo=iso8601.UTC))
  elif argstr == 'NEVER' or YYYYMMDD_PATTERN.match(argstr):
    if argstr == 'NEVER':
      argstr = NEVER_DATE
    try:
      dateTime = datetime.datetime.strptime(argstr, YYYYMMDD_FORMAT)
    except ValueError:
      return (False, YYYYMMDD_FORMAT_REQUIRED)
    argstr = ISOformatTimeStamp(dateTime.replace(tzinfo=iso8601.UTC))
  try:
    iso8601.parse_date(argstr)
    return (True, argstr.replace(' ', 'T'))
  except (iso8601.ParseError, OverflowError):
    return (False, YYYYMMDD_FORMAT_REQUIRED)

def getRowFilterTimeOrDeltaFromNow(argstr):
  argstr = argstr.strip().upper()
  if argstr in TODAY_NOW or argstr[0] in PLUS_MINUS:
    deltaTime = getDelta(argstr, DELTA_TIME_PATTERN)
    if deltaTime is None:
      return (False, DELTA_TIME_FORMAT_REQUIRED)
    argstr = ISOformatTimeStamp(deltaTime)
  elif argstr == 'NEVER':
    argstr = NEVER_TIME
  elif YYYYMMDD_PATTERN.match(argstr):
    try:
      dateTime = datetime.datetime.strptime(argstr, YYYYMMDD_FORMAT)
    except ValueError:
      return (False, YYYYMMDD_FORMAT_REQUIRED)
    argstr = ISOformatTimeStamp(dateTime.replace(tzinfo=GC.Values[GC.TIMEZONE]))
  try:
    iso8601.parse_date(argstr)
    return (True, argstr.replace(' ', 'T'))
  except (iso8601.ParseError, OverflowError):
    return (False, YYYYMMDDTHHMMSS_FORMAT_REQUIRED)

def mapQueryRelativeTimes(query, keywords):
  QUOTES = '\'"'
  for kw in keywords:
    pattern = re.compile(rf'({kw})\s*([<>]=?|=|!=)\s*[{QUOTES}]?(now|today|[+-]\d+[mhdwy])', re.IGNORECASE)
    pos = 0
    while True:
      mg = pattern.search(query, pos)
      if not mg:
        break
      if mg.groups()[2] is not None:
        deltaTime = getDelta(mg.group(3).upper(), DELTA_TIME_PATTERN)
        if deltaTime:
          query = query[:mg.start(3)]+ISOformatTimeStamp(deltaTime)+query[mg.end(3):]
      pos = mg.end()
  return query

class StartEndTime():
  def __init__(self, startkw='starttime', endkw='endtime', mode='time'):
    self.startTime = self.endTime = self.startDateTime = self.endDateTime = None
    self._startkw = startkw
    self._endkw = endkw
    self._getValueOrDeltaFromNow = getTimeOrDeltaFromNow if mode == 'time' else getDateOrDeltaFromNow

  def Get(self, myarg):
    if myarg in {'start', self._startkw}:
      self.startDateTime, _, self.startTime = self._getValueOrDeltaFromNow(True)
    elif myarg in {'end', self._endkw}:
      self.endDateTime, _, self.endTime = self._getValueOrDeltaFromNow(True)
    elif myarg == 'yesterday':
      currDate = todaysDate()
      self.startDateTime = currDate+datetime.timedelta(days=-1)
      self.startTime = ISOformatTimeStamp(self.startDateTime)
      self.endDateTime = currDate+datetime.timedelta(seconds=-1)
      self.endTime = ISOformatTimeStamp(self.endDateTime)
    elif myarg == 'today':
      currDate = todaysDate()
      self.startDateTime = currDate
      self.startTime = ISOformatTimeStamp(self.startDateTime)
    elif myarg == 'range':
      self.startDateTime, _, self.startTime = self._getValueOrDeltaFromNow(True)
      self.endDateTime, _, self.endTime = self._getValueOrDeltaFromNow(True)
    else: #elif myarg in {'thismonth', 'previousmonths'}
      if myarg == 'thismonth':
        firstMonth = 0
      else:
        firstMonth = getInteger(minVal=1, maxVal=6)
      currDate = todaysDate()
      self.startDateTime = currDate+relativedelta(months=-firstMonth, day=1, hour=0, minute=0, second=0, microsecond=0)
      self.startTime = ISOformatTimeStamp(self.startDateTime)
      if myarg == 'thismonth':
        self.endDateTime = todaysTime()
      else:
        self.endDateTime = currDate+relativedelta(day=1, hour=23, minute=59, second=59, microsecond=0)+relativedelta(days=-1)
      self.endTime = ISOformatTimeStamp(self.endDateTime)
    if self.startDateTime and self.endDateTime and self.endDateTime < self.startDateTime:
      Cmd.Backup()
      usageErrorExit(Msg.INVALID_DATE_TIME_RANGE.format(self._endkw, self.endTime, self._startkw, self.startTime))

EVENTID_PATTERN = re.compile(r'^[a-v0-9]{5,1024}$')
EVENTID_FORMAT_REQUIRED = '[a-v0-9]{5,1024}'

def getEventID():
  if Cmd.ArgumentsRemaining():
    tg = EVENTID_PATTERN.match(Cmd.Current().strip())
    if tg:
      Cmd.Advance()
      return tg.group(0)
    invalidArgumentExit(EVENTID_FORMAT_REQUIRED)
  missingArgumentExit(EVENTID_FORMAT_REQUIRED)

EVENT_TIME_FORMAT_REQUIRED = 'allday yyyy-mm-dd | '+YYYYMMDDTHHMMSS_FORMAT_REQUIRED

def getEventTime():
  if Cmd.ArgumentsRemaining():
    if Cmd.Current().strip().lower() == 'allday':
      Cmd.Advance()
      return {'date': getYYYYMMDD()}
    return {'dateTime': getTimeOrDeltaFromNow()}
  missingArgumentExit(EVENT_TIME_FORMAT_REQUIRED)

AGE_TIME_PATTERN = re.compile(r'^(\d+)([mhdw])$')
AGE_TIME_FORMAT_REQUIRED = '<Number>(m|h|d|w)'

def getAgeTime():
  if Cmd.ArgumentsRemaining():
    tg = AGE_TIME_PATTERN.match(Cmd.Current().strip().lower())
    if tg:
      age = int(tg.group(1))
      age_unit = tg.group(2)
      now = int(time.time())
      if age_unit == 'm':
        age = now-(age*SECONDS_PER_MINUTE)
      elif age_unit == 'h':
        age = now-(age*SECONDS_PER_HOUR)
      elif age_unit == 'd':
        age = now-(age*SECONDS_PER_DAY)
      else: # age_unit == 'w':
        age = now-(age*SECONDS_PER_WEEK)
      Cmd.Advance()
      return age*1000
    invalidArgumentExit(AGE_TIME_FORMAT_REQUIRED)
  missingArgumentExit(AGE_TIME_FORMAT_REQUIRED)

CALENDAR_REMINDER_METHODS = ['email', 'popup']
CALENDAR_REMINDER_MAX_MINUTES = 40320

def getCalendarReminder(allowClearNone=False):
  methods = CALENDAR_REMINDER_METHODS[:]
  if allowClearNone:
    methods += Cmd.CLEAR_NONE_ARGUMENT
  if Cmd.ArgumentsRemaining():
    method = Cmd.Current().strip()
    if not method.isdigit():
      method = getChoice(methods)
      minutes = getInteger(minVal=0, maxVal=CALENDAR_REMINDER_MAX_MINUTES)
    else:
      minutes = getInteger(minVal=0, maxVal=CALENDAR_REMINDER_MAX_MINUTES)
      method = getChoice(methods)
    return {'method': method, 'minutes': minutes}
  missingChoiceExit(methods)

def getCharacter():
  if Cmd.ArgumentsRemaining():
    argstr = codecs.escape_decode(bytes(Cmd.Current(), UTF8))[0].decode(UTF8)
    if argstr:
      if len(argstr) == 1:
        Cmd.Advance()
        return argstr
      invalidArgumentExit(f'{integerLimits(1, 1, Msg.STRING_LENGTH)} for {Cmd.OB_CHARACTER}')
    emptyArgumentExit(Cmd.OB_CHARACTER)
  missingArgumentExit(Cmd.OB_CHARACTER)

def getDelimiter():
  if not checkArgumentPresent('delimiter'):
    return None
  return getCharacter()

def getJSON(deleteFields):
  if not checkArgumentPresent('file'):
    encoding = getCharSet()
    if not Cmd.ArgumentsRemaining():
      missingArgumentExit(Cmd.OB_JSON_DATA)
    argstr = Cmd.Current()
#    argstr = Cmd.Current().replace(r'\\"', r'\"')
    Cmd.Advance()
    try:
      if encoding == UTF8:
        jsonData = json.loads(argstr)
      else:
        jsonData = json.loads(argstr.encode(encoding).decode(UTF8))
    except (IndexError, KeyError, SyntaxError, TypeError, ValueError) as e:
      Cmd.Backup()
      usageErrorExit(f'{str(e)}: {argstr if encoding == UTF8 else argstr.encode(encoding).decode(UTF8)}')
  else:
    filename = getString(Cmd.OB_FILE_NAME)
    encoding = getCharSet()
    try:
      jsonData = json.loads(readFile(filename, encoding=encoding))
    except (IndexError, KeyError, SyntaxError, TypeError, ValueError) as e:
      Cmd.Backup()
      usageErrorExit(Msg.JSON_ERROR.format(str(e), filename))
  for field in deleteFields:
    jsonData.pop(field, None)
  return jsonData

def getMatchSkipFields(fieldNames):
  matchFields = {}
  skipFields = {}
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg in {'matchfield', 'skipfield'}:
      matchField = getString(Cmd.OB_FIELD_NAME).strip('~')
      if (not matchField) or (matchField not in fieldNames):
        csvFieldErrorExit(matchField, fieldNames, backupArg=True)
      if myarg == 'matchfield':
        matchFields[matchField] = getREPattern()
      else:
        skipFields[matchField] = getREPattern()
    else:
      Cmd.Backup()
      break
  return (matchFields, skipFields)

def checkMatchSkipFields(row, fieldnames, matchFields, skipFields):
  for matchField, matchPattern in matchFields.items():
    if (matchField not in row) or not matchPattern.search(row[matchField]):
      return False
  for skipField, matchPattern in skipFields.items():
    if (skipField in row) and matchPattern.search(row[skipField]):
      return False
  if fieldnames and (GC.Values[GC.CSV_INPUT_ROW_FILTER] or GC.Values[GC.CSV_INPUT_ROW_DROP_FILTER]):
    return RowFilterMatch(row, fieldnames,
                          GC.Values[GC.CSV_INPUT_ROW_FILTER], GC.Values[GC.CSV_INPUT_ROW_FILTER_MODE],
                          GC.Values[GC.CSV_INPUT_ROW_DROP_FILTER], GC.Values[GC.CSV_INPUT_ROW_DROP_FILTER_MODE])
  return True

def checkSubkeyField():
  if not GM.Globals[GM.CSV_SUBKEY_FIELD]:
    Cmd.Backup()
    usageErrorExit(Msg.NO_CSV_FILE_SUBKEYS_SAVED)
  chkSubkeyField = getString(Cmd.OB_FIELD_NAME, checkBlank=True)
  if chkSubkeyField != GM.Globals[GM.CSV_SUBKEY_FIELD]:
    Cmd.Backup()
    usageErrorExit(Msg.SUBKEY_FIELD_MISMATCH.format(chkSubkeyField, GM.Globals[GM.CSV_SUBKEY_FIELD]))

def checkDataField():
  if not GM.Globals[GM.CSV_DATA_FIELD]:
    Cmd.Backup()
    usageErrorExit(Msg.NO_CSV_FILE_DATA_SAVED)
  chkDataField = getString(Cmd.OB_FIELD_NAME, checkBlank=True)
  if chkDataField != GM.Globals[GM.CSV_DATA_FIELD]:
    Cmd.Backup()
    usageErrorExit(Msg.DATA_FIELD_MISMATCH.format(chkDataField, GM.Globals[GM.CSV_DATA_FIELD]))

MAX_MESSAGE_BYTES_PATTERN = re.compile(r'^(\d+)([mkb]?)$')
MAX_MESSAGE_BYTES_FORMAT_REQUIRED = '<Number>[m|k|b]'

def getMaxMessageBytes(oneKiloBytes, oneMegaBytes):
  if Cmd.ArgumentsRemaining():
    tg = MAX_MESSAGE_BYTES_PATTERN.match(Cmd.Current().strip().lower())
    if tg:
      mmb = int(tg.group(1))
      mmb_unit = tg.group(2)
      if mmb_unit == 'm':
        mmb *= oneMegaBytes
      elif mmb_unit == 'k':
        mmb *= oneKiloBytes
      Cmd.Advance()
      return mmb
    invalidArgumentExit(MAX_MESSAGE_BYTES_FORMAT_REQUIRED)
  missingArgumentExit(MAX_MESSAGE_BYTES_FORMAT_REQUIRED)

# Get domain from email address
def getEmailAddressDomain(emailAddress):
  atLoc = emailAddress.find('@')
  if atLoc == -1:
    return GC.Values[GC.DOMAIN]
  return emailAddress[atLoc+1:].lower()

# Get user name from email address
def getEmailAddressUsername(emailAddress):
  atLoc = emailAddress.find('@')
  if atLoc == -1:
    return emailAddress.lower()
  return emailAddress[:atLoc].lower()

# Split email address into user and domain
def splitEmailAddress(emailAddress):
  atLoc = emailAddress.find('@')
  if atLoc == -1:
    return (emailAddress.lower(), GC.Values[GC.DOMAIN])
  return (emailAddress[:atLoc].lower(), emailAddress[atLoc+1:].lower())

def formatFileSize(fileSize):
  if fileSize == 0:
    return '0kb'
  if fileSize < ONE_KILO_10_BYTES:
    return '1kb'
  if fileSize < ONE_MEGA_10_BYTES:
    return f'{fileSize//ONE_KILO_10_BYTES}kb'
  if fileSize < ONE_GIGA_10_BYTES:
    return f'{fileSize//ONE_MEGA_10_BYTES}mb'
  return f'{fileSize//ONE_GIGA_10_BYTES}gb'

def formatLocalTime(dateTimeStr):
  if dateTimeStr in {NEVER_TIME, NEVER_TIME_NOMS}:
    return GC.Values[GC.NEVER_TIME]
  try:
    timestamp, _ = iso8601.parse_date(dateTimeStr)
    if not GC.Values[GC.OUTPUT_TIMEFORMAT]:
      if GM.Globals[GM.CONVERT_TO_LOCAL_TIME]:
        return ISOformatTimeStamp(timestamp.astimezone(GC.Values[GC.TIMEZONE]))
      return timestamp.strftime(YYYYMMDDTHHMMSSZ_FORMAT)
    if GM.Globals[GM.CONVERT_TO_LOCAL_TIME]:
      return timestamp.astimezone(GC.Values[GC.TIMEZONE]).strftime(GC.Values[GC.OUTPUT_TIMEFORMAT])
    return timestamp.strftime(GC.Values[GC.OUTPUT_TIMEFORMAT])
  except (iso8601.ParseError, OverflowError):
    return dateTimeStr

def formatLocalSecondsTimestamp(timestamp):
  if not GC.Values[GC.OUTPUT_TIMEFORMAT]:
    return ISOformatTimeStamp(datetime.datetime.fromtimestamp(int(timestamp), GC.Values[GC.TIMEZONE]))
  return datetime.datetime.fromtimestamp(int(timestamp), GC.Values[GC.TIMEZONE]).strftime(GC.Values[GC.OUTPUT_TIMEFORMAT])

def formatLocalTimestamp(timestamp):
  if not GC.Values[GC.OUTPUT_TIMEFORMAT]:
    return ISOformatTimeStamp(datetime.datetime.fromtimestamp(int(timestamp)//1000, GC.Values[GC.TIMEZONE]))
  return datetime.datetime.fromtimestamp(int(timestamp)//1000, GC.Values[GC.TIMEZONE]).strftime(GC.Values[GC.OUTPUT_TIMEFORMAT])

def formatLocalTimestampUTC(timestamp):
  return ISOformatTimeStamp(datetime.datetime.fromtimestamp(int(timestamp)//1000, iso8601.UTC))

def formatLocalDatestamp(timestamp):
  try:
    if not GC.Values[GC.OUTPUT_DATEFORMAT]:
      return datetime.datetime.fromtimestamp(int(timestamp)//1000, GC.Values[GC.TIMEZONE]).strftime(YYYYMMDD_FORMAT)
    return datetime.datetime.fromtimestamp(int(timestamp)//1000, GC.Values[GC.TIMEZONE]).strftime(GC.Values[GC.OUTPUT_DATEFORMAT])
  except OverflowError:
    return NEVER_DATE

def formatMaxMessageBytes(maxMessageBytes, oneKiloBytes, oneMegaBytes):
  if maxMessageBytes < oneKiloBytes:
    return maxMessageBytes
  if maxMessageBytes < oneMegaBytes:
    return f'{maxMessageBytes//oneKiloBytes}K'
  return f'{maxMessageBytes//oneMegaBytes}M'

def formatMilliSeconds(millis):
  seconds, millis = divmod(millis, 1000)
  minutes, seconds = divmod(seconds, 60)
  hours, minutes = divmod(minutes, 60)
  return f'{hours:02d}:{minutes:02d}:{seconds:02d}'

def getPhraseDNEorSNA(email):
  return Msg.DOES_NOT_EXIST if getEmailAddressDomain(email) == GC.Values[GC.DOMAIN] else Msg.SERVICE_NOT_APPLICABLE

def formatHTTPError(http_status, reason, message):
  return f'{http_status}: {reason} - {message}'

def getHTTPError(responses, http_status, reason, message):
  if reason in responses:
    return responses[reason]
  return formatHTTPError(http_status, reason, message)

# Warnings
def badRequestWarning(entityType, itemType, itemValue):
  printWarningMessage(BAD_REQUEST_RC,
                      f'{Msg.GOT} 0 {Ent.Plural(entityType)}: {Msg.INVALID} {Ent.Singular(itemType)} - {itemValue}')

def emptyQuery(query, entityType):
  return f'{Ent.Singular(Ent.QUERY)} ({query}) {Msg.NO_ENTITIES_FOUND.format(Ent.Plural(entityType))}'

def invalidQuery(query):
  return f'{Ent.Singular(Ent.QUERY)} ({query}) {Msg.INVALID}'

def invalidMember(query):
  if query:
    badRequestWarning(Ent.GROUP, Ent.QUERY, invalidQuery(query))
    return True
  return False

def invalidUserSchema(schema):
  if isinstance(schema, list):
    return f'{Ent.Singular(Ent.USER_SCHEMA)} ({",".join(schema)}) {Msg.INVALID}'
  return f'{Ent.Singular(Ent.USER_SCHEMA)} {schema}) {Msg.INVALID}'

def userServiceNotEnabledWarning(entityName, service, i=0, count=0):
  setSysExitRC(SERVICE_NOT_APPLICABLE_RC)
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 [Ent.Singular(Ent.USER), entityName, Msg.SERVICE_NOT_ENABLED.format(service)],
                                 currentCountNL(i, count)))

def userAlertsServiceNotEnabledWarning(entityName, i=0, count=0):
  userServiceNotEnabledWarning(entityName, 'Alerts', i, count)

def userAnalyticsServiceNotEnabledWarning(entityName, i=0, count=0):
  userServiceNotEnabledWarning(entityName, 'Alerts', i, count)

def userCalServiceNotEnabledWarning(entityName, i=0, count=0):
  userServiceNotEnabledWarning(entityName, 'Calendar', i, count)

def userChatServiceNotEnabledWarning(entityName, i=0, count=0):
  userServiceNotEnabledWarning(entityName, 'Chat', i, count)

def userContactDelegateServiceNotEnabledWarning(entityName, i=0, count=0):
  userServiceNotEnabledWarning(entityName, 'Contact Delegate', i, count)

def userDriveServiceNotEnabledWarning(user, errMessage, i=0, count=0):
#  if errMessage.find('Drive apps') == -1 and errMessage.find('Active session is invalid') == -1:
#    entityServiceNotApplicableWarning(Ent.USER, user, i, count)
  if errMessage.find('Drive apps') >= 0 or errMessage.find('Active session is invalid') >= 0:
    userServiceNotEnabledWarning(user, 'Drive', i, count)
  else:
    entityActionNotPerformedWarning([Ent.USER, user], errMessage, i, count)

def userKeepServiceNotEnabledWarning(entityName, i=0, count=0):
  userServiceNotEnabledWarning(entityName, 'Keep', i, count)

def userGmailServiceNotEnabledWarning(entityName, i=0, count=0):
  userServiceNotEnabledWarning(entityName, 'Gmail', i, count)

def userLookerStudioServiceNotEnabledWarning(entityName, i=0, count=0):
  userServiceNotEnabledWarning(entityName, 'Looker Studio', i, count)

def userPeopleServiceNotEnabledWarning(entityName, i=0, count=0):
  userServiceNotEnabledWarning(entityName, 'People', i, count)

def userTasksServiceNotEnabledWarning(entityName, i=0, count=0):
  userServiceNotEnabledWarning(entityName, 'Tasks', i, count)

def userYouTubeServiceNotEnabledWarning(entityName, i=0, count=0):
  userServiceNotEnabledWarning(entityName, 'YouTube', i, count)

def entityServiceNotApplicableWarning(entityType, entityName, i=0, count=0):
  setSysExitRC(SERVICE_NOT_APPLICABLE_RC)
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 [Ent.Singular(entityType), entityName, Msg.SERVICE_NOT_APPLICABLE],
                                 currentCountNL(i, count)))

def entityDoesNotExistWarning(entityType, entityName, i=0, count=0):
  setSysExitRC(ENTITY_DOES_NOT_EXIST_RC)
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 [Ent.Singular(entityType), entityName, Msg.DOES_NOT_EXIST],
                                 currentCountNL(i, count)))

def entityListDoesNotExistWarning(entityValueList, i=0, count=0):
  setSysExitRC(ENTITY_DOES_NOT_EXIST_RC)
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[Msg.DOES_NOT_EXIST],
                                 currentCountNL(i, count)))

def entityUnknownWarning(entityType, entityName, i=0, count=0):
  domain = getEmailAddressDomain(entityName)
  if (domain.endswith(GC.Values[GC.DOMAIN])) or (domain.endswith('google.com')):
    entityDoesNotExistWarning(entityType, entityName, i, count)
  else:
    entityServiceNotApplicableWarning(entityType, entityName, i, count)

def entityOrEntityUnknownWarning(entity1Type, entity1Name, entity2Type, entity2Name, i=0, count=0):
  setSysExitRC(ENTITY_DOES_NOT_EXIST_RC)
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 [f'{Msg.EITHER} {Ent.Singular(entity1Type)}', entity1Name, getPhraseDNEorSNA(entity1Name), None,
                                  f'{Msg.OR} {Ent.Singular(entity2Type)}', entity2Name, getPhraseDNEorSNA(entity2Name)],
                                 currentCountNL(i, count)))

def entityDoesNotHaveItemWarning(entityValueList, i=0, count=0):
  setSysExitRC(ENTITY_DOES_NOT_EXIST_RC)
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[Msg.DOES_NOT_EXIST],
                                 currentCountNL(i, count)))

def duplicateAliasGroupUserWarning(cd, entityValueList, i=0, count=0):
  email = entityValueList[1]
  try:
    result = callGAPI(cd.users(), 'get',
                      throwReasons=GAPI.USER_GET_THROW_REASONS,
                      userKey=email, fields='id,primaryEmail')
    if (result['primaryEmail'].lower() == email) or (result['id'] == email):
      kvList = [Ent.USER, email]
    else:
      kvList = [Ent.USER_ALIAS, email, Ent.USER, result['primaryEmail']]
  except (GAPI.userNotFound, GAPI.badRequest,
          GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden, GAPI.backendError, GAPI.systemError):
    try:
      result = callGAPI(cd.groups(), 'get',
                        throwReasons=GAPI.GROUP_GET_THROW_REASONS,
                        groupKey=email, fields='id,email')
      if (result['email'].lower() == email) or (result['id'] == email):
        kvList = [Ent.GROUP, email]
      else:
        kvList = [Ent.GROUP_ALIAS, email, Ent.GROUP, result['email']]
    except (GAPI.groupNotFound,
            GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden, GAPI.badRequest):
      kvList = [Ent.EMAIL, email]
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+
                                 [Act.Failed(), Msg.DUPLICATE]+
                                 Ent.FormatEntityValueList(kvList),
                                 currentCountNL(i, count)))
  setSysExitRC(ENTITY_DUPLICATE_RC)
  return kvList[0]

def entityDuplicateWarning(entityValueList, i=0, count=0):
  setSysExitRC(ENTITY_DUPLICATE_RC)
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[Act.Failed(), Msg.DUPLICATE],
                                 currentCountNL(i, count)))

def entityActionFailedWarning(entityValueList, errMessage, i=0, count=0):
  setSysExitRC(ACTION_FAILED_RC)
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[Act.Failed(), errMessage],
                                 currentCountNL(i, count)))

def entityModifierItemValueListActionFailedWarning(entityValueList, modifier, infoTypeValueList, errMessage, i=0, count=0):
  setSysExitRC(ACTION_FAILED_RC)
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.ToPerform()} {modifier}', None]+Ent.FormatEntityValueList(infoTypeValueList)+[Act.Failed(), errMessage],
                                 currentCountNL(i, count)))

def entityModifierActionFailedWarning(entityValueList, modifier, errMessage, i=0, count=0):
  setSysExitRC(ACTION_FAILED_RC)
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.ToPerform()} {modifier}', Act.Failed(), errMessage],
                                 currentCountNL(i, count)))

def entityModifierNewValueActionFailedWarning(entityValueList, modifier, newValue, errMessage, i=0, count=0):
  setSysExitRC(ACTION_FAILED_RC)
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.ToPerform()} {modifier}', newValue, Act.Failed(), errMessage],
                                 currentCountNL(i, count)))

def entityNumEntitiesActionFailedWarning(entityType, entityName, itemType, itemCount, errMessage, i=0, count=0):
  setSysExitRC(ACTION_FAILED_RC)
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 [Ent.Singular(entityType), entityName,
                                  Ent.Choose(itemType, itemCount), itemCount,
                                  Act.Failed(), errMessage],
                                 currentCountNL(i, count)))

def entityActionNotPerformedWarning(entityValueList, errMessage, i=0, count=0):
  setSysExitRC(ACTION_NOT_PERFORMED_RC)
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[Act.NotPerformed(), errMessage],
                                 currentCountNL(i, count)))

def entityItemValueListActionNotPerformedWarning(entityValueList, infoTypeValueList, errMessage, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[Act.NotPerformed(), '']+Ent.FormatEntityValueList(infoTypeValueList)+[errMessage],
                                 currentCountNL(i, count)))

def entityModifierItemValueListActionNotPerformedWarning(entityValueList, modifier, infoTypeValueList, errMessage, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.NotPerformed()} {modifier}', None]+Ent.FormatEntityValueList(infoTypeValueList)+[errMessage],
                                 currentCountNL(i, count)))

def entityNumEntitiesActionNotPerformedWarning(entityValueList, itemType, itemCount, errMessage, i=0, count=0):
  setSysExitRC(ACTION_NOT_PERFORMED_RC)
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[Ent.Choose(itemType, itemCount), itemCount, Act.NotPerformed(), errMessage],
                                 currentCountNL(i, count)))

def entityBadRequestWarning(entityValueList, errMessage, i=0, count=0):
  setSysExitRC(BAD_REQUEST_RC)
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[ERROR, errMessage],
                                 currentCountNL(i, count)))

# Getting ... utilities
def printGettingAllAccountEntities(entityType, query='', qualifier='', accountType=Ent.ACCOUNT):
  if GC.Values[GC.SHOW_GETTINGS]:
    if query:
      Ent.SetGettingQuery(entityType, query)
    elif qualifier:
      Ent.SetGettingQualifier(entityType, qualifier)
    else:
      Ent.SetGetting(entityType)
    writeStderr(f'{Msg.GETTING_ALL} {Ent.PluralGetting()}{Ent.GettingPreQualifier()}{Ent.MayTakeTime(accountType)}\n')

def printGotAccountEntities(count):
  if GC.Values[GC.SHOW_GETTINGS]:
    writeStderr(f'{Msg.GOT} {count} {Ent.ChooseGetting(count)}{Ent.GettingPostQualifier()}\n')

def setGettingAllEntityItemsForWhom(entityItem, forWhom, query='', qualifier=''):
  if GC.Values[GC.SHOW_GETTINGS]:
    if query:
      Ent.SetGettingQuery(entityItem, query)
    elif qualifier:
      Ent.SetGettingQualifier(entityItem, qualifier)
    else:
      Ent.SetGetting(entityItem)
    Ent.SetGettingForWhom(forWhom)

def printGettingAllEntityItemsForWhom(entityItem, forWhom, i=0, count=0, query='', qualifier='', entityType=None):
  if GC.Values[GC.SHOW_GETTINGS]:
    setGettingAllEntityItemsForWhom(entityItem, forWhom, query=query, qualifier=qualifier)
    writeStderr(f'{Msg.GETTING_ALL} {Ent.PluralGetting()}{Ent.GettingPreQualifier()} {Msg.FOR} {forWhom}{Ent.MayTakeTime(entityType)}{currentCountNL(i, count)}')

def printGotEntityItemsForWhom(count):
  if GC.Values[GC.SHOW_GETTINGS]:
    writeStderr(f'{Msg.GOT} {count} {Ent.ChooseGetting(count)}{Ent.GettingPostQualifier()} {Msg.FOR} {Ent.GettingForWhom()}\n')

def printGettingEntityItem(entityType, entityItem, i=0, count=0):
  if GC.Values[GC.SHOW_GETTINGS]:
    writeStderr(f'{Msg.GETTING} {Ent.Singular(entityType)} {entityItem}{currentCountNL(i, count)}')

def printGettingEntityItemForWhom(entityItem, forWhom, i=0, count=0):
  if GC.Values[GC.SHOW_GETTINGS]:
    Ent.SetGetting(entityItem)
    Ent.SetGettingForWhom(forWhom)
    writeStderr(f'{Msg.GETTING} {Ent.PluralGetting()} {Msg.FOR} {forWhom}{currentCountNL(i, count)}')

def stderrEntityMessage(entityValueList, message, i=0, count=0):
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[message],
                                 currentCountNL(i, count)))

FIRST_ITEM_MARKER = '%%first_item%%'
LAST_ITEM_MARKER = '%%last_item%%'
TOTAL_ITEMS_MARKER = '%%total_items%%'

def getPageMessage(showFirstLastItems=False, showDate=None):
  if not GC.Values[GC.SHOW_GETTINGS]:
    return None
  pageMessage = f'{Msg.GOT} {TOTAL_ITEMS_MARKER} {{0}}'
  if showDate:
    pageMessage += f' on {showDate}'
  if showFirstLastItems:
    pageMessage += f': {FIRST_ITEM_MARKER} - {LAST_ITEM_MARKER}'
  else:
    pageMessage += '...'
  if GC.Values[GC.SHOW_GETTINGS_GOT_NL]:
    pageMessage += '\n'
  else:
    GM.Globals[GM.LAST_GOT_MSG_LEN] = 0
  return pageMessage

def getPageMessageForWhom(forWhom=None, showFirstLastItems=False, showDate=None, clearLastGotMsgLen=True):
  if not GC.Values[GC.SHOW_GETTINGS]:
    return None
  if forWhom:
    Ent.SetGettingForWhom(forWhom)
  pageMessage = f'{Msg.GOT} {TOTAL_ITEMS_MARKER} {{0}}{Ent.GettingPostQualifier()} {Msg.FOR} {Ent.GettingForWhom()}'
  if showDate:
    pageMessage += f' on {showDate}'
  if showFirstLastItems:
    pageMessage += f': {FIRST_ITEM_MARKER} - {LAST_ITEM_MARKER}'
  else:
    pageMessage += '...'
  if GC.Values[GC.SHOW_GETTINGS_GOT_NL]:
    pageMessage += '\n'
  elif clearLastGotMsgLen:
    GM.Globals[GM.LAST_GOT_MSG_LEN] = 0
  return pageMessage

def printLine(message):
  writeStdout(message+'\n')

def printBlankLine():
  writeStdout('\n')

def printKeyValueList(kvList):
  writeStdout(formatKeyValueList(Ind.Spaces(), kvList, '\n'))

def printKeyValueListWithCount(kvList, i, count):
  writeStdout(formatKeyValueList(Ind.Spaces(), kvList, currentCountNL(i, count)))

def printKeyValueDict(kvDict):
  for key, value in kvDict.items():
    writeStdout(formatKeyValueList(Ind.Spaces(), [key, value], '\n'))

def printKeyValueWithCRsNLs(key, value):
  if value.find('\n') >= 0 or value.find('\r') >= 0:
    if GC.Values[GC.SHOW_CONVERT_CR_NL]:
      printKeyValueList([key, escapeCRsNLs(value)])
    else:
      printKeyValueList([key, ''])
      Ind.Increment()
      printKeyValueList([Ind.MultiLineText(value)])
      Ind.Decrement()
  else:
    printKeyValueList([key, value])

def printJSONKey(key):
  writeStdout(formatKeyValueList(Ind.Spaces(), [key, None], ''))

def printJSONValue(value):
  writeStdout(formatKeyValueList(' ', [value], '\n'))

def printEntity(entityValueList, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList),
                                 currentCountNL(i, count)))

def printEntityMessage(entityValueList, message, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[message],
                                 currentCountNL(i, count)))

def printEntitiesCount(entityType, entityList):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 [Ent.Plural(entityType), None if entityList is None else f'({len(entityList)})'],
                                 '\n'))

def printEntityKVList(entityValueList, infoKVList, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+infoKVList,
                                 currentCountNL(i, count)))

def performAction(entityType, entityValue, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 [f'{Act.ToPerform()} {Ent.Singular(entityType)} {entityValue}'],
                                 currentCountNL(i, count)))

def performActionNumItems(itemCount, itemType, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 [f'{Act.ToPerform()} {itemCount} {Ent.Choose(itemType, itemCount)}'],
                                 currentCountNL(i, count)))

def performActionModifierNumItems(modifier, itemCount, itemType, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 [f'{Act.ToPerform()} {modifier} {itemCount} {Ent.Choose(itemType, itemCount)}'],
                                 currentCountNL(i, count)))

def actionPerformedNumItems(itemCount, itemType, i=0, count=0):
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 [f'{itemCount} {Ent.Choose(itemType, itemCount)} {Act.Performed()} '],
                                 currentCountNL(i, count)))

def actionFailedNumItems(itemCount, itemType, errMessage, i=0, count=0):
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 [f'{itemCount} {Ent.Choose(itemType, itemCount)} {Act.Failed()}: {errMessage} '],
                                 currentCountNL(i, count)))

def actionNotPerformedNumItemsWarning(itemCount, itemType, errMessage, i=0, count=0):
  setSysExitRC(ACTION_NOT_PERFORMED_RC)
  writeStderr(formatKeyValueList(Ind.Spaces(),
                                 [Ent.Choose(itemType, itemCount), itemCount, Act.NotPerformed(), errMessage],
                                 currentCountNL(i, count)))

def entityPerformAction(entityValueList, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.ToPerform()}'],
                                 currentCountNL(i, count)))

def entityPerformActionNumItems(entityValueList, itemCount, itemType, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.ToPerform()} {itemCount} {Ent.Choose(itemType, itemCount)}'],
                                 currentCountNL(i, count)))

def entityPerformActionModifierNumItems(entityValueList, modifier, itemCount, itemType, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.ToPerform()} {modifier} {itemCount} {Ent.Choose(itemType, itemCount)}'],
                                 currentCountNL(i, count)))

def entityPerformActionNumItemsModifier(entityValueList, itemCount, itemType, modifier, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.ToPerform()} {itemCount} {Ent.Choose(itemType, itemCount)} {modifier}'],
                                 currentCountNL(i, count)))

def entityPerformActionSubItemModifierNumItems(entityValueList, subitemType, modifier, itemCount, itemType, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.ToPerform()} {Ent.Plural(subitemType)} {modifier} {itemCount} {Ent.Choose(itemType, itemCount)}'],
                                 currentCountNL(i, count)))

def entityPerformActionSubItemModifierNumItemsModifierNewValue(entityValueList, subitemType, modifier1, itemCount, itemType, modifier2, newValue, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+
                                 [f'{Act.ToPerform()} {Ent.Plural(subitemType)} {modifier1} {itemCount} {Ent.Choose(itemType, itemCount)} {modifier2}', newValue],
                                 currentCountNL(i, count)))

def entityPerformActionModifierNumItemsModifier(entityValueList, modifier1, itemCount, itemType, modifier2, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.ToPerform()} {modifier1} {itemCount} {Ent.Choose(itemType, itemCount)} {modifier2}'],
                                 currentCountNL(i, count)))

def entityPerformActionModifierItemValueList(entityValueList, modifier, infoTypeValueList, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.ToPerform()} {modifier}', None]+Ent.FormatEntityValueList(infoTypeValueList),
                                 currentCountNL(i, count)))

def entityPerformActionModifierNewValue(entityValueList, modifier, newValue, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.ToPerform()} {modifier}', newValue],
                                 currentCountNL(i, count)))

def entityPerformActionModifierNewValueItemValueList(entityValueList, modifier, newValue, infoTypeValueList, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.ToPerform()} {modifier}', newValue]+Ent.FormatEntityValueList(infoTypeValueList),
                                 currentCountNL(i, count)))

def entityPerformActionItemValue(entityValueList, itemType, itemValue, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[Act.ToPerform(), None, Ent.Singular(itemType), itemValue],
                                 currentCountNL(i, count)))

def entityPerformActionInfo(entityValueList, infoValue, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[Act.ToPerform(), infoValue],
                                 currentCountNL(i, count)))

def entityActionPerformed(entityValueList, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[Act.Performed()],
                                 currentCountNL(i, count)))

def entityActionPerformedMessage(entityValueList, message, i=0, count=0):
  if message:
    writeStdout(formatKeyValueList(Ind.Spaces(),
                                   Ent.FormatEntityValueList(entityValueList)+[Act.Performed(), message],
                                   currentCountNL(i, count)))
  else:
    writeStdout(formatKeyValueList(Ind.Spaces(),
                                   Ent.FormatEntityValueList(entityValueList)+[Act.Performed()],
                                   currentCountNL(i, count)))

def entityNumItemsActionPerformed(entityValueList, itemCount, itemType, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{itemCount} {Ent.Choose(itemType, itemCount)} {Act.Performed()}'],
                                 currentCountNL(i, count)))

def entityModifierActionPerformed(entityValueList, modifier, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.Performed()} {modifier}', None],
                                 currentCountNL(i, count)))

def entityModifierItemValueListActionPerformed(entityValueList, modifier, infoTypeValueList, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.Performed()} {modifier}', None]+Ent.FormatEntityValueList(infoTypeValueList),
                                 currentCountNL(i, count)))

def entityModifierNewValueActionPerformed(entityValueList, modifier, newValue, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.Performed()} {modifier}', newValue],
                                 currentCountNL(i, count)))

def entityModifierNewValueItemValueListActionPerformed(entityValueList, modifier, newValue, infoTypeValueList, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.Performed()} {modifier}', newValue]+Ent.FormatEntityValueList(infoTypeValueList),
                                 currentCountNL(i, count)))

def entityModifierNewValueKeyValueActionPerformed(entityValueList, modifier, newValue, infoKey, infoValue, i=0, count=0):
  writeStdout(formatKeyValueList(Ind.Spaces(),
                                 Ent.FormatEntityValueList(entityValueList)+[f'{Act.Performed()} {modifier}', newValue, infoKey, infoValue],
                                 currentCountNL(i, count)))

def cleanFilename(filename):
  return sanitize_filename(filename, '_')

def setFilePath(fileName):
  if fileName.startswith('./') or fileName.startswith('.\\'):
    fileName = os.path.join(os.getcwd(), fileName[2:])
  else:
    fileName = os.path.expanduser(fileName)
  if not os.path.isabs(fileName):
    fileName = os.path.join(GC.Values[GC.DRIVE_DIR], fileName)
  return fileName

def uniqueFilename(targetFolder, filetitle, overwrite, extension=None):
  filename = filetitle
  y = 0
  while True:
    if extension is not None and filename.lower()[-len(extension):] != extension.lower():
      filename += extension
    filepath = os.path.join(targetFolder, filename)
    if overwrite or not os.path.isfile(filepath):
      return (filepath, filename)
    y += 1
    filename = f'({y})-{filetitle}'

def cleanFilepath(filepath):
  return sanitize_filepath(filepath, platform='auto')

def fileErrorMessage(filename, e, entityType=Ent.FILE):
  return f'{Ent.Singular(entityType)}: {filename}, {str(e)}'

def fdErrorMessage(f, defaultFilename, e):
  return fileErrorMessage(getattr(f, 'name') if hasattr(f, 'name') else defaultFilename, e)

# Set file encoding to handle UTF8 BOM
def setEncoding(mode, encoding):
  if 'b' in mode:
    return {}
  if not encoding:
    encoding = GM.Globals[GM.SYS_ENCODING]
  if 'r' in mode and encoding.lower().replace('-', '') == 'utf8':
    encoding = UTF8_SIG
  return {'encoding': encoding}

def StringIOobject(initbuff=None):
  if initbuff is None:
    return io.StringIO()
  return io.StringIO(initbuff)

# Open a file
def openFile(filename, mode=DEFAULT_FILE_READ_MODE, encoding=None, errors=None, newline=None,
             continueOnError=False, displayError=True, stripUTFBOM=False):
  try:
    if filename != '-':
      kwargs = setEncoding(mode, encoding)
      f = open(os.path.expanduser(filename), mode, errors=errors, newline=newline, **kwargs)
      if stripUTFBOM:
        if 'b' in mode:
          if f.read(3) != b'\xef\xbb\xbf':
            f.seek(0)
        elif not kwargs['encoding'].lower().startswith('utf'):
          if f.read(3).encode('iso-8859-1', 'replace') != codecs.BOM_UTF8:
            f.seek(0)
        else:
          if f.read(1) != '\ufeff':
            f.seek(0)
      return f
    if 'r' in mode:
      return StringIOobject(str(sys.stdin.read()))
    if 'b' not in mode:
      return sys.stdout
    return os.fdopen(os.dup(sys.stdout.fileno()), 'wb')
  except (IOError, LookupError, UnicodeDecodeError, UnicodeError) as e:
    if continueOnError:
      if displayError:
        stderrWarningMsg(fileErrorMessage(filename, e))
        setSysExitRC(FILE_ERROR_RC)
      return None
    systemErrorExit(FILE_ERROR_RC, fileErrorMessage(filename, e))

# Close a file
def closeFile(f, forceFlush=False):
  try:
    if forceFlush:
      # Necessary to make sure file is flushed by both Python and OS
      # https://stackoverflow.com/a/13762137/1503886
      f.flush()
      os.fsync(f.fileno())
    f.close()
    return True
  except IOError as e:
    stderrErrorMsg(fdErrorMessage(f, UNKNOWN, e))
    setSysExitRC(FILE_ERROR_RC)
    return False

# Read a file
def readFile(filename, mode=DEFAULT_FILE_READ_MODE, encoding=None, newline=None,
             continueOnError=False, displayError=True):
  try:
    if filename != '-':
      kwargs = setEncoding(mode, encoding)
      with open(os.path.expanduser(filename), mode, newline=newline, **kwargs) as f:
        return f.read()
    return str(sys.stdin.read())
  except (IOError, LookupError, UnicodeDecodeError, UnicodeError) as e:
    if continueOnError:
      if displayError:
        stderrWarningMsg(fileErrorMessage(filename, e))
        setSysExitRC(FILE_ERROR_RC)
      return None
    systemErrorExit(FILE_ERROR_RC, fileErrorMessage(filename, e))

# Write a file
def writeFile(filename, data, mode=DEFAULT_FILE_WRITE_MODE,
              continueOnError=False, displayError=True):
  try:
    if filename != '-':
      kwargs = setEncoding(mode, None)
      with open(os.path.expanduser(filename), mode, **kwargs) as f:
        f.write(data)
      return True
    GM.Globals[GM.STDOUT].get(GM.REDIRECT_MULTI_FD, sys.stdout).write(data)
    return True
  except (IOError, LookupError, UnicodeDecodeError, UnicodeError) as e:
    if continueOnError:
      if displayError:
        stderrErrorMsg(fileErrorMessage(filename, e))
      setSysExitRC(FILE_ERROR_RC)
      return False
    systemErrorExit(FILE_ERROR_RC, fileErrorMessage(filename, e))

# Write a file, return error
def writeFileReturnError(filename, data, mode=DEFAULT_FILE_WRITE_MODE):
  try:
    kwargs = {'encoding': GM.Globals[GM.SYS_ENCODING]} if 'b' not in mode else {}
    with open(os.path.expanduser(filename), mode, **kwargs) as f:
      f.write(data)
    return (True, None)
  except (IOError, LookupError, UnicodeDecodeError, UnicodeError) as e:
    return (False, e)

# Delete a file
def deleteFile(filename, continueOnError=False, displayError=True):
  if os.path.isfile(filename):
    try:
      os.remove(filename)
    except OSError as e:
      if continueOnError:
        if displayError:
          stderrWarningMsg(fileErrorMessage(filename, e))
        return
      systemErrorExit(FILE_ERROR_RC, fileErrorMessage(filename, e))

def getGDocSheetDataRetryWarning(entityValueList, errMsg, i=0, count=0):
  action = Act.Get()
  Act.Set(Act.RETRIEVE_DATA)
  stderrWarningMsg(formatKeyValueList(Ind.Spaces(),
                                      Ent.FormatEntityValueList(entityValueList)+[Act.NotPerformed(), errMsg, 'Retry', ''],
                                      currentCountNL(i, count)))
  Act.Set(action)

def getGDocSheetDataFailedExit(entityValueList, errMsg, i=0, count=0):
  Act.Set(Act.RETRIEVE_DATA)
  systemErrorExit(ACTION_FAILED_RC, formatKeyValueList(Ind.Spaces(),
                                                       Ent.FormatEntityValueList(entityValueList)+[Act.NotPerformed(), errMsg],
                                                       currentCountNL(i, count)))

GDOC_FORMAT_MIME_TYPES = {
  'gcsv': MIMETYPE_TEXT_CSV,
  'gdoc': MIMETYPE_TEXT_PLAIN,
  'ghtml': MIMETYPE_TEXT_HTML,
  }

# gdoc <EmailAddress> <DriveFileIDEntity>|<DriveFileNameEntity>
def getGDocData(gformat):
  mimeType = GDOC_FORMAT_MIME_TYPES[gformat]
  user = getEmailAddress()
  fileIdEntity = getDriveFileEntity(queryShortcutsOK=False)
  user, drive, jcount = _validateUserGetFileIDs(user, 0, 0, fileIdEntity)
  if not drive:
    sys.exit(GM.Globals[GM.SYSEXITRC])
  if jcount == 0:
    getGDocSheetDataFailedExit([Ent.USER, user], Msg.NO_ENTITIES_FOUND.format(Ent.Singular(Ent.DRIVE_FILE)))
  if jcount > 1:
    getGDocSheetDataFailedExit([Ent.USER, user], Msg.MULTIPLE_ENTITIES_FOUND.format(Ent.Plural(Ent.DRIVE_FILE), jcount, ','.join(fileIdEntity['list'])))
  fileId = fileIdEntity['list'][0]
  try:
    result = callGAPI(drive.files(), 'get',
                      throwReasons=GAPI.DRIVE_GET_THROW_REASONS,
                      fileId=fileId, fields='name,mimeType,exportLinks',
                      supportsAllDrives=True)
# Google Doc
    if 'exportLinks' in result:
      if mimeType not in result['exportLinks']:
        getGDocSheetDataFailedExit([Ent.USER, user, Ent.DRIVE_FILE, result['name']],
                                   Msg.INVALID_MIMETYPE.format(result['mimeType'], mimeType))
      f = TemporaryFile(mode='w+', encoding=UTF8)
      _, content = drive._http.request(uri=result['exportLinks'][mimeType], method='GET')
      f.write(content.decode(UTF8_SIG))
      f.seek(0)
      return f
# Drive File
    if result['mimeType'] != mimeType:
      getGDocSheetDataFailedExit([Ent.USER, user, Ent.DRIVE_FILE, result['name']],
                                 Msg.INVALID_MIMETYPE.format(result['mimeType'], mimeType))
    fb = TemporaryFile(mode='wb+')
    request = drive.files().get_media(fileId=fileId)
    downloader = googleapiclient.http.MediaIoBaseDownload(fb, request)
    done = False
    while not done:
      _, done = downloader.next_chunk()
    f = TemporaryFile(mode='w+', encoding=UTF8)
    fb.seek(0)
    f.write(fb.read().decode(UTF8_SIG))
    fb.close()
    f.seek(0)
    return f
  except GAPI.fileNotFound:
    getGDocSheetDataFailedExit([Ent.USER, user, Ent.DOCUMENT, fileId], Msg.DOES_NOT_EXIST)
  except (IOError, httplib2.HttpLib2Error, google.auth.exceptions.TransportError, RuntimeError) as e:
    if f:
      f.close()
    getGDocSheetDataFailedExit([Ent.USER, user, Ent.DOCUMENT, fileId], str(e))
  except (GAPI.serviceNotAvailable, GAPI.authError, GAPI.domainPolicy) as e:
    userDriveServiceNotEnabledWarning(user, str(e))
    sys.exit(GM.Globals[GM.SYSEXITRC])

HTML_TITLE_PATTERN = re.compile(r'.*<title>(.+)</title>')

# gsheet <EmailAddress> <DriveFileIDEntity>|<DriveFileNameEntity> <SheetEntity>
def getGSheetData():
  user = getEmailAddress()
  fileIdEntity = getDriveFileEntity(queryShortcutsOK=False)
  sheetEntity = getSheetEntity(False)
  user, drive, jcount = _validateUserGetFileIDs(user, 0, 0, fileIdEntity)
  if not drive:
    sys.exit(GM.Globals[GM.SYSEXITRC])
  if jcount == 0:
    getGDocSheetDataFailedExit([Ent.USER, user], Msg.NO_ENTITIES_FOUND.format(Ent.Singular(Ent.DRIVE_FILE)))
  if jcount > 1:
    getGDocSheetDataFailedExit([Ent.USER, user], Msg.MULTIPLE_ENTITIES_FOUND.format(Ent.Plural(Ent.DRIVE_FILE), jcount, ','.join(fileIdEntity['list'])))
  _, sheet = buildGAPIServiceObject(API.SHEETS, user)
  if not sheet:
    sys.exit(GM.Globals[GM.SYSEXITRC])
  fileId = fileIdEntity['list'][0]
  try:
    result = callGAPI(drive.files(), 'get',
                      throwReasons=GAPI.DRIVE_GET_THROW_REASONS,
                      fileId=fileId, fields='name,mimeType', supportsAllDrives=True)
    if result['mimeType'] != MIMETYPE_GA_SPREADSHEET:
      getGDocSheetDataFailedExit([Ent.USER, user, Ent.DRIVE_FILE, result['name']],
                                 Msg.INVALID_MIMETYPE.format(result['mimeType'], MIMETYPE_GA_SPREADSHEET))
    spreadsheet = callGAPI(sheet.spreadsheets(), 'get',
                           throwReasons=GAPI.SHEETS_ACCESS_THROW_REASONS,
                           spreadsheetId=fileId, fields='spreadsheetUrl,sheets(properties(sheetId,title))')
    sheetId = getSheetIdFromSheetEntity(spreadsheet, sheetEntity)
    if sheetId is None:
      getGDocSheetDataFailedExit([Ent.USER, user, Ent.SPREADSHEET, result['name'], sheetEntity['sheetType'], sheetEntity['sheetValue']], Msg.NOT_FOUND)
    spreadsheetUrl = f'{re.sub("/edit.*$", "/export", spreadsheet["spreadsheetUrl"])}?format=csv&id={fileId}&gid={sheetId}'
    f = TemporaryFile(mode='w+', encoding=UTF8)
    if GC.Values[GC.DEBUG_LEVEL] > 0:
      sys.stderr.write(f'Debug: spreadsheetUrl: {spreadsheetUrl}\n')
    triesLimit = 3
    for n in range(1, triesLimit+1):
      _, content = drive._http.request(uri=spreadsheetUrl, method='GET')
# Check for HTML error message instead of data
      if content[0:15] != b'<!DOCTYPE html>':
        break
      tg = HTML_TITLE_PATTERN.match(content[0:600].decode('utf-8'))
      errMsg = tg.group(1) if tg else 'Unknown error'
      getGDocSheetDataRetryWarning([Ent.USER, user, Ent.SPREADSHEET, result['name'], sheetEntity['sheetType'], sheetEntity['sheetValue']], errMsg, n, triesLimit)
      time.sleep(20)
    else:
      getGDocSheetDataFailedExit([Ent.USER, user, Ent.SPREADSHEET, result['name'], sheetEntity['sheetType'], sheetEntity['sheetValue']], errMsg)
    f.write(content.decode(UTF8_SIG))
    f.seek(0)
    return f
  except GAPI.fileNotFound:
    getGDocSheetDataFailedExit([Ent.USER, user, Ent.SPREADSHEET, fileId], Msg.DOES_NOT_EXIST)
  except (GAPI.notFound, GAPI.forbidden, GAPI.permissionDenied,
          GAPI.internalError, GAPI.insufficientFilePermissions, GAPI.badRequest,
          GAPI.invalid, GAPI.invalidArgument, GAPI.failedPrecondition) as e:
    getGDocSheetDataFailedExit([Ent.USER, user, Ent.SPREADSHEET, fileId, sheetEntity['sheetType'], sheetEntity['sheetValue']], str(e))
  except (IOError, httplib2.HttpLib2Error) as e:
    if f:
      f.close()
    getGDocSheetDataFailedExit([Ent.USER, user, Ent.SPREADSHEET, fileId, sheetEntity['sheetType'], sheetEntity['sheetValue']], str(e))
  except (GAPI.serviceNotAvailable, GAPI.authError, GAPI.domainPolicy) as e:
    userDriveServiceNotEnabledWarning(user, str(e))
    sys.exit(GM.Globals[GM.SYSEXITRC])


BUCKET_OBJECT_PATTERNS = [
  {'pattern': re.compile(r'https://storage.(?:googleapis|cloud.google).com/(.+?)/(.+)'), 'unquote': True},
  {'pattern': re.compile(r'gs://(.+?)/(.+)'), 'unquote': False},
  {'pattern': re.compile(r'(.+?)/(.+)'), 'unquote': False},
  ]

def getBucketObjectName():
  uri = getString(Cmd.OB_STRING)
  for pattern in BUCKET_OBJECT_PATTERNS:
    mg = re.search(pattern['pattern'], uri)
    if mg:
      bucket = mg.group(1)
      s_object = mg.group(2) if not pattern['unquote'] else unquote(mg.group(2))
      return (bucket, s_object, f'{bucket}/{s_object}')
  systemErrorExit(ACTION_NOT_PERFORMED_RC, f'Invalid <StorageBucketObjectName>: {uri}')

GCS_FORMAT_MIME_TYPES = {
  'gcscsv': MIMETYPE_TEXT_CSV,
  'gcsdoc': MIMETYPE_TEXT_PLAIN,
  'gcshtml': MIMETYPE_TEXT_HTML,
  }

# gcscsv|gcshtml|gcsdoc <StorageBucketObjectName>
def getStorageFileData(gcsformat, returnData=True):
  mimeType = GCS_FORMAT_MIME_TYPES[gcsformat]
  bucket, s_object, bucketObject = getBucketObjectName()
  s = buildGAPIObject(API.STORAGEREAD)
  try:
    result = callGAPI(s.objects(), 'get',
                      throwReasons=[GAPI.NOT_FOUND, GAPI.FORBIDDEN],
                      bucket=bucket, object=s_object, projection='noAcl', fields='contentType')
  except GAPI.notFound:
    entityDoesNotExistExit(Ent.CLOUD_STORAGE_FILE, bucketObject)
  except GAPI.forbidden as e:
    entityActionFailedExit([Ent.CLOUD_STORAGE_FILE, bucketObject], str(e))
  if result['contentType'] != mimeType:
    getGDocSheetDataFailedExit([Ent.CLOUD_STORAGE_FILE, bucketObject],
                               Msg.INVALID_MIMETYPE.format(result['contentType'], mimeType))
  fb = TemporaryFile(mode='wb+')
  try:
    request = s.objects().get_media(bucket=bucket, object=s_object)
    downloader = googleapiclient.http.MediaIoBaseDownload(fb, request)
    done = False
    while not done:
      _, done = downloader.next_chunk()
    fb.seek(0)
    if returnData:
      data = fb.read().decode(UTF8)
      fb.close()
      return data
    f = TemporaryFile(mode='w+', encoding=UTF8)
    f.write(fb.read().decode(UTF8_SIG))
    fb.close()
    f.seek(0)
    return f
  except googleapiclient.http.HttpError as e:
    mg = HTTP_ERROR_PATTERN.match(str(e))
    getGDocSheetDataFailedExit([Ent.CLOUD_STORAGE_FILE, bucketObject], mg.group(1) if mg else str(e))

# <CSVFileInput>
def openCSVFileReader(filename, fieldnames=None):
  filenameLower = filename.lower()
  if filenameLower == 'gsheet':
    f = getGSheetData()
    getCharSet()
  elif filenameLower in {'gcsv', 'gdoc'}:
    f = getGDocData(filenameLower)
    getCharSet()
  elif filenameLower in {'gcscsv', 'gcsdoc'}:
    f = getStorageFileData(filenameLower, False)
    getCharSet()
  else:
    encoding = getCharSet()
    f = openFile(filename, mode=DEFAULT_CSV_READ_MODE, encoding=encoding)
  if checkArgumentPresent('warnifnodata'):
    loc = f.tell()
    try:
      if not f.readline() or not f.readline():
        stderrWarningMsg(fileErrorMessage(filename, Msg.NO_CSV_FILE_DATA_FOUND))
        sys.exit(NO_ENTITIES_FOUND_RC)
      f.seek(loc)
    except (IOError, UnicodeDecodeError, UnicodeError) as e:
      systemErrorExit(FILE_ERROR_RC, fileErrorMessage(filename, e))
  if checkArgumentPresent('columndelimiter'):
    columnDelimiter = getCharacter()
  else:
    columnDelimiter = GC.Values[GC.CSV_INPUT_COLUMN_DELIMITER]
  if checkArgumentPresent('noescapechar'):
    noEscapeChar = getBoolean()
  else:
    noEscapeChar = GC.Values[GC.CSV_INPUT_NO_ESCAPE_CHAR]
  if checkArgumentPresent('quotechar'):
    quotechar = getCharacter()
  else:
    quotechar = GC.Values[GC.CSV_INPUT_QUOTE_CHAR]
  if not checkArgumentPresent('endcsv') and checkArgumentPresent('fields'):
    fieldnames = shlexSplitList(getString(Cmd.OB_FIELD_NAME_LIST))
  try:
    csvFile = csv.DictReader(f, fieldnames=fieldnames,
                             delimiter=columnDelimiter,
                             escapechar='\\' if not noEscapeChar else None,
                             quotechar=quotechar)
    return (f, csvFile, csvFile.fieldnames if csvFile.fieldnames is not None else [])
  except (csv.Error, UnicodeDecodeError, UnicodeError) as e:
    systemErrorExit(FILE_ERROR_RC, e)

def incrAPICallsRetryData(errMsg, delta):
  GM.Globals[GM.API_CALLS_RETRY_DATA].setdefault(errMsg, [0, 0.0])
  GM.Globals[GM.API_CALLS_RETRY_DATA][errMsg][0] += 1
  GM.Globals[GM.API_CALLS_RETRY_DATA][errMsg][1] += delta

def initAPICallsRateCheck():
  GM.Globals[GM.RATE_CHECK_COUNT] = 0
  GM.Globals[GM.RATE_CHECK_START] = time.time()

def checkAPICallsRate():
  GM.Globals[GM.RATE_CHECK_COUNT] += 1
  if GM.Globals[GM.RATE_CHECK_COUNT] >= GC.Values[GC.API_CALLS_RATE_LIMIT]:
    current = time.time()
    delta = int(current-GM.Globals[GM.RATE_CHECK_START])
    if 0 <= delta < 60:
      delta = (60-delta)+3
      error_message = f'API calls per 60 seconds limit {GC.Values[GC.API_CALLS_RATE_LIMIT]} exceeded'
      writeStderr(f'{WARNING_PREFIX}{error_message}: Backing off: {delta} seconds\n')
      flushStderr()
      time.sleep(delta)
      if GC.Values[GC.SHOW_API_CALLS_RETRY_DATA]:
        incrAPICallsRetryData(error_message, delta)
      GM.Globals[GM.RATE_CHECK_START] = time.time()
    else:
      GM.Globals[GM.RATE_CHECK_START] = current
    GM.Globals[GM.RATE_CHECK_COUNT] = 0

def openGAMCommandLog(Globals, name):
  try:
    Globals[GM.CMDLOG_LOGGER] = logging.getLogger(name)
    Globals[GM.CMDLOG_LOGGER].setLevel(logging.INFO)
    Globals[GM.CMDLOG_HANDLER] = RotatingFileHandler(GC.Values[GC.CMDLOG],
                                                     maxBytes=1024*GC.Values[GC.CMDLOG_MAX_KILO_BYTES],
                                                     backupCount=GC.Values[GC.CMDLOG_MAX_BACKUPS],
                                                     encoding=GC.Values[GC.CHARSET])
    Globals[GM.CMDLOG_LOGGER].addHandler(Globals[GM.CMDLOG_HANDLER])
  except Exception as e:
    Globals[GM.CMDLOG_LOGGER] = None
    systemErrorExit(CONFIG_ERROR_RC, Msg.LOGGING_INITIALIZATION_ERROR.format(str(e)))

def writeGAMCommandLog(Globals, logCmd, sysRC):
  Globals[GM.CMDLOG_LOGGER].info(f'{currentISOformatTimeStamp()},{sysRC},{logCmd}')

def closeGAMCommandLog(Globals):
  try:
    Globals[GM.CMDLOG_HANDLER].flush()
    Globals[GM.CMDLOG_HANDLER].close()
    Globals[GM.CMDLOG_LOGGER].removeHandler(Globals[GM.CMDLOG_HANDLER])
  except Exception:
    pass
  Globals[GM.CMDLOG_LOGGER] = None

# Set global variables from config file
# Return True if there are additional commands on the command line
def SetGlobalVariables():

  def _stringInQuotes(value):
    return (len(value) > 1) and (((value.startswith('"') and value.endswith('"'))) or ((value.startswith("'") and value.endswith("'"))))

  def _stripStringQuotes(value):
    if _stringInQuotes(value):
      return value[1:-1]
    return value

  def _quoteStringIfLeadingTrailingBlanks(value):
    if not value:
      return "''"
    if _stringInQuotes(value):
      return value
    if (value[0] != ' ') and (value[-1] != ' '):
      return value
    return f"'{value}'"

  def _getDefault(itemName, itemEntry, oldGamPath):
    if GC.VAR_SIGFILE in itemEntry:
      GC.Defaults[itemName] = itemEntry[GC.VAR_SFFT][os.path.isfile(os.path.join(oldGamPath, itemEntry[GC.VAR_SIGFILE]))]
    elif GC.VAR_ENVVAR in itemEntry:
      value = os.environ.get(itemEntry[GC.VAR_ENVVAR], GC.Defaults[itemName])
      if itemEntry[GC.VAR_TYPE] in [GC.TYPE_INTEGER, GC.TYPE_FLOAT]:
        try:
          number = int(value) if itemEntry[GC.VAR_TYPE] == GC.TYPE_INTEGER else float(value)
          minVal, maxVal = itemEntry[GC.VAR_LIMITS]
          if (minVal is not None) and (number < minVal):
            number = minVal
          elif (maxVal is not None) and (number > maxVal):
            number = maxVal
        except ValueError:
          number = GC.Defaults[itemName]
        value = str(number)
      elif itemEntry[GC.VAR_TYPE] == GC.TYPE_STRING:
        value = _quoteStringIfLeadingTrailingBlanks(value)
      GC.Defaults[itemName] = value

  def _selectSection():
    value = getString(Cmd.OB_SECTION_NAME, minLen=0)
    if (not value) or (value.upper() == configparser.DEFAULTSECT):
      return configparser.DEFAULTSECT
    if GM.Globals[GM.PARSER].has_section(value):
      return value
    Cmd.Backup()
    usageErrorExit(formatKeyValueList('', [Ent.Singular(Ent.SECTION), value, Msg.NOT_FOUND], ''))

  def _showSections():
    printKeyValueList([Ent.Singular(Ent.CONFIG_FILE), GM.Globals[GM.GAM_CFG_FILE]])
    Ind.Increment()
    for section in [configparser.DEFAULTSECT]+sorted(GM.Globals[GM.PARSER].sections()):
      printKeyValueList([f'{section}{" *" if section == sectionName else ""}'])
    Ind.Decrement()

  def _checkMakeDir(itemName):
    if not os.path.isdir(GC.Defaults[itemName]):
      try:
        os.makedirs(GC.Defaults[itemName])
        printKeyValueList([Act.PerformedName(Act.CREATE), GC.Defaults[itemName]])
      except OSError as e:
        if not os.path.isdir(GC.Defaults[itemName]):
          systemErrorExit(FILE_ERROR_RC, e)

  def _copyCfgFile(srcFile, targetDir, oldGamPath):
    if (not srcFile) or os.path.isabs(srcFile):
      return
    dstFile = os.path.join(GC.Defaults[targetDir], srcFile)
    if os.path.isfile(dstFile):
      return
    srcFile = os.path.join(oldGamPath, srcFile)
    if not os.path.isfile(srcFile):
      return
    data = readFile(srcFile, continueOnError=True, displayError=False)
    if (data is not None) and writeFile(dstFile, data, continueOnError=True):
      printKeyValueList([Act.PerformedName(Act.COPY), srcFile, Msg.TO, dstFile])

  def _printValueError(sectionName, itemName, value, errMessage, sysRC=CONFIG_ERROR_RC):
    kvlMsg = formatKeyValueList('',
                                [Ent.Singular(Ent.CONFIG_FILE), GM.Globals[GM.GAM_CFG_FILE],
                                 Ent.Singular(Ent.SECTION), sectionName,
                                 Ent.Singular(Ent.ITEM), itemName,
                                 Ent.Singular(Ent.VALUE), value,
                                 errMessage],
                                '')
    if sysRC != 0:
      status['errors'] = True
      printErrorMessage(sysRC, kvlMsg)
    else:
      writeStderr(formatKeyValueList(Ind.Spaces(), [WARNING, kvlMsg], '\n'))

  def _getCfgBoolean(sectionName, itemName):
    value = GM.Globals[GM.PARSER].get(sectionName, itemName).lower()
    if value in TRUE_VALUES:
      return True
    if value in FALSE_VALUES:
      return False
    _printValueError(sectionName, itemName, value, f'{Msg.EXPECTED}: {formatChoiceList(TRUE_FALSE)}')
    return False

  def _getCfgCharacter(sectionName, itemName):
    value = codecs.escape_decode(bytes(_stripStringQuotes(GM.Globals[GM.PARSER].get(sectionName, itemName)), UTF8))[0].decode(UTF8)
    if not value and (itemName == 'csv_output_field_delimiter'):
      return ' '
    if not value and (itemName in {'csv_input_escape_char', 'csv_output_escape_char'}):
      return None
    if len(value) == 1:
      return value
    _printValueError(sectionName, itemName, f'"{value}"', f'{Msg.EXPECTED}: {integerLimits(1, 1, Msg.STRING_LENGTH)}')
    return ''

  def _getCfgChoice(sectionName, itemName):
    value = _stripStringQuotes(GM.Globals[GM.PARSER].get(sectionName, itemName)).lower()
    choices = GC.VAR_INFO[itemName][GC.VAR_CHOICES]
    if value in choices:
      return choices[value]
    _printValueError(sectionName, itemName, f'"{value}"', f'{Msg.EXPECTED}: {",".join(choices)}')
    return ''

  def _getCfgLocale(sectionName, itemName):
    value = _stripStringQuotes(GM.Globals[GM.PARSER].get(sectionName, itemName)).lower().replace('_', '-')
    if value in LOCALE_CODES_MAP:
      return LOCALE_CODES_MAP[value]
    _printValueError(sectionName, itemName, f'"{value}"', f'{Msg.EXPECTED}: {",".join(LOCALE_CODES_MAP)}')
    return ''

  def _getCfgNumber(sectionName, itemName):
    value = GM.Globals[GM.PARSER].get(sectionName, itemName)
    minVal, maxVal = GC.VAR_INFO[itemName][GC.VAR_LIMITS]
    try:
      number = int(value) if GC.VAR_INFO[itemName][GC.VAR_TYPE] == GC.TYPE_INTEGER else float(value)
      if ((minVal is None) or (number >= minVal)) and ((maxVal is None) or (number <= maxVal)):
        return number
      if (minVal is not None) and (number < minVal):
        number = minVal
      else:
        number = maxVal
      _printValueError(sectionName, itemName, value, f'{Msg.EXPECTED}: {integerLimits(minVal, maxVal)}, {Msg.USED}: {number}', sysRC=0)
      return number
    except ValueError:
      pass
    _printValueError(sectionName, itemName, value, f'{Msg.EXPECTED}: {integerLimits(minVal, maxVal)}')
    return 0

  def _getCfgHeaderFilter(sectionName, itemName):
    value = GM.Globals[GM.PARSER].get(sectionName, itemName)
    headerFilters = []
    if not value or (len(value) == 2 and _stringInQuotes(value)):
      return headerFilters
    splitStatus, filters = shlexSplitListStatus(value)
    if splitStatus:
      for filterStr in filters:
        try:
          headerFilters.append(re.compile(filterStr, re.IGNORECASE))
        except re.error as e:
          _printValueError(sectionName, itemName, f'"{filterStr}"', f'{Msg.INVALID_RE}: {e}')
    else:
      _printValueError(sectionName, itemName, f'"{value}"', f'{Msg.INVALID_LIST}: {filters}')
    return headerFilters

  def _getCfgHeaderFilterFromForce(sectionName, itemName):
    headerFilters = []
    for filterStr in GC.Values[itemName]:
      try:
        headerFilters.append(re.compile(fr'^{filterStr}$'))
      except re.error as e:
        _printValueError(sectionName, itemName, f'"{filterStr}"', f'{Msg.INVALID_RE}: {e}')
    return headerFilters

  ROW_FILTER_ANY_ALL_PATTERN = re.compile(r'^(any:|all:)(.+)$', re.IGNORECASE)
  ROW_FILTER_COMP_PATTERN = re.compile(r'^(date|time|count|length)\s*([<>]=?|=|!=)(.+)$', re.IGNORECASE)
  ROW_FILTER_RANGE_PATTERN = re.compile(r'^(daterange|timerange|countrange|lengthrange)(=|!=)(\S+)/(\S+)$', re.IGNORECASE)
  ROW_FILTER_TIMEOFDAYRANGE_PATTERN = re.compile(r'^(timeofdayrange)(=|!=)(\d\d):(\d\d)/(\d\d):(\d\d)$', re.IGNORECASE)
  ROW_FILTER_BOOL_PATTERN = re.compile(r'^(boolean):(.+)$', re.IGNORECASE)
  ROW_FILTER_TEXT_PATTERN = re.compile(r'^(text)([<>]=?|=|!=)(.*)$', re.IGNORECASE)
  ROW_FILTER_TEXTRANGE_PATTERN = re.compile(r'^(textrange)(=|!=)(.*)/(.*)$', re.IGNORECASE)
  ROW_FILTER_RE_PATTERN = re.compile(r'^(regex|regexcs|notregex|notregexcs):(.*)$', re.IGNORECASE)
  ROW_FILTER_DATA_PATTERN = re.compile(r'^(data|notdata):(list|file|csvfile) +(.+)$', re.IGNORECASE)
  REGEX_CHARS = '^$*+|$[{('

  def _getCfgRowFilter(sectionName, itemName):
    value = GM.Globals[GM.PARSER].get(sectionName, itemName)
    rowFilters = []
    if not value:
      return rowFilters
    if value.startswith('{'):
      try:
        filterDict = json.loads(value.encode('unicode-escape').decode(UTF8))
      except (IndexError, KeyError, SyntaxError, TypeError, ValueError) as e:
        _printValueError(sectionName, itemName, f'"{value}"', f'{Msg.FAILED_TO_PARSE_AS_JSON}: {str(e)}')
        return rowFilters
    else:
      filterDict = {}
      status, filterList = shlexSplitListStatus(value)
      if not status:
        _printValueError(sectionName, itemName, f'"{value}"', f'{Msg.FAILED_TO_PARSE_AS_LIST}: {str(filterList)}')
        return rowFilters
      for filterVal in filterList:
        if not filterVal:
          continue
        try:
          filterTokens = shlexSplitList(filterVal, ':')
          column = filterTokens[0]
          filterStr = ':'.join(filterTokens[1:])
        except ValueError:
          _printValueError(sectionName, itemName, f'"{filterVal}"', f'{Msg.EXPECTED}: column:filter')
          continue
        filterDict[column] = filterStr
    for column, filterStr in filterDict.items():
      for c in REGEX_CHARS:
        if c in column:
          columnPat = column
          break
      else:
        columnPat = f'^{column}$'
      try:
        columnPat = re.compile(columnPat, re.IGNORECASE)
      except re.error as e:
        _printValueError(sectionName, itemName, f'"{column}"', f'{Msg.INVALID_RE}: {e}')
        continue
      anyMatch = True
      mg = ROW_FILTER_ANY_ALL_PATTERN.match(filterStr)
      if mg:
        anyMatch = mg.group(1).lower() == 'any:'
        filterStr = mg.group(2)
      mg = ROW_FILTER_COMP_PATTERN.match(filterStr)
      if mg:
        filterType = mg.group(1).lower()
        if filterType in {'date', 'time'}:
          if filterType == 'date':
            valid, filterValue = getRowFilterDateOrDeltaFromNow(mg.group(3))
          else:
            valid, filterValue = getRowFilterTimeOrDeltaFromNow(mg.group(3))
          if valid:
            rowFilters.append((columnPat, anyMatch, filterType, mg.group(2), filterValue))
          else:
            _printValueError(sectionName, itemName, f'"{column}": "{filterStr}"', f'{Msg.EXPECTED}: {filterValue}')
        else: # filterType in {'count', 'length'}:
          if mg.group(3).isdigit():
            rowFilters.append((columnPat, anyMatch, filterType, mg.group(2), int(mg.group(3))))
          else:
            _printValueError(sectionName, itemName, f'"{column}": "{filterStr}"', f'{Msg.EXPECTED}: <Number>')
        continue
      mg = ROW_FILTER_TEXT_PATTERN.match(filterStr)
      if mg:
        filterType = mg.group(1).lower()
        rowFilters.append((columnPat, anyMatch, filterType, mg.group(2), mg.group(3)))
        continue
      mg = ROW_FILTER_TEXTRANGE_PATTERN.match(filterStr)
      if mg:
        filterType = mg.group(1).lower()
        rowFilters.append((columnPat, anyMatch, filterType, mg.group(2), mg.group(3), mg.group(4)))
        continue
      mg = ROW_FILTER_RANGE_PATTERN.match(filterStr)
      if mg:
        filterType = mg.group(1).lower()
        if filterType in {'daterange', 'timerange'}:
          if filterType == 'daterange':
            valid1, filterValue1 = getRowFilterDateOrDeltaFromNow(mg.group(3))
            valid2, filterValue2 = getRowFilterDateOrDeltaFromNow(mg.group(4))
          else:
            valid1, filterValue1 = getRowFilterTimeOrDeltaFromNow(mg.group(3))
            valid2, filterValue2 = getRowFilterTimeOrDeltaFromNow(mg.group(4))
          if valid1 and valid2:
            rowFilters.append((columnPat, anyMatch, filterType, mg.group(2), filterValue1, filterValue2))
          else:
            _printValueError(sectionName, itemName, f'"{column}": "{filterStr}"', f'{Msg.EXPECTED}: {filterValue1}/{filterValue2}')
        else: #countrange|lengthrange
          if mg.group(3).isdigit() and mg.group(4).isdigit():
            rowFilters.append((columnPat, anyMatch, filterType, mg.group(2), int(mg.group(3)), int(mg.group(4))))
          else:
            _printValueError(sectionName, itemName, f'"{column}": "{filterStr}"', f'{Msg.EXPECTED}: <Number>/<Number>')
        continue
      mg = ROW_FILTER_TIMEOFDAYRANGE_PATTERN.match(filterStr)
      if mg:
        filterType = mg.group(1).lower()
        startHour = int(mg.group(3))
        startMinute = int(mg.group(4))
        endHour = int(mg.group(5))
        endMinute = int(mg.group(6))
        if startHour > 23 or startMinute > 59 or endHour > 23 or endMinute > 59 or \
           endHour < startHour or (endHour == startHour and endMinute < startMinute):
          Cmd.Backup()
          usageErrorExit(Msg.INVALID_TIMEOFDAY_RANGE.format(f'{startHour:02d}:{startMinute:02d}', f'{endHour:02d}:{endMinute:02d}'))
        rowFilters.append((columnPat, anyMatch, filterType, mg.group(2), f'{startHour:02d}:{startMinute:02d}', f'{endHour:02d}:{endMinute:02d}'))
        continue
      mg = ROW_FILTER_BOOL_PATTERN.match(filterStr)
      if mg:
        filterType = mg.group(1).lower()
        filterValue = mg.group(2).lower()
        if filterValue in TRUE_VALUES:
          rowFilters.append((columnPat, anyMatch, filterType, True))
        elif filterValue in FALSE_VALUES:
          rowFilters.append((columnPat, anyMatch, filterType, False))
        else:
          _printValueError(sectionName, itemName, f'"{column}": "{filterStr}"', f'{Msg.EXPECTED}: <Boolean>')
        continue
      mg = ROW_FILTER_RE_PATTERN.match(filterStr)
      if mg:
        filterType = mg.group(1).lower()
        try:
          if filterType.endswith('cs'):
            filterType = filterType[0:-2]
            flags = 0
          else:
            flags = re.IGNORECASE
          rowFilters.append((columnPat, anyMatch, filterType, re.compile(mg.group(2), flags)))
        except re.error as e:
          _printValueError(sectionName, itemName, f'"{column}": "{filterStr}"', f'{Msg.INVALID_RE}: {e}')
        continue
      mg = ROW_FILTER_DATA_PATTERN.match(filterStr)
      if mg:
        filterType = mg.group(1).lower()
        filterSubType = mg.group(2).lower()
        if filterSubType == 'list':
          rowFilters.append((columnPat, anyMatch, filterType, set(shlexSplitList(mg.group(3)))))
          continue
        Cmd.MergeArguments(shlexSplitList(mg.group(3), ' '))
        if filterSubType == 'file':
          rowFilters.append((columnPat, anyMatch, filterType, getEntitiesFromFile(False, returnSet=True)))
        else: #elif filterSubType == 'csvfile':
          rowFilters.append((columnPat, anyMatch, filterType, getEntitiesFromCSVFile(False, returnSet=True)))
        Cmd.RestoreArguments()
        continue
      _printValueError(sectionName, itemName, f'"{column}": "{filterStr}"', f'{Msg.EXPECTED}: <RowValueFilter>')
    return rowFilters

  def _getCfgSection(sectionName, itemName):
    value = _stripStringQuotes(GM.Globals[GM.PARSER].get(sectionName, itemName))
    if (not value) or (value.upper() == configparser.DEFAULTSECT):
      return configparser.DEFAULTSECT
    if GM.Globals[GM.PARSER].has_section(value):
      return value
    _printValueError(sectionName, itemName, value, Msg.NOT_FOUND)
    return configparser.DEFAULTSECT

  def _getCfgPassword(sectionName, itemName):
    value = GM.Globals[GM.PARSER].get(sectionName, itemName)
    if isinstance(value, bytes):
      return value
    value = _stripStringQuotes(value)
    if value.startswith("b'") and value.endswith("'"):
      return bytes(value[2:-1], UTF8)
    if value:
      return value
    return ''

  def _validateLicenseSKUs(sectionName, itemName, skuList):
    GM.Globals[GM.LICENSE_SKUS] = []
    for sku in skuList.split(','):
      if '/' not in sku:
        productId, sku = SKU.getProductAndSKU(sku)
        if not productId:
          _printValueError(sectionName, itemName, sku, f'{Msg.EXPECTED}: {",".join(SKU.getSortedSKUList())}')
      else:
        (productId, sku) = sku.split('/')
      if (productId, sku) not in GM.Globals[GM.LICENSE_SKUS]:
        GM.Globals[GM.LICENSE_SKUS].append((productId, sku))

  def _getCfgString(sectionName, itemName):
    value = _stripStringQuotes(GM.Globals[GM.PARSER].get(sectionName, itemName))
    if itemName == GC.DOMAIN:
      value = value.strip()
    minLen, maxLen = GC.VAR_INFO[itemName].get(GC.VAR_LIMITS, (None, None))
    if ((minLen is None) or (len(value) >= minLen)) and ((maxLen is None) or (len(value) <= maxLen)):
      if itemName == GC.LICENSE_SKUS and value:
        _validateLicenseSKUs(sectionName, itemName, value)
      return value
    _printValueError(sectionName, itemName, f'"{value}"', f'{Msg.EXPECTED}: {integerLimits(minLen, maxLen, Msg.STRING_LENGTH)}')
    return ''

  def _getCfgStringList(sectionName, itemName):
    value = GM.Globals[GM.PARSER].get(sectionName, itemName)
    stringlist = []
    if not value or (len(value) == 2 and _stringInQuotes(value)):
      return stringlist
    splitStatus, stringlist = shlexSplitListStatus(value)
    if not splitStatus:
      _printValueError(sectionName, itemName, f'"{value}"', f'{Msg.INVALID_LIST}: {stringlist}')
    return stringlist

  def _getCfgTimezone(sectionName, itemName):
    value = _stripStringQuotes(GM.Globals[GM.PARSER].get(sectionName, itemName).lower())
    if value == 'utc':
      GM.Globals[GM.CONVERT_TO_LOCAL_TIME] = False
      return iso8601.UTC
    GM.Globals[GM.CONVERT_TO_LOCAL_TIME] = True
    if value == 'local':
      return iso8601.Local
    try:
      return iso8601.parse_timezone_str(value)
    except (iso8601.ParseError, OverflowError):
      _printValueError(sectionName, itemName, value, f'{Msg.EXPECTED}: {TIMEZONE_FORMAT_REQUIRED}')
      GM.Globals[GM.CONVERT_TO_LOCAL_TIME] = False
      return iso8601.UTC

  def _getCfgDirectory(sectionName, itemName):
    dirPath = os.path.expanduser(_stripStringQuotes(GM.Globals[GM.PARSER].get(sectionName, itemName)))
    if (not dirPath) and (itemName in {GC.GMAIL_CSE_INCERT_DIR, GC.GMAIL_CSE_INKEY_DIR}):
      return dirPath
    if (not dirPath) or (not os.path.isabs(dirPath) and dirPath != '.'):
      if (sectionName != configparser.DEFAULTSECT) and (GM.Globals[GM.PARSER].has_option(sectionName, itemName)):
        dirPath = os.path.join(os.path.expanduser(_stripStringQuotes(GM.Globals[GM.PARSER].get(configparser.DEFAULTSECT, itemName))), dirPath)
      if not os.path.isabs(dirPath):
        dirPath = os.path.join(GM.Globals[GM.GAM_CFG_PATH], dirPath)
    return dirPath

  def _getCfgFile(sectionName, itemName):
    value = os.path.expanduser(_stripStringQuotes(GM.Globals[GM.PARSER].get(sectionName, itemName)))
    if value and not os.path.isabs(value):
      value = os.path.expanduser(os.path.join(_getCfgDirectory(sectionName, GC.CONFIG_DIR), value))
    elif not value and itemName == GC.CACERTS_PEM:
      if hasattr(sys, '_MEIPASS'):
        value = os.path.join(sys._MEIPASS, GC.FN_CACERTS_PEM) #pylint: disable=no-member
      else:
        value = os.path.join(GM.Globals[GM.GAM_PATH], GC.FN_CACERTS_PEM)
    return value

  def _readGamCfgFile(config, fileName):
    try:
      with open(fileName, DEFAULT_FILE_READ_MODE, encoding=GM.Globals[GM.SYS_ENCODING]) as f:
        config.read_file(f)
    except (configparser.DuplicateOptionError, configparser.DuplicateSectionError,
            configparser.MissingSectionHeaderError, configparser.ParsingError) as e:
      systemErrorExit(CONFIG_ERROR_RC, formatKeyValueList('',
                                                          [Ent.Singular(Ent.CONFIG_FILE), fileName,
                                                           Msg.INVALID, str(e)],
                                                          ''))
    except IOError as e:
      systemErrorExit(FILE_ERROR_RC, fileErrorMessage(fileName, e, Ent.CONFIG_FILE))

  def _writeGamCfgFile(config, fileName, action):
    GM.Globals[GM.SECTION] = None # No need to save section for inner gams
    try:
      with open(fileName, DEFAULT_FILE_WRITE_MODE, encoding=GM.Globals[GM.SYS_ENCODING]) as f:
        config.write(f)
      printKeyValueList([Ent.Singular(Ent.CONFIG_FILE), fileName, Act.PerformedName(action)])
    except IOError as e:
      stderrErrorMsg(fileErrorMessage(fileName, e, Ent.CONFIG_FILE))

  def _verifyValues(sectionName, inputFilterSectionName, outputFilterSectionName):
    printKeyValueList([Ent.Singular(Ent.SECTION), sectionName]) # Do not use printEntity
    Ind.Increment()
    for itemName, itemEntry in GC.VAR_INFO.items():
      sectName = sectionName
      if itemName in GC.CSV_INPUT_ROW_FILTER_ITEMS:
        if inputFilterSectionName:
          sectName = inputFilterSectionName
      elif itemName in GC.CSV_OUTPUT_ROW_FILTER_ITEMS:
        if outputFilterSectionName:
          sectName = outputFilterSectionName
      cfgValue = GM.Globals[GM.PARSER].get(sectName, itemName)
      varType = itemEntry[GC.VAR_TYPE]
      if varType == GC.TYPE_CHOICE:
        for choice, value in itemEntry[GC.VAR_CHOICES].items():
          if cfgValue == value:
            cfgValue = choice
            break
      elif varType not in [GC.TYPE_BOOLEAN, GC.TYPE_INTEGER, GC.TYPE_FLOAT, GC.TYPE_PASSWORD]:
        cfgValue = _quoteStringIfLeadingTrailingBlanks(cfgValue)
      if varType == GC.TYPE_FILE:
        expdValue = _getCfgFile(sectName, itemName)
        if cfgValue not in ("''", expdValue):
          cfgValue = f'{cfgValue} ; {expdValue}'
      elif varType == GC.TYPE_DIRECTORY:
        expdValue = _getCfgDirectory(sectName, itemName)
        if cfgValue not in ("''", expdValue):
          cfgValue = f'{cfgValue} ; {expdValue}'
      elif (itemName == GC.SECTION) and (sectName != configparser.DEFAULTSECT):
        continue
      printLine(f'{Ind.Spaces()}{itemName} = {cfgValue}')
    Ind.Decrement()

  def _chkCfgDirectories(sectionName):
    for itemName, itemEntry in GC.VAR_INFO.items():
      if itemEntry[GC.VAR_TYPE] == GC.TYPE_DIRECTORY:
        dirPath = GC.Values[itemName]
        if (not dirPath) and (itemName in {GC.GMAIL_CSE_INCERT_DIR, GC.GMAIL_CSE_INKEY_DIR}):
          return
        if (itemName != GC.CACHE_DIR or not GC.Values[GC.NO_CACHE]) and not os.path.isdir(dirPath):
          writeStderr(formatKeyValueList(WARNING_PREFIX,
                                         [Ent.Singular(Ent.CONFIG_FILE), GM.Globals[GM.GAM_CFG_FILE],
                                          Ent.Singular(Ent.SECTION), sectionName,
                                          Ent.Singular(Ent.ITEM), itemName,
                                          Ent.Singular(Ent.VALUE), dirPath,
                                          Msg.INVALID_PATH],
                                         '\n'))

  def _chkCfgFiles(sectionName):
    for itemName, itemEntry in GC.VAR_INFO.items():
      if itemEntry[GC.VAR_TYPE] == GC.TYPE_FILE:
        fileName = GC.Values[itemName]
        if (not fileName) and (itemName in {GC.EXTRA_ARGS, GC.CMDLOG}):
          continue
        if itemName == GC.CLIENT_SECRETS_JSON: # Added 6.57.01
          continue
        if GC.Values[GC.ENABLE_DASA] and itemName == GC.OAUTH2_TXT:
          continue
        if not os.path.isfile(fileName):
          writeStderr(formatKeyValueList([WARNING_PREFIX, ERROR_PREFIX][itemName == GC.CACERTS_PEM],
                                         [Ent.Singular(Ent.CONFIG_FILE), GM.Globals[GM.GAM_CFG_FILE],
                                          Ent.Singular(Ent.SECTION), sectionName,
                                          Ent.Singular(Ent.ITEM), itemName,
                                          Ent.Singular(Ent.VALUE), fileName,
                                          Msg.NOT_FOUND],
                                         '\n'))
          if itemName == GC.CACERTS_PEM:
            status['errors'] = True
        elif not os.access(fileName, itemEntry[GC.VAR_ACCESS]):
          if itemEntry[GC.VAR_ACCESS] == os.R_OK | os.W_OK:
            accessMsg = Msg.NEED_READ_WRITE_ACCESS
          elif itemEntry[GC.VAR_ACCESS] == os.R_OK:
            accessMsg = Msg.NEED_READ_ACCESS
          else:
            accessMsg = Msg.NEED_WRITE_ACCESS
          writeStderr(formatKeyValueList(ERROR_PREFIX,
                                         [Ent.Singular(Ent.CONFIG_FILE), GM.Globals[GM.GAM_CFG_FILE],
                                          Ent.Singular(Ent.SECTION), sectionName,
                                          Ent.Singular(Ent.ITEM), itemName,
                                          Ent.Singular(Ent.VALUE), fileName,
                                          accessMsg],
                                         '\n'))
          status['errors'] = True

  def _setCSVFile(fileName, mode, encoding, writeHeader, multi):
    if fileName != '-':
      fileName = setFilePath(fileName)
    GM.Globals[GM.CSVFILE][GM.REDIRECT_NAME] = fileName
    GM.Globals[GM.CSVFILE][GM.REDIRECT_MODE] = mode
    GM.Globals[GM.CSVFILE][GM.REDIRECT_ENCODING] = encoding
    GM.Globals[GM.CSVFILE][GM.REDIRECT_WRITE_HEADER] = writeHeader
    GM.Globals[GM.CSVFILE][GM.REDIRECT_MULTIPROCESS] = multi
    GM.Globals[GM.CSVFILE][GM.REDIRECT_QUEUE] = None

  def _setSTDFile(stdtype, fileName, mode, multi):
    if stdtype == GM.STDOUT:
      GM.Globals[GM.SAVED_STDOUT] = None
    GM.Globals[stdtype][GM.REDIRECT_STD] = False
    if fileName == 'null':
      GM.Globals[stdtype][GM.REDIRECT_FD] = open(os.devnull, mode, encoding=UTF8)
    elif fileName == '-':
      GM.Globals[stdtype][GM.REDIRECT_STD] = True
      if stdtype == GM.STDOUT:
        GM.Globals[stdtype][GM.REDIRECT_FD] = os.fdopen(os.dup(sys.stdout.fileno()), mode, encoding=GM.Globals[GM.SYS_ENCODING])
      else:
        GM.Globals[stdtype][GM.REDIRECT_FD] = os.fdopen(os.dup(sys.stderr.fileno()), mode, encoding=GM.Globals[GM.SYS_ENCODING])
    else:
      fileName = setFilePath(fileName)
      if multi and mode == DEFAULT_FILE_WRITE_MODE:
        deleteFile(fileName)
        mode = DEFAULT_FILE_APPEND_MODE
      GM.Globals[stdtype][GM.REDIRECT_FD] = openFile(fileName, mode)
    GM.Globals[stdtype][GM.REDIRECT_MULTI_FD] = GM.Globals[stdtype][GM.REDIRECT_FD] if not multi else StringIOobject()
    if (stdtype == GM.STDOUT) and (GC.Values[GC.DEBUG_LEVEL] > 0):
      GM.Globals[GM.SAVED_STDOUT] = sys.stdout
      sys.stdout = GM.Globals[stdtype][GM.REDIRECT_MULTI_FD]
    GM.Globals[stdtype][GM.REDIRECT_NAME] = fileName
    GM.Globals[stdtype][GM.REDIRECT_MODE] = mode
    GM.Globals[stdtype][GM.REDIRECT_MULTIPROCESS] = multi
    GM.Globals[stdtype][GM.REDIRECT_QUEUE] = 'stdout' if stdtype == GM.STDOUT else 'stderr'

  MULTIPROCESS_EXIT_COMP_PATTERN = re.compile(r'^rc([<>]=?|=|!=)(.+)$', re.IGNORECASE)
  MULTIPROCESS_EXIT_RANGE_PATTERN = re.compile(r'^rcrange(=|!=)(\S+)/(\S+)$', re.IGNORECASE)

  def _setMultiprocessExit():
    rcStr = getString(Cmd.OB_STRING)
    mg = MULTIPROCESS_EXIT_COMP_PATTERN.match(rcStr)
    if mg:
      if not mg.group(2).isdigit():
        usageErrorExit(f'{Msg.EXPECTED}: rc<Operator><Value>')
      GM.Globals[GM.MULTIPROCESS_EXIT_CONDITION] = {'comp': mg.group(1), 'value': int(mg.group(2))}
      return
    mg = MULTIPROCESS_EXIT_RANGE_PATTERN.match(rcStr)
    if mg:
      if not mg.group(2).isdigit() or not  mg.group(3).isdigit():
        usageErrorExit(f'{Msg.EXPECTED}: rcrange<Operator><Value>/Value>')
      GM.Globals[GM.MULTIPROCESS_EXIT_CONDITION] = {'range': mg.group(1), 'low': int(mg.group(2)), 'high': int(mg.group(3))}
      return
    usageErrorExit(f'{Msg.EXPECTED}: (rc<Operator><Value>)|(rcrange<Operator><Value>/Value>)')

  if not GM.Globals[GM.PARSER]:
    homePath = os.path.expanduser('~')
    GM.Globals[GM.GAM_CFG_PATH] = os.environ.get(EV_GAMCFGDIR, None)
    if GM.Globals[GM.GAM_CFG_PATH]:
      GM.Globals[GM.GAM_CFG_PATH] = os.path.expanduser(GM.Globals[GM.GAM_CFG_PATH])
    else:
      GM.Globals[GM.GAM_CFG_PATH] = os.path.join(homePath, '.gam')
    GC.Defaults[GC.CONFIG_DIR] = GM.Globals[GM.GAM_CFG_PATH]
    GC.Defaults[GC.CACHE_DIR] = os.path.join(GM.Globals[GM.GAM_CFG_PATH], 'gamcache')
    GC.Defaults[GC.DRIVE_DIR] = os.path.join(homePath, 'Downloads')
    GM.Globals[GM.GAM_CFG_FILE] = os.path.join(GM.Globals[GM.GAM_CFG_PATH], FN_GAM_CFG)
    if not os.path.isfile(GM.Globals[GM.GAM_CFG_FILE]):
      for itemName, itemEntry in GC.VAR_INFO.items():
        if itemEntry[GC.VAR_TYPE] == GC.TYPE_DIRECTORY:
          _getDefault(itemName, itemEntry, None)
      oldGamPath = os.environ.get(EV_OLDGAMPATH, GC.Defaults[GC.CONFIG_DIR])
      for itemName, itemEntry in GC.VAR_INFO.items():
        if itemEntry[GC.VAR_TYPE] != GC.TYPE_DIRECTORY:
          _getDefault(itemName, itemEntry, oldGamPath)
      GM.Globals[GM.PARSER] = configparser.RawConfigParser(defaults=collections.OrderedDict(sorted(list(GC.Defaults.items()), key=lambda t: t[0])))
      _checkMakeDir(GC.CONFIG_DIR)
      _checkMakeDir(GC.CACHE_DIR)
      _checkMakeDir(GC.DRIVE_DIR)
      for itemName, itemEntry in GC.VAR_INFO.items():
        if itemEntry[GC.VAR_TYPE] == GC.TYPE_FILE:
          srcFile = os.path.expanduser(_stripStringQuotes(GM.Globals[GM.PARSER].get(configparser.DEFAULTSECT, itemName)))
          _copyCfgFile(srcFile, GC.CONFIG_DIR, oldGamPath)
      _writeGamCfgFile(GM.Globals[GM.PARSER], GM.Globals[GM.GAM_CFG_FILE], Act.INITIALIZE)
    else:
      GM.Globals[GM.PARSER] = configparser.RawConfigParser(defaults=collections.OrderedDict(sorted(list(GC.Defaults.items()), key=lambda t: t[0])))
      _readGamCfgFile(GM.Globals[GM.PARSER], GM.Globals[GM.GAM_CFG_FILE])
  status = {'errors': False}
  inputFilterSectionName = outputFilterSectionName = None
  GM.Globals[GM.GAM_CFG_SECTION] = os.environ.get(EV_GAMCFGSECTION, None)
  if GM.Globals[GM.GAM_CFG_SECTION]:
    sectionName = GM.Globals[GM.GAM_CFG_SECTION]
    GM.Globals[GM.SECTION] = sectionName # Save section for inner gams
    if not GM.Globals[GM.PARSER].has_section(sectionName):
      usageErrorExit(formatKeyValueList('', [EV_GAMCFGSECTION, sectionName, Msg.NOT_FOUND], ''))
    if checkArgumentPresent(Cmd.SELECT_CMD):
      Cmd.Backup()
      usageErrorExit(formatKeyValueList('', [EV_GAMCFGSECTION, sectionName, 'select', Msg.NOT_ALLOWED], ''))
  else:
    sectionName = _getCfgSection(configparser.DEFAULTSECT, GC.SECTION)
# select <SectionName> [save] [verify]
    if checkArgumentPresent(Cmd.SELECT_CMD):
      sectionName = _selectSection()
      GM.Globals[GM.SECTION] = sectionName # Save section for inner gams
      while Cmd.ArgumentsRemaining():
        if checkArgumentPresent('save'):
          GM.Globals[GM.PARSER].set(configparser.DEFAULTSECT, GC.SECTION, sectionName)
          _writeGamCfgFile(GM.Globals[GM.PARSER], GM.Globals[GM.GAM_CFG_FILE], Act.SAVE)
        elif checkArgumentPresent('verify'):
          _verifyValues(sectionName, inputFilterSectionName, outputFilterSectionName)
        else:
          break
  GM.Globals[GM.GAM_CFG_SECTION_NAME] = sectionName
# showsections
  if checkArgumentPresent(Cmd.SHOWSECTIONS_CMD):
    _showSections()
# selectfilter|selectoutputfilter|selectinputfilter <SectionName>
  while True:
    filterCommand = getChoice([Cmd.SELECTFILTER_CMD, Cmd.SELECTOUTPUTFILTER_CMD, Cmd.SELECTINPUTFILTER_CMD], defaultChoice=None)
    if filterCommand is None:
      break
    if filterCommand != Cmd.SELECTINPUTFILTER_CMD:
      outputFilterSectionName = _selectSection()
    else:
      inputFilterSectionName = _selectSection()
# Handle todrive_nobrowser and todrive_noemail if not present
  value = GM.Globals[GM.PARSER].get(configparser.DEFAULTSECT, GC.TODRIVE_NOBROWSER)
  if value == '':
    GM.Globals[GM.PARSER].set(configparser.DEFAULTSECT, GC.TODRIVE_NOBROWSER, str(_getCfgBoolean(configparser.DEFAULTSECT, GC.NO_BROWSER)).lower())
  value = GM.Globals[GM.PARSER].get(configparser.DEFAULTSECT, GC.TODRIVE_NOEMAIL)
  if value == '':
    GM.Globals[GM.PARSER].set(configparser.DEFAULTSECT, GC.TODRIVE_NOEMAIL, str(not _getCfgBoolean(configparser.DEFAULTSECT, GC.NO_BROWSER)).lower())
# Handle todrive_sheet_timestamp and todrive_sheet_timeformat if not present
  for section in [sectionName, configparser.DEFAULTSECT]:
    value = GM.Globals[GM.PARSER].get(section, GC.TODRIVE_SHEET_TIMESTAMP)
    if value == 'copy':
      GM.Globals[GM.PARSER].set(section, GC.TODRIVE_SHEET_TIMESTAMP, str(_getCfgBoolean(section, GC.TODRIVE_TIMESTAMP)).lower())
    value = GM.Globals[GM.PARSER].get(section, GC.TODRIVE_SHEET_TIMEFORMAT)
    if value == 'copy':
      GM.Globals[GM.PARSER].set(section, GC.TODRIVE_SHEET_TIMEFORMAT, _getCfgString(section, GC.TODRIVE_TIMEFORMAT))
# Fix mistyped keyword cmdlog_max__backups
  for section in [configparser.DEFAULTSECT, sectionName]:
    if GM.Globals[GM.PARSER].has_option(section, GC.CMDLOG_MAX__BACKUPS):
      GM.Globals[GM.PARSER].set(section, GC.CMDLOG_MAX_BACKUPS, GM.Globals[GM.PARSER].get(section, GC.CMDLOG_MAX__BACKUPS))
      GM.Globals[GM.PARSER].remove_option(section, GC.CMDLOG_MAX__BACKUPS)
# config (<VariableName> [=] <Value>)* [save] [verify]
  if checkArgumentPresent(Cmd.CONFIG_CMD):
    while Cmd.ArgumentsRemaining():
      if checkArgumentPresent('save'):
        _writeGamCfgFile(GM.Globals[GM.PARSER], GM.Globals[GM.GAM_CFG_FILE], Act.SAVE)
      elif checkArgumentPresent('verify'):
        _verifyValues(sectionName, inputFilterSectionName, outputFilterSectionName)
      else:
        itemName = getChoice(GC.VAR_INFO, defaultChoice=None)
        if itemName is None:
          break
        itemEntry = GC.VAR_INFO[itemName]
        checkArgumentPresent('=')
        varType = itemEntry[GC.VAR_TYPE]
        if varType == GC.TYPE_BOOLEAN:
          value = TRUE if getBoolean(None) else FALSE
        elif varType == GC.TYPE_CHARACTER:
          value = getCharacter()
        elif varType == GC.TYPE_CHOICE:
          value = getChoice(itemEntry[GC.VAR_CHOICES])
        elif varType == GC.TYPE_INTEGER:
          minVal, maxVal = itemEntry[GC.VAR_LIMITS]
          value = str(getInteger(minVal=minVal, maxVal=maxVal))
        elif varType == GC.TYPE_FLOAT:
          minVal, maxVal = itemEntry[GC.VAR_LIMITS]
          value = str(getFloat(minVal=minVal, maxVal=maxVal))
        elif varType == GC.TYPE_LOCALE:
          value = getLanguageCode(LOCALE_CODES_MAP)
        elif varType == GC.TYPE_PASSWORD:
          minLen, maxLen = itemEntry[GC.VAR_LIMITS]
          value = getString(Cmd.OB_STRING, checkBlank=True, minLen=minLen, maxLen=maxLen)
          if value and value.startswith("b'") and value.endswith("'"):
            value = bytes(value[2:-1], UTF8)
        elif varType == GC.TYPE_TIMEZONE:
          value = getString(Cmd.OB_STRING, checkBlank=True)
        else: # GC.TYPE_STRING, GC.TYPE_STRINGLIST
          minLen, maxLen = itemEntry.get(GC.VAR_LIMITS, (0, None))
          value = _quoteStringIfLeadingTrailingBlanks(getString(Cmd.OB_STRING, minLen=minLen, maxLen=maxLen))
        GM.Globals[GM.PARSER].set(sectionName, itemName, value)
  prevExtraArgsTxt = GC.Values.get(GC.EXTRA_ARGS, None)
  prevOauth2serviceJson = GC.Values.get(GC.OAUTH2SERVICE_JSON, None)
# Assign global variables, directories, timezone first as other variables depend on them
  for itemName, itemEntry in sorted(GC.VAR_INFO.items()):
    varType = itemEntry[GC.VAR_TYPE]
    if varType == GC.TYPE_DIRECTORY:
      GC.Values[itemName] = _getCfgDirectory(sectionName, itemName)
    elif varType == GC.TYPE_TIMEZONE:
      GC.Values[itemName] = _getCfgTimezone(sectionName, itemName)
  GM.Globals[GM.DATETIME_NOW] = datetime.datetime.now(GC.Values[GC.TIMEZONE])
# Everything else except row filters
  for itemName, itemEntry in sorted(GC.VAR_INFO.items()):
    varType = itemEntry[GC.VAR_TYPE]
    if varType == GC.TYPE_BOOLEAN:
      GC.Values[itemName] = _getCfgBoolean(sectionName, itemName)
    elif varType == GC.TYPE_CHARACTER:
      GC.Values[itemName] = _getCfgCharacter(sectionName, itemName)
    elif varType == GC.TYPE_CHOICE:
      GC.Values[itemName] = _getCfgChoice(sectionName, itemName)
    elif varType in [GC.TYPE_INTEGER, GC.TYPE_FLOAT]:
      GC.Values[itemName] = _getCfgNumber(sectionName, itemName)
    elif varType == GC.TYPE_HEADERFILTER:
      GC.Values[itemName] = _getCfgHeaderFilter(sectionName, itemName)
    elif varType == GC.TYPE_LOCALE:
      GC.Values[itemName] = _getCfgLocale(sectionName, itemName)
    elif varType == GC.TYPE_PASSWORD:
      GC.Values[itemName] = _getCfgPassword(sectionName, itemName)
    elif varType == GC.TYPE_STRING:
      GC.Values[itemName] = _getCfgString(sectionName, itemName)
    elif varType in {GC.TYPE_STRINGLIST, GC.TYPE_HEADERFORCE, GC.TYPE_HEADERORDER}:
      GC.Values[itemName] = _getCfgStringList(sectionName, itemName)
    elif varType == GC.TYPE_FILE:
      GC.Values[itemName] = _getCfgFile(sectionName, itemName)
# Row filters
  for itemName, itemEntry in sorted(GC.VAR_INFO.items()):
    varType = itemEntry[GC.VAR_TYPE]
    if varType == GC.TYPE_ROWFILTER:
      GC.Values[itemName] = _getCfgRowFilter(sectionName, itemName)
# Process selectfilter|selectoutputfilter|selectinputfilter
  if inputFilterSectionName:
    GC.Values[GC.CSV_INPUT_ROW_FILTER] = _getCfgRowFilter(inputFilterSectionName, GC.CSV_INPUT_ROW_FILTER)
    GC.Values[GC.CSV_INPUT_ROW_FILTER_MODE] = _getCfgChoice(inputFilterSectionName, GC.CSV_INPUT_ROW_FILTER_MODE)
    GC.Values[GC.CSV_INPUT_ROW_DROP_FILTER] = _getCfgRowFilter(inputFilterSectionName, GC.CSV_INPUT_ROW_DROP_FILTER)
    GC.Values[GC.CSV_INPUT_ROW_DROP_FILTER_MODE] = _getCfgChoice(inputFilterSectionName, GC.CSV_INPUT_ROW_DROP_FILTER_MODE)
    GC.Values[GC.CSV_INPUT_ROW_LIMIT] = _getCfgNumber(inputFilterSectionName, GC.CSV_INPUT_ROW_LIMIT)
  if outputFilterSectionName:
    GC.Values[GC.CSV_OUTPUT_HEADER_FORCE] = _getCfgStringList(outputFilterSectionName, GC.CSV_OUTPUT_HEADER_FORCE)
    if GC.Values[GC.CSV_OUTPUT_HEADER_FORCE]:
      GC.Values[GC.CSV_OUTPUT_HEADER_FILTER] = _getCfgHeaderFilterFromForce(outputFilterSectionName, GC.CSV_OUTPUT_HEADER_FORCE)
    else:
      GC.Values[GC.CSV_OUTPUT_HEADER_FILTER] = _getCfgHeaderFilter(outputFilterSectionName, GC.CSV_OUTPUT_HEADER_FILTER)
    GC.Values[GC.CSV_OUTPUT_HEADER_DROP_FILTER] = _getCfgHeaderFilter(outputFilterSectionName, GC.CSV_OUTPUT_HEADER_DROP_FILTER)
    GC.Values[GC.CSV_OUTPUT_HEADER_ORDER] = _getCfgStringList(outputFilterSectionName, GC.CSV_OUTPUT_HEADER_ORDER)
    GC.Values[GC.CSV_OUTPUT_ROW_FILTER] = _getCfgRowFilter(outputFilterSectionName, GC.CSV_OUTPUT_ROW_FILTER)
    GC.Values[GC.CSV_OUTPUT_ROW_FILTER_MODE] = _getCfgChoice(outputFilterSectionName, GC.CSV_OUTPUT_ROW_FILTER_MODE)
    GC.Values[GC.CSV_OUTPUT_ROW_DROP_FILTER] = _getCfgRowFilter(outputFilterSectionName, GC.CSV_OUTPUT_ROW_DROP_FILTER)
    GC.Values[GC.CSV_OUTPUT_ROW_DROP_FILTER_MODE] = _getCfgChoice(outputFilterSectionName, GC.CSV_OUTPUT_ROW_DROP_FILTER_MODE)
    GC.Values[GC.CSV_OUTPUT_ROW_LIMIT] = _getCfgNumber(outputFilterSectionName, GC.CSV_OUTPUT_ROW_LIMIT)
    GC.Values[GC.CSV_OUTPUT_SORT_HEADERS] = _getCfgStringList(outputFilterSectionName, GC.CSV_OUTPUT_SORT_HEADERS)
  elif GC.Values[GC.CSV_OUTPUT_HEADER_FORCE]:
    GC.Values[GC.CSV_OUTPUT_HEADER_FILTER] = _getCfgHeaderFilterFromForce(sectionName, GC.CSV_OUTPUT_HEADER_FORCE)
  if status['errors']:
    sys.exit(CONFIG_ERROR_RC)
# Global values cleanup
  GC.Values[GC.DOMAIN] = GC.Values[GC.DOMAIN].lower()
  if not GC.Values[GC.SMTP_FQDN]:
    GC.Values[GC.SMTP_FQDN] = None
# Inherit debug_level, output_dateformat, output_timeformat if not locally defined
  if GM.Globals[GM.PID] != 0:
    if GC.Values[GC.DEBUG_LEVEL] == 0:
      GC.Values[GC.DEBUG_LEVEL] = GM.Globals[GM.DEBUG_LEVEL]
    if not GC.Values[GC.OUTPUT_DATEFORMAT]:
      GC.Values[GC.OUTPUT_DATEFORMAT] = GM.Globals[GM.OUTPUT_DATEFORMAT]
    if not GC.Values[GC.OUTPUT_TIMEFORMAT]:
      GC.Values[GC.OUTPUT_TIMEFORMAT] = GM.Globals[GM.OUTPUT_TIMEFORMAT]
# Define lockfile: oauth2.txt.lock
  GM.Globals[GM.OAUTH2_TXT_LOCK] = f'{GC.Values[GC.OAUTH2_TXT]}.lock'
# Override httplib2 settings
  httplib2.debuglevel = GC.Values[GC.DEBUG_LEVEL]
# Reset global variables if required
  if prevExtraArgsTxt != GC.Values[GC.EXTRA_ARGS]:
    GM.Globals[GM.EXTRA_ARGS_LIST] = [('prettyPrint', GC.Values[GC.DEBUG_LEVEL] > 0)]
    if GC.Values[GC.EXTRA_ARGS]:
      ea_config = configparser.ConfigParser()
      ea_config.optionxform = str
      ea_config.read(GC.Values[GC.EXTRA_ARGS])
      GM.Globals[GM.EXTRA_ARGS_LIST].extend(ea_config.items('extra-args'))
  if prevOauth2serviceJson != GC.Values[GC.OAUTH2SERVICE_JSON]:
    GM.Globals[GM.OAUTH2SERVICE_JSON_DATA] = {}
    GM.Globals[GM.OAUTH2SERVICE_CLIENT_ID] = None
  Cmd.SetEncoding(GM.Globals[GM.SYS_ENCODING])
# multiprocessexit (rc<Operator><Number>)|(rcrange=<Number>/<Number>)|(rcrange!=<Number>/<Number>)
  if checkArgumentPresent(Cmd.MULTIPROCESSEXIT_CMD):
    _setMultiprocessExit()
# redirect csv <FileName> [multiprocess] [append] [noheader] [charset <CharSet>]
#	       [columndelimiter <Character>] [quotechar <Character>]] [noescapechar [<Boolean>]]
#	       [sortheaders <StringList>] [timestampcolumn <String>] [transpose [<Boolean>]]
#	       [todrive <ToDriveAttribute>*]
# redirect stdout <FileName> [multiprocess] [append]
# redirect stdout null
# redirect stderr <FileName> [multiprocess] [append]
# redirect stderr stdout
# redirect stderr null
  while checkArgumentPresent(Cmd.REDIRECT_CMD):
    myarg = getChoice(['csv', 'stdout', 'stderr'])
    filename = re.sub(r'{{Section}}', sectionName, getString(Cmd.OB_FILE_NAME, checkBlank=True))
    if myarg == 'csv':
      multi = checkArgumentPresent('multiprocess')
      mode = DEFAULT_FILE_APPEND_MODE if checkArgumentPresent('append') else DEFAULT_FILE_WRITE_MODE
      writeHeader = not checkArgumentPresent('noheader')
      encoding = getCharSet()
      if checkArgumentPresent('columndelimiter'):
        GM.Globals[GM.CSV_OUTPUT_COLUMN_DELIMITER] = GC.Values[GC.CSV_OUTPUT_COLUMN_DELIMITER] = getCharacter()
      if checkArgumentPresent('quotechar'):
        GM.Globals[GM.CSV_OUTPUT_QUOTE_CHAR] = GC.Values[GC.CSV_OUTPUT_QUOTE_CHAR] = getCharacter()
      if checkArgumentPresent('noescapechar'):
        GM.Globals[GM.CSV_OUTPUT_NO_ESCAPE_CHAR] = GC.Values[GC.CSV_OUTPUT_NO_ESCAPE_CHAR] = getBoolean()
      if checkArgumentPresent('sortheaders'):
        GM.Globals[GM.CSV_OUTPUT_SORT_HEADERS] = GC.Values[GC.CSV_OUTPUT_SORT_HEADERS] = getString(Cmd.OB_STRING_LIST, minLen=0).replace(',', ' ').split()
      if checkArgumentPresent('timestampcolumn'):
        GM.Globals[GM.CSV_OUTPUT_TIMESTAMP_COLUMN] = GC.Values[GC.CSV_OUTPUT_TIMESTAMP_COLUMN] = getString(Cmd.OB_STRING, minLen=0)
      if checkArgumentPresent('transpose'):
        GM.Globals[GM.CSV_OUTPUT_TRANSPOSE] = getBoolean()
      _setCSVFile(filename, mode, encoding, writeHeader, multi)
      GM.Globals[GM.CSVFILE][GM.REDIRECT_QUEUE_CSVPF] = CSVPrintFile()
      if checkArgumentPresent('todrive'):
        GM.Globals[GM.CSVFILE][GM.REDIRECT_QUEUE_CSVPF].GetTodriveParameters()
        GM.Globals[GM.CSV_TODRIVE] = GM.Globals[GM.CSVFILE][GM.REDIRECT_QUEUE_CSVPF].todrive.copy()
    elif myarg == 'stdout':
      if filename.lower() == 'null':
        multi = checkArgumentPresent('multiprocess')
        _setSTDFile(GM.STDOUT, 'null', DEFAULT_FILE_WRITE_MODE, multi)
      else:
        multi = checkArgumentPresent('multiprocess')
        mode = DEFAULT_FILE_APPEND_MODE if checkArgumentPresent('append') else DEFAULT_FILE_WRITE_MODE
        _setSTDFile(GM.STDOUT, filename, mode, multi)
    else: # myarg == 'stderr'
      if filename.lower() == 'null':
        multi = checkArgumentPresent('multiprocess')
        _setSTDFile(GM.STDERR, 'null', DEFAULT_FILE_WRITE_MODE, multi)
      elif filename.lower() != 'stdout':
        multi = checkArgumentPresent('multiprocess')
        mode = DEFAULT_FILE_APPEND_MODE if checkArgumentPresent('append') else DEFAULT_FILE_WRITE_MODE
        _setSTDFile(GM.STDERR, filename, mode, multi)
      else:
        multi = checkArgumentPresent('multiprocess')
        if not GM.Globals[GM.STDOUT]:
          _setSTDFile(GM.STDOUT, '-', DEFAULT_FILE_WRITE_MODE, multi)
        GM.Globals[GM.STDERR] = GM.Globals[GM.STDOUT].copy()
        GM.Globals[GM.STDERR][GM.REDIRECT_NAME] = 'stdout'
  if not GM.Globals[GM.STDOUT]:
    _setSTDFile(GM.STDOUT, '-', DEFAULT_FILE_WRITE_MODE, False)
  if not GM.Globals[GM.STDERR]:
    _setSTDFile(GM.STDERR, '-', DEFAULT_FILE_WRITE_MODE, False)
# If both csv and stdout are redirected to - with same multiprocess setting and csv doesn't have any todrive parameters, collapse csv onto stdout
  if (GM.Globals[GM.PID] == 0  and GM.Globals[GM.CSVFILE] and
      GM.Globals[GM.CSVFILE][GM.REDIRECT_NAME] == '-' and GM.Globals[GM.STDOUT][GM.REDIRECT_NAME] == '-' and
      GM.Globals[GM.CSVFILE][GM.REDIRECT_MULTIPROCESS] == GM.Globals[GM.STDOUT][GM.REDIRECT_MULTIPROCESS] and
      GM.Globals[GM.CSVFILE].get(GM.REDIRECT_QUEUE_CSVPF) and not GM.Globals[GM.CSVFILE][GM.REDIRECT_QUEUE_CSVPF].todrive):
    _setCSVFile('-', GM.Globals[GM.STDOUT].get(GM.REDIRECT_MODE, DEFAULT_FILE_WRITE_MODE), GC.Values[GC.CHARSET],
                GM.Globals[GM.CSVFILE].get(GM.REDIRECT_WRITE_HEADER, True), GM.Globals[GM.STDOUT][GM.REDIRECT_MULTIPROCESS])
  elif not GM.Globals[GM.CSVFILE]:
    _setCSVFile('-', GM.Globals[GM.STDOUT].get(GM.REDIRECT_MODE, DEFAULT_FILE_WRITE_MODE), GC.Values[GC.CHARSET], True, False)
  initAPICallsRateCheck()
# Main process
# Clear input row filters/limit from parser, children can define but shouldn't inherit global value
# Clear output header/row filters/limit from parser, children can define or they will inherit global value if not defined
  if GM.Globals[GM.PID] == 0:
    for itemName, itemEntry in sorted(GC.VAR_INFO.items()):
      varType = itemEntry[GC.VAR_TYPE]
      if varType in {GC.TYPE_HEADERFILTER, GC.TYPE_HEADERFORCE, GC.TYPE_HEADERORDER, GC.TYPE_ROWFILTER}:
        GM.Globals[GM.PARSER].set(sectionName, itemName, '')
      elif (varType == GC.TYPE_INTEGER) and itemName in {GC.CSV_INPUT_ROW_LIMIT, GC.CSV_OUTPUT_ROW_LIMIT}:
        GM.Globals[GM.PARSER].set(sectionName, itemName, '0')
# Child process
# Inherit main process output header/row filters/limit, print defaults if not locally defined
  else:
    if not GC.Values[GC.CSV_OUTPUT_HEADER_FILTER]:
      GC.Values[GC.CSV_OUTPUT_HEADER_FILTER] = GM.Globals[GM.CSV_OUTPUT_HEADER_FILTER][:]
    if not GC.Values[GC.CSV_OUTPUT_HEADER_DROP_FILTER]:
      GC.Values[GC.CSV_OUTPUT_HEADER_DROP_FILTER] = GM.Globals[GM.CSV_OUTPUT_HEADER_DROP_FILTER][:]
    if not GC.Values[GC.CSV_OUTPUT_HEADER_FORCE]:
      GC.Values[GC.CSV_OUTPUT_HEADER_FORCE] = GM.Globals[GM.CSV_OUTPUT_HEADER_FORCE][:]
    if not GC.Values[GC.CSV_OUTPUT_HEADER_ORDER]:
      GC.Values[GC.CSV_OUTPUT_HEADER_ORDER] = GM.Globals[GM.CSV_OUTPUT_HEADER_ORDER][:]
    if not GC.Values[GC.CSV_OUTPUT_ROW_FILTER]:
      GC.Values[GC.CSV_OUTPUT_ROW_FILTER] = GM.Globals[GM.CSV_OUTPUT_ROW_FILTER][:]
      GC.Values[GC.CSV_OUTPUT_ROW_FILTER_MODE] = GM.Globals[GM.CSV_OUTPUT_ROW_FILTER_MODE]
    if not GC.Values[GC.CSV_OUTPUT_ROW_DROP_FILTER]:
      GC.Values[GC.CSV_OUTPUT_ROW_DROP_FILTER] = GM.Globals[GM.CSV_OUTPUT_ROW_DROP_FILTER][:]
      GC.Values[GC.CSV_OUTPUT_ROW_DROP_FILTER_MODE] = GM.Globals[GM.CSV_OUTPUT_ROW_DROP_FILTER_MODE]
    if not GC.Values[GC.CSV_OUTPUT_ROW_LIMIT]:
      GC.Values[GC.CSV_OUTPUT_ROW_LIMIT] = GM.Globals[GM.CSV_OUTPUT_ROW_LIMIT]
    if not GC.Values[GC.PRINT_AGU_DOMAINS]:
      GC.Values[GC.PRINT_AGU_DOMAINS] = GM.Globals[GM.PRINT_AGU_DOMAINS]
    if not GC.Values[GC.PRINT_CROS_OUS]:
      GC.Values[GC.PRINT_CROS_OUS] = GM.Globals[GM.PRINT_CROS_OUS]
    if not GC.Values[GC.PRINT_CROS_OUS_AND_CHILDREN]:
      GC.Values[GC.PRINT_CROS_OUS_AND_CHILDREN] = GM.Globals[GM.PRINT_CROS_OUS_AND_CHILDREN]
    GC.Values[GC.SHOW_GETTINGS] = GM.Globals[GM.SHOW_GETTINGS]
    GC.Values[GC.SHOW_GETTINGS_GOT_NL] = GM.Globals[GM.SHOW_GETTINGS_GOT_NL]
# customer_id, domain and admin_email must be set when enable_dasa = true
  if GC.Values[GC.ENABLE_DASA]:
    errors = 0
    for itemName in [GC.CUSTOMER_ID, GC.DOMAIN, GC.ADMIN_EMAIL]:
      if not GC.Values[itemName] or (itemName == GC.CUSTOMER_ID and GC.Values[itemName] == GC.MY_CUSTOMER):
        stderrErrorMsg(formatKeyValueList('',
                                          [Ent.Singular(Ent.CONFIG_FILE), GM.Globals[GM.GAM_CFG_FILE],
                                           Ent.Singular(Ent.SECTION), sectionName,
                                           itemName, GC.Values[itemName] or '""',
                                           GC.ENABLE_DASA, GC.Values[GC.ENABLE_DASA],
                                           Msg.NOT_COMPATIBLE],
                                          '\n'))
        errors += 1
    if errors:
      sys.exit(USAGE_ERROR_RC)
# If no select/options commands were executed or some were and there are more arguments on the command line,
# warn if the json files are missing and return True
  if (Cmd.Location() == 1) or (Cmd.ArgumentsRemaining()):
    _chkCfgDirectories(sectionName)
    _chkCfgFiles(sectionName)
    if status['errors']:
      sys.exit(CONFIG_ERROR_RC)
    if GC.Values[GC.NO_CACHE]:
      GM.Globals[GM.CACHE_DIR] = None
      GM.Globals[GM.CACHE_DISCOVERY_ONLY] = False
    else:
      GM.Globals[GM.CACHE_DIR] = GC.Values[GC.CACHE_DIR]
      GM.Globals[GM.CACHE_DISCOVERY_ONLY] = GC.Values[GC.CACHE_DISCOVERY_ONLY]
# Set environment variables so GData API can find cacerts.pem
    os.environ['REQUESTS_CA_BUNDLE'] = GC.Values[GC.CACERTS_PEM]
    os.environ['DEFAULT_CA_BUNDLE_PATH'] = GC.Values[GC.CACERTS_PEM]
    os.environ['HTTPLIB2_CA_CERTS'] = GC.Values[GC.CACERTS_PEM]
    os.environ['SSL_CERT_FILE'] = GC.Values[GC.CACERTS_PEM]
    httplib2.CA_CERTS = GC.Values[GC.CACERTS_PEM]
# Needs to be set so oauthlib doesn't puke when Google changes our scopes
    os.environ['OAUTHLIB_RELAX_TOKEN_SCOPE'] = 'true'
# Set up command logging at top level only
    if (GM.Globals[GM.PID] == 0) and GC.Values[GC.CMDLOG]:
      openGAMCommandLog(GM.Globals, 'mainlog')
    return True
# We're done, nothing else to do
  return False

def handleServerError(e):
  errMsg = str(e)
  if 'setting tls' not in errMsg:
    systemErrorExit(NETWORK_ERROR_RC, errMsg)
  stderrErrorMsg(errMsg)
  writeStderr(Msg.DISABLE_TLS_MIN_MAX)
  systemErrorExit(NETWORK_ERROR_RC, None)

def getHttpObj(cache=None, timeout=None, override_min_tls=None, override_max_tls=None):
  tls_minimum_version = override_min_tls if override_min_tls else GC.Values[GC.TLS_MIN_VERSION] if GC.Values[GC.TLS_MIN_VERSION] else None
  tls_maximum_version = override_max_tls if override_max_tls else GC.Values[GC.TLS_MAX_VERSION] if GC.Values[GC.TLS_MAX_VERSION] else None
  httpObj = httplib2.Http(cache=cache,
                          timeout=timeout,
                          ca_certs=GC.Values[GC.CACERTS_PEM],
                          disable_ssl_certificate_validation=GC.Values[GC.NO_VERIFY_SSL],
                          tls_maximum_version=tls_maximum_version,
                          tls_minimum_version=tls_minimum_version)
  httpObj.redirect_codes = set(httpObj.redirect_codes) - {308}
  return httpObj

def _force_user_agent(user_agent):
  """Creates a decorator which can force a user agent in HTTP headers."""

  def decorator(request_method):
    """Wraps a request method to insert a user-agent in HTTP headers."""

    def wrapped_request_method(*args, **kwargs):
      """Modifies HTTP headers to include a specified user-agent."""
      if kwargs.get('headers') is not None:
        if kwargs['headers'].get('user-agent'):
          if user_agent not in kwargs['headers']['user-agent']:
            # Save the existing user-agent header and tack on our own.
            kwargs['headers']['user-agent'] = f'{user_agent} {kwargs["headers"]["user-agent"]}'
        else:
          kwargs['headers']['user-agent'] = user_agent
      else:
        kwargs['headers'] = {'user-agent': user_agent}
      return request_method(*args, **kwargs)

    return wrapped_request_method

  return decorator

class transportAgentRequest(google_auth_httplib2.Request):
  """A Request which forces a user agent."""

  @_force_user_agent(GAM_USER_AGENT)
  def __call__(self, *args, **kwargs): #pylint: disable=arguments-differ
    """Inserts the GAM user-agent header in requests."""
    return super().__call__(*args, **kwargs)

class transportAuthorizedHttp(google_auth_httplib2.AuthorizedHttp):
  """An AuthorizedHttp which forces a user agent during requests."""

  @_force_user_agent(GAM_USER_AGENT)
  def request(self, *args, **kwargs): #pylint: disable=arguments-differ
    """Inserts the GAM user-agent header in requests."""
    return super().request(*args, **kwargs)

def transportCreateRequest(httpObj=None):
  """Creates a uniform Request object with a default http, if not provided.

  Args:
    httpObj: Optional httplib2.Http compatible object to be used with the request.
      If not provided, a default HTTP will be used.

  Returns:
    Request: A google_auth_httplib2.Request compatible Request.
  """
  if not httpObj:
    httpObj = getHttpObj()
  return transportAgentRequest(httpObj)

def doGAMCheckForUpdates(forceCheck):
  def _gamLatestVersionNotAvailable():
    if forceCheck:
      systemErrorExit(NETWORK_ERROR_RC, Msg.GAM_LATEST_VERSION_NOT_AVAILABLE)

  try:
    _, c = getHttpObj(timeout=10).request(GAM_LATEST_RELEASE, 'GET', headers={'Accept': 'application/vnd.github.v3.text+json'})
    try:
      release_data = json.loads(c)
    except (IndexError, KeyError, SyntaxError, TypeError, ValueError):
      _gamLatestVersionNotAvailable()
      return
    if not isinstance(release_data, dict) or 'tag_name' not in release_data:
      _gamLatestVersionNotAvailable()
      return
    current_version = __version__
    latest_version = release_data['tag_name']
    if latest_version[0].lower() == 'v':
      latest_version = latest_version[1:]
      printKeyValueList(['Version Check', None])
      Ind.Increment()
      printKeyValueList(['Current', current_version])
      printKeyValueList([' Latest', latest_version])
      Ind.Decrement()
    if forceCheck < 0:
      setSysExitRC(1 if latest_version > current_version else 0)
      return
  except (httplib2.HttpLib2Error, httplib2.ServerNotFoundError,
          google.auth.exceptions.TransportError,
          RuntimeError, ConnectionError, OSError) as e:
    if forceCheck:
      handleServerError(e)

_DEFAULT_TOKEN_LIFETIME_SECS = 3600  # 1 hour in seconds

class signjwtJWTCredentials(google.auth.jwt.Credentials):
  ''' Class used for DASA '''
  def _make_jwt(self):
    now = datetime.datetime.utcnow()
    lifetime = datetime.timedelta(seconds=self._token_lifetime)
    expiry = now + lifetime
    payload = {
      "iat": google.auth._helpers.datetime_to_secs(now),
      "exp": google.auth._helpers.datetime_to_secs(expiry),
      "iss": self._issuer,
      "sub": self._subject,
    }
    if self._audience:
      payload["aud"] = self._audience
    payload.update(self._additional_claims)
    jwt = self._signer.sign(payload)
    return jwt, expiry

# Some Workforce Identity Federation endpoints such as GitHub Actions
# only allow TLS 1.2 as of April 2023.

def getTLSv1_2Request():
  httpc = getHttpObj(override_min_tls='TLSv1_2')
  return transportCreateRequest(httpc)

class signjwtCredentials(google.oauth2.service_account.Credentials):
  ''' Class used for DwD '''

  def _make_authorization_grant_assertion(self):
    now = datetime.datetime.utcnow()
    lifetime = datetime.timedelta(seconds=_DEFAULT_TOKEN_LIFETIME_SECS)
    expiry = now + lifetime
    payload = {
        "iat": google.auth._helpers.datetime_to_secs(now),
        "exp": google.auth._helpers.datetime_to_secs(expiry),
        "iss": self._service_account_email,
        "aud": API.GOOGLE_OAUTH2_TOKEN_ENDPOINT,
        "scope": google.auth._helpers.scopes_to_string(self._scopes or ()),
    }
    payload.update(self._additional_claims)
    # The subject can be a user email for domain-wide delegation.
    if self._subject:
      payload.setdefault("sub", self._subject)
    token = self._signer(payload)
    return token

class signjwtSignJwt(google.auth.crypt.Signer):
  ''' Signer class for SignJWT '''
  def __init__(self, service_account_info):
    self.service_account_email = service_account_info['client_email']
    self.name = f'projects/-/serviceAccounts/{self.service_account_email}'
    self._key_id = None

  @property  # type: ignore
  def key_id(self):
    return self._key_id

  def sign(self, message):
    ''' Call IAM Credentials SignJWT API to get our signed JWT '''
    try:
      credentials, _ = google.auth.default(scopes=[API.IAM_SCOPE],
                                           request=getTLSv1_2Request())
    except (google.auth.exceptions.DefaultCredentialsError, google.auth.exceptions.RefreshError) as e:
      systemErrorExit(API_ACCESS_DENIED_RC, str(e))
    httpObj = transportAuthorizedHttp(credentials, http=getHttpObj(override_min_tls='TLSv1_2'))
    iamc = getService(API.IAM_CREDENTIALS, httpObj)
    response = callGAPI(iamc.projects().serviceAccounts(), 'signJwt',
                        name=self.name, body={'payload': json.dumps(message)})
    signed_jwt = response.get('signedJwt')
    return signed_jwt

def handleOAuthTokenError(e, softErrors, displayError=False, i=0, count=0):
  errMsg = str(e).replace('.', '')
  if ((errMsg in API.OAUTH2_TOKEN_ERRORS) or
      errMsg.startswith('Invalid response') or
      errMsg.startswith('invalid_request: Invalid impersonation &quot;sub&quot; field')):
    if not GM.Globals[GM.CURRENT_SVCACCT_USER]:
      ClientAPIAccessDeniedExit()
    # 403 Forbidden, API disabled, user not enabled
    # 400 Bad Request, user not defined
    if softErrors:
      entityActionFailedWarning([Ent.USER, GM.Globals[GM.CURRENT_SVCACCT_USER], Ent.USER, None], errMsg, i, count)
      return None
    systemErrorExit(SERVICE_NOT_APPLICABLE_RC, Msg.SERVICE_NOT_APPLICABLE_THIS_ADDRESS.format(GM.Globals[GM.CURRENT_SVCACCT_USER]))
  if errMsg in API.OAUTH2_UNAUTHORIZED_ERRORS:
    if not GM.Globals[GM.CURRENT_SVCACCT_USER]:
      ClientAPIAccessDeniedExit()
    # 401 Unauthorized, API disabled, user enabled
    if softErrors:
      if displayError:
        apiOrScopes = API.getAPIName(GM.Globals[GM.CURRENT_SVCACCT_API]) if GM.Globals[GM.CURRENT_SVCACCT_API] else ','.join(sorted(GM.Globals[GM.CURRENT_SVCACCT_API_SCOPES]))
        userServiceNotEnabledWarning(GM.Globals[GM.CURRENT_SVCACCT_USER], apiOrScopes, i, count)
      return None
    SvcAcctAPIAccessDeniedExit()
  if errMsg in API.REFRESH_PERM_ERRORS:
    if softErrors:
      return None
    if not GM.Globals[GM.CURRENT_SVCACCT_USER]:
      expiredRevokedOauth2TxtExit()
  stderrErrorMsg(f'Authentication Token Error - {errMsg}')
  APIAccessDeniedExit()

def getOauth2TxtCredentials(exitOnError=True, api=None, noDASA=False, refreshOnly=False, noScopes=False):
  if not noDASA and GC.Values[GC.ENABLE_DASA]:
    jsonData = readFile(GC.Values[GC.OAUTH2SERVICE_JSON], continueOnError=True, displayError=False)
    if jsonData:
      try:
        if api in API.APIS_NEEDING_ACCESS_TOKEN:
          return (False, getSvcAcctCredentials(API.APIS_NEEDING_ACCESS_TOKEN[api], userEmail=None, forceOauth=True))
        jsonDict = json.loads(jsonData)
        api, _, _ = API.getVersion(api)
        audience = f'https://{api}.googleapis.com/'
        key_type = jsonDict.get('key_type', 'default')
        if key_type == 'default':
          return (True, JWTCredentials.from_service_account_info(jsonDict, audience=audience))
        if key_type == 'yubikey':
          yksigner = yubikey.YubiKey(jsonDict)
          return (True, JWTCredentials._from_signer_and_info(yksigner, jsonDict, audience=audience))
        if key_type == 'signjwt':
          sjsigner = signjwtSignJwt(jsonDict)
          return (True, signjwtJWTCredentials._from_signer_and_info(sjsigner, jsonDict, audience=audience))
      except (IndexError, KeyError, SyntaxError, TypeError, ValueError) as e:
        invalidOauth2serviceJsonExit(str(e))
    invalidOauth2serviceJsonExit(Msg.NO_DATA)
  jsonData = readFile(GC.Values[GC.OAUTH2_TXT], continueOnError=True, displayError=False)
  if jsonData:
    try:
      jsonDict = json.loads(jsonData)
      if noScopes:
        jsonDict['scopes'] = []
      if 'client_id' in jsonDict:
        if not refreshOnly:
          if set(jsonDict.get('scopes', API.REQUIRED_SCOPES)) == API.REQUIRED_SCOPES_SET:
            if exitOnError:
              systemErrorExit(OAUTH2_TXT_REQUIRED_RC, Msg.NO_CLIENT_ACCESS_ALLOWED)
            return (False, None)
        else:
          GM.Globals[GM.CREDENTIALS_SCOPES] = set(jsonDict.pop('scopes', API.REQUIRED_SCOPES))
        token_expiry = jsonDict.get('token_expiry', REFRESH_EXPIRY)
        if GC.Values[GC.TRUNCATE_CLIENT_ID]:
          # chop off .apps.googleusercontent.com suffix as it's not needed and we need to keep things short for the Auth URL.
          jsonDict['client_id'] = re.sub(r'\.apps\.googleusercontent\.com$', '', jsonDict['client_id'])
        creds = google.oauth2.credentials.Credentials.from_authorized_user_info(jsonDict)
        if 'id_token_jwt' not in jsonDict:
          creds.token = jsonDict['token']
          creds._id_token = jsonDict['id_token']
          GM.Globals[GM.DECODED_ID_TOKEN] = jsonDict['decoded_id_token']
        else:
          creds.token = jsonDict['access_token']
          creds._id_token = jsonDict['id_token_jwt']
          GM.Globals[GM.DECODED_ID_TOKEN] = jsonDict['id_token']
        creds.expiry = datetime.datetime.strptime(token_expiry, YYYYMMDDTHHMMSSZ_FORMAT)
        return (not noScopes, creds)
      if jsonDict and exitOnError:
        invalidOauth2TxtExit(Msg.INVALID)
    except (IndexError, KeyError, SyntaxError, TypeError, ValueError) as e:
      if exitOnError:
        invalidOauth2TxtExit(str(e))
  if exitOnError:
    systemErrorExit(OAUTH2_TXT_REQUIRED_RC, Msg.NO_CLIENT_ACCESS_ALLOWED)
  return (False, None)

def _getValueFromOAuth(field, credentials=None):
  if not GM.Globals[GM.DECODED_ID_TOKEN]:
    request = transportCreateRequest()
    if credentials is None:
      credentials = getClientCredentials(refreshOnly=True)
    elif credentials.expired:
      credentials.refresh(request)
    try:
      GM.Globals[GM.DECODED_ID_TOKEN] = google.oauth2.id_token.verify_oauth2_token(credentials.id_token, request,
                                                                                   clock_skew_in_seconds=GC.Values[GC.CLOCK_SKEW_IN_SECONDS])
    except ValueError as e:
      if 'Token used too early' in str(e):
        stderrErrorMsg(Msg.PLEASE_CORRECT_YOUR_SYSTEM_TIME)
      systemErrorExit(SYSTEM_ERROR_RC, str(e))
  return GM.Globals[GM.DECODED_ID_TOKEN].get(field, UNKNOWN)

def _getAdminEmail():
  if GC.Values[GC.ADMIN_EMAIL]:
    return GC.Values[GC.ADMIN_EMAIL]
  return _getValueFromOAuth('email')

def writeClientCredentials(creds, filename):
  creds_data = {
    'client_id': creds.client_id,
    'client_secret': creds.client_secret,
    'id_token': creds.id_token,
    'refresh_token': creds.refresh_token,
    'scopes': sorted(creds.scopes or GM.Globals[GM.CREDENTIALS_SCOPES]),
    'token': creds.token,
    'token_expiry': creds.expiry.strftime(YYYYMMDDTHHMMSSZ_FORMAT),
    'token_uri': creds.token_uri,
    }
  expected_iss = ['https://accounts.google.com', 'accounts.google.com']
  if _getValueFromOAuth('iss', creds) not in expected_iss:
    systemErrorExit(OAUTH2_TXT_REQUIRED_RC, f'Wrong OAuth 2.0 credentials issuer. Got {_getValueFromOAuth("iss", creds)} expected one of {", ".join(expected_iss)}')
  request = transportCreateRequest()
  try:
    creds_data['decoded_id_token'] = google.oauth2.id_token.verify_oauth2_token(creds.id_token, request,
                                                                                clock_skew_in_seconds=GC.Values[GC.CLOCK_SKEW_IN_SECONDS])
  except ValueError as e:
    if 'Token used too early' in str(e):
      stderrErrorMsg(Msg.PLEASE_CORRECT_YOUR_SYSTEM_TIME)
    systemErrorExit(SYSTEM_ERROR_RC, str(e))
  GM.Globals[GM.DECODED_ID_TOKEN] = creds_data['decoded_id_token']
  if filename != '-':
    writeFile(filename, json.dumps(creds_data, indent=2, sort_keys=True)+'\n')
  else:
    writeStdout(json.dumps(creds_data, ensure_ascii=False, sort_keys=True, indent=2)+'\n')

URL_SHORTENER_ENDPOINT = 'https://gam-shortn.appspot.com/create'

def shortenURL(long_url):
  if GC.Values[GC.NO_SHORT_URLS]:
    return long_url
  httpObj = getHttpObj(timeout=10)
  try:
    payload = json.dumps({'long_url': long_url})
    resp, content = httpObj.request(URL_SHORTENER_ENDPOINT, 'POST',
                                    payload,
                                    headers={'Content-Type': 'application/json',
                                             'User-Agent': GAM_USER_AGENT})
  except:
    return long_url
  if resp.status != 200:
    return long_url
  try:
    if isinstance(content, bytes):
      content = content.decode()
    return json.loads(content).get('short_url', long_url)
  except:
    return long_url

def runSqliteQuery(db_file, query):
  conn = sqlite3.connect(db_file)
  curr = conn.cursor()
  curr.execute(query)
  return curr.fetchone()[0]

def refreshCredentialsWithReauth(credentials):
  def gcloudError():
    writeStderr(f'Failed to run gcloud as {admin_email}. Please make sure it\'s setup')
    e = Msg.REAUTHENTICATION_IS_NEEDED
    handleOAuthTokenError(e, False)

  writeStderr(Msg.CALLING_GCLOUD_FOR_REAUTH)
  if 'termios' in sys.modules:
    old_settings = termios.tcgetattr(sys.stdin)
  admin_email = _getAdminEmail()
  # First makes sure gcloud has a valid access token and thus
  # should also have a valid RAPT token
  try:
    devnull = open(os.devnull, 'w', encoding=UTF8)
    subprocess.run(['gcloud',
                    'auth',
                    'print-identity-token',
                    '--no-user-output-enabled'],
                   stderr=devnull,
                   check=False)
    devnull.close()
    # now determine gcloud's config path and token file
    gcloud_path_result = subprocess.run(['gcloud',
                                         'info',
                                         '--format=value(config.paths.global_config_dir)'],
            capture_output=True, check=False)
  except KeyboardInterrupt as e:
    # avoids loss of terminal echo on *nix
    if 'termios' in sys.modules:
      termios.tcsetattr(sys.stdin, termios.TCSADRAIN, old_settings)
    printBlankLine()
    raise KeyboardInterrupt from e
  token_path = gcloud_path_result.stdout.decode().strip()
  if not token_path:
    gcloudError()
  token_file = f'{token_path}/access_tokens.db'
  try:
    credentials._rapt_token = runSqliteQuery(token_file,
            f'SELECT rapt_token FROM access_tokens WHERE account_id = "{admin_email}"')
  except TypeError:
    gcloudError()
  if not credentials._rapt_token:
    systemErrorExit(SYSTEM_ERROR_RC,
            'Failed to retrieve reauth token from gcloud. You may need to wait until gcloud is also prompted for reauth.')

def getClientCredentials(forceRefresh=False, forceWrite=False, filename=None, api=None, noDASA=False, refreshOnly=False, noScopes=False):
  """Gets OAuth2 credentials which are guaranteed to be fresh and valid.
     Locks during read and possible write so that only one process will
     attempt refresh/write when running in parallel. """
  lock = FileLock(GM.Globals[GM.OAUTH2_TXT_LOCK])
  with lock:
    writeCreds, credentials = getOauth2TxtCredentials(api=api, noDASA=noDASA, refreshOnly=refreshOnly, noScopes=noScopes)
    if not credentials:
      invalidOauth2TxtExit('')
    if credentials.expired or forceRefresh:
      triesLimit = 3
      for n in range(1, triesLimit+1):
        try:
          credentials.refresh(transportCreateRequest())
          if writeCreds or forceWrite:
            writeClientCredentials(credentials, filename or GC.Values[GC.OAUTH2_TXT])
          break
        except (httplib2.HttpLib2Error, google.auth.exceptions.TransportError, RuntimeError) as e:
          if n != triesLimit:
            waitOnFailure(n, triesLimit, NETWORK_ERROR_RC, str(e))
            continue
          handleServerError(e)
        except google.auth.exceptions.RefreshError as e:
          if isinstance(e.args, tuple):
            e = e.args[0]
          if 'Reauthentication is needed' in str(e):
            if GC.Values[GC.ENABLE_GCLOUD_REAUTH]:
              refreshCredentialsWithReauth(credentials)
              continue
            e = Msg.REAUTHENTICATION_IS_NEEDED
          handleOAuthTokenError(e, False)
  return credentials

def waitOnFailure(n, triesLimit, error_code, error_message):
  delta = min(2 ** n, 60)+float(random.randint(1, 1000))/1000
  if n > 3:
    writeStderr(f'Temporary error: {error_code} - {error_message}, Backing off: {int(delta)} seconds, Retry: {n}/{triesLimit}\n')
    flushStderr()
  time.sleep(delta)
  if GC.Values[GC.SHOW_API_CALLS_RETRY_DATA]:
    incrAPICallsRetryData(error_message, delta)

def clearServiceCache(service):
  if hasattr(service._http, 'http') and hasattr(service._http.http, 'cache'):
    if service._http.http.cache is None:
      return False
    service._http.http.cache = None
    return True
  if hasattr(service._http, 'cache'):
    if service._http.cache is None:
      return False
    service._http.cache = None
    return True
  return False

DISCOVERY_URIS = [googleapiclient.discovery.V1_DISCOVERY_URI, googleapiclient.discovery.V2_DISCOVERY_URI]

# Used for API.CLOUDRESOURCEMANAGER, API.SERVICEUSAGE, API.IAM
def getAPIService(api, httpObj):
  api, version, v2discovery = API.getVersion(api)
  return googleapiclient.discovery.build(api, version, http=httpObj, cache_discovery=False,
                                         discoveryServiceUrl=DISCOVERY_URIS[v2discovery], static_discovery=False)

def getService(api, httpObj):
### Drive v3beta
#  mapDriveURL = api == API.DRIVE3 and GC.Values[GC.DRIVE_V3_BETA]
  hasLocalJSON = API.hasLocalJSON(api)
  api, version, v2discovery = API.getVersion(api)
  if api in GM.Globals[GM.CURRENT_API_SERVICES] and version in GM.Globals[GM.CURRENT_API_SERVICES][api]:
    service = googleapiclient.discovery.build_from_document(GM.Globals[GM.CURRENT_API_SERVICES][api][version], http=httpObj)
    if GM.Globals[GM.CACHE_DISCOVERY_ONLY]:
      clearServiceCache(service)
    return service
  if not hasLocalJSON:
    triesLimit = 3
    for n in range(1, triesLimit+1):
      try:
        service = googleapiclient.discovery.build(api, version, http=httpObj, cache_discovery=False,
                                                  discoveryServiceUrl=DISCOVERY_URIS[v2discovery], static_discovery=False)
        GM.Globals[GM.CURRENT_API_SERVICES].setdefault(api, {})
        GM.Globals[GM.CURRENT_API_SERVICES][api][version] = service._rootDesc.copy()
### Drive v3beta
#        if mapDriveURL:
#          setattr(service, '_baseUrl', getattr(service, '_baseUrl').replace('/v3/', '/v3beta/'))
        if GM.Globals[GM.CACHE_DISCOVERY_ONLY]:
          clearServiceCache(service)
        return service
      except googleapiclient.errors.UnknownApiNameOrVersion as e:
        systemErrorExit(GOOGLE_API_ERROR_RC, Msg.UNKNOWN_API_OR_VERSION.format(str(e), __author__))
      except (googleapiclient.errors.InvalidJsonError, KeyError, ValueError) as e:
        if n != triesLimit:
          waitOnFailure(n, triesLimit, INVALID_JSON_RC, str(e))
          continue
        systemErrorExit(INVALID_JSON_RC, str(e))
      except (http_client.ResponseNotReady, OSError, googleapiclient.errors.HttpError) as e:
        errMsg = f'Connection error: {str(e) or repr(e)}'
        if n != triesLimit:
          waitOnFailure(n, triesLimit, SOCKET_ERROR_RC, errMsg)
          continue
        systemErrorExit(SOCKET_ERROR_RC, errMsg)
      except (httplib2.HttpLib2Error, google.auth.exceptions.TransportError, RuntimeError) as e:
        if n != triesLimit:
          httpObj.connections = {}
          waitOnFailure(n, triesLimit, NETWORK_ERROR_RC, str(e))
          continue
        handleServerError(e)
  disc_file, discovery = readDiscoveryFile(f'{api}-{version}')
  try:
    service = googleapiclient.discovery.build_from_document(discovery, http=httpObj)
    GM.Globals[GM.CURRENT_API_SERVICES].setdefault(api, {})
    GM.Globals[GM.CURRENT_API_SERVICES][api][version] = service._rootDesc.copy()
    if GM.Globals[GM.CACHE_DISCOVERY_ONLY]:
      clearServiceCache(service)
    return service
  except (googleapiclient.errors.InvalidJsonError, KeyError, ValueError) as e:
    invalidDiscoveryJsonExit(disc_file, str(e))
  except IOError as e:
    systemErrorExit(FILE_ERROR_RC, str(e))

def defaultSvcAcctScopes():
  scopesList = API.getSvcAcctScopesList(GC.Values[GC.USER_SERVICE_ACCOUNT_ACCESS_ONLY], False)
  saScopes = {}
  for scope in scopesList:
    if not scope.get('offByDefault'):
      saScopes.setdefault(scope['api'], [])
      saScopes[scope['api']].append(scope['scope'])
  saScopes[API.DRIVEACTIVITY].append(API.DRIVE_SCOPE)
  saScopes[API.DRIVE2] = saScopes[API.DRIVE3]
  saScopes[API.DRIVETD] = saScopes[API.DRIVE3]
  saScopes[API.SHEETSTD] = saScopes[API.SHEETS]
  return saScopes

def _getSvcAcctData():
  if not GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]:
    jsonData = readFile(GC.Values[GC.OAUTH2SERVICE_JSON], continueOnError=True, displayError=True)
    if not jsonData:
      invalidOauth2serviceJsonExit(Msg.NO_DATA)
    try:
      GM.Globals[GM.OAUTH2SERVICE_JSON_DATA] = json.loads(jsonData)
    except (IndexError, KeyError, SyntaxError, TypeError, ValueError) as e:
      invalidOauth2serviceJsonExit(str(e))
    if not GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]:
      systemErrorExit(OAUTH2SERVICE_JSON_REQUIRED_RC, Msg.NO_SVCACCT_ACCESS_ALLOWED)
    requiredFields = ['client_email', 'client_id', 'project_id', 'token_uri']
    key_type = GM.Globals[GM.OAUTH2SERVICE_JSON_DATA].get('key_type', 'default')
    if key_type == 'default':
      requiredFields.extend(['private_key', 'private_key_id'])
    missingFields = []
    for field in requiredFields:
      if field not in GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]:
        missingFields.append(field)
    if missingFields:
      invalidOauth2serviceJsonExit(Msg.MISSING_FIELDS.format(','.join(missingFields)))
# Some old oauth2service.json files have: 'https://accounts.google.com/o/oauth2/auth' which no longer works
    if GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['token_uri'] == 'https://accounts.google.com/o/oauth2/auth':
      GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['token_uri'] = API.GOOGLE_OAUTH2_TOKEN_ENDPOINT
    if API.OAUTH2SA_SCOPES not in GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]:
      GM.Globals[GM.SVCACCT_SCOPES_DEFINED] = False
      GM.Globals[GM.SVCACCT_SCOPES] = defaultSvcAcctScopes()
    else:
      GM.Globals[GM.SVCACCT_SCOPES_DEFINED] = True
      GM.Globals[GM.SVCACCT_SCOPES] = GM.Globals[GM.OAUTH2SERVICE_JSON_DATA].pop(API.OAUTH2SA_SCOPES)

def getSvcAcctCredentials(scopesOrAPI, userEmail, softErrors=False, forceOauth=False):
  _getSvcAcctData()
  if isinstance(scopesOrAPI, str):
    GM.Globals[GM.CURRENT_SVCACCT_API] = scopesOrAPI
    if scopesOrAPI not in API.JWT_APIS:
      GM.Globals[GM.CURRENT_SVCACCT_API_SCOPES] = GM.Globals[GM.SVCACCT_SCOPES].get(scopesOrAPI, [])
    else:
      GM.Globals[GM.CURRENT_SVCACCT_API_SCOPES] = API.JWT_APIS[scopesOrAPI]
    if scopesOrAPI != API.CHAT_EVENTS and not GM.Globals[GM.CURRENT_SVCACCT_API_SCOPES]:
      if softErrors:
        return None
      SvcAcctAPIAccessDeniedExit()
    if scopesOrAPI in {API.PEOPLE, API.PEOPLE_DIRECTORY, API.PEOPLE_OTHERCONTACTS}:
      GM.Globals[GM.CURRENT_SVCACCT_API_SCOPES].append(API.USERINFO_PROFILE_SCOPE)
      if scopesOrAPI in {API.PEOPLE_OTHERCONTACTS}:
        GM.Globals[GM.CURRENT_SVCACCT_API_SCOPES].append(API.PEOPLE_SCOPE)
    elif scopesOrAPI == API.CHAT_EVENTS:
      for chatAPI in [API.CHAT_SPACES, API.CHAT_MEMBERSHIPS, API.CHAT_MESSAGES]:
        GM.Globals[GM.CURRENT_SVCACCT_API_SCOPES].extend(GM.Globals[GM.SVCACCT_SCOPES].get(chatAPI, []))
  else:
    GM.Globals[GM.CURRENT_SVCACCT_API] = ''
    GM.Globals[GM.CURRENT_SVCACCT_API_SCOPES] = scopesOrAPI
  key_type = GM.Globals[GM.OAUTH2SERVICE_JSON_DATA].get('key_type', 'default')
  if not GM.Globals[GM.CURRENT_SVCACCT_API] or scopesOrAPI not in API.JWT_APIS or forceOauth:
    try:
      if key_type == 'default':
        credentials = google.oauth2.service_account.Credentials.from_service_account_info(GM.Globals[GM.OAUTH2SERVICE_JSON_DATA])
      elif key_type == 'yubikey':
        yksigner = yubikey.YubiKey(GM.Globals[GM.OAUTH2SERVICE_JSON_DATA])
        credentials = google.oauth2.service_account.Credentials._from_signer_and_info(yksigner,
                                                                                      GM.Globals[GM.OAUTH2SERVICE_JSON_DATA])
      elif key_type == 'signjwt':
        sjsigner = signjwtSignJwt(GM.Globals[GM.OAUTH2SERVICE_JSON_DATA])
        credentials = signjwtCredentials._from_signer_and_info(sjsigner.sign,
                                                               GM.Globals[GM.OAUTH2SERVICE_JSON_DATA])
    except (ValueError, IndexError, KeyError) as e:
      if softErrors:
        return None
      invalidOauth2serviceJsonExit(str(e))
    credentials = credentials.with_scopes(GM.Globals[GM.CURRENT_SVCACCT_API_SCOPES])
  else:
    audience = f'https://{scopesOrAPI}.googleapis.com/'
    try:
      if key_type == 'default':
        credentials = JWTCredentials.from_service_account_info(GM.Globals[GM.OAUTH2SERVICE_JSON_DATA],
                                                               audience=audience)
      elif key_type == 'yubikey':
        yksigner = yubikey.YubiKey(GM.Globals[GM.OAUTH2SERVICE_JSON_DATA])
        credentials = JWTCredentials._from_signer_and_info(yksigner,
                                                           GM.Globals[GM.OAUTH2SERVICE_JSON_DATA],
                                                           audience=audience)
      elif key_type == 'signjwt':
        sjsigner = signjwtSignJwt(GM.Globals[GM.OAUTH2SERVICE_JSON_DATA])
        credentials = signjwtJWTCredentials._from_signer_and_info(sjsigner,
                                                                  GM.Globals[GM.OAUTH2SERVICE_JSON_DATA],
                                                                  audience=audience)
      credentials.project_id = GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['project_id']
    except (ValueError, IndexError, KeyError) as e:
      if softErrors:
        return None
      invalidOauth2serviceJsonExit(str(e))
  GM.Globals[GM.CURRENT_SVCACCT_USER] = userEmail
  if userEmail:
    credentials = credentials.with_subject(userEmail)
  GM.Globals[GM.ADMIN] = GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['client_email']
  GM.Globals[GM.OAUTH2SERVICE_CLIENT_ID] = GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['client_id']
  return credentials

def getGDataOAuthToken(gdataObj, credentials=None):
  if not credentials:
    credentials = getClientCredentials(refreshOnly=True)
  try:
    credentials.refresh(transportCreateRequest())
  except (httplib2.HttpLib2Error, google.auth.exceptions.TransportError, RuntimeError) as e:
    handleServerError(e)
  except google.auth.exceptions.RefreshError as e:
    if isinstance(e.args, tuple):
      e = e.args[0]
    handleOAuthTokenError(e, False)
  gdataObj.additional_headers['Authorization'] = f'Bearer {credentials.token}'
  if not GC.Values[GC.DOMAIN]:
    GC.Values[GC.DOMAIN] = GM.Globals[GM.DECODED_ID_TOKEN].get('hd', 'UNKNOWN').lower()
  if not GC.Values[GC.CUSTOMER_ID]:
    GC.Values[GC.CUSTOMER_ID] = GC.MY_CUSTOMER
  GM.Globals[GM.ADMIN] = GM.Globals[GM.DECODED_ID_TOKEN].get('email', 'UNKNOWN').lower()
  GM.Globals[GM.OAUTH2_CLIENT_ID] = credentials.client_id
  gdataObj.domain = GC.Values[GC.DOMAIN]
  gdataObj.source = GAM_USER_AGENT
  return True

def checkGDataError(e, service):
  error = e.args
  reason = error[0].get('reason', '')
  body = error[0].get('body', '').decode(UTF8)
  # First check for errors that need special handling
  if reason in ['Token invalid - Invalid token: Stateless token expired', 'Token invalid - Invalid token: Token not found', 'gone']:
    keep_domain = service.domain
    getGDataOAuthToken(service)
    service.domain = keep_domain
    return (GDATA.TOKEN_EXPIRED, reason)
  error_code = getattr(e, 'error_code', 600)
  if GC.Values[GC.DEBUG_LEVEL] > 0:
    writeStdout(f'{ERROR_PREFIX} {error_code}: {reason}, {body}\n')
  if error_code == 600:
    if (body.startswith('Quota exceeded for the current request') or
        body.startswith('Quota exceeded for quota metric') or
        body.startswith('Request rate higher than configured')):
      return (GDATA.QUOTA_EXCEEDED, body)
    if (body.startswith('Photo delete failed') or
        body.startswith('Upload photo failed') or
        body.startswith('Photo query failed')):
      return (GDATA.NOT_FOUND, body)
    if body.startswith(GDATA.API_DEPRECATED_MSG):
      return (GDATA.API_DEPRECATED, body)
    if reason == 'Too Many Requests':
      return (GDATA.QUOTA_EXCEEDED, reason)
    if reason == 'Bad Gateway':
      return (GDATA.BAD_GATEWAY, reason)
    if reason == 'Gateway Timeout':
      return (GDATA.GATEWAY_TIMEOUT, reason)
    if reason == 'Service Unavailable':
      return (GDATA.SERVICE_UNAVAILABLE, reason)
    if reason == 'Service <jotspot> disabled by G Suite admin.':
      return (GDATA.FORBIDDEN, reason)
    if reason == 'Internal Server Error':
      return (GDATA.INTERNAL_SERVER_ERROR, reason)
    if reason == 'Token invalid - Invalid token: Token disabled, revoked, or expired.':
      return (GDATA.TOKEN_INVALID, 'Token disabled, revoked, or expired. Please delete and re-create oauth.txt')
    if reason == 'Token invalid - AuthSub token has wrong scope':
      return (GDATA.INSUFFICIENT_PERMISSIONS, reason)
    if reason.startswith('Only administrators can request entries belonging to'):
      return (GDATA.INSUFFICIENT_PERMISSIONS, reason)
    if reason == 'You are not authorized to access this API':
      return (GDATA.INSUFFICIENT_PERMISSIONS, reason)
    if reason == 'Invalid domain.':
      return (GDATA.INVALID_DOMAIN, reason)
    if reason.startswith('You are not authorized to perform operations on the domain'):
      return (GDATA.INVALID_DOMAIN, reason)
    if reason == 'Bad Request':
      if 'already exists' in body:
        return (GDATA.ENTITY_EXISTS, Msg.DUPLICATE)
      return (GDATA.BAD_REQUEST, body)
    if reason == 'Forbidden':
      return (GDATA.FORBIDDEN, body)
    if reason == 'Not Found':
      return (GDATA.NOT_FOUND, Msg.DOES_NOT_EXIST)
    if reason == 'Not Implemented':
      return (GDATA.NOT_IMPLEMENTED, body)
    if reason == 'Precondition Failed':
      return (GDATA.PRECONDITION_FAILED, reason)
  elif error_code == 602:
    if body.startswith(GDATA.API_DEPRECATED_MSG):
      return (GDATA.API_DEPRECATED, body)
    if reason == 'Bad Request':
      return (GDATA.BAD_REQUEST, body)
  elif error_code == 610:
    if reason == 'Service <jotspot> disabled by G Suite admin.':
      return (GDATA.FORBIDDEN, reason)

  # We got a "normal" error, define the mapping below
  error_code_map = {
    1000: reason,
    1001: reason,
    1002: 'Unauthorized and forbidden',
    1100: 'User deleted recently',
    1200: 'Domain user limit exceeded',
    1201: 'Domain alias limit exceeded',
    1202: 'Domain suspended',
    1203: 'Domain feature unavailable',
    1300: f'Entity {getattr(e, "invalidInput", "<unknown>")} exists',
    1301: f'Entity {getattr(e, "invalidInput", "<unknown>")} Does Not Exist',
    1302: 'Entity Name Is Reserved',
    1303: f'Entity {getattr(e, "invalidInput", "<unknown>")} name not valid',
    1306: f'{getattr(e, "invalidInput", "<unknown>")} has members. Cannot delete.',
    1317: f'Invalid input {getattr(e, "invalidInput", "<unknown>")}, reason {getattr(e, "reason", "<unknown>")}',
    1400: 'Invalid Given Name',
    1401: 'Invalid Family Name',
    1402: 'Invalid Password',
    1403: 'Invalid Username',
    1404: 'Invalid Hash Function Name',
    1405: 'Invalid Hash Digest Length',
    1406: 'Invalid Email Address',
    1407: 'Invalid Query Parameter Value',
    1408: 'Invalid SSO Signing Key',
    1409: 'Invalid Encryption Public Key',
    1410: 'Feature Unavailable For User',
    1411: 'Invalid Encryption Public Key Format',
    1500: 'Too Many Recipients On Email List',
    1501: 'Too Many Aliases For User',
    1502: 'Too Many Delegates For User',
    1601: 'Duplicate Destinations',
    1602: 'Too Many Destinations',
    1603: 'Invalid Route Address',
    1700: 'Group Cannot Contain Cycle',
    1800: 'Group Cannot Contain Cycle',
    1801: f'Invalid value {getattr(e, "invalidInput", "<unknown>")}',
  }
  return (error_code, error_code_map.get(error_code, f'Unknown Error: {str(e)}'))

def callGData(service, function,
              bailOnInternalServerError=False, softErrors=False,
              throwErrors=None, retryErrors=None, triesLimit=0,
              **kwargs):
  if throwErrors is None:
    throwErrors = []
  if retryErrors is None:
    retryErrors = []
  if triesLimit == 0:
    triesLimit = GC.Values[GC.API_CALLS_TRIES_LIMIT]
  allRetryErrors = GDATA.NON_TERMINATING_ERRORS+retryErrors
  method = getattr(service, function)
  if GC.Values[GC.API_CALLS_RATE_CHECK]:
    checkAPICallsRate()
  for n in range(1, triesLimit+1):
    try:
      return method(**kwargs)
    except (gdata.service.RequestError, gdata.apps.service.AppsForYourDomainException) as e:
      error_code, error_message = checkGDataError(e, service)
      if (n != triesLimit) and (error_code in allRetryErrors):
        if (error_code == GDATA.INTERNAL_SERVER_ERROR and
            bailOnInternalServerError and n == GC.Values[GC.BAIL_ON_INTERNAL_ERROR_TRIES]):
          raise GDATA.ERROR_CODE_EXCEPTION_MAP[error_code](error_message)
        waitOnFailure(n, triesLimit, error_code, error_message)
        continue
      if error_code in throwErrors:
        if error_code in GDATA.ERROR_CODE_EXCEPTION_MAP:
          raise GDATA.ERROR_CODE_EXCEPTION_MAP[error_code](error_message)
        raise
      if softErrors:
        stderrErrorMsg(f'{error_code} - {error_message}{["", ": Giving up."][n > 1]}')
        return None
      if error_code == GDATA.INSUFFICIENT_PERMISSIONS:
        APIAccessDeniedExit()
      systemErrorExit(GOOGLE_API_ERROR_RC, f'{error_code} - {error_message}')
    except (httplib2.HttpLib2Error, google.auth.exceptions.TransportError, RuntimeError) as e:
      if n != triesLimit:
        waitOnFailure(n, triesLimit, NETWORK_ERROR_RC, str(e))
        continue
      handleServerError(e)
    except google.auth.exceptions.RefreshError as e:
      if isinstance(e.args, tuple):
        e = e.args[0]
      handleOAuthTokenError(e, GDATA.SERVICE_NOT_APPLICABLE in throwErrors)
      raise GDATA.ERROR_CODE_EXCEPTION_MAP[GDATA.SERVICE_NOT_APPLICABLE](str(e))
    except (http_client.ResponseNotReady, OSError) as e:
      errMsg = f'Connection error: {str(e) or repr(e)}'
      if n != triesLimit:
        waitOnFailure(n, triesLimit, SOCKET_ERROR_RC, errMsg)
        continue
      if softErrors:
        writeStderr(f'\n{ERROR_PREFIX}{errMsg} - Giving up.\n')
        return None
      systemErrorExit(SOCKET_ERROR_RC, errMsg)

def writeGotMessage(msg):
  if GC.Values[GC.SHOW_GETTINGS_GOT_NL]:
    writeStderr(msg)
  else:
    writeStderr('\r')
    msgLen = len(msg)
    if msgLen < GM.Globals[GM.LAST_GOT_MSG_LEN]:
      writeStderr(msg+' '*(GM.Globals[GM.LAST_GOT_MSG_LEN]-msgLen))
    else:
      writeStderr(msg)
    GM.Globals[GM.LAST_GOT_MSG_LEN] = msgLen
  flushStderr()

def callGDataPages(service, function,
                   pageMessage=None,
                   softErrors=False, throwErrors=None, retryErrors=None,
                   uri=None,
                   **kwargs):
  if throwErrors is None:
    throwErrors = []
  if retryErrors is None:
    retryErrors = []
  nextLink = None
  allResults = []
  totalItems = 0
  while True:
    this_page = callGData(service, function,
                          softErrors=softErrors, throwErrors=throwErrors, retryErrors=retryErrors,
                          uri=uri,
                          **kwargs)
    if this_page:
      nextLink = this_page.GetNextLink()
      pageItems = len(this_page.entry)
      if pageItems == 0:
        nextLink = None
      totalItems += pageItems
      allResults.extend(this_page.entry)
    else:
      nextLink = None
      pageItems = 0
    if pageMessage:
      show_message = pageMessage.replace(TOTAL_ITEMS_MARKER, str(totalItems))
      writeGotMessage(show_message.format(Ent.ChooseGetting(totalItems)))
    if nextLink is None:
      if pageMessage and (pageMessage[-1] != '\n'):
        writeStderr('\r\n')
        flushStderr()
      return allResults
    uri = nextLink.href
    if 'url_params' in kwargs:
      kwargs['url_params'].pop('start-index', None)

def checkGAPIError(e, softErrors=False, retryOnHttpError=False, mapNotFound=True):
  def makeErrorDict(code, reason, message):
    return {'error': {'code': code, 'errors': [{'reason': reason, 'message': message}]}}

  try:
    error = json.loads(e.content.decode(UTF8))
    if GC.Values[GC.DEBUG_LEVEL] > 0:
      writeStdout(f'{ERROR_PREFIX} JSON: {str(error)}\n')
  except (IndexError, KeyError, SyntaxError, TypeError, ValueError):
    eContent = e.content.decode(UTF8) if isinstance(e.content, bytes) else e.content
    lContent = eContent.lower()
    if GC.Values[GC.DEBUG_LEVEL] > 0:
      writeStdout(f'{ERROR_PREFIX} HTTP: {str(eContent)}\n')
    if eContent[0:15] != '<!DOCTYPE html>':
      if (e.resp['status'] == '403') and (lContent.startswith('request rate higher than configured')):
        return (e.resp['status'], GAPI.QUOTA_EXCEEDED, eContent)
      if (e.resp['status'] == '429') and (lContent.startswith('quota exceeded for quota metric')):
        return (e.resp['status'], GAPI.QUOTA_EXCEEDED, eContent)
      if (e.resp['status'] == '502') and ('bad gateway' in lContent):
        return (e.resp['status'], GAPI.BAD_GATEWAY, eContent)
      if (e.resp['status'] == '503') and (lContent.startswith('quota exceeded for the current request')):
        return (e.resp['status'], GAPI.QUOTA_EXCEEDED, eContent)
      if (e.resp['status'] == '504') and ('gateway timeout' in lContent):
        return (e.resp['status'], GAPI.GATEWAY_TIMEOUT, eContent)
    else:
      tg = HTML_TITLE_PATTERN.match(lContent)
      lContent = tg.group(1) if tg else 'bad request'
    if (e.resp['status'] == '403') and ('invalid domain.' in lContent):
      error = makeErrorDict(403, GAPI.NOT_FOUND, 'Domain not found')
    elif (e.resp['status'] == '403') and ('domain cannot use apis.' in lContent):
      error = makeErrorDict(403, GAPI.DOMAIN_CANNOT_USE_APIS, 'Domain cannot use apis')
    elif (e.resp['status'] == '400') and ('invalidssosigningkey' in lContent):
      error = makeErrorDict(400, GAPI.INVALID, 'InvalidSsoSigningKey')
    elif (e.resp['status'] == '400') and ('unknownerror' in lContent):
      error = makeErrorDict(400, GAPI.INVALID, 'UnknownError')
    elif (e.resp['status'] == '400') and ('featureunavailableforuser' in lContent):
      error = makeErrorDict(400, GAPI.SERVICE_NOT_AVAILABLE, 'Feature Unavailable For User')
    elif (e.resp['status'] == '400') and ('entitydoesnotexist' in lContent):
      error = makeErrorDict(400, GAPI.NOT_FOUND, 'Entity Does Not Exist')
    elif (e.resp['status'] == '400') and ('entitynamenotvalid' in lContent):
      error = makeErrorDict(400, GAPI.INVALID_INPUT, 'Entity Name Not Valid')
    elif (e.resp['status'] == '400') and ('failed to parse Content-Range header' in lContent):
      error = makeErrorDict(400, GAPI.BAD_REQUEST, 'Failed to parse Content-Range header')
    elif (e.resp['status'] == '400') and ('request contains an invalid argument' in lContent):
      error = makeErrorDict(400, GAPI.INVALID_ARGUMENT, 'Request contains an invalid argument')
    elif (e.resp['status'] == '404') and ('not found' in lContent):
      error = makeErrorDict(404, GAPI.NOT_FOUND, lContent)
    elif (e.resp['status'] == '404') and ('bad request' in lContent):
      error = makeErrorDict(404, GAPI.BAD_REQUEST, lContent)
    elif retryOnHttpError:
      return (-1, None, eContent)
    elif softErrors:
      stderrErrorMsg(eContent)
      return (0, None, None)
    else:
      systemErrorExit(HTTP_ERROR_RC, eContent)
  requiredScopes = ''
  wwwAuthenticate = e.resp.get('www-authenticate', '')
  if 'insufficient_scope' in wwwAuthenticate:
    mg = re.match(r'.+scope="(.+)"', wwwAuthenticate)
    if mg:
      requiredScopes = mg.group(1).split(' ')
  if 'error' in error:
    http_status = error['error']['code']
    reason = ''
    if 'errors' in error['error'] and 'message' in error['error']['errors'][0]:
      message = error['error']['errors'][0]['message']
      if 'reason' in error['error']['errors'][0]:
        reason = error['error']['errors'][0]['reason']
    elif 'errors' in error['error'] and 'Unknown Error' in error['error']['message'] and 'reason' in error['error']['errors'][0]:
      message = error['error']['errors'][0]['reason']
    else:
      message = error['error']['message']
    status = error['error'].get('status', '')
    lmessage = message.lower() if message is not None else ''
    if http_status == 500:
      if not lmessage or status == 'UNKNOWN':
        if not lmessage:
          message = Msg.UNKNOWN
        error = makeErrorDict(http_status, GAPI.UNKNOWN_ERROR, message)
      elif 'backend error' in lmessage:
        error = makeErrorDict(http_status, GAPI.BACKEND_ERROR, message)
      elif 'internal error encountered' in lmessage:
        error = makeErrorDict(http_status, GAPI.INTERNAL_ERROR, message)
      elif 'role assignment exists: roleassignment' in lmessage:
        error = makeErrorDict(http_status, GAPI.DUPLICATE, message)
      elif 'role assignment exists: roleid' in lmessage:
        error = makeErrorDict(http_status, GAPI.DUPLICATE, message)
      elif 'operation not supported' in lmessage:
        error = makeErrorDict(http_status, GAPI.OPERATION_NOT_SUPPORTED, message)
      elif 'failed status in update settings response' in lmessage:
        error = makeErrorDict(http_status, GAPI.INVALID_INPUT, message)
      elif 'cannot delete a field in use.resource.fields' in lmessage:
        error = makeErrorDict(http_status, GAPI.FIELD_IN_USE, message)
      elif status == 'INTERNAL':
        error = makeErrorDict(http_status, GAPI.INTERNAL_ERROR, message)
    elif http_status == 502:
      if 'bad gateway' in lmessage:
        error = makeErrorDict(http_status, GAPI.BAD_GATEWAY, message)
    elif http_status == 503:
      if message.startswith('quota exceeded for the current request'):
        error = makeErrorDict(http_status, GAPI.QUOTA_EXCEEDED, message)
      elif status == 'UNAVAILABLE' or 'the service is currently unavailable' in lmessage:
        error = makeErrorDict(http_status, GAPI.SERVICE_NOT_AVAILABLE, message)
    elif http_status == 504:
      if 'gateway timeout' in lmessage:
        error = makeErrorDict(http_status, GAPI.GATEWAY_TIMEOUT, message)
    elif http_status == 400:
      if '@attachmentnotvisible' in lmessage:
        error = makeErrorDict(http_status, GAPI.BAD_REQUEST, message)
      elif status == 'INVALID_ARGUMENT':
        error = makeErrorDict(http_status, GAPI.INVALID_ARGUMENT, message)
      elif status == 'FAILED_PRECONDITION' or 'precondition check failed' in lmessage:
        error = makeErrorDict(http_status, GAPI.FAILED_PRECONDITION, message)
      elif 'does not match' in lmessage or 'invalid' in lmessage:
        error = makeErrorDict(http_status, GAPI.INVALID, message)
    elif http_status == 401:
      if 'active session is invalid' in lmessage and reason == 'authError':
#        message += ' Drive SDK API access disabled'
#        message = Msg.SERVICE_NOT_ENABLED.format('Drive')
        error = makeErrorDict(http_status, GAPI.AUTH_ERROR, message)
      elif status == 'PERMISSION_DENIED':
        error = makeErrorDict(http_status, GAPI.PERMISSION_DENIED, message)
      elif status == 'UNAUTHENTICATED':
        error = makeErrorDict(http_status, GAPI.AUTH_ERROR, message)
    elif http_status == 403:
      if 'quota exceeded for quota metric' in lmessage:
        error = makeErrorDict(http_status, GAPI.QUOTA_EXCEEDED, message)
      elif 'the authenticated user cannot access this service' in lmessage:
        error = makeErrorDict(http_status, GAPI.SERVICE_NOT_AVAILABLE, message)
      elif status == 'PERMISSION_DENIED' or 'the caller does not have permission' in lmessage or 'permission iam.serviceaccountkeys' in lmessage:
        if requiredScopes:
          message += f', {Msg.NO_SCOPES_FOR_API.format(API.findAPIforScope(requiredScopes))}'
        error = makeErrorDict(http_status, GAPI.PERMISSION_DENIED, message)
    elif http_status == 404:
      if status == 'NOT_FOUND' or 'requested entity was not found' in lmessage or 'does not exist' in lmessage:
        error = makeErrorDict(http_status, GAPI.NOT_FOUND, message)
    elif http_status == 409:
      if status == 'ALREADY_EXISTS' or 'requested entity already exists' in lmessage:
        error = makeErrorDict(http_status, GAPI.ALREADY_EXISTS, message)
      elif status == 'ABORTED' or 'the operation was aborted' in lmessage:
        error = makeErrorDict(http_status, GAPI.ABORTED, message)
    elif http_status == 412:
      if 'insufficient archived user licenses' in lmessage:
        error = makeErrorDict(http_status, GAPI.INSUFFICIENT_ARCHIVED_USER_LICENSES, message)
    elif http_status == 413:
      if 'request too large' in lmessage:
        error = makeErrorDict(http_status, GAPI.UPLOAD_TOO_LARGE, message)
    elif http_status == 429:
      if status == 'RESOURCE_EXHAUSTED' or 'quota exceeded' in lmessage or 'insufficient quota' in lmessage:
        error = makeErrorDict(http_status, GAPI.QUOTA_EXCEEDED, message)
  else:
    if 'error_description' in error:
      if error['error_description'] == 'Invalid Value':
        message = error['error_description']
        http_status = 400
        error = makeErrorDict(http_status, GAPI.INVALID, message)
      else:
        systemErrorExit(GOOGLE_API_ERROR_RC, str(error))
    else:
      systemErrorExit(GOOGLE_API_ERROR_RC, str(error))
  try:
    reason = error['error']['errors'][0]['reason']
    for messageItem in GAPI.REASON_MESSAGE_MAP.get(reason, []):
      if messageItem[0] in message:
        if reason in [GAPI.NOT_FOUND, GAPI.RESOURCE_NOT_FOUND] and mapNotFound:
          message = Msg.DOES_NOT_EXIST
        reason = messageItem[1]
        break
    if reason == GAPI.INVALID_SHARING_REQUEST:
      loc = message.find('User message: ')
      if loc != -1:
        message = message[loc+15:]
    else:
      loc = message.find('User message: ""')
      if loc != -1:
        message = message[:loc+14]+f'"{reason}"'
  except KeyError:
    reason = f'{http_status}'
  return (http_status, reason, message)

def callGAPI(service, function,
             bailOnInternalError=False, bailOnTransientError=False, bailOnInvalidError=False,
             softErrors=False, mapNotFound=True,
             throwReasons=None, retryReasons=None, triesLimit=0,
             **kwargs):
  if throwReasons is None:
    throwReasons = []
  if retryReasons is None:
    retryReasons = []
  if triesLimit == 0:
    triesLimit = GC.Values[GC.API_CALLS_TRIES_LIMIT]
  allRetryReasons = GAPI.DEFAULT_RETRY_REASONS+retryReasons
  method = getattr(service, function)
  svcparms = dict(list(kwargs.items())+GM.Globals[GM.EXTRA_ARGS_LIST])
  if GC.Values[GC.API_CALLS_RATE_CHECK]:
    checkAPICallsRate()
  for n in range(1, triesLimit+1):
    try:
      return method(**svcparms).execute()
    except googleapiclient.errors.HttpError as e:
      http_status, reason, message = checkGAPIError(e, softErrors=softErrors, retryOnHttpError=n < 3, mapNotFound=mapNotFound)
      if http_status == -1:
        # The error detail indicated that we should retry this request
        # We'll refresh credentials and make another pass
        try:
#          service._http.credentials.refresh(getHttpObj())
          service._http.credentials.refresh(transportCreateRequest())
        except TypeError:
          systemErrorExit(HTTP_ERROR_RC, message)
        continue
      if http_status == 0:
        return None
      if (n != triesLimit) and ((reason in allRetryReasons) or
                             (GC.Values[GC.RETRY_API_SERVICE_NOT_AVAILABLE] and (reason == GAPI.SERVICE_NOT_AVAILABLE))):
        if (reason in [GAPI.INTERNAL_ERROR, GAPI.BACKEND_ERROR] and
            bailOnInternalError and n == GC.Values[GC.BAIL_ON_INTERNAL_ERROR_TRIES]):
          raise GAPI.REASON_EXCEPTION_MAP[reason](message)
        if (reason in [GAPI.INVALID] and
            bailOnInvalidError and n == GC.Values[GC.BAIL_ON_INTERNAL_ERROR_TRIES]):
          raise GAPI.REASON_EXCEPTION_MAP[reason](message)
        waitOnFailure(n, triesLimit, reason, message)
        if reason == GAPI.TRANSIENT_ERROR and bailOnTransientError:
          raise GAPI.REASON_EXCEPTION_MAP[reason](message)
        continue
      if reason in throwReasons:
        if reason in GAPI.REASON_EXCEPTION_MAP:
          raise GAPI.REASON_EXCEPTION_MAP[reason](message)
        raise e
      if softErrors:
        stderrErrorMsg(f'{http_status}: {reason} - {message}{["", ": Giving up."][n > 1]}')
        return None
      if reason == GAPI.INSUFFICIENT_PERMISSIONS:
        APIAccessDeniedExit()
      systemErrorExit(HTTP_ERROR_RC, formatHTTPError(http_status, reason, message))
    except googleapiclient.errors.MediaUploadSizeError as e:
      raise e
    except (httplib2.HttpLib2Error, google.auth.exceptions.TransportError, RuntimeError) as e:
      if n != triesLimit:
        service._http.connections = {}
        waitOnFailure(n, triesLimit, NETWORK_ERROR_RC, str(e))
        continue
      handleServerError(e)
    except google.auth.exceptions.RefreshError as e:
      if isinstance(e.args, tuple):
        e = e.args[0]
      handleOAuthTokenError(e, GAPI.SERVICE_NOT_AVAILABLE in throwReasons)
      raise GAPI.REASON_EXCEPTION_MAP[GAPI.SERVICE_NOT_AVAILABLE](str(e))
    except (http_client.ResponseNotReady, OSError) as e:
      errMsg = f'Connection error: {str(e) or repr(e)}'
      if n != triesLimit:
        waitOnFailure(n, triesLimit, SOCKET_ERROR_RC, errMsg)
        continue
      if softErrors:
        writeStderr(f'\n{ERROR_PREFIX}{errMsg} - Giving up.\n')
        return None
      systemErrorExit(SOCKET_ERROR_RC, errMsg)
    except ValueError as e:
      if clearServiceCache(service):
        continue
      systemErrorExit(GOOGLE_API_ERROR_RC, str(e))
    except TypeError as e:
      systemErrorExit(GOOGLE_API_ERROR_RC, str(e))

def _showGAPIpagesResult(results, pageItems, totalItems, pageMessage, messageAttribute, entityType):
  showMessage = pageMessage.replace(TOTAL_ITEMS_MARKER, str(totalItems))
  if pageItems:
    if messageAttribute:
      firstItem = results[0] if pageItems > 0 else {}
      lastItem = results[-1] if pageItems > 1 else firstItem
      if isinstance(messageAttribute, str):
        firstItem = str(firstItem.get(messageAttribute, ''))
        lastItem = str(lastItem.get(messageAttribute, ''))
      else:
        for attr in messageAttribute:
          firstItem = firstItem.get(attr, {})
          lastItem = lastItem.get(attr, {})
        firstItem = str(firstItem)
        lastItem = str(lastItem)
      showMessage = showMessage.replace(FIRST_ITEM_MARKER, firstItem)
      showMessage = showMessage.replace(LAST_ITEM_MARKER, lastItem)
  else:
    showMessage = showMessage.replace(FIRST_ITEM_MARKER, '')
    showMessage = showMessage.replace(LAST_ITEM_MARKER, '')
  writeGotMessage(showMessage.replace('{0}', str(Ent.Choose(entityType, totalItems))))

def _processGAPIpagesResult(results, items, allResults, totalItems, pageMessage, messageAttribute, entityType):
  if results:
    pageToken = results.get('nextPageToken')
    if items in results:
      pageItems = len(results[items])
      totalItems += pageItems
      if allResults is not None:
        allResults.extend(results[items])
    else:
      results = {items: []}
      pageItems = 0
  else:
    pageToken = None
    results = {items: []}
    pageItems = 0
  if pageMessage:
    _showGAPIpagesResult(results[items], pageItems, totalItems, pageMessage, messageAttribute, entityType)
  return (pageToken, totalItems)

def _finalizeGAPIpagesResult(pageMessage):
  if pageMessage and (pageMessage[-1] != '\n'):
    writeStderr('\r\n')
    flushStderr()

def callGAPIpages(service, function, items,
                  pageMessage=None, messageAttribute=None, maxItems=0, noFinalize=False,
                  throwReasons=None, retryReasons=None,
                  pageArgsInBody=False,
                  **kwargs):
  if throwReasons is None:
    throwReasons = []
  if retryReasons is None:
    retryReasons = []
  allResults = []
  totalItems = 0
  maxArg = ''
  if maxItems:
    maxResults = kwargs.get('maxResults', 0)
    if maxResults:
      maxArg = 'maxResults'
    else:
      maxResults = kwargs.get('pageSize', 0)
      if maxResults:
        maxArg = 'pageSize'
  if pageArgsInBody:
    kwargs.setdefault('body', {})
  entityType = Ent.Getting() if pageMessage else None
  while True:
    if maxArg and maxItems-totalItems < maxResults:
      kwargs[maxArg] = maxItems-totalItems
    results = callGAPI(service, function,
                       throwReasons=throwReasons, retryReasons=retryReasons,
                       **kwargs)
    pageToken, totalItems = _processGAPIpagesResult(results, items, allResults, totalItems, pageMessage, messageAttribute, entityType)
    if not pageToken or (maxItems and totalItems >= maxItems):
      if not noFinalize:
        _finalizeGAPIpagesResult(pageMessage)
      return allResults
    if pageArgsInBody:
      kwargs['body']['pageToken'] = pageToken
    else:
      kwargs['pageToken'] = pageToken

def yieldGAPIpages(service, function, items,
                   pageMessage=None, messageAttribute=None, maxItems=0, noFinalize=False,
                   throwReasons=None, retryReasons=None,
                   pageArgsInBody=False,
                   **kwargs):
  if throwReasons is None:
    throwReasons = []
  if retryReasons is None:
    retryReasons = []
  totalItems = 0
  maxArg = ''
  if maxItems:
    maxResults = kwargs.get('maxResults', 0)
    if maxResults:
      maxArg = 'maxResults'
    else:
      maxResults = kwargs.get('pageSize', 0)
      if maxResults:
        maxArg = 'pageSize'
  if pageArgsInBody:
    kwargs.setdefault('body', {})
  entityType = Ent.Getting() if pageMessage else None
  while True:
    if maxArg and maxItems-totalItems < maxResults:
      kwargs[maxArg] = maxItems-totalItems
    results = callGAPI(service, function,
                       throwReasons=throwReasons, retryReasons=retryReasons,
                       **kwargs)
    if results:
      pageToken = results.get('nextPageToken')
      if items in results:
        pageItems = len(results[items])
        totalItems += pageItems
      else:
        results = {items: []}
        pageItems = 0
    else:
      pageToken = None
      results = {items: []}
      pageItems = 0
    if pageMessage:
      _showGAPIpagesResult(results[items], pageItems, totalItems, pageMessage, messageAttribute, entityType)
    yield results[items]
    if not pageToken or (maxItems and totalItems >= maxItems):
      if not noFinalize:
        _finalizeGAPIpagesResult(pageMessage)
      return
    if pageArgsInBody:
      kwargs['body']['pageToken'] = pageToken
    else:
      kwargs['pageToken'] = pageToken

def callGAPIitems(service, function, items,
                  throwReasons=None, retryReasons=None,
                  **kwargs):
  if throwReasons is None:
    throwReasons = []
  if retryReasons is None:
    retryReasons = []
  results = callGAPI(service, function,
                     throwReasons=throwReasons, retryReasons=retryReasons,
                     **kwargs)
  if results:
    return results.get(items, [])
  return []

def readDiscoveryFile(api_version):
  disc_filename = f'{api_version}.json'
  disc_file = os.path.join(GM.Globals[GM.GAM_PATH], disc_filename)
  if hasattr(sys, '_MEIPASS'):
    json_string = readFile(os.path.join(sys._MEIPASS, disc_filename), continueOnError=True, displayError=True) #pylint: disable=no-member
  elif os.path.isfile(disc_file):
    json_string = readFile(disc_file, continueOnError=True, displayError=True)
  else:
    json_string = None
  if not json_string:
    invalidDiscoveryJsonExit(disc_file, Msg.NO_DATA)
  try:
    discovery = json.loads(json_string)
    return (disc_file, discovery)
  except (IndexError, KeyError, SyntaxError, TypeError, ValueError) as e:
    invalidDiscoveryJsonExit(disc_file, str(e))

def buildGAPIObject(api, credentials=None):
  if credentials is None:
    credentials = getClientCredentials(api=api, refreshOnly=True)
  httpObj = transportAuthorizedHttp(credentials, http=getHttpObj(cache=GM.Globals[GM.CACHE_DIR]))
  service = getService(api, httpObj)
  if not GC.Values[GC.ENABLE_DASA]:
    try:
      API_Scopes = set(list(service._rootDesc['auth']['oauth2']['scopes']))
    except KeyError:
      API_Scopes = set(API.VAULT_SCOPES) if api == API.VAULT else set()
    GM.Globals[GM.CURRENT_CLIENT_API] = api
    GM.Globals[GM.CURRENT_CLIENT_API_SCOPES] = API_Scopes.intersection(GM.Globals[GM.CREDENTIALS_SCOPES])
    if api not in API.SCOPELESS_APIS and not GM.Globals[GM.CURRENT_CLIENT_API_SCOPES]:
      systemErrorExit(NO_SCOPES_FOR_API_RC, Msg.NO_SCOPES_FOR_API.format(API.getAPIName(api)))
    if not GC.Values[GC.DOMAIN]:
      GC.Values[GC.DOMAIN] = GM.Globals[GM.DECODED_ID_TOKEN].get('hd', 'UNKNOWN').lower()
    if not GC.Values[GC.CUSTOMER_ID]:
      GC.Values[GC.CUSTOMER_ID] = GC.MY_CUSTOMER
    GM.Globals[GM.ADMIN] = GM.Globals[GM.DECODED_ID_TOKEN].get('email', 'UNKNOWN').lower()
    GM.Globals[GM.OAUTH2_CLIENT_ID] = credentials.client_id
  return service

def getSaUser(user):
  currentClientAPI = GM.Globals[GM.CURRENT_CLIENT_API]
  currentClientAPIScopes = GM.Globals[GM.CURRENT_CLIENT_API_SCOPES]
  userEmail = convertUIDtoEmailAddress(user) if user else None
  GM.Globals[GM.CURRENT_CLIENT_API] = currentClientAPI
  GM.Globals[GM.CURRENT_CLIENT_API_SCOPES] = currentClientAPIScopes
  return userEmail

def buildGAPIServiceObject(api, user, i=0, count=0, displayError=True):
  userEmail = getSaUser(user)
  httpObj = getHttpObj(cache=GM.Globals[GM.CACHE_DIR])
  service = getService(api, httpObj)
  if api == API.MEET_BETA:
    api = API.MEET
  credentials = getSvcAcctCredentials(api, userEmail)
  request = transportCreateRequest(httpObj)
  triesLimit = 3
  for n in range(1, triesLimit+1):
    try:
      credentials.refresh(request)
      service._http = transportAuthorizedHttp(credentials, http=httpObj)
      return (userEmail, service)
    except (httplib2.HttpLib2Error, google.auth.exceptions.TransportError, RuntimeError) as e:
      if n != triesLimit:
        httpObj.connections = {}
        waitOnFailure(n, triesLimit, NETWORK_ERROR_RC, str(e))
        continue
      handleServerError(e)
    except google.auth.exceptions.RefreshError as e:
      if isinstance(e.args, tuple):
        e = e.args[0]
      if n < triesLimit:
        if isinstance(e, str):
          eContent = e
        else:
          eContent = e.content.decode(UTF8) if isinstance(e.content, bytes) else e.content
        if eContent[0:15] == '<!DOCTYPE html>':
          if GC.Values[GC.DEBUG_LEVEL] > 0:
            writeStdout(f'{ERROR_PREFIX} HTTP: {str(eContent)}\n')
          lContent = eContent.lower()
          tg = HTML_TITLE_PATTERN.match(lContent)
          lContent = tg.group(1) if tg else ''
          if lContent.startswith('Error 502 (Server Error)'):
            time.sleep(30)
            continue
      handleOAuthTokenError(e, True, displayError, i, count)
      return (userEmail, None)

def buildGAPIObjectNoAuthentication(api):
  httpObj = getHttpObj(cache=GM.Globals[GM.CACHE_DIR])
  service = getService(api, httpObj)
  return service

def initGDataObject(gdataObj, api):
  GM.Globals[GM.CURRENT_CLIENT_API] = api
  credentials = getClientCredentials(noDASA=True, refreshOnly=True)
  GM.Globals[GM.CURRENT_CLIENT_API_SCOPES] = API.getClientScopesSet(api).intersection(GM.Globals[GM.CREDENTIALS_SCOPES])
  if not GM.Globals[GM.CURRENT_CLIENT_API_SCOPES]:
    systemErrorExit(NO_SCOPES_FOR_API_RC, Msg.NO_SCOPES_FOR_API.format(API.getAPIName(api)))
  getGDataOAuthToken(gdataObj, credentials)
  if GC.Values[GC.DEBUG_LEVEL] > 0:
    gdataObj.debug = True
  return gdataObj

def getGDataUserCredentials(api, user, i, count):
  userEmail = getSaUser(user)
  credentials = getSvcAcctCredentials(api, userEmail)
  request = transportCreateRequest()
  try:
    credentials.refresh(request)
    return (userEmail, credentials)
  except (httplib2.HttpLib2Error, google.auth.exceptions.TransportError, RuntimeError) as e:
    handleServerError(e)
  except google.auth.exceptions.RefreshError as e:
    if isinstance(e.args, tuple):
      e = e.args[0]
    handleOAuthTokenError(e, True, True, i, count)
    return (userEmail, None)

def getContactsObject():
  contactsObject = initGDataObject(gdata.apps.contacts.service.ContactsService(contactFeed=True),
                                   API.CONTACTS)
  return (GC.Values[GC.DOMAIN], contactsObject)

def getContactsQuery(**kwargs):
  if GC.Values[GC.NO_VERIFY_SSL]:
    ssl._create_default_https_context = ssl._create_unverified_context
  return gdata.apps.contacts.service.ContactsQuery(**kwargs)

def getEmailAuditObject():
  return initGDataObject(gdata.apps.audit.service.AuditService(), API.EMAIL_AUDIT)

def getUserEmailFromID(uid, cd):
  try:
    result = callGAPI(cd.users(), 'get',
                      throwReasons=GAPI.USER_GET_THROW_REASONS,
                      userKey=uid, fields='primaryEmail')
    return result.get('primaryEmail')
  except (GAPI.userNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden,
          GAPI.badRequest, GAPI.backendError, GAPI.systemError):
    return None

def getGroupEmailFromID(uid, cd):
  try:
    result = callGAPI(cd.groups(), 'get',
                      throwReasons=GAPI.GROUP_GET_THROW_REASONS,
                      groupKey=uid, fields='email')
    return result.get('email')
  except (GAPI.groupNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden, GAPI.badRequest):
    return None

def getServiceAccountEmailFromID(account_id, sal=None):
  if sal is None:
    sal = buildGAPIObject(API.SERVICEACCOUNTLOOKUP)
  try:
    certs = callGAPI(sal.serviceaccounts(), 'lookup',
                     throwReasons = [GAPI.BAD_REQUEST, GAPI.NOT_FOUND, GAPI.RESOURCE_NOT_FOUND,  GAPI.INVALID_ARGUMENT],
                     account=account_id)
  except (GAPI.badRequest, GAPI.notFound, GAPI.resourceNotFound, GAPI.invalidArgument):
    return None
  sa_cn_rx = r'CN=(.+)\.(.+)\.iam\.gservice.*'
  sa_emails = []
  for _, raw_cert in certs.items():
    cert = x509.load_pem_x509_certificate(raw_cert.encode(), default_backend())
    # suppress crytography warning due to long service account email
    with warnings.catch_warnings():
      warnings.filterwarnings('ignore', message='.*Attribute\'s length.*')
      mg = re.match(sa_cn_rx, cert.issuer.rfc4514_string())
    if mg:
      sa_email = f'{mg.group(1)}@{mg.group(2)}.iam.gserviceaccount.com'
      if sa_email not in sa_emails:
        sa_emails.append(sa_email)
  return GC.Values[GC.CSV_OUTPUT_FIELD_DELIMITER].join(sa_emails)

# Convert UID to email address and type
def convertUIDtoEmailAddressWithType(emailAddressOrUID, cd=None, sal=None, emailTypes=None,
                                     checkForCustomerId=False, ciGroupsAPI=False, aliasAllowed=True):
  if emailTypes is None:
    emailTypes = ['user']
  elif not isinstance(emailTypes, list):
    emailTypes = [emailTypes] if emailTypes != 'any' else ['user', 'group']
  if checkForCustomerId and (emailAddressOrUID == GC.Values[GC.CUSTOMER_ID]):
    return (emailAddressOrUID, 'email')
  normalizedEmailAddressOrUID = normalizeEmailAddressOrUID(emailAddressOrUID, ciGroupsAPI=ciGroupsAPI)
  if ciGroupsAPI and emailAddressOrUID.startswith('groups/'):
    return emailAddressOrUID
  if normalizedEmailAddressOrUID.find('@') > 0 and aliasAllowed:
    return (normalizedEmailAddressOrUID, 'email')
  if cd is None:
    cd = buildGAPIObject(API.DIRECTORY)
  if 'user' in emailTypes and 'group' in emailTypes:
    # Google User IDs *TEND* to be integers while groups tend to have letters
    # thus we can optimize which check we try first. We'll still check
    # both since there is no guarantee this will always be true.
    if normalizedEmailAddressOrUID.isdigit():
      uid = getUserEmailFromID(normalizedEmailAddressOrUID, cd)
      if uid:
        return (uid, 'user')
      uid = getGroupEmailFromID(normalizedEmailAddressOrUID, cd)
      if uid:
        return (uid, 'group')
    else:
      uid = getGroupEmailFromID(normalizedEmailAddressOrUID, cd)
      if uid:
        return (uid, 'group')
      uid = getUserEmailFromID(normalizedEmailAddressOrUID, cd)
      if uid:
        return (uid, 'user')
  elif 'user' in emailTypes:
    uid = getUserEmailFromID(normalizedEmailAddressOrUID, cd)
    if uid:
      return (uid, 'user')
  elif 'group' in emailTypes:
    uid = getGroupEmailFromID(normalizedEmailAddressOrUID, cd)
    if uid:
      return (uid, 'group')
  if 'resource' in emailTypes:
    try:
      result = callGAPI(cd.resources().calendars(), 'get',
                        throwReasons=[GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN],
                        calendarResourceId=normalizedEmailAddressOrUID,
                        customer=GC.Values[GC.CUSTOMER_ID], fields='resourceEmail')
      if 'resourceEmail' in result:
        return (result['resourceEmail'].lower(), 'resource')
    except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden):
      pass
  if 'serviceaccount' in emailTypes:
    uid = getServiceAccountEmailFromID(normalizedEmailAddressOrUID, sal)
    if uid:
      return (uid, 'serviceaccount')
  return (normalizedEmailAddressOrUID, 'unknown')

NON_EMAIL_MEMBER_PREFIXES = (
                              "cbcm-browser.",
                              "chrome-os-device.",
                            )
# Convert UID to email address
def convertUIDtoEmailAddress(emailAddressOrUID, cd=None, emailTypes=None,
                             checkForCustomerId=False, ciGroupsAPI=False, aliasAllowed=True):
  if ciGroupsAPI:
    if emailAddressOrUID.startswith(NON_EMAIL_MEMBER_PREFIXES):
      return emailAddressOrUID
    normalizedEmailAddressOrUID = normalizeEmailAddressOrUID(emailAddressOrUID, ciGroupsAPI=ciGroupsAPI)
    if normalizedEmailAddressOrUID.startswith(NON_EMAIL_MEMBER_PREFIXES):
      return normalizedEmailAddressOrUID
  email, _ = convertUIDtoEmailAddressWithType(emailAddressOrUID, cd, emailTypes,
                                              checkForCustomerId, ciGroupsAPI, aliasAllowed)
  return email

# Convert email address to User/Group UID; called immediately after getting email address from command line
def convertEmailAddressToUID(emailAddressOrUID, cd=None, emailType='user', savedLocation=None):
  normalizedEmailAddressOrUID = normalizeEmailAddressOrUID(emailAddressOrUID)
  if normalizedEmailAddressOrUID.find('@') == -1:
    return normalizedEmailAddressOrUID
  if cd is None:
    cd = buildGAPIObject(API.DIRECTORY)
  if emailType != 'group':
    try:
      return callGAPI(cd.users(), 'get',
                      throwReasons=GAPI.USER_GET_THROW_REASONS,
                      userKey=normalizedEmailAddressOrUID, fields='id')['id']
    except (GAPI.userNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden,
            GAPI.badRequest, GAPI.backendError, GAPI.systemError):
      if emailType == 'user':
        if savedLocation is not None:
          Cmd.SetLocation(savedLocation)
        entityDoesNotExistExit(Ent.USER, normalizedEmailAddressOrUID, errMsg=getPhraseDNEorSNA(normalizedEmailAddressOrUID))
  try:
    return callGAPI(cd.groups(), 'get',
                    throwReasons=GAPI.GROUP_GET_THROW_REASONS, retryReasons=GAPI.GROUP_GET_RETRY_REASONS,
                    groupKey=normalizedEmailAddressOrUID, fields='id')['id']
  except (GAPI.groupNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden, GAPI.badRequest, GAPI.invalid, GAPI.systemError):
    if savedLocation is not None:
      Cmd.SetLocation(savedLocation)
    entityDoesNotExistExit([Ent.USER, Ent.GROUP][emailType == 'group'], normalizedEmailAddressOrUID, errMsg=getPhraseDNEorSNA(normalizedEmailAddressOrUID))

# Convert User UID from API call to email address
def convertUserIDtoEmail(uid, cd=None):
  primaryEmail = GM.Globals[GM.MAP_USER_ID_TO_NAME].get(uid)
  if not primaryEmail:
    if cd is None:
      cd = buildGAPIObject(API.DIRECTORY)
    try:
      primaryEmail = callGAPI(cd.users(), 'get',
                              throwReasons=GAPI.USER_GET_THROW_REASONS,
                              userKey=uid, fields='primaryEmail')['primaryEmail']
    except (GAPI.userNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden,
            GAPI.badRequest, GAPI.backendError, GAPI.systemError):
      primaryEmail = f'uid:{uid}'
    GM.Globals[GM.MAP_USER_ID_TO_NAME][uid] = primaryEmail
  return primaryEmail

# Convert UID to split email address
# Return (foo@bar.com, foo, bar.com)
def splitEmailAddressOrUID(emailAddressOrUID):
  normalizedEmailAddressOrUID = normalizeEmailAddressOrUID(emailAddressOrUID)
  atLoc = normalizedEmailAddressOrUID.find('@')
  if atLoc > 0:
    return (normalizedEmailAddressOrUID, normalizedEmailAddressOrUID[:atLoc], normalizedEmailAddressOrUID[atLoc+1:])
  try:
    cd = buildGAPIObject(API.DIRECTORY)
    result = callGAPI(cd.users(), 'get',
                      throwReasons=GAPI.USER_GET_THROW_REASONS,
                      userKey=normalizedEmailAddressOrUID, fields='primaryEmail')
    if 'primaryEmail' in result:
      normalizedEmailAddressOrUID = result['primaryEmail'].lower()
      atLoc = normalizedEmailAddressOrUID.find('@')
      return (normalizedEmailAddressOrUID, normalizedEmailAddressOrUID[:atLoc], normalizedEmailAddressOrUID[atLoc+1:])
  except (GAPI.userNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden,
          GAPI.badRequest, GAPI.backendError, GAPI.systemError):
    pass
  return (normalizedEmailAddressOrUID, normalizedEmailAddressOrUID, GC.Values[GC.DOMAIN])

# Convert Org Unit Id to Org Unit Path
def convertOrgUnitIDtoPath(cd, orgUnitId):
  if orgUnitId.lower().startswith('orgunits/'):
    orgUnitId = f'id:{orgUnitId[9:]}'
  orgUnitPath = GM.Globals[GM.MAP_ORGUNIT_ID_TO_NAME].get(orgUnitId)
  if not orgUnitPath:
    if cd is None:
      cd = buildGAPIObject(API.DIRECTORY)
    try:
      orgUnitPath = callGAPI(cd.orgunits(), 'get',
                             throwReasons=GAPI.ORGUNIT_GET_THROW_REASONS,
                             customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath=orgUnitId, fields='orgUnitPath')['orgUnitPath']
    except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError, GAPI.badRequest, GAPI.invalidCustomerId, GAPI.loginRequired):
      orgUnitPath = orgUnitId
    GM.Globals[GM.MAP_ORGUNIT_ID_TO_NAME][orgUnitId] = orgUnitPath
  return orgUnitPath

def shlexSplitList(entity, dataDelimiter=' ,'):
  lexer = shlex.shlex(entity, posix=True)
  lexer.whitespace = dataDelimiter
  lexer.whitespace_split = True
  try:
    return list(lexer)
  except ValueError as e:
    Cmd.Backup()
    usageErrorExit(str(e))

def shlexSplitListStatus(entity, dataDelimiter=' ,'):
  lexer = shlex.shlex(entity, posix=True)
  lexer.whitespace = dataDelimiter
  lexer.whitespace_split = True
  try:
    return (True, list(lexer))
  except ValueError as e:
    return (False, str(e))

def getQueries(myarg):
  if myarg in {'query', 'filter'}:
    return [getString(Cmd.OB_QUERY)]
  return shlexSplitList(getString(Cmd.OB_QUERY_LIST))

def convertEntityToList(entity, shlexSplit=False, nonListEntityType=False):
  if not entity:
    return []
  if isinstance(entity, (list, set, dict)):
    return list(entity)
  if nonListEntityType:
    return [entity.strip()]
  if not shlexSplit:
    return entity.replace(',', ' ').split()
  return shlexSplitList(entity)

GROUP_ROLES_MAP = {
  'owner': Ent.ROLE_OWNER,
  'owners': Ent.ROLE_OWNER,
  'manager': Ent.ROLE_MANAGER,
  'managers': Ent.ROLE_MANAGER,
  'member': Ent.ROLE_MEMBER,
  'members': Ent.ROLE_MEMBER,
  }
ALL_GROUP_ROLES = {Ent.ROLE_MANAGER, Ent.ROLE_MEMBER, Ent.ROLE_OWNER}

def _getRoleVerification(memberRoles, fields):
  if memberRoles and memberRoles.find(Ent.ROLE_MEMBER) != -1:
    return (set(memberRoles.split(',')), None, fields if fields.find('role') != -1 else fields[:-1]+',role)')
  return (set(), memberRoles, fields)

def _getCIRoleVerification(memberRoles):
  if memberRoles:
    return set(memberRoles.split(','))
  return set()

def _checkMemberStatusIsSuspendedIsArchived(memberStatus, isSuspended, isArchived):
  if isSuspended is None and isArchived is None:
    return True
  if isSuspended is not None and isArchived is not None:
    if isSuspended == isArchived:
      if not isSuspended:
        return memberStatus not in {'SUSPENDED', 'ARCHIVED'}
      return memberStatus in {'SUSPENDED', 'ARCHIVED'}
    if isSuspended:
      return memberStatus == 'SUSPENDED'
    return memberStatus == 'ARCHIVED'
  if isSuspended is not None:
    if (not isSuspended and memberStatus != 'SUSPENDED') or (isSuspended and memberStatus == 'SUSPENDED'):
      return True
  if isArchived is not None:
    if (not isArchived and memberStatus != 'ARCHIVED') or (isArchived and memberStatus == 'ARCHIVED'):
      return True
  return False

def _checkMemberIsSuspendedIsArchived(member, isSuspended, isArchived):
  return _checkMemberStatusIsSuspendedIsArchived(member.get('status', 'UNKNOWN'), isSuspended, isArchived)

def _checkMemberRole(member, validRoles):
  return not validRoles or member.get('role', Ent.ROLE_MEMBER) in validRoles

def _checkMemberRoleIsSuspendedIsArchived(member, validRoles, isSuspended, isArchived):
  return _checkMemberRole(member, validRoles) and _checkMemberIsSuspendedIsArchived(member, isSuspended, isArchived)

def _checkMemberCategory(member, memberDisplayOptions):
  member_email = member.get('email', member.get('id', ''))
  if member_email.find('@') > 0:
    _, domain = member_email.lower().split('@', 1)
    category = 'internal' if domain in memberDisplayOptions['internalDomains'] else 'external'
  else:
    category = 'internal'
  if memberDisplayOptions[category]:
    member['category'] = category
    return True
  return False

def _checkCIMemberCategory(member, memberDisplayOptions):
  member_email = member.get('preferredMemberKey', {}).get('id', '')
  if member_email.find('@') > 0:
    _, domain = member_email.lower().split('@', 1)
    category = 'internal' if domain in memberDisplayOptions['internalDomains'] else 'external'
  else:
    category = 'internal'
  if memberDisplayOptions[category]:
    member['category'] = category
    return True
  return False

def getCIGroupMemberRoleFixType(member):
  ''' fixes missing type and returns the highest role of member '''
  if 'type' not in member:
    if member['preferredMemberKey']['id'] == GC.Values[GC.CUSTOMER_ID]:
      member['type'] = Ent.TYPE_CUSTOMER
    else:
      member['type'] = Ent.TYPE_OTHER
  roles = {}
  memberRoles = member.get('roles', [{'name': Ent.ROLE_MEMBER}])
  for role in memberRoles:
    roles[role['name']] = role
  for a_role in [Ent.ROLE_OWNER, Ent.ROLE_MANAGER, Ent.ROLE_MEMBER]:
    if a_role in roles:
      member['role'] = a_role
      if 'expiryDetail' in roles[a_role]:
        member['expireTime'] = roles[a_role]['expiryDetail']['expireTime']
      return
  member['role'] = memberRoles[0]['name']

def getCIGroupTransitiveMemberRoleFixType(groupName, tmember):
  ''' map transitive member to normal member '''
  tid = tmember['preferredMemberKey'][0].get('id', GC.Values[GC.CUSTOMER_ID]) if tmember['preferredMemberKey'] else ''
  if '/' in tmember['member']:
    ttype, tname = tmember['member'].split('/')
  else:
    ttype = ''
    tname = tmember['member']
  member = {'name': f'{groupName}/membershipd/{tname}', 'preferredMemberKey': {'id': tid}}
  if 'type' not in tmember:
    if tid == GC.Values[GC.CUSTOMER_ID]:
      member['type'] = Ent.TYPE_CUSTOMER
    elif ttype == 'users':
      member['type'] = Ent.TYPE_USER if not tid.endswith('.iam.gserviceaccount.com') else Ent.TYPE_SERVICE_ACCOUNT
    elif ttype == 'groups':
      member['type'] = Ent.TYPE_GROUP
    elif tid.startswith('cbcm-browser.'):
      member['type'] = Ent.TYPE_CBCM_BROWSER
    else:
      member['type'] = Ent.TYPE_OTHER
  else:
    member['type'] = tmember['type']
  if 'roles' in tmember:
    memberRoles = []
    for trole in tmember['roles']:
      if 'role' in trole:
        trole['name'] = trole.pop('role')
      if trole['name'] == 'ADMIN':
        trole['name'] = Ent.ROLE_MANAGER
      memberRoles.append(trole)
  else:
    memberRoles = [{'name': Ent.ROLE_MEMBER}]
  roles = {}
  for role in memberRoles:
    roles[role['name']] = role
  for a_role in [Ent.ROLE_OWNER, Ent.ROLE_MANAGER, Ent.ROLE_MEMBER]:
    if a_role in roles:
      member['role'] = a_role
      if 'expiryDetail' in roles[a_role]:
        member['expireTime'] = roles[a_role]['expiryDetail']['expireTime']
      break
  else:
    member['role'] = memberRoles[0]['name']
  return member

def convertGroupCloudIDToEmail(ci, group, i=0, count=0):
  if not group.startswith('groups/'):
    group = normalizeEmailAddressOrUID(group, ciGroupsAPI=True)
    if not group.startswith('groups/'):
      return (ci, None, group)
  if not ci:
    ci = buildGAPIObject(API.CLOUDIDENTITY_GROUPS)
  try:
    ciGroup = callGAPI(ci.groups(), 'get',
                       throwReasons=GAPI.CIGROUP_GET_THROW_REASONS, retryReasons=GAPI.CIGROUP_RETRY_REASONS,
                       name=group, fields='groupKey(id)')
    return (ci, None, ciGroup['groupKey']['id'])
  except (GAPI.notFound, GAPI.domainNotFound, GAPI.domainCannotUseApis,
          GAPI.forbidden, GAPI.badRequest, GAPI.invalid,
          GAPI.systemError, GAPI.permissionDenied, GAPI.serviceNotAvailable) as e:
    action = Act.Get()
    Act.Set(Act.LOOKUP)
    entityActionFailedWarning([Ent.CLOUD_IDENTITY_GROUP, group, Ent.GROUP, None], str(e), i, count)
    Act.Set(action)
    return (ci, None, None)

def convertGroupEmailToCloudID(ci, group, i=0, count=0):
  group = normalizeEmailAddressOrUID(group, ciGroupsAPI=True)
  if not group.startswith('groups/') and group.find('@') == -1:
    group = 'groups/'+group
  if group.startswith('groups/'):
    ci, _, groupEmail = convertGroupCloudIDToEmail(ci, group, i, count)
    return (ci, group, groupEmail)
  if not ci:
    ci = buildGAPIObject(API.CLOUDIDENTITY_GROUPS)
  try:
    ciGroup = callGAPI(ci.groups(), 'lookup',
                       throwReasons=GAPI.CIGROUP_GET_THROW_REASONS, retryReasons=GAPI.CIGROUP_RETRY_REASONS,
                       groupKey_id=group, fields='name')
    return (ci, ciGroup['name'], group)
  except (GAPI.notFound, GAPI.domainNotFound, GAPI.domainCannotUseApis,
          GAPI.forbidden, GAPI.badRequest, GAPI.invalid,
          GAPI.systemError, GAPI.failedPrecondition, GAPI.permissionDenied, GAPI.serviceNotAvailable) as e:
    action = Act.Get()
    Act.Set(Act.LOOKUP)
    entityActionFailedWarning([Ent.GROUP, group, Ent.CLOUD_IDENTITY_GROUP, None], str(e), i, count)
    Act.Set(action)
    return (ci, None, None)

CIGROUP_DISCUSSION_FORUM_LABEL = 'cloudidentity.googleapis.com/groups.discussion_forum'
CIGROUP_DYNAMIC_LABEL = 'cloudidentity.googleapis.com/groups.dynamic'
CIGROUP_SECURITY_LABEL = 'cloudidentity.googleapis.com/groups.security'
CIGROUP_LOCKED_LABEL = 'cloudidentity.googleapis.com/groups.locked'

def getCIGroupMembershipGraph(ci, member):
  if not ci:
    ci = buildGAPIObject(API.CLOUDIDENTITY_GROUPS)
  parent = 'groups/-'
  try:
    result = callGAPI(ci.groups().memberships(), 'getMembershipGraph',
                      throwReasons=GAPI.CIGROUP_LIST_THROW_REASONS, retryReasons=GAPI.CIGROUP_RETRY_REASONS,
                      parent=parent,
                      query=f"member_key_id == '{member}' && '{CIGROUP_DISCUSSION_FORUM_LABEL}' in labels")
    return (ci, result.get('response', {}))
  except (GAPI.resourceNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis,
          GAPI.forbidden, GAPI.badRequest, GAPI.invalid, GAPI.invalidArgument,
          GAPI.systemError, GAPI.permissionDenied, GAPI.serviceNotAvailable) as e:
    action = Act.Get()
    Act.Set(Act.LOOKUP)
    entityActionFailedWarning([Ent.CLOUD_IDENTITY_GROUP, parent], str(e))
    Act.Set(action)
    return (ci, None)

def checkGroupExists(cd, ci, ciGroupsAPI, group, i=0, count=0):
  group = normalizeEmailAddressOrUID(group, ciGroupsAPI=ciGroupsAPI)
  if not ciGroupsAPI:
    if not group.startswith('groups/'):
      try:
        result = callGAPI(cd.groups(), 'get',
                          throwReasons=GAPI.GROUP_GET_THROW_REASONS, retryReasons=GAPI.GROUP_GET_RETRY_REASONS,
                          groupKey=group, fields='email')
        return (ci, None, result['email'])
      except (GAPI.groupNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden, GAPI.badRequest, GAPI.invalid, GAPI.systemError):
        entityUnknownWarning(Ent.GROUP, group, i, count)
        return (ci, None, None)
    else:
      ci, _, groupEmail = convertGroupCloudIDToEmail(ci, group, i, count)
      return (ci, None, groupEmail)
  else:
    if not group.startswith('groups/') and group.find('@') == -1:
      group = 'groups/'+group
    if group.startswith('groups/'):
      try:
        result = callGAPI(ci.groups(), 'get',
                          throwReasons=GAPI.CIGROUP_GET_THROW_REASONS, retryReasons=GAPI.CIGROUP_RETRY_REASONS,
                          name=group, fields='name,groupKey(id)')
        return (ci, result['name'], result['groupKey']['id'])
      except (GAPI.notFound, GAPI.domainNotFound, GAPI.domainCannotUseApis,
              GAPI.forbidden, GAPI.badRequest, GAPI.invalid,
              GAPI.systemError, GAPI.permissionDenied, GAPI.serviceNotAvailable):
        entityUnknownWarning(Ent.CLOUD_IDENTITY_GROUP, group, i, count)
        return (ci, None, None)
    else:
      return convertGroupEmailToCloudID(ci, group, i, count)

# Turn the entity into a list of Users/CrOS devices
def getItemsToModify(entityType, entity, memberRoles=None, isSuspended=None, isArchived=None,
                     groupMemberType=Ent.TYPE_USER, noListConversion=False, recursive=False, noCLArgs=False):
  def _incrEntityDoesNotExist(entityType):
    entityError['entityType'] = entityType
    entityError[ENTITY_ERROR_DNE] += 1

  def _showInvalidEntity(entityType, entityName):
    entityError['entityType'] = entityType
    entityError[ENTITY_ERROR_INVALID] += 1
    printErrorMessage(INVALID_ENTITY_RC, formatKeyValueList('', [Ent.Singular(entityType), entityName, Msg.INVALID], ''))

  def _addGroupUsersToUsers(group, domains, recursive, includeDerivedMembership):
    printGettingAllEntityItemsForWhom(memberRoles if memberRoles else Ent.ROLE_MANAGER_MEMBER_OWNER, group, entityType=Ent.GROUP)
    validRoles, listRoles, listFields = _getRoleVerification(memberRoles, 'nextPageToken,members(email,type,status)')
    try:
      result = callGAPIpages(cd.members(), 'list', 'members',
                             pageMessage=getPageMessageForWhom(),
                             throwReasons=GAPI.MEMBERS_THROW_REASONS, retryReasons=GAPI.MEMBERS_RETRY_REASONS,
                             includeDerivedMembership=includeDerivedMembership,
                             groupKey=group, roles=listRoles, fields=listFields, maxResults=GC.Values[GC.MEMBER_MAX_RESULTS])
    except (GAPI.groupNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.invalid, GAPI.forbidden, GAPI.serviceNotAvailable):
      entityUnknownWarning(Ent.GROUP, group)
      _incrEntityDoesNotExist(Ent.GROUP)
      return
    for member in result:
      if member['type'] == Ent.TYPE_USER:
        email = member['email'].lower()
        if email in entitySet:
          continue
        if _checkMemberRoleIsSuspendedIsArchived(member, validRoles, isSuspended, isArchived):
          if domains:
            _, domain = splitEmailAddress(email)
            if domain not in domains:
              continue
          entitySet.add(email)
          entityList.append(email)
      elif recursive and member['type'] == Ent.TYPE_GROUP:
        _addGroupUsersToUsers(member['email'], domains, recursive, includeDerivedMembership)

  def _addCIGroupUsersToUsers(groupName, groupEmail, recursive):
    printGettingAllEntityItemsForWhom(memberRoles if memberRoles else Ent.ROLE_MANAGER_MEMBER_OWNER, groupEmail, entityType=Ent.CLOUD_IDENTITY_GROUP)
    validRoles = _getCIRoleVerification(memberRoles)
    try:
      result = callGAPIpages(ci.groups().memberships(), 'list', 'memberships',
                             pageMessage=getPageMessageForWhom(),
                             throwReasons=GAPI.CIGROUP_LIST_THROW_REASONS, retryReasons=GAPI.CIGROUP_RETRY_REASONS,
                             parent=groupName, view='FULL',
                             fields='nextPageToken,memberships(name,preferredMemberKey(id),roles(name),type)', pageSize=GC.Values[GC.MEMBER_MAX_RESULTS])
    except (GAPI.resourceNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis,
            GAPI.forbidden, GAPI.badRequest, GAPI.invalid, GAPI.invalidArgument,
            GAPI.systemError, GAPI.permissionDenied, GAPI.serviceNotAvailable):
      entityUnknownWarning(Ent.CLOUD_IDENTITY_GROUP, groupEmail)
      _incrEntityDoesNotExist(Ent.CLOUD_IDENTITY_GROUP)
      return
    for member in result:
      getCIGroupMemberRoleFixType(member)
      if member['type'] == Ent.TYPE_USER:
        email = member.get('preferredMemberKey', {}).get('id', '')
        if (email and _checkMemberRole(member, validRoles) and email not in entitySet):
          entitySet.add(email)
          entityList.append(email)
      elif recursive and member['type'] == Ent.TYPE_GROUP and _checkMemberRole(member, validRoles):
        _, gname = member['name'].rsplit('/', 1)
        _addCIGroupUsersToUsers(f'groups/{gname}', f'groups/{gname}', recursive)

  GM.Globals[GM.CLASSROOM_SERVICE_NOT_AVAILABLE] = False
  ENTITY_ERROR_DNE = 'doesNotExist'
  ENTITY_ERROR_INVALID = 'invalid'
  entityError = {'entityType': None, ENTITY_ERROR_DNE: 0, ENTITY_ERROR_INVALID: 0}
  entityList = []
  entitySet = set()
  entityLocation = Cmd.Location()
  if entityType in {Cmd.ENTITY_USER, Cmd.ENTITY_USERS}:
    if not GC.Values[GC.USER_SERVICE_ACCOUNT_ACCESS_ONLY] and not GC.Values[GC.DOMAIN]:
      buildGAPIObject(API.DIRECTORY)
    result = convertEntityToList(entity, nonListEntityType=entityType == Cmd.ENTITY_USER)
    for user in result:
      if validateEmailAddressOrUID(user):
        if user not in entitySet:
          entitySet.add(user)
          entityList.append(user)
      else:
        _showInvalidEntity(Ent.USER, user)
    if GC.Values[GC.USER_SERVICE_ACCOUNT_ACCESS_ONLY]:
      return entityList
  elif entityType in {Cmd.ENTITY_ALL_USERS, Cmd.ENTITY_ALL_USERS_NS, Cmd.ENTITY_ALL_USERS_NS_SUSP, Cmd.ENTITY_ALL_USERS_SUSP}:
    cd = buildGAPIObject(API.DIRECTORY)
    if entityType == Cmd.ENTITY_ALL_USERS and isSuspended is not None:
      query = f'isSuspended={isSuspended}'
    else:
      query = Cmd.ALL_USERS_QUERY_MAP[entityType]
    printGettingAllAccountEntities(Ent.USER)
    try:
      result = callGAPIpages(cd.users(), 'list', 'users',
                             pageMessage=getPageMessage(),
                             throwReasons=[GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN],
                             retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                             customer=GC.Values[GC.CUSTOMER_ID],
                             query=query, orderBy='email', fields='nextPageToken,users(primaryEmail,archived)',
                             maxResults=GC.Values[GC.USER_MAX_RESULTS])
    except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden):
      accessErrorExit(cd)
    entityList = [user['primaryEmail'] for user in result if isArchived is None or isArchived == user['archived']]
    printGotAccountEntities(len(entityList))
  elif entityType in {Cmd.ENTITY_DOMAINS, Cmd.ENTITY_DOMAINS_NS, Cmd.ENTITY_DOMAINS_SUSP}:
    if entityType == Cmd.ENTITY_DOMAINS_NS:
      query = 'isSuspended=False'
    elif entityType == Cmd.ENTITY_DOMAINS_SUSP:
      query = 'isSuspended=True'
    elif isSuspended is not None:
      query = f'isSuspended={isSuspended}'
    else:
      query = None
    cd = buildGAPIObject(API.DIRECTORY)
    domains = convertEntityToList(entity)
    for domain in domains:
      printGettingAllEntityItemsForWhom(Ent.USER, domain, entityType=Ent.DOMAIN)
      try:
        result = callGAPIpages(cd.users(), 'list', 'users',
                               pageMessage=getPageMessageForWhom(),
                               throwReasons=[GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.DOMAIN_NOT_FOUND, GAPI.FORBIDDEN],
                               retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                               domain=domain,
                               query=query, orderBy='email', fields='nextPageToken,users(primaryEmail,archived)',
                               maxResults=GC.Values[GC.USER_MAX_RESULTS])
      except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.domainNotFound, GAPI.forbidden):
        checkEntityDNEorAccessErrorExit(cd, Ent.DOMAIN, domain)
        _incrEntityDoesNotExist(Ent.DOMAIN)
        continue
      entityList = [user['primaryEmail'] for user in result if isArchived is None or isArchived == user['archived']]
      printGotAccountEntities(len(entityList))
  elif entityType in {Cmd.ENTITY_GROUP, Cmd.ENTITY_GROUPS,
                      Cmd.ENTITY_GROUP_NS, Cmd.ENTITY_GROUPS_NS,
                      Cmd.ENTITY_GROUP_SUSP, Cmd.ENTITY_GROUPS_SUSP,
                      Cmd.ENTITY_GROUP_INDE, Cmd.ENTITY_GROUPS_INDE}:
    if entityType in {Cmd.ENTITY_GROUP_NS, Cmd.ENTITY_GROUPS_NS}:
      isSuspended = False
    elif entityType in {Cmd.ENTITY_GROUP_SUSP, Cmd.ENTITY_GROUPS_SUSP}:
      isSuspended = True
    includeDerivedMembership = entityType in {Cmd.ENTITY_GROUP_INDE, Cmd.ENTITY_GROUPS_INDE}
    cd = buildGAPIObject(API.DIRECTORY)
    groups = convertEntityToList(entity, nonListEntityType=entityType in {Cmd.ENTITY_GROUP, Cmd.ENTITY_GROUP_NS, Cmd.ENTITY_GROUP_SUSP, Cmd.ENTITY_GROUP_INDE})
    for group in groups:
      if validateEmailAddressOrUID(group, checkPeople=False):
        group = normalizeEmailAddressOrUID(group)
        printGettingAllEntityItemsForWhom(memberRoles if memberRoles else Ent.ROLE_MANAGER_MEMBER_OWNER, group, entityType=Ent.GROUP)
        validRoles, listRoles, listFields = _getRoleVerification(memberRoles, 'nextPageToken,members(email,id,type,status)')
        try:
          result = callGAPIpages(cd.members(), 'list', 'members',
                                 pageMessage=getPageMessageForWhom(),
                                 throwReasons=GAPI.MEMBERS_THROW_REASONS, retryReasons=GAPI.MEMBERS_RETRY_REASONS,
                                 includeDerivedMembership=includeDerivedMembership,
                                 groupKey=group, roles=listRoles, fields=listFields, maxResults=GC.Values[GC.MEMBER_MAX_RESULTS])
        except (GAPI.groupNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.invalid, GAPI.forbidden, GAPI.serviceNotAvailable):
          entityUnknownWarning(Ent.GROUP, group)
          _incrEntityDoesNotExist(Ent.GROUP)
          continue
        for member in result:
          email = member['email'].lower() if member['type'] != Ent.TYPE_CUSTOMER else member['id']
          if ((groupMemberType in ('ALL', member['type'])) and
              (not includeDerivedMembership or (member['type'] == Ent.TYPE_USER)) and
              _checkMemberRoleIsSuspendedIsArchived(member, validRoles, isSuspended, isArchived) and
              email not in entitySet):
            entitySet.add(email)
            entityList.append(email)
      else:
        _showInvalidEntity(Ent.GROUP, group)
  elif entityType in {Cmd.ENTITY_GROUP_USERS, Cmd.ENTITY_GROUP_USERS_NS, Cmd.ENTITY_GROUP_USERS_SUSP, Cmd.ENTITY_GROUP_USERS_SELECT}:
    if entityType == Cmd.ENTITY_GROUP_USERS_NS:
      isSuspended = False
    elif entityType == Cmd.ENTITY_GROUP_USERS_SUSP:
      isSuspended = True
    cd = buildGAPIObject(API.DIRECTORY)
    groups = convertEntityToList(entity)
    includeDerivedMembership = False
    domains = []
    rolesSet = set()
    if not noCLArgs:
      while Cmd.ArgumentsRemaining():
        myarg = getArgument()
        if myarg in GROUP_ROLES_MAP:
          rolesSet.add(GROUP_ROLES_MAP[myarg])
        elif myarg == 'primarydomain':
          domains.append(GC.Values[GC.DOMAIN])
        elif myarg == 'domains':
          domains.extend(getEntityList(Cmd.OB_DOMAIN_NAME_ENTITY))
        elif myarg == 'recursive':
          recursive = True
          includeDerivedMembership = False
        elif myarg == 'includederivedmembership':
          includeDerivedMembership = True
          recursive = False
        elif entityType == Cmd.ENTITY_GROUP_USERS_SELECT and myarg in SUSPENDED_ARGUMENTS:
          isSuspended = _getIsSuspended(myarg)
        elif entityType == Cmd.ENTITY_GROUP_USERS_SELECT and myarg in ARCHIVED_ARGUMENTS:
          isArchived = _getIsArchived(myarg)
        elif myarg == 'end':
          break
        else:
          Cmd.Backup()
          missingArgumentExit('end')
    if rolesSet:
      memberRoles = ','.join(sorted(rolesSet))
    for group in groups:
      if validateEmailAddressOrUID(group, checkPeople=False):
        _addGroupUsersToUsers(normalizeEmailAddressOrUID(group), domains, recursive, includeDerivedMembership)
      else:
        _showInvalidEntity(Ent.GROUP, group)
  elif entityType in {Cmd.ENTITY_CIGROUP, Cmd.ENTITY_CIGROUPS}:
    ci = buildGAPIObject(API.CLOUDIDENTITY_GROUPS)
    groups = convertEntityToList(entity, nonListEntityType=entityType in {Cmd.ENTITY_CIGROUP})
    for group in groups:
      if validateEmailAddressOrUID(group, checkPeople=False, ciGroupsAPI=True):
        _, name, groupEmail = convertGroupEmailToCloudID(ci, group)
        printGettingAllEntityItemsForWhom(memberRoles if memberRoles else Ent.ROLE_MANAGER_MEMBER_OWNER, groupEmail, entityType=Ent.CLOUD_IDENTITY_GROUP)
        validRoles = _getCIRoleVerification(memberRoles)
        try:
          result = callGAPIpages(ci.groups().memberships(), 'list', 'memberships',
                                 pageMessage=getPageMessageForWhom(),
                                 throwReasons=GAPI.CIGROUP_LIST_THROW_REASONS, retryReasons=GAPI.CIGROUP_RETRY_REASONS,
                                 parent=name, view='FULL',
                                 fields='nextPageToken,memberships(preferredMemberKey(id),roles(name),type)',
                                 pageSize=GC.Values[GC.MEMBER_MAX_RESULTS])
        except (GAPI.resourceNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis,
                GAPI.forbidden, GAPI.badRequest, GAPI.invalid, GAPI.invalidArgument,
                GAPI.systemError, GAPI.permissionDenied, GAPI.serviceNotAvailable):
          entityUnknownWarning(Ent.CLOUD_IDENTITY_GROUP, groupEmail)
          _incrEntityDoesNotExist(Ent.CLOUD_IDENTITY_GROUP)
          continue
        for member in result:
          getCIGroupMemberRoleFixType(member)
          email = member.get('preferredMemberKey', {}).get('id', '')
          if (email and (groupMemberType in ('ALL', member['type'])) and
              _checkMemberRole(member, validRoles) and email not in entitySet):
            entitySet.add(email)
            entityList.append(email)
      else:
        _showInvalidEntity(Ent.CLOUD_IDENTITY_GROUP, groupEmail)
  elif entityType in {Cmd.ENTITY_CIGROUP_USERS}:
    ci = buildGAPIObject(API.CLOUDIDENTITY_GROUPS)
    groups = convertEntityToList(entity)
    rolesSet = set()
    if not noCLArgs:
      while Cmd.ArgumentsRemaining():
        myarg = getArgument()
        if myarg in GROUP_ROLES_MAP:
          rolesSet.add(GROUP_ROLES_MAP[myarg])
        elif myarg == 'recursive':
          recursive = True
        elif myarg == 'end':
          break
        else:
          Cmd.Backup()
          missingArgumentExit('end')
    if rolesSet:
      memberRoles = ','.join(sorted(rolesSet))
    for group in groups:
      _, name, groupEmail = convertGroupEmailToCloudID(ci, group)
      if name:
        _addCIGroupUsersToUsers(name, groupEmail, recursive)
      else:
        _showInvalidEntity(Ent.GROUP, group)
  elif entityType in {Cmd.ENTITY_OU, Cmd.ENTITY_OUS, Cmd.ENTITY_OU_AND_CHILDREN, Cmd.ENTITY_OUS_AND_CHILDREN,
                      Cmd.ENTITY_OU_NS, Cmd.ENTITY_OUS_NS, Cmd.ENTITY_OU_AND_CHILDREN_NS, Cmd.ENTITY_OUS_AND_CHILDREN_NS,
                      Cmd.ENTITY_OU_SUSP, Cmd.ENTITY_OUS_SUSP, Cmd.ENTITY_OU_AND_CHILDREN_SUSP, Cmd.ENTITY_OUS_AND_CHILDREN_SUSP}:
    if entityType in {Cmd.ENTITY_OU_NS, Cmd.ENTITY_OUS_NS, Cmd.ENTITY_OU_AND_CHILDREN_NS, Cmd.ENTITY_OUS_AND_CHILDREN_NS}:
      isSuspended = False
    elif entityType in {Cmd.ENTITY_OU_SUSP, Cmd.ENTITY_OUS_SUSP, Cmd.ENTITY_OU_AND_CHILDREN_SUSP, Cmd.ENTITY_OUS_AND_CHILDREN_SUSP}:
      isSuspended = True
    cd = buildGAPIObject(API.DIRECTORY)
    ous = convertEntityToList(entity, shlexSplit=True, nonListEntityType=entityType in {Cmd.ENTITY_OU, Cmd.ENTITY_OU_AND_CHILDREN,
                                                                                        Cmd.ENTITY_OU_NS, Cmd.ENTITY_OU_AND_CHILDREN_NS,
                                                                                        Cmd.ENTITY_OU_SUSP, Cmd.ENTITY_OU_AND_CHILDREN_SUSP})
    directlyInOU = entityType in {Cmd.ENTITY_OU, Cmd.ENTITY_OUS, Cmd.ENTITY_OU_NS, Cmd.ENTITY_OUS_NS, Cmd.ENTITY_OU_SUSP, Cmd.ENTITY_OUS_SUSP}
    qualifier = Msg.DIRECTLY_IN_THE.format(Ent.Singular(Ent.ORGANIZATIONAL_UNIT)) if directlyInOU else Msg.IN_THE.format(Ent.Singular(Ent.ORGANIZATIONAL_UNIT))
    fields = 'nextPageToken,users(primaryEmail,orgUnitPath,archived)' if directlyInOU else 'nextPageToken,users(primaryEmail,archived)'
    prevLen = 0
    for ou in ous:
      ou = makeOrgUnitPathAbsolute(ou)
      if ou.startswith('id:'):
        try:
          ou = callGAPI(cd.orgunits(), 'get',
                        throwReasons=GAPI.ORGUNIT_GET_THROW_REASONS,
                        customerId=GC.Values[GC.CUSTOMER_ID],
                        orgUnitPath=ou, fields='orgUnitPath')['orgUnitPath']
        except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError, GAPI.badRequest,
                GAPI.invalidCustomerId, GAPI.loginRequired):
          checkEntityDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, ou)
          _incrEntityDoesNotExist(Ent.ORGANIZATIONAL_UNIT)
          continue
      ouLower = ou.lower()
      printGettingAllEntityItemsForWhom(Ent.USER, ou, qualifier=Msg.IN_THE.format(Ent.Singular(Ent.ORGANIZATIONAL_UNIT)),
                                        entityType=Ent.ORGANIZATIONAL_UNIT)
      pageMessage = getPageMessageForWhom()
      usersInOU = 0
      try:
        feed = yieldGAPIpages(cd.users(), 'list', 'users',
                              pageMessage=pageMessage, messageAttribute='primaryEmail',
                              throwReasons=[GAPI.INVALID_ORGUNIT, GAPI.ORGUNIT_NOT_FOUND,
                                            GAPI.INVALID_INPUT, GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN],
                              retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                              customer=GC.Values[GC.CUSTOMER_ID], query=orgUnitPathQuery(ou, isSuspended), orderBy='email',
                              fields=fields, maxResults=GC.Values[GC.USER_MAX_RESULTS])
        for users in feed:
          if directlyInOU:
            for user in users:
              if ouLower == user.get('orgUnitPath', '').lower() and (isArchived is None or isArchived == user['archived']):
                usersInOU += 1
                entityList.append(user['primaryEmail'])
          elif isArchived is None:
            entityList.extend([user['primaryEmail'] for user in users])
            usersInOU += len(users)
          else:
            for user in users:
              if isArchived == user['archived']:
                usersInOU += 1
                entityList.append(user['primaryEmail'])
        setGettingAllEntityItemsForWhom(Ent.USER, ou, qualifier=qualifier)
        printGotEntityItemsForWhom(usersInOU)
      except (GAPI.invalidInput, GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError, GAPI.badRequest,
              GAPI.invalidCustomerId, GAPI.loginRequired, GAPI.resourceNotFound, GAPI.forbidden):
        checkEntityDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, ou)
        _incrEntityDoesNotExist(Ent.ORGANIZATIONAL_UNIT)
  elif entityType in {Cmd.ENTITY_QUERY, Cmd.ENTITY_QUERIES}:
    cd = buildGAPIObject(API.DIRECTORY)
    queries = convertEntityToList(entity, shlexSplit=True, nonListEntityType=entityType == Cmd.ENTITY_QUERY)
    prevLen = 0
    for query in queries:
      printGettingAllAccountEntities(Ent.USER, query)
      try:
        result = callGAPIpages(cd.users(), 'list', 'users',
                               pageMessage=getPageMessage(),
                               throwReasons=[GAPI.INVALID_ORGUNIT, GAPI.ORGUNIT_NOT_FOUND,
                                             GAPI.INVALID_INPUT, GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN],
                               retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                               customer=GC.Values[GC.CUSTOMER_ID], query=query, orderBy='email',
                               fields='nextPageToken,users(primaryEmail,suspended,archived)',
                               maxResults=GC.Values[GC.USER_MAX_RESULTS])
      except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.invalidInput):
        Cmd.Backup()
        usageErrorExit(Msg.INVALID_QUERY)
      except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden):
        accessErrorExit(cd)
      for user in result:
        email = user['primaryEmail']
        if ((isSuspended is None or isSuspended == user['suspended']) and
            (isArchived is None or isArchived == user['archived']) and
            email not in entitySet):
          entitySet.add(email)
          entityList.append(email)
      totalLen = len(entityList)
      printGotAccountEntities(totalLen-prevLen)
      prevLen = totalLen
  elif entityType == Cmd.ENTITY_LICENSES:
    skusList = []
    for item in entity.split(','):
      productId, sku = SKU.getProductAndSKU(item)
      if not productId:
        _incrEntityDoesNotExist(Ent.SKU)
      elif (productId, sku) not in skusList:
        skusList.append((productId, sku))
    if skusList:
      entityList = doPrintLicenses(returnFields=['userId'], skus=skusList)
  elif entityType in {Cmd.ENTITY_COURSEPARTICIPANTS, Cmd.ENTITY_TEACHERS, Cmd.ENTITY_STUDENTS}:
    croom = buildGAPIObject(API.CLASSROOM)
    if not noListConversion:
      courseIdList = convertEntityToList(entity)
    else:
      courseIdList = [entity]
    _, _, coursesInfo = _getCoursesOwnerInfo(croom, courseIdList, GC.Values[GC.USE_COURSE_OWNER_ACCESS])
    for courseId, courseInfo in coursesInfo.items():
      try:
        if entityType in {Cmd.ENTITY_COURSEPARTICIPANTS, Cmd.ENTITY_TEACHERS}:
          printGettingAllEntityItemsForWhom(Ent.TEACHER, removeCourseIdScope(courseId), entityType=Ent.COURSE)
          result = callGAPIpages(courseInfo['croom'].courses().teachers(), 'list', 'teachers',
                                 pageMessage=getPageMessageForWhom(),
                                 throwReasons=[GAPI.NOT_FOUND, GAPI.BAD_REQUEST, GAPI.SERVICE_NOT_AVAILABLE,
                                               GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                                 retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                                 courseId=courseId, fields='nextPageToken,teachers/profile/emailAddress',
                                 pageSize=GC.Values[GC.CLASSROOM_MAX_RESULTS])
          for teacher in result:
            email = teacher['profile'].get('emailAddress', None)
            if email and (email not in entitySet):
              entitySet.add(email)
              entityList.append(email)
        if entityType in {Cmd.ENTITY_COURSEPARTICIPANTS, Cmd.ENTITY_STUDENTS}:
          printGettingAllEntityItemsForWhom(Ent.STUDENT, removeCourseIdScope(courseId), entityType=Ent.COURSE)
          result = callGAPIpages(courseInfo['croom'].courses().students(), 'list', 'students',
                                 pageMessage=getPageMessageForWhom(),
                                 throwReasons=[GAPI.NOT_FOUND, GAPI.BAD_REQUEST, GAPI.SERVICE_NOT_AVAILABLE,
                                               GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                                 retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                                 courseId=courseId, fields='nextPageToken,students/profile/emailAddress',
                                 pageSize=GC.Values[GC.CLASSROOM_MAX_RESULTS])
          for student in result:
            email = student['profile'].get('emailAddress', None)
            if email and (email not in entitySet):
              entitySet.add(email)
              entityList.append(email)
      except GAPI.notFound:
        entityDoesNotExistWarning(Ent.COURSE, removeCourseIdScope(courseId))
        _incrEntityDoesNotExist(Ent.COURSE)
      except GAPI.serviceNotAvailable as e:
        entityActionNotPerformedWarning([Ent.COURSE, removeCourseIdScope(courseId)], str(e))
        GM.Globals[GM.CLASSROOM_SERVICE_NOT_AVAILABLE] = True
        break
      except (GAPI.forbidden, GAPI.permissionDenied, GAPI.badRequest) as e:
        ClientAPIAccessDeniedExit(str(e))
  elif entityType == Cmd.ENTITY_CROS:
    buildGAPIObject(API.DIRECTORY)
    result = convertEntityToList(entity)
    for deviceId in result:
      if deviceId not in entitySet:
        entitySet.add(deviceId)
        entityList.append(deviceId)
  elif entityType == Cmd.ENTITY_ALL_CROS:
    cd = buildGAPIObject(API.DIRECTORY)
    printGettingAllAccountEntities(Ent.CROS_DEVICE)
    try:
      result = callGAPIpages(cd.chromeosdevices(), 'list', 'chromeosdevices',
                             pageMessage=getPageMessage(),
                             throwReasons=[GAPI.INVALID_INPUT, GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN],
                             retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                             customerId=GC.Values[GC.CUSTOMER_ID],
                             fields='nextPageToken,chromeosdevices(deviceId)',
                             maxResults=GC.Values[GC.DEVICE_MAX_RESULTS])
    except (GAPI.invalidInput, GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden):
      accessErrorExit(cd)
    entityList = [device['deviceId'] for device in result]
  elif entityType in {Cmd.ENTITY_CROS_QUERY, Cmd.ENTITY_CROS_QUERIES, Cmd.ENTITY_CROS_SN}:
    cd = buildGAPIObject(API.DIRECTORY)
    queries = convertEntityToList(entity, shlexSplit=entityType == Cmd.ENTITY_CROS_QUERIES,
                                  nonListEntityType=entityType == Cmd.ENTITY_CROS_QUERY)
    if entityType == Cmd.ENTITY_CROS_SN:
      queries = [f'id:{query}' for query in queries]
    prevLen = 0
    for query in queries:
      printGettingAllAccountEntities(Ent.CROS_DEVICE, query)
      try:
        result = callGAPIpages(cd.chromeosdevices(), 'list', 'chromeosdevices',
                               pageMessage=getPageMessage(),
                               throwReasons=[GAPI.INVALID_INPUT, GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN],
                               retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                               customerId=GC.Values[GC.CUSTOMER_ID], query=query,
                               fields='nextPageToken,chromeosdevices(deviceId)',
                               maxResults=GC.Values[GC.DEVICE_MAX_RESULTS])
      except GAPI.invalidInput:
        Cmd.Backup()
        usageErrorExit(Msg.INVALID_QUERY)
      except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden):
        accessErrorExit(cd)
      for device in result:
        deviceId = device['deviceId']
        if deviceId not in entitySet:
          entitySet.add(deviceId)
          entityList.append(deviceId)
      totalLen = len(entityList)
      printGotAccountEntities(totalLen-prevLen)
      prevLen = totalLen
  elif entityType in {Cmd.ENTITY_CROS_OU, Cmd.ENTITY_CROS_OU_AND_CHILDREN, Cmd.ENTITY_CROS_OUS, Cmd.ENTITY_CROS_OUS_AND_CHILDREN,
                      Cmd.ENTITY_CROS_OU_QUERY, Cmd.ENTITY_CROS_OU_AND_CHILDREN_QUERY, Cmd.ENTITY_CROS_OUS_QUERY, Cmd.ENTITY_CROS_OUS_AND_CHILDREN_QUERY,
                      Cmd.ENTITY_CROS_OU_QUERIES, Cmd.ENTITY_CROS_OU_AND_CHILDREN_QUERIES, Cmd.ENTITY_CROS_OUS_QUERIES, Cmd.ENTITY_CROS_OUS_AND_CHILDREN_QUERIES}:
    cd = buildGAPIObject(API.DIRECTORY)
    ous = convertEntityToList(entity, shlexSplit=True,
                              nonListEntityType=entityType in {Cmd.ENTITY_CROS_OU, Cmd.ENTITY_CROS_OU_AND_CHILDREN,
                                                               Cmd.ENTITY_CROS_OU_QUERY, Cmd.ENTITY_CROS_OU_AND_CHILDREN_QUERY,
                                                               Cmd.ENTITY_CROS_OU_QUERIES, Cmd.ENTITY_CROS_OU_AND_CHILDREN_QUERIES})
    numOus = len(ous)
    includeChildOrgunits = entityType in {Cmd.ENTITY_CROS_OU_AND_CHILDREN, Cmd.ENTITY_CROS_OUS_AND_CHILDREN,
                                          Cmd.ENTITY_CROS_OU_AND_CHILDREN_QUERY, Cmd.ENTITY_CROS_OUS_AND_CHILDREN_QUERY,
                                          Cmd.ENTITY_CROS_OU_AND_CHILDREN_QUERIES, Cmd.ENTITY_CROS_OUS_AND_CHILDREN_QUERIES}
    allQualifier = Msg.DIRECTLY_IN_THE.format(Ent.Choose(Ent.ORGANIZATIONAL_UNIT, numOus)) if not includeChildOrgunits else Msg.IN_THE.format(Ent.Choose(Ent.ORGANIZATIONAL_UNIT, numOus))
    if entityType in {Cmd.ENTITY_CROS_OU_QUERY, Cmd.ENTITY_CROS_OU_AND_CHILDREN_QUERY, Cmd.ENTITY_CROS_OUS_QUERY, Cmd.ENTITY_CROS_OUS_AND_CHILDREN_QUERY}:
      queries = getQueries('query')
    elif entityType in {Cmd.ENTITY_CROS_OU_QUERIES, Cmd.ENTITY_CROS_OU_AND_CHILDREN_QUERIES, Cmd.ENTITY_CROS_OUS_QUERIES, Cmd.ENTITY_CROS_OUS_AND_CHILDREN_QUERIES}:
      queries = getQueries('queries')
    else:
      queries = [None]
    for ou in ous:
      ou = makeOrgUnitPathAbsolute(ou)
      oneQualifier = Msg.DIRECTLY_IN_THE.format(Ent.Singular(Ent.ORGANIZATIONAL_UNIT)) if not includeChildOrgunits else Msg.IN_THE.format(Ent.Singular(Ent.ORGANIZATIONAL_UNIT))
      for query in queries:
        printGettingAllEntityItemsForWhom(Ent.CROS_DEVICE, ou,
                                          query=query, qualifier=oneQualifier, entityType=Ent.ORGANIZATIONAL_UNIT)
        try:
          result = callGAPIpages(cd.chromeosdevices(), 'list', 'chromeosdevices',
                                 pageMessage=getPageMessageForWhom(),
                                 throwReasons=[GAPI.INVALID_INPUT, GAPI.INVALID_ORGUNIT, GAPI.ORGUNIT_NOT_FOUND,
                                               GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN],
                                 retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                                 customerId=GC.Values[GC.CUSTOMER_ID], query=query,
                                 orgUnitPath=ou, includeChildOrgunits=includeChildOrgunits,
                                 fields='nextPageToken,chromeosdevices(deviceId)', maxResults=GC.Values[GC.DEVICE_MAX_RESULTS])
        except GAPI.invalidInput:
          Cmd.Backup()
          usageErrorExit(Msg.INVALID_QUERY)
        except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden):
          checkEntityDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, ou)
          _incrEntityDoesNotExist(Ent.ORGANIZATIONAL_UNIT)
          continue
        if query is None:
          entityList.extend([device['deviceId'] for device in result])
        else:
          for device in result:
            deviceId = device['deviceId']
            if deviceId not in entitySet:
              entitySet.add(deviceId)
              entityList.append(deviceId)
    Ent.SetGettingQualifier(Ent.CROS_DEVICE, allQualifier)
    Ent.SetGettingForWhom(','.join(ous))
    printGotEntityItemsForWhom(len(entityList))
  elif entityType == Cmd.ENTITY_BROWSER:
    result = convertEntityToList(entity)
    for deviceId in result:
      if deviceId not in entitySet:
        entitySet.add(deviceId)
        entityList.append(deviceId)
  elif entityType in {Cmd.ENTITY_BROWSER_OU, Cmd.ENTITY_BROWSER_OUS}:
    cbcm = buildGAPIObject(API.CBCM)
    customerId = _getCustomerIdNoC()
    ous = convertEntityToList(entity, shlexSplit=True, nonListEntityType=entityType == Cmd.ENTITY_BROWSER_OU)
    numOus = len(ous)
    allQualifier = Msg.DIRECTLY_IN_THE.format(Ent.Choose(Ent.ORGANIZATIONAL_UNIT, numOus))
    oneQualifier = Msg.DIRECTLY_IN_THE.format(Ent.Singular(Ent.ORGANIZATIONAL_UNIT))
    for ou in ous:
      ou = makeOrgUnitPathAbsolute(ou)
      printGettingAllEntityItemsForWhom(Ent.CHROME_BROWSER, ou, qualifier=oneQualifier, entityType=Ent.ORGANIZATIONAL_UNIT)
      try:
        result = callGAPIpages(cbcm.chromebrowsers(), 'list', 'browsers',
                               pageMessage=getPageMessageForWhom(),
                               throwReasons=[GAPI.BAD_REQUEST, GAPI.INVALID_ORGUNIT, GAPI.FORBIDDEN],
                               retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                               customer=customerId, orgUnitPath=ou, projection='BASIC',
                               orderBy='id', sortOrder='ASCENDING', fields='nextPageToken,browsers(deviceId)')
      except (GAPI.badRequest, GAPI.invalidOrgunit, GAPI.forbidden):
        checkEntityDNEorAccessErrorExit(None, Ent.ORGANIZATIONAL_UNIT, ou)
        _incrEntityDoesNotExist(Ent.ORGANIZATIONAL_UNIT)
        continue
      entityList.extend([browser['deviceId'] for browser in result])
    Ent.SetGettingQualifier(Ent.CHROME_BROWSER, allQualifier)
    Ent.SetGettingForWhom(','.join(ous))
    printGotEntityItemsForWhom(len(entityList))
  elif entityType in {Cmd.ENTITY_BROWSER_QUERY, Cmd.ENTITY_BROWSER_QUERIES}:
    cbcm = buildGAPIObject(API.CBCM)
    customerId = _getCustomerIdNoC()
    queries = convertEntityToList(entity, shlexSplit=entityType == Cmd.ENTITY_BROWSER_QUERIES,
                                  nonListEntityType=entityType == Cmd.ENTITY_BROWSER_QUERY)
    prevLen = 0
    for query in queries:
      printGettingAllAccountEntities(Ent.CHROME_BROWSER, query)
      try:
        result = callGAPIpages(cbcm.chromebrowsers(), 'list', 'browsers',
                               pageMessage=getPageMessage(),
                               throwReasons=[GAPI.INVALID_INPUT, GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN],
                               retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                               customer=customerId, query=query, projection='BASIC',
                               orderBy='id', sortOrder='ASCENDING', fields='nextPageToken,browsers(deviceId)')
      except GAPI.invalidInput:
        Cmd.Backup()
        usageErrorExit(Msg.INVALID_QUERY)
      except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden) as e:
        accessErrorExitNonDirectory(API.CBCM, str(e))
      for device in result:
        deviceId = device['deviceId']
        if deviceId not in entitySet:
          entitySet.add(deviceId)
          entityList.append(deviceId)
      totalLen = len(entityList)
      printGotAccountEntities(totalLen-prevLen)
      prevLen = totalLen
  else:
    systemErrorExit(UNKNOWN_ERROR_RC, 'getItemsToModify coding error')
  for errorType in [ENTITY_ERROR_DNE, ENTITY_ERROR_INVALID]:
    if entityError[errorType] > 0:
      Cmd.SetLocation(entityLocation-1)
      writeStderr(Cmd.CommandLineWithBadArgumentMarked(False))
      count = entityError[errorType]
      if errorType == ENTITY_ERROR_DNE:
        stderrErrorMsg(Msg.BAD_ENTITIES_IN_SOURCE.format(count, Ent.Choose(entityError['entityType'], count),
                                                         Msg.DO_NOT_EXIST if count != 1 else Msg.DOES_NOT_EXIST))
        sys.exit(ENTITY_DOES_NOT_EXIST_RC)
      else:
        stderrErrorMsg(Msg.BAD_ENTITIES_IN_SOURCE.format(count, Msg.INVALID, Ent.Choose(entityError['entityType'], count)))
        sys.exit(INVALID_ENTITY_RC)
  return entityList

def splitEntityList(entity, dataDelimiter):
  if not entity:
    return []
  if not dataDelimiter:
    return [entity]
  return entity.split(dataDelimiter)

def splitEntityListShlex(entity, dataDelimiter):
  if not entity:
    return (True, [])
  if not dataDelimiter:
    return (True, [entity])
  return shlexSplitListStatus(entity, dataDelimiter)

def fileDataErrorExit(filename, row, itemName, value, errMessage):
  if itemName:
    systemErrorExit(DATA_ERROR_RC,
                    formatKeyValueList('',
                                       [Ent.Singular(Ent.FILE), filename,
                                        Ent.Singular(Ent.ROW), row,
                                        Ent.Singular(Ent.ITEM), itemName,
                                        Ent.Singular(Ent.VALUE), value,
                                        errMessage],
                                       ''))
  else:
    systemErrorExit(DATA_ERROR_RC,
                    formatKeyValueList('',
                                       [Ent.Singular(Ent.FILE), filename,
                                        Ent.Singular(Ent.ROW), row,
                                        Ent.Singular(Ent.VALUE), value,
                                        errMessage],
                                       ''))

# <FileSelector>
def getEntitiesFromFile(shlexSplit, returnSet=False):
  filename = getString(Cmd.OB_FILE_NAME)
  filenameLower = filename.lower()
  if filenameLower not in {'gcsv', 'gdoc', 'gcscsv', 'gcsdoc'}:
    encoding = getCharSet()
    f = openFile(filename, encoding=encoding, stripUTFBOM=True)
  elif filenameLower in {'gcsv', 'gdoc'}:
    f = getGDocData(filenameLower)
    getCharSet()
  else: #filenameLower in {'gcscsv', 'gcsdoc'}:
    f = getStorageFileData(filenameLower)
    getCharSet()
  dataDelimiter = getDelimiter()
  entitySet = set()
  entityList = []
  i = 0
  for row in f:
    i += 1
    if shlexSplit:
      splitStatus, itemList = splitEntityListShlex(row.strip(), dataDelimiter)
      if not splitStatus:
        fileDataErrorExit(filename, i, None, row.strip(), f'{Msg.INVALID_LIST}: {itemList}')
    else:
      itemList = splitEntityList(row.strip(), dataDelimiter)
    for item in itemList:
      item = item.strip()
      if item and (item not in entitySet):
        entitySet.add(item)
        entityList.append(item)
  closeFile(f)
  return entityList if not returnSet else entitySet

# <CSVFileSelector>
def getEntitiesFromCSVFile(shlexSplit, returnSet=False):
  fileFieldName = getString(Cmd.OB_FILE_NAME_FIELD_NAME)
  if platform.system() == 'Windows' and not fileFieldName.startswith('-:'):
    drive, fileFieldName = os.path.splitdrive(fileFieldName)
  else:
    drive = ''
  if fileFieldName.find(':') == -1:
    Cmd.Backup()
    invalidArgumentExit(Cmd.OB_FILE_NAME_FIELD_NAME)
  fileFieldNameList = fileFieldName.split(':')
  filename = drive+fileFieldNameList[0]
  f, csvFile, fieldnames = openCSVFileReader(filename)
  for fieldName in fileFieldNameList[1:]:
    if fieldName not in fieldnames:
      csvFieldErrorExit(fieldName, fieldnames, backupArg=True, checkForCharset=True)
  matchFields, skipFields = getMatchSkipFields(fieldnames)
  dataDelimiter = getDelimiter()
  entitySet = set()
  entityList = []
  i = 1
  for row in csvFile:
    i += 1
    if checkMatchSkipFields(row, None, matchFields, skipFields):
      for fieldName in fileFieldNameList[1:]:
        if shlexSplit:
          splitStatus, itemList = splitEntityListShlex(row[fieldName].strip(), dataDelimiter)
          if not splitStatus:
            fileDataErrorExit(filename, i, fieldName, row[fieldName].strip(), f'{Msg.INVALID_LIST}: {itemList}')
        else:
          itemList = splitEntityList(row[fieldName].strip(), dataDelimiter)
        for item in itemList:
          item = item.strip()
          if item and (item not in entitySet):
            entitySet.add(item)
            entityList.append(item)
  closeFile(f)
  return entityList if not returnSet else entitySet

# <CSVFileSelector>
#	keyfield <FieldName> [keypattern <RESearchPattern>] [keyvalue <RESubstitution>] [delimiter <Character>]
#	subkeyfield <FieldName> [keypattern <RESearchPattern>] [keyvalue <RESubstitution>] [delimiter <Character>]
#	(matchfield|skipfield <FieldName> <RESearchPattern>)*
#	[datafield <FieldName>(:<FieldName>)* [delimiter <Character>]]
def getEntitiesFromCSVbyField():

  def getKeyFieldInfo(keyword, required, globalKeyField):
    if not checkArgumentPresent(keyword, required=required):
      GM.Globals[globalKeyField] = None
      return (None, None, None, None)
    keyField = GM.Globals[globalKeyField] = getString(Cmd.OB_FIELD_NAME)
    if keyField not in fieldnames:
      csvFieldErrorExit(keyField, fieldnames, backupArg=True)
    if checkArgumentPresent('keypattern'):
      keyPattern = getREPattern()
    else:
      keyPattern = None
    if checkArgumentPresent('keyvalue'):
      keyValue = getString(Cmd.OB_STRING)
    else:
      keyValue = keyField
    keyDelimiter = getDelimiter()
    return (keyField, keyPattern, keyValue, keyDelimiter)

  def getKeyList(row, keyField, keyPattern, keyValue, keyDelimiter, matchFields, skipFields):
    item = row[keyField].strip()
    if not item:
      return []
    if not checkMatchSkipFields(row, None, matchFields, skipFields):
      return []
    if keyPattern:
      keyList = [keyPattern.sub(keyValue, keyItem.strip()) for keyItem in splitEntityList(item, keyDelimiter)]
    else:
      keyList = [re.sub(keyField, keyItem.strip(), keyValue) for keyItem in splitEntityList(item, keyDelimiter)]
    return [key for key in keyList if key]

  filename = getString(Cmd.OB_FILE_NAME)
  f, csvFile, fieldnames = openCSVFileReader(filename)
  mainKeyField, mainKeyPattern, mainKeyValue, mainKeyDelimiter = getKeyFieldInfo('keyfield', True, GM.CSV_KEY_FIELD)
  subKeyField, subKeyPattern, subKeyValue, subKeyDelimiter = getKeyFieldInfo('subkeyfield', False, GM.CSV_SUBKEY_FIELD)
  matchFields, skipFields = getMatchSkipFields(fieldnames)
  if checkArgumentPresent('datafield'):
    if GM.Globals[GM.CSV_DATA_DICT]:
      csvDataAlreadySavedErrorExit()
    GM.Globals[GM.CSV_DATA_FIELD] = getString(Cmd.OB_FIELD_NAME, checkBlank=True)
    dataFields = GM.Globals[GM.CSV_DATA_FIELD].split(':')
    for dataField in dataFields:
      if dataField not in fieldnames:
        csvFieldErrorExit(dataField, fieldnames, backupArg=True)
    dataDelimiter = getDelimiter()
  else:
    GM.Globals[GM.CSV_DATA_FIELD] = None
    dataFields = []
    dataDelimiter = None
  entitySet = set()
  entityList = []
  csvDataKeys = {}
  GM.Globals[GM.CSV_DATA_DICT] = {}
  if not subKeyField:
    for row in csvFile:
      mainKeyList = getKeyList(row, mainKeyField, mainKeyPattern, mainKeyValue, mainKeyDelimiter, matchFields, skipFields)
      if not mainKeyList:
        continue
      for mainKey in mainKeyList:
        if mainKey not in entitySet:
          entitySet.add(mainKey)
          entityList.append(mainKey)
          if GM.Globals[GM.CSV_DATA_FIELD]:
            csvDataKeys[mainKey] = set()
            GM.Globals[GM.CSV_DATA_DICT][mainKey] = []
      for dataField in dataFields:
        if dataField in row:
          dataList = splitEntityList(row[dataField].strip(), dataDelimiter)
          for dataValue in dataList:
            dataValue = dataValue.strip()
            if not dataValue:
              continue
            for mainKey in mainKeyList:
              if dataValue not in csvDataKeys[mainKey]:
                csvDataKeys[mainKey].add(dataValue)
                GM.Globals[GM.CSV_DATA_DICT][mainKey].append(dataValue)
  else:
    csvSubKeys = {}
    for row in csvFile:
      mainKeyList = getKeyList(row, mainKeyField, mainKeyPattern, mainKeyValue, mainKeyDelimiter, matchFields, skipFields)
      if not mainKeyList:
        continue
      for mainKey in mainKeyList:
        if mainKey not in entitySet:
          entitySet.add(mainKey)
          entityList.append(mainKey)
          csvSubKeys[mainKey] = set()
          csvDataKeys[mainKey] = {}
          GM.Globals[GM.CSV_DATA_DICT][mainKey] = {}
      subKeyList = getKeyList(row, subKeyField, subKeyPattern, subKeyValue, subKeyDelimiter, {}, {})
      if not subKeyList:
        continue
      for mainKey in mainKeyList:
        for subKey in subKeyList:
          if subKey not in csvSubKeys[mainKey]:
            csvSubKeys[mainKey].add(subKey)
            if GM.Globals[GM.CSV_DATA_FIELD]:
              csvDataKeys[mainKey][subKey] = set()
              GM.Globals[GM.CSV_DATA_DICT][mainKey][subKey] = []
      for dataField in dataFields:
        if dataField in row:
          dataList = splitEntityList(row[dataField].strip(), dataDelimiter)
          for dataValue in dataList:
            dataValue = dataValue.strip()
            if not dataValue:
              continue
            for mainKey in mainKeyList:
              for subKey in subKeyList:
                if dataValue not in csvDataKeys[mainKey][subKey]:
                  csvDataKeys[mainKey][subKey].add(dataValue)
                  GM.Globals[GM.CSV_DATA_DICT][mainKey][subKey].append(dataValue)
  closeFile(f)
  return entityList

# Typically used to map courseparticipants to students or teachers
def mapEntityType(entityType, typeMap):
  if (typeMap is not None) and (entityType in typeMap):
    return typeMap[entityType]
  return entityType

def getEntityArgument(entityList):
  if entityList is None:
    return (0, 0, entityList)
  if isinstance(entityList, dict):
    clLoc = Cmd.Location()
    Cmd.SetLocation(GM.Globals[GM.ENTITY_CL_DELAY_START])
    entityList = getItemsToModify(**entityList)
    Cmd.SetLocation(clLoc)
  return (0, len(entityList), entityList)

def getEntityToModify(defaultEntityType=None, browserAllowed=False, crosAllowed=False, userAllowed=True,
                      typeMap=None, isSuspended=None, isArchived=None, groupMemberType=Ent.TYPE_USER, delayGet=False):
  if GC.Values[GC.USER_SERVICE_ACCOUNT_ACCESS_ONLY]:
    crosAllowed = False
    selectorChoices = Cmd.SERVICE_ACCOUNT_ONLY_ENTITY_SELECTORS[:]
  else:
    selectorChoices = Cmd.BASE_ENTITY_SELECTORS[:]
  if userAllowed:
    selectorChoices += Cmd.USER_ENTITY_SELECTORS+Cmd.USER_CSVDATA_ENTITY_SELECTORS
  if crosAllowed:
    selectorChoices += Cmd.CROS_ENTITY_SELECTORS+Cmd.CROS_CSVDATA_ENTITY_SELECTORS
  if browserAllowed:
    selectorChoices = Cmd.BROWSER_ENTITY_SELECTORS
  entitySelector = getChoice(selectorChoices, defaultChoice=None)
  if entitySelector:
    choices = []
    if entitySelector == Cmd.ENTITY_SELECTOR_ALL:
      if userAllowed:
        choices += Cmd.USER_ENTITY_SELECTOR_ALL_SUBTYPES
      if crosAllowed:
        choices += Cmd.CROS_ENTITY_SELECTOR_ALL_SUBTYPES
      entityType = Cmd.ENTITY_SELECTOR_ALL_SUBTYPES_MAP[getChoice(choices)]
      if not delayGet:
        return (Cmd.ENTITY_USERS if entityType != Cmd.ENTITY_ALL_CROS else Cmd.ENTITY_CROS,
                getItemsToModify(entityType, None))
      GM.Globals[GM.ENTITY_CL_DELAY_START] = Cmd.Location()
      buildGAPIObject(API.DIRECTORY)
      return (Cmd.ENTITY_USERS if entityType != Cmd.ENTITY_ALL_CROS else Cmd.ENTITY_CROS,
              {'entityType': entityType, 'entity': None})
    if userAllowed:
      if entitySelector == Cmd.ENTITY_SELECTOR_FILE:
        return (Cmd.ENTITY_USERS, getItemsToModify(Cmd.ENTITY_USERS, getEntitiesFromFile(False)))
      if entitySelector in [Cmd.ENTITY_SELECTOR_CSV, Cmd.ENTITY_SELECTOR_CSVFILE]:
        return (Cmd.ENTITY_USERS, getItemsToModify(Cmd.ENTITY_USERS, getEntitiesFromCSVFile(False)))
    if crosAllowed:
      if entitySelector == Cmd.ENTITY_SELECTOR_CROSFILE:
        return (Cmd.ENTITY_CROS, getEntitiesFromFile(False))
      if entitySelector in [Cmd.ENTITY_SELECTOR_CROSCSV, Cmd.ENTITY_SELECTOR_CROSCSVFILE]:
        return (Cmd.ENTITY_CROS, getEntitiesFromCSVFile(False))
      if entitySelector == Cmd.ENTITY_SELECTOR_CROSFILE_SN:
        return (Cmd.ENTITY_CROS, getItemsToModify(Cmd.ENTITY_CROS_SN, getEntitiesFromFile(False)))
      if entitySelector in [Cmd.ENTITY_SELECTOR_CROSCSV_SN, Cmd.ENTITY_SELECTOR_CROSCSVFILE_SN]:
        return (Cmd.ENTITY_CROS, getItemsToModify(Cmd.ENTITY_CROS_SN, getEntitiesFromCSVFile(False)))
    if browserAllowed:
      if entitySelector == Cmd.ENTITY_SELECTOR_FILE:
        return (Cmd.ENTITY_BROWSER, getEntitiesFromFile(False))
      if entitySelector in [Cmd.ENTITY_SELECTOR_CSV, Cmd.ENTITY_SELECTOR_CSVFILE]:
        return (Cmd.ENTITY_BROWSER, getEntitiesFromCSVFile(False))
    if entitySelector == Cmd.ENTITY_SELECTOR_DATAFILE:
      if userAllowed:
        choices += Cmd.USER_ENTITY_SELECTOR_DATAFILE_CSVKMD_SUBTYPES if not GC.Values[GC.USER_SERVICE_ACCOUNT_ACCESS_ONLY] else [Cmd.ENTITY_USERS]
      if crosAllowed:
        choices += Cmd.CROS_ENTITY_SELECTOR_DATAFILE_CSVKMD_SUBTYPES
      entityType = mapEntityType(getChoice(choices), typeMap)
      return (Cmd.ENTITY_USERS if entityType not in Cmd.CROS_ENTITY_SELECTOR_DATAFILE_CSVKMD_SUBTYPES else Cmd.ENTITY_CROS,
              getItemsToModify(entityType, getEntitiesFromFile(shlexSplit=entityType in [Cmd.ENTITY_OUS, Cmd.ENTITY_OUS_AND_CHILDREN,
                                                                                         Cmd.ENTITY_OUS_NS, Cmd.ENTITY_OUS_AND_CHILDREN_NS,
                                                                                         Cmd.ENTITY_OUS_SUSP, Cmd.ENTITY_OUS_AND_CHILDREN_SUSP,
                                                                                         Cmd.ENTITY_CROS_OUS, Cmd.ENTITY_CROS_OUS_AND_CHILDREN])))
    if entitySelector == Cmd.ENTITY_SELECTOR_CSVDATAFILE:
      if userAllowed:
        choices += Cmd.USER_ENTITY_SELECTOR_DATAFILE_CSVKMD_SUBTYPES if not GC.Values[GC.USER_SERVICE_ACCOUNT_ACCESS_ONLY] else [Cmd.ENTITY_USERS]
      if crosAllowed:
        choices += Cmd.CROS_ENTITY_SELECTOR_DATAFILE_CSVKMD_SUBTYPES
      entityType = mapEntityType(getChoice(choices), typeMap)
      return (Cmd.ENTITY_USERS if entityType not in Cmd.CROS_ENTITY_SELECTOR_DATAFILE_CSVKMD_SUBTYPES else Cmd.ENTITY_CROS,
              getItemsToModify(entityType, getEntitiesFromCSVFile(shlexSplit=entityType in [Cmd.ENTITY_OUS, Cmd.ENTITY_OUS_AND_CHILDREN,
                                                                                            Cmd.ENTITY_OUS_NS, Cmd.ENTITY_OUS_AND_CHILDREN_NS,
                                                                                            Cmd.ENTITY_OUS_SUSP, Cmd.ENTITY_OUS_AND_CHILDREN_SUSP,
                                                                                            Cmd.ENTITY_CROS_OUS, Cmd.ENTITY_CROS_OUS_AND_CHILDREN])))
    if entitySelector == Cmd.ENTITY_SELECTOR_CSVKMD:
      if userAllowed:
        choices += Cmd.USER_ENTITY_SELECTOR_DATAFILE_CSVKMD_SUBTYPES if not GC.Values[GC.USER_SERVICE_ACCOUNT_ACCESS_ONLY] else [Cmd.ENTITY_USERS]
      if crosAllowed:
        choices += Cmd.CROS_ENTITY_SELECTOR_DATAFILE_CSVKMD_SUBTYPES
      entityType = mapEntityType(getChoice(choices, choiceAliases=Cmd.ENTITY_ALIAS_MAP), typeMap)
      return (Cmd.ENTITY_USERS if entityType not in Cmd.CROS_ENTITY_SELECTOR_DATAFILE_CSVKMD_SUBTYPES else Cmd.ENTITY_CROS,
              getItemsToModify(entityType, getEntitiesFromCSVbyField()))
    if entitySelector in [Cmd.ENTITY_SELECTOR_CSVDATA, Cmd.ENTITY_SELECTOR_CROSCSVDATA]:
      checkDataField()
      return (Cmd.ENTITY_USERS if entitySelector == Cmd.ENTITY_SELECTOR_CSVDATA else Cmd.ENTITY_CROS,
              GM.Globals[GM.CSV_DATA_DICT])
  entityChoices = []
  if userAllowed:
    entityChoices += Cmd.USER_ENTITIES if not GC.Values[GC.USER_SERVICE_ACCOUNT_ACCESS_ONLY] else [Cmd.ENTITY_USER, Cmd.ENTITY_USERS]
  if crosAllowed:
    entityChoices += Cmd.CROS_ENTITIES
  if browserAllowed:
    entityChoices += Cmd.BROWSER_ENTITIES
  entityType = mapEntityType(getChoice(entityChoices, choiceAliases=Cmd.ENTITY_ALIAS_MAP, defaultChoice=defaultEntityType), typeMap)
  if not entityType:
    invalidChoiceExit(Cmd.Current(), selectorChoices+entityChoices, False)
  if entityType not in Cmd.CROS_ENTITIES+Cmd.BROWSER_ENTITIES:
    entityClass = Cmd.ENTITY_USERS
    if entityType == Cmd.ENTITY_OAUTHUSER:
      return (entityClass, [_getAdminEmail()])
    entityItem = getString(Cmd.OB_USER_ENTITY, minLen=0)
  elif entityType in Cmd.CROS_ENTITIES:
    entityClass = Cmd.ENTITY_CROS
    entityItem = getString(Cmd.OB_CROS_ENTITY, minLen=0)
  else:
    entityClass = Cmd.ENTITY_BROWSER
    entityItem = getString(Cmd.OB_BROWSER_ENTITY, minLen=0)
  if not delayGet:
    if entityClass == Cmd.ENTITY_USERS:
      return (entityClass, getItemsToModify(entityType, entityItem,
                                            isSuspended=isSuspended, isArchived=isArchived, groupMemberType=groupMemberType))
    return (entityClass, getItemsToModify(entityType, entityItem))
  GM.Globals[GM.ENTITY_CL_DELAY_START] = Cmd.Location()
  if not GC.Values[GC.USER_SERVICE_ACCOUNT_ACCESS_ONLY]:
    buildGAPIObject(API.DIRECTORY)
  if entityClass == Cmd.ENTITY_USERS:
    if entityType in [Cmd.ENTITY_GROUP_USERS,
                      Cmd.ENTITY_GROUP_USERS_NS, Cmd.ENTITY_GROUP_USERS_SUSP,
                      Cmd.ENTITY_GROUP_USERS_SELECT,
                      Cmd.ENTITY_CIGROUP_USERS]:
      # Skip over sub-arguments
      while Cmd.ArgumentsRemaining():
        myarg = getArgument()
        if myarg in GROUP_ROLES_MAP or myarg in {'primarydomain', 'recursive', 'includederivedmembership'}:
          pass
        elif myarg == 'domains':
          Cmd.Advance()
        elif ((entityType == Cmd.ENTITY_GROUP_USERS_SELECT) and
              (myarg in SUSPENDED_ARGUMENTS) or (myarg in ARCHIVED_ARGUMENTS)):
          if myarg in {'issuspended', 'isarchived'}:
            if Cmd.PeekArgumentPresent(TRUE_VALUES) or Cmd.PeekArgumentPresent(FALSE_VALUES):
              Cmd.Advance()
        elif myarg == 'end':
          break
        else:
          Cmd.Backup()
          missingArgumentExit('end')
    return (entityClass,
            {'entityType': entityType, 'entity': entityItem, 'isSuspended': isSuspended, 'isArchived': isArchived,
             'groupMemberType': groupMemberType})
  if entityClass == Cmd.ENTITY_CROS:
    if entityType in {Cmd.ENTITY_CROS_OU_QUERY, Cmd.ENTITY_CROS_OU_AND_CHILDREN_QUERY, Cmd.ENTITY_CROS_OUS_QUERY, Cmd.ENTITY_CROS_OUS_AND_CHILDREN_QUERY,
                      Cmd.ENTITY_CROS_OU_QUERIES, Cmd.ENTITY_CROS_OU_AND_CHILDREN_QUERIES, Cmd.ENTITY_CROS_OUS_QUERIES, Cmd.ENTITY_CROS_OUS_AND_CHILDREN_QUERIES}:
      Cmd.Advance()
  return (entityClass,
          {'entityType': entityType, 'entity': entityItem})

def getEntitySelector():
  return getChoice(Cmd.ENTITY_LIST_SELECTORS, defaultChoice=None)

def getEntitySelection(entitySelector, shlexSplit):
  if entitySelector in [Cmd.ENTITY_SELECTOR_FILE]:
    return getEntitiesFromFile(shlexSplit)
  if entitySelector in [Cmd.ENTITY_SELECTOR_CSV, Cmd.ENTITY_SELECTOR_CSVFILE]:
    return getEntitiesFromCSVFile(shlexSplit)
  if entitySelector == Cmd.ENTITY_SELECTOR_CSVKMD:
    return getEntitiesFromCSVbyField()
  if entitySelector in [Cmd.ENTITY_SELECTOR_CSVSUBKEY]:
    checkSubkeyField()
    return GM.Globals[GM.CSV_DATA_DICT]
  if entitySelector in [Cmd.ENTITY_SELECTOR_CSVDATA]:
    checkDataField()
    return GM.Globals[GM.CSV_DATA_DICT]
  return []

def getEntityList(item, shlexSplit=False):
  entitySelector = getEntitySelector()
  if entitySelector:
    return getEntitySelection(entitySelector, shlexSplit)
  return convertEntityToList(getString(item, minLen=0), shlexSplit=shlexSplit)

def getNormalizedEmailAddressEntity(shlexSplit=False, noUid=True, noLower=False):
  return [normalizeEmailAddressOrUID(emailAddress, noUid=noUid, noLower=noLower) for emailAddress in getEntityList(Cmd.OB_EMAIL_ADDRESS_ENTITY, shlexSplit)]

def getUserObjectEntity(clObject, itemType, shlexSplit=False):
  entity = {'item': itemType, 'list': getEntityList(clObject, shlexSplit), 'dict': None}
  if isinstance(entity['list'], dict):
    entity['dict'] = entity['list']
  return entity

def _validateUserGetObjectList(user, i, count, entity, api=API.GMAIL, showAction=True):
  if entity['dict']:
    entityList = entity['dict'][user]
  else:
    entityList = entity['list']
  user, svc = buildGAPIServiceObject(api, user, i, count)
  if not svc:
    return (user, None, [], 0)
  jcount = len(entityList)
  if showAction:
    entityPerformActionNumItems([Ent.USER, user], jcount, entity['item'], i, count)
  if jcount == 0:
    setSysExitRC(NO_ENTITIES_FOUND_RC)
  return (user, svc, entityList, jcount)

def _validateUserGetMessageIds(user, i, count, entity):
  if entity:
    if entity['dict']:
      entityList = entity['dict'][user]
    else:
      entityList = entity['list']
  else:
    entityList = []
  user, gmail = buildGAPIServiceObject(API.GMAIL, user, i, count)
  if not gmail:
    return (user, None, None)
  return (user, gmail, entityList)

def checkUserExists(cd, user, entityType=Ent.USER, i=0, count=0):
  user = normalizeEmailAddressOrUID(user)
  try:
    return callGAPI(cd.users(), 'get',
                    throwReasons=GAPI.USER_GET_THROW_REASONS,
                    userKey=user, fields='primaryEmail')['primaryEmail']
  except (GAPI.userNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden,
          GAPI.badRequest, GAPI.backendError, GAPI.systemError):
    entityUnknownWarning(entityType, user, i, count)
    return None

def checkUserSuspended(cd, user, entityType=Ent.USER, i=0, count=0):
  user = normalizeEmailAddressOrUID(user)
  try:
    return callGAPI(cd.users(), 'get',
                    throwReasons=GAPI.USER_GET_THROW_REASONS,
                    userKey=user, fields='suspended')['suspended']
  except (GAPI.userNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden,
          GAPI.badRequest, GAPI.backendError, GAPI.systemError):
    entityUnknownWarning(entityType, user, i, count)
    return None

# Add attachements to an email message
def _addAttachmentsToMessage(message, attachments):
  for attachment in attachments:
    try:
      attachFilename = attachment[0]
      attachContentType, attachEncoding = mimetypes.guess_type(attachFilename)
      if attachContentType is None or attachEncoding is not None:
        attachContentType = 'application/octet-stream'
      main_type, sub_type = attachContentType.split('/', 1)
      if main_type == 'text':
        msg = MIMEText(readFile(attachFilename, 'r', attachment[1]), _subtype=sub_type, _charset=UTF8)
      elif main_type == 'image':
        msg = MIMEImage(readFile(attachFilename, 'rb'), _subtype=sub_type)
      elif main_type == 'audio':
        msg = MIMEAudio(readFile(attachFilename, 'rb'), _subtype=sub_type)
      elif main_type == 'application':
        msg = MIMEApplication(readFile(attachFilename, 'rb'), _subtype=sub_type)
      else:
        msg = MIMEBase(main_type, sub_type)
        msg.set_payload(readFile(attachFilename, 'rb'))
      msg.add_header('Content-Disposition', 'attachment', filename=os.path.basename(attachFilename))
      message.attach(msg)
    except (IOError, UnicodeDecodeError) as e:
      usageErrorExit(f'{attachFilename}: {str(e)}')

# Add embedded images to an email message
def _addEmbeddedImagesToMessage(message, embeddedImages):
  for embeddedImage in embeddedImages:
    try:
      imageFilename = embeddedImage[0]
      imageContentType, imageEncoding = mimetypes.guess_type(imageFilename)
      if imageContentType is None or imageEncoding is not None:
        imageContentType = 'application/octet-stream'
      main_type, sub_type = imageContentType.split('/', 1)
      if main_type == 'image':
        msg = MIMEImage(readFile(imageFilename, 'rb'), _subtype=sub_type)
      else:
        msg = MIMEBase(main_type, sub_type)
        msg.set_payload(readFile(imageFilename, 'rb'))
      msg.add_header('Content-Disposition', 'attachment', filename=os.path.basename(imageFilename))
      msg.add_header('Content-ID', f'<{embeddedImage[1]}>')
      message.attach(msg)
    except (IOError, UnicodeDecodeError) as e:
      usageErrorExit(f'{imageFilename}: {str(e)}')

NAME_EMAIL_ADDRESS_PATTERN = re.compile(r'^.*<(.+)>$')

# Send an email
def send_email(msgSubject, msgBody, msgTo, i=0, count=0, clientAccess=False, msgFrom=None, msgReplyTo=None,
               html=False, charset=UTF8, attachments=None, embeddedImages=None,
               msgHeaders=None, ccRecipients=None, bccRecipients=None, mailBox=None):
  def checkResult(entityType, recipients):
    if not recipients:
      return
    toSent = set(recipients.split(','))
    toFailed = {}
    for addr, err in result.items():
      if addr in toSent:
        toSent.remove(addr)
        toFailed[addr] = f'{err[0]}: {err[1]}'
    if toSent:
      entityActionPerformed([entityType, ','.join(toSent), Ent.MESSAGE, msgSubject], i, count)
    for addr, errMsg in toFailed.items():
      entityActionFailedWarning([entityType, addr, Ent.MESSAGE, msgSubject], errMsg, i, count)

  def cleanAddr(emailAddr):
    match = NAME_EMAIL_ADDRESS_PATTERN.match(emailAddr)
    if match:
      return match.group(1)
    return emailAddr

  if msgFrom is None:
    msgFrom = _getAdminEmail()
  # Force ASCII for RFC compliance
  # xmlcharref seems to work to display at least
  # some unicode in HTML body and is ignored in
  # plain text body.
#  msgBody = msgBody.encode('ascii', 'xmlcharrefreplace').decode(UTF8)
  if not attachments and not embeddedImages:
    message = MIMEText(msgBody, ['plain', 'html'][html], charset)
  else:
    message = MIMEMultipart()
    msg = MIMEText(msgBody, ['plain', 'html'][html], charset)
    message.attach(msg)
    if attachments:
      _addAttachmentsToMessage(message, attachments)
    if embeddedImages:
      _addEmbeddedImagesToMessage(message, embeddedImages)
  message['Subject'] = msgSubject
  message['From'] = msgFrom
  if msgReplyTo is not None:
    message['Reply-To'] = msgReplyTo
  if ccRecipients:
    message['Cc'] = ccRecipients.lower()
  if bccRecipients:
    message['Bcc'] = bccRecipients.lower()
  if msgHeaders:
    for header, value in msgHeaders.items():
      if header not in {'Subject', 'From', 'To', 'Reply-To', 'Cc', 'Bcc'}:
        message[header] = value
  if mailBox is None:
    mailBox = msgFrom
  mailBoxAddr = normalizeEmailAddressOrUID(cleanAddr(mailBox), noUid=True, noLower=True)
  action = Act.Get()
  Act.Set(Act.SENDEMAIL)
  if not GC.Values[GC.SMTP_HOST]:
    if not clientAccess:
      userId, gmail = buildGAPIServiceObject(API.GMAIL, mailBoxAddr)
      if not gmail:
        return
    else:
      userId = mailBoxAddr
      gmail = buildGAPIObject(API.GMAIL)
    message['To'] = msgTo if msgTo else userId
    try:
      result = callGAPI(gmail.users().messages(), 'send',
                        throwReasons=[GAPI.SERVICE_NOT_AVAILABLE, GAPI.AUTH_ERROR, GAPI.DOMAIN_POLICY,
                                      GAPI.INVALID, GAPI.INVALID_ARGUMENT, GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                        userId=userId, body={'raw': base64.urlsafe_b64encode(message.as_bytes()).decode()}, fields='id')
      entityActionPerformedMessage([Ent.RECIPIENT, msgTo, Ent.MESSAGE, msgSubject], f"{result['id']}", i, count)
    except (GAPI.serviceNotAvailable, GAPI.authError, GAPI.domainPolicy,
            GAPI.invalid, GAPI.invalidArgument, GAPI.forbidden, GAPI.permissionDenied) as e:
      entityActionFailedWarning([Ent.RECIPIENT, msgTo, Ent.MESSAGE, msgSubject], str(e), i, count)
  else:
    message['To'] = msgTo if msgTo else mailBoxAddr
    server = None
    try:
      server = smtplib.SMTP(GC.Values[GC.SMTP_HOST], 587, GC.Values[GC.SMTP_FQDN])
      if GC.Values[GC.DEBUG_LEVEL] > 0:
        server.set_debuglevel(1)
      server.starttls(context=ssl.create_default_context(cafile=GC.Values[GC.CACERTS_PEM]))
      if GC.Values[GC.SMTP_USERNAME] and GC.Values[GC.SMTP_PASSWORD]:
        if isinstance(GC.Values[GC.SMTP_PASSWORD], bytes):
          server.login(GC.Values[GC.SMTP_USERNAME], base64.b64decode(GC.Values[GC.SMTP_PASSWORD]).decode(UTF8))
        else:
          server.login(GC.Values[GC.SMTP_USERNAME], GC.Values[GC.SMTP_PASSWORD])
      result = server.send_message(message)
      checkResult(Ent.RECIPIENT, message['To'])
      checkResult(Ent.RECIPIENT_CC, ccRecipients)
      checkResult(Ent.RECIPIENT_BCC, bccRecipients)
    except smtplib.SMTPException as e:
      entityActionFailedWarning([Ent.RECIPIENT, msgTo, Ent.MESSAGE, msgSubject], str(e), i, count)
    if server:
      try:
        server.quit()
      except Exception:
        pass
  Act.Set(action)

def addFieldToFieldsList(fieldName, fieldsChoiceMap, fieldsList):
  fields = fieldsChoiceMap[fieldName.lower()]
  if isinstance(fields, list):
    fieldsList.extend(fields)
  else:
    fieldsList.append(fields)

def _getFieldsList():
  return getString(Cmd.OB_FIELD_NAME_LIST).lower().replace('_', '').replace(',', ' ').split()

def _getRawFields(requiredField=None):
  rawFields = getString(Cmd.OB_FIELDS)
  if requiredField is None or requiredField in rawFields:
    return rawFields
  return f'{requiredField},{rawFields}'

def CheckInputRowFilterHeaders(titlesList, rowFilter, rowDropFilter):
  status = True
  for filterVal in rowFilter:
    columns = [t for t in titlesList if filterVal[0].match(t)]
    if not columns:
      stderrErrorMsg(Msg.COLUMN_DOES_NOT_MATCH_ANY_INPUT_COLUMNS.format(GC.CSV_INPUT_ROW_FILTER, filterVal[0].pattern))
      status = False
  for filterVal in rowDropFilter:
    columns = [t for t in titlesList if filterVal[0].match(t)]
    if not columns:
      stderrErrorMsg(Msg.COLUMN_DOES_NOT_MATCH_ANY_INPUT_COLUMNS.format(GC.CSV_INPUT_ROW_DROP_FILTER, filterVal[0].pattern))
      status = False
  if not status:
    sys.exit(USAGE_ERROR_RC)

def RowFilterMatch(row, titlesList, rowFilter, rowFilterModeAll, rowDropFilter, rowDropFilterModeAll):
  def rowRegexFilterMatch(filterPattern):
    if anyMatch:
      for column in columns:
        if filterPattern.search(str(row.get(column, ''))):
          return True
      return False
    for column in columns:
      if not filterPattern.search(str(row.get(column, ''))):
        return False
    return True

  def rowNotRegexFilterMatch(filterPattern):
    if anyMatch:
      for column in columns:
        if filterPattern.search(str(row.get(column, ''))):
          return False
      return True
    for column in columns:
      if not filterPattern.search(str(row.get(column, ''))):
        return True

    return False

  def stripTimeFromDateTime(rowDate):
    if YYYYMMDD_PATTERN.match(rowDate):
      try:
        rowTime = datetime.datetime.strptime(rowDate, YYYYMMDD_FORMAT)
      except ValueError:
        return None
    else:
      try:
        rowTime, _ = iso8601.parse_date(rowDate)
      except (iso8601.ParseError, OverflowError):
        return None
    return ISOformatTimeStamp(datetime.datetime(rowTime.year, rowTime.month, rowTime.day, tzinfo=iso8601.UTC))

  def rowDateTimeFilterMatch(dateMode, op, filterDate):
    def checkMatch(rowDate):
      if not rowDate or not isinstance(rowDate, str):
        return False
      if rowDate == GC.Values[GC.NEVER_TIME]:
        rowDate = NEVER_TIME
      if dateMode:
        rowDate = stripTimeFromDateTime(rowDate)
        if not rowDate:
          return False
      if op == '<':
        return rowDate < filterDate
      if op == '<=':
        return rowDate <= filterDate
      if op == '>':
        return rowDate > filterDate
      if op == '>=':
        return rowDate >= filterDate
      if op == '!=':
        return rowDate != filterDate
      return rowDate == filterDate

    if anyMatch:
      for column in columns:
        if checkMatch(row.get(column, '')):
          return True
      return False
    for column in columns:
      if not checkMatch(row.get(column, '')):
        return False
    return True

  def rowDateTimeRangeFilterMatch(dateMode, op, filterDateL, filterDateR):
    def checkMatch(rowDate):
      if not rowDate or not isinstance(rowDate, str):
        return False
      if rowDate == GC.Values[GC.NEVER_TIME]:
        rowDate = NEVER_TIME
      if dateMode:
        rowDate = stripTimeFromDateTime(rowDate)
        if not rowDate:
          return False
      if op == '!=':
        return not filterDateL <= rowDate <= filterDateR
      return filterDateL <= rowDate <= filterDateR

    if anyMatch:
      for column in columns:
        if checkMatch(row.get(column, '')):
          return True
      return False
    for column in columns:
      if not checkMatch(row.get(column, '')):
        return False
    return True

  def getHourMinuteFromDateTime(rowDate):
    if YYYYMMDD_PATTERN.match(rowDate):
      return None
    try:
      rowTime, _ = iso8601.parse_date(rowDate)
    except (iso8601.ParseError, OverflowError):
      return None
    return f'{rowTime.hour:02d}:{rowTime.minute:02d}'

  def rowTimeOfDayRangeFilterMatch(op, startHourMinute, endHourMinute):
    def checkMatch(rowDate):
      if not rowDate or not isinstance(rowDate, str) or rowDate == GC.Values[GC.NEVER_TIME]:
        return False
      rowHourMinute = getHourMinuteFromDateTime(rowDate)
      if not rowHourMinute:
        return False
      if op == '!=':
        return not startHourMinute <= rowHourMinute <= endHourMinute
      return startHourMinute <= rowHourMinute <= endHourMinute

    if anyMatch:
      for column in columns:
        if checkMatch(row.get(column, '')):
          return True
      return False
    for column in columns:
      if not checkMatch(row.get(column, '')):
        return False
    return True

  def rowCountFilterMatch(op, filterCount):
    def checkMatch(rowCount):
      if isinstance(rowCount, str):
##### Blank = 0
        if not rowCount:
          rowCount = '0'
        elif not rowCount.isdigit():
          return False
        rowCount = int(rowCount)
      elif not isinstance(rowCount, int):
        return False
      if op == '<':
        return rowCount < filterCount
      if op == '<=':
        return rowCount <= filterCount
      if op == '>':
        return rowCount > filterCount
      if op == '>=':
        return rowCount >= filterCount
      if op == '!=':
        return rowCount != filterCount
      return rowCount == filterCount

    if anyMatch:
      for column in columns:
        if checkMatch(row.get(column, 0)):
          return True
      return False
    for column in columns:
      if not checkMatch(row.get(column, 0)):
        return False
    return True

  def rowCountRangeFilterMatch(op, filterCountL, filterCountR):
    def checkMatch(rowCount):
      if isinstance(rowCount, str):
        if not rowCount.isdigit():
          return False
        rowCount = int(rowCount)
      elif not isinstance(rowCount, int):
        return False
      if op == '!=':
        return not filterCountL <= rowCount <= filterCountR
      return filterCountL <= rowCount <= filterCountR

    if anyMatch:
      for column in columns:
        if checkMatch(row.get(column, 0)):
          return True
      return False
    for column in columns:
      if not checkMatch(row.get(column, 0)):
        return False
    return True

  def rowLengthFilterMatch(op, filterLength):
    def checkMatch(rowString):
      if not isinstance(rowString, str):
        return False
      rowLength = len(rowString)
      if op == '<':
        return rowLength < filterLength
      if op == '<=':
        return rowLength <= filterLength
      if op == '>':
        return rowLength > filterLength
      if op == '>=':
        return rowLength >= filterLength
      if op == '!=':
        return rowLength != filterLength
      return rowLength == filterLength

    if anyMatch:
      for column in columns:
        if checkMatch(row.get(column, '')):
          return True
      return False
    for column in columns:
      if not checkMatch(row.get(column, '')):
        return False
    return True

  def rowLengthRangeFilterMatch(op, filterLengthL, filterLengthR):
    def checkMatch(rowString):
      if not isinstance(rowString, str):
        return False
      rowLength = len(rowString)
      if op == '!=':
        return not filterLengthL <= rowLength <= filterLengthR
      return filterLengthL <= rowLength <= filterLengthR

    if anyMatch:
      for column in columns:
        if checkMatch(row.get(column, '')):
          return True
      return False
    for column in columns:
      if not checkMatch(row.get(column, '')):
        return False
    return True

  def rowBooleanFilterMatch(filterBoolean):
    def checkMatch(rowBoolean):
      if isinstance(rowBoolean, bool):
        return rowBoolean == filterBoolean
      if isinstance(rowBoolean, str):
        if rowBoolean.lower() in TRUE_FALSE:
          return rowBoolean.capitalize() == str(filterBoolean)
##### Blank = False
        if not rowBoolean:
          return not filterBoolean
      return False

    if anyMatch:
      for column in columns:
        if checkMatch(row.get(column, False)):
          return True
      return False
    for column in columns:
      if not checkMatch(row.get(column, False)):
        return False
    return True

  def rowDataFilterMatch(filterData):
    if anyMatch:
      for column in columns:
        if str(row.get(column, '')) in filterData:
          return True
      return False
    for column in columns:
      if not str(row.get(column, '')) in filterData:
        return False
    return True

  def rowNotDataFilterMatch(filterData):
    if anyMatch:
      for column in columns:
        if str(row.get(column, '')) in filterData:
          return False
      return True
    for column in columns:
      if not str(row.get(column, '')) in filterData:
        return True
    return False

  def rowTextFilterMatch(op, filterText):
    def checkMatch(rowText):
      if not isinstance(rowText, str):
        rowText = str(rowText)
      if op == '<':
        return rowText < filterText
      if op == '<=':
        return rowText <= filterText
      if op == '>':
        return rowText > filterText
      if op == '>=':
        return rowText >= filterText
      if op == '!=':
        return rowText != filterText
      return rowText == filterText

    if anyMatch:
      for column in columns:
        if checkMatch(row.get(column, '')):
          return True
      return False
    for column in columns:
      if not checkMatch(row.get(column, '')):
        return False
    return True

  def rowTextRangeFilterMatch(op, filterTextL, filterTextR):
    def checkMatch(rowText):
      if not isinstance(rowText, str):
        rowText = str(rowText)
      if op == '!=':
        return not filterTextL <= rowText <= filterTextR
      return filterTextL <= rowText <= filterTextR

    if anyMatch:
      for column in columns:
        if checkMatch(row.get(column, '')):
          return True
      return False
    for column in columns:
      if not checkMatch(row.get(column, '')):
        return False
    return True

  def filterMatch(filterVal):
    if filterVal[2] == 'regex':
      if rowRegexFilterMatch(filterVal[3]):
        return True
    elif filterVal[2] == 'notregex':
      if rowNotRegexFilterMatch(filterVal[3]):
        return True
    elif filterVal[2] in {'date', 'time'}:
      if rowDateTimeFilterMatch(filterVal[2] == 'date', filterVal[3], filterVal[4]):
        return True
    elif filterVal[2] in {'daterange', 'timerange'}:
      if rowDateTimeRangeFilterMatch(filterVal[2] == 'date', filterVal[3], filterVal[4], filterVal[5]):
        return True
    elif filterVal[2] == 'timeofdayrange':
      if rowTimeOfDayRangeFilterMatch(filterVal[3], filterVal[4], filterVal[5]):
        return True
    elif filterVal[2] == 'count':
      if rowCountFilterMatch(filterVal[3], filterVal[4]):
        return True
    elif filterVal[2] == 'countrange':
      if rowCountRangeFilterMatch(filterVal[3], filterVal[4], filterVal[5]):
        return True
    elif filterVal[2] == 'length':
      if rowLengthFilterMatch(filterVal[3], filterVal[4]):
        return True
    elif filterVal[2] == 'lengthrange':
      if rowLengthRangeFilterMatch(filterVal[3], filterVal[4], filterVal[5]):
        return True
    elif filterVal[2] == 'boolean':
      if rowBooleanFilterMatch(filterVal[3]):
        return True
    elif filterVal[2] == 'data':
      if rowDataFilterMatch(filterVal[3]):
        return True
    elif filterVal[2] == 'notdata':
      if rowNotDataFilterMatch(filterVal[3]):
        return True
    elif filterVal[2] == 'text':
      if rowTextFilterMatch(filterVal[3], filterVal[4]):
        return True
    elif filterVal[2] == 'textrange':
      if rowTextRangeFilterMatch(filterVal[3], filterVal[4], filterVal[5]):
        return True
    return False

  if rowFilter:
    anyMatches = False
    for filterVal in rowFilter:
      columns = [t for t in titlesList if filterVal[0].match(t)]
      if not columns:
        columns = [None]
      anyMatch = filterVal[1]
      if filterMatch(filterVal):
        if not rowFilterModeAll: # Any - any match selects
          anyMatches = True
          break
      else:
        if rowFilterModeAll: # All - any match failure doesn't select
          return False
    if not rowFilterModeAll and not anyMatches: # Any - no matches doesn't select
      return False
  if rowDropFilter:
    allMatches = True
    for filterVal in rowDropFilter:
      columns = [t for t in titlesList if filterVal[0].match(t)]
      if not columns:
        columns = [None]
      anyMatch = filterVal[1]
      if filterMatch(filterVal):
        if not rowDropFilterModeAll: # Any - any match drops
          return False
      else:
        if rowDropFilterModeAll: # All - any match failure doesn't drop
          allMatches = False
          break
    if rowDropFilterModeAll and allMatches: # All - all matches drops
      return False
  return True

# myarg is command line argument
# fieldChoiceMap maps myarg to API field names
#FIELD_CHOICE_MAP = {
#  'foo': 'foo',
#  'foobar': 'fooBar',
#  }
# fieldsList is the list of API fields
def getFieldsList(myarg, fieldsChoiceMap, fieldsList, initialField=None, fieldsArg='fields', onlyFieldsArg=False):
  def addInitialField():
    if isinstance(initialField, list):
      fieldsList.extend(initialField)
    else:
      fieldsList.append(initialField)

  def addMappedFields(mappedFields):
    if isinstance(mappedFields, list):
      fieldsList.extend(mappedFields)
    else:
      fieldsList.append(mappedFields)

  if not onlyFieldsArg and myarg in fieldsChoiceMap:
    if not fieldsList and initialField is not None:
      addInitialField()
    addMappedFields(fieldsChoiceMap[myarg])
  elif myarg == fieldsArg:
    if not fieldsList and initialField is not None:
      addInitialField()
    for field in _getFieldsList():
      if field in fieldsChoiceMap:
        addMappedFields(fieldsChoiceMap[field])
      else:
        invalidChoiceExit(field, fieldsChoiceMap, True)
  else:
    return False
  return True

def getFieldsFromFieldsList(fieldsList):
  if fieldsList:
    return ','.join(set(fieldsList)).replace('.', '/')
  return None

def getItemFieldsFromFieldsList(item, fieldsList, returnItemIfNoneList=False):
  if fieldsList:
    return f'nextPageToken,{item}({",".join(set(fieldsList))})'.replace('.', '/')
  if not returnItemIfNoneList:
    return None
  return f'nextPageToken,{item}'

class CSVPrintFile():

  def __init__(self, titles=None, sortTitles=None, indexedTitles=None):
    self.rows = []
    self.rowCount = 0
    self.outputTranspose = GM.Globals[GM.CSV_OUTPUT_TRANSPOSE]
    self.todrive = GM.Globals[GM.CSV_TODRIVE]
    self.titlesSet = set()
    self.titlesList = []
    self.JSONtitlesSet = set()
    self.JSONtitlesList = []
    self.sortHeaders = []
    self.SetHeaderForce(GC.Values[GC.CSV_OUTPUT_HEADER_FORCE])
    if not self.headerForce and titles is not None:
      self.SetTitles(titles)
      self.SetJSONTitles(titles)
    self.SetHeaderOrder(GC.Values[GC.CSV_OUTPUT_HEADER_ORDER])
    if GM.Globals.get(GM.CSV_OUTPUT_COLUMN_DELIMITER) is None:
      GM.Globals[GM.CSV_OUTPUT_COLUMN_DELIMITER] = GC.Values.get(GC.CSV_OUTPUT_COLUMN_DELIMITER, ',')
    self.SetColumnDelimiter(GM.Globals[GM.CSV_OUTPUT_COLUMN_DELIMITER])
    if GM.Globals.get(GM.CSV_OUTPUT_QUOTE_CHAR) is None:
      GM.Globals[GM.CSV_OUTPUT_QUOTE_CHAR] = GC.Values.get(GC.CSV_OUTPUT_QUOTE_CHAR, '"')
    if GM.Globals.get(GM.CSV_OUTPUT_NO_ESCAPE_CHAR) is None:
      GM.Globals[GM.CSV_OUTPUT_NO_ESCAPE_CHAR] = GC.Values.get(GC.CSV_OUTPUT_NO_ESCAPE_CHAR, False)
    self.SetNoEscapeChar(GM.Globals[GM.CSV_OUTPUT_NO_ESCAPE_CHAR])
    self.SetQuoteChar(GM.Globals[GM.CSV_OUTPUT_QUOTE_CHAR])
#    if GM.Globals.get(GM.CSV_OUTPUT_SORT_HEADERS) is None:
    if not GM.Globals.get(GM.CSV_OUTPUT_SORT_HEADERS):
      GM.Globals[GM.CSV_OUTPUT_SORT_HEADERS] = GC.Values.get(GC.CSV_OUTPUT_SORT_HEADERS, [])
    self.SetSortHeaders(GM.Globals[GM.CSV_OUTPUT_SORT_HEADERS])
#    if GM.Globals.get(GM.CSV_OUTPUT_TIMESTAMP_COLUMN) is None:
    if not GM.Globals.get(GM.CSV_OUTPUT_TIMESTAMP_COLUMN):
      GM.Globals[GM.CSV_OUTPUT_TIMESTAMP_COLUMN] = GC.Values.get(GC.CSV_OUTPUT_TIMESTAMP_COLUMN, '')
    self.SetTimestampColumn(GM.Globals[GM.CSV_OUTPUT_TIMESTAMP_COLUMN])
    self.SetFormatJSON(False)
    self.SetMapDrive3Titles(False)
    self.SetNodataFields(False, None, None, None, False)
    self.SetFixPaths(False)
    self.SetShowPermissionsLast(False)
    self.sortTitlesSet = set()
    self.sortTitlesList = []
    if sortTitles is not None:
      if not isinstance(sortTitles, str) or sortTitles != 'sortall':
        self.SetSortTitles(sortTitles)
      else:
        self.SetSortAllTitles()
    self.SetIndexedTitles(indexedTitles if indexedTitles is not None else [])
    self.SetHeaderFilter(GC.Values[GC.CSV_OUTPUT_HEADER_FILTER])
    self.SetHeaderDropFilter(GC.Values[GC.CSV_OUTPUT_HEADER_DROP_FILTER])
    self.SetRowFilter(GC.Values[GC.CSV_OUTPUT_ROW_FILTER], GC.Values[GC.CSV_OUTPUT_ROW_FILTER_MODE])
    self.SetRowDropFilter(GC.Values[GC.CSV_OUTPUT_ROW_DROP_FILTER], GC.Values[GC.CSV_OUTPUT_ROW_DROP_FILTER_MODE])
    self.SetRowLimit(GC.Values[GC.CSV_OUTPUT_ROW_LIMIT])
    self.SetZeroBlankMimeTypeCounts(False)

  def AddTitle(self, title):
    self.titlesSet.add(title)
    self.titlesList.append(title)

  def AddTitles(self, titles):
    for title in titles if isinstance(titles, list) else [titles]:
      if title not in self.titlesSet:
        self.AddTitle(title)

  def SetTitles(self, titles):
    self.titlesSet = set()
    self.titlesList = []
    self.AddTitles(titles)

  def RemoveTitles(self, titles):
    for title in titles if isinstance(titles, list) else [titles]:
      if title in self.titlesSet:
        self.titlesSet.remove(title)
        self.titlesList.remove(title)

  def MoveTitlesToEnd(self, titles):
    self.RemoveTitles(titles)
    self.AddTitles(titles)

  def MapTitles(self, ov, nv):
    if ov in self.titlesSet:
      self.titlesSet.remove(ov)
      self.titlesSet.add(nv)
      for i, v in enumerate(self.titlesList):
        if v == ov:
          self.titlesList[i] = nv
          break

  def AddSortTitle(self, title):
    self.sortTitlesSet.add(title)
    self.sortTitlesList.append(title)

  def AddSortTitles(self, titles):
    for title in titles if isinstance(titles, list) else [titles]:
      if title not in self.sortTitlesSet:
        self.AddSortTitle(title)

  def SetSortTitles(self, titles):
    self.sortTitlesSet = set()
    self.sortTitlesList = []
    self.AddSortTitles(titles)

  def SetSortAllTitles(self):
    self.sortTitlesList = self.titlesList[:]
    self.sortTitlesSet = set(self.sortTitlesList)

  def SetMapDrive3Titles(self, mapDrive3Titles):
    self.mapDrive3Titles = mapDrive3Titles

  def MapDrive3TitlesToDrive2(self):
    _mapDrive3TitlesToDrive2(self.titlesList, API.DRIVE3_TO_DRIVE2_FILES_FIELDS_MAP)
    self.titlesSet = set(self.titlesList)

  def AddJSONTitle(self, title):
    self.JSONtitlesSet.add(title)
    self.JSONtitlesList.append(title)

  def AddJSONTitles(self, titles):
    for title in titles if isinstance(titles, list) else [titles]:
      if title not in self.JSONtitlesSet:
        self.AddJSONTitle(title)

  def RemoveJSONTitles(self, titles):
    for title in titles if isinstance(titles, list) else [titles]:
      if title in self.JSONtitlesSet:
        self.JSONtitlesSet.remove(title)
        self.JSONtitlesList.remove(title)

  def MoveJSONTitlesToEnd(self, titles):
    for title in titles if isinstance(titles, list) else [titles]:
      if title in self.JSONtitlesList:
        self.JSONtitlesList.remove(title)
      self.JSONtitlesList.append(title)

  def SetJSONTitles(self, titles):
    self.JSONtitlesSet = set()
    self.JSONtitlesList = []
    self.AddJSONTitles(titles)

# fieldName is command line argument
# fieldNameMap maps fieldName to API field names; CSV file header will be API field name
#ARGUMENT_TO_PROPERTY_MAP = {
#  'admincreated': 'adminCreated',
#  'aliases': ['aliases', 'nonEditableAliases'],
#  }
# fieldsList is the list of API fields
  def AddField(self, fieldName, fieldNameMap, fieldsList):
    fields = fieldNameMap[fieldName.lower()]
    if isinstance(fields, list):
      for field in fields:
        if field not in fieldsList:
          fieldsList.append(field)
          self.AddTitles(field.replace('.', GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]))
    elif fields not in fieldsList:
      fieldsList.append(fields)
      self.AddTitles(fields.replace('.', GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]))

  def addInitialField(self, initialField, fieldsChoiceMap, fieldsList):
    if isinstance(initialField, list):
      for field in initialField:
        self.AddField(field, fieldsChoiceMap, fieldsList)
    else:
      self.AddField(initialField, fieldsChoiceMap, fieldsList)

  def GetFieldsListTitles(self, fieldName, fieldsChoiceMap, fieldsList, initialField=None):
    if fieldName in fieldsChoiceMap:
      if not fieldsList and initialField is not None:
        self.addInitialField(initialField, fieldsChoiceMap, fieldsList)
      self.AddField(fieldName, fieldsChoiceMap, fieldsList)
    elif fieldName == 'fields':
      if not fieldsList and initialField is not None:
        self.addInitialField(initialField, fieldsChoiceMap, fieldsList)
      for field in _getFieldsList():
        if field in fieldsChoiceMap:
          self.AddField(field, fieldsChoiceMap, fieldsList)
        else:
          invalidChoiceExit(field, fieldsChoiceMap, True)
    else:
      return False
    return True

  TDSHEET_ENTITY_MAP = {'tdsheet': 'sheetEntity', 'tdbackupsheet': 'backupSheetEntity', 'tdcopysheet': 'copySheetEntity'}
  TDSHARE_ACL_ROLES_MAP = {
    'commenter': 'commenter',
    'contributor': 'writer',
    'editor': 'writer',
    'read': 'reader',
    'reader': 'reader',
    'viewer': 'reader',
    'writer': 'writer',
    }


  def GetTodriveParameters(self):
    def invalidTodriveFileIdExit(entityValueList, message, location):
      Cmd.SetLocation(location-1)
      usageErrorExit(formatKeyValueList('', Ent.FormatEntityValueList([Ent.DRIVE_FILE_ID, self.todrive['fileId']]+entityValueList)+[message], ''))

    def invalidTodriveParentExit(entityType, message):
      Cmd.SetLocation(tdparentLocation-1)
      if not localParent:
        usageErrorExit(Msg.INVALID_ENTITY.format(Ent.Singular(entityType),
                                                 formatKeyValueList('',
                                                                    [Ent.Singular(Ent.CONFIG_FILE), GM.Globals[GM.GAM_CFG_FILE],
                                                                     Ent.Singular(Ent.ITEM), GC.TODRIVE_PARENT,
                                                                     Ent.Singular(Ent.VALUE), self.todrive['parent'],
                                                                     message],
                                                                    '')))
      else:
        usageErrorExit(Msg.INVALID_ENTITY.format(Ent.Singular(entityType), message))

    def invalidTodriveUserExit(entityType, message):
      Cmd.SetLocation(tduserLocation-1)
      if not localUser:
        usageErrorExit(Msg.INVALID_ENTITY.format(Ent.Singular(entityType),
                                                 formatKeyValueList('',
                                                                    [Ent.Singular(Ent.CONFIG_FILE), GM.Globals[GM.GAM_CFG_FILE],
                                                                     Ent.Singular(Ent.ITEM), GC.TODRIVE_USER,
                                                                     Ent.Singular(Ent.VALUE), self.todrive['user'],
                                                                     message],
                                                                    '')))
      else:
        usageErrorExit(Msg.INVALID_ENTITY.format(Ent.Singular(entityType), message))

    def getDriveObject():
      if not GC.Values[GC.TODRIVE_CLIENTACCESS]:
        _, drive = buildGAPIServiceObject(API.DRIVETD, self.todrive['user'])
        if not drive:
          invalidTodriveUserExit(Ent.USER, Msg.NOT_FOUND)
      else:
        drive = buildGAPIObject(API.DRIVE3)
      return drive

    CELL_WRAP_MAP = {'clip': 'CLIP', 'overflow': 'OVERFLOW_CELL', 'overflowcell': 'OVERFLOW_CELL', 'wrap': 'WRAP'}
    CELL_NUMBER_FORMAT_MAP = {'text': 'TEXT', 'number': 'NUMBER'}

    localUser = localParent = False
    tdfileidLocation = tdparentLocation = tdaddsheetLocation = tdupdatesheetLocation = tduserLocation = Cmd.Location()
    tdsheetLocation = {}
    for sheetEntity in self.TDSHEET_ENTITY_MAP.values():
      tdsheetLocation[sheetEntity] = Cmd.Location()
    self.todrive = {'user': GC.Values[GC.TODRIVE_USER], 'title': None, 'description': None,
                    'sheetEntity': None, 'addsheet': False, 'updatesheet': False, 'sheettitle': None,
                    'cellwrap': None, 'cellnumberformat': None, 'clearfilter': GC.Values[GC.TODRIVE_CLEARFILTER],
                    'backupSheetEntity': None, 'copySheetEntity': None,
                    'locale': GC.Values[GC.TODRIVE_LOCALE], 'timeZone': GC.Values[GC.TODRIVE_TIMEZONE],
                    'timestamp': GC.Values[GC.TODRIVE_TIMESTAMP], 'timeformat': GC.Values[GC.TODRIVE_TIMEFORMAT],
                    'noescapechar': GC.Values[GC.TODRIVE_NO_ESCAPE_CHAR],
                    'daysoffset': None, 'hoursoffset': None,
                    'sheettimestamp': GC.Values[GC.TODRIVE_SHEET_TIMESTAMP], 'sheettimeformat': GC.Values[GC.TODRIVE_SHEET_TIMEFORMAT],
                    'sheetdaysoffset': None, 'sheethoursoffset': None,
                    'fileId': None, 'parentId': None, 'parent': GC.Values[GC.TODRIVE_PARENT], 'retaintitle': False,
                    'localcopy': GC.Values[GC.TODRIVE_LOCALCOPY], 'uploadnodata': GC.Values[GC.TODRIVE_UPLOAD_NODATA],
                    'nobrowser': GC.Values[GC.TODRIVE_NOBROWSER], 'noemail': GC.Values[GC.TODRIVE_NOEMAIL], 'returnidonly': False,
                    'alert': [], 'share': [], 'notify': False, 'subject': None, 'from': None}
    while Cmd.ArgumentsRemaining():
      myarg = getArgument()
      if myarg == 'tduser':
        self.todrive['user'] = getString(Cmd.OB_EMAIL_ADDRESS)
        tduserLocation = Cmd.Location()
        localUser = True
      elif myarg == 'tdtitle':
        self.todrive['title'] = getString(Cmd.OB_STRING, minLen=0)
      elif myarg == 'tddescription':
        self.todrive['description'] = getString(Cmd.OB_STRING)
      elif myarg in self.TDSHEET_ENTITY_MAP:
        sheetEntity = self.TDSHEET_ENTITY_MAP[myarg]
        tdsheetLocation[sheetEntity] = Cmd.Location()
        self.todrive[sheetEntity] = getSheetEntity(True)
      elif myarg == 'tdaddsheet':
        tdaddsheetLocation = Cmd.Location()
        self.todrive['addsheet'] = getBoolean()
        if self.todrive['addsheet']:
          self.todrive['updatesheet'] = False
      elif myarg == 'tdupdatesheet':
        tdupdatesheetLocation = Cmd.Location()
        self.todrive['updatesheet'] = getBoolean()
        if self.todrive['updatesheet']:
          self.todrive['addsheet'] = False
      elif myarg == 'tdcellwrap':
        self.todrive['cellwrap'] = getChoice(CELL_WRAP_MAP, mapChoice=True)
      elif myarg == 'tdcellnumberformat':
        self.todrive['cellnumberformat'] = getChoice(CELL_NUMBER_FORMAT_MAP, mapChoice=True)
      elif myarg == 'tdclearfilter':
        self.todrive['clearfilter'] = getBoolean()
      elif myarg == 'tdlocale':
        self.todrive['locale'] = getLanguageCode(LOCALE_CODES_MAP)
      elif myarg == 'tdtimezone':
        self.todrive['timeZone'] = getString(Cmd.OB_STRING, minLen=0)
      elif myarg == 'tdtimestamp':
        self.todrive['timestamp'] = getBoolean()
      elif myarg == 'tdtimeformat':
        self.todrive['timeformat'] = getString(Cmd.OB_STRING, minLen=0)
      elif myarg == 'tdsheettitle':
        self.todrive['sheettitle'] = getString(Cmd.OB_STRING, minLen=0)
      elif myarg == 'tdsheettimestamp':
        self.todrive['sheettimestamp'] = getBoolean()
      elif myarg == 'tdsheettimeformat':
        self.todrive['sheettimeformat'] = getString(Cmd.OB_STRING, minLen=0)
      elif myarg == 'tddaysoffset':
        self.todrive['daysoffset'] = getInteger(minVal=0)
      elif myarg == 'tdhoursoffset':
        self.todrive['hoursoffset'] = getInteger(minVal=0)
      elif myarg == 'tdsheetdaysoffset':
        self.todrive['sheetdaysoffset'] = getInteger(minVal=0)
      elif myarg == 'tdsheethoursoffset':
        self.todrive['sheethoursoffset'] = getInteger(minVal=0)
      elif myarg == 'tdfileid':
        self.todrive['fileId'] = getString(Cmd.OB_DRIVE_FILE_ID)
        tdfileidLocation = Cmd.Location()
      elif myarg == 'tdretaintitle':
        self.todrive['retaintitle'] = getBoolean()
      elif myarg == 'tdparent':
        self.todrive['parent'] = escapeDriveFileName(getString(Cmd.OB_DRIVE_FOLDER_NAME, minLen=0))
        tdparentLocation = Cmd.Location()
        localParent = True
      elif myarg == 'tdlocalcopy':
        self.todrive['localcopy'] = getBoolean()
      elif myarg == 'tduploadnodata':
        self.todrive['uploadnodata'] = getBoolean()
      elif myarg == 'tdnobrowser':
        self.todrive['nobrowser'] = getBoolean()
      elif myarg == 'tdnoemail':
        self.todrive['noemail'] = getBoolean()
      elif myarg == 'tdreturnidonly':
        self.todrive['returnidonly'] = getBoolean()
      elif myarg == 'tdnoescapechar':
        self.todrive['noescapechar'] = getBoolean()
      elif myarg == 'tdalert':
        self.todrive['alert'].append({'emailAddress': normalizeEmailAddressOrUID(getString(Cmd.OB_EMAIL_ADDRESS))})
      elif myarg == 'tdshare':
        self.todrive['share'].append({'emailAddress': normalizeEmailAddressOrUID(getString(Cmd.OB_EMAIL_ADDRESS)),
                                      'type': 'user',
                                      'role': getChoice(self.TDSHARE_ACL_ROLES_MAP, mapChoice=True)})
      elif myarg == 'tdnotify':
        self.todrive['notify'] = getBoolean()
      elif myarg == 'tdsubject':
        self.todrive['subject'] = getString(Cmd.OB_STRING, minLen=0)
      elif myarg == 'tdfrom':
        self.todrive['from'] = getString(Cmd.OB_EMAIL_ADDRESS)
      else:
        Cmd.Backup()
        break
    if self.todrive['addsheet']:
      if not self.todrive['fileId']:
        Cmd.SetLocation(tdaddsheetLocation-1)
        missingArgumentExit('tdfileid')
      if self.todrive['sheetEntity'] and self.todrive['sheetEntity']['sheetId']:
        Cmd.SetLocation(tdsheetLocation[sheetEntity]-1)
        invalidArgumentExit('tdsheet <String>')
    if self.todrive['updatesheet'] and (not self.todrive['fileId'] or not self.todrive['sheetEntity']):
      Cmd.SetLocation(tdupdatesheetLocation-1)
      missingArgumentExit('tdfileid and tdsheet')
    if self.todrive['sheetEntity'] and self.todrive['sheetEntity']['sheetId'] and (not self.todrive['fileId'] or not self.todrive['updatesheet']):
      Cmd.SetLocation(tdsheetLocation['sheetEntity']-1)
      missingArgumentExit('tdfileid and tdupdatesheet')
    if not self.todrive['user'] or GC.Values[GC.TODRIVE_CLIENTACCESS]:
      self.todrive['user'] = _getAdminEmail()
    if not GC.Values[GC.USER_SERVICE_ACCOUNT_ACCESS_ONLY] and not GC.Values[GC.TODRIVE_CLIENTACCESS]:
      user = checkUserExists(buildGAPIObject(API.DIRECTORY), self.todrive['user'])
      if not user:
        invalidTodriveUserExit(Ent.USER, Msg.NOT_FOUND)
      self.todrive['user'] = user
    else:
      self.todrive['user'] = normalizeEmailAddressOrUID(self.todrive['user'])
    if self.todrive['fileId']:
      drive = getDriveObject()
      try:
        result = callGAPI(drive.files(), 'get',
                          throwReasons=GAPI.DRIVE_GET_THROW_REASONS,
                          fileId=self.todrive['fileId'], fields='id,mimeType,capabilities(canEdit)', supportsAllDrives=True)
        if result['mimeType'] == MIMETYPE_GA_FOLDER:
          invalidTodriveFileIdExit([], Msg.NOT_AN_ENTITY.format(Ent.Singular(Ent.DRIVE_FILE)), tdfileidLocation)
        if not result['capabilities']['canEdit']:
          invalidTodriveFileIdExit([], Msg.NOT_WRITABLE, tdfileidLocation)
        if self.todrive['sheetEntity']:
          if result['mimeType'] != MIMETYPE_GA_SPREADSHEET:
            invalidTodriveFileIdExit([], f'{Msg.NOT_A} {Ent.Singular(Ent.SPREADSHEET)}', tdfileidLocation)
          if not GC.Values[GC.TODRIVE_CLIENTACCESS]:
            _, sheet = buildGAPIServiceObject(API.SHEETSTD, self.todrive['user'])
            if sheet is None:
              invalidTodriveUserExit(Ent.USER, Msg.NOT_FOUND)
          else:
            sheet = buildGAPIObject(API.SHEETS)
          try:
            spreadsheet = callGAPI(sheet.spreadsheets(), 'get',
                                   throwReasons=GAPI.SHEETS_ACCESS_THROW_REASONS,
                                   spreadsheetId=self.todrive['fileId'],
                                   fields='spreadsheetUrl,sheets(properties(sheetId,title),protectedRanges(range(sheetId),requestingUserCanEdit))')
            for sheetEntity in self.TDSHEET_ENTITY_MAP.values():
              if self.todrive[sheetEntity]:
                sheetId = getSheetIdFromSheetEntity(spreadsheet, self.todrive[sheetEntity])
                if sheetId is None:
                  if not self.todrive['addsheet'] and ((sheetEntity != 'sheetEntity') or (self.todrive[sheetEntity]['sheetType'] == Ent.SHEET_ID)):
                    invalidTodriveFileIdExit([self.todrive[sheetEntity]['sheetType'], self.todrive[sheetEntity]['sheetValue']], Msg.NOT_FOUND, tdsheetLocation[sheetEntity])
                else:
                  if self.todrive['addsheet']:
                    invalidTodriveFileIdExit([self.todrive[sheetEntity]['sheetType'], self.todrive[sheetEntity]['sheetValue']], Msg.ALREADY_EXISTS, tdsheetLocation[sheetEntity])
                  if protectedSheetId(spreadsheet, sheetId):
                    invalidTodriveFileIdExit([self.todrive[sheetEntity]['sheetType'], self.todrive[sheetEntity]['sheetValue']], Msg.NOT_WRITABLE, tdsheetLocation[sheetEntity])
                self.todrive[sheetEntity]['sheetId'] = sheetId
          except (GAPI.notFound, GAPI.forbidden, GAPI.permissionDenied,
                  GAPI.internalError, GAPI.insufficientFilePermissions, GAPI.badRequest,
                  GAPI.invalid, GAPI.invalidArgument, GAPI.failedPrecondition) as e:
            invalidTodriveFileIdExit([], str(e), tdfileidLocation)
      except GAPI.fileNotFound:
        invalidTodriveFileIdExit([], Msg.NOT_FOUND, tdfileidLocation)
      except (GAPI.serviceNotAvailable, GAPI.authError, GAPI.domainPolicy) as e:
        invalidTodriveUserExit(Ent.USER, str(e))
    elif not self.todrive['parent'] or self.todrive['parent'] == ROOT:
      self.todrive['parentId'] = ROOT
    else:
      drive = getDriveObject()
      if self.todrive['parent'].startswith('id:'):
        try:
          result = callGAPI(drive.files(), 'get',
                            throwReasons=GAPI.DRIVE_USER_THROW_REASONS+[GAPI.FILE_NOT_FOUND, GAPI.INVALID],
                            fileId=self.todrive['parent'][3:], fields='id,mimeType,capabilities(canEdit)', supportsAllDrives=True)
        except GAPI.fileNotFound:
          invalidTodriveParentExit(Ent.DRIVE_FOLDER_ID, Msg.NOT_FOUND)
        except GAPI.invalid as e:
          invalidTodriveParentExit(Ent.DRIVE_FOLDER_ID, str(e))
        except (GAPI.serviceNotAvailable, GAPI.authError, GAPI.domainPolicy) as e:
          invalidTodriveUserExit(Ent.USER, str(e))
        if result['mimeType'] != MIMETYPE_GA_FOLDER:
          invalidTodriveParentExit(Ent.DRIVE_FOLDER_ID, Msg.NOT_AN_ENTITY.format(Ent.Singular(Ent.DRIVE_FOLDER)))
        if not result['capabilities']['canEdit']:
          invalidTodriveParentExit(Ent.DRIVE_FOLDER_ID, Msg.NOT_WRITABLE)
        self.todrive['parentId'] = result['id']
      else:
        try:
          results = callGAPIpages(drive.files(), 'list', 'files',
                                  throwReasons=GAPI.DRIVE_USER_THROW_REASONS+[GAPI.INVALID_QUERY],
                                  retryReasons=[GAPI.UNKNOWN_ERROR],
                                  q=f"name = '{self.todrive['parent']}'",
                                  fields='nextPageToken,files(id,mimeType,capabilities(canEdit))',
                                  pageSize=1, supportsAllDrives=True)
        except GAPI.invalidQuery:
          invalidTodriveParentExit(Ent.DRIVE_FOLDER_NAME, Msg.NOT_FOUND)
        except (GAPI.serviceNotAvailable, GAPI.authError, GAPI.domainPolicy) as e:
          invalidTodriveUserExit(Ent.USER, str(e))
        if not results:
          invalidTodriveParentExit(Ent.DRIVE_FOLDER_NAME, Msg.NOT_FOUND)
        if results[0]['mimeType'] != MIMETYPE_GA_FOLDER:
          invalidTodriveParentExit(Ent.DRIVE_FOLDER_NAME, Msg.NOT_AN_ENTITY.format(Ent.Singular(Ent.DRIVE_FOLDER)))
        if not results[0]['capabilities']['canEdit']:
          invalidTodriveParentExit(Ent.DRIVE_FOLDER_NAME, Msg.NOT_WRITABLE)
        self.todrive['parentId'] = results[0]['id']

  def SortTitles(self):
    if not self.sortTitlesList:
      return
    restoreTitles = []
    for title in self.sortTitlesList:
      if title in self.titlesSet:
        self.titlesList.remove(title)
        restoreTitles.append(title)
    self.titlesList.sort()
    for title in restoreTitles[::-1]:
      self.titlesList.insert(0, title)

  def RemoveIndexedTitles(self, titles):
    for title in titles if isinstance(titles, list) else [titles]:
      if title in self.indexedTitles:
        self.indexedTitles.remove(title)

  def SetIndexedTitles(self, indexedTitles):
    self.indexedTitles = indexedTitles

  def SortIndexedTitles(self, titlesList):
    for field in self.indexedTitles:
      fieldDotN = re.compile(fr'({field}){GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}(\d+)(.*)')
      indexes = []
      subtitles = []
      for i, v in enumerate(titlesList):
        mg = fieldDotN.match(v)
        if mg:
          indexes.append(i)
          subtitles.append(mg.groups(''))
      for i, ii in enumerate(indexes):
        titlesList[ii] = [f'{subtitle[0]}{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{subtitle[1]}{subtitle[2]}' for subtitle in sorted(subtitles, key=lambda k: (int(k[1]), k[2]))][i]

  @staticmethod
  def FixPathsTitles(titlesList):
# Put paths before path.0
    try:
      index = titlesList.index(f'path{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}0')
      titlesList.remove('paths')
      titlesList.insert(index, 'paths')
    except ValueError:
      pass

  def FixNodataTitles(self):
    if self.mapNodataFields:
      titles = []
      addPermissionsTitle = not self.oneItemPerRow
      for field in self.nodataFields:
        if field.find('(') != -1:
          field, subFields = field.split('(', 1)
          if field in self.driveListFields:
            if field != 'permissions':
              titles.append(field)
            elif addPermissionsTitle:
              titles.append(field)
              addPermissionsTitle = False
            titles.extend([f'{field}{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}0{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{subField}' for subField in subFields[:-1].split(',') if subField])
          else:
            titles.extend([f'{field}{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{subField}' for subField in subFields[:-1].split(',') if subField])
        elif field.find('.') != -1:
          field, subField = field.split('.', 1)
          if field in self.driveListFields:
            if field != 'permissions':
              titles.append(field)
            elif addPermissionsTitle:
              titles.append(field)
              addPermissionsTitle = False
            titles.append(f'{field}{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}0{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{subField}')
          else:
            titles.append(f'{field}{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{subField}')
        elif field.lower() in self.driveSubfieldsChoiceMap:
          if field in self.driveListFields:
            if field != 'permissions':
              titles.append(field)
            elif addPermissionsTitle:
              titles.append(field)
              addPermissionsTitle = False
            for subField in self.driveSubfieldsChoiceMap[field.lower()].values():
              if not isinstance(subField, list):
                titles.append(f'{field}{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}0{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{subField}')
              else:
                titles.extend([f'{field}{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}0{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{subSubField}' for subSubField in subField])
          else:
            for subField in self.driveSubfieldsChoiceMap[field.lower()].values():
              if not isinstance(subField, list):
                titles.append(f'{field}{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{subField}')
              else:
                titles.extend([f'{field}{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{subSubField}' for subSubField in subField])
        else:
          titles.append(field)
        if self.oneItemPerRow:
          for i, title in enumerate(titles):
            if title.startswith('permissions.0'):
              titles[i] = title.replace('permissions.0', 'permission')
      if not self.formatJSON:
        self.SetTitles(titles)
        self.SetSortTitles(['Owner', 'id', 'name', 'title'])
        self.SortTitles()
      else:
        self.SetJSONTitles(titles)
    else:
      self.SetTitles(self.nodataFields)
      self.SetJSONTitles(self.nodataFields)

  def MovePermsToEnd(self):
# Put permissions at end of titles
    try:
      last = len(self.titlesList)
      start = end = self.titlesList.index('permissions')
      while end < last and self.titlesList[end].startswith('permissions'):
        end += 1
      self.titlesList = self.titlesList[:start]+self.titlesList[end:]+self.titlesList[start:end]
    except ValueError:
      pass

  def SetColumnDelimiter(self, columnDelimiter):
    self.columnDelimiter = columnDelimiter

  def SetNoEscapeChar(self, noEscapeChar):
    self.noEscapeChar = noEscapeChar

  def SetQuoteChar(self, quoteChar):
    self.quoteChar = quoteChar

  def SetTimestampColumn(self, timestampColumn):
    self.timestampColumn = timestampColumn
    if not GC.Values[GC.OUTPUT_TIMEFORMAT]:
      self.todaysTime = ISOformatTimeStamp(todaysTime())
    else:
      self.todaysTime = todaysTime().strftime(GC.Values[GC.OUTPUT_TIMEFORMAT])

  def SetSortHeaders(self, sortHeaders):
    self.sortHeaders = sortHeaders

  def SetFormatJSON(self, formatJSON):
    self.formatJSON = formatJSON

  def SetNodataFields(self, mapNodataFields, nodataFields, driveListFields, driveSubfieldsChoiceMap, oneItemPerRow):
    self.mapNodataFields = mapNodataFields
    self.nodataFields = nodataFields
    self.driveListFields = driveListFields
    self.driveSubfieldsChoiceMap = driveSubfieldsChoiceMap
    self.oneItemPerRow = oneItemPerRow

  def SetFixPaths(self, fixPaths):
    self.fixPaths = fixPaths

  def SetShowPermissionsLast(self, showPermissionsLast):
    self.showPermissionsLast = showPermissionsLast

  def FixCourseAliasesTitles(self):
# Put Aliases.* after Aliases
    try:
      aliasesIndex = self.sortTitlesList.index('Aliases')
      index = self.titlesList.index('Aliases.0')
      tempSortTitlesList = self.sortTitlesList[:]
      self.SetSortTitles(tempSortTitlesList[:aliasesIndex+1])
      while self.titlesList[index].startswith('Aliases.'):
        self.AddSortTitle(self.titlesList[index])
        index += 1
      self.AddSortTitles(tempSortTitlesList[aliasesIndex+1:])
    except ValueError:
      pass

  def RearrangeCourseTitles(self, ttitles, stitles):
# Put teachers and students after courseMaterialSets if present, otherwise at end
    for title in ttitles['list']:
      if title in self.titlesList:
        self.titlesList.remove(title)
    for title in stitles['list']:
      if title in self.titlesList:
        self.titlesList.remove(title)
    try:
      cmsIndex = self.titlesList.index('courseMaterialSets')
      self.titlesList = self.titlesList[:cmsIndex]+ttitles['list']+stitles['list']+self.titlesList[cmsIndex:]
    except ValueError:
      self.titlesList.extend(ttitles['list'])
      self.titlesList.extend(stitles['list'])

  def SortRows(self, title, reverse):
    if title in self.titlesSet:
      self.rows.sort(key=lambda k: k[title], reverse=reverse)

  def SortRowsTwoTitles(self, title1, title2, reverse):
    if title1 in self.titlesSet and title2 in self.titlesSet:
      self.rows.sort(key=lambda k: (k[title1], k[title2]), reverse=reverse)

  def SetRowFilter(self, rowFilter, rowFilterMode):
    self.rowFilter = rowFilter
    self.rowFilterMode = rowFilterMode

  def SetRowDropFilter(self, rowDropFilter, rowDropFilterMode):
    self.rowDropFilter = rowDropFilter
    self.rowDropFilterMode = rowDropFilterMode

  def SetRowLimit(self, rowLimit):
    self.rowLimit = rowLimit

  def AppendRow(self, row):
    if self.timestampColumn:
      row[self.timestampColumn] = self.todaysTime
    if not self.rowLimit or self.rowCount < self.rowLimit:
      self.rowCount +=1
      self.rows.append(row)

  def WriteRowNoFilter(self, row):
    self.AppendRow(row)

  def WriteRow(self, row):
    if RowFilterMatch(row, self.titlesList, self.rowFilter, self.rowFilterMode, self.rowDropFilter, self.rowDropFilterMode):
      self.AppendRow(row)

  def WriteRowTitles(self, row):
    for title in row:
      if title not in self.titlesSet:
        self.AddTitle(title)
    if RowFilterMatch(row, self.titlesList, self.rowFilter, self.rowFilterMode, self.rowDropFilter, self.rowDropFilterMode):
      self.AppendRow(row)

  def WriteRowTitlesNoFilter(self, row):
    for title in row:
      if title not in self.titlesSet:
        self.AddTitle(title)
    self.AppendRow(row)

  def WriteRowTitlesJSONNoFilter(self, row):
    for title in row:
      if title not in self.JSONtitlesSet:
        self.AddJSONTitle(title)
    self.AppendRow(row)

  def CheckRowTitles(self, row):
    if not self.rowFilter and not self.rowDropFilter:
      return True
    for title in row:
      if title not in self.titlesSet:
        self.AddTitle(title)
    return RowFilterMatch(row, self.titlesList, self.rowFilter, self.rowFilterMode, self.rowDropFilter, self.rowDropFilterMode)

  def UpdateMimeTypeCounts(self, row, mimeTypeInfo, sizeField):
    saveList = self.titlesList[:]
    saveSet = set(self.titlesSet)
    for title in row:
      if title not in self.titlesSet:
        self.AddTitle(title)
    if RowFilterMatch(row, self.titlesList, self.rowFilter, self.rowFilterMode, self.rowDropFilter, self.rowDropFilterMode):
      mimeTypeInfo.setdefault(row['mimeType'], {'count': 0, 'size': 0})
      mimeTypeInfo[row['mimeType']]['count'] += 1
      mimeTypeInfo[row['mimeType']]['size'] += int(row.get(sizeField, '0'))
    self.titlesList = saveList[:]
    self.titlesSet = set(saveSet)

  def SetZeroBlankMimeTypeCounts(self, zeroBlankMimeTypeCounts):
    self.zeroBlankMimeTypeCounts = zeroBlankMimeTypeCounts

  def ZeroBlankMimeTypeCounts(self):
    for row in self.rows:
      for title in self.titlesList:
        if title not in self.sortTitlesSet and title not in row:
          row[title] = 0

  def CheckOutputRowFilterHeaders(self):
    for filterVal in self.rowFilter:
      columns = [t for t in self.titlesList if filterVal[0].match(t)]
      if not columns:
        stderrWarningMsg(Msg.COLUMN_DOES_NOT_MATCH_ANY_OUTPUT_COLUMNS.format(GC.CSV_OUTPUT_ROW_FILTER, filterVal[0].pattern))
    for filterVal in self.rowDropFilter:
      columns = [t for t in self.titlesList if filterVal[0].match(t)]
      if not columns:
        stderrWarningMsg(Msg.COLUMN_DOES_NOT_MATCH_ANY_OUTPUT_COLUMNS.format(GC.CSV_OUTPUT_ROW_DROP_FILTER, filterVal[0].pattern))

  def SetHeaderFilter(self, headerFilter):
    self.headerFilter = headerFilter

  def SetHeaderDropFilter(self, headerDropFilter):
    self.headerDropFilter = headerDropFilter

  def SetHeaderForce(self, headerForce):
    self.headerForce = headerForce
    self.SetTitles(headerForce)
    self.SetJSONTitles(headerForce)

  def SetHeaderOrder(self, headerOrder):
    self.headerOrder = headerOrder

  def orderHeaders(self, titlesList):
    for title in self.headerOrder:
      if title in titlesList:
        titlesList.remove(title)
    return self.headerOrder+titlesList

  @staticmethod
  def HeaderFilterMatch(filters, title):
    for filterStr in filters:
      if filterStr.match(title):
        return True
    return False

  def FilterHeaders(self):
    if self.headerDropFilter:
      self.titlesList = [t for t in self.titlesList if not self.HeaderFilterMatch(self.headerDropFilter, t)]
    if self.headerFilter:
      self.titlesList = [t for t in self.titlesList if self.HeaderFilterMatch(self.headerFilter, t)]
    self.titlesSet = set(self.titlesList)
    if not self.titlesSet:
      systemErrorExit(USAGE_ERROR_RC, Msg.NO_COLUMNS_SELECTED_WITH_CSV_OUTPUT_HEADER_FILTER.format(GC.CSV_OUTPUT_HEADER_FILTER, GC.CSV_OUTPUT_HEADER_DROP_FILTER))

  def FilterJSONHeaders(self):
    if self.headerDropFilter:
      self.JSONtitlesList = [t for t in self.JSONtitlesList if not self.HeaderFilterMatch(self.headerDropFilter, t)]
    if self.headerFilter:
      self.JSONtitlesList = [t for t in self.JSONtitlesList if self.HeaderFilterMatch(self.headerFilter, t)]
    self.JSONtitlesSet = set(self.JSONtitlesList)
    if not self.JSONtitlesSet:
      systemErrorExit(USAGE_ERROR_RC, Msg.NO_COLUMNS_SELECTED_WITH_CSV_OUTPUT_HEADER_FILTER.format(GC.CSV_OUTPUT_HEADER_FILTER, GC.CSV_OUTPUT_HEADER_DROP_FILTER))

  def writeCSVfile(self, list_type, clearRowFilters=False):

    def todriveCSVErrorExit(entityValueList, errMsg):
      systemErrorExit(ACTION_FAILED_RC, formatKeyValueList(Ind.Spaces(),
                                                           Ent.FormatEntityValueList(entityValueList)+[Act.NotPerformed(), errMsg],
                                                           currentCountNL(0, 0)))

    @staticmethod
    def itemgetter(*items):
      if len(items) == 1:
        item = items[0]
        def g(obj):
          return obj.get(item, '')
      else:
        def g(obj):
          return tuple(obj.get(item, '') for item in items)
      return g

    def writeCSVData(writer):
      try:
        if not self.outputTranspose:
          if GM.Globals[GM.CSVFILE][GM.REDIRECT_WRITE_HEADER]:
            writer.writerow(dict((item, item) for item in writer.fieldnames))
          if not self.sortHeaders:
            writer.writerows(self.rows)
          else:
            for row in sorted(self.rows, key=itemgetter(*self.sortHeaders)):
              writer.writerow(row)
        else:
          writer.writerows(self.rows)
        return True
      except IOError as e:
        stderrErrorMsg(e)
        return False

    def setDialect(lineterminator, noEscapeChar):
      writerDialect = {
        'delimiter': self.columnDelimiter,
        'doublequote': True,
        'escapechar': '\\' if not noEscapeChar else None,
        'lineterminator': lineterminator,
        'quotechar': self.quoteChar,
        'quoting': csv.QUOTE_MINIMAL,
        'skipinitialspace': False,
        'strict': False}
      return writerDialect

    def normalizeSortHeaders():
      if self.sortHeaders:
        writerKeyMap = {}
        for k in titlesList:
          writerKeyMap[k.lower()] = k
        self.sortHeaders = [writerKeyMap[k.lower()] for k in self.sortHeaders if k.lower() in writerKeyMap]

    def writeCSVToStdout():
      csvFile = StringIOobject()
      writerDialect = setDialect('\n', self.noEscapeChar)
      writer = csv.DictWriter(csvFile, titlesList, extrasaction=extrasaction, **writerDialect)
      if writeCSVData(writer):
        try:
          GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD].write(csvFile.getvalue())
        except IOError as e:
          stderrErrorMsg(fdErrorMessage(GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD], 'stdout', e))
          setSysExitRC(FILE_ERROR_RC)
      closeFile(csvFile)

    def writeCSVToFile():
      csvFile = openFile(GM.Globals[GM.CSVFILE][GM.REDIRECT_NAME], GM.Globals[GM.CSVFILE][GM.REDIRECT_MODE], newline='',
                         encoding=GM.Globals[GM.CSVFILE][GM.REDIRECT_ENCODING], errors='backslashreplace',
                         continueOnError=True)
      if csvFile:
        writerDialect = setDialect(str(GC.Values[GC.CSV_OUTPUT_LINE_TERMINATOR]), self.noEscapeChar)
        writer = csv.DictWriter(csvFile, titlesList, extrasaction=extrasaction, **writerDialect)
        writeCSVData(writer)
        closeFile(csvFile)

    def writeCSVToDrive():
      numRows = len(self.rows)
      numColumns = len(titlesList)
      if numRows == 0 and not self.todrive['uploadnodata']:
        printKeyValueList([Msg.NO_CSV_DATA_TO_UPLOAD])
        setSysExitRC(NO_CSV_DATA_TO_UPLOAD_RC)
        return
      if self.todrive['addsheet'] or self.todrive['updatesheet']:
        csvFile = TemporaryFile(mode='w+', encoding=UTF8)
      else:
        csvFile = StringIOobject()
      writerDialect = setDialect('\n', self.todrive['noescapechar'])
      writer = csv.DictWriter(csvFile, titlesList, extrasaction=extrasaction, **writerDialect)
      if writeCSVData(writer):
        if ((self.todrive['title'] is None) or
             (not self.todrive['title'] and not self.todrive['timestamp'])):
          title = f'{GC.Values[GC.DOMAIN]} - {list_type}'
        else:
          title = self.todrive['title']
        if ((self.todrive['sheettitle'] is None) or
            (not self.todrive['sheettitle'] and not self.todrive['sheettimestamp'])):
          if ((self.todrive['sheetEntity'] is None) or
              (not self.todrive['sheetEntity']['sheetTitle'])):
            sheetTitle = title
          else:
            sheetTitle = self.todrive['sheetEntity']['sheetTitle']
        else:
          sheetTitle = self.todrive['sheettitle']
        tdbasetime = tdtime = datetime.datetime.now(GC.Values[GC.TIMEZONE])
        if self.todrive['daysoffset'] is not None or self.todrive['hoursoffset'] is not None:
          tdtime = tdbasetime+relativedelta(days=-self.todrive['daysoffset'] if self.todrive['daysoffset'] is not None else 0,
                                            hours=-self.todrive['hoursoffset'] if self.todrive['hoursoffset'] is not None else 0)
        if self.todrive['timestamp']:
          if title:
            title += ' - '
          if not self.todrive['timeformat']:
            title += ISOformatTimeStamp(tdtime)
          else:
            title += tdtime.strftime(self.todrive['timeformat'])
        if self.todrive['sheettimestamp']:
          if self.todrive['sheetdaysoffset'] is not None or self.todrive['sheethoursoffset'] is not None:
            tdtime = tdbasetime+relativedelta(days=-self.todrive['sheetdaysoffset'] if self.todrive['sheetdaysoffset'] is not None else 0,
                                              hours=-self.todrive['sheethoursoffset'] if self.todrive['sheethoursoffset'] is not None else 0)
          if sheetTitle:
            sheetTitle += ' - '
          if not self.todrive['sheettimeformat']:
            sheetTitle += ISOformatTimeStamp(tdtime)
          else:
            sheetTitle += tdtime.strftime(self.todrive['sheettimeformat'])
        action = Act.Get()
        if not GC.Values[GC.TODRIVE_CLIENTACCESS]:
          user, drive = buildGAPIServiceObject(API.DRIVETD, self.todrive['user'])
          if not drive:
            closeFile(csvFile)
            return
        else:
          user = self.todrive['user']
          drive = buildGAPIObject(API.DRIVE3)
        importSize = csvFile.tell()
# Add/Update sheet
        try:
          if self.todrive['addsheet'] or self.todrive['updatesheet']:
            Act.Set(Act.CREATE if self.todrive['addsheet'] else Act.UPDATE)
            result = callGAPI(drive.about(), 'get',
                              throwReasons=GAPI.DRIVE_USER_THROW_REASONS,
                              fields='maxImportSizes')
            if numRows*numColumns > MAX_GOOGLE_SHEET_CELLS or importSize > int(result['maxImportSizes'][MIMETYPE_GA_SPREADSHEET]):
              todriveCSVErrorExit([Ent.USER, user], Msg.RESULTS_TOO_LARGE_FOR_GOOGLE_SPREADSHEET)
            fields = ','.join(['id', 'mimeType', 'webViewLink', 'name', 'capabilities(canEdit)'])
            body = {'description': self.todrive['description']}
            if body['description'] is None:
              body['description'] = Cmd.QuotedArgumentList(Cmd.AllArguments())
            if not self.todrive['retaintitle']:
              body['name'] = title
            result = callGAPI(drive.files(), 'update',
                              throwReasons=GAPI.DRIVE_USER_THROW_REASONS+[GAPI.INSUFFICIENT_PERMISSIONS, GAPI.INSUFFICIENT_PARENT_PERMISSIONS,
                                                                          GAPI.FILE_NOT_FOUND, GAPI.UNKNOWN_ERROR],
                              fileId=self.todrive['fileId'], body=body, fields=fields, supportsAllDrives=True)
            entityValueList = [Ent.USER, user, Ent.DRIVE_FILE_ID, self.todrive['fileId']]
            if not result['capabilities']['canEdit']:
              todriveCSVErrorExit(entityValueList, Msg.NOT_WRITABLE)
            if result['mimeType'] != MIMETYPE_GA_SPREADSHEET:
              todriveCSVErrorExit(entityValueList, f'{Msg.NOT_A} {Ent.Singular(Ent.SPREADSHEET)}')
            if not GC.Values[GC.TODRIVE_CLIENTACCESS]:
              _, sheet = buildGAPIServiceObject(API.SHEETSTD, user)
              if sheet is None:
                return
            else:
              sheet = buildGAPIObject(API.SHEETS)
            csvFile.seek(0)
            spreadsheet = None
            if self.todrive['updatesheet']:
              for sheetEntity in self.TDSHEET_ENTITY_MAP.values():
                if self.todrive[sheetEntity]:
                  entityValueList = [Ent.USER, user, Ent.SPREADSHEET, title, self.todrive[sheetEntity]['sheetType'], self.todrive[sheetEntity]['sheetValue']]
                  if spreadsheet is None:
                    spreadsheet = callGAPI(sheet.spreadsheets(), 'get',
                                           throwReasons=GAPI.SHEETS_ACCESS_THROW_REASONS,
                                           spreadsheetId=self.todrive['fileId'],
                                           fields='spreadsheetUrl,sheets(properties(sheetId,title),protectedRanges(range(sheetId),requestingUserCanEdit))')
                  sheetId = getSheetIdFromSheetEntity(spreadsheet, self.todrive[sheetEntity])
                  if sheetId is None:
                    if ((sheetEntity != 'sheetEntity') or (self.todrive[sheetEntity]['sheetType'] == Ent.SHEET_ID)):
                      todriveCSVErrorExit(entityValueList, Msg.NOT_FOUND)
                    self.todrive['addsheet'] = True
                  else:
                    if protectedSheetId(spreadsheet, sheetId):
                      todriveCSVErrorExit(entityValueList, Msg.NOT_WRITABLE)
                    self.todrive[sheetEntity]['sheetId'] = sheetId
            if self.todrive['addsheet']:
              body = {'requests': [{'addSheet': {'properties': {'title': sheetTitle, 'sheetType': 'GRID'}}}]}
              try:
                addresult = callGAPI(sheet.spreadsheets(), 'batchUpdate',
                                     throwReasons=GAPI.SHEETS_ACCESS_THROW_REASONS,
                                     spreadsheetId=self.todrive['fileId'], body=body)
                self.todrive['sheetEntity'] = {'sheetId': addresult['replies'][0]['addSheet']['properties']['sheetId']}
              except (GAPI.notFound, GAPI.forbidden, GAPI.permissionDenied,
                      GAPI.internalError, GAPI.insufficientFilePermissions, GAPI.insufficientParentPermissions, GAPI.badRequest,
                      GAPI.invalid, GAPI.invalidArgument, GAPI.failedPrecondition) as e:
                todriveCSVErrorExit(entityValueList, str(e))
            body = {'requests': []}
            if not self.todrive['addsheet']:
              if self.todrive['backupSheetEntity']:
                body['requests'].append({'copyPaste': {'source': {'sheetId': self.todrive['sheetEntity']['sheetId']},
                                                       'destination': {'sheetId': self.todrive['backupSheetEntity']['sheetId']}, 'pasteType': 'PASTE_NORMAL'}})
              if self.todrive['clearfilter']:
                body['requests'].append({'clearBasicFilter': {'sheetId': self.todrive['sheetEntity']['sheetId']}})
              if self.todrive['sheettitle']:
                body['requests'].append({'updateSheetProperties':
                                           {'properties': {'sheetId': self.todrive['sheetEntity']['sheetId'], 'title': sheetTitle}, 'fields': 'title'}})
            body['requests'].append({'updateCells': {'range': {'sheetId': self.todrive['sheetEntity']['sheetId']}, 'fields': '*'}})
            if self.todrive['cellwrap']:
              body['requests'].append({'repeatCell': {'range': {'sheetId': self.todrive['sheetEntity']['sheetId']},
                                                      'fields': 'userEnteredFormat.wrapStrategy',
                                                      'cell': {'userEnteredFormat': {'wrapStrategy': self.todrive['cellwrap']}}}})
            if self.todrive['cellnumberformat']:
              body['requests'].append({'repeatCell': {'range': {'sheetId': self.todrive['sheetEntity']['sheetId']},
                                                      'fields': 'userEnteredFormat.numberFormat',
                                                      'cell': {'userEnteredFormat': {'numberFormat': {'type': self.todrive['cellnumberformat']}}}}})
            body['requests'].append({'pasteData': {'coordinate': {'sheetId': self.todrive['sheetEntity']['sheetId'], 'rowIndex': '0', 'columnIndex': '0'},
                                                   'data': csvFile.read(), 'type': 'PASTE_NORMAL', 'delimiter': self.columnDelimiter}})
            if self.todrive['copySheetEntity']:
              body['requests'].append({'copyPaste': {'source': {'sheetId': self.todrive['sheetEntity']['sheetId']},
                                                     'destination': {'sheetId': self.todrive['copySheetEntity']['sheetId']}, 'pasteType': 'PASTE_NORMAL'}})
            try:
              callGAPI(sheet.spreadsheets(), 'batchUpdate',
                       throwReasons=GAPI.SHEETS_ACCESS_THROW_REASONS,
                       spreadsheetId=self.todrive['fileId'], body=body)
            except (GAPI.notFound, GAPI.forbidden, GAPI.permissionDenied,
                    GAPI.internalError, GAPI.insufficientFilePermissions, GAPI.badRequest,
                    GAPI.invalid, GAPI.invalidArgument, GAPI.failedPrecondition) as e:
              todriveCSVErrorExit(entityValueList, str(e))
            closeFile(csvFile)
# Create/update file
          else:
            if GC.Values[GC.TODRIVE_CONVERSION]:
              result = callGAPI(drive.about(), 'get',
                                throwReasons=GAPI.DRIVE_USER_THROW_REASONS,
                                fields='maxImportSizes')
              if numRows*len(titlesList) > MAX_GOOGLE_SHEET_CELLS or importSize > int(result['maxImportSizes'][MIMETYPE_GA_SPREADSHEET]):
                printKeyValueList([WARNING, Msg.RESULTS_TOO_LARGE_FOR_GOOGLE_SPREADSHEET])
                mimeType = 'text/csv'
              else:
                mimeType = MIMETYPE_GA_SPREADSHEET
            else:
              mimeType = 'text/csv'
            fields = ','.join(['id', 'mimeType', 'webViewLink'])
            body = {'description': self.todrive['description'], 'mimeType': mimeType}
            if body['description'] is None:
              body['description'] = Cmd.QuotedArgumentList(Cmd.AllArguments())
            if not self.todrive['fileId'] or not self.todrive['retaintitle']:
              body['name'] = title
            try:
              if not self.todrive['fileId']:
                Act.Set(Act.CREATE)
                body['parents'] = [self.todrive['parentId']]
                result = callGAPI(drive.files(), 'create',
                                  bailOnInternalError=True,
                                  throwReasons=GAPI.DRIVE_USER_THROW_REASONS+[GAPI.FORBIDDEN, GAPI.INSUFFICIENT_PERMISSIONS, GAPI.INSUFFICIENT_PARENT_PERMISSIONS,
                                                                              GAPI.FILE_NOT_FOUND, GAPI.UNKNOWN_ERROR, GAPI.INTERNAL_ERROR, GAPI.STORAGE_QUOTA_EXCEEDED,
                                                                              GAPI.TEAMDRIVE_FILE_LIMIT_EXCEEDED, GAPI.TEAMDRIVE_HIERARCHY_TOO_DEEP],
                                  body=body,
                                  media_body=googleapiclient.http.MediaIoBaseUpload(io.BytesIO(csvFile.getvalue().encode()), mimetype='text/csv', resumable=True),
                                  fields=fields, supportsAllDrives=True)
              else:
                Act.Set(Act.UPDATE)
                result = callGAPI(drive.files(), 'update',
                                  bailOnInternalError=True,
                                  throwReasons=GAPI.DRIVE_USER_THROW_REASONS+[GAPI.INSUFFICIENT_PERMISSIONS, GAPI.INSUFFICIENT_PARENT_PERMISSIONS,
                                                                              GAPI.FILE_NOT_FOUND, GAPI.UNKNOWN_ERROR, GAPI.INTERNAL_ERROR],
                                  fileId=self.todrive['fileId'],
                                  body=body,
                                  media_body=googleapiclient.http.MediaIoBaseUpload(io.BytesIO(csvFile.getvalue().encode()), mimetype='text/csv', resumable=True),
                                  fields=fields, supportsAllDrives=True)
              spreadsheetId = result['id']
            except GAPI.internalError as e:
              entityActionFailedWarning([Ent.DRIVE_FILE, body['name']], Msg.UPLOAD_CSV_FILE_INTERNAL_ERROR.format(str(e), str(numRows)))
              closeFile(csvFile)
              return
            closeFile(csvFile)
            if not self.todrive['fileId'] and self.todrive['share']:
              Act.Set(Act.SHARE)
              for share in self.todrive['share']:
                if share['emailAddress'] != user:
                  try:
                    callGAPI(drive.permissions(), 'create',
                             bailOnInternalError=True,
                             throwReasons=GAPI.DRIVE_ACCESS_THROW_REASONS+GAPI.DRIVE3_CREATE_ACL_THROW_REASONS,
                             fileId=spreadsheetId, sendNotificationEmail=False, body=share, fields='', supportsAllDrives=True)
                    entityActionPerformed([Ent.USER, user, Ent.SPREADSHEET, title,
                                           Ent.TARGET_USER, share['emailAddress'], Ent.ROLE, share['role']])
                  except (GAPI.badRequest, GAPI.invalid, GAPI.fileNotFound, GAPI.forbidden, GAPI.internalError,
                          GAPI.insufficientFilePermissions, GAPI.insufficientParentPermissions, GAPI.unknownError, GAPI.ownershipChangeAcrossDomainNotPermitted,
                          GAPI.teamDriveDomainUsersOnlyRestriction, GAPI.teamDriveTeamMembersOnlyRestriction,
                          GAPI.targetUserRoleLimitedByLicenseRestriction, GAPI.insufficientAdministratorPrivileges, GAPI.sharingRateLimitExceeded,
                          GAPI.publishOutNotPermitted, GAPI.shareInNotPermitted, GAPI.shareOutNotPermitted, GAPI.shareOutNotPermittedToUser,
                          GAPI.cannotShareTeamDriveTopFolderWithAnyoneOrDomains, GAPI.cannotShareTeamDriveWithNonGoogleAccounts,
                          GAPI.ownerOnTeamDriveItemNotSupported,
                          GAPI.organizerOnNonTeamDriveNotSupported, GAPI.organizerOnNonTeamDriveItemNotSupported,
                          GAPI.fileOrganizerNotYetEnabledForThisTeamDrive,
                          GAPI.fileOrganizerOnFoldersInSharedDriveOnly,
                          GAPI.fileOrganizerOnNonTeamDriveNotSupported,
                          GAPI.cannotModifyInheritedPermission,
                          GAPI.teamDrivesFolderSharingNotSupported, GAPI.invalidLinkVisibility,
                          GAPI.invalidSharingRequest, GAPI.fileNeverWritable, GAPI.abusiveContentRestriction) as e:
                    entityActionFailedWarning([Ent.USER, user, Ent.SPREADSHEET, title,
                                               Ent.TARGET_USER, share['emailAddress'], Ent.ROLE, share['role']],
                                              str(e))
            if ((result['mimeType'] == MIMETYPE_GA_SPREADSHEET) and
                (self.todrive['sheetEntity'] or self.todrive['locale'] or self.todrive['timeZone'] or
                 self.todrive['sheettitle'] or self.todrive['cellwrap'] or self.todrive['cellnumberformat'])):
              if not GC.Values[GC.TODRIVE_CLIENTACCESS]:
                _, sheet = buildGAPIServiceObject(API.SHEETSTD, user)
                if sheet is None:
                  return
              else:
                sheet = buildGAPIObject(API.SHEETS)
              try:
                body = {'requests': []}
                if self.todrive['sheetEntity'] or self.todrive['sheettitle'] or self.todrive['cellwrap']:
                  spreadsheet = callGAPI(sheet.spreadsheets(), 'get',
                                         throwReasons=GAPI.SHEETS_ACCESS_THROW_REASONS,
                                         spreadsheetId=spreadsheetId, fields='sheets/properties')
                  spreadsheet['sheets'][0]['properties']['title'] = sheetTitle
                  body['requests'].append({'updateSheetProperties':
                                           {'properties': spreadsheet['sheets'][0]['properties'], 'fields': 'title'}})
                  if self.todrive['cellwrap']:
                    body['requests'].append({'repeatCell': {'range': {'sheetId': spreadsheet['sheets'][0]['properties']['sheetId']},
                                                            'fields': 'userEnteredFormat.wrapStrategy',
                                                            'cell': {'userEnteredFormat': {'wrapStrategy': self.todrive['cellwrap']}}}})
                if self.todrive['locale']:
                  body['requests'].append({'updateSpreadsheetProperties':
                                             {'properties': {'locale': self.todrive['locale']}, 'fields': 'locale'}})
                if self.todrive['timeZone']:
                  body['requests'].append({'updateSpreadsheetProperties':
                                             {'properties': {'timeZone': self.todrive['timeZone']}, 'fields': 'timeZone'}})
                if body['requests']:
                  callGAPI(sheet.spreadsheets(), 'batchUpdate',
                           throwReasons=GAPI.SHEETS_ACCESS_THROW_REASONS,
                           spreadsheetId=spreadsheetId, body=body)
              except (GAPI.notFound, GAPI.forbidden, GAPI.permissionDenied,
                      GAPI.internalError, GAPI.insufficientFilePermissions, GAPI.badRequest,
                      GAPI.invalid, GAPI.invalidArgument, GAPI.failedPrecondition,
                      GAPI.teamDriveFileLimitExceeded, GAPI.teamDriveHierarchyTooDeep) as e:
                todriveCSVErrorExit([Ent.USER, user, Ent.SPREADSHEET, title], str(e))
          Act.Set(action)
          file_url = result['webViewLink']
          msg_txt = f'{Msg.DATA_UPLOADED_TO_DRIVE_FILE}:\n{file_url}'
          if not self.todrive['returnidonly']:
            printKeyValueList([msg_txt])
          else:
            if self.todrive['fileId']:
              writeStdout(f'{self.todrive["fileId"]}\n')
            else:
              writeStdout(f'{spreadsheetId}\n')
          if not self.todrive['subject']:
            subject = title
          else:
            subject = self.todrive['subject'].replace('#file#', title).replace('#sheet#', sheetTitle)
          if not self.todrive['noemail']:
            send_email(subject, msg_txt, user, clientAccess=GC.Values[GC.TODRIVE_CLIENTACCESS], msgFrom=self.todrive['from'])
          if self.todrive['notify']:
            for recipient in self.todrive['share']+self.todrive['alert']:
              if recipient['emailAddress'] != user:
                send_email(subject, msg_txt, recipient['emailAddress'], clientAccess=GC.Values[GC.TODRIVE_CLIENTACCESS], msgFrom=self.todrive['from'])
          if not self.todrive['nobrowser']:
            webbrowser.open(file_url)
        except (GAPI.forbidden, GAPI.insufficientPermissions):
          printWarningMessage(INSUFFICIENT_PERMISSIONS_RC, Msg.INSUFFICIENT_PERMISSIONS_TO_PERFORM_TASK)
        except (GAPI.fileNotFound, GAPI.unknownError, GAPI.internalError, GAPI.storageQuotaExceeded) as e:
          if not self.todrive['fileId']:
            entityActionFailedWarning([Ent.DRIVE_FOLDER, self.todrive['parentId']], str(e))
          else:
            entityActionFailedWarning([Ent.DRIVE_FILE, self.todrive['fileId']], str(e))
        except (GAPI.serviceNotAvailable, GAPI.authError, GAPI.domainPolicy) as e:
          userDriveServiceNotEnabledWarning(user, str(e), 0, 0)
      else:
        closeFile(csvFile)

    if GM.Globals[GM.CSVFILE][GM.REDIRECT_QUEUE] is not None:
      GM.Globals[GM.CSVFILE][GM.REDIRECT_QUEUE].put((GM.REDIRECT_QUEUE_NAME, list_type))
      GM.Globals[GM.CSVFILE][GM.REDIRECT_QUEUE].put((GM.REDIRECT_QUEUE_TODRIVE, self.todrive))
      GM.Globals[GM.CSVFILE][GM.REDIRECT_QUEUE].put((GM.REDIRECT_QUEUE_CSVPF,
                                                     (self.titlesList, self.sortTitlesList, self.indexedTitles,
                                                      self.formatJSON, self.JSONtitlesList,
                                                      self.columnDelimiter, self.noEscapeChar, self.quoteChar,
                                                      self.sortHeaders, self.timestampColumn,
                                                      self.mapDrive3Titles,
                                                      self.fixPaths,
                                                      self.mapNodataFields,
                                                      self.nodataFields,
                                                      self.driveListFields,
                                                      self.driveSubfieldsChoiceMap,
                                                      self.oneItemPerRow,
                                                      self.showPermissionsLast,
                                                      self.zeroBlankMimeTypeCounts)))
      if clearRowFilters:
        GM.Globals[GM.CSVFILE][GM.REDIRECT_QUEUE].put((GM.REDIRECT_QUEUE_CLEAR_ROW_FILTERS, clearRowFilters))
      GM.Globals[GM.CSVFILE][GM.REDIRECT_QUEUE].put((GM.REDIRECT_QUEUE_DATA, self.rows))
      return
    if self.zeroBlankMimeTypeCounts:
      self.ZeroBlankMimeTypeCounts()
    if not clearRowFilters and (self.rowFilter or self.rowDropFilter):
      self.CheckOutputRowFilterHeaders()
    if self.headerFilter or self.headerDropFilter:
      if not self.formatJSON:
        self.FilterHeaders()
      else:
        self.FilterJSONHeaders()
      extrasaction = 'ignore'
    else:
      extrasaction = 'raise'
    if not self.formatJSON:
      if not self.headerForce:
        self.SortTitles()
        self.SortIndexedTitles(self.titlesList)
        if self.fixPaths:
          self.FixPathsTitles(self.titlesList)
        if self.showPermissionsLast:
          self.MovePermsToEnd()
        if not self.rows and self.nodataFields is not None:
          self.FixNodataTitles()
        if self.mapDrive3Titles:
          self. MapDrive3TitlesToDrive2()
      else:
        self.titlesList = self.headerForce
      if self.timestampColumn:
        self.AddTitle(self.timestampColumn)
      if self.headerOrder:
        self.titlesList = self.orderHeaders(self.titlesList)
      titlesList = self.titlesList
    else:
      if not self.headerForce:
        if self.fixPaths:
          self.FixPathsTitles(self.JSONtitlesList)
        if not self.rows and self.nodataFields is not None:
          self.FixNodataTitles()
      else:
        self.JSONtitlesList = self.headerForce
      if self.timestampColumn:
        for i, v in enumerate(self.JSONtitlesList):
          if v.startswith('JSON'):
            self.JSONtitlesList.insert(i, self.timestampColumn)
            self.JSONtitlesSet.add(self.timestampColumn)
            break
        else:
          self.AddJSONTitle(self.timestampColumn)
      if self.headerOrder:
        self.JSONtitlesList = self.orderHeaders(self.JSONtitlesList)
      titlesList = self.JSONtitlesList
    normalizeSortHeaders()
    if self.outputTranspose:
      newRows = []
      newTitlesList = list(range(len(self.rows) + 1))
      for title in titlesList:
        i = 0
        newRow = {i: title}
        for row in self.rows:
          i += 1
          newRow[i] = row.get(title)
        newRows.append(newRow)
      titlesList = newTitlesList
      self.rows = newRows
    if (not self.todrive) or self.todrive['localcopy']:
      if GM.Globals[GM.CSVFILE][GM.REDIRECT_NAME] == '-':
        if GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD]:
          writeCSVToStdout()
        else:
          GM.Globals[GM.CSVFILE][GM.REDIRECT_NAME] = GM.Globals[GM.STDOUT][GM.REDIRECT_NAME]
          writeCSVToFile()
      else:
        writeCSVToFile()
    if self.todrive:
      writeCSVToDrive()
    if GM.Globals[GM.CSVFILE][GM.REDIRECT_MODE] == DEFAULT_FILE_APPEND_MODE:
      GM.Globals[GM.CSVFILE][GM.REDIRECT_WRITE_HEADER] = False

def writeEntityNoHeaderCSVFile(entityType, entityList):
  csvPF = CSVPrintFile(entityType)
  _, _, entityList = getEntityArgument(entityList)
  if entityType == Ent.USER:
    for entity in entityList:
      csvPF.WriteRowNoFilter({entityType: normalizeEmailAddressOrUID(entity)})
  else:
    for entity in entityList:
      csvPF.WriteRowNoFilter({entityType: entity})
  GM.Globals[GM.CSVFILE][GM.REDIRECT_WRITE_HEADER] = False
  csvPF.writeCSVfile(Ent.Plural(entityType))

def getTodriveOnly(csvPF):
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if csvPF and myarg == 'todrive':
      csvPF.GetTodriveParameters()
    else:
      unknownArgumentExit()

DEFAULT_SKIP_OBJECTS = {'kind', 'etag', 'etags', '@type'}

# Clean a JSON object
def cleanJSON(topStructure, listLimit=None, skipObjects=None, timeObjects=None):
  def _clean(structure, key, subSkipObjects):
    if not isinstance(structure, (dict, list)):
      if key not in timeObjects:
        if isinstance(structure, str) and GC.Values[GC.CSV_OUTPUT_CONVERT_CR_NL]:
          return escapeCRsNLs(structure)
        return structure
      if isinstance(structure, str) and not structure.isdigit():
        return formatLocalTime(structure)
      return formatLocalTimestamp(structure)
    if isinstance(structure, list):
      listLen = len(structure)
      listLen = min(listLen, listLimit or listLen)
      return [_clean(v, '', DEFAULT_SKIP_OBJECTS) for v in structure[0:listLen]]
    return {k: _clean(v, k, DEFAULT_SKIP_OBJECTS) for k, v in sorted(structure.items()) if k not in subSkipObjects}

  timeObjects = timeObjects or set()
  return _clean(topStructure, '', DEFAULT_SKIP_OBJECTS.union(skipObjects or set()))

# Flatten a JSON object
def flattenJSON(topStructure, flattened=None,
                listLimit=None, skipObjects=None, timeObjects=None, noLenObjects=None, simpleLists=None, delimiter=None):
  def _flatten(structure, key, path):
    if not isinstance(structure, (dict, list)):
      if key not in timeObjects:
        if isinstance(structure, str):
          if GC.Values[GC.CSV_OUTPUT_CONVERT_CR_NL] and (structure.find('\n') >= 0 or structure.find('\r') >= 0):
            flattened[path] = escapeCRsNLs(structure)
          else:
            flattened[path] = structure
        else:
          flattened[path] = structure
      else:
        if isinstance(structure, str) and not structure.isdigit():
          flattened[path] = formatLocalTime(structure)
        else:
          flattened[path] = formatLocalTimestamp(structure)
    elif isinstance(structure, list):
      listLen = len(structure)
      listLen = min(listLen, listLimit or listLen)
      if key in simpleLists:
        flattened[path] = delimiter.join(structure[:listLen])
      else:
        if key not in noLenObjects:
          flattened[path] = listLen
        for i in range(listLen):
          _flatten(structure[i], '', f'{path}{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{i}')
    else:
      if structure:
        for k, v in sorted(structure.items()):
          if k not in DEFAULT_SKIP_OBJECTS:
            _flatten(v, k, f'{path}{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{k}')
      else:
        flattened[path] = ''

  flattened = flattened or {}
  allSkipObjects = DEFAULT_SKIP_OBJECTS.union(skipObjects or set())
  timeObjects = timeObjects or set()
  noLenObjects = noLenObjects or set()
  simpleLists = simpleLists or set()
  for k, v in sorted(topStructure.items()):
    if k not in allSkipObjects:
      _flatten(v, k, k)
  return flattened

# Show a json object
def showJSON(showName, showValue, skipObjects=None, timeObjects=None,
             simpleLists=None, dictObjectsKey=None, sortDictKeys=True):
  def _show(objectName, objectValue, subObjectKey, level, subSkipObjects):
    if objectName in subSkipObjects:
      return
    if objectName is not None:
      printJSONKey(objectName)
      subObjectKey = dictObjectsKey.get(objectName)
    if isinstance(objectValue, list):
      if objectName in simpleLists:
        printJSONValue(' '.join(objectValue))
        return
      if len(objectValue) == 1 and isinstance(objectValue[0], (str, bool, float, int)):
        if objectName is not None:
          printJSONValue(objectValue[0])
        else:
          printKeyValueList([objectValue[0]])
        return
      if objectName is not None:
        printBlankLine()
        Ind.Increment()
      for subValue in objectValue:
        if isinstance(subValue, (str, bool, float, int)):
          printKeyValueList([subValue])
        else:
          _show(None, subValue, subObjectKey, level+1, DEFAULT_SKIP_OBJECTS)
      if objectName is not None:
        Ind.Decrement()
    elif isinstance(objectValue, dict):
      indentAfterFirst = unindentAfterLast = False
      if objectName is not None:
        printBlankLine()
        Ind.Increment()
      elif level > 0:
        indentAfterFirst = unindentAfterLast = True
      subObjects = sorted(objectValue) if sortDictKeys else objectValue.keys()
      if subObjectKey and (subObjectKey in subObjects):
        subObjects.remove(subObjectKey)
        subObjects.insert(0, subObjectKey)
        subObjectKey = None
      for subObject in subObjects:
        if subObject not in subSkipObjects:
          _show(subObject, objectValue[subObject], subObjectKey, level+1, DEFAULT_SKIP_OBJECTS)
          if indentAfterFirst:
            Ind.Increment()
            indentAfterFirst = False
      if objectName is not None or ((not indentAfterFirst) and unindentAfterLast):
        Ind.Decrement()
    else:
      if objectName not in timeObjects:
        if isinstance(objectValue, str) and (objectValue.find('\n') >= 0 or objectValue.find('\r') >= 0):
          if GC.Values[GC.SHOW_CONVERT_CR_NL]:
            printJSONValue(escapeCRsNLs(objectValue))
          else:
            printBlankLine()
            Ind.Increment()
            printKeyValueList([Ind.MultiLineText(objectValue)])
            Ind.Decrement()
        else:
          printJSONValue(objectValue if objectValue is not None else '')
      else:
        if isinstance(objectValue, str) and not objectValue.isdigit():
          printJSONValue(formatLocalTime(objectValue))
        else:
          printJSONValue(formatLocalTimestamp(objectValue))

  timeObjects = timeObjects or set()
  simpleLists = simpleLists or set()
  dictObjectsKey = dictObjectsKey or {}
  _show(showName, showValue, None, 0, DEFAULT_SKIP_OBJECTS.union(skipObjects or set()))

class FormatJSONQuoteChar():

  def __init__(self, csvPF=None, formatJSONOnly=False):
    self.SetCsvPF(csvPF)
    self.SetFormatJSON(False)
    self.SetQuoteChar(GM.Globals.get(GM.CSV_OUTPUT_QUOTE_CHAR, GC.Values.get(GC.CSV_OUTPUT_QUOTE_CHAR, '"')))
    if not formatJSONOnly:
      return
    while Cmd.ArgumentsRemaining():
      myarg = getArgument()
      if myarg == 'formatjson':
        self.SetFormatJSON(True)
        return
      unknownArgumentExit()

  def SetCsvPF(self, csvPF):
    self.csvPF = csvPF

  def SetFormatJSON(self, formatJSON):
    self.formatJSON = formatJSON
    if self.csvPF:
      self.csvPF.SetFormatJSON(formatJSON)

  def GetFormatJSON(self, myarg):
    if myarg == 'formatjson':
      self.SetFormatJSON(True)
      return
    unknownArgumentExit()

  def SetQuoteChar(self, quoteChar):
    self.quoteChar = quoteChar
    if self.csvPF:
      self.csvPF.SetQuoteChar(quoteChar)

  def GetQuoteChar(self, myarg):
    if self.csvPF and myarg == 'quotechar':
      self.SetQuoteChar(getCharacter())
      return
    unknownArgumentExit()

  def GetFormatJSONQuoteChar(self, myarg, addTitle=False, noExit=False):
    if myarg == 'formatjson':
      self.SetFormatJSON(True)
      if self.csvPF and addTitle:
        self.csvPF.AddJSONTitles('JSON')
      return True
    if self.csvPF and myarg == 'quotechar':
      self.SetQuoteChar(getCharacter())
      return True
    if noExit:
      return False
    unknownArgumentExit()

# Batch processing request_id fields
RI_ENTITY = 0
RI_I = 1
RI_COUNT = 2
RI_J = 3
RI_JCOUNT = 4
RI_ITEM = 5
RI_ROLE = 6
RI_OPTION = 7

def batchRequestID(entityName, i, count, j, jcount, item, role=None, option=None):
  if role is None and option is None:
    return f'{entityName}\n{i}\n{count}\n{j}\n{jcount}\n{item}'
  return f'{entityName}\n{i}\n{count}\n{j}\n{jcount}\n{item}\n{role}\n{option}'

TIME_OFFSET_UNITS = [('day', SECONDS_PER_DAY), ('hour', SECONDS_PER_HOUR), ('minute', SECONDS_PER_MINUTE), ('second', 1)]

def getLocalGoogleTimeOffset(testLocation=GOOGLE_TIMECHECK_LOCATION):
  # If local time is well off, it breaks https because the server certificate will be seen as too old or new and thus invalid; http doesn't have that issue.
  # Try with http first, if time is close (<MAX_LOCAL_GOOGLE_TIME_OFFSET seconds), retry with https as it should be OK
  httpObj = getHttpObj()
  for prot in ['http', 'https']:
    try:
      headerData = httpObj.request(f'{prot}://'+testLocation, 'HEAD')
      googleUTC = datetime.datetime.strptime(headerData[0]['date'], '%a, %d %b %Y %H:%M:%S %Z').replace(tzinfo=iso8601.UTC)
    except (httplib2.HttpLib2Error, RuntimeError) as e:
      handleServerError(e)
    except httplib2.socks.HTTPError as e:
      # If user has specified an HTTPS proxy, the http request will probably fail as httplib2
      # turns a GET into a CONNECT which is not valid for an http address
      if prot == 'http':
        continue
      handleServerError(e)
    except (ValueError, KeyError):
      if prot == 'http':
        continue
      systemErrorExit(NETWORK_ERROR_RC, Msg.INVALID_HTTP_HEADER.format(str(headerData)))
    offset = remainder = int(abs((datetime.datetime.now(iso8601.UTC)-googleUTC).total_seconds()))
    if offset < MAX_LOCAL_GOOGLE_TIME_OFFSET and prot == 'http':
      continue
    timeoff = []
    for tou in TIME_OFFSET_UNITS:
      uval, remainder = divmod(remainder, tou[1])
      if uval:
        timeoff.append(f'{uval} {tou[0]}{"s" if uval != 1 else ""}')
    if not timeoff:
      timeoff.append(Msg.LESS_THAN_1_SECOND)
    nicetime = ', '.join(timeoff)
    return (offset, nicetime)

def _getServerTLSUsed(location):
  url = 'https://'+location
  _, netloc, _, _, _, _ = urlparse(url)
  conn = 'https:'+netloc
  httpObj = getHttpObj()
  triesLimit = 5
  for n in range(1, triesLimit+1):
    try:
      httpObj.request(url, headers={'user-agent': GAM_USER_AGENT})
      cipher_name, tls_ver, _ = httpObj.connections[conn].sock.cipher()
      return tls_ver, cipher_name
    except (httplib2.HttpLib2Error, RuntimeError) as e:
      if n != triesLimit:
        httpObj.connections = {}
        waitOnFailure(n, triesLimit, NETWORK_ERROR_RC, str(e))
        continue
      handleServerError(e)

MACOS_CODENAMES = {
  10: {
    6:  'Snow Leopard',
    7:  'Lion',
    8:  'Mountain Lion',
    9:  'Mavericks',
    10: 'Yosemite',
    11: 'El Capitan',
    12: 'Sierra',
    13: 'High Sierra',
    14: 'Mojave',
    15: 'Catalina',
    16: 'Big Sur'
    },
  11: 'Big Sur',
  12: 'Monterey',
  13: 'Ventura',
  14: 'Sonoma',
  15: 'Sequoia',
  }

def getOSPlatform():
  myos = platform.system()
  if myos == 'Linux':
    pltfrm = ' '.join(distro.linux_distribution(full_distribution_name=False)).title()
  elif myos == 'Windows':
    pltfrm = ' '.join(platform.win32_ver())
  elif myos == 'Darwin':
    myos = 'MacOS'
    mac_ver = platform.mac_ver()[0]
    major_ver = int(mac_ver.split('.')[0]) # macver 10.14.6 == major_ver 10
    minor_ver = int(mac_ver.split('.')[1]) # macver 10.14.6 == minor_ver 14
    if major_ver == 10:
      codename = MACOS_CODENAMES[major_ver].get(minor_ver, '')
    else:
      codename = MACOS_CODENAMES.get(major_ver, '')
    pltfrm = ' '.join([codename, mac_ver])
  else:
    pltfrm = platform.platform()
  return f'{myos} {pltfrm}'

# gam checkconnection
def doCheckConnection():

  def check_host(host):
    nonlocal try_count, okay, not_okay, success_count
    try_count += 1
    dns_err = None
    ip = 'unknown'
    try:
      ip = socket.getaddrinfo(host, None)[0][-1][0] # works with ipv6
    except socket.gaierror as e:
      dns_err = f'{not_okay}\n   DNS failure: {str(e)}\n'
    except Exception as e:
      dns_err = f'{not_okay}\n   Unknown DNS failure: {str(e)}\n'
    check_line = f'Checking {host} ({ip}) ({try_count})...'
    writeStdout(f'{check_line:<100}')
    flushStdout()
    if dns_err:
      writeStdout(dns_err)
      return
    gen_firewall = 'You probably have security software or a firewall on your machine or network that is preventing GAM from making Internet connections. Check your network configuration or try running GAM on a hotspot or home network to see if the problem exists only on your organization\'s network.'
    try:
      if host.startswith('http'):
        url = host
      else:
        url = f'https://{host}:443/'
      httpObj.request(url, 'HEAD', headers=headers)
      success_count += 1
      writeStdout(f'{okay}\n')
    except ConnectionRefusedError:
      writeStdout(f'{not_okay}\n    Connection refused. {gen_firewall}\n')
    except ConnectionResetError:
      writeStdout(f'{not_okay}\n    Connection reset by peer. {gen_firewall}\n')
    except httplib2.error.ServerNotFoundError:
      writeStdout(f'{not_okay}\n    Failed to find server. Your DNS is probably misconfigured.\n')
    except ssl.SSLError as e:
      if e.reason == 'SSLV3_ALERT_HANDSHAKE_FAILURE':
        writeStdout(f'{not_okay}\n    GAM expects to connect with TLS 1.3 or newer and that failed. If your firewall / proxy server is not compatible with TLS 1.3 then you can tell GAM to allow TLS 1.2 by setting tls_min_version = TLSv1.2 in gam.cfg.\n')
      elif e.reason == 'CERTIFICATE_VERIFY_FAILED':
        writeStdout(f'{not_okay}\n    Certificate verification failed. If you are behind a firewall / proxy server that does TLS / SSL inspection you may need to point GAM at your certificate authority file by setting cacerts_pem = /path/to/your/certauth.pem in gam.cfg.\n')
      elif e.strerror and e.strerror.startswith('TLS/SSL connection has been closed\n'):
        writeStdout(f'{not_okay}\n    TLS connection was closed. {gen_firewall}\n')
      else:
        writeStdout(f'{not_okay}\n    {str(e)}\n')
    except TimeoutError:
      writeStdout(f'{not_okay}\n    Timed out trying to connect to host\n')
    except Exception as e:
      writeStdout(f'{not_okay}\n    {str(e)}\n')

  try_count = 0
  httpObj = getHttpObj(timeout=30)
  httpObj.follow_redirects = False
  headers = {'user-agent': GAM_USER_AGENT}
  okay = createGreenText('OK')
  not_okay = createRedText('ERROR')
  success_count = 0
  initial_hosts = ['api.github.com',
           'raw.githubusercontent.com',
           'accounts.google.com',
           'oauth2.googleapis.com',
           'www.googleapis.com']
  for host in initial_hosts:
    check_host(host)
  api_hosts = ['apps-apis.google.com',
               'www.google.com']
  for host in api_hosts:
    check_host(host)
  # For v2 discovery APIs, GAM gets discovery file from <api>.googleapis.com so
  # add those domains.
  disc_hosts = []
  for api, config in API._INFO.items():
    if config.get('v2discovery') and not config.get('localdiscovery'):
      if mapped_api := config.get('mappedAPI'):
        api = mapped_api
      host = f'{api}.googleapis.com'
      if host not in disc_hosts:
        disc_hosts.append(host)
  for host in disc_hosts:
    check_host(host)
  checked_hosts = initial_hosts + api_hosts + disc_hosts
  # now we need to "build" each API and check it's base URL host
  # if we haven't already. This may not be any hosts at all but
  # to ensure we are checking all hosts GAM may use we should
  # keep this.
  for api in API._INFO:
    if api in [API.CONTACTS, API.EMAIL_AUDIT]:
      continue
    svc = getService(api, httpObj)
    base_url = svc._rootDesc.get('baseUrl')
    parsed_base_url = urlparse(base_url)
    base_host = parsed_base_url.netloc
    if base_host not in checked_hosts:
      writeStdout(f'Checking {base_host} for {api}\n')
      check_host(base_host)
      checked_hosts.append(base_host)
  if success_count == try_count:
    writeStdout(createGreenText('All hosts passed!\n'))
  else:
    systemErrorExit(3, createYellowText('Some hosts failed to connect! Please follow the recommendations for those hosts to correct any issues and try again.'))

# gam comment
def doComment():
  writeStdout(Cmd.QuotedArgumentList(Cmd.Remaining())+'\n')

# gam version [check|checkrc|simple|extended] [timeoffset] [nooffseterror] [location <HostName>]
def doVersion(checkForArgs=True):
  forceCheck = 0
  extended = noOffsetError = timeOffset = simple = False
  testLocation = GOOGLE_TIMECHECK_LOCATION
  if checkForArgs:
    while Cmd.ArgumentsRemaining():
      myarg = getArgument()
      if myarg == 'check':
        forceCheck = 1
      elif myarg == 'checkrc':
        forceCheck = -1
      elif myarg == 'simple':
        simple = True
      elif myarg == 'extended':
        extended = timeOffset = True
      elif myarg == 'timeoffset':
        timeOffset = True
      elif myarg == 'nooffseterror':
        noOffsetError = True
      elif myarg == 'location':
        testLocation = getString(Cmd.OB_HOST_NAME)
      else:
        unknownArgumentExit()
  if simple:
    writeStdout(__version__)
    return
  writeStdout((f'{GAM} {__version__} - {GAM_URL} - {GM.Globals[GM.GAM_TYPE]}\n'
               f'{__author__}\n'
               f'Python {sys.version_info[0]}.{sys.version_info[1]}.{sys.version_info[2]} {struct.calcsize("P")*8}-bit {sys.version_info[3]}\n'
               f'{getOSPlatform()} {platform.machine()}\n'
               f'Path: {GM.Globals[GM.GAM_PATH]}\n'
               f'{Ent.Singular(Ent.CONFIG_FILE)}: {GM.Globals[GM.GAM_CFG_FILE]}, {Ent.Singular(Ent.SECTION)}: {GM.Globals[GM.GAM_CFG_SECTION_NAME]}, '
               f'{GC.CUSTOMER_ID}: {GC.Values[GC.CUSTOMER_ID]}, {GC.DOMAIN}: {GC.Values[GC.DOMAIN]}\n'
               f'Time: {ISOformatTimeStamp(todaysTime())}\n'
               ))
  if sys.platform.startswith('win') and str(struct.calcsize('P')*8).find('32') != -1 and platform.machine().find('64') != -1:
    printKeyValueList([Msg.UPDATE_GAM_TO_64BIT])
  if timeOffset:
    offsetSeconds, offsetFormatted = getLocalGoogleTimeOffset(testLocation)
    printKeyValueList([Msg.YOUR_SYSTEM_TIME_DIFFERS_FROM_GOOGLE.format(testLocation, offsetFormatted)])
    if offsetSeconds > MAX_LOCAL_GOOGLE_TIME_OFFSET:
      if not noOffsetError:
        systemErrorExit(NETWORK_ERROR_RC, Msg.PLEASE_CORRECT_YOUR_SYSTEM_TIME)
      stderrWarningMsg(Msg.PLEASE_CORRECT_YOUR_SYSTEM_TIME)
  if forceCheck:
    doGAMCheckForUpdates(forceCheck)
  if extended:
    printKeyValueList([ssl.OPENSSL_VERSION])
    tls_ver, cipher_name = _getServerTLSUsed(testLocation)
    for lib in glverlibs.GAM_VER_LIBS:
      try:
        writeStdout(f'{lib} {lib_version(lib)}\n')
      except:
        pass
    printKeyValueList([f'{testLocation} connects using {tls_ver} {cipher_name}'])

# gam help
def doUsage():
  printBlankLine()
  doVersion(checkForArgs=False)
  writeStdout(Msg.HELP_SYNTAX.format(os.path.join(GM.Globals[GM.GAM_PATH], FN_GAMCOMMANDS_TXT)))
  writeStdout(Msg.HELP_WIKI.format(GAM_WIKI))

class NullHandler(logging.Handler):
  def emit(self, record):
    pass

def initializeLogging():
  nh = NullHandler()
  logging.getLogger().addHandler(nh)

def saveNonPickleableValues():
  savedValues = {GM.STDOUT: {}, GM.STDERR: {}, GM.SAVED_STDOUT: None,
                 GM.CMDLOG_HANDLER: None, GM.CMDLOG_LOGGER: None}
  savedValues[GM.SAVED_STDOUT] = GM.Globals[GM.SAVED_STDOUT]
  GM.Globals[GM.SAVED_STDOUT] = None
  savedValues[GM.STDOUT][GM.REDIRECT_FD] = GM.Globals[GM.STDOUT].get(GM.REDIRECT_FD, None)
  GM.Globals[GM.STDOUT].pop(GM.REDIRECT_FD, None)
  savedValues[GM.STDERR][GM.REDIRECT_FD] = GM.Globals[GM.STDERR].get(GM.REDIRECT_FD, None)
  GM.Globals[GM.STDERR].pop(GM.REDIRECT_FD, None)
  savedValues[GM.STDOUT][GM.REDIRECT_MULTI_FD] = GM.Globals[GM.STDOUT].get(GM.REDIRECT_MULTI_FD, None)
  GM.Globals[GM.STDOUT].pop(GM.REDIRECT_MULTI_FD, None)
  savedValues[GM.STDERR][GM.REDIRECT_MULTI_FD] = GM.Globals[GM.STDERR].get(GM.REDIRECT_MULTI_FD, None)
  GM.Globals[GM.STDERR].pop(GM.REDIRECT_MULTI_FD, None)
  savedValues[GM.CMDLOG_HANDLER] = GM.Globals[GM.CMDLOG_HANDLER]
  GM.Globals[GM.CMDLOG_HANDLER] = None
  savedValues[GM.CMDLOG_LOGGER] = GM.Globals[GM.CMDLOG_LOGGER]
  GM.Globals[GM.CMDLOG_LOGGER] = None
  return savedValues

def restoreNonPickleableValues(savedValues):
  GM.Globals[GM.SAVED_STDOUT] = savedValues[GM.SAVED_STDOUT]
  GM.Globals[GM.STDOUT][GM.REDIRECT_FD] = savedValues[GM.STDOUT][GM.REDIRECT_FD]
  GM.Globals[GM.STDERR][GM.REDIRECT_FD] = savedValues[GM.STDERR][GM.REDIRECT_FD]
  GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD] = savedValues[GM.STDOUT][GM.REDIRECT_MULTI_FD]
  GM.Globals[GM.STDERR][GM.REDIRECT_MULTI_FD] = savedValues[GM.STDERR][GM.REDIRECT_MULTI_FD]
  GM.Globals[GM.CMDLOG_HANDLER] = savedValues[GM.CMDLOG_HANDLER]
  GM.Globals[GM.CMDLOG_LOGGER] = savedValues[GM.CMDLOG_LOGGER]

def CSVFileQueueHandler(mpQueue, mpQueueStdout, mpQueueStderr, csvPF, datetimeNow, tzinfo, output_timeformat):
  global Cmd

  def reopenSTDFile(stdtype):
    if GM.Globals[stdtype][GM.REDIRECT_NAME] == 'null':
      GM.Globals[stdtype][GM.REDIRECT_FD] = open(os.devnull, GM.Globals[stdtype][GM.REDIRECT_MODE], encoding=UTF8)
    elif GM.Globals[stdtype][GM.REDIRECT_NAME] == '-':
      GM.Globals[stdtype][GM.REDIRECT_FD] = os.fdopen(os.dup([sys.stderr.fileno(), sys.stdout.fileno()][stdtype == GM.STDOUT]),
                                                      GM.Globals[stdtype][GM.REDIRECT_MODE], encoding=GM.Globals[GM.SYS_ENCODING])
    elif stdtype == GM.STDERR and GM.Globals[stdtype][GM.REDIRECT_NAME] == 'stdout':
      GM.Globals[stdtype][GM.REDIRECT_FD] = GM.Globals[GM.STDOUT][GM.REDIRECT_FD]
    else:
      GM.Globals[stdtype][GM.REDIRECT_FD] = openFile(GM.Globals[stdtype][GM.REDIRECT_NAME], GM.Globals[stdtype][GM.REDIRECT_MODE])
    if stdtype == GM.STDERR and GM.Globals[stdtype][GM.REDIRECT_NAME] == 'stdout':
      GM.Globals[stdtype][GM.REDIRECT_MULTI_FD] = GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD]
    else:
      GM.Globals[stdtype][GM.REDIRECT_MULTI_FD] = GM.Globals[stdtype][GM.REDIRECT_FD] if not GM.Globals[stdtype][GM.REDIRECT_MULTIPROCESS] else StringIOobject()

  GM.Globals[GM.DATETIME_NOW] = datetimeNow
  GC.Values[GC.TIMEZONE] = tzinfo
  GC.Values[GC.OUTPUT_TIMEFORMAT] = output_timeformat
  clearRowFilters = False
#  if sys.platform.startswith('win'):
#    signal.signal(signal.SIGINT, signal.SIG_IGN)
  if multiprocessing.get_start_method() == 'spawn':
    signal.signal(signal.SIGINT, signal.SIG_IGN)
    Cmd = glclargs.GamCLArgs()
  else:
    csvPF.SetColumnDelimiter(GC.Values[GC.CSV_OUTPUT_COLUMN_DELIMITER])
    csvPF.SetNoEscapeChar(GC.Values[GC.CSV_OUTPUT_NO_ESCAPE_CHAR])
    csvPF.SetQuoteChar(GC.Values[GC.CSV_OUTPUT_QUOTE_CHAR])
    csvPF.SetSortHeaders(GC.Values[GC.CSV_OUTPUT_SORT_HEADERS])
    csvPF.SetTimestampColumn(GC.Values[GC.CSV_OUTPUT_TIMESTAMP_COLUMN])
    csvPF.SetHeaderFilter(GC.Values[GC.CSV_OUTPUT_HEADER_FILTER])
    csvPF.SetHeaderDropFilter(GC.Values[GC.CSV_OUTPUT_HEADER_DROP_FILTER])
    csvPF.SetRowFilter(GC.Values[GC.CSV_OUTPUT_ROW_FILTER], GC.Values[GC.CSV_OUTPUT_ROW_FILTER_MODE])
    csvPF.SetRowDropFilter(GC.Values[GC.CSV_OUTPUT_ROW_DROP_FILTER], GC.Values[GC.CSV_OUTPUT_ROW_DROP_FILTER_MODE])
    csvPF.SetRowLimit(GC.Values[GC.CSV_OUTPUT_ROW_LIMIT])
  list_type = 'CSV'
  while True:
    dataType, dataItem = mpQueue.get()
    if dataType == GM.REDIRECT_QUEUE_NAME:
      list_type = dataItem
    elif dataType == GM.REDIRECT_QUEUE_TODRIVE:
      csvPF.todrive = dataItem
    elif dataType == GM.REDIRECT_QUEUE_CSVPF:
      csvPF.AddTitles(dataItem[0])
      csvPF.SetSortTitles(dataItem[1])
      csvPF.SetIndexedTitles(dataItem[2])
      csvPF.SetFormatJSON(dataItem[3])
      csvPF.AddJSONTitles(dataItem[4])
      csvPF.SetColumnDelimiter(dataItem[5])
      csvPF.SetNoEscapeChar(dataItem[6])
      csvPF.SetQuoteChar(dataItem[7])
      csvPF.SetSortHeaders(dataItem[8])
      csvPF.SetTimestampColumn(dataItem[9])
      csvPF.SetMapDrive3Titles(dataItem[10])
      csvPF.SetFixPaths(dataItem[11])
      csvPF.SetNodataFields(dataItem[12], dataItem[13], dataItem[14], dataItem[15], dataItem[16])
      csvPF.SetShowPermissionsLast(dataItem[17])
      csvPF.SetZeroBlankMimeTypeCounts(dataItem[18])
    elif dataType == GM.REDIRECT_QUEUE_DATA:
      csvPF.rows.extend(dataItem)
    elif dataType == GM.REDIRECT_QUEUE_ARGS:
      Cmd.InitializeArguments(dataItem)
    elif dataType == GM.REDIRECT_QUEUE_GLOBALS:
      GM.Globals = dataItem
      if multiprocessing.get_start_method() == 'spawn':
        reopenSTDFile(GM.STDOUT)
        reopenSTDFile(GM.STDERR)
    elif dataType == GM.REDIRECT_QUEUE_VALUES:
      GC.Values = dataItem
      csvPF.SetColumnDelimiter(GC.Values[GC.CSV_OUTPUT_COLUMN_DELIMITER])
      csvPF.SetNoEscapeChar(GC.Values[GC.CSV_OUTPUT_NO_ESCAPE_CHAR])
      csvPF.SetQuoteChar(GC.Values[GC.CSV_OUTPUT_QUOTE_CHAR])
      csvPF.SetSortHeaders(GC.Values[GC.CSV_OUTPUT_SORT_HEADERS])
      csvPF.SetTimestampColumn(GC.Values[GC.CSV_OUTPUT_TIMESTAMP_COLUMN])
      csvPF.SetHeaderFilter(GC.Values[GC.CSV_OUTPUT_HEADER_FILTER])
      csvPF.SetHeaderDropFilter(GC.Values[GC.CSV_OUTPUT_HEADER_DROP_FILTER])
      if not clearRowFilters:
        csvPF.SetRowFilter(GC.Values[GC.CSV_OUTPUT_ROW_FILTER], GC.Values[GC.CSV_OUTPUT_ROW_FILTER_MODE])
        csvPF.SetRowDropFilter(GC.Values[GC.CSV_OUTPUT_ROW_DROP_FILTER], GC.Values[GC.CSV_OUTPUT_ROW_DROP_FILTER_MODE])
      else:
        csvPF.SetRowFilter([], GC.Values[GC.CSV_OUTPUT_ROW_FILTER_MODE])
        csvPF.SetRowDropFilter([], GC.Values[GC.CSV_OUTPUT_ROW_DROP_FILTER_MODE])
      csvPF.SetRowLimit(GC.Values[GC.CSV_OUTPUT_ROW_LIMIT])
    elif dataType == GM.REDIRECT_QUEUE_CLEAR_ROW_FILTERS:
      clearRowFilters = dataItem
    else: #GM.REDIRECT_QUEUE_EOF
      break
  csvPF.writeCSVfile(list_type)
  if mpQueueStdout:
    mpQueueStdout.put((0, GM.REDIRECT_QUEUE_DATA, GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD].getvalue()))
  else:
    flushStdout()
  if mpQueueStderr and mpQueueStderr is not mpQueueStdout:
    mpQueueStderr.put((0, GM.REDIRECT_QUEUE_DATA, GM.Globals[GM.STDERR][GM.REDIRECT_MULTI_FD].getvalue()))
  else:
    flushStderr()

def initializeCSVFileQueueHandler(mpManager, mpQueueStdout, mpQueueStderr):
  mpQueue = mpManager.Queue()
  mpQueueHandler = multiprocessing.Process(target=CSVFileQueueHandler,
                                           args=(mpQueue, mpQueueStdout, mpQueueStderr,
                                                 GM.Globals[GM.CSVFILE][GM.REDIRECT_QUEUE_CSVPF],
                                                 GM.Globals[GM.DATETIME_NOW], GC.Values[GC.TIMEZONE],
                                                 GC.Values[GC.OUTPUT_TIMEFORMAT]))
  mpQueueHandler.start()
  return (mpQueue, mpQueueHandler)

def terminateCSVFileQueueHandler(mpQueue, mpQueueHandler):
  GM.Globals[GM.PARSER] = None
  GM.Globals[GM.CSVFILE][GM.REDIRECT_QUEUE] = None
  if multiprocessing.get_start_method() == 'spawn':
    mpQueue.put((GM.REDIRECT_QUEUE_ARGS, Cmd.AllArguments()))
    savedValues = saveNonPickleableValues()
    mpQueue.put((GM.REDIRECT_QUEUE_GLOBALS, GM.Globals))
    restoreNonPickleableValues(savedValues)
    mpQueue.put((GM.REDIRECT_QUEUE_VALUES, GC.Values))
  mpQueue.put((GM.REDIRECT_QUEUE_EOF, None))
  mpQueueHandler.join()

def StdQueueHandler(mpQueue, stdtype, gmGlobals, gcValues):

  PROCESS_MSG = '{0}: {1:6d}, {2:>5s}: {3}, RC: {4:3d}, Cmd: {5}\n'

  def _writeData(data):
    fd.write(data)

  def _writePidData(pid, data):
    try:
      if pid != 0 and GC.Values[GC.SHOW_MULTIPROCESS_INFO]:
        _writeData(PROCESS_MSG.format(pidData[pid]['queue'], pid, 'Start', pidData[pid]['start'], 0, pidData[pid]['cmd']))
      if data[1] is not None:
        _writeData(data[1])
      if GC.Values[GC.SHOW_MULTIPROCESS_INFO]:
        _writeData(PROCESS_MSG.format(pidData[pid]['queue'], pid, 'End', currentISOformatTimeStamp(), data[0], pidData[pid]['cmd']))
      fd.flush()
    except IOError as e:
      systemErrorExit(FILE_ERROR_RC, fdErrorMessage(fd, GM.Globals[stdtype][GM.REDIRECT_NAME], e))

#  if sys.platform.startswith('win'):
#    signal.signal(signal.SIGINT, signal.SIG_IGN)
  if multiprocessing.get_start_method() == 'spawn':
    signal.signal(signal.SIGINT, signal.SIG_IGN)
    GM.Globals = gmGlobals.copy()
    GC.Values = gcValues.copy()
  pid0DataItem = [KEYBOARD_INTERRUPT_RC, None]
  pidData = {}
  if multiprocessing.get_start_method() == 'spawn':
    if GM.Globals[stdtype][GM.REDIRECT_NAME] == 'null':
      fd = open(os.devnull, GM.Globals[stdtype][GM.REDIRECT_MODE], encoding=UTF8)
    elif GM.Globals[stdtype][GM.REDIRECT_NAME] == '-':
      fd = os.fdopen(os.dup([sys.stderr.fileno(), sys.stdout.fileno()][GM.Globals[stdtype][GM.REDIRECT_QUEUE] == 'stdout']),
                     GM.Globals[stdtype][GM.REDIRECT_MODE], encoding=GM.Globals[GM.SYS_ENCODING])
    elif GM.Globals[stdtype][GM.REDIRECT_NAME] == 'stdout' and GM.Globals[stdtype][GM.REDIRECT_QUEUE] == 'stderr':
      fd = os.fdopen(os.dup(sys.stdout.fileno()), GM.Globals[stdtype][GM.REDIRECT_MODE], encoding=GM.Globals[GM.SYS_ENCODING])
    else:
      fd = openFile(GM.Globals[stdtype][GM.REDIRECT_NAME], GM.Globals[stdtype][GM.REDIRECT_MODE])
  else:
    fd = GM.Globals[stdtype][GM.REDIRECT_FD]
  while True:
    try:
      pid, dataType, dataItem = mpQueue.get()
    except (EOFError, ValueError):
      break
    if dataType == GM.REDIRECT_QUEUE_START:
      pidData[pid] = {'queue': GM.Globals[stdtype][GM.REDIRECT_QUEUE],
                      'start': currentISOformatTimeStamp(),
                      'cmd': Cmd.QuotedArgumentList(dataItem)}
      if pid == 0 and GC.Values[GC.SHOW_MULTIPROCESS_INFO]:
        fd.write(PROCESS_MSG.format(pidData[pid]['queue'], pid, 'Start', pidData[pid]['start'], 0, pidData[pid]['cmd']))
    elif dataType == GM.REDIRECT_QUEUE_DATA:
      _writeData(dataItem)
    elif dataType == GM.REDIRECT_QUEUE_END:
      if pid != 0:
        _writePidData(pid, dataItem)
        del pidData[pid]
      else:
        pid0DataItem = dataItem
    else: #GM.REDIRECT_QUEUE_EOF
      break
  for pid in pidData:
    if pid != 0:
      _writePidData(pid, [KEYBOARD_INTERRUPT_RC, None])
  _writePidData(0, pid0DataItem)
  if fd not in [sys.stdout, sys.stderr]:
    try:
      fd.flush()
      fd.close()
    except IOError:
      pass
  GM.Globals[stdtype][GM.REDIRECT_FD] = None

def initializeStdQueueHandler(mpManager, stdtype, gmGlobals, gcValues):
  mpQueue = mpManager.Queue()
  mpQueueHandler = multiprocessing.Process(target=StdQueueHandler, args=(mpQueue, stdtype, gmGlobals, gcValues))
  mpQueueHandler.start()
  return (mpQueue, mpQueueHandler)

def batchWriteStderr(data):
  try:
    sys.stderr.write(data)
    sys.stderr.flush()
  except IOError as e:
    systemErrorExit(FILE_ERROR_RC, fileErrorMessage('stderr', e))

def writeStdQueueHandler(mpQueue, item):
  while True:
    try:
      mpQueue.put(item)
      return
    except Exception as e:
      time.sleep(1)
      batchWriteStderr(f'{currentISOformatTimeStamp()},{item[0]}/{GM.Globals[GM.NUM_BATCH_ITEMS]},Error,{str(e)}\n')

def terminateStdQueueHandler(mpQueue, mpQueueHandler):
  mpQueue.put((0, GM.REDIRECT_QUEUE_EOF, None))
  mpQueueHandler.join()

def ProcessGAMCommandMulti(pid, numItems, logCmd, mpQueueCSVFile, mpQueueStdout, mpQueueStderr,
                           debugLevel, todrive, printAguDomains,
                           printCrosOUs, printCrosOUsAndChildren,
                           output_dateformat, output_timeformat,
                           csvColumnDelimiter, csvNoEscapeChar, csvQuoteChar,
                           csvSortHeaders, csvTimestampColumn,
                           csvHeaderFilter, csvHeaderDropFilter,
                           csvHeaderForce, csvHeaderOrder,
                           csvRowFilter, csvRowFilterMode, csvRowDropFilter, csvRowDropFilterMode,
                           csvRowLimit,
                           showGettings, showGettingsGotNL,
                           args):
  global mplock

  with mplock:
    initializeLogging()
#    if sys.platform.startswith('win'):
    if multiprocessing.get_start_method() == 'spawn':
      signal.signal(signal.SIGINT, signal.SIG_IGN)
    GM.Globals[GM.API_CALLS_RETRY_DATA] = {}
    GM.Globals[GM.CMDLOG_LOGGER] = None
    GM.Globals[GM.CSVFILE] = {}
    GM.Globals[GM.CSV_DATA_DICT] = {}
    GM.Globals[GM.CSV_KEY_FIELD] = None
    GM.Globals[GM.CSV_SUBKEY_FIELD] = None
    GM.Globals[GM.CSV_DATA_FIELD] = None
    GM.Globals[GM.CSV_OUTPUT_COLUMN_DELIMITER] = csvColumnDelimiter
    GM.Globals[GM.CSV_OUTPUT_NO_ESCAPE_CHAR] = csvNoEscapeChar
    GM.Globals[GM.CSV_OUTPUT_HEADER_DROP_FILTER] = csvHeaderDropFilter[:]
    GM.Globals[GM.CSV_OUTPUT_HEADER_FILTER] = csvHeaderFilter[:]
    GM.Globals[GM.CSV_OUTPUT_HEADER_FORCE] = csvHeaderForce[:]
    GM.Globals[GM.CSV_OUTPUT_HEADER_ORDER] = csvHeaderOrder[:]
    GM.Globals[GM.CSV_OUTPUT_QUOTE_CHAR] = csvQuoteChar
    GM.Globals[GM.CSV_OUTPUT_ROW_DROP_FILTER] = csvRowDropFilter[:]
    GM.Globals[GM.CSV_OUTPUT_ROW_DROP_FILTER_MODE] = csvRowDropFilterMode
    GM.Globals[GM.CSV_OUTPUT_ROW_FILTER] = csvRowFilter[:]
    GM.Globals[GM.CSV_OUTPUT_ROW_FILTER_MODE] = csvRowFilterMode
    GM.Globals[GM.CSV_OUTPUT_ROW_LIMIT] = csvRowLimit
    GM.Globals[GM.CSV_OUTPUT_SORT_HEADERS] = csvSortHeaders[:]
    GM.Globals[GM.CSV_OUTPUT_TIMESTAMP_COLUMN] = csvTimestampColumn
    GM.Globals[GM.CSV_TODRIVE] = todrive.copy()
    GM.Globals[GM.DEBUG_LEVEL] = debugLevel
    GM.Globals[GM.OUTPUT_DATEFORMAT] = output_dateformat
    GM.Globals[GM.OUTPUT_TIMEFORMAT] = output_timeformat
    GM.Globals[GM.NUM_BATCH_ITEMS] = numItems
    GM.Globals[GM.PID] = pid
    GM.Globals[GM.PRINT_AGU_DOMAINS] = printAguDomains[:]
    GM.Globals[GM.PRINT_CROS_OUS] = printCrosOUs[:]
    GM.Globals[GM.PRINT_CROS_OUS_AND_CHILDREN] = printCrosOUsAndChildren[:]
    GM.Globals[GM.SAVED_STDOUT] = None
    GM.Globals[GM.SHOW_GETTINGS] = showGettings
    GM.Globals[GM.SHOW_GETTINGS_GOT_NL] = showGettingsGotNL
    GM.Globals[GM.SYSEXITRC] = 0
    GM.Globals[GM.PARSER] = None
    if mpQueueCSVFile:
      GM.Globals[GM.CSVFILE][GM.REDIRECT_QUEUE] = mpQueueCSVFile
    if mpQueueStdout:
      GM.Globals[GM.STDOUT] = {GM.REDIRECT_NAME: '', GM.REDIRECT_FD: None, GM.REDIRECT_MULTI_FD: StringIOobject()}
      if debugLevel:
        sys.stdout = GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD]
#      mpQueueStdout.put((pid, GM.REDIRECT_QUEUE_START, args))
      writeStdQueueHandler(mpQueueStdout,(pid, GM.REDIRECT_QUEUE_START, args))
    else:
      GM.Globals[GM.STDOUT] = {}
    if mpQueueStderr:
      if mpQueueStderr is not mpQueueStdout:
        GM.Globals[GM.STDERR] = {GM.REDIRECT_NAME: '', GM.REDIRECT_FD: None, GM.REDIRECT_MULTI_FD: StringIOobject()}
#        mpQueueStderr.put((pid, GM.REDIRECT_QUEUE_START, args))
        writeStdQueueHandler(mpQueueStderr, (pid, GM.REDIRECT_QUEUE_START, args))
      else:
        GM.Globals[GM.STDERR][GM.REDIRECT_MULTI_FD] = GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD]
    else:
      GM.Globals[GM.STDERR] = {}
  sysRC = ProcessGAMCommand(args)
  with mplock:
    if mpQueueStdout:
#      mpQueueStdout.put((pid, GM.REDIRECT_QUEUE_END, [sysRC, GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD].getvalue()]))
      writeStdQueueHandler(mpQueueStdout, (pid, GM.REDIRECT_QUEUE_END, [sysRC, GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD].getvalue()]))
      GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD].close()
      GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD] = None
    if mpQueueStderr and mpQueueStderr is not mpQueueStdout:
#      mpQueueStderr.put((pid, GM.REDIRECT_QUEUE_END, [sysRC, GM.Globals[GM.STDERR][GM.REDIRECT_MULTI_FD].getvalue()]))
      writeStdQueueHandler(mpQueueStderr, (pid, GM.REDIRECT_QUEUE_END, [sysRC, GM.Globals[GM.STDERR][GM.REDIRECT_MULTI_FD].getvalue()]))
      GM.Globals[GM.STDERR][GM.REDIRECT_MULTI_FD].close()
      GM.Globals[GM.STDERR][GM.REDIRECT_MULTI_FD] = None
  return (pid, sysRC, logCmd)

ERROR_PLURAL_SINGULAR = [Msg.ERRORS, Msg.ERROR]
PROCESS_PLURAL_SINGULAR = [Msg.PROCESSES, Msg.PROCESS]
THREAD_PLURAL_SINGULAR = [Msg.THREADS, Msg.THREAD]

def checkChildProcessRC(rc):
# Comparison
  if 'comp' in GM.Globals[GM.MULTIPROCESS_EXIT_CONDITION]:
    op = GM.Globals[GM.MULTIPROCESS_EXIT_CONDITION]['comp']
    value = GM.Globals[GM.MULTIPROCESS_EXIT_CONDITION]['value']
    if op == '<':
      return rc < value
    if op == '<=':
      return rc <= value
    if op == '>':
      return rc > value
    if op == '>=':
      return rc >= value
    if op == '!=':
      return rc != value
    return rc == value
# Range
  op = GM.Globals[GM.MULTIPROCESS_EXIT_CONDITION]['range']
  low = GM.Globals[GM.MULTIPROCESS_EXIT_CONDITION]['low']
  high = GM.Globals[GM.MULTIPROCESS_EXIT_CONDITION]['high']
  if op == '!=':
    return not low <= rc <= high
  return low <= rc <= high

def initGamWorker(l):
  global mplock
  mplock = l

def MultiprocessGAMCommands(items, showCmds):
  def poolCallback(result):
    poolProcessResults[0] -= 1
    if showCmds:
      batchWriteStderr(f'{currentISOformatTimeStamp()},{result[0]}/{numItems},End,{result[1]},{result[2]}\n')
    if GM.Globals[GM.CMDLOG_LOGGER]:
      GM.Globals[GM.CMDLOG_LOGGER].info(f'{currentISOformatTimeStamp()},{result[1]},{result[2]}')
    if GM.Globals[GM.MULTIPROCESS_EXIT_CONDITION] is not None and checkChildProcessRC(result[1]):
      GM.Globals[GM.MULTIPROCESS_EXIT_PROCESSING] = True

  def signal_handler(*_):
    nonlocal controlC
    controlC = True

  def handleControlC(source):
    nonlocal controlC
    batchWriteStderr(f'Control-C (Multiprocess-{source})\n')
    setSysExitRC(KEYBOARD_INTERRUPT_RC)
    batchWriteStderr(Msg.BATCH_CSV_TERMINATE_N_PROCESSES.format(currentISOformatTimeStamp(),
                                                                numItems, poolProcessResults[0],
                                                                PROCESS_PLURAL_SINGULAR[poolProcessResults[0] == 1]))
    pool.terminate()
    controlC = False

  if not items:
    return
  GM.Globals[GM.NUM_BATCH_ITEMS] = numItems = len(items)
  numPoolProcesses = min(numItems, GC.Values[GC.NUM_THREADS])
  if GC.Values[GC.MULTIPROCESS_POOL_LIMIT] == -1:
    parallelPoolProcesses = -1
  elif GC.Values[GC.MULTIPROCESS_POOL_LIMIT] == 0:
    parallelPoolProcesses = numPoolProcesses
  else:
    parallelPoolProcesses = min(numItems, GC.Values[GC.MULTIPROCESS_POOL_LIMIT])
#  origSigintHandler = signal.signal(signal.SIGINT, signal.SIG_IGN)
  signal.signal(signal.SIGINT, signal.SIG_IGN)
  mpManager = multiprocessing.Manager()
  l = mpManager.Lock()
  try:
    if multiprocessing.get_start_method() == 'spawn':
      pool = mpManager.Pool(processes=numPoolProcesses, initializer=initGamWorker, initargs=(l,), maxtasksperchild=200)
    else:
      pool = multiprocessing.Pool(processes=numPoolProcesses, initializer=initGamWorker, initargs=(l,), maxtasksperchild=200)
  except IOError as e:
    systemErrorExit(FILE_ERROR_RC, e)
  except AssertionError as e:
    Cmd.SetLocation(0)
    usageErrorExit(str(e))
  if multiprocessing.get_start_method() == 'spawn':
    savedValues = saveNonPickleableValues()
  if GM.Globals[GM.STDOUT][GM.REDIRECT_MULTIPROCESS]:
    mpQueueStdout, mpQueueHandlerStdout = initializeStdQueueHandler(mpManager, GM.STDOUT, GM.Globals, GC.Values)
    mpQueueStdout.put((0, GM.REDIRECT_QUEUE_START, Cmd.AllArguments()))
  else:
    mpQueueStdout = None
  if GM.Globals[GM.STDERR][GM.REDIRECT_MULTIPROCESS]:
    if GM.Globals[GM.STDERR][GM.REDIRECT_NAME] != 'stdout':
      mpQueueStderr, mpQueueHandlerStderr = initializeStdQueueHandler(mpManager, GM.STDERR, GM.Globals, GC.Values)
      mpQueueStderr.put((0, GM.REDIRECT_QUEUE_START, Cmd.AllArguments()))
    else:
      mpQueueStderr = mpQueueStdout
  else:
    mpQueueStderr = None
  if multiprocessing.get_start_method() == 'spawn':
    restoreNonPickleableValues(savedValues)
  if mpQueueStdout:
    mpQueueStdout.put((0, GM.REDIRECT_QUEUE_DATA, GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD].getvalue()))
    GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD].truncate(0)
  if mpQueueStderr and mpQueueStderr is not mpQueueStdout:
    mpQueueStderr.put((0, GM.REDIRECT_QUEUE_DATA, GM.Globals[GM.STDERR][GM.REDIRECT_MULTI_FD].getvalue()))
    GM.Globals[GM.STDERR][GM.REDIRECT_MULTI_FD].truncate(0)
  if GM.Globals[GM.CSVFILE][GM.REDIRECT_MULTIPROCESS]:
    mpQueueCSVFile, mpQueueHandlerCSVFile = initializeCSVFileQueueHandler(mpManager, mpQueueStdout, mpQueueStderr)
  else:
    mpQueueCSVFile = None
#  signal.signal(signal.SIGINT, origSigintHandler)
  controlC = False
  signal.signal(signal.SIGINT, signal_handler)
  batchWriteStderr(Msg.USING_N_PROCESSES.format(currentISOformatTimeStamp(),
                                                numItems, numPoolProcesses,
                                                PROCESS_PLURAL_SINGULAR[numPoolProcesses == 1]))
  try:
    pid = 0
    poolProcessResults = {pid: 0}
    for item in items:
      if GM.Globals[GM.MULTIPROCESS_EXIT_PROCESSING]:
        break
      if controlC:
        break
      if item[0] == Cmd.COMMIT_BATCH_CMD:
        batchWriteStderr(Msg.COMMIT_BATCH_WAIT_N_PROCESSES.format(currentISOformatTimeStamp(),
                                                                  numItems, poolProcessResults[0],
                                                                  PROCESS_PLURAL_SINGULAR[poolProcessResults[0] == 1]))
        while poolProcessResults[0] > 0:
          time.sleep(1)
          completedProcesses = []
          for p, result in poolProcessResults.items():
            if p != 0 and result.ready():
              poolCallback(result.get())
              completedProcesses.append(p)
          for p in completedProcesses:
            del poolProcessResults[p]
        batchWriteStderr(Msg.COMMIT_BATCH_COMPLETE.format(currentISOformatTimeStamp(), numItems, Msg.PROCESSES))
        if len(item) > 1:
          readStdin(f'{currentISOformatTimeStamp()},0/{numItems},{Cmd.QuotedArgumentList(item[1:])}')
        continue
      if item[0] == Cmd.PRINT_CMD:
        batchWriteStderr(Cmd.QuotedArgumentList(item[1:])+'\n')
        continue
      if item[0] == Cmd.SLEEP_CMD:
        batchWriteStderr(f'{currentISOformatTimeStamp()},0/{numItems},Sleepiing {item[1]} seconds\n')
        time.sleep(int(item[1]))
        continue
      pid += 1
      if not showCmds and ((pid % 100 == 0) or (pid == numItems)):
        batchWriteStderr(Msg.PROCESSING_ITEM_N_OF_M.format(currentISOformatTimeStamp(), pid, numItems))
      if showCmds or GM.Globals[GM.CMDLOG_LOGGER]:
        logCmd = Cmd.QuotedArgumentList(item)
        if showCmds:
          batchWriteStderr(f'{currentISOformatTimeStamp()},{pid}/{numItems},Start,0,{logCmd}\n')
      else:
        logCmd = ''
      poolProcessResults[pid] = pool.apply_async(ProcessGAMCommandMulti,
                                                 [pid, numItems, logCmd, mpQueueCSVFile, mpQueueStdout, mpQueueStderr,
                                                  GC.Values[GC.DEBUG_LEVEL], GM.Globals[GM.CSV_TODRIVE],
                                                  GC.Values[GC.PRINT_AGU_DOMAINS],
                                                  GC.Values[GC.PRINT_CROS_OUS], GC.Values[GC.PRINT_CROS_OUS_AND_CHILDREN],
                                                  GC.Values[GC.OUTPUT_DATEFORMAT], GC.Values[GC.OUTPUT_TIMEFORMAT],
                                                  GC.Values[GC.CSV_OUTPUT_COLUMN_DELIMITER],
                                                  GC.Values[GC.CSV_OUTPUT_NO_ESCAPE_CHAR],
                                                  GC.Values[GC.CSV_OUTPUT_QUOTE_CHAR],
                                                  GC.Values[GC.CSV_OUTPUT_SORT_HEADERS],
                                                  GC.Values[GC.CSV_OUTPUT_TIMESTAMP_COLUMN],
                                                  GC.Values[GC.CSV_OUTPUT_HEADER_FILTER],
                                                  GC.Values[GC.CSV_OUTPUT_HEADER_DROP_FILTER],
                                                  GC.Values[GC.CSV_OUTPUT_HEADER_FORCE],
                                                  GC.Values[GC.CSV_OUTPUT_HEADER_ORDER],
                                                  GC.Values[GC.CSV_OUTPUT_ROW_FILTER],
                                                  GC.Values[GC.CSV_OUTPUT_ROW_FILTER_MODE],
                                                  GC.Values[GC.CSV_OUTPUT_ROW_DROP_FILTER],
                                                  GC.Values[GC.CSV_OUTPUT_ROW_DROP_FILTER_MODE],
                                                  GC.Values[GC.CSV_OUTPUT_ROW_LIMIT],
                                                  GC.Values[GC.SHOW_GETTINGS], GC.Values[GC.SHOW_GETTINGS_GOT_NL],
                                                  item])
      poolProcessResults[0] += 1
      if parallelPoolProcesses > 0:
        while poolProcessResults[0] == parallelPoolProcesses:
          completedProcesses = []
          for p, result in poolProcessResults.items():
            if p != 0 and result.ready():
              poolCallback(result.get())
              completedProcesses.append(p)
          if completedProcesses:
            for p in completedProcesses:
              del poolProcessResults[p]
            break
          time.sleep(1)
    processWaitStart = time.time()
    if not controlC:
      if GC.Values[GC.PROCESS_WAIT_LIMIT] > 0:
        waitRemaining = GC.Values[GC.PROCESS_WAIT_LIMIT]
      else:
        waitRemaining = 'unlimited'
      while poolProcessResults[0] > 0:
        batchWriteStderr(Msg.BATCH_CSV_WAIT_N_PROCESSES.format(currentISOformatTimeStamp(),
                                                               numItems, poolProcessResults[0],
                                                               PROCESS_PLURAL_SINGULAR[poolProcessResults[0] == 1],
                                                               Msg.BATCH_CSV_WAIT_LIMIT.format(waitRemaining)))
        completedProcesses = []
        for p, result in poolProcessResults.items():
          if p != 0 and result.ready():
            poolCallback(result.get())
            completedProcesses.append(p)
        for p in completedProcesses:
          del poolProcessResults[p]
        if poolProcessResults[0] > 0:
          if controlC:
            handleControlC('SIG')
            break
          time.sleep(5)
          if GC.Values[GC.PROCESS_WAIT_LIMIT] > 0:
            delta = int(time.time()-processWaitStart)
            if delta >= GC.Values[GC.PROCESS_WAIT_LIMIT]:
              batchWriteStderr(Msg.BATCH_CSV_TERMINATE_N_PROCESSES.format(currentISOformatTimeStamp(),
                                                                          numItems, poolProcessResults[0],
                                                                          PROCESS_PLURAL_SINGULAR[poolProcessResults[0] == 1]))
              pool.terminate()
              break
            waitRemaining = GC.Values[GC.PROCESS_WAIT_LIMIT] - delta
      pool.close()
    else:
      handleControlC('SIG')
  except KeyboardInterrupt:
    handleControlC('KBI')
  pool.join()
  batchWriteStderr(Msg.BATCH_CSV_PROCESSING_COMPLETE.format(currentISOformatTimeStamp(), numItems))
  if mpQueueCSVFile:
    terminateCSVFileQueueHandler(mpQueueCSVFile, mpQueueHandlerCSVFile)
  if mpQueueStdout:
    mpQueueStdout.put((0, GM.REDIRECT_QUEUE_END, [GM.Globals[GM.SYSEXITRC], GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD].getvalue()]))
    GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD].close()
    GM.Globals[GM.STDOUT][GM.REDIRECT_MULTI_FD] = None
    terminateStdQueueHandler(mpQueueStdout, mpQueueHandlerStdout)
  if mpQueueStderr and mpQueueStderr is not mpQueueStdout:
    mpQueueStderr.put((0, GM.REDIRECT_QUEUE_END, [GM.Globals[GM.SYSEXITRC], GM.Globals[GM.STDERR][GM.REDIRECT_MULTI_FD].getvalue()]))
    GM.Globals[GM.STDERR][GM.REDIRECT_MULTI_FD].close()
    GM.Globals[GM.STDERR][GM.REDIRECT_MULTI_FD] = None
    terminateStdQueueHandler(mpQueueStderr, mpQueueHandlerStderr)

def threadBatchWorker(showCmds=False, numItems=0):
  while True:
    pid, item, logCmd = GM.Globals[GM.TBATCH_QUEUE].get()
    try:
      sysRC = subprocess.call(item, stdout=GM.Globals[GM.STDOUT].get(GM.REDIRECT_MULTI_FD, sys.stdout),
                              stderr=GM.Globals[GM.STDERR].get(GM.REDIRECT_MULTI_FD, sys.stderr))
      if showCmds:
        batchWriteStderr(f'{currentISOformatTimeStamp()},{pid}/{numItems},End,{sysRC},{logCmd}\n')
      if GM.Globals[GM.MULTIPROCESS_EXIT_CONDITION] is not None and checkChildProcessRC(sysRC):
        GM.Globals[GM.MULTIPROCESS_EXIT_PROCESSING] = True
    except Exception as e:
      batchWriteStderr(f'{currentISOformatTimeStamp()},{pid}/{numItems},Error,{str(e)},{logCmd}\n')
    GM.Globals[GM.TBATCH_QUEUE].task_done()

BATCH_COMMANDS = [Cmd.GAM_CMD, Cmd.COMMIT_BATCH_CMD, Cmd.PRINT_CMD, Cmd.SLEEP_CMD]
TBATCH_COMMANDS = [Cmd.GAM_CMD, Cmd.COMMIT_BATCH_CMD, Cmd.EXECUTE_CMD, Cmd.PRINT_CMD, Cmd.SLEEP_CMD]

def ThreadBatchGAMCommands(items, showCmds):
  if not items:
    return
  pythonCmd = [sys.executable]
  if not getattr(sys, 'frozen', False): # we're not frozen
    pythonCmd.append(os.path.realpath(Cmd.Argument(0)))
  GM.Globals[GM.NUM_BATCH_ITEMS] = numItems = len(items)
  numWorkerThreads = min(numItems, GC.Values[GC.NUM_TBATCH_THREADS])
# GM.Globals[GM.TBATCH_QUEUE].put() gets blocked when trying to create more items than there are workers
  GM.Globals[GM.TBATCH_QUEUE] = queue.Queue(maxsize=numWorkerThreads)
  batchWriteStderr(Msg.USING_N_PROCESSES.format(currentISOformatTimeStamp(),
                                                numItems, numWorkerThreads,
                                                THREAD_PLURAL_SINGULAR[numWorkerThreads == 1]))
  for _ in range(numWorkerThreads):
    t = threading.Thread(target=threadBatchWorker, kwargs={'showCmds': showCmds, 'numItems': numItems})
    t.daemon = True
    t.start()
  pid = 0
  numThreadsInUse = 0
  for item in items:
    if GM.Globals[GM.MULTIPROCESS_EXIT_PROCESSING]:
      break
    if item[0] == Cmd.COMMIT_BATCH_CMD:
      batchWriteStderr(Msg.COMMIT_BATCH_WAIT_N_PROCESSES.format(currentISOformatTimeStamp(),
                                                                numItems, numThreadsInUse,
                                                                THREAD_PLURAL_SINGULAR[numThreadsInUse == 1]))
      GM.Globals[GM.TBATCH_QUEUE].join()
      batchWriteStderr(Msg.COMMIT_BATCH_COMPLETE.format(currentISOformatTimeStamp(), numItems, Msg.THREADS))
      numThreadsInUse = 0
      if len(item) > 1:
        readStdin(f'{currentISOformatTimeStamp()},0/{numItems},{Cmd.QuotedArgumentList(item[1:])}')
      continue
    if item[0] == Cmd.PRINT_CMD:
      batchWriteStderr(f'{currentISOformatTimeStamp()},0/{numItems},{Cmd.QuotedArgumentList(item[1:])}\n')
      continue
    if item[0] == Cmd.SLEEP_CMD:
      batchWriteStderr(f'{currentISOformatTimeStamp()},0/{numItems},Sleeping {item[1]} seconds\n')
      time.sleep(int(item[1]))
      continue
    pid += 1
    if not showCmds and ((pid % 100 == 0) or (pid == numItems)):
      batchWriteStderr(Msg.PROCESSING_ITEM_N_OF_M.format(currentISOformatTimeStamp(), pid, numItems))
    if showCmds:
      logCmd = Cmd.QuotedArgumentList(item)
      batchWriteStderr(f'{currentISOformatTimeStamp()},{pid}/{numItems},Start,{Cmd.QuotedArgumentList(item)}\n')
    else:
      logCmd = ''
    if item[0] == Cmd.GAM_CMD:
      GM.Globals[GM.TBATCH_QUEUE].put((pid, pythonCmd+item[1:], logCmd))
    else:
      GM.Globals[GM.TBATCH_QUEUE].put((pid, item[1:], logCmd))
    numThreadsInUse += 1
  GM.Globals[GM.TBATCH_QUEUE].join()
  if showCmds:
    batchWriteStderr(f'{currentISOformatTimeStamp()},0/{numItems},Complete\n')

def _getShowCommands():
  if checkArgumentPresent('showcmds'):
    return getBoolean()
  return GC.Values[GC.SHOW_COMMANDS]

def _getSkipRows():
  if checkArgumentPresent('skiprows'):
    return getInteger(minVal=0)
#  return GC.Values[GC.CSV_INPUT_ROW_SKIP]
  return 0

def _getMaxRows():
  if checkArgumentPresent('maxrows'):
    return getInteger(minVal=0)
  return GC.Values[GC.CSV_INPUT_ROW_LIMIT]

# gam batch <BatchContent> [showcmds [<Boolean>]]
def doBatch(threadBatch=False):
  filename = getString(Cmd.OB_FILE_NAME)
  if (filename == '-') and (GC.Values[GC.DEBUG_LEVEL] > 0):
    Cmd.Backup()
    usageErrorExit(Msg.BATCH_CSV_LOOP_DASH_DEBUG_INCOMPATIBLE.format(Cmd.BATCH_CMD))
  filenameLower = filename.lower()
  if filenameLower not in {'gdoc', 'gcsdoc'}:
    encoding = getCharSet()
    f = openFile(filename, encoding=encoding, stripUTFBOM=True)
  elif filenameLower == 'gdoc':
    f = getGDocData(filenameLower)
    getCharSet()
  else: #filenameLower == 'gcsdoc':
    f = getStorageFileData(filenameLower)
    getCharSet()
  showCmds = _getShowCommands()
  checkForExtraneousArguments()
  validCommands = BATCH_COMMANDS if not threadBatch else TBATCH_COMMANDS
  kwValues = {}
  items = []
  errors = 0
  try:
    for line in f:
      if line.startswith('#'):
        continue
      if kwValues:
        for kw, value in kwValues.items():
          line = line.replace(f'%{kw}%', value)
      try:
        argv = shlex.split(line)
      except ValueError as e:
        writeStderr(f'Command: >>>{line.strip()}<<<\n')
        writeStderr(f'{ERROR_PREFIX}{str(e)}\n')
        errors += 1
        continue
      if argv:
        cmd = argv[0].strip().lower()
        if cmd == Cmd.SET_CMD:
          if len(argv) == 3:
            kwValues[argv[1]] = argv[2]
          else:
            writeStderr(f'Command: >>>{Cmd.QuotedArgumentList([argv[0]])}<<< {Cmd.QuotedArgumentList(argv[1:])}\n')
            writeStderr(f'{ERROR_PREFIX}{Cmd.ARGUMENT_ERROR_NAMES[Cmd.ARGUMENT_INVALID][1]}: {Msg.EXPECTED} <{Cmd.SET_CMD} keyword value>)>\n')
            errors += 1
          continue
        if cmd == Cmd.CLEAR_CMD:
          if len(argv) == 2:
            kwValues.pop(argv[1], None)
          else:
            writeStderr(f'Command: >>>{Cmd.QuotedArgumentList([argv[0]])}<<< {Cmd.QuotedArgumentList(argv[1:])}\n')
            writeStderr(f'{ERROR_PREFIX}{Cmd.ARGUMENT_ERROR_NAMES[Cmd.ARGUMENT_INVALID][1]}: {Msg.EXPECTED} <{Cmd.CLEAR_CMD} keyword>)>\n')
            errors += 1
          continue
        if cmd == Cmd.SLEEP_CMD:
          if len(argv) != 2 or not argv[1].isdigit():
            writeStderr(f'Command: >>>{Cmd.QuotedArgumentList([argv[0]])}<<< {Cmd.QuotedArgumentList(argv[1:])}\n')
            writeStderr(f'{ERROR_PREFIX}{Cmd.ARGUMENT_ERROR_NAMES[Cmd.ARGUMENT_INVALID][1]}: {Msg.EXPECTED} <{Cmd.SLEEP_CMD} integer>)>\n')
            errors += 1
            continue
        if (not cmd) or ((len(argv) == 1) and (cmd not in [Cmd.COMMIT_BATCH_CMD, Cmd.PRINT_CMD])):
          continue
        if cmd in validCommands:
          items.append(argv)
        else:
          writeStderr(f'Command: >>>{Cmd.QuotedArgumentList([argv[0]])}<<< {Cmd.QuotedArgumentList(argv[1:])}\n')
          writeStderr(f'{ERROR_PREFIX}{Cmd.ARGUMENT_ERROR_NAMES[Cmd.ARGUMENT_INVALID][1]}: {Msg.EXPECTED} <{formatChoiceList(validCommands)}>\n')
          errors += 1
  except IOError as e:
    systemErrorExit(FILE_ERROR_RC, fileErrorMessage(filename, e))
  closeFile(f)
  if errors == 0:
    if not threadBatch:
      MultiprocessGAMCommands(items, showCmds)
    else:
      ThreadBatchGAMCommands(items, showCmds)
  else:
    writeStderr(Msg.BATCH_NOT_PROCESSED_ERRORS.format(ERROR_PREFIX, filename, errors, ERROR_PLURAL_SINGULAR[errors == 1]))
    setSysExitRC(USAGE_ERROR_RC)

# gam tbatch <BatchContent> [showcmds [<Boolean>]]
def doThreadBatch():
  adjustRedirectedSTDFilesIfNotMultiprocessing()
  doBatch(True)

def doAutoBatch(entityType, entityList, CL_command):
  remaining = Cmd.Remaining()
  items = []
  initial_argv = [Cmd.GAM_CMD]
  if GM.Globals[GM.SECTION] and not GM.Globals[GM.GAM_CFG_SECTION]:
    initial_argv.extend([Cmd.SELECT_CMD, GM.Globals[GM.SECTION]])
  for entity in entityList:
    items.append(initial_argv+[entityType, entity, CL_command]+remaining)
  MultiprocessGAMCommands(items, GC.Values[GC.SHOW_COMMANDS])

# Process command line arguments, find substitutions
# An argument containing instances of ~~xxx~!~pattern~!~replacement~~ has ~~...~~ replaced by re.sub(pattern, replacement, value of field xxx from the CSV file)
# For example, ~~primaryEmail~!~^(.+)@(.+)$~!~\1 AT \2~~ would replace foo@bar.com (from the primaryEmail column) with foo AT bar.com
# An argument containing instances of ~~xxx~~ has xxx replaced by the value of field xxx from the CSV file
# An argument containing exactly ~xxx is replaced by the value of field xxx from the CSV file
# Otherwise, the argument is preserved as is

SUB_PATTERN = re.compile(r'~~(.+?)~~')
RE_PATTERN = re.compile(r'~~(.+?)~!~(.+?)~!~(.+?)~~')
SUB_TYPE = 'sub'
RE_TYPE = 're'

# SubFields is a dictionary; the key is the argument number, the value is a list of tuples that mark
# the substition (type, fieldname, start, end). Type is 'sub' for simple substitution, 're' for regex substitution.
# Example: update user '~User' address type work unstructured '~~Street~~, ~~City~~, ~~State~~ ~~ZIP~~' primary
# {2: [('sub', 'User', 0, 5)], 7: [('sub', 'Street', 0, 10), ('sub', 'City', 12, 20), ('sub', 'State', 22, 31), ('sub', 'ZIP', 32, 39)]}
def getSubFields(initial_argv, fieldNames):
  subFields = {}
  GAM_argv = initial_argv[:]
  GAM_argvI = len(GAM_argv)
  while Cmd.ArgumentsRemaining():
    myarg = Cmd.Current()
    if not myarg:
      GAM_argv.append(myarg)
    elif SUB_PATTERN.search(myarg):
      pos = 0
      subFields.setdefault(GAM_argvI, [])
      while True:
        submatch = SUB_PATTERN.search(myarg, pos)
        if not submatch:
          break
        rematch = RE_PATTERN.match(submatch.group(0))
        if not rematch:
          fieldName = submatch.group(1)
          if fieldName not in fieldNames:
            csvFieldErrorExit(fieldName, fieldNames)
          subFields[GAM_argvI].append((SUB_TYPE, fieldName, submatch.start(), submatch.end()))
        else:
          fieldName = rematch.group(1)
          if fieldName not in fieldNames:
            csvFieldErrorExit(fieldName, fieldNames)
          try:
            re.compile(rematch.group(2))
            subFields[GAM_argvI].append((RE_TYPE, fieldName, submatch.start(), submatch.end(), rematch.group(2), rematch.group(3)))
          except re.error as e:
            usageErrorExit(f'{Cmd.OB_RE_PATTERN} {Msg.ERROR}: {e}')
        pos = submatch.end()
      GAM_argv.append(myarg)
    elif myarg[0] == '~':
      fieldName = myarg[1:]
      if fieldName in fieldNames:
        subFields[GAM_argvI] = [(SUB_TYPE, fieldName, 0, len(myarg))]
        GAM_argv.append(myarg)
      else:
        csvFieldErrorExit(fieldName, fieldNames)
    else:
      GAM_argv.append(myarg)
    GAM_argvI += 1
    Cmd.Advance()
  return(GAM_argv, subFields)

def processSubFields(GAM_argv, row, subFields):
  argv = GAM_argv[:]
  for GAM_argvI, fields in subFields.items():
    oargv = argv[GAM_argvI][:]
    argv[GAM_argvI] = ''
    pos = 0
    for field in fields:
      argv[GAM_argvI] += oargv[pos:field[2]]
      if field[0] == SUB_TYPE:
        if row[field[1]]:
          argv[GAM_argvI] += row[field[1]]
      else:
        if row[field[1]]:
          argv[GAM_argvI] += re.sub(field[4], field[5], row[field[1]])
      pos = field[3]
    argv[GAM_argvI] += oargv[pos:]
  return argv

# gam csv <CSVLoopContent> [warnifnodata]
#	[columndelimiter <Character>] [quotechar <Character>] [fields <FieldNameList>]
#	(matchfield|skipfield <FieldName> <RESearchPattern>)* [showcmds [<Boolean>]]
#	[skiprows <Integer>] [maxrows <Integer>]
#	gam <GAM argument list>
def doCSV(testMode=False):
  filename = getString(Cmd.OB_FILE_NAME)
  if (filename == '-') and (GC.Values[GC.DEBUG_LEVEL] > 0):
    Cmd.Backup()
    usageErrorExit(Msg.BATCH_CSV_LOOP_DASH_DEBUG_INCOMPATIBLE.format(Cmd.CSV_CMD))
  f, csvFile, fieldnames = openCSVFileReader(filename)
  matchFields, skipFields = getMatchSkipFields(fieldnames)
  showCmds = _getShowCommands()
  skipRows = _getSkipRows()
  maxRows = _getMaxRows()
  checkArgumentPresent(Cmd.GAM_CMD, required=True)
  if not Cmd.ArgumentsRemaining():
    missingArgumentExit(Cmd.OB_GAM_ARGUMENT_LIST)
  initial_argv = [Cmd.GAM_CMD]
  if GM.Globals[GM.SECTION] and not GM.Globals[GM.GAM_CFG_SECTION] and not Cmd.PeekArgumentPresent(Cmd.SELECT_CMD):
    initial_argv.extend([Cmd.SELECT_CMD, GM.Globals[GM.SECTION]])
  GAM_argv, subFields = getSubFields(initial_argv, fieldnames)
  if GC.Values[GC.CSV_INPUT_ROW_FILTER] or GC.Values[GC.CSV_INPUT_ROW_DROP_FILTER]:
    CheckInputRowFilterHeaders(fieldnames, GC.Values[GC.CSV_INPUT_ROW_FILTER], GC.Values[GC.CSV_INPUT_ROW_DROP_FILTER])
  items = []
  i = 0
  for row in csvFile:
    if checkMatchSkipFields(row, fieldnames, matchFields, skipFields):
      i += 1
      if skipRows:
        if i <= skipRows:
          continue
        i = 1
        skipRows = 0
      items.append(processSubFields(GAM_argv, row, subFields))
      if maxRows and i >= maxRows:
        break
  closeFile(f)
  if not testMode:
    MultiprocessGAMCommands(items, showCmds)
  else:
    numItems = min(len(items), 10)
    writeStdout(Msg.CSV_FILE_HEADERS.format(filename))
    Ind.Increment()
    for field in fieldnames:
      writeStdout(f'{Ind.Spaces()}{field}\n')
    Ind.Decrement()
    writeStdout(Msg.CSV_SAMPLE_COMMANDS.format(numItems, GAM))
    Ind.Increment()
    for i in range(numItems):
      writeStdout(f'{Ind.Spaces()}{Cmd.QuotedArgumentList(items[i])}\n')
    Ind.Decrement()

def doCSVTest():
  doCSV(testMode=True)

# gam loop <CSVLoopContent> [warnifnodata]
#	[columndelimiter <Character>] [quotechar <Character>] [fields <FieldNameList>]
#	(matchfield|skipfield <FieldName> <RESearchPattern>)* [showcmds [<Boolean>]]
#	[skiprows <Integer>] [maxrows <Integer>]
#	gam <GAM argument list>
def doLoop(loopCmd):
  filename = getString(Cmd.OB_FILE_NAME)
  if (filename == '-') and (GC.Values[GC.DEBUG_LEVEL] > 0):
    Cmd.Backup()
    usageErrorExit(Msg.BATCH_CSV_LOOP_DASH_DEBUG_INCOMPATIBLE.format(Cmd.LOOP_CMD))
  f, csvFile, fieldnames = openCSVFileReader(filename)
  matchFields, skipFields = getMatchSkipFields(fieldnames)
  showCmds = _getShowCommands()
  skipRows = _getSkipRows()
  maxRows = _getMaxRows()
  checkArgumentPresent(Cmd.GAM_CMD, required=True)
  if not Cmd.ArgumentsRemaining():
    missingArgumentExit(Cmd.OB_GAM_ARGUMENT_LIST)
  if GC.Values[GC.CSV_INPUT_ROW_FILTER] or GC.Values[GC.CSV_INPUT_ROW_DROP_FILTER]:
    CheckInputRowFilterHeaders(fieldnames, GC.Values[GC.CSV_INPUT_ROW_FILTER], GC.Values[GC.CSV_INPUT_ROW_DROP_FILTER])
  choice = Cmd.Current().strip().lower()
  if choice == Cmd.LOOP_CMD:
    usageErrorExit(Msg.NESTED_LOOP_CMD_NOT_ALLOWED)
# gam loop ... gam redirect|select|config ... process gam.cfg on each iteration
# gam redirect|select|config ... loop ... gam redirect|select|config ... process gam.cfg on each iteration
# gam loop ... gam !redirect|select|config ... no further processing of gam.cfg
# gam redirect|select|config ... loop ... gam !redirect|select|config ... no further processing of gam.cfg
  processGamCfg = choice in Cmd.GAM_META_COMMANDS
  GAM_argv, subFields = getSubFields([Cmd.GAM_CMD], fieldnames)
  multi = GM.Globals[GM.CSVFILE][GM.REDIRECT_MULTIPROCESS]
  if multi:
    mpManager = multiprocessing.Manager()
    mpQueue, mpQueueHandler = initializeCSVFileQueueHandler(mpManager, None, None)
  else:
    mpQueue = None
  GM.Globals[GM.CSVFILE][GM.REDIRECT_QUEUE] = mpQueue
# Set up command logging at top level only
  if GM.Globals[GM.CMDLOG_LOGGER]:
    LoopGlobals = GM.Globals
  else:
    LoopGlobals = {GM.CMDLOG_LOGGER: None, GM.CMDLOG_HANDLER: None}
    if (GM.Globals[GM.PID] > 0) and GC.Values[GC.CMDLOG]:
      openGAMCommandLog(LoopGlobals, 'looplog')
  if LoopGlobals[GM.CMDLOG_LOGGER]:
    writeGAMCommandLog(LoopGlobals, loopCmd, '*')
  if not showCmds:
    i = 0
    for row in csvFile:
      if checkMatchSkipFields(row, fieldnames, matchFields, skipFields):
        i += 1
        if skipRows:
          if i <= skipRows:
            continue
          i = 1
          skipRows = 0
        item = processSubFields(GAM_argv, row, subFields)
        logCmd = Cmd.QuotedArgumentList(item)
        if i % 100 == 0:
          batchWriteStderr(Msg.PROCESSING_ITEM_N.format(currentISOformatTimeStamp(), i))
        sysRC = ProcessGAMCommand(item, processGamCfg=processGamCfg, inLoop=True)
        if (GM.Globals[GM.PID] > 0) and LoopGlobals[GM.CMDLOG_LOGGER]:
          writeGAMCommandLog(LoopGlobals, logCmd, sysRC)
        if (sysRC > 0) and (GM.Globals[GM.SYSEXITRC] <= HARD_ERROR_RC):
          break
        if maxRows and i >= maxRows:
          break
    closeFile(f)
  else:
    items = []
    i = 0
    for row in csvFile:
      if checkMatchSkipFields(row, fieldnames, matchFields, skipFields):
        i += 1
        if skipRows:
          if i <= skipRows:
            continue
          i = 1
          skipRows = 0
        items.append(processSubFields(GAM_argv, row, subFields))
        if maxRows and i >= maxRows:
          break
    closeFile(f)
    numItems = len(items)
    pid = 0
    for item in items:
      pid += 1
      logCmd = Cmd.QuotedArgumentList(item)
      batchWriteStderr(f'{currentISOformatTimeStamp()},{pid}/{numItems},Start,0,{logCmd}\n')
      sysRC = ProcessGAMCommand(item, processGamCfg=processGamCfg, inLoop=True)
      batchWriteStderr(f'{currentISOformatTimeStamp()},{pid}/{numItems},End,{sysRC},{logCmd}\n')
      if (GM.Globals[GM.PID] > 0) and LoopGlobals[GM.CMDLOG_LOGGER]:
        writeGAMCommandLog(LoopGlobals, logCmd, sysRC)
      if (sysRC > 0) and (GM.Globals[GM.SYSEXITRC] <= HARD_ERROR_RC):
        break
  if (GM.Globals[GM.PID] > 0) and LoopGlobals[GM.CMDLOG_LOGGER]:
    closeGAMCommandLog(LoopGlobals)
  if multi:
    terminateCSVFileQueueHandler(mpQueue, mpQueueHandler)

def _doList(entityList, entityType):
  buildGAPIObject(API.DIRECTORY)
  if GM.Globals[GM.CSV_DATA_DICT]:
    keyField = GM.Globals[GM.CSV_KEY_FIELD]
    dataField = GM.Globals[GM.CSV_DATA_FIELD]
  else:
    keyField = 'Entity'
    dataField = 'Data'
  csvPF = CSVPrintFile(keyField)
  if checkArgumentPresent('todrive'):
    csvPF.GetTodriveParameters()
  if entityList is None:
    entityList = getEntityList(Cmd.OB_ENTITY)
  showData = checkArgumentPresent('data')
  if showData:
    if not entityType:
      itemType, itemList = getEntityToModify(crosAllowed=True)
    else:
      itemType = None
      itemList = getEntityList(Cmd.OB_ENTITY)
    entityItemLists = itemList if isinstance(itemList, dict) else None
    csvPF.AddTitle(dataField)
  else:
    entityItemLists = None
  dataDelimiter = getDelimiter()
  checkForExtraneousArguments()
  _, _, entityList = getEntityArgument(entityList)
  for entity in entityList:
    entityEmail = normalizeEmailAddressOrUID(entity)
    if showData:
      if entityItemLists:
        if entity not in entityItemLists:
          csvPF.WriteRow({keyField: entityEmail})
          continue
        itemList = entityItemLists[entity]
        if itemType == Cmd.ENTITY_USERS:
          for i, item in enumerate(itemList):
            itemList[i] = normalizeEmailAddressOrUID(item)
      if dataDelimiter:
        csvPF.WriteRow({keyField: entityEmail, dataField: dataDelimiter.join(itemList)})
      else:
        for item in itemList:
          csvPF.WriteRow({keyField: entityEmail, dataField: item})
    else:
      csvPF.WriteRow({keyField: entityEmail})
  csvPF.writeCSVfile('Entity')

# gam list [todrive <ToDriveAttribute>*] <EntityList> [data <CrOSTypeEntity>|<UserTypeEntity> [delimiter <Character>]]
def doListType():
  _doList(None, None)

# gam <CrOSTypeEntity> list [todrive <ToDriveAttribute>*] [data <EntityList> [delimiter <Character>]]
def doListCrOS(entityList):
  _doList(entityList, Cmd.ENTITY_CROS)

# gam <UserTypeEntity> list [todrive <ToDriveAttribute>*] [data <EntityList> [delimiter <Character>]]
def doListUser(entityList):
  _doList(entityList, Cmd.ENTITY_USERS)

def _showCount(entityList, entityType):
  buildGAPIObject(API.DIRECTORY)
  checkForExtraneousArguments()
  _, count, entityList = getEntityArgument(entityList)
  actionPerformedNumItems(count, entityType)

# gam <CrOSTypeEntity> show count
def showCountCrOS(entityList):
  _showCount(entityList, Ent.CHROME_DEVICE)

# gam <UserTypeEntity> show count
def showCountUser(entityList):
  _showCount(entityList, Ent.USER)

VALIDEMAIL_PATTERN = re.compile(r'^[^@]+@[^@]+\.[^@]+$')

def _getValidateLoginHint(login_hint, projectId=None):
  while True:
    if not login_hint:
      if not projectId:
        login_hint = readStdin(Msg.ENTER_GSUITE_ADMIN_EMAIL_ADDRESS).strip()
      else:
        login_hint = readStdin(Msg.ENTER_MANAGE_GCP_PROJECT_EMAIL_ADDRESS.format(projectId)).strip()
    if login_hint.find('@') == -1 and GC.Values[GC.DOMAIN]:
      login_hint = f'{login_hint}@{GC.Values[GC.DOMAIN]}'
    if VALIDEMAIL_PATTERN.match(login_hint):
      return login_hint
    sys.stdout.write(f'{ERROR_PREFIX}Invalid email address: {login_hint}\n')
    login_hint = None

def getOAuthClientIDAndSecret():
  cs_data = readFile(GC.Values[GC.CLIENT_SECRETS_JSON], continueOnError=True, displayError=True)
  if not cs_data:
    invalidClientSecretsJsonExit(Msg.NO_DATA)
  try:
    cs_json = json.loads(cs_data)
    if not cs_json:
      systemErrorExit(CLIENT_SECRETS_JSON_REQUIRED_RC, Msg.NO_CLIENT_ACCESS_CREATE_UPDATE_ALLOWED)
    return (cs_json['installed']['client_id'], cs_json['installed']['client_secret'])
  except (IndexError, KeyError, SyntaxError, TypeError, ValueError) as e:
    invalidClientSecretsJsonExit(str(e))

def getScopesFromUser(scopesList, clientAccess, currentScopes=None):
  OAUTH2_CMDS = ['s', 'u', 'e', 'c']
  oauth2_menu = ''
  numScopes = len(scopesList)
  for a_scope in scopesList:
    oauth2_menu += f"[%%s] %2d)  {a_scope['name']}"
    if a_scope['subscopes']:
      oauth2_menu += f' (supports {" and ".join(a_scope["subscopes"])})'
    oauth2_menu += '\n'
  oauth2_menu += '''
Select an unselected scope [ ] by entering a number; yields [*]
For scopes that support readonly, enter a number and an 'r' to grant read-only access; yields [R]
For scopes that support action, enter a number and an 'a' to grant action-only access; yields [A]
Clear read-only access [R] or action-only access [A] from a scope by entering a number; yields [*]
Unselect a selected scope [*] by entering a number; yields [ ]
Select all default scopes by entering an 's'; yields [*] for default scopes, [ ] for others
Unselect all scopes by entering a 'u'; yields [ ] for all scopes
Exit without changes/authorization by entering an 'e'
Continue to authorization by entering a 'c'
'''
  if clientAccess:
    oauth2_menu += '''  Note, if all scopes are selected, Google will probably generate an authorization error
'''
  menu = oauth2_menu % tuple(range(numScopes))
  selectedScopes = ['*'] * numScopes
  if currentScopes is None and clientAccess:
    lock = FileLock(GM.Globals[GM.OAUTH2_TXT_LOCK])
    with lock:
      _, credentials = getOauth2TxtCredentials(exitOnError=False)
      if credentials and credentials.scopes is not None:
        currentScopes = sorted(credentials.scopes)
  if currentScopes is not None:
    if clientAccess:
      i = 0
      for a_scope in scopesList:
        selectedScopes[i] = ' '
        possibleScope = a_scope['scope']
        for currentScope in currentScopes:
          if currentScope == possibleScope:
            selectedScopes[i] = '*'
            break
          if 'readonly' in a_scope['subscopes']:
            if currentScope == possibleScope+'.readonly':
              selectedScopes[i] = 'R'
              break
          if 'action' in a_scope['subscopes']:
            if currentScope == possibleScope+'.action':
              selectedScopes[i] = 'A'
              break
        i += 1
    else:
      i = 0
      for a_scope in scopesList:
        selectedScopes[i] = ' '
        api = a_scope['api']
        possibleScope = a_scope['scope']
        if api in currentScopes:
          for scope in currentScopes[api]:
            if scope == possibleScope:
              selectedScopes[i] = '*'
              break
            if 'readonly' in a_scope['subscopes']:
              if (scope == possibleScope+'.readonly') or (scope == a_scope.get('roscope')):
                selectedScopes[i] = 'R'
                break
        i += 1
  else:
    i = 0
    for a_scope in scopesList:
      if a_scope.get('offByDefault'):
        selectedScopes[i] = ' '
      elif a_scope.get('roByDefault'):
        selectedScopes[i] = 'R'
      else:
        selectedScopes[i] = '*'
      i += 1
  prompt = f'\nPlease enter 0-{numScopes-1}[a|r] or {"|".join(OAUTH2_CMDS)}: '
  while True:
    os.system(['clear', 'cls'][sys.platform.startswith('win')])
    sys.stdout.write(menu % tuple(selectedScopes))
    while True:
      choice = readStdin(prompt)
      if choice:
        selection = choice.lower()
        if selection.find('r') >= 0:
          mode = 'R'
          selection = selection.replace('r', '')
        elif selection.find('a') >= 0:
          mode = 'A'
          selection = selection.replace('a', '')
        else:
          mode = ' '
        if selection and selection.isdigit():
          selection = int(selection)
        if isinstance(selection, int) and selection < numScopes:
          if mode == 'R':
            if 'readonly' not in scopesList[selection]['subscopes']:
              sys.stdout.write(f'{ERROR_PREFIX}Scope {selection} does not support read-only mode!\n')
              continue
          elif mode == 'A':
            if 'action' not in scopesList[selection]['subscopes']:
              sys.stdout.write(f'{ERROR_PREFIX}Scope {selection} does not support action-only mode!\n')
              continue
          elif selectedScopes[selection] != '*':
            mode = '*'
          else:
            mode = ' '
          selectedScopes[selection] = mode
          break
        if isinstance(selection, str) and selection in OAUTH2_CMDS:
          if selection == 's':
            i = 0
            for a_scope in scopesList:
              selectedScopes[i] = ' ' if a_scope.get('offByDefault', False) else '*'
              i += 1
          elif selection == 'u':
            for i in range(numScopes):
              selectedScopes[i] = ' '
          elif selection == 'e':
            return None
          break
        sys.stdout.write(f'{ERROR_PREFIX}Invalid input "{choice}"\n')
    if selection == 'c':
      break
  return selectedScopes

def _localhost_to_ip():
  '''returns IPv4 or IPv6 loopback address which localhost resolves to.
     If localhost does not resolve to valid loopback IP address then returns
     127.0.0.1'''
  # TODO gethostbyname() will only ever return ipv4
  # find a way to support IPv6 here and get preferred IP
  # note that IPv6 may be broken on some systems also :-(
  # for now IPv4 should do.
  local_ip = socket.gethostbyname('localhost')
#  local_ip = socket.getaddrinfo('localhost', None)[0][-1][0] # works with ipv6, makes wsgiref fail
  if not ipaddress.ip_address(local_ip).is_loopback:
    local_ip = '127.0.0.1'
  return local_ip

def _waitForHttpClient(d):
  wsgi_app = google_auth_oauthlib.flow._RedirectWSGIApp(Msg.AUTHENTICATION_FLOW_COMPLETE_CLOSE_BROWSER.format(GAM))
  wsgiref.simple_server.WSGIServer.allow_reuse_address = False
  # Convert hostname to IP since apparently binding to the IP
  # reduces odds of firewall blocking us
  local_ip = _localhost_to_ip()
  for port in range(8080, 8099):
    try:
      local_server = wsgiref.simple_server.make_server(
        local_ip,
        port,
        wsgi_app,
        handler_class=wsgiref.simple_server.WSGIRequestHandler
        )
      break
    except OSError:
      pass
  redirect_uri_format = "http://{}:{}/" if d['trailing_slash'] else "http://{}:{}"
  # provide redirect_uri to main process so it can formulate auth_url
  d['redirect_uri'] = redirect_uri_format.format(*local_server.server_address)
  # wait until main process provides auth_url
  # so we can open it in web browser.
  while 'auth_url' not in d:
    time.sleep(0.1)
  if d['open_browser']:
    webbrowser.open(d['auth_url'], new=1, autoraise=True)
  try:
    local_server.handle_request()
    authorization_response = wsgi_app.last_request_uri.replace("http", "https")
    d['code'] = authorization_response
  except:
    pass
  local_server.server_close()

def _waitForUserInput(d):
  sys.stdin = open(0, DEFAULT_FILE_READ_MODE, encoding=UTF8)
  d['code'] = readStdin(Msg.ENTER_VERIFICATION_CODE_OR_URL)

class _GamOauthFlow(google_auth_oauthlib.flow.InstalledAppFlow):
  def run_dual(self, **kwargs):
    mgr = multiprocessing.Manager()
    d = mgr.dict()
    d['trailing_slash'] = True
    d['open_browser'] = not GC.Values[GC.NO_BROWSER]
    httpClientProcess = multiprocessing.Process(target=_waitForHttpClient, args=(d,))
    userInputProcess = multiprocessing.Process(target=_waitForUserInput, args=(d,))
    httpClientProcess.start()
    # we need to wait until web server starts on avail port
    # so we know redirect_uri to use
    while 'redirect_uri' not in d:
      time.sleep(0.1)
    self.redirect_uri = d['redirect_uri']
    d['auth_url'], _ = super().authorization_url(**kwargs)
    d['auth_url'] = shortenURL(d['auth_url'])
    print(Msg.OAUTH2_GO_TO_LINK_MESSAGE.format(url=d['auth_url']))
    userInputProcess.start()
    userInput = False
    checkHttp = checkUser = True
    alive = 2
    while alive > 0:
      time.sleep(0.1)
      if checkHttp and not httpClientProcess.is_alive():
        if 'code' in d:
          if checkUser:
            userInputProcess.terminate()
          break
        checkHttp = False
        alive -= 1
      if checkUser and not userInputProcess.is_alive():
        userInput = True
        if 'code' in d:
          if checkHttp:
            httpClientProcess.terminate()
          break
        checkUser = False
        alive -= 1
    if 'code' not in d:
      systemErrorExit(SYSTEM_ERROR_RC, Msg.AUTHENTICATION_FLOW_FAILED)
    while True:
      code = d['code']
      if code.startswith('http'):
        parsed_url = urlparse(code)
        parsed_params = parse_qs(parsed_url.query)
        code = parsed_params.get('code', [None])[0]
      try:
        fetch_args = {'code': code}
        if GC.Values[GC.CACERTS_PEM]:
          fetch_args['verify'] = GC.Values[GC.CACERTS_PEM]
        self.fetch_token(**fetch_args)
        break
      except Exception as e:
        if not userInput:
          systemErrorExit(INVALID_TOKEN_RC, str(e))
        stderrErrorMsg(str(e))
        _waitForUserInput(d)
    print(Msg.AUTHENTICATION_FLOW_COMPLETE)
    return self.credentials

class Credentials(google.oauth2.credentials.Credentials):
  """Google OAuth2.0 Credentials with GAM-specific properties and methods."""

  def __init__(self,
               token,
               refresh_token=None,
               id_token=None,
               token_uri=None,
               client_id=None,
               client_secret=None,
               scopes=None,
               quota_project_id=None,
               expiry=None,
               id_token_data=None,
               filename=None):
    """A thread-safe OAuth2.0 credentials object.

    Credentials adds additional utility properties and methods to a
    standard OAuth2.0 credentials object. When used to store credentials on
    disk, it implements a file lock to avoid collision during writes.

    Args:
      token: Optional String, The OAuth 2.0 access token. Can be None if refresh
        information is provided.
      refresh_token: String, The OAuth 2.0 refresh token. If specified,
        credentials can be refreshed.
      id_token: String, The Open ID Connect ID Token.
      token_uri: String, The OAuth 2.0 authorization server's token endpoint
        URI. Must be specified for refresh, can be left as None if the token can
        not be refreshed.
      client_id: String, The OAuth 2.0 client ID. Must be specified for refresh,
        can be left as None if the token can not be refreshed.
      client_secret: String, The OAuth 2.0 client secret. Must be specified for
        refresh, can be left as None if the token can not be refreshed.
      scopes: Sequence[str], The scopes used to obtain authorization.
        This parameter is used by :meth:`has_scopes`. OAuth 2.0 credentials can
          not request additional scopes after authorization. The scopes must be
          derivable from the refresh token if refresh information is provided
          (e.g. The refresh token scopes are a superset of this or contain a
          wild card scope like
            'https://www.googleapis.com/auth/any-api').
      quota_project_id: String, The project ID used for quota and billing. This
        project may be different from the project used to create the
        credentials.
      expiry: datetime.datetime, The time at which the provided token will
        expire.
      id_token_data: Oauth2.0 ID Token data which was previously fetched for
        this access token against the google.oauth2.id_token library.
      filename: String, Path to a file that will be used to store the
        credentials. If provided, a lock file of the same name and a ".lock"
          extension will be created for concurrency controls. Note: New
            credentials are not saved to disk until write() or refresh() are
            called.

    Raises:
      TypeError: If id_token_data is not the required dict type.
    """
    super().__init__(token=token,
                     refresh_token=refresh_token,
                     id_token=id_token,
                     token_uri=token_uri,
                     client_id=client_id,
                     client_secret=client_secret,
                     scopes=scopes,
                     quota_project_id=quota_project_id)

    # Load data not restored by the super class
    self.expiry = expiry
    if id_token_data and not isinstance(id_token_data, dict):
      raise TypeError(f'Expected type id_token_data dict but received {type(id_token_data)}')
    self._id_token_data = id_token_data.copy() if id_token_data else None

    # If a filename is provided, use a lock file to control concurrent access
    # to the resource. If no filename is provided, use a thread lock that has
    # the same interface as FileLock in order to simplify the implementation.
    if filename:
      # Convert relative paths into absolute
      self._filename = os.path.abspath(filename)
    else:
      self._filename = None

  # Use a property to prevent external mutation of the filename.
  @property
  def filename(self):
    return self._filename

  @classmethod
  def from_authorized_user_info_gam(cls, info, filename=None):
    """Generates Credentials from JSON containing authorized user info.

    Args:
      info: Dict, authorized user info in Google format.
      filename: String, the filename used to store these credentials on disk. If
        no filename is provided, the credentials will not be saved to disk.

    Raises:
      ValueError: If missing fields are detected in the info.
    """
    # We need all of these keys
    keys_needed = {'client_id', 'client_secret'}
    # We need 1 or more of these keys
    keys_need_one_of = {'refresh_token', 'auth_token', 'token'}
    missing = keys_needed.difference(info.keys())
    has_one_of = set(info) & keys_need_one_of
    if missing or not has_one_of:
      raise ValueError(
        'Authorized user info was not in the expected format, missing '
        f'fields {", ".join(missing)} and one of {", ".join(keys_need_one_of)}.')

    expiry = info.get('token_expiry')
    if expiry:
      # Convert the raw expiry to datetime
      expiry = datetime.datetime.strptime(expiry, YYYYMMDDTHHMMSSZ_FORMAT)
    id_token_data = info.get('decoded_id_token')

    # Provide backwards compatibility with field names when loading from JSON.
    # Some field names may be different, depending on when/how the credentials
    # were pickled.
    return cls(token=info.get('token', info.get('auth_token', '')),
               refresh_token=info.get('refresh_token', ''),
               id_token=info.get('id_token_jwt', info.get('id_token')),
               token_uri=info.get('token_uri'),
               client_id=info['client_id'],
               client_secret=info['client_secret'],
               scopes=info.get('scopes'),
               quota_project_id=info.get('quota_project_id'),
               expiry=expiry,
               id_token_data=id_token_data,
               filename=filename)

  @classmethod
  def from_google_oauth2_credentials(cls, credentials, filename=None):
    """Generates Credentials from a google.oauth2.Credentials object."""
    info = json.loads(credentials.to_json())
    # Add properties which are not exported with the native to_json() output.
    info['id_token'] = credentials.id_token
    if credentials.expiry:
      info['token_expiry'] = credentials.expiry.strftime(YYYYMMDDTHHMMSSZ_FORMAT)
    info['quota_project_id'] = credentials.quota_project_id

    return cls.from_authorized_user_info_gam(info, filename=filename)

  @classmethod
  def from_client_secrets(cls,
                          client_id,
                          client_secret,
                          scopes,
                          access_type='offline',
                          login_hint=None,
                          filename=None,
                          open_browser=True):
    """Runs an OAuth Flow from client secrets to generate credentials.

    Args:
      client_id: String, The OAuth2.0 Client ID.
      client_secret: String, The OAuth2.0 Client Secret.
      scopes: Sequence[str], A list of scopes to include in the credentials.
      access_type: String, 'offline' or 'online'.  Indicates whether your
        application can refresh access tokens when the user is not present at
        the browser. Valid parameter values are online, which is the default
        value, and offline.  Set the value to offline if your application needs
        to refresh access tokens when the user is not present at the browser.
        This is the method of refreshing access tokens described later in this
        document. This value instructs the Google authorization server to return
        a refresh token and an access token the first time that your application
        exchanges an authorization code for tokens.
      login_hint: String, The email address that will be displayed on the Google
        login page as a hint for the user to login to the correct account.
      filename: String, the path to a file to use to save the credentials.
      open_browser: Boolean: whether or not GAM should try to open the browser
        automatically.

    Returns:
      Credentials
    """
    client_config = {
      'installed': {
        'client_id': client_id,
        'client_secret': client_secret,
        'redirect_uris': ['http://localhost'],
        'auth_uri': API.GOOGLE_OAUTH2_ENDPOINT,
        'token_uri': API.GOOGLE_OAUTH2_TOKEN_ENDPOINT,
        }
    }

    flow = _GamOauthFlow.from_client_config(client_config,
                                            scopes,
                                            autogenerate_code_verifier=True)
    flow_kwargs = {'access_type': access_type,
                   'open_browser': open_browser}
    if login_hint:
      flow_kwargs['login_hint'] = login_hint
    flow.run_dual(**flow_kwargs)
    return cls.from_google_oauth2_credentials(flow.credentials, filename=filename)

  def to_json(self, strip=None):
    """Creates a JSON representation of a Credentials.

    Args:
        strip: Sequence[str], Optional list of members to exclude from the
          generated JSON.

    Returns:
        str: A JSON representation of this instance, suitable to pass to
             from_json().
    """
    expiry = self.expiry.strftime(YYYYMMDDTHHMMSSZ_FORMAT) if self.expiry else None
    prep = {
      'token': self.token,
      'refresh_token': self.refresh_token,
      'token_uri': self.token_uri,
      'client_id': self.client_id,
      'client_secret': self.client_secret,
      'id_token': self.id_token,
      # Google auth doesn't currently give us scopes back on refresh.
      # 'scopes': sorted(self.scopes),
      'token_expiry': expiry,
      'decoded_id_token': self._id_token_data,
      }

    # Remove empty entries
    prep = {k: v for k, v in prep.items() if v is not None}

    # Remove entries that explicitly need to be removed
    if strip is not None:
      prep = {k: v for k, v in prep.items() if k not in strip}

    return json.dumps(prep, indent=2, sort_keys=True)

def doOAuthRequest(currentScopes, login_hint, verifyScopes=False):
  client_id, client_secret = getOAuthClientIDAndSecret()
  scopesList = API.getClientScopesList(GC.Values[GC.TODRIVE_CLIENTACCESS])
  if not currentScopes or verifyScopes:
    selectedScopes = getScopesFromUser(scopesList, True, currentScopes)
    if selectedScopes is None:
      return False
    scopes = set(API.REQUIRED_SCOPES)
    i = 0
    for scope in scopesList:
      if selectedScopes[i] == '*':
        if scope['scope']:
          scopes.add(scope['scope'])
      elif selectedScopes[i] == 'R':
        scopes.add(f'{scope["scope"]}.readonly')
      elif selectedScopes[i] == 'A':
        scopes.add(f'{scope["scope"]}.action')
      i += 1
  else:
    scopes = set(currentScopes+API.REQUIRED_SCOPES)
  if API.STORAGE_READWRITE_SCOPE in scopes:
    scopes.discard(API.STORAGE_READONLY_SCOPE)
  login_hint = _getValidateLoginHint(login_hint)
# Needs to be set so oauthlib doesn't puke when Google changes our scopes
  os.environ['OAUTHLIB_RELAX_TOKEN_SCOPE'] = 'true'
  credentials = Credentials.from_client_secrets(
    client_id,
    client_secret,
    scopes=list(scopes),
    access_type='offline',
    login_hint=login_hint,
    open_browser=not GC.Values[GC.NO_BROWSER])
  lock = FileLock(GM.Globals[GM.OAUTH2_TXT_LOCK])
  with lock:
    writeClientCredentials(credentials, GC.Values[GC.OAUTH2_TXT])
  entityActionPerformed([Ent.OAUTH2_TXT_FILE, GC.Values[GC.OAUTH2_TXT]])
  return True

# gam oauth|oauth2 create|request [<EmailAddress>]
# gam oauth|oauth2 create|request [admin <EmailAddress>] [scope|scopes <APIScopeURLList>]
def doOAuthCreate():
  if not Cmd.PeekArgumentPresent(['admin', 'scope', 'scopes']):
    login_hint = getEmailAddress(noUid=True, optional=True)
    scopes = None
    checkForExtraneousArguments()
  else:
    login_hint = None
    scopes = []
    scopesList = API.getClientScopesList(GC.Values[GC.TODRIVE_CLIENTACCESS])
    while Cmd.ArgumentsRemaining():
      myarg = getArgument()
      if myarg == 'admin':
        login_hint = getEmailAddress(noUid=True)
      elif myarg in {'scope', 'scopes'}:
        for uscope in getString(Cmd.OB_API_SCOPE_URL_LIST).lower().replace(',', ' ').split():
          if uscope in {'openid', 'email', API.USERINFO_EMAIL_SCOPE, 'profile', API.USERINFO_PROFILE_SCOPE}:
            continue
          for scope in scopesList:
            if ((uscope == scope['scope']) or
                (uscope.endswith('.action') and 'action' in scope['subscopes']) or
                (uscope.endswith('.readonly') and 'readonly' in scope['subscopes'])):
              scopes.append(uscope)
              break
          else:
            invalidChoiceExit(uscope, API.getClientScopesURLs(GC.Values[GC.TODRIVE_CLIENTACCESS]), True)
      else:
        unknownArgumentExit()
    if len(scopes) == 0:
      scopes = None
  doOAuthRequest(scopes, login_hint)

def exitIfNoOauth2Txt():
  if not os.path.isfile(GC.Values[GC.OAUTH2_TXT]):
    entityActionNotPerformedWarning([Ent.OAUTH2_TXT_FILE, GC.Values[GC.OAUTH2_TXT]], Msg.DOES_NOT_EXIST)
    sys.exit(GM.Globals[GM.SYSEXITRC])

# gam oauth|oauth2 delete|revoke
def doOAuthDelete():
  checkForExtraneousArguments()
  exitIfNoOauth2Txt()
  lock = FileLock(GM.Globals[GM.OAUTH2_TXT_LOCK], timeout=10)
  with lock:
    _, credentials = getOauth2TxtCredentials(noScopes=True)
    if not credentials:
      return
    entityType = Ent.OAUTH2_TXT_FILE
    entityName = GC.Values[GC.OAUTH2_TXT]
    sys.stdout.write(f'{Ent.Singular(entityType)}: {entityName}, will be Deleted in 3...')
    sys.stdout.flush()
    time.sleep(1)
    sys.stdout.write('2...')
    sys.stdout.flush()
    time.sleep(1)
    sys.stdout.write('1...')
    sys.stdout.flush()
    time.sleep(1)
    sys.stdout.write('boom!\n')
    sys.stdout.flush()
    httpObj = getHttpObj()
    params = {'token': credentials.refresh_token}
    revoke_uri = f'https://accounts.google.com/o/oauth2/revoke?{urlencode(params)}'
    httpObj.request(revoke_uri, 'GET')
    deleteFile(GC.Values[GC.OAUTH2_TXT], continueOnError=True)
    entityActionPerformed([entityType, entityName])

# gam oauth|oauth2 info|verify [showsecret] [accesstoken <AccessToken> idtoken <IDToken>] [showdetails]
def doOAuthInfo():
  credentials = access_token = id_token = None
  showDetails = showSecret = False
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'accesstoken':
      access_token = getString(Cmd.OB_ACCESS_TOKEN)
    elif myarg == 'idtoken':
      id_token = getString(Cmd.OB_ID_TOKEN)
    elif myarg == 'showdetails':
      showDetails = True
    elif myarg == 'showsecret':
      showSecret = True
    else:
      unknownArgumentExit()
  exitIfNoOauth2Txt()
  if not access_token and not id_token:
    credentials = getClientCredentials(noScopes=True)
    access_token = credentials.token
    printEntity([Ent.OAUTH2_TXT_FILE, GC.Values[GC.OAUTH2_TXT]])
  oa2 = buildGAPIObject(API.OAUTH2)
  try:
    token_info = callGAPI(oa2, 'tokeninfo',
                          throwReasons=[GAPI.INVALID],
                          access_token=access_token, id_token=id_token)
  except GAPI.invalid as e:
    entityActionFailedExit([Ent.OAUTH2_TXT_FILE, GC.Values[GC.OAUTH2_TXT]], str(e))
  if 'issued_to' in token_info:
    printKeyValueList(['Client ID', token_info['issued_to']])
  if credentials is not None and showSecret:
    printKeyValueList(['Secret', credentials.client_secret])
  if 'scope' in token_info:
    scopes = token_info['scope'].split(' ')
    printKeyValueList(['Scopes', len(scopes)])
    Ind.Increment()
    for scope in sorted(scopes):
      printKeyValueList([scope])
    Ind.Decrement()
  if 'email' in token_info:
    printKeyValueList(['Google Workspace Admin', f'{token_info["email"]}'])
  if 'expires_in' in token_info:
    printKeyValueList(['Expires', ISOformatTimeStamp((datetime.datetime.now()+datetime.timedelta(seconds=token_info['expires_in'])).replace(tzinfo=GC.Values[GC.TIMEZONE]))])
  if showDetails:
    for k, v in sorted(token_info.items()):
      if k not in  ['email', 'expires_in', 'issued_to', 'scope']:
        printKeyValueList([k, v])
  printBlankLine()

# gam oauth|oauth2 update [<EmailAddress>]
# gam oauth|oauth2 update [admin <EmailAddress>]
def doOAuthUpdate():
  if Cmd.PeekArgumentPresent(['admin']):
    Cmd.Advance()
    login_hint = getEmailAddress(noUid=True)
  else:
    login_hint = getEmailAddress(noUid=True, optional=True)
  checkForExtraneousArguments()
  exitIfNoOauth2Txt()
  lock = FileLock(GM.Globals[GM.OAUTH2_TXT_LOCK])
  with lock:
    jsonData = readFile(GC.Values[GC.OAUTH2_TXT], continueOnError=True, displayError=False)
  if not jsonData:
    invalidOauth2TxtExit(Msg.NO_DATA)
  try:
    jsonDict = json.loads(jsonData)
    if 'client_id' in jsonDict:
      if 'scopes' in jsonDict:
        currentScopes = jsonDict['scopes']
      else:
        currentScopes = API.getClientScopesURLs(GC.Values[GC.TODRIVE_CLIENTACCESS])
    else:
      currentScopes = []
  except (AttributeError, IndexError, KeyError, SyntaxError, TypeError, ValueError) as e:
    invalidOauth2TxtExit(str(e))
  if not doOAuthRequest(currentScopes, login_hint, verifyScopes=True):
    entityActionNotPerformedWarning([Ent.OAUTH2_TXT_FILE, GC.Values[GC.OAUTH2_TXT]], Msg.USER_CANCELLED)
    sys.exit(GM.Globals[GM.SYSEXITRC])

# gam oauth|oauth2 refresh
def doOAuthRefresh():
  checkForExtraneousArguments()
  exitIfNoOauth2Txt()
  getClientCredentials(forceRefresh=True, forceWrite=True, filename=GC.Values[GC.OAUTH2_TXT], refreshOnly=True)
  entityActionPerformed([Ent.OAUTH2_TXT_FILE, GC.Values[GC.OAUTH2_TXT]])

# gam oauth|oauth2 export [<FileName>]
def doOAuthExport():
  if Cmd.ArgumentsRemaining():
    filename = getString(Cmd.OB_FILE_NAME)
    checkForExtraneousArguments()
  else:
    filename = GC.Values[GC.OAUTH2_TXT]
  getClientCredentials(forceRefresh=True, forceWrite=True, filename=filename, refreshOnly=True)
  if filename != '-':
    entityModifierNewValueActionPerformed([Ent.OAUTH2_TXT_FILE, GC.Values[GC.OAUTH2_TXT]], Act.MODIFIER_TO, filename)

def getCRMService(login_hint):
  scopes = [API.CLOUD_PLATFORM_SCOPE]
  client_id = GAM_PROJECT_CREATION_CLIENT_ID
  client_secret = 'qM3dP8f_4qedwzWQE1VR4zzU'
  credentials = Credentials.from_client_secrets(
    client_id,
    client_secret,
    scopes=scopes,
    access_type='online',
    login_hint=login_hint,
    open_browser=not GC.Values[GC.NO_BROWSER])
  httpObj = transportAuthorizedHttp(credentials, http=getHttpObj())
  return (httpObj, getAPIService(API.CLOUDRESOURCEMANAGER, httpObj))

def enableGAMProjectAPIs(httpObj, projectId, login_hint, checkEnabled, i=0, count=0):
  apis = API.PROJECT_APIS[:]
  projectName = f'projects/{projectId}'
  serveu = getAPIService(API.SERVICEUSAGE, httpObj)
  status = True
  if checkEnabled:
    try:
      services = callGAPIpages(serveu.services(), 'list', 'services',
                               throwReasons=[GAPI.NOT_FOUND, GAPI.PERMISSION_DENIED],
                               parent=projectName, filter='state:ENABLED',
                               fields='nextPageToken,services(name)')
      Act.Set(Act.CHECK)
      jcount = len(services)
      entityPerformActionNumItems([Ent.PROJECT, projectId], jcount, Ent.API, i, count)
      Ind.Increment()
      j = 0
      for service in sorted(services, key=lambda k: k['name']):
        j += 1
        if 'name' in service:
          serviceName = service['name'].split('/')[-1]
          if serviceName in apis:
            printEntityKVList([Ent.API, serviceName], ['Already enabled'], j, jcount)
            apis.remove(serviceName)
          else:
            printEntityKVList([Ent.API, serviceName], ['Already enabled (non-GAM which is fine)'], j, jcount)
      Ind.Decrement()
    except (GAPI.notFound, GAPI.permissionDenied) as e:
      entityActionFailedWarning([Ent.PROJECT, projectId], str(e), i, count)
      status = False
  jcount = len(apis)
  if status and jcount > 0:
    Act.Set(Act.ENABLE)
    entityPerformActionNumItems([Ent.PROJECT, projectId], jcount, Ent.API, i, count)
    failed = 0
    Ind.Increment()
    j = 0
    for api in apis:
      j += 1
      serviceName = f'projects/{projectId}/services/{api}'
      while True:
        try:
          callGAPI(serveu.services(), 'enable',
                   throwReasons=[GAPI.FAILED_PRECONDITION, GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED, GAPI.INTERNAL_ERROR],
                   retryReasons=[GAPI.INTERNAL_ERROR],
                   name=serviceName)
          entityActionPerformed([Ent.API, api], j, jcount)
          break
        except GAPI.failedPrecondition as e:
          entityActionFailedWarning([Ent.API, api], str(e), j, jcount)
          readStdin(Msg.ACCEPT_CLOUD_TOS.format(login_hint))
        except (GAPI.forbidden, GAPI.permissionDenied, GAPI.internalError) as e:
          entityActionFailedWarning([Ent.API, api], str(e), j, jcount)
          failed += 1
          break
    Ind.Decrement()
    if not checkEnabled:
      status = failed <= 2
    else:
      status = failed == 0
  return status

# gam enable apis [auto|manual]
def doEnableAPIs():
  automatic = None
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'auto':
      automatic = True
    elif myarg == 'manual':
      automatic = False
    else:
      unknownArgumentExit()
  request = getTLSv1_2Request()
  try:
    _, projectId = google.auth.default(scopes=[API.IAM_SCOPE], request=request)
  except (google.auth.exceptions.DefaultCredentialsError, google.auth.exceptions.RefreshError):
    projectId = readStdin(Msg.WHAT_IS_YOUR_PROJECT_ID).strip()
  while automatic is None:
    a_or_m = readStdin(Msg.ENABLE_PROJECT_APIS_AUTOMATICALLY_OR_MANUALLY).strip().lower()
    if a_or_m.startswith('a'):
      automatic = True
      break
    if a_or_m.startswith('m'):
      automatic = False
      break
    writeStdout(Msg.PLEASE_ENTER_A_OR_M)
  if automatic:
    login_hint = _getValidateLoginHint(None)
    httpObj, _ = getCRMService(login_hint)
    enableGAMProjectAPIs(httpObj, projectId, login_hint, True)
  else:
    apis = API.PROJECT_APIS[:]
    chunk_size = 20
    writeStdout('Using an account with project access, please use ALL of these URLs to enable 20 APIs at a time:\n\n')
    for chunk in range(0, len(apis), chunk_size):
      apiid = ",".join(apis[chunk:chunk+chunk_size])
      url = f'https://console.cloud.google.com/apis/enableflow?apiid={apiid}&project={projectId}'
      writeStdout(f'    {url}\n\n')

def _waitForSvcAcctCompletion(i):
  sleep_time = i*5
  if i > 3:
    sys.stdout.write(Msg.WAITING_FOR_ITEM_CREATION_TO_COMPLETE_SLEEPING.format(Ent.Singular(Ent.SVCACCT), sleep_time))
  time.sleep(sleep_time)

def _grantRotateRights(iam, projectId, service_account, account_type='serviceAccount'):
  body = {'policy': {'bindings': [{'role': 'roles/iam.serviceAccountKeyAdmin',
                                   'members': [f'{account_type}:{service_account}']}]}}
  maxRetries = 10
  kvList = [Ent.PROJECT, projectId, Ent.SVCACCT, service_account]
  printEntityMessage(kvList, Msg.GRANTING_RIGHTS_TO_ROTATE_ITS_OWN_PRIVATE_KEY.format('Granting'))
  for retry in range(1, maxRetries+1):
    try:
      callGAPI(iam.projects().serviceAccounts(), 'setIamPolicy',
               throwReasons=[GAPI.INVALID_ARGUMENT],
               resource=f'projects/{projectId}/serviceAccounts/{service_account}', body=body)
      printEntityMessage(kvList, Msg.GRANTING_RIGHTS_TO_ROTATE_ITS_OWN_PRIVATE_KEY.format('Granted'))
      return True
    except GAPI.invalidArgument as e:
      entityActionFailedWarning(kvList, str(e))
      if 'does not exist' not in str(e) or retry == maxRetries:
        return False
      _waitForSvcAcctCompletion(retry)
    except Exception as e:
      entityActionFailedWarning(kvList, str(e))
      return False

def _createOauth2serviceJSON(httpObj, projectInfo, svcAcctInfo, create_key=True):
  iam = getAPIService(API.IAM, httpObj)
  try:
    service_account = callGAPI(iam.projects().serviceAccounts(), 'create',
                               throwReasons=[GAPI.NOT_FOUND, GAPI.PERMISSION_DENIED, GAPI.ALREADY_EXISTS],
                               name=f'projects/{projectInfo["projectId"]}',
                               body={'accountId': svcAcctInfo['name'],
                                     'serviceAccount': {'displayName': svcAcctInfo['displayName'],
                                                        'description': svcAcctInfo['description']}})
    entityActionPerformed([Ent.PROJECT, projectInfo['projectId'], Ent.SVCACCT, service_account['name'].rsplit('/', 1)[-1]])
  except (GAPI.notFound, GAPI.permissionDenied) as e:
    entityActionFailedWarning([Ent.PROJECT, projectInfo['projectId']], str(e))
    return False
  except GAPI.alreadyExists as e:
    entityActionFailedWarning([Ent.PROJECT, projectInfo['projectId'], Ent.SVCACCT, svcAcctInfo['name']], str(e))
    writeStderr(Msg.RERUN_THE_COMMAND_AND_SPECIFY_A_NEW_SANAME)
    return False
  GM.Globals[GM.SVCACCT_SCOPES_DEFINED] = False
  if create_key and not doProcessSvcAcctKeys(mode='retainexisting', iam=iam,
                                             projectId=service_account['projectId'],
                                             clientEmail=service_account['email'],
                                             clientId=service_account['uniqueId']):
    return False
  sa_email = service_account['name'].rsplit('/', 1)[-1]
  return _grantRotateRights(iam, projectInfo['projectId'], sa_email)

def _createClientSecretsOauth2service(httpObj, login_hint, appInfo, projectInfo, svcAcctInfo, create_key=True):
  def _checkClientAndSecret(csHttpObj, client_id, client_secret):
    post_data = {'client_id': client_id, 'client_secret': client_secret,
                 'code': 'ThisIsAnInvalidCodeOnlyBeingUsedToTestIfClientAndSecretAreValid',
                 'redirect_uri': 'http://127.0.0.1:8080', 'grant_type': 'authorization_code'}
    _, content = csHttpObj.request(API.GOOGLE_OAUTH2_TOKEN_ENDPOINT, 'POST', urlencode(post_data),
                                   headers={'Content-type': 'application/x-www-form-urlencoded'})
    try:
      content = json.loads(content)
    except (IndexError, KeyError, SyntaxError, TypeError, ValueError) as e:
      sys.stderr.write(f'{str(e)}: {content}')
      return False
    if not 'error' in content or not 'error_description' in content:
      sys.stderr.write(f'Unknown error: {content}\n')
      return False
    if content['error'] == 'invalid_grant':
      return True
    if content['error_description'] == 'The OAuth client was not found.':
      sys.stderr.write(Msg.IS_NOT_A_VALID_CLIENT_ID.format(client_id))
      return False
    if content['error_description'] == 'Unauthorized':
      sys.stderr.write(Msg.IS_NOT_A_VALID_CLIENT_SECRET.format(client_secret))
      return False
    sys.stderr.write(f'Unknown error: {content}\n')
    return False

  if not enableGAMProjectAPIs(httpObj, projectInfo['projectId'], login_hint, False):
    return
  sys.stdout.write(Msg.SETTING_GAM_PROJECT_CONSENT_SCREEN_CREATING_CLIENT)
  console_url = f'https://console.cloud.google.com/auth/clients?project={projectInfo["projectId"]}&authuser={login_hint}'
  csHttpObj = getHttpObj()
  while True:
    sys.stdout.write(Msg.CREATE_CLIENT_INSTRUCTIONS.format(console_url, appInfo['applicationTitle'], appInfo['supportEmail']))
    client_id = readStdin(Msg.ENTER_YOUR_CLIENT_ID).strip()
    if not client_id:
      client_id = readStdin('').strip()
    client_secret = readStdin(Msg.ENTER_YOUR_CLIENT_SECRET).strip()
    if not client_secret:
      client_secret = readStdin('').strip()
    client_valid = _checkClientAndSecret(csHttpObj, client_id, client_secret)
    if client_valid:
      break
    sys.stdout.write('\n')
  cs_data = f'''{{
    "installed": {{
        "auth_provider_x509_cert_url": "{API.GOOGLE_AUTH_PROVIDER_X509_CERT_URL}",
        "auth_uri": "{API.GOOGLE_OAUTH2_ENDPOINT}",
        "client_id": "{client_id}",
        "client_secret": "{client_secret}",
        "created_by": "{login_hint}",
        "project_id": "{projectInfo['projectId']}",
        "token_uri": "{API.GOOGLE_OAUTH2_TOKEN_ENDPOINT}"
    }}
}}'''
  writeFile(GC.Values[GC.CLIENT_SECRETS_JSON], cs_data, continueOnError=False)
  sys.stdout.write(Msg.TRUST_GAM_CLIENT_ID.format(GAM, client_id))
  readStdin('')
  if not _createOauth2serviceJSON(httpObj, projectInfo, svcAcctInfo, create_key):
    return
  sys.stdout.write(Msg.YOUR_GAM_PROJECT_IS_CREATED_AND_READY_TO_USE)

def _getProjects(crm, pfilter, returnNF=False):
  try:
    projects = callGAPIpages(crm.projects(), 'search', 'projects',
                             throwReasons=[GAPI.BAD_REQUEST, GAPI.INVALID_ARGUMENT, GAPI.PERMISSION_DENIED],
                             query=pfilter)
    if projects:
      return projects
    if (not pfilter) or pfilter == GAM_PROJECT_FILTER:
      return []
    if pfilter.startswith('id:'):
      projects = [callGAPI(crm.projects(), 'get',
                           throwReasons=[GAPI.BAD_REQUEST, GAPI.INVALID_ARGUMENT, GAPI.PERMISSION_DENIED],
                           name=f'projects/{pfilter[3:]}')]
      if projects or not returnNF:
        return projects
    return []
  except (GAPI.badRequest, GAPI.invalidArgument) as e:
    entityActionFailedExit([Ent.PROJECT, pfilter], str(e))
  except GAPI.permissionDenied:
    if (not pfilter) or (not pfilter.startswith('id:')) or (not returnNF):
      return []
  return [{'projectId': pfilter[3:], 'state': 'NF'}]

def _checkProjectFound(project, i, count):
  if project.get('state', '') != 'NF':
    return True
  entityActionFailedWarning([Ent.PROJECT, project['projectId']], Msg.DOES_NOT_EXIST, i, count)
  return False

def convertGCPFolderNameToID(parent, crm):
  folders = callGAPIpages(crm.folders(), 'search', 'folders',
                          query=f'displayName="{parent}"')
  if not folders:
    entityActionFailedExit([Ent.PROJECT_FOLDER, parent], Msg.NOT_FOUND)
  jcount = len(folders)
  if jcount > 1:
    entityActionNotPerformedWarning([Ent.PROJECT_FOLDER, parent],
                                    Msg.PLEASE_SELECT_ENTITY_TO_PROCESS.format(jcount, Ent.Plural(Ent.PROJECT_FOLDER), 'use in create', 'parent <String>'))
    Ind.Increment()
    j = 0
    for folder in folders:
      j += 1
      printKeyValueListWithCount(['Name', folder['name'], 'ID', folder['displayName']], j, jcount)
    Ind.Decrement()
    systemErrorExit(MULTIPLE_PROJECT_FOLDERS_FOUND_RC, None)
  return folders[0]['name']

PROJECTID_PATTERN = re.compile(r'^[a-z][a-z0-9-]{4,28}[a-z0-9]$')
PROJECTID_FORMAT_REQUIRED = '[a-z][a-z0-9-]{4,28}[a-z0-9]'
def _checkProjectId(projectId):
  if not PROJECTID_PATTERN.match(projectId):
    Cmd.Backup()
    invalidArgumentExit(PROJECTID_FORMAT_REQUIRED)

PROJECTNAME_PATTERN = re.compile('^[a-zA-Z0-9 '+"'"+'"!-]{4,30}$')
PROJECTNAME_FORMAT_REQUIRED = '[a-zA-Z0-9 \'"!-]{4,30}'
def _checkProjectName(projectName):
  if not PROJECTNAME_PATTERN.match(projectName):
    Cmd.Backup()
    invalidArgumentExit(PROJECTNAME_FORMAT_REQUIRED)

def _getSvcAcctInfo(myarg, svcAcctInfo):
  if myarg == 'saname':
    svcAcctInfo['name'] = getString(Cmd.OB_STRING, minLen=6, maxLen=30)
    _checkProjectId(svcAcctInfo['name'])
  elif myarg == 'sadisplayname':
    svcAcctInfo['displayName'] = getString(Cmd.OB_STRING, maxLen=100)
  elif myarg == 'sadescription':
    svcAcctInfo['description'] = getString(Cmd.OB_STRING, maxLen=256)
  else:
    return False
  return True

def _getAppInfo(myarg, appInfo):
  if myarg == 'appname':
    appInfo['applicationTitle'] = getString(Cmd.OB_STRING)
  elif myarg == 'supportemail':
    appInfo['supportEmail'] = getEmailAddress(noUid=True)
  else:
    return False
  return True

def _generateProjectSvcAcctId(prefix):
  return f'{prefix}-{"".join(random.choice(LOWERNUMERIC_CHARS) for _ in range(5))}'

def _getLoginHintProjectInfo(createCmd):
  login_hint = None
  create_key = True
  appInfo = {'applicationTitle': '', 'supportEmail': ''}
  projectInfo = {'projectId': '', 'parent': '', 'name': ''}
  svcAcctInfo = {'name': '', 'displayName': '', 'description': ''}
  if not Cmd.PeekArgumentPresent(['admin', 'appname', 'supportemail', 'project', 'parent',
                                  'projectname', 'saname', 'sadisplayname', 'sadescription',
                                  'algorithm', 'localkeysize', 'validityhours', 'yubikey', 'nokey']):
    login_hint = getString(Cmd.OB_EMAIL_ADDRESS, optional=True)
    if login_hint and login_hint.find('@') == -1:
      Cmd.Backup()
      login_hint = None
    projectInfo['projectId'] = getString(Cmd.OB_STRING, optional=True, minLen=6, maxLen=30).strip()
    if projectInfo['projectId']:
      _checkProjectId(projectInfo['projectId'])
    checkForExtraneousArguments()
  else:
    while Cmd.ArgumentsRemaining():
      myarg = getArgument()
      if myarg == 'admin':
        login_hint = getEmailAddress(noUid=True)
      elif myarg == 'nokey':
        create_key = False
      elif myarg == 'project':
        projectInfo['projectId'] = getString(Cmd.OB_STRING, minLen=6, maxLen=30)
        _checkProjectId(projectInfo['projectId'])
      elif createCmd and myarg == 'parent':
        projectInfo['parent'] = getString(Cmd.OB_STRING)
      elif myarg == 'projectname':
        projectInfo['name'] = getString(Cmd.OB_STRING, minLen=4, maxLen=30)
        _checkProjectName(projectInfo['name'])
      elif _getSvcAcctInfo(myarg, svcAcctInfo):
        pass
      elif _getAppInfo(myarg, appInfo):
        pass
      elif myarg in {'algorithm', 'localkeysize', 'validityhours', 'yubikey'}:
        Cmd.Backup()
        break
      else:
        unknownArgumentExit()
  if not projectInfo['projectId']:
    if createCmd:
      projectInfo['projectId'] = _generateProjectSvcAcctId('gam-project')
    else:
      projectInfo['projectId'] = readStdin(Msg.WHAT_IS_YOUR_PROJECT_ID).strip()
      if not PROJECTID_PATTERN.match(projectInfo['projectId']):
        systemErrorExit(USAGE_ERROR_RC, f'{Cmd.ARGUMENT_ERROR_NAMES[Cmd.ARGUMENT_INVALID][1]} {Cmd.OB_PROJECT_ID}: {Msg.EXPECTED} <{PROJECTID_FORMAT_REQUIRED}>')
  if not projectInfo['name']:
    projectInfo['name'] = 'GAM Project' if not GC.Values[GC.USE_PROJECTID_AS_NAME] else projectInfo['projectId']
  if not svcAcctInfo['name']:
    svcAcctInfo['name'] = projectInfo['projectId']
  if not svcAcctInfo['displayName']:
    svcAcctInfo['displayName'] = projectInfo['name']
  if not svcAcctInfo['description']:
    svcAcctInfo['description'] = svcAcctInfo['displayName']
  login_hint = _getValidateLoginHint(login_hint, projectInfo['projectId'])
  if not appInfo['applicationTitle']:
    appInfo['applicationTitle'] = 'GAM' if not GC.Values[GC.USE_PROJECTID_AS_NAME] else projectInfo['projectId']
  if not appInfo['supportEmail']:
    appInfo['supportEmail'] = login_hint
  httpObj, crm = getCRMService(login_hint)
  if projectInfo['parent'] and not projectInfo['parent'].startswith('organizations/') and not projectInfo['parent'].startswith('folders/'):
    projectInfo['parent'] = convertGCPFolderNameToID(projectInfo['parent'], crm)
  projects = _getProjects(crm, f'id:{projectInfo["projectId"]}')
  if not createCmd:
    if not projects:
      entityActionFailedExit([Ent.USER, login_hint, Ent.PROJECT, projectInfo['projectId']], Msg.DOES_NOT_EXIST)
    if projects[0]['state'] != 'ACTIVE':
      entityActionFailedExit([Ent.USER, login_hint, Ent.PROJECT, projectInfo['projectId']], Msg.NOT_ACTIVE)
  else:
    if projects:
      entityActionFailedExit([Ent.USER, login_hint, Ent.PROJECT, projectInfo['projectId']], Msg.DUPLICATE)
  return (crm, httpObj, login_hint, appInfo, projectInfo, svcAcctInfo, create_key)

def _getCurrentProjectId():
  jsonData = readFile(GC.Values[GC.OAUTH2SERVICE_JSON], continueOnError=True, displayError=False)
  if jsonData:
    try:
      return json.loads(jsonData)['project_id']
    except (IndexError, KeyError, SyntaxError, TypeError, ValueError):
      pass
  jsonData = readFile(GC.Values[GC.CLIENT_SECRETS_JSON], continueOnError=True, displayError=True)
  if not jsonData:
    invalidClientSecretsJsonExit(Msg.NO_DATA)
  try:
    return json.loads(jsonData)['installed']['project_id']
  except (IndexError, KeyError, SyntaxError, TypeError, ValueError) as e:
    invalidClientSecretsJsonExit(str(e))

GAM_PROJECT_FILTER = 'id:gam-project-*'
PROJECTID_FILTER_REQUIRED = '<ProjectIDEntity>'
PROJECTS_CREATESVCACCT_OPTIONS = {'saname', 'sadisplayname', 'sadescription'}
PROJECTS_DELETESVCACCT_OPTIONS = {'saemail', 'saname', 'sauniqueid'}
PROJECTS_PRINTSHOW_OPTIONS = {'showsakeys', 'showiampolicies', 'onememberperrow', 'states', 'todrive', 'delimiter', 'formatjson', 'quotechar'}

def _getLoginHintProjects(createSvcAcctCmd=False, deleteSvcAcctCmd=False, printShowCmd=False, readOnly=False):
  if checkArgumentPresent(['admin']):
    login_hint = getEmailAddress(noUid=True)
  else:
    login_hint = getString(Cmd.OB_EMAIL_ADDRESS, optional=True)
  if login_hint and login_hint.find('@') == -1:
    Cmd.Backup()
    login_hint = None
  if readOnly and login_hint and login_hint != _getAdminEmail():
    readOnly = False
  projectIds = None
  pfilter = getString(Cmd.OB_STRING, optional=True)
  if not pfilter:
    pfilter = 'current' if not printShowCmd else GAM_PROJECT_FILTER
  elif printShowCmd and pfilter in PROJECTS_PRINTSHOW_OPTIONS:
    pfilter = GAM_PROJECT_FILTER
    Cmd.Backup()
  elif createSvcAcctCmd and pfilter in PROJECTS_CREATESVCACCT_OPTIONS:
    pfilter = 'current'
    Cmd.Backup()
  elif deleteSvcAcctCmd and pfilter in PROJECTS_DELETESVCACCT_OPTIONS:
    pfilter = 'current'
    Cmd.Backup()
  elif printShowCmd and pfilter.lower() == 'all':
    pfilter = None
  elif pfilter.lower() == 'current':
    pfilter = 'current'
  elif pfilter.lower() == 'gam':
    pfilter = GAM_PROJECT_FILTER
  elif pfilter.lower() == 'filter':
    pfilter = getString(Cmd.OB_STRING)
  elif pfilter.lower() == 'select':
    projectIds = getEntityList(Cmd.OB_PROJECT_ID_ENTITY, False)
    projectId = None
  elif PROJECTID_PATTERN.match(pfilter):
    pfilter = f'id:{pfilter}'
  elif pfilter.startswith('id:') and PROJECTID_PATTERN.match(pfilter[3:]):
    pass
  else:
    Cmd.Backup()
    invalidArgumentExit(['', 'all|'][printShowCmd]+PROJECTID_FILTER_REQUIRED)
  if not printShowCmd and not createSvcAcctCmd and not deleteSvcAcctCmd:
    checkForExtraneousArguments()
  if projectIds is None:
    if pfilter in {'current', 'id:current'}:
      projectId = _getCurrentProjectId()
    else:
      projectId = f'filter {pfilter or "all"}'
  login_hint = _getValidateLoginHint(login_hint, projectId)
  crm = None
  if readOnly:
    _, crm = buildGAPIServiceObject(API.CLOUDRESOURCEMANAGER, None)
    if crm:
      httpObj = crm._http
  if not crm:
    httpObj, crm = getCRMService(login_hint)
  if projectIds is None:
    if pfilter in {'current', 'id:current'}:
      if not printShowCmd:
        projects = [{'projectId': projectId}]
      else:
        projects = _getProjects(crm, f'id:{projectId}', returnNF=True)
    else:
      projects = _getProjects(crm, pfilter, returnNF=printShowCmd)
  else:
    projects = []
    for projectId in projectIds:
      projects.extend(_getProjects(crm, f'id:{projectId}', returnNF=True))
  return (crm, httpObj, login_hint, projects)

def _checkForExistingProjectFiles(projectFiles):
  for a_file in projectFiles:
    if os.path.exists(a_file):
      systemErrorExit(JSON_ALREADY_EXISTS_RC, Msg.AUTHORIZATION_FILE_ALREADY_EXISTS.format(a_file, Act.ToPerform()))

def getGCPOrg(crm, login_hint, login_domain):
  try:
    getorg = callGAPI(crm.organizations(), 'search',
                      throwReasons=[GAPI.INVALID_ARGUMENT, GAPI.PERMISSION_DENIED],
                      query=f'domain:{login_domain}')
  except (GAPI.invalidArgument, GAPI.permissionDenied) as e:
    entityActionFailedExit([Ent.USER, login_hint, Ent.DOMAIN, login_domain], str(e))
  try:
    organization = getorg['organizations'][0]['name']
    sys.stdout.write(Msg.YOUR_ORGANIZATION_NAME_IS.format(organization))
    return organization
  except (KeyError, IndexError):
    systemErrorExit(3, Msg.YOU_HAVE_NO_RIGHTS_TO_CREATE_PROJECTS_AND_YOU_ARE_NOT_A_SUPER_ADMIN)

# gam create gcpfolder <String>
# gam create gcpfolder [admin <EmailAddress] folder <String>
def doCreateGCPFolder():
  login_hint = None
  if not Cmd.PeekArgumentPresent(['admin', 'folder']):
    name = getString(Cmd.OB_STRING)
    checkForExtraneousArguments()
  else:
    name = ''
    while Cmd.ArgumentsRemaining():
      myarg = getArgument()
      if myarg == 'admin':
        login_hint = getEmailAddress(noUid=True)
      elif myarg == 'folder':
        name = getString(Cmd.OB_STRING)
      else:
        unknownArgumentExit()
    if not name:
      missingChoiceExit('folder')
  login_hint = _getValidateLoginHint(login_hint)
  login_domain = getEmailAddressDomain(login_hint)
  _, crm = getCRMService(login_hint)
  organization = getGCPOrg(crm, login_hint, login_domain)
  try:
    result = callGAPI(crm.folders(), 'create',
                      throwReasons=[GAPI.INVALID_ARGUMENT, GAPI.PERMISSION_DENIED],
                      body={'parent': organization, 'displayName': name})
  except (GAPI.invalidArgument, GAPI.permissionDenied) as e:
    entityActionFailedExit([Ent.USER, login_hint, Ent.GCP_FOLDER, name], str(e))
  entityActionPerformed([Ent.USER, login_hint, Ent.GCP_FOLDER, name, Ent.GCP_FOLDER_NAME, result['name']])

# gam create project [<EmailAddress>] [<ProjectID>]
# gam create project [admin <EmailAddress>] [project <ProjectID>]
#	[appname <String>] [supportemail <EmailAddress>]
#	[projectname <ProjectName>] [parent <String>]
#	[saname <ServiceAccountName>] [sadisplayname <ServiceAccountDisplayName>] [sadescription <ServiceAccountDescription>]
#	[(algorithm KEY_ALG_RSA_1024|KEY_ALG_RSA_2048)|
#	 (localkeysize 1024|2048|4096 [validityhours <Number>])|
#	 (yubikey yubikey_pin yubikey_slot AUTHENTICATION yubikey_serialnumber <String>)|
#	 nokey]
def doCreateProject():
  _checkForExistingProjectFiles([GC.Values[GC.OAUTH2SERVICE_JSON], GC.Values[GC.CLIENT_SECRETS_JSON]])
  sys.stdout.write(Msg.TRUST_GAM_CLIENT_ID.format(GAM_PROJECT_CREATION, GAM_PROJECT_CREATION_CLIENT_ID))
  readStdin('')
  crm, httpObj, login_hint, appInfo, projectInfo, svcAcctInfo, create_key = _getLoginHintProjectInfo(True)
  login_domain = getEmailAddressDomain(login_hint)
  body = {'projectId': projectInfo['projectId'], 'displayName': projectInfo['name']}
  if projectInfo['parent']:
    body['parent'] = projectInfo['parent']
  while True:
    create_again = False
    sys.stdout.write(Msg.CREATING_PROJECT.format(body['displayName']))
    try:
      create_operation = callGAPI(crm.projects(), 'create',
                                  throwReasons=[GAPI.BAD_REQUEST, GAPI.ALREADY_EXISTS,
                                                GAPI.FAILED_PRECONDITION, GAPI.PERMISSION_DENIED],
                                  body=body)
    except (GAPI.badRequest, GAPI.alreadyExists, GAPI.failedPrecondition, GAPI.permissionDenied) as e:
      entityActionFailedExit([Ent.USER, login_hint, Ent.PROJECT, projectInfo['projectId']], str(e))
    operation_name = create_operation['name']
    time.sleep(5) # Google recommends always waiting at least 5 seconds
    for i in range(1, 10):
      sys.stdout.write(Msg.CHECKING_PROJECT_CREATION_STATUS)
      status = callGAPI(crm.operations(), 'get',
                        name=operation_name)
      if 'error' in status:
        if status['error'].get('message', '') == 'No permission to create project in organization':
          sys.stdout.write(Msg.NO_RIGHTS_GOOGLE_CLOUD_ORGANIZATION)
          organization = getGCPOrg(crm, login_hint, login_domain)
          org_policy = callGAPI(crm.organizations(), 'getIamPolicy',
                                resource=organization)
          if 'bindings' not in org_policy:
            org_policy['bindings'] = []
            sys.stdout.write(Msg.LOOKS_LIKE_NO_ONE_HAS_RIGHTS_TO_YOUR_GOOGLE_CLOUD_ORGANIZATION_ATTEMPTING_TO_GIVE_YOU_CREATE_RIGHTS)
          else:
            sys.stdout.write(Msg.THE_FOLLOWING_RIGHTS_SEEM_TO_EXIST)
            for a_policy in org_policy['bindings']:
              if 'role' in a_policy:
                sys.stdout.write(f'  Role: {a_policy["role"]}\n')
              if 'members' in a_policy:
                sys.stdout.write('  Members:\n')
                for member in a_policy['members']:
                  sys.stdout.write(f'    {member}\n')
          my_role = 'roles/resourcemanager.projectCreator'
          sys.stdout.write(Msg.GIVING_LOGIN_HINT_THE_CREATOR_ROLE.format(login_hint, my_role))
          org_policy['bindings'].append({'role': my_role, 'members': [f'user:{login_hint}']})
          callGAPI(crm.organizations(), 'setIamPolicy',
                   resource=organization, body={'policy': org_policy})
          create_again = True
          break
        try:
          if status['error']['details'][0]['violations'][0]['description'] == 'Callers must accept Terms of Service':
            readStdin(Msg.ACCEPT_CLOUD_TOS.format(login_hint))
            create_again = True
            break
        except (IndexError, KeyError):
          pass
        systemErrorExit(1, str(status)+'\n')
      if status.get('done', False):
        break
      sleep_time = min(2 ** i, 60)
      sys.stdout.write(Msg.PROJECT_STILL_BEING_CREATED_SLEEPING.format(sleep_time))
      time.sleep(sleep_time)
    if create_again:
      continue
    if not status.get('done', False):
      systemErrorExit(1, Msg.FAILED_TO_CREATE_PROJECT.format(status))
    elif 'error' in status:
      systemErrorExit(2, status['error']+'\n')
    break
# Try to set policy on project to allow Service Account Key Upload
#  orgp = getAPIService(API.ORGPOLICY, httpObj)
#  projectParent = f"projects/{projectInfo['projectId']}"
#  policyName = f'{projectParent}/policies/iam.managed.disableServiceAccountKeyUpload'
#  try:
#    result = callGAPI(orgp.projects().policies(), 'get',
#                      throwReasons=[GAPI.NOT_FOUND, GAPI.FAILED_PRECONDITION, GAPI.PERMISSION_DENIED],
#                      name=policyName)
#    if result['spec']['rules'][0]['enforce']:
#      callGAPI(orgp.projects().policies(), 'patch',
#               throwReasons=[GAPI.FAILED_PRECONDITION, GAPI.PERMISSION_DENIED],
#               name=policyName, body={'spec': {'rules': [{'enforce': False}]}}, updateMask='policy.spec')
#  except GAPI.notFound:
#    callGAPI(orgp.projects().policies(), 'create',
#             throwReasons=[GAPI.BAD_REQUEST, GAPI.FAILED_PRECONDITION, GAPI.PERMISSION_DENIED],
#             parent=projectParent, body={'name': policyName, 'spec': {'rules': [{'enforce': False}]}})
#  except (GAPI.badRequest, GAPI.failedPrecondition, GAPI.permissionDenied):
#    pass
# Create client_secrets.json and oauth2service.json
  _createClientSecretsOauth2service(httpObj, login_hint, appInfo, projectInfo, svcAcctInfo, create_key)

# gam use project [<EmailAddress>] [<ProjectID>]
# gam use project [admin <EmailAddress>] [project <ProjectID>]
#	[appname <String>] [supportemail <EmailAddress>]
#	[saname <ServiceAccountName>] [sadisplayname <ServiceAccountDisplayName>] [sadescription <ServiceAccountDescription>]
#	[(algorithm KEY_ALG_RSA_1024|KEY_ALG_RSA_2048)|
#	 (localkeysize 1024|2048|4096 [validityhours <Number>])|
#	 (yubikey yubikey_pin yubikey_slot AUTHENTICATION yubikey_serialnumber <String>)]
def doUseProject():
  _checkForExistingProjectFiles([GC.Values[GC.OAUTH2SERVICE_JSON], GC.Values[GC.CLIENT_SECRETS_JSON]])
  _, httpObj, login_hint, appInfo, projectInfo, svcAcctInfo, create_key = _getLoginHintProjectInfo(False)
  _createClientSecretsOauth2service(httpObj, login_hint, appInfo, projectInfo, svcAcctInfo, create_key)

# gam update project [[admin] <EmailAddress>] [<ProjectIDEntity>]
def doUpdateProject():
  _, httpObj, login_hint, projects = _getLoginHintProjects()
  count = len(projects)
  entityPerformActionNumItems([Ent.USER, login_hint], count, Ent.PROJECT)
  Ind.Increment()
  i = 0
  for project in projects:
    i += 1
    if not _checkProjectFound(project, i, count):
      continue
    projectId = project['projectId']
    Act.Set(Act.UPDATE)
    if not enableGAMProjectAPIs(httpObj, projectId, login_hint, True, i, count):
      continue
    iam = getAPIService(API.IAM, httpObj)
    _getSvcAcctData() # needed to read in GM.OAUTH2SERVICE_JSON_DATA
    _grantRotateRights(iam, projectId, GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['client_email'])
  Ind.Decrement()

# gam delete project [[admin] <EmailAddress>] [<ProjectIDEntity>]
def doDeleteProject():
  crm, _, login_hint, projects = _getLoginHintProjects()
  count = len(projects)
  entityPerformActionNumItems([Ent.USER, login_hint], count, Ent.PROJECT)
  Ind.Increment()
  i = 0
  for project in projects:
    i += 1
    if not _checkProjectFound(project, i, count):
      continue
    projectId = project['projectId']
    try:
      callGAPI(crm.projects(), 'delete',
               throwReasons=[GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
               name=project['name'])
      entityActionPerformed([Ent.PROJECT, projectId])
    except (GAPI.forbidden, GAPI.permissionDenied) as e:
      entityActionFailedWarning([Ent.PROJECT, projectId], str(e))
  Ind.Decrement()

PROJECT_TIMEOBJECTS = ['createTime']
PROJECT_STATE_CHOICE_MAP = {
  'all': {'ACTIVE', 'DELETE_REQUESTED'},
  'active': {'ACTIVE'},
  'deleterequested': {'DELETE_REQUESTED'}
  }

# gam print projects [[admin] <EmailAddress>] [all|<ProjectIDEntity>] [todrive <ToDriveAttribute>*]
#	[states all|active|deleterequested] [showiampolicies 0|1|3 [onememberperrow]]
#	[delimiter <Character>] [formatjson [quotechar <Character>]]
# gam show projects [[admin] <EmailAddress>] [all|<ProjectIDEntity>]
#	[states all|active|deleterequested] [showiampolicies 0|1|3]
def doPrintShowProjects():
  def _getProjectPolicies(crm, project, policyBody, i, count):
    try:
      policy = callGAPI(crm.projects(), 'getIamPolicy',
                        throwReasons=[GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                        resource=project['name'], body=policyBody)
      return policy
    except (GAPI.forbidden, GAPI.permissionDenied) as e:
      entityActionFailedWarning([Ent.PROJECT, project['projectId'], Ent.IAM_POLICY, None], str(e), i, count)
    return {}

  readOnly = not Cmd.ArgumentIsAhead('showiampolicies')
  crm, _, login_hint, projects = _getLoginHintProjects(printShowCmd=True, readOnly=readOnly)
  csvPF = CSVPrintFile(['User', 'projectId']) if Act.csvFormat() else None
  FJQC = FormatJSONQuoteChar(csvPF)
  oneMemberPerRow = False
  showIAMPolicies = -1
  lifecycleStates = PROJECT_STATE_CHOICE_MAP['active']
  policy = None
  delimiter = GC.Values[GC.CSV_OUTPUT_FIELD_DELIMITER]
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if csvPF and myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif csvPF and myarg == 'onememberperrow':
      oneMemberPerRow = True
    elif myarg == 'states':
      lifecycleStates = getChoice(PROJECT_STATE_CHOICE_MAP, mapChoice=True)
    elif myarg == 'showiampolicies':
      showIAMPolicies = int(getChoice(['0', '1', '3']))
      policyBody = {'options': {"requestedPolicyVersion": showIAMPolicies}}
    elif myarg == 'delimiter':
      delimiter = getCharacter()
    else:
      FJQC.GetFormatJSONQuoteChar(myarg, True)
  if not csvPF:
    count = len(projects)
    entityPerformActionNumItems([Ent.USER, login_hint], count, Ent.PROJECT)
    Ind.Increment()
    i = 0
    for project in projects:
      i += 1
      if not _checkProjectFound(project, i, count):
        continue
      if project['state'] not in lifecycleStates:
        continue
      projectId = project['projectId']
      if showIAMPolicies >= 0:
        policy = _getProjectPolicies(crm, project, policyBody, i, count)
      printEntity([Ent.PROJECT, projectId], i, count)
      Ind.Increment()
      printKeyValueList(['name', project['name']])
      printKeyValueList(['displayName', project['displayName']])
      for field in ['createTime', 'updateTime', 'deleteTime']:
        if field in project:
          printKeyValueList([field, formatLocalTime(project[field])])
      printKeyValueList(['state', project['state']])
      jcount = len(project.get('labels', []))
      if jcount > 0:
        printKeyValueList(['labels', jcount])
        Ind.Increment()
        for k, v in project['labels'].items():
          printKeyValueList([k, v])
        Ind.Decrement()
      if 'parent' in project:
        printKeyValueList(['parent', project['parent']])
      if policy:
        printKeyValueList([Ent.Singular(Ent.IAM_POLICY), ''])
        Ind.Increment()
        bindings = policy.get('bindings', [])
        jcount = len(bindings)
        printKeyValueList(['version', policy['version']])
        printKeyValueList(['bindings', jcount])
        Ind.Increment()
        j = 0
        for binding in bindings:
          j += 1
          printKeyValueListWithCount(['role', binding['role']], j, jcount)
          Ind.Increment()
          for member in binding.get('members', []):
            printKeyValueList(['member', member])
          if 'condition' in binding:
            printKeyValueList(['condition', ''])
            Ind.Increment()
            for k, v in binding['condition'].items():
              printKeyValueList([k, v])
            Ind.Decrement()
          Ind.Decrement()
        Ind.Decrement()
        Ind.Decrement()
      Ind.Decrement()
    Ind.Decrement()
  else:
    if not FJQC.formatJSON:
      csvPF.AddTitles(['projectId', 'name', 'displayName', 'createTime', 'updateTime', 'deleteTime', 'state'])
      csvPF.SetSortAllTitles()
    count = len(projects)
    i = 0
    for project in projects:
      i += 1
      if not _checkProjectFound(project, i, count):
        continue
      if project['state'] not in lifecycleStates:
        continue
      projectId = project['projectId']
      if showIAMPolicies >= 0:
        policy = _getProjectPolicies(crm, project, policyBody, i, count)
      if FJQC.formatJSON:
        if policy is not None:
          project['policy'] = policy
        row = flattenJSON(project, flattened={'User': login_hint}, timeObjects=PROJECT_TIMEOBJECTS)
        if csvPF.CheckRowTitles(row):
          csvPF.WriteRowNoFilter({'User': login_hint, 'projectId': projectId,
                                  'JSON': json.dumps(cleanJSON(project),
                                                     ensure_ascii=False, sort_keys=True)})
        continue
      row = flattenJSON(project, flattened={'User': login_hint}, timeObjects=PROJECT_TIMEOBJECTS)
      if not policy:
        csvPF.WriteRowTitles(row)
        continue
      row[f'policy{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}version'] = policy['version']
      for binding in policy.get('bindings', []):
        prow = row.copy()
        prow[f'policy{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}role'] = binding['role']
        if 'condition' in binding:
          for k, v in binding['condition'].items():
            prow[f'policy{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}condition{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{k}'] = v
        members = binding.get('members', [])
        if not oneMemberPerRow:
          prow[f'policy{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}members'] = delimiter.join(members)
          csvPF.WriteRowTitles(prow)
        else:
          for member in members:
            mrow = prow.copy()
            mrow[f'policy{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}member'] = member
            csvPF.WriteRowTitles(mrow)
    csvPF.writeCSVfile('Projects')

# gam info currentprojectid
def doInfoCurrentProjectId():
  checkForExtraneousArguments()
  printEntity([Ent.PROJECT_ID, _getCurrentProjectId()])

# gam create svcacct [[admin] <EmailAddress>] [<ProjectIDEntity>]
#	[saname <ServiceAccountName>] [sadisplayname <ServiceAccountDisplayName>] [sadescription <ServiceAccountDescription>]
#	[(algorithm KEY_ALG_RSA_1024|KEY_ALG_RSA_2048)|
#	 (localkeysize 1024|2048|4096 [validityhours <Number>])|
#	 (yubikey yubikey_pin yubikey_slot AUTHENTICATION yubikey_serialnumber <String>)]
def doCreateSvcAcct():
  _checkForExistingProjectFiles([GC.Values[GC.OAUTH2SERVICE_JSON]])
  _, httpObj, login_hint, projects = _getLoginHintProjects(createSvcAcctCmd=True)
  svcAcctInfo = {'name': '', 'displayName': '', 'description': ''}
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if _getSvcAcctInfo(myarg, svcAcctInfo):
      pass
    else:
      unknownArgumentExit()
  if not svcAcctInfo['name']:
    svcAcctInfo['name'] = _generateProjectSvcAcctId('gam-svcacct')
  if not svcAcctInfo['displayName']:
    svcAcctInfo['displayName'] = svcAcctInfo['name']
  if not svcAcctInfo['description']:
    svcAcctInfo['description'] = svcAcctInfo['displayName']
  count = len(projects)
  entityPerformActionSubItemModifierNumItems([Ent.USER, login_hint], Ent.SVCACCT, Act.MODIFIER_TO, count, Ent.PROJECT)
  Ind.Increment()
  i = 0
  for project in projects:
    i += 1
    if not _checkProjectFound(project, i, count):
      continue
    projectInfo = {'projectId': project['projectId']}
    _createOauth2serviceJSON(httpObj, projectInfo, svcAcctInfo)
  Ind.Decrement()

# gam delete svcacct [[admin] <EmailAddress>] [<ProjectIDEntity>]
#	(saemail <ServiceAccountEmail>)|(saname <ServiceAccountName>)|(sauniqueid <ServiceAccountUniqueID>)
def doDeleteSvcAcct():
  _, httpObj, login_hint, projects = _getLoginHintProjects(deleteSvcAcctCmd=True)
  iam = getAPIService(API.IAM, httpObj)
  clientEmail = clientId = clientName = None
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'saemail':
      clientEmail = getEmailAddress(noUid=True)
      clientName = clientId = None
    elif myarg == 'saname':
      clientName = getString(Cmd.OB_STRING, minLen=6, maxLen=30).strip()
      _checkProjectId(clientName)
      clientEmail = clientId = None
    elif myarg == 'sauniqueid':
      clientId = getInteger(minVal=0)
      clientEmail = clientName = None
    else:
      unknownArgumentExit()
  if not clientEmail and not clientId and not clientName:
    missingArgumentExit('email|name|uniqueid')
  count = len(projects)
  entityPerformActionSubItemModifierNumItems([Ent.USER, login_hint], Ent.SVCACCT, Act.MODIFIER_FROM, count, Ent.PROJECT)
  Ind.Increment()
  i = 0
  for project in projects:
    i += 1
    if not _checkProjectFound(project, i, count):
      continue
    projectId = project['projectId']
    try:
      if clientEmail:
        saName = clientEmail
      elif clientName:
        saName = f'{clientName}@{projectId}.iam.gserviceaccount.com'
      else: #clientId
        saName = clientId
      callGAPI(iam.projects().serviceAccounts(), 'delete',
               throwReasons=[GAPI.NOT_FOUND, GAPI.BAD_REQUEST],
               name=f'projects/{projectId}/serviceAccounts/{saName}')
      entityActionPerformed([Ent.PROJECT, projectId, Ent.SVCACCT, saName], i, count)
    except (GAPI.notFound, GAPI.badRequest) as e:
      entityActionFailedWarning([Ent.PROJECT, projectId, Ent.SVCACCT, saName], str(e), i, count)
    Ind.Decrement()

def _getSvcAcctKeyProjectClientFields():
  return (GM.Globals[GM.OAUTH2SERVICE_JSON_DATA].get('private_key_id', ''),
          GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['project_id'],
          GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['client_email'],
          GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['client_id'])

# gam <UserTypeEntity> check serviceaccount (scope|scopes <APIScopeURLList>)* [usecolor]
# gam <UserTypeEntity> update serviceaccount (scope|scopes <APIScopeURLList>)* [usecolor]
def checkServiceAccount(users):
  def printMessage(message):
    writeStdout(Ind.Spaces()+message+'\n')

  def printPassFail(description, result):
    writeStdout(Ind.Spaces()+f'{description:73} {result}'+'\n')

  def authorizeScopes(message):
    long_url = ('https://admin.google.com/ac/owl/domainwidedelegation'
                f'?clientScopeToAdd={",".join(sorted(checkScopes))}'
                f'&clientIdToAdd={service_account}&overwriteClientId=true')
    if GC.Values[GC.DOMAIN]:
      long_url += f'&dn={GC.Values[GC.DOMAIN]}'
    long_url += f'&authuser={_getAdminEmail()}'
    short_url = shortenURL(long_url)
    printLine(message.format('', short_url))

  credentials = getSvcAcctCredentials([API.USERINFO_EMAIL_SCOPE], None, forceOauth=True)
  allScopes = API.getSvcAcctScopes(GC.Values[GC.USER_SERVICE_ACCOUNT_ACCESS_ONLY], Act.Get() == Act.UPDATE)
  checkScopesSet = set()
  saScopes = {}
  checkDeprecatedScopes = True
  useColor = False
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg in {'scope', 'scopes'}:
      checkDeprecatedScopes = False
      for scope in getString(Cmd.OB_API_SCOPE_URL_LIST).lower().replace(',', ' ').split():
        api = API.getSvcAcctScopeAPI(scope)
        if api is not None:
          saScopes.setdefault(api, [])
          saScopes[api].append(scope)
          checkScopesSet.add(scope)
        else:
          invalidChoiceExit(scope, allScopes, True)
    elif myarg == 'usecolor':
      useColor = True
    else:
      unknownArgumentExit()
  if useColor:
    testPass = createGreenText('PASS')
    testFail = createRedText('FAIL')
    testWarn = createYellowText('WARN')
    testDeprecated = createRedText('DEPRECATED')
  else:
    testPass = 'PASS'
    testFail = 'FAIL'
    testWarn = 'WARN'
    testDeprecated = 'DEPRECATED'
  if Act.Get() == Act.CHECK:
    if not checkScopesSet:
      for scope in GM.Globals[GM.SVCACCT_SCOPES].values():
        checkScopesSet.update(scope)
  else:
    if not checkScopesSet:
      scopesList = API.getSvcAcctScopesList(GC.Values[GC.USER_SERVICE_ACCOUNT_ACCESS_ONLY], True)
      selectedScopes = getScopesFromUser(scopesList, False, GM.Globals[GM.SVCACCT_SCOPES] if GM.Globals[GM.SVCACCT_SCOPES_DEFINED] else None)
      if selectedScopes is None:
        return False
      i = 0
      for scope in scopesList:
        if selectedScopes[i] == '*':
          saScopes.setdefault(scope['api'], [])
          saScopes[scope['api']].append(scope['scope'])
          checkScopesSet.add(scope['scope'])
        elif selectedScopes[i] == 'R':
          saScopes.setdefault(scope['api'], [])
          if 'roscope' not in scope:
            saScopes[scope['api']].append(f'{scope["scope"]}.readonly')
            checkScopesSet.add(f'{scope["scope"]}.readonly')
          else:
            saScopes[scope['api']].append(scope['roscope'])
            checkScopesSet.add(scope['roscope'])
        i += 1
    if API.DRIVEACTIVITY in saScopes and API.DRIVE3 in saScopes:
      saScopes[API.DRIVEACTIVITY].append(API.DRIVE_SCOPE)
    if API.DRIVE3 in saScopes:
      saScopes[API.DRIVE2] = saScopes[API.DRIVE3]
    GM.Globals[GM.OAUTH2SERVICE_JSON_DATA][API.OAUTH2SA_SCOPES] = saScopes
    writeFile(GC.Values[GC.OAUTH2SERVICE_JSON],
              json.dumps(GM.Globals[GM.OAUTH2SERVICE_JSON_DATA], ensure_ascii=False, sort_keys=True, indent=2),
              continueOnError=False)
  checkScopes = sorted(checkScopesSet)
  jcount = len(checkScopes)
  printMessage(Msg.SYSTEM_TIME_STATUS)
  offsetSeconds, offsetFormatted = getLocalGoogleTimeOffset()
  if offsetSeconds <= MAX_LOCAL_GOOGLE_TIME_OFFSET:
    timeStatus = testPass
  else:
    timeStatus = testFail
  Ind.Increment()
  printPassFail(Msg.YOUR_SYSTEM_TIME_DIFFERS_FROM_GOOGLE.format(GOOGLE_TIMECHECK_LOCATION, offsetFormatted), timeStatus)
  Ind.Decrement()
  oa2 = buildGAPIObject(API.OAUTH2)
  printMessage(Msg.SERVICE_ACCOUNT_PRIVATE_KEY_AUTHENTICATION)
  # We are explicitly not doing DwD here, just confirming service account can auth
  auth_error = ''
  try:
    request = transportCreateRequest()
    credentials.refresh(request)
    sa_token_info = callGAPI(oa2, 'tokeninfo', access_token=credentials.token)
    if sa_token_info:
      saTokenStatus = testPass
    else:
      saTokenStatus = testFail
  except (httplib2.HttpLib2Error, google.auth.exceptions.TransportError, RuntimeError) as e:
    handleServerError(e)
  except google.auth.exceptions.RefreshError as e:
    saTokenStatus = testFail
    if isinstance(e.args, tuple):
      e = e.args[0]
    auth_error = ' - '+str(e)
  Ind.Increment()
  printPassFail(f'Authentication{auth_error}', saTokenStatus)
  Ind.Decrement()
  if saTokenStatus == testFail:
    invalidOauth2serviceJsonExit(f'Authentication{auth_error}')
  _getSvcAcctData() # needed to read in GM.OAUTH2SERVICE_JSON_DATA
  if API.IAM not in GM.Globals[GM.SVCACCT_SCOPES]:
    GM.Globals[GM.SVCACCT_SCOPES][API.IAM] = [API.IAM_SCOPE]
  key_type = GM.Globals[GM.OAUTH2SERVICE_JSON_DATA].get('key_type', 'default')
  if key_type == 'default':
    printMessage(Msg.SERVICE_ACCOUNT_CHECK_PRIVATE_KEY_AGE)
    _, iam = buildGAPIServiceObject(API.IAM, None)
    currentPrivateKeyId, projectId, _, clientId = _getSvcAcctKeyProjectClientFields()
    name = f'projects/{projectId}/serviceAccounts/{clientId}/keys/{currentPrivateKeyId}'
    Ind.Increment()
    try:
      key = callGAPI(iam.projects().serviceAccounts().keys(), 'get',
                     throwReasons=[GAPI.BAD_REQUEST, GAPI.INVALID, GAPI.NOT_FOUND,
                                   GAPI.PERMISSION_DENIED, GAPI.SERVICE_NOT_AVAILABLE],
                     name=name, fields='validAfterTime')
      key_created, _ = iso8601.parse_date(key['validAfterTime'])
      key_age = todaysTime()-key_created
      printPassFail(Msg.SERVICE_ACCOUNT_PRIVATE_KEY_AGE.format(key_age.days), testWarn if key_age.days > 30 else testPass)
    except GAPI.permissionDenied:
      printMessage(Msg.UPDATE_PROJECT_TO_VIEW_MANAGE_SAKEYS)
      printPassFail(Msg.SERVICE_ACCOUNT_PRIVATE_KEY_AGE.format('UNKNOWN'), testWarn)
    except (GAPI.badRequest, GAPI.invalid, GAPI.notFound) as e:
      entityActionFailedWarning([Ent.PROJECT, GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['project_id'],
                                 Ent.SVCACCT, GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['client_email']],
                                str(e))
      printPassFail(Msg.SERVICE_ACCOUNT_PRIVATE_KEY_AGE.format('UNKNOWN'), testWarn)
    except GAPI.serviceNotAvailable as e:
      entityActionFailedExit([Ent.PROJECT, GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['project_id'],
                              Ent.SVCACCT, GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['client_email']],
                             str(e))
  else:
    printPassFail(Msg.SERVICE_ACCOUNT_SKIPPING_KEY_AGE_CHECK.format(key_type), testPass)
  Ind.Decrement()
  i, count, users = getEntityArgument(users)
  for user in users:
    i += 1
    allScopesPass = True
    user = convertUIDtoEmailAddress(user)
    printKeyValueListWithCount([Msg.DOMAIN_WIDE_DELEGATION_AUTHENTICATION, '',
                                Ent.Singular(Ent.USER), user,
                                Ent.Choose(Ent.SCOPE, jcount), jcount],
                               i, count)
    Ind.Increment()
    j = 0
    for scope in checkScopes:
      j += 1
      # try with and without email scope
      for scopes in [[scope, API.USERINFO_EMAIL_SCOPE], [scope]]:
        try:
          credentials = getSvcAcctCredentials(scopes, user)
          credentials.refresh(request)
          break
        except (httplib2.HttpLib2Error, google.auth.exceptions.TransportError, RuntimeError) as e:
          handleServerError(e)
        except google.auth.exceptions.RefreshError:
          continue
      if credentials.token:
        token_info = callGAPI(oa2, 'tokeninfo', access_token=credentials.token)
        if scope in token_info.get('scope', '').split(' ') and user == token_info.get('email', user).lower():
          scopeStatus = testPass
        else:
          scopeStatus = testFail
          allScopesPass = False
      else:
        scopeStatus = testFail
        allScopesPass = False
      printPassFail(scope, f'{scopeStatus}{currentCount(j, jcount)}')
    Ind.Decrement()
    if checkDeprecatedScopes:
      deprecatedScopes = sorted(API.DEPRECATED_SCOPES)
      jcount = len(deprecatedScopes)
      printKeyValueListWithCount([Msg.DEPRECATED_SCOPES, '',
                                  Ent.Singular(Ent.USER), user,
                                  Ent.Choose(Ent.SCOPE, jcount), jcount],
                                 i, count)
      Ind.Increment()
      j = 0
      for scope in deprecatedScopes:
        j += 1
        # try with and without email scope
        for scopes in [[scope, API.USERINFO_EMAIL_SCOPE], [scope]]:
          try:
            credentials = getSvcAcctCredentials(scopes, user)
            credentials.refresh(request)
            break
          except (httplib2.HttpLib2Error, google.auth.exceptions.TransportError, RuntimeError) as e:
            handleServerError(e)
          except google.auth.exceptions.RefreshError:
            continue
        if credentials.token:
          token_info = callGAPI(oa2, 'tokeninfo', access_token=credentials.token)
          if scope in token_info.get('scope', '').split(' ') and user == token_info.get('email', user).lower():
            scopeStatus = testDeprecated
            allScopesPass = False
          else:
            scopeStatus = testPass
        else:
          scopeStatus = testPass
        printPassFail(scope, f'{scopeStatus}{currentCount(j, jcount)}')
      Ind.Decrement()
    service_account = GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['client_id']
    if allScopesPass:
      if Act.Get() == Act.CHECK:
        printLine(Msg.SCOPE_AUTHORIZATION_PASSED.format(service_account))
      else:
        authorizeScopes(Msg.SCOPE_AUTHORIZATION_UPDATE_PASSED)
    else:
      # Tack on email scope for more accurate checking
      checkScopes.append(API.USERINFO_EMAIL_SCOPE)
      setSysExitRC(SCOPES_NOT_AUTHORIZED_RC)
      authorizeScopes(Msg.SCOPE_AUTHORIZATION_FAILED)
    printBlankLine()

# gam check svcacct <UserTypeEntity> (scope|scopes <APIScopeURLList>)*
# gam update svcacct <UserTypeEntity> (scope|scopes <APIScopeURLList>)*
def doCheckUpdateSvcAcct():
  _, entityList = getEntityToModify(defaultEntityType=Cmd.ENTITY_USER)
  checkServiceAccount(entityList)

def _getSAKeys(iam, projectId, clientEmail, name, keyTypes):
  try:
    keys = callGAPIitems(iam.projects().serviceAccounts().keys(), 'list', 'keys',
                         throwReasons=[GAPI.BAD_REQUEST, GAPI.PERMISSION_DENIED],
                         name=name, fields='*', keyTypes=keyTypes)
    return (True, keys)
  except GAPI.permissionDenied:
    entityActionFailedWarning([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], Msg.UPDATE_PROJECT_TO_VIEW_MANAGE_SAKEYS)
  except GAPI.badRequest as e:
    entityActionFailedWarning([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], str(e))
  return (False, None)

SVCACCT_KEY_TIME_OBJECTS = {'validAfterTime', 'validBeforeTime'}

def _showSAKeys(keys, count, currentPrivateKeyId):
  Ind.Increment()
  i = 0
  for key in keys:
    i += 1
    keyName = key.pop('name').rsplit('/', 1)[-1]
    printKeyValueListWithCount(['name', keyName], i, count)
    Ind.Increment()
    for k, v in sorted(key.items()):
      if k not in SVCACCT_KEY_TIME_OBJECTS:
        printKeyValueList([k, v])
      else:
        printKeyValueList([k, formatLocalTime(v)])
    if keyName == currentPrivateKeyId:
      printKeyValueList(['usedToAuthenticateThisRequest', True])
    Ind.Decrement()
  Ind.Decrement()

SVCACCT_DISPLAY_FIELDS = ['displayName', 'description', 'oauth2ClientId', 'uniqueId', 'disabled']
SVCACCT_KEY_TYPE_CHOICE_MAP = {
  'all': None,
  'system': 'SYSTEM_MANAGED',
  'systemmanaged': 'SYSTEM_MANAGED',
  'user': 'USER_MANAGED',
  'usermanaged': 'USER_MANAGED'
  }

# gam print svcaccts [[admin] <EmailAddress>] [all|<ProjectIDEntity>]
#	[showsakeys all|system|user]
#	[todrive <ToDriveAttribute>*] [formatjson [quotechar <Character>]]
# gam show svcaccts [<EmailAddress>] [all|<ProjectIDEntity>]
#	[showsakeys all|system|user]
def doPrintShowSvcAccts():
  _, httpObj, login_hint, projects = _getLoginHintProjects(printShowCmd=True, readOnly=False)
  csvPF = CSVPrintFile(['User', 'projectId']) if Act.csvFormat() else None
  FJQC = FormatJSONQuoteChar(csvPF)
  iam = getAPIService(API.IAM, httpObj)
  keyTypes = None
  showSAKeys = False
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if csvPF and myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif myarg == 'showsakeys':
      keyTypes = getChoice(SVCACCT_KEY_TYPE_CHOICE_MAP, mapChoice=True)
      showSAKeys = True
    else:
      FJQC.GetFormatJSONQuoteChar(myarg, True)
  count = len(projects)
  if not csvPF:
    entityPerformActionSubItemModifierNumItems([Ent.USER, login_hint], Ent.SVCACCT, Act.MODIFIER_FOR, count, Ent.PROJECT)
  else:
    csvPF.AddTitles(['projectId']+SVCACCT_DISPLAY_FIELDS)
    csvPF.SetSortAllTitles()
  i = 0
  for project in projects:
    i += 1
    if not _checkProjectFound(project, i, count):
      continue
    projectId = project['projectId']
    if csvPF:
      printGettingAllEntityItemsForWhom(Ent.SVCACCT, projectId, i, count)
    if project['state'] != 'ACTIVE':
      entityActionNotPerformedWarning([Ent.PROJECT, projectId], Msg.DELETED, i, count)
      continue
    try:
      svcAccts = callGAPIpages(iam.projects().serviceAccounts(), 'list', 'accounts',
                               throwReasons=[GAPI.NOT_FOUND, GAPI.PERMISSION_DENIED],
                               name=f'projects/{projectId}')
      jcount = len(svcAccts)
      if not csvPF:
        entityPerformActionNumItems([Ent.PROJECT, projectId], jcount, Ent.SVCACCT, i, count)
        Ind.Increment()
        j = 0
        for svcAcct in svcAccts:
          j += 1
          printKeyValueListWithCount(['email', svcAcct['email']], j, jcount)
          Ind.Increment()
          for field in SVCACCT_DISPLAY_FIELDS:
            if field in svcAcct:
              printKeyValueList([field, svcAcct[field]])
          if showSAKeys:
            name = f"projects/{projectId}/serviceAccounts/{svcAcct['oauth2ClientId']}"
            status, keys = _getSAKeys(iam, projectId, svcAcct['email'], name, keyTypes)
            if status:
              kcount = len(keys)
              if kcount > 0:
                printKeyValueList([Ent.Choose(Ent.SVCACCT_KEY, kcount), kcount])
                _showSAKeys(keys, kcount, '')
          Ind.Decrement()
        Ind.Decrement()
      else:
        for svcAcct in svcAccts:
          if showSAKeys:
            name = f"projects/{projectId}/serviceAccounts/{svcAcct['oauth2ClientId']}"
            status, keys = _getSAKeys(iam, projectId, svcAcct['email'], name, keyTypes)
            if status:
              svcAcct['keys'] = keys
          row = flattenJSON(svcAcct, flattened={'User': login_hint}, timeObjects=SVCACCT_KEY_TIME_OBJECTS)
          if not FJQC.formatJSON:
            csvPF.WriteRowTitles(row)
          elif csvPF.CheckRowTitles(row):
            csvPF.WriteRowNoFilter({'User': login_hint, 'projectId': projectId,
                                    'JSON': json.dumps(cleanJSON(svcAcct, timeObjects=SVCACCT_KEY_TIME_OBJECTS),
                                                       ensure_ascii=False, sort_keys=True)})
    except (GAPI.notFound, GAPI.permissionDenied) as e:
      entityActionFailedWarning([Ent.PROJECT, projectId], str(e), i, count)
  if csvPF:
    csvPF.writeCSVfile('Service Accounts')

def _generatePrivateKeyAndPublicCert(projectId, clientEmail, name, key_size, b64enc_pub=True, validityHours=0):
  if projectId:
    printEntityMessage([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], Msg.GENERATING_NEW_PRIVATE_KEY)
  else:
    writeStdout(Msg.GENERATING_NEW_PRIVATE_KEY+'\n')
  private_key = rsa.generate_private_key(public_exponent=65537, key_size=key_size, backend=default_backend())
  private_pem = private_key.private_bytes(encoding=serialization.Encoding.PEM,
                                          format=serialization.PrivateFormat.PKCS8,
                                          encryption_algorithm=serialization.NoEncryption()).decode()

  if projectId:
    printEntityMessage([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], Msg.EXTRACTING_PUBLIC_CERTIFICATE)
  else:
    writeStdout(Msg.EXTRACTING_PUBLIC_CERTIFICATE+'\n')
  public_key = private_key.public_key()
  builder = x509.CertificateBuilder()
  # suppress cryptography warnings on service account email length
  with warnings.catch_warnings():
    warnings.filterwarnings('ignore', message='.*Attribute\'s length.*')
    builder = builder.subject_name(x509.Name([x509.NameAttribute(NameOID.COMMON_NAME,
                                                                 name,
                                                                 _validate=False)]))
    builder = builder.issuer_name(x509.Name([x509.NameAttribute(NameOID.COMMON_NAME,
                                                                name,
                                                                _validate=False)]))
  # Gooogle seems to enforce the not before date strictly. Set the not before
  # date to be UTC two minutes ago which should cover any clock skew.
  now = datetime.datetime.utcnow()
  builder = builder.not_valid_before(now - datetime.timedelta(minutes=2))
  # Google defaults to 12/31/9999 date for end time if there's no
  # policy to restrict key age
  if validityHours:
    expires = now + datetime.timedelta(hours=validityHours) - datetime.timedelta(minutes=2)
    builder = builder.not_valid_after(expires)
  else:
    builder = builder.not_valid_after(datetime.datetime(9999, 12, 31, 23, 59))
  builder = builder.serial_number(x509.random_serial_number())
  builder = builder.public_key(public_key)
  builder = builder.add_extension(x509.BasicConstraints(ca=False, path_length=None), critical=True)
  builder = builder.add_extension(x509.KeyUsage(key_cert_sign=False,
                                                crl_sign=False, digital_signature=True, content_commitment=False,
                                                key_encipherment=False, data_encipherment=False, key_agreement=False,
                                                encipher_only=False, decipher_only=False), critical=True)
  builder = builder.add_extension(x509.ExtendedKeyUsage([x509.oid.ExtendedKeyUsageOID.SERVER_AUTH]), critical=True)
  certificate = builder.sign(private_key=private_key, algorithm=hashes.SHA256(), backend=default_backend())
  public_cert_pem = certificate.public_bytes(serialization.Encoding.PEM).decode()
  if projectId:
    printEntityMessage([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], Msg.DONE_GENERATING_PRIVATE_KEY_AND_PUBLIC_CERTIFICATE)
  else:
    writeStdout(Msg.DONE_GENERATING_PRIVATE_KEY_AND_PUBLIC_CERTIFICATE+'\n')
  if not b64enc_pub:
    return (private_pem, public_cert_pem)
  publicKeyData = base64.b64encode(public_cert_pem.encode())
  if isinstance(publicKeyData, bytes):
    publicKeyData = publicKeyData.decode()
  return (private_pem, publicKeyData)

def _formatOAuth2ServiceData(service_data):
  quotedEmail = quote(service_data.get('client_email', ''))
  service_data['auth_provider_x509_cert_url'] = API.GOOGLE_AUTH_PROVIDER_X509_CERT_URL
  service_data['auth_uri'] = API.GOOGLE_OAUTH2_ENDPOINT
  service_data['client_x509_cert_url'] = f'https://www.googleapis.com/robot/v1/metadata/x509/{quotedEmail}'
  service_data['token_uri'] = API.GOOGLE_OAUTH2_TOKEN_ENDPOINT
  service_data['type'] = 'service_account'
  GM.Globals[GM.OAUTH2SERVICE_JSON_DATA] = service_data.copy()
  return json.dumps(GM.Globals[GM.OAUTH2SERVICE_JSON_DATA], indent=2, sort_keys=True)

def doProcessSvcAcctKeys(mode=None, iam=None, projectId=None, clientEmail=None, clientId=None):
  def getSAKeyParms(body, new_data):
    nonlocal local_key_size, validityHours
    while Cmd.ArgumentsRemaining():
      myarg = getArgument()
      if myarg == 'algorithm':
        body['keyAlgorithm'] = getChoice(["key_alg_rsa_1024", "key_alg_rsa_2048"]).upper()
        local_key_size = 0
      elif myarg == 'localkeysize':
        local_key_size = int(getChoice(['1024', '2048', '4096']))
      elif myarg == 'validityhours':
        validityHours = getInteger()
      elif myarg == 'yubikey':
        new_data['key_type'] = 'yubikey'
      elif myarg == 'yubikeyslot':
        new_data['yubikey_slot'] = getString(Cmd.OB_STRING).upper()
      elif myarg == 'yubikeypin':
        new_data['yubikey_pin'] = readStdin('Enter your YubiKey PIN: ')
      elif myarg == 'yubikeyserialnumber':
        new_data['yubikey_serial_number'] = getInteger()
      else:
        unknownArgumentExit()

  local_key_size = 2048
  validityHours = 0
  body = {}
  if mode is None:
    mode = getChoice(['retainnone', 'retainexisting', 'replacecurrent'])
  if iam is None or mode == 'upload':
    if iam is None:
      _, iam = buildGAPIServiceObject(API.IAM, None)
    _getSvcAcctData()
    currentPrivateKeyId, projectId, clientEmail, clientId = _getSvcAcctKeyProjectClientFields()
    # dict() ensures we have a real copy, not pointer
    new_data = dict(GM.Globals[GM.OAUTH2SERVICE_JSON_DATA])
    # assume default key type unless we are told otherwise
    new_data['key_type'] = 'default'
    getSAKeyParms(body, new_data)
  else:
    new_data = {
      'client_email': clientEmail,
      'project_id': projectId,
      'client_id': clientId,
      'key_type': 'default'
    }
    getSAKeyParms(body, new_data)
  name = f'projects/{projectId}/serviceAccounts/{clientId}'
  if mode != 'retainexisting':
    try:
      keys = callGAPIitems(iam.projects().serviceAccounts().keys(), 'list', 'keys',
                           throwReasons=[GAPI.BAD_REQUEST, GAPI.PERMISSION_DENIED],
                           name=name, keyTypes='USER_MANAGED')
    except GAPI.permissionDenied:
      entityActionFailedWarning([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], Msg.UPDATE_PROJECT_TO_VIEW_MANAGE_SAKEYS)
      return False
    except GAPI.badRequest as e:
      entityActionFailedWarning([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], str(e))
      return False
  if new_data.get('key_type') == 'yubikey':
    # Use yubikey private key
    new_data['yubikey_key_type'] = f'RSA{local_key_size}'
    new_data.pop('private_key', None)
    yk = yubikey.YubiKey(new_data)
    if 'yubikey_serial_number' not in new_data:
      new_data['yubikey_serial_number'] = yk.get_serial_number()
      yk = yubikey.YubiKey(new_data)
    if 'yubikey_slot' not in new_data:
      new_data['yubikey_slot'] = 'AUTHENTICATION'
    publicKeyData = yk.get_certificate()
  elif local_key_size:
    # Generate private key locally, store in file
    new_data['private_key'], publicKeyData = _generatePrivateKeyAndPublicCert(projectId, clientEmail, name,
                                                                              local_key_size, validityHours=validityHours)
    new_data['key_type'] = 'default'
    for key in list(new_data):
      if key.startswith('yubikey_'):
        new_data.pop(key, None)
  if local_key_size:
    Act.Set(Act.UPLOAD)
    maxRetries = 10
    printEntityMessage([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], Msg.UPLOADING_NEW_PUBLIC_CERTIFICATE_TO_GOOGLE)
    for retry in range(1, maxRetries+1):
      try:
        result = callGAPI(iam.projects().serviceAccounts().keys(), 'upload',
                          throwReasons=[GAPI.NOT_FOUND, GAPI.BAD_REQUEST, GAPI.PERMISSION_DENIED, GAPI.FAILED_PRECONDITION],
                          name=name, body={'publicKeyData': publicKeyData})
        newPrivateKeyId = result['name'].rsplit('/', 1)[-1]
        break
      except GAPI.notFound as e:
        if retry == maxRetries:
          entityActionFailedWarning([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], str(e))
          return False
        _waitForSvcAcctCompletion(retry)
      except GAPI.permissionDenied:
        if retry == maxRetries:
          entityActionFailedWarning([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], Msg.UPDATE_PROJECT_TO_VIEW_MANAGE_SAKEYS)
          return False
        _waitForSvcAcctCompletion(retry)
      except GAPI.badRequest as e:
        entityActionFailedWarning([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], str(e))
        return False
      except GAPI.failedPrecondition as e:
        entityActionFailedWarning([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], str(e))
        if 'iam.disableServiceAccountKeyUpload' not in str(e) and 'iam.managed.disableServiceAccountKeyUpload' not in str(e):
          return False
        if retry == maxRetries or mode != 'upload':
          sys.stdout.write(Msg.ENABLE_SERVICE_ACCOUNT_PRIVATE_KEY_UPLOAD.format(projectId))
          new_data['private_key'] = ''
          newPrivateKeyId = ''
          break
        _waitForSvcAcctCompletion(retry)
    new_data['private_key_id'] = newPrivateKeyId
    oauth2service_data = _formatOAuth2ServiceData(new_data)
  else:
    Act.Set(Act.CREATE)
    maxRetries = 10
    for retry in range(1, maxRetries+1):
      try:
        result = callGAPI(iam.projects().serviceAccounts().keys(), 'create',
                          throwReasons=[GAPI.BAD_REQUEST, GAPI.PERMISSION_DENIED],
                          name=name, body=body)
        newPrivateKeyId = result['name'].rsplit('/', 1)[-1]
        break
      except GAPI.permissionDenied:
        if retry == maxRetries:
          entityActionFailedWarning([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], Msg.UPDATE_PROJECT_TO_VIEW_MANAGE_SAKEYS)
          return False
        _waitForSvcAcctCompletion(retry)
      except GAPI.badRequest as e:
        entityActionFailedWarning([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], str(e))
        return False
    oauth2service_data = base64.b64decode(result['privateKeyData']).decode(UTF8)
  if newPrivateKeyId != '':
    entityActionPerformed([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail, Ent.SVCACCT_KEY, newPrivateKeyId])
  if GM.Globals[GM.SVCACCT_SCOPES_DEFINED]:
    try:
      GM.Globals[GM.OAUTH2SERVICE_JSON_DATA] = json.loads(oauth2service_data)
    except (IndexError, KeyError, SyntaxError, TypeError, ValueError) as e:
      invalidOauth2serviceJsonExit(str(e))
    GM.Globals[GM.OAUTH2SERVICE_JSON_DATA][API.OAUTH2SA_SCOPES] = GM.Globals[GM.SVCACCT_SCOPES]
    oauth2service_data = json.dumps(GM.Globals[GM.OAUTH2SERVICE_JSON_DATA], ensure_ascii=False, sort_keys=True, indent=2)
  writeFile(GC.Values[GC.OAUTH2SERVICE_JSON], oauth2service_data, continueOnError=False)
  Act.Set(Act.UPDATE)
  entityActionPerformed([Ent.OAUTH2SERVICE_JSON_FILE, GC.Values[GC.OAUTH2SERVICE_JSON],
                         Ent.SVCACCT_KEY, newPrivateKeyId])
  if mode in {'retainexisting', 'upload'}:
    return newPrivateKeyId != ''
  Act.Set(Act.REVOKE)
  count = len(keys) if mode == 'retainnone' else 1
  entityPerformActionNumItems([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], count, Ent.SVCACCT_KEY)
  Ind.Increment()
  i = 0
  for key in keys:
    keyName = key['name'].rsplit('/', 1)[-1]
    if mode == 'retainnone' or keyName == currentPrivateKeyId and keyName != newPrivateKeyId:
      i += 1
      maxRetries = 5
      for retry in range(1, maxRetries+1):
        try:
          callGAPI(iam.projects().serviceAccounts().keys(), 'delete',
                   throwReasons=[GAPI.BAD_REQUEST, GAPI.PERMISSION_DENIED],
                   name=key['name'])
          entityActionPerformed([Ent.SVCACCT_KEY, keyName], i, count)
          break
        except GAPI.permissionDenied:
          if retry == maxRetries:
            entityActionFailedWarning([Ent.SVCACCT_KEY, keyName], Msg.UPDATE_PROJECT_TO_VIEW_MANAGE_SAKEYS)
            break
          _waitForSvcAcctCompletion(retry)
        except GAPI.badRequest as e:
          entityActionFailedWarning([Ent.SVCACCT_KEY, keyName], str(e), i, count)
          break
      if mode != 'retainnone':
        break
  Ind.Decrement()
  return True

# gam create sakey|sakeys
# gam rotate sakey|sakeys retain_existing
#	(algorithm KEY_ALG_RSA_1024|KEY_ALG_RSA_2048)|
#	(localkeysize 1024|2048|4096 [validityhours <Number>])|
#	(yubikey yubikey_pin yubikey_slot AUTHENTICATION yubikey_serialnumber <String>)
def doCreateSvcAcctKeys():
  doProcessSvcAcctKeys(mode='retainexisting')

# gam update sakey|sakeys
# gam rotate sakey|sakeys replace_current
#	(algorithm KEY_ALG_RSA_1024|KEY_ALG_RSA_2048)|
#	(localkeysize 1024|2048|4096 [validityhours <Number>])|
#	(yubikey yubikey_pin yubikey_slot AUTHENTICATION yubikey_serialnumber <String>)
def doUpdateSvcAcctKeys():
  doProcessSvcAcctKeys(mode='replacecurrent')

# gam replace sakey|sakeys
# gam rotate sakey|sakeys retain_none
#	(algorithm KEY_ALG_RSA_1024|KEY_ALG_RSA_2048)|
#	(localkeysize 1024|2048|4096 [validityhours <Number>])|
#	(yubikey yubikey_pin yubikey_slot AUTHENTICATION yubikey_serialnumber <String>)
def doReplaceSvcAcctKeys():
  doProcessSvcAcctKeys(mode='retainnone')

# gam upload sakey|sakeys [admin <EmailAddress>]
#	(algorithm KEY_ALG_RSA_1024|KEY_ALG_RSA_2048)|
#	(localkeysize 1024|2048|4096 [validityhours <Number>])|
#	(yubikey yubikey_pin yubikey_slot AUTHENTICATION yubikey_serialnumber <String>)
def doUploadSvcAcctKeys():
  login_hint = getEmailAddress(noUid=True) if checkArgumentPresent(['admin']) else None
  httpObj, _ = getCRMService(login_hint)
  iam = getAPIService(API.IAM, httpObj)
  if doProcessSvcAcctKeys(mode='upload', iam=iam):
    sa_email = GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['client_email']
    _grantRotateRights(iam, GM.Globals[GM.OAUTH2SERVICE_JSON_DATA]['project_id'], sa_email)
    sys.stdout.write(Msg.YOUR_GAM_PROJECT_IS_CREATED_AND_READY_TO_USE)

# gam delete sakeys <ServiceAccountKeyList>
def doDeleteSvcAcctKeys():
  _, iam = buildGAPIServiceObject(API.IAM, None)
  doit = False
  keyList = []
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'doit':
      doit = True
    else:
      Cmd.Backup()
      keyList.extend(getString(Cmd.OB_SERVICE_ACCOUNT_KEY_LIST, minLen=0).strip().replace(',', ' ').split())
  currentPrivateKeyId, projectId, clientEmail, clientId = _getSvcAcctKeyProjectClientFields()
  name = f'projects/{projectId}/serviceAccounts/{clientId}'
  try:
    keys = callGAPIitems(iam.projects().serviceAccounts().keys(), 'list', 'keys',
                         throwReasons=[GAPI.BAD_REQUEST, GAPI.PERMISSION_DENIED],
                         name=name, keyTypes='USER_MANAGED')
  except GAPI.permissionDenied:
    entityActionFailedWarning([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], Msg.UPDATE_PROJECT_TO_VIEW_MANAGE_SAKEYS)
    return
  except GAPI.badRequest as e:
    entityActionFailedWarning([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], str(e))
    return
  Act.Set(Act.REVOKE)
  count = len(keyList)
  entityPerformActionNumItems([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], count, Ent.SVCACCT_KEY)
  Ind.Increment()
  i = 0
  for dkeyName in keyList:
    i += 1
    for key in keys:
      keyName = key['name'].rsplit('/', 1)[-1]
      if keyName == dkeyName:
        if keyName == currentPrivateKeyId and not doit:
          entityActionNotPerformedWarning([Ent.SVCACCT_KEY, keyName],
                                          Msg.USE_DOIT_ARGUMENT_TO_PERFORM_ACTION+Msg.ON_CURRENT_PRIVATE_KEY, i, count)
          break
        try:
          callGAPI(iam.projects().serviceAccounts().keys(), 'delete',
                   throwReasons=[GAPI.BAD_REQUEST, GAPI.PERMISSION_DENIED],
                   name=key['name'])
          entityActionPerformed([Ent.SVCACCT_KEY, keyName], i, count)
        except GAPI.permissionDenied:
          entityActionFailedWarning([Ent.SVCACCT_KEY, keyName], Msg.UPDATE_PROJECT_TO_VIEW_MANAGE_SAKEYS)
        except GAPI.badRequest as e:
          entityActionFailedWarning([Ent.SVCACCT_KEY, keyName], str(e), i, count)
        break
    else:
      entityActionNotPerformedWarning([Ent.SVCACCT_KEY, dkeyName], Msg.NOT_FOUND, i, count)
  Ind.Decrement()

# gam show sakeys [all|system|user]
def doShowSvcAcctKeys():
  _, iam = buildGAPIServiceObject(API.IAM, None)
  keyTypes = None
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg in SVCACCT_KEY_TYPE_CHOICE_MAP:
      keyTypes = SVCACCT_KEY_TYPE_CHOICE_MAP[myarg]
    else:
      unknownArgumentExit()
  currentPrivateKeyId, projectId, clientEmail, clientId = _getSvcAcctKeyProjectClientFields()
  name = f'projects/{projectId}/serviceAccounts/{clientId}'
  status, keys = _getSAKeys(iam, projectId, clientEmail, name, keyTypes)
  if not status:
    return
  count = len(keys)
  entityPerformActionNumItems([Ent.PROJECT, projectId, Ent.SVCACCT, clientEmail], count, Ent.SVCACCT_KEY)
  if count > 0:
    _showSAKeys(keys, count, currentPrivateKeyId)

# gam create gcpserviceaccount|signjwtserviceaccount
def doCreateGCPServiceAccount():
  checkForExtraneousArguments()
  _checkForExistingProjectFiles([GC.Values[GC.OAUTH2SERVICE_JSON]])
  sa_info = {'key_type': 'signjwt', 'token_uri': API.GOOGLE_OAUTH2_TOKEN_ENDPOINT, 'type': 'service_account'}
  request = getTLSv1_2Request()
  try:
    credentials, sa_info['project_id'] = google.auth.default(scopes=[API.IAM_SCOPE], request=request)
  except (google.auth.exceptions.DefaultCredentialsError, google.auth.exceptions.RefreshError) as e:
    systemErrorExit(API_ACCESS_DENIED_RC, str(e))
  credentials.refresh(request)
  sa_info['client_email'] = credentials.service_account_email
  oa2 = buildGAPIObjectNoAuthentication(API.OAUTH2)
  try:
    token_info = callGAPI(oa2, 'tokeninfo',
                          throwReasons=[GAPI.INVALID],
                          access_token=credentials.token)
  except GAPI.invalid as e:
    systemErrorExit(API_ACCESS_DENIED_RC, str(e))
  sa_info['client_id'] = token_info['issued_to']
  sa_output = json.dumps(sa_info, ensure_ascii=False, sort_keys=True, indent=2)
  writeStdout(f'Writing SignJWT service account data:\n\n{sa_output}\n')
  writeFile(GC.Values[GC.OAUTH2SERVICE_JSON], sa_output, continueOnError=False)

# Audit command utilities
def getAuditParameters(emailAddressRequired=True, requestIdRequired=True, destUserRequired=False):
  auditObject = getEmailAuditObject()
  emailAddress = getEmailAddress(noUid=True, optional=not emailAddressRequired)
  parameters = {}
  if emailAddress:
    parameters['auditUser'] = emailAddress
    parameters['auditUserName'], auditObject.domain = splitEmailAddress(emailAddress)
    if requestIdRequired:
      parameters['requestId'] = getString(Cmd.OB_REQUEST_ID)
    if destUserRequired:
      destEmailAddress = getEmailAddress(noUid=True)
      parameters['auditDestUser'] = destEmailAddress
      parameters['auditDestUserName'], destDomain = splitEmailAddress(destEmailAddress)
      if auditObject.domain != destDomain:
        Cmd.Backup()
        invalidArgumentExit(f'{parameters["auditDestUserName"]}@{auditObject.domain}')
  return (auditObject, parameters)

# Audit monitor command utilities
def _showMailboxMonitorRequestStatus(request, i=0, count=0):
  printKeyValueListWithCount(['Destination', normalizeEmailAddressOrUID(request['destUserName'])], i, count)
  Ind.Increment()
  printKeyValueList(['Begin', request.get('beginDate', 'immediately')])
  printKeyValueList(['End', request['endDate']])
  printKeyValueList(['Monitor Incoming', request['outgoingEmailMonitorLevel']])
  printKeyValueList(['Monitor Outgoing', request['incomingEmailMonitorLevel']])
  printKeyValueList(['Monitor Chats', request.get('chatMonitorLevel', 'NONE')])
  printKeyValueList(['Monitor Drafts', request.get('draftMonitorLevel', 'NONE')])
  Ind.Decrement()

# gam audit monitor create <EmailAddress> <DestEmailAddress> [begin <DateTime>] [end <DateTime>] [incoming_headers] [outgoing_headers] [nochats] [nodrafts] [chat_headers] [draft_headers]
def doCreateMonitor():
  auditObject, parameters = getAuditParameters(emailAddressRequired=True, requestIdRequired=False, destUserRequired=True)
  #end_date defaults to 30 days in the future...
  end_date = (GM.Globals[GM.DATETIME_NOW]+datetime.timedelta(days=30)).strftime(YYYYMMDD_HHMM_FORMAT)
  begin_date = None
  incoming_headers_only = outgoing_headers_only = drafts_headers_only = chats_headers_only = False
  drafts = chats = True
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'begin':
      begin_date = getYYYYMMDD_HHMM()
    elif myarg == 'end':
      end_date = getYYYYMMDD_HHMM()
    elif myarg == 'incomingheaders':
      incoming_headers_only = True
    elif myarg == 'outgoingheaders':
      outgoing_headers_only = True
    elif myarg == 'nochats':
      chats = False
    elif myarg == 'nodrafts':
      drafts = False
    elif myarg == 'chatheaders':
      chats_headers_only = True
    elif myarg == 'draftheaders':
      drafts_headers_only = True
    else:
      unknownArgumentExit()
  try:
    request = callGData(auditObject, 'createEmailMonitor',
                        throwErrors=[GDATA.INVALID_VALUE, GDATA.INVALID_INPUT, GDATA.DOES_NOT_EXIST, GDATA.INVALID_DOMAIN],
                        source_user=parameters['auditUserName'], destination_user=parameters['auditDestUserName'], end_date=end_date, begin_date=begin_date,
                        incoming_headers_only=incoming_headers_only, outgoing_headers_only=outgoing_headers_only,
                        drafts=drafts, drafts_headers_only=drafts_headers_only, chats=chats, chats_headers_only=chats_headers_only)
    entityActionPerformed([Ent.USER, parameters['auditUser'], Ent.AUDIT_MONITOR_REQUEST, None])
    Ind.Increment()
    _showMailboxMonitorRequestStatus(request)
    Ind.Decrement()
  except (GDATA.invalidValue, GDATA.invalidInput) as e:
    entityActionFailedWarning([Ent.USER, parameters['auditUser'], Ent.AUDIT_MONITOR_REQUEST, None], str(e))
  except (GDATA.doesNotExist, GDATA.invalidDomain) as e:
    if str(e).find(parameters['auditUser']) != -1:
      entityUnknownWarning(Ent.USER, parameters['auditUser'])
    else:
      entityActionFailedWarning([Ent.USER, parameters['auditUser'], Ent.AUDIT_MONITOR_REQUEST, None], str(e))

# gam audit monitor delete <EmailAddress> <DestEmailAddress>
def doDeleteMonitor():
  auditObject, parameters = getAuditParameters(emailAddressRequired=True, requestIdRequired=False, destUserRequired=True)
  checkForExtraneousArguments()
  try:
    callGData(auditObject, 'deleteEmailMonitor',
              throwErrors=[GDATA.INVALID_INPUT, GDATA.DOES_NOT_EXIST, GDATA.INVALID_DOMAIN],
              source_user=parameters['auditUserName'], destination_user=parameters['auditDestUserName'])
    entityActionPerformed([Ent.USER, parameters['auditUser'], Ent.AUDIT_MONITOR_REQUEST, parameters['auditDestUser']])
  except GDATA.invalidInput as e:
    entityActionFailedWarning([Ent.USER, parameters['auditUser'], Ent.AUDIT_MONITOR_REQUEST, None], str(e))
  except (GDATA.doesNotExist, GDATA.invalidDomain) as e:
    if str(e).find(parameters['auditUser']) != -1:
      entityUnknownWarning(Ent.USER, parameters['auditUser'])
    else:
      entityActionFailedWarning([Ent.USER, parameters['auditUser'], Ent.AUDIT_MONITOR_REQUEST, None], str(e))

# gam audit monitor list <EmailAddress>
def doShowMonitors():
  auditObject, parameters = getAuditParameters(emailAddressRequired=True, requestIdRequired=False, destUserRequired=False)
  checkForExtraneousArguments()
  try:
    results = callGData(auditObject, 'getEmailMonitors',
                        throwErrors=[GDATA.DOES_NOT_EXIST, GDATA.INVALID_DOMAIN],
                        user=parameters['auditUserName'])
    jcount = len(results) if (results) else 0
    entityPerformActionNumItems([Ent.USER, parameters['auditUser']], jcount, Ent.AUDIT_MONITOR_REQUEST)
    if jcount == 0:
      setSysExitRC(NO_ENTITIES_FOUND_RC)
      return
    Ind.Increment()
    j = 0
    for request in results:
      j += 1
      _showMailboxMonitorRequestStatus(request, j, jcount)
    Ind.Decrement()
  except (GDATA.doesNotExist, GDATA.invalidDomain):
    entityUnknownWarning(Ent.USER, parameters['auditUser'])

# gam whatis <EmailItem> [noinfo] [noinvitablecheck]
def doWhatIs():
  def _showPrimaryType(entityType, email):
    printEntity([entityType, email])

  def _showAliasType(entityType, email, primaryEntityType, primaryEmail):
    printEntity([entityType, email, primaryEntityType, primaryEmail])

  cd = buildGAPIObject(API.DIRECTORY)
  email = getEmailAddress()
  showInfo = invitableCheck = True
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'noinfo':
      showInfo = False
    elif myarg == 'noinvitablecheck':
      invitableCheck = False
    else:
      unknownArgumentExit()
  try:
    result = callGAPI(cd.users(), 'get',
                      throwReasons=GAPI.USER_GET_THROW_REASONS,
                      userKey=email, fields='id,primaryEmail')
    if (result['primaryEmail'].lower() == email) or (result['id'] == email):
      if showInfo:
        infoUsers(entityList=[email])
      else:
        _showPrimaryType(Ent.USER, email)
      setSysExitRC(ENTITY_IS_A_USER_RC)
    else:
      if showInfo:
        infoAliases(entityList=[email])
      else:
        _showAliasType(Ent.USER_ALIAS, email, Ent.USER, result['primaryEmail'])
      setSysExitRC(ENTITY_IS_A_USER_ALIAS_RC)
    return
  except (GAPI.userNotFound, GAPI.badRequest):
    pass
  except (GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden,
          GAPI.backendError, GAPI.systemError):
    entityUnknownWarning(Ent.EMAIL, email)
    setSysExitRC(ENTITY_IS_UKNOWN_RC)
    return
  try:
    result = callGAPI(cd.groups(), 'get',
                      throwReasons=GAPI.GROUP_GET_THROW_REASONS,
                      groupKey=email, fields='id,email')
    if (result['email'].lower() == email) or (result['id'] == email):
      if showInfo:
        infoGroups([email])
      else:
        _showPrimaryType(Ent.GROUP, email)
      setSysExitRC(ENTITY_IS_A_GROUP_RC)
    else:
      if showInfo:
        infoAliases(entityList=[email])
      else:
        _showAliasType(Ent.GROUP_ALIAS, email, Ent.GROUP, result['email'])
      setSysExitRC(ENTITY_IS_A_GROUP_ALIAS_RC)
    return
  except (GAPI.groupNotFound, GAPI.forbidden):
    pass
  except (GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.badRequest):
    entityUnknownWarning(Ent.EMAIL, email)
    setSysExitRC(ENTITY_IS_UKNOWN_RC)
    return
  if not invitableCheck:
    isInvitableUser = False
  else:
    isInvitableUser, ci = _getIsInvitableUser(None, email)
  if isInvitableUser:
    if showInfo:
      name, user, ci = _getCIUserInvitationsEntity(ci, email)
      infoCIUserInvitations(name, user, ci, None)
    else:
      _showPrimaryType(Ent.USER_INVITATION, email)
    setSysExitRC(ENTITY_IS_AN_UNMANAGED_ACCOUNT_RC)
  else:
    entityUnknownWarning(Ent.EMAIL, email)
    setSysExitRC(ENTITY_IS_UKNOWN_RC)

def _adjustTryDate(errMsg, numDateChanges, limitDateChanges, prevTryDate):
  match_date = re.match('Data for dates later than (.*) is not yet available. Please check back later', errMsg)
  if match_date:
    tryDate = match_date.group(1)
  else:
    match_date = re.match('Start date can not be later than (.*)', errMsg)
    if match_date:
      tryDate = match_date.group(1)
    else:
      match_date = re.match('End date greater than LastReportedDate.', errMsg)
      if match_date:
        tryDateTime = datetime.datetime.strptime(prevTryDate, YYYYMMDD_FORMAT)-datetime.timedelta(days=1)
        tryDate = tryDateTime.strftime(YYYYMMDD_FORMAT)
  if (not match_date) or (numDateChanges > limitDateChanges >= 0):
    printWarningMessage(DATA_NOT_AVALIABLE_RC, errMsg)
    return None
  return tryDate

def _checkDataRequiredServices(result, tryDate, dataRequiredServices, parameterServices=None, checkUserEmail=False):
# -1: Data not available:
#  0: Backup to earlier date
#  1: Data available
  oneDay = datetime.timedelta(days=1)
  dataWarnings = result.get('warnings', [])
  usageReports = result.get('usageReports', [])
  # move to day before if we don't have at least one usageReport with parameters
  if not usageReports or not usageReports[0].get('parameters', []):
    tryDateTime = datetime.datetime.strptime(tryDate, YYYYMMDD_FORMAT)-oneDay
    return (0, tryDateTime.strftime(YYYYMMDD_FORMAT), None)
  for warning in dataWarnings:
    if warning['code'] == 'PARTIAL_DATA_AVAILABLE':
      for app in warning['data']:
        if app['key'] == 'application' and app['value'] != 'docs' and app['value'] in dataRequiredServices:
          tryDateTime = datetime.datetime.strptime(tryDate, YYYYMMDD_FORMAT)-oneDay
          return (0, tryDateTime.strftime(YYYYMMDD_FORMAT), None)
    elif warning['code'] == 'DATA_NOT_AVAILABLE':
      for app in warning['data']:
        if app['key'] == 'application' and app['value'] != 'docs' and app['value'] in dataRequiredServices:
          return (-1, tryDate, None)
  if parameterServices:
    requiredServices = parameterServices.copy()
    for item in usageReports[0].get('parameters', []):
      if 'name' not in item:
        continue
      service, _ = item['name'].split(':', 1)
      if service in requiredServices:
        requiredServices.remove(service)
        if not requiredServices:
          break
    else:
      tryDateTime = datetime.datetime.strptime(tryDate, YYYYMMDD_FORMAT)-oneDay
      return (0, tryDateTime.strftime(YYYYMMDD_FORMAT), None)
  if checkUserEmail:
    if 'entity' not in usageReports[0] or 'userEmail' not in usageReports[0]['entity']:
      tryDateTime = datetime.datetime.strptime(tryDate, YYYYMMDD_FORMAT)-oneDay
      return (0, tryDateTime.strftime(YYYYMMDD_FORMAT), None)
  return (1, tryDate, usageReports)

CUSTOMER_REPORT_SERVICES = {
  'accounts',
  'app_maker',
  'apps_scripts',
  'calendar',
  'chat',
  'classroom',
  'cros',
  'device_management',
  'docs',
  'drive',
  'gmail',
  'gplus',
  'meet',
  'sites',
  }

USER_REPORT_SERVICES = {
  'accounts',
  'chat',
  'classroom',
  'docs',
  'drive',
  'gmail',
  'gplus',
  }

CUSTOMER_USER_CHOICES = {'customer', 'user'}

# gam report usageparameters customer|user [todrive <ToDriveAttribute>*]
def doReportUsageParameters():
  report = getChoice(CUSTOMER_USER_CHOICES)
  csvPF = CSVPrintFile(['parameter'], 'sortall')
  getTodriveOnly(csvPF)
  rep = buildGAPIObject(API.REPORTS)
  if report == 'customer':
    service = rep.customerUsageReports()
    dataRequiredServices = CUSTOMER_REPORT_SERVICES
    kwargs = {}
  else: # 'user'
    service = rep.userUsageReport()
    dataRequiredServices = USER_REPORT_SERVICES
    kwargs = {'userKey': _getAdminEmail()}
  customerId = GC.Values[GC.CUSTOMER_ID]
  if customerId == GC.MY_CUSTOMER:
    customerId = None
  tryDate = todaysDate().strftime(YYYYMMDD_FORMAT)
  allParameters = set()
  while True:
    try:
      result = callGAPI(service, 'get',
                        throwReasons=[GAPI.INVALID, GAPI.BAD_REQUEST],
                        date=tryDate, customerId=customerId, fields='warnings,usageReports(parameters(name))', **kwargs)
      fullData, tryDate, usageReports = _checkDataRequiredServices(result, tryDate, dataRequiredServices)
      if fullData < 0:
        printWarningMessage(DATA_NOT_AVALIABLE_RC, Msg.NO_USAGE_PARAMETERS_DATA_AVAILABLE)
        return
      if usageReports:
        for parameter in usageReports[0]['parameters']:
          name = parameter.get('name')
          if name:
            allParameters.add(name)
      if fullData == 1:
        break
    except GAPI.badRequest:
      printErrorMessage(BAD_REQUEST_RC, Msg.BAD_REQUEST)
      return
    except GAPI.invalid as e:
      tryDate = _adjustTryDate(str(e), 0, -1, tryDate)
      if not tryDate:
        break
  for parameter in sorted(allParameters):
    csvPF.WriteRow({'parameter': parameter})
  csvPF.writeCSVfile(f'{report.capitalize()} Report Usage Parameters')

def getUserOrgUnits(cd, orgUnit, orgUnitId):
  try:
    if orgUnit == orgUnitId:
      orgUnit = callGAPI(cd.orgunits(), 'get',
                         throwReasons=GAPI.ORGUNIT_GET_THROW_REASONS,
                         customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath=orgUnit, fields='orgUnitPath')['orgUnitPath']
    printGettingAllEntityItemsForWhom(Ent.USER, orgUnit, qualifier=Msg.IN_THE.format(Ent.Singular(Ent.ORGANIZATIONAL_UNIT)),
                                      entityType=Ent.ORGANIZATIONAL_UNIT)
    result = callGAPIpages(cd.users(), 'list', 'users',
                           pageMessage=getPageMessageForWhom(),
                           throwReasons=[GAPI.INVALID_ORGUNIT, GAPI.ORGUNIT_NOT_FOUND,
                                         GAPI.INVALID_INPUT, GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN],
                           retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                           customer=GC.Values[GC.CUSTOMER_ID], query=orgUnitPathQuery(orgUnit, None), orderBy='email',
                           fields='nextPageToken,users(primaryEmail,orgUnitPath)', maxResults=GC.Values[GC.USER_MAX_RESULTS])
    userOrgUnits = {}
    for user in result:
      userOrgUnits[user['primaryEmail']] = user['orgUnitPath']
    return userOrgUnits
  except (GAPI.badRequest, GAPI.invalidInput, GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError,
          GAPI.invalidCustomerId, GAPI.loginRequired, GAPI.resourceNotFound, GAPI.forbidden):
    checkEntityDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, orgUnit)

# Convert report mb item to gb
def convertReportMBtoGB(name, item):
  if item is not None:
    item['intValue'] = f"{int(item['intValue'])/1024:.2f}"
  return name.replace('_in_mb', '_in_gb')

REPORTS_PARAMETERS_SIMPLE_TYPES = ['intValue', 'boolValue', 'datetimeValue', 'stringValue']

# gam report usage user [todrive <ToDriveAttribute>*]
#	[(user all|<UserItem>)|(orgunit|org|ou <OrgUnitPath> [showorgunit])|(select <UserTypeEntity>)]
#	[([start|startdate <Date>] [end|enddate <Date>])|(range <Date> <Date>)|
#	 thismonth|(previousmonths <Integer>)]
#	[fields|parameters <String>)]
#	[convertmbtogb]
# gam report usage customer [todrive <ToDriveAttribute>*]
#	[([start|startdate <Date>] [end|enddate <Date>])|(range <Date> <Date>)|
#	 thismonth|(previousmonths <Integer>)]
#	[fields|parameters <String>)]
#	[convertmbtogb]
def doReportUsage():
  def usageEntitySelectors():
    selectorChoices = Cmd.USER_ENTITY_SELECTORS+Cmd.USER_CSVDATA_ENTITY_SELECTORS
    if GC.Values[GC.USER_SERVICE_ACCOUNT_ACCESS_ONLY]:
      selectorChoices += Cmd.SERVICE_ACCOUNT_ONLY_ENTITY_SELECTORS[:]+[Cmd.ENTITY_USER, Cmd.ENTITY_USERS]
    else:
      selectorChoices += Cmd.BASE_ENTITY_SELECTORS[:]+Cmd.USER_ENTITIES[:]
    return selectorChoices

  def validateYYYYMMDD(argstr):
    if argstr in TODAY_NOW or argstr[0] in PLUS_MINUS:
      if argstr == 'NOW':
        argstr = 'TODAY'
      deltaDate = getDelta(argstr, DELTA_DATE_PATTERN)
      if deltaDate is None:
        Cmd.Backup()
        invalidArgumentExit(DELTA_DATE_FORMAT_REQUIRED)
      return deltaDate
    try:
      argDate = datetime.datetime.strptime(argstr, YYYYMMDD_FORMAT)
      return datetime.datetime(argDate.year, argDate.month, argDate.day, tzinfo=GC.Values[GC.TIMEZONE])
    except ValueError:
      Cmd.Backup()
      invalidArgumentExit(YYYYMMDD_FORMAT_REQUIRED)

  report = getChoice(CUSTOMER_USER_CHOICES)
  rep = buildGAPIObject(API.REPORTS)
  titles = ['date']
  if report == 'customer':
    fullDataServices = CUSTOMER_REPORT_SERVICES
    userReports = False
    service = rep.customerUsageReports()
    kwargs = [{}]
  else: # 'user'
    fullDataServices = USER_REPORT_SERVICES
    userReports = True
    service = rep.userUsageReport()
    kwargs = [{'userKey': 'all'}]
    titles.append('user')
  csvPF = CSVPrintFile()
  customerId = GC.Values[GC.CUSTOMER_ID]
  if customerId == GC.MY_CUSTOMER:
    customerId = None
  parameters = set()
  convertMbToGb = select = showOrgUnit = False
  userKey = 'all'
  cd = orgUnit = orgUnitId = None
  userOrgUnits = {}
  startEndTime = StartEndTime('startdate', 'enddate', 'date')
  skipDayNumbers = []
  skipDates = set()
  oneDay = datetime.timedelta(days=1)
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if csvPF and myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif myarg in {'start', 'startdate', 'end', 'enddate', 'range', 'thismonth', 'previousmonths'}:
      startEndTime.Get(myarg)
    elif userReports and myarg in {'ou', 'org', 'orgunit'}:
      if cd is None:
        cd = buildGAPIObject(API.DIRECTORY)
      orgUnit, orgUnitId = getOrgUnitId(cd)
      select = False
    elif userReports and myarg == 'showorgunit':
      showOrgUnit = True
    elif myarg in {'fields', 'parameters'}:
      for field in getString(Cmd.OB_STRING).replace(',', ' ').split():
        if ':' in field:
          repsvc, _ = field.split(':', 1)
          if repsvc in fullDataServices:
            parameters.add(field)
          else:
            invalidChoiceExit(repsvc, fullDataServices, True)
        else:
          Cmd.Backup()
          invalidArgumentExit('service:parameter')
    elif myarg == 'skipdates':
      for skip in getString(Cmd.OB_STRING).upper().split(','):
        if skip.find(':') == -1:
          skipDates.add(validateYYYYMMDD(skip))
        else:
          skipStart, skipEnd = skip.split(':', 1)
          skipStartDate = validateYYYYMMDD(skipStart)
          skipEndDate = validateYYYYMMDD(skipEnd)
          if skipEndDate < skipStartDate:
            Cmd.Backup()
            usageErrorExit(Msg.INVALID_DATE_TIME_RANGE.format(myarg, skipEnd, myarg, skipStart))
          while skipStartDate <= skipEndDate:
            skipDates.add(skipStartDate)
            skipStartDate += oneDay
    elif myarg == 'skipdaysofweek':
      skipdaynames = getString(Cmd.OB_STRING).split(',')
      dow = [d.lower() for d in calendarlib.day_abbr]
      skipDayNumbers = [dow.index(d) for d in skipdaynames if d in dow]
    elif userReports and myarg == 'user':
      userKey = getString(Cmd.OB_EMAIL_ADDRESS)
      orgUnit = orgUnitId = None
      select = False
    elif userReports and (myarg == 'select' or myarg in usageEntitySelectors()):
      if myarg != 'select':
        Cmd.Backup()
      _, users = getEntityToModify(defaultEntityType=Cmd.ENTITY_USERS)
      orgUnit = orgUnitId = None
      select = True
    elif myarg == 'convertmbtogb':
      convertMbToGb = True
    else:
      unknownArgumentExit()
  if startEndTime.endDateTime is None:
    startEndTime.endDateTime = todaysDate()
  if startEndTime.startDateTime is None:
    startEndTime.startDateTime = startEndTime.endDateTime+datetime.timedelta(days=-30)
  startDateTime = startEndTime.startDateTime
  startDate = startDateTime.strftime(YYYYMMDD_FORMAT)
  endDateTime = startEndTime.endDateTime
  endDate = endDateTime.strftime(YYYYMMDD_FORMAT)
  startUseDate = endUseDate = None
  if not orgUnitId:
    showOrgUnit = False
  if userReports:
    if select:
      Ent.SetGetting(Ent.REPORT)
      kwargs = [{'userKey': normalizeEmailAddressOrUID(user)} for user in users]
    elif userKey == 'all':
      if orgUnitId:
        kwargs[0]['orgUnitID'] = orgUnitId
        userOrgUnits = getUserOrgUnits(cd, orgUnit, orgUnitId)
        forWhom = f'users in orgUnit {orgUnit}'
      else:
        forWhom = 'all users'
      printGettingEntityItemForWhom(Ent.REPORT, forWhom)
    else:
      Ent.SetGetting(Ent.REPORT)
      kwargs = [{'userKey': normalizeEmailAddressOrUID(userKey)}]
      printGettingEntityItemForWhom(Ent.REPORT, kwargs[0]['userKey'])
    if showOrgUnit:
      titles.append('orgUnitPath')
  else:
    pageMessage = None
  csvPF.SetTitles(titles)
  csvPF.SetSortAllTitles()
  parameters = ','.join(parameters) if parameters else None
  while startDateTime <= endDateTime:
    if startDateTime.weekday() in skipDayNumbers or startDateTime in skipDates:
      startDateTime += oneDay
      continue
    useDate = startDateTime.strftime(YYYYMMDD_FORMAT)
    startDateTime += oneDay
    try:
      for kwarg in kwargs:
        if userReports:
          if not select and userKey == 'all':
            pageMessage = getPageMessageForWhom(forWhom, showDate=useDate)
          else:
            pageMessage = getPageMessageForWhom(kwarg['userKey'], showDate=useDate)
        try:
          usage = callGAPIpages(service, 'get', 'usageReports',
                                pageMessage=pageMessage,
                                throwReasons=[GAPI.INVALID, GAPI.INVALID_INPUT, GAPI.BAD_REQUEST, GAPI.FORBIDDEN],
                                retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                                customerId=customerId, date=useDate,
                                parameters=parameters, **kwarg)
        except GAPI.badRequest:
          continue
        for entity in usage:
          row = {'date': useDate}
          if userReports:
            if 'userEmail' in entity['entity']:
              row['user'] = entity['entity']['userEmail']
              if showOrgUnit:
                row['orgUnitPath'] = userOrgUnits.get(row['user'], UNKNOWN)
            else:
              row['user'] = UNKNOWN
          for item in entity.get('parameters', []):
            if 'name' not in item:
              continue
            name = item['name']
            if name == 'cros:device_version_distribution':
              versions = {}
              for version in item['msgValue']:
                versions[version['version_number']] = version['num_devices']
              for k, v in sorted(versions.items(), reverse=True):
                title = f'cros:num_devices_chrome_{k}'
                row[title] = v
            else:
              for ptype in REPORTS_PARAMETERS_SIMPLE_TYPES:
                if ptype in item:
                  if ptype != 'datetimeValue':
                    if convertMbToGb and name.endswith('_in_mb'):
                      name = convertReportMBtoGB(name, item)
                    row[name] = item[ptype]
                  else:
                    row[name] = formatLocalTime(item[ptype])
                  break
              else:
                row[name] = ''
          if not startUseDate:
            startUseDate = useDate
          endUseDate = useDate
          csvPF.WriteRowTitles(row)
    except GAPI.invalid as e:
      stderrWarningMsg(str(e))
      break
    except GAPI.invalidInput as e:
      systemErrorExit(GOOGLE_API_ERROR_RC, str(e))
    except GAPI.forbidden as e:
      accessErrorExit(None, str(e))
  if startUseDate:
    reportName = f'{report.capitalize()} Usage Report - {startUseDate}:{endUseDate}'
  else:
    reportName = f'{report.capitalize()} Usage Report - {startDate}:{endDate} - No Data'
  csvPF.writeCSVfile(reportName)

NL_SPACES_PATTERN = re.compile(r'\n +')
DISABLED_REASON_TIME_PATTERN = re.compile(r'.*(\d{4}/\d{2}/\d{2}-\d{2}:\d{2}:\d{2})')

REPORT_ALIASES_CHOICE_MAP = {
  'access': 'accesstransparency',
  'calendars': 'calendar',
  'cloud': 'gcp',
  'currents': 'gplus',
  'customers': 'customer',
  'domain': 'customer',
  'devices': 'mobile',
  'doc': 'drive',
  'docs': 'drive',
  'enterprisegroups': 'groupsenterprise',
  'gemini': 'geminiinworkspaceapps',
  'geminiforworkspace': 'geminiinworkspaceapps',
  'group': 'groups',
  'google+': 'gplus',
  'hangoutsmeet': 'meet',
  'logins': 'login',
  'lookerstudio': 'datastudio',
  'oauthtoken': 'token',
  'tokens': 'token',
  'users': 'user',
  }

REPORT_CHOICE_MAP = {
  'accesstransparency': 'access_transparency',
  'admin': 'admin',
  'calendar': 'calendar',
  'chat': 'chat',
  'chrome': 'chrome',
  'contextawareaccess': 'context_aware_access',
  'customer': 'customer',
  'datastudio': 'data_studio',
  'drive': 'drive',
  'gcp': 'gcp',
  'geminiinworkspaceapps': 'gemini_in_workspace_apps',
  'gplus': 'gplus',
  'groups': 'groups',
  'groupsenterprise': 'groups_enterprise',
  'jamboard': 'jamboard',
  'keep': 'keep',
  'login': 'login',
  'meet': 'meet',
  'mobile': 'mobile',
  'rules': 'rules',
  'saml': 'saml',
  'token': 'token',
  'usage': 'usage',
  'usageparameters': 'usageparameters',
  'user': 'user',
  'useraccounts': 'user_accounts',
  'vault': 'vault',
  }

REPORT_ACTIVITIES_UPPERCASE_EVENTS = {
  'access_transparency',
  'admin',
  'chrome',
  'context_aware_access',
  'data_studio',
  'gcp',
  'jamboard',
  'mobile'
  }

REPORT_ACTIVITIES_TIME_OBJECTS = {'time'}

# gam report <ActivityApplictionName> [todrive <ToDriveAttribute>*]
#	[(user all|<UserItem>)|(orgunit|org|ou <OrgUnitPath> [showorgunit])|(select <UserTypeEntity>)]
#	[([start <Time>] [end <Time>])|(range <Time> <Time>)|
#	 yesterday|today|thismonth|(previousmonths <Integer>)]
#	[filter <String> (filtertime<String> <Time>)*]
#	[event|events <EventNameList>] [ip <String>]
#	[groupidfilter <String>]
#	[maxactivities <Number>] [maxevents <Number>] [maxresults <Number>]
#	[countsonly [bydate|summary] [eventrowfilter]]
#	(addcsvdata <FieldName> <String>)* [shownoactivities]
# gam report users|user [todrive <ToDriveAttribute>*]
#	[(user all|<UserItem>)|(orgunit|org|ou <OrgUnitPath> [showorgunit])|(select <UserTypeEntity>)]
#	[allverifyuser <UserItem>]
#	[(date <Date>)|(range <Date> <Date>)|
#	 yesterday|today|thismonth|(previousmonths <Integer>)]
#	[nodatechange | (fulldatarequired all|<UserServiceNameList>)]
#	[filter <String> (filtertime<String> <Time>)*]
#	[(fields|parameters <String>)|(services <UserServiceNameList>)]
#	[aggregatebydate|aggregatebyuser [Boolean]]
#	[maxresults <Number>]
#	[convertmbtogb]
# gam report customers|customer|domain [todrive <ToDriveAttribute>*]
#	[(date <Date>)|(range <Date> <Date>)|
#	 yesterday|today|thismonth|(previousmonths <Integer>)]
#	[nodatechange | (fulldatarequired all|<CustomerServiceNameList>)]
#	[(fields|parameters <String>)|(services <CustomerServiceNameList>)] [noauthorizedapps]
#	[convertmbtogb]
def doReport():
  def processUserUsage(usage, lastDate):
    if not usage:
      return (True, lastDate)
    if lastDate == usage[0]['date']:
      return (False, lastDate)
    lastDate = usage[0]['date']
    for user_report in usage:
      if 'entity' not in user_report:
        continue
      row = {'date': user_report['date']}
      if 'userEmail' in user_report['entity']:
        row['email'] = user_report['entity']['userEmail']
        if showOrgUnit:
          row['orgUnitPath'] = userOrgUnits.get(row['email'], UNKNOWN)
      else:
        row['email'] = UNKNOWN
      for item in user_report.get('parameters', []):
        if 'name' not in item:
          continue
        name = item['name']
        repsvc, _ = name.split(':', 1)
        if repsvc not in includeServices:
          continue
        if name == 'accounts:disabled_reason' and 'stringValue' in item:
          mg = DISABLED_REASON_TIME_PATTERN.match(item['stringValue'])
          if mg:
            try:
              disabledTime = formatLocalTime(datetime.datetime.strptime(mg.group(1), '%Y/%m/%d-%H:%M:%S').replace(tzinfo=iso8601.UTC).strftime(YYYYMMDDTHHMMSSZ_FORMAT))
              row['accounts:disabled_time'] = disabledTime
              csvPF.AddTitles('accounts:disabled_time')
            except ValueError:
              pass
        elif convertMbToGb and name.endswith('_in_mb'):
          name = convertReportMBtoGB(name, item)
        csvPF.AddTitles(name)
        for ptype in REPORTS_PARAMETERS_SIMPLE_TYPES:
          if ptype in item:
            if ptype != 'datetimeValue':
              row[name] = item[ptype]
            else:
              row[name] = formatLocalTime(item[ptype])
            break
        else:
          row[name] = ''
      csvPF.WriteRow(row)
    return (True, lastDate)

  def processAggregateUserUsageByUser(usage, lastDate):
    if not usage:
      return (True, lastDate)
    if lastDate == usage[0]['date']:
      return (False, lastDate)
    lastDate = usage[0]['date']
    for user_report in usage:
      if 'entity' not in user_report:
        continue
      if 'userEmail' not in user_report['entity']:
        continue
      email = user_report['entity']['userEmail']
      for item in user_report.get('parameters', []):
        if 'name' not in item:
          continue
        name = item['name']
        repsvc, _ = name.split(':', 1)
        if repsvc not in includeServices:
          continue
        if 'intValue' in item:
          if convertMbToGb and name.endswith('_in_mb'):
            name = convertReportMBtoGB(name, None)
          csvPF.AddTitles(name)
          eventCounts.setdefault(email, {})
          eventCounts[email].setdefault(name, 0)
          eventCounts[email][name] += int(item['intValue'])
    return (True, lastDate)

  def processAggregateUserUsageByDate(usage, lastDate):
    if not usage:
      return (True, lastDate)
    if lastDate == usage[0]['date']:
      return (False, lastDate)
    lastDate = usage[0]['date']
    for user_report in usage:
      if 'entity' not in user_report:
        continue
      for item in user_report.get('parameters', []):
        if 'name' not in item:
          continue
        name = item['name']
        repsvc, _ = name.split(':', 1)
        if repsvc not in includeServices:
          continue
        if 'intValue' in item:
          if convertMbToGb and name.endswith('_in_mb'):
            name = convertReportMBtoGB(name, None)
          csvPF.AddTitles(name)
          eventCounts.setdefault(lastDate, {})
          eventCounts[lastDate].setdefault(name, 0)
          eventCounts[lastDate][name] += int(item['intValue'])
    return (True, lastDate)

  def processCustomerUsageOneRow(usage, lastDate):
    if not usage:
      return (True, lastDate)
    if lastDate == usage[0]['date']:
      return (False, lastDate)
    lastDate = usage[0]['date']
    row = {'date': lastDate}
    for item in usage[0].get('parameters', []):
      if 'name' not in item:
        continue
      name = item['name']
      repsvc, _ = name.split(':', 1)
      if repsvc not in includeServices:
        continue
      for ptype in REPORTS_PARAMETERS_SIMPLE_TYPES:
        if ptype in item:
          if convertMbToGb and name.endswith('_in_mb'):
            name = convertReportMBtoGB(name, item)
          csvPF.AddTitles(name)
          if ptype != 'datetimeValue':
            row[name] = item[ptype]
          else:
            row[name] = formatLocalTime(item[ptype])
          break
      else:
        if 'msgValue' in item:
          if name == 'accounts:authorized_apps':
            if noAuthorizedApps:
              continue
            for app in item['msgValue']:
              appName = f'App: {escapeCRsNLs(app["client_name"])}'
              for key in ['num_users', 'client_id']:
                title = f'{appName}{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{key}'
                csvPF.AddTitles(title)
                row[title] = app[key]
          elif name == 'cros:device_version_distribution':
            versions = {}
            for version in item['msgValue']:
              versions[version['version_number']] = version['num_devices']
            for k, v in sorted(versions.items(), reverse=True):
              title = f'cros:device_version{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{k}'
              csvPF.AddTitles(title)
              row[title] = v
          else:
            values = []
            for subitem in item['msgValue']:
              if 'count' in subitem:
                mycount = myvalue = None
                for key, value in subitem.items():
                  if key == 'count':
                    mycount = value
                  else:
                    myvalue = value
                  if mycount and myvalue:
                    values.append(f'{myvalue}:{mycount}')
                value = ' '.join(values)
              elif 'version_number' in subitem and 'num_devices' in subitem:
                values.append(f'{subitem["version_number"]}:{subitem["num_devices"]}')
              else:
                continue
              value = ' '.join(sorted(values, reverse=True))
            csvPF.AddTitles(name)
            row['name'] = value
    csvPF.WriteRow(row)
    return (True, lastDate)

  def processCustomerUsage(usage, lastDate):
    if not usage:
      return (True, lastDate)
    if lastDate == usage[0]['date']:
      return (False, lastDate)
    lastDate = usage[0]['date']
    for item in usage[0].get('parameters', []):
      if 'name' not in item:
        continue
      name = item['name']
      repsvc, _ = name.split(':', 1)
      if repsvc not in includeServices:
        continue
      for ptype in REPORTS_PARAMETERS_SIMPLE_TYPES:
        if ptype in item:
          if ptype != 'datetimeValue':
            if convertMbToGb and name.endswith('_in_mb'):
              name = convertReportMBtoGB(name, item)
            csvPF.WriteRow({'date': lastDate, 'name': name, 'value': item[ptype]})
          else:
            csvPF.WriteRow({'date': lastDate, 'name': name, 'value': formatLocalTime(item[ptype])})
          break
      else:
        if 'msgValue' in item:
          if name == 'accounts:authorized_apps':
            if noAuthorizedApps:
              continue
            for subitem in item['msgValue']:
              app = {'date': lastDate}
              for an_item in subitem:
                if an_item == 'client_name':
                  app['name'] = f'App: {escapeCRsNLs(subitem[an_item])}'
                elif an_item == 'num_users':
                  app['value'] = f'{subitem[an_item]} users'
                elif an_item == 'client_id':
                  app['client_id'] = subitem[an_item]
              authorizedApps.append(app)
          elif name == 'cros:device_version_distribution':
            values = []
            for subitem in item['msgValue']:
              values.append(f'{subitem["version_number"]}:{subitem["num_devices"]}')
            csvPF.WriteRow({'date': lastDate, 'name': name, 'value': ' '.join(sorted(values, reverse=True))})
          else:
            values = []
            for subitem in item['msgValue']:
              if 'count' in subitem:
                mycount = myvalue = None
                for key, value in subitem.items():
                  if key == 'count':
                    mycount = value
                  else:
                    myvalue = value
                  if mycount and myvalue:
                    values.append(f'{myvalue}:{mycount}')
              else:
                continue
            csvPF.WriteRow({'date': lastDate, 'name': name, 'value': ' '.join(sorted(values, reverse=True))})
    csvPF.SortRowsTwoTitles('date', 'name', False)
    if authorizedApps:
      csvPF.AddTitle('client_id')
      for row in sorted(authorizedApps, key=lambda k: (k['date'], k['name'].lower())):
        csvPF.WriteRow(row)
    return (True, lastDate)

  # dynamically extend our choices with other reports Google dynamically adds
  rep = buildGAPIObject(API.REPORTS)
  dyn_choices = rep._rootDesc \
          .get('resources', {}) \
          .get('activities', {}) \
          .get('methods', {}) \
          .get('list', {}) \
          .get('parameters', {}) \
          .get('applicationName', {}) \
          .get('enum', [])
  for dyn_choice in dyn_choices:
    if dyn_choice.replace('_', '') not in REPORT_CHOICE_MAP and \
       dyn_choice not in REPORT_CHOICE_MAP.values():
      REPORT_CHOICE_MAP[dyn_choice.replace('_', '')] = dyn_choice
  report = getChoice(REPORT_CHOICE_MAP, choiceAliases=REPORT_ALIASES_CHOICE_MAP, mapChoice=True)
  if report == 'usage':
    doReportUsage()
    return
  if report == 'usageparameters':
    doReportUsageParameters()
    return
  customerId = GC.Values[GC.CUSTOMER_ID]
  if customerId == GC.MY_CUSTOMER:
    customerId = None
  csvPF = CSVPrintFile()
  filters = actorIpAddress = groupIdFilter = orgUnit = orgUnitId = None
  showOrgUnit = False
  parameters = set()
  parameterServices = set()
  eventCounts = {}
  eventNames = []
  startEndTime = StartEndTime('start', 'end')
  oneDay = datetime.timedelta(days=1)
  filterTimes = {}
  maxActivities = maxEvents = 0
  maxResults = 1000
  aggregateByDate = aggregateByUser = convertMbToGb = countsOnly = countsByDate = countsSummary = \
    eventRowFilter = exitUserLoop = noAuthorizedApps = normalizeUsers = select = userCustomerRange = False
  limitDateChanges = -1
  allVerifyUser = userKey = 'all'
  cd = orgUnit = orgUnitId = None
  userOrgUnits = {}
  customerReports = userReports = False
  if report == 'customer':
    customerReports = True
    service = rep.customerUsageReports()
    fullDataServices = CUSTOMER_REPORT_SERVICES
  elif report == 'user':
    userReports = True
    service = rep.userUsageReport()
    fullDataServices = USER_REPORT_SERVICES
  else:
    service = rep.activities()
  usageReports = customerReports or userReports
  activityReports = not usageReports
  dataRequiredServices = set()
  addCSVData = {}
  showNoActivities = False
  if usageReports:
    includeServices = set()
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif myarg in {'range', 'thismonth', 'previousmonths'}:
      startEndTime.Get(myarg)
      userCustomerRange = True
    elif myarg in {'ou', 'org', 'orgunit'}:
      if cd is None:
        cd = buildGAPIObject(API.DIRECTORY)
      orgUnit, orgUnitId = getOrgUnitId(cd)
      select = False
    elif myarg == 'showorgunit':
      showOrgUnit = True
    elif usageReports and myarg in {'date', 'yesterday', 'today'}:
      startEndTime.Get('start' if myarg == 'date' else myarg)
      startEndTime.endDateTime = startEndTime.startDateTime
      userCustomerRange = False
    elif usageReports and myarg in {'nodatechange', 'limitdatechanges'}:
      if myarg == 'nodatechange':
        limitDateChanges = 0
      else:
        limitDateChanges = getInteger(minVal=-1)
      if (limitDateChanges == 0) and (startEndTime.startDateTime is not None) and (startEndTime.endDateTime == startEndTime.startDateTime):
        userCustomerRange = True
    elif usageReports and myarg in {'fields', 'parameters'}:
      for field in getString(Cmd.OB_STRING).replace(',', ' ').split():
        if ':' in field:
          repsvc, _ = field.split(':', 1)
          if repsvc in fullDataServices:
            parameters.add(field)
            parameterServices.add(repsvc)
            includeServices.add(repsvc)
          else:
            invalidChoiceExit(repsvc, fullDataServices, True)
        else:
          Cmd.Backup()
          invalidArgumentExit('service:parameter')
    elif usageReports and myarg == 'fulldatarequired':
      fdr = getString(Cmd.OB_SERVICE_NAME_LIST, minLen=0).lower()
      if fdr:
        if fdr != 'all':
          for repsvc in fdr.replace(',', ' ').split():
            if repsvc in fullDataServices:
              dataRequiredServices.add(repsvc)
            else:
              invalidChoiceExit(repsvc, fullDataServices, True)
        else:
          dataRequiredServices = fullDataServices
    elif usageReports and myarg in {'service', 'services'}:
      for repsvc in getString(Cmd.OB_SERVICE_NAME_LIST).lower().replace(',', ' ').split():
        if repsvc in fullDataServices:
          parameterServices.add(repsvc)
          includeServices.add(repsvc)
        else:
          invalidChoiceExit(repsvc, fullDataServices, True)
    elif usageReports and myarg == 'convertmbtogb':
      convertMbToGb = True
    elif customerReports and myarg == 'noauthorizedapps':
      noAuthorizedApps = True
    elif activityReports and myarg == 'maxactivities':
      maxActivities = getInteger(minVal=0)
    elif activityReports and myarg == 'maxevents':
      maxEvents = getInteger(minVal=0)
    elif activityReports and myarg in {'start', 'starttime', 'end', 'endtime', 'yesterday', 'today'}:
      startEndTime.Get(myarg)
    elif activityReports and myarg in {'event', 'events'}:
      for event in getString(Cmd.OB_EVENT_NAME_LIST).replace(',', ' ').split():
        event = event.lower() if report not in REPORT_ACTIVITIES_UPPERCASE_EVENTS else event.upper()
        if event not in eventNames:
          eventNames.append(event)
    elif activityReports and myarg == 'ip':
      actorIpAddress = getString(Cmd.OB_STRING)
    elif activityReports and myarg == 'countsonly':
      countsOnly = True
    elif activityReports and myarg == 'bydate':
      countsByDate = True
    elif activityReports and myarg == 'summary':
      countsSummary = True
    elif activityReports and myarg == 'eventrowfilter':
      eventRowFilter = True
    elif activityReports and myarg == 'groupidfilter':
      groupIdFilter = getString(Cmd.OB_STRING)
    elif activityReports and myarg == 'addcsvdata':
      k = getString(Cmd.OB_STRING)
      addCSVData[k] = getString(Cmd.OB_STRING, minLen=0)
    elif activityReports and myarg == 'shownoactivities':
      showNoActivities = True
    elif not customerReports and myarg.startswith('filtertime'):
      filterTimes[myarg] = getTimeOrDeltaFromNow()
    elif not customerReports and myarg in {'filter', 'filters'}:
      filters = getString(Cmd.OB_STRING)
    elif not customerReports and myarg == 'maxresults':
      maxResults = getInteger(minVal=1, maxVal=1000)
    elif not customerReports and myarg == 'user':
      userKey = getString(Cmd.OB_EMAIL_ADDRESS)
      orgUnit = orgUnitId = None
      select = False
    elif not customerReports and myarg == 'select':
      _, users = getEntityToModify(defaultEntityType=Cmd.ENTITY_USERS)
      orgUnit = orgUnitId = None
      select = True
    elif userReports and myarg == 'aggregatebydate':
      aggregateByDate = getBoolean()
    elif userReports and myarg == 'aggregatebyuser':
      aggregateByUser = getBoolean()
    elif userReports and myarg == 'allverifyuser':
      allVerifyUser = getEmailAddress()
    else:
      unknownArgumentExit()
  if aggregateByDate and aggregateByUser:
    usageErrorExit(Msg.ARE_MUTUALLY_EXCLUSIVE.format('aggregateByDate', 'aggregateByUser'))
  if countsOnly and countsByDate and countsSummary:
    usageErrorExit(Msg.ARE_MUTUALLY_EXCLUSIVE.format('bydate', 'summary'))
  parameters = ','.join(parameters) if parameters else None
  if usageReports and not includeServices:
    includeServices = set(fullDataServices)
  if filterTimes and filters is not None:
    for filterTimeName, filterTimeValue in filterTimes.items():
      filters = filters.replace(f'#{filterTimeName}#', filterTimeValue)
  if not orgUnitId:
    showOrgUnit = False
  if userReports:
    if startEndTime.startDateTime is None:
      startEndTime.startDateTime = startEndTime.endDateTime = todaysDate()
    if select:
      normalizeUsers = True
      orgUnitId = None
      Ent.SetGetting(Ent.REPORT)
    elif userKey == 'all':
      if orgUnitId:
        if showOrgUnit:
          userOrgUnits = getUserOrgUnits(cd, orgUnit, orgUnitId)
        forWhom = f'users in orgUnit {orgUnit}'
      else:
        forWhom = 'all users'
      printGettingEntityItemForWhom(Ent.REPORT, forWhom)
      users = ['all']
    else:
      Ent.SetGetting(Ent.REPORT)
      users = [normalizeEmailAddressOrUID(userKey)]
      orgUnitId = None
    if aggregateByDate:
      titles = ['date']
    elif aggregateByUser:
      titles = ['email'] if not showOrgUnit else ['email', 'orgUnitPath']
    else:
      titles = ['email', 'date'] if not showOrgUnit else ['email', 'orgUnitPath', 'date']
    csvPF.SetTitles(titles)
    csvPF.SetSortAllTitles()
    i = 0
    count = len(users)
    for user in users:
      i += 1
      if normalizeUsers:
        user = normalizeEmailAddressOrUID(user)
      if user != 'all':
        printGettingEntityItemForWhom(Ent.REPORT, user, i, count)
        verifyUser = user
      else:
        verifyUser = allVerifyUser
      startDateTime = startEndTime.startDateTime
      endDateTime = startEndTime.endDateTime
      lastDate = None
      numDateChanges = 0
      while startDateTime <= endDateTime:
        tryDate = startDateTime.strftime(YYYYMMDD_FORMAT)
        try:
          if not userCustomerRange:
            result = callGAPI(service, 'get',
                              throwReasons=[GAPI.INVALID, GAPI.INVALID_INPUT, GAPI.BAD_REQUEST, GAPI.FORBIDDEN],
                              userKey=verifyUser, date=tryDate, customerId=customerId,
                              orgUnitID=orgUnitId, parameters=parameters,
                              fields='warnings,usageReports', maxResults=1)
            prevTryDate = tryDate
            fullData, tryDate, usageReports = _checkDataRequiredServices(result, tryDate,
                                                                         dataRequiredServices, parameterServices, True)
            if fullData < 0:
              printWarningMessage(DATA_NOT_AVALIABLE_RC, Msg.NO_REPORT_AVAILABLE.format(report))
              break
            numDateChanges += 1
            if fullData == 0:
              if numDateChanges > limitDateChanges >= 0:
                break
              startDateTime = endDateTime = datetime.datetime.strptime(tryDate, YYYYMMDD_FORMAT)
              continue
          if not select and userKey == 'all':
            pageMessage = getPageMessageForWhom(forWhom, showDate=tryDate)
          else:
            pageMessage = getPageMessageForWhom(user, showDate=tryDate)
          prevTryDate = tryDate
          usage = callGAPIpages(service, 'get', 'usageReports',
                                pageMessage=pageMessage,
                                throwReasons=[GAPI.INVALID, GAPI.INVALID_INPUT, GAPI.BAD_REQUEST, GAPI.FORBIDDEN],
                                retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                                userKey=user, date=tryDate, customerId=customerId,
                                orgUnitID=orgUnitId, filters=filters, parameters=parameters,
                                maxResults=maxResults)
          if aggregateByDate:
            status, lastDate = processAggregateUserUsageByDate(usage, lastDate)
          elif aggregateByUser:
            status, lastDate = processAggregateUserUsageByUser(usage, lastDate)
          else:
            status, lastDate = processUserUsage(usage, lastDate)
          if not status:
            break
        except GAPI.invalid as e:
          if userCustomerRange:
            break
          numDateChanges += 1
          tryDate = _adjustTryDate(str(e), numDateChanges, limitDateChanges, tryDate)
          if not tryDate:
            break
          startDateTime = endDateTime = datetime.datetime.strptime(tryDate, YYYYMMDD_FORMAT)
          continue
        except GAPI.invalidInput as e:
          systemErrorExit(GOOGLE_API_ERROR_RC, str(e))
        except GAPI.badRequest:
          if user != 'all':
            entityUnknownWarning(Ent.USER, user, i, count)
          else:
            printErrorMessage(BAD_REQUEST_RC, Msg.BAD_REQUEST)
            exitUserLoop = True
          break
        except GAPI.forbidden as e:
          accessErrorExit(None, str(e))
        startDateTime += oneDay
      if exitUserLoop:
        break
      if user != 'all' and lastDate is None and GC.Values[GC.CSV_OUTPUT_USERS_AUDIT]:
        csvPF.WriteRowNoFilter({'date': prevTryDate, 'email': user})
    if aggregateByDate:
      for usageDate, events in eventCounts.items():
        row = {'date': usageDate}
        for event, count in events.items():
          if convertMbToGb and event.endswith('_in_gb'):
            count = f'{count/1024:.2f}'
          row[event] = count
        csvPF.WriteRow(row)
      csvPF.SortRows('date', False)
      csvPF.writeCSVfile(f'User Reports Aggregate - {tryDate}')
    elif aggregateByUser:
      for email, events in eventCounts.items():
        row = {'email': email}
        if showOrgUnit:
          row['orgUnitPath'] = userOrgUnits.get(email, UNKNOWN)
        for event, count in events.items():
          if convertMbToGb and event.endswith('_in_gb'):
            count = f'{count/1024:.2f}'
          row[event] = count
        csvPF.WriteRow(row)
      csvPF.SortRows('email', False)
      csvPF.writeCSVfile('User Reports Aggregate - User')
    else:
      csvPF.SortRowsTwoTitles('email', 'date', False)
      csvPF.writeCSVfile(f'User Reports - {tryDate}')
  elif customerReports:
    if startEndTime.startDateTime is None:
      startEndTime.startDateTime = startEndTime.endDateTime = todaysDate()
    csvPF.SetTitles('date')
    if not userCustomerRange or (startEndTime.startDateTime == startEndTime.endDateTime):
      csvPF.AddTitles(['name', 'value'])
    authorizedApps = []
    startDateTime = startEndTime.startDateTime
    endDateTime = startEndTime.endDateTime
    lastDate = None
    numDateChanges = 0
    while startDateTime <= endDateTime:
      tryDate = startDateTime.strftime(YYYYMMDD_FORMAT)
      try:
        if not userCustomerRange:
          result = callGAPI(service, 'get',
                            throwReasons=[GAPI.INVALID, GAPI.INVALID_INPUT, GAPI.BAD_REQUEST, GAPI.FORBIDDEN],
                            date=tryDate, customerId=customerId, parameters=parameters, fields='warnings,usageReports')
          fullData, tryDate, usageReports = _checkDataRequiredServices(result, tryDate,
                                                                       dataRequiredServices, parameterServices)
          if fullData < 0:
            printWarningMessage(DATA_NOT_AVALIABLE_RC, Msg.NO_REPORT_AVAILABLE.format(report))
            break
          numDateChanges += 1
          if fullData == 0:
            if numDateChanges > limitDateChanges >= 0:
              break
            startDateTime = endDateTime = datetime.datetime.strptime(tryDate, YYYYMMDD_FORMAT)
            continue
        usage = callGAPIpages(service, 'get', 'usageReports',
                              throwReasons=[GAPI.INVALID, GAPI.INVALID_INPUT, GAPI.FORBIDDEN],
                              date=tryDate, customerId=customerId, parameters=parameters)
        if not userCustomerRange or (startEndTime.startDateTime == startEndTime.endDateTime):
          status, lastDate = processCustomerUsage(usage, lastDate)
        else:
          status, lastDate = processCustomerUsageOneRow(usage, lastDate)
        if not status:
          break
      except GAPI.invalid as e:
        if userCustomerRange:
          break
        numDateChanges += 1
        tryDate = _adjustTryDate(str(e), numDateChanges, limitDateChanges, tryDate)
        if not tryDate:
          break
        startDateTime = endDateTime = datetime.datetime.strptime(tryDate, YYYYMMDD_FORMAT)
        continue
      except GAPI.invalidInput as e:
        systemErrorExit(GOOGLE_API_ERROR_RC, str(e))
      except GAPI.forbidden as e:
        accessErrorExit(None, str(e))
      startDateTime += oneDay
    csvPF.writeCSVfile(f'Customer Report - {tryDate}')
  else: # activityReports
    csvPF.SetTitles('name')
    if addCSVData:
      csvPF.AddTitles(sorted(addCSVData.keys()))
    if select:
      pageMessage = None
      normalizeUsers = True
      orgUnitId = None
    elif userKey == 'all':
      if orgUnitId:
        if showOrgUnit:
          userOrgUnits = getUserOrgUnits(cd, orgUnit, orgUnitId)
        printGettingEntityItemForWhom(Ent.REPORT, f'users in orgUnit {orgUnit}')
      else:
        printGettingEntityItemForWhom(Ent.REPORT, 'all users')
      pageMessage = getPageMessage()
      users = ['all']
    else:
      Ent.SetGetting(Ent.ACTIVITY)
      pageMessage = getPageMessage()
      users = [normalizeEmailAddressOrUID(userKey)]
      orgUnitId = None
    zeroEventCounts = {}
    if not eventNames:
      eventNames.append(None)
    else:
      for eventName in eventNames:
        zeroEventCounts[eventName] = 0
    i = 0
    count = len(users)
    for user in users:
      i += 1
      if normalizeUsers:
        user = normalizeEmailAddressOrUID(user)
      if select or user != 'all':
        printGettingEntityItemForWhom(Ent.ACTIVITY, user, i, count)
      for eventName in eventNames:
        try:
          feed = callGAPIpages(service, 'list', 'items',
                               pageMessage=pageMessage, maxItems=maxActivities,
                               throwReasons=[GAPI.BAD_REQUEST, GAPI.INVALID, GAPI.INVALID_INPUT, GAPI.AUTH_ERROR, GAPI.SERVICE_NOT_AVAILABLE],
                               retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                               applicationName=report, userKey=user, customerId=customerId,
                               actorIpAddress=actorIpAddress, orgUnitID=orgUnitId,
                               startTime=startEndTime.startTime, endTime=startEndTime.endTime,
                               eventName=eventName, filters=filters, groupIdFilter=groupIdFilter,
                               maxResults=maxResults)
        except GAPI.badRequest:
          if user != 'all':
            entityUnknownWarning(Ent.USER, user, i, count)
            continue
          printErrorMessage(BAD_REQUEST_RC, Msg.BAD_REQUEST)
          break
        except (GAPI.invalid, GAPI.invalidInput, GAPI.serviceNotAvailable) as e:
          systemErrorExit(GOOGLE_API_ERROR_RC, str(e))
        except GAPI.authError:
          accessErrorExit(None)
        for activity in feed:
          events = activity.pop('events')
          actor = activity['actor'].get('email')
          if not actor:
            actor = 'id:'+activity['actor'].get('profileId', UNKNOWN)
          if showOrgUnit:
            activity['actor']['orgUnitPath'] = userOrgUnits.get(actor, UNKNOWN)
          if countsOnly and countsByDate:
            eventTime = activity.get('id', {}).get('time', UNKNOWN)
            if eventTime != UNKNOWN:
              try:
                eventTime, _ = iso8601.parse_date(eventTime)
              except (iso8601.ParseError, OverflowError):
                eventTime = UNKNOWN
            if eventTime != UNKNOWN:
              eventDate = eventTime.strftime(YYYYMMDD_FORMAT)
            else:
              eventDate = UNKNOWN
          if not countsOnly or eventRowFilter:
            activity_row = flattenJSON(activity, timeObjects=REPORT_ACTIVITIES_TIME_OBJECTS)
            purge_parameters = True
            numEvents = 0
            for event in events:
              numEvents += 1
              for item in event.get('parameters', []):
                itemSet = set(item)
                if not itemSet.symmetric_difference({'name'}):
                  event[item['name']] = ''
                elif not itemSet.symmetric_difference({'value', 'name'}):
                  event[item['name']] = NL_SPACES_PATTERN.sub('', item['value'])
                elif not itemSet.symmetric_difference({'intValue', 'name'}):
                  if item['name'] in {'start_time', 'end_time'}:
                    val = item.get('intValue')
                    if val is not None:
                      val = int(val)
                      if val >= 62135683200:
                        event[item['name']] = ISOformatTimeStamp(datetime.datetime.fromtimestamp(val-62135683200, GC.Values[GC.TIMEZONE]))
                      else:
                        event[item['name']] = val
                  else:
                    event[item['name']] = item['intValue']
                elif not itemSet.symmetric_difference({'boolValue', 'name'}):
                  event[item['name']] = item['boolValue']
                elif not itemSet.symmetric_difference({'multiValue', 'name'}):
                  event[item['name']] = ' '.join(item['multiValue'])
                elif item['name'] == 'scope_data':
                  parts = {}
                  for message in item['multiMessageValue']:
                    for mess in message['parameter']:
                      value = mess.get('value', ' '.join(mess.get('multiValue', [])))
                      parts[mess['name']] = parts.get(mess['name'], [])+[value]
                  for part, v in parts.items():
                    if part == 'scope_name':
                      part = 'scope'
                    event[part] = ' '.join(v)
                else:
                  purge_parameters = False
              if purge_parameters:
                event.pop('parameters', None)
              row = flattenJSON(event)
              row.update(activity_row)
              if not countsOnly:
                if addCSVData:
                  row.update(addCSVData)
                csvPF.WriteRowTitles(row)
                if numEvents >= maxEvents > 0:
                  break
              elif csvPF.CheckRowTitles(row):
                eventName = event['name']
                if not countsSummary:
                  eventCounts.setdefault(actor, {})
                  if not countsByDate:
                    eventCounts[actor].setdefault(eventName, 0)
                    eventCounts[actor][eventName] += 1
                  else:
                    eventCounts[actor].setdefault(eventDate, {})
                    eventCounts[actor][eventDate].setdefault(eventName, 0)
                    eventCounts[actor][eventDate][eventName] += 1
                else:
                  eventCounts.setdefault(eventName, 0)
                  eventCounts[eventName] += 1
          elif not countsSummary:
            eventCounts.setdefault(actor, {})
            if not countsByDate:
              for event in events:
                eventName = event['name']
                eventCounts[actor].setdefault(eventName, 0)
                eventCounts[actor][eventName] += 1
            else:
              for event in events:
                eventName = event['name']
                eventCounts[actor].setdefault(eventDate, {})
                eventCounts[actor][eventDate].setdefault(eventName, 0)
                eventCounts[actor][eventDate][eventName] += 1
          else:
            for event in events:
              eventCounts.setdefault(event['name'], 0)
              eventCounts[event['name']] += 1
    if not countsOnly:
      if not csvPF.rows and showNoActivities:
        row = {'name': 'NoActivities'}
        if addCSVData:
          row.update(addCSVData)
        csvPF.WriteRowTitles(row)
    else:
      if eventRowFilter:
        csvPF.SetRowFilter([], GC.Values[GC.CSV_OUTPUT_ROW_FILTER_MODE])
        csvPF.SetRowDropFilter([], GC.Values[GC.CSV_OUTPUT_ROW_DROP_FILTER_MODE])
      if not countsSummary:
        titles = ['emailAddress']
        if countsOnly and countsByDate:
          titles.append('date')
        csvPF.SetTitles(titles)
        csvPF.SetSortTitles(titles)
        if addCSVData:
          csvPF.AddTitles(sorted(addCSVData.keys()))
        if eventCounts:
          if not countsByDate:
            for actor, events in eventCounts.items():
              row = {'emailAddress': actor}
              row.update(zeroEventCounts)
              for event, count in events.items():
                row[event] = count
              if addCSVData:
                row.update(addCSVData)
              csvPF.WriteRowTitles(row)
          else:
            for actor, eventDates in eventCounts.items():
              for eventDate, events in eventDates.items():
                row = {'emailAddress': actor, 'date': eventDate}
                row.update(zeroEventCounts)
                for event, count in events.items():
                  row[event] = count
                if addCSVData:
                  row.update(addCSVData)
                csvPF.WriteRowTitles(row)
        elif showNoActivities:
          row = {'emailAddress': 'NoActivities'}
          if addCSVData:
            row.update(addCSVData)
            csvPF.WriteRow(row)
      else:
        csvPF.SetTitles(['event', 'count'])
        if addCSVData:
          csvPF.AddTitles(sorted(addCSVData.keys()))
        if eventCounts:
          for event, count in sorted(eventCounts.items()):
            row = {'event': event, 'count': count}
            if addCSVData:
              row.update(addCSVData)
            csvPF.WriteRow(row)
        elif showNoActivities:
          row = {'event': 'NoActivities', 'count': 0}
          if addCSVData:
            row.update(addCSVData)
          csvPF.WriteRow(row)
    csvPF.writeCSVfile(f'{report.capitalize()} Activity Report', eventRowFilter)

# Substitute for #user#, #email#, #usernamne#
def _substituteForUser(field, user, userName):
  if field.find('#') == -1:
    return field
  return field.replace('#user#', user).replace('#email#', user).replace('#username#', userName)

# Tag utilities
TAG_ADDRESS_ARGUMENT_TO_FIELD_MAP = {
  'country': 'country',
  'countrycode': 'countryCode',
  'customtype': 'customType',
  'extendedaddress': 'extendedAddress',
  'formatted': 'formatted',
  'locality': 'locality',
  'pobox': 'poBox',
  'postalcode': 'postalCode',
  'primary': 'primary',
  'region': 'region',
  'streetaddress': 'streetAddress',
  'type': 'type',
  }

TAG_EMAIL_ARGUMENT_TO_FIELD_MAP = {
  'domain': 'domain',
  'primaryemail': 'primaryEmail',
  'username': 'username',
  }

TAG_EXTERNALID_ARGUMENT_TO_FIELD_MAP = {
  'customtype': 'customType',
  'type': 'type',
  'value': 'value',
  }

TAG_GENDER_ARGUMENT_TO_FIELD_MAP = {
  'addressmeas': 'addressMeAs',
  'customgender': 'customGender',
  'type': 'type',
  }

TAG_IM_ARGUMENT_TO_FIELD_MAP = {
  'customprotocol': 'customProtocol',
  'customtype': 'customType',
  'im': 'im',
  'protocol': 'protocol',
  'primary': 'primary',
  'type': 'type',
  }

TAG_KEYWORD_ARGUMENT_TO_FIELD_MAP = {
  'customtype': 'customType',
  'type': 'type',
  'value': 'value',
  }

TAG_LOCATION_ARGUMENT_TO_FIELD_MAP = {
  'area': 'area',
  'buildingid': 'buildingId',
  'buildingname': 'buildingName',
  'customtype': 'customType',
  'deskcode': 'deskCode',
  'floorname': 'floorName',
  'floorsection': 'floorSection',
  'type': 'type',
  }

TAG_NAME_ARGUMENT_TO_FIELD_MAP = {
  'familyname': 'familyName',
  'fullname': 'fullName',
  'givenname': 'givenName',
  }

TAG_ORGANIZATION_ARGUMENT_TO_FIELD_MAP = {
  'costcenter': 'costCenter',
  'costcentre': 'costCenter',
  'customtype': 'customType',
  'department': 'department',
  'description': 'description',
  'domain': 'domain',
  'fulltimeequivalent': 'fullTimeEquivalent',
  'location': 'location',
  'name': 'name',
  'primary': 'primary',
  'symbol': 'symbol',
  'title': 'title',
  'type': 'type',
  }

TAG_OTHEREMAIL_ARGUMENT_TO_FIELD_MAP = {
  'address': 'address',
  'customtype': 'customType',
  'primary': 'primary',
  'type': 'type',
  }

TAG_PHONE_ARGUMENT_TO_FIELD_MAP = {
  'customtype': 'customType',
  'primary': 'primary',
  'type': 'type',
  'value': 'value',
  }

TAG_POSIXACCOUNT_ARGUMENT_TO_FIELD_MAP = {
  'accountid': 'accountId',
  'gecos': 'gecos',
  'gid': 'gid',
  'homedirectory': 'homeDirectory',
  'operatingsystemtype': 'operatingSystemType',
  'primary': 'primary',
  'shell': 'shell',
  'systemid': 'systemId',
  'uid': 'uid',
  'username': 'username',
  }

TAG_RELATION_ARGUMENT_TO_FIELD_MAP = {
  'customtype': 'customType',
  'type': 'type',
  'value': 'value',
  }

TAG_SSHPUBLICKEY_ARGUMENT_TO_FIELD_MAP = {
  'expirationtimeusec': 'expirationTimeUsec',
  'fingerprint': 'fingerprint',
  'key': 'key',
  }

TAG_WEBSITE_ARGUMENT_TO_FIELD_MAP = {
  'customtype': 'customType',
  'primary': 'primary',
  'type': 'type',
  'value': 'value',
  }

TAG_FIELD_SUBFIELD_CHOICE_MAP = {
  'address': ('addresses', TAG_ADDRESS_ARGUMENT_TO_FIELD_MAP),
  'addresses': ('addresses', TAG_ADDRESS_ARGUMENT_TO_FIELD_MAP),
  'email': ('primaryEmail', TAG_EMAIL_ARGUMENT_TO_FIELD_MAP),
  'externalid': ('externalIds', TAG_EXTERNALID_ARGUMENT_TO_FIELD_MAP),
  'externalids': ('externalIds', TAG_EXTERNALID_ARGUMENT_TO_FIELD_MAP),
  'gender': ('gender', TAG_GENDER_ARGUMENT_TO_FIELD_MAP),
  'im': ('ims', TAG_IM_ARGUMENT_TO_FIELD_MAP),
  'ims': ('ims', TAG_IM_ARGUMENT_TO_FIELD_MAP),
  'keyword': ('keywords', TAG_KEYWORD_ARGUMENT_TO_FIELD_MAP),
  'keywords': ('keywords', TAG_KEYWORD_ARGUMENT_TO_FIELD_MAP),
  'location': ('locations', TAG_LOCATION_ARGUMENT_TO_FIELD_MAP),
  'locations': ('locations', TAG_LOCATION_ARGUMENT_TO_FIELD_MAP),
  'name': ('name', TAG_NAME_ARGUMENT_TO_FIELD_MAP),
  'organization': ('organizations', TAG_ORGANIZATION_ARGUMENT_TO_FIELD_MAP),
  'organizations': ('organizations', TAG_ORGANIZATION_ARGUMENT_TO_FIELD_MAP),
  'organisation': ('organizations', TAG_ORGANIZATION_ARGUMENT_TO_FIELD_MAP),
  'organisations': ('organizations', TAG_ORGANIZATION_ARGUMENT_TO_FIELD_MAP),
  'otheremail': ('emails', TAG_OTHEREMAIL_ARGUMENT_TO_FIELD_MAP),
  'otheremails': ('emails', TAG_OTHEREMAIL_ARGUMENT_TO_FIELD_MAP),
  'phone': ('phones', TAG_PHONE_ARGUMENT_TO_FIELD_MAP),
  'phones': ('phones', TAG_PHONE_ARGUMENT_TO_FIELD_MAP),
  'photourl': ('thumbnailPhotoUrl', {'': 'thumbnailPhotoUrl'}),
  'posix': ('posixAccounts', TAG_POSIXACCOUNT_ARGUMENT_TO_FIELD_MAP),
  'posixaccounts': ('posixAccounts', TAG_POSIXACCOUNT_ARGUMENT_TO_FIELD_MAP),
  'relation': ('relations', TAG_RELATION_ARGUMENT_TO_FIELD_MAP),
  'relations': ('relations', TAG_RELATION_ARGUMENT_TO_FIELD_MAP),
  'sshkeys': ('sshPublicKeys', TAG_SSHPUBLICKEY_ARGUMENT_TO_FIELD_MAP),
  'sshpublickeys': ('sshPublicKeys', TAG_SSHPUBLICKEY_ARGUMENT_TO_FIELD_MAP),
  'website': ('websites', TAG_WEBSITE_ARGUMENT_TO_FIELD_MAP),
  'websites': ('websites', TAG_WEBSITE_ARGUMENT_TO_FIELD_MAP),
  }

def _initTagReplacements():
  return {'cd': None, 'tags': {}, 'subs': False,
          'fieldsSet': set(), 'fields': '',
          'schemasSet': set(), 'customFieldMask': None}

def _getTagReplacement(myarg, tagReplacements, allowSubs):
  if myarg == 'replace':
    trregex = None
  elif myarg == 'replaceregex':
    trregex = getREPatternSubstitution(re.IGNORECASE)
  else:
    return False
  matchTag = getString(Cmd.OB_TAG)
  matchReplacement = getString(Cmd.OB_STRING, minLen=0)
  if matchReplacement.startswith('field:'):
    if not allowSubs:
      usageErrorExit(Msg.USER_SUBS_NOT_ALLOWED_TAG_REPLACEMENT)
    tagReplacements['subs'] = True
    field = matchReplacement[6:].strip().lower()
    if field.find('.') != -1:
      args = field.split('.', 3)
      field = args[0]
      subfield = args[1]
      if len(args) == 2:
        matchfield = matchvalue = ''
      elif len(args) == 4:
        matchfield = args[2]
        matchvalue = args[3]
        if matchfield == 'primary':
          matchvalue = matchvalue.lower()
          if matchvalue == TRUE:
            matchvalue = True
          elif matchvalue == FALSE:
            matchvalue = ''
          else:
            invalidChoiceExit(matchvalue, TRUE_FALSE, True)
      else:
        Cmd.Backup()
        usageErrorExit(Msg.INVALID_TAG_SPECIFICATION)
    elif field == 'photourl':
      subfield = matchfield = matchvalue = ''
    else:
      field = ''
    if not field or field not in TAG_FIELD_SUBFIELD_CHOICE_MAP:
      invalidChoiceExit(field, TAG_FIELD_SUBFIELD_CHOICE_MAP, True)
    field, subfieldsChoiceMap = TAG_FIELD_SUBFIELD_CHOICE_MAP[field]
    if subfield not in subfieldsChoiceMap:
      invalidChoiceExit(subfield, subfieldsChoiceMap, True)
    subfield = subfieldsChoiceMap[subfield]
    if matchfield:
      if matchfield not in subfieldsChoiceMap:
        invalidChoiceExit(matchfield, subfieldsChoiceMap, True)
      matchfield = subfieldsChoiceMap[matchfield]
    tagReplacements['fieldsSet'].add(field)
    tagReplacements['fields'] = ','.join(tagReplacements['fieldsSet'])
    tagReplacements['tags'][matchTag] = {'field': field, 'subfield': subfield,
                                         'matchfield': matchfield, 'matchvalue': matchvalue, 'value': '',
                                         'trregex': trregex}
    if field == 'locations' and subfield == 'buildingName':
      _makeBuildingIdNameMap()
  elif matchReplacement.startswith('schema:'):
    if not allowSubs:
      usageErrorExit(Msg.USER_SUBS_NOT_ALLOWED_TAG_REPLACEMENT)
    tagReplacements['subs'] = True
    matchReplacement = matchReplacement[7:].strip()
    if matchReplacement.find('.') != -1:
      schemaName, schemaField = matchReplacement.split('.', 1)
    else:
      schemaName = ''
    if not schemaName or not schemaField:
      invalidArgumentExit(Cmd.OB_SCHEMA_NAME_FIELD_NAME)
    tagReplacements['fieldsSet'].add('customSchemas')
    tagReplacements['fields'] = ','.join(tagReplacements['fieldsSet'])
    tagReplacements['schemasSet'].add(schemaName)
    tagReplacements['customFieldMask'] = ','.join(tagReplacements['schemasSet'])
    tagReplacements['tags'][matchTag] = {'schema': schemaName, 'schemafield': schemaField, 'value': '',
                                         'trregex': trregex}
  elif ((matchReplacement.find('#') >= 0) and
        (matchReplacement.find('#user#') >= 0) or (matchReplacement.find('#email#') >= 0) or (matchReplacement.find('#username#') >= 0)):
    if not allowSubs:
      usageErrorExit(Msg.USER_SUBS_NOT_ALLOWED_TAG_REPLACEMENT)
    tagReplacements['subs'] = True
    tagReplacements['tags'][matchTag] = {'template': matchReplacement, 'value': '',
                                         'trregex': trregex}
  else:
    if trregex is None:
      tagReplacements['tags'][matchTag] = {'value': matchReplacement}
    else:
      tagReplacements['tags'][matchTag] = {'value': re.sub(trregex[0], trregex[1], matchReplacement)}
  return True

def _getTagReplacementFieldValues(user, i, count, tagReplacements, results=None):
  if results is None:
    if tagReplacements['fields'] and tagReplacements['fields'] != 'primaryEmail':
      if not tagReplacements['cd']:
        tagReplacements['cd'] = buildGAPIObject(API.DIRECTORY)
      try:
        results = callGAPI(tagReplacements['cd'].users(), 'get',
                           throwReasons=GAPI.USER_GET_THROW_REASONS,
                           userKey=user, projection='custom', customFieldMask=tagReplacements['customFieldMask'], fields=tagReplacements['fields'])
      except (GAPI.userNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden, GAPI.badRequest, GAPI.backendError, GAPI.systemError):
        entityUnknownWarning(Ent.USER, user, i, count)
        return
    else:
      results = {'primaryEmail': user}
  userName, domain = splitEmailAddress(user)
  for tag in tagReplacements['tags'].values():
    if tag.get('field'):
      field = tag['field']
      if field == 'primaryEmail':
        subfield = tag['subfield']
        if subfield == 'username':
          tag['value'] = userName
        elif subfield == 'domain':
          tag['value'] = domain
        else:
          tag['value'] = user
      else:
        if field in ['addresses', 'emails', 'ims', 'organizations', 'phones', 'posixAccounts', 'websites']:
          items = results.get(field, [])
          if not tag['matchfield']:
            for data in items:
              if data.get('primary'):
                break
            else:
              if items:
                data = items[0]
              else:
                data = {}
          else:
            for data in items:
              if data.get(tag['matchfield'], '') == tag['matchvalue']:
                break
            else:
              data = {}
        elif field in ['externalIds', 'relations', 'sshPublicKeys']:
          items = results.get(field, [])
          if not tag['matchfield']:
            if items:
              data = items[0]
            else:
              data = {}
          else:
            for data in items:
              if data.get(tag['matchfield'], '') == tag['matchvalue']:
                break
            else:
              data = {}
        elif field in ['keywords', 'locations']:
          items = results.get(field, [])
          if not tag['matchfield']:
            if items:
              data = items[0]
              data['buildingName'] = GM.Globals[GM.MAP_BUILDING_ID_TO_NAME].get(data.get('buildingId', ''), '')
            else:
              data = {}
          else:
            for data in items:
              if data.get(tag['matchfield'], '') == tag['matchvalue']:
                break
            else:
              data = {}
        elif field == 'thumbnailPhotoUrl':
          data = results
        else:
          data = results.get(field, {})
        tag['value'] = str(data.get(tag['subfield'], ''))
    elif tag.get('schema'):
      tag['value'] = str(results.get('customSchemas', {}).get(tag['schema'], {}).get(tag['schemafield'], ''))
    elif tag.get('template'):
      tag['value'] = _substituteForUser(tag['template'], user, userName)
    trregex = tag.get('trregex', None)
    if trregex is not None:
      tag['value'] = re.sub(trregex[0], trregex[1], tag['value'])

RTL_PATTERN = re.compile(r'(?s){RTL}.*?{/RTL}')
RT_PATTERN = re.compile(r'(?s){RT}.*?{/RT}')
TAG_REPLACE_PATTERN = re.compile(r'{(.+?)}')
RT_MARKERS = {'RT', '/RT', 'RTL', '/RTL'}
PC_PATTERN = re.compile(r'(?s){PC}.*?{/PC}')
UC_PATTERN = re.compile(r'(?s){UC}.*?{/UC}')
LC_PATTERN = re.compile(r'(?s){LC}.*?{/LC}')
CASE_MARKERS = {'PC', '/PC', 'UC', '/UC', 'LC', '/LC'}
SKIP_PATTERNS = [re.compile(r'<head>.*?</head>', flags=re.IGNORECASE),
                 re.compile(r'<script>.*?</script>', flags=re.IGNORECASE)]

def _processTagReplacements(tagReplacements, message):
  def pcase(trstring):
    new = ''
    # state = True: Upshift 1st letter found
    # state = False: Downshift subsequent letters
    state = True
    for c in trstring:
      if state:
        if c.isalpha():
          new += c.upper()
          state = False
        else:
          new += c
      else:
        if c.isalpha():
          new += c.lower()
        else:
          state = True
          new += c
    return new

  def ucase(trstring):
    return trstring.upper()

  def lcase(trstring):
    return trstring.lower()

  def _processCase(message, casePattern, caseFunc):
# Find all {xC}.*?{/xC} sequences
    pos = 0
    while True:
      match = casePattern.search(message, pos)
      if not match:
        return message
      start, end = match.span()
      for skipArea in skipAreas:
        if start >= skipArea[0] and end <= skipArea[1]:
          break
      else:
        message = message[:start]+caseFunc(message[start+4:end-5])+message[end:]
      pos = end

# Identify areas of message to avoid replacements
  skipAreas = []
  for pattern in SKIP_PATTERNS:
    pos = 0
    while True:
      match = pattern.search(message, pos)
      if not match:
        break
      skipAreas.append(match.span())
      pos = match.end()
  skipTags = set()
# Find all {tag}, note replacement value and starting location; note tags in skipAreas
  tagFields = []
  tagSubs = {}
  pos = 0
  while True:
    match = TAG_REPLACE_PATTERN.search(message, pos)
    if not match:
      break
    start, end = match.span()
    tag = match.group(1)
    if tag in CASE_MARKERS:
      pass
    elif tag not in RT_MARKERS:
      for skipArea in skipAreas:
        if start >= skipArea[0] and end <= skipArea[1]:
          skipTags.add(tag)
          break
      else:
        tagSubs.setdefault(tag, tagReplacements['tags'].get(tag, {'value': ''})['value'])
        tagFields.append((tagSubs[tag], start))
    pos = end
# Find all {RT}.*?{/RT} sequences
# If any non-empty {tag} replacement value falls between them, then mark {RT} and {/RT} to be stripped
# Otherwise, mark the entire {RT}.*?{/RT} sequence to be stripped
  rtStrips = []
  pos = 0
  while True:
    match = RT_PATTERN.search(message, pos)
    if not match:
      break
    start, end = match.span()
    stripEntireRT = True
    hasTags = False
    for tagField in tagFields:
      if tagField[1] >= end:
        break
      if tagField[1] >= start:
        hasTags = True
        if tagField[0]:
          rtStrips.append((False, start, start+4))
          rtStrips.append((False, end-5, end))
          stripEntireRT = False
          break
    if stripEntireRT:
      if hasTags or start+4 == end-5:
        rtStrips.append((True, start, end))
      else:
        rtStrips.append((False, start, start+4))
        rtStrips.append((False, end-5, end))
    pos = end
# Find all {RTL}.*?{/RTL} sequences
# If any non-empty {RT}...{tag}... {/RT} falls between them, then mark {RTL} and {/RTL} to be stripped
# Otherwise, mark the entire {RTL}.*{/RTL} sequence to be stripped
  rtlStrips = []
  pos = 0
  while True:
    match = RTL_PATTERN.search(message, pos)
    if not match:
      break
    start, end = match.span()
    stripEntireRTL = True
    hasTags = False
    for tagField in tagFields:
      if tagField[1] >= end:
        break
      if tagField[1] >= start:
        hasTags = True
        if tagField[0]:
          rtlStrips.append((False, start, start+5, end-6, end))
          stripEntireRTL = False
          break
    if stripEntireRTL:
      for rtStrip in rtStrips:
        if rtStrip[1] >= end:
          break
        if rtStrip[1] >= start:
          hasTags = True
          if not rtStrip[0]:
            rtlStrips.append((False, start, start+5, end-6, end))
            stripEntireRTL = False
            break
    if stripEntireRTL:
      if hasTags or start+5 == end-6:
        rtlStrips.append((True, start, end))
      else:
        rtlStrips.append((False, start, start+5, end-6, end))
    pos = end
  if rtlStrips:
    allStrips = []
    i = 0
    l = len(rtStrips)
    for rtlStrip in rtlStrips:
      while i < l and rtStrips[i][1] < rtlStrip[1]:
        allStrips.append(rtStrips[i])
        i += 1
      allStrips.append((False, rtlStrip[1], rtlStrip[2]))
      if not rtlStrip[0]:
        while i < l and rtStrips[i][1] < rtlStrip[3]:
          allStrips.append(rtStrips[i])
          i += 1
        allStrips.append((False, rtlStrip[3], rtlStrip[4]))
      else:
        while i < l and rtStrips[i][1] < rtlStrip[2]:
          i += 1
    while i < l:
      allStrips.append(rtStrips[i])
      i += 1
  else:
    allStrips = rtStrips
# Strip {RTL} {/RTL}, {RT} {/RT}, {RTL}.*?{/RTL}, {RT}.*?{/RT} sequences
  for rtStrip in allStrips[::-1]:
    message = message[:rtStrip[1]]+message[rtStrip[2]:]
# Strip {RTL} {/RTL}, {RT} {/RT}, {RTL}.*?{/RTL}, {RT}.*?{/RT} sequences
# Make {tag} replacements; ignore tags in skipAreas
  pos = 0
  while True:
    match = TAG_REPLACE_PATTERN.search(message, pos)
    if not match:
      break
    start, end = match.span()
    tag = match.group(1)
    if tag in CASE_MARKERS:
      pos = end
    elif tag not in RT_MARKERS:
      if tag not in skipTags:
        message = re.sub(match.group(0), tagSubs[tag], message)
        pos = start+1
      else:
        pos = end
    else:
# Replace invalid RT tags with ERROR(RT)
      message = re.sub(match.group(0), f'ERROR({tag})', message)
      pos = start+1
# Process case changes
  message = _processCase(message, PC_PATTERN, pcase)
  message = _processCase(message, UC_PATTERN, ucase)
  message = _processCase(message, LC_PATTERN, lcase)
  return message

def sendCreateUpdateUserNotification(body, basenotify, tagReplacements, i=0, count=0, msgFrom=None, createMessage=True):
  def _makeSubstitutions(field):
    notify[field] = _substituteForUser(notify[field], body['primaryEmail'], userName)
    notify[field] = notify[field].replace('#domain#', domain)
    notify[field] = notify[field].replace('#givenname#', body['name'].get('givenName', ''))
    notify[field] = notify[field].replace('#familyname#', body['name'].get('familyName', ''))

  def _makePasswordSubstitutions(field, html):
    if not html:
      notify[field] = notify[field].replace('#password#', notify['password'])
    else:
      notify[field] = notify[field].replace('#password#', notify['password'].replace('&', '&amp;').replace('<', '&lt;').replace('>', '&gt;'))

  userName, domain = splitEmailAddress(body['primaryEmail'])
  notify = basenotify.copy()
  if not notify['subject']:
    notify['subject'] = Msg.CREATE_USER_NOTIFY_SUBJECT if createMessage else Msg.UPDATE_USER_PASSWORD_CHANGE_NOTIFY_SUBJECT
  _makeSubstitutions('subject')
  if not notify['message']:
    notify['message'] = Msg.CREATE_USER_NOTIFY_MESSAGE if createMessage else Msg.UPDATE_USER_PASSWORD_CHANGE_NOTIFY_MESSAGE
  elif notify['html']:
    notify['message'] = notify['message'].replace('\r', '').replace('\\n', '<br/>')
  else:
    notify['message'] = notify['message'].replace('\r', '').replace('\\n', '\n')
  _makeSubstitutions('message')
  if tagReplacements['subs']:
    _getTagReplacementFieldValues(body['primaryEmail'], i, count, tagReplacements, body if createMessage else None)
  notify['subject'] = _processTagReplacements(tagReplacements, notify['subject'])
  notify['message'] = _processTagReplacements(tagReplacements, notify['message'])
  _makePasswordSubstitutions('subject', False)
  _makePasswordSubstitutions('message', notify['html'])
  if 'from' in notify:
    msgFrom = notify['from']
  msgReplyTo = notify.get('replyto', None)
  mailBox = notify.get('mailbox', None)
  for recipient in notify['recipients']:
    send_email(notify['subject'], notify['message'], recipient, i, count,
               msgFrom=msgFrom, msgReplyTo=msgReplyTo, html=notify['html'], charset=notify['charset'], mailBox=mailBox)

def getRecipients():
  if checkArgumentPresent('select'):
    _, recipients = getEntityToModify(defaultEntityType=Cmd.ENTITY_USERS)
    return [normalizeEmailAddressOrUID(emailAddress, noUid=True, noLower=True) for emailAddress in recipients]
  return getNormalizedEmailAddressEntity(shlexSplit=True, noLower=True)

# gam sendemail [recipient|to] <RecipientEntity> [from <EmailAddress>] [mailbox <EmailAddress>] [replyto <EmailAddress>]
#	[cc <RecipientEntity>] [bcc <RecipientEntity>] [singlemessage]
#	[subject <String>]
#	[<MessageContent>]
#	(replace <Tag> <String>)*
#	(replaceregex <REMatchPattern> <RESubstitution>  <Tag> <String>)*
#	[html [<Boolean>]] (attach <FileName> [charset <CharSet>])*
#	(embedimage <FileName> <String>)*
#	[newuser <EmailAddress> firstname|givenname <String> lastname|familyname <string> password <Password>]
#	(<SMTPDateHeader> <Time>)* (<SMTPHeader> <String>)* (header <String> <String>)*
# gam <UserTypeEntity> sendemail recipient|to <RecipientEntity> [replyto <EmailAddress>]
#	[cc <RecipientEntity>] [bcc <RecipientEntity>] [singlemessage]
#	[subject <String>]
#	[<MessageContent>]
#	(replace <Tag> <String>)*
#	(replaceregex <REMatchPattern> <RESubstitution>  <Tag> <String>)*
#	[html [<Boolean>]] (attach <FileName> [charset <CharSet>])*
#	(embedimage <FileName> <String>)*
#	[newuser <EmailAddress> firstname|givenname <String> lastname|familyname <string> password <Password>]
#	(<SMTPDateHeader> <Time>)* (<SMTPHeader> <String>)* (header <String> <String>)*
# gam <UserTypeEntity> sendemail from <EmailAddress> [replyto <EmailAddress>]
#	[cc <RecipientEntity>] [bcc <RecipientEntity>] [singlemessage]
#	[subject <String>]
#	[<MessageContent>]
#	(replace <Tag> <String>)*
#	(replaceregex <REMatchPattern> <RESubstitution>  <Tag> <String>)*
#	[html [<Boolean>]] (attach <FileName> [charset <CharSet>])*
#	(embedimage <FileName> <String>)*
#	[newuser <EmailAddress> firstname|givenname <String> lastname|familyname <string> password <Password>]
#	(<SMTPDateHeader> <Time>)* (<SMTPHeader> <String>)* (header <String> <String>)*
def doSendEmail(users=None):
  body = {}
  notify = {'subject': '', 'message': '', 'html': False, 'charset': UTF8, 'password': ''}
  msgFroms = [_getAdminEmail()]
  count = 1
  if users is None:
    checkArgumentPresent({'recipient', 'recipients', 'to'})
    recipients = getRecipients()
  else:
    _, count, entityList = getEntityArgument(users)
    if checkArgumentPresent({'recipient', 'recipients', 'to'}):
      msgFroms = [normalizeEmailAddressOrUID(entity) for entity in entityList]
      recipients = getRecipients()
    elif checkArgumentPresent({'from'}):
      recipients = [normalizeEmailAddressOrUID(entity) for entity in entityList]
      msgFroms = [getString(Cmd.OB_EMAIL_ADDRESS)]
      count = 1
    else:
      missingArgumentExit('recipient|to|from')
  msgHeaders = {}
  ccRecipients = []
  bccRecipients = []
  mailBox = None
  msgReplyTo = None
  singleMessage = False
  tagReplacements = _initTagReplacements()
  attachments = []
  embeddedImages = []
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if users is None and myarg == 'from':
      msgFroms = [getString(Cmd.OB_EMAIL_ADDRESS)]
      count = 1
    elif myarg == 'replyto':
      msgReplyTo = getString(Cmd.OB_EMAIL_ADDRESS)
    elif myarg == 'subject':
      notify['subject'] = getString(Cmd.OB_STRING)
    elif myarg in SORF_MSG_FILE_ARGUMENTS:
      notify['message'], notify['charset'], notify['html'] = getStringOrFile(myarg)
    elif myarg == 'cc':
      ccRecipients = getRecipients()
    elif myarg == 'bcc':
      bccRecipients = getRecipients()
    elif myarg == 'mailbox':
      mailBox = getString(Cmd.OB_EMAIL_ADDRESS)
    elif myarg == 'singlemessage':
      singleMessage = True
    elif myarg == 'html':
      notify['html'] = getBoolean()
    elif myarg == 'newuser':
      body['primaryEmail'] = getEmailAddress()
    elif myarg in {'firstname', 'givenname'}:
      body.setdefault('name', {})
      body['name']['givenName'] = getString(Cmd.OB_STRING, minLen=0, maxLen=60)
    elif myarg in {'lastname', 'familyname'}:
      body.setdefault('name', {})
      body['name']['familyName'] = getString(Cmd.OB_STRING, minLen=0, maxLen=60)
    elif myarg in {'password', 'notifypassword'}:
      body['password'] = notify['password'] = getString(Cmd.OB_PASSWORD, maxLen=100)
    elif _getTagReplacement(myarg, tagReplacements, False):
      pass
    elif myarg == 'attach':
      attachments.append((getFilename(), getCharSet()))
    elif myarg == 'embedimage':
      embeddedImages.append((getFilename(), getString(Cmd.OB_STRING)))
    elif myarg in SMTP_HEADERS_MAP:
      if myarg in SMTP_DATE_HEADERS:
        msgDate, _, _ = getTimeOrDeltaFromNow(True)
        msgHeaders[SMTP_HEADERS_MAP[myarg]] = formatdate(time.mktime(msgDate.timetuple()) + msgDate.microsecond/1E6, True)
      else:
        msgHeaders[SMTP_HEADERS_MAP[myarg]] = getString(Cmd.OB_STRING)
    elif myarg == 'header':
      header = getString(Cmd.OB_STRING, minLen=1)
      msgHeaders[SMTP_HEADERS_MAP.get(header.lower(), header)] = getString(Cmd.OB_STRING)
    else:
      unknownArgumentExit()
  notify['message'] = notify['message'].replace('\r', '').replace('\\n', '\n')
  if tagReplacements['tags']:
    notify['message'] = _processTagReplacements(tagReplacements, notify['message'])
  if tagReplacements['tags']:
    notify['subject'] = _processTagReplacements(tagReplacements, notify['subject'])
  jcount = len(recipients)
  if body.get('primaryEmail'):
    if (recipients and ('password' in body) and ('name' in body) and ('givenName' in body['name']) and ('familyName' in body['name'])):
      notify['recipients'] = recipients
      sendCreateUpdateUserNotification(body, notify, tagReplacements, msgFrom=msgFroms[0])
    else:
      usageErrorExit(Msg.NEWUSER_REQUIREMENTS, True)
    return
  if ccRecipients or bccRecipients:
    singleMessage = True
  i = 0
  for msgFrom in msgFroms:
    i += 1
    if singleMessage:
      entityPerformActionModifierNumItems([Ent.USER, msgFrom],
                                          Act.MODIFIER_TO, jcount+len(ccRecipients)+len(bccRecipients), Ent.RECIPIENT, i, count)
      send_email(notify['subject'], notify['message'], ','.join(recipients), i, count,
                 msgFrom=msgFrom, msgReplyTo=msgReplyTo, html=notify['html'], charset=notify['charset'],
                 attachments=attachments, embeddedImages=embeddedImages, msgHeaders=msgHeaders,
                 ccRecipients=','.join(ccRecipients), bccRecipients=','.join(bccRecipients),
                 mailBox=mailBox)
    else:
      entityPerformActionModifierNumItems([Ent.USER, msgFrom], Act.MODIFIER_TO, jcount, Ent.RECIPIENT, i, count)
      Ind.Increment()
      j = 0
      for recipient in recipients:
        j += 1
        send_email(notify['subject'], notify['message'], recipient, j, jcount,
                   msgFrom=msgFrom, msgReplyTo=msgReplyTo, html=notify['html'], charset=notify['charset'],
                   attachments=attachments, embeddedImages=embeddedImages, msgHeaders=msgHeaders, mailBox=mailBox)
      Ind.Decrement()

ADDRESS_FIELDS_PRINT_ORDER = ['contactName', 'organizationName', 'addressLine1', 'addressLine2', 'addressLine3', 'locality', 'region', 'postalCode', 'countryCode']

def _showCustomerAddressPhoneNumber(customerInfo):
  if 'postalAddress' in customerInfo:
    printKeyValueList(['Address', None])
    Ind.Increment()
    for field in ADDRESS_FIELDS_PRINT_ORDER:
      if field in customerInfo['postalAddress']:
        printKeyValueList([field, customerInfo['postalAddress'][field]])
    Ind.Decrement()
  if 'phoneNumber' in customerInfo:
    printKeyValueList(['Phone', customerInfo['phoneNumber']])

ADDRESS_FIELDS_ARGUMENT_MAP = {
  'contact': 'contactName', 'contactname': 'contactName',
  'name': 'organizationName', 'organizationname': 'organizationName', 'organisationname': 'organizationName',
  'address': 'addressLine1', 'address1': 'addressLine1', 'addressline1': 'addressLine1',
  'address2': 'addressLine2', 'addressline2': 'addressLine2',
  'address3': 'addressLine3', 'addressline3': 'addressLine3',
  'city': 'locality', 'locality': 'locality',
  'state': 'region', 'region': 'region',
  'zipcode': 'postalCode', 'postal': 'postalCode', 'postalcode': 'postalCode',
  'country': 'countryCode', 'countrycode': 'countryCode',
  }

def _getResoldCustomerAttr():
  body = {}
  customerAuthToken = None
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg in ADDRESS_FIELDS_ARGUMENT_MAP:
      body.setdefault('postalAddress', {})
      body['postalAddress'][ADDRESS_FIELDS_ARGUMENT_MAP[myarg]] = getString(Cmd.OB_STRING, minLen=0, maxLen=255)
    elif myarg in {'email', 'alternateemail'}:
      body['alternateEmail'] = getEmailAddress(noUid=True)
    elif myarg in {'phone', 'phonenumber'}:
      body['phoneNumber'] = getString(Cmd.OB_STRING, minLen=0)
    elif myarg in {'customerauthtoken', 'transfertoken'}:
      customerAuthToken = getString(Cmd.OB_STRING)
    else:
      unknownArgumentExit()
  return customerAuthToken, body

# gam create resoldcustomer <CustomerDomain> (customer_auth_token <String>) <ResoldCustomerAttribute>+
def doCreateResoldCustomer():
  res = buildGAPIObject(API.RESELLER)
  customerDomain = getString('customerDomain')
  customerAuthToken, body = _getResoldCustomerAttr()
  body['customerDomain'] = customerDomain
  try:
    result = callGAPI(res.customers(), 'insert',
                      throwReasons=GAPI.RESELLER_THROW_REASONS,
                      body=body, customerAuthToken=customerAuthToken, fields='customerId')
    entityActionPerformed([Ent.CUSTOMER_DOMAIN, body['customerDomain'], Ent.CUSTOMER_ID, result['customerId']])
  except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden, GAPI.invalid) as e:
    entityActionFailedWarning([Ent.CUSTOMER_DOMAIN, body['customerDomain']], str(e))

# gam update resoldcustomer <CustomerID> <ResoldCustomerAttribute>+
def doUpdateResoldCustomer():
  res = buildGAPIObject(API.RESELLER)
  customerId = getString(Cmd.OB_CUSTOMER_ID)
  _, body = _getResoldCustomerAttr()
  try:
    callGAPI(res.customers(), 'patch',
             throwReasons=GAPI.RESELLER_THROW_REASONS,
             customerId=customerId, body=body, fields='')
    entityActionPerformed([Ent.CUSTOMER_ID, customerId])
  except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden, GAPI.invalid) as e:
    entityActionFailedWarning([Ent.CUSTOMER_ID, customerId], str(e))

# gam info resoldcustomer <CustomerID> [formatjson]
def doInfoResoldCustomer():
  res = buildGAPIObject(API.RESELLER)
  customerId = getString(Cmd.OB_CUSTOMER_ID)
  FJQC = FormatJSONQuoteChar()
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    FJQC.GetFormatJSON(myarg)
  try:
    customerInfo = callGAPI(res.customers(), 'get',
                            throwReasons=GAPI.RESELLER_THROW_REASONS,
                            customerId=customerId)
    if not FJQC.formatJSON:
      printKeyValueList(['Customer ID', customerInfo['customerId']])
      printKeyValueList(['Customer Type', customerInfo['customerType']])
      printKeyValueList(['Customer Domain', customerInfo['customerDomain']])
      if 'customerDomainVerified' in customerInfo:
        printKeyValueList(['Customer Domain Verified', customerInfo['customerDomainVerified']])
      _showCustomerAddressPhoneNumber(customerInfo)
      primaryEmail = customerInfo.get('primaryAdmin', {}).get('primaryEmail')
      if primaryEmail:
        printKeyValueList(['Customer Primary Email', primaryEmail])
      if 'alternateEmail' in customerInfo:
        printKeyValueList(['Customer Alternate Email', customerInfo['alternateEmail']])
      printKeyValueList(['Customer Admin Console URL', customerInfo['resourceUiUrl']])
    else:
      printLine(json.dumps(cleanJSON(customerInfo), ensure_ascii=False, sort_keys=False))
  except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden, GAPI.invalid) as e:
    entityActionFailedWarning([Ent.CUSTOMER_ID, customerId], str(e))

def getCustomerSubscription(res):
  customerId = getString(Cmd.OB_CUSTOMER_ID)
  productId, skuId = SKU.getProductAndSKU(getString(Cmd.OB_SKU_ID))
  if not productId:
    invalidChoiceExit(skuId, SKU.getSortedSKUList(), True)
  try:
    subscriptions = callGAPIpages(res.subscriptions(), 'list', 'subscriptions',
                                  throwReasons=GAPI.RESELLER_THROW_REASONS,
                                  customerId=customerId, fields='nextPageToken,subscriptions(skuId,subscriptionId,plan(planName))')
  except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden, GAPI.invalid) as e:
    entityActionFailedWarning([Ent.SUBSCRIPTION, None], str(e))
    sys.exit(GM.Globals[GM.SYSEXITRC])
  for subscription in subscriptions:
    if skuId == subscription['skuId']:
      return (customerId, skuId, subscription['subscriptionId'], subscription['plan']['planName'])
  Cmd.Backup()
  usageErrorExit(f'{Ent.FormatEntityValueList([Ent.CUSTOMER_ID, customerId, Ent.SKU, skuId])}, {Msg.SUBSCRIPTION_NOT_FOUND}')

PLAN_NAME_MAP = {
  'annualmonthlypay': 'ANNUAL_MONTHLY_PAY',
  'annualyearlypay': 'ANNUAL_YEARLY_PAY',
  'flexible': 'FLEXIBLE',
  'free': 'FREE',
  'trial': 'TRIAL',
  }

def _getResoldSubscriptionAttr(customerId):
  body = {'customerId': customerId,
          'plan': {},
          'seats': {},
          'skuId': None,
         }
  customerAuthToken = None
  seats1 = seats2 = None
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg in {'deal', 'dealcode'}:
      body['dealCode'] = getString('dealCode')
    elif myarg in {'plan', 'planname'}:
      body['plan']['planName'] = getChoice(PLAN_NAME_MAP, mapChoice=True)
    elif myarg in {'purchaseorderid', 'po'}:
      body['purchaseOrderId'] = getString('purchaseOrderId')
    elif myarg == 'seats':
      seats1 = getInteger(minVal=0)
      if Cmd.ArgumentsRemaining() and Cmd.Current().isdigit():
        seats2 = getInteger(minVal=0)
    elif myarg in {'sku', 'skuid'}:
      productId, body['skuId'] = SKU.getProductAndSKU(getString(Cmd.OB_SKU_ID))
      if not productId:
        invalidChoiceExit(body['skuId'], SKU.getSortedSKUList(), True)
    elif myarg in {'customerauthtoken', 'transfertoken'}:
      customerAuthToken = getString('customer_auth_token')
    else:
      unknownArgumentExit()
  for field in ['plan', 'skuId']:
    if not body[field]:
      missingArgumentExit(field.lower())
  if seats1 is None:
    missingArgumentExit('seats')
  if body['plan']['planName'].startswith('ANNUAL'):
    body['seats']['numberOfSeats'] = seats1
  else:
    body['seats']['maximumNumberOfSeats'] = seats1 if seats2 is None else seats2
  return customerAuthToken, body

SUBSCRIPTION_SKIP_OBJECTS = {'customerId', 'skuId', 'subscriptionId'}
SUBSCRIPTION_TIME_OBJECTS = {'creationTime', 'startTime', 'endTime', 'trialEndTime', 'transferabilityExpirationTime'}

def _showSubscription(subscription, FJQC=None):
  if FJQC is not None and FJQC.formatJSON:
    printLine(json.dumps(cleanJSON(subscription, timeObjects=SUBSCRIPTION_TIME_OBJECTS), ensure_ascii=False, sort_keys=False))
    return
  Ind.Increment()
  printEntity([Ent.SUBSCRIPTION, subscription['subscriptionId']])
  showJSON(None, subscription, SUBSCRIPTION_SKIP_OBJECTS, SUBSCRIPTION_TIME_OBJECTS)
  Ind.Decrement()

# gam create resoldsubscription <CustomerID> (sku <SKUID>)
#	 (plan annual_monthly_pay|annual_yearly_pay|flexible|trial) (seats <Number>)
#	 [customer_auth_token <String>] [deal <String>] [purchaseorderid <String>]
def doCreateResoldSubscription():
  res = buildGAPIObject(API.RESELLER)
  customerId = getString(Cmd.OB_CUSTOMER_ID)
  customerAuthToken, body = _getResoldSubscriptionAttr(customerId)
  try:
    subscription = callGAPI(res.subscriptions(), 'insert',
                            throwReasons=GAPI.RESELLER_THROW_REASONS,
                            customerId=customerId, customerAuthToken=customerAuthToken, body=body)
    entityActionPerformed([Ent.CUSTOMER_ID, customerId, Ent.SKU, subscription['skuId']])
    _showSubscription(subscription)
  except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden, GAPI.invalid) as e:
    entityActionFailedWarning([Ent.CUSTOMER_ID, customerId], str(e))

RENEWAL_TYPE_MAP = {
  'autorenewmonthlypay': 'AUTO_RENEW_MONTHLY_PAY',
  'autorenewyearlypay': 'AUTO_RENEW_YEARLY_PAY',
  'cancel': 'CANCEL',
  'renewcurrentusersmonthlypay': 'RENEW_CURRENT_USERS_MONTHLY_PAY',
  'renewcurrentusersyearlypay': 'RENEW_CURRENT_USERS_YEARLY_PAY',
  'switchtopayasyougo': 'SWITCH_TO_PAY_AS_YOU_GO',
  }

# gam update resoldsubscription <CustomerID> <SKUID>
#	activate|suspend|startpaidservice|
#	(renewal auto_renew_monthly_pay|auto_renew_yearly_pay|cancel|renew_current_users_monthly_pay|renew_current_users_yearly_pay|switch_to_pay_as_you_go)|
#	(seats <Number>)|
#	(plan annual_monthly_pay|annual_yearly_pay|flexible|trial|free [deal <String>] [purchaseorderid <String>] [seats <Number>])
def doUpdateResoldSubscription():
  def _getSeats():
    seats1 = getInteger(minVal=0)
    if Cmd.ArgumentsRemaining() and Cmd.Current().isdigit():
      seats2 = getInteger(minVal=0)
    else:
      seats2 = None
    if planName.startswith('ANNUAL'):
      return {'numberOfSeats': seats1}
    return {'maximumNumberOfSeats': seats1 if seats2 is None else seats2}

  res = buildGAPIObject(API.RESELLER)
  function = None
  customerId, skuId, subscriptionId, planName = getCustomerSubscription(res)
  kwargs = {}
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'activate':
      function = 'activate'
    elif myarg == 'suspend':
      function = 'suspend'
    elif myarg == 'startpaidservice':
      function = 'startPaidService'
    elif myarg in {'renewal', 'renewaltype'}:
      function = 'changeRenewalSettings'
      kwargs['body'] = {'renewalType': getChoice(RENEWAL_TYPE_MAP, mapChoice=True)}
    elif myarg == 'seats':
      function = 'changeSeats'
      kwargs['body'] =  _getSeats()
    elif myarg == 'plan':
      function = 'changePlan'
      planName = getChoice(PLAN_NAME_MAP, mapChoice=True)
      kwargs['body'] = {'planName': planName}
      while Cmd.ArgumentsRemaining():
        planarg = getArgument()
        if planarg == 'seats':
          kwargs['body']['seats'] = _getSeats()
        elif planarg in {'purchaseorderid', 'po'}:
          kwargs['body']['purchaseOrderId'] = getString('purchaseOrderId')
        elif planarg in {'dealcode', 'deal'}:
          kwargs['body']['dealCode'] = getString('dealCode')
        else:
          unknownArgumentExit()
    else:
      unknownArgumentExit()
  try:
    subscription = callGAPI(res.subscriptions(), function,
                            throwReasons=GAPI.RESELLER_THROW_REASONS,
                            customerId=customerId, subscriptionId=subscriptionId, **kwargs)
    entityActionPerformed([Ent.CUSTOMER_ID, customerId, Ent.SKU, skuId])
    if subscription:
      _showSubscription(subscription)
  except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden, GAPI.invalid) as e:
    entityActionFailedWarning([Ent.CUSTOMER_ID, customerId], str(e))

DELETION_TYPE_MAP = {
  'cancel': 'cancel',
  'downgrade': 'downgrade',
  'transfertodirect': 'transfer_to_direct',
  }

# gam delete resoldsubscription <CustomerID> <SKUID> cancel|downgrade|transfer_to_direct
def doDeleteResoldSubscription():
  res = buildGAPIObject(API.RESELLER)
  customerId, skuId, subscriptionId, _ = getCustomerSubscription(res)
  deletionType = getChoice(DELETION_TYPE_MAP, mapChoice=True)
  checkForExtraneousArguments()
  try:
    callGAPI(res.subscriptions(), 'delete',
             throwReasons=GAPI.RESELLER_THROW_REASONS,
             customerId=customerId, subscriptionId=subscriptionId, deletionType=deletionType)
    entityActionPerformed([Ent.CUSTOMER_ID, customerId, Ent.SKU, skuId])
  except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden, GAPI.invalid) as e:
    entityActionFailedWarning([Ent.CUSTOMER_ID, customerId, Ent.SKU, skuId], str(e))

# gam info resoldsubscription <CustomerID> <SKUID>
def doInfoResoldSubscription():
  res = buildGAPIObject(API.RESELLER)
  customerId, skuId, subscriptionId, _ = getCustomerSubscription(res)
  FJQC = FormatJSONQuoteChar()
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    FJQC.GetFormatJSON(myarg)
  try:
    subscription = callGAPI(res.subscriptions(), 'get',
                            throwReasons=GAPI.RESELLER_THROW_REASONS,
                            customerId=customerId, subscriptionId=subscriptionId)
    if not FJQC.formatJSON:
      printEntity([Ent.CUSTOMER_ID, customerId, Ent.SKU, skuId])
    _showSubscription(subscription, FJQC)
  except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden, GAPI.invalid) as e:
    entityActionFailedWarning([Ent.CUSTOMER_ID, customerId, Ent.SKU, skuId], str(e))

PRINT_RESOLD_SUBSCRIPTIONS_TITLES = ['customerId', 'skuId', 'subscriptionId']

# gam print resoldsubscriptions [todrive <ToDriveAttribute>*]
#	[customerid <CustomerID>] [customer_auth_token <String>] [customer_prefix <String>]
#	[maxresults <Number>]
#	[formatjson [quotechar <Character>]]
# gam show resoldsubscriptions
#	[customerid <CustomerID>] [customer_auth_token <String>] [customer_prefix <String>]
#	[maxresults <Number>]
#	[formatjson]
def doPrintShowResoldSubscriptions():
  res = buildGAPIObject(API.RESELLER)
  kwargs = {'maxResults': 100}
  csvPF = CSVPrintFile(PRINT_RESOLD_SUBSCRIPTIONS_TITLES, 'sortall') if Act.csvFormat() else None
  FJQC = FormatJSONQuoteChar(csvPF)
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if csvPF and myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif myarg == 'customerid':
      kwargs['customerId'] = getString(Cmd.OB_CUSTOMER_ID)
    elif myarg in {'customerauthtoken', 'transfertoken'}:
      kwargs['customerAuthToken'] = getString(Cmd.OB_CUSTOMER_AUTH_TOKEN)
    elif myarg == 'customerprefix':
      kwargs['customerNamePrefix'] = getString(Cmd.OB_STRING)
    elif myarg == 'maxresults':
      kwargs['maxResults'] = getInteger(minVal=1, maxVal=100)
    else:
      FJQC.GetFormatJSONQuoteChar(myarg, True)
  try:
    subscriptions = callGAPIpages(res.subscriptions(), 'list', 'subscriptions',
                                  throwReasons=GAPI.RESELLER_THROW_REASONS,
                                  fields='nextPageToken,subscriptions', **kwargs)
  except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden, GAPI.invalid) as e:
    entityActionFailedWarning([Ent.SUBSCRIPTION, None], str(e))
    return
  jcount = len(subscriptions)
  if not csvPF:
    if not FJQC.formatJSON:
      performActionNumItems(jcount, Ent.SUBSCRIPTION)
    Ind.Increment()
    j = 0
    for subscription in subscriptions:
      j += 1
      if not FJQC.formatJSON:
        printEntity([Ent.CUSTOMER_ID, subscription['customerId'], Ent.SKU, subscription['skuId']], j, jcount)
      _showSubscription(subscription, FJQC)
    Ind.Decrement()
  else:
    for subscription in subscriptions:
      row = flattenJSON(subscription, timeObjects=SUBSCRIPTION_TIME_OBJECTS)
      if not FJQC.formatJSON:
        csvPF.WriteRowTitles(row)
      elif csvPF.CheckRowTitles(row):
        csvPF.WriteRowNoFilter({'customerId': subscription['customerId'],
                                'skuId': subscription['skuId'],
                                'subscriptionId': subscription['subscriptionId'],
                                'JSON': json.dumps(cleanJSON(subscription, timeObjects=SUBSCRIPTION_TIME_OBJECTS), ensure_ascii=False, sort_keys=True)})
    csvPF.writeCSVfile('Resold Subscriptions')

def normalizeChannelResellerID(resellerId):
  if resellerId.startswith('accounts/'):
    return resellerId
  return f'accounts/{resellerId}'

def normalizeChannelCustomerID(customerId):
  if customerId.startswith('customers/'):
    return customerId
  return f'customers/{customerId}'

def normalizeChannelProductID(productId):
  if productId.startswith('products/'):
    return productId
  return f'products/{productId}'

CHANNEL_ENTITY_MAP = {
  Ent.CHANNEL_CUSTOMER:
    {'JSONtitles': ['name', 'domain', 'JSON'],
     'timeObjects': ['createTime', 'updateTime'],
     'items': 'customers',
     'pageSize': 50,
     'maxPageSize': 50,
     'fields': {
        'name': 'name',
        'orgdisplayname': 'orgDisplayName',
        'orgpostaladdress': 'orgPostalAddress',
        'primarycontactinfo': 'primaryContactInfo',
        'alternateemail': 'alternateEmail',
        'domain': 'domain',
        'createtime': 'createTime',
        'updatetime': 'updateTime',
        'cloudidentityid': 'cloudIdentityId',
        'languagecode': 'languageCode',
        'cloudidentityinfo': 'cloudIdentityInfo',
        'channelpartnerid': 'channelPartnerId',
        }
     },
  Ent.CHANNEL_CUSTOMER_ENTITLEMENT:
    {'JSONtitles': ['name', 'offer', 'JSON'],
     'timeObjects': ['createTime', 'updateTime', 'startTime', 'endTime'],
     'items': 'entitlements',
     'pageSize': 100,
     'maxPageSize': 100,
     'fields': {
        'name': 'name',
        'createtime': 'createTime',
        'updatetime': 'updateTime',
        'offer': 'offer',
        'commitmentsettings': 'commitmentSettings',
        'provisioningstate': 'provisioningState',
        'provisionedservice': 'provisionedService',
        'suspensionreasons': 'suspensionReasons',
        'purchaseorderid': 'purchaseOrderId',
        'trialsettings': 'trialSettings',
        'associationinfo': 'associationInfo',
        'parameters': 'parameters',
        }
     },
  Ent.CHANNEL_OFFER:
    {'JSONtitles': ['name', 'sku', 'JSON'],
     'timeObjects': ['startTime', 'endTime'],
     'items': 'offers',
     'pageSize': 1000,
     'maxPageSize': 1000,
     'fields': {
        'name': 'name',
        'marketinginfo': 'marketingInfo',
        'sku': 'sku',
        'plan': 'plan',
        'constraints': 'constraints',
        'pricebyresources': 'priceByResources',
        'starttime': 'startTime',
        'endtime': 'endTime',
        'parameterdefinitions': 'parameterDefinitions',
        }
     },
  Ent.CHANNEL_PRODUCT:
    {'JSONtitles': ['name', 'JSON'],
     'timeObjects': None,
     'items': 'products',
     'pageSize': 1000,
     'maxPageSize': 1000,
     'fields': {
        'name': 'name',
        'marketinginfo': 'marketingInfo',
        }
     },
  Ent.CHANNEL_SKU:
    {'JSOBtitles': ['name', 'JSON'],
     'timeObjects': None,
     'items': 'skus',
     'pageSize': 1000,
     'maxPageSize': 1000,
     'fields': {
        'name': 'name',
        'marketinginfo': 'marketingInfo',
        'product': 'product',
        }
     }
  }

def doPrintShowChannelItems(entityType):
  cchan = buildGAPIObject(API.CLOUDCHANNEL)
  if entityType == Ent.CHANNEL_CUSTOMER:
    service = cchan.accounts().customers()
  elif entityType == Ent.CHANNEL_CUSTOMER_ENTITLEMENT:
    service = cchan.accounts().customers().entitlements()
  elif entityType == Ent.CHANNEL_OFFER:
    service = cchan.accounts().offers()
  elif entityType == Ent.CHANNEL_PRODUCT:
    service = cchan.products()
  else: #Ent.CHANNEL_SKU
    service = cchan.products().skus()
  channelEntityMap = CHANNEL_ENTITY_MAP[entityType]
#  csvPF = CSVPrintFile(channelEntityMap['titles'], 'sortall') if Act.csvFormat() else None
  csvPF = CSVPrintFile(['name'], 'sortall') if Act.csvFormat() else None
  FJQC = FormatJSONQuoteChar(csvPF)
  fieldsList = []
  resellerId = normalizeChannelResellerID(GC.Values[GC.RESELLER_ID] if GC.Values[GC.RESELLER_ID] else GC.Values[GC.CUSTOMER_ID])
  customerId = normalizeChannelCustomerID(GC.Values[GC.CHANNEL_CUSTOMER_ID])
  name = None
  productId = 'products/-'
  kwargs = {'pageSize': channelEntityMap['pageSize']}
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if csvPF and myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif myarg == 'resellerid':
      resellerId = normalizeChannelResellerID(getString(Cmd.OB_RESELLER_ID))
    elif (entityType == Ent.CHANNEL_CUSTOMER_ENTITLEMENT) and myarg in {'customerid', 'channelcustomerid'}:
      customerId = normalizeChannelCustomerID(getString(Cmd.OB_CHANNEL_CUSTOMER_ID))
    elif (entityType == Ent.CHANNEL_CUSTOMER_ENTITLEMENT) and myarg == 'name':
      name = getString(Cmd.OB_STRING)
      parent = name.split('/')
      if (len(parent) != 4) or (parent[0] != 'accounts') or (not parent[1]) or (parent[2] != 'customers') or (not parent[3]):
        Cmd.Backup()
        usageErrorExit(Msg.INVALID_RESELLER_CUSTOMER_NAME)
    elif (entityType in {Ent.CHANNEL_OFFER, Ent.CHANNEL_PRODUCT, Ent.CHANNEL_SKU}) and myarg == 'language':
      kwargs['languageCode'] = getLanguageCode(LANGUAGE_CODES_MAP)
    elif (entityType in {Ent.CHANNEL_CUSTOMER, Ent.CHANNEL_OFFER}) and myarg == 'filter':
      kwargs['filter'] = getString(Cmd.OB_STRING)
    elif (entityType == Ent.CHANNEL_SKU) and myarg == 'productid':
      productId = normalizeChannelProductID(getString(Cmd.OB_PRODUCT_ID))
    elif myarg == 'fields':
      if not fieldsList:
        fieldsList.append('name')
      for field in _getFieldsList():
        if field in channelEntityMap['fields']:
          fieldsList.append(channelEntityMap['fields'][field])
        else:
          invalidChoiceExit(field, list(channelEntityMap['fields']), True)
    elif myarg == 'maxresults':
      kwargs['pageSize'] = getInteger(minVal=1, maxVal=channelEntityMap['maxPageSize'])
    else:
      FJQC.GetFormatJSONQuoteChar(myarg, True)
  if entityType != Ent.CHANNEL_CUSTOMER_ENTITLEMENT:
    entityName = resellerId
    if entityType in {Ent.CHANNEL_CUSTOMER, Ent.CHANNEL_OFFER}:
      kwargs['parent'] = resellerId
    else:
      kwargs['account'] = resellerId
      if entityType == Ent.CHANNEL_SKU:
        kwargs['parent'] = productId
  else:
    if not name and customerId == 'customers/':
      missingArgumentExit('channelcustomerid')
    entityName = kwargs['parent'] = name if name else f'{resellerId}/{customerId}'
  fields = getItemFieldsFromFieldsList(channelEntityMap['items'], fieldsList)
#  if csvPF and FJQC.formatJSON and not fieldsList:
#    csvPF.SetJSONTitles(channelEntityMap['JSONtitles'])
  try:
    results = callGAPIpages(service, 'list', channelEntityMap['items'],
                            bailOnInternalError=True,
                            throwReasons=[GAPI.PERMISSION_DENIED, GAPI.INVALID_ARGUMENT, GAPI.BAD_REQUEST, GAPI.INTERNAL_ERROR, GAPI.NOT_FOUND],
                            fields=fields, **kwargs)
  except (GAPI.permissionDenied, GAPI.invalidArgument, GAPI.badRequest, GAPI.internalError, GAPI.notFound) as e:
    entityActionFailedWarning([entityType, entityName], str(e))
    return
  jcount = len(results)
  if not csvPF:
    if not FJQC.formatJSON:
      performActionNumItems(jcount, entityType)
    Ind.Increment()
    j = 0
    for item in results:
      j += 1
      if not FJQC.formatJSON:
        printEntity([entityType, item['name']], j, jcount)
        Ind.Increment()
        showJSON(None, item, timeObjects=channelEntityMap['timeObjects'])
        Ind.Decrement()
      else:
        printLine(json.dumps(cleanJSON(item, timeObjects=channelEntityMap['timeObjects']),
                             ensure_ascii=False, sort_keys=False))
    Ind.Decrement()
  else:
    for item in results:
      row = flattenJSON(item, timeObjects=channelEntityMap['timeObjects'])
      if not FJQC.formatJSON:
        csvPF.WriteRowTitles(row)
      elif csvPF.CheckRowTitles(row):
        row = {'name': item['name'],
               'JSON': json.dumps(cleanJSON(item, timeObjects=channelEntityMap['timeObjects']),
                                  ensure_ascii=False, sort_keys=True)}
#        if not fieldsList:
#          if entityType == Ent.CHANNEL_CUSTOMER:
#            row.update({'domain': item['domain']})
#          elif entityType == Ent.CHANNEL_CUSTOMER_ENTITLEMENT:
#            row.update({'offer': item['offer']})
#          elif entityType == Ent.CHANNEL_OFFER:
#            row.update({'sku': item['sku']})
        csvPF.WriteRowNoFilter(row)
    csvPF.writeCSVfile(Ent.Plural(entityType))

# gam print channelcustomers [todrive <ToDriveAttribute>*]
#	[resellerid <ResellerID>] [filter <String>]
#	[fields <ChannelCustomerFieldList>]
#	[maxresults <Integer>]
#	[formatjson [quotechar <Character>]]
# gam show channelcustomers
#	[resellerid <ResellerID>] [filter <String>]
#	[fields <ChannelCustomerFieldList>]
#	[maxresults <Integer>]
#	[formatjson]
def doPrintShowChannelCustomers():
  doPrintShowChannelItems(Ent.CHANNEL_CUSTOMER)

# gam print channelcustomercentitlements [todrive <ToDriveAttribute>*]
#	([resellerid <ResellerID>] [customerid <ChannelCustomerID>])|
#	(name accounts/<ResellerID>/customers/<ChannelCustomerID>)
#	[fields <ChannelCustomerEntitlementFieldList>]
#	[maxresults <Integer>]
#	[formatjson [quotechar <Character>]]
# gam show channelcustomerentitlements
#	([resellerid <ResellerID>] [customerid <ChannelCustomerID>])|
#	(name accounts/<ResellerID>/customers/<ChannelCustomerID>)
#	[fields <ChannelCustomerEntitlementFieldList>]
#	[maxresults <Integer>]
#	[formatjson]
def doPrintShowChannelCustomerEntitlements():
  doPrintShowChannelItems(Ent.CHANNEL_CUSTOMER_ENTITLEMENT)

# gam print channeloffers [todrive <ToDriveAttribute>*]
#	[resellerid <ResellerID>] [filter <String>] [language <LanguageCode]
#	[fields <ChannelOfferFieldList>]
#	[maxresults <Integer>]
#	[formatjson [quotechar <Character>]]
# gam show channeloffers
#	[resellerid <ResellerID>] [filter <String>] [language <LanguageCode]
#	[fields <ChannelOfferFieldList>]
#	[maxresults <Integer>]
#	[formatjson]
def doPrintShowChannelOffers():
  doPrintShowChannelItems(Ent.CHANNEL_OFFER)

# gam print channelproducts [todrive <ToDriveAttribute>*]
#	[resellerid <ResellerID>] [language <LanguageCode]
#	[fields <ChannelProductFieldList>]
#	[maxresults <Integer>]
#	[formatjson [quotechar <Character>]]
# gam show channelproducts
#	[resellerid <ResellerID>] [language <LanguageCode]
#	[fields <ChannelProductFieldList>]
#	[maxresults <Integer>]
#	[formatjson]
def doPrintShowChannelProducts():
  doPrintShowChannelItems(Ent.CHANNEL_PRODUCT)

# gam print channelskus [todrive <ToDriveAttribute>*]
#	[resellerid <ResellerID>] [language <LanguageCode] [productid <ProductID>]
#	[fields <ChannelSKUFieldList>]
#	[maxresults <Integer>]
#	[formatjson [quotechar <Character>]]
# gam show channelskus
#	[resellerid <ResellerID>] [language <LanguageCode] [productid <ProductID>]
#	[fields <ChannelSKUFieldList>]
#	[maxresults <Integer>]
#	[formatjson]
def doPrintShowChannelSKUs():
  doPrintShowChannelItems(Ent.CHANNEL_SKU)

ANALYTIC_ENTITY_MAP = {
  Ent.ANALYTIC_ACCOUNT:
    {'titles': ['User', 'name', 'displayName', 'createTime', 'updateTime', 'regionCode', 'deleted'],
     'JSONtitles': ['User', 'name', 'displayName', 'JSON'],
     'timeObjects': ['createTime', 'updateTime'],
     'items': 'accounts',
     'pageSize': 50,
     'maxPageSize': 200,
     },
  Ent.ANALYTIC_ACCOUNT_SUMMARY:
    {'titles': ['User', 'name', 'displayName', 'account'],
     'JSONtitles': ['User', 'name', 'displayName', 'account', 'JSON'],
     'timeObjects': ['createTime', 'updateTime', 'deleteTime', 'expireTime'],
     'items': 'accountSummaries',
     'pageSize': 50,
     'maxPageSize': 200,
     },
  Ent.ANALYTIC_DATASTREAM:
    {'titles': ['User', 'name', 'displayName', 'type', 'createTime', 'updateTime'],
     'JSONtitles': ['User', 'name', 'displayName', 'type', 'JSON'],
     'timeObjects': ['createTime', 'updateTime'],
     'items': 'dataStreams',
     'pageSize': 50,
     'maxPageSize': 200,
     },
  Ent.ANALYTIC_PROPERTY:
    {'titles': ['User', 'name', 'displayName', 'createTime', 'updateTime', 'propertyType', 'parent'],
     'JSONtitles': ['User', 'name', 'displayName', 'propertyType', 'parent', 'JSON'],
     'timeObjects': ['createTime', 'updateTime', 'deleteTime', 'expireTime'],
     'items': 'properties',
     'pageSize': 50,
     'maxPageSize': 200,
     },
  }

def printShowAnalyticItems(users, entityType):
  analyticEntityMap = ANALYTIC_ENTITY_MAP[entityType]
  csvPF = CSVPrintFile(analyticEntityMap['titles'], 'sortall') if Act.csvFormat() else None
  FJQC = FormatJSONQuoteChar(csvPF)
  kwargs = {'pageSize': analyticEntityMap['pageSize']}
  if entityType in {Ent.ANALYTIC_ACCOUNT, Ent.ANALYTIC_PROPERTY}:
    kwargs['showDeleted'] = False
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if csvPF and myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif myarg == 'maxresults':
      kwargs['pageSize'] = getInteger(minVal=1, maxVal=analyticEntityMap['maxPageSize'])
    elif entityType in {Ent.ANALYTIC_ACCOUNT, Ent.ANALYTIC_PROPERTY} and myarg == 'showdeleted':
      kwargs['showDeleted'] = getBoolean()
    elif entityType == Ent.ANALYTIC_PROPERTY and myarg == 'filter':
      kwargs['filter'] = getString(Cmd.OB_STRING)
    elif entityType == Ent.ANALYTIC_DATASTREAM and myarg == 'parent':
      kwargs['parent'] = getString(Cmd.OB_STRING)
    else:
      FJQC.GetFormatJSONQuoteChar(myarg, True)
  if entityType == Ent.ANALYTIC_PROPERTY and 'filter' not in kwargs:
    missingArgumentExit('filter')
  if entityType == Ent.ANALYTIC_DATASTREAM and 'parent' not in kwargs:
    missingArgumentExit('parent')
  if csvPF and FJQC.formatJSON:
    csvPF.SetJSONTitles(analyticEntityMap['JSONtitles'])
  i, count, users = getEntityArgument(users)
  for user in users:
    i += 1
    user, analytics = buildGAPIServiceObject(API.ANALYTICS_ADMIN, user, i, count)
    if not analytics:
      continue
    if entityType == Ent.ANALYTIC_ACCOUNT:
      service = analytics.accounts()
    elif entityType == Ent.ANALYTIC_ACCOUNT_SUMMARY:
      service = analytics.accountSummaries()
    elif entityType == Ent.ANALYTIC_DATASTREAM:
      service = analytics.properties().dataStreams()
    else: #  entityType == Ent.ANALYTIC_PROPERTY:
      service = analytics.properties()
    if csvPF:
      printGettingAllEntityItemsForWhom(entityType, user, i, count)
      pageMessage = getPageMessageForWhom()
    else:
      pageMessage = None
    try:
      results = callGAPIpages(service, 'list', analyticEntityMap['items'],
                              pageMessage=pageMessage,
                              throwReasons=[GAPI.PERMISSION_DENIED, GAPI.INVALID_ARGUMENT, GAPI.BAD_REQUEST, GAPI.INTERNAL_ERROR,
                                            GAPI.SERVICE_NOT_AVAILABLE],
                              **kwargs)
    except (GAPI.permissionDenied, GAPI.invalidArgument, GAPI.badRequest, GAPI.internalError) as e:
      entityActionFailedWarning([Ent.USER, user, entityType, None], str(e), i, count)
      continue
    except GAPI.serviceNotAvailable:
      userAnalyticsServiceNotEnabledWarning(user, i, count)
      continue
    jcount = len(results)
    if not csvPF:
      if not FJQC.formatJSON:
        entityPerformActionNumItems([Ent.USER, user], jcount, entityType)
      Ind.Increment()
      j = 0
      for item in results:
        j += 1
        if not FJQC.formatJSON:
          printEntity([entityType, item['name']], j, jcount)
          Ind.Increment()
          showJSON(None, item, timeObjects=analyticEntityMap['timeObjects'])
          Ind.Decrement()
        else:
          printLine(json.dumps(cleanJSON(item, timeObjects=analyticEntityMap['timeObjects']),
                               ensure_ascii=False, sort_keys=False))
      Ind.Decrement()
    else:
      for item in results:
        row = flattenJSON(item, flattened={'User': user}, timeObjects=analyticEntityMap['timeObjects'])
        if not FJQC.formatJSON:
          csvPF.WriteRowTitles(row)
        elif csvPF.CheckRowTitles(row):
          row = {'User': user, 'name': item['name'], 'displayName': item['displayName']}
          for field in analyticEntityMap['JSONtitles'][2:-1]:
            row[field] = item[field]
          row['JSON'] = json.dumps(cleanJSON(item, timeObjects=analyticEntityMap['timeObjects']),
                                   ensure_ascii=False, sort_keys=True)
          csvPF.WriteRowNoFilter(row)
  if csvPF:
    csvPF.writeCSVfile(Ent.Plural(entityType))

# gam <UserTypeEntity> print analyticaccounts [todrive <ToDriveAttribute>*]
#	[maxresults <Integer>] [showdeleted [<Boolean>]]
#	[formatjson [quotechar <Character>]]
# gam <UserTypeEntity> show analyticaccounts
#	[maxresults <Integer>] [showdeleted [<Boolean>]]
#	[formatjson]
def printShowAnalyticAccounts(users):
  printShowAnalyticItems(users, Ent.ANALYTIC_ACCOUNT)

# gam <UserTypeEntity> print analyticaccountsummaries [todrive <ToDriveAttribute>*]
#	[maxresults <Integer>]
#	[formatjson [quotechar <Character>]]
# gam <UserTypeEntity> show analyticaccountsummaries
#	[maxresults <Integer>]
#	[formatjson]
def printShowAnalyticAccountSummaries(users):
  printShowAnalyticItems(users, Ent.ANALYTIC_ACCOUNT_SUMMARY)

# gam <UserTypeEntity> print analyticproperties [todrive <ToDriveAttribute>*]
#	filter <String>
#	[maxresults <Integer>] [showdeleted [<Boolean>]]
#	[formatjson [quotechar <Character>]]
# gam <UserTypeEntity> show analyticproperties
#	filter <String>
#	[maxresults <Integer>] [showdeleted [<Boolean>]]
#	[formatjson]
def printShowAnalyticProperties(users):
  printShowAnalyticItems(users, Ent.ANALYTIC_PROPERTY)

# gam <UserTypeEntity> print analyticdatastreams [todrive <ToDriveAttribute>*]
#	parent <String>
#	[maxresults <Integer>]
#	[formatjson [quotechar <Character>]]
# gam <UserTypeEntity> show analyticdatastreams
#	parent <String>
#	[maxresults <Integer>]
#	[formatjson]
def printShowAnalyticDatastreams(users):
  printShowAnalyticItems(users, Ent.ANALYTIC_DATASTREAM)

# gam create domainalias|aliasdomain <DomainAlias> <DomainName>
def doCreateDomainAlias():
  cd = buildGAPIObject(API.DIRECTORY)
  body = {'domainAliasName': getString(Cmd.OB_DOMAIN_ALIAS)}
  body['parentDomainName'] = getString(Cmd.OB_DOMAIN_NAME)
  checkForExtraneousArguments()
  try:
    callGAPI(cd.domainAliases(), 'insert',
             throwReasons=[GAPI.DOMAIN_NOT_FOUND, GAPI.DUPLICATE, GAPI.INVALID, GAPI.CONFLICT,
                           GAPI.BAD_REQUEST, GAPI.NOT_FOUND,
                           GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
             customer=GC.Values[GC.CUSTOMER_ID], body=body, fields='')
    entityActionPerformed([Ent.DOMAIN, body['parentDomainName'], Ent.DOMAIN_ALIAS, body['domainAliasName']])
  except GAPI.domainNotFound:
    entityActionFailedWarning([Ent.DOMAIN, body['parentDomainName']], Msg.DOES_NOT_EXIST)
  except GAPI.duplicate:
    entityActionFailedWarning([Ent.DOMAIN, body['parentDomainName'], Ent.DOMAIN_ALIAS, body['domainAliasName']], Msg.DUPLICATE)
  except (GAPI.invalid, GAPI.conflict) as e:
    entityActionFailedWarning([Ent.DOMAIN, body['parentDomainName'], Ent.DOMAIN_ALIAS, body['domainAliasName']], str(e))
  except (GAPI.badRequest, GAPI.notFound) as e:
    accessErrorExit(cd, str(e))
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))

# gam delete domainalias|aliasdomain <DomainAlias>
def doDeleteDomainAlias():
  cd = buildGAPIObject(API.DIRECTORY)
  domainAliasName = getString(Cmd.OB_DOMAIN_ALIAS)
  checkForExtraneousArguments()
  try:
    callGAPI(cd.domainAliases(), 'delete',
             throwReasons=[GAPI.DOMAIN_ALIAS_NOT_FOUND, GAPI.BAD_REQUEST, GAPI.NOT_FOUND,
                           GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
             customer=GC.Values[GC.CUSTOMER_ID], domainAliasName=domainAliasName)
    entityActionPerformed([Ent.DOMAIN_ALIAS, domainAliasName])
  except GAPI.domainAliasNotFound:
    entityActionFailedWarning([Ent.DOMAIN_ALIAS, domainAliasName], Msg.DOES_NOT_EXIST)
  except (GAPI.badRequest, GAPI.notFound) as e:
    accessErrorExit(cd, str(e))
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))

DOMAIN_TIME_OBJECTS = {'creationTime'}
DOMAIN_ALIAS_PRINT_ORDER = ['parentDomainName', 'creationTime', 'verified']
DOMAIN_ALIAS_SKIP_OBJECTS = {'domainAliasName'}

def _showDomainAlias(alias, FJQC, aliasSkipObjects, i=0, count=0):
  if FJQC.formatJSON:
    printLine(json.dumps(cleanJSON(alias, timeObjects=DOMAIN_TIME_OBJECTS), ensure_ascii=False, sort_keys=True))
    return
  printEntity([Ent.DOMAIN_ALIAS, alias['domainAliasName']], i, count)
  Ind.Increment()
  if 'creationTime' in alias:
    alias['creationTime'] = formatLocalTimestamp(alias['creationTime'])
  for field in DOMAIN_ALIAS_PRINT_ORDER:
    if field in alias:
      printKeyValueList([field, alias[field]])
      aliasSkipObjects.add(field)
  showJSON(None, alias, aliasSkipObjects)
  Ind.Decrement()

# gam info domainalias|aliasdomain <DomainAlias> [formatjson]
def doInfoDomainAlias():
  cd = buildGAPIObject(API.DIRECTORY)
  domainAliasName = getString(Cmd.OB_DOMAIN_ALIAS)
  FJQC = FormatJSONQuoteChar(formatJSONOnly=True)
  try:
    result = callGAPI(cd.domainAliases(), 'get',
                      throwReasons=[GAPI.DOMAIN_ALIAS_NOT_FOUND, GAPI.BAD_REQUEST, GAPI.NOT_FOUND,
                                    GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                      customer=GC.Values[GC.CUSTOMER_ID], domainAliasName=domainAliasName)
    aliasSkipObjects = DOMAIN_ALIAS_SKIP_OBJECTS
    _showDomainAlias(result, FJQC, aliasSkipObjects)
  except GAPI.domainAliasNotFound:
    entityActionFailedWarning([Ent.DOMAIN_ALIAS, domainAliasName], Msg.DOES_NOT_EXIST)
  except (GAPI.badRequest, GAPI.notFound) as e:
    accessErrorExit(cd, str(e))
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))

def _printDomain(domain, csvPF):
  row = {}
  for attr in domain:
    if attr not in DEFAULT_SKIP_OBJECTS:
      if attr in DOMAIN_TIME_OBJECTS:
        row[attr] = formatLocalTimestamp(domain[attr])
      else:
        row[attr] = domain[attr]
      csvPF.AddTitles(attr)
  csvPF.WriteRow(row)

DOMAIN_ALIAS_SORT_TITLES = ['domainAliasName', 'parentDomainName', 'creationTime', 'verified']

# gam print domainaliases [todrive <ToDriveAttribute>*]
#	[formatjson [quotechar <Character>]]
#	[showitemcountonly]
# gam show domainaliases
#	[formatjson]
#	[showitemcountonly]
def doPrintShowDomainAliases():
  cd = buildGAPIObject(API.DIRECTORY)
  csvPF = CSVPrintFile(['domainAliasName'], DOMAIN_ALIAS_SORT_TITLES) if Act.csvFormat() else None
  FJQC = FormatJSONQuoteChar(csvPF)
  showItemCountOnly = False
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if csvPF and myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif myarg == 'showitemcountonly':
      showItemCountOnly = True
    else:
      FJQC.GetFormatJSONQuoteChar(myarg, True)
  try:
    domainAliases = callGAPIitems(cd.domainAliases(), 'list', 'domainAliases',
                                  throwReasons=[GAPI.BAD_REQUEST, GAPI.NOT_FOUND,
                                                GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                                  customer=GC.Values[GC.CUSTOMER_ID])
    count = len(domainAliases)
    if showItemCountOnly:
      writeStdout(f'{count}\n')
      return
    i = 0
    for domainAlias in domainAliases:
      i += 1
      if not csvPF:
        aliasSkipObjects = DOMAIN_ALIAS_SKIP_OBJECTS
        _showDomainAlias(domainAlias, FJQC, aliasSkipObjects, i, count)
      elif not FJQC.formatJSON:
        _printDomain(domainAlias, csvPF)
      else:
        csvPF.WriteRowNoFilter({'domainAliasName': domainAlias['domainAliasName'],
                                'JSON': json.dumps(cleanJSON(domainAlias, timeObjects=DOMAIN_TIME_OBJECTS),
                                                   ensure_ascii=False, sort_keys=True)})
  except (GAPI.badRequest, GAPI.notFound) as e:
    accessErrorExit(cd, str(e))
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))
  if csvPF:
    csvPF.writeCSVfile('Domain Aliases')

# gam create domain <DomainName>
def doCreateDomain():
  cd = buildGAPIObject(API.DIRECTORY)
  body = {'domainName': getString(Cmd.OB_DOMAIN_NAME)}
  checkForExtraneousArguments()
  try:
    callGAPI(cd.domains(), 'insert',
             throwReasons=[GAPI.DUPLICATE, GAPI.CONFLICT,
                           GAPI.DOMAIN_NOT_FOUND, GAPI.BAD_REQUEST, GAPI.NOT_FOUND,
                           GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
             customer=GC.Values[GC.CUSTOMER_ID], body=body, fields='')
    entityActionPerformed([Ent.DOMAIN, body['domainName']])
  except GAPI.duplicate:
    entityDuplicateWarning([Ent.DOMAIN, body['domainName']])
  except GAPI.conflict as e:
    entityActionFailedWarning([Ent.DOMAIN, body['domainName']], str(e))
  except (GAPI.domainNotFound, GAPI.badRequest, GAPI.notFound) as e:
    accessErrorExit(cd, str(e))
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))

# gam update domain <DomainName> primary
def doUpdateDomain():
  cd = buildGAPIObject(API.DIRECTORY)
  domainName = getString(Cmd.OB_DOMAIN_NAME)
  body = {}
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'primary':
      body['customerDomain'] = domainName
    else:
      unknownArgumentExit()
  if not body:
    missingArgumentExit('primary')
  try:
    callGAPI(cd.customers(), 'update',
             throwReasons=[GAPI.DOMAIN_NOT_VERIFIED_SECONDARY, GAPI.BAD_REQUEST,
                           GAPI.RESOURCE_NOT_FOUND, GAPI.INVALID_INPUT,
                           GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
             customerKey=GC.Values[GC.CUSTOMER_ID], body=body, fields='')
    entityActionPerformedMessage([Ent.DOMAIN, domainName], Msg.NOW_THE_PRIMARY_DOMAIN)
  except GAPI.domainNotVerifiedSecondary:
    entityActionFailedWarning([Ent.DOMAIN, domainName], Msg.DOMAIN_NOT_VERIFIED_SECONDARY)
  except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.invalidInput) as e:
    accessErrorExit(cd, str(e))
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))

# gam delete domain <DomainName>
def doDeleteDomain():
  cd = buildGAPIObject(API.DIRECTORY)
  domainName = getString(Cmd.OB_DOMAIN_NAME)
  checkForExtraneousArguments()
  try:
    callGAPI(cd.domains(), 'delete',
             throwReasons=[GAPI.BAD_REQUEST, GAPI.NOT_FOUND,
                           GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
             customer=GC.Values[GC.CUSTOMER_ID], domainName=domainName)
    entityActionPerformed([Ent.DOMAIN, domainName])
  except (GAPI.badRequest, GAPI.notFound) as e:
    accessErrorExit(cd, str(e))
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))

CUSTOMER_LICENSE_MAP = {
  'accounts:num_users': 'Total Users',
  'accounts:gsuite_basic_total_licenses': 'G Suite Basic Licenses',
  'accounts:gsuite_basic_used_licenses': 'G Suite Basic Users',
  'accounts:gsuite_enterprise_total_licenses': 'Workspace Enterprise Plus Licenses',
  'accounts:gsuite_enterprise_used_licenses': 'Workspace Enterprise Plus Users',
  'accounts:gsuite_unlimited_total_licenses': 'G Suite Business Licenses',
  'accounts:gsuite_unlimited_used_licenses': 'G Suite Business Users',
  'accounts:vault_total_licenses': 'Google Vault Licenses',
  }

def _showCustomerLicenseInfo(customerInfo, FJQC):
  def numUsersAvailable(result):
    usageReports = result.get('usageReports', [])
    if usageReports:
      for item in usageReports[0].get('parameters', []):
        if item['name'] == 'accounts:num_users':
          return usageReports
    return None

  rep = buildGAPIObject(API.REPORTS)
  parameters = ','.join(CUSTOMER_LICENSE_MAP)
  tryDate = todaysDate().strftime(YYYYMMDD_FORMAT)
  dataRequiredServices = {'accounts'}
  while True:
    try:
      result = callGAPI(rep.customerUsageReports(), 'get',
                        throwReasons=[GAPI.INVALID, GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                        date=tryDate, customerId=customerInfo['id'],
                        fields='warnings,usageReports', parameters=parameters)
      usageReports = numUsersAvailable(result)
      if usageReports:
        break
      fullData, tryDate, usageReports = _checkDataRequiredServices(result, tryDate, dataRequiredServices)
      if fullData < 0:
        printWarningMessage(DATA_NOT_AVALIABLE_RC, Msg.NO_USER_COUNTS_DATA_AVAILABLE)
        return
      if fullData == 0:
        continue
      break
    except GAPI.invalid as e:
      tryDate = _adjustTryDate(str(e), 0, -1, tryDate)
      if not tryDate:
        return
      continue
    except (GAPI.forbidden, GAPI.permissionDenied) as e:
      ClientAPIAccessDeniedExit(str(e))
  if not FJQC.formatJSON:
    printKeyValueList([f'User counts as of {tryDate}:'])
    Ind.Increment()
  for item in usageReports[0]['parameters']:
    api_name = CUSTOMER_LICENSE_MAP.get(item['name'])
    api_value = int(item.get('intValue', '0'))
    if api_name and api_value:
      if not FJQC.formatJSON:
        printKeyValueList([api_name, f'{api_value:,}'])
      else:
        customerInfo[item['name']] = api_value
  if not FJQC.formatJSON:
    Ind.Decrement()

def setTrueCustomerId(cd=None):
  if GC.Values[GC.CUSTOMER_ID] == GC.MY_CUSTOMER:
    if not cd:
      cd = buildGAPIObject(API.DIRECTORY)
    try:
      customerInfo = callGAPI(cd.customers(), 'get',
                              throwReasons=[GAPI.BAD_REQUEST, GAPI.INVALID_INPUT, GAPI.RESOURCE_NOT_FOUND,
                                            GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                              customerKey=GC.MY_CUSTOMER,
                              fields='id')
      GC.Values[GC.CUSTOMER_ID] = customerInfo['id']
    except (GAPI.badRequest, GAPI.invalidInput, GAPI.resourceNotFound):
      pass
    except (GAPI.forbidden, GAPI.permissionDenied) as e:
      ClientAPIAccessDeniedExit(str(e))

def _getCustomerId():
  customerId = GC.Values[GC.CUSTOMER_ID]
  if customerId != GC.MY_CUSTOMER and customerId[0] != 'C':
    customerId = 'C' + customerId
  return customerId

def _getCustomerIdNoC():
  customerId = GC.Values[GC.CUSTOMER_ID]
  if customerId[0] == 'C':
    return customerId[1:]
  return customerId

def _getCustomersCustomerIdNoC():
  customerId = GC.Values[GC.CUSTOMER_ID]
  if customerId.startswith('C'):
    customerId = customerId[1:]
  return f'customers/{customerId}'

def _getCustomersCustomerIdWithC():
  customerId = GC.Values[GC.CUSTOMER_ID]
  if customerId != GC.MY_CUSTOMER and customerId[0] != 'C':
    customerId = 'C' + customerId
  return f'customers/{customerId}'

# gam info customer [formatjson]
def doInfoCustomer(returnCustomerInfo=None, FJQC=None):
  cd = buildGAPIObject(API.DIRECTORY)
  customerId = _getCustomerId()
  if FJQC is None:
    FJQC = FormatJSONQuoteChar(formatJSONOnly=True)
  try:
    customerInfo = callGAPI(cd.customers(), 'get',
                            throwReasons=[GAPI.BAD_REQUEST, GAPI.INVALID_INPUT, GAPI.RESOURCE_NOT_FOUND,
                                          GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                            customerKey=customerId)
    if 'customerCreationTime' in customerInfo:
      customerInfo['customerCreationTime'] = formatLocalTime(customerInfo['customerCreationTime'])
    else:
      customerInfo['customerCreationTime'] =  UNKNOWN
    primaryDomain = {'domainName': UNKNOWN, 'verified': UNKNOWN}
    try:
      domains = callGAPIitems(cd.domains(), 'list', 'domains',
                              throwReasons=[GAPI.BAD_REQUEST, GAPI.NOT_FOUND,
                                            GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                              customer=customerInfo['id'], fields='domains(creationTime,domainName,isPrimary,verified)')
      for domain in domains:
        if domain.get('isPrimary'):
          primaryDomain = domain
          break
      # From Jay Lee
      # If customer has changed primary domain, customerCreationTime is date of current primary being added, not customer create date.
      # We should get all domains and use oldest date
      customerCreationTime = UNKNOWN
      for domain in domains:
        domainCreationTime = formatLocalTimestampUTC(domain['creationTime'])
        if customerCreationTime == UNKNOWN or domainCreationTime < customerCreationTime:
          customerCreationTime = domainCreationTime
      customerInfo['customerCreationTime'] = formatLocalTime(customerCreationTime)
    except (GAPI.badRequest, GAPI.notFound):
      pass
    customerInfo['customerDomain'] = primaryDomain['domainName']
    customerInfo['verified'] = primaryDomain['verified']
    if FJQC.formatJSON:
      _showCustomerLicenseInfo(customerInfo, FJQC)
      if returnCustomerInfo is not None:
        returnCustomerInfo.update(customerInfo)
        return
      printLine(json.dumps(cleanJSON(customerInfo), ensure_ascii=False, sort_keys=True))
      return
    printKeyValueList(['Customer ID', customerInfo['id']])
    printKeyValueList(['Primary Domain', customerInfo['customerDomain']])
    printKeyValueList(['Primary Domain Verified', customerInfo['verified']])
    printKeyValueList(['Customer Creation Time', customerInfo['customerCreationTime']])
    printKeyValueList(['Default Language', customerInfo.get('language', 'Unset or Unknown (defaults to en)')])
    _showCustomerAddressPhoneNumber(customerInfo)
    printKeyValueList(['Admin Secondary Email', customerInfo.get('alternateEmail', UNKNOWN)])
    _showCustomerLicenseInfo(customerInfo, FJQC)
  except (GAPI.badRequest, GAPI.invalidInput, GAPI.domainNotFound, GAPI.notFound, GAPI.resourceNotFound):
    accessErrorExit(cd)
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))

# gam update customer [primary <DomainName>] [adminsecondaryemail|alternateemail <EmailAddress>] [language <LanguageCode] [phone|phonenumber <String>]
#	[contact|contactname <String>] [name|organizationname <String>]
#	[address1|addressline1 <String>] [address2|addressline2 <String>] [address3|addressline3 <String>]
#	[locality <String>] [region <String>] [postalcode <String>] [country|countrycode <String>]
def doUpdateCustomer():
  cd = buildGAPIObject(API.DIRECTORY)
  customerId = _getCustomerId()
  body = {}
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg in ADDRESS_FIELDS_ARGUMENT_MAP:
      body.setdefault('postalAddress', {})
      body['postalAddress'][ADDRESS_FIELDS_ARGUMENT_MAP[myarg]] = getString(Cmd.OB_STRING, minLen=0)
    elif myarg == 'primary':
      body['customerDomain'] = getString(Cmd.OB_DOMAIN_NAME)
    elif myarg in {'adminsecondaryemail', 'alternateemail'}:
      body['alternateEmail'] = getEmailAddress(noUid=True)
    elif myarg in {'phone', 'phonenumber'}:
      body['phoneNumber'] = getString(Cmd.OB_STRING, minLen=0)
    elif myarg == 'language':
      body['language'] = getLanguageCode(LANGUAGE_CODES_MAP)
    else:
      unknownArgumentExit()
  if body:
    try:
      callGAPI(cd.customers(), 'patch',
               throwReasons=[GAPI.DOMAIN_NOT_VERIFIED_SECONDARY, GAPI.INVALID, GAPI.INVALID_INPUT, GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND,
                             GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
               customerKey=customerId, body=body, fields='')
      entityActionPerformed([Ent.CUSTOMER_ID, GC.Values[GC.CUSTOMER_ID]])
    except GAPI.domainNotVerifiedSecondary:
      entityActionFailedWarning([Ent.CUSTOMER_ID, GC.Values[GC.CUSTOMER_ID], Ent.DOMAIN, body['customerDomain']], Msg.DOMAIN_NOT_VERIFIED_SECONDARY)
    except (GAPI.invalid, GAPI.invalidInput) as e:
      entityActionFailedWarning([Ent.CUSTOMER_ID, GC.Values[GC.CUSTOMER_ID]], str(e))
    except (GAPI.badRequest, GAPI.resourceNotFound):
      accessErrorExit(cd)
    except (GAPI.forbidden, GAPI.permissionDenied) as e:
      ClientAPIAccessDeniedExit(str(e))

# gam info instance [formatjson]
def doInfoInstance():
  FJQC = FormatJSONQuoteChar(formatJSONOnly=True)
  customerInfo = None if not FJQC.formatJSON else {}
  doInfoCustomer(customerInfo, FJQC)
  if FJQC.formatJSON:
    printLine(json.dumps(cleanJSON(customerInfo), ensure_ascii=False, sort_keys=True))

DOMAIN_PRINT_ORDER = ['customerDomain', 'creationTime', 'isPrimary', 'verified']
DOMAIN_SKIP_OBJECTS = {'domainName', 'domainAliases'}

def _showDomain(result, FJQC, i=0, count=0):
  if FJQC.formatJSON:
    printLine(json.dumps(cleanJSON(result, timeObjects=DOMAIN_TIME_OBJECTS), ensure_ascii=False, sort_keys=True))
    return
  skipObjects = DOMAIN_SKIP_OBJECTS
  printEntity([Ent.DOMAIN, result['domainName']], i, count)
  Ind.Increment()
  if 'creationTime' in result:
    result['creationTime'] = formatLocalTimestamp(result['creationTime'])
  for field in DOMAIN_PRINT_ORDER:
    if field in result:
      printKeyValueList([field, result[field]])
      skipObjects.add(field)
  field = 'domainAliases'
  aliases = result.get(field)
  if aliases:
    skipObjects.add(field)
    aliasSkipObjects = DOMAIN_ALIAS_SKIP_OBJECTS
    for alias in aliases:
      _showDomainAlias(alias, FJQC, aliasSkipObjects)
      showJSON(None, alias, aliasSkipObjects)
  showJSON(None, result, skipObjects)
  Ind.Decrement()

# gam info domain [<DomainName>] [formatjson]
def doInfoDomain():
  if (not Cmd.ArgumentsRemaining()) or (Cmd.Current().lower() == 'formatjson'):
    doInfoInstance()
    return
  cd = buildGAPIObject(API.DIRECTORY)
  domainName = getString(Cmd.OB_DOMAIN_NAME)
  FJQC = FormatJSONQuoteChar(formatJSONOnly=True)
  try:
    result = callGAPI(cd.domains(), 'get',
                      throwReasons=[GAPI.DOMAIN_NOT_FOUND, GAPI.BAD_REQUEST, GAPI.NOT_FOUND,
                                    GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                      customer=GC.Values[GC.CUSTOMER_ID], domainName=domainName)
    _showDomain(result, FJQC)
  except GAPI.domainNotFound:
    entityActionFailedWarning([Ent.DOMAIN, domainName], Msg.DOES_NOT_EXIST)
  except (GAPI.badRequest, GAPI.notFound):
    accessErrorExit(cd)
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))

DOMAIN_SORT_TITLES = ['domainName', 'parentDomainName', 'creationTime', 'type', 'verified']

# gam print domains [todrive <ToDriveAttribute>*]
#	[formatjson [quotechar <Character>]]
#	[showitemcountonly]
# gam show domains
#	[formatjson]
#	[showitemcountonly]
def doPrintShowDomains():
  cd = buildGAPIObject(API.DIRECTORY)
  csvPF = CSVPrintFile(['domainName'], DOMAIN_SORT_TITLES) if Act.csvFormat() else None
  FJQC = FormatJSONQuoteChar(csvPF)
  showItemCountOnly = False
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if csvPF and myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif myarg == 'showitemcountonly':
      showItemCountOnly = True
    else:
      FJQC.GetFormatJSONQuoteChar(myarg, True)
  try:
    domains = callGAPIitems(cd.domains(), 'list', 'domains',
                            throwReasons=[GAPI.DOMAIN_NOT_FOUND, GAPI.BAD_REQUEST, GAPI.NOT_FOUND, GAPI.FORBIDDEN],
                            customer=GC.Values[GC.CUSTOMER_ID])
    count = len(domains)
    if showItemCountOnly:
      writeStdout(f'{count}\n')
      return
    i = 0
    for domain in domains:
      i += 1
      if not csvPF:
        _showDomain(domain, FJQC, i, count)
      elif not FJQC.formatJSON:
        domain['type'] = 'primary' if domain.pop('isPrimary') else 'secondary'
        domainAliases = domain.pop('domainAliases', [])
        _printDomain(domain, csvPF)
        for domainAlias in domainAliases:
          domainAlias['type'] = 'alias'
          domainAlias['domainName'] = domainAlias.pop('domainAliasName')
          _printDomain(domainAlias, csvPF)
      else:
        csvPF.WriteRowNoFilter({'domainName': domain['domainName'],
                                'JSON': json.dumps(cleanJSON(domain, timeObjects=DOMAIN_TIME_OBJECTS),
                                                   ensure_ascii=False, sort_keys=True)})
  except (GAPI.badRequest, GAPI.notFound, GAPI.domainNotFound) as e:
    accessErrorExit(cd, str(e))
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))
  if csvPF:
    csvPF.writeCSVfile('Domains')

PRINT_PRIVILEGES_FIELDS = ['serviceId', 'serviceName', 'privilegeName', 'isOuScopable', 'childPrivileges']

def _listPrivileges(cd):
  fields = f'items({",".join(PRINT_PRIVILEGES_FIELDS)})'
  try:
    return callGAPIitems(cd.privileges(), 'list', 'items',
                         throwReasons=[GAPI.BAD_REQUEST, GAPI.CUSTOMER_NOT_FOUND,
                                       GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                         customer=GC.Values[GC.CUSTOMER_ID], fields=fields)
  except (GAPI.badRequest, GAPI.customerNotFound):
    accessErrorExit(cd)
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))

# gam print privileges [todrive <ToDriveAttribute>*]
# gam show privileges
def doPrintShowPrivileges():
  def _showPrivilege(privilege, i, count):
    printEntity([Ent.PRIVILEGE, privilege['privilegeName']], i, count)
    Ind.Increment()
    printKeyValueList(['serviceId', privilege['serviceId']])
    printKeyValueList(['serviceName', privilege.get('serviceName', UNKNOWN)])
    printKeyValueList(['isOuScopable', privilege['isOuScopable']])
    jcount = len(privilege.get('childPrivileges', []))
    if jcount > 0:
      printKeyValueList(['childPrivileges', jcount])
      Ind.Increment()
      j = 0
      for childPrivilege in privilege['childPrivileges']:
        j += 1
        _showPrivilege(childPrivilege, j, jcount)
      Ind.Decrement()
    Ind.Decrement()

  cd = buildGAPIObject(API.DIRECTORY)
  csvPF = CSVPrintFile(PRINT_PRIVILEGES_FIELDS, 'sortall') if Act.csvFormat() else None
  getTodriveOnly(csvPF)
  privileges = _listPrivileges(cd)
  if not csvPF:
    count = len(privileges)
    performActionNumItems(count, Ent.PRIVILEGE)
    Ind.Increment()
    i = 0
    for privilege in privileges:
      i += 1
      _showPrivilege(privilege, i, count)
    Ind.Decrement()
  else:
    for privilege in privileges:
      csvPF.WriteRowTitles(flattenJSON(privilege))
  if csvPF:
    csvPF.writeCSVfile('Privileges')

def makeRoleIdNameMap():
  GM.Globals[GM.MAKE_ROLE_ID_NAME_MAP] = False
  cd = buildGAPIObject(API.DIRECTORY)
  try:
    result = callGAPIpages(cd.roles(), 'list', 'items',
                           throwReasons=[GAPI.BAD_REQUEST, GAPI.CUSTOMER_NOT_FOUND,
                                         GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                           customer=GC.Values[GC.CUSTOMER_ID],
                           fields='nextPageToken,items(roleId,roleName)',
                           maxResults=100)
  except (GAPI.badRequest, GAPI.customerNotFound):
    accessErrorExit(cd)
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))
  for role in result:
    GM.Globals[GM.MAP_ROLE_ID_TO_NAME][role['roleId']] = role['roleName']
    GM.Globals[GM.MAP_ROLE_NAME_TO_ID][role['roleName'].lower()] = role['roleId']

def role_from_roleid(roleid):
  if GM.Globals[GM.MAKE_ROLE_ID_NAME_MAP]:
    makeRoleIdNameMap()
  return GM.Globals[GM.MAP_ROLE_ID_TO_NAME].get(roleid, roleid)

def roleid_from_role(role):
  if GM.Globals[GM.MAKE_ROLE_ID_NAME_MAP]:
    makeRoleIdNameMap()
  return GM.Globals[GM.MAP_ROLE_NAME_TO_ID].get(role.lower(), None)

def getRoleId():
  role = getString(Cmd.OB_ROLE_ITEM)
  cg = UID_PATTERN.match(role)
  if cg:
    roleId = cg.group(1)
  else:
    roleId = roleid_from_role(role)
    if not roleId:
      invalidChoiceExit(role, GM.Globals[GM.MAP_ROLE_NAME_TO_ID], True)
  return (role, roleId)

# gam create adminrole <String> [description <String>]
#	privileges all|all_ou|<PrivilegesList>|(select <FileSelector>|<CSVFileSelector>)
# gam update adminrole <RoleItem> [name <String>] [description <String>]
#	[privileges all|all_ou|<PrivilegesList>|(select <FileSelector>|<CSVFileSelector>)]
def doCreateUpdateAdminRoles():
  def expandChildPrivileges(privilege):
    for childPrivilege in privilege.get('childPrivileges', []):
      childPrivileges[childPrivilege['privilegeName']] = childPrivilege['serviceId']
      expandChildPrivileges(childPrivilege)

  cd = buildGAPIObject(API.DIRECTORY)
  updateCmd = Act.Get() == Act.UPDATE
  if not updateCmd:
    body = {'roleName': getString(Cmd.OB_STRING)}
  else:
    body = {}
    _, roleId = getRoleId()
  allPrivileges = {}
  ouPrivileges = {}
  childPrivileges = {}
  for privilege in _listPrivileges(cd):
    allPrivileges[privilege['privilegeName']] = privilege['serviceId']
    if privilege['isOuScopable']:
      ouPrivileges[privilege['privilegeName']] = privilege['serviceId']
    expandChildPrivileges(privilege)
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'privileges':
      privs = getString(Cmd.OB_PRIVILEGE_LIST).upper()
      if privs == 'ALL':
        body['rolePrivileges'] = [{'privilegeName': p, 'serviceId': v} for p, v in allPrivileges.items()]
      elif privs == 'ALL_OU':
        body['rolePrivileges'] = [{'privilegeName': p, 'serviceId': v} for p, v in ouPrivileges.items()]
      else:
        if privs == 'SELECT':
          privsList = [p.upper() for p in getEntityList(Cmd.OB_PRIVILEGE_LIST)]
        else:
          privsList = privs.replace(',', ' ').split()
        body.setdefault('rolePrivileges', [])
        for p in privsList:
          if p in allPrivileges:
            body['rolePrivileges'].append({'privilegeName': p, 'serviceId': allPrivileges[p]})
          elif p in ouPrivileges:
            body['rolePrivileges'].append({'privilegeName': p, 'serviceId': ouPrivileges[p]})
          elif p in childPrivileges:
            body['rolePrivileges'].append({'privilegeName': p, 'serviceId': childPrivileges[p]})
          elif ':' in p:
            priv, serv = p.split(':')
            body['rolePrivileges'].append({'privilegeName': priv, 'serviceId': serv.lower()})
          elif p == 'SUPPORT':
            pass
          else:
            invalidChoiceExit(p, list(allPrivileges.keys())+list(ouPrivileges.keys())+list(childPrivileges.keys()), True)
    elif myarg == 'description':
      body['roleDescription'] = getString(Cmd.OB_STRING)
    elif myarg == 'name':
      body['roleName'] = getString(Cmd.OB_STRING)
    else:
      unknownArgumentExit()
  if not updateCmd and not body.get('rolePrivileges'):
    missingArgumentExit('privileges')
  try:
    if not updateCmd:
      result = callGAPI(cd.roles(), 'insert',
                        throwReasons=[GAPI.BAD_REQUEST, GAPI.CUSTOMER_NOT_FOUND,
                                      GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED]+[GAPI.DUPLICATE],
                        customer=GC.Values[GC.CUSTOMER_ID], body=body, fields='roleId,roleName')
    else:
      result = callGAPI(cd.roles(), 'patch',
                        throwReasons=[GAPI.BAD_REQUEST, GAPI.CUSTOMER_NOT_FOUND,
                                      GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED]+[GAPI.NOT_FOUND, GAPI.FAILED_PRECONDITION, GAPI.CONFLICT],
                        customer=GC.Values[GC.CUSTOMER_ID], roleId=roleId, body=body, fields='roleId,roleName')
    entityActionPerformed([Ent.ADMIN_ROLE, f"{result['roleName']}({result['roleId']})"])
  except GAPI.duplicate as e:
    entityActionFailedWarning([Ent.ADMIN_ROLE, f"{body['roleName']}"], str(e))
  except (GAPI.notFound, GAPI.failedPrecondition, GAPI.conflict) as e:
    entityActionFailedWarning([Ent.ADMIN_ROLE, roleId], str(e))
  except (GAPI.badRequest, GAPI.customerNotFound):
    accessErrorExit(cd)
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))

# gam delete adminrole <RoleItem>
def doDeleteAdminRole():
  cd = buildGAPIObject(API.DIRECTORY)
  role, roleId = getRoleId()
  checkForExtraneousArguments()
  try:
    callGAPI(cd.roles(), 'delete',
             throwReasons=[GAPI.BAD_REQUEST, GAPI.CUSTOMER_NOT_FOUND,
                           GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED]+[GAPI.NOT_FOUND, GAPI.FAILED_PRECONDITION],
             customer=GC.Values[GC.CUSTOMER_ID], roleId=roleId)
    entityActionPerformed([Ent.ADMIN_ROLE, f"{role}({roleId})"])
  except (GAPI.notFound, GAPI.failedPrecondition) as e:
    entityActionFailedWarning([Ent.ADMIN_ROLE, roleId], str(e))
  except (GAPI.badRequest, GAPI.customerNotFound):
    accessErrorExit(cd)
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))

PRINT_ADMIN_ROLES_FIELDS = ['roleId', 'roleName', 'roleDescription', 'isSuperAdminRole', 'isSystemRole']

def _showAdminRole(role, i=0, count=0):
  printEntity([Ent.ADMIN_ROLE, role['roleName']], i, count)
  Ind.Increment()
  for field in PRINT_ADMIN_ROLES_FIELDS:
    if field != 'roleName' and field in role:
      printKeyValueList([field, role[field]])
  jcount = len(role.get('rolePrivileges', []))
  if jcount > 0:
    printKeyValueList(['rolePrivileges', jcount])
    Ind.Increment()
    j = 0
    for rolePrivilege in role['rolePrivileges']:
      j += 1
      printKeyValueList(['privilegeName', rolePrivilege['privilegeName']])
      Ind.Increment()
      printKeyValueList(['serviceId', rolePrivilege['serviceId']])
      Ind.Decrement()
    Ind.Decrement()
  Ind.Decrement()

# gam info adminrole <RoleItem> [privileges]
# gam print adminroles|roles [todrive <ToDriveAttribute>*]
#	[role <RoleItem>] [privileges] [oneitemperrow]
# gam show adminroles|roles
#	[role <RoleItem>] [privileges]
def doInfoPrintShowAdminRoles():
  cd = buildGAPIObject(API.DIRECTORY)
  fieldsList = PRINT_ADMIN_ROLES_FIELDS[:]
  csvPF = CSVPrintFile(fieldsList, PRINT_ADMIN_ROLES_FIELDS) if Act.csvFormat() else None
  oneItemPerRow = False
  if Act.Get() != Act.INFO:
    roleId = None
  else:
    _, roleId = getRoleId()
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if csvPF and myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif roleId is None and myarg == 'role':
      _, roleId = getRoleId()
    elif myarg == 'privileges':
      fieldsList.append('rolePrivileges')
    elif myarg == 'oneitemperrow':
      oneItemPerRow = True
    else:
      unknownArgumentExit()
  if csvPF and 'rolePrivileges' in fieldsList:
    if not oneItemPerRow:
      csvPF.AddTitles(['rolePrivileges'])
    else:
      csvPF.AddTitles(['privilegeName', 'serviceId'])
  try:
    if roleId is None:
      fields = getItemFieldsFromFieldsList('items', fieldsList)
      printGettingAllAccountEntities(Ent.ADMIN_ROLE)
      roles = callGAPIpages(cd.roles(), 'list', 'items',
                            pageMessage=getPageMessage(),
                            throwReasons=[GAPI.BAD_REQUEST, GAPI.CUSTOMER_NOT_FOUND,
                                          GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                            customer=GC.Values[GC.CUSTOMER_ID], fields=fields)
    else:
      fields = getFieldsFromFieldsList(fieldsList)
      roles = [callGAPI(cd.roles(), 'get',
                        throwReasons=[GAPI.NOT_FOUND, GAPI.FAILED_PRECONDITION,
                                      GAPI.BAD_REQUEST, GAPI.CUSTOMER_NOT_FOUND,
                                      GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                        customer=GC.Values[GC.CUSTOMER_ID], roleId=roleId, fields=fields)]
  except (GAPI.notFound, GAPI.failedPrecondition) as e:
    entityActionFailedWarning([Ent.ADMIN_ROLE, roleId], str(e))
  except (GAPI.badRequest, GAPI.customerNotFound):
    accessErrorExit(cd)
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))
  for role in roles:
    role.setdefault('isSuperAdminRole', False)
    role.setdefault('isSystemRole', False)
  if not csvPF:
    count = len(roles)
    performActionNumItems(count, Ent.ADMIN_ROLE)
    Ind.Increment()
    i = 0
    for role in roles:
      i += 1
      _showAdminRole(role, i, count)
    Ind.Decrement()
  else:
    for role in roles:
      if not oneItemPerRow or 'rolePrivileges' not in role:
        csvPF.WriteRowTitles(flattenJSON(role))
      else:
        privileges = role.pop('rolePrivileges')
        baserow = flattenJSON(role)
        for privilege in privileges:
          row = flattenJSON(privilege, flattened=baserow.copy())
          csvPF.WriteRowTitles(row)
  if csvPF:
    csvPF.writeCSVfile('Admin Roles')

ADMIN_SCOPE_TYPE_CHOICE_MAP = {'customer': 'CUSTOMER', 'orgunit': 'ORG_UNIT', 'org': 'ORG_UNIT', 'ou': 'ORG_UNIT'}

SECURITY_GROUP_CONDITION = "api.getAttribute('cloudidentity.googleapis.com/groups.labels', []).hasAny(['groups.security']) && resource.type == 'cloudidentity.googleapis.com/Group'"
NONSECURITY_GROUP_CONDITION = f'!{SECURITY_GROUP_CONDITION}'
ADMIN_CONDITION_CHOICE_MAP = {
  'securitygroup': SECURITY_GROUP_CONDITION,
  'nonsecuritygroup': NONSECURITY_GROUP_CONDITION,
  }

# gam create admin <EmailAddress>|<UniqueID> <RoleItem> customer|(org_unit <OrgUnitItem>)
#	[condition securitygroup|nonsecuritygroup]
def doCreateAdmin():
  cd = buildGAPIObject(API.DIRECTORY)
  user = getEmailAddress(returnUIDprefix='uid:')
  body = {'assignedTo': convertEmailAddressToUID(user, cd, emailType='any')}
  role, roleId = getRoleId()
  body['roleId'] = roleId
  body['scopeType'] = getChoice(ADMIN_SCOPE_TYPE_CHOICE_MAP, mapChoice=True)
  if body['scopeType'] == 'ORG_UNIT':
    orgUnit, orgUnitId = getOrgUnitId(cd)
    body['orgUnitId'] = orgUnitId[3:]
    scope = f'ORG_UNIT {orgUnit}'
  else:
    scope = 'CUSTOMER'
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'condition':
      body['condition'] = getChoice(ADMIN_CONDITION_CHOICE_MAP, mapChoice=True)
    else:
      unknownArgumentExit()
  try:
    result = callGAPI(cd.roleAssignments(), 'insert',
                      throwReasons=[GAPI.INTERNAL_ERROR, GAPI.BAD_REQUEST, GAPI.CUSTOMER_NOT_FOUND,
                                    GAPI.CUSTOMER_EXCEEDED_ROLE_ASSIGNMENTS_LIMIT, GAPI.SERVICE_NOT_AVAILABLE,
                                    GAPI.INVALID_ORGUNIT, GAPI.DUPLICATE, GAPI.CONDITION_NOT_MET,
                                    GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                      retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                      customer=GC.Values[GC.CUSTOMER_ID], body=body, fields='roleAssignmentId,assigneeType')
    assigneeType = result.get('assigneeType')
    if assigneeType == 'user':
      entityType = Ent.USER
    elif assigneeType == 'group':
      entityType = Ent.GROUP
    else:
      entityType = Ent.ADMINISTRATOR
    entityActionPerformedMessage([Ent.ADMIN_ROLE_ASSIGNMENT, result['roleAssignmentId']],
                                 f'{Ent.Singular(entityType)} {user}, {Ent.Singular(Ent.ADMIN_ROLE)} {role}, {Ent.Singular(Ent.SCOPE)} {scope}')
  except GAPI.internalError:
    pass
  except (GAPI.customerExceededRoleAssignmentsLimit, GAPI.serviceNotAvailable, GAPI.conditionNotMet) as e:
    entityActionFailedWarning([Ent.ADMINISTRATOR, user, Ent.ADMIN_ROLE, role], str(e))
  except GAPI.invalidOrgunit:
    entityActionFailedWarning([Ent.ADMINISTRATOR, user], Msg.INVALID_ORGUNIT)
  except GAPI.duplicate:
    entityActionFailedWarning([Ent.ADMINISTRATOR, user, Ent.ADMIN_ROLE, role], Msg.DUPLICATE)
  except (GAPI.badRequest, GAPI.customerNotFound):
    accessErrorExit(cd)
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))

# gam delete admin <RoleAssignmentId>
def doDeleteAdmin():
  cd = buildGAPIObject(API.DIRECTORY)
  roleAssignmentId = getString(Cmd.OB_ROLE_ASSIGNMENT_ID)
  checkForExtraneousArguments()
  try:
    callGAPI(cd.roleAssignments(), 'delete',
             throwReasons=[GAPI.NOT_FOUND, GAPI.OPERATION_NOT_SUPPORTED,
                           GAPI.INVALID_INPUT, GAPI.SERVICE_NOT_AVAILABLE, GAPI.RESOURCE_NOT_FOUND,
                           GAPI.FAILED_PRECONDITION, GAPI.BAD_REQUEST, GAPI.CUSTOMER_NOT_FOUND,
                           GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
             retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
             customer=GC.Values[GC.CUSTOMER_ID], roleAssignmentId=roleAssignmentId)
    entityActionPerformed([Ent.ADMIN_ROLE_ASSIGNMENT, roleAssignmentId])
  except (GAPI.notFound, GAPI.operationNotSupported, GAPI.invalidInput,
          GAPI.serviceNotAvailable, GAPI.resourceNotFound, GAPI.failedPrecondition) as e:
    entityActionFailedWarning([Ent.ADMIN_ROLE_ASSIGNMENT, roleAssignmentId], str(e))
  except (GAPI.badRequest, GAPI.customerNotFound):
    accessErrorExit(cd)
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))

ASSIGNEE_EMAILTYPE_TOFIELD_MAP = {
  'user': 'assignedToUser',
  'group': 'assignedToGroup',
  'serviceaccount': 'assignedToServiceAccount',
  }
PRINT_ADMIN_FIELDS = ['roleAssignmentId', 'roleId', 'assignedTo', 'scopeType', 'orgUnitId']
PRINT_ADMIN_TITLES = ['roleAssignmentId', 'roleId', 'role',
                      'assignedTo', 'assignedToUser', 'assignedToGroup', 'assignedToServiceAccount', 'assignedToUnknown',
                      'scopeType', 'orgUnitId', 'orgUnit']

# gam print admins [todrive <ToDriveAttribute>*]
#	[user|group <EmailAddress>|<UniqueID>] [role <RoleItem>] [condition]
#	[privileges] [oneitemperrow]
# gam show admins
#	[user|group <EmailAddress>|<UniqueID>] [role <RoleItem>] [condition] [privileges]
def doPrintShowAdmins():
  def _getPrivileges(admin):
    if showPrivileges:
      roleId = admin['roleId']
      if roleId not in rolePrivileges:
        try:
          rolePrivileges[roleId] = callGAPI(cd.roles(), 'get',
                                            throwReasons=[GAPI.NOT_FOUND, GAPI.FAILED_PRECONDITION,
                                                          GAPI.SERVICE_NOT_AVAILABLE, GAPI.BAD_REQUEST, GAPI.CUSTOMER_NOT_FOUND,
                                                          GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                                            retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                                            customer=GC.Values[GC.CUSTOMER_ID],
                                            roleId=roleId,
                                            fields='rolePrivileges')
        except (GAPI.notFound, GAPI.failedPrecondition, GAPI.serviceNotAvailable) as e:
          entityActionFailedExit([Ent.USER, userKey, Ent.ADMIN_ROLE, admin['roleId']], str(e))
          rolePrivileges[roleId] = None
        except (GAPI.badRequest, GAPI.customerNotFound):
          accessErrorExit(cd)
        except (GAPI.forbidden, GAPI.permissionDenied) as e:
          ClientAPIAccessDeniedExit(str(e))
      return rolePrivileges[roleId]

  def _setNamesFromIds(admin, privileges):
    admin['role'] = role_from_roleid(admin['roleId'])
    assignedTo = admin['assignedTo']
    admin['assignedToUnknown'] = False
    if assignedTo not in assignedToIdEmailMap:
      assigneeType = admin.get('assigneeType')
      assignedToField = ASSIGNEE_EMAILTYPE_TOFIELD_MAP.get(assigneeType, None)
      assigneeEmail, assigneeType = convertUIDtoEmailAddressWithType(f'uid:{assignedTo}', cd, sal,
                                                                     emailTypes=list(ASSIGNEE_EMAILTYPE_TOFIELD_MAP.keys()))
      if not assignedToField and assigneeType in ASSIGNEE_EMAILTYPE_TOFIELD_MAP:
        assignedToField = ASSIGNEE_EMAILTYPE_TOFIELD_MAP[assigneeType]
      if assigneeType == 'unknown':
        assignedToField = 'assignedToUnknown'
        assigneeEmail = True
      assignedToIdEmailMap[assignedTo] = {'assignedToField': assignedToField, 'assigneeEmail': assigneeEmail}
    admin[assignedToIdEmailMap[assignedTo]['assignedToField']] = assignedToIdEmailMap[assignedTo]['assigneeEmail']
    if privileges is not None:
      admin.update(privileges)
    if 'orgUnitId' in admin:
      admin['orgUnit'] = convertOrgUnitIDtoPath(cd, f'id:{admin["orgUnitId"]}')
    if 'condition' in admin:
      if admin['condition'] == SECURITY_GROUP_CONDITION:
        admin['condition'] = 'securitygroup'
      elif admin['condition'] == NONSECURITY_GROUP_CONDITION:
        admin['condition'] = 'nonsecuritygroup'

  cd = buildGAPIObject(API.DIRECTORY)
  sal = buildGAPIObject(API.SERVICEACCOUNTLOOKUP)
  csvPF = CSVPrintFile(PRINT_ADMIN_TITLES) if Act.csvFormat() else None
  roleId = None
  userKey = None
  oneItemPerRow = showPrivileges = False
  kwargs = {}
  rolePrivileges = {}
  fieldsList = PRINT_ADMIN_FIELDS
  assignedToIdEmailMap = {}
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if csvPF and myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif myarg in {'user', 'group'}:
      userKey = kwargs['userKey'] = getEmailAddress()
    elif myarg == 'role':
      _, roleId = getRoleId()
    elif myarg == 'condition':
      fieldsList.append('condition')
      if csvPF:
        csvPF.AddTitle('condition')
    elif myarg == 'privileges':
      showPrivileges = True
    elif myarg == 'oneitemperrow':
      oneItemPerRow = True
    else:
      unknownArgumentExit()
  if roleId and not kwargs:
    kwargs['roleId'] = roleId
    roleId = None
  fields = getItemFieldsFromFieldsList('items', fieldsList)
  printGettingAllAccountEntities(Ent.ADMIN_ROLE_ASSIGNMENT)
  try:
    admins = callGAPIpages(cd.roleAssignments(), 'list', 'items',
                           pageMessage=getPageMessage(),
                           throwReasons=[GAPI.INVALID, GAPI.USER_NOT_FOUND,
                                         GAPI.FORBIDDEN, GAPI.SERVICE_NOT_AVAILABLE,
                                         GAPI.BAD_REQUEST, GAPI.CUSTOMER_NOT_FOUND,
                                         GAPI.FORBIDDEN, GAPI.PERMISSION_DENIED],
                           retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                           customer=GC.Values[GC.CUSTOMER_ID], fields=fields, **kwargs)
  except (GAPI.invalid, GAPI.userNotFound):
    entityUnknownWarning(Ent.ADMINISTRATOR, userKey)
    return
  except (GAPI.serviceNotAvailable) as e:
    entityActionFailedExit([Ent.ADMINISTRATOR, userKey, Ent.ADMIN_ROLE, roleId], str(e))
  except (GAPI.badRequest, GAPI.customerNotFound):
    accessErrorExit(cd)
  except (GAPI.forbidden, GAPI.permissionDenied) as e:
    ClientAPIAccessDeniedExit(str(e))
  if not csvPF:
    count = len(admins)
    performActionNumItems(count, Ent.ADMIN_ROLE_ASSIGNMENT)
    Ind.Increment()
    i = 0
    for admin in admins:
      i += 1
      if roleId and roleId != admin['roleId']:
        continue
      _setNamesFromIds(admin, _getPrivileges(admin))
      printEntity([Ent.ADMIN_ROLE_ASSIGNMENT, admin['roleAssignmentId']], i, count)
      Ind.Increment()
      for field in PRINT_ADMIN_TITLES:
        if field in admin:
          if field == 'roleAssignmentId':
            continue
          if field != 'rolePrivileges':
            printKeyValueList([field, admin[field]])
          else:
            showJSON(None, admin[field])
      Ind.Decrement()
    Ind.Decrement()
  else:
    for admin in admins:
      if roleId and roleId != admin['roleId']:
        continue
      _setNamesFromIds(admin, _getPrivileges(admin))
      if not oneItemPerRow or 'rolePrivileges' not in admin:
        csvPF.WriteRowTitles(flattenJSON(admin))
      else:
        privileges = admin.pop('rolePrivileges')
        baserow = flattenJSON(admin)
        for privilege in privileges:
          row = flattenJSON(privilege, flattened=baserow.copy())
          csvPF.WriteRowTitles(row)
  if csvPF:
    csvPF.writeCSVfile('Admins')

def getTransferApplications(dt):
  try:
    return callGAPIpages(dt.applications(), 'list', 'applications',
                         throwReasons=[GAPI.UNKNOWN_ERROR, GAPI.FORBIDDEN],
                         customerId=GC.Values[GC.CUSTOMER_ID], fields='applications(id,name,transferParams)')
  except (GAPI.unknownError, GAPI.forbidden):
    accessErrorExit(None)

def _convertTransferAppIDtoName(apps, appID):
  for app in apps:
    if appID == app['id']:
      return app['name']
  return f'applicationId: {appID}'

DRIVE_AND_DOCS_APP_NAME = 'drive and docs'
GOOGLE_LOOKER_STUDIO_APP_NAME = 'looker studio'

SERVICE_NAME_CHOICE_MAP = {
  'datastudio': GOOGLE_LOOKER_STUDIO_APP_NAME,
  'drive': DRIVE_AND_DOCS_APP_NAME,
  'googledrive': DRIVE_AND_DOCS_APP_NAME,
  'gdrive': DRIVE_AND_DOCS_APP_NAME,
  'lookerstudio': GOOGLE_LOOKER_STUDIO_APP_NAME,
  }

def _validateTransferAppName(apps, appName):
  appName = appName.strip().lower()
  appName = SERVICE_NAME_CHOICE_MAP.get(appName, appName)
  appNameList = []
  for app in apps:
    if appName == app['name'].lower():
      return (app['name'], app['id'])
    appNameList.append(app['name'].lower())
  invalidChoiceExit(appName, appNameList, True)

PRIVACY_LEVEL_CHOICE_MAP = {
  'private': ['PRIVATE'],
  'shared': ['SHARED'],
  'all': ['PRIVATE', 'SHARED'],
  }

# gam create datatransfer|transfer <OldOwnerID> <ServiceNameList> <NewOwnerID>
#	[private|shared|all] [release_resources] (<ParameterKey> <ParameterValue>)*
#	[wait <Integer> <Integer>]
def doCreateDataTransfer():
  def _assignAppParameter(key, value, doubleBackup=False):
    keyValid = False
    for app in apps:
      for params in app.get('transferParams', []):
        if key == params['key']:
          appIndex = appIndicies.get(app['id'])
          if appIndex is not None:
            body['applicationDataTransfers'][appIndex].setdefault('applicationTransferParams', [])
            body['applicationDataTransfers'][appIndex]['applicationTransferParams'].append({'key': key, 'value': value})
            keyValid = True
          break
    if not keyValid:
      Cmd.Backup()
      if doubleBackup:
        Cmd.Backup()
      usageErrorExit(Msg.NO_DATA_TRANSFER_APP_FOR_PARAMETER.format(key))

  dt = buildGAPIObject(API.DATATRANSFER)
  apps = getTransferApplications(dt)
  old_owner = getEmailAddress(returnUIDprefix='uid:')
  body = {'oldOwnerUserId': convertEmailAddressToUID(old_owner)}
  appIndicies = {}
  appNameList = []
  waitInterval = waitRetries = 0
  i = 0
  body['applicationDataTransfers'] = []
  for appName in getString(Cmd.OB_SERVICE_NAME_LIST).split(','):
    appName, appId = _validateTransferAppName(apps, appName)
    body['applicationDataTransfers'].append({'applicationId': appId})
    appIndicies[appId] = i
    i += 1
    appNameList.append(appName)
  new_owner = getEmailAddress(returnUIDprefix='uid:')
  body['newOwnerUserId'] = convertEmailAddressToUID(new_owner)
  if body['oldOwnerUserId'] == body['newOwnerUserId']:
    Cmd.Backup()
    usageErrorExit(Msg.NEW_OWNER_MUST_DIFFER_FROM_OLD_OWNER)
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg in PRIVACY_LEVEL_CHOICE_MAP:
      _assignAppParameter('PRIVACY_LEVEL', PRIVACY_LEVEL_CHOICE_MAP[myarg])
    elif myarg == 'releaseresources':
      if getBoolean():
        _assignAppParameter('RELEASE_RESOURCES', ['TRUE'])
    elif myarg == 'wait':
      waitInterval = getInteger(minVal=5, maxVal=60)
      waitRetries = getInteger(minVal=0)
    else:
      _assignAppParameter(Cmd.Previous().upper(), getString(Cmd.OB_PARAMETER_VALUE).upper().split(','), True)
  try:
    result = callGAPI(dt.transfers(), 'insert',
                      throwReasons=[GAPI.UNKNOWN_ERROR, GAPI.FORBIDDEN],
                      body=body, fields='id')
  except (GAPI.unknownError, GAPI.forbidden) as e:
    entityActionFailedExit([Ent.USER, old_owner], str(e))
  entityActionPerformed([Ent.TRANSFER_REQUEST, None])
  Ind.Increment()
  printEntity([Ent.TRANSFER_ID, result['id']])
  printEntity([Ent.SERVICE, ','.join(appNameList)])
  printKeyValueList([Msg.FROM, old_owner])
  printKeyValueList([Msg.TO, new_owner])
  Ind.Decrement()
  if waitRetries == 0:
    return
  retry = 0
  status = 'inProgress'
  dtId = result['id']
  while True:
    writeStderr(Ind.Spaces()+Msg.WAITING_FOR_DATA_TRANSFER_TO_COMPLETE_SLEEPING.format(waitInterval))
    time.sleep(waitInterval)
    try:
      result = callGAPI(dt.transfers(), 'get',
                        throwReasons=[GAPI.NOT_FOUND],
                        dataTransferId=dtId, fields='overallTransferStatusCode')
      if result['overallTransferStatusCode'] == 'completed':
        status = result['overallTransferStatusCode']
        break
      retry += 1
      if retry >= waitRetries:
        break
    except GAPI.notFound:
      entityActionFailedWarning([Ent.TRANSFER_ID, dtId], Msg.DOES_NOT_EXIST)
      break
  printEntity([Ent.TRANSFER_ID, dtId, Ent.STATUS, status])

def _showTransfer(apps, transfer, i, count):
  printEntity([Ent.TRANSFER_ID, transfer['id']], i, count)
  Ind.Increment()
  printKeyValueList(['Request Time', formatLocalTime(transfer['requestTime'])])
  printKeyValueList(['Old Owner', convertUserIDtoEmail(transfer['oldOwnerUserId'])])
  printKeyValueList(['New Owner', convertUserIDtoEmail(transfer['newOwnerUserId'])])
  printKeyValueList(['Overall Transfer Status', transfer['overallTransferStatusCode']])
  for app in transfer['applicationDataTransfers']:
    printKeyValueList(['Application', _convertTransferAppIDtoName(apps, app['applicationId'])])
    Ind.Increment()
    printKeyValueList(['Status', app['applicationTransferStatus']])
    printKeyValueList(['Parameters'])
    Ind.Increment()
    if 'applicationTransferParams' in app:
      for param in app['applicationTransferParams']:
        key = param['key']
        value = param.get('value', [])
        if value:
          printKeyValueList([key, ','.join(value)])
        else:
          printKeyValueList([key])
    else:
      printKeyValueList(['None'])
    Ind.Decrement()
    Ind.Decrement()
  Ind.Decrement()

# gam info datatransfer|transfer <TransferID>
def doInfoDataTransfer():
  dt = buildGAPIObject(API.DATATRANSFER)
  apps = getTransferApplications(dt)
  dtId = getString(Cmd.OB_TRANSFER_ID)
  checkForExtraneousArguments()
  try:
    transfer = callGAPI(dt.transfers(), 'get',
                        throwReasons=[GAPI.NOT_FOUND],
                        dataTransferId=dtId)
    _showTransfer(apps, transfer, 0, 0)
  except GAPI.notFound:
    entityActionFailedWarning([Ent.TRANSFER_ID, dtId], Msg.DOES_NOT_EXIST)

DATA_TRANSFER_STATUS_MAP = {
  'completed': 'completed',
  'failed': 'failed',
#  'pending': 'pending',
  'inprogress': 'inProgress',
  }
DATA_TRANSFER_SORT_TITLES = ['id', 'requestTime', 'oldOwnerUserEmail', 'newOwnerUserEmail',
                             'overallTransferStatusCode', 'application', 'applicationId', 'status']

# gam print datatransfers|transfers [todrive <ToDriveAttribute>*]
#	[olduser|oldowner <UserItem>] [newuser|newowner <UserItem>]
#	[status <String>] [delimiter <Character>]
#	(addcsvdata <FieldName> <String>)*
# gam show datatransfers|transfers
#	[olduser|oldowner <UserItem>] [newuser|newowner <UserItem>]
#	[status <String>] [delimiter <Character>]
def doPrintShowDataTransfers():
  dt = buildGAPIObject(API.DATATRANSFER)
  apps = getTransferApplications(dt)
  newOwnerUserId = None
  oldOwnerUserId = None
  status = None
  csvPF = CSVPrintFile(['id'], DATA_TRANSFER_SORT_TITLES) if Act.csvFormat() else None
  delimiter = GC.Values[GC.CSV_OUTPUT_FIELD_DELIMITER]
  addCSVData = {}
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if csvPF and myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif myarg in {'olduser', 'oldowner'}:
      oldOwnerUserId = convertEmailAddressToUID(getEmailAddress(returnUIDprefix='uid:'))
    elif myarg in {'newuser', 'newowner'}:
      newOwnerUserId = convertEmailAddressToUID(getEmailAddress(returnUIDprefix='uid:'))
    elif myarg == 'status':
      status = getChoice(DATA_TRANSFER_STATUS_MAP, mapChoice=True)
    elif myarg == 'delimiter':
      delimiter = getCharacter()
    elif csvPF and myarg == 'addcsvdata':
      k = getString(Cmd.OB_STRING)
      addCSVData[k] = getString(Cmd.OB_STRING, minLen=0)
    else:
      unknownArgumentExit()
  try:
    transfers = callGAPIpages(dt.transfers(), 'list', 'dataTransfers',
                              throwReasons=[GAPI.UNKNOWN_ERROR, GAPI.FORBIDDEN, GAPI.INVALID_INPUT],
                              customerId=GC.Values[GC.CUSTOMER_ID], status=status,
                              newOwnerUserId=newOwnerUserId, oldOwnerUserId=oldOwnerUserId)
  except (GAPI.unknownError, GAPI.forbidden, GAPI.invalidInput) as e:
    entityActionFailedExit([Ent.TRANSFER_REQUEST, None], str(e))
  if not csvPF:
    count = len(transfers)
    performActionNumItems(count, Ent.TRANSFER_REQUEST)
    Ind.Increment()
    i = 0
    for transfer in sorted(transfers, key=lambda k: k['requestTime']):
      i += 1
      _showTransfer(apps, transfer, i, count)
    Ind.Decrement()
  else:
    for transfer in sorted(transfers, key=lambda k: k['requestTime']):
      row = {}
      row['id'] = transfer['id']
      row['requestTime'] = formatLocalTime(transfer['requestTime'])
      row['oldOwnerUserEmail'] = convertUserIDtoEmail(transfer['oldOwnerUserId'])
      row['newOwnerUserEmail'] = convertUserIDtoEmail(transfer['newOwnerUserId'])
      row['overallTransferStatusCode'] = transfer['overallTransferStatusCode']
      if addCSVData:
        row.update(addCSVData)
      for app in transfer['applicationDataTransfers']:
        xrow = row.copy()
        xrow['application'] = _convertTransferAppIDtoName(apps, app['applicationId'])
        xrow['applicationId'] = app['applicationId']
        xrow['status'] = app['applicationTransferStatus']
        for param in app.get('applicationTransferParams', []):
          key = param['key']
          xrow[key] = delimiter.join(param.get('value', [] if key != 'RELEASE_RESOURCES' else ['TRUE']))
        csvPF.WriteRowTitles(xrow)
  if csvPF:
    csvPF.writeCSVfile('Data Transfers')

# gam show transferapps
def doShowTransferApps():
  dt = buildGAPIObject(API.DATATRANSFER)
  checkForExtraneousArguments()
  Act.Set(Act.SHOW)
  try:
    apps = callGAPIpages(dt.applications(), 'list', 'applications',
                         throwReasons=[GAPI.UNKNOWN_ERROR, GAPI.FORBIDDEN],
                         customerId=GC.Values[GC.CUSTOMER_ID], fields='applications(id,name,transferParams)')
  except (GAPI.unknownError, GAPI.forbidden):
    accessErrorExit(None)
  count = len(apps)
  performActionNumItems(count, Ent.TRANSFER_APPLICATION)
  Ind.Increment()
  i = 0
  for app in apps:
    i += 1
    printKeyValueListWithCount([app['name']], i, count)
    Ind.Increment()
    printKeyValueList(['id', app['id']])
    transferParams = app.get('transferParams', [])
    if transferParams:
      printKeyValueList(['Parameters'])
      Ind.Increment()
      for param in transferParams:
        printKeyValueList(['key', param['key']])
        Ind.Increment()
        printKeyValueList(['value', ','.join(param['value'])])
        Ind.Decrement()
      Ind.Decrement()
    Ind.Decrement()
  Ind.Decrement()

def _getOrgInheritance(myarg, body):
  if myarg == 'noinherit':
    Cmd.Backup()
    deprecatedArgumentExit(myarg)
  elif myarg == 'inherit':
    body['blockInheritance'] = False
  elif myarg in {'blockinheritance', 'inheritanceblocked'}:
    location = Cmd.Location()-1
    if getBoolean():
      Cmd.SetLocation(location)
      deprecatedArgumentExit(myarg)
    body['blockInheritance'] = False
  else:
    return False
  return True

# gam create org|ou <String> [description <String>] [parent <OrgUnitItem>] [inherit|(blockinheritance False)] [buildpath]
def doCreateOrg():

  def _createOrg(body, parentPath, fullPath):
    try:
      callGAPI(cd.orgunits(), 'insert',
               throwReasons=[GAPI.INVALID_PARENT_ORGUNIT, GAPI.INVALID_ORGUNIT, GAPI.BACKEND_ERROR, GAPI.BAD_REQUEST, GAPI.INVALID_CUSTOMER_ID, GAPI.LOGIN_REQUIRED],
               customerId=GC.Values[GC.CUSTOMER_ID], body=body, fields='')
      entityActionPerformed([Ent.ORGANIZATIONAL_UNIT, fullPath])
    except GAPI.invalidParentOrgunit:
      entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, fullPath, Ent.PARENT_ORGANIZATIONAL_UNIT, parentPath], Msg.ENTITY_DOES_NOT_EXIST.format(Ent.Singular(Ent.PARENT_ORGANIZATIONAL_UNIT)))
      return False
    except (GAPI.invalidOrgunit, GAPI.backendError):
      entityDuplicateWarning([Ent.ORGANIZATIONAL_UNIT, fullPath])
    except (GAPI.badRequest, GAPI.invalidCustomerId, GAPI.loginRequired):
      checkEntityAFDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, fullPath)
    return True

  cd = buildGAPIObject(API.DIRECTORY)
  name = getOrgUnitItem(pathOnly=True, absolutePath=False)
  parent = ''
  body = {}
  buildPath = False
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'description':
      body['description'] = getStringWithCRsNLs()
    elif myarg == 'parent':
      parent = getOrgUnitItem()
    elif _getOrgInheritance(myarg, body):
      pass
    elif myarg == 'buildpath':
      buildPath = True
    else:
      unknownArgumentExit()
  if parent.startswith('id:'):
    parentPath = None
    try:
      parentPath = callGAPI(cd.orgunits(), 'get',
                            throwReasons=GAPI.ORGUNIT_GET_THROW_REASONS,
                            customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath=parent, fields='orgUnitPath')['orgUnitPath']
    except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError):
      pass
    except (GAPI.badRequest, GAPI.invalidCustomerId, GAPI.loginRequired):
      errMsg = accessErrorMessage(cd)
      if errMsg:
        systemErrorExit(INVALID_DOMAIN_RC, errMsg)
    if not parentPath and not buildPath:
      entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, name, Ent.PARENT_ORGANIZATIONAL_UNIT, parent], Msg.ENTITY_DOES_NOT_EXIST.format(Ent.Singular(Ent.PARENT_ORGANIZATIONAL_UNIT)))
      return
    parent = parentPath
  if parent == '/':
    orgUnitPath = parent+name
  else:
    orgUnitPath = parent+'/'+name
  if orgUnitPath.count('/') > 1:
    body['parentOrgUnitPath'], body['name'] = orgUnitPath.rsplit('/', 1)
  else:
    body['parentOrgUnitPath'] = '/'
    body['name'] = orgUnitPath[1:]
  parent = body['parentOrgUnitPath']
  if _createOrg(body, parent, orgUnitPath) or not buildPath:
    return
  description = body.pop('description', None)
  fullPath = '/'
  getPath = ''
  orgNames = orgUnitPath.split('/')
  n = len(orgNames)-1
  for i in range(1, n+1):
    body['parentOrgUnitPath'] = fullPath
    if fullPath != '/':
      fullPath += '/'
    fullPath += orgNames[i]
    if getPath != '':
      getPath += '/'
    getPath += orgNames[i]
    try:
      callGAPI(cd.orgunits(), 'get',
               throwReasons=GAPI.ORGUNIT_GET_THROW_REASONS,
               customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath=encodeOrgUnitPath(getPath), fields='')
      printKeyValueList([Ent.Singular(Ent.ORGANIZATIONAL_UNIT), fullPath, Msg.EXISTS])
    except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError):
      body['name'] = orgNames[i]
      if i == n and description:
        body['description'] = description
      if not _createOrg(body, body['parentOrgUnitPath'], fullPath):
        return
    except (GAPI.badRequest, GAPI.invalidCustomerId, GAPI.loginRequired):
      checkEntityAFDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, fullPath)

def checkOrgUnitPathExists(cd, orgUnitPath, i=0, count=0, showError=False):
  if orgUnitPath == '/':
    _, orgUnitId = getOrgUnitId(cd, orgUnitPath)
    return (True, orgUnitPath, orgUnitId)
  try:
    orgUnit = callGAPI(cd.orgunits(), 'get',
                       throwReasons=GAPI.ORGUNIT_GET_THROW_REASONS,
                       customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath=encodeOrgUnitPath(makeOrgUnitPathRelative(orgUnitPath)),
                       fields='orgUnitPath,orgUnitId')
    return (True, orgUnit['orgUnitPath'], orgUnit['orgUnitId'])
  except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError):
    pass
  except (GAPI.badRequest, GAPI.invalidCustomerId, GAPI.loginRequired):
    errMsg = accessErrorMessage(cd)
    if errMsg:
      systemErrorExit(INVALID_DOMAIN_RC, errMsg)
  if showError:
    entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, orgUnitPath], Msg.DOES_NOT_EXIST, i, count)
  return (False, orgUnitPath, orgUnitPath)

def _batchMoveCrOSesToOrgUnit(cd, orgUnitPath, orgUnitId, i, count, items, quickCrOSMove, fromOrgUnitPath=None):
  def _callbackMoveCrOSesToOrgUnit(request_id, _, exception):
    ri = request_id.splitlines()
    if exception is None:
      if not fromOrgUnitPath:
        entityActionPerformed([Ent.ORGANIZATIONAL_UNIT, orgUnitPath, Ent.CROS_DEVICE, ri[RI_ITEM]], int(ri[RI_J]), int(ri[RI_JCOUNT]))
      else:
        entityModifierActionPerformed([Ent.ORGANIZATIONAL_UNIT, fromOrgUnitPath, Ent.CROS_DEVICE, ri[RI_ITEM]], toOrgUnitPath, int(ri[RI_J]), int(ri[RI_JCOUNT]))
    else:
      http_status, reason, message = checkGAPIError(exception)
      if reason in [GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN]:
        checkEntityItemValueAFDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, orgUnitPath, Ent.CROS_DEVICE, ri[RI_ITEM], int(ri[RI_J]), int(ri[RI_JCOUNT]))
      else:
        errMsg = getHTTPError({}, http_status, reason, message)
        if not fromOrgUnitPath:
          entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, orgUnitPath, Ent.CROS_DEVICE, ri[RI_ITEM]], errMsg, int(ri[RI_J]), int(ri[RI_JCOUNT]))
        else:
          entityModifierActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, fromOrgUnitPath, Ent.CROS_DEVICE, ri[RI_ITEM]], toOrgUnitPath, errMsg, int(ri[RI_J]), int(ri[RI_JCOUNT]))

  jcount = len(items)
  if not fromOrgUnitPath:
    entityPerformActionNumItems([Ent.ORGANIZATIONAL_UNIT, orgUnitPath], jcount, Ent.CROS_DEVICE, i, count)
  else:
    toOrgUnitPath = f'{Act.MODIFIER_TO} {Ent.Singular(Ent.ORGANIZATIONAL_UNIT)}: {orgUnitPath}'
    entityPerformActionNumItemsModifier([Ent.ORGANIZATIONAL_UNIT, fromOrgUnitPath], jcount, Ent.CROS_DEVICE, toOrgUnitPath, i, count)
  Ind.Increment()
  if not quickCrOSMove:
    svcargs = dict([('customerId', GC.Values[GC.CUSTOMER_ID]),
                    ('deviceId', None),
                    ('fields', '')]+GM.Globals[GM.EXTRA_ARGS_LIST])
    if not GC.Values[GC.UPDATE_CROS_OU_WITH_ID]:
      svcargs['body'] = {'orgUnitPath': orgUnitPath}
      method = getattr(cd.chromeosdevices(), 'update')
    else:
      svcargs['body'] = {'orgUnitPath': orgUnitPath, 'orgUnitId': orgUnitId}
      method = getattr(cd.chromeosdevices(), 'patch')
    dbatch = cd.new_batch_http_request(callback=_callbackMoveCrOSesToOrgUnit)
    bcount = 0
    j = 0
    for deviceId in items:
      j += 1
      svcparms = svcargs.copy()
      svcparms['deviceId'] = deviceId
      dbatch.add(method(**svcparms), request_id=batchRequestID('', 0, 0, j, jcount, deviceId))
      bcount += 1
      if bcount >= GC.Values[GC.BATCH_SIZE]:
        executeBatch(dbatch)
        dbatch = cd.new_batch_http_request(callback=_callbackMoveCrOSesToOrgUnit)
        bcount = 0
    if bcount > 0:
      dbatch.execute()
  else:
    bcount = 0
    j = 0
    while bcount < jcount:
      kcount = min(jcount-bcount, GC.Values[GC.BATCH_SIZE])
      try:
        deviceIds = items[bcount:bcount+kcount]
        callGAPI(cd.chromeosdevices(), 'moveDevicesToOu',
                 throwReasons=[GAPI.INVALID_ORGUNIT, GAPI.INVALID_INPUT, GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN],
                 customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath=orgUnitPath,
                 body={'deviceIds': deviceIds})
        for deviceId in deviceIds:
          j += 1
          entityActionPerformed([Ent.ORGANIZATIONAL_UNIT, orgUnitPath, Ent.CROS_DEVICE, deviceId], j, jcount)
        bcount += kcount
      except GAPI.invalidOrgunit:
        entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, orgUnitPath], Msg.INVALID_ORGUNIT, i, count)
        break
      except GAPI.invalidInput as e:
        entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, orgUnitPath, Ent.CROS_DEVICE, None], str(e), i, count)
        break
      except GAPI.resourceNotFound as e:
        entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, orgUnitPath, Ent.CROS_DEVICE, ','.join(deviceIds)], str(e), i, count)
        break
      except (GAPI.badRequest, GAPI.forbidden):
        checkEntityAFDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, orgUnitPath, i, count)
        bcount += kcount
  Ind.Decrement()

def _batchMoveUsersToOrgUnit(cd, orgUnitPath, i, count, items, fromOrgUnitPath=None):
  _MOVE_USER_REASON_TO_MESSAGE_MAP = {GAPI.USER_NOT_FOUND: Msg.DOES_NOT_EXIST, GAPI.DOMAIN_NOT_FOUND: Msg.SERVICE_NOT_APPLICABLE, GAPI.FORBIDDEN: Msg.SERVICE_NOT_APPLICABLE}
  def _callbackMoveUsersToOrgUnit(request_id, _, exception):
    ri = request_id.splitlines()
    if exception is None:
      if not fromOrgUnitPath:
        entityActionPerformed([Ent.ORGANIZATIONAL_UNIT, orgUnitPath, Ent.USER, ri[RI_ITEM]], int(ri[RI_J]), int(ri[RI_JCOUNT]))
      else:
        entityModifierActionPerformed([Ent.ORGANIZATIONAL_UNIT, fromOrgUnitPath, Ent.USER, ri[RI_ITEM]], toOrgUnitPath, int(ri[RI_J]), int(ri[RI_JCOUNT]))
    else:
      http_status, reason, message = checkGAPIError(exception)
      errMsg = getHTTPError(_MOVE_USER_REASON_TO_MESSAGE_MAP, http_status, reason, message)
      if not fromOrgUnitPath:
        entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, orgUnitPath, Ent.USER, ri[RI_ITEM]], errMsg, int(ri[RI_J]), int(ri[RI_JCOUNT]))
      else:
        entityModifierActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, fromOrgUnitPath, Ent.USER, ri[RI_ITEM]], toOrgUnitPath, errMsg, int(ri[RI_J]), int(ri[RI_JCOUNT]))

  jcount = len(items)
  if not fromOrgUnitPath:
    entityPerformActionNumItems([Ent.ORGANIZATIONAL_UNIT, orgUnitPath], jcount, Ent.USER, i, count)
  else:
    toOrgUnitPath = f'{Act.MODIFIER_TO} {Ent.Singular(Ent.ORGANIZATIONAL_UNIT)}: {orgUnitPath}'
    entityPerformActionNumItemsModifier([Ent.ORGANIZATIONAL_UNIT, fromOrgUnitPath], jcount, Ent.USER, toOrgUnitPath, i, count)
  Ind.Increment()
  svcargs = dict([('userKey', None), ('body', {'orgUnitPath': orgUnitPath}), ('fields', '')]+GM.Globals[GM.EXTRA_ARGS_LIST])
  method = getattr(cd.users(), 'update')
  dbatch = cd.new_batch_http_request(callback=_callbackMoveUsersToOrgUnit)
  bcount = 0
  j = 0
  for user in items:
    j += 1
    svcparms = svcargs.copy()
    svcparms['userKey'] = normalizeEmailAddressOrUID(user)
    dbatch.add(method(**svcparms), request_id=batchRequestID('', 0, 0, j, jcount, svcparms['userKey']))
    bcount += 1
    if bcount >= GC.Values[GC.BATCH_SIZE]:
      executeBatch(dbatch)
      dbatch = cd.new_batch_http_request(callback=_callbackMoveUsersToOrgUnit)
      bcount = 0
  if bcount > 0:
    dbatch.execute()
  Ind.Decrement()

def _doUpdateOrgs(entityList):
  cd = buildGAPIObject(API.DIRECTORY)
  if checkArgumentPresent(['move', 'add']):
    entityType, items = getEntityToModify(defaultEntityType=Cmd.ENTITY_USERS, crosAllowed=True)
    orgItemLists = items if isinstance(items, dict) else None
    quickCrOSMove = GC.Values[GC.QUICK_CROS_MOVE]
    while Cmd.ArgumentsRemaining():
      myarg = getArgument()
      if entityType == Cmd.ENTITY_CROS and myarg == 'quickcrosmove':
        quickCrOSMove = getBoolean()
      else:
        unknownArgumentExit()
    Act.Set(Act.ADD)
    i = 0
    count = len(entityList)
    for orgUnitPath in entityList:
      i += 1
      if orgItemLists:
        items = orgItemLists[orgUnitPath]
      status, orgUnitPath, orgUnitId = checkOrgUnitPathExists(cd, orgUnitPath, i, count, True)
      if not status:
        continue
      if entityType == Cmd.ENTITY_USERS:
        _batchMoveUsersToOrgUnit(cd, orgUnitPath, i, count, items)
      else:
        _batchMoveCrOSesToOrgUnit(cd, orgUnitPath, orgUnitId, i, count, items, quickCrOSMove)
  elif checkArgumentPresent(['sync']):
    entityType, syncMembers = getEntityToModify(defaultEntityType=Cmd.ENTITY_USERS, crosAllowed=True)
    cmdEntityType = Cmd.ENTITY_OU if entityType == Cmd.ENTITY_USERS else Cmd.ENTITY_CROS_OU
    orgItemLists = syncMembers if isinstance(syncMembers, dict) else None
    if orgItemLists is None:
      syncMembersSet = set(syncMembers)
    removeToOrgUnitPath = '/'
    removeToOrgUnitId = None
    quickCrOSMove = GC.Values[GC.QUICK_CROS_MOVE]
    while Cmd.ArgumentsRemaining():
      myarg = getArgument()
      if entityType == Cmd.ENTITY_CROS and myarg == 'quickcrosmove':
        quickCrOSMove = getBoolean()
      elif myarg == 'removetoou':
        status, removeToOrgUnitPath, removeToOrgUnitId = checkOrgUnitPathExists(cd, getOrgUnitItem())
        if not status:
          entityDoesNotExistExit(Ent.ORGANIZATIONAL_UNIT, removeToOrgUnitPath)
      else:
        unknownArgumentExit()
    if entityType == Cmd.ENTITY_CROS and not removeToOrgUnitId:
      _, removeToOrgUnitPath, removeToOrgUnitId = checkOrgUnitPathExists(cd, removeToOrgUnitPath)
    i = 0
    count = len(entityList)
    for orgUnitPath in entityList:
      i += 1
      if orgItemLists:
        syncMembersSet = set(orgItemLists[orgUnitPath])
      status, orgUnitPath, orgUnitId = checkOrgUnitPathExists(cd, orgUnitPath, i, count, True)
      if not status:
        continue
      currentMembersSet = set(getItemsToModify(cmdEntityType, orgUnitPath))
      if entityType == Cmd.ENTITY_USERS:
        Act.Set(Act.ADD)
        _batchMoveUsersToOrgUnit(cd, orgUnitPath, i, count, list(syncMembersSet-currentMembersSet))
        Act.Set(Act.REMOVE)
        _batchMoveUsersToOrgUnit(cd, removeToOrgUnitPath, i, count, list(currentMembersSet-syncMembersSet), orgUnitPath)
      else:
        Act.Set(Act.ADD)
        _batchMoveCrOSesToOrgUnit(cd, orgUnitPath, orgUnitId, i, count, list(syncMembersSet-currentMembersSet), quickCrOSMove)
        Act.Set(Act.REMOVE)
        _batchMoveCrOSesToOrgUnit(cd, removeToOrgUnitPath, removeToOrgUnitId, i, count, list(currentMembersSet-syncMembersSet), quickCrOSMove, orgUnitPath)
  else:
    body = {}
    while Cmd.ArgumentsRemaining():
      myarg = getArgument()
      if myarg == 'name':
        body['name'] = getString(Cmd.OB_STRING)
      elif myarg == 'description':
        body['description'] = getStringWithCRsNLs()
      elif myarg == 'parent':
        parent = getOrgUnitItem()
        if parent.startswith('id:'):
          body['parentOrgUnitId'] = parent
        else:
          body['parentOrgUnitPath'] = parent
      elif _getOrgInheritance(myarg, body):
        pass
      else:
        unknownArgumentExit()
    i = 0
    count = len(entityList)
    for orgUnitPath in entityList:
      i += 1
      try:
        callGAPI(cd.orgunits(), 'update',
                 throwReasons=[GAPI.INVALID_ORGUNIT, GAPI.ORGUNIT_NOT_FOUND, GAPI.BACKEND_ERROR, GAPI.INVALID_ORGUNIT_NAME,
                               GAPI.CONDITION_NOT_MET, GAPI.BAD_REQUEST, GAPI.INVALID_CUSTOMER_ID, GAPI.LOGIN_REQUIRED],
                 customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath=encodeOrgUnitPath(makeOrgUnitPathRelative(orgUnitPath)), body=body, fields='')
        entityActionPerformed([Ent.ORGANIZATIONAL_UNIT, orgUnitPath], i, count)
      except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError):
        entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, orgUnitPath], Msg.DOES_NOT_EXIST, i, count)
      except GAPI.invalidOrgunitName as e:
        entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, orgUnitPath, Ent.NAME, body['name']], str(e), i, count)
      except GAPI.conditionNotMet as e:
        entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, orgUnitPath], str(e), i, count)
      except (GAPI.badRequest, GAPI.invalidCustomerId, GAPI.loginRequired):
        checkEntityAFDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, orgUnitPath)

# gam update orgs|ous <OrgUnitEntity> [name <String>] [description <String>] [parent <OrgUnitItem>] [inherit|(blockinheritance False)]
# gam update orgs|ous <OrgUnitEntity> add|move <CrosTypeEntity> [quickcrosmove [<Boolean>]]
# gam update orgs|ous <OrgUnitEntity> add|move <UserTypeEntity>
# gam update orgs|ous <OrgUnitEntity> sync <CrosTypeEntity> [removetoou <OrgUnitItem>] [quickcrosmove [<Boolean>]]
# gam update orgs|ous <OrgUnitEntity> sync <UserTypeEntity> [removetoou <OrgUnitItem>]
def doUpdateOrgs():
  _doUpdateOrgs(getEntityList(Cmd.OB_ORGUNIT_ENTITY, shlexSplit=True))

# gam update org|ou <OrgUnitItem> [name <String>] [description <String>]  [parent <OrgUnitItem>] [inherit|(blockinheritance False)]
# gam update org|ou <OrgUnitItem> add|move <CrosTypeEntity> [quickcrosmove [<Boolean>]]
# gam update org|ou <OrgUnitItem> add|move <UserTypeEntity>
# gam update org|ou <OrgUnitItem> sync <CrosTypeEntity> [removetoou <OrgUnitItem>] [quickcrosmove [<Boolean>]]
# gam update org|ou <OrgUnitItem> sync <UserTypeEntity> [removetoou <OrgUnitItem>]
def doUpdateOrg():
  _doUpdateOrgs([getOrgUnitItem()])

def _doDeleteOrgs(entityList):
  cd = buildGAPIObject(API.DIRECTORY)
  checkForExtraneousArguments()
  i = 0
  count = len(entityList)
  for orgUnitPath in entityList:
    i += 1
    try:
      orgUnitPath = makeOrgUnitPathAbsolute(orgUnitPath)
      callGAPI(cd.orgunits(), 'delete',
               throwReasons=[GAPI.CONDITION_NOT_MET, GAPI.INVALID_ORGUNIT, GAPI.ORGUNIT_NOT_FOUND, GAPI.BACKEND_ERROR,
                             GAPI.INVALID_CUSTOMER_ID, GAPI.SERVICE_NOT_AVAILABLE,
                             GAPI.BAD_REQUEST,  GAPI.LOGIN_REQUIRED],
               retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
               customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath=encodeOrgUnitPath(makeOrgUnitPathRelative(orgUnitPath)))
      entityActionPerformed([Ent.ORGANIZATIONAL_UNIT, orgUnitPath], i, count)
    except GAPI.conditionNotMet:
      entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, orgUnitPath], Msg.HAS_CHILD_ORGS.format(Ent.Plural(Ent.ORGANIZATIONAL_UNIT)), i, count)
    except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError):
      entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, orgUnitPath], Msg.DOES_NOT_EXIST, i, count)
    except (GAPI.invalidCustomerId, GAPI.serviceNotAvailable) as e:
### Check for my_customer
      entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, orgUnitPath, Ent.CUSTOMER_ID, GC.Values[GC.CUSTOMER_ID]], str(e), i, count)
    except (GAPI.badRequest, GAPI.loginRequired):
      checkEntityAFDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, orgUnitPath)

# gam delete orgs|ous <OrgUnitEntity>
def doDeleteOrgs():
  _doDeleteOrgs(getEntityList(Cmd.OB_ORGUNIT_ENTITY, shlexSplit=True))

# gam delete org|ou <OrgUnitItem>
def doDeleteOrg():
  _doDeleteOrgs([getOrgUnitItem()])

ORG_FIELD_INFO_ORDER = ['orgUnitId', 'name', 'description', 'parentOrgUnitPath', 'parentOrgUnitId', 'blockInheritance']
ORG_FIELDS_WITH_CRS_NLS = {'description'}

def _doInfoOrgs(entityList):
  cd = buildGAPIObject(API.DIRECTORY)
  getUsers = True
  isSuspended = None
  entityType = Ent.USER
  showChildren = False
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'nousers':
      getUsers = False
    elif myarg in SUSPENDED_ARGUMENTS:
      isSuspended = _getIsSuspended(myarg)
      entityType = Ent.USER_SUSPENDED if isSuspended else Ent.USER_NOT_SUSPENDED
    elif myarg in {'children', 'child'}:
      showChildren = True
    else:
      unknownArgumentExit()
  i = 0
  count = len(entityList)
  for origOrgUnitPath in entityList:
    i += 1
    try:
      if origOrgUnitPath == '/':
        _, orgUnitPath = getOrgUnitId(cd, origOrgUnitPath)
      else:
        orgUnitPath = makeOrgUnitPathRelative(origOrgUnitPath)
      result = callGAPI(cd.orgunits(), 'get',
                        throwReasons=GAPI.ORGUNIT_GET_THROW_REASONS,
                        customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath=encodeOrgUnitPath(orgUnitPath))
      if 'orgUnitPath' not in result:
        entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, origOrgUnitPath], Msg.DOES_NOT_EXIST, i, count)
        continue
      printEntity([Ent.ORGANIZATIONAL_UNIT, result['orgUnitPath']], i, count)
      Ind.Increment()
      for field in ORG_FIELD_INFO_ORDER:
        value = result.get(field, None)
        if value is not None:
          if field not in ORG_FIELDS_WITH_CRS_NLS:
            printKeyValueList([field, value])
          else:
            printKeyValueWithCRsNLs(field, value)
      if getUsers:
        orgUnitPath = result['orgUnitPath']
        users = callGAPIpages(cd.users(), 'list', 'users',
                              throwReasons=[GAPI.BAD_REQUEST, GAPI.INVALID_INPUT, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN],
                              retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                              customer=GC.Values[GC.CUSTOMER_ID], query=orgUnitPathQuery(orgUnitPath, isSuspended), orderBy='email',
                              fields='nextPageToken,users(primaryEmail,orgUnitPath)', maxResults=GC.Values[GC.USER_MAX_RESULTS])
        printEntitiesCount(entityType, None)
        usersInOU = 0
        Ind.Increment()
        orgUnitPath = orgUnitPath.lower()
        for user in users:
          if orgUnitPath == user['orgUnitPath'].lower():
            printKeyValueList([user['primaryEmail']])
            usersInOU += 1
          elif showChildren:
            printKeyValueList([f'{user["primaryEmail"]} (child)'])
            usersInOU += 1
        Ind.Decrement()
        printKeyValueList([Msg.TOTAL_ITEMS_IN_ENTITY.format(Ent.Plural(entityType), Ent.Singular(Ent.ORGANIZATIONAL_UNIT)), usersInOU])
      Ind.Decrement()
    except (GAPI.invalidInput, GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError):
      entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, orgUnitPath], Msg.DOES_NOT_EXIST, i, count)
    except (GAPI.badRequest, GAPI.invalidCustomerId, GAPI.loginRequired, GAPI.resourceNotFound, GAPI.forbidden):
      checkEntityAFDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, orgUnitPath)

# gam info orgs|ous <OrgUnitEntity> [nousers|notsuspended|suspended] [children|child]
def doInfoOrgs():
  _doInfoOrgs(getEntityList(Cmd.OB_ORGUNIT_ENTITY, shlexSplit=True))

# gam info org|ou <OrgUnitItem> [nousers|notsuspended|suspended] [children|child]
def doInfoOrg():
  _doInfoOrgs([getOrgUnitItem()])

ORG_ARGUMENT_TO_FIELD_MAP = {
  'blockinheritance': 'blockInheritance',
  'inheritanceblocked': 'blockInheritance',
  'inherit': 'blockInheritance',
  'description': 'description',
  'id': 'orgUnitId',
  'name': 'name',
  'orgunitid': 'orgUnitId',
  'orgunitpath': 'orgUnitPath',
  'path': 'orgUnitPath',
  'parentorgunitid': 'parentOrgUnitId',
  'parentid': 'parentOrgUnitId',
  'parentorgunitpath': 'parentOrgUnitPath',
  'parent': 'parentOrgUnitPath',
  }
ORG_FIELD_PRINT_ORDER = ['orgUnitPath', 'orgUnitId', 'name', 'description', 'parentOrgUnitPath', 'parentOrgUnitId', 'blockInheritance']
PRINT_ORGS_DEFAULT_FIELDS = ['orgUnitPath', 'orgUnitId', 'name', 'parentOrgUnitId']

ORG_UNIT_SELECTOR_FIELD = 'orgUnitSelector'
PRINT_OUS_SELECTOR_CHOICES = [
  Cmd.ENTITY_CROS_OU, Cmd.ENTITY_CROS_OU_AND_CHILDREN,
  Cmd.ENTITY_OU, Cmd.ENTITY_OU_NS, Cmd.ENTITY_OU_SUSP,
  Cmd.ENTITY_OU_AND_CHILDREN, Cmd.ENTITY_OU_AND_CHILDREN_NS, Cmd.ENTITY_OU_AND_CHILDREN_SUSP,
  ]

def _getOrgUnits(cd, orgUnitPath, fieldsList, listType, showParent, batchSubOrgs, childSelector=None, parentSelector=None):
  def _callbackListOrgUnits(request_id, response, exception):
    ri = request_id.splitlines()
    if exception is None:
      orgUnits.extend(response.get('organizationUnits', []))
    else:
      http_status, reason, message = checkGAPIError(exception)
      errMsg = getHTTPError({}, http_status, reason, message)
      if reason not in GAPI.DEFAULT_RETRY_REASONS:
        if reason in [GAPI.BAD_REQUEST, GAPI.INVALID_CUSTOMER_ID, GAPI.LOGIN_REQUIRED]:
          accessErrorExit(cd)
        entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, topLevelOrgUnits[int(ri[RI_I])]], errMsg)
        return
      waitOnFailure(1, 10, reason, message)
      try:
        response = callGAPI(cd.orgunits(), 'list',
                            throwReasons=GAPI.ORGUNIT_GET_THROW_REASONS,
                            customerId=GC.Values[GC.CUSTOMER_ID], type='all', orgUnitPath=topLevelOrgUnits[int(ri[RI_I])], fields=listfields)
        orgUnits.extend(response.get('organizationUnits', []))
      except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError):
        entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, topLevelOrgUnits[int(ri[RI_I])]], Msg.DOES_NOT_EXIST)
      except (GAPI.badRequest, GAPI.invalidCustomerId, GAPI.loginRequired):
        accessErrorExit(cd)

  def _batchListOrgUnits():
    svcargs = dict([('customerId', GC.Values[GC.CUSTOMER_ID]), ('orgUnitPath', None), ('type', 'all'), ('fields', listfields)]+GM.Globals[GM.EXTRA_ARGS_LIST])
    method = getattr(cd.orgunits(), 'list')
    dbatch = cd.new_batch_http_request(callback=_callbackListOrgUnits)
    bcount = 0
    i = 0
    for orgUnitPath in topLevelOrgUnits:
      svcparms = svcargs.copy()
      svcparms['orgUnitPath'] = orgUnitPath
      dbatch.add(method(**svcparms), request_id=batchRequestID('', i, 0, 0, 0, ''))
      bcount += 1
      i += 1
      if bcount >= GC.Values[GC.BATCH_SIZE]:
        executeBatch(dbatch)
        dbatch = cd.new_batch_http_request(callback=_callbackListOrgUnits)
        bcount = 0
    if bcount > 0:
      dbatch.execute()

  deleteOrgUnitId = deleteParentOrgUnitId = False
  if showParent:
    localFieldsList = fieldsList[:]
    if 'orgUnitId' not in fieldsList:
      localFieldsList.append('orgUnitId')
      deleteOrgUnitId = True
    if 'parentOrgUnitId' not in fieldsList:
      localFieldsList.append('parentOrgUnitId')
      deleteParentOrgUnitId = True
    fields = getFieldsFromFieldsList(localFieldsList)
  else:
    fields = getFieldsFromFieldsList(fieldsList)
  listfields = f'organizationUnits({fields})'
  if listType == 'all' and  orgUnitPath == '/':
    printGettingAllAccountEntities(Ent.ORGANIZATIONAL_UNIT)
  else:
    printGettingAllEntityItemsForWhom(Ent.CHILD_ORGANIZATIONAL_UNIT, orgUnitPath,
                                      qualifier=' (Direct Children)' if listType == 'children' else '', entityType=Ent.ORGANIZATIONAL_UNIT)
  if listType == 'children':
    batchSubOrgs = False
  try:
    orgs = callGAPI(cd.orgunits(), 'list',
                    throwReasons=GAPI.ORGUNIT_GET_THROW_REASONS,
                    customerId=GC.Values[GC.CUSTOMER_ID], type=listType if not batchSubOrgs else 'children',
                    orgUnitPath=orgUnitPath, fields=listfields)
  except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError):
    entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, orgUnitPath], Msg.DOES_NOT_EXIST)
    return None
  except (GAPI.badRequest, GAPI.invalidCustomerId, GAPI.loginRequired):
    accessErrorExit(cd)
  orgUnits = orgs.get('organizationUnits', [])
  topLevelOrgUnits = [orgUnit['orgUnitPath'] for orgUnit in orgUnits]
  if batchSubOrgs:
    _batchListOrgUnits()
  if showParent:
    parentOrgIds = []
    retrievedOrgIds = []
    if not orgUnits:
      topLevelOrgId = getTopLevelOrgId(cd, orgUnitPath)
      if topLevelOrgId:
        parentOrgIds.append(topLevelOrgId)
    for orgUnit in orgUnits:
      retrievedOrgIds.append(orgUnit['orgUnitId'])
      if orgUnit['parentOrgUnitId'] not in parentOrgIds:
        parentOrgIds.append(orgUnit['parentOrgUnitId'])
    missing_parents = set(parentOrgIds)-set(retrievedOrgIds)
    for missing_parent in missing_parents:
      try:
        result = callGAPI(cd.orgunits(), 'get',
                          throwReasons=GAPI.ORGUNIT_GET_THROW_REASONS,
                          customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath=missing_parent, fields=fields)
        orgUnits.append(result)
      except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.backendError,
              GAPI.badRequest, GAPI.invalidCustomerId, GAPI.loginRequired):
        pass
  if listType == 'all' and  orgUnitPath == '/':
    printGotAccountEntities(len(orgUnits))
  else:
    printGotEntityItemsForWhom(len(orgUnits))
  if childSelector is not None:
    for orgUnit in orgUnits:
      orgUnit[ORG_UNIT_SELECTOR_FIELD] = childSelector if orgUnit['orgUnitPath'] != orgUnitPath else parentSelector
  if deleteOrgUnitId or deleteParentOrgUnitId:
    for orgUnit in orgUnits:
      if deleteOrgUnitId:
        orgUnit.pop('orgUnitId', None)
      if deleteParentOrgUnitId:
        orgUnit.pop('parentOrgUnitId', None)
  return orgUnits

def getOrgUnitIdToPathMap(cd=None):
  if cd is None:
    cd = buildGAPIObject(API.DIRECTORY)
  orgUnits = _getOrgUnits(cd, '/', ['orgUnitPath', 'orgUnitId'], 'all', True, False)
  return {ou['orgUnitId']:ou['orgUnitPath'] for ou in orgUnits}

# gam print orgs|ous [todrive <ToDriveAttribute>*]
#	[fromparent <OrgUnitItem>] [showparent] [toplevelonly]
#	[parentselector <OrgUnitSelector> childselector <OrgUnitSelector>]
#	[allfields|<OrgUnitFieldName>*|(fields <OrgUnitFieldNameList>)] [convertcrnl] [batchsuborgs [<Boolean>]]
#	[mincroscount <Number>] [maxcroscount <Number>]
#	[minusercount <Number>] [maxusercount <Number>]
# 	[showitemcountonly]
def doPrintOrgs():
  cd = buildGAPIObject(API.DIRECTORY)
  convertCRNL = GC.Values[GC.CSV_OUTPUT_CONVERT_CR_NL]
  fieldsList = []
  csvPF = CSVPrintFile(sortTitles=ORG_FIELD_PRINT_ORDER)
  orgUnitPath = '/'
  listType = 'all'
  batchSubOrgs = showParent = False
  childSelector = parentSelector = None
  minCrOSCounts = maxCrOSCounts = minUserCounts = maxUserCounts = -1
  crosCounts = {}
  userCounts = {}
  showItemCountOnly = False
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif myarg == 'fromparent':
      orgUnitPath = getOrgUnitItem()
    elif myarg == 'showparent':
      showParent = getBoolean()
    elif myarg == 'parentselector':
      parentSelector = getChoice(PRINT_OUS_SELECTOR_CHOICES)
    elif myarg == 'childselector':
      childSelector = getChoice(PRINT_OUS_SELECTOR_CHOICES)
    elif myarg == 'mincroscount':
      minCrOSCounts = getInteger(minVal=-1)
    elif myarg == 'maxcroscount':
      maxCrOSCounts = getInteger(minVal=-1)
    elif myarg == 'minusercount':
      minUserCounts = getInteger(minVal=-1)
    elif myarg == 'maxusercount':
      maxUserCounts = getInteger(minVal=-1)
    elif myarg == 'batchsuborgs':
      batchSubOrgs = getBoolean()
    elif myarg == 'toplevelonly':
      listType = 'children'
    elif myarg == 'allfields':
      fieldsList = []
      csvPF.SetTitles(fieldsList)
      for field in ORG_FIELD_PRINT_ORDER:
        csvPF.AddField(field, ORG_ARGUMENT_TO_FIELD_MAP, fieldsList)
    elif myarg in ORG_ARGUMENT_TO_FIELD_MAP:
      if not fieldsList:
        csvPF.AddField('orgUnitPath', ORG_ARGUMENT_TO_FIELD_MAP, fieldsList)
      csvPF.AddField(myarg, ORG_ARGUMENT_TO_FIELD_MAP, fieldsList)
    elif myarg == 'fields':
      if not fieldsList:
        csvPF.AddField('orgUnitPath', ORG_ARGUMENT_TO_FIELD_MAP, fieldsList)
      for field in _getFieldsList():
        if field in ORG_ARGUMENT_TO_FIELD_MAP:
          csvPF.AddField(field, ORG_ARGUMENT_TO_FIELD_MAP, fieldsList)
        else:
          invalidChoiceExit(field, list(ORG_ARGUMENT_TO_FIELD_MAP), True)
    elif myarg in {'convertcrnl', 'converttextnl'}:
      convertCRNL = True
    elif myarg == 'showitemcountonly':
      showItemCountOnly = True
    else:
      unknownArgumentExit()
  if childSelector:
    if showParent and parentSelector is None:
      missingArgumentExit('parentselector')
    csvPF.AddTitle(ORG_UNIT_SELECTOR_FIELD)
    csvPF.AddSortTitle(ORG_UNIT_SELECTOR_FIELD)
  showCrOSCounts = (minCrOSCounts >= 0 or maxCrOSCounts >= 0)
  showUserCounts = (minUserCounts >= 0 or maxUserCounts >= 0)
  if not fieldsList:
    for field in PRINT_ORGS_DEFAULT_FIELDS:
      csvPF.AddField(field, ORG_ARGUMENT_TO_FIELD_MAP, fieldsList)
  orgUnits = _getOrgUnits(cd, orgUnitPath, fieldsList, listType, showParent, batchSubOrgs, childSelector, parentSelector)
  if showItemCountOnly:
    writeStdout(f'{0 if orgUnits is None else (len(orgUnits))}\n')
    return
  if orgUnits is None:
    return
  if showUserCounts:
    for orgUnit in orgUnits:
      userCounts[orgUnit['orgUnitPath']] = [0, 0]
    qualifier = Msg.IN_THE.format(Ent.Singular(Ent.ORGANIZATIONAL_UNIT))
    printGettingAllEntityItemsForWhom(Ent.USER, orgUnitPath, qualifier=qualifier, entityType=Ent.ORGANIZATIONAL_UNIT)
    pageMessage = getPageMessageForWhom()
    try:
      feed = yieldGAPIpages(cd.users(), 'list', 'users',
                            pageMessage=pageMessage,
                            throwReasons=[GAPI.INVALID_ORGUNIT, GAPI.ORGUNIT_NOT_FOUND,
                                          GAPI.INVALID_INPUT, GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN],
                            retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                            customer=GC.Values[GC.CUSTOMER_ID], query=orgUnitPathQuery(orgUnitPath, None), orderBy='email',
                            fields='nextPageToken,users(orgUnitPath,suspended)', maxResults=GC.Values[GC.USER_MAX_RESULTS])
      for users in feed:
        for user in users:
          if user['orgUnitPath'] in userCounts:
            userCounts[user['orgUnitPath']][user['suspended']] += 1
    except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.invalidInput, GAPI.badRequest, GAPI.backendError,
            GAPI.invalidCustomerId, GAPI.loginRequired, GAPI.resourceNotFound, GAPI.forbidden):
      checkEntityDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, orgUnitPath)
  for orgUnit in sorted(orgUnits, key=lambda k: k['orgUnitPath']):
    orgUnitPath = orgUnit['orgUnitPath']
    if showCrOSCounts:
      crosCounts[orgUnit['orgUnitPath']] = {}
      printGettingAllEntityItemsForWhom(Ent.CROS_DEVICE, orgUnitPath, entityType=Ent.ORGANIZATIONAL_UNIT)
      pageMessage = getPageMessageForWhom()
      pageToken = None
      totalItems = 0
      tokenRetries = 0
      while True:
        try:
          feed = callGAPI(cd.chromeosdevices(), 'list', 'chromeosdevices',
                          throwReasons=[GAPI.INVALID_INPUT, GAPI.INVALID_ORGUNIT, GAPI.ORGUNIT_NOT_FOUND,
                                        GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN],
                          retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                          pageToken=pageToken,
                          customerId=GC.Values[GC.CUSTOMER_ID], orgUnitPath=orgUnitPath, includeChildOrgunits=False,
                          fields='nextPageToken,chromeosdevices(status)', maxResults=GC.Values[GC.DEVICE_MAX_RESULTS])
          tokenRetries = 0
          pageToken, totalItems = _processGAPIpagesResult(feed, 'chromeosdevices', None, totalItems, pageMessage, None, Ent.CROS_DEVICE)
          if feed:
            for cros in feed.get('chromeosdevices', []):
              crosCounts[orgUnitPath].setdefault(cros['status'], 0)
              crosCounts[orgUnitPath][cros['status']] += 1
            del feed
          if not pageToken:
            _finalizeGAPIpagesResult(pageMessage)
            break
        except GAPI.invalidInput as e:
          message = str(e)
# Invalid Input: xyz - Check for invalid pageToken!!
# 0123456789012345
          if message[15:] == pageToken:
            tokenRetries += 1
            if tokenRetries <= 2:
              writeStderr(f'{WARNING_PREFIX}{Msg.LIST_CHROMEOS_INVALID_INPUT_PAGE_TOKEN_RETRY}')
              time.sleep(tokenRetries*5)
              continue
          entityActionFailedWarning([Ent.ORGANIZATIONAL_UNIT, orgUnitPath, Ent.CROS_DEVICE, None], message)
          break
        except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.badRequest, GAPI.backendError,
                GAPI.invalidCustomerId, GAPI.loginRequired, GAPI.resourceNotFound, GAPI.forbidden):
          checkEntityDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, orgUnitPath)
          break
    row = {}
    for field in fieldsList:
      if convertCRNL and field in ORG_FIELDS_WITH_CRS_NLS:
        row[field] = escapeCRsNLs(orgUnit.get(field, ''))
      else:
        row[field] = orgUnit.get(field, '')
    if childSelector:
      row[ORG_UNIT_SELECTOR_FIELD] = orgUnit[ORG_UNIT_SELECTOR_FIELD]
    if showCrOSCounts or showUserCounts:
      if showCrOSCounts:
        total = 0
        for k, v in sorted(crosCounts[orgUnitPath].items()):
          row[f'CrOS{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{k}'] = v
          total += v
        row[f'CrOS{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}Total'] = total
        if ((minCrOSCounts != -1 and total < minCrOSCounts) or
            (maxCrOSCounts != -1 and total > maxCrOSCounts)):
          continue
      if showUserCounts:
        row[f'Users{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}NotSuspended'] = userCounts[orgUnitPath][0]
        row[f'Users{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}Suspended'] = userCounts[orgUnitPath][1]
        row[f'Users{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}Total'] = total = userCounts[orgUnitPath][0]+userCounts[orgUnitPath][1]
        if ((minUserCounts != -1 and total < minUserCounts) or
            (maxUserCounts != -1 and total > maxUserCounts)):
          continue
      csvPF.WriteRowTitles(row)
    else:
      csvPF.WriteRow(row)
  csvPF.writeCSVfile('Orgs')

# gam show orgtree [fromparent <OrgUnitItem>] [batchsuborgs [Boolean>]]
def doShowOrgTree():
  def addOrgUnitToTree(orgPathList, i, n, tree):
    if orgPathList[i] not in tree:
      tree[orgPathList[i]] = {}
    if i < n:
      addOrgUnitToTree(orgPathList, i+1, n, tree[orgPathList[i]])

  def printOrgUnit(parentOrgUnit, tree):
    printKeyValueList([parentOrgUnit])
    Ind.Increment()
    for childOrgUnit in sorted(tree[parentOrgUnit]):
      printOrgUnit(childOrgUnit, tree[parentOrgUnit])
    Ind.Decrement()

  cd = buildGAPIObject(API.DIRECTORY)
  orgUnitPath = '/'
  fieldsList = ['orgUnitPath']
  listType = 'all'
  batchSubOrgs = False
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'fromparent':
      orgUnitPath = getOrgUnitItem()
    elif myarg == 'batchsuborgs':
      batchSubOrgs = getBoolean()
    else:
      unknownArgumentExit()
  orgUnits = _getOrgUnits(cd, orgUnitPath, fieldsList, listType, False, batchSubOrgs)
  if orgUnits is None:
    return
  orgTree = {}
  for orgUnit in orgUnits:
    orgPath = orgUnit['orgUnitPath'].split('/')
    addOrgUnitToTree(orgPath, 1, len(orgPath)-1, orgTree)
  for org in sorted(orgTree):
    printOrgUnit(org, orgTree)

ORG_ITEMS_FIELD_MAP = {
  'browsers': 'browsers',
  'devices': 'devices',
  'shareddrives': 'sharedDrives',
  'subous': 'subOus',
  'users': 'users',
  }

# gam check ou|org <OrgUnitItem> [todrive <ToDriveAttribute>*]
#	[<OrgUnitCheckName>*|(fields <OrgUnitCheckNameList>)]
#	[filename <FileName>] [movetoou <OrgUnitItem>]
#	[formatjson [quotechar <Character>]]
def doCheckOrgUnit():
  def writeCommandInfo(field):
    nonlocal commitBatch
    if commitBatch:
      f.write(f'{Cmd.COMMIT_BATCH_CMD}\n')
    else:
      commitBatch = True
    f.write(f'{Cmd.PRINT_CMD} Move {field} from {orgUnitPath} to {moveToOrgUnitPath}\n')

  cd = buildGAPIObject(API.DIRECTORY)
  csvPF = CSVPrintFile(['orgUnitPath', 'orgUnitId', 'empty'])
  FJQC = FormatJSONQuoteChar(csvPF)
  f = orgUnitPath = None
  fieldsList = []
  titlesList = []
  status, orgUnitPath, orgUnitId = checkOrgUnitPathExists(cd, getOrgUnitItem())
  if not status:
    entityDoesNotExistExit(Ent.ORGANIZATIONAL_UNIT, orgUnitPath)
  orgUnitPathLower = orgUnitPath.lower()
  fileName = 'CleanOuBatch.txt'
  moveToOrgUnitPath = moveToOrgUnitPathLower = None
  commitBatch = False
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if csvPF and myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif myarg in ORG_ITEMS_FIELD_MAP:
      fieldsList.append(myarg)
    elif myarg == 'fields':
      for field in _getFieldsList():
        if field in ORG_ITEMS_FIELD_MAP:
          fieldsList.append(field)
        else:
          invalidChoiceExit(field, list(ORG_ITEMS_FIELD_MAP), True)
    elif myarg == 'filename':
      fileName = setFilePath(getString(Cmd.OB_FILE_NAME))
    elif myarg == 'movetoou':
      movetoouLocation = Cmd.Location()
      status, moveToOrgUnitPath, _ = checkOrgUnitPathExists(cd, getOrgUnitItem())
      moveToOrgUnitPathLower = moveToOrgUnitPath.lower()
      if not status:
        entityDoesNotExistExit(Ent.ORGANIZATIONAL_UNIT, moveToOrgUnitPath)
    else:
      FJQC.GetFormatJSONQuoteChar(myarg, True)
  if not fieldsList:
    fieldsList = ORG_ITEMS_FIELD_MAP.keys()
  if moveToOrgUnitPath is not None:
    Cmd.SetLocation(movetoouLocation)
    if orgUnitPathLower == moveToOrgUnitPathLower:
      usageErrorExit(Msg.OU_AND_MOVETOOU_CANNOT_BE_IDENTICAL.format(orgUnitPath, moveToOrgUnitPath))
    if 'subous' in fieldsList and moveToOrgUnitPathLower.startswith(orgUnitPathLower):
      usageErrorExit(Msg.OU_SUBOUS_CANNOT_BE_MOVED_TO_MOVETOOU.format(orgUnitPath, moveToOrgUnitPath))
    fileName = setFilePath(fileName)
    f = openFile(fileName, DEFAULT_FILE_WRITE_MODE)
  orgUnitItemCounts = {}
  for field in sorted(fieldsList):
    title = ORG_ITEMS_FIELD_MAP[field]
    orgUnitItemCounts[title] = 0
    if not FJQC.formatJSON:
      titlesList.append(title)
  if 'browsers' in fieldsList:
    cbcm = buildGAPIObject(API.CBCM)
    customerId = _getCustomerIdNoC()
    printGettingAllEntityItemsForWhom(Ent.CHROME_BROWSER, orgUnitPath, entityType=Ent.ORGANIZATIONAL_UNIT)
    pageMessage = getPageMessage()
    try:
      feed = yieldGAPIpages(cbcm.chromebrowsers(), 'list', 'browsers',
                            pageMessage=pageMessage, messageAttribute='deviceId',
                            throwReasons=[GAPI.INVALID_INPUT, GAPI.BAD_REQUEST, GAPI.INVALID_ORGUNIT, GAPI.FORBIDDEN],
                            retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                            customer=customerId, orgUnitPath=orgUnitPath, projection='BASIC',
                            fields='nextPageToken,browsers(deviceId)')
      for browsers in feed:
        orgUnitItemCounts['browsers'] += len(browsers)
      if f is not None and orgUnitItemCounts['browsers'] > 0:
        writeCommandInfo('browsers')
        f.write(f'gam move browsers ou {moveToOrgUnitPath} browserou {orgUnitPath}\n')
    except (GAPI.invalidInput, GAPI.forbidden) as e:
      entityActionFailedWarning([Ent.CHROME_BROWSER, None], str(e))
    except GAPI.invalidOrgunit  as e:
      entityActionFailedExit([Ent.CHROME_BROWSER, None], str(e))
    except (GAPI.badRequest, GAPI.resourceNotFound):
      accessErrorExit(None)
  if 'devices' in fieldsList:
    printGettingAllEntityItemsForWhom(Ent.CROS_DEVICE, orgUnitPath, entityType=Ent.ORGANIZATIONAL_UNIT)
    pageMessage = getPageMessageForWhom()
    pageToken = None
    totalItems = 0
    tokenRetries = 0
    while True:
      try:
        feed = callGAPI(cd.chromeosdevices(), 'list',
                        throwReasons=[GAPI.INVALID_INPUT, GAPI.INVALID_ORGUNIT,
                                      GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN],
                        retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                        pageToken=pageToken, customerId=GC.Values[GC.CUSTOMER_ID],
                        orgUnitPath=orgUnitPath, fields='nextPageToken,chromeosdevices(deviceId)', maxResults=GC.Values[GC.DEVICE_MAX_RESULTS])
        tokenRetries = 0
        pageToken, totalItems = _processGAPIpagesResult(feed, 'chromeosdevices', None, totalItems, pageMessage, None, Ent.CROS_DEVICE)
        if feed:
          orgUnitItemCounts['devices'] += len(feed.get('chromeosdevices', []))
          del feed
        if not pageToken:
          _finalizeGAPIpagesResult(pageMessage)
          printGotAccountEntities(totalItems)
          if f is not None and orgUnitItemCounts['devices'] > 0:
            writeCommandInfo('devices')
            f.write(f'gam update  ou {moveToOrgUnitPath} add cros_ou {orgUnitPath}\n')
          break
      except GAPI.invalidInput as e:
        message = str(e)
# Invalid Input: xyz - Check for invalid pageToken!!
# 0123456789012345
        if message[15:] == pageToken:
          tokenRetries += 1
          if tokenRetries <= 2:
            writeStderr(f'{WARNING_PREFIX}{Msg.LIST_CHROMEOS_INVALID_INPUT_PAGE_TOKEN_RETRY}')
            time.sleep(tokenRetries*5)
            continue
          entityActionFailedWarning([Ent.CROS_DEVICE, None], message)
          break
        entityActionFailedWarning([Ent.CROS_DEVICE, None], message)
        break
      except GAPI.invalidOrgunit as e:
        entityActionFailedExit([Ent.CROS_DEVICE, None], str(e))
      except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden):
        accessErrorExit(cd)
  if 'shareddrives' in fieldsList:
    ci = buildGAPIObject(API.CLOUDIDENTITY_ORGUNITS_BETA)
    printGettingAllEntityItemsForWhom(Ent.SHAREDDRIVE, orgUnitPath, entityType=Ent.ORGANIZATIONAL_UNIT)
    sds = callGAPIpages(ci.orgUnits().memberships(), 'list', 'orgMemberships',
                        pageMessage=getPageMessageForWhom(),
                        parent=f'orgUnits/{orgUnitId[3:]}',
                        customer=_getCustomersCustomerIdWithC(),
                        filter="type == 'shared_drive'")
    orgUnitItemCounts['sharedDrives'] = len(sds)
    if f is not None and orgUnitItemCounts['sharedDrives'] > 0:
      writeCommandInfo('Shared Drives')
      for sd in sds:
        name = sd['name'].split(';')[1]
        f.write(f'gam update shareddrive {name} ou {moveToOrgUnitPath}\n')
  if 'subous' in fieldsList:
    subOus = _getOrgUnits(cd, orgUnitPath, ['orgUnitPath'], 'children', False, False, None, None)
    orgUnitItemCounts['subOus'] = len(subOus)
    if f is not None and orgUnitItemCounts['subOus'] > 0:
      writeCommandInfo('Sub OrgUnit')
      for ou in subOus:
        f.write(f'gam update ou {ou["orgUnitPath"]} parent {moveToOrgUnitPath}\n')
  if 'users' in fieldsList:
    printGettingAllEntityItemsForWhom(Ent.USER, orgUnitPath, entityType=Ent.ORGANIZATIONAL_UNIT)
    pageMessage = getPageMessageForWhom()
    try:
      feed = yieldGAPIpages(cd.users(), 'list', 'users',
                            pageMessage=pageMessage,
                            throwReasons=[GAPI.INVALID_ORGUNIT, GAPI.ORGUNIT_NOT_FOUND,
                                          GAPI.INVALID_INPUT, GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN],
                            retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                            customer=GC.Values[GC.CUSTOMER_ID], query=orgUnitPathQuery(orgUnitPath, None),
                            fields='nextPageToken,users(orgUnitPath)', maxResults=GC.Values[GC.USER_MAX_RESULTS])
      for users in feed:
        for user in users:
          if orgUnitPathLower == user.get('orgUnitPath', '').lower():
            orgUnitItemCounts['users'] += 1
      if f is not None and orgUnitItemCounts['users'] > 0:
        writeCommandInfo('users')
        f.write(f'gam update ou {moveToOrgUnitPath} add ou {orgUnitPath}\n')
    except (GAPI.invalidOrgunit, GAPI.orgunitNotFound, GAPI.invalidInput, GAPI.badRequest, GAPI.backendError,
            GAPI.invalidCustomerId, GAPI.loginRequired, GAPI.resourceNotFound, GAPI.forbidden):
      checkEntityDNEorAccessErrorExit(cd, Ent.ORGANIZATIONAL_UNIT, orgUnitPath)
  if f is not None:
    closeFile(f)
    writeStderr(Msg.GAM_BATCH_FILE_WRITTEN.format(fileName))
  empty = True
  for count in orgUnitItemCounts.values():
    if count > 0:
      empty = False
      break
  baseRow = {'orgUnitPath': orgUnitPath, 'orgUnitId': orgUnitId, 'empty': empty}
  row = flattenJSON(orgUnitItemCounts, baseRow.copy())
  if not FJQC.formatJSON:
    csvPF.WriteRowTitles(row)
  elif csvPF.CheckRowTitles(row):
    baseRow['JSON'] = json.dumps(cleanJSON(orgUnitItemCounts), ensure_ascii=False, sort_keys=True)
    csvPF.WriteRowNoFilter(baseRow)
  csvPF.writeCSVfile(f'OrgUnit {orgUnitPath} Item Counts')
  if not empty and GM.Globals[GM.SYSEXITRC] == 0:
    setSysExitRC(ORGUNIT_NOT_EMPTY_RC)

ALIAS_TARGET_TYPES = ['user', 'group', 'target']

# gam create aliases|nicknames <EmailAddressEntity> user|group|target <UniqueID>|<EmailAddress>
#	[verifynotinvitable]
# gam update aliases|nicknames <EmailAddressEntity> user|group|target <UniqueID>|<EmailAddress>
#	[notargetverify] [waitafterdelete <Integer>]
def doCreateUpdateAliases():
  def verifyAliasTargetExists():
    if targetType != 'group':
      try:
        callGAPI(cd.users(), 'get',
                 throwReasons=GAPI.USER_GET_THROW_REASONS,
                 userKey=targetEmail, fields='primaryEmail')
        return 'user'
      except (GAPI.userNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden,
              GAPI.badRequest, GAPI.backendError, GAPI.systemError):
        if targetType == 'user':
          return None
    try:
      callGAPI(cd.groups(), 'get',
               throwReasons=GAPI.GROUP_GET_THROW_REASONS, retryReasons=GAPI.GROUP_GET_RETRY_REASONS,
               groupKey=targetEmail, fields='email')
      return 'group'
    except (GAPI.groupNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden,
            GAPI.badRequest, GAPI.invalid, GAPI.systemError):
      return None

  def deleteAliasOnUpdate():
# User alias
    if targetType != 'group':
      try:
        callGAPI(cd.users().aliases(), 'delete',
                 throwReasons=[GAPI.USER_NOT_FOUND, GAPI.BAD_REQUEST, GAPI.INVALID, GAPI.FORBIDDEN, GAPI.INVALID_RESOURCE,
                               GAPI.CONDITION_NOT_MET],
                 userKey=aliasEmail, alias=aliasEmail)
        printEntityKVList([Ent.USER_ALIAS, aliasEmail], [Act.PerformedName(Act.DELETE)], i, count)
        time.sleep(waitAfterDelete)
        return True
      except GAPI.conditionNotMet as e:
        entityActionFailedWarning([Ent.USER_ALIAS, aliasEmail], str(e), i, count)
        return False
      except (GAPI.userNotFound, GAPI.badRequest, GAPI.invalid, GAPI.forbidden, GAPI.invalidResource):
        if targetType == 'user':
          entityUnknownWarning(Ent.USER_ALIAS, aliasEmail, i, count)
          return False
# Group alias
    try:
      callGAPI(cd.groups().aliases(), 'delete',
               throwReasons=[GAPI.GROUP_NOT_FOUND, GAPI.BAD_REQUEST, GAPI.INVALID, GAPI.FORBIDDEN, GAPI.INVALID_RESOURCE,
                             GAPI.CONDITION_NOT_MET],
               groupKey=aliasEmail, alias=aliasEmail)
      time.sleep(waitAfterDelete)
      return True
    except GAPI.conditionNotMet as e:
      entityActionFailedWarning([Ent.GROUP_ALIAS, aliasEmail], str(e), i, count)
      return False
    except GAPI.forbidden:
      entityUnknownWarning(Ent.GROUP_ALIAS, aliasEmail, i, count)
      return False
    except (GAPI.groupNotFound, GAPI.badRequest, GAPI.invalid, GAPI.invalidResource):
      entityUnknownWarning(Ent.GROUP_ALIAS, aliasEmail, i, count)
      return False

  cd = buildGAPIObject(API.DIRECTORY)
  ci = None
  updateCmd = Act.Get() == Act.UPDATE
  aliasList = getEntityList(Cmd.OB_EMAIL_ADDRESS_ENTITY)
  targetType = getChoice(ALIAS_TARGET_TYPES)
  targetEmails = getEntityList(Cmd.OB_GROUP_ENTITY)
  entityLists = targetEmails if isinstance(targetEmails, dict) else None
  verifyNotInvitable = False
  verifyTarget = updateCmd
  waitAfterDelete = 2
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if (not updateCmd) and myarg == 'verifynotinvitable':
      verifyNotInvitable = True
    elif updateCmd and myarg == 'notargetverify':
      verifyTarget = False
    elif updateCmd and myarg == 'waitafterdelete':
      waitAfterDelete = getInteger(minVal=2, maxVal=10)
    else:
      unknownArgumentExit()
  i = 0
  count = len(aliasList)
  for aliasEmail in aliasList:
    i += 1
    if entityLists:
      targetEmails = entityLists[aliasEmail]
    aliasEmail = normalizeEmailAddressOrUID(aliasEmail, noUid=True, noLower=True)
    if verifyNotInvitable:
      isInvitableUser, ci = _getIsInvitableUser(ci, aliasEmail)
      if isInvitableUser:
        entityActionNotPerformedWarning([Ent.ALIAS_EMAIL, aliasEmail], Msg.EMAIL_ADDRESS_IS_UNMANAGED_ACCOUNT)
        continue
    body = {'alias': aliasEmail}
    jcount = len(targetEmails)
    if jcount > 0:
# Only process first target
      targetEmail = normalizeEmailAddressOrUID(targetEmails[0])
      if verifyTarget:
        targetType = verifyAliasTargetExists()
        if targetType is None:
          entityUnknownWarning(Ent.ALIAS_TARGET, targetEmail, i, count)
          continue
      if updateCmd and not deleteAliasOnUpdate():
        continue
# User alias
      if targetType != 'group':
        try:
          callGAPI(cd.users().aliases(), 'insert',
                   throwReasons=[GAPI.USER_NOT_FOUND, GAPI.BAD_REQUEST,
                                 GAPI.INVALID, GAPI.INVALID_INPUT, GAPI.FORBIDDEN, GAPI.DUPLICATE,
                                 GAPI.CONDITION_NOT_MET, GAPI.LIMIT_EXCEEDED],
                   userKey=targetEmail, body=body, fields='')
          entityActionPerformed([Ent.USER_ALIAS, aliasEmail, Ent.USER, targetEmail], i, count)
          continue
        except (GAPI.conditionNotMet, GAPI.limitExceeded) as e:
          entityActionFailedWarning([Ent.USER_ALIAS, aliasEmail, Ent.USER, targetEmail], str(e), i, count)
          continue
        except GAPI.duplicate:
          duplicateAliasGroupUserWarning(cd, [Ent.USER_ALIAS, aliasEmail, Ent.USER, targetEmail], i, count)
          continue
        except (GAPI.invalid, GAPI.invalidInput):
          entityActionFailedWarning([Ent.USER_ALIAS, aliasEmail, Ent.USER, targetEmail], Msg.INVALID_ALIAS, i, count)
          continue
        except (GAPI.userNotFound, GAPI.badRequest, GAPI.forbidden):
          if targetType == 'user':
            entityUnknownWarning(Ent.ALIAS_TARGET, targetEmail, i, count)
            continue
# Group alias
      try:
        callGAPI(cd.groups().aliases(), 'insert',
                 throwReasons=[GAPI.GROUP_NOT_FOUND, GAPI.USER_NOT_FOUND, GAPI.BAD_REQUEST,
                               GAPI.INVALID, GAPI.INVALID_INPUT, GAPI.FORBIDDEN, GAPI.DUPLICATE,
                               GAPI.CONDITION_NOT_MET, GAPI.LIMIT_EXCEEDED],
                 groupKey=targetEmail, body=body, fields='')
        entityActionPerformed([Ent.GROUP_ALIAS, aliasEmail, Ent.GROUP, targetEmail], i, count)
      except (GAPI.conditionNotMet, GAPI.limitExceeded) as e:
        entityActionFailedWarning([Ent.GROUP_ALIAS, aliasEmail, Ent.GROUP, targetEmail], str(e), i, count)
      except GAPI.duplicate:
        duplicateAliasGroupUserWarning(cd, [Ent.GROUP_ALIAS, aliasEmail, Ent.GROUP, targetEmail], i, count)
      except (GAPI.invalid, GAPI.invalidInput):
        entityActionFailedWarning([Ent.GROUP_ALIAS, aliasEmail, Ent.GROUP, targetEmail], Msg.INVALID_ALIAS, i, count)
      except (GAPI.groupNotFound, GAPI.userNotFound, GAPI.badRequest, GAPI.forbidden):
        entityUnknownWarning(Ent.ALIAS_TARGET, targetEmail, i, count)

# gam delete aliases|nicknames [user|group|target] <EmailAddressEntity>
def doDeleteAliases():
  cd = buildGAPIObject(API.DIRECTORY)
  targetType = getChoice(ALIAS_TARGET_TYPES, defaultChoice='target')
  entityList = getEntityList(Cmd.OB_EMAIL_ADDRESS_ENTITY)
  checkForExtraneousArguments()
  i = 0
  count = len(entityList)
  for aliasEmail in entityList:
    i += 1
    aliasEmail = normalizeEmailAddressOrUID(aliasEmail, noUid=True)
    aliasDeleted = False
    if targetType != 'group':
      try:
        result = callGAPI(cd.users().aliases(), 'list',
                          throwReasons=[GAPI.USER_NOT_FOUND, GAPI.BAD_REQUEST, GAPI.INVALID, GAPI.FORBIDDEN, GAPI.INVALID_RESOURCE,
                                        GAPI.CONDITION_NOT_MET],
                          userKey=aliasEmail, fields='aliases(alias)')
        for aliasEntry in result.get('aliases', []):
          if aliasEmail == aliasEntry['alias'].lower():
            aliasEmail = aliasEntry['alias']
            callGAPI(cd.users().aliases(), 'delete',
                     throwReasons=[GAPI.USER_NOT_FOUND, GAPI.BAD_REQUEST, GAPI.INVALID, GAPI.FORBIDDEN, GAPI.INVALID_RESOURCE,
                                   GAPI.CONDITION_NOT_MET],
                     userKey=aliasEmail, alias=aliasEmail)
            entityActionPerformed([Ent.USER_ALIAS, aliasEmail], i, count)
            aliasDeleted = True
            break
        if aliasDeleted:
          continue
      except GAPI.conditionNotMet as e:
        entityActionFailedWarning([Ent.USER_ALIAS, aliasEmail], str(e), i, count)
        continue
      except (GAPI.userNotFound, GAPI.badRequest, GAPI.invalid, GAPI.forbidden, GAPI.invalidResource):
        pass
      if targetType == 'user':
        entityUnknownWarning(Ent.USER_ALIAS, aliasEmail, i, count)
        continue
    try:
      result = callGAPI(cd.groups().aliases(), 'list',
                        throwReasons=[GAPI.GROUP_NOT_FOUND, GAPI.USER_NOT_FOUND, GAPI.BAD_REQUEST, GAPI.INVALID, GAPI.FORBIDDEN, GAPI.INVALID_RESOURCE,
                                      GAPI.CONDITION_NOT_MET],
                        groupKey=aliasEmail, fields='aliases(alias)')
      for aliasEntry in result.get('aliases', []):
        if aliasEmail == aliasEntry['alias'].lower():
          aliasEmail = aliasEntry['alias']
          callGAPI(cd.groups().aliases(), 'delete',
                   throwReasons=[GAPI.GROUP_NOT_FOUND, GAPI.USER_NOT_FOUND, GAPI.BAD_REQUEST, GAPI.INVALID, GAPI.FORBIDDEN, GAPI.INVALID_RESOURCE,
                                 GAPI.CONDITION_NOT_MET],
                   groupKey=aliasEmail, alias=aliasEmail)
          entityActionPerformed([Ent.GROUP_ALIAS, aliasEmail], i, count)
          aliasDeleted = True
          break
      if aliasDeleted:
        continue
    except GAPI.conditionNotMet as e:
      entityActionFailedWarning([Ent.GROUP_ALIAS, aliasEmail], str(e), i, count)
      continue
    except (GAPI.groupNotFound, GAPI.userNotFound, GAPI.badRequest, GAPI.invalid, GAPI.forbidden, GAPI.invalidResource):
      pass
    if targetType == 'group':
      entityUnknownWarning(Ent.GROUP_ALIAS, aliasEmail, i, count)
      continue
    entityUnknownWarning(Ent.ALIAS, aliasEmail, i, count)

# gam remove aliases|nicknames <EmailAddress> user|group <EmailAddressEntity>
def doRemoveAliases():
  cd = buildGAPIObject(API.DIRECTORY)
  targetEmail = getEmailAddress()
  targetType = getChoice(['user', 'group'])
  entityList = getEntityList(Cmd.OB_EMAIL_ADDRESS_ENTITY)
  checkForExtraneousArguments()
  count = len(entityList)
  i = 0
  if targetType == 'user':
    try:
      for aliasEmail in entityList:
        i += 1
        aliasEmail = normalizeEmailAddressOrUID(aliasEmail, noUid=True)
        callGAPI(cd.users().aliases(), 'delete',
                 throwReasons=[GAPI.USER_NOT_FOUND, GAPI.BAD_REQUEST, GAPI.INVALID, GAPI.FORBIDDEN, GAPI.INVALID_RESOURCE,
                               GAPI.CONDITION_NOT_MET],
                 userKey=targetEmail, alias=aliasEmail)
        entityActionPerformed([Ent.USER, targetEmail, Ent.USER_ALIAS, aliasEmail], i, count)
    except (GAPI.userNotFound, GAPI.badRequest, GAPI.invalid, GAPI.forbidden, GAPI.conditionNotMet) as e:
      entityActionFailedWarning([Ent.USER, targetEmail, Ent.USER_ALIAS, aliasEmail], str(e), i, count)
    except GAPI.invalidResource:
      entityActionFailedWarning([Ent.USER, targetEmail, Ent.USER_ALIAS, aliasEmail], Msg.DOES_NOT_EXIST, i, count)
  else:
    try:
      for aliasEmail in entityList:
        i += 1
        aliasEmail = normalizeEmailAddressOrUID(aliasEmail, noUid=True)
        callGAPI(cd.groups().aliases(), 'delete',
                 throwReasons=[GAPI.GROUP_NOT_FOUND, GAPI.USER_NOT_FOUND, GAPI.BAD_REQUEST, GAPI.INVALID, GAPI.FORBIDDEN, GAPI.INVALID_RESOURCE,
                               GAPI.CONDITION_NOT_MET],
                 groupKey=targetEmail, alias=aliasEmail)
        entityActionPerformed([Ent.GROUP, targetEmail, Ent.GROUP_ALIAS, aliasEmail], i, count)
    except (GAPI.groupNotFound, GAPI.userNotFound, GAPI.badRequest, GAPI.invalid, GAPI.forbidden, GAPI.conditionNotMet) as e:
      entityActionFailedWarning([Ent.GROUP, targetEmail, Ent.GROUP_ALIAS, aliasEmail], str(e), i, count)
    except GAPI.invalidResource:
      entityActionFailedWarning([Ent.GROUP, targetEmail, Ent.GROUP_ALIAS, aliasEmail], Msg.DOES_NOT_EXIST, i, count)

def _addUserAliases(cd, user, aliasList, i, count):
  jcount = len(aliasList)
  entityPerformActionNumItems([Ent.USER, user], jcount, Ent.USER_ALIAS, i, count)
  Ind.Increment()
  j = 0
  for aliasEmail in aliasList:
    j += 1
    aliasEmail = normalizeEmailAddressOrUID(aliasEmail, noUid=True, noLower=True)
    body = {'alias': aliasEmail}
    try:
      callGAPI(cd.users().aliases(), 'insert',
               throwReasons=[GAPI.USER_NOT_FOUND, GAPI.BAD_REQUEST,
                             GAPI.INVALID, GAPI.INVALID_INPUT, GAPI.FORBIDDEN, GAPI.DUPLICATE,
                             GAPI.CONDITION_NOT_MET, GAPI.LIMIT_EXCEEDED],
               userKey=user, body=body, fields='')
      entityActionPerformed([Ent.USER, user, Ent.USER_ALIAS, aliasEmail], j, jcount)
    except (GAPI.conditionNotMet, GAPI.limitExceeded) as e:
      entityActionFailedWarning([Ent.USER, user, Ent.USER_ALIAS, aliasEmail], str(e), j, jcount)
    except GAPI.duplicate:
      duplicateAliasGroupUserWarning(cd, [Ent.USER, user, Ent.USER_ALIAS, aliasEmail], j, jcount)
    except (GAPI.invalid, GAPI.invalidInput):
      entityActionFailedWarning([Ent.USER, user, Ent.USER_ALIAS, aliasEmail], Msg.INVALID_ALIAS, j, jcount)
    except (GAPI.userNotFound, GAPI.badRequest, GAPI.forbidden):
      entityUnknownWarning(Ent.USER, user, i, count)
  Ind.Decrement()

# gam <UserTypeEntity> delete alias|aliases
def deleteUsersAliases(users):
  cd = buildGAPIObject(API.DIRECTORY)
  checkForExtraneousArguments()
  i, count, users = getEntityArgument(users)
  for user in users:
    i += 1
    user = normalizeEmailAddressOrUID(user)
    try:
      user_aliases = callGAPI(cd.users(), 'get',
                              throwReasons=GAPI.USER_GET_THROW_REASONS,
                              userKey=user, fields='id,primaryEmail,aliases')
      user_id = user_aliases['id']
      user_primary = user_aliases['primaryEmail']
      jcount = len(user_aliases['aliases']) if ('aliases' in user_aliases) else 0
      entityPerformActionNumItems([Ent.USER, user_primary], jcount, Ent.ALIAS, i, count)
      if jcount == 0:
        setSysExitRC(NO_ENTITIES_FOUND_RC)
        continue
      Ind.Increment()
      j = 0
      for an_alias in user_aliases['aliases']:
        j += 1
        try:
          callGAPI(cd.users().aliases(), 'delete',
                   throwReasons=[GAPI.RESOURCE_ID_NOT_FOUND],
                   userKey=user_id, alias=an_alias)
          entityActionPerformed([Ent.USER, user_primary, Ent.ALIAS, an_alias], j, jcount)
        except GAPI.resourceIdNotFound:
          entityActionFailedWarning([Ent.USER, user_primary, Ent.ALIAS, an_alias], Msg.DOES_NOT_EXIST, j, jcount)
      Ind.Decrement()
    except (GAPI.userNotFound, GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden,
            GAPI.badRequest, GAPI.backendError, GAPI.systemError):
      entityUnknownWarning(Ent.USER, user, i, count)

def infoAliases(entityList):

  def _showAliasInfo(uid, email, aliasEmail, entityType, aliasEntityType, i, count):
    if email.lower() != aliasEmail:
      printEntity([aliasEntityType, aliasEmail], i, count)
      Ind.Increment()
      printEntity([entityType, email])
      printEntity([Ent.UNIQUE_ID, uid])
      Ind.Decrement()
    else:
      setSysExitRC(ENTITY_IS_NOT_AN_ALIAS_RC)
      printEntityKVList([Ent.EMAIL, aliasEmail],
                        [f'Is a {Ent.Singular(entityType)}, not a {Ent.Singular(aliasEntityType)}'],
                        i, count)

  cd = buildGAPIObject(API.DIRECTORY)
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
# Ignore info group/user arguments that may have come from whatis
    if (myarg in INFO_GROUP_OPTIONS) or (myarg in INFO_USER_OPTIONS):
      if myarg == 'schemas':
        getString(Cmd.OB_SCHEMA_NAME_LIST)
    else:
      unknownArgumentExit()
  i = 0
  count = len(entityList)
  for aliasEmail in entityList:
    i += 1
    aliasEmail = normalizeEmailAddressOrUID(aliasEmail, noUid=True, noLower=True)
    try:
      result = callGAPI(cd.users(), 'get',
                        throwReasons=GAPI.USER_GET_THROW_REASONS,
                        userKey=aliasEmail, fields='id,primaryEmail')
      _showAliasInfo(result['id'], result['primaryEmail'], aliasEmail, Ent.USER_EMAIL, Ent.USER_ALIAS, i, count)
      continue
    except (GAPI.userNotFound, GAPI.badRequest):
      pass
    except (GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden,
            GAPI.backendError, GAPI.systemError):
      entityUnknownWarning(Ent.USER_ALIAS, aliasEmail, i, count)
      continue
    try:
      result = callGAPI(cd.groups(), 'get',
                        throwReasons=GAPI.GROUP_GET_THROW_REASONS,
                        groupKey=aliasEmail, fields='id,email')
      _showAliasInfo(result['id'], result['email'], aliasEmail, Ent.GROUP_EMAIL, Ent.GROUP_ALIAS, i, count)
      continue
    except GAPI.groupNotFound:
      pass
    except (GAPI.domainNotFound, GAPI.domainCannotUseApis, GAPI.forbidden, GAPI.badRequest):
      entityUnknownWarning(Ent.GROUP_ALIAS, aliasEmail, i, count)
      continue
    entityUnknownWarning(Ent.EMAIL, aliasEmail, i, count)

# gam info aliases|nicknames <EmailAddressEntity>
def doInfoAliases():
  infoAliases(getEntityList(Cmd.OB_EMAIL_ADDRESS_ENTITY))

def initUserGroupDomainQueryFilters():
  if not GC.Values[GC.PRINT_AGU_DOMAINS]:
    return {'list': [{'customer': GC.Values[GC.CUSTOMER_ID]}], 'queries': [None]}
  return {'list': [{'domain': domain.lower()} for domain in GC.Values[GC.PRINT_AGU_DOMAINS].replace(',', ' ').split()], 'queries': [None]}

def getUserGroupDomainQueryFilters(myarg, kwargsDict):
  if myarg in {'domain', 'domains'}:
    kwargsDict['list'] = [{'domain': domain.lower()} for domain in getEntityList(Cmd.OB_DOMAIN_NAME_ENTITY)]
  elif myarg in {'query', 'queries'}:
    kwargsDict['queries'] = getQueries(myarg)
  else:
    return False
  return True

def makeUserGroupDomainQueryFilters(kwargsDict):
  kwargsQueries = []
  for kwargs in kwargsDict['list']:
    for query in kwargsDict['queries']:
      kwargsQueries.append((kwargs, query))
  return kwargsQueries

def userFilters(kwargs, query, orgUnitPath, isSuspended):
  queryTitle = ''
  if kwargs.get('domain'):
    queryTitle += f'domain={kwargs["domain"]}, '
  if orgUnitPath is not None:
    if query is not None and query.find(orgUnitPath) == -1:
      query += f" orgUnitPath='{orgUnitPath}'"
    else:
      if query is None:
        query = ''
      else:
        query += ' '
      query += f"orgUnitPath='{orgUnitPath}'"
  if isSuspended is not None:
    if query is None:
      query = ''
    else:
      query += ' '
    query += f'isSuspended={isSuspended}'
  if query is not None:
    queryTitle += f'query="{query}", '
  if queryTitle:
    return query, queryTitle[:-2]
  return query, queryTitle

# gam print aliases|nicknames [todrive <ToDriveAttribute>*]
#	([domain|domains <DomainNameEntity>] [(query <QueryUser>)|(queries <QueryUserList>)]
#	 [limittoou <OrgUnitItem>])
#	[user|users <EmailAddressList>] [group|groups <EmailAddressList>]
#	[select <UserTypeEntity>]
#	[issuspended <Boolean>] [aliasmatchpattern <REMatchPattern>]
#	[shownoneditable] [nogroups] [nousers]
#	[onerowpertarget] [delimiter <Character>]
#	[suppressnoaliasrows]
#	(addcsvdata <FieldName> <String>)*
def doPrintAliases():
  def writeAliases(target, targetEmail, targetType):
    if not oneRowPerTarget:
      for alias in target.get('aliases', []):
        if aliasMatchPattern.match(alias):
          row = {'Alias': alias, 'Target': targetEmail, 'TargetType': targetType}
          if addCSVData:
            row.update(addCSVData)
          csvPF.WriteRow(row)
      if showNonEditable:
        for alias in target.get('nonEditableAliases', []):
          if aliasMatchPattern.match(alias):
            row = {'NonEditableAlias': alias, 'Target': targetEmail, 'TargetType': targetType}
            if addCSVData:
              row.update(addCSVData)
            csvPF.WriteRow(row)
    else:
      aliases = [alias for alias in target.get('aliases', []) if aliasMatchPattern.match(alias)]
      if showNonEditable:
        nealiases = [alias for alias in target.get('nonEditableAliases', []) if aliasMatchPattern.match(alias)]
      else:
        nealiases = []
      if suppressNoAliasRows and not aliases and not nealiases:
        return
      row = {'Target': targetEmail, 'TargetType': targetType, 'Aliases': delimiter.join(aliases)}
      if showNonEditable:
        row['NonEditableAliases'] = delimiter.join(nealiases)
      if addCSVData:
        row.update(addCSVData)
      csvPF.WriteRow(row)

  cd = buildGAPIObject(API.DIRECTORY)
  csvPF = CSVPrintFile()
  userFields = ['primaryEmail', 'aliases']
  groupFields = ['email', 'aliases']
  oneRowPerTarget = showNonEditable = suppressNoAliasRows = False
  kwargsDict = initUserGroupDomainQueryFilters()
  getGroups = getUsers = True
  groups = []
  users = []
  aliasMatchPattern = re.compile(r'^.*$')
  isSuspended = orgUnitPath = None
  addCSVData = {}
  delimiter = GC.Values[GC.CSV_OUTPUT_FIELD_DELIMITER]
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif myarg == 'shownoneditable':
      showNonEditable= True
      userFields.append('nonEditableAliases')
      groupFields.append('nonEditableAliases')
    elif myarg == 'nogroups':
      getGroups = False
    elif myarg == 'nousers':
      getUsers = False
    elif myarg == 'limittoou':
      orgUnitPath = getOrgUnitItem(pathOnly=True, cd=cd)
      orgUnitPathLower = orgUnitPath.lower()
      userFields.append('orgUnitPath')
      getGroups = False
    elif getUserGroupDomainQueryFilters(myarg, kwargsDict):
      pass
    elif myarg == 'select':
      _, users = getEntityToModify(defaultEntityType=Cmd.ENTITY_USERS)
    elif myarg == 'issuspended':
      isSuspended = getBoolean()
    elif myarg in {'user','users'}:
      users.extend(convertEntityToList(getString(Cmd.OB_EMAIL_ADDRESS_LIST, minLen=0)))
    elif myarg in {'group', 'groups'}:
      groups.extend(convertEntityToList(getString(Cmd.OB_EMAIL_ADDRESS_LIST, minLen=0)))
    elif myarg == 'aliasmatchpattern':
      aliasMatchPattern = getREPattern(re.IGNORECASE)
    elif myarg == 'onerowpertarget':
      oneRowPerTarget = True
    elif myarg == 'suppressnoaliasrows':
      suppressNoAliasRows = True
    elif myarg == 'addcsvdata':
      k = getString(Cmd.OB_STRING)
      addCSVData[k] = getString(Cmd.OB_STRING, minLen=0)
    elif myarg == 'delimiter':
      delimiter = getCharacter()
    else:
      unknownArgumentExit()
  if (users or groups) and kwargsDict['queries'][0] is None:
    getUsers = getGroups = False
  if not oneRowPerTarget:
    titlesList = ['Alias', 'Target', 'TargetType']
    if showNonEditable:
      titlesList.insert(1, 'NonEditableAlias')
  else:
    titlesList = ['Target', 'TargetType', 'Aliases']
    if showNonEditable:
      titlesList.append('NonEditableAliases')
  csvPF.SetTitles(titlesList)
  if addCSVData:
    csvPF.AddTitles(sorted(addCSVData.keys()))
  if getUsers:
    for kwargsQuery in makeUserGroupDomainQueryFilters(kwargsDict):
      kwargs = kwargsQuery[0]
      query = kwargsQuery[1]
      query, pquery = userFilters(kwargs, query, orgUnitPath, isSuspended)
      printGettingAllAccountEntities(Ent.USER, pquery)
      try:
        entityList = callGAPIpages(cd.users(), 'list', 'users',
                                   pageMessage=getPageMessage(showFirstLastItems=True), messageAttribute='primaryEmail',
                                   throwReasons=[GAPI.INVALID_ORGUNIT, GAPI.INVALID_INPUT, GAPI.DOMAIN_NOT_FOUND,
                                                 GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN, GAPI.BAD_REQUEST,
                                                 GAPI.UNKNOWN_ERROR, GAPI.FAILED_PRECONDITION],
                                   retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS+[GAPI.UNKNOWN_ERROR, GAPI.FAILED_PRECONDITION],
                                   query=query, orderBy='email',
                                   fields=f'nextPageToken,users({",".join(userFields)})',
                                   maxResults=GC.Values[GC.USER_MAX_RESULTS], **kwargs)
        for user in entityList:
          if orgUnitPath is None or orgUnitPathLower == user.get('orgUnitPath', '').lower():
            writeAliases(user, user['primaryEmail'], 'User')
      except (GAPI.invalidOrgunit, GAPI.invalidInput):
        entityActionFailedWarning([Ent.ALIAS, None], invalidQuery(query))
        continue
      except GAPI.domainNotFound as e:
        entityActionFailedWarning([Ent.ALIAS, None, Ent.DOMAIN, kwargs['domain']], str(e))
        continue
      except (GAPI.unknownError, GAPI.failedPrecondition) as e:
        entityActionFailedExit([Ent.USER, None], str(e))
      except (GAPI.resourceNotFound, GAPI.forbidden, GAPI.badRequest):
        accessErrorExit(cd)
  count = len(users)
  i = 0
  for user in users:
    i += 1
    user = normalizeEmailAddressOrUID(user)
    printGettingEntityItemForWhom(Ent.USER_ALIAS, user, i, count)
    try:
      result = callGAPI(cd.users().aliases(), 'list',
                        throwReasons=[GAPI.USER_NOT_FOUND, GAPI.BAD_REQUEST, GAPI.INVALID, GAPI.FORBIDDEN, GAPI.INVALID_RESOURCE,
                                      GAPI.CONDITION_NOT_MET],
                        userKey=user, fields='aliases(alias)')
      aliases = {'aliases': [alias['alias'] for alias in result.get('aliases', [])]}
      writeAliases(aliases, user, 'User')
    except (GAPI.userNotFound, GAPI.badRequest, GAPI.invalid, GAPI.forbidden, GAPI.invalidResource, GAPI.conditionNotMet) as e:
      entityActionFailedWarning([Ent.USER, user], str(e), i, count)
  if getGroups:
    for kwargsQuery in makeUserGroupDomainQueryFilters(kwargsDict):
      kwargs = kwargsQuery[0]
      query = kwargsQuery[1]
      query, pquery = groupFilters(kwargs, query)
      printGettingAllAccountEntities(Ent.GROUP, pquery)
      try:
        entityList = callGAPIpages(cd.groups(), 'list', 'groups',
                                   pageMessage=getPageMessage(showFirstLastItems=True), messageAttribute='email',
                                   throwReasons=GAPI.GROUP_LIST_USERKEY_THROW_REASONS,
                                   retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                                   query=query, orderBy='email',
                                   fields=f'nextPageToken,groups({",".join(groupFields)})', **kwargs)
        for group in entityList:
          writeAliases(group, group['email'], 'Group')
      except (GAPI.invalidMember, GAPI.invalidInput) as e:
        if not invalidMember(query):
          entityActionFailedExit([Ent.GROUP, None], str(e))
      except GAPI.domainNotFound as e:
        entityActionFailedWarning([Ent.ALIAS, None, Ent.DOMAIN, kwargs['domain']], str(e))
        continue
      except (GAPI.resourceNotFound, GAPI.forbidden, GAPI.badRequest):
        accessErrorExit(cd)
  count = len(groups)
  i = 0
  for group in groups:
    i += 1
    group = normalizeEmailAddressOrUID(group)
    printGettingEntityItemForWhom(Ent.GROUP_ALIAS, group, i, count)
    try:
      result = callGAPI(cd.groups().aliases(), 'list',
                        throwReasons=[GAPI.GROUP_NOT_FOUND, GAPI.BAD_REQUEST, GAPI.INVALID, GAPI.FORBIDDEN, GAPI.INVALID_RESOURCE,
                                      GAPI.CONDITION_NOT_MET],
                        groupKey=group, fields='aliases(alias)')
      aliases = {'aliases': [alias['alias'] for alias in result.get('aliases', [])]}
      writeAliases(aliases, group, 'Group')
    except (GAPI.groupNotFound, GAPI.badRequest, GAPI.invalid, GAPI.forbidden, GAPI.invalidResource, GAPI.conditionNotMet) as e:
      entityActionFailedWarning([Ent.GROUP, group], str(e), i, count)
  csvPF.writeCSVfile('Aliases')

# gam print addresses [todrive <ToDriveAttribute>*]
#	[domain <DomainName>]
def doPrintAddresses():
  cd = buildGAPIObject(API.DIRECTORY)
  kwargs = {'customer': GC.Values[GC.CUSTOMER_ID]}
  csvPF = CSVPrintFile()
  titlesList = ['Type', 'Email', 'Target']
  userFields = ['primaryEmail', 'aliases', 'nonEditableAliases', 'suspended']
  groupFields = ['email', 'aliases', 'nonEditableAliases']
  domainFields = ['domainName', 'isPrimary', 'domainAliases']
  resourceFields = ['resourceEmail']
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif myarg == 'domain':
      kwargs['domain'] = getString(Cmd.OB_DOMAIN_NAME).lower()
      kwargs.pop('customer', None)
    else:
      unknownArgumentExit()
  csvPF.SetTitles(titlesList)
  printGettingAllAccountEntities(Ent.USER)
  try:
    entityList = callGAPIpages(cd.users(), 'list', 'users',
                               pageMessage=getPageMessage(showFirstLastItems=True), messageAttribute='primaryEmail',
                               throwReasons=[GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN, GAPI.BAD_REQUEST,
                                             GAPI.UNKNOWN_ERROR, GAPI.FAILED_PRECONDITION],
                               retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS+[GAPI.UNKNOWN_ERROR, GAPI.FAILED_PRECONDITION],
                               orderBy='email', fields=f'nextPageToken,users({",".join(userFields)})',
                               maxResults=GC.Values[GC.USER_MAX_RESULTS], **kwargs)
  except (GAPI.unknownError, GAPI.failedPrecondition) as e:
    entityActionFailedExit([Ent.USER, None], str(e))
  except (GAPI.resourceNotFound, GAPI.forbidden, GAPI.badRequest):
    accessErrorExit(cd)
  for user in entityList:
    userEmail = user['primaryEmail']
    prefix = '' if not user['suspended'] else 'Suspended'
    csvPF.WriteRow({'Type': f'{prefix}User', 'Email': userEmail})
    for alias in user.get('aliases', []):
      csvPF.WriteRow({'Type': f'{prefix}UserAlias', 'Email': alias, 'Target': userEmail})
    for alias in user.get('nonEditableAliases', []):
      csvPF.WriteRow({'Type': f'{prefix}UserNEAlias', 'Email': alias, 'Target': userEmail})
  printGettingAllAccountEntities(Ent.GROUP)
  try:
    entityList = callGAPIpages(cd.groups(), 'list', 'groups',
                               pageMessage=getPageMessage(showFirstLastItems=True), messageAttribute='email',
                               throwReasons=GAPI.GROUP_LIST_THROW_REASONS,
                               retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                               orderBy='email', fields=f'nextPageToken,groups({",".join(groupFields)})', **kwargs)
  except (GAPI.resourceNotFound, GAPI.domainNotFound, GAPI.forbidden, GAPI.badRequest):
    accessErrorExit(cd)
  for group in entityList:
    groupEmail = group['email']
    csvPF.WriteRow({'Type': 'Group', 'Email': groupEmail})
    for alias in group.get('aliases', []):
      csvPF.WriteRow({'Type': 'GroupAlias', 'Email': alias, 'Target': groupEmail})
    for alias in group.get('nonEditableAliases', []):
      csvPF.WriteRow({'Type': 'GroupNEAlias', 'Email': alias, 'Target': groupEmail})
  printGettingAllAccountEntities(Ent.RESOURCE_CALENDAR)
  try:
    entityList = callGAPIpages(cd.resources().calendars(), 'list', 'items',
                               pageMessage=getPageMessage(showFirstLastItems=True), messageAttribute='resourceEmail',
                               throwReasons=[GAPI.BAD_REQUEST, GAPI.RESOURCE_NOT_FOUND, GAPI.FORBIDDEN, GAPI.INVALID_INPUT],
                               customer=GC.Values[GC.CUSTOMER_ID], fields=f'nextPageToken,items({",".join(resourceFields)})')
  except (GAPI.badRequest, GAPI.resourceNotFound, GAPI.forbidden):
    accessErrorExit(cd)
  except GAPI.invalidInput as e:
    entityActionFailedWarning([Ent.RESOURCE_CALENDAR, ''], str(e))
    return
  for resource in entityList:
    csvPF.WriteRow({'Type': 'Resource', 'Email': resource['resourceEmail']})
  try:
    entityList = callGAPIitems(cd.domains(), 'list', 'domains',
                               throwReasons=[GAPI.BAD_REQUEST, GAPI.NOT_FOUND, GAPI.FORBIDDEN],
                               customer=GC.Values[GC.CUSTOMER_ID], fields=f'domains({",".join(domainFields)})')
  except (GAPI.badRequest, GAPI.notFound, GAPI.forbidden):
    accessErrorExit(cd)
  for domain in entityList:
    domainEmail = domain['domainName']
    csvPF.WriteRow({'Type': 'DomainPrimary' if domain['isPrimary'] else 'DomainSecondary', 'Email': domainEmail})
    for alias in domain.get('domainAliases', []):
      csvPF.WriteRow({'Type': 'DomainAlias', 'Email': alias['domainAliasName'], 'Target': domainEmail})
  csvPF.SortRowsTwoTitles('Type', 'Email', False)
  csvPF.writeCSVfile('Addresses')

# Contact commands utilities
#
def _getCreateContactReturnOptions(parameters):
  myarg = getArgument()
  if myarg == 'returnidonly':
    parameters['returnIdOnly'] = True
  elif myarg == 'csv':
    parameters['csvPF'] = CSVPrintFile(parameters['titles'], 'sortall')
  elif parameters['csvPF'] and myarg == 'todrive':
    parameters['csvPF'].GetTodriveParameters()
  elif parameters['csvPF'] and myarg == 'addcsvdata':
    k = getString(Cmd.OB_STRING)
    parameters['addCSVData'][k] = getString(Cmd.OB_STRING, minLen=0)
  else:
    return False
  return True
#
CONTACT_JSON = 'JSON'

CONTACT_ID = 'ContactID'
CONTACT_UPDATED = 'Updated'
CONTACT_NAME_PREFIX = 'Name Prefix'
CONTACT_GIVEN_NAME = 'Given Name'
CONTACT_ADDITIONAL_NAME = 'Additional Name'
CONTACT_FAMILY_NAME = 'Family Name'
CONTACT_NAME_SUFFIX = 'Name Suffix'
CONTACT_NAME = 'Name'
CONTACT_NICKNAME = 'Nickname'
CONTACT_MAIDENNAME = 'Maiden Name'
CONTACT_SHORTNAME = 'Short Name'
CONTACT_INITIALS = 'Initials'
CONTACT_BIRTHDAY = 'Birthday'
CONTACT_GENDER = 'Gender'
CONTACT_LOCATION = 'Location'
CONTACT_PRIORITY = 'Priority'
CONTACT_SENSITIVITY = 'Sensitivity'
CONTACT_SUBJECT = 'Subject'
CONTACT_LANGUAGE = 'Language'
CONTACT_NOTES = 'Notes'
CONTACT_OCCUPATION = 'Occupation'
CONTACT_BILLING_INFORMATION = 'Billing Information'
CONTACT_MILEAGE = 'Mileage'
CONTACT_DIRECTORY_SERVER = 'Directory Server'
CONTACT_ADDRESSES = 'Addresses'
CONTACT_CALENDARS = 'Calendars'
CONTACT_EMAILS = 'Emails'
CONTACT_EXTERNALIDS = 'External IDs'
CONTACT_EVENTS = 'Events'
CONTACT_HOBBIES = 'Hobbies'
CONTACT_IMS = 'IMs'
CONTACT_JOTS = 'Jots'
CONTACT_ORGANIZATIONS = 'Organizations'
CONTACT_PHONES = 'Phones'
CONTACT_RELATIONS = 'Relations'
CONTACT_USER_DEFINED_FIELDS = 'User Defined Fields'
CONTACT_WEBSITES = 'Websites'
#
class ContactsManager():

  CONTACT_ARGUMENT_TO_PROPERTY_MAP = {
    'json': CONTACT_JSON,
    'name': CONTACT_NAME,
    'prefix': CONTACT_NAME_PREFIX,
    'givenname': CONTACT_GIVEN_NAME,
    'additionalname': CONTACT_ADDITIONAL_NAME,
    'familyname': CONTACT_FAMILY_NAME,
    'firstname': CONTACT_GIVEN_NAME,
    'middlename': CONTACT_ADDITIONAL_NAME,
    'lastname': CONTACT_FAMILY_NAME,
    'suffix': CONTACT_NAME_SUFFIX,
    'nickname': CONTACT_NICKNAME,
    'maidenname': CONTACT_MAIDENNAME,
    'shortname': CONTACT_SHORTNAME,
    'initials': CONTACT_INITIALS,
    'birthday': CONTACT_BIRTHDAY,
    'gender': CONTACT_GENDER,
    'location': CONTACT_LOCATION,
    'priority': CONTACT_PRIORITY,
    'sensitivity': CONTACT_SENSITIVITY,
    'subject': CONTACT_SUBJECT,
    'language': CONTACT_LANGUAGE,
    'note': CONTACT_NOTES,
    'notes': CONTACT_NOTES,
    'occupation': CONTACT_OCCUPATION,
    'billinginfo': CONTACT_BILLING_INFORMATION,
    'mileage': CONTACT_MILEAGE,
    'directoryserver': CONTACT_DIRECTORY_SERVER,
    'address': CONTACT_ADDRESSES,
    'addresses': CONTACT_ADDRESSES,
    'calendar': CONTACT_CALENDARS,
    'calendars': CONTACT_CALENDARS,
    'email': CONTACT_EMAILS,
    'emails': CONTACT_EMAILS,
    'externalid': CONTACT_EXTERNALIDS,
    'externalids': CONTACT_EXTERNALIDS,
    'event': CONTACT_EVENTS,
    'events': CONTACT_EVENTS,
    'hobby': CONTACT_HOBBIES,
    'hobbies': CONTACT_HOBBIES,
    'im': CONTACT_IMS,
    'ims': CONTACT_IMS,
    'jot': CONTACT_JOTS,
    'jots': CONTACT_JOTS,
    'organization': CONTACT_ORGANIZATIONS,
    'organizations': CONTACT_ORGANIZATIONS,
    'organisation': CONTACT_ORGANIZATIONS,
    'organisations': CONTACT_ORGANIZATIONS,
    'phone': CONTACT_PHONES,
    'phones': CONTACT_PHONES,
    'relation': CONTACT_RELATIONS,
    'relations': CONTACT_RELATIONS,
    'userdefinedfield': CONTACT_USER_DEFINED_FIELDS,
    'userdefinedfields': CONTACT_USER_DEFINED_FIELDS,
    'website': CONTACT_WEBSITES,
    'websites': CONTACT_WEBSITES,
    'updated': CONTACT_UPDATED,
    }

  GENDER_CHOICE_MAP = {'male': 'male', 'female': 'female'}

  PRIORITY_CHOICE_MAP = {'low': 'low', 'normal': 'normal', 'high': 'high'}

  SENSITIVITY_CHOICE_MAP = {
    'confidential': 'confidential',
    'normal': 'normal',
    'personal': 'personal',
    'private': 'private',
    }

  CONTACT_NAME_FIELDS = (
    CONTACT_NAME_PREFIX,
    CONTACT_GIVEN_NAME,
    CONTACT_ADDITIONAL_NAME,
    CONTACT_FAMILY_NAME,
    CONTACT_NAME_SUFFIX,
    )

  ADDRESS_TYPE_ARGUMENT_TO_REL = {
    'work': gdata.apps.contacts.REL_WORK,
    'home': gdata.apps.contacts.REL_HOME,
    'other': gdata.apps.contacts.REL_OTHER,
    }

  ADDRESS_REL_TO_TYPE_ARGUMENT = {
    gdata.apps.contacts.REL_WORK: 'work',
    gdata.apps.contacts.REL_HOME: 'home',
    gdata.apps.contacts.REL_OTHER: 'other',
    }

  ADDRESS_ARGUMENT_TO_FIELD_MAP = {
    'streetaddress': 'street',
    'pobox': 'pobox',
    'neighborhood': 'neighborhood',
    'locality': 'city',
    'region': 'region',
    'postalcode': 'postcode',
    'country': 'country',
    'formatted': 'value', 'unstructured': 'value',
    }

  ADDRESS_FIELD_TO_ARGUMENT_MAP = {
    'street': 'streetaddress',
    'pobox': 'pobox',
    'neighborhood': 'neighborhood',
    'city': 'locality',
    'region': 'region',
    'postcode': 'postalcode',
    'country': 'country',
    }

  ADDRESS_FIELD_PRINT_ORDER = [
    'street',
    'pobox',
    'neighborhood',
    'city',
    'region',
    'postcode',
    'country',
    ]

  CALENDAR_TYPE_ARGUMENT_TO_REL = {
    'work': 'work',
    'home': 'home',
    'free-busy': 'free-busy',
    }

  CALENDAR_REL_TO_TYPE_ARGUMENT = {
    'work': 'work',
    'home': 'home',
    'free-busy': 'free-busy',
    }

  EMAIL_TYPE_ARGUMENT_TO_REL = {
    'work': gdata.apps.contacts.REL_WORK,
    'home': gdata.apps.contacts.REL_HOME,
    'other': gdata.apps.contacts.REL_OTHER,
    }

  EMAIL_REL_TO_TYPE_ARGUMENT = {
    gdata.apps.contacts.REL_WORK: 'work',
    gdata.apps.contacts.REL_HOME: 'home',
    gdata.apps.contacts.REL_OTHER: 'other',
    }

  EVENT_TYPE_ARGUMENT_TO_REL = {
    'anniversary': 'anniversary',
    'other': 'other',
    }

  EVENT_REL_TO_TYPE_ARGUMENT = {
    'anniversary': 'anniversary',
    'other': 'other',
    }

  EXTERNALID_TYPE_ARGUMENT_TO_REL = {
    'account': 'account',
    'customer': 'customer',
    'network': 'network',
    'organization': 'organization',
    'organisation': 'organization',
    }

  EXTERNALID_REL_TO_TYPE_ARGUMENT = {
    'account': 'account',
    'customer': 'customer',
    'network': 'network',
    'organization': 'organization',
    'organisation': 'organization',
    }

  IM_TYPE_ARGUMENT_TO_REL = {
    'work': gdata.apps.contacts.REL_WORK,
    'home': gdata.apps.contacts.REL_HOME,
    'other': gdata.apps.contacts.REL_OTHER,
    }

  IM_REL_TO_TYPE_ARGUMENT = {
    gdata.apps.contacts.REL_WORK: 'work',
    gdata.apps.contacts.REL_HOME: 'home',
    gdata.apps.contacts.REL_OTHER: 'other',
    }

  IM_PROTOCOL_TO_REL_MAP = {
    'aim': gdata.apps.contacts.IM_AIM,
    'gtalk': gdata.apps.contacts.IM_GOOGLE_TALK,
    'icq': gdata.apps.contacts.IM_ICQ,
    'jabber': gdata.apps.contacts.IM_JABBER,
    'msn': gdata.apps.contacts.IM_MSN,
    'netmeeting': gdata.apps.contacts.IM_NETMEETING,
    'qq': gdata.apps.contacts.IM_QQ,
    'skype': gdata.apps.contacts.IM_SKYPE,
    'xmpp': gdata.apps.contacts.IM_JABBER,
    'yahoo': gdata.apps.contacts.IM_YAHOO,
    }

  IM_REL_TO_PROTOCOL_MAP = {
    gdata.apps.contacts.IM_AIM: 'aim',
    gdata.apps.contacts.IM_GOOGLE_TALK: 'gtalk',
    gdata.apps.contacts.IM_ICQ: 'icq',
    gdata.apps.contacts.IM_JABBER: 'jabber',
    gdata.apps.contacts.IM_MSN: 'msn',
    gdata.apps.contacts.IM_NETMEETING: 'netmeeting',
    gdata.apps.contacts.IM_QQ: 'qq',
    gdata.apps.contacts.IM_SKYPE: 'skype',
    gdata.apps.contacts.IM_YAHOO: 'yahoo',
    }

  JOT_TYPE_ARGUMENT_TO_REL = {
    'work': 'work',
    'home': 'home',
    'other': 'other',
    'keywords': 'keywords',
    'user': 'user',
    }

  JOT_REL_TO_TYPE_ARGUMENT = {
    'work': 'work',
    'home': 'home',
    'other': 'other',
    'keywords': 'keywords',
    'user': 'user',
    }

  ORGANIZATION_TYPE_ARGUMENT_TO_REL = {
    'work': gdata.apps.contacts.REL_WORK,
    'other': gdata.apps.contacts.REL_OTHER,
    }

  ORGANIZATION_REL_TO_TYPE_ARGUMENT = {
    gdata.apps.contacts.REL_WORK: 'work',
    gdata.apps.contacts.REL_OTHER: 'other',
    }

  ORGANIZATION_ARGUMENT_TO_FIELD_MAP = {
    'location': 'where',
    'department': 'department',
    'title': 'title',
    'jobdescription': 'jobdescription',
    'symbol': 'symbol',
    }

  ORGANIZATION_FIELD_TO_ARGUMENT_MAP = {
    'where': 'location',
    'department': 'department',
    'title': 'title',
    'jobdescription': 'jobdescription',
    'symbol': 'symbol',
    }

  ORGANIZATION_FIELD_PRINT_ORDER = [
    'where',
    'department',
    'title',
    'jobdescription',
    'symbol',
    ]

  PHONE_TYPE_ARGUMENT_TO_REL = {
    'work': gdata.apps.contacts.PHONE_WORK,
    'home': gdata.apps.contacts.PHONE_HOME,
    'other': gdata.apps.contacts.PHONE_OTHER,
    'fax': gdata.apps.contacts.PHONE_FAX,
    'home_fax': gdata.apps.contacts.PHONE_HOME_FAX,
    'work_fax': gdata.apps.contacts.PHONE_WORK_FAX,
    'other_fax': gdata.apps.contacts.PHONE_OTHER_FAX,
    'main': gdata.apps.contacts.PHONE_MAIN,
    'company_main': gdata.apps.contacts.PHONE_COMPANY_MAIN,
    'assistant': gdata.apps.contacts.PHONE_ASSISTANT,
    'mobile': gdata.apps.contacts.PHONE_MOBILE,
    'work_mobile': gdata.apps.contacts.PHONE_WORK_MOBILE,
    'pager': gdata.apps.contacts.PHONE_PAGER,
    'work_pager': gdata.apps.contacts.PHONE_WORK_PAGER,
    'car': gdata.apps.contacts.PHONE_CAR,
    'radio': gdata.apps.contacts.PHONE_RADIO,
    'callback': gdata.apps.contacts.PHONE_CALLBACK,
    'isdn': gdata.apps.contacts.PHONE_ISDN,
    'telex': gdata.apps.contacts.PHONE_TELEX,
    'tty_tdd': gdata.apps.contacts.PHONE_TTY_TDD,
    }

  PHONE_REL_TO_TYPE_ARGUMENT = {
    gdata.apps.contacts.PHONE_WORK: 'work',
    gdata.apps.contacts.PHONE_HOME: 'home',
    gdata.apps.contacts.PHONE_OTHER: 'other',
    gdata.apps.contacts.PHONE_FAX: 'fax',
    gdata.apps.contacts.PHONE_HOME_FAX: 'home_fax',
    gdata.apps.contacts.PHONE_WORK_FAX: 'work_fax',
    gdata.apps.contacts.PHONE_OTHER_FAX: 'other_fax',
    gdata.apps.contacts.PHONE_MAIN: 'main',
    gdata.apps.contacts.PHONE_COMPANY_MAIN: 'company_main',
    gdata.apps.contacts.PHONE_ASSISTANT: 'assistant',
    gdata.apps.contacts.PHONE_MOBILE: 'mobile',
    gdata.apps.contacts.PHONE_WORK_MOBILE: 'work_mobile',
    gdata.apps.contacts.PHONE_PAGER: 'pager',
    gdata.apps.contacts.PHONE_WORK_PAGER: 'work_pager',
    gdata.apps.contacts.PHONE_CAR: 'car',
    gdata.apps.contacts.PHONE_RADIO: 'radio',
    gdata.apps.contacts.PHONE_CALLBACK: 'callback',
    gdata.apps.contacts.PHONE_ISDN: 'isdn',
    gdata.apps.contacts.PHONE_TELEX: 'telex',
    gdata.apps.contacts.PHONE_TTY_TDD: 'tty_tdd',
    }

  RELATION_TYPE_ARGUMENT_TO_REL = {
    'spouse': 'spouse',
    'child': 'child',
    'mother': 'mother',
    'father': 'father',
    'parent': 'parent',
    'brother': 'brother',
    'sister': 'sister',
    'friend': 'friend',
    'relative': 'relative',
    'manager': 'manager',
    'assistant': 'assistant',
    'referredby': 'referred-by',
    'partner': 'partner',
    'domesticpartner': 'domestic-partner',
    }

  RELATION_REL_TO_TYPE_ARGUMENT = {
    'spouse' : 'spouse',
    'child' : 'child',
    'mother' : 'mother',
    'father' : 'father',
    'parent' : 'parent',
    'brother' : 'brother',
    'sister' : 'sister',
    'friend' : 'friend',
    'relative' : 'relative',
    'manager' : 'manager',
    'assistant' : 'assistant',
    'referred-by' : 'referred_by',
    'partner' : 'partner',
    'domestic-partner' : 'domestic_partner',
    }

  WEBSITE_TYPE_ARGUMENT_TO_REL = {
    'home-page': 'home-page',
    'blog': 'blog',
    'profile': 'profile',
    'work': 'work',
    'home': 'home',
    'other': 'other',
    'ftp': 'ftp',
    'reservations': 'reservations',
    'app-install-page': 'app-install-page',
    }

  WEBSITE_REL_TO_TYPE_ARGUMENT = {
    'home-page': 'home-page',
    'blog': 'blog',
    'profile': 'profile',
    'work': 'work',
    'home': 'home',
    'other': 'other',
    'ftp': 'ftp',
    'reservations': 'reservations',
    'app-install-page': 'app-install-page',
    }

  CONTACT_NAME_PROPERTY_PRINT_ORDER = [
    CONTACT_UPDATED,
    CONTACT_NAME,
    CONTACT_NAME_PREFIX,
    CONTACT_GIVEN_NAME,
    CONTACT_ADDITIONAL_NAME,
    CONTACT_FAMILY_NAME,
    CONTACT_NAME_SUFFIX,
    CONTACT_NICKNAME,
    CONTACT_MAIDENNAME,
    CONTACT_SHORTNAME,
    CONTACT_INITIALS,
    CONTACT_BIRTHDAY,
    CONTACT_GENDER,
    CONTACT_LOCATION,
    CONTACT_PRIORITY,
    CONTACT_SENSITIVITY,
    CONTACT_SUBJECT,
    CONTACT_LANGUAGE,
    CONTACT_NOTES,
    CONTACT_OCCUPATION,
    CONTACT_BILLING_INFORMATION,
    CONTACT_MILEAGE,
    CONTACT_DIRECTORY_SERVER,
    ]

  CONTACT_ARRAY_PROPERTY_PRINT_ORDER = [
    CONTACT_ADDRESSES,
    CONTACT_EMAILS,
    CONTACT_IMS,
    CONTACT_PHONES,
    CONTACT_CALENDARS,
    CONTACT_ORGANIZATIONS,
    CONTACT_EXTERNALIDS,
    CONTACT_EVENTS,
    CONTACT_HOBBIES,
    CONTACT_JOTS,
    CONTACT_RELATIONS,
    CONTACT_WEBSITES,
    CONTACT_USER_DEFINED_FIELDS,
    ]

  CONTACT_ARRAY_PROPERTIES = {
    CONTACT_ADDRESSES: {'relMap': ADDRESS_REL_TO_TYPE_ARGUMENT, 'infoTitle': 'formatted', 'primary': True},
    CONTACT_EMAILS: {'relMap': EMAIL_REL_TO_TYPE_ARGUMENT, 'infoTitle': 'address', 'primary': True},
    CONTACT_IMS: {'relMap': IM_REL_TO_TYPE_ARGUMENT, 'infoTitle': 'address', 'primary': True},
    CONTACT_PHONES: {'relMap': PHONE_REL_TO_TYPE_ARGUMENT, 'infoTitle': 'value', 'primary': True},
    CONTACT_CALENDARS: {'relMap': CALENDAR_REL_TO_TYPE_ARGUMENT, 'infoTitle': 'address', 'primary': True},
    CONTACT_ORGANIZATIONS: {'relMap': ORGANIZATION_REL_TO_TYPE_ARGUMENT, 'infoTitle': 'name', 'primary': True},
    CONTACT_EXTERNALIDS: {'relMap': EXTERNALID_REL_TO_TYPE_ARGUMENT, 'infoTitle': 'value', 'primary': False},
    CONTACT_EVENTS: {'relMap': EVENT_REL_TO_TYPE_ARGUMENT, 'infoTitle': 'date', 'primary': False},
    CONTACT_HOBBIES: {'relMap': None, 'infoTitle': 'value', 'primary': False},
    CONTACT_JOTS: {'relMap': JOT_REL_TO_TYPE_ARGUMENT, 'infoTitle': 'value', 'primary': False},
    CONTACT_RELATIONS: {'relMap': RELATION_REL_TO_TYPE_ARGUMENT, 'infoTitle': 'value', 'primary': False},
    CONTACT_USER_DEFINED_FIELDS: {'relMap': None, 'infoTitle': 'value', 'primary': False},
    CONTACT_WEBSITES: {'relMap': WEBSITE_REL_TO_TYPE_ARGUMENT, 'infoTitle': 'value', 'primary': True},
    }

  @staticmethod
  def GetContactShortId(contactEntry):
    full_id = contactEntry.id.text
    return full_id[full_id.rfind('/')+1:]

  @staticmethod
  def GetContactFields(parameters=None):

    fields = {}

    def CheckClearFieldsList(fieldName):
      if checkArgumentPresent(Cmd.CLEAR_NONE_ARGUMENT):
        fields.pop(fieldName, None)
        fields[fieldName] = []
        return True
      return False

    def InitArrayItem(choices):
      item = {}
      rel = getChoice(choices, mapChoice=True, defaultChoice=None)
      if rel:
        item['rel'] = rel
        item['label'] = None
      else:
        item['rel'] = None
        item['label'] = getString(Cmd.OB_STRING)
      return item

    def PrimaryNotPrimary(pnp, entry):
      if pnp == 'notprimary':
        entry['primary'] = 'false'
        return True
      if pnp == 'primary':
        entry['primary'] = 'true'
        primary['location'] = Cmd.Location()
        return True
      return False

    def GetPrimaryNotPrimaryChoice(entry):
      if not getChoice({'primary': True, 'notprimary': False}, mapChoice=True):
        entry['primary'] = 'false'
      else:
        entry['primary'] = 'true'
        primary['location'] = Cmd.Location()

    def AppendItemToFieldsList(fieldName, fieldValue, checkBlankField=None):
      fields.setdefault(fieldName, [])
      if checkBlankField is None or fieldValue[checkBlankField]:
        if isinstance(fieldValue, dict) and fieldValue.get('primary', 'false') == 'true':
          for citem in fields[fieldName]:
            if citem.get('primary', 'false') == 'true':
              Cmd.SetLocation(primary['location']-1)
              usageErrorExit(Msg.MULTIPLE_ITEMS_MARKED_PRIMARY.format(fieldName))
        fields[fieldName].append(fieldValue)

    primary = {}
    while Cmd.ArgumentsRemaining():
      if parameters is not None:
        if _getCreateContactReturnOptions(parameters):
          continue
        Cmd.Backup()
      fieldName = getChoice(ContactsManager.CONTACT_ARGUMENT_TO_PROPERTY_MAP, mapChoice=True)
      if fieldName == CONTACT_BIRTHDAY:
        fields[fieldName] = getYYYYMMDD(minLen=0)
      elif fieldName == CONTACT_GENDER:
        fields[fieldName] = getChoice(ContactsManager.GENDER_CHOICE_MAP, mapChoice=True)
      elif fieldName == CONTACT_PRIORITY:
        fields[fieldName] = getChoice(ContactsManager.PRIORITY_CHOICE_MAP, mapChoice=True)
      elif fieldName == CONTACT_SENSITIVITY:
        fields[fieldName] = getChoice(ContactsManager.SENSITIVITY_CHOICE_MAP, mapChoice=True)
      elif fieldName == CONTACT_LANGUAGE:
        fields[fieldName] = getLanguageCode(LANGUAGE_CODES_MAP)
      elif fieldName == CONTACT_NOTES:
        fields[fieldName] = getStringWithCRsNLsOrFile()[0]
      elif fieldName == CONTACT_ADDRESSES:
        if CheckClearFieldsList(fieldName):
          continue
        entry = InitArrayItem(ContactsManager.ADDRESS_TYPE_ARGUMENT_TO_REL)
        entry['primary'] = 'false'
        while Cmd.ArgumentsRemaining():
          argument = getArgument()
          if argument in ContactsManager.ADDRESS_ARGUMENT_TO_FIELD_MAP:
            value = getString(Cmd.OB_STRING, minLen=0)
            if value:
              entry[ContactsManager.ADDRESS_ARGUMENT_TO_FIELD_MAP[argument]] = value.replace('\\n', '\n')
          elif PrimaryNotPrimary(argument, entry):
            break
          else:
            unknownArgumentExit()
        AppendItemToFieldsList(fieldName, entry)
      elif fieldName == CONTACT_CALENDARS:
        if CheckClearFieldsList(fieldName):
          continue
        entry = InitArrayItem(ContactsManager.CALENDAR_TYPE_ARGUMENT_TO_REL)
        entry['value'] = getString(Cmd.OB_STRING, minLen=0)
        GetPrimaryNotPrimaryChoice(entry)
        AppendItemToFieldsList(fieldName, entry, 'value')
      elif fieldName == CONTACT_EMAILS:
        if CheckClearFieldsList(fieldName):
          continue
        entry = InitArrayItem(ContactsManager.EMAIL_TYPE_ARGUMENT_TO_REL)
        entry['value'] = getEmailAddress(noUid=True, minLen=0)
        GetPrimaryNotPrimaryChoice(entry)
        AppendItemToFieldsList(fieldName, entry, 'value')
      elif fieldName == CONTACT_EVENTS:
        if CheckClearFieldsList(fieldName):
          continue
        entry = InitArrayItem(ContactsManager.EVENT_TYPE_ARGUMENT_TO_REL)
        entry['value'] = getYYYYMMDD(minLen=0)
        AppendItemToFieldsList(fieldName, entry, 'value')
      elif fieldName == CONTACT_EXTERNALIDS:
        if CheckClearFieldsList(fieldName):
          continue
        entry = InitArrayItem(ContactsManager.EXTERNALID_TYPE_ARGUMENT_TO_REL)
        entry['value'] = getString(Cmd.OB_STRING, minLen=0)
        AppendItemToFieldsList(fieldName, entry, 'value')
      elif fieldName == CONTACT_HOBBIES:
        if CheckClearFieldsList(fieldName):
          continue
        entry = {'value': getString(Cmd.OB_STRING, minLen=0)}
        AppendItemToFieldsList(fieldName, entry, 'value')
      elif fieldName == CONTACT_IMS:
        if CheckClearFieldsList(fieldName):
          continue
        entry = InitArrayItem(ContactsManager.IM_TYPE_ARGUMENT_TO_REL)
        entry['protocol'] = getChoice(ContactsManager.IM_PROTOCOL_TO_REL_MAP, mapChoice=True)
        entry['value'] = getString(Cmd.OB_STRING, minLen=0)
        GetPrimaryNotPrimaryChoice(entry)
        AppendItemToFieldsList(fieldName, entry, 'value')
      elif fieldName == CONTACT_JOTS:
        if CheckClearFieldsList(fieldName):
          continue
        entry = {'rel': getChoice(ContactsManager.JOT_TYPE_ARGUMENT_TO_REL, mapChoice=True)}
        entry['value'] = getString(Cmd.OB_STRING, minLen=0)
        AppendItemToFieldsList(fieldName, entry, 'value')
      elif fieldName == CONTACT_ORGANIZATIONS:
        if CheckClearFieldsList(fieldName):
          continue
        entry = InitArrayItem(ContactsManager.ORGANIZATION_TYPE_ARGUMENT_TO_REL)
        entry['primary'] = 'false'
        entry['value'] = getString(Cmd.OB_STRING, minLen=0)
        while Cmd.ArgumentsRemaining():
          argument = getArgument()
          if argument in ContactsManager.ORGANIZATION_ARGUMENT_TO_FIELD_MAP:
            value = getString(Cmd.OB_STRING, minLen=0)
            if value:
              entry[ContactsManager.ORGANIZATION_ARGUMENT_TO_FIELD_MAP[argument]] = value
          elif PrimaryNotPrimary(argument, entry):
            break
          else:
            unknownArgumentExit()
        AppendItemToFieldsList(fieldName, entry, 'value')
      elif fieldName == CONTACT_PHONES:
        if CheckClearFieldsList(fieldName):
          continue
        entry = InitArrayItem(ContactsManager.PHONE_TYPE_ARGUMENT_TO_REL)
        entry['value'] = getString(Cmd.OB_STRING, minLen=0)
        GetPrimaryNotPrimaryChoice(entry)
        AppendItemToFieldsList(fieldName, entry, 'value')
      elif fieldName == CONTACT_RELATIONS:
        if CheckClearFieldsList(fieldName):
          continue
        entry = InitArrayItem(ContactsManager.RELATION_TYPE_ARGUMENT_TO_REL)
        entry['value'] = getString(Cmd.OB_STRING, minLen=0)
        AppendItemToFieldsList(fieldName, entry, 'value')
      elif fieldName == CONTACT_USER_DEFINED_FIELDS:
        if CheckClearFieldsList(fieldName):
          continue
        entry = {'rel': getString(Cmd.OB_STRING, minLen=0), 'value': getString(Cmd.OB_STRING, minLen=0)}
        if not entry['rel'] or entry['rel'].lower() == 'none':
          entry['rel'] = None
        AppendItemToFieldsList(fieldName, entry, 'value')
      elif fieldName == CONTACT_WEBSITES:
        if CheckClearFieldsList(fieldName):
          continue
        entry = InitArrayItem(ContactsManager.WEBSITE_TYPE_ARGUMENT_TO_REL)
        entry['value'] = getString(Cmd.OB_STRING, minLen=0)
        GetPrimaryNotPrimaryChoice(entry)
        AppendItemToFieldsList(fieldName, entry, 'value')
      else:
        fields[fieldName] = getString(Cmd.OB_STRING, minLen=0)
    return fields

  @staticmethod
  def FieldsToContact(fields):
    def GetField(fieldName):
      return fields.get(fieldName)

    def SetClassAttribute(value, fieldClass, processNLs, attr):
      if value:
        if processNLs:
          value = value.replace('\\n', '\n')
        if attr == 'text':
          return fieldClass(text=value)
        if attr == 'code':
          return fieldClass(code=value)
        if attr == 'rel':
          return fieldClass(rel=value)
        if attr == 'value':
          return fieldClass(value=value)
        if attr == 'value_string':
          return fieldClass(value_string=value)
        if attr == 'when':
          return fieldClass(when=value)
      return None

    def GetContactField(fieldName, fieldClass, processNLs=False, attr='text'):
      return SetClassAttribute(fields.get(fieldName), fieldClass, processNLs, attr)

    def GetListEntryField(entry, fieldName, fieldClass, processNLs=False, attr='text'):
      return SetClassAttribute(entry.get(fieldName), fieldClass, processNLs, attr)

    contactEntry = gdata.apps.contacts.ContactEntry()
    value = GetField(CONTACT_NAME)
    if not value:
      value = ' '.join([fields[fieldName] for fieldName in ContactsManager.CONTACT_NAME_FIELDS if fieldName in fields])
    contactEntry.name = gdata.apps.contacts.Name(full_name=gdata.apps.contacts.FullName(text=value))
    contactEntry.name.name_prefix = GetContactField(CONTACT_NAME_PREFIX, gdata.apps.contacts.NamePrefix)
    contactEntry.name.given_name = GetContactField(CONTACT_GIVEN_NAME, gdata.apps.contacts.GivenName)
    contactEntry.name.additional_name = GetContactField(CONTACT_ADDITIONAL_NAME, gdata.apps.contacts.AdditionalName)
    contactEntry.name.family_name = GetContactField(CONTACT_FAMILY_NAME, gdata.apps.contacts.FamilyName)
    contactEntry.name.name_suffix = GetContactField(CONTACT_NAME_SUFFIX, gdata.apps.contacts.NameSuffix)
    contactEntry.nickname = GetContactField(CONTACT_NICKNAME, gdata.apps.contacts.Nickname)
    contactEntry.maidenName = GetContactField(CONTACT_MAIDENNAME, gdata.apps.contacts.MaidenName)
    contactEntry.shortName = GetContactField(CONTACT_SHORTNAME, gdata.apps.contacts.ShortName)
    contactEntry.initials = GetContactField(CONTACT_INITIALS, gdata.apps.contacts.Initials)
    contactEntry.birthday = GetContactField(CONTACT_BIRTHDAY, gdata.apps.contacts.Birthday, attr='when')
    contactEntry.gender = GetContactField(CONTACT_GENDER, gdata.apps.contacts.Gender, attr='value')
    contactEntry.where = GetContactField(CONTACT_LOCATION, gdata.apps.contacts.Where, attr='value_string')
    contactEntry.priority = GetContactField(CONTACT_PRIORITY, gdata.apps.contacts.Priority, attr='rel')
    contactEntry.sensitivity = GetContactField(CONTACT_SENSITIVITY, gdata.apps.contacts.Sensitivity, attr='rel')
    contactEntry.subject = GetContactField(CONTACT_SUBJECT, gdata.apps.contacts.Subject)
    contactEntry.language = GetContactField(CONTACT_LANGUAGE, gdata.apps.contacts.Language, attr='code')
    contactEntry.content = GetContactField(CONTACT_NOTES, gdata.apps.contacts.Content, processNLs=True)
    contactEntry.occupation = GetContactField(CONTACT_OCCUPATION, gdata.apps.contacts.Occupation)
    contactEntry.billingInformation = GetContactField(CONTACT_BILLING_INFORMATION, gdata.apps.contacts.BillingInformation, processNLs=True)
    contactEntry.mileage = GetContactField(CONTACT_MILEAGE, gdata.apps.contacts.Mileage)
    contactEntry.directoryServer = GetContactField(CONTACT_DIRECTORY_SERVER, gdata.apps.contacts.DirectoryServer)
    value = GetField(CONTACT_ADDRESSES)
    if value:
      for address in value:
        street = GetListEntryField(address, 'street', gdata.apps.contacts.Street)
        pobox = GetListEntryField(address, 'pobox', gdata.apps.contacts.PoBox)
        neighborhood = GetListEntryField(address, 'neighborhood', gdata.apps.contacts.Neighborhood)
        city = GetListEntryField(address, 'city', gdata.apps.contacts.City)
        region = GetListEntryField(address, 'region', gdata.apps.contacts.Region)
        postcode = GetListEntryField(address, 'postcode', gdata.apps.contacts.Postcode)
        country = GetListEntryField(address, 'country', gdata.apps.contacts.Country)
        formatted_address = GetListEntryField(address, 'value', gdata.apps.contacts.FormattedAddress, processNLs=True)
        contactEntry.structuredPostalAddress.append(gdata.apps.contacts.StructuredPostalAddress(street=street, pobox=pobox, neighborhood=neighborhood,
                                                                                                city=city, region=region,
                                                                                                postcode=postcode, country=country,
                                                                                                formatted_address=formatted_address,
                                                                                                rel=address['rel'], label=address['label'], primary=address['primary']))
    value = GetField(CONTACT_CALENDARS)
    if value:
      for calendarLink in value:
        contactEntry.calendarLink.append(gdata.apps.contacts.CalendarLink(href=calendarLink['value'], rel=calendarLink['rel'], label=calendarLink['label'], primary=calendarLink['primary']))
    value = GetField(CONTACT_EMAILS)
    if value:
      for emailaddr in value:
        contactEntry.email.append(gdata.apps.contacts.Email(address=emailaddr['value'], rel=emailaddr['rel'], label=emailaddr['label'], primary=emailaddr['primary']))
    value = GetField(CONTACT_EXTERNALIDS)
    if value:
      for externalid in value:
        contactEntry.externalId.append(gdata.apps.contacts.ExternalId(value=externalid['value'], rel=externalid['rel'], label=externalid['label']))
    value = GetField(CONTACT_EVENTS)
    if value:
      for event in value:
        contactEntry.event.append(gdata.apps.contacts.Event(rel=event['rel'], label=event['label'],
                                                            when=gdata.apps.contacts.When(startTime=event['value'])))
    value = GetField(CONTACT_HOBBIES)
    if value:
      for hobby in value:
        contactEntry.hobby.append(gdata.apps.contacts.Hobby(text=hobby['value']))
    value = GetField(CONTACT_IMS)
    if value:
      for im in value:
        contactEntry.im.append(gdata.apps.contacts.IM(address=im['value'], protocol=im['protocol'], rel=im['rel'], label=im['label'], primary=im['primary']))
    value = GetField(CONTACT_JOTS)
    if value:
      for jot in value:
        contactEntry.jot.append(gdata.apps.contacts.Jot(text=jot['value'], rel=jot['rel']))
    value = GetField(CONTACT_ORGANIZATIONS)
    if value:
      for organization in value:
        org_name = gdata.apps.contacts.OrgName(text=organization['value'])
        department = GetListEntryField(organization, 'department', gdata.apps.contacts.OrgDepartment)
        title = GetListEntryField(organization, 'title', gdata.apps.contacts.OrgTitle)
        job_description = GetListEntryField(organization, 'jobdescription', gdata.apps.contacts.OrgJobDescription)
        symbol = GetListEntryField(organization, 'symbol', gdata.apps.contacts.OrgSymbol)
        where = GetListEntryField(organization, 'where', gdata.apps.contacts.Where, attr='value_string')
        contactEntry.organization.append(gdata.apps.contacts.Organization(name=org_name, department=department,
                                                                          title=title, job_description=job_description,
                                                                          symbol=symbol, where=where,
                                                                          rel=organization['rel'], label=organization['label'], primary=organization['primary']))
    value = GetField(CONTACT_PHONES)
    if value:
      for phone in value:
        contactEntry.phoneNumber.append(gdata.apps.contacts.PhoneNumber(text=phone['value'], rel=phone['rel'], label=phone['label'], primary=phone['primary']))
    value = GetField(CONTACT_RELATIONS)
    if value:
      for relation in value:
        contactEntry.relation.append(gdata.apps.contacts.Relation(text=relation['value'], rel=relation['rel'], label=relation['label']))
    value = GetField(CONTACT_USER_DEFINED_FIELDS)
    if value:
      for userdefinedfield in value:
        contactEntry.userDefinedField.append(gdata.apps.contacts.UserDefinedField(key=userdefinedfield['rel'], value=userdefinedfield['value']))
    value = GetField(CONTACT_WEBSITES)
    if value:
      for website in value:
        contactEntry.website.append(gdata.apps.contacts.Website(href=website['value'], rel=website['rel'], label=website['label'], primary=website['primary']))
    return contactEntry

  @staticmethod
  def ContactToFields(contactEntry):
    fields = {}
    def GetContactField(fieldName, attrlist):
      objAttr = contactEntry
      for attr in attrlist:
        objAttr = getattr(objAttr, attr)
        if not objAttr:
          return
      fields[fieldName] = objAttr

    def GetListEntryField(entry, attrlist):
      objAttr = entry
      for attr in attrlist:
        objAttr = getattr(objAttr, attr)
        if not objAttr:
          return None
      return objAttr

    def AppendItemToFieldsList(fieldName, fieldValue):
      fields.setdefault(fieldName, [])
      fields[fieldName].append(fieldValue)

    fields[CONTACT_ID] = ContactsManager.GetContactShortId(contactEntry)
    GetContactField(CONTACT_UPDATED, ['updated', 'text'])
    if not contactEntry.deleted:
      GetContactField(CONTACT_NAME, ['title', 'text'])
    else:
      fields[CONTACT_NAME] = 'Deleted'
    GetContactField(CONTACT_NAME_PREFIX, ['name', 'name_prefix', 'text'])
    GetContactField(CONTACT_GIVEN_NAME, ['name', 'given_name', 'text'])
    GetContactField(CONTACT_ADDITIONAL_NAME, ['name', 'additional_name', 'text'])
    GetContactField(CONTACT_FAMILY_NAME, ['name', 'family_name', 'text'])
    GetContactField(CONTACT_NAME_SUFFIX, ['name', 'name_suffix', 'text'])
    GetContactField(CONTACT_NICKNAME, ['nickname', 'text'])
    GetContactField(CONTACT_MAIDENNAME, ['maidenName', 'text'])
    GetContactField(CONTACT_SHORTNAME, ['shortName', 'text'])
    GetContactField(CONTACT_INITIALS, ['initials', 'text'])
    GetContactField(CONTACT_BIRTHDAY, ['birthday', 'when'])
    GetContactField(CONTACT_GENDER, ['gender', 'value'])
    GetContactField(CONTACT_SUBJECT, ['subject', 'text'])
    GetContactField(CONTACT_LANGUAGE, ['language', 'code'])
    GetContactField(CONTACT_PRIORITY, ['priority', 'rel'])
    GetContactField(CONTACT_SENSITIVITY, ['sensitivity', 'rel'])
    GetContactField(CONTACT_NOTES, ['content', 'text'])
    GetContactField(CONTACT_LOCATION, ['where', 'value_string'])
    GetContactField(CONTACT_OCCUPATION, ['occupation', 'text'])
    GetContactField(CONTACT_BILLING_INFORMATION, ['billingInformation', 'text'])
    GetContactField(CONTACT_MILEAGE, ['mileage', 'text'])
    GetContactField(CONTACT_DIRECTORY_SERVER, ['directoryServer', 'text'])
    for address in contactEntry.structuredPostalAddress:
      AppendItemToFieldsList(CONTACT_ADDRESSES,
                             {'rel': address.rel,
                              'label': address.label,
                              'value': GetListEntryField(address, ['formatted_address', 'text']),
                              'street': GetListEntryField(address, ['street', 'text']),
                              'pobox': GetListEntryField(address, ['pobox', 'text']),
                              'neighborhood': GetListEntryField(address, ['neighborhood', 'text']),
                              'city': GetListEntryField(address, ['city', 'text']),
                              'region': GetListEntryField(address, ['region', 'text']),
                              'postcode': GetListEntryField(address, ['postcode', 'text']),
                              'country': GetListEntryField(address, ['country', 'text']),
                              'primary': address.primary})
    for calendarLink in contactEntry.calendarLink:
      AppendItemToFieldsList(CONTACT_CALENDARS,
                             {'rel': calendarLink.rel,
                              'label': calendarLink.label,
                              'value': calendarLink.href,
                              'primary': calendarLink.primary})
    for emailaddr in contactEntry.email:
      AppendItemToFieldsList(CONTACT_EMAILS,
                             {'rel': emailaddr.rel,
                              'label': emailaddr.label,
                              'value': emailaddr.address,
                              'primary': emailaddr.primary})
    for externalid in contactEntry.externalId:
      AppendItemToFieldsList(CONTACT_EXTERNALIDS,
                             {'rel': externalid.rel,
                              'label': externalid.label,
                              'value': externalid.value})
    for event in contactEntry.event:
      AppendItemToFieldsList(CONTACT_EVENTS,
                             {'rel': event.rel,
                              'label': event.label,
                              'value': GetListEntryField(event, ['when', 'startTime'])})
    for hobby in contactEntry.hobby:
      AppendItemToFieldsList(CONTACT_HOBBIES,
                             {'value': hobby.text})
    for im in contactEntry.im:
      AppendItemToFieldsList(CONTACT_IMS,
                             {'rel': im.rel,
                              'label': im.label,
                              'value': im.address,
                              'protocol': im.protocol,
                              'primary': im.primary})
    for jot in contactEntry.jot:
      AppendItemToFieldsList(CONTACT_JOTS,
                             {'rel': jot.rel,
                              'value': jot.text})
    for organization in contactEntry.organization:
      AppendItemToFieldsList(CONTACT_ORGANIZATIONS,
                             {'rel': organization.rel,
                              'label': organization.label,
                              'value': GetListEntryField(organization, ['name', 'text']),
                              'department': GetListEntryField(organization, ['department', 'text']),
                              'title': GetListEntryField(organization, ['title', 'text']),
                              'symbol': GetListEntryField(organization, ['symbol', 'text']),
                              'jobdescription': GetListEntryField(organization, ['job_description', 'text']),
                              'where': GetListEntryField(organization, ['where', 'value_string']),
                              'primary': organization.primary})
    for phone in contactEntry.phoneNumber:
      AppendItemToFieldsList(CONTACT_PHONES,
                             {'rel': phone.rel,
                              'label': phone.label,
                              'value': phone.text,
                              'primary': phone.primary})
    for relation in contactEntry.relation:
      AppendItemToFieldsList(CONTACT_RELATIONS,
                             {'rel': relation.rel,
                              'label': relation.label,
                              'value': relation.text})
    for userdefinedfield in contactEntry.userDefinedField:
      AppendItemToFieldsList(CONTACT_USER_DEFINED_FIELDS,
                             {'rel': userdefinedfield.key,
                              'value': userdefinedfield.value})
    for website in contactEntry.website:
      AppendItemToFieldsList(CONTACT_WEBSITES,
                             {'rel': website.rel,
                              'label': website.label,
                              'value': website.href,
                              'primary': website.primary})
    return fields

CONTACTS_PROJECTION_CHOICE_MAP = {'basic': 'thin', 'thin': 'thin', 'full': 'full'}
CONTACTS_ORDERBY_CHOICE_MAP = {'lastmodified': 'lastmodified'}

def normalizeContactId(contactId):
  if contactId.startswith('id:'):
    return contactId[3:]
  return contactId

def _initContactQueryAttributes():
  return {'query': None, 'projection': 'full', 'url_params': {'max-results': str(GC.Values[GC.CONTACT_MAX_RESULTS])},
          'emailMatchPattern': None, 'emailMatchType': None}

def _getContactQueryAttributes(contactQuery, myarg, unknownAction, printShowCmd):
  if myarg == 'query':
    contactQuery['query'] = getString(Cmd.OB_QUERY)
  elif myarg == 'emailmatchpattern':
    contactQuery['emailMatchPattern'] = getREPattern(re.IGNORECASE)
  elif myarg == 'emailmatchtype':
    contactQuery['emailMatchType'] = getString(Cmd.OB_CONTACT_EMAIL_TYPE)
  elif myarg == 'updatedmin':
    contactQuery['url_params']['updated-min'] = getYYYYMMDD()
  elif myarg == 'endquery':
    return False
  elif not printShowCmd:
    if unknownAction < 0:
      unknownArgumentExit()
    if unknownAction > 0:
      Cmd.Backup()
    return False
  elif myarg == 'orderby':
    contactQuery['url_params']['orderby'], contactQuery['url_params']['sortorder'] = getOrderBySortOrder(CONTACTS_ORDERBY_CHOICE_MAP, 'ascending', False)
  elif myarg in CONTACTS_PROJECTION_CHOICE_MAP:
    contactQuery['projection'] = CONTACTS_PROJECTION_CHOICE_MAP[myarg]
  elif myarg == 'showdeleted':
    contactQuery['url_params']['showdeleted'] = 'true'
  else:
    if unknownAction < 0:
      unknownArgumentExit()
    if unknownAction > 0:
      Cmd.Backup()
    return False
  return True

CONTACT_SELECT_ARGUMENTS = {'query', 'emailmatchpattern', 'emailmatchtype', 'updatedmin'}

def _getContactEntityList(unknownAction, printShowCmd):
  contactQuery = _initContactQueryAttributes()
  if Cmd.PeekArgumentPresent(CONTACT_SELECT_ARGUMENTS):
    entityList = None
    queriedContacts = True
    while Cmd.ArgumentsRemaining():
      myarg = getArgument()
      if not _getContactQueryAttributes(contactQuery, myarg, unknownAction, printShowCmd):
        break
  else:
    entityList = getEntityList(Cmd.OB_CONTACT_ENTITY)
    queriedContacts = False
    if unknownAction < 0:
      checkForExtraneousArguments()
  return (entityList, contactQuery, queriedContacts)

def queryContacts(contactsObject, contactQuery):
  entityType = Ent.DOMAIN
  user = GC.Values[GC.DOMAIN]
  if contactQuery['query']:
    uri = getContactsQuery(feed=contactsObject.GetContactFeedUri(contact_list=user, projection=contactQuery['projection']),
                           text_query=contactQuery['query']).ToUri()
  else:
    uri = contactsObject.GetContactFeedUri(contact_list=user, projection=contactQuery['projection'])
  printGettingAllEntityItemsForWhom(Ent.CONTACT, user, query=contactQuery['query'])
  try:
    entityList = callGDataPages(contactsObject, 'GetContactsFeed',
                                pageMessage=getPageMessageForWhom(),
                                throwErrors=[GDATA.BAD_REQUEST, GDATA.FORBIDDEN],
                                retryErrors=[GDATA.INTERNAL_SERVER_ERROR],
                                uri=uri, url_params=contactQuery['url_params'])
    return entityList
  except GDATA.badRequest as e:
    entityActionFailedWarning([entityType, user, Ent.CONTACT, ''], str(e))
  except GDATA.forbidden:
    entityServiceNotApplicableWarning(entityType, user)
  return None

def localContactSelects(contactsManager, contactQuery, fields):
  if contactQuery['emailMatchPattern']:
    emailMatchType = contactQuery['emailMatchType']
    for item in fields.get(CONTACT_EMAILS, []):
      if contactQuery['emailMatchPattern'].match(item['value']):
        if (not emailMatchType or
            emailMatchType == item.get('label') or
            emailMatchType == contactsManager.CONTACT_ARRAY_PROPERTIES[CONTACT_EMAILS]['relMap'].get(item['rel'], 'custom')):
          break
    else:
      return False
  return True

def countLocalContactSelects(contactsManager, contacts, contactQuery):
  if contacts is not None and contactQuery:
    jcount = 0
    for contact in contacts:
      fields = contactsManager.ContactToFields(contact)
      if localContactSelects(contactsManager, contactQuery, fields):
        jcount += 1
  else:
    jcount = len(contacts) if contacts is not None else 0
  return jcount

def clearEmailAddressMatches(contactsManager, contactClear, fields):
  savedAddresses = []
  updateRequired = False
  emailMatchType = contactClear['emailClearType']
  for item in fields.get(CONTACT_EMAILS, []):
    if (contactClear['emailClearPattern'].match(item['value']) and
        (not emailMatchType or
         emailMatchType == item.get('label') or
         emailMatchType == contactsManager.CONTACT_ARRAY_PROPERTIES[CONTACT_EMAILS]['relMap'].get(item['rel'], 'custom'))):
      updateRequired = True
    else:
      savedAddresses.append(item)
  if updateRequired:
    fields[CONTACT_EMAILS] = savedAddresses
  return updateRequired

def dedupEmailAddressMatches(contactsManager, emailMatchType, fields):
  sai = -1
  savedAddresses = []
  matches = {}
  updateRequired = False
  for item in fields.get(CONTACT_EMAILS, []):
    emailAddr = item['value']
    emailType = item.get('label')
    if emailType is None:
      emailType = contactsManager.CONTACT_ARRAY_PROPERTIES[CONTACT_EMAILS]['relMap'].get(item['rel'], 'custom')
    if (emailAddr in matches) and (not emailMatchType or emailType in matches[emailAddr]['types']):
      if item['primary'] == 'true':
        savedAddresses[matches[emailAddr]['sai']]['primary'] = 'true'
      updateRequired = True
    else:
      savedAddresses.append(item)
      sai += 1
      matches.setdefault(emailAddr, {'types': set(), 'sai': sai})
      matches[emailAddr]['types'].add(emailType)
  if updateRequired:
    fields[CONTACT_EMAILS] = savedAddresses
  return updateRequired

# gam create contact <ContactAttribute>+
#	[(csv [todrive <ToDriveAttribute>*] (addcsvdata <FieldName> <String>)*))| returnidonly]
def doCreateDomainContact():
  entityType = Ent.DOMAIN
  contactsManager = ContactsManager()
  parameters = {'csvPF': None, 'titles': ['Domain', CONTACT_ID], 'addCSVData': {}, 'returnIdOnly': False}
  fields = contactsManager.GetContactFields(parameters)
  csvPF = parameters['csvPF']
  addCSVData = parameters['addCSVData']
  if addCSVData:
    csvPF.AddTitles(sorted(addCSVData.keys()))
  returnIdOnly = parameters['returnIdOnly']
  contactEntry = contactsManager.FieldsToContact(fields)
  user, contactsObject = getContactsObject()
  try:
    contact = callGData(contactsObject, 'CreateContact',
                        throwErrors=[GDATA.BAD_REQUEST, GDATA.SERVICE_NOT_APPLICABLE, GDATA.FORBIDDEN],
                        retryErrors=[GDATA.INTERNAL_SERVER_ERROR],
                        new_contact=contactEntry, insert_uri=contactsObject.GetContactFeedUri(contact_list=user))
    contactId = contactsManager.GetContactShortId(contact)
    if returnIdOnly:
      writeStdout(f'{contactId}\n')
    elif not csvPF:
      entityActionPerformed([entityType, user, Ent.CONTACT, contactId])
    else:
      row = {'Domain': user, CONTACT_ID: contactId}
      if addCSVData:
        row.update(addCSVData)
      csvPF.WriteRow(row)
  except GDATA.badRequest as e:
    entityActionFailedWarning([entityType, user, Ent.CONTACT, ''], str(e))
  except GDATA.forbidden:
    entityServiceNotApplicableWarning(entityType, user)
  except GDATA.serviceNotApplicable:
    entityUnknownWarning(entityType, user)
  if csvPF:
    csvPF.writeCSVfile('Contacts')

def _clearUpdateContacts(updateContacts):
  entityType = Ent.DOMAIN
  contactsManager = ContactsManager()
  entityList, contactQuery, queriedContacts = _getContactEntityList(1, False)
  if updateContacts:
    update_fields = contactsManager.GetContactFields()
  else:
    contactClear = {'emailClearPattern': contactQuery['emailMatchPattern'], 'emailClearType': contactQuery['emailMatchType']}
    deleteClearedContactsWithNoEmails = False
    while Cmd.ArgumentsRemaining():
      myarg = getArgument()
      if myarg == 'emailclearpattern':
        contactClear['emailClearPattern'] = getREPattern(re.IGNORECASE)
      elif myarg == 'emailcleartype':
        contactClear['emailClearType'] = getString(Cmd.OB_CONTACT_EMAIL_TYPE)
      elif myarg == 'deleteclearedcontactswithnoemails':
        deleteClearedContactsWithNoEmails = True
      else:
        unknownArgumentExit()
    if not contactClear['emailClearPattern']:
      missingArgumentExit('emailclearpattern')
  user, contactsObject = getContactsObject()
  if queriedContacts:
    entityList = queryContacts(contactsObject, contactQuery)
    if entityList is None:
      return
  j = 0
  jcount = len(entityList)
  entityPerformActionModifierNumItems([entityType, user], Msg.MAXIMUM_OF, jcount, Ent.CONTACT)
  if jcount == 0:
    setSysExitRC(NO_ENTITIES_FOUND_RC)
    return
  Ind.Increment()
  for contact in entityList:
    j += 1
    try:
      if not queriedContacts:
        contactId = normalizeContactId(contact)
        contact = callGData(contactsObject, 'GetContact',
                            throwErrors=[GDATA.NOT_FOUND, GDATA.BAD_REQUEST, GDATA.SERVICE_NOT_APPLICABLE, GDATA.FORBIDDEN, GDATA.NOT_IMPLEMENTED],
                            retryErrors=[GDATA.INTERNAL_SERVER_ERROR],
                            uri=contactsObject.GetContactFeedUri(contact_list=user, contactId=contactId))
        fields = contactsManager.ContactToFields(contact)
      else:
        contactId = contactsManager.GetContactShortId(contact)
        fields = contactsManager.ContactToFields(contact)
        if not localContactSelects(contactsManager, contactQuery, fields):
          continue
      if updateContacts:
##### Zip
        for field, value in update_fields.items():
          fields[field] = value
        contactEntry = contactsManager.FieldsToContact(fields)
      else:
        if not clearEmailAddressMatches(contactsManager, contactClear, fields):
          continue
        if deleteClearedContactsWithNoEmails and not fields[CONTACT_EMAILS]:
          Act.Set(Act.DELETE)
          callGData(contactsObject, 'DeleteContact',
                    throwErrors=[GDATA.NOT_FOUND, GDATA.SERVICE_NOT_APPLICABLE, GDATA.FORBIDDEN],
                    edit_uri=contactsObject.GetContactFeedUri(contact_list=user, contactId=contactId), extra_headers={'If-Match': contact.etag})
          entityActionPerformed([entityType, user, Ent.CONTACT, contactId], j, jcount)
          continue
        contactEntry = contactsManager.FieldsToContact(fields)
      contactEntry.category = contact.category
      contactEntry.link = contact.link
      contactEntry.etag = contact.etag
      contactEntry.id = contact.id
      Act.Set(Act.UPDATE)
      callGData(contactsObject, 'UpdateContact',
                throwErrors=[GDATA.NOT_FOUND, GDATA.BAD_REQUEST, GDATA.PRECONDITION_FAILED, GDATA.SERVICE_NOT_APPLICABLE, GDATA.FORBIDDEN],
                edit_uri=contactsObject.GetContactFeedUri(contact_list=user, contactId=contactId), updated_contact=contactEntry, extra_headers={'If-Match': contact.etag})
      entityActionPerformed([entityType, user, Ent.CONTACT, contactId], j, jcount)
    except (GDATA.notFound, GDATA.badRequest, GDATA.preconditionFailed) as e:
      entityActionFailedWarning([entityType, user, Ent.CONTACT, contactId], str(e), j, jcount)
    except (GDATA.forbidden, GDATA.notImplemented):
      entityServiceNotApplicableWarning(entityType, user)
      break
    except GDATA.serviceNotApplicable:
      entityUnknownWarning(entityType, user)
      break
  Ind.Decrement()

# gam clear contacts <ContactEntity>|<ContactSelection>
#	[clearmatchpattern <REMatchPattern>] [clearmatchtype work|home|other|<String>]
#	[deleteclearedcontactswithnoemails]
def doClearDomainContacts():
  _clearUpdateContacts(False)

# gam update contacts <ContactEntity>|<ContactSelection> <ContactAttribute>+
def doUpdateDomainContacts():
  _clearUpdateContacts(True)

# gam dedup contacts <ContactEntity>|<ContactSelection> [matchType [<Boolean>]]
def doDedupDomainContacts():
  entityType = Ent.DOMAIN
  contactsManager = ContactsManager()
  contactQuery = _initContactQueryAttributes()
  emailMatchType = False
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'matchtype':
      emailMatchType = getBoolean()
    else:
      _getContactQueryAttributes(contactQuery, myarg, -1, False)
  user, contactsObject = getContactsObject()
  contacts = queryContacts(contactsObject, contactQuery)
  if contacts is None:
    return
  j = 0
  jcount = len(contacts)
  entityPerformActionModifierNumItems([entityType, user], Msg.MAXIMUM_OF, jcount, Ent.CONTACT)
  if jcount == 0:
    setSysExitRC(NO_ENTITIES_FOUND_RC)
    return
  Ind.Increment()
  for contact in contacts:
    j += 1
    try:
      fields = contactsManager.ContactToFields(contact)
      if not localContactSelects(contactsManager, contactQuery, fields):
        continue
      if not dedupEmailAddressMatches(contactsManager, emailMatchType, fields):
        continue
      contactId = fields[CONTACT_ID]
      contactEntry = contactsManager.FieldsToContact(fields)
      contactEntry.category = contact.category
      contactEntry.link = contact.link
      contactEntry.etag = contact.etag
      contactEntry.id = contact.id
      Act.Set(Act.UPDATE)
      callGData(contactsObject, 'UpdateContact',
                throwErrors=[GDATA.NOT_FOUND, GDATA.BAD_REQUEST, GDATA.PRECONDITION_FAILED, GDATA.SERVICE_NOT_APPLICABLE, GDATA.FORBIDDEN],
                edit_uri=contactsObject.GetContactFeedUri(contact_list=user, contactId=contactId), updated_contact=contactEntry, extra_headers={'If-Match': contact.etag})
      entityActionPerformed([entityType, user, Ent.CONTACT, contactId], j, jcount)
    except (GDATA.notFound, GDATA.badRequest, GDATA.preconditionFailed) as e:
      entityActionFailedWarning([entityType, user, Ent.CONTACT, contactId], str(e), j, jcount)
    except (GDATA.forbidden, GDATA.notImplemented):
      entityServiceNotApplicableWarning(entityType, user)
      break
    except GDATA.serviceNotApplicable:
      entityUnknownWarning(entityType, user)
      break
  Ind.Decrement()

# gam delete contacts <ContactEntity>|<ContactSelection>
def doDeleteDomainContacts():
  entityType = Ent.DOMAIN
  contactsManager = ContactsManager()
  entityList, contactQuery, queriedContacts = _getContactEntityList(-1, False)
  user, contactsObject = getContactsObject()
  if queriedContacts:
    entityList = queryContacts(contactsObject, contactQuery)
    if entityList is None:
      return
  j = 0
  jcount = len(entityList)
  entityPerformActionModifierNumItems([entityType, user], Msg.MAXIMUM_OF, jcount, Ent.CONTACT)
  if jcount == 0:
    setSysExitRC(NO_ENTITIES_FOUND_RC)
    return
  Ind.Increment()
  for contact in entityList:
    j += 1
    try:
      if not queriedContacts:
        contactId = normalizeContactId(contact)
        contact = callGData(contactsObject, 'GetContact',
                            throwErrors=[GDATA.NOT_FOUND, GDATA.BAD_REQUEST, GDATA.SERVICE_NOT_APPLICABLE, GDATA.FORBIDDEN, GDATA.NOT_IMPLEMENTED],
                            retryErrors=[GDATA.INTERNAL_SERVER_ERROR],
                            uri=contactsObject.GetContactFeedUri(contact_list=user, contactId=contactId))
      else:
        contactId = contactsManager.GetContactShortId(contact)
        fields = contactsManager.ContactToFields(contact)
        if not localContactSelects(contactsManager, contactQuery, fields):
          continue
      callGData(contactsObject, 'DeleteContact',
                throwErrors=[GDATA.NOT_FOUND, GDATA.SERVICE_NOT_APPLICABLE, GDATA.FORBIDDEN],
                edit_uri=contactsObject.GetContactFeedUri(contact_list=user, contactId=contactId), extra_headers={'If-Match': contact.etag})
      entityActionPerformed([entityType, user, Ent.CONTACT, contactId], j, jcount)
    except (GDATA.notFound, GDATA.badRequest) as e:
      entityActionFailedWarning([entityType, user, Ent.CONTACT, contactId], str(e), j, jcount)
    except (GDATA.forbidden, GDATA.notImplemented):
      entityServiceNotApplicableWarning(entityType, user)
      break
    except GDATA.serviceNotApplicable:
      entityUnknownWarning(entityType, user)
      break
  Ind.Decrement()

CONTACT_TIME_OBJECTS = {CONTACT_UPDATED}
CONTACT_FIELDS_WITH_CRS_NLS = {CONTACT_NOTES, CONTACT_BILLING_INFORMATION}

def _showContact(contactsManager, fields, displayFieldsList, j, jcount, FJQC):
  if FJQC.formatJSON:
    printLine(json.dumps(cleanJSON(fields, timeObjects=CONTACT_TIME_OBJECTS), ensure_ascii=False, sort_keys=True))
    return
  printEntity([Ent.CONTACT, fields[CONTACT_ID]], j, jcount)
  Ind.Increment()
  for key in contactsManager.CONTACT_NAME_PROPERTY_PRINT_ORDER:
    if displayFieldsList and key not in displayFieldsList:
      continue
    if key in fields:
      if key in CONTACT_TIME_OBJECTS:
        printKeyValueList([key, formatLocalTime(fields[key])])
      elif key not in CONTACT_FIELDS_WITH_CRS_NLS:
        printKeyValueList([key, fields[key]])
      else:
        printKeyValueWithCRsNLs(key, fields[key])
  for key in contactsManager.CONTACT_ARRAY_PROPERTY_PRINT_ORDER:
    if displayFieldsList and key not in displayFieldsList:
      continue
    if key in fields:
      keymap = contactsManager.CONTACT_ARRAY_PROPERTIES[key]
      printKeyValueList([key, None])
      Ind.Increment()
      for item in fields[key]:
        fn = item.get('label')
        if keymap['relMap']:
          if not fn:
            fn = keymap['relMap'].get(item['rel'], 'custom')
          printKeyValueList(['type', fn])
          Ind.Increment()
        if keymap['primary']:
          printKeyValueList(['rank', ['notprimary', 'primary'][item['primary'] == 'true']])
        value = item['value']
        if value is None:
          value = ''
        if key == CONTACT_IMS:
          printKeyValueList(['protocol', contactsManager.IM_REL_TO_PROTOCOL_MAP.get(item['protocol'], item['protocol'])])
          printKeyValueList([keymap['infoTitle'], value])
        elif key == CONTACT_ADDRESSES:
          printKeyValueWithCRsNLs(keymap['infoTitle'], value)
          for org_key in contactsManager.ADDRESS_FIELD_PRINT_ORDER:
            if item[org_key]:
              printKeyValueList([contactsManager.ADDRESS_FIELD_TO_ARGUMENT_MAP[org_key], item[org_key]])
        elif key == CONTACT_ORGANIZATIONS:
          printKeyValueList([keymap['infoTitle'], value])
          for org_key in contactsManager.ORGANIZATION_FIELD_PRINT_ORDER:
            if item[org_key]:
              printKeyValueList([contactsManager.ORGANIZATION_FIELD_TO_ARGUMENT_MAP[org_key], item[org_key]])
        elif key == CONTACT_USER_DEFINED_FIELDS:
          printKeyValueList([item.get('rel') or 'None', value])
        else:
          printKeyValueList([keymap['infoTitle'], value])
        if keymap['relMap']:
          Ind.Decrement()
      Ind.Decrement()
  Ind.Decrement()

def _getContactFieldsList(contactsManager, displayFieldsList):
  for field in _getFieldsList():
    if field in contactsManager.CONTACT_ARGUMENT_TO_PROPERTY_MAP:
      displayFieldsList.append(contactsManager.CONTACT_ARGUMENT_TO_PROPERTY_MAP[field])
    else:
      invalidChoiceExit(field, contactsManager.CONTACT_ARGUMENT_TO_PROPERTY_MAP, True)

# gam info contacts <ContactEntity>
#	[basic|full]
#	[fields <ContactFieldNameList>] [formatjson]
def doInfoDomainContacts():
  entityType = Ent.DOMAIN
  contactsManager = ContactsManager()
  entityList = getEntityList(Cmd.OB_CONTACT_ENTITY)
  contactQuery = _initContactQueryAttributes()
  FJQC = FormatJSONQuoteChar()
  displayFieldsList = []
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg in CONTACTS_PROJECTION_CHOICE_MAP:
      contactQuery['projection'] = CONTACTS_PROJECTION_CHOICE_MAP[myarg]
    elif myarg == 'fields':
      _getContactFieldsList(contactsManager, displayFieldsList)
    else:
      FJQC.GetFormatJSON(myarg)
  user, contactsObject = getContactsObject()
  j = 0
  jcount = len(entityList)
  if not FJQC.formatJSON:
    entityPerformActionNumItems([entityType, user], jcount, Ent.CONTACT)
  if jcount == 0:
    setSysExitRC(NO_ENTITIES_FOUND_RC)
    return
  Ind.Increment()
  for contact in entityList:
    j += 1
    try:
      contactId = normalizeContactId(contact)
      contact = callGData(contactsObject, 'GetContact',
                          bailOnInternalServerError=True,
                          throwErrors=[GDATA.NOT_FOUND, GDATA.BAD_REQUEST, GDATA.SERVICE_NOT_APPLICABLE,
                                       GDATA.FORBIDDEN, GDATA.NOT_IMPLEMENTED, GDATA.INTERNAL_SERVER_ERROR],
                          retryErrors=[GDATA.INTERNAL_SERVER_ERROR],
                          uri=contactsObject.GetContactFeedUri(contact_list=user, contactId=contactId, projection=contactQuery['projection']))
      fields = contactsManager.ContactToFields(contact)
      _showContact(contactsManager, fields, displayFieldsList, j, jcount, FJQC)
    except (GDATA.notFound, GDATA.badRequest, GDATA.forbidden, GDATA.notImplemented, GDATA.internalServerError) as e:
      entityActionFailedWarning([entityType, user, Ent.CONTACT, contactId], str(e), j, jcount)
    except GDATA.serviceNotApplicable:
      entityUnknownWarning(entityType, user)
      break
  Ind.Decrement()

# gam print contacts [todrive <ToDriveAttribute>*] <ContactSelection>
#	[basic|full|countsonly] [showdeleted] [orderby <ContactOrderByFieldName> [ascending|descending]]
#	[fields <ContactFieldNameList>] [formatjson [quotechar <Character>]]
# gam show contacts <ContactSelection>
#	[basic|full|countsonly] [showdeleted] [orderby <ContactOrderByFieldName> [ascending|descending]]
#	[fields <ContactFieldNameList>] [formatjson]
def doPrintShowDomainContacts():
  entityType = Ent.DOMAIN
  entityTypeName = Ent.Singular(entityType)
  contactsManager = ContactsManager()
  csvPF = CSVPrintFile([entityTypeName, CONTACT_ID, CONTACT_NAME], 'sortall',
                       contactsManager.CONTACT_ARRAY_PROPERTY_PRINT_ORDER) if Act.csvFormat() else None
  FJQC = FormatJSONQuoteChar(csvPF)
  CSVTitle = 'Contacts'
  contactQuery = _initContactQueryAttributes()
  countsOnly = False
  displayFieldsList = []
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if csvPF and myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif myarg == 'fields':
      _getContactFieldsList(contactsManager, displayFieldsList)
    elif myarg == 'countsonly':
      countsOnly = True
      contactQuery['projection'] = CONTACTS_PROJECTION_CHOICE_MAP['basic']
      if csvPF:
        csvPF.SetTitles([entityTypeName, CSVTitle])
    elif _getContactQueryAttributes(contactQuery, myarg, 0, True):
      pass
    else:
      FJQC.GetFormatJSONQuoteChar(myarg, True)
  user, contactsObject = getContactsObject()
  contacts = queryContacts(contactsObject, contactQuery)
  if countsOnly:
    jcount = countLocalContactSelects(contactsManager, contacts, contactQuery)
    if csvPF:
      csvPF.WriteRowTitles({entityTypeName: user, CSVTitle: jcount})
    else:
      printEntityKVList([entityType, user], [CSVTitle, jcount])
  elif contacts is not None:
    jcount = len(contacts)
    if not csvPF:
      if not FJQC.formatJSON:
        entityPerformActionModifierNumItems([entityType, user], Msg.MAXIMUM_OF, jcount, Ent.CONTACT)
      Ind.Increment()
      j = 0
      for contact in contacts:
        j += 1
        fields = contactsManager.ContactToFields(contact)
        if not localContactSelects(contactsManager, contactQuery, fields):
          continue
        _showContact(contactsManager, fields, displayFieldsList, j, jcount, FJQC)
      Ind.Decrement()
    elif contacts:
      for contact in contacts:
        fields = contactsManager.ContactToFields(contact)
        if not localContactSelects(contactsManager, contactQuery, fields):
          continue
        contactRow = {entityTypeName: user, CONTACT_ID: fields[CONTACT_ID]}
        for key in contactsManager.CONTACT_NAME_PROPERTY_PRINT_ORDER:
          if displayFieldsList and key not in displayFieldsList:
            continue
          if key in fields:
            if key == CONTACT_UPDATED:
              contactRow[key] = formatLocalTime(fields[key])
            elif key not in (CONTACT_NOTES, CONTACT_BILLING_INFORMATION):
              contactRow[key] = fields[key]
            else:
              contactRow[key] = escapeCRsNLs(fields[key])
        for key in contactsManager.CONTACT_ARRAY_PROPERTY_PRINT_ORDER:
          if displayFieldsList and key not in displayFieldsList:
            continue
          if key in fields:
            keymap = contactsManager.CONTACT_ARRAY_PROPERTIES[key]
            j = 0
            contactRow[f'{key}{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{j}{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}count'] = len(fields[key])
            for item in fields[key]:
              j += 1
              fn = f'{key}{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}{j}{GC.Values[GC.CSV_OUTPUT_SUBFIELD_DELIMITER]}'
              fnt = item.get('label')
              if fnt:
                contactRow[fn+'type'] = fnt
              elif keymap['relMap']:
                contactRow[fn+'type'] = keymap['relMap'].get(item['rel'], 'custom')
              if keymap['primary']:
                contactRow[fn+'rank'] = 'primary' if item['primary'] == 'true' else 'notprimary'
              value = item['value']
              if value is None:
                value = ''
              if key == CONTACT_IMS:
                contactRow[fn+'protocol'] = contactsManager.IM_REL_TO_PROTOCOL_MAP.get(item['protocol'], item['protocol'])
                contactRow[fn+keymap['infoTitle']] = value
              elif key == CONTACT_ADDRESSES:
                contactRow[fn+keymap['infoTitle']] = escapeCRsNLs(value)
                for org_key in contactsManager.ADDRESS_FIELD_PRINT_ORDER:
                  if item[org_key]:
                    contactRow[fn+contactsManager.ADDRESS_FIELD_TO_ARGUMENT_MAP[org_key]] = escapeCRsNLs(item[org_key])
              elif key == CONTACT_ORGANIZATIONS:
                contactRow[fn+keymap['infoTitle']] = value
                for org_key in contactsManager.ORGANIZATION_FIELD_PRINT_ORDER:
                  if item[org_key]:
                    contactRow[fn+contactsManager.ORGANIZATION_FIELD_TO_ARGUMENT_MAP[org_key]] = item[org_key]
              elif key == CONTACT_USER_DEFINED_FIELDS:
                contactRow[fn+'type'] = item.get('rel') or 'None'
                contactRow[fn+keymap['infoTitle']] = value
              else:
                contactRow[fn+keymap['infoTitle']] = value
        if not FJQC.formatJSON:
          csvPF.WriteRowTitles(contactRow)
        elif csvPF.CheckRowTitles(contactRow):
          csvPF.WriteRowNoFilter({entityTypeName: user, CONTACT_ID: fields[CONTACT_ID],
                                  CONTACT_NAME: fields.get(CONTACT_NAME, ''),
                                  'JSON': json.dumps(cleanJSON(fields, timeObjects=CONTACT_TIME_OBJECTS),
                                                     ensure_ascii=False, sort_keys=True)})
  if csvPF:
    csvPF.writeCSVfile(CSVTitle)

# Prople commands utilities
#
def normalizePeopleResourceName(resourceName):
  if resourceName.startswith('people/'):
    return resourceName
  return f'people/{resourceName}'

def normalizeContactGroupResourceName(resourceName):
  if resourceName.startswith('contactGroups/'):
    return resourceName
  return f'contactGroups/{resourceName}'

def normalizeOtherContactsResourceName(resourceName):
  if resourceName.startswith('otherContacts/'):
    return resourceName
  return f'otherContacts/{resourceName}'

PEOPLE_JSON = 'JSON'

PEOPLE_ADDRESSES = 'addresses'
PEOPLE_BIOGRAPHIES = 'biographies'
PEOPLE_BIRTHDAYS = 'birthdays'
PEOPLE_CALENDAR_URLS = 'calendarUrls'
PEOPLE_CLIENT_DATA = 'clientData'
PEOPLE_COVER_PHOTOS = 'coverPhotos'
PEOPLE_EMAIL_ADDRESSES = 'emailAddresses'
PEOPLE_EVENTS = 'events'
PEOPLE_EXTERNAL_IDS = 'externalIds'
PEOPLE_FILE_ASES = 'fileAses'
PEOPLE_GENDERS = 'genders'
PEOPLE_IM_CLIENTS = 'imClients'
PEOPLE_INTERESTS = 'interests'
PEOPLE_LOCALES = 'locales'
PEOPLE_LOCATIONS = 'locations'
PEOPLE_MEMBERSHIPS = 'memberships'
PEOPLE_METADATA = 'metadata'
PEOPLE_MISC_KEYWORDS = 'miscKeywords'
PEOPLE_MISC_KEYWORDS_BILLING_INFORMATION = PEOPLE_MISC_KEYWORDS+'.OUTLOOK_BILLING_INFORMATION'
PEOPLE_MISC_KEYWORDS_DIRECTORY_SERVER = PEOPLE_MISC_KEYWORDS+'.OUTLOOK_DIRECTORY_SERVER'
PEOPLE_MISC_KEYWORDS_JOT = PEOPLE_MISC_KEYWORDS+'.jot'
PEOPLE_MISC_KEYWORDS_MILEAGE = PEOPLE_MISC_KEYWORDS+'.OUTLOOK_MILEAGE'
PEOPLE_MISC_KEYWORDS_PRIORITY = PEOPLE_MISC_KEYWORDS+'.OUTLOOK_PRIORITY'
PEOPLE_MISC_KEYWORDS_SENSITIVITY = PEOPLE_MISC_KEYWORDS+'.OUTLOOK_SENSITIVITY'
PEOPLE_MISC_KEYWORDS_SUBJECT = PEOPLE_MISC_KEYWORDS+'.OUTLOOK_SUBJECT'
PEOPLE_NAMES = 'names'
PEOPLE_NAMES_FAMILY_NAME = PEOPLE_NAMES+'.familyName'
PEOPLE_NAMES_GIVEN_NAME = PEOPLE_NAMES+'.givenName'
PEOPLE_NAMES_HONORIFIC_PREFIX = PEOPLE_NAMES+'.honorificPrefix'
PEOPLE_NAMES_HONORIFIC_SUFFIX = PEOPLE_NAMES+'.honorificSuffix'
PEOPLE_NAMES_MIDDLE_NAME = PEOPLE_NAMES+'.middleName'
PEOPLE_NAMES_PHONETIC_FAMILY_NAME = PEOPLE_NAMES+'.phoneticFamilyName'
PEOPLE_NAMES_PHONETIC_GIVEN_NAME = PEOPLE_NAMES+'.phoneticGivenName'
PEOPLE_NAMES_PHONETIC_HONORIFIC_PREFIX = PEOPLE_NAMES+'.phoneticHonorificPrefix'
PEOPLE_NAMES_PHONETIC_HONORIFIC_SUFFIX = PEOPLE_NAMES+'.phoneticHonorificSuffix'
PEOPLE_NAMES_PHONETIC_MIDDLE_NAME = PEOPLE_NAMES+'.phoneticMiddleName'
PEOPLE_NAMES_UNSTRUCTURED_NAME = PEOPLE_NAMES+'.unstructuredName'
PEOPLE_NICKNAMES = 'nicknames'
PEOPLE_NICKNAMES_INITIALS = PEOPLE_NICKNAMES+'.INITIALS'
PEOPLE_NICKNAMES_MAIDENNAME = PEOPLE_NICKNAMES+'.MAIDEN_NAME'
PEOPLE_NICKNAMES_NICKNAME = PEOPLE_NICKNAMES+'.DEFAULT'
PEOPLE_NICKNAMES_SHORTNAME = PEOPLE_NICKNAMES+'.SHORT_NAME'
PEOPLE_OCCUPATIONS = 'occupations'
PEOPLE_ORGANIZATIONS = 'organizations'
PEOPLE_PHONE_NUMBERS = 'phoneNumbers'
PEOPLE_PHOTOS = 'photos'
PEOPLE_RELATIONS = 'relations'
PEOPLE_SIP_ADDRESSES = 'sipAddresses'
PEOPLE_SKILLS = 'skills'
PEOPLE_UPDATE_TIME = 'updateTime'
PEOPLE_URLS = 'urls'
PEOPLE_USER_DEFINED = 'userDefined'

PEOPLE_GROUPS = 'ContactGroups'
PEOPLE_GROUPS_LIST = 'ContactGroupsList'
PEOPLE_ADD_GROUPS = 'ContactAddGroups'
PEOPLE_ADD_GROUPS_LIST = 'ContactAddGroupsList'
PEOPLE_REMOVE_GROUPS = 'ContactRemoveGroups'
PEOPLE_REMOVE_GROUPS_LIST = 'ContactRemoveGroupsList'

PEOPLE_GROUP_NAME = 'name'
PEOPLE_GROUP_CLIENT_DATA = 'clientData'
#
class PeopleManager():
  PEOPLE_ARGUMENT_TO_PROPERTY_MAP = {
    'json': PEOPLE_JSON,
    'additionalname': PEOPLE_NAMES_MIDDLE_NAME,
    'address': PEOPLE_ADDRESSES,
    'addresses': PEOPLE_ADDRESSES,
    'billinginfo': PEOPLE_MISC_KEYWORDS_BILLING_INFORMATION,
    'biography': PEOPLE_BIOGRAPHIES,
    'biographies': PEOPLE_BIOGRAPHIES,
    'birthday': PEOPLE_BIRTHDAYS,
    'birthdays': PEOPLE_BIRTHDAYS,
    'calendar': PEOPLE_CALENDAR_URLS,
    'calendars': PEOPLE_CALENDAR_URLS,
    'clientdata': PEOPLE_CLIENT_DATA,
#    'coverphoto': PEOPLE_COVER_PHOTOS,
#    'coverphotos': PEOPLE_COVER_PHOTOS,
    'directoryserver': PEOPLE_MISC_KEYWORDS_DIRECTORY_SERVER,
    'email': PEOPLE_EMAIL_ADDRESSES,
    'emails': PEOPLE_EMAIL_ADDRESSES,
    'emailadresses': PEOPLE_EMAIL_ADDRESSES,
    'event': PEOPLE_EVENTS,
    'events': PEOPLE_EVENTS,
    'externalid': PEOPLE_EXTERNAL_IDS,
    'externalids': PEOPLE_EXTERNAL_IDS,
    'familyname': PEOPLE_NAMES_FAMILY_NAME,
    'fileas': PEOPLE_FILE_ASES,
    'firstname': PEOPLE_NAMES_GIVEN_NAME,
    'gender': PEOPLE_GENDERS,
    'genders': PEOPLE_GENDERS,
    'givenname': PEOPLE_NAMES_GIVEN_NAME,
    'hobby': PEOPLE_INTERESTS,
    'hobbies': PEOPLE_INTERESTS,
    'im': PEOPLE_IM_CLIENTS,
    'ims': PEOPLE_IM_CLIENTS,
    'imclients': 'imClients',
    'initials': PEOPLE_NICKNAMES_INITIALS,
    'interests': PEOPLE_INTERESTS,
    'jot': PEOPLE_MISC_KEYWORDS_JOT,
    'jots': PEOPLE_MISC_KEYWORDS_JOT,
    'language': PEOPLE_LOCALES,
    'lastname': PEOPLE_NAMES_FAMILY_NAME,
    'locale': PEOPLE_LOCALES,
    'location': PEOPLE_LOCATIONS,
    'locations': PEOPLE_LOCATIONS,
    'maidenname': PEOPLE_NICKNAMES_MAIDENNAME,
    'middlename': PEOPLE_NAMES_MIDDLE_NAME,
    'mileage': PEOPLE_MISC_KEYWORDS_MILEAGE,
    'misckeywords': PEOPLE_MISC_KEYWORDS,
    'name': PEOPLE_NAMES_UNSTRUCTURED_NAME,
    'names': PEOPLE_NAMES_UNSTRUCTURED_NAME,
    'nickname': PEOPLE_NICKNAMES_NICKNAME,
    'nicknames': PEOPLE_NICKNAMES_NICKNAME,
    'note': PEOPLE_BIOGRAPHIES,
    'notes': PEOPLE_BIOGRAPHIES,
    'occupation': PEOPLE_OCCUPATIONS,
    'occupations': PEOPLE_OCCUPATIONS,
    'organization': PEOPLE_ORGANIZATIONS,
    'organizations': PEOPLE_ORGANIZATIONS,
    'organisation': PEOPLE_ORGANIZATIONS,
    'organisations': PEOPLE_ORGANIZATIONS,
    'phone': PEOPLE_PHONE_NUMBERS,
    'phones': PEOPLE_PHONE_NUMBERS,
    'phonenumbers': PEOPLE_PHONE_NUMBERS,
#    'photo': PEOPLE_PHOTOS,
#    'photos': PEOPLE_PHOTOS,
    'prefix': PEOPLE_NAMES_HONORIFIC_PREFIX,
    'priority': PEOPLE_MISC_KEYWORDS_PRIORITY,
    'relation': PEOPLE_RELATIONS,
    'relations': PEOPLE_RELATIONS,
    'sensitivity': PEOPLE_MISC_KEYWORDS_SENSITIVITY,
    'shortname': PEOPLE_NICKNAMES_SHORTNAME,
    'sipaddress': PEOPLE_SIP_ADDRESSES,
    'sipaddresses': PEOPLE_SIP_ADDRESSES,
    'skills': PEOPLE_SKILLS,
    'subject': PEOPLE_MISC_KEYWORDS_SUBJECT,
    'suffix': PEOPLE_NAMES_HONORIFIC_SUFFIX,
    'url': PEOPLE_URLS,
    'urls': PEOPLE_URLS,
    'userdefined': PEOPLE_USER_DEFINED,
    'userdefinedfield': PEOPLE_USER_DEFINED,
    'userdefinedfields': PEOPLE_USER_DEFINED,
    'website': PEOPLE_URLS,
    'websites': PEOPLE_URLS,
    'contactgroup': PEOPLE_GROUPS,
    'contactgroups': PEOPLE_GROUPS,
    'addcontactgroup': PEOPLE_ADD_GROUPS,
    'addcontactgroups': PEOPLE_ADD_GROUPS,
    'removecontactgroup': PEOPLE_REMOVE_GROUPS,
    'removecontactgroups': PEOPLE_REMOVE_GROUPS,
    }

  ADDRESS_ARGUMENT_TO_FIELD_MAP = {
    'formatted': 'formattedValue',
    'unstructured': 'formattedValue',
    'pobox': 'poBox',
    'street': 'streetAddress',
    'streetaddress': 'streetAddress',
    'extended': 'extendedAddress',
    'neighborhood': 'extendedAddress',
    'city': 'city',
    'locality': 'city',
    'region': 'region',
    'postalcode': 'postalCode',
    'country': 'country',
    'countrycode': 'countryCode',
    }

  JOT_TYPE_MAP = {
    'work': 'WORK',
    'home': 'HOME',
    'other': 'OTHER',
    'keyword': 'KEYWORD',
    'keywords': 'KEYWORD',
    'user': 'USER',
    }

  IM_PROTOCOLS = {
    'aim': 'aim',
    'googletalk': 'googleTalk',
    'gtalk': 'googleTalk',
    'icq': 'icq',
    'jabber': 'jabber',
    'msn': 'msn',
    'netmeeting': 'netMeeting',
    'qq': 'qq',
    'skype': 'skype',
    'xmpp': 'jabber',
    'yahoo': 'yahoo',
    }

  ORGANIZATION_ARGUMENT_TO_FIELD_MAP = {
    'startdate': 'startDate',
    'enddate': 'endDate',
    'current': 'current',
    'phoneticname': 'phoneticName',
    'title': 'title',
    'department': 'department',
    'jobdescription': 'jobDescription',
    'symbol': 'symbol',
    'domain': 'domain',
    'location': 'location',
    }

# Fields with a key and value
  KEY_VALUE_FIELDS = {
    PEOPLE_CLIENT_DATA,
    PEOPLE_USER_DEFINED,
    }

# Fields with a type and value
  TYPE_VALUE_FIELDS = {
    PEOPLE_EVENTS: {
      'anniversary': 'anniversary',
      'other': 'other',
      },
    PEOPLE_EXTERNAL_IDS: {
      'account': 'account',
      'customer': 'customer',
      'loginid': 'loginId',
      'network': 'network',
      'organization': 'organization',
      'organisation': 'organization',
      },
    PEOPLE_RELATIONS: {
      'spouse' : 'spouse',
      'child' : 'child',
      'mother' : 'mother',
      'father' : 'father',
      'parent' : 'parent',
      'brother' : 'brother',
      'sister' : 'sister',
      'friend' : 'friend',
      'relative' : 'relative',
      'domesticpartner' : 'domesticPartner',
      'manager' : 'manager',
      'assistant' : 'assistant',
      'referredby' : 'referredBy',
      'partner' : 'partner',
      },
    PEOPLE_SIP_ADDRESSES: {
      'work': 'work',
      'home': 'home',
      'other': 'other',
      'mobile': 'mobile',
      },
    }

# Fields with a type, value and primary|notprimary; some fields may have additional arguments
  TYPE_VALUE_PNP_FIELDS = {
    PEOPLE_ADDRESSES: {
      'work': 'work',
      'home': 'home',
      'other': 'other',
      },
    PEOPLE_CALENDAR_URLS: {
      'work': 'work',
      'home': 'home',
      'freebusy': 'freeBusy',
      },
    PEOPLE_EMAIL_ADDRESSES: {
      'work': 'work',
      'home': 'home',
      'other': 'other',
      },
    PEOPLE_IM_CLIENTS: {
      'work': 'work',
      'home': 'home',
      'other': 'other',
      },
    PEOPLE_ORGANIZATIONS: {
      'school': 'school',
      'work': 'work',
      'other': 'other',
      },
    PEOPLE_PHONE_NUMBERS: {
      'work': 'work',
      'home': 'home',
      'other': 'other',
      'googlevoice': 'googleVoice',
      'fax': 'homeFax',
      'homefax': 'homeFax',
      'workfax': 'workFax',
      'otherfax': 'otherFax',
      'main': 'main',
      'company_main': 'companyMain',
      'assistant': 'assistant',
      'mobile': 'mobile',
      'workmobile': 'workMobile',
      'pager': 'pager',
      'workpager': 'workPager',
      'car': 'car',
      'radio': 'radio',
      'callback': 'callback',
      'isdn': 'isdn',
      'telex': 'telex',
      'ttytdd': 'TTY_TDD',
      },
    PEOPLE_SIP_ADDRESSES: {
      'work': 'work',
      'home': 'home',
      'mobile': 'mobile',
      'other': 'other',
      },
    PEOPLE_URLS: {
      'appinstallpage': 'appInstallPage',
      'blog': 'blog',
      'ftp': 'ftp',
      'home': 'homePage',
      'homepage': 'homePage',
      'other': 'other',
      'profile': 'profile',
      'reservations': 'reservations',
      'resume': 'resume',
      'work': 'work',
      }
    }

# Fields that allow an empty type
  EMPTY_TYPE_ALLOWED_FIELDS = {PEOPLE_ADDRESSES, PEOPLE_EMAIL_ADDRESSES, PEOPLE_PHONE_NUMBERS, PEOPLE_URLS}

# Fields with just a URL
#  URL_FIELDS = {
#    PEOPLE_COVER_PHOTOS,
#    PEOPLE_PHOTOS,
#    }

# Fields with a single value
  SINGLE_VALUE_FIELDS = {
    PEOPLE_FILE_ASES,
    }

# Fields with multiple values
  MULTI_VALUE_FIELDS = {
    PEOPLE_INTERESTS,
    PEOPLE_LOCALES,
    PEOPLE_LOCATIONS,
    PEOPLE_OCCUPATIONS,
    PEOPLE_SKILLS,
    }

  @staticmethod
  def GetPersonFields(entityType, allowAddRemove, parameters=None):
    person = {}
    contactGroupsLists = {
      PEOPLE_GROUPS_LIST: [],
      PEOPLE_ADD_GROUPS_LIST: [],
      PEOPLE_REMOVE_GROUPS_LIST: []
      }
    locations = {'primary': None}

    def CheckClearPersonField(fieldName):
      if checkArgumentPresent(Cmd.CLEAR_NONE_ARGUMENT):
        person.pop(fieldName, None)
        person[fieldName] = []
        return True
      return False

    def GetSingleFieldEntry(fieldName):
      person.setdefault(fieldName, [])
      if not person[fieldName]:
        person[fieldName].append({})
      return person[fieldName][0]

    def InitArrayFieldEntry(choices, typeMinLen=1):
      entry = {'metadata': {'primary': False}}
      if choices is not None:
        ftype = getChoice(choices, mapChoice=True, defaultChoice=None)
        if ftype:
          entry['type'] = ftype
        else:
          entry['type'] = getString(Cmd.OB_STRING, minLen=typeMinLen)
      return entry

    def GetMultiFieldEntry(fieldName):
      person.setdefault(fieldName, [])
      person[fieldName].append({})
      return person[fieldName][-1]

    def getDate(entry, fieldName):
      event = getYYYYMMDD(minLen=0, returnDateTime=True)
      if event:
        entry[fieldName] = {'year': event.year, 'month': event.month, 'day': event.day}

    def PrimaryNotPrimary(pnp, entry):
      if pnp == 'notprimary':
        entry['metadata']['primary'] = 'false'
        return True
      if pnp == 'primary':
        entry['metadata']['primary'] = 'true'
        locations['primary'] = Cmd.Location()
        return True
      return False

    def GetPrimaryNotPrimaryChoice(entry):
      pnp = getChoice({'primary': True, 'notprimary': False}, mapChoice=True)
      entry['metadata']['primary'] = pnp
      if pnp:
        locations['primary'] = Cmd.Location()

    def AppendArrayEntryToFields(fieldName, entry, checkBlankField=None):
      person.setdefault(fieldName, [])
      if checkBlankField is None or entry[checkBlankField]:
        if entry.get('metadata', {}).get('primary', False):
          for centry in person[fieldName][1:]:
            if centry.get('metadata', {}).get('primary', False):
              Cmd.SetLocation(locations['primary']-1)
              usageErrorExit(Msg.MULTIPLE_ITEMS_MARKED_PRIMARY.format(fieldName))
        person[fieldName].append(entry)

    while Cmd.ArgumentsRemaining():
      if parameters is not None:
        if _getCreateContactReturnOptions(parameters):
          continue
        Cmd.Backup()
      locations['fieldName'] = Cmd.Location()
      fieldName = getChoice(PeopleManager.PEOPLE_ARGUMENT_TO_PROPERTY_MAP, mapChoice=True)
      if '.' in fieldName:
        fieldName, subFieldName = fieldName.split('.')
      if fieldName == PEOPLE_ADDRESSES:
        if CheckClearPersonField(fieldName):
          continue
        entry = InitArrayFieldEntry(PeopleManager.TYPE_VALUE_PNP_FIELDS[fieldName], typeMinLen=0)
        while Cmd.ArgumentsRemaining():
          argument = getArgument()
          if argument in PeopleManager.ADDRESS_ARGUMENT_TO_FIELD_MAP:
            subFieldName = PeopleManager.ADDRESS_ARGUMENT_TO_FIELD_MAP[argument]
            value = getString(Cmd.OB_STRING, minLen=0)
            if value: ### Delete?
              entry[subFieldName] = value.replace('\\n', '\n')
          elif PrimaryNotPrimary(argument, entry):
            break
          else:
            unknownArgumentExit()
        AppendArrayEntryToFields(fieldName, entry, None)
      elif fieldName == PEOPLE_BIRTHDAYS:
        entry = GetSingleFieldEntry(fieldName)
        getDate(entry, 'date')
      elif fieldName == PEOPLE_BIOGRAPHIES:
        entry = GetSingleFieldEntry(fieldName)
        text, _, html = getStringWithCRsNLsOrFile()
        entry['value' ] = text
        entry['contentType'] = ['TEXT_PLAIN', 'TEXT_HTML'][html]
      elif fieldName == PEOPLE_GENDERS:
        entry = GetSingleFieldEntry(fieldName)
        entry['value'] = getString(Cmd.OB_STRING, minLen=0)
      elif fieldName == PEOPLE_MISC_KEYWORDS:
        entry = GetMultiFieldEntry(fieldName)
        if subFieldName == 'jot':
          subFieldName = getChoice(PeopleManager.JOT_TYPE_MAP, mapChoice=True)
        entry['value'] = getString(Cmd.OB_STRING, minLen=0)
        entry['type'] = subFieldName
      elif fieldName == PEOPLE_NAMES:
        entry = GetSingleFieldEntry(fieldName)
        entry[subFieldName] = getString(Cmd.OB_STRING, minLen=0)
      elif fieldName == PEOPLE_NICKNAMES:
        entry = GetMultiFieldEntry(fieldName)
        entry['value'] = getString(Cmd.OB_STRING, minLen=0)
        entry['type'] = subFieldName
      elif fieldName == PEOPLE_ORGANIZATIONS:
        entry = InitArrayFieldEntry(PeopleManager.TYPE_VALUE_PNP_FIELDS[fieldName])
        entry['name'] = getString(Cmd.OB_STRING, minLen=0)
        while Cmd.ArgumentsRemaining():
          argument = getArgument()
          if argument in PeopleManager.ORGANIZATION_ARGUMENT_TO_FIELD_MAP:
            subFieldName = PeopleManager.ORGANIZATION_ARGUMENT_TO_FIELD_MAP[argument]
            if subFieldName == 'current':
              entry[subFieldName] = getBoolean()
            elif subFieldName in {'startDate', 'endDate'}:
              getDate(entry, subFieldName)
            else:
              value = getString(Cmd.OB_STRING, minLen=0)
              if value: ### Delete?
                entry[subFieldName] = value
          elif PrimaryNotPrimary(argument, entry):
            break
          else:
            unknownArgumentExit()
        AppendArrayEntryToFields(fieldName, entry, None)
      elif fieldName in PeopleManager.KEY_VALUE_FIELDS:
        if CheckClearPersonField(fieldName):
          continue
        entry = InitArrayFieldEntry(None)
        entry['key'] = getString(Cmd.OB_STRING, minLen=1)
        entry['value'] = getString(Cmd.OB_STRING, minLen=0)
        AppendArrayEntryToFields(fieldName, entry, 'value')
      elif fieldName in PeopleManager.TYPE_VALUE_FIELDS:
        if CheckClearPersonField(fieldName):
          continue
        entry = InitArrayFieldEntry(PeopleManager.TYPE_VALUE_FIELDS[fieldName])
        if fieldName == PEOPLE_EVENTS:
          checkBlankField = 'date'
          getDate(entry, checkBlankField)
        elif fieldName == PEOPLE_RELATIONS:
          checkBlankField = 'type'
          entry['person'] = getString(Cmd.OB_STRING, minLen=0)
        else:
          checkBlankField = 'value'
          entry['value'] = getString(Cmd.OB_STRING, minLen=0)
        AppendArrayEntryToFields(fieldName, entry, checkBlankField)
      elif fieldName in PeopleManager.TYPE_VALUE_PNP_FIELDS:
        if CheckClearPersonField(fieldName):
          continue
        entry = InitArrayFieldEntry(PeopleManager.TYPE_VALUE_PNP_FIELDS[fieldName],
                                    typeMinLen=0 if fieldName in PeopleManager.EMPTY_TYPE_ALLOWED_FIELDS else 1)
        if fieldName == PEOPLE_IM_CLIENTS:
          checkBlankField = None
          entry['protocol'] = getChoice(PeopleManager.IM_PROTOCOLS, mapChoice=True)
          entry['username'] = getString(Cmd.OB_STRING, minLen=0)
        elif fieldName == PEOPLE_EMAIL_ADDRESSES:
          checkBlankField = 'value'
          entry[checkBlankField] = getString(Cmd.OB_STRING, minLen=0)
          if checkArgumentPresent(['displayname']):
            entry['displayName'] = getString(Cmd.OB_STRING, minLen=0)
        elif fieldName == PEOPLE_CALENDAR_URLS:
          checkBlankField = 'url'
          entry[checkBlankField] = getString(Cmd.OB_STRING, minLen=0)
        else:
          checkBlankField = 'value'
          entry[checkBlankField] = getString(Cmd.OB_STRING, minLen=0)
        GetPrimaryNotPrimaryChoice(entry)
        AppendArrayEntryToFields(fieldName, entry, checkBlankField)
      elif fieldName in PeopleManager.SINGLE_VALUE_FIELDS:
        entry = GetSingleFieldEntry(fieldName)
        entry['value'] = getString(Cmd.OB_STRING, minLen=0)
      elif fieldName in PeopleManager.MULTI_VALUE_FIELDS:
        if CheckClearPersonField(fieldName):
          continue
        entry = InitArrayFieldEntry(None)
        entry['value'] = getString(Cmd.OB_STRING, minLen=0)
        AppendArrayEntryToFields(fieldName, entry, 'value')
#      elif fieldName in PeopleManager.URL_FIELDS:
#        if CheckClearPersonField(fieldName):
#          continue
#        entry = InitArrayFieldEntry(None)
#        entry['url'] = getString(Cmd.OB_STRING, minLen=0)
#        entry['default'] = False
#        AppendArrayEntryToFields(fieldName, entry, 'url')
      elif fieldName == PEOPLE_GROUPS:
        if entityType != Ent.USER:
          Cmd.Backup()
          unknownArgumentExit()
        contactGroupsLists[PEOPLE_GROUPS_LIST].append(getString(Cmd.OB_STRING))
      elif fieldName == PEOPLE_ADD_GROUPS:
        if not allowAddRemove:
          unknownArgumentExit()
        if entityType != Ent.USER:
          Cmd.Backup()
          unknownArgumentExit()
        contactGroupsLists[PEOPLE_ADD_GROUPS_LIST].append(getString(Cmd.OB_STRING))
      elif fieldName == PEOPLE_REMOVE_GROUPS:
        if not allowAddRemove:
          unknownArgumentExit()
        if entityType != Ent.USER:
          Cmd.Backup()
          unknownArgumentExit()
        contactGroupsLists[PEOPLE_REMOVE_GROUPS_LIST].append(getString(Cmd.OB_STRING))
      elif fieldName == PEOPLE_JSON:
        jsonData = getJSON(['resourceName', 'etag', 'metadata', PEOPLE_COVER_PHOTOS, PEOPLE_PHOTOS, PEOPLE_UPDATE_TIME])
        for membership in jsonData.pop('memberships', []):
          contactGroupName = membership.get('contactGroupMembership', {}).get('contactGroupName', '')
          if contactGroupName:
            contactGroupsLists[PEOPLE_GROUPS_LIST].append(contactGroupName)
        newClientData = []
        for clientData in jsonData.pop('clientData', []):
          if clientData['key'] not in {'ContactId', 'CtsContactHash'}:
            newClientData.append({'key': clientData['key'], 'value': clientData['value']})
        if newClientData:
          person.setdefault(PEOPLE_CLIENT_DATA, [])
          person[PEOPLE_CLIENT_DATA].extend(newClientData)
        person.update(jsonData)
    return (person, set(person.keys()), contactGroupsLists)

  PEOPLE_GROUP_ARGUMENT_TO_PROPERTY_MAP = {
    'json': PEOPLE_JSON,
    'name': PEOPLE_GROUP_NAME,
    'clientdata': PEOPLE_GROUP_CLIENT_DATA,
    }

  @staticmethod
  def AddContactGroupsToContact(contactEntry, contactGroupsList):
    contactEntry[PEOPLE_MEMBERSHIPS] = []
    for groupId in contactGroupsList:
      if groupId != 'clear':
        contactEntry[PEOPLE_MEMBERSHIPS].append({'contactGroupMembership': {'contactGroupResourceName': groupId}})
      else:
        contactEntry[PEOPLE_MEMBERSHIPS] = []

  @staticmethod
  def AddFilteredContactGroupsToContact(contactEntry, contactGroupsList, contactRemoveGroupsList):
    contactEntry[PEOPLE_MEMBERSHIPS] = []
    for groupId in contactGroupsList:
      if groupId not in contactRemoveGroupsList:
        contactEntry[PEOPLE_MEMBERSHIPS].append({'contactGroupMembership': {'contactGroupResourceName': groupId}})

  @staticmethod
  def AddAdditionalContactGroupsToContact(contactEntry, contactGroupsList):
    for groupId in contactGroupsList:
      contactEntry[PEOPLE_MEMBERSHIPS].append({'contactGroupMembership': {'contactGroupResourceName': groupId}})

  @staticmethod
  def GetContactGroupFields(parameters=None):
    contactGroup = {}
    while Cmd.ArgumentsRemaining():
      if parameters is not None:
        if _getCreateContactReturnOptions(parameters):
          continue
        Cmd.Backup()
      fieldName = getChoice(PeopleManager.PEOPLE_GROUP_ARGUMENT_TO_PROPERTY_MAP, mapChoice=True)
      if fieldName == PEOPLE_GROUP_NAME:
        contactGroup[PEOPLE_GROUP_NAME] = getString(Cmd.OB_STRING)
      elif fieldName == PEOPLE_GROUP_CLIENT_DATA:
        entry = {}
        entry['key'] = getString(Cmd.OB_STRING, minLen=1)
        entry['value'] = getString(Cmd.OB_STRING, minLen=0)
        if entry['value']:
          contactGroup.setdefault(fieldName, [])
          contactGroup[fieldName].append(entry)
      elif fieldName == PEOPLE_JSON:
        jsonData = getJSON(['resourceName', 'etag', 'metadata', 'formattedName', 'memberResourceNames',  'memberCount'])
        if jsonData.get('groupType', '') != 'SYSTEM_CONTACT_GROUP':
          contactGroup[PEOPLE_GROUP_NAME] = jsonData['name']
    return (contactGroup, ','.join(contactGroup.keys()))

PEOPLE_DIRECTORY_SOURCES_CHOICE_MAP = {
  'contact': 'DIRECTORY_SOURCE_TYPE_DOMAIN_CONTACT',
  'contacts': 'DIRECTORY_SOURCE_TYPE_DOMAIN_CONTACT',
  'domaincontact': 'DIRECTORY_SOURCE_TYPE_DOMAIN_CONTACT',
  'comaincontacts': 'DIRECTORY_SOURCE_TYPE_DOMAIN_CONTACT',
  'profile': 'DIRECTORY_SOURCE_TYPE_DOMAIN_PROFILE',
  'profiles': 'DIRECTORY_SOURCE_TYPE_DOMAIN_PROFILE'
  }

PEOPLE_READ_SOURCES_CHOICE_MAP = {
  'contact': 'READ_SOURCE_TYPE_CONTACT',
  'contacts': 'READ_SOURCE_TYPE_CONTACT',
  'domaincontact': 'READ_SOURCE_TYPE_DOMAIN_CONTACT',
  'domaincontacts': 'READ_SOURCE_TYPE_DOMAIN_CONTACT',
  'profile': 'READ_SOURCE_TYPE_PROFILE',
  'profiles': 'READ_SOURCE_TYPE_PROFILE'
  }

PEOPLE_DIRECTORY_MERGE_SOURCES_CHOICE_MAP = {
  'contact': 'DIRECTORY_MERGE_SOURCE_TYPE_CONTACT',
  'contacts': 'DIRECTORY_MERGE_SOURCE_TYPE_CONTACT',
  }

def _initPeopleContactQueryAttributes(printShowCmd):
  return {'query': None, 'updateTime': None,
          'contactGroupSelect': None, 'contactGroupFilter': None, 'group': None, 'dropMemberships': False,
          'mainContacts': True, 'otherContacts': printShowCmd, 'emailMatchPattern': None, 'emailMatchType': None}

def _getPeopleContactQueryAttributes(contactQuery, myarg, entityType, unknownAction, printShowCmd):
  if myarg == 'query':
    contactQuery['query'] = getString(Cmd.OB_QUERY)
  elif myarg in {'contactgroup', 'selectcontactgroup'}:
    if entityType == Ent.USER:
      contactQuery['contactGroupSelect'] = getString(Cmd.OB_CONTACT_GROUP_ITEM)
      contactQuery['contactGroupFilter'] = None
      contactQuery['mainContacts'] = True
      contactQuery['otherContacts'] = False
    else:
      unknownArgumentExit()
  elif myarg == 'emailmatchpattern':
    contactQuery['emailMatchPattern'] = getREPattern(re.IGNORECASE)
  elif myarg == 'emailmatchtype':
    contactQuery['emailMatchType'] = getString(Cmd.OB_CONTACT_EMAIL_TYPE)
  elif myarg == 'updatedmin':
    deprecatedArgument(myarg)
    getYYYYMMDD()
  elif myarg == 'endquery':
    return False
  elif not printShowCmd:
    if unknownAction < 0:
      unknownArgumentExit()
    if unknownAction > 0:
      Cmd.Backup()
    return False
  elif myarg == 'filtercontactgroup':
    if entityType == Ent.USER:
      contactQuery['contactGroupFilter'] = getString(Cmd.OB_CONTACT_GROUP_ITEM)
      contactQuery['contactGroupSelect'] = None
      contactQuery['mainContacts'] = True
      contactQuery['otherContacts'] = False
    else:
      unknownArgumentExit()
  elif myarg in {'maincontacts', 'selectmaincontacts'}:
    if entityType == Ent.USER:
      contactQuery['contactGroupSelect'] = None
      contactQuery['contactGroupFilter'] = None
      contactQuery['mainContacts'] = True
      contactQuery['otherContacts'] = False
    else:
      unknownArgumentExit()
  elif myarg in {'othercontacts', 'selectothercontacts'}:
    if entityType == Ent.USER:
      contactQuery['contactGroupSelect'] = None
      contactQuery['contactGroupFilter'] = None
      contactQuery['mainContacts'] = False
      contactQuery['otherContacts'] = True
    else:
      unknownArgumentExit()
  elif myarg == 'orderby':
    getOrderBySortOrder(CONTACTS_ORDERBY_CHOICE_MAP, 'ascending', False)
    deprecatedArgument(myarg)
  elif myarg in CONTACTS_PROJECTION_CHOICE_MAP:
    deprecatedArgument(myarg)
  elif myarg == 'showdeleted':
    deprecatedArgument(myarg)
  else:
    if unknownAction < 0:
      unknownArgumentExit()
    if unknownAction > 0:
      Cmd.Backup()
    return False
  return True

PEOPLE_CONTACT_SELECT_ARGUMENTS = {
  'query', 'contactgroup', 'selectcontactgroup',
  'maincontacts', 'selectmaincontacts',
  'othercontacts', 'selectothercontacts',
  'emailmatchpattern', 'emailmatchtype',
  }
PEOPLE_CONTACT_DEPRECATED_SELECT_ARGUMENTS = {
  'orderby', 'basic', 'thin', 'full', 'showdeleted',
  }

def _getPeopleContactEntityList(entityType, unknownAction, noEntityArguments=None):
  contactQuery = _initPeopleContactQueryAttributes(False)
  if noEntityArguments is not None and (not Cmd.ArgumentsRemaining() or Cmd.PeekArgumentPresent(noEntityArguments)):
    # <PeopleResourceNameEntity>|<PeopleUserContactSelection> are optional in dedup|replacedomain contacts
    entityList = None
    queriedContacts = True
  elif Cmd.PeekArgumentPresent(PEOPLE_CONTACT_SELECT_ARGUMENTS.union(PEOPLE_CONTACT_DEPRECATED_SELECT_ARGUMENTS)):
    entityList = None
    queriedContacts = True
    while Cmd.ArgumentsRemaining():
      myarg = getArgument()
      if not _getPeopleContactQueryAttributes(contactQuery, myarg, entityType, unknownAction, False):
        break
  else:
    entityList = getEntityList(Cmd.OB_CONTACT_ENTITY)
    queriedContacts = False
    if unknownAction < 0:
      checkForExtraneousArguments()
  return (entityList, entityList if isinstance(entityList, dict) else None, contactQuery, queriedContacts)

def _initPeopleOtherContactQueryAttributes():
  return {'query': None,
          'otherContacts': True, 'emailMatchPattern': None, 'emailMatchType': None}

def _getPeopleOtherContactQueryAttributes(contactQuery, myarg, unknownAction):
  if myarg == 'query':
    contactQuery['query'] = getString(Cmd.OB_QUERY)
  elif myarg == 'emailmatchpattern':
    contactQuery['emailMatchPattern'] = getREPattern(re.IGNORECASE)
  elif myarg == 'emailmatchtype':
    contactQuery['emailMatchType'] = getString(Cmd.OB_CONTACT_EMAIL_TYPE)
  elif myarg == 'endquery':
    return False
  else:
    if unknownAction < 0:
      unknownArgumentExit()
    if unknownAction > 0:
      Cmd.Backup()
    return False
  return True

PEOPLE_OTHERCONTACT_SELECT_ARGUMENTS = {'query', 'emailmatchpattern', 'emailmatchtype'}

def _getPeopleOtherContactEntityList(unknownAction):
  contactQuery = _initPeopleOtherContactQueryAttributes()
  if Cmd.PeekArgumentPresent(PEOPLE_OTHERCONTACT_SELECT_ARGUMENTS):
    entityList = None
    queriedContacts = True
    while Cmd.ArgumentsRemaining():
      myarg = getArgument()
      if not _getPeopleOtherContactQueryAttributes(contactQuery, myarg, unknownAction):
        break
  else:
    entityList = getEntityList(Cmd.OB_CONTACT_ENTITY)
    queriedContacts = False
    if unknownAction < 0:
      checkForExtraneousArguments()
  return (entityList, entityList if isinstance(entityList, dict) else None, contactQuery, queriedContacts)

def _getPeopleOtherContacts(people, entityType, user, i=0, count=0):
  try:
    printGettingAllEntityItemsForWhom(Ent.OTHER_CONTACT, user, i, count)
    results = callGAPIpages(people.otherContacts(), 'list', 'otherContacts',
                            pageMessage=getPageMessageForWhom(),
                            throwReasons=GAPI.PEOPLE_ACCESS_THROW_REASONS,
                            retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                            pageSize=1000,
                            readMask='emailAddresses', fields='nextPageToken,otherContacts(etag,resourceName,emailAddresses(value,type))')
    otherContacts = {}
    for contact in results:
      resourceName = contact.pop('resourceName')
      otherContacts[resourceName] = contact
    return otherContacts
  except GAPI.permissionDenied as e:
    ClientAPIAccessDeniedExit(str(e))
  except (GAPI.serviceNotAvailable, GAPI.forbidden):
    entityUnknownWarning(entityType, user, i, count)
  return None

def queryPeopleContacts(people, contactQuery, fields, sortOrder, entityType, user, i=0, count=0):
  sources = [PEOPLE_READ_SOURCES_CHOICE_MAP['domaincontact' if entityType == Ent.DOMAIN else 'contact']]
  printGettingAllEntityItemsForWhom(Ent.PEOPLE_CONTACT, user, i, count, query=contactQuery['query'])
  pageMessage = getPageMessageForWhom()
  try:
# Contact group not selected
    if not contactQuery['contactGroupSelect']:
      if not contactQuery['query']:
        results = callGAPIpages(people.people().connections(), 'list', 'connections',
                                pageMessage=pageMessage,
                                throwReasons=GAPI.PEOPLE_ACCESS_THROW_REASONS,
                                retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                                pageSize=GC.Values[GC.PEOPLE_MAX_RESULTS],
                                resourceName='people/me', sources=sources, personFields=fields,
                                sortOrder=sortOrder, fields='nextPageToken,connections')
        if contactQuery['contactGroupFilter']:
          entityList = []
          for person in results:
            for membership in person.get(PEOPLE_MEMBERSHIPS, []):
              if membership.get('contactGroupMembership', {}).get('contactGroupResourceName', '') == contactQuery['group']:
                if contactQuery['dropMemberships']:
                  person.pop(PEOPLE_MEMBERSHIPS)
                entityList.append(person)
                break
        else:
          entityList = results
      else:
        results = callGAPI(people.people(), 'searchContacts',
                           throwReasons=GAPI.PEOPLE_ACCESS_THROW_REASONS,
                           sources=sources, readMask=fields, query=contactQuery['query'])
        entityList = [person['person'] for person in results.get('results', [])]
      totalItems = len(entityList)
# Contact group selected
    else:
      totalItems = callGAPI(people.contactGroups(), 'get',
                            throwReasons=GAPI.PEOPLE_ACCESS_THROW_REASONS,
                            resourceName=contactQuery['group'], groupFields='memberCount').get('memberCount', 0)
      entityList = []
      if totalItems > 0:
        results = callGAPI(people.contactGroups(), 'get',
                           throwReasons=GAPI.PEOPLE_ACCESS_THROW_REASONS,
                           resourceName=contactQuery['group'], maxMembers=totalItems, groupFields='name')
        for resourceName in results.get('memberResourceNames', []):
          result = callGAPI(people.people(), 'get',
                            retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                            resourceName=resourceName, sources=sources, personFields=fields)

          entityList.append(result)
    if pageMessage and (contactQuery['contactGroupSelect'] or contactQuery['contactGroupFilter'] or contactQuery['query']):
      showMessage = pageMessage.replace(TOTAL_ITEMS_MARKER, str(totalItems))
      writeGotMessage(showMessage.replace('{0}', str(Ent.Choose(Ent.PEOPLE_CONTACT, totalItems))))
    return entityList
  except (GAPI.permissionDenied, GAPI.failedPrecondition) as e:
    ClientAPIAccessDeniedExit(str(e))
  except (GAPI.serviceNotAvailable, GAPI.forbidden):
    entityUnknownWarning(entityType, user, i, count)
  return None

def queryPeopleOtherContacts(people, contactQuery, fields, entityType, user, i=0, count=0):
  sources = [PEOPLE_READ_SOURCES_CHOICE_MAP['contact']]
  printGettingAllEntityItemsForWhom(Ent.OTHER_CONTACT, user, i, count, query=contactQuery['query'])
  pageMessage = getPageMessageForWhom()
  try:
    if not contactQuery['query']:
      entityList = callGAPIpages(people.otherContacts(), 'list', 'otherContacts',
                                 pageMessage=pageMessage,
                                 throwReasons=GAPI.PEOPLE_ACCESS_THROW_REASONS,
                                 retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                                 pageSize=GC.Values[GC.PEOPLE_MAX_RESULTS],
                                 readMask=fields, fields='nextPageToken,otherContacts', sources=sources)
    else:
      results = callGAPI(people.otherContacts(), 'search',
                         throwReasons=GAPI.PEOPLE_ACCESS_THROW_REASONS,
                         pageSize=30, readMask=fields, query=contactQuery['query'])
      entityList = [person['person'] for person in results.get('results', [])]
      totalItems = len(entityList)
      if pageMessage:
        showMessage = pageMessage.replace(TOTAL_ITEMS_MARKER, str(totalItems))
        writeGotMessage(showMessage.replace('{0}', str(Ent.Choose(Ent.OTHER_CONTACT, totalItems))))
    return entityList
  except GAPI.permissionDenied as e:
    ClientAPIAccessDeniedExit(str(e))
  except GAPI.forbidden:
    userPeopleServiceNotEnabledWarning(user, i, count)
  except GAPI.serviceNotAvailable:
    entityUnknownWarning(entityType, user, i, count)
  return None

def getPeopleContactGroupsInfo(people, entityType, entityName, i, count):
  contactGroupIDs = {}
  contactGroupNames = {}
  try:
    groups = callGAPIpages(people.contactGroups(), 'list', 'contactGroups',
                           throwReasons=GAPI.PEOPLE_ACCESS_THROW_REASONS,
                           pageSize=GC.Values[GC.PEOPLE_MAX_RESULTS],
                           groupFields='name', fields='nextPageToken,contactGroups(resourceName,name,formattedName)')
    if groups:
      for group in groups:
        contactGroupIDs[group['resourceName']] = group['formattedName']
        contactGroupNames.setdefault(group['formattedName'], [])
        contactGroupNames[group['formattedName']].append(group['resourceName'])
        if group['formattedName'] != group['name']:
          contactGroupNames.setdefault(group['name'], [])
          contactGroupNames[group['name']].append(group['resourceName'])
  except GAPI.permissionDenied as e:
    ClientAPIAccessDeniedExit(str(e))
  except GAPI.forbidden:
    userPeopleServiceNotEnabledWarning(entityName, i, count)
    return (contactGroupIDs, False)
  except GAPI.serviceNotAvailable:
    entityUnknownWarning(entityType, entityName, i, count)
    return (contactGroupIDs, False)
  return (contactGroupIDs, contactGroupNames)

def validatePeopleContactGroup(people, contactGroupName,
                               contactGroupIDs, contactGroupNames, entityType, entityName, i, count):
  if not contactGroupNames:
    contactGroupIDs, contactGroupNames = getPeopleContactGroupsInfo(people, entityType, entityName, i, count)
    if contactGroupNames is False:
      return (None, contactGroupIDs, contactGroupNames)
  if contactGroupName == 'clear':
    return (contactGroupName, contactGroupIDs, contactGroupNames)
  cg = UID_PATTERN.match(contactGroupName)
  if cg:
    contactGroupName = cg.group(1)
    if contactGroupName in contactGroupIDs:
      return (contactGroupName, contactGroupIDs, contactGroupNames)
    normalizedContactGroupName = normalizeContactGroupResourceName(contactGroupName)
    if normalizedContactGroupName in contactGroupIDs:
      return (normalizedContactGroupName, contactGroupIDs, contactGroupNames)
  else:
    if contactGroupName in contactGroupIDs:
      return (contactGroupName, contactGroupIDs, contactGroupNames)
    if contactGroupName in contactGroupNames:
      return (contactGroupNames[contactGroupName][0], contactGroupIDs, contactGroupNames)
    normalizedContactGroupName = normalizeContactGroupResourceName(contactGroupName)
    if normalizedContactGroupName != contactGroupName and normalizedContactGroupName in contactGroupIDs:
      return (normalizedContactGroupName, contactGroupIDs, contactGroupNames)
  return (None, contactGroupIDs, contactGroupNames)

def validatePeopleContactGroupsList(people, contactId,
                                    contactGroupsList, entityType, entityName, i, count):
  result = True
  contactGroupIDs = contactGroupNames = None
  validatedContactGroupsList = []
  for contactGroup in contactGroupsList:
    groupId, contactGroupIDs, contactGroupNames = validatePeopleContactGroup(people, contactGroup,
                                                                             contactGroupIDs, contactGroupNames, entityType, entityName, i, count)
    if groupId:
      validatedContactGroupsList.append(groupId)
    else:
      if contactGroupNames:
        entityActionNotPerformedWarning([entityType, entityName, Ent.CONTACT, contactId],
                                        Ent.TypeNameMessage(Ent.CONTACT_GROUP, contactGroup, Msg.DOES_NOT_EXIST))
      result = False
  return (result, validatedContactGroupsList, contactGroupIDs)

# gam <UserTypeEntity> create contact <PeopleContactAttribute>+
#	(contactgroup <ContactGroupItem>)*
#	[(csv [todrive <ToDriveAttribute>*] (addcsvdata <FieldName> <String>)*))| returnidonly]
def createUserPeopleContact(users):
  entityType = Ent.USER
  peopleManager = PeopleManager()
  peopleEntityType = Ent.CONTACT
  sources = PEOPLE_READ_SOURCES_CHOICE_MAP['contact']
  parameters = {'csvPF': None, 'titles': ['User', 'resourceName'], 'addCSVData': {}, 'returnIdOnly': False}
  body, personFields, contactGroupsLists = peopleManager.GetPersonFields(entityType, False, parameters)
  csvPF = parameters['csvPF']
  addCSVData = parameters['addCSVData']
  if addCSVData:
    csvPF.AddTitles(sorted(addCSVData.keys()))
  returnIdOnly = parameters['returnIdOnly']
  i, count, users = getEntityArgument(users)
  for user in users:
    i += 1
    user, people = buildGAPIServiceObject(API.PEOPLE, user, i, count)
    if not people:
      continue
    if contactGroupsLists[PEOPLE_GROUPS_LIST]:
      result, validatedContactGroupsList, _ = validatePeopleContactGroupsList(people, '',
                                                                              contactGroupsLists[PEOPLE_GROUPS_LIST], entityType, user, i, count)
      if not result:
        continue
      peopleManager.AddContactGroupsToContact(body, validatedContactGroupsList)
      personFields.add(PEOPLE_MEMBERSHIPS)
    else:
      personFields.discard(PEOPLE_MEMBERSHIPS)
    try:
      result = callGAPI(people.people(), 'createContact',
                        throwReasons=GAPI.PEOPLE_ACCESS_THROW_REASONS+[GAPI.INVALID_ARGUMENT],
                        retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                        personFields=','.join(personFields), body=body, sources=sources)
      resourceName = result['resourceName']
      if returnIdOnly:
        writeStdout(f'{resourceName}\n')
      elif not csvPF:
        entityActionPerformed([entityType, user, peopleEntityType, resourceName], i, count)
      else:
        row = {'User': user, 'resourceName': resourceName}
        if addCSVData:
          row.update(addCSVData)
        csvPF.WriteRow(row)
    except GAPI.invalidArgument as e:
      entityActionFailedWarning([entityType, user, peopleEntityType, None], str(e), i, count)
    except (GAPI.serviceNotAvailable, GAPI.forbidden, GAPI.permissionDenied, GAPI.failedPrecondition) as e:
      ClientAPIAccessDeniedExit(str(e))
  if csvPF:
    csvPF.writeCSVfile('People Contacts')

def localPeopleContactSelects(contactQuery, contact):
  if contactQuery['emailMatchPattern']:
    emailMatchType = contactQuery['emailMatchType']
    for item in contact.get(PEOPLE_EMAIL_ADDRESSES, []):
      if contactQuery['emailMatchPattern'].match(item['value']):
        if (not emailMatchType or emailMatchType == item.get('type', '')):
          break
    else:
      return False
  return True

def countLocalPeopleContactSelects(contactQuery, contacts):
  if contacts is not None and contactQuery['emailMatchPattern']:
    jcount = 0
    for contact in contacts:
      if localPeopleContactSelects(contactQuery, contact):
        jcount += 1
  else:
    jcount = len(contacts) if contacts is not None else 0
  return jcount

def clearPeopleEmailAddressMatches(contactClear, contact):
  savedAddresses = []
  updateRequired = False
  emailMatchType = contactClear['emailClearType']
  for item in contact.get(PEOPLE_EMAIL_ADDRESSES, []):
    if (contactClear['emailClearPattern'].match(item['value']) and
        (not emailMatchType or emailMatchType == item.get('type', ''))):
      updateRequired = True
    else:
      savedAddresses.append(item)
  if updateRequired:
    contact[PEOPLE_EMAIL_ADDRESSES] = savedAddresses
  return updateRequired

def _clearUpdatePeopleContacts(users, updateContacts):
  action = Act.Get()
  entityType = Ent.USER
  peopleManager = PeopleManager()
  peopleEntityType = Ent.PEOPLE_CONTACT
  sources = PEOPLE_READ_SOURCES_CHOICE_MAP['contact']
  entityList, resourceNameLists, contactQuery, queriedContacts = _getPeopleContactEntityList(entityType, 1)
  if updateContacts:
    body, updatePersonFields, contactGroupsLists = peopleManager.GetPersonFields(entityType, True)
  else:
    contactClear = {'emailClearPattern': contactQuery['emailMatchPattern'], 'emailClearType': contactQuery['emailMatchType']}
    deleteClearedContactsWithNoEmails = False
    while Cmd.ArgumentsRemaining():
      myarg = getArgument()
      if myarg == 'emailclearpattern':
        contactClear['emailClearPattern'] = getREPattern(re.IGNORECASE)
      elif myarg == 'emailcleartype':
        contactClear['emailClearType'] = getString(Cmd.OB_CONTACT_EMAIL_TYPE)
      elif myarg == 'deleteclearedcontactswithnoemails':
        deleteClearedContactsWithNoEmails = True
      else:
        unknownArgumentExit()
    if not contactClear['emailClearPattern']:
      missingArgumentExit('emailclearpattern')
  i, count, users = getEntityArgument(users)
  for user in users:
    i += 1
    if resourceNameLists:
      entityList = resourceNameLists[user]
    user, people = buildGAPIServiceObject(API.PEOPLE, user, i, count)
    if not people:
      continue
    if contactQuery['contactGroupSelect']:
      groupId, _, contactGroupNames = validatePeopleContactGroup(people, contactQuery['contactGroupSelect'],
                                                                 None, None, entityType, user, i, count)
      if not groupId:
        if contactGroupNames:
          entityActionFailedWarning([entityType, user, Ent.CONTACT_GROUP, contactQuery['contactGroupSelect']], Msg.DOES_NOT_EXIST, i, count)
        continue
      contactQuery['group'] = groupId
    if queriedContacts:
      entityList = queryPeopleContacts(people, contactQuery, 'emailAddresses,memberships', None, entityType, user, i, count)
      if entityList is None:
        continue
    Act.Set(action)
    j = 0
    jcount = len(entityList)
    entityPerformActionModifierNumItems([entityType, user], Msg.MAXIMUM_OF, jcount, peopleEntityType, i, count)
    if jcount == 0:
      setSysExitRC(NO_ENTITIES_FOUND_RC)
      continue
    validatedContactGroupsLists = {
      PEOPLE_GROUPS_LIST: [],
      PEOPLE_ADD_GROUPS_LIST: [],
      PEOPLE_REMOVE_GROUPS_LIST: []
      }
    Ind.Increment()
    for contact in entityList:
      j += 1
      try:
        if not queriedContacts:
          resourceName = contact
          contact = callGAPI(people.people(), 'get',
                             bailOnInternalError=True,
                             throwReasons=[GAPI.NOT_FOUND, GAPI.INTERNAL_ERROR]+GAPI.PEOPLE_ACCESS_THROW_REASONS,
                             retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                             resourceName=contact, sources=sources, personFields='emailAddresses,memberships')
        else:
          if not localPeopleContactSelects(contactQuery, contact):
            continue
          resourceName = contact['resourceName']
        if updateContacts:
          body['etag'] = contact['etag']
          existingContactGroupsList = []
          for contactGroup in contact.get(PEOPLE_MEMBERSHIPS, []):
            if 'contactGroupMembership' in contactGroup:
              existingContactGroupsList.append(contactGroup['contactGroupMembership']['contactGroupResourceName'])
          groupError = False
          for field in [PEOPLE_GROUPS_LIST, PEOPLE_ADD_GROUPS_LIST, PEOPLE_REMOVE_GROUPS_LIST]:
            if contactGroupsLists[field] and not validatedContactGroupsLists[field]:
              status, validatedContactGroupsLists[field], _ = validatePeopleContactGroupsList(people, resourceName,
                                                                                              contactGroupsLists[field], entityType, user, i, count)
              if not status:
                groupError = True
          if groupError:
            break
          if validatedContactGroupsLists[PEOPLE_GROUPS_LIST]:
            peopleManager.AddContactGroupsToContact(body, validatedContactGroupsLists[PEOPLE_GROUPS_LIST])
            updatePersonFields.add(PEOPLE_MEMBERSHIPS)
          elif validatedContactGroupsLists[PEOPLE_ADD_GROUPS_LIST] or validatedContactGroupsLists[PEOPLE_REMOVE_GROUPS_LIST]:
            body[PEOPLE_MEMBERSHIPS] = []
            if contact.get(PEOPLE_MEMBERSHIPS):
              peopleManager.AddFilteredContactGroupsToContact(body, existingContactGroupsList,
                                                              validatedContactGroupsLists[PEOPLE_REMOVE_GROUPS_LIST])
            if validatedContactGroupsLists[PEOPLE_ADD_GROUPS_LIST]:
              peopleManager.AddAdditionalContactGroupsToContact(body, validatedContactGroupsLists[PEOPLE_ADD_GROUPS_LIST])
            updatePersonFields.add(PEOPLE_MEMBERSHIPS)
          elif existingContactGroupsList:
            updatePersonFields.discard(PEOPLE_MEMBERSHIPS)
        else:
          if not clearPeopleEmailAddressMatches(contactClear, contact):
            continue
          if deleteClearedContactsWithNoEmails and not contact[PEOPLE_EMAIL_ADDRESSES]:
            Act.Set(Act.DELETE)
            callGAPI(people.people(), 'deleteContact',
                     bailOnInternalError=True,
                     throwReasons=[GAPI.NOT_FOUND, GAPI.INTERNAL_ERROR]+GAPI.PEOPLE_ACCESS_THROW_REASONS,
                     retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                     resourceName=resourceName)
            entityActionPerformed([entityType, user, peopleEntityType, resourceName], j, jcount)
            continue
          body = contact
          updatePersonFields = [PEOPLE_EMAIL_ADDRESSES]
        person = callGAPI(people.people(), 'updateContact',
                          throwReasons=[GAPI.INVALID_ARGUMENT, GAPI.NOT_FOUND, GAPI.INTERNAL_ERROR]+GAPI.PEOPLE_ACCESS_THROW_REASONS,
                          retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                          resourceName=resourceName,
                          updatePersonFields=','.join(updatePersonFields), body=body, sources=sources)
        entityActionPerformed([entityType, user, peopleEntityType, person['resourceName']], j, jcount)
      except GAPI.invalidArgument as e:
        entityActionFailedWarning([entityType, user, peopleEntityType, resourceName], str(e), j, jcount)
      except (GAPI.notFound, GAPI.internalError):
        entityActionFailedWarning([entityType, user, peopleEntityType, resourceName], Msg.DOES_NOT_EXIST, j, jcount)
      except (GAPI.serviceNotAvailable, GAPI.forbidden, GAPI.permissionDenied, GAPI.failedPrecondition) as e:
        ClientAPIAccessDeniedExit(str(e))
    Ind.Decrement()

# gam <UserTypeEntity> clear contacts <PeopleResourceNameEntity>|<PeopleUserContactSelection>
#	[emailclearpattern <REMatchPattern>] [emailcleartype work|home|other|<String>]
#	[deleteclearedcontactswithnoemails]
def clearUserPeopleContacts(users):
  _clearUpdatePeopleContacts(users, False)

# gam <UserTypeEntity> update contacts <PeopleResourceNameEntity>|(<PeopleUserContactSelection> endquery)
#	<PeopleContactAttribute>+
#	(contactgroup <ContactGroupItem>)*|((addcontactgroup <ContactGroupItem>)* (removecontactgroup <ContactGroupItem>)*)
# gam <UserTypeEntity> update contacts
def updateUserPeopleContacts(users):
  _clearUpdatePeopleContacts(users, True)

def dedupPeopleEmailAddressMatches(emailMatchType, contact):
  sai = -1
  savedAddresses = []
  matches = {}
  updateRequired = False
  for item in contact.get(PEOPLE_EMAIL_ADDRESSES, []):
    emailAddr = item['value']
    emailType = item.get('type', '')
    if (emailAddr in matches) and (not emailMatchType or emailType in matches[emailAddr]['types']):
      if item['metadata'].get('primary', False):
        savedAddresses[matches[emailAddr]['sai']]['metadata']['primary'] = True
      updateRequired = True
    else:
      savedAddresses.append(item)
      sai += 1
      matches.setdefault(emailAddr, {'types': set(), 'sai': sai})
      matches[emailAddr]['types'].add(emailType)
  if updateRequired:
    contact[PEOPLE_EMAIL_ADDRESSES] = savedAddresses
  return updateRequired

def replaceDomainPeopleEmailAddressMatches(contactQuery, contact, replaceDomains):
  updateRequired = False
  if contactQuery['emailMatchPattern']:
    emailMatchType = contactQuery['emailMatchType']
    for item in contact.get(PEOPLE_EMAIL_ADDRESSES, []):
      emailAddr = item['value']
      emailType = item.get('type', '')
      if contactQuery['emailMatchPattern'].match(emailAddr):
        userName, domain = splitEmailAddress(emailAddr)
        domain = domain.lower()
        if ((domain in replaceDomains) and
            (not emailMatchType or emailType == emailMatchType)):
          item['value'] = f'{userName}@{replaceDomains[domain]}'
          updateRequired = True
  else:
    for item in contact.get(PEOPLE_EMAIL_ADDRESSES, []):
      emailAddr = item['value']
      emailType = item.get('type', '')
      userName, domain = splitEmailAddress(emailAddr)
      domain = domain.lower()
      if domain in replaceDomains:
        item['value'] = f'{userName}@{replaceDomains[domain]}'
        updateRequired = True
  return updateRequired

# gam <UserTypeEntity> dedup contacts
#	[<PeopleResourceNameEntity>|<PeopleUserContactSelection>]
#	[matchType [<Boolean>]]
# gam <UserTypeEntity> replacedomain contacts
#	[<PeopleResourceNameEntity>|<PeopleUserContactSelection>]
#	(domain <DomainName> <DomainName>)+
def dedupReplaceDomainUserPeopleContacts(users):
  action = Act.Get()
  entityType = Ent.USER
  peopleEntityType = Ent.PEOPLE_CONTACT
  sources = PEOPLE_READ_SOURCES_CHOICE_MAP['contact']
  entityList, resourceNameLists, contactQuery, queriedContacts = _getPeopleContactEntityList(entityType, 1, {'matchtype', 'domain'})
  emailMatchType = False
  replaceDomains = {}
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if action == Act.DEDUP and myarg == 'matchtype':
      emailMatchType = getBoolean()
    elif action == Act.REPLACE_DOMAIN and myarg == 'domain':
      domain = getString(Cmd.OB_DOMAIN_NAME).lower()
      replaceDomains[domain] = getString(Cmd.OB_DOMAIN_NAME).lower()
    else:
      unknownArgumentExit()
  if action == Act.REPLACE_DOMAIN and not replaceDomains:
    missingArgumentExit('domain')
  i, count, users = getEntityArgument(users)
  for user in users:
    i += 1
    if resourceNameLists:
      entityList = resourceNameLists[user]
    user, people = buildGAPIServiceObject(API.PEOPLE, user, i, count)
    if not people:
      continue
    if contactQuery['contactGroupSelect']:
      groupId, _, contactGroupNames = validatePeopleContactGroup(people, contactQuery['contactGroupSelect'],
                                                                 None, None, entityType, user, i, count)
      if not groupId:
        if contactGroupNames:
          entityActionFailedWarning([entityType, user, Ent.CONTACT_GROUP, contactQuery['contactGroupSelect']], Msg.DOES_NOT_EXIST, i, count)
        continue
      contactQuery['group'] = groupId
    if queriedContacts:
      entityList = queryPeopleContacts(people, contactQuery, 'emailAddresses,memberships', None, entityType, user, i, count)
      if entityList is None:
        continue
    Act.Set(action)
    j = 0
    jcount = len(entityList)
    entityPerformActionModifierNumItems([entityType, user], Msg.MAXIMUM_OF, jcount, peopleEntityType, i, count)
    if jcount == 0:
      setSysExitRC(NO_ENTITIES_FOUND_RC)
      continue
    Act.Set(Act.UPDATE)
    Ind.Increment()
    for contact in entityList:
      j += 1
      try:
        if not queriedContacts:
          resourceName = contact
          contact = callGAPI(people.people(), 'get',
                             bailOnInternalError=True,
                             throwReasons=[GAPI.NOT_FOUND, GAPI.INTERNAL_ERROR]+GAPI.PEOPLE_ACCESS_THROW_REASONS,
                             retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                             resourceName=contact, sources=sources, personFields='emailAddresses,memberships')
        else:
          if action == Act.DEDUP and not localPeopleContactSelects(contactQuery, contact):
            continue
          resourceName = contact['resourceName']
        if action == Act.DEDUP:
          if not dedupPeopleEmailAddressMatches(emailMatchType, contact):
            continue
        else:
          if not replaceDomainPeopleEmailAddressMatches(contactQuery, contact, replaceDomains):
            continue
        Act.Set(Act.UPDATE)
        callGAPI(people.people(), 'updateContact',
                 throwReasons=[GAPI.INVALID_ARGUMENT, GAPI.NOT_FOUND, GAPI.INTERNAL_ERROR]+GAPI.PEOPLE_ACCESS_THROW_REASONS,
                 retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                 resourceName=resourceName,
                 updatePersonFields='emailAddresses', body=contact)
        entityActionPerformed([entityType, user, peopleEntityType, resourceName], j, jcount)
      except GAPI.invalidArgument as e:
        entityActionFailedWarning([entityType, user, peopleEntityType, resourceName], str(e), j, jcount)
      except (GAPI.notFound, GAPI.internalError):
        entityActionFailedWarning([entityType, user, peopleEntityType, resourceName], Msg.DOES_NOT_EXIST, j, jcount)
      except (GAPI.serviceNotAvailable, GAPI.forbidden, GAPI.permissionDenied, GAPI.failedPrecondition) as e:
        ClientAPIAccessDeniedExit(str(e))
    Ind.Decrement()

# gam <UserTypeEntity> delete contacts <PeopleResourceNameEntity>|<PeopleUserContactSelection>
def deleteUserPeopleContacts(users):
  entityType = Ent.USER
  peopleEntityType = Ent.PEOPLE_CONTACT
  entityList, resourceNameLists, contactQuery, queriedContacts = _getPeopleContactEntityList(entityType, -1)
  i, count, users = getEntityArgument(users)
  for user in users:
    i += 1
    if resourceNameLists:
      entityList = resourceNameLists[user]
    user, people = buildGAPIServiceObject(API.PEOPLE, user, i, count)
    if not people:
      continue
    if contactQuery['contactGroupSelect']:
      groupId, _, contactGroupNames = validatePeopleContactGroup(people, contactQuery['contactGroupSelect'],
                                                                 None, None, entityType, user, i, count)
      if not groupId:
        if contactGroupNames:
          entityActionFailedWarning([entityType, user, Ent.CONTACT_GROUP, contactQuery['contactGroupSelect']], Msg.DOES_NOT_EXIST, i, count)
        continue
      contactQuery['group'] = groupId
    if queriedContacts:
      entityList = queryPeopleContacts(people, contactQuery, 'emailAddresses', None, entityType, user, i, count)
      if entityList is None:
        continue
    j = 0
    jcount = len(entityList)
    entityPerformActionModifierNumItems([entityType, user], Msg.MAXIMUM_OF, jcount, peopleEntityType, i, count)
    if jcount == 0:
      setSysExitRC(NO_ENTITIES_FOUND_RC)
      continue
    Ind.Increment()
    for contact in entityList:
      j += 1
      if isinstance(contact, dict):
        if not localPeopleContactSelects(contactQuery, contact):
          continue
        resourceName = contact['resourceName']
      else:
        resourceName = normalizePeopleResourceName(contact)
      try:
        callGAPI(people.people(), 'deleteContact',
                 bailOnInternalError=True,
                 throwReasons=[GAPI.NOT_FOUND, GAPI.INTERNAL_ERROR]+GAPI.PEOPLE_ACCESS_THROW_REASONS,
                 retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                 resourceName=resourceName)
        entityActionPerformed([entityType, user, peopleEntityType, resourceName], j, jcount)
      except (GAPI.notFound, GAPI.internalError):
        entityActionFailedWarning([entityType, user, peopleEntityType, resourceName], Msg.DOES_NOT_EXIST, j, jcount)
      except (GAPI.serviceNotAvailable, GAPI.forbidden, GAPI.permissionDenied, GAPI.failedPrecondition) as e:
        ClientAPIAccessDeniedExit(str(e))
    Ind.Decrement()

def _initPersonMetadataParameters():
  return {'strip': True, 'mapUpdateTime': False, 'sourceTypes': set()}

def _processPersonMetadata(person, parameters):
  metadata = person.get(PEOPLE_METADATA, None)
  if metadata is not None:
    if parameters['mapUpdateTime']:
      sources = person[PEOPLE_METADATA].get('sources', [])
      if sources and sources[0].get(PEOPLE_UPDATE_TIME, None) is not None:
        person[PEOPLE_UPDATE_TIME] = formatLocalTime(sources[0][PEOPLE_UPDATE_TIME])
  if parameters['sourceTypes']:
    stripKeys = []
    for k, v in person.items():
      if isinstance(v, list):
        person[k] = []
        for entry in v:
          if isinstance(entry, dict):
            if entry.get('metadata', {}).get('source', {}).get('type', None) in parameters['sourceTypes']:
              person[k].append(entry)
          else:
            person[k].append(entry)
        if not person[k]:
          stripKeys.append(k)
    for k in stripKeys:
      person.pop(k, None)
  if parameters['strip']:
    person.pop(PEOPLE_METADATA, None)
    for v in person.values():
      if isinstance(v, list):
        for entry in v:
          if isinstance(entry, dict):
            entry.pop(PEOPLE_METADATA, None)

def addContactGroupNamesToContacts(contacts, contactGroupIDs, showContactGroupNamesList):
  for contact in contacts:
    if showContactGroupNamesList:
      contact[PEOPLE_GROUPS_LIST] = []
    for membership in contact.get('memberships', []):
      if 'contactGroupMembership' in membership:
        membership['contactGroupMembership']['contactGroupName'] = contactGroupIDs.get(membership['contactGroupMembership']['contactGroupResourceName'], UNKNOWN)
        if showContactGroupNamesList:
          contact[PEOPLE_GROUPS_LIST].append(membership['contactGroupMembership']['contactGroupName'])

def _printPerson(entityTypeName, user, person, csvPF, FJQC, parameters):
  _processPersonMetadata(person, parameters)
  row = flattenJSON(person, flattened={entityTypeName: user})
  if not FJQC.formatJSON:
    csvPF.WriteRowTitles(row)
  elif csvPF.CheckRowTitles(row):
    csvPF.WriteRowNoFilter({entityTypeName: user, 'resourceName': person['resourceName'],
                            'JSON': json.dumps(cleanJSON(person),
                                               ensure_ascii=False, sort_keys=True)})

PEOPLE_CONTACT_OBJECT_KEYS = {
  'addresses': 'type',
  'calendarUrls': 'type',
  'emailAddresses': 'type',
  'events': 'type',
  'externalIds': 'type',
  'genders': 'value',
  'imClients': 'type',
  'locations': 'type',
  'miscKeywords': 'type',
  'nicknames': 'type',
  'organizations': 'type',
  'relations': 'type',
  'urls': 'type',
  'userDefined': 'key',
  }

def _showPerson(userEntityType, user, entityType, person, i, count, FJQC, parameters):
  _processPersonMetadata(person, parameters)
  if not FJQC.formatJSON:
    printEntity([userEntityType, user, entityType, person['resourceName']], i, count)
    Ind.Increment()
    showJSON(None, person, dictObjectsKey=PEOPLE_CONTACT_OBJECT_KEYS)
    Ind.Decrement()
  else:
    printLine(json.dumps(cleanJSON(person), ensure_ascii=False, sort_keys=True))

def _printPersonEntityList(entityType, entityList, userEntityType, user, i, count, csvPF, FJQC, parameters, contactQuery):
  if not csvPF:
    jcount = len(entityList)
    if not FJQC.formatJSON:
      entityPerformActionModifierNumItems([userEntityType, user], Msg.MAXIMUM_OF, jcount, entityType, i, count)
    Ind.Increment()
    j = 0
    for person in entityList:
      j += 1
      if not contactQuery or localPeopleContactSelects(contactQuery, person):
        _showPerson(userEntityType, user, entityType, person, j, jcount, FJQC, parameters)
    Ind.Decrement()
  else:
    entityTypeName = Ent.Singular(userEntityType)
    for person in entityList:
      if not contactQuery or localPeopleContactSelects(contactQuery, person):
        _printPerson(entityTypeName, user, person, csvPF, FJQC, parameters)

PEOPLE_FIELDS_CHOICE_MAP = {
  'additionalname': PEOPLE_NAMES,
  'address': PEOPLE_ADDRESSES,
  'addresses': PEOPLE_ADDRESSES,
  'ageranges': 'ageRanges',
  'billinginfo': PEOPLE_MISC_KEYWORDS,
  'biography': PEOPLE_BIOGRAPHIES,
  'biographies': PEOPLE_BIOGRAPHIES,
  'birthday': PEOPLE_BIRTHDAYS,
  'birthdays': PEOPLE_BIRTHDAYS,
  'calendar': PEOPLE_CALENDAR_URLS,
  'calendars': PEOPLE_CALENDAR_URLS,
  'calendarurls': PEOPLE_CALENDAR_URLS,
  'clientdata': PEOPLE_CLIENT_DATA,
  'coverphotos': PEOPLE_COVER_PHOTOS,
  'directoryserver': PEOPLE_MISC_KEYWORDS,
  'email': PEOPLE_EMAIL_ADDRESSES,
  'emails': PEOPLE_EMAIL_ADDRESSES,
  'emailaddresses': PEOPLE_EMAIL_ADDRESSES,
  'event': PEOPLE_EVENTS,
  'events': PEOPLE_EVENTS,
  'externalid': PEOPLE_EXTERNAL_IDS,
  'externalids': PEOPLE_EXTERNAL_IDS,
  'familyname': PEOPLE_NAMES,
  'fileas': PEOPLE_FILE_ASES,
  'firstname': PEOPLE_NAMES,
  'gender': PEOPLE_GENDERS,
  'genders': PEOPLE_GENDERS,
  'givenname': PEOPLE_NAMES,
  'hobby': PEOPLE_INTERESTS,
  'hobbies': PEOPLE_INTERESTS,
  'im': PEOPLE_IM_CLIENTS,
  'ims': PEOPLE_IM_CLIENTS,
  'imclients': PEOPLE_IM_CLIENTS,
  'initials': PEOPLE_NICKNAMES,
  'interests': PEOPLE_INTERESTS,
  'jot': PEOPLE_MISC_KEYWORDS,
  'jots': PEOPLE_MISC_KEYWORDS,
  'language': PEOPLE_LOCALES,
  'languages': PEOPLE_LOCALES,
  'lastname': PEOPLE_NAMES,
  'locales': PEOPLE_LOCALES,
  'location': PEOPLE_LOCATIONS,
  'locations': PEOPLE_LOCATIONS,
  'maidenname': PEOPLE_NAMES,
  'memberships': PEOPLE_MEMBERSHIPS,
  'metadata': PEOPLE_METADATA,
  'middlename': PEOPLE_NAMES,
  'mileage': PEOPLE_MISC_KEYWORDS,
  'misckeywords': PEOPLE_MISC_KEYWORDS,
  'name': PEOPLE_NAMES,
  'names': PEOPLE_NAMES,
  'nickname': PEOPLE_NICKNAMES,
  'nicknames': PEOPLE_NICKNAMES,
  'note': PEOPLE_BIOGRAPHIES,
  'notes': PEOPLE_BIOGRAPHIES,
  'occupation': PEOPLE_OCCUPATIONS,
  'occupations': PEOPLE_OCCUPATIONS,
  'organization': PEOPLE_ORGANIZATIONS,
  'organizations': PEOPLE_ORGANIZATIONS,
  'organisation': PEOPLE_ORGANIZATIONS,
  'organisations': PEOPLE_ORGANIZATIONS,
  'phone': PEOPLE_PHONE_NUMBERS,
  'phones': PEOPLE_PHONE_NUMBERS,
  'phonenumbers': PEOPLE_PHONE_NUMBERS,
  'photo': PEOPLE_PHOTOS,
  'photos': PEOPLE_PHOTOS,
  'prefix': PEOPLE_NAMES,
  'priority': PEOPLE_MISC_KEYWORDS,
  'relation': PEOPLE_RELATIONS,
  'relations': PEOPLE_RELATIONS,
  'sensitivity': PEOPLE_MISC_KEYWORDS,
  'shortname': PEOPLE_NICKNAMES,
  'sipaddress': PEOPLE_SIP_ADDRESSES,
  'sipaddresses': PEOPLE_SIP_ADDRESSES,
  'skills': PEOPLE_SKILLS,
  'subject': PEOPLE_MISC_KEYWORDS,
  'suffix': PEOPLE_NAMES,
  'updated': PEOPLE_UPDATE_TIME,
  'updatetime': PEOPLE_UPDATE_TIME,
  'urls': PEOPLE_URLS,
  'userdefined': PEOPLE_USER_DEFINED,
  'userdefinedfield': PEOPLE_USER_DEFINED,
  'userdefinedfields': PEOPLE_USER_DEFINED,
  'website': PEOPLE_URLS,
  'websites': PEOPLE_URLS,
  }

PEOPLE_OTHER_CONTACTS_FIELDS_CHOICE_MAP = {
  'email': PEOPLE_EMAIL_ADDRESSES,
  'emails': PEOPLE_EMAIL_ADDRESSES,
  'emailaddresses': PEOPLE_EMAIL_ADDRESSES,
  'metadata': PEOPLE_METADATA,
  'names': PEOPLE_NAMES,
  'phone': PEOPLE_PHONE_NUMBERS,
  'phones': PEOPLE_PHONE_NUMBERS,
  'phonenumbers': PEOPLE_PHONE_NUMBERS,
  'photo': PEOPLE_PHOTOS,
  'photos': PEOPLE_PHOTOS,
  }

PEOPLE_CONTACTS_DEFAULT_FIELDS = ['names', 'emailaddresses', 'phonenumbers']

PEOPLE_ORDERBY_CHOICE_MAP = {
  'firstname': 'FIRST_NAME_ASCENDING',
  'lastname': 'LAST_NAME_ASCENDING',
  'lastmodified': 'LAST_MODIFIED_',
  }

def getPersonFieldsList(myarg, fieldsChoiceMap, fieldsList, initialField=None, fieldsArg='fields'):
  if fieldsList is None:
    fieldsList = []
  return getFieldsList(myarg, fieldsChoiceMap, fieldsList, initialField, fieldsArg)

def _getPersonFields(fieldsChoiceMap, defaultFields, fieldsList, parameters):
  if fieldsList is None:
    fieldsList = []
    for field in fieldsChoiceMap:
      addFieldToFieldsList(field, fieldsChoiceMap, fieldsList)
  elif not fieldsList:
    for field in defaultFields:
      addFieldToFieldsList(field, fieldsChoiceMap, fieldsList)
  fieldsList = list(set(fieldsList))
  if PEOPLE_UPDATE_TIME in fieldsList:
    parameters['mapUpdateTime'] = True
    fieldsList.remove(PEOPLE_UPDATE_TIME)
    fieldsList.append(PEOPLE_METADATA)
  return ','.join(fieldsList)

def _infoPeople(users, entityType, source):
  if entityType == Ent.DOMAIN:
    people = buildGAPIObject(API.PEOPLE)
  peopleEntityType = Ent.DOMAIN_PROFILE if source == 'profile' else Ent.PEOPLE_CONTACT
  sources = [PEOPLE_READ_SOURCES_CHOICE_MAP[source]]
  entityList = getEntityList(Cmd.OB_CONTACT_ENTITY)
  resourceNameLists = entityList if isinstance(entityList, dict) else None
  showContactGroups = False
  FJQC = FormatJSONQuoteChar()
  fieldsList = []
  parameters = _initPersonMetadataParameters()
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if myarg == 'allfields':
      fieldsList = None
    elif getPersonFieldsList(myarg, PEOPLE_FIELDS_CHOICE_MAP, fieldsList):
      pass
    elif myarg == 'showgroups':
      showContactGroups = True
    elif myarg == 'showmetadata':
      parameters['strip'] = False
    else:
      FJQC.GetFormatJSON(myarg)
  fields = _getPersonFields(PEOPLE_FIELDS_CHOICE_MAP, PEOPLE_CONTACTS_DEFAULT_FIELDS, fieldsList, parameters)
  i, count, users = getEntityArgument(users)
  for user in users:
    i += 1
    if resourceNameLists:
      entityList = resourceNameLists[user]
    contactGroupIDs = contactGroupNames = None
    if entityType != Ent.DOMAIN:
      user, people = buildGAPIServiceObject(API.PEOPLE, user, i, count)
      if not people:
        continue
      if showContactGroups:
        contactGroupIDs, contactGroupNames = getPeopleContactGroupsInfo(people, entityType, user, i, count)
        if contactGroupNames is False:
          continue
    j = 0
    jcount = len(entityList)
    if not FJQC.formatJSON:
      entityPerformActionNumItems([entityType, user], jcount, peopleEntityType, i, count)
    if jcount == 0:
      setSysExitRC(NO_ENTITIES_FOUND_RC)
      continue
    Ind.Increment()
    for contact in entityList:
      j += 1
      if isinstance(contact, dict):
        resourceName = contact['resourceName']
      else:
        resourceName = normalizePeopleResourceName(contact)
      try:
        result = callGAPI(people.people(), 'get',
                          bailOnInternalError=True,
                          throwReasons=[GAPI.NOT_FOUND, GAPI.INTERNAL_ERROR, GAPI.INVALID_ARGUMENT]+GAPI.PEOPLE_ACCESS_THROW_REASONS,
                          retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                          resourceName=resourceName, sources=sources, personFields=fields)
      except (GAPI.notFound, GAPI.internalError):
        entityActionFailedWarning([entityType, user, peopleEntityType, resourceName], Msg.DOES_NOT_EXIST, j, jcount)
        continue
      except GAPI.invalidArgument as e:
        entityActionFailedWarning([entityType, user, peopleEntityType, resourceName], str(e), j, jcount)
        continue
      except (GAPI.serviceNotAvailable, GAPI.forbidden, GAPI.permissionDenied, GAPI.failedPrecondition) as e:
        ClientAPIAccessDeniedExit(str(e))
      if showContactGroups and contactGroupIDs:
        addContactGroupNamesToContacts([result], contactGroupIDs, False)
      _showPerson(entityType, user, peopleEntityType, result, j, jcount, FJQC, parameters)
    Ind.Decrement()

# gam <UserTypeEntity> info contacts <PeopleResourceNameEntity>
#	[showgroups]
#	[allFields|(fields <PeopleFieldNameList>)] [showmetadata]
#	[formatjson]
def infoUserPeopleContacts(users):
  _infoPeople(users, Ent.USER, 'contact')

# gam <UserTypeEntity> print contacts [todrive <ToDriveAttribute>*] <PeoplePrintShowUserContactSelection>
#	[showgroups|showgroupnameslist] [orderby firstname|lastname|(lastmodified ascending)|(lastnodified descending)
#	[countsonly|allfields|fields <PeopleFieldNameList>] [showmetadata]
#	[formatjson [quotechar <Character>]]
# gam <UserTypeEntity> show contacts <PeoplePrintShowUserContactSelection>
#	[showgroups] [orderby firstname|lastname|(lastmodified ascending)|(lastnodified descending)
#	[countsonly|allfields|(fields <PeopleFieldNameList>)] [showmetadata]
#	[formatjson]
def printShowUserPeopleContacts(users):
  entityType = Ent.USER
  entityTypeName = Ent.Singular(entityType)
  csvPF = CSVPrintFile([entityTypeName, 'resourceName'], 'sortall') if Act.csvFormat() else None
  FJQC = FormatJSONQuoteChar(csvPF)
  CSVTitle = 'People Contacts'
  fieldsList = []
  parameters = _initPersonMetadataParameters()
  sortOrder = None
  countsOnly = showContactGroups = showContactGroupNamesList = False
  contactQuery = _initPeopleContactQueryAttributes(True)
  while Cmd.ArgumentsRemaining():
    myarg = getArgument()
    if csvPF and myarg == 'todrive':
      csvPF.GetTodriveParameters()
    elif myarg == 'showgroups':
      showContactGroups = True
    elif myarg == 'showgroupnameslist':
      showContactGroups = showContactGroupNamesList = True
    elif myarg == 'allfields':
      fieldsList = None
    elif getPersonFieldsList(myarg, PEOPLE_FIELDS_CHOICE_MAP, fieldsList):
      pass
    elif myarg == 'countsonly':
      countsOnly = True
      if csvPF:
        csvPF.SetTitles([entityTypeName, CSVTitle])
    elif myarg == 'showmetadata':
      parameters['strip'] = False
    elif myarg == 'orderby':
      sortOrder = getChoice(PEOPLE_ORDERBY_CHOICE_MAP, mapChoice=True)
      if sortOrder == 'LAST_MODIFIED_':
        sortOrder += getChoice(SORTORDER_CHOICE_MAP, defaultChoice='DESCENDING', mapChoice=True)
    elif _getPeopleContactQueryAttributes(contactQuery, myarg, entityType, 0, True):
      pass
    else:
      FJQC.GetFormatJSONQuoteChar(myarg, True)
  if countsOnly:
    fieldsList = ['emailAddresses']
  if contactQuery['mainContacts']:
    fields = _getPersonFields(PEOPLE_FIELDS_CHOICE_MAP, PEOPLE_CONTACTS_DEFAULT_FIELDS, fieldsList, parameters)
    if contactQuery['contactGroupFilter'] and 'memberships' not in fields:
      fields += ',memberships'
      contactQuery['dropMemberships'] = True
  if contactQuery['otherContacts']:
    if not fieldsList:
      ofields = _getPersonFields(PEOPLE_OTHER_CONTACTS_FIELDS_CHOICE_MAP, PEOPLE_CONTACTS_DEFAULT_FIELDS, fieldsList, parameters)
    else:
      ofields = getFieldsFromFieldsList([PEOPLE_OTHER_CONTACTS_FIELDS_CHOICE_MAP[field.lower()] for field in fieldsList if field.lower() in PEOPLE_OTHER_CONTACTS_FIELDS_CHOICE_MAP])
  i, count, users = getEntityArgument(users)
  for user in users:
    i += 1
    user, people = buildGAPIServiceObject(API.PEOPLE, user, i, count)
    if not people:
      continue
    if contactQuery['otherContacts']:
      _, opeople = buildGAPIServiceObject(API.PEOPLE_OTHERCONTACTS, user, i, count)
      if not opeople:
        continue
    contactGroupIDs = contactGroupNames = None
    if showContactGroups:
      contactGroupIDs, contactGroupNames = getPeopleContactGroupsInfo(people, entityType, user, i, count)
      if contactGroupNames is False:
        continue
    contactGroupSelectFilter = contactQuery['contactGroupSelect'] or contactQuery['contactGroupFilter']
    if contactGroupSelectFilter:
      groupId, _, contactGroupNames =\
        validatePeopleContactGroup(people, contactGroupSelectFilter,
                                   contactGroupIDs, contactGroupNames, entityType, user, i, count)
      if not groupId:
        if contactGroupNames:
          entityActionFailedWarning([entityType, user, Ent.CONTACT_GROUP, contactGroupSelectFilter], Msg.DOES_NOT_EXIST, i, count)
        continue
      contactQuery['group'] = groupId
    if contactQuery['mainContacts']:
      contacts = queryPeopleContacts(people, contactQuery, fields, sortOrder, entityType, user, i, count)
    else:
      contacts = []
    if contactQuery['otherContacts']:
      ocontacts = queryPeopleOtherContacts(opeople, contactQuery, ofields, entityType, user, i, count)
    else:
      ocontacts = []
    if countsOnly:
      jcount = countLocalPeopleContactSelects(contactQuery, contacts)+countLocalPeopleContactSelects(contactQuery, ocontacts)
      if csvPF:
        csvPF.WriteRowTitles({entityTypeName: user, CSVTitle: jcount})
      else:
        printEntityKVList([entityType, user], [CSVTitle, jcount], i, count)
    elif contacts is not None or ocontacts is not None:
      if not csvPF:
        if contacts is not None and contactQuery['mainContacts']:
          if showContactGroups and contactGroupIDs:
            addContactGroupNamesToContacts(contacts, contactGroupIDs, False)
          _printPersonEntityList(Ent.PEOPLE_CONTACT, contacts, entityType, user, i, count, csvPF, FJQC, parameters, contactQuery)
        if ocontacts is not None and contactQuery['otherContacts']:
          _printPersonEntityList(Ent.OTHER_CONTACT, ocontacts, entityType, user, i, count, csvPF, FJQC, parameters, contactQuery)
      elif contacts or ocontacts:
        if contacts:
          if showContactGroups and contactGroupIDs:
            addContactGroupNamesToContacts(contacts, contactGroupIDs, showContactGroupNamesList and FJQC.formatJSON)
          _printPersonEntityList(Ent.PEOPLE_CONTACT, contacts, entityType, user, i, count, csvPF, FJQC, parameters, contactQuery)
        if ocontacts:
          _printPersonEntityList(Ent.OTHER_CONTACT, ocontacts, entityType, user, i, count, csvPF, FJQC, parameters, contactQuery)
      elif GC.Values[GC.CSV_OUTPUT_USERS_AUDIT]:
        csvPF.WriteRowNoFilter({Ent.Singular(entityType): user})
  if csvPF:
    csvPF.writeCSVfile(CSVTitle)

CONTACTGROUPS_MYCONTACTS_ID = 'contactGroups/myContacts'
CONTACTGROUPS_MYCONTACTS_NAME = 'My Contacts'

# gam <UserTypeEntity> copy othercontacts
#	<OtherContactResourceNameEntity>|<OtherContactSelection>
def copyUserPeopleOtherContacts(users):
  entityType = Ent.USER
  peopleEntityType = Ent.OTHER_CONTACT
  sources = [PEOPLE_READ_SOURCES_CHOICE_MAP['contact']]
  copyMask = ['emailAddresses', 'names', 'phoneNumbers']
  entityList, resourceNameLists, contactQuery, queriedContacts = _getPeopleOtherContactEntityList(-1)
  checkForExtraneousArguments()
  i, count, users = getEntityArgument(users)
  for user in users:
    i += 1
    if resourceNameLists:
      entityList = resourceNameLists[user]
    user, people = buildGAPIServiceObject(API.PEOPLE_OTHERCONTACTS, user, i, count)
    if not people:
      continue
    if queriedContacts:
      entityList = queryPeopleOtherContacts(people, contactQuery, 'emailAddresses', entityType, user, i, count)
      if entityList is None:
        continue
    j = 0
    jcount = len(entityList)
    entityPerformActionModifierNumItems([entityType, user], Msg.MAXIMUM_OF, jcount, peopleEntityType, i, count)
    if jcount == 0:
      setSysExitRC(NO_ENTITIES_FOUND_RC)
      continue
    Ind.Increment()
    for contact in entityList:
      j += 1
      if isinstance(contact, dict):
        if not localPeopleContactSelects(contactQuery, contact):
          continue
        resourceName = contact['resourceName']
      else:
        resourceName = normalizeOtherContactsResourceName(contact)
      try:
        callGAPI(people.otherContacts(), 'copyOtherContactToMyContactsGroup',
                 bailOnInternalError=True,
                 throwReasons=[GAPI.NOT_FOUND, GAPI.INTERNAL_ERROR]+GAPI.PEOPLE_ACCESS_THROW_REASONS,
                 resourceName=resourceName, body={'copyMask': ','.join(copyMask), 'sources': sources})
        entityModifierNewValueActionPerformed([entityType, user, peopleEntityType, resourceName], Act.MODIFIER_TO, CONTACTGROUPS_MYCONTACTS_NAME)
      except (GAPI.notFound, GAPI.internalError):
        entityActionFailedWarning([entityType, user, peopleEntityType, resourceName], Msg.DOES_NOT_EXIST, j, jcount)
        continue
      except (GAPI.serviceNotAvailable, GAPI.forbidden, GAPI.permissionDenied, GAPI.failedPrecondition) as e:
        ClientAPIAccessDeniedExit(str(e))
    Ind.Decrement()

# gam <UserTypeEntity> delete othercontacts
#	<OtherContactResourceNameEntity>|<OtherContactSelection>
# gam <UserTypeEntity> move othercontacts
#	<OtherContactResourceNameEntity>|<OtherContactSelection>
# gam <UserTypeEntity> update othercontacts
#	<OtherResourceNameEntity>|<OtherContactSelection>
#	<PeopleContactAttribute>*
#	(contactgroup <ContactGroupItem>)*
def processUserPeopleOtherContacts(users):
  action = Act.Get()
  entityType = Ent.USER
  peopleEntityType = Ent.OTHER_CONTACT
  sources = PEOPLE_READ_SOURCES_CHOICE_MAP['contact']
  entityList, resourceNameLists, contactQuery, queriedContacts = _getPeopleOtherContactEntityList(1)
  if action == Act.UPDATE:
    Act.Set(Act.UPDATE_MOVE)
    peopleManager = PeopleManager()
    body, updatePersonFields, contactGroupsLists = peopleManager.GetPersonFields(entityType, False)
  else:
    body = {PEOPLE_MEMBERSHIPS: [{'contactGroupMembership': {'contactGroupResourceName': CONTACTGROUPS_MYCONTACTS_ID}}]}
    updatePersonFields = [PEOPLE_MEMBERSHIPS]
    checkForExtraneousArguments()
  validatedContactGroupsList = [CONTACTGROUPS_MYCONTACTS_ID]
  contactGroupIDs = {CONTACTGROUPS_MYCONTACTS_ID: CONTACTGROUPS_MYCONTACTS_NAME}
  i, count, users = getEntityArgument(users)
  for user in users:
    i += 1
    if resourceNameLists:
      entityList = resourceNameLists[user]
    user, people = buildGAPIServiceObject(API.PEOPLE_OTHERCONTACTS, user, i, count)
    if not people:
      continue
    _, upeople = buildGAPIServiceObject(API.PEOPLE, user, i, count)
    if not upeople:
      continue
    if queriedContacts:
      entityList = queryPeopleOtherContacts(people, contactQuery, 'emailAddresses', entityType, user, i, count)
      if entityList is None:
        continue
    else:
      otherContacts = _getPeopleOtherContacts(people, entityType, user, i=0, count=0)
      if otherContacts is None:
        continue
    if action == Act.UPDATE:
      if contactGroupsLists[PEOPLE_GROUPS_LIST]:
        result, validatedContactGroupsList, contactGroupIDs =\
          validatePeopleContactGroupsList(people, '', contactGroupsLists[PEOPLE_GROUPS_LIST], entityType, user, i, count)
        if not result:
          continue
      if CONTACTGROUPS_MYCONTACTS_ID not in validatedContactGroupsList:
        validatedContactGroupsList.insert(0, CONTACTGROUPS_MYCONTACTS_ID)
      updatePersonFields.add(PEOPLE_MEMBERSHIPS)
    contactGroupNamesList = []
    for resourceName in validatedContactGroupsList:
      contactGroupNamesList.append(contactGroupIDs[resourceName])
    contactGroupNames = ','.join(contactGroupNamesList)
    j = 0
    jcount = len(entityList)
    entityPerformActionModifierNumItems([entityType, user], Msg.MAXIMUM_OF, jcount, peopleEntityType, i, count)
    if jcount == 0:
      setSysExitRC(NO_ENTITIES_FOUND_RC)
      continue
    Ind.Increment()
    for contact in entityList:
      j += 1
      if isinstance(contact, dict):
        if not localPeopleContactSelects(contactQuery, contact):
          continue
        resourceName = contact['resourceName']
      else:
        resourceName = normalizeOtherContactsResourceName(contact)
        contact = otherContacts.get(resourceName)
        if contact is None:
          entityActionFailedWarning([entityType, user, peopleEntityType, resourceName], Msg.DOES_NOT_EXIST, j, jcount)
          continue
      peopleResourceName = resourceName.replace('otherContacts', 'people')
      body['etag'] = contact['etag']
      if action == Act.UPDATE and validatedContactGroupsList:
        peopleManager.AddContactGroupsToContact(body, validatedContactGroupsList)
      try:
        callGAPI(upeople.people(), 'updateContact',
                 throwReasons=[GAPI.INVALID_ARGUMENT, GAPI.NOT_FOUND, GAPI.INTERNAL_ERROR]+GAPI.PEOPLE_ACCESS_THROW_REASONS,
                 retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                 resourceName=peopleResourceName,
                 updatePersonFields=','.join(updatePersonFields), body=body, sources=sources)
        if action != Act.DELETE:
          entityModifierNewValueActionPerformed([entityType, user, peopleEntityType, resourceName],
                                                Act.MODIFIER_TO, contactGroupNames, j, jcount)
        else:
          maxRetries = 5
          for retry in range(1, maxRetries+1):
            try:
              callGAPI(upeople.people(), 'deleteContact',
                       bailOnInternalError=True,
                       throwReasons=[GAPI.NOT_FOUND, GAPI.INTERNAL_ERROR]+GAPI.PEOPLE_ACCESS_THROW_REASONS,
                       retryReasons=GAPI.SERVICE_NOT_AVAILABLE_RETRY_REASONS,
                       resourceName=peopleResourceName)
              entityActionPerformed([entityType, user, peopleEntityType, resourceName], j, jcount)
              break
            except (GAPI.notFound, GAPI.internalError):
              if retry == maxRetries:
                entityActionFailedWarning([entityType, user, peopleEntityType, resourceName], Msg.DOES_NOT_EXIST, 