from abc import abstractmethod

from relationalai.early_access.dsl.core.exprs import contextStack
from relationalai.early_access.dsl.core.namespaces import Namespace
from relationalai.early_access.dsl.core.rules import Rule
from relationalai.early_access.dsl.core.utils import camel_to_snake


def addrule(rul): return contextStack.root_context().addrule(rul)


# A @rule decorator that can be used to register a rule, optionally with one or more
# annotations.
#
# Examples:
#   @rule(Annotation.INNER_LOOP_NON_STRATIFIED, ...)
#   def my_rule():
#       ...
#
#   @rule()
#   def my_other_rule():
#       ...
def rule(*annotations):
    def decorator(func):
        rule_instance = addrule(func)
        for annotation in annotations:
            rule_instance.annotate(annotation)
        return func

    return decorator


class AbstractRelation:
    class NullContainer:
        def name(self): return ""

        def entity(self): return False

    # A Relation can serve as a context for declaring rules that
    # populate that relation and also for declaring constraints
    # that pertain only to this single relation -- e.g., functional
    # and mandatory constraints.
    #
    def __enter__(self):
        contextStack.push(self)
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        contextStack.pop()
        pass

    # A Relation can be "invoked" in the body of a rule definition to
    # declare what will be an Atom in the Schema generated by the rule
    # def.
    #
    # Note that because Schemas expect these Atoms to be defined on
    # RelationComponents, we have to make sure that such a component
    # exists in the Schema in context.
    #
    def __call__(self, *args):
        context = contextStack.root_context()
        if context is not None:
            return context.build_atom(self, args)
        return None

    def __init__(self, namespace, name, sig):
        self._relname = name
        self._container = namespace
        self._signature = sig
        self._rules = []

    # The "rule" to add is a function object with a number of parameters that should
    # equal the number of roles in this Relation.
    #
    def addrule(self, rule):
        if not isinstance(rule, Rule):
            # Assume that rule is a Python function that, when invoked can build a rule
            rule = Rule(self).elaborate(rule)

        self._rules.append(rule)
        return rule

    def rules(self):
        return self._rules

    def arity(self):
        return self._signature.arity()

    def binary(self) -> bool:
        return self.arity() == 2

    def display(self):
        return self.qualified_name()

    def entityid(self):
        return hash(self.qualified_name())

    # Generates a new Rule that populates this relation, returning a tuple whose
    # first component is the Rule object and whose next n-1 components are the
    # free variables of this rule in order. Allows one to construct new rules that
    # populate a relation and get a handle on all free variables individually using
    # one method call, e.g.:
    #
    #   (rule, x, y, z) = relation.genrule()
    #
    def genrule(self):
        rule = Rule(self)
        self._rules.append(rule)
        tup = [rule]
        for v in rule.freevars():
            tup.append(v)
        return tuple(tup)

    def pprint(self):
        return self.display() + self._signature.display()

    # Returns the fully-qualified name that will be used in queries to
    # refer to this Relation.
    #
    def qualified_name(self):
        if not isinstance(self._container, Namespace):
            raise Exception(f"Cannot find containing namespace for Relation named {self._relname}")

        if self._container == Namespace.top:
            return self._relname
        else:
            # TODO: [VAMI] We changed `::` to `__` to spead up E2E testing. We should change it back to `::` before release.
            return f"{self._container.qualified_name()}__{self._relname}"

    def signature(self):
        return self._signature

    def rel_name(self):
        return self._relname

    def first(self):
        return self._signature.first()

class Relation(AbstractRelation):

    def __init__(self, namespace, reading, relationship, functional=False):
        super().__init__(namespace, reading.to_rel_name(),
                         RelationSignature(*reading.types(), functional=functional))
        self._reading = reading
        self._relationship = relationship

    def __getattr__(self, item):
        if item in self.__dict__:
            return self.__dict__[item]
        return self._relationship.role_by_name(item)

    def reading(self):
        return self._reading

    def relationship(self):
        return self._relationship

class AssertedRelation(AbstractRelation):
    """
    A relation that's assumed to be defined in the target system, but cannot be verified.

    This is used as a common parent class for such relations, and the compiler makes sure to
    not generate any unnecessary constructs for them.
    """
    def __init__(self, namespace, name, sig):
        super().__init__(namespace, name, sig)

class ExternalRelation(AssertedRelation):

    def __init__(self, namespace, name, sig):
        super().__init__(namespace, name, sig)

class ExportRelation(AssertedRelation):

    def __init__(self, namespace, name, sig):
        super().__init__(namespace, name, sig)

class EntityPopulationRelation(AbstractRelation):

    def __init__(self, namespace, name, sig):
        super().__init__(namespace, name, sig)

class EntityInstanceRelation(AbstractRelation):

    def __init__(self, namespace, name, sig):
        type = sig.first()
        # Instance relations are in type's namespace
        type_namespace = Namespace(camel_to_snake(type.display()), namespace)
        super().__init__(type_namespace, name, sig)

class AbstractRelationSignature:

    def __init__(self, *types):
        self._types = types

    def __call__(self, *args, **kwargs):
        seq = [self]
        for i in range(len(args)):
            seq.append(args[i])
        context = contextStack.root_context()
        if context is not None:
            return context.build_relation_variable(seq, kwargs)
        return None

    def arity(self):
        return len(self._types)

    def entityid(self):
        relhash = hash([r.entityid() for r in self._types])
        if self.functional():
            return hash((relhash, True))
        else:
            return hash((relhash, False))

    def relational(self) -> bool:
        return True

    # An n-ary AbstractRelationSignature is functional if its n-th column
    # is functionally dependent on the first n-1 columns.
    #
    @abstractmethod
    def functional(self) -> bool:
        """Return True when relation is functional otherwise False."""

    def domainsig(self):
        if self.functional():
            a = self.arity()
            return self._types[0:a - 1]
        else:
            return self._types

    def rangesig(self):
        if self.functional():
            return self._types[self.arity() - 1]
        else:
            return None

    def types(self):
        return self._types

    def display(self):
        dsp = [r.display() for r in self.domainsig() if r is not None]
        dom = "(" + ", ".join(dsp) + ")"
        range_sig = self.rangesig()
        if range_sig is not None:
            return f"{dom} -> {range_sig.display()}"
        else:
            return dom

    # An AbstractRelationSignature is a subtype of another if the two have the
    # same number of types and in each corresponding type pair (t1, t2),
    # the type that plays t1 is a subtype of the type that plays t2.
    #
    def subtype_of(self, sig2):
        if self.arity() != sig2.arity():
            return False

        for i in range(self.arity()):
            if not self._types[i].subtype_of(sig2._types[i]):
                return False

        return True

    def first(self):
        return self._types[0]

class RelationSignature(AbstractRelationSignature):

    def __init__(self, *types, functional=False):
        super().__init__(*types)
        self._is_functional = functional

    def set_functional(self, functional: bool) -> None:
        self._is_functional = functional

    def functional(self): return self._is_functional
