"""low level hooks into auto-generated API code

Functions here should not be used directly. Rather, interact
with these functions via api.py
"""

import inspect
import json
from pathlib import Path
import sys

from beartype import beartype
from box import Box
from deeporigin_data import (
    AuthenticationError,
    ConflictError,
    DeeporiginData,
    NotFoundError,
)

from deeporigin import auth
from deeporigin.exceptions import DeepOriginException
from deeporigin.utils.core import PersistentDict, _get_method, _redact_responses

# this list specifies methods in the low level API that should
# not be automatically wrapped
IGNORED_METHODS = {
    "close",
    "get",
    "post",
    "put",
    "delete",
    "get_api_list",
    "is_closed",
    "platform_headers",
    "patch",
    "with_options",
}

# this list specifies methods that should be overridden
# by functions in api.py
OVERRIDDEN_METHODS = {
    "list_rows",
    "convert_id_format",
    "create_workspace",
    "create_database",
    "list_files",
    "add_database_column",  # this method is broken in the autogenerated code
}


@beartype
def _get_client_methods() -> set:
    # the only reason we're creating this client is to
    # extract methods from it. So no need to
    # authenticate
    client = DeeporiginData(token="", org_id="").with_raw_response
    methods = set(
        [
            attr
            for attr in dir(client)
            if callable(getattr(client, attr)) and not attr.startswith("_")
        ]
    ).difference(IGNORED_METHODS)

    return methods


def _get_default_client(
    *,
    client=None,
    refresh: bool = False,
    use_async: bool = False,
):
    """Internal function to instantiate client

    Creates and returns an authenticated client if
    not provided with one.

    Warning: Internal function
        Do not use this function

    Args:
        client: None, or a Client


    """
    if client is None:
        tokens = auth.get_tokens(refresh=refresh)
        access_token = tokens["access"]

        import httpx

        from deeporigin.config import get_value

        value = get_value()

        org_id = value["organization_id"]
        base_url = httpx.URL.join(
            value["api_endpoint"],
            value["nucleus_api_route"],
        )

        if use_async:
            from deeporigin_data import AsyncDeeporiginData

            client = AsyncDeeporiginData(
                token=access_token,
                org_id=org_id,
                base_url=base_url,
            )
        else:
            client = DeeporiginData(
                token=access_token,
                org_id=org_id,
                base_url=base_url,
            ).with_raw_response

    return client


def _create_function(method_path):
    """utility function the dynamically creates functions
    that wrap low-level functions in the DeepOrigin data API"""

    # we're constructing a client solely for the purposes
    # of inspecting its methods and extracting
    # function signatures. So we don't need any
    # authentication
    client = DeeporiginData(token="", org_id="").with_raw_response
    method = _get_method(client, method_path)

    signature = inspect.signature(method)

    def dynamic_function(
        *,
        client=None,
        _stash: bool = False,
        **kwargs,
    ):
        if client is None:
            client = _get_default_client(
                refresh=False,
            )
        method = _get_method(client, method_path)
        # call the low level API method

        try:
            response = method(**kwargs)
        except AuthenticationError as error:
            if "expired token" in error.message:
                # TODO: this is a hack to refresh the token
                # This can likely be safely removed, and I do not think
                # this code block is ever reached, because the client
                # is configured to refresh the token when it expires.

                print("⚠️ Token expired. Refreshing credentials...")
                tokens = auth.read_cached_tokens()
                tokens["access"] = auth.refresh_tokens(tokens["refresh"])

                # cache to disk
                auth.cache_tokens(tokens)

                # clear the cache from memory
                # so that we force the client to read the new
                # token from disk
                auth.get_tokens.cache_clear()

                # configure the client to use the new access
                # token. we need to regenerate the client
                # because there's no way to modify the token
                client = _get_default_client(refresh=False)

                method = _get_method(client, method_path)
                response = method(**kwargs)
            else:
                raise error
        except NotFoundError as error:
            message = error.body["errors"][0]["title"]
            raise DeepOriginException(
                message,
                fix="The resource could not be found. Please check the ID and organization.",
                title="Deep Origin error: [Resource not found]",
            ) from error
        except ConflictError as error:
            message = error.body["errors"][0]["title"]
            raise DeepOriginException(
                message,
                fix="The name you are trying to use cannot be used because it is already in use. Please try another name.",
                title="Deep Origin error: [Name conflicts]",
            ) from error

        if not isinstance(response, dict):
            if "json" in response.headers["content-type"]:
                response = response.json()
            elif "text" in response.headers["content-type"]:
                response = response.text()
            else:
                raise DeepOriginException(
                    f"Uncertain response type: {response.headers}"
                )

        if isinstance(response, dict) and "data" in response.keys():
            response = response["data"]

        if isinstance(response, list):
            response = [Box(item) for item in response]
        elif isinstance(response, dict):
            response = Box(response)

        if _stash:
            print("Stashing response")
            stash_loc = (
                Path(__file__).resolve().parent.parent.parent
                / "tests"
                / "fixtures"
                / "responses"
                / f"{method_path}.json"
            )
            stashed_data = PersistentDict(stash_loc)
            key = json.dumps(kwargs, sort_keys=True)

            if isinstance(response, list):
                stashed_response = response[:10]
            else:
                stashed_response = response

            stashed_data[key] = _redact_responses(stashed_response)

        return response

    # attach the signature of the underlying method to the
    # function so that IDEs can display it properly
    dynamic_function.__signature__ = signature

    return dynamic_function


module_name = sys.modules[__name__]


methods = _get_client_methods()
for method in methods:
    setattr(module_name, method, _create_function(method))


__all__ = list((methods | {"_get_default_client"}).difference(OVERRIDDEN_METHODS))
