#!/usr/bin/env python3

"""
Tools for performing coverage QC checks on targeted deep sequencing data.
"""

from pathlib import Path
import rich_click as click
from .panelIndexer import buildPanelIndex
from .coverageQC import computeCoverage


@click.group(name="DeepSeqCoverageQC")
@click.version_option()
def DeepSeqCoverageQC():
    """
    Compute coverageQC for targeted deep sequencing data
    """
    pass


@DeepSeqCoverageQC.command(
        name="buildIndex",
        help="Generate index for the panel"
)
@click.option(
    "--panelFile", type=Path,
    help="File with panel regions. "
    "File should have four tab-delimited columns: Chr, Start, End, Gene",
    required=True
)
@click.option(
    "--outfile", type=Path, default=None,
    help="Output file name. ",
    required=False
)
@click.option(
    "--padding", type=int, default=0,
    help="Padding (bp) for regions. ",
    required=False
)
@click.option(
    "--chr/--no-chr", type=bool, default=False,
    help="Add 'chr' prefix",
    required=False
)
def buildIndex(panelfile, outfile, padding, chr):
    buildPanelIndex(
        panelFile=panelfile,
        outfile=outfile,
        padding=padding,
        chrPrefix=chr
    )


@DeepSeqCoverageQC.command(
        name="computeCoverageQC",
        help="Compute coverageQC metrics across panel regions."
)
@click.option(
    "--panelPosFile", type=Path,
    help="File with panel positions as generated by buildIndex command."
    "File should have six columns: Chr.Pos, Chr, Start, End, Gene, "
    "RegionLength",
    required=True
)
@click.option(
    "--summaryPosFile", type=Path,
    help="File with unique positions in the panel to compute sample level"
    " summary statistics. File should have one column: "
    "Chr.Pos, as generated by buildIndex command. ",
    required=True
)
@click.option(
    "--countFile", type=Path, multiple=True,
    help="Count file(s) generated by SequencErr program. File(s) should've"
    " seven columns: Chr, Pos, A_Q_30, C_Q_30, G_Q_30, T_Q_30, N_Q_30",
    required=True
)
@click.option(
    "--outdir", type=Path, default=None,
    help="Output directory path. ",
    required=False
)
@click.option(
    "--outSummary/--no-outSummary", type=bool, default=False,
    help="Output counts at summary positions",
    required=False
)
def computeCoverageQC(
    panelposfile, summaryposfile, countfile, outdir, outsummary
):
    computeCoverage(
        panelPosFile=panelposfile,
        summaryPosFile=summaryposfile,
        countFile=countfile,
        outdir=outdir,
        outSummaryCounts=outsummary
    )


if __name__ == "__main__":
    DeepSeqCoverageQC()
