import sys
from enum import Enum, auto
from dataclasses import dataclass
from typing import List, Any, Optional


class TokenType(Enum):
    # Literals
    STRING = auto()
    NUMBER = auto()
    IDENTIFIER = auto()
    
    # Keywords
    PRINT = auto()
    SET = auto()
    IF = auto()
    ELSE = auto()
    ENDIF = auto()
    THEN = auto()
    WHILE = auto()
    ENDWHILE = auto()
    FOR = auto()
    IN = auto()
    ENDFOR = auto()
    FUNC = auto()
    ENDFUNC = auto()
    CALL = auto()
    RETURN = auto()
    INPUT = auto()
    
    # Operators
    EQUALS = auto()
    NOT_EQUALS = auto()
    GREATER = auto()
    GREATER_EQUALS = auto()
    LESS = auto()
    LESS_EQUALS = auto()
    CONTAINS = auto()
    
    # Special
    NEWLINE = auto()
    EOF = auto()
    SEMICOLON = auto()


@dataclass
class Token:
    type: TokenType
    value: Any
    line: int
    column: int
    
    def __repr__(self):
        return f"Token({self.type.name}, {self.value!r}, {self.line}:{self.column})"


class Lexer:
    """Converts source code into tokens"""
    
    KEYWORDS = {
        'print': TokenType.PRINT,
        'set': TokenType.SET,
        'if': TokenType.IF,
        'else': TokenType.ELSE,
        'endif': TokenType.ENDIF,
        'then': TokenType.THEN,
        'while': TokenType.WHILE,
        'endwhile': TokenType.ENDWHILE,
        'for': TokenType.FOR,
        'in': TokenType.IN,
        'endfor': TokenType.ENDFOR,
        'func': TokenType.FUNC,
        'endfunc': TokenType.ENDFUNC,
        'call': TokenType.CALL,
        'return': TokenType.RETURN,
        'input': TokenType.INPUT,
        'equals': TokenType.EQUALS,
        'notequals': TokenType.NOT_EQUALS,
        'greater': TokenType.GREATER,
        'greaterequals': TokenType.GREATER_EQUALS,
        'less': TokenType.LESS,
        'lessequals': TokenType.LESS_EQUALS,
        'contains': TokenType.CONTAINS,
    }
    
    OPERATORS = {
        '==': TokenType.EQUALS,
        '!=': TokenType.NOT_EQUALS,
        '>=': TokenType.GREATER_EQUALS,
        '>': TokenType.GREATER,
        '<=': TokenType.LESS_EQUALS,
        '<': TokenType.LESS,
    }
    
    def __init__(self, source: str):
        self.source = source
        self.pos = 0
        self.line = 1
        self.column = 1
        self.tokens: List[Token] = []
    
    def current_char(self) -> Optional[str]:
        if self.pos >= len(self.source):
            return None
        return self.source[self.pos]
    
    def peek(self, offset: int = 1) -> Optional[str]:
        pos = self.pos + offset
        if pos >= len(self.source):
            return None
        return self.source[pos]
    
    def advance(self):
        if self.pos < len(self.source) and self.source[self.pos] == '\n':
            self.line += 1
            self.column = 1
        else:
            self.column += 1
        self.pos += 1
    
    def skip_whitespace(self):
        while self.current_char() and self.current_char() in ' \t\r':
            self.advance()
    
    def skip_comment(self):
        if self.current_char() == '#':
            while self.current_char() and self.current_char() != '\n':
                self.advance()
    
    def read_string(self, quote_char: str) -> str:
        value = ""
        self.advance()  # Skip opening quote
        
        while self.current_char() and self.current_char() != quote_char:
            if self.current_char() == '\\':
                self.advance()
                if self.current_char():
                    # Handle escape sequences
                    escape_map = {'n': '\n', 't': '\t', 'r': '\r', '\\': '\\'}
                    value += escape_map.get(self.current_char(), self.current_char())
                    self.advance()
            else:
                value += self.current_char()
                self.advance()
        
        if self.current_char() == quote_char:
            self.advance()  # Skip closing quote
        
        return value
    
    def read_number(self) -> float:
        value = ""
        has_dot = False
        
        while self.current_char() and (self.current_char().isdigit() or self.current_char() == '.'):
            if self.current_char() == '.':
                if has_dot:
                    break
                has_dot = True
            value += self.current_char()
            self.advance()
        
        return float(value) if has_dot else int(value)
    
    def read_identifier(self) -> str:
        value = ""
        while self.current_char() and (self.current_char().isalnum() or self.current_char() == '_'):
            value += self.current_char()
            self.advance()
        return value
    
    def read_operator(self) -> Optional[TokenType]:
        # Try two-character operators first
        two_char = self.source[self.pos:self.pos+2]
        if two_char in self.OPERATORS:
            self.advance()
            self.advance()
            return self.OPERATORS[two_char]
        
        # Try single-character operators
        one_char = self.current_char()
        if one_char in self.OPERATORS:
            self.advance()
            return self.OPERATORS[one_char]
        
        return None
    
    def tokenize(self) -> List[Token]:
        while self.current_char():
            start_line = self.line
            start_column = self.column
            
            # Skip whitespace
            if self.current_char() in ' \t\r':
                self.skip_whitespace()
                continue
            
            # Skip comments
            if self.current_char() == '#':
                self.skip_comment()
                continue
            
            # Newlines
            if self.current_char() == '\n':
                self.tokens.append(Token(TokenType.NEWLINE, '\n', start_line, start_column))
                self.advance()
                continue
            
            # Semicolons
            if self.current_char() == ';':
                self.tokens.append(Token(TokenType.SEMICOLON, ';', start_line, start_column))
                self.advance()
                continue
            
            # Strings
            if self.current_char() in '"\'':
                quote = self.current_char()
                value = self.read_string(quote)
                self.tokens.append(Token(TokenType.STRING, value, start_line, start_column))
                continue
            
            # Numbers
            if self.current_char().isdigit():
                value = self.read_number()
                self.tokens.append(Token(TokenType.NUMBER, value, start_line, start_column))
                continue
            
            # Operators
            op_type = self.read_operator()
            if op_type:
                self.tokens.append(Token(op_type, op_type.name, start_line, start_column))
                continue
            
            # Identifiers and keywords
            if self.current_char().isalpha() or self.current_char() == '_':
                value = self.read_identifier()
                token_type = self.KEYWORDS.get(value.lower(), TokenType.IDENTIFIER)
                self.tokens.append(Token(token_type, value, start_line, start_column))
                continue
            
            # Unknown character
            print(f"Warning: Unknown character '{self.current_char()}' at {start_line}:{start_column}")
            self.advance()
        
        self.tokens.append(Token(TokenType.EOF, None, self.line, self.column))
        return self.tokens


class Parser:
    """Parses tokens into an AST or executes directly"""
    
    def __init__(self, tokens: List[Token]):
        self.tokens = tokens
        self.pos = 0
        self.variables = {}
        self.functions = {}
    
    def current_token(self) -> Token:
        if self.pos >= len(self.tokens):
            return self.tokens[-1]  # EOF
        return self.tokens[self.pos]
    
    def peek(self, offset: int = 1) -> Token:
        pos = self.pos + offset
        if pos >= len(self.tokens):
            return self.tokens[-1]
        return self.tokens[pos]
    
    def advance(self):
        if self.pos < len(self.tokens) - 1:
            self.pos += 1
    
    def skip_newlines(self):
        while self.current_token().type in [TokenType.NEWLINE, TokenType.SEMICOLON]:
            self.advance()
    
    def expect(self, token_type: TokenType) -> Token:
        token = self.current_token()
        if token.type != token_type:
            raise SyntaxError(f"Expected {token_type.name}, got {token.type.name} at {token.line}:{token.column}")
        self.advance()
        return token
    
    def resolve_value(self, token: Token):
        """Resolve a token to its actual value"""
        if token.type == TokenType.STRING:
            return token.value
        elif token.type == TokenType.NUMBER:
            return token.value
        elif token.type == TokenType.IDENTIFIER:
            return self.variables.get(token.value, token.value)
        return token.value
    
    def parse_expression(self) -> Any:
        """Parse a simple expression (could be extended for full expression parsing)"""
        token = self.current_token()
        self.advance()
        return self.resolve_value(token)
    
    def evaluate_condition(self, left_token: Token, op_token: Token, right_token: Token) -> bool:
        """Evaluate a comparison condition"""
        left = self.resolve_value(left_token)
        right = self.resolve_value(right_token)
        
        # Try numeric comparison
        try:
            left_num = float(left) if isinstance(left, str) and left.replace('.', '', 1).replace('-', '', 1).isdigit() else left
            right_num = float(right) if isinstance(right, str) and right.replace('.', '', 1).replace('-', '', 1).isdigit() else right
        except:
            left_num, right_num = left, right
        
        if op_token.type in [TokenType.EQUALS]:
            return str(left) == str(right)
        elif op_token.type == TokenType.NOT_EQUALS:
            return str(left) != str(right)
        elif op_token.type == TokenType.GREATER:
            return left_num > right_num
        elif op_token.type == TokenType.GREATER_EQUALS:
            return left_num >= right_num
        elif op_token.type == TokenType.LESS:
            return left_num < right_num
        elif op_token.type == TokenType.LESS_EQUALS:
            return left_num <= right_num
        elif op_token.type == TokenType.CONTAINS:
            return str(right) in str(left)
        elif op_token.type == TokenType.IN:
            return str(left) in str(right)
        
        return False
    
    def execute(self):
        """Main execution loop"""
        while self.current_token().type != TokenType.EOF:
            self.skip_newlines()
            if self.current_token().type == TokenType.EOF:
                break
            self.execute_statement()
    
    def execute_statement(self):
        """Execute a single statement"""
        token = self.current_token()
        
        if token.type == TokenType.PRINT:
            self.execute_print()
        elif token.type == TokenType.SET:
            self.execute_set()
        elif token.type == TokenType.IF:
            self.execute_if()
        elif token.type == TokenType.WHILE:
            self.execute_while()
        elif token.type == TokenType.FOR:
            self.execute_for()
        elif token.type == TokenType.INPUT:
            self.execute_input()
        elif token.type == TokenType.FUNC:
            self.execute_func_def()
        elif token.type == TokenType.CALL:
            self.execute_call()
        else:
            # Skip unknown tokens
            self.advance()
    
    def execute_print(self):
        """Execute print statement"""
        self.advance()  # Skip 'print'
        
        parts = []
        while self.current_token().type not in [TokenType.NEWLINE, TokenType.SEMICOLON, TokenType.EOF]:
            parts.append(str(self.resolve_value(self.current_token())))
            self.advance()
        
        print(" ".join(parts))
    
    def execute_set(self):
        """Execute variable assignment"""
        self.advance()  # Skip 'set'
        var_name = self.expect(TokenType.IDENTIFIER).value
        value = self.parse_expression()
        self.variables[var_name] = value
    
    def execute_if(self):
        """Execute if statement"""
        self.advance()  # Skip 'if'
        
        # Parse condition
        left = self.current_token()
        self.advance()
        op = self.current_token()
        self.advance()
        right = self.current_token()
        self.advance()
        
        condition_met = self.evaluate_condition(left, op, right)
        
        # Check for single-line if with 'then'
        if self.current_token().type == TokenType.THEN:
            self.advance()  # Skip 'then'
            if condition_met:
                while self.current_token().type not in [TokenType.NEWLINE, TokenType.EOF]:
                    self.execute_statement()
            else:
                # Skip to end of line
                while self.current_token().type not in [TokenType.NEWLINE, TokenType.EOF]:
                    self.advance()
        else:
            # Multi-line if
            self.skip_newlines()
            
            if condition_met:
                # Execute if block
                while self.current_token().type not in [TokenType.ELSE, TokenType.ENDIF, TokenType.EOF]:
                    self.execute_statement()
                    self.skip_newlines()
                
                # Skip else block if present
                if self.current_token().type == TokenType.ELSE:
                    self.advance()
                    self.skip_newlines()
                    depth = 1
                    while depth > 0 and self.current_token().type != TokenType.EOF:
                        if self.current_token().type == TokenType.IF:
                            depth += 1
                        elif self.current_token().type == TokenType.ENDIF:
                            depth -= 1
                        if depth > 0:
                            self.advance()
            else:
                # Skip if block
                depth = 1
                while depth > 0 and self.current_token().type != TokenType.EOF:
                    if self.current_token().type == TokenType.IF:
                        depth += 1
                    elif self.current_token().type == TokenType.ELSE and depth == 1:
                        self.advance()
                        self.skip_newlines()
                        # Execute else block
                        while self.current_token().type not in [TokenType.ENDIF, TokenType.EOF]:
                            self.execute_statement()
                            self.skip_newlines()
                        break
                    elif self.current_token().type == TokenType.ENDIF:
                        depth -= 1
                    if depth > 0:
                        self.advance()
            
            if self.current_token().type == TokenType.ENDIF:
                self.advance()
    
    def execute_while(self):
        """Execute while loop"""
        loop_start = self.pos
        self.advance()  # Skip 'while'
        
        # Parse condition tokens
        cond_start = self.pos
        left_token = self.current_token()
        self.advance()
        op_token = self.current_token()
        self.advance()
        right_token = self.current_token()
        self.advance()
        
        self.skip_newlines()
        body_start = self.pos
        
        # Find endwhile
        depth = 1
        while depth > 0 and self.current_token().type != TokenType.EOF:
            if self.current_token().type == TokenType.WHILE:
                depth += 1
            elif self.current_token().type == TokenType.ENDWHILE:
                depth -= 1
            if depth > 0:
                self.advance()
        
        endwhile_pos = self.pos
        
        # Execute loop
        while True:
            # Re-evaluate condition
            self.pos = cond_start
            left = self.current_token()
            self.advance()
            op = self.current_token()
            self.advance()
            right = self.current_token()
            
            if not self.evaluate_condition(left, op, right):
                break
            
            # Execute body
            self.pos = body_start
            while self.pos < endwhile_pos:
                self.skip_newlines()
                if self.pos >= endwhile_pos:
                    break
                self.execute_statement()
        
        self.pos = endwhile_pos
        if self.current_token().type == TokenType.ENDWHILE:
            self.advance()
    
    def execute_for(self):
        """Execute for loop"""
        self.advance()  # Skip 'for'
        var_name = self.expect(TokenType.IDENTIFIER).value
        self.expect(TokenType.IN)
        
        # Get iterable
        iterable_token = self.current_token()
        self.advance()
        iterable = self.resolve_value(iterable_token)
        
        # Try to evaluate as range or list
        if isinstance(iterable, str):
            try:
                iterable = eval(iterable, {"__builtins__": None, "range": range}, self.variables)
            except:
                pass
        
        self.skip_newlines()
        body_start = self.pos
        
        # Find endfor
        depth = 1
        while depth > 0 and self.current_token().type != TokenType.EOF:
            if self.current_token().type == TokenType.FOR:
                depth += 1
            elif self.current_token().type == TokenType.ENDFOR:
                depth -= 1
            if depth > 0:
                self.advance()
        
        endfor_pos = self.pos
        
        # Execute loop
        for value in iterable:
            self.variables[var_name] = value
            self.pos = body_start
            while self.pos < endfor_pos:
                self.skip_newlines()
                if self.pos >= endfor_pos:
                    break
                self.execute_statement()
        
        self.pos = endfor_pos
        if self.current_token().type == TokenType.ENDFOR:
            self.advance()
    
    def execute_input(self):
        """Execute input statement"""
        self.advance()  # Skip 'input'
        var_name = self.expect(TokenType.IDENTIFIER).value
        
        # Collect prompt
        prompt_parts = []
        while self.current_token().type not in [TokenType.NEWLINE, TokenType.EOF]:
            prompt_parts.append(str(self.resolve_value(self.current_token())))
            self.advance()
        
        prompt = " ".join(prompt_parts)
        self.variables[var_name] = input(prompt)
    
    def execute_func_def(self):
        """Skip function definition (store for later)"""
        self.advance()  # Skip 'func'
        # For now, just skip to endfunc
        depth = 1
        while depth > 0 and self.current_token().type != TokenType.EOF:
            if self.current_token().type == TokenType.FUNC:
                depth += 1
            elif self.current_token().type == TokenType.ENDFUNC:
                depth -= 1
            self.advance()
    
    def execute_call(self):
        """Execute function call (placeholder)"""
        self.advance()  # Skip 'call'
        # Skip to end of line
        while self.current_token().type not in [TokenType.NEWLINE, TokenType.EOF]:
            self.advance()


def ifs(source: str):
    """Main interpreter function"""
    lexer = Lexer(source)
    tokens = lexer.tokenize()
    
    # Debug: print tokens
    # for token in tokens:
    #     print(token)
    
    parser = Parser(tokens)
    parser.execute()


def main():
    """CLI entry point"""
    if len(sys.argv) < 2:
        print("Usage: python -m cerona.main <filename>")
        sys.exit(1)

    filename = sys.argv[1]
    try:
        with open(filename, "r") as file:
            source = file.read()
    except FileNotFoundError:
        print(f"File '{filename}' not found.")
        sys.exit(1)

    ifs(source)


if __name__ == "__main__":
    main()
