import jax
import jax.numpy as jnp
import tensorflow_probability.substrates.jax as tfp
from tensorflow_probability.substrates.jax.internal import (
    parameter_properties,
    tensorshape_util,
)

tfd = tfp.distributions


def _log_choose(N, k):  # noqa: N803
    return (
        jax.lax.lgamma(N + 1.0)
        - jax.lax.lgamma(k + 1.0)
        - jax.lax.lgamma(N - k + 1.0)
    )


class UniformKCategorical(tfd.Distribution):
    def __init__(
        self,
        k,
        mask,
        float_dtype=jnp.float32,
        validate_args=False,
        allow_nan_stats=True,
        name="UniformKCategorical",
    ):
        """Uniform K-Categorical distribution.

        Given a set of items indexed $1,...,n$ and a boolean mask of the same
            shape sample $k$ indices without replacement.

        :param k: the number of indices to sample
        :param mask: a boolean mask with `True` where an element is valid,
                     otherwise `False`
        :param validate_args: Whether to validate args
        :param allow_nan_stats: allow nan stats
        :param name: name of the distribution

        Example 1: Generate 4 samples of size k given a mask
            import jax
            import jax.numpy as jnp
            from jax import random
            from gemlib.distributions.kcategorical import UniformKCategorical

            # Mask determines which indices are valid and returned by sample().
            # Below combinations of the indices 0, 3, 4, and 6 will be realised
            # when sampling.
            mask = [True, False, False, True, True, False, True]
            X = UniformKCategorical(k=3, mask=mask)
            x = X.sample(4)
            print(x)

        Example 2: Probability of a given sample
            import jax
            import jax.numpy as jnp
            from jax import random
            from gemlib.distributions.kcategorical import UniformKCategorical

            # Probability of drawing an unordered sample of
            # size k from N items where N equals the number
            # of True states in the mask determines N.
            mask = [True, False, False, True, True, False, True]
            sample = jnp.array[4, 3, 6])
            X = UniformKCategorical(k=sample.shape[-1], mask=mask)
            lp = X.log_prob(sample)
            print('prob:', jnp.exp(lp))

        """
        parameters = dict(locals())
        self._mask = jnp.array(mask)
        self._k = k
        self._float_dtype = float_dtype
        dtype = jnp.int32

        super().__init__(
            dtype=dtype,
            validate_args=validate_args,
            reparameterization_type=tfd.FULLY_REPARAMETERIZED,
            allow_nan_stats=allow_nan_stats,
            parameters=parameters,
            name=name,
        )

    @classmethod
    def _parameter_properties(cls, dtype, num_classes=None):  # noqa: ARG003
        return {
            "k": parameter_properties.ParameterProperties(),
            "mask": parameter_properties.ParameterProperties(
                default_constraining_bijector_fn=parameter_properties.BIJECTOR_NOT_IMPLEMENTED,
                event_ndims=1,
            ),
        }

    @property
    def k(self):
        return self._parameters["k"]

    @property
    def mask(self):
        return self._parameters["mask"]

    def _batch_shape(self):
        return jnp.shape(self._mask)[:-1]

    def _event_shape(self):
        return tensorshape_util.constant_value_as_shape(
            jnp.expand_dims(self._k, axis=0)
        )

    def _sample_n(self, n, seed=None):
        seed = tfp.random.sanitize_seed(seed, salt="KCategorical._sample_n")
        u = tfd.Uniform(
            low=jnp.zeros(self._mask.shape, dtype=jnp.float32),
            high=jnp.ones(self._mask.shape, dtype=jnp.float32),
        ).sample((n,), seed=seed)
        u = u * self._parameters["mask"]

        _, x = jax.lax.top_k(u, k=self._k)
        return x

    def _log_prob(self, _):
        N = jnp.count_nonzero(self._mask, axis=-1)
        return -_log_choose(N, self._k).astype(self._float_dtype)
