# DRES: Danish Resilient Encryption Scheme

[![PyPI version](https://img.shields.io/pypi/v/dres.svg?style=flat-square)](https://pypi.org/project/dres/)
[![Python Version](https://img.shields.io/pypi/pyversions/dres.svg?style=flat-square)](https://pypi.org/project/dres/)
[![License](https://img.shields.io/pypi/l/dres.svg?style=flat-square)](https://github.com/1Danish-00/HtmlWebShot/blob/main/LICENSE)
<!-- [![Build Status](https://img.shields.io/github/actions/workflow/status/your-username/DRES/python-package.yml?branch=main&style=flat-square)](https://github.com/your-username/DRES/actions) -->

A pure-Python hybrid encryption library created for academic research, featuring a pluggable, custom-designed cryptographic layer.

---

### 🎓 An Academic Project with a Unique Core

`DRES` is not just *another* encryption library. It was built as part of an M.Tech project to demonstrate a mature understanding of both established cryptographic principles and novel algorithm design.

It operates in two distinct modes:

1.  **Standard Mode `🛡️`**:
    A robust, industry-standard hybrid encryption scheme. It combines Diffie-Hellman, AES-128-CBC, and an Encrypt-then-MAC scheme with HMAC-SHA256. This mode is secure by modern standards.

2.  **Academic Mode `🔬`**:
    This mode adds the project's unique contribution: the **DOL (Danish Obfuscation Layer)**. The DOL is a custom-designed, key-derived stream cipher that pre-encrypts the data *before* it's passed to AES. This demonstrates a novel, layered approach to encryption, perfect for analysis and research.

---

### Features

* **Pure Python**: No external dependencies needed.
* **Hybrid Encryption**: Combines the efficiency of symmetric AES with the security of asymmetric Diffie-Hellman key exchange.
* **Perfect Forward Secrecy**: Uses ephemeral keys for each session, so a compromised long-term key cannot decrypt past messages.
* **Authenticated Encryption**: Implements an **Encrypt-then-MAC** scheme using HMAC-SHA256 to prevent tampering.
* **Secure Key Derivation**: Uses HKDF to derive separate, cryptographically isolated keys for AES, HMAC, and the DOL.
* **Novel Academic Component**: Includes the pluggable "Danish Obfuscation Layer" (DOL) for research and analysis.

### Installation

```bash
pip install dres
````

### Quick Start: Standard Mode `🛡️`

This example shows a simple, secure encryption from Alice to Bob.

```python
from dres import DRESCipher, KeyExchange
from dres.exceptions import AuthenticationError

# 1. Initialize the cipher engine
cipher = DRESCipher()

# 2. Both parties generate their long-term key pairs.
#    (They would share their public keys beforehand)
alice_private, alice_public = KeyExchange.generate_keypair()
bob_private, bob_public = KeyExchange.generate_keypair()

# 3. Alice encrypts a message for Bob using his public key.
#    We explicitly set academic_mode=False for standard security.
message = b"This is a standard, secure message for Bob."
print("Encrypting (Standard Mode)...")

encrypted_package = cipher.encrypt(
    message,
    bob_public,
    academic_mode=False  # Use the standard AES-only mode
)

# 4. Bob receives the package and decrypts it with his private key.
print("Decrypting...")
try:
    decrypted_message = cipher.decrypt(encrypted_package, bob_private)
    
    print(f"\nSuccess! Decrypted: '{decrypted_message.decode()}'")
    assert message == decrypted_message

except AuthenticationError:
    print("\n[!] FATAL: Message authentication failed! Package was tampered with.")
except Exception as e:
    print(f"\n[!] An error occurred: {e}")
```

### Advanced Usage: Academic Mode `🔬`

This is the core of the M.Tech project. To use your custom layer, simply set the `academic_mode` flag to `True`.

```python
# ... (setup is the same as above) ...

# Alice encrypts a message using the DOL + AES
message_academic = b"This message is secured by the custom DOL + AES."
print("\nEncrypting (Academic Mode)...")

academic_package = cipher.encrypt(
    message_academic,
    bob_public,
    academic_mode=True  # Use the custom Danish Obfuscation Layer
)

# Bob decrypts. The library automatically detects the mode.
print("Decrypting...")
try:
    decrypted_academic = cipher.decrypt(academic_package, bob_private)
    
    print(f"\nSuccess! Decrypted: '{decrypted_academic.decode()}'")
    assert message_academic == decrypted_academic

except AuthenticationError:
    print("\n[!] FATAL: Message authentication failed! Package was tampered with.")
```

-----

### How It Works: The DRES Pipeline

`DRES` follows a modern cryptographic pipeline.

1.  **Key Exchange**: Alice and Bob use **Diffie-Hellman** to establish a mutual `shared_secret`.
2.  **Key Derivation**: The `shared_secret` is fed into **HKDF (HMAC-based KDF)** to "split" it into three cryptographically separate keys:
      * `aes_key` (for the block cipher)
      * `hmac_key` (for the authentication tag)
      * `dol_key` (for the custom stream cipher)
3.  **Encryption (Academic Mode Pipeline)**:
    ```
    [Plaintext]
         |
    (XOR w/ DOL Keystream)  <- [DOL 🔬] (Your custom HASH-PRNG)
         |
    [Obfuscated Text]
         |
    (Encrypt w/ AES-CBC)   <- [AES 🛡️]
         |
    [Ciphertext]
         |
    (HMAC(IV + Ciphertext)) <- [HMAC 🏷️]
         |
    [Final Package: Flag + IV + HMAC + Ciphertext]
    ```

#### What is the Danish Obfuscation Layer (DOL)?

The **DOL** is the novel component of this project. It is a simple **stream cipher** that uses a HASH-PRNG (Pseudo-Random Number Generator).

It works by generating a unique "keystream" of pseudo-random bytes:

  * `Keystream Block 1 = SHA256(dol_key + 0)`
  * `Keystream Block 2 = SHA256(dol_key + 1)`
  * ...

This keystream is then **XORed** against the plaintext. The resulting obfuscated text is then passed to the standard AES-CBC algorithm for the second layer of encryption.

### License

This project is open-sourced under the **MIT License**. See the [LICENSE](https://www.google.com/search?q=LICENSE) file for more details.

-----

*This library was created with ❤️ by **Danish** as part of a Master of Technology project.*
