import 'svelte/internal/disclose-version';
import * as $ from 'svelte/internal/client';
import * as d3 from 'd3';
import { map, extent, InternSet, range, rollup, scaleLinear, scaleBand, rgb, interpolateInferno, scaleOrdinal } from 'd3';
import { descending, sum, extent as extent$1 } from 'd3-array';

// Helper: convert [0–1, 0–1, 0–1] → "rgb(R, G, B)"
function rgbArrayToCss(rgbArray) {
    const [r, g, b] = rgbArray.map(v => Math.round(v * 255));
    return `rgb(${r}, ${g}, ${b})`;
  }

function style(props) {
  return Object.entries(props)
    .map(([key, value]) => `${key.replace(/([A-Z])/g, '-$1').toLowerCase()}: ${value}`)
    .join('; ');
}

  // // imported from matlab version. Normalized RGB color definitions
  const rawColors = {
    blue: [43, 103, 198].map(v => v / 255),
    red: [198, 43, 103].map(v => v / 255),

    paleblue: [195, 230, 243].map(v => v / 255),
    palered: [255, 204, 204].map(v => v / 255),

    lightergrey: [1, 1, 1].map(v => v * 0.96),
    lightishgrey: [1, 1, 1].map(v => v * 0.93),
    lightgrey: [1, 1, 1].map(v => v * 0.90),

    lightgreyer: [1, 1, 1].map(v => v * 0.85),
    lightgreyish: [1, 1, 1].map(v => v * 0.80),

    grey: [1, 1, 1].map(v => v * 0.75),
    darkgrey: [1, 1, 1].map(v => v * 0.55),
    darkergrey: [1, 1, 1].map(v => v * 0.50),
    verydarkgrey: [1, 1, 1].map(v => v * 0.55),
    superdarkgrey: [1, 1, 1].map(v => v * 0.55),
    reallyverdarkgrey: [1, 1, 1].map(v => v * 0.05),

    orange: [255, 116, 0].map(v => v / 255)
  };

  // Create CSS strings for each
  const cssColors = {};
  for (const [key, rgb] of Object.entries(rawColors)) {
    cssColors[key] = rgbArrayToCss(rgb);
  }

  // Export both raw RGB arrays and CSS strings
  const alloColors = {
    css: cssColors     // e.g., colors.css.blue → "rgb(43, 103, 198)"
  };

// System font stack in order of preference
const alloFonts = {
  family: `"EB Garamond", "Garamond", "Century Schoolbook L", "URW Bookman L", "Bookman Old Style", "Times", serif`,
  sizes: {
    md: "14px",
    lg: "16px"}

};

var root_1$7 = $.from_svg(`<g><line x1="0" x2="0" y1="0" y2="6" stroke-width="0.5"></line></g><g><text dx="5" dy="13" text-anchor="start"> </text></g>`, 1);
var root$8 = $.from_svg(`<g class="axis x"><!><g class="xlab"><text dy="45">Rank r</text><text dy="63">for</text><text dy="80"> </text><text dy="60">more →</text><text dy="75">frequent</text><text dy="60">← less</text><text dy="75">frequent</text></g></g>`);

function AxisX($$anchor, $$props) {
	$.push($$props, true);

	let logFormat10 = $.derived(() => $$props.scale.tickFormat());
	let xTicks = $.derived(() => $$props.scale.ticks().filter((t) => t >= 1 && Number.isInteger(Math.log10(t))));

	let axisStyles = () => style({
		fontFamily: alloFonts.family,
		fontSize: alloFonts.sizes.lg,
		fill: alloColors.css.darkergrey,
		textAnchor: "middle"
	});

	let helperText = () => style({
		fontFamily: alloFonts.family,
		fontSize: alloFonts.sizes.md,
		fill: alloColors.css.darkgrey,
		textAnchor: "middle"
	});

	var g = root$8();
	var node = $.child(g);

	$.each(node, 17, () => $.get(xTicks), $.index, ($$anchor, tick) => {
		var fragment = root_1$7();
		var g_1 = $.first_child(fragment);
		var line = $.child(g_1);

		$.reset(g_1);

		var g_2 = $.sibling(g_1);
		var text = $.child(g_2);
		var text_1 = $.child(text, true);

		$.reset(text);
		$.reset(g_2);

		$.template_effect(
			($0, $1, $2) => {
				$.set_attribute(g_1, 'transform', `translate(${$0 ?? ''}, 0)`);
				$.set_attribute(line, 'stroke', alloColors.css.darkergrey);
				$.set_attribute(g_2, 'transform', `translate(${$1 ?? ''}, 0) scale(-1,1) rotate(45)`);
				$.set_attribute(text, 'font-family', alloFonts.family);
				$.set_attribute(text, 'font-size', alloFonts.sizes.md);
				$.set_attribute(text, 'fill', alloColors.css.darkergrey);
				$.set_text(text_1, $2);
			},
			[
				() => $$props.scale($.get(tick)),
				() => $$props.scale($.get(tick)),
				() => $.get(logFormat10)($.get(tick))
			]
		);

		$.append($$anchor, fragment);
	});

	var g_3 = $.sibling(node);
	var text_2 = $.child(g_3);
	var text_3 = $.sibling(text_2);
	var text_4 = $.sibling(text_3);
	var text_5 = $.child(text_4, true);

	$.reset(text_4);

	var text_6 = $.sibling(text_4);
	var text_7 = $.sibling(text_6);
	var text_8 = $.sibling(text_7);

	$.set_attribute(text_8, 'x', 40);

	var text_9 = $.sibling(text_8);

	$.set_attribute(text_9, 'x', 40);
	$.reset(g_3);
	$.reset(g);

	$.template_effect(
		($0, $1, $2, $3, $4, $5, $6) => {
			$.set_attribute(g, 'transform', `translate(0, ${$$props.innerHeight ?? ''})`);
			$.set_attribute(text_2, 'x', $$props.innerHeight / 2);
			$.set_attribute(text_2, 'transform', `scale(-1,1) translate(-${$$props.innerHeight ?? ''}, 0)`);
			$.set_style(text_2, $0);
			$.set_attribute(text_3, 'x', $$props.innerHeight / 2);
			$.set_attribute(text_3, 'transform', `scale(-1,1) translate(-${$$props.innerHeight ?? ''}, 0)`);
			$.set_style(text_3, $1);
			$.set_attribute(text_4, 'x', $$props.innerHeight / 2);
			$.set_attribute(text_4, 'transform', `scale(-1,1) translate(-${$$props.innerHeight ?? ''}, 0)`);
			$.set_style(text_4, $2);
			$.set_text(text_5, $$props.title[1]);
			$.set_attribute(text_6, 'x', $$props.innerHeight - 40);
			$.set_attribute(text_6, 'transform', `scale(-1,1) translate(-${$$props.innerHeight ?? ''}, 0)`);
			$.set_style(text_6, $3);
			$.set_attribute(text_7, 'x', $$props.innerHeight - 40);
			$.set_attribute(text_7, 'transform', `scale(-1,1) translate(-${$$props.innerHeight ?? ''}, 0)`);
			$.set_style(text_7, $4);
			$.set_attribute(text_8, 'transform', `scale(-1,1) translate(-${$$props.innerHeight ?? ''}, 0)`);
			$.set_style(text_8, $5);
			$.set_attribute(text_9, 'transform', `scale(-1,1) translate(-${$$props.innerHeight ?? ''}, 0)`);
			$.set_style(text_9, $6);
		},
		[
			axisStyles,
			axisStyles,
			axisStyles,
			helperText,
			helperText,
			helperText,
			helperText
		]
	);

	$.append($$anchor, g);
	$.pop();
}

var root_1$6 = $.from_svg(`<g><line x1="0" x2="-6" y1="0" y2="0" stroke-width="0.5"></line></g><g><text dx="-5" dy="13" text-anchor="end"> </text></g>`, 1);
var root$7 = $.from_svg(`<g class="axis y"><!><g class="ylab" transform="rotate(90)"><text dy="45">Rank r</text><text dy="63">for</text><text dy="80"> </text><text dy="60">less →</text><text dy="75">frequent</text><text dy="60">← more</text><text dy="75">frequent</text></g></g>`);

function AxisY($$anchor, $$props) {
	$.push($$props, true);

	let logFormat10 = $.derived(() => $$props.scale.tickFormat());
	let yTicks = $.derived(() => $$props.scale.ticks().filter((t) => t >= 1 && Number.isInteger(Math.log10(t))));

	let axisStyles = () => style({
		fontFamily: alloFonts.family,
		fontSize: alloFonts.sizes.lg,
		fill: alloColors.css.darkergrey,
		textAnchor: "middle"
	});

	let helperText = () => style({
		fontFamily: alloFonts.family,
		fontSize: alloFonts.sizes.md,
		fill: alloColors.css.darkgrey,
		textAnchor: "middle"
	});

	var g = root$7();
	var node = $.child(g);

	$.each(node, 17, () => $.get(yTicks), $.index, ($$anchor, tick) => {
		var fragment = root_1$6();
		var g_1 = $.first_child(fragment);
		var line = $.child(g_1);

		$.reset(g_1);

		var g_2 = $.sibling(g_1);
		var text = $.child(g_2);
		var text_1 = $.child(text, true);

		$.reset(text);
		$.reset(g_2);

		$.template_effect(
			($0, $1, $2) => {
				$.set_attribute(g_1, 'transform', `translate(0, ${$0 ?? ''})`);
				$.set_attribute(line, 'stroke', alloColors.css.darkergrey);
				$.set_attribute(g_2, 'transform', `translate(0, ${$1 ?? ''}) rotate(45)`);
				$.set_attribute(text, 'font-family', alloFonts.family);
				$.set_attribute(text, 'font-size', alloFonts.sizes.md);
				$.set_attribute(text, 'fill', alloColors.css.darkergrey);
				$.set_text(text_1, $2);
			},
			[
				() => $$props.scale($.get(tick)),
				() => $$props.scale($.get(tick)),
				() => $.get(logFormat10)($.get(tick))
			]
		);

		$.append($$anchor, fragment);
	});

	var g_3 = $.sibling(node);
	var text_2 = $.child(g_3);
	var text_3 = $.sibling(text_2);
	var text_4 = $.sibling(text_3);
	var text_5 = $.child(text_4, true);

	$.reset(text_4);

	var text_6 = $.sibling(text_4);
	var text_7 = $.sibling(text_6);
	var text_8 = $.sibling(text_7);

	$.set_attribute(text_8, 'x', 40);

	var text_9 = $.sibling(text_8);

	$.set_attribute(text_9, 'x', 40);
	$.reset(g_3);
	$.reset(g);

	$.template_effect(
		($0, $1, $2, $3, $4, $5, $6) => {
			$.set_attribute(g, 'transform', `translate(${$$props.innerHeight ?? ''}, 0) scale(-1, 1)`);
			$.set_attribute(text_2, 'x', $$props.innerHeight / 2);
			$.set_style(text_2, $0);
			$.set_attribute(text_3, 'x', $$props.innerHeight / 2);
			$.set_style(text_3, $1);
			$.set_attribute(text_4, 'x', $$props.innerHeight / 2);
			$.set_style(text_4, $2);
			$.set_text(text_5, $$props.title[0]);
			$.set_attribute(text_6, 'x', $$props.innerHeight - 40);
			$.set_style(text_6, $3);
			$.set_attribute(text_7, 'x', $$props.innerHeight - 40);
			$.set_style(text_7, $4);
			$.set_style(text_8, $5);
			$.set_style(text_9, $6);
		},
		[
			axisStyles,
			axisStyles,
			axisStyles,
			helperText,
			helperText,
			helperText,
			helperText
		]
	);

	$.append($$anchor, g);
	$.pop();
}

var root_1$5 = $.from_svg(`<g><line y2="0" stroke="#d3d3d3" stroke-dasharray="1,3"></line></g><g><line x2="0" stroke="#d3d3d3" stroke-dasharray="1,3"></line></g>`, 1);
var root$6 = $.from_svg(`<g class="grid"></g>`);

function Grid($$anchor, $$props) {
	$.push($$props, true);

	let xygridLines = $.derived(() => $$props.scale.ticks($$props.ncells / 2));
	var g = root$6();

	$.each(g, 21, () => $.get(xygridLines), $.index, ($$anchor, yline, index) => {
		var fragment = root_1$5();
		var g_1 = $.first_child(fragment);
		var line = $.child(g_1);

		$.set_attribute(line, 'opacity', index === 0 ? 0 : 1);
		$.reset(g_1);

		var g_2 = $.sibling(g_1);
		var line_1 = $.child(g_2);

		$.set_attribute(line_1, 'opacity', index === 0 ? 0 : 1);
		$.reset(g_2);

		$.template_effect(
			($0, $1) => {
				$.set_attribute(g_1, 'transform', `translate(${$0 ?? ''}, 0)`);
				$.set_attribute(line, 'y1', $$props.height);
				$.set_attribute(g_2, 'transform', `translate(0, ${$1 ?? ''})`);
				$.set_attribute(line_1, 'x1', $$props.height);
			},
			[
				() => $$props.wxy($.get(yline)),
				() => $$props.wxy($.get(yline))
			]
		);

		$.append($$anchor, fragment);
	});

	$.reset(g);
	$.append($$anchor, g);
	$.pop();
}

var root_1$4 = $.from_svg(`<path fill="none" stroke="grey" stroke-width="0.55" stroke-opacity="0.9"></path>`);
var root$5 = $.from_svg(`<g class="contours"></g>`);

function Contours($$anchor, $$props) {
	$.push($$props, true);

	function alpha_norm_type2(x1, x2, alpha) {
		if (alpha == 0) {
			return Math.abs(Math.log(x1 / x2));
		} else if (alpha === Infinity) {
			return x1 === x2 ? 0 : Math.max(x1, x2);
		} else {
			const prefactor = (alpha + 1) / alpha;
			const power = 1 / (alpha + 1);

			return prefactor * Math.abs(Math.pow(x1, alpha) - Math.pow(x2, alpha)) ** power;
		}
	}

	function make_grid(Ninset, tmpr1, tmpr2, alpha, divnorm) {
		const deltamatrix = Array.from({ length: Ninset }, () => Array(Ninset).fill(0));

		for (let i = 0; i < Ninset; i++) {
			for (let j = 0; j < Ninset; j++) {
				const divElem = alpha_norm_type2(1 / tmpr1[i], 1 / tmpr2[j], alpha);

				deltamatrix[i][j] = divElem / divnorm;
			}

			deltamatrix[i][i] = -1;

			if (i < Ninset - 1) {
				deltamatrix[i][i + 1] = -1;
				deltamatrix[i + 1][i] = -1;
			}
		}

		return deltamatrix;
	}

	function filter_contours(tmpcontours, Ninset, maxlog10) {
		const chart2val = d3.scaleLinear().domain([0, Ninset]).range([0, maxlog10]);
		let out = [];

		tmpcontours.forEach((contour) => {
			contour.coordinates.forEach((pair, i) => {
				const tmpr1 = pair[0].map((d) => d[0]);
				const tmpr2 = pair[0].map((d) => d[1]);
				const filteredPairs = [];

				for (let index = 0; index < tmpr1.length - 1; index++) {
					const x1 = chart2val(tmpr1[index]);
					const x2 = chart2val(tmpr2[index]);
					const tmpxrot = Math.abs(x2 - x1) / Math.sqrt(2);

					if (Math.abs(tmpxrot) >= 0.1 & x1 != maxlog10 & x2 != 0 & x1 != 0 & x2 != maxlog10) {
						filteredPairs.push([x1, x2]);
					}
				}

				if (filteredPairs.length > 0) {
					out.push(filteredPairs);
				}
			});
		});

		return out;
	}

	function get_contours(alpha, maxlog10, divnorm) {
		const Ninset = 10 ** 3;
		const tmpr1 = d3.range(0, 1000).map((d) => Math.pow(10, d / 999 * 5));
		const tmpr2 = d3.range(0, 1000).map((d) => Math.pow(10, d / 999 * 5));
		const Ncontours = 10;
		const scale = d3.scaleLinear().domain([0, Ncontours + 1]).range([1, tmpr1.length]);
		const contour_indices = d3.range(Ncontours + 2).map((i) => Math.round(scale(i)));
		const grid = make_grid(Ninset, tmpr1, tmpr2, alpha, divnorm);
		const indices = contour_indices.slice(1, -1);
		const lastRow = grid[grid.length - 1];
		const heights = indices.map((index) => lastRow[index]);
		const logTmpr = tmpr1.map(Math.log10);
		const contourGenerator = d3.contours().size([logTmpr.length, logTmpr.length]).thresholds(heights);
		const flatDeltamatrix = grid.flat();
		const tmpcontours = contourGenerator(flatDeltamatrix);

		return filter_contours(tmpcontours, Ninset, maxlog10);
	}

	// Only calculate contours in browser
	let mycontours = $.derived(() => get_contours($$props.alpha, $$props.maxlog10, $$props.divnorm));
	const x = $.derived(() => d3.scaleLinear([0, $$props.maxlog10], [0, $$props.DiamondInnerHeight]));
	const y = $.derived(() => d3.scaleLinear([$$props.maxlog10, 0], [$$props.DiamondInnerHeight, 0]));
	const pathData = d3.line().x((d, i) => $.get(x)(d[0])).y((d, i) => $.get(y)(d[1]));
	var g = root$5();

	$.each(g, 21, () => $.get(mycontours), $.index, ($$anchor, contour) => {
		var path = root_1$4();

		$.template_effect(($0) => $.set_attribute(path, 'd', $0), [() => pathData($.get(contour))]);
		$.append($$anchor, path);
	});

	$.reset(g);
	$.append($$anchor, g);
	$.pop();
}

function updateTooltipPosition(event, tooltipVisible, tooltipX, tooltipY) {
	if ($.get(tooltipVisible)) {
		$.set(tooltipX, event.clientX + 15);
		$.set(tooltipY, event.clientY - 10);
	}
}

var root_1$3 = $.from_svg(`<rect></rect>`);
var root_2$3 = $.from_svg(`<rect role="button" tabindex="0"></rect>`);
var root_4$1 = $.from_svg(`<text dy="5"> </text>`);
var root_5 = $.from_html(`<div><!></div>`);
var root$4 = $.from_html(`<div style="position: relative;"><svg xmlns="http://www.w3.org/2000/svg" style="overflow: visible; display: block;"><g><polygon class="diamond-background grey-triangle" fill-opacity="0.8" stroke="black" stroke-width="0.5"></polygon><polygon class="diamond-background blue-triangle" fill-opacity="0.8" stroke="black" stroke-width="0.5"></polygon><!><!><!><!><!><!><line x1="0" y1="0"></line><!></g></svg> <!></div>`);

function Diamond($$anchor, $$props) {
	$.push($$props, true);

	let DiamondHeight = $.prop($$props, 'DiamondHeight', 3, 600),
		marginInner = $.prop($$props, 'marginInner', 3, 160),
		marginDiamond = $.prop($$props, 'marginDiamond', 3, 40),
		highlightedTerm = $.prop($$props, 'highlightedTerm', 3, null),
		highlightedSystem = $.prop($$props, 'highlightedSystem', 3, null),
		labelThreshold = $.prop($$props, 'labelThreshold', 3, Infinity);

	// Extract data from dat object
	let diamond_dat = $.derived(() => $$props.dat.counts);
	$.derived(() => $$props.dat.deltas);
	// Calculate derived dimensions (matching D3 version exactly)
	let innerHeight = $.derived(() => DiamondHeight() - marginInner());
	let diamondHeight = $.derived(() => $.get(innerHeight) - marginDiamond());

	// Move rin function here before it's used
	function rin(arr1, arr2) {
		return Array.from(arr1, (x) => arr2.indexOf(x) !== -1);
	}

	// Move filter_labs function here and make it work without relevant_types parameter
	function filter_labs(d, relevant_types_array) {
		if (!relevant_types_array || relevant_types_array.length === 0) return false;

		// Check if the cell has too many types (exceeds threshold)
		const typeCount = d.types.split(",").length;

		if (typeCount > labelThreshold()) return false;
		return rin(relevant_types_array, d.types.split(",")).some((x) => x === true);
	}

	function get_relevant_types(diamond_dat) {
		const ncells = d3.max(diamond_dat, (d) => d.x1);
		const cumbin = d3.range(0, ncells, 1.5);
		const relevant_types = [];
		// Count how often each type appears across all cells (for selectivity)
		const type_frequency = new Map();

		diamond_dat.filter((d) => d.value > 0).forEach((d) => {
			d.types.split(",").forEach((type) => {
				const clean_type = type.trim();

				type_frequency.set(clean_type, (type_frequency.get(clean_type) || 0) + 1);
			});
		});

		for (let sys of ["right", "left"]) {

			for (let i = 1; i < cumbin.length; i++) {
				const filtered_dat = diamond_dat.filter((d) => d.value > 0 && d.which_sys == sys).filter((d) => d.coord_on_diag >= cumbin[i - 1] && d.coord_on_diag < cumbin[i]);

				if (filtered_dat.length > 0) {
					const cos_dists = filtered_dat.map((d) => d.cos_dist);
					const max_dist = cos_dists.reduce((a, b) => Math.max(a, b));
					const max_dist_idx = cos_dists.indexOf(max_dist);
					const types = filtered_dat[max_dist_idx]['types'].split(",").map((t) => t.trim());
					// Instead of random selection, pick the LEAST common type
					// This will give us more distinctive labels
					let selected_type = types[0]; // fallback
					let min_frequency = Infinity;

					for (const type of types) {
						const freq = type_frequency.get(type) || 0;

						if (freq < min_frequency && freq > 0) {
							min_frequency = freq;
							selected_type = type;
						}
					}

					relevant_types.push(selected_type);
				}
			}
		}

		return relevant_types;
	}

	// Now derive relevant_types from diamond_dat
	let relevant_types = $.derived(() => get_relevant_types($.get(diamond_dat)));
	// Calculate filtered cells count AFTER relevant_types is defined
	$.derived(() => $.get(relevant_types) ? $.get(diamond_dat).filter((d) => filter_labs(d, $.get(relevant_types))).length : 0);
	// Wrangling data
	let ncells = $.derived(() => d3.max($.get(diamond_dat), (d) => d.x1));
	let max_rank = $.derived(() => d3.max($.get(diamond_dat), (d) => d.rank_L[1]));
	let rounded_max_rank = $.derived(() => 10 ** Math.ceil(Math.log10($.get(max_rank))));
	let xyDomain = $.derived(() => [1, $.get(rounded_max_rank)]);
	// Scales (matching D3 version dimensions)
	let xy = $.derived(() => d3.scaleBand().domain($.get(diamond_dat).map((d) => d.y1)).range([0, $.get(diamondHeight)]));
	let logScale = $.derived(() => d3.scaleLog().domain($.get(xyDomain)).range([0, $.get(innerHeight)]).nice());
	let linScale = $.derived(() => d3.scaleLinear().domain([0, $.get(ncells) - 1]).range([0, $.get(innerHeight)]));
	let wxy = $.derived(() => d3.scaleBand().domain(d3.range($.get(ncells))).range([0, $.get(innerHeight)]));
	let color_scale = d3.scaleSequentialLog().domain([$.get(rounded_max_rank), 1]).interpolator(d3.interpolateInferno);

	// Background triangles
	let blue_triangle = $.derived(() => [
		[
			$.get(innerHeight),
			$.get(innerHeight)
		],
		[0, 0],
		[0, $.get(innerHeight)]
	].join(" "));

	let grey_triangle = $.derived(() => [
		[
			$.get(innerHeight),
			$.get(innerHeight)
		],
		[0, 0],
		[$.get(innerHeight), 0]
	].join(" "));

	// TOOLTIP
	let tooltipVisible = $.state(false);
	let tooltipContent = $.state('');
	let tooltipX = $.state(0);
	let tooltipY = $.state(0);

	function showTooltip(event, d) {
		if (d.value === 0) return;

		const tokens = d.types.split(",");
		const displayTokens = tokens.length < 50 ? tokens.slice(0, 8).join(", ") : tokens.slice(0, 8).join(", ") + " ...";

		$.set(tooltipContent, `
        <div style="color: rgb(89, 89, 89); font-size: 11px;">Types: ${displayTokens}</div>
        `);

		$.set(tooltipX, event.clientX + 15);
		$.set(tooltipY, event.clientY - 10);
		$.set(tooltipVisible, true);
	}

	function hideTooltip() {
		$.set(tooltipVisible, false);
	}

	var div = root$4();
	var svg = $.child(div);
	var g = $.child(svg);
	var polygon = $.child(g);
	var polygon_1 = $.sibling(polygon);
	var node = $.sibling(polygon_1);

	AxisX(node, {
		get innerHeight() {
			return $.get(innerHeight);
		},
		get scale() {
			return $.get(logScale);
		},
		get title() {
			return $$props.title;
		}
	});

	var node_1 = $.sibling(node);

	AxisY(node_1, {
		get innerHeight() {
			return $.get(innerHeight);
		},
		get scale() {
			return $.get(logScale);
		},
		get title() {
			return $$props.title;
		}
	});

	var node_2 = $.sibling(node_1);

	Grid(node_2, {
		get height() {
			return $.get(innerHeight);
		},
		get wxy() {
			return $.get(wxy);
		},
		get ncells() {
			return $.get(ncells);
		},
		get scale() {
			return $.get(linScale);
		}
	});

	var node_3 = $.sibling(node_2);

	$.each(node_3, 17, () => $.get(diamond_dat), $.index, ($$anchor, d) => {
		var rect = root_1$3();
		const typesList = $.derived(() => $.get(d).types.split(',').map((t) => t.trim()));
		const isHighlighted = $.derived(() => highlightedTerm() && $.get(typesList).includes(highlightedTerm()) && (!highlightedSystem() || $.get(d).which_sys === highlightedSystem()));
		let classes;

		$.template_effect(
			($0, $1, $2, $3, $4, $5) => {
				classes = $.set_class(rect, 0, 'diamond-cell', null, classes, $0);
				$.set_attribute(rect, 'x', $1);
				$.set_attribute(rect, 'y', $2);
				$.set_attribute(rect, 'width', $3);
				$.set_attribute(rect, 'height', $4);
				$.set_attribute(rect, 'fill', $5);
				$.set_attribute(rect, 'stroke', $.get(isHighlighted) ? "#ff6b6b" : "none");
				$.set_attribute(rect, 'stroke-width', $.get(isHighlighted) ? 2 : 0);
				$.set_style(rect, `opacity: ${highlightedTerm() && !$.get(isHighlighted) ? 0.3 : 1};`);
			},
			[
				() => ({ highlighted: $.get(isHighlighted) }),
				() => $.get(xy)($.get(d).x1),
				() => $.get(xy)($.get(d).y1),
				() => $.get(xy).bandwidth(),
				() => $.get(xy).bandwidth(),
				() => $.get(d).value === 0 ? "none" : color_scale($.get(d).value)
			]
		);

		$.append($$anchor, rect);
	});

	var node_4 = $.sibling(node_3);

	$.each(node_4, 17, () => $.get(diamond_dat), $.index, ($$anchor, d) => {
		var rect_1 = root_2$3();

		rect_1.__mousemove = [
			updateTooltipPosition,
			tooltipVisible,
			tooltipX,
			tooltipY
		];

		$.template_effect(
			($0, $1, $2, $3) => {
				$.set_attribute(rect_1, 'x', $0);
				$.set_attribute(rect_1, 'y', $1);
				$.set_attribute(rect_1, 'width', $2);
				$.set_attribute(rect_1, 'height', $3);
				$.set_attribute(rect_1, 'fill', $.get(d).value > 0 ? 'rgba(255,255,255,0.001)' : 'none');
				$.set_attribute(rect_1, 'stroke', $.get(d).value > 0 ? alloColors.css.darkergrey : 'none');
				$.set_attribute(rect_1, 'stroke-width', $.get(d).value > 0 ? '1.18' : '0');
				$.set_attribute(rect_1, 'stroke-opacity', $.get(d).value > 0 ? '0.4' : '0');
				$.set_style(rect_1, `cursor: ${$.get(d).value > 0 ? 'pointer' : 'default'};`);
			},
			[
				() => $.get(xy)($.get(d).x1),
				() => $.get(xy)($.get(d).y1),
				() => $.get(xy).bandwidth(),
				() => $.get(xy).bandwidth()
			]
		);

		$.event('mouseenter', rect_1, (e) => showTooltip(e, $.get(d)));
		$.event('mouseleave', rect_1, hideTooltip);
		$.append($$anchor, rect_1);
	});

	var node_5 = $.sibling(node_4);

	{
		var consequent = ($$anchor) => {
			var fragment = $.comment();
			var node_6 = $.first_child(fragment);

			$.each(node_6, 17, () => $.get(diamond_dat).filter((d) => filter_labs(d, $.get(relevant_types))), $.index, ($$anchor, d) => {
				var text = root_4$1();
				var text_1 = $.child(text, true);

				$.reset(text);

				$.template_effect(
					($0, $1, $2, $3, $4, $5) => {
						$.set_attribute(text, 'x', $0);
						$.set_attribute(text, 'y', $1);
						$.set_attribute(text, 'dx', $.get(d).x1 - $.get(d).y1 <= 0 ? 5 : -5);
						$.set_attribute(text, 'text-anchor', $.get(d).x1 - $.get(d).y1 <= 0 ? "start" : "end");
						$.set_attribute(text, 'transform', `scale(1,-1) rotate(-90) rotate(-45, ${$2 ?? ''}, ${$3 ?? ''}) translate(${$4 ?? ''}, 0)`);
						$.set_style(text, `font-family: ${alloFonts.family}; font-size: 12px; fill: ${alloColors.css.darkergrey ?? ''};`);
						$.set_text(text_1, $5);
					},
					[
						() => $.get(xy)($.get(d).x1),
						() => Number.isInteger($.get(d).coord_on_diag) ? $.get(xy)($.get(d).y1) : $.get(xy)($.get(d).y1) - 1,
						() => $.get(xy)($.get(d).x1),
						() => $.get(xy)($.get(d).y1),
						() => $.get(d).which_sys === "right" ? $.get(xy)(Math.sqrt($.get(d).cos_dist)) * 1.5 : -$.get(xy)(Math.sqrt($.get(d).cos_dist)) * 1.5,
						() => $.get(d).types.split(",")[0]
					]
				);

				$.append($$anchor, text);
			});

			$.append($$anchor, fragment);
		};

		$.if(node_5, ($$render) => {
			if ($.get(relevant_types)) $$render(consequent);
		});
	}

	var line = $.sibling(node_5);
	var node_7 = $.sibling(line);

	Contours(node_7, {
		get alpha() {
			return $$props.alpha;
		},
		get maxlog10() {
			return $$props.maxlog10;
		},
		get divnorm() {
			return $$props.divnorm;
		},
		get DiamondInnerHeight() {
			return $.get(innerHeight);
		}
	});

	$.reset(g);
	$.reset(svg);

	var node_8 = $.sibling(svg, 2);

	{
		var consequent_1 = ($$anchor) => {
			var div_1 = root_5();
			var node_9 = $.child(div_1);

			$.html(node_9, () => $.get(tooltipContent));
			$.reset(div_1);

			$.template_effect(() => $.set_style(div_1, `
                position: fixed;
                left: ${$.get(tooltipX) ?? ''}px;
                top: ${$.get(tooltipY) ?? ''}px;
                background: white;
                border: 1px solid rgb(200, 200, 200);
                border-radius: 6px;
                padding: 10px 12px;
                font-family: 'EB Garamond', serif;
                font-size: 12px;
                line-height: 1.5;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
                pointer-events: none;
                z-index: 1000;
                max-width: 280px;
            `));

			$.append($$anchor, div_1);
		};

		$.if(node_8, ($$render) => {
			if ($.get(tooltipVisible)) $$render(consequent_1);
		});
	}

	$.reset(div);

	$.template_effect(() => {
		$.set_attribute(svg, 'width', DiamondHeight());
		$.set_attribute(svg, 'height', DiamondHeight());
		$.set_attribute(svg, 'viewBox', `0 0 ${DiamondHeight() ?? ''} ${DiamondHeight() ?? ''}`);
		$.set_attribute(g, 'transform', `matrix(-0.7071, 0.7071, 0.7071, 0.7071, ${DiamondHeight() * 0.5}, ${DiamondHeight() * 0.067})`);
		$.set_attribute(polygon, 'points', $.get(grey_triangle));
		$.set_attribute(polygon, 'fill', alloColors.css.lightgrey);
		$.set_attribute(polygon_1, 'points', $.get(blue_triangle));
		$.set_attribute(polygon_1, 'fill', alloColors.css.paleblue);
		$.set_attribute(line, 'x2', $.get(innerHeight) - 7);
		$.set_attribute(line, 'y2', $.get(innerHeight) - 7);
		$.set_style(line, `stroke: ${alloColors.css.verydarkgrey ?? ''}; stroke-width: 0.5;`);
	});

	$.append($$anchor, div);
	$.pop();
}

$.delegate(['mousemove']);

var root_1$2 = $.from_svg(`<line y1="0" y2="6" style="stroke: currentColor; stroke-width: 1;"></line><line class="wordshift-grid-line" y1="0"></line><text y="-12" text-anchor="middle"> </text>`, 1);
var on_click = (e, onBarClick, $$props, i, Y) => onBarClick()?.(e, $$props.barData[$.get(i)], $.get(Y)[$.get(i)]);
var root_2$2 = $.from_svg(`<rect></rect>`);
var root_4 = $.from_svg(`<text dy="0.32em"> </text>`);
var root_3 = $.from_svg(`<g class="wordshift-label-group"><text dy="0.32em"> </text><!></g>`);
var root$3 = $.from_svg(`<svg style="overflow: visible; display: block;"><g class="wordshift-container"><g class="wordshift-axis x"><!><text y="-35" text-anchor="middle"> </text></g><!><g class="wordshift-y-axis"></g></g></svg>`);

function Wordshift($$anchor, $$props) {
	$.push($$props, true);

	let x = $.prop($$props, 'x', 3, (d) => d.metric),
		y = $.prop($$props, 'y', 3, (d) => d.type),
		marginTop = $.prop($$props, 'marginTop', 3, 50),
		marginRight = $.prop($$props, 'marginRight', 3, 60),
		marginBottom = $.prop($$props, 'marginBottom', 3, 40),
		marginLeft = $.prop($$props, 'marginLeft', 3, 70),
		width = $.prop($$props, 'width', 3, 360),
		xLabel = $.prop($$props, 'xLabel', 3, '← System 1 · Divergence contribution · System 2 →'),
		yPadding = $.prop($$props, 'yPadding', 3, 0),
		colors = $.prop($$props, 'colors', 19, () => [
			alloColors.css.lightgrey,
			alloColors.css.paleblue
		]),
		barHeightFactor = $.prop($$props, 'barHeightFactor', 3, 0.7),
		onBarClick = $.prop($$props, 'onBarClick', 3, undefined),
		selectedBar = $.prop($$props, 'selectedBar', 3, null);

	let X = $.derived(() => d3.map($$props.barData, x()));
	let Y = $.derived(() => d3.map($$props.barData, y()));
	let computedXDomain = $.derived(() => $$props.xDomain || d3.extent($.get(X)));
	let yDomain = $.derived(() => new d3.InternSet($.get(Y)));
	const xAxisYOffset = 10; // Space below x-axis
	const bandHeight = 18; // Fixed band height
	const shiftSvgBy = 12; // shift svg up to align with system titles
	const barYOffset = 10; // Additional offset just for bars
	let compactHeight = $.derived(() => $.get(yDomain).size * bandHeight);
	let innerWidth = $.derived(() => width() - marginLeft() - marginRight());
	let innerHeight = $.derived(() => $.get(compactHeight) + xAxisYOffset);
	let computedHeight = $.derived(() => $.get(innerHeight) + marginTop() + marginBottom());
	// Compute ranges exactly like D3
	let xRange = $.derived(() => [0, $.get(innerWidth)]);

	let yRange = $.derived(() => [
		xAxisYOffset + barYOffset,
		xAxisYOffset + barYOffset + $.get(compactHeight)
	]);

	// Filter indices and create lookup
	let I = $.derived(() => d3.range($.get(X).length).filter((i) => $.get(yDomain).has($.get(Y)[i])));
	let YX = $.derived(() => d3.rollup($.get(I), ([i]) => $.get(X)[i], (i) => $.get(Y)[i]));
	// Scales
	let xScale = $.derived(() => d3.scaleLinear($.get(computedXDomain), $.get(xRange)));
	let yScale = $.derived(() => d3.scaleBand().domain($.get(yDomain)).range($.get(yRange)).padding(yPadding()));
	let format = $.derived(() => d3.format(".2%"));
	let xTicks = $.derived(() => $.get(xScale).ticks(width() / 80));

	// Helper function matching D3 logic exactly
	function parseLabelData(label) {
		const splitIndex = label.indexOf(' ');
		let name_y, numbers_y;

		if (splitIndex === -1) {
			name_y = label;
			numbers_y = "";
		} else {
			name_y = label.slice(0, splitIndex);
			numbers_y = label.slice(splitIndex + 1).trim();

			// Strip first and last characters from numbers_y if possible
			if (numbers_y.length > 2) {
				numbers_y = numbers_y.slice(1, numbers_y.length - 1);
			}
		}

		return { name_y, numbers_y };
	}

	let finalHeight = $.derived(() => $$props.height || $.get(computedHeight));
	var svg = root$3();
	var g = $.child(svg);
	var g_1 = $.child(g);

	$.set_attribute(g_1, 'transform', 'translate(0, 10)');

	var node = $.child(g_1);

	$.each(node, 17, () => $.get(xTicks), $.index, ($$anchor, tick) => {
		var fragment = root_1$2();
		var line = $.first_child(fragment);
		var line_1 = $.sibling(line);
		var text = $.sibling(line_1);
		var text_1 = $.child(text, true);

		$.reset(text);

		$.template_effect(
			($0, $1, $2, $3, $4, $5) => {
				$.set_attribute(line, 'x1', $0);
				$.set_attribute(line, 'x2', $1);
				$.set_attribute(line_1, 'x1', $2);
				$.set_attribute(line_1, 'x2', $3);
				$.set_attribute(line_1, 'y2', $.get(innerHeight) - xAxisYOffset + barYOffset);
				$.set_style(line_1, $.get(tick) === 0 ? `stroke: ${alloColors.css.verydarkgrey}; stroke-width: 1; stroke-opacity: 0.8;` : `stroke: currentColor; stroke-opacity: 0.1;`);
				$.set_attribute(text, 'x', $4);
				$.set_style(text, `font-family: ${alloFonts.family}; font-size: 12px; fill: ${alloColors.css.verydarkgrey ?? ''};`);
				$.set_text(text_1, $5);
			},
			[
				() => $.get(xScale)($.get(tick)),
				() => $.get(xScale)($.get(tick)),
				() => $.get(xScale)($.get(tick)),
				() => $.get(xScale)($.get(tick)),
				() => $.get(xScale)($.get(tick)),
				() => $.get(format)($.get(tick))
			]
		);

		$.append($$anchor, fragment);
	});

	var text_2 = $.sibling(node);
	var text_3 = $.child(text_2, true);

	$.reset(text_2);
	$.reset(g_1);

	var node_1 = $.sibling(g_1);

	$.each(node_1, 17, () => $.get(I), $.index, ($$anchor, i) => {
		var rect = root_2$2();
		const isSelected = $.derived(() => selectedBar() === $.get(Y)[$.get(i)]);
		let classes;

		rect.__click = [on_click, onBarClick, $$props, i, Y];

		$.template_effect(
			($0, $1, $2, $3, $4) => {
				classes = $.set_class(rect, 0, 'wordshift-bar svelte-1xsy6uz', null, classes, $0);
				$.set_attribute(rect, 'x', $1);
				$.set_attribute(rect, 'y', $2);
				$.set_attribute(rect, 'fill', colors()[$.get(X)[$.get(i)] > 0 ? colors().length - 1 : 0]);
				$.set_attribute(rect, 'width', $3);
				$.set_attribute(rect, 'height', $4);

				$.set_style(rect, `
                    mix-blend-mode: multiply;
                    cursor: ${onBarClick() ? 'pointer' : 'default'};
                    opacity: ${$.get(isSelected) ? 1 : selectedBar() ? 0.3 : 1};
                    transition: opacity 0.2s ease, stroke 0.2s ease;
                `);

				$.set_attribute(rect, 'stroke', $.get(isSelected) ? '#ff6b6b' : 'none');
				$.set_attribute(rect, 'stroke-width', $.get(isSelected) ? '2' : '0');
			},
			[
				() => ({ selected: $.get(isSelected) }),
				() => Math.min($.get(xScale)(0), $.get(xScale)($.get(X)[$.get(i)])),
				() => $.get(yScale)($.get(Y)[$.get(i)]) + ($.get(yScale).bandwidth() - $.get(yScale).bandwidth() * barHeightFactor()) / 2,
				() => Math.abs($.get(xScale)($.get(X)[$.get(i)]) - $.get(xScale)(0)),
				() => $.get(yScale).bandwidth() * barHeightFactor()
			]
		);

		$.append($$anchor, rect);
	});

	var g_2 = $.sibling(node_1);

	$.each(g_2, 21, () => $.get(yScale).domain(), $.index, ($$anchor, label) => {
		var g_3 = root_3();
		const labelData = $.derived(() => parseLabelData($.get(label)));
		const xValue = $.derived(() => $.get(YX).get($.get(label)));
		var text_4 = $.child(g_3);
		var text_5 = $.child(text_4, true);

		$.reset(text_4);

		var node_2 = $.sibling(text_4);

		{
			var consequent = ($$anchor) => {
				var text_6 = root_4();
				var text_7 = $.child(text_6, true);

				$.reset(text_6);

				$.template_effect(() => {
					$.set_attribute(text_6, 'x', $.get(xValue) > 0 ? -6 : 6);
					$.set_attribute(text_6, 'text-anchor', $.get(xValue) > 0 ? "end" : "start");
					$.set_style(text_6, `font-family: ${alloFonts.family}; font-size: 14px; fill: ${alloColors.css.darkergrey ?? ''}; opacity: 0.5;`);
					$.set_text(text_7, $.get(labelData).numbers_y);
				});

				$.append($$anchor, text_6);
			};

			$.if(node_2, ($$render) => {
				if ($.get(labelData).numbers_y) $$render(consequent);
			});
		}

		$.reset(g_3);

		$.template_effect(
			($0) => {
				$.set_attribute(g_3, 'transform', `translate(0, ${$0 ?? ''})`);
				$.set_attribute(text_4, 'x', $.get(xValue) > 0 ? 6 : -6);
				$.set_attribute(text_4, 'text-anchor', $.get(xValue) > 0 ? "start" : "end");
				$.set_style(text_4, `font-family: ${alloFonts.family}; font-size: 14px; fill: ${alloColors.css.verydarkgrey ?? ''};`);
				$.set_text(text_5, $.get(labelData).name_y);
			},
			[
				() => $.get(yScale)($.get(label)) + $.get(yScale).bandwidth() / 2
			]
		);

		$.append($$anchor, g_3);
	});

	$.reset(g_2);
	$.reset(g);
	$.reset(svg);

	$.template_effect(
		($0, $1) => {
			$.set_attribute(svg, 'width', width());
			$.set_attribute(svg, 'height', $.get(finalHeight));
			$.set_attribute(svg, 'viewBox', `0 0 ${width() ?? ''} ${$.get(finalHeight) ?? ''}`);
			$.set_attribute(g, 'transform', `translate(${marginLeft() ?? ''}, ${marginTop() - shiftSvgBy})`);
			$.set_attribute(text_2, 'x', $0);
			$.set_style(text_2, `font-family: ${alloFonts.family}; font-size: 16px; fill: ${alloColors.css.verydarkgrey ?? ''};`);
			$.set_text(text_3, xLabel());
			$.set_attribute(g_2, 'transform', `translate(${$1 ?? ''}, 0)`);
		},
		[
			() => $.get(xScale)(0),
			() => $.get(xScale)(0)
		]
	);

	$.append($$anchor, svg);
	$.pop();
}

$.delegate(['click']);

var root_1$1 = $.from_svg(`<rect></rect><text dy="0.35em"> </text>`, 1);
var root_2$1 = $.from_svg(`<text x="0" dy="0.35em" text-anchor="middle"> </text>`);
var root$2 = $.from_svg(`<svg style="overflow: visible; display: block;"><g class="balance-chart"><!><g class="y-axis"></g></g></svg>`);

function DivergingBarChart($$anchor, $$props) {
	$.push($$props, true);

	let x = $.prop($$props, 'x', 3, (d) => d.frequency),
		y = $.prop($$props, 'y', 3, (d) => d.y_coord),
		marginTop = $.prop($$props, 'marginTop', 3, 0),
		marginRight = $.prop($$props, 'marginRight', 3, 40),
		marginBottom = $.prop($$props, 'marginBottom', 3, 10),
		marginLeft = $.prop($$props, 'marginLeft', 3, 40),
		width = $.prop($$props, 'width', 3, 200),
		yPadding = $.prop($$props, 'yPadding', 3, 0.5),
		colors = $.prop($$props, 'colors', 19, () => [
			alloColors.css.lightgrey,
			alloColors.css.paleblue
		]);

	// Compute values (matching D3 version exactly)
	let X = $.derived(() => map($$props.data, x()));
	let Y = $.derived(() => map($$props.data, y()));
	// Compute domains
	let xDomain = $.derived(() => extent($.get(X)));
	let yDomain = $.derived(() => new InternSet($.get(Y)));

	// Compute dimensions
	let xRange = $.derived(() => [
		marginLeft(),
		width() - marginRight()
	]);

	let height = $.derived(() => Math.ceil(($.get(yDomain).size + yPadding()) * 25) + marginTop() + marginBottom());

	let yRange = $.derived(() => [
		marginTop(),
		$.get(height) - marginBottom()
	]);

	// Filter indices and create lookup
	let I = $.derived(() => range($.get(X).length).filter((i) => $.get(yDomain).has($.get(Y)[i])));
	let YX = $.derived(() => rollup($.get(I), ([i]) => $.get(X)[i], (i) => $.get(Y)[i]));
	// Scales
	let xScale = $.derived(() => scaleLinear($.get(xDomain), $.get(xRange)));
	let yScale = $.derived(() => scaleBand().domain($.get(yDomain)).range($.get(yRange)).padding(yPadding()));
	let format = $.derived(() => $.get(xScale).tickFormat(100, "%"));
	var svg = root$2();
	var g = $.child(svg);
	var node = $.child(g);

	$.each(node, 17, () => $.get(I), $.index, ($$anchor, i) => {
		var fragment = root_1$1();
		var rect = $.first_child(fragment);
		var text = $.sibling(rect);
		var text_1 = $.child(text, true);

		$.reset(text);

		$.template_effect(
			($0, $1, $2, $3, $4, $5, $6) => {
				$.set_attribute(rect, 'x', $0);
				$.set_attribute(rect, 'y', $1);
				$.set_attribute(rect, 'fill', colors()[$.get(X)[$.get(i)] > 0 ? colors().length - 1 : 0]);
				$.set_attribute(rect, 'width', $2);
				$.set_attribute(rect, 'height', $3);
				$.set_attribute(text, 'x', $4);
				$.set_attribute(text, 'y', $5);
				$.set_attribute(text, 'text-anchor', $.get(X)[$.get(i)] < 0 ? "end" : "start");
				$.set_style(text, `font-family: ${alloFonts.family}; font-size: 12px; fill: ${alloColors.css.darkergrey ?? ''}; opacity: 0.5;`);
				$.set_text(text_1, $6);
			},
			[
				() => Math.min($.get(xScale)(0), $.get(xScale)($.get(X)[$.get(i)])),
				() => $.get(yScale)($.get(Y)[$.get(i)]),
				() => Math.abs($.get(xScale)($.get(X)[$.get(i)]) - $.get(xScale)(0)),
				() => $.get(yScale).bandwidth(),
				() => $.get(xScale)($.get(X)[$.get(i)]) + Math.sign($.get(X)[$.get(i)] - 0) * 4,
				() => $.get(yScale)($.get(Y)[$.get(i)]) + $.get(yScale).bandwidth() / 2,
				() => $.get(format)(Math.abs($.get(X)[$.get(i)]))
			]
		);

		$.append($$anchor, fragment);
	});

	var g_1 = $.sibling(node);

	$.each(g_1, 21, () => $.get(yScale).domain(), $.index, ($$anchor, label) => {
		var text_2 = root_2$1();
		var text_3 = $.child(text_2, true);

		$.reset(text_2);

		$.template_effect(
			($0, $1) => {
				$.set_attribute(text_2, 'y', $0);
				$.set_attribute(text_2, 'opacity', $1);
				$.set_style(text_2, `font-family: ${alloFonts.family}; font-size: 14px; fill: ${alloColors.css.darkergrey ?? ''};`);
				$.set_text(text_3, $.get(label));
			},
			[
				() => $.get(yScale)($.get(label)) + $.get(yScale).bandwidth() / 2,
				() => $.get(YX).get($.get(label)) ? "0.5" : "1"
			]
		);

		$.append($$anchor, text_2);
	});

	$.reset(g_1);
	$.reset(g);
	$.reset(svg);

	$.template_effect(
		($0) => {
			$.set_attribute(svg, 'width', width());
			$.set_attribute(svg, 'height', $.get(height));
			$.set_attribute(svg, 'viewBox', `0 0 ${width() ?? ''} ${$.get(height) ?? ''}`);
			$.set_attribute(g_1, 'transform', `translate(${$0 ?? ''}, -12)`);
		},
		[() => $.get(xScale)(0)]
	);

	$.append($$anchor, svg);
	$.pop();
}

var root_1 = $.from_svg(`<rect transform="rotate(-90) translate(-70,0)" style="stroke: black; stroke-width: 0.65; shape-rendering: crispEdges;"></rect>`);
var root_2 = $.from_svg(`<g class="tick"><text dx="30" dy="-30" transform="rotate(90)"> </text></g>`);
var root$1 = $.from_svg(`<svg style="overflow: visible; display: block;"><!><g transform="rotate(-90) translate(-60,5)"><!><text class="title" dx="30" dy="-5" transform="rotate(90)">Counts per cell</text></g></svg>`);

function Legend($$anchor, $$props) {
	$.push($$props, true);

	let tickSize = $.prop($$props, 'tickSize', 3, 0),
		height = $.prop($$props, 'height', 19, () => 44 + tickSize()),
		width = $.prop($$props, 'width', 3, 300),
		marginTop = $.prop($$props, 'marginTop', 3, 13),
		marginBottom = $.prop($$props, 'marginBottom', 19, () => 16 + tickSize()),
		marginLeft = $.prop($$props, 'marginLeft', 3, 0),
		N_CATEGO = $.prop($$props, 'N_CATEGO', 3, 20);

	const myramp = range(N_CATEGO()).map((i) => rgb(interpolateInferno(i / (N_CATEGO() - 1))).hex());
	const color = scaleOrdinal(range(N_CATEGO()), myramp);
	let x = $.derived(() => scaleBand().domain(color.domain()).rangeRound([marginLeft(), width() - 100]));
	let x2 = $.derived(() => scaleBand().domain(range($$props.max_count_log).map((i) => 10 ** i).sort((a, b) => b - a)).rangeRound([marginLeft() - 40, width() - 90]));
	var svg = root$1();
	var node = $.child(svg);

	$.each(node, 17, () => color.domain(), $.index, ($$anchor, d) => {
		var rect = root_1();

		$.template_effect(
			($0, $1, $2, $3) => {
				$.set_attribute(rect, 'x', $0);
				$.set_attribute(rect, 'y', marginTop());
				$.set_attribute(rect, 'width', $1);
				$.set_attribute(rect, 'height', $2);
				$.set_attribute(rect, 'fill', $3);
			},
			[
				() => $.get(x)($.get(d)),
				() => Math.max(0, $.get(x).bandwidth()),
				() => $.get(x).bandwidth(),
				() => color($.get(d))
			]
		);

		$.append($$anchor, rect);
	});

	var g = $.sibling(node);
	var node_1 = $.child(g);

	$.each(node_1, 17, () => $.get(x2).domain(), $.index, ($$anchor, tick) => {
		var g_1 = root_2();
		var text = $.child(g_1);
		var text_1 = $.child(text, true);

		$.reset(text);
		$.reset(g_1);

		$.template_effect(
			($0) => {
				$.set_attribute(g_1, 'transform', `translate(${$0 ?? ''}, 0)`);
				$.set_style(text, `font-family: ${alloFonts.family}; font-size: 14px; fill: ${alloColors.css.verydarkgrey ?? ''}; text-anchor: start;`);
				$.set_text(text_1, $.get(tick));
			},
			[() => $.get(x2)($.get(tick))]
		);

		$.append($$anchor, g_1);
	});

	var text_2 = $.sibling(node_1);

	$.reset(g);
	$.reset(svg);

	$.template_effect(() => {
		$.set_attribute(svg, 'width', width());
		$.set_attribute(svg, 'height', height());
		$.set_attribute(svg, 'viewBox', `0 0 ${width() ?? ''} ${height() ?? ''}`);
		$.set_attribute(text_2, 'x', marginLeft() - 25);
		$.set_attribute(text_2, 'y', marginTop() + marginBottom());
		$.set_style(text_2, `font-family: ${alloFonts.family}; font-size: 14px; fill: ${alloColors.css.verydarkgrey ?? ''}; text-anchor: start;`);
	});

	$.append($$anchor, svg);
	$.pop();
}

var root = $.from_html(`<div id="allotaxonometer-dashboard" style="position: relative; margin: 0; padding: 0;"><div style="display:flex; flex-wrap: wrap; align-items:center; justify-content: center; row-gap: 50px;"><div id="diamond-group" style="margin-top:20px; margin-right: -50px;"><div style="display:flex; gap: 10em; justify-content: center; margin-bottom: -50px; margin-right: 55px; position: relative;"><div style="position: relative;"><div> </div> <div><div style="margin-bottom: 0.5em;"> </div> <div> </div></div></div> <div> </div></div> <div id="diamondplot"><!></div> <div style="display: flex; gap: 13em; justify-content: center;"><div id="legend" style="margin-left: -50px;"><!></div> <div id="balance"><!></div></div></div> <div style="margin-top:60px; overflow: visible;"><div id="wordshift" style="overflow: visible;"><!></div></div></div></div>`);

function Dashboard($$anchor, $$props) {
	$.push($$props, true);

	let dat = $.prop($$props, 'dat', 3, null),
		alpha = $.prop($$props, 'alpha', 3, 0.58),
		divnorm = $.prop($$props, 'divnorm', 3, 1),
		barData = $.prop($$props, 'barData', 19, () => []),
		balanceData = $.prop($$props, 'balanceData', 19, () => []),
		xDomain = $.prop($$props, 'xDomain', 3, undefined),
		instrumentText = $.prop($$props, 'instrumentText', 3, 'Instrument: Rank-Turbulence Divergence'),
		title = $.prop($$props, 'title', 19, () => ['System 1', 'System 2']),
		maxlog10 = $.prop($$props, 'maxlog10', 3, 0);
		$.prop($$props, 'height', 3, 815);
		$.prop($$props, 'width', 3, 1200);
		let DashboardHeight = $.prop($$props, 'DashboardHeight', 3, 815),
		DashboardWidth = $.prop($$props, 'DashboardWidth', 3, 1200),
		DiamondHeight = $.prop($$props, 'DiamondHeight', 3, 600),
		DiamondWidth = $.prop($$props, 'DiamondWidth', 3, 600),
		WordshiftWidth = $.prop($$props, 'WordshiftWidth', 3, 640),
		marginInner = $.prop($$props, 'marginInner', 3, 160),
		marginDiamond = $.prop($$props, 'marginDiamond', 3, 40),
		max_count_log = $.prop($$props, 'max_count_log', 3, undefined),
		labelThreshold = $.prop($$props, 'labelThreshold', 3, Infinity);

	let max_shift = $.derived(() => barData().length > 0 ? Math.max(...barData().map((d) => Math.abs(d.metric))) : 1);

	let wordshiftXDomain = $.derived(() => xDomain() || [
		-$.get(max_shift) * 1.5,
		$.get(max_shift) * 1.5
	]);

	var div = root();
	var div_1 = $.child(div);
	var div_2 = $.child(div_1);
	var div_3 = $.child(div_2);
	var div_4 = $.child(div_3);
	var div_5 = $.child(div_4);
	var text = $.child(div_5, true);

	$.reset(div_5);

	var div_6 = $.sibling(div_5, 2);
	var div_7 = $.child(div_6);
	var text_1 = $.child(div_7, true);

	$.reset(div_7);

	var div_8 = $.sibling(div_7, 2);
	var text_2 = $.child(div_8);

	$.reset(div_8);
	$.reset(div_6);
	$.reset(div_4);

	var div_9 = $.sibling(div_4, 2);
	var text_3 = $.child(div_9, true);

	$.reset(div_9);
	$.reset(div_3);

	var div_10 = $.sibling(div_3, 2);
	var node = $.child(div_10);

	Diamond(node, {
		get dat() {
			return dat();
		},
		get alpha() {
			return alpha();
		},
		get divnorm() {
			return divnorm();
		},
		get title() {
			return title();
		},
		get maxlog10() {
			return maxlog10();
		},
		get DiamondHeight() {
			return DiamondHeight();
		},
		get marginInner() {
			return marginInner();
		},
		get marginDiamond() {
			return marginDiamond();
		},
		get labelThreshold() {
			return labelThreshold();
		}
	});

	$.reset(div_10);

	var div_11 = $.sibling(div_10, 2);
	var div_12 = $.child(div_11);
	var node_1 = $.child(div_12);
	const expression = $.derived(() => max_count_log() || 5);

	Legend(node_1, {
		get diamond_dat() {
			return dat().counts;
		},
		get DiamondHeight() {
			return DiamondHeight();
		},
		get max_count_log() {
			return $.get(expression);
		}
	});

	$.reset(div_12);

	var div_13 = $.sibling(div_12, 2);
	var node_2 = $.child(div_13);

	DivergingBarChart(node_2, {
		get data() {
			return balanceData();
		},
		get DiamondHeight() {
			return DiamondHeight();
		},
		get DiamondWidth() {
			return DiamondWidth();
		}
	});

	$.reset(div_13);
	$.reset(div_11);
	$.reset(div_2);

	var div_14 = $.sibling(div_2, 2);
	var div_15 = $.child(div_14);
	var node_3 = $.child(div_15);

	Wordshift(node_3, {
		get barData() {
			return barData();
		},
		get DashboardHeight() {
			return DashboardHeight();
		},
		get DashboardWidth() {
			return DashboardWidth();
		},
		get xDomain() {
			return $.get(wordshiftXDomain);
		},
		get width() {
			return WordshiftWidth();
		},
		marginLeft: 110
	});

	$.reset(div_15);
	$.reset(div_14);
	$.reset(div_1);
	$.reset(div);

	$.template_effect(() => {
		$.set_style(div_5, `font-family: ${alloFonts.family}; font-size: 16px; color: ${alloColors.css.superdarkgrey ?? ''};`);
		$.set_text(text, title()[0]);
		$.set_style(div_6, `position: absolute; top: 100%; left: -8em; margin-top: 3em; font-family: ${alloFonts.family}; font-size: 14px; color: ${alloColors.css.darkgrey ?? ''}; width: 150px; line-height: 1;`);
		$.set_text(text_1, instrumentText());
		$.set_text(text_2, `α = ${alpha() ?? ''}`);
		$.set_style(div_9, `font-family: ${alloFonts.family}; font-size: 16px; color: ${alloColors.css.superdarkgrey ?? ''};`);
		$.set_text(text_3, title()[1]);
	});

	$.append($$anchor, div);
	$.pop();
}

// Takes arrays, returns a Set object containing the union of both arrays
 function getUnions(x,y) {
  let a = new Set(x); // convert array x to a Set object
  let b = new Set(y); // convert array y to a Set object
  return new Set([...a, ...b]); // return a new Set object containing the union of a and b
}

// Takes arrays, returns a Set object
function setdiff(x,y) {
  let a = new Set(x); // convert array x to a Set object
  let b = new Set(y); // convert array y to a Set object
  // return a new Set object containing elements in a that are not present in b
  return new Set(       
    [...a].filter(x => !b.has(x)));
} 

function which(x) {
  // Which indices are TRUE?
  // Description:
  //   Give the ‘TRUE’ indices of a logical object, allowing for array indices.
  // Arguments:
  //   x: a ‘logical’ vector or array.
  return x.reduce(
      (out, bool, index) => bool ? out.concat(index) : out, 
      []
    )
}


function matlab_sort(A, rev = false) {
    if (A.length === 0) return { value: [], orig_idx: [] };
    if (A.length === 1) return { value: [...A], orig_idx: [0] };
    
    // Create array of [value, originalIndex] pairs
    const indexedArray = A.map((value, index) => ({ value, index }));
    
    // Sort by value (ascending or descending)
    if (rev) {
        indexedArray.sort((a, b) => b.value - a.value);
    } else {
        indexedArray.sort((a, b) => a.value - b.value);
    }
    
    // Extract sorted values and original indices
    const value = indexedArray.map(item => item.value);
    const orig_idx = indexedArray.map(item => item.index);
    
    return { value, orig_idx };
}

function tiedrank(arr) {
    if (arr.length === 0) return [];
    
    // Group values by their actual value
    const valueMap = new Map();
    arr.forEach((value, index) => {
        if (!valueMap.has(value)) {
            valueMap.set(value, []);
        }
        valueMap.get(value).push(index);
    });
    
    // Sort values in descending order (to match original behavior)
    const sortedValues = [...valueMap.keys()].sort((a, b) => b - a);
    
    // Calculate ranks
    const ranks = new Array(arr.length);
    let currentRank = 1;
    
    for (const value of sortedValues) {
        const indices = valueMap.get(value);
        const tieCount = indices.length;
        
        // Calculate average rank for tied values
        const avgRank = currentRank + (tieCount - 1) / 2;
        
        // Assign the average rank to all tied indices
        indices.forEach(index => {
            ranks[index] = avgRank;
        });
        
        currentRank += tieCount;
    }
    
    return ranks;
}

function rank_maxlog10(mixedelements) {
  // Get maximum of log10 ranks from both systems, then round up
  // Use loop to avoid stack overflow with large arrays (Math.max(...array) fails for 100K+ items)
  let max1 = mixedelements[[0]].ranks[0];
  for (let i = 1; i < mixedelements[[0]].ranks.length; i++) {
    if (mixedelements[[0]].ranks[i] > max1) max1 = mixedelements[[0]].ranks[i];
  }

  let max2 = mixedelements[[1]].ranks[0];
  for (let i = 1; i < mixedelements[[1]].ranks.length; i++) {
    if (mixedelements[[1]].ranks[i] > max2) max2 = mixedelements[[1]].ranks[i];
  }

  const logged_max1 = Math.log10(max1);
  const logged_max2 = Math.log10(max2);
  return Math.ceil(Math.max(logged_max1, logged_max2))
}

function rin(arr1, arr2) {
  // Find element arr1 presents in arr2, i.e. arr1 %in% arr2
  //
  // examples
  // A = ["bob", "george", "jesus"]
  // B = ["bob", "jesus", "terrence"]
  // rin(A, B)
  // [true, false, true]
  return Array.from(arr1, (x) => {
    return arr2.indexOf(x) == -1 ? false : true
  })
}

function zeros(length){
  // Create array of all zeros. Similar to matlab.
  function createArray(length) {
    var arr = new Array(length || 0),
        i = length;
    if (arguments.length > 1) {
        var args = Array.prototype.slice.call(arguments, 1);
        while(i--) arr[length-1 - i] = createArray.apply(this, args);
    }
    return arr;
  }
  let empty_mat = createArray(length,length);
  return Array.from(empty_mat, arr => arr.fill(0))
}

// Normalize a dataset: calculate totalunique and probs if missing
function normalizeDataset(data) {
    // If data already has all required fields, return as-is
    if (data.every(item => item.probs !== undefined && item.totalunique !== undefined)) {
        return data;
    }

    // Calculate total count for probability calculation
    const totalCount = data.reduce((sum, item) => sum + item.counts, 0);
    const totalunique = data.length;

    // Return normalized data with calculated fields
    return data.map(item => ({
        types: item.types,
        counts: item.counts,
        totalunique: item.totalunique ?? totalunique,
        probs: item.probs ?? (item.counts / totalCount)
    }));
}

// Optimized combElems - Single-pass approach (Opt2)
// This version shows 20-50% performance improvements in most scenarios
// Now supports optional totalunique and probs fields
function combElems(elem1, elem2) {
    // Normalize datasets: calculate totalunique and probs if missing
    const normalized1 = normalizeDataset(elem1);
    const normalized2 = normalizeDataset(elem2);

    // Build union and collect data in single pass
    const typeMap = new Map();

    // Process first dataset
    for (const item of normalized1) {
        typeMap.set(item.types, {
            counts1: item.counts,
            probs1: item.probs,
            counts2: 0,
            probs2: 0
        });
    }

    // Process second dataset, updating existing or adding new
    for (const item of normalized2) {
        const existing = typeMap.get(item.types);
        if (existing) {
            existing.counts2 = item.counts;
            existing.probs2 = item.probs;
        } else {
            typeMap.set(item.types, {
                counts1: 0,
                probs1: 0,
                counts2: item.counts,
                probs2: item.probs
            });
        }
    }
    
    // Extract to arrays efficiently
    const unionArray = [...typeMap.keys()];
    const len = unionArray.length;
    const counts1 = new Array(len);
    const counts2 = new Array(len);
    const probs1 = new Array(len);
    const probs2 = new Array(len);
    
    let i = 0;
    for (const [type, data] of typeMap) {
        unionArray[i] = type;
        counts1[i] = data.counts1;
        counts2[i] = data.counts2;
        probs1[i] = data.probs1;
        probs2[i] = data.probs2;
        i++;
    }
    
    return [
        {
            types: unionArray,
            counts: counts1,
            probs: probs1,
            ranks: tiedrank(counts1),
            totalunique: len
        },
        {
            types: unionArray, 
            counts: counts2,
            probs: probs2,
            ranks: tiedrank(counts2),
            totalunique: len
        }
    ];
}


// helpers to wrangle data for the balance plot
function balanceDat(elem1, elem2) {
  const types_1 = elem1.map(d => d.types);
  const types_2 = elem2.map(d => d.types);

  const union_types = getUnions(types_1, types_2);
  const tot_types = types_1.length+types_2.length;

  return [
    { y_coord: "total count",     frequency: +(types_2.length / tot_types).toFixed(3) },
    { y_coord: "total count",     frequency: -(types_1.length / tot_types).toFixed(3) },
    { y_coord: "all types",       frequency: +(types_2.length / union_types.size).toFixed(3) },
    { y_coord: "all types",       frequency: -(types_1.length / union_types.size).toFixed(3) },
    { y_coord: "exclusive types", frequency: +(setdiff(types_2, types_1).size / types_2.length).toFixed(3) },
    { y_coord: "exclusive types", frequency: -(setdiff(types_1, types_2).size / types_1.length).toFixed(3) }
  ]
}

// helper to wrangle the data for the wordshift plot
function wordShift_dat(me, dat) {
  const out = [];
  for (let i=0; i < me[0]['types'].length; i++) {
    const rank_diff = me[0]['ranks'][i]-me[1]['ranks'][i];
    out.push({
      'type': `${me[0]['types'][i]} (${me[0]['ranks'][i]} ⇋ ${me[1]['ranks'][i]})` ,
      'rank_diff': rank_diff,
      'metric': rank_diff < 0 ? -dat.deltas[i] : dat.deltas[i],
    });
  }

  return out.slice().sort((a, b) => descending(Math.abs(a.metric), Math.abs(b.metric)))
}

// WASM module cache
let wasmModule = null;
let wasmInitialized = false;

// Eagerly try to load WASM (but don't block)
(async function initWASM() {
  try {
    const module = await import('./allotaxonometer_wasm-BfhQFmx3.js');
    await module.default(); // Initialize WASM
    wasmModule = module;
    wasmInitialized = true;
    console.log('✅ WASM module loaded successfully');
  } catch (e) {
    console.warn('⚠️  WASM not available, using JS fallback:', e.message);
    wasmModule = null;
    wasmInitialized = false;
  }
})();

// JavaScript fallback implementation
function divElems(inv_r1, inv_r2, alpha) {
  if (alpha === Infinity) {
      return inv_r1.map((d,i) => inv_r1[i] == inv_r2[i] ? 0 : Math.max(inv_r1[i], inv_r2[i]))
  } else if (alpha == 0) {
      const x_max = inv_r1.map((d,i) => Math.max(1 / inv_r1[i], 1 / inv_r2[i]));
      const x_min = inv_r1.map((d,i) => Math.min(1 / inv_r1[i], 1 / inv_r2[i]));
      return inv_r1.map((d,i) => Math.log10(x_max[i] / x_min[i]))
  } else {
      return inv_r1.map((d,i) => (alpha+1) / alpha * Math.abs(inv_r1[i]**alpha - inv_r2[i]**alpha)**(1. / (alpha+1)))
    }
}

function norm_divElems(mixedelements, inv_r1, inv_r2, alpha) {
  const c1 = mixedelements[0]['counts'];
  const c2 = mixedelements[1]['counts'];

  const indices1 = which(c1.map(d => d > 0));
  const indices2 = which(c2.map(d => d > 0));

  const N1 = indices1.length;
  const N2 = indices2.length;

  function calc_disjoint(N1, N2) {
    return( 1 / (N2 + N1/2) )
    }

  const inv_r1_disjoint = calc_disjoint(N1, N2);
  const inv_r2_disjoint = calc_disjoint(N2, N1);

  if (alpha === Infinity) {
      return sum(indices1.map((i) => inv_r1[i])) + sum(indices2.map((i) => inv_r2[i]))
  } else if (alpha === 0) {
      const term1 = sum(
        indices1.map((i) => Math.abs(Math.log(inv_r1[i] / inv_r2_disjoint)))
      );
      const term2 = sum(
        indices2.map((i) => Math.abs(Math.log(inv_r2[i] / inv_r1_disjoint)))
      );
      return term1 + term2
  } else {
      const term1 = (alpha+1)/alpha * sum(
        indices1.map((i) => inv_r1[i]).map( d => (Math.abs(d**alpha) - inv_r2_disjoint**alpha)**(1./(alpha+1) ))
      );
      const term2 = (alpha+1)/alpha * sum(
        indices2.map((i) => inv_r2[i]).map(d => Math.abs(inv_r1_disjoint**alpha - d**alpha)**(1. / (alpha+1)))
      );
      return term1 + term2
    }
}

// JS implementation (fallback)
function rank_turbulence_divergence_js(mixedelements, alpha) {
  const inv_r1 = mixedelements[0]['ranks'].map(d => Math.pow(d, -1));
  const inv_r2 = mixedelements[1]['ranks'].map(d => Math.pow(d, -1));

  const divergence_elements = divElems(inv_r1, inv_r2, alpha);
  const normalization = norm_divElems(mixedelements, inv_r1, inv_r2, alpha);

  return {
    'divergence_elements': divergence_elements.map(d => d / normalization),
    'normalization': normalization
  }
}

// Main export: SYNCHRONOUS - uses WASM if available, otherwise JS
function rank_turbulence_divergence(mixedelements, alpha) {
  // If WASM is loaded, use it
  if (wasmInitialized && wasmModule) {
    try {
      // Call WASM implementation
      const result = wasmModule.rank_turbulence_divergence(
        mixedelements[0]['ranks'],
        mixedelements[1]['ranks'],
        mixedelements[0]['counts'],
        mixedelements[1]['counts'],
        alpha
      );

      // WASM returns a Map, convert to plain object to match expected interface
      if (result instanceof Map) {
        return {
          divergence_elements: result.get('divergence_elements'),
          normalization: result.get('normalization')
        };
      }

      // Handle plain object (in case WASM serialization changes in the future)
      if (result && typeof result === 'object' && result.divergence_elements) {
        return {
          divergence_elements: result.divergence_elements,
          normalization: result.normalization
        };
      }

      // Unexpected format, fall back to JS
      console.warn('⚠️  WASM returned unexpected format, using JS fallback. Got:', result);
      return rank_turbulence_divergence_js(mixedelements, alpha);
    } catch (e) {
      console.warn('⚠️  WASM execution failed, using JS fallback:', e);
      return rank_turbulence_divergence_js(mixedelements, alpha);
    }
  } else {
    // Use JS fallback
    return rank_turbulence_divergence_js(mixedelements, alpha);
  }
}

function rank2coord(rank) { return Math.floor(Math.log10(rank) / (1/15)) }

// Augment information already in `me` class with coordinates. 
function diamond_counts(mixedelements) {

  let maxlog10 = rank_maxlog10(mixedelements);
  if (maxlog10 < 1) maxlog10 = 1;
  
  const CELL_LENGTH = 1/15;
  const Ncells = Math.floor(maxlog10/CELL_LENGTH) + 1;
  
  // Pre-compute all coordinates and group by them
  const coordGroups = new Map();
  for (let i = 0; i < mixedelements[0]['ranks'].length; i++) {
      const x1 = rank2coord(mixedelements[1]['ranks'][i]); // System 1 → X
      const y1 = rank2coord(mixedelements[0]['ranks'][i]); // System 0 → Y
      const key = `${x1},${y1}`;
      
      if (!coordGroups.has(key)) {
          coordGroups.set(key, []);
      }
      
      coordGroups.get(key).push({
          types: mixedelements[0]['types'][i],
          x1: x1,
          y1: y1,
          rank1: mixedelements[0]['ranks'][i],
          rank2: mixedelements[1]['ranks'][i]
      });
  }
  
  // Generate final result directly without intermediate arrays
  const result = [];
  for (let i = 0; i < Ncells; i++) {
      for (let j = 0; j < Ncells; j++) {
          const key = `${i},${j}`;
          const items = coordGroups.get(key);
          
          if (!items) {
              result.push({
                  x1: i,
                  y1: j,
                  coord_on_diag: (j+i)/2,
                  cos_dist: (i-j)**2,
                  rank: "",
                  rank_L: "",
                  rank_R: "",
                  value: 0,
                  types: "",
                  which_sys: i - j <= 0 ? "right" : "left"
              });
          } else {
              result.push({
                  x1: i,
                  y1: j,
                  coord_on_diag: (j+i)/2,
                  cos_dist: (i-j)**2,
                  rank: `(${items[0].rank1}, ${items[0].rank2})`,
                  rank_L: extent$1(items.map(d => d.rank1)),
                  rank_R: extent$1(items.map(d => d.rank2)),
                  value: items.length,
                  types: items.map(d => d.types).join(', '),
                  which_sys: i - j <= 0 ? "right" : "left"
              });
          }
      }
  }
  
  return result;
}


// we expect wordshift to be of the form { divergence_elements: [ length of type ], normalization: float }
function diamond_count(mixedelements, wordshift) {
 
    let deltas = wordshift["divergence_elements"];
    let sorted_div = matlab_sort(deltas, true);
    let indices_deltas = sorted_div.orig_idx;
    
    const len = indices_deltas.length;
    const reordered = {
        deltas: new Array(len),
        mixedelements: [
            { types: new Array(len), counts: new Array(len), ranks: new Array(len), probs: new Array(len) },
            { types: new Array(len), counts: new Array(len), ranks: new Array(len), probs: new Array(len) }
        ]
    };
    
    for (let i = 0; i < len; i++) {
        const idx = indices_deltas[i];
        reordered.deltas[i] = deltas[idx];
        reordered.mixedelements[0].types[i] = mixedelements[0]['types'][idx];
        reordered.mixedelements[0].counts[i] = mixedelements[0]['counts'][idx];
        reordered.mixedelements[0].ranks[i] = mixedelements[0]['ranks'][idx];
        reordered.mixedelements[0].probs[i] = mixedelements[0]['probs'][idx];
        reordered.mixedelements[1].types[i] = mixedelements[1]['types'][idx];
        reordered.mixedelements[1].counts[i] = mixedelements[1]['counts'][idx];
        reordered.mixedelements[1].ranks[i] = mixedelements[1]['ranks'][idx];
        reordered.mixedelements[1].probs[i] = mixedelements[1]['probs'][idx];
    }
    
    // Update original arrays
    Object.assign(mixedelements[0], reordered.mixedelements[0]);
    Object.assign(mixedelements[1], reordered.mixedelements[1]);
    deltas = reordered.deltas;
    
    const deltas_loss = [...deltas];
    [...deltas];
    
    for (let i = 0; i < len; i++) {
        if (mixedelements[0]['ranks'][i] > mixedelements[1]['ranks'][i]) {
            deltas_loss[i] = -1;
        }
        // Fix apparent bug: comparing mixedelements[1]['ranks'][i] < mixedelements[1]['ranks'][i] always false
        if (mixedelements[1]['ranks'][i] < mixedelements[0]['ranks'][i]) ;
    }
    
    const counts = diamond_counts(mixedelements);

  // Find max manually to avoid stack overflow with large arrays
  let max_delta_loss = deltas_loss[0];
  for (let i = 1; i < deltas_loss.length; i++) {
    if (deltas_loss[i] > max_delta_loss) max_delta_loss = deltas_loss[i];
  }

  return({'counts': counts, 'deltas': deltas, 'max_delta_loss': max_delta_loss})
}

/* allotaxonograph.svelte.ts generated by Svelte v5.34.3 */

class Allotaxonograph {
	#sys1 = $.state(null);

	get sys1() {
		return $.get(this.#sys1);
	}

	set sys1(value) {
		$.set(this.#sys1, value, true);
	}

	#sys2 = $.state(null);

	get sys2() {
		return $.get(this.#sys2);
	}

	set sys2(value) {
		$.set(this.#sys2, value, true);
	}

	#title = $.state($.proxy(["System 1", "System 2"]));

	get title() {
		return $.get(this.#title);
	}

	set title(value) {
		$.set(this.#title, value, true);
	}

	#alpha = $.state(0.58);

	get alpha() {
		return $.get(this.#alpha);
	}

	set alpha(value) {
		$.set(this.#alpha, value, true);
	}

	#topN = $.state(30);

	get topN() {
		return $.get(this.#topN);
	}

	set topN(value) {
		$.set(this.#topN, value, true);
	}

	#me = $.derived(() => this.sys1 && this.sys2 ? combElems(this.sys1, this.sys2) : null);

	get me() {
		return $.get(this.#me);
	}

	set me(value) {
		$.set(this.#me, value);
	}

	#rtd = $.derived(() => this.me ? rank_turbulence_divergence(this.me, this.alpha) : null);

	get rtd() {
		return $.get(this.#rtd);
	}

	set rtd(value) {
		$.set(this.#rtd, value);
	}

	#dat = $.derived(() => this.me && this.rtd ? diamond_count(this.me, this.rtd) : null);

	get dat() {
		return $.get(this.#dat);
	}

	set dat(value) {
		$.set(this.#dat, value);
	}

	#barData = $.derived(() => this.me && this.dat ? wordShift_dat(this.me, this.dat).slice(0, this.topN) : []);

	get barData() {
		return $.get(this.#barData);
	}

	set barData(value) {
		$.set(this.#barData, value);
	}

	#balanceData = $.derived(() => this.sys1 && this.sys2 ? balanceDat(this.sys1, this.sys2) : []);

	get balanceData() {
		return $.get(this.#balanceData);
	}

	set balanceData(value) {
		$.set(this.#balanceData, value);
	}

	#maxlog10 = $.derived(() => this.me
		? Math.ceil(d3.max([
			Math.log10(d3.max(this.me[0].ranks)),
			Math.log10(d3.max(this.me[1].ranks))
		]))
		: 0);

	get maxlog10() {
		return $.get(this.#maxlog10);
	}

	set maxlog10(value) {
		$.set(this.#maxlog10, value);
	}

	#max_count_log = $.derived(() => this.dat ? Math.ceil(Math.log10(d3.max(this.dat.counts, (d) => d.value))) + 1 : 2);

	get max_count_log() {
		return $.get(this.#max_count_log);
	}

	set max_count_log(value) {
		$.set(this.#max_count_log, value);
	}

	#max_shift = $.derived(() => this.barData.length > 0 ? d3.max(this.barData, (d) => Math.abs(d.metric)) : 1);

	get max_shift() {
		return $.get(this.#max_shift);
	}

	set max_shift(value) {
		$.set(this.#max_shift, value);
	}

	#divnorm = $.derived(() => this.rtd?.normalization);

	get divnorm() {
		return $.get(this.#divnorm);
	}

	set divnorm(value) {
		$.set(this.#divnorm, value);
	}

	#xDomain = $.derived(() => [-this.max_shift * 1.5, this.max_shift * 1.5]);

	get xDomain() {
		return $.get(this.#xDomain);
	}

	set xDomain(value) {
		$.set(this.#xDomain, value);
	}

	#isDataReady = $.derived(() => this.dat && this.barData && this.balanceData && this.me && this.rtd);

	get isDataReady() {
		return $.get(this.#isDataReady);
	}

	set isDataReady(value) {
		$.set(this.#isDataReady, value);
	}

	constructor(sys1, sys2, options = {}) {
		if (sys1) this.sys1 = sys1;
		if (sys2) this.sys2 = sys2;
		if (options.alpha !== void 0) this.alpha = options.alpha;
		if (options.title) this.title = options.title;
		if (options.topN !== void 0) this.topN = options.topN;
	}

	// Methods
	updateData(sys1, sys2, titles) {
		this.sys1 = sys1;
		this.sys2 = sys2;
		if (titles) this.title = titles;
	}

	setAlpha(alpha) {
		this.alpha = alpha;
	}

	setTopN(n) {
		this.topN = n;
	}

	exportData() {
		return {
			rtd: this.rtd,
			barData: this.barData,
			balanceData: this.balanceData,
			meta: {
				alpha: this.alpha,
				titles: this.title,
				maxlog10: this.maxlog10,
				max_count_log: this.max_count_log
			}
		};
	}

	// Static factory methods
	static fromData(data1, data2, options) {
		return new Allotaxonograph(data1, data2, options);
	}

	static createComparison(data1, data2, preset = "standard") {
		const presets = {
			standard: { alpha: 0.58 },
			sensitive: { alpha: 0.1 },
			robust: { alpha: Infinity }
		};

		return new Allotaxonograph(data1, data2, presets[preset]);
	}
}

export { Allotaxonograph, Dashboard, Diamond, DivergingBarChart, Legend, Wordshift, balanceDat, combElems, diamond_count, getUnions, matlab_sort, rank_maxlog10, rank_turbulence_divergence, rin, setdiff, tiedrank, which, wordShift_dat, zeros };
