import * as $ from "svelte/internal/server";
import * as d3 from "d3";
import { map, extent, InternSet, range, rollup, scaleLinear, scaleBand, rgb, interpolateInferno, scaleOrdinal } from "d3";
import { descending, sum, extent as extent$1 } from "d3-array";
function rgbArrayToCss(rgbArray) {
  const [r, g, b] = rgbArray.map((v) => Math.round(v * 255));
  return `rgb(${r}, ${g}, ${b})`;
}
function style(props) {
  return Object.entries(props).map(([key, value]) => `${key.replace(/([A-Z])/g, "-$1").toLowerCase()}: ${value}`).join("; ");
}
const rawColors = {
  blue: [43, 103, 198].map((v) => v / 255),
  red: [198, 43, 103].map((v) => v / 255),
  paleblue: [195, 230, 243].map((v) => v / 255),
  palered: [255, 204, 204].map((v) => v / 255),
  lightergrey: [1, 1, 1].map((v) => v * 0.96),
  lightishgrey: [1, 1, 1].map((v) => v * 0.93),
  lightgrey: [1, 1, 1].map((v) => v * 0.9),
  lightgreyer: [1, 1, 1].map((v) => v * 0.85),
  lightgreyish: [1, 1, 1].map((v) => v * 0.8),
  grey: [1, 1, 1].map((v) => v * 0.75),
  darkgrey: [1, 1, 1].map((v) => v * 0.55),
  darkergrey: [1, 1, 1].map((v) => v * 0.5),
  verydarkgrey: [1, 1, 1].map((v) => v * 0.55),
  superdarkgrey: [1, 1, 1].map((v) => v * 0.55),
  reallyverdarkgrey: [1, 1, 1].map((v) => v * 0.05),
  orange: [255, 116, 0].map((v) => v / 255)
};
const cssColors = {};
for (const [key, rgb2] of Object.entries(rawColors)) {
  cssColors[key] = rgbArrayToCss(rgb2);
}
const alloColors = {
  css: cssColors
  // e.g., colors.css.blue → "rgb(43, 103, 198)"
};
const alloFonts = {
  family: `"EB Garamond", "Garamond", "Century Schoolbook L", "URW Bookman L", "Bookman Old Style", "Times", serif`,
  sizes: {
    md: "14px",
    lg: "16px"
  }
};
function AxisX($$payload, $$props) {
  $.push();
  let { innerHeight, scale, title } = $$props;
  let logFormat10 = scale.tickFormat();
  let xTicks = scale.ticks().filter((t) => t >= 1 && Number.isInteger(Math.log10(t)));
  let axisStyles = () => style({
    fontFamily: alloFonts.family,
    fontSize: alloFonts.sizes.lg,
    fill: alloColors.css.darkergrey,
    textAnchor: "middle"
  });
  let helperText = () => style({
    fontFamily: alloFonts.family,
    fontSize: alloFonts.sizes.md,
    fill: alloColors.css.darkgrey,
    textAnchor: "middle"
  });
  const each_array = $.ensure_array_like(xTicks);
  $$payload.out += `<g class="axis x"${$.attr("transform", `translate(0, ${$.stringify(innerHeight)})`)}><!--[-->`;
  for (let index = 0, $$length = each_array.length; index < $$length; index++) {
    let tick = each_array[index];
    $$payload.out += `<g${$.attr("transform", `translate(${$.stringify(scale(tick))}, 0)`)}><line x1="0" x2="0" y1="0" y2="6"${$.attr("stroke", alloColors.css.darkergrey)} stroke-width="0.5"></line></g><g${$.attr("transform", `translate(${$.stringify(scale(tick))}, 0) scale(-1,1) rotate(45)`)}><text dx="5" dy="13" text-anchor="start"${$.attr("font-family", alloFonts.family)}${$.attr("font-size", alloFonts.sizes.md)}${$.attr("fill", alloColors.css.darkergrey)}>${$.escape(logFormat10(tick))}</text></g>`;
  }
  $$payload.out += `<!--]--><g class="xlab"><text${$.attr("x", innerHeight / 2)} dy="45"${$.attr("transform", `scale(-1,1) translate(-${$.stringify(innerHeight)}, 0)`)}${$.attr_style(axisStyles())}>Rank r</text><text${$.attr("x", innerHeight / 2)} dy="63"${$.attr("transform", `scale(-1,1) translate(-${$.stringify(innerHeight)}, 0)`)}${$.attr_style(axisStyles())}>for</text><text${$.attr("x", innerHeight / 2)} dy="80"${$.attr("transform", `scale(-1,1) translate(-${$.stringify(innerHeight)}, 0)`)}${$.attr_style(axisStyles())}>${$.escape(title[1])}</text><text${$.attr("x", innerHeight - 40)} dy="60"${$.attr("transform", `scale(-1,1) translate(-${$.stringify(innerHeight)}, 0)`)}${$.attr_style(helperText())}>more →</text><text${$.attr("x", innerHeight - 40)} dy="75"${$.attr("transform", `scale(-1,1) translate(-${$.stringify(innerHeight)}, 0)`)}${$.attr_style(helperText())}>frequent</text><text${$.attr("x", 40)} dy="60"${$.attr("transform", `scale(-1,1) translate(-${$.stringify(innerHeight)}, 0)`)}${$.attr_style(helperText())}>← less</text><text${$.attr("x", 40)} dy="75"${$.attr("transform", `scale(-1,1) translate(-${$.stringify(innerHeight)}, 0)`)}${$.attr_style(helperText())}>frequent</text></g></g>`;
  $.pop();
}
function AxisY($$payload, $$props) {
  $.push();
  let { innerHeight, scale, title } = $$props;
  let logFormat10 = scale.tickFormat();
  let yTicks = scale.ticks().filter((t) => t >= 1 && Number.isInteger(Math.log10(t)));
  let axisStyles = () => style({
    fontFamily: alloFonts.family,
    fontSize: alloFonts.sizes.lg,
    fill: alloColors.css.darkergrey,
    textAnchor: "middle"
  });
  let helperText = () => style({
    fontFamily: alloFonts.family,
    fontSize: alloFonts.sizes.md,
    fill: alloColors.css.darkgrey,
    textAnchor: "middle"
  });
  const each_array = $.ensure_array_like(yTicks);
  $$payload.out += `<g class="axis y"${$.attr("transform", `translate(${$.stringify(innerHeight)}, 0) scale(-1, 1)`)}><!--[-->`;
  for (let index = 0, $$length = each_array.length; index < $$length; index++) {
    let tick = each_array[index];
    $$payload.out += `<g${$.attr("transform", `translate(0, ${$.stringify(scale(tick))})`)}><line x1="0" x2="-6" y1="0" y2="0"${$.attr("stroke", alloColors.css.darkergrey)} stroke-width="0.5"></line></g><g${$.attr("transform", `translate(0, ${$.stringify(scale(tick))}) rotate(45)`)}><text dx="-5" dy="13" text-anchor="end"${$.attr("font-family", alloFonts.family)}${$.attr("font-size", alloFonts.sizes.md)}${$.attr("fill", alloColors.css.darkergrey)}>${$.escape(logFormat10(tick))}</text></g>`;
  }
  $$payload.out += `<!--]--><g class="ylab" transform="rotate(90)"><text${$.attr("x", innerHeight / 2)} dy="45"${$.attr_style(axisStyles())}>Rank r</text><text${$.attr("x", innerHeight / 2)} dy="63"${$.attr_style(axisStyles())}>for</text><text${$.attr("x", innerHeight / 2)} dy="80"${$.attr_style(axisStyles())}>${$.escape(title[0])}</text><text${$.attr("x", innerHeight - 40)} dy="60"${$.attr_style(helperText())}>less →</text><text${$.attr("x", innerHeight - 40)} dy="75"${$.attr_style(helperText())}>frequent</text><text${$.attr("x", 40)} dy="60"${$.attr_style(helperText())}>← more</text><text${$.attr("x", 40)} dy="75"${$.attr_style(helperText())}>frequent</text></g></g>`;
  $.pop();
}
function Grid($$payload, $$props) {
  $.push();
  let { scale, height, wxy, ncells } = $$props;
  let xygridLines = scale.ticks(ncells / 2);
  const each_array = $.ensure_array_like(xygridLines);
  $$payload.out += `<g class="grid"><!--[-->`;
  for (let index = 0, $$length = each_array.length; index < $$length; index++) {
    let yline = each_array[index];
    $$payload.out += `<g${$.attr("transform", `translate(${$.stringify(wxy(yline))}, 0)`)}><line${$.attr("y1", height)} y2="0" stroke="#d3d3d3" stroke-dasharray="1,3"${$.attr("opacity", index === 0 ? 0 : 1)}></line></g><g${$.attr("transform", `translate(0, ${$.stringify(wxy(yline))})`)}><line${$.attr("x1", height)} x2="0" stroke="#d3d3d3" stroke-dasharray="1,3"${$.attr("opacity", index === 0 ? 0 : 1)}></line></g>`;
  }
  $$payload.out += `<!--]--></g>`;
  $.pop();
}
function Contours($$payload, $$props) {
  $.push();
  let {
    alpha,
    maxlog10,
    divnorm,
    DiamondInnerHeight
  } = $$props;
  function alpha_norm_type2(x1, x2, alpha2) {
    if (alpha2 == 0) {
      return Math.abs(Math.log(x1 / x2));
    } else if (alpha2 === Infinity) {
      return x1 === x2 ? 0 : Math.max(x1, x2);
    } else {
      const prefactor = (alpha2 + 1) / alpha2;
      const power = 1 / (alpha2 + 1);
      return prefactor * Math.abs(Math.pow(x1, alpha2) - Math.pow(x2, alpha2)) ** power;
    }
  }
  function make_grid(Ninset, tmpr1, tmpr2, alpha2, divnorm2) {
    const deltamatrix = Array.from({ length: Ninset }, () => Array(Ninset).fill(0));
    for (let i = 0; i < Ninset; i++) {
      for (let j = 0; j < Ninset; j++) {
        const divElem = alpha_norm_type2(1 / tmpr1[i], 1 / tmpr2[j], alpha2);
        deltamatrix[i][j] = divElem / divnorm2;
      }
      deltamatrix[i][i] = -1;
      if (i < Ninset - 1) {
        deltamatrix[i][i + 1] = -1;
        deltamatrix[i + 1][i] = -1;
      }
    }
    return deltamatrix;
  }
  function filter_contours(tmpcontours, Ninset, maxlog102) {
    const chart2val = d3.scaleLinear().domain([0, Ninset]).range([0, maxlog102]);
    let out = [];
    tmpcontours.forEach((contour) => {
      contour.coordinates.forEach((pair, i) => {
        const tmpr1 = pair[0].map((d) => d[0]);
        const tmpr2 = pair[0].map((d) => d[1]);
        const filteredPairs = [];
        for (let index = 0; index < tmpr1.length - 1; index++) {
          const x1 = chart2val(tmpr1[index]);
          const x2 = chart2val(tmpr2[index]);
          const tmpxrot = Math.abs(x2 - x1) / Math.sqrt(2);
          if (Math.abs(tmpxrot) >= 0.1 & x1 != maxlog102 & x2 != 0 & x1 != 0 & x2 != maxlog102) {
            filteredPairs.push([x1, x2]);
          }
        }
        if (filteredPairs.length > 0) {
          out.push(filteredPairs);
        }
      });
    });
    return out;
  }
  function get_contours(alpha2, maxlog102, divnorm2) {
    const Ninset = 10 ** 3;
    const tmpr1 = d3.range(0, 1e3).map((d) => Math.pow(10, d / 999 * 5));
    const tmpr2 = d3.range(0, 1e3).map((d) => Math.pow(10, d / 999 * 5));
    const Ncontours = 10;
    const scale = d3.scaleLinear().domain([0, Ncontours + 1]).range([1, tmpr1.length]);
    const contour_indices = d3.range(Ncontours + 2).map((i) => Math.round(scale(i)));
    const grid = make_grid(Ninset, tmpr1, tmpr2, alpha2, divnorm2);
    const indices = contour_indices.slice(1, -1);
    const lastRow = grid[grid.length - 1];
    const heights = indices.map((index) => lastRow[index]);
    const logTmpr = tmpr1.map(Math.log10);
    const contourGenerator = d3.contours().size([logTmpr.length, logTmpr.length]).thresholds(heights);
    const flatDeltamatrix = grid.flat();
    const tmpcontours = contourGenerator(flatDeltamatrix);
    return filter_contours(tmpcontours, Ninset, maxlog102);
  }
  let mycontours = get_contours(alpha, maxlog10, divnorm);
  const x = d3.scaleLinear([0, maxlog10], [0, DiamondInnerHeight]);
  const y = d3.scaleLinear([maxlog10, 0], [DiamondInnerHeight, 0]);
  const pathData = d3.line().x((d, i) => x(d[0])).y((d, i) => y(d[1]));
  const each_array = $.ensure_array_like(mycontours);
  $$payload.out += `<g class="contours"><!--[-->`;
  for (let index = 0, $$length = each_array.length; index < $$length; index++) {
    let contour = each_array[index];
    $$payload.out += `<path fill="none" stroke="grey"${$.attr("d", pathData(contour))} stroke-width="0.55" stroke-opacity="0.9"></path>`;
  }
  $$payload.out += `<!--]--></g>`;
  $.pop();
}
function Diamond($$payload, $$props) {
  $.push();
  let {
    dat,
    alpha,
    divnorm,
    title,
    maxlog10,
    DiamondHeight = 600,
    marginInner = 160,
    marginDiamond = 40,
    highlightedTerm = null,
    highlightedSystem = null,
    // 'left' or 'right'
    labelThreshold = Infinity
    // Maximum number of types per cell to show labels (default: show all)
  } = $$props;
  let diamond_dat = dat.counts;
  dat.deltas;
  let innerHeight = DiamondHeight - marginInner;
  let diamondHeight = innerHeight - marginDiamond;
  function rin2(arr1, arr2) {
    return Array.from(arr1, (x) => arr2.indexOf(x) !== -1);
  }
  function filter_labs(d, relevant_types_array) {
    if (!relevant_types_array || relevant_types_array.length === 0) return false;
    const typeCount = d.types.split(",").length;
    if (typeCount > labelThreshold) return false;
    return rin2(relevant_types_array, d.types.split(",")).some((x) => x === true);
  }
  function get_relevant_types(diamond_dat2) {
    const ncells2 = d3.max(diamond_dat2, (d) => d.x1);
    const cumbin = d3.range(0, ncells2, 1.5);
    const relevant_types2 = [];
    const type_frequency = /* @__PURE__ */ new Map();
    diamond_dat2.filter((d) => d.value > 0).forEach((d) => {
      d.types.split(",").forEach((type) => {
        const clean_type = type.trim();
        type_frequency.set(clean_type, (type_frequency.get(clean_type) || 0) + 1);
      });
    });
    for (let sys of ["right", "left"]) {
      for (let i = 1; i < cumbin.length; i++) {
        const filtered_dat = diamond_dat2.filter((d) => d.value > 0 && d.which_sys == sys).filter((d) => d.coord_on_diag >= cumbin[i - 1] && d.coord_on_diag < cumbin[i]);
        if (filtered_dat.length > 0) {
          const cos_dists = filtered_dat.map((d) => d.cos_dist);
          const max_dist = cos_dists.reduce((a, b) => Math.max(a, b));
          const max_dist_idx = cos_dists.indexOf(max_dist);
          const types = filtered_dat[max_dist_idx]["types"].split(",").map((t) => t.trim());
          let selected_type = types[0];
          let min_frequency = Infinity;
          for (const type of types) {
            const freq = type_frequency.get(type) || 0;
            if (freq < min_frequency && freq > 0) {
              min_frequency = freq;
              selected_type = type;
            }
          }
          relevant_types2.push(selected_type);
        }
      }
    }
    return relevant_types2;
  }
  let relevant_types = get_relevant_types(diamond_dat);
  relevant_types ? diamond_dat.filter((d) => filter_labs(d, relevant_types)).length : 0;
  let ncells = d3.max(diamond_dat, (d) => d.x1);
  let max_rank = d3.max(diamond_dat, (d) => d.rank_L[1]);
  let rounded_max_rank = 10 ** Math.ceil(Math.log10(max_rank));
  let xyDomain = [1, rounded_max_rank];
  let xy = d3.scaleBand().domain(diamond_dat.map((d) => d.y1)).range([0, diamondHeight]);
  let logScale = d3.scaleLog().domain(xyDomain).range([0, innerHeight]).nice();
  let linScale = d3.scaleLinear().domain([0, ncells - 1]).range([0, innerHeight]);
  let wxy = d3.scaleBand().domain(d3.range(ncells)).range([0, innerHeight]);
  let color_scale = d3.scaleSequentialLog().domain([rounded_max_rank, 1]).interpolator(d3.interpolateInferno);
  let blue_triangle = [
    [innerHeight, innerHeight],
    [0, 0],
    [0, innerHeight]
  ].join(" ");
  let grey_triangle = [
    [innerHeight, innerHeight],
    [0, 0],
    [innerHeight, 0]
  ].join(" ");
  const each_array = $.ensure_array_like(diamond_dat);
  const each_array_1 = $.ensure_array_like(diamond_dat);
  $$payload.out += `<div style="position: relative;"><svg xmlns="http://www.w3.org/2000/svg"${$.attr("width", DiamondHeight)}${$.attr("height", DiamondHeight)}${$.attr("viewBox", `0 0 ${$.stringify(DiamondHeight)} ${$.stringify(DiamondHeight)}`)} style="overflow: visible; display: block;"><g${$.attr("transform", `matrix(-0.7071, 0.7071, 0.7071, 0.7071, ${$.stringify(DiamondHeight * 0.5)}, ${$.stringify(DiamondHeight * 0.067)})`)}><polygon class="diamond-background grey-triangle"${$.attr("points", grey_triangle)}${$.attr("fill", alloColors.css.lightgrey)} fill-opacity="0.8" stroke="black" stroke-width="0.5"></polygon><polygon class="diamond-background blue-triangle"${$.attr("points", blue_triangle)}${$.attr("fill", alloColors.css.paleblue)} fill-opacity="0.8" stroke="black" stroke-width="0.5"></polygon>`;
  AxisX($$payload, { innerHeight, scale: logScale, title });
  $$payload.out += `<!---->`;
  AxisY($$payload, { innerHeight, scale: logScale, title });
  $$payload.out += `<!---->`;
  Grid($$payload, {
    height: innerHeight,
    wxy,
    ncells,
    scale: linScale
  });
  $$payload.out += `<!----><!--[-->`;
  for (let $$index = 0, $$length = each_array.length; $$index < $$length; $$index++) {
    let d = each_array[$$index];
    const typesList = d.types.split(",").map((t) => t.trim());
    const isHighlighted = highlightedTerm && typesList.includes(highlightedTerm) && (!highlightedSystem || d.which_sys === highlightedSystem);
    $$payload.out += `<rect${$.attr_class("diamond-cell", void 0, { "highlighted": isHighlighted })}${$.attr("x", xy(d.x1))}${$.attr("y", xy(d.y1))}${$.attr("width", xy.bandwidth())}${$.attr("height", xy.bandwidth())}${$.attr("fill", d.value === 0 ? "none" : color_scale(d.value))}${$.attr("stroke", isHighlighted ? "#ff6b6b" : "none")}${$.attr("stroke-width", isHighlighted ? 2 : 0)}${$.attr_style(`opacity: ${$.stringify(highlightedTerm && !isHighlighted ? 0.3 : 1)};`)}></rect>`;
  }
  $$payload.out += `<!--]--><!--[-->`;
  for (let $$index_1 = 0, $$length = each_array_1.length; $$index_1 < $$length; $$index_1++) {
    let d = each_array_1[$$index_1];
    $$payload.out += `<rect${$.attr("x", xy(d.x1))}${$.attr("y", xy(d.y1))}${$.attr("width", xy.bandwidth())}${$.attr("height", xy.bandwidth())}${$.attr("fill", d.value > 0 ? "rgba(255,255,255,0.001)" : "none")}${$.attr("stroke", d.value > 0 ? alloColors.css.darkergrey : "none")}${$.attr("stroke-width", d.value > 0 ? "1.18" : "0")}${$.attr("stroke-opacity", d.value > 0 ? "0.4" : "0")}${$.attr_style(`cursor: ${$.stringify(d.value > 0 ? "pointer" : "default")};`)} role="button" tabindex="0"></rect>`;
  }
  $$payload.out += `<!--]-->`;
  if (relevant_types) {
    $$payload.out += "<!--[-->";
    const each_array_2 = $.ensure_array_like(diamond_dat.filter((d) => filter_labs(d, relevant_types)));
    $$payload.out += `<!--[-->`;
    for (let $$index_2 = 0, $$length = each_array_2.length; $$index_2 < $$length; $$index_2++) {
      let d = each_array_2[$$index_2];
      $$payload.out += `<text${$.attr("x", xy(d.x1))}${$.attr("y", Number.isInteger(d.coord_on_diag) ? xy(d.y1) : xy(d.y1) - 1)}${$.attr("dx", d.x1 - d.y1 <= 0 ? 5 : -5)} dy="5"${$.attr("text-anchor", d.x1 - d.y1 <= 0 ? "start" : "end")}${$.attr("transform", `scale(1,-1) rotate(-90) rotate(-45, ${$.stringify(xy(d.x1))}, ${$.stringify(xy(d.y1))}) translate(${$.stringify(d.which_sys === "right" ? xy(Math.sqrt(d.cos_dist)) * 1.5 : -xy(Math.sqrt(d.cos_dist)) * 1.5)}, 0)`)}${$.attr_style(`font-family: ${$.stringify(alloFonts.family)}; font-size: 12px; fill: ${$.stringify(alloColors.css.darkergrey)};`)}>${$.escape(d.types.split(",")[0])}</text>`;
    }
    $$payload.out += `<!--]-->`;
  } else {
    $$payload.out += "<!--[!-->";
  }
  $$payload.out += `<!--]--><line x1="0" y1="0"${$.attr("x2", innerHeight - 7)}${$.attr("y2", innerHeight - 7)}${$.attr_style(`stroke: ${$.stringify(alloColors.css.verydarkgrey)}; stroke-width: 0.5;`)}></line>`;
  Contours($$payload, {
    alpha,
    maxlog10,
    divnorm,
    DiamondInnerHeight: innerHeight
  });
  $$payload.out += `<!----></g></svg> `;
  {
    $$payload.out += "<!--[!-->";
  }
  $$payload.out += `<!--]--></div>`;
  $.pop();
}
function Wordshift($$payload, $$props) {
  $.push();
  let {
    barData,
    x = (d) => d.metric,
    y = (d) => d.type,
    xDomain,
    marginTop = 50,
    marginRight = 60,
    marginBottom = 40,
    marginLeft = 70,
    width = 360,
    height,
    xLabel = "← System 1 · Divergence contribution · System 2 →",
    yPadding = 0,
    colors = [
      alloColors.css.lightgrey,
      alloColors.css.paleblue
    ],
    barHeightFactor = 0.7,
    onBarClick = void 0,
    // (event, data, label) => void
    selectedBar = null
    // Track selected bar (pass label)
  } = $$props;
  let X = d3.map(barData, x);
  let Y = d3.map(barData, y);
  let computedXDomain = xDomain || d3.extent(X);
  let yDomain = new d3.InternSet(Y);
  const xAxisYOffset = 10;
  const bandHeight = 18;
  const shiftSvgBy = 12;
  const barYOffset = 10;
  let compactHeight = yDomain.size * bandHeight;
  let innerWidth = width - marginLeft - marginRight;
  let innerHeight = compactHeight + xAxisYOffset;
  let computedHeight = innerHeight + marginTop + marginBottom;
  let xRange = [0, innerWidth];
  let yRange = [
    xAxisYOffset + barYOffset,
    xAxisYOffset + barYOffset + compactHeight
  ];
  let I = d3.range(X.length).filter((i) => yDomain.has(Y[i]));
  let YX = d3.rollup(I, ([i]) => X[i], (i) => Y[i]);
  let xScale = d3.scaleLinear(computedXDomain, xRange);
  let yScale = d3.scaleBand().domain(yDomain).range(yRange).padding(yPadding);
  let format = d3.format(".2%");
  let xTicks = xScale.ticks(width / 80);
  function parseLabelData(label) {
    const splitIndex = label.indexOf(" ");
    let name_y, numbers_y;
    if (splitIndex === -1) {
      name_y = label;
      numbers_y = "";
    } else {
      name_y = label.slice(0, splitIndex);
      numbers_y = label.slice(splitIndex + 1).trim();
      if (numbers_y.length > 2) {
        numbers_y = numbers_y.slice(1, numbers_y.length - 1);
      }
    }
    return { name_y, numbers_y };
  }
  let finalHeight = height || computedHeight;
  const each_array = $.ensure_array_like(xTicks);
  const each_array_1 = $.ensure_array_like(I);
  const each_array_2 = $.ensure_array_like(yScale.domain());
  $$payload.out += `<svg${$.attr("width", width)}${$.attr("height", finalHeight)}${$.attr("viewBox", `0 0 ${$.stringify(width)} ${$.stringify(finalHeight)}`)} style="overflow: visible; display: block;"><g class="wordshift-container"${$.attr("transform", `translate(${$.stringify(marginLeft)}, ${$.stringify(marginTop - shiftSvgBy)})`)}><g class="wordshift-axis x"${$.attr("transform", `translate(0, ${$.stringify(xAxisYOffset)})`)}><!--[-->`;
  for (let $$index = 0, $$length = each_array.length; $$index < $$length; $$index++) {
    let tick = each_array[$$index];
    $$payload.out += `<line${$.attr("x1", xScale(tick))} y1="0"${$.attr("x2", xScale(tick))} y2="6" style="stroke: currentColor; stroke-width: 1;"></line><line class="wordshift-grid-line"${$.attr("x1", xScale(tick))} y1="0"${$.attr("x2", xScale(tick))}${$.attr("y2", innerHeight - xAxisYOffset + barYOffset)}${$.attr_style(tick === 0 ? `stroke: ${alloColors.css.verydarkgrey}; stroke-width: 1; stroke-opacity: 0.8;` : `stroke: currentColor; stroke-opacity: 0.1;`)}></line><text${$.attr("x", xScale(tick))} y="-12" text-anchor="middle"${$.attr_style(`font-family: ${$.stringify(alloFonts.family)}; font-size: 12px; fill: ${$.stringify(alloColors.css.verydarkgrey)};`)}>${$.escape(format(tick))}</text>`;
  }
  $$payload.out += `<!--]--><text${$.attr("x", xScale(0))} y="-35" text-anchor="middle"${$.attr_style(`font-family: ${$.stringify(alloFonts.family)}; font-size: 16px; fill: ${$.stringify(alloColors.css.verydarkgrey)};`)}>${$.escape(xLabel)}</text></g><!--[-->`;
  for (let $$index_1 = 0, $$length = each_array_1.length; $$index_1 < $$length; $$index_1++) {
    let i = each_array_1[$$index_1];
    const isSelected = selectedBar === Y[i];
    $$payload.out += `<rect${$.attr_class("wordshift-bar svelte-1xsy6uz", void 0, { "selected": isSelected })}${$.attr("x", Math.min(xScale(0), xScale(X[i])))}${$.attr("y", yScale(Y[i]) + (yScale.bandwidth() - yScale.bandwidth() * barHeightFactor) / 2)}${$.attr("fill", colors[X[i] > 0 ? colors.length - 1 : 0])}${$.attr("width", Math.abs(xScale(X[i]) - xScale(0)))}${$.attr("height", yScale.bandwidth() * barHeightFactor)}${$.attr_style(` mix-blend-mode: multiply; cursor: ${$.stringify(onBarClick ? "pointer" : "default")}; opacity: ${$.stringify(isSelected ? 1 : selectedBar ? 0.3 : 1)}; transition: opacity 0.2s ease, stroke 0.2s ease; `)}${$.attr("stroke", isSelected ? "#ff6b6b" : "none")}${$.attr("stroke-width", isSelected ? "2" : "0")}></rect>`;
  }
  $$payload.out += `<!--]--><g class="wordshift-y-axis"${$.attr("transform", `translate(${$.stringify(xScale(0))}, 0)`)}><!--[-->`;
  for (let $$index_2 = 0, $$length = each_array_2.length; $$index_2 < $$length; $$index_2++) {
    let label = each_array_2[$$index_2];
    const labelData = parseLabelData(label);
    const xValue = YX.get(label);
    $$payload.out += `<g class="wordshift-label-group"${$.attr("transform", `translate(0, ${$.stringify(yScale(label) + yScale.bandwidth() / 2)})`)}><text${$.attr("x", xValue > 0 ? 6 : -6)} dy="0.32em"${$.attr("text-anchor", xValue > 0 ? "start" : "end")}${$.attr_style(`font-family: ${$.stringify(alloFonts.family)}; font-size: 14px; fill: ${$.stringify(alloColors.css.verydarkgrey)};`)}>${$.escape(labelData.name_y)}</text>`;
    if (labelData.numbers_y) {
      $$payload.out += "<!--[-->";
      $$payload.out += `<text${$.attr("x", xValue > 0 ? -6 : 6)} dy="0.32em"${$.attr("text-anchor", xValue > 0 ? "end" : "start")}${$.attr_style(`font-family: ${$.stringify(alloFonts.family)}; font-size: 14px; fill: ${$.stringify(alloColors.css.darkergrey)}; opacity: 0.5;`)}>${$.escape(labelData.numbers_y)}</text>`;
    } else {
      $$payload.out += "<!--[!-->";
    }
    $$payload.out += `<!--]--></g>`;
  }
  $$payload.out += `<!--]--></g></g></svg>`;
  $.pop();
}
function DivergingBarChart($$payload, $$props) {
  $.push();
  let {
    data,
    x = (d) => d.frequency,
    y = (d) => d.y_coord,
    marginTop = 0,
    // Match original
    marginRight = 40,
    marginBottom = 10,
    marginLeft = 40,
    width = 200,
    yPadding = 0.5,
    colors = [
      alloColors.css.lightgrey,
      alloColors.css.paleblue
    ]
  } = $$props;
  let X = map(data, x);
  let Y = map(data, y);
  let xDomain = extent(X);
  let yDomain = new InternSet(Y);
  let xRange = [marginLeft, width - marginRight];
  let height = Math.ceil((yDomain.size + yPadding) * 25) + marginTop + marginBottom;
  let yRange = [marginTop, height - marginBottom];
  let I = range(X.length).filter((i) => yDomain.has(Y[i]));
  let YX = rollup(I, ([i]) => X[i], (i) => Y[i]);
  let xScale = scaleLinear(xDomain, xRange);
  let yScale = scaleBand().domain(yDomain).range(yRange).padding(yPadding);
  let format = xScale.tickFormat(100, "%");
  const each_array = $.ensure_array_like(I);
  const each_array_1 = $.ensure_array_like(yScale.domain());
  $$payload.out += `<svg${$.attr("width", width)}${$.attr("height", height)}${$.attr("viewBox", `0 0 ${$.stringify(width)} ${$.stringify(height)}`)} style="overflow: visible; display: block;"><g class="balance-chart"><!--[-->`;
  for (let $$index = 0, $$length = each_array.length; $$index < $$length; $$index++) {
    let i = each_array[$$index];
    $$payload.out += `<rect${$.attr("x", Math.min(xScale(0), xScale(X[i])))}${$.attr("y", yScale(Y[i]))}${$.attr("fill", colors[X[i] > 0 ? colors.length - 1 : 0])}${$.attr("width", Math.abs(xScale(X[i]) - xScale(0)))}${$.attr("height", yScale.bandwidth())}></rect><text${$.attr("x", xScale(X[i]) + Math.sign(X[i] - 0) * 4)}${$.attr("y", yScale(Y[i]) + yScale.bandwidth() / 2)} dy="0.35em"${$.attr("text-anchor", X[i] < 0 ? "end" : "start")}${$.attr_style(`font-family: ${$.stringify(alloFonts.family)}; font-size: 12px; fill: ${$.stringify(alloColors.css.darkergrey)}; opacity: 0.5;`)}>${$.escape(format(Math.abs(X[i])))}</text>`;
  }
  $$payload.out += `<!--]--><g class="y-axis"${$.attr("transform", `translate(${$.stringify(xScale(0))}, -12)`)}><!--[-->`;
  for (let $$index_1 = 0, $$length = each_array_1.length; $$index_1 < $$length; $$index_1++) {
    let label = each_array_1[$$index_1];
    $$payload.out += `<text x="0"${$.attr("y", yScale(label) + yScale.bandwidth() / 2)} dy="0.35em" text-anchor="middle"${$.attr("opacity", YX.get(label) ? "0.5" : "1")}${$.attr_style(`font-family: ${$.stringify(alloFonts.family)}; font-size: 14px; fill: ${$.stringify(alloColors.css.darkergrey)};`)}>${$.escape(label)}</text>`;
  }
  $$payload.out += `<!--]--></g></g></svg>`;
  $.pop();
}
function Legend($$payload, $$props) {
  $.push();
  let {
    diamond_dat,
    max_count_log,
    tickSize = 0,
    height = 44 + tickSize,
    width = 300,
    marginTop = 13,
    marginBottom = 16 + tickSize,
    marginLeft = 0,
    N_CATEGO = 20
  } = $$props;
  const myramp = range(N_CATEGO).map((i) => rgb(interpolateInferno(i / (N_CATEGO - 1))).hex());
  const color = scaleOrdinal(range(N_CATEGO), myramp);
  let x = scaleBand().domain(color.domain()).rangeRound([marginLeft, width - 100]);
  let x2 = scaleBand().domain(range(max_count_log).map((i) => 10 ** i).sort((a, b) => b - a)).rangeRound([marginLeft - 40, width - 90]);
  const each_array = $.ensure_array_like(color.domain());
  const each_array_1 = $.ensure_array_like(x2.domain());
  $$payload.out += `<svg${$.attr("width", width)}${$.attr("height", height)}${$.attr("viewBox", `0 0 ${$.stringify(width)} ${$.stringify(height)}`)} style="overflow: visible; display: block;"><!--[-->`;
  for (let $$index = 0, $$length = each_array.length; $$index < $$length; $$index++) {
    let d = each_array[$$index];
    $$payload.out += `<rect${$.attr("x", x(d))}${$.attr("y", marginTop)}${$.attr("width", Math.max(0, x.bandwidth()))}${$.attr("height", x.bandwidth())}${$.attr("fill", color(d))} transform="rotate(-90) translate(-70,0)" style="stroke: black; stroke-width: 0.65; shape-rendering: crispEdges;"></rect>`;
  }
  $$payload.out += `<!--]--><g transform="rotate(-90) translate(-60,5)"><!--[-->`;
  for (let $$index_1 = 0, $$length = each_array_1.length; $$index_1 < $$length; $$index_1++) {
    let tick = each_array_1[$$index_1];
    $$payload.out += `<g class="tick"${$.attr("transform", `translate(${$.stringify(x2(tick))}, 0)`)}><text dx="30" dy="-30" transform="rotate(90)"${$.attr_style(`font-family: ${$.stringify(alloFonts.family)}; font-size: 14px; fill: ${$.stringify(alloColors.css.verydarkgrey)}; text-anchor: start;`)}>${$.escape(tick)}</text></g>`;
  }
  $$payload.out += `<!--]--><text class="title"${$.attr("x", marginLeft - 25)}${$.attr("y", marginTop + marginBottom)} dx="30" dy="-5" transform="rotate(90)"${$.attr_style(`font-family: ${$.stringify(alloFonts.family)}; font-size: 14px; fill: ${$.stringify(alloColors.css.verydarkgrey)}; text-anchor: start;`)}>Counts per cell</text></g></svg>`;
  $.pop();
}
function Dashboard($$payload, $$props) {
  $.push();
  let {
    dat = null,
    alpha = 0.58,
    divnorm = 1,
    barData = [],
    balanceData = [],
    xDomain = void 0,
    instrumentText = "Instrument: Rank-Turbulence Divergence",
    title = ["System 1", "System 2"],
    maxlog10 = 0,
    height = 815,
    width = 1200,
    DashboardHeight = 815,
    DashboardWidth = 1200,
    DiamondHeight = 600,
    DiamondWidth = 600,
    WordshiftWidth = 640,
    marginInner = 160,
    marginDiamond = 40,
    max_count_log = void 0,
    labelThreshold = Infinity
  } = $$props;
  let max_shift = barData.length > 0 ? Math.max(...barData.map((d) => Math.abs(d.metric))) : 1;
  let wordshiftXDomain = xDomain || [-max_shift * 1.5, max_shift * 1.5];
  $$payload.out += `<div id="allotaxonometer-dashboard" style="position: relative; margin: 0; padding: 0;"><div style="display:flex; flex-wrap: wrap; align-items:center; justify-content: center; row-gap: 50px;"><div id="diamond-group" style="margin-top:20px; margin-right: -50px;"><div style="display:flex; gap: 10em; justify-content: center; margin-bottom: -50px; margin-right: 55px; position: relative;"><div style="position: relative;"><div${$.attr_style(`font-family: ${$.stringify(alloFonts.family)}; font-size: 16px; color: ${$.stringify(alloColors.css.superdarkgrey)};`)}>${$.escape(title[0])}</div> <div${$.attr_style(`position: absolute; top: 100%; left: -8em; margin-top: 3em; font-family: ${$.stringify(alloFonts.family)}; font-size: 14px; color: ${$.stringify(alloColors.css.darkgrey)}; width: 150px; line-height: 1;`)}><div style="margin-bottom: 0.5em;">${$.escape(instrumentText)}</div> <div>α = ${$.escape(alpha)}</div></div></div> <div${$.attr_style(`font-family: ${$.stringify(alloFonts.family)}; font-size: 16px; color: ${$.stringify(alloColors.css.superdarkgrey)};`)}>${$.escape(title[1])}</div></div> <div id="diamondplot">`;
  Diamond($$payload, {
    dat,
    alpha,
    divnorm,
    title,
    maxlog10,
    DiamondHeight,
    marginInner,
    marginDiamond,
    labelThreshold
  });
  $$payload.out += `<!----></div> <div style="display: flex; gap: 13em; justify-content: center;"><div id="legend" style="margin-left: -50px;">`;
  Legend($$payload, {
    diamond_dat: dat.counts,
    max_count_log: max_count_log || 5
  });
  $$payload.out += `<!----></div> <div id="balance">`;
  DivergingBarChart($$payload, {
    data: balanceData
  });
  $$payload.out += `<!----></div></div></div> <div style="margin-top:60px; overflow: visible;"><div id="wordshift" style="overflow: visible;">`;
  Wordshift($$payload, {
    barData,
    xDomain: wordshiftXDomain,
    width: WordshiftWidth,
    marginLeft: 110
  });
  $$payload.out += `<!----></div></div></div></div>`;
  $.pop();
}
function getUnions(x, y) {
  let a = new Set(x);
  let b = new Set(y);
  return /* @__PURE__ */ new Set([...a, ...b]);
}
function setdiff(x, y) {
  let a = new Set(x);
  let b = new Set(y);
  return new Set(
    [...a].filter((x2) => !b.has(x2))
  );
}
function which(x) {
  return x.reduce(
    (out, bool, index) => bool ? out.concat(index) : out,
    []
  );
}
function matlab_sort(A, rev = false) {
  if (A.length === 0) return { value: [], orig_idx: [] };
  if (A.length === 1) return { value: [...A], orig_idx: [0] };
  const indexedArray = A.map((value2, index) => ({ value: value2, index }));
  if (rev) {
    indexedArray.sort((a, b) => b.value - a.value);
  } else {
    indexedArray.sort((a, b) => a.value - b.value);
  }
  const value = indexedArray.map((item) => item.value);
  const orig_idx = indexedArray.map((item) => item.index);
  return { value, orig_idx };
}
function tiedrank(arr) {
  if (arr.length === 0) return [];
  const valueMap = /* @__PURE__ */ new Map();
  arr.forEach((value, index) => {
    if (!valueMap.has(value)) {
      valueMap.set(value, []);
    }
    valueMap.get(value).push(index);
  });
  const sortedValues = [...valueMap.keys()].sort((a, b) => b - a);
  const ranks = new Array(arr.length);
  let currentRank = 1;
  for (const value of sortedValues) {
    const indices = valueMap.get(value);
    const tieCount = indices.length;
    const avgRank = currentRank + (tieCount - 1) / 2;
    indices.forEach((index) => {
      ranks[index] = avgRank;
    });
    currentRank += tieCount;
  }
  return ranks;
}
function rank_maxlog10(mixedelements) {
  let max1 = mixedelements[[0]].ranks[0];
  for (let i = 1; i < mixedelements[[0]].ranks.length; i++) {
    if (mixedelements[[0]].ranks[i] > max1) max1 = mixedelements[[0]].ranks[i];
  }
  let max2 = mixedelements[[1]].ranks[0];
  for (let i = 1; i < mixedelements[[1]].ranks.length; i++) {
    if (mixedelements[[1]].ranks[i] > max2) max2 = mixedelements[[1]].ranks[i];
  }
  const logged_max1 = Math.log10(max1);
  const logged_max2 = Math.log10(max2);
  return Math.ceil(Math.max(logged_max1, logged_max2));
}
function rin(arr1, arr2) {
  return Array.from(arr1, (x) => {
    return arr2.indexOf(x) == -1 ? false : true;
  });
}
function zeros(length) {
  function createArray(length2) {
    var arr = new Array(length2 || 0), i = length2;
    if (arguments.length > 1) {
      var args = Array.prototype.slice.call(arguments, 1);
      while (i--) arr[length2 - 1 - i] = createArray.apply(this, args);
    }
    return arr;
  }
  let empty_mat = createArray(length, length);
  return Array.from(empty_mat, (arr) => arr.fill(0));
}
function normalizeDataset(data) {
  if (data.every((item) => item.probs !== void 0 && item.totalunique !== void 0)) {
    return data;
  }
  const totalCount = data.reduce((sum2, item) => sum2 + item.counts, 0);
  const totalunique = data.length;
  return data.map((item) => ({
    types: item.types,
    counts: item.counts,
    totalunique: item.totalunique ?? totalunique,
    probs: item.probs ?? item.counts / totalCount
  }));
}
function combElems(elem1, elem2) {
  const normalized1 = normalizeDataset(elem1);
  const normalized2 = normalizeDataset(elem2);
  const typeMap = /* @__PURE__ */ new Map();
  for (const item of normalized1) {
    typeMap.set(item.types, {
      counts1: item.counts,
      probs1: item.probs,
      counts2: 0,
      probs2: 0
    });
  }
  for (const item of normalized2) {
    const existing = typeMap.get(item.types);
    if (existing) {
      existing.counts2 = item.counts;
      existing.probs2 = item.probs;
    } else {
      typeMap.set(item.types, {
        counts1: 0,
        probs1: 0,
        counts2: item.counts,
        probs2: item.probs
      });
    }
  }
  const unionArray = [...typeMap.keys()];
  const len = unionArray.length;
  const counts1 = new Array(len);
  const counts2 = new Array(len);
  const probs1 = new Array(len);
  const probs2 = new Array(len);
  let i = 0;
  for (const [type, data] of typeMap) {
    unionArray[i] = type;
    counts1[i] = data.counts1;
    counts2[i] = data.counts2;
    probs1[i] = data.probs1;
    probs2[i] = data.probs2;
    i++;
  }
  return [
    {
      types: unionArray,
      counts: counts1,
      probs: probs1,
      ranks: tiedrank(counts1),
      totalunique: len
    },
    {
      types: unionArray,
      counts: counts2,
      probs: probs2,
      ranks: tiedrank(counts2),
      totalunique: len
    }
  ];
}
function balanceDat(elem1, elem2) {
  const types_1 = elem1.map((d) => d.types);
  const types_2 = elem2.map((d) => d.types);
  const union_types = getUnions(types_1, types_2);
  const tot_types = types_1.length + types_2.length;
  return [
    { y_coord: "total count", frequency: +(types_2.length / tot_types).toFixed(3) },
    { y_coord: "total count", frequency: -(types_1.length / tot_types).toFixed(3) },
    { y_coord: "all types", frequency: +(types_2.length / union_types.size).toFixed(3) },
    { y_coord: "all types", frequency: -(types_1.length / union_types.size).toFixed(3) },
    { y_coord: "exclusive types", frequency: +(setdiff(types_2, types_1).size / types_2.length).toFixed(3) },
    { y_coord: "exclusive types", frequency: -(setdiff(types_1, types_2).size / types_1.length).toFixed(3) }
  ];
}
function wordShift_dat(me, dat) {
  const out = [];
  for (let i = 0; i < me[0]["types"].length; i++) {
    const rank_diff = me[0]["ranks"][i] - me[1]["ranks"][i];
    out.push({
      "type": `${me[0]["types"][i]} (${me[0]["ranks"][i]} ⇋ ${me[1]["ranks"][i]})`,
      "rank_diff": rank_diff,
      "metric": rank_diff < 0 ? -dat.deltas[i] : dat.deltas[i]
    });
  }
  return out.slice().sort((a, b) => descending(Math.abs(a.metric), Math.abs(b.metric)));
}
let wasmModule = null;
let wasmInitialized = false;
(async function initWASM() {
  try {
    const module = await import("./assets/allotaxonometer_wasm-BIgsRP_p.js");
    await module.default();
    wasmModule = module;
    wasmInitialized = true;
    console.log("✅ WASM module loaded successfully");
  } catch (e) {
    console.warn("⚠️  WASM not available, using JS fallback:", e.message);
    wasmModule = null;
    wasmInitialized = false;
  }
})();
function divElems(inv_r1, inv_r2, alpha) {
  if (alpha === Infinity) {
    return inv_r1.map((d, i) => inv_r1[i] == inv_r2[i] ? 0 : Math.max(inv_r1[i], inv_r2[i]));
  } else if (alpha == 0) {
    const x_max = inv_r1.map((d, i) => Math.max(1 / inv_r1[i], 1 / inv_r2[i]));
    const x_min = inv_r1.map((d, i) => Math.min(1 / inv_r1[i], 1 / inv_r2[i]));
    return inv_r1.map((d, i) => Math.log10(x_max[i] / x_min[i]));
  } else {
    return inv_r1.map((d, i) => (alpha + 1) / alpha * Math.abs(inv_r1[i] ** alpha - inv_r2[i] ** alpha) ** (1 / (alpha + 1)));
  }
}
function norm_divElems(mixedelements, inv_r1, inv_r2, alpha) {
  const c1 = mixedelements[0]["counts"];
  const c2 = mixedelements[1]["counts"];
  const indices1 = which(c1.map((d) => d > 0));
  const indices2 = which(c2.map((d) => d > 0));
  const N1 = indices1.length;
  const N2 = indices2.length;
  function calc_disjoint(N12, N22) {
    return 1 / (N22 + N12 / 2);
  }
  const inv_r1_disjoint = calc_disjoint(N1, N2);
  const inv_r2_disjoint = calc_disjoint(N2, N1);
  if (alpha === Infinity) {
    return sum(indices1.map((i) => inv_r1[i])) + sum(indices2.map((i) => inv_r2[i]));
  } else if (alpha === 0) {
    const term1 = sum(
      indices1.map((i) => Math.abs(Math.log(inv_r1[i] / inv_r2_disjoint)))
    );
    const term2 = sum(
      indices2.map((i) => Math.abs(Math.log(inv_r2[i] / inv_r1_disjoint)))
    );
    return term1 + term2;
  } else {
    const term1 = (alpha + 1) / alpha * sum(
      indices1.map((i) => inv_r1[i]).map((d) => (Math.abs(d ** alpha) - inv_r2_disjoint ** alpha) ** (1 / (alpha + 1)))
    );
    const term2 = (alpha + 1) / alpha * sum(
      indices2.map((i) => inv_r2[i]).map((d) => Math.abs(inv_r1_disjoint ** alpha - d ** alpha) ** (1 / (alpha + 1)))
    );
    return term1 + term2;
  }
}
function rank_turbulence_divergence_js(mixedelements, alpha) {
  const inv_r1 = mixedelements[0]["ranks"].map((d) => Math.pow(d, -1));
  const inv_r2 = mixedelements[1]["ranks"].map((d) => Math.pow(d, -1));
  const divergence_elements = divElems(inv_r1, inv_r2, alpha);
  const normalization = norm_divElems(mixedelements, inv_r1, inv_r2, alpha);
  return {
    "divergence_elements": divergence_elements.map((d) => d / normalization),
    "normalization": normalization
  };
}
function rank_turbulence_divergence(mixedelements, alpha) {
  if (wasmInitialized && wasmModule) {
    try {
      const result = wasmModule.rank_turbulence_divergence(
        mixedelements[0]["ranks"],
        mixedelements[1]["ranks"],
        mixedelements[0]["counts"],
        mixedelements[1]["counts"],
        alpha
      );
      if (result instanceof Map) {
        return {
          divergence_elements: result.get("divergence_elements"),
          normalization: result.get("normalization")
        };
      }
      if (result && typeof result === "object" && result.divergence_elements) {
        return {
          divergence_elements: result.divergence_elements,
          normalization: result.normalization
        };
      }
      console.warn("⚠️  WASM returned unexpected format, using JS fallback. Got:", result);
      return rank_turbulence_divergence_js(mixedelements, alpha);
    } catch (e) {
      console.warn("⚠️  WASM execution failed, using JS fallback:", e);
      return rank_turbulence_divergence_js(mixedelements, alpha);
    }
  } else {
    return rank_turbulence_divergence_js(mixedelements, alpha);
  }
}
function rank2coord(rank) {
  return Math.floor(Math.log10(rank) / (1 / 15));
}
function diamond_counts(mixedelements) {
  let maxlog10 = rank_maxlog10(mixedelements);
  if (maxlog10 < 1) maxlog10 = 1;
  const CELL_LENGTH = 1 / 15;
  const Ncells = Math.floor(maxlog10 / CELL_LENGTH) + 1;
  const coordGroups = /* @__PURE__ */ new Map();
  for (let i = 0; i < mixedelements[0]["ranks"].length; i++) {
    const x1 = rank2coord(mixedelements[1]["ranks"][i]);
    const y1 = rank2coord(mixedelements[0]["ranks"][i]);
    const key = `${x1},${y1}`;
    if (!coordGroups.has(key)) {
      coordGroups.set(key, []);
    }
    coordGroups.get(key).push({
      types: mixedelements[0]["types"][i],
      x1,
      y1,
      rank1: mixedelements[0]["ranks"][i],
      rank2: mixedelements[1]["ranks"][i]
    });
  }
  const result = [];
  for (let i = 0; i < Ncells; i++) {
    for (let j = 0; j < Ncells; j++) {
      const key = `${i},${j}`;
      const items = coordGroups.get(key);
      if (!items) {
        result.push({
          x1: i,
          y1: j,
          coord_on_diag: (j + i) / 2,
          cos_dist: (i - j) ** 2,
          rank: "",
          rank_L: "",
          rank_R: "",
          value: 0,
          types: "",
          which_sys: i - j <= 0 ? "right" : "left"
        });
      } else {
        result.push({
          x1: i,
          y1: j,
          coord_on_diag: (j + i) / 2,
          cos_dist: (i - j) ** 2,
          rank: `(${items[0].rank1}, ${items[0].rank2})`,
          rank_L: extent$1(items.map((d) => d.rank1)),
          rank_R: extent$1(items.map((d) => d.rank2)),
          value: items.length,
          types: items.map((d) => d.types).join(", "),
          which_sys: i - j <= 0 ? "right" : "left"
        });
      }
    }
  }
  return result;
}
function diamond_count(mixedelements, wordshift) {
  let deltas = wordshift["divergence_elements"];
  let sorted_div = matlab_sort(deltas, true);
  let indices_deltas = sorted_div.orig_idx;
  const len = indices_deltas.length;
  const reordered = {
    deltas: new Array(len),
    mixedelements: [
      { types: new Array(len), counts: new Array(len), ranks: new Array(len), probs: new Array(len) },
      { types: new Array(len), counts: new Array(len), ranks: new Array(len), probs: new Array(len) }
    ]
  };
  for (let i = 0; i < len; i++) {
    const idx = indices_deltas[i];
    reordered.deltas[i] = deltas[idx];
    reordered.mixedelements[0].types[i] = mixedelements[0]["types"][idx];
    reordered.mixedelements[0].counts[i] = mixedelements[0]["counts"][idx];
    reordered.mixedelements[0].ranks[i] = mixedelements[0]["ranks"][idx];
    reordered.mixedelements[0].probs[i] = mixedelements[0]["probs"][idx];
    reordered.mixedelements[1].types[i] = mixedelements[1]["types"][idx];
    reordered.mixedelements[1].counts[i] = mixedelements[1]["counts"][idx];
    reordered.mixedelements[1].ranks[i] = mixedelements[1]["ranks"][idx];
    reordered.mixedelements[1].probs[i] = mixedelements[1]["probs"][idx];
  }
  Object.assign(mixedelements[0], reordered.mixedelements[0]);
  Object.assign(mixedelements[1], reordered.mixedelements[1]);
  deltas = reordered.deltas;
  const deltas_loss = [...deltas];
  [...deltas];
  for (let i = 0; i < len; i++) {
    if (mixedelements[0]["ranks"][i] > mixedelements[1]["ranks"][i]) {
      deltas_loss[i] = -1;
    }
    if (mixedelements[1]["ranks"][i] < mixedelements[0]["ranks"][i]) ;
  }
  const counts = diamond_counts(mixedelements);
  let max_delta_loss = deltas_loss[0];
  for (let i = 1; i < deltas_loss.length; i++) {
    if (deltas_loss[i] > max_delta_loss) max_delta_loss = deltas_loss[i];
  }
  return { "counts": counts, "deltas": deltas, "max_delta_loss": max_delta_loss };
}
class Allotaxonograph {
  // Core data state
  sys1 = null;
  sys2 = null;
  title = ["System 1", "System 2"];
  // Configuration
  alpha = 0.58;
  topN = 30;
  #me = $.derived(() => this.sys1 && this.sys2 ? combElems(this.sys1, this.sys2) : null);
  get me() {
    return this.#me();
  }
  set me($$value) {
    return this.#me($$value);
  }
  #rtd = $.derived(() => this.me ? rank_turbulence_divergence(this.me, this.alpha) : null);
  get rtd() {
    return this.#rtd();
  }
  set rtd($$value) {
    return this.#rtd($$value);
  }
  #dat = $.derived(() => this.me && this.rtd ? diamond_count(this.me, this.rtd) : null);
  get dat() {
    return this.#dat();
  }
  set dat($$value) {
    return this.#dat($$value);
  }
  #barData = $.derived(() => this.me && this.dat ? wordShift_dat(this.me, this.dat).slice(0, this.topN) : []);
  get barData() {
    return this.#barData();
  }
  set barData($$value) {
    return this.#barData($$value);
  }
  #balanceData = $.derived(() => this.sys1 && this.sys2 ? balanceDat(this.sys1, this.sys2) : []);
  get balanceData() {
    return this.#balanceData();
  }
  set balanceData($$value) {
    return this.#balanceData($$value);
  }
  #maxlog10 = $.derived(() => this.me ? Math.ceil(d3.max([
    Math.log10(d3.max(this.me[0].ranks)),
    Math.log10(d3.max(this.me[1].ranks))
  ])) : 0);
  get maxlog10() {
    return this.#maxlog10();
  }
  set maxlog10($$value) {
    return this.#maxlog10($$value);
  }
  #max_count_log = $.derived(() => this.dat ? Math.ceil(Math.log10(d3.max(this.dat.counts, (d) => d.value))) + 1 : 2);
  get max_count_log() {
    return this.#max_count_log();
  }
  set max_count_log($$value) {
    return this.#max_count_log($$value);
  }
  #max_shift = $.derived(() => this.barData.length > 0 ? d3.max(this.barData, (d) => Math.abs(d.metric)) : 1);
  get max_shift() {
    return this.#max_shift();
  }
  set max_shift($$value) {
    return this.#max_shift($$value);
  }
  #divnorm = $.derived(() => this.rtd?.normalization);
  get divnorm() {
    return this.#divnorm();
  }
  set divnorm($$value) {
    return this.#divnorm($$value);
  }
  #xDomain = $.derived(() => [-this.max_shift * 1.5, this.max_shift * 1.5]);
  get xDomain() {
    return this.#xDomain();
  }
  set xDomain($$value) {
    return this.#xDomain($$value);
  }
  #isDataReady = $.derived(() => this.dat && this.barData && this.balanceData && this.me && this.rtd);
  get isDataReady() {
    return this.#isDataReady();
  }
  set isDataReady($$value) {
    return this.#isDataReady($$value);
  }
  constructor(sys1, sys2, options = {}) {
    if (sys1) this.sys1 = sys1;
    if (sys2) this.sys2 = sys2;
    if (options.alpha !== void 0) this.alpha = options.alpha;
    if (options.title) this.title = options.title;
    if (options.topN !== void 0) this.topN = options.topN;
  }
  // Methods
  updateData(sys1, sys2, titles) {
    this.sys1 = sys1;
    this.sys2 = sys2;
    if (titles) this.title = titles;
  }
  setAlpha(alpha) {
    this.alpha = alpha;
  }
  setTopN(n) {
    this.topN = n;
  }
  exportData() {
    return {
      rtd: this.rtd,
      barData: this.barData,
      balanceData: this.balanceData,
      meta: {
        alpha: this.alpha,
        titles: this.title,
        maxlog10: this.maxlog10,
        max_count_log: this.max_count_log
      }
    };
  }
  // Static factory methods
  static fromData(data1, data2, options) {
    return new Allotaxonograph(data1, data2, options);
  }
  static createComparison(data1, data2, preset = "standard") {
    const presets = {
      standard: { alpha: 0.58 },
      sensitive: { alpha: 0.1 },
      robust: { alpha: Infinity }
    };
    return new Allotaxonograph(data1, data2, presets[preset]);
  }
}
export {
  Allotaxonograph,
  Dashboard,
  Diamond,
  DivergingBarChart,
  Legend,
  Wordshift,
  balanceDat,
  combElems,
  diamond_count,
  getUnions,
  matlab_sort,
  rank_maxlog10,
  rank_turbulence_divergence,
  rin,
  setdiff,
  tiedrank,
  which,
  wordShift_dat,
  zeros
};
