from __future__ import annotations 

import re
import sys
from datetime import (
    date,
    datetime,
    time
)
from decimal import Decimal 
from enum import Enum 
from typing import (
    Any,
    ClassVar,
    Literal,
    Optional,
    Union
)

from pydantic import (
    BaseModel,
    ConfigDict,
    Field,
    RootModel,
    field_validator
)


metamodel_version = "None"
version = "11.9.0"


class ConfiguredBaseModel(BaseModel):
    model_config = ConfigDict(
        validate_assignment = True,
        validate_default = True,
        extra = "forbid",
        arbitrary_types_allowed = True,
        use_enum_values = True,
        strict = False,
    )
    pass




class LinkMLMeta(RootModel):
    root: dict[str, Any] = {}
    model_config = ConfigDict(frozen=True)

    def __getattr__(self, key:str):
        return getattr(self.root, key)

    def __getitem__(self, key:str):
        return self.root[key]

    def __setitem__(self, key:str, value):
        self.root[key] = value

    def __contains__(self, key:str) -> bool:
        return key in self.root


linkml_meta = LinkMLMeta({'default_prefix': 'nmdc',
     'default_range': 'string',
     'description': 'The NMDC Schema is a foundational framework designed to '
                    'standardize metadata for the National Microbiome Data  '
                    'Collaborative (NMDC) and collaborating data providors. By '
                    'establishing a structured approach to metadata, the NMDC '
                    'Schema enables researchers to organize,  share, and interpret '
                    'complex datasets with consistency and clarity. The NMDC '
                    'Schema is critical substrate used to facilitate  '
                    'interoperability and collaboration, as it provide a common '
                    'language for data exchange across systems and disciplines.  '
                    'In the context of the NMDC, this schema supports the '
                    'integration of microbiome data from medicine, agriculture,  '
                    'bioenergy, and environmental science into a cohesive '
                    'platform.',
     'emit_prefixes': ['KEGG.ORTHOLOGY',
                       'MASSIVE',
                       'biosample',
                       'cas',
                       'doi',
                       'gnps.task',
                       'gold',
                       'img.taxon',
                       'jgi.proposal',
                       'kegg',
                       'rdf',
                       'rdfs',
                       'skos',
                       'xsd'],
     'id': 'https://w3id.org/nmdc/nmdc',
     'license': 'https://creativecommons.org/publicdomain/zero/1.0/',
     'name': 'NMDC',
     'notes': ['not importing any MIxS terms where the relationship between the '
               "name (SCN) and the id isn't 1:1"],
     'prefixes': {'BFO': {'prefix_prefix': 'BFO',
                          'prefix_reference': 'http://purl.obolibrary.org/obo/BFO_'},
                  'CATH': {'prefix_prefix': 'CATH',
                           'prefix_reference': 'https://bioregistry.io/cath:'},
                  'CHEBI': {'prefix_prefix': 'CHEBI',
                            'prefix_reference': 'http://purl.obolibrary.org/obo/CHEBI_'},
                  'CHEMBL.COMPOUND': {'prefix_prefix': 'CHEMBL.COMPOUND',
                                      'prefix_reference': 'https://bioregistry.io/chembl.compound:'},
                  'CHMO': {'prefix_prefix': 'CHMO',
                           'prefix_reference': 'http://purl.obolibrary.org/obo/CHMO_'},
                  'COG': {'prefix_prefix': 'COG',
                          'prefix_reference': 'https://bioregistry.io/cog:'},
                  'Contaminant': {'prefix_prefix': 'Contaminant',
                                  'prefix_reference': 'http://example.org/contaminant/'},
                  'DRUGBANK': {'prefix_prefix': 'DRUGBANK',
                               'prefix_reference': 'https://bioregistry.io/drugbank:'},
                  'EC': {'prefix_prefix': 'EC',
                         'prefix_reference': 'https://bioregistry.io/eccode:'},
                  'EFO': {'prefix_prefix': 'EFO',
                          'prefix_reference': 'http://www.ebi.ac.uk/efo/'},
                  'EGGNOG': {'prefix_prefix': 'EGGNOG',
                             'prefix_reference': 'https://bioregistry.io/eggnog:'},
                  'ENVO': {'prefix_prefix': 'ENVO',
                           'prefix_reference': 'http://purl.obolibrary.org/obo/ENVO_'},
                  'FBcv': {'prefix_prefix': 'FBcv',
                           'prefix_reference': 'http://purl.obolibrary.org/obo/FBcv_'},
                  'FMA': {'prefix_prefix': 'FMA',
                          'prefix_reference': 'http://purl.obolibrary.org/obo/FMA_'},
                  'GENEPIO': {'prefix_prefix': 'GENEPIO',
                              'prefix_reference': 'http://purl.obolibrary.org/obo/GENEPIO_'},
                  'GO': {'prefix_prefix': 'GO',
                         'prefix_reference': 'http://purl.obolibrary.org/obo/GO_'},
                  'HMDB': {'prefix_prefix': 'HMDB',
                           'prefix_reference': 'https://bioregistry.io/hmdb:'},
                  'ISA': {'prefix_prefix': 'ISA',
                          'prefix_reference': 'http://example.org/isa/'},
                  'KEGG.COMPOUND': {'prefix_prefix': 'KEGG.COMPOUND',
                                    'prefix_reference': 'https://bioregistry.io/kegg.compound:'},
                  'KEGG.MODULE': {'prefix_prefix': 'KEGG.MODULE',
                                  'prefix_reference': 'https://bioregistry.io/kegg.module:'},
                  'KEGG.ORTHOLOGY': {'prefix_prefix': 'KEGG.ORTHOLOGY',
                                     'prefix_reference': 'https://bioregistry.io/kegg.orthology:'},
                  'KEGG.REACTION': {'prefix_prefix': 'KEGG.REACTION',
                                    'prefix_reference': 'https://bioregistry.io/kegg.reaction:'},
                  'KEGG_PATHWAY': {'prefix_prefix': 'KEGG_PATHWAY',
                                   'prefix_reference': 'https://bioregistry.io/kegg.pathway:'},
                  'MASSIVE': {'prefix_prefix': 'MASSIVE',
                              'prefix_reference': 'https://bioregistry.io/reference/massive:'},
                  'MCO': {'prefix_prefix': 'MCO',
                          'prefix_reference': 'http://purl.obolibrary.org/obo/MICRO_'},
                  'MESH': {'prefix_prefix': 'MESH',
                           'prefix_reference': 'https://bioregistry.io/mesh:'},
                  'MISO': {'prefix_prefix': 'MISO',
                           'prefix_reference': 'http://purl.obolibrary.org/obo/MISO_'},
                  'MIXS': {'prefix_prefix': 'MIXS',
                           'prefix_reference': 'https://w3id.org/mixs/'},
                  'MS': {'prefix_prefix': 'MS',
                         'prefix_reference': 'http://purl.obolibrary.org/obo/MS_'},
                  'MetaCyc': {'prefix_prefix': 'MetaCyc',
                              'prefix_reference': 'https://bioregistry.io/metacyc.compound:'},
                  'MetaNetX': {'prefix_prefix': 'MetaNetX',
                               'prefix_reference': 'http://example.org/metanetx/'},
                  'NCBI': {'prefix_prefix': 'NCBI',
                           'prefix_reference': 'http://example.org/ncbitaxon/'},
                  'NCBITaxon': {'prefix_prefix': 'NCBITaxon',
                                'prefix_reference': 'http://purl.obolibrary.org/obo/NCBITaxon_'},
                  'NCIT': {'prefix_prefix': 'NCIT',
                           'prefix_reference': 'http://purl.obolibrary.org/obo/NCIT_'},
                  'OBI': {'prefix_prefix': 'OBI',
                          'prefix_reference': 'http://purl.obolibrary.org/obo/OBI_'},
                  'OMIT': {'prefix_prefix': 'OMIT',
                           'prefix_reference': 'http://purl.obolibrary.org/obo/OMIT_'},
                  'ORCID': {'prefix_prefix': 'ORCID',
                            'prefix_reference': 'https://orcid.org/'},
                  'PANTHER.FAMILY': {'prefix_prefix': 'PANTHER.FAMILY',
                                     'prefix_reference': 'https://bioregistry.io/panther.family:'},
                  'PATO': {'prefix_prefix': 'PATO',
                           'prefix_reference': 'http://purl.obolibrary.org/obo/PATO_'},
                  'PFAM': {'prefix_prefix': 'PFAM',
                           'prefix_reference': 'https://bioregistry.io/pfam:'},
                  'PFAM.CLAN': {'prefix_prefix': 'PFAM.CLAN',
                                'prefix_reference': 'https://bioregistry.io/pfam.clan:'},
                  'PO': {'prefix_prefix': 'PO',
                         'prefix_reference': 'http://purl.obolibrary.org/obo/PO_'},
                  'PR': {'prefix_prefix': 'PR',
                         'prefix_reference': 'http://purl.obolibrary.org/obo/PR_'},
                  'PUBCHEM.COMPOUND': {'prefix_prefix': 'PUBCHEM.COMPOUND',
                                       'prefix_reference': 'https://bioregistry.io/pubchem.compound:'},
                  'RHEA': {'prefix_prefix': 'RHEA',
                           'prefix_reference': 'https://bioregistry.io/rhea:'},
                  'RO': {'prefix_prefix': 'RO',
                         'prefix_reference': 'http://purl.obolibrary.org/obo/RO_'},
                  'RetroRules': {'prefix_prefix': 'RetroRules',
                                 'prefix_reference': 'http://example.org/retrorules/'},
                  'SEED': {'prefix_prefix': 'SEED',
                           'prefix_reference': 'https://bioregistry.io/seed:'},
                  'SIO': {'prefix_prefix': 'SIO',
                          'prefix_reference': 'http://semanticscience.org/resource/SIO_'},
                  'SO': {'prefix_prefix': 'SO',
                         'prefix_reference': 'http://purl.obolibrary.org/obo/SO_'},
                  'SUPFAM': {'prefix_prefix': 'SUPFAM',
                             'prefix_reference': 'https://bioregistry.io/supfam:'},
                  'TAXRANK': {'prefix_prefix': 'TAXRANK',
                              'prefix_reference': 'http://purl.obolibrary.org/obo/TAXRANK_'},
                  'TIGRFAM': {'prefix_prefix': 'TIGRFAM',
                              'prefix_reference': 'https://bioregistry.io/tigrfam:'},
                  'UBERON': {'prefix_prefix': 'UBERON',
                             'prefix_reference': 'http://purl.obolibrary.org/obo/UBERON_'},
                  'UO': {'prefix_prefix': 'UO',
                         'prefix_reference': 'http://purl.obolibrary.org/obo/UO_'},
                  'UniProtKB': {'prefix_prefix': 'UniProtKB',
                                'prefix_reference': 'https://bioregistry.io/uniprot:'},
                  'biolink': {'prefix_prefix': 'biolink',
                              'prefix_reference': 'https://w3id.org/biolink/vocab/'},
                  'bioproject': {'prefix_prefix': 'bioproject',
                                 'prefix_reference': 'https://bioregistry.io/bioproject:'},
                  'biosample': {'prefix_prefix': 'biosample',
                                'prefix_reference': 'https://bioregistry.io/biosample:'},
                  'cas': {'prefix_prefix': 'cas',
                          'prefix_reference': 'https://bioregistry.io/cas:'},
                  'dcterms': {'prefix_prefix': 'dcterms',
                              'prefix_reference': 'http://purl.org/dc/terms/'},
                  'doi': {'prefix_prefix': 'doi',
                          'prefix_reference': 'https://bioregistry.io/doi:'},
                  'edam.data': {'prefix_prefix': 'edam.data',
                                'prefix_reference': 'http://edamontology.org/data_'},
                  'edam.format': {'prefix_prefix': 'edam.format',
                                  'prefix_reference': 'http://edamontology.org/format_'},
                  'emsl': {'prefix_prefix': 'emsl',
                           'prefix_reference': 'http://example.org/emsl_in_mongodb/'},
                  'emsl.project': {'prefix_prefix': 'emsl.project',
                                   'prefix_reference': 'https://bioregistry.io/emsl.project:'},
                  'emsl_uuid_like': {'prefix_prefix': 'emsl_uuid_like',
                                     'prefix_reference': 'http://example.org/emsl_uuid_like/'},
                  'generic': {'prefix_prefix': 'generic',
                              'prefix_reference': 'http://example.org/generic/'},
                  'gnps.task': {'prefix_prefix': 'gnps.task',
                                'prefix_reference': 'https://bioregistry.io/gnps.task:'},
                  'gold': {'prefix_prefix': 'gold',
                           'prefix_reference': 'https://bioregistry.io/gold:'},
                  'gtpo': {'prefix_prefix': 'gtpo',
                           'prefix_reference': 'http://example.org/gtpo/'},
                  'igsn': {'prefix_prefix': 'igsn',
                           'prefix_reference': 'https://app.geosamples.org/sample/igsn/'},
                  'img.taxon': {'prefix_prefix': 'img.taxon',
                                'prefix_reference': 'https://bioregistry.io/img.taxon:'},
                  'insdc.sra': {'prefix_prefix': 'insdc.sra',
                                'prefix_reference': 'https://bioregistry.io/insdc.sra:'},
                  'jgi': {'prefix_prefix': 'jgi',
                          'prefix_reference': 'http://example.org/jgi/'},
                  'jgi.analysis': {'prefix_prefix': 'jgi.analysis',
                                   'prefix_reference': 'https://data.jgi.doe.gov/search?q='},
                  'jgi.proposal': {'prefix_prefix': 'jgi.proposal',
                                   'prefix_reference': 'https://bioregistry.io/jgi.proposal:'},
                  'kegg': {'prefix_prefix': 'kegg',
                           'prefix_reference': 'https://bioregistry.io/kegg:'},
                  'linkml': {'prefix_prefix': 'linkml',
                             'prefix_reference': 'https://w3id.org/linkml/'},
                  'mgnify.analysis': {'prefix_prefix': 'mgnify.analysis',
                                      'prefix_reference': 'https://bioregistry.io/mgnify.analysis:'},
                  'mgnify.proj': {'prefix_prefix': 'mgnify.proj',
                                  'prefix_reference': 'https://bioregistry.io/mgnify.proj:'},
                  'my_emsl': {'prefix_prefix': 'my_emsl',
                              'prefix_reference': 'https://release.my.emsl.pnnl.gov/released_data/'},
                  'neon.identifier': {'prefix_prefix': 'neon.identifier',
                                      'prefix_reference': 'http://example.org/neon/identifier/'},
                  'neon.schema': {'prefix_prefix': 'neon.schema',
                                  'prefix_reference': 'http://example.org/neon/schema/'},
                  'nmdc': {'prefix_prefix': 'nmdc',
                           'prefix_reference': 'https://w3id.org/nmdc/'},
                  'owl': {'prefix_prefix': 'owl',
                          'prefix_reference': 'http://www.w3.org/2002/07/owl#'},
                  'prov': {'prefix_prefix': 'prov',
                           'prefix_reference': 'http://www.w3.org/ns/prov#'},
                  'qud': {'prefix_prefix': 'qud',
                          'prefix_reference': 'http://qudt.org/1.1/schema/qudt#'},
                  'rdf': {'prefix_prefix': 'rdf',
                          'prefix_reference': 'http://www.w3.org/1999/02/22-rdf-syntax-ns#'},
                  'rdfs': {'prefix_prefix': 'rdfs',
                           'prefix_reference': 'http://www.w3.org/2000/01/rdf-schema#'},
                  'ror': {'prefix_prefix': 'ror',
                          'prefix_reference': 'https://bioregistry.io/ror:'},
                  'schema': {'prefix_prefix': 'schema',
                             'prefix_reference': 'http://schema.org/'},
                  'shex': {'prefix_prefix': 'shex',
                           'prefix_reference': 'http://www.w3.org/ns/shex#'},
                  'skos': {'prefix_prefix': 'skos',
                           'prefix_reference': 'http://www.w3.org/2004/02/skos/core#'},
                  'wgs84': {'prefix_prefix': 'wgs84',
                            'prefix_reference': 'http://www.w3.org/2003/01/geo/wgs84_pos#'},
                  'wikidata': {'prefix_prefix': 'wikidata',
                               'prefix_reference': 'http://www.wikidata.org/entity/'},
                  'xsd': {'prefix_prefix': 'xsd',
                          'prefix_reference': 'http://www.w3.org/2001/XMLSchema#'}},
     'settings': {'id_blade': {'setting_key': 'id_blade',
                               'setting_value': '([A-Za-z0-9]{1,})'},
                  'id_locus': {'setting_key': 'id_locus',
                               'setting_value': '(_[A-Za-z0-9_\\.-]+)?$'},
                  'id_nmdc_prefix': {'setting_key': 'id_nmdc_prefix',
                                     'setting_value': '^(nmdc)'},
                  'id_shoulder': {'setting_key': 'id_shoulder',
                                  'setting_value': '([0-9][a-z]{0,6}[0-9])'},
                  'id_version': {'setting_key': 'id_version',
                                 'setting_value': '(\\.[0-9]{1,})'}},
     'source_file': 'nmdc_schema/nmdc_materialized_patterns.yaml',
     'title': 'NMDC Schema',
     'types': {'boolean': {'base': 'Bool',
                           'description': 'A binary (true or false) value',
                           'exact_mappings': ['schema:Boolean'],
                           'from_schema': 'https://w3id.org/nmdc/nmdc',
                           'name': 'boolean',
                           'notes': ['If you are authoring schemas in LinkML YAML, '
                                     'the type is referenced with the lower case '
                                     '"boolean".'],
                           'repr': 'bool',
                           'uri': 'xsd:boolean'},
               'bytes': {'base': 'int',
                         'description': 'An integer value that corresponds to a '
                                        'size in bytes',
                         'from_schema': 'https://w3id.org/nmdc/nmdc',
                         'name': 'bytes',
                         'see_also': ['UO:0000233'],
                         'uri': 'xsd:long'},
               'curie': {'base': 'Curie',
                         'comments': ['in RDF serializations this MUST be expanded '
                                      'to a URI',
                                      'in non-RDF serializations MAY be serialized '
                                      'as the compact representation'],
                         'conforms_to': 'https://www.w3.org/TR/curie/',
                         'description': 'a compact URI',
                         'from_schema': 'https://w3id.org/nmdc/nmdc',
                         'name': 'curie',
                         'notes': ['If you are authoring schemas in LinkML YAML, '
                                   'the type is referenced with the lower case '
                                   '"curie".'],
                         'repr': 'str',
                         'uri': 'xsd:string'},
               'date': {'base': 'XSDDate',
                        'description': 'a date (year, month and day) in an '
                                       'idealized calendar',
                        'exact_mappings': ['schema:Date'],
                        'from_schema': 'https://w3id.org/nmdc/nmdc',
                        'name': 'date',
                        'notes': ["URI is dateTime because OWL reasoners don't "
                                  'work with straight date or time',
                                  'If you are authoring schemas in LinkML YAML, '
                                  'the type is referenced with the lower case '
                                  '"date".'],
                        'repr': 'str',
                        'uri': 'xsd:date'},
               'date_or_datetime': {'base': 'str',
                                    'description': 'Either a date or a datetime',
                                    'from_schema': 'https://w3id.org/nmdc/nmdc',
                                    'name': 'date_or_datetime',
                                    'notes': ['If you are authoring schemas in '
                                              'LinkML YAML, the type is referenced '
                                              'with the lower case '
                                              '"date_or_datetime".'],
                                    'repr': 'str',
                                    'uri': 'linkml:DateOrDatetime'},
               'datetime': {'base': 'XSDDateTime',
                            'description': 'The combination of a date and time',
                            'exact_mappings': ['schema:DateTime'],
                            'from_schema': 'https://w3id.org/nmdc/nmdc',
                            'name': 'datetime',
                            'notes': ['If you are authoring schemas in LinkML '
                                      'YAML, the type is referenced with the lower '
                                      'case "datetime".'],
                            'repr': 'str',
                            'uri': 'xsd:dateTime'},
               'decimal': {'base': 'Decimal',
                           'broad_mappings': ['schema:Number'],
                           'description': 'A real number with arbitrary precision '
                                          'that conforms to the xsd:decimal '
                                          'specification',
                           'from_schema': 'https://w3id.org/nmdc/nmdc',
                           'name': 'decimal',
                           'notes': ['If you are authoring schemas in LinkML YAML, '
                                     'the type is referenced with the lower case '
                                     '"decimal".'],
                           'uri': 'xsd:decimal'},
               'decimal_degree': {'base': 'float',
                                  'description': 'A decimal degree expresses '
                                                 'latitude or longitude as decimal '
                                                 'fractions.',
                                  'from_schema': 'https://w3id.org/nmdc/nmdc',
                                  'name': 'decimal_degree',
                                  'see_also': ['https://en.wikipedia.org/wiki/Decimal_degrees'],
                                  'uri': 'xsd:decimal'},
               'double': {'base': 'float',
                          'close_mappings': ['schema:Float'],
                          'description': 'A real number that conforms to the '
                                         'xsd:double specification',
                          'from_schema': 'https://w3id.org/nmdc/nmdc',
                          'name': 'double',
                          'notes': ['If you are authoring schemas in LinkML YAML, '
                                    'the type is referenced with the lower case '
                                    '"double".'],
                          'uri': 'xsd:double'},
               'external_identifier': {'description': 'A CURIE representing an '
                                                      'external identifier',
                                       'from_schema': 'https://w3id.org/nmdc/nmdc',
                                       'name': 'external_identifier',
                                       'pattern': '^[a-zA-Z0-9][a-zA-Z0-9_\\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\\-\\/\\.,]*$',
                                       'see_also': ['https://microbiomedata.github.io/nmdc-schema/identifiers/'],
                                       'typeof': 'uriorcurie',
                                       'uri': 'xsd:anyURI'},
               'float': {'base': 'float',
                         'description': 'A real number that conforms to the '
                                        'xsd:float specification',
                         'exact_mappings': ['schema:Float'],
                         'from_schema': 'https://w3id.org/nmdc/nmdc',
                         'name': 'float',
                         'notes': ['If you are authoring schemas in LinkML YAML, '
                                   'the type is referenced with the lower case '
                                   '"float".'],
                         'uri': 'xsd:float'},
               'integer': {'base': 'int',
                           'description': 'An integer',
                           'exact_mappings': ['schema:Integer'],
                           'from_schema': 'https://w3id.org/nmdc/nmdc',
                           'name': 'integer',
                           'notes': ['If you are authoring schemas in LinkML YAML, '
                                     'the type is referenced with the lower case '
                                     '"integer".'],
                           'uri': 'xsd:integer'},
               'jsonpath': {'base': 'str',
                            'conforms_to': 'https://www.ietf.org/archive/id/draft-goessner-dispatch-jsonpath-00.html',
                            'description': 'A string encoding a JSON Path. The '
                                           'value of the string MUST conform to '
                                           'JSON Point syntax and SHOULD '
                                           'dereference to zero or more valid '
                                           'objects within the current instance '
                                           'document when encoded in tree form.',
                            'from_schema': 'https://w3id.org/nmdc/nmdc',
                            'name': 'jsonpath',
                            'notes': ['If you are authoring schemas in LinkML '
                                      'YAML, the type is referenced with the lower '
                                      'case "jsonpath".'],
                            'repr': 'str',
                            'uri': 'xsd:string'},
               'jsonpointer': {'base': 'str',
                               'conforms_to': 'https://datatracker.ietf.org/doc/html/rfc6901',
                               'description': 'A string encoding a JSON Pointer. '
                                              'The value of the string MUST '
                                              'conform to JSON Point syntax and '
                                              'SHOULD dereference to a valid '
                                              'object within the current instance '
                                              'document when encoded in tree form.',
                               'from_schema': 'https://w3id.org/nmdc/nmdc',
                               'name': 'jsonpointer',
                               'notes': ['If you are authoring schemas in LinkML '
                                         'YAML, the type is referenced with the '
                                         'lower case "jsonpointer".'],
                               'repr': 'str',
                               'uri': 'xsd:string'},
               'language_code': {'base': 'str',
                                 'description': 'A language code conforming to '
                                                'ISO_639-1',
                                 'from_schema': 'https://w3id.org/nmdc/nmdc',
                                 'name': 'language_code',
                                 'see_also': ['https://en.wikipedia.org/wiki/ISO_639-1'],
                                 'uri': 'xsd:language'},
               'ncname': {'base': 'NCName',
                          'description': 'Prefix part of CURIE',
                          'from_schema': 'https://w3id.org/nmdc/nmdc',
                          'name': 'ncname',
                          'notes': ['If you are authoring schemas in LinkML YAML, '
                                    'the type is referenced with the lower case '
                                    '"ncname".'],
                          'repr': 'str',
                          'uri': 'xsd:string'},
               'nodeidentifier': {'base': 'NodeIdentifier',
                                  'description': 'A URI, CURIE or BNODE that '
                                                 'represents a node in a model.',
                                  'from_schema': 'https://w3id.org/nmdc/nmdc',
                                  'name': 'nodeidentifier',
                                  'notes': ['If you are authoring schemas in '
                                            'LinkML YAML, the type is referenced '
                                            'with the lower case '
                                            '"nodeidentifier".'],
                                  'repr': 'str',
                                  'uri': 'shex:nonLiteral'},
               'objectidentifier': {'base': 'ElementIdentifier',
                                    'comments': ['Used for inheritance and type '
                                                 'checking'],
                                    'description': 'A URI or CURIE that represents '
                                                   'an object in the model.',
                                    'from_schema': 'https://w3id.org/nmdc/nmdc',
                                    'name': 'objectidentifier',
                                    'notes': ['If you are authoring schemas in '
                                              'LinkML YAML, the type is referenced '
                                              'with the lower case '
                                              '"objectidentifier".'],
                                    'repr': 'str',
                                    'uri': 'shex:iri'},
               'sparqlpath': {'base': 'str',
                              'conforms_to': 'https://www.w3.org/TR/sparql11-query/#propertypaths',
                              'description': 'A string encoding a SPARQL Property '
                                             'Path. The value of the string MUST '
                                             'conform to SPARQL syntax and SHOULD '
                                             'dereference to zero or more valid '
                                             'objects within the current instance '
                                             'document when encoded as RDF.',
                              'from_schema': 'https://w3id.org/nmdc/nmdc',
                              'name': 'sparqlpath',
                              'notes': ['If you are authoring schemas in LinkML '
                                        'YAML, the type is referenced with the '
                                        'lower case "sparqlpath".'],
                              'repr': 'str',
                              'uri': 'xsd:string'},
               'string': {'base': 'str',
                          'description': 'A character string',
                          'exact_mappings': ['schema:Text'],
                          'from_schema': 'https://w3id.org/nmdc/nmdc',
                          'name': 'string',
                          'notes': ['In RDF serializations, a slot with range of '
                                    'string is treated as a literal or type '
                                    'xsd:string.   If you are authoring schemas in '
                                    'LinkML YAML, the type is referenced with the '
                                    'lower case "string".'],
                          'uri': 'xsd:string'},
               'time': {'base': 'XSDTime',
                        'description': 'A time object represents a (local) time of '
                                       'day, independent of any particular day',
                        'exact_mappings': ['schema:Time'],
                        'from_schema': 'https://w3id.org/nmdc/nmdc',
                        'name': 'time',
                        'notes': ['URI is dateTime because OWL reasoners do not '
                                  'work with straight date or time',
                                  'If you are authoring schemas in LinkML YAML, '
                                  'the type is referenced with the lower case '
                                  '"time".'],
                        'repr': 'str',
                        'uri': 'xsd:time'},
               'unit': {'base': 'str',
                        'from_schema': 'https://w3id.org/nmdc/nmdc',
                        'mappings': ['qud:Unit', 'UO:0000000'],
                        'name': 'unit',
                        'uri': 'xsd:string'},
               'uri': {'base': 'URI',
                       'close_mappings': ['schema:URL'],
                       'comments': ['in RDF serializations a slot with range of '
                                    'uri is treated as a literal or type '
                                    'xsd:anyURI unless it is an identifier or a '
                                    'reference to an identifier, in which case it '
                                    'is translated directly to a node'],
                       'conforms_to': 'https://www.ietf.org/rfc/rfc3987.txt',
                       'description': 'a complete URI',
                       'from_schema': 'https://w3id.org/nmdc/nmdc',
                       'name': 'uri',
                       'notes': ['If you are authoring schemas in LinkML YAML, the '
                                 'type is referenced with the lower case "uri".'],
                       'repr': 'str',
                       'uri': 'xsd:anyURI'},
               'uriorcurie': {'base': 'URIorCURIE',
                              'description': 'a URI or a CURIE',
                              'from_schema': 'https://w3id.org/nmdc/nmdc',
                              'name': 'uriorcurie',
                              'notes': ['If you are authoring schemas in LinkML '
                                        'YAML, the type is referenced with the '
                                        'lower case "uriorcurie".'],
                              'repr': 'str',
                              'uri': 'xsd:anyURI'}}} )

class CalibrationTargetEnum(str, Enum):
    mSOLIDUSz = "mass_charge_ratio"
    retention_time = "retention_time"
    retention_index = "retention_index"


class CalibrationStandardEnum(str, Enum):
    fames = "fames"
    alkanes = "alkanes"


class StrandedOrientationEnum(str, Enum):
    """
    This enumeration specifies information about stranded RNA library preparations.
    """
    antisense_orientation = "antisense orientation"
    """
    Orientation that is complementary (non-coding) to a sequence of messenger RNA.
    """
    sense_orientation = "sense orientation"
    """
    Orientation that corresponds to the coding sequence of messenger RNA.
    """


class MassSpectrometryAcquisitionStrategyEnum(str, Enum):
    data_independent_acquisition = "data_independent_acquisition"
    """
    ['Data independent mass spectrometer acquisition method wherein the full mass range is fragmented. Examples of such an approach include MS^E, AIF, and bbCID.']
    """
    data_dependent_acquisition = "data_dependent_acquisition"
    """
    Mass spectrometer data acquisition method wherein MSn spectra are triggered based on the m/z of precursor ions detected in the same run.
    """
    full_scan_only = "full_scan_only"
    """
    Mass spectrometer data acquisition method wherein only MS1 data are acquired.
    """


class ResolutionCategoryEnum(str, Enum):
    high = "high"
    """
    higher than unit resolution
    """
    low = "low"
    """
    at unit resolution
    """


class MassAnalyzerEnum(str, Enum):
    time_of_flight = "time_of_flight"
    """
    Instrument that separates ions by m/z in a field-free region after acceleration to a fixed acceleration energy.
    """
    quadrupole = "quadrupole"
    """
    A mass spectrometer that consists of four parallel rods whose centers form the corners of a square and whose opposing poles are connected. The voltage applied to the rods is a superposition of a static potential and a sinusoidal radio frequency potential. The motion of an ion in the x and y dimensions is described by the Matthieu equation whose solutions show that ions in a particular m/z range can be transmitted along the z axis.
    """
    Orbitrap = "Orbitrap"
    """
    An ion trapping device that consists of an outer barrel-like electrode and a coaxial inner spindle-like electrode that form an electrostatic field with quadro-logarithmic potential distribution. The frequency of harmonic oscillations of the orbitally trapped ions along the axis of the electrostatic field is independent of the ion velocity and is inversely proportional to the square root of m/z so that the trap can be used as a mass analyzer.
    """
    ion_cyclotron_resonance = "ion_cyclotron_resonance"
    """
    A mass spectrometer based on the principle of ion cyclotron resonance in which an ion in a magnetic field moves in a circular orbit at a frequency characteristic of its m/z value. Ions are coherently excited to a larger radius orbit using a pulse of radio frequency energy and their image charge is detected on receiver plates as a time domain signal. Fourier transformation of the time domain signal results in a frequency domain signal which is converted to a mass spectrum based in the inverse relationship between frequency and m/z.
    """
    ion_trap = "ion_trap"
    """
    A device for spatially confining ions using electric and magnetic fields alone or in combination.
    """


class IonizationSourceEnum(str, Enum):
    electrospray_ionization = "electrospray_ionization"
    matrix_assisted_laser_desorption_ionization = "matrix_assisted_laser_desorption_ionization"
    atmospheric_pressure_photo_ionization = "atmospheric_pressure_photo_ionization"
    atmospheric_pressure_chemical_ionization = "atmospheric_pressure_chemical_ionization"
    electron_ionization = "electron_ionization"


class MassSpectrumCollectionModeEnum(str, Enum):
    full_profile = "full_profile"
    reduced_profile = "reduced_profile"
    centroid = "centroid"


class PolarityModeEnum(str, Enum):
    positive = "positive"
    negative = "negative"


class EluentIntroductionCategoryEnum(str, Enum):
    liquid_chromatography = "liquid_chromatography"
    """
    The processed sample is introduced into the mass spectrometer through a liquid chromatography process.
    """
    gas_chromatography = "gas_chromatography"
    """
    The processed sample is introduced into the mass spectrometer through a gas chromatography process.
    """
    direct_infusion_syringe = "direct_infusion_syringe"
    """
    The processed sample is introduced into the mass spectrometer through a direct infusion process using a syringe.
    """
    direct_infusion_autosampler = "direct_infusion_autosampler"
    """
    The processed sample is introduced into the mass spectrometer through a direct infusion process using an autosampler.
    """


class LibraryTypeEnum(str, Enum):
    DNA = "DNA"
    RNA = "RNA"


class ContainerCategoryEnum(str, Enum):
    """
    The permitted types of containers used in processing metabolomic samples.
    """
    v_bottom_conical_tube = "v-bottom_conical_tube"
    falcon_tube = "falcon_tube"


class SeparationMethodEnum(str, Enum):
    """
    The tool/substance used to separate or filter a solution or mixture.
    """
    ptfe_96_well_filter_plate = "ptfe_96_well_filter_plate"
    syringe = "syringe"


class StationaryPhaseEnum(str, Enum):
    """
    The type of stationary phase used in a chromatography process.
    """
    BEH_HILIC = "BEH-HILIC"
    """
    Hydrophilic Interaction Chromatography (HILIC) employing BEH (Bridged Ethylene Hybrid) particles as the stationary phase.
    """
    C18 = "C18"
    """
    A stationary phase consisting of octadecyl chains (C18) bonded to silica particles.
    """
    C8 = "C8"
    """
    A stationary phase consisting of octyl chains (C8) bonded to silica particles.
    """
    C4 = "C4"
    """
    A stationary phase consisting of butyl chains (C4) bonded to silica particles.
    """
    C2 = "C2"
    """
    A stationary phase consisting of ethyl chains (C2) bonded to silica particles.
    """
    C1 = "C1"
    """
    A stationary phase consisting of methyl chains (C1) bonded to silica particles.
    """
    C30 = "C30"
    """
    A stationary phase consisting of triacontyl chains (C30) bonded to silica particles.
    """
    C60 = "C60"
    """
    A stationary phase consisting of hexatriacontyl chains (C60) bonded to silica particles.
    """
    CNT = "CNT"
    """
    Carbon Nanotube stationary phase.
    """
    CN = "CN"
    """
    Cyano (CN) bonded stationary phase.
    """
    Diol = "Diol"
    """
    A stationary phase with diol (1,2-diol) functional groups.
    """
    HILIC = "HILIC"
    """
    Hydrophilic Interaction Chromatography (HILIC) stationary phase.
    """
    NH2 = "NH2"
    """
    Amino (NH2) bonded stationary phase.
    """
    Phenyl = "Phenyl"
    """
    Phenyl bonded stationary phase.
    """
    Polysiloxane = "Polysiloxane"
    """
    A stationary phase made of polysiloxane, usually used in gas chromatography.
    """
    PS_DVB = "PS-DVB"
    """
    Polystyrene-divinylbenzene stationary phase, often used in solid-phase extraction, including proprietary Priority PolLutant (PPL).
    """
    SAX = "SAX"
    """
    Strong Anion Exchange (SAX) stationary phase.
    """
    SCX = "SCX"
    """
    Strong Cation Exchange (SCX) stationary phase.
    """
    Silica = "Silica"
    """
    A stationary phase made of silica, commonly used in chromatography.
    """
    WCX = "WCX"
    """
    Weak Cation Exchange (WCX) stationary phase.
    """
    WAX = "WAX"
    """
    Weak Anion Exchange (WAX) stationary phase.
    """
    ZIC_HILIC = "ZIC-HILIC"
    """
    Zwitterionic Hydrophilic Interaction Chromatography (ZIC-HILIC) stationary phase.
    """
    ZIC_pHILIC = "ZIC-pHILIC"
    """
    Zwitterionic pH-Responsive Hydrophilic Interaction Chromatography (ZIC-pHILIC) stationary phase.
    """
    ZIC_cHILIC = "ZIC-cHILIC"
    """
    Zwitterionic Charged Hydrophilic Interaction Chromatography (ZIC-cHILIC) stationary phase.
    """


class ProtocolCategoryEnum(str, Enum):
    """
    The possible protocols that may be followed for an assay.
    """
    mplex = "mplex"
    derivatization = "derivatization"
    filter_clean_up = "filter_clean_up"
    organic_matter_extraction = "organic_matter_extraction"
    solid_phase_extraction = "solid_phase_extraction"
    phosphorus_extraction = "phosphorus_extraction"
    ph_measurement = "ph_measurement"
    respiration_measurement = "respiration_measurement"
    texture_measurement = "texture_measurement"
    dna_extraction = "dna_extraction"
    phenol_chloroform_extraction = "phenol_chloroform_extraction"


class ChromatographicCategoryEnum(str, Enum):
    liquid_chromatography = "liquid_chromatography"
    gas_chromatography = "gas_chromatography"
    solid_phase_extraction = "solid_phase_extraction"


class SamplePortionEnum(str, Enum):
    supernatant = "supernatant"
    pellet = "pellet"
    Organic_layer = "organic_layer"
    """
    The portion of a mixture containing dissolved organic material
    """
    Aqueous_layer = "aqueous_layer"
    """
    The portion of a mixture containing molecules dissolved in water
    """
    Interlayer = "interlayer"
    """
    The layer of material between liquid layers of a separated mixture
    """
    Chloroform_layer = "chloroform_layer"
    """
    The portion of a mixture containing molecules dissolved in chloroform
    """
    Methanol_layer = "methanol_layer"
    """
    The portion of a mixture containing molecules dissolved in methanol
    """


class BinQualityEnum(str, Enum):
    HQ = "HQ"
    """
    Metagenome-assembled genome is high quality based on MIMAG standards (https://doi.org/10.1038/nbt.3893)
    """
    MQ = "MQ"
    """
    Metagenome-assembled genome is medium quality based on MIMAG standards (https://doi.org/10.1038/nbt.3893)
    """
    LQ = "LQ"
    """
    Metagenome-assembled genome is low quality based on MIMAG standards (https://doi.org/10.1038/nbt.3893)
    """


class ChemicalConversionCategoryEnum(str, Enum):
    addition = "addition"
    substitution = "substitution"
    acid_base = "acid_base"
    reduction_oxidation = "reduction_oxidation"
    combustion = "combustion"
    decomposition = "decomposition"
    protease_cleavage = "protease_cleavage"
    """
    an enzymatic cleavage which relies on an enzyme with protease activity to act on proteins and to produce polypeptides (protein fragments).
    """


class BiosampleCategoryEnum(str, Enum):
    """
    Funding-based, sample location-based, or experimental method-based defined categories
    """
    National_Science_FoundationAPOSTROPHEs_Long_Term_Ecological_Research_Network = "LTER"
    SIP = "SIP"
    Department_of_Energy_Office_of_Science_Biological_and_Environmental_Research_Program_Laboratory_Science_Focus_Areas = "SFA"
    """
    Science Focus Area projects funded through the Department of Energy Office of Science Biological and Environmental Research Program
    """
    Facilities_Integrating_Collaborations_for_User_Science = "FICUS"
    National_Science_FoundationAPOSTROPHEs_National_Ecological_Observatory_Network = "NEON"
    Bioenergy_Research_Centers = "BRC"
    """
    Bioenergy Research Centers funded by the Biological Systems Science Division of the U.S. Department of Energy's Biological and Environmental Research Program.
    """


class SubstanceRoleEnum(str, Enum):
    buffer = "buffer"
    """
    Maintains the pH of the solution within a specific range to stabilize analytes or reactions.
    """
    acid = "acid"
    """
    Donates a proton or accepts an electron pair in a chemical reaction.
    """
    base = "base"
    """
    Accepts a proton or donates an electron pair in a chemical reaction.
    """
    ms_proteolytic_enzyme = "ms_proteolytic_enzyme"
    """
    Enzyme that catalyzes the hydrolysis of proteins and is used in mass spectrometry based proteomics
    """
    solvent = "solvent"
    """
    Dissolves the sample or reagents to facilitate reactions or extraction.
    """
    surfactant = "surfactant"
    """
    Reduces surface tension and aids in the solubilization of substances.
    """
    derivatizing_agent = "derivatizing_agent"
    """
    Chemically modifies analytes to improve detection or separation.
    """
    solubilizing_agent = "solubilizing_agent"


class SampleStateEnum(str, Enum):
    solid = "solid"
    liquid = "liquid"
    gas = "gas"


class ChemicalEntityEnum(str, Enum):
    acetonitrile = "acetonitrile"
    acetic_acid = "acetic_acid"
    alphaLP = "alphaLP"
    ammonium_acetate = "ammonium_acetate"
    ammonium_bicarbonate = "ammonium_bicarbonate"
    Arg_C = "Arg-C"
    Asp_N = "Asp-N"
    chloroform = "chloroform"
    chymotrypsin = "chymotrypsin"
    formic_acid = "formic_acid"
    glucose = "glucose"
    Glu_C = "Glu-C"
    hydrochloric_acid = "hydrochloric_acid"
    isopropyl_alcohol = "isopropyl_alcohol"
    Lys_C = "Lys-C"
    Lys_N = "Lys-N"
    methanol = "methanol"
    medronic_acid = "medronic_acid"
    phosphoric_acid = "phosphoric_acid"
    trypsin = "trypsin"
    water = "water"


class ArchStrucEnum(str, Enum):
    building = "building"
    shed = "shed"
    home = "home"


class BiolStatEnum(str, Enum):
    wild = "wild"
    natural = "natural"
    semi_natural = "semi-natural"
    inbred_line = "inbred line"
    breederAPOSTROPHEs_line = "breeder's line"
    hybrid = "hybrid"
    clonal_selection = "clonal selection"
    mutant = "mutant"


class BioticRelationshipEnum(str, Enum):
    free_living = "free living"
    parasite = "parasite"
    commensal = "commensal"
    symbiont = "symbiont"


class BuildDocsEnum(str, Enum):
    building_information_model = "building information model"
    commissioning_report = "commissioning report"
    complaint_logs = "complaint logs"
    contract_administration = "contract administration"
    cost_estimate = "cost estimate"
    janitorial_schedules_or_logs = "janitorial schedules or logs"
    maintenance_plans = "maintenance plans"
    schedule = "schedule"
    sections = "sections"
    shop_drawings = "shop drawings"
    submittals = "submittals"
    ventilation_system = "ventilation system"
    windows = "windows"


class BuildOccupTypeEnum(str, Enum):
    office = "office"
    market = "market"
    restaurant = "restaurant"
    residence = "residence"
    school = "school"
    residential = "residential"
    commercial = "commercial"
    low_rise = "low rise"
    high_rise = "high rise"
    wood_framed = "wood framed"
    health_care = "health care"
    airport = "airport"
    sports_complex = "sports complex"


class BuildingSettingEnum(str, Enum):
    urban = "urban"
    suburban = "suburban"
    exurban = "exurban"
    rural = "rural"


class CeilCondEnum(str, Enum):
    new = "new"
    visible_wear = "visible wear"
    needs_repair = "needs repair"
    damaged = "damaged"
    rupture = "rupture"


class CeilFinishMatEnum(str, Enum):
    drywall = "drywall"
    mineral_fibre = "mineral fibre"
    tiles = "tiles"
    PVC = "PVC"
    plasterboard = "plasterboard"
    metal = "metal"
    fiberglass = "fiberglass"
    stucco = "stucco"
    mineral_woolSOLIDUScalcium_silicate = "mineral wool/calcium silicate"
    wood = "wood"


class CeilTextureEnum(str, Enum):
    crows_feet = "crows feet"
    crows_foot_stomp = "crows-foot stomp"
    double_skip = "double skip"
    hawk_and_trowel = "hawk and trowel"
    knockdown = "knockdown"
    popcorn = "popcorn"
    orange_peel = "orange peel"
    rosebud_stomp = "rosebud stomp"
    Santa_Fe_texture = "Santa-Fe texture"
    skip_trowel = "skip trowel"
    smooth = "smooth"
    stomp_knockdown = "stomp knockdown"
    swirl = "swirl"


class CeilTypeEnum(str, Enum):
    cathedral = "cathedral"
    dropped = "dropped"
    concave = "concave"
    barrel_shaped = "barrel-shaped"
    coffered = "coffered"
    cove = "cove"
    stretched = "stretched"


class CurLandUseEnum(str, Enum):
    badlands = "badlands"
    cities = "cities"
    conifers = "conifers"
    crop_trees = "crop trees"
    farmstead = "farmstead"
    gravel = "gravel"
    hardwoods = "hardwoods"
    hayland = "hayland"
    horticultural_plants = "horticultural plants"
    industrial_areas = "industrial areas"
    intermixed_hardwood_and_conifers = "intermixed hardwood and conifers"
    marshlands = "marshlands"
    meadows = "meadows"
    minesSOLIDUSquarries = "mines/quarries"
    mudflats = "mudflats"
    oil_waste_areas = "oil waste areas"
    pastureland = "pastureland"
    permanent_snow_or_ice = "permanent snow or ice"
    rainforest = "rainforest"
    rangeland = "rangeland"
    roadsSOLIDUSrailroads = "roads/railroads"
    rock = "rock"
    row_crops = "row crops"
    saline_seeps = "saline seeps"
    salt_flats = "salt flats"
    sand = "sand"
    shrub_crops = "shrub crops"
    shrub_land = "shrub land"
    small_grains = "small grains"
    successional_shrub_land = "successional shrub land"
    swamp = "swamp"
    tropical = "tropical"
    tundra = "tundra"
    vegetable_crops = "vegetable crops"
    vine_crops = "vine crops"


class DeposEnvEnum(str, Enum):
    Continental___Alluvial = "Continental - Alluvial"
    Continental___Aeolian = "Continental - Aeolian"
    Continental___Fluvial = "Continental - Fluvial"
    Continental___Lacustrine = "Continental - Lacustrine"
    Transitional___Deltaic = "Transitional - Deltaic"
    Transitional___Tidal = "Transitional - Tidal"
    Transitional___Lagoonal = "Transitional - Lagoonal"
    Transitional___Beach = "Transitional - Beach"
    Transitional___Lake = "Transitional - Lake"
    Marine___Shallow = "Marine - Shallow"
    Marine___Deep = "Marine - Deep"
    Marine___Reef = "Marine - Reef"
    Other___Evaporite = "Other - Evaporite"
    Other___Glacial = "Other - Glacial"
    Other___Volcanic = "Other - Volcanic"
    other = "other"


class DoorCompTypeEnum(str, Enum):
    metal_covered = "metal covered"
    revolving = "revolving"
    sliding = "sliding"
    telescopic = "telescopic"


class DoorCondEnum(str, Enum):
    damaged = "damaged"
    needs_repair = "needs repair"
    new = "new"
    rupture = "rupture"
    visible_wear = "visible wear"


class DoorDirectEnum(str, Enum):
    inward = "inward"
    outward = "outward"
    sideways = "sideways"


class DoorLocEnum(str, Enum):
    north = "north"
    south = "south"
    east = "east"
    west = "west"


class DoorMatEnum(str, Enum):
    aluminum = "aluminum"
    cellular_PVC = "cellular PVC"
    engineered_plastic = "engineered plastic"
    fiberboard = "fiberboard"
    fiberglass = "fiberglass"
    metal = "metal"
    thermoplastic_alloy = "thermoplastic alloy"
    vinyl = "vinyl"
    wood = "wood"
    woodSOLIDUSplastic_composite = "wood/plastic composite"


class DoorMoveEnum(str, Enum):
    collapsible = "collapsible"
    folding = "folding"
    revolving = "revolving"
    rolling_shutter = "rolling shutter"
    sliding = "sliding"
    swinging = "swinging"


class DoorTypeEnum(str, Enum):
    composite = "composite"
    metal = "metal"
    wooden = "wooden"


class DoorTypeMetalEnum(str, Enum):
    collapsible = "collapsible"
    corrugated_steel = "corrugated steel"
    hollow = "hollow"
    rolling_shutters = "rolling shutters"
    steel_plate = "steel plate"


class DoorTypeWoodEnum(str, Enum):
    bettened_and_ledged = "bettened and ledged"
    battened = "battened"
    ledged_and_braced = "ledged and braced"
    ledged_and_framed = "ledged and framed"
    ledged_braced_and_frame = "ledged, braced and frame"
    framed_and_paneled = "framed and paneled"
    glashed_or_sash = "glashed or sash"
    flush = "flush"
    louvered = "louvered"
    wire_gauged = "wire gauged"


class DrainageClassEnum(str, Enum):
    very_poorly = "very poorly"
    poorly = "poorly"
    somewhat_poorly = "somewhat poorly"
    moderately_well = "moderately well"
    well = "well"
    excessively_drained = "excessively drained"


class DrawingsEnum(str, Enum):
    operation = "operation"
    as_built = "as built"
    construction = "construction"
    bid = "bid"
    design = "design"
    building_navigation_map = "building navigation map"
    diagram = "diagram"
    sketch = "sketch"


class ExtWallOrientEnum(str, Enum):
    north = "north"
    south = "south"
    east = "east"
    west = "west"
    northeast = "northeast"
    southeast = "southeast"
    southwest = "southwest"
    northwest = "northwest"


class ExtWindowOrientEnum(str, Enum):
    north = "north"
    south = "south"
    east = "east"
    west = "west"
    northeast = "northeast"
    southeast = "southeast"
    southwest = "southwest"
    northwest = "northwest"


class FaoClassEnum(str, Enum):
    Acrisols = "Acrisols"
    Andosols = "Andosols"
    Arenosols = "Arenosols"
    Cambisols = "Cambisols"
    Chernozems = "Chernozems"
    Ferralsols = "Ferralsols"
    Fluvisols = "Fluvisols"
    Gleysols = "Gleysols"
    Greyzems = "Greyzems"
    Gypsisols = "Gypsisols"
    Histosols = "Histosols"
    Kastanozems = "Kastanozems"
    Lithosols = "Lithosols"
    Luvisols = "Luvisols"
    Nitosols = "Nitosols"
    Phaeozems = "Phaeozems"
    Planosols = "Planosols"
    Podzols = "Podzols"
    Podzoluvisols = "Podzoluvisols"
    Rankers = "Rankers"
    Regosols = "Regosols"
    Rendzinas = "Rendzinas"
    Solonchaks = "Solonchaks"
    Solonetz = "Solonetz"
    Vertisols = "Vertisols"
    Yermosols = "Yermosols"


class FilterTypeEnum(str, Enum):
    particulate_air_filter = "particulate air filter"
    chemical_air_filter = "chemical air filter"
    low_MERV_pleated_media = "low-MERV pleated media"
    HEPA = "HEPA"
    electrostatic = "electrostatic"
    gas_phase_or_ultraviolet_air_treatments = "gas-phase or ultraviolet air treatments"


class FloorCondEnum(str, Enum):
    new = "new"
    visible_wear = "visible wear"
    needs_repair = "needs repair"
    damaged = "damaged"
    rupture = "rupture"


class FloorFinishMatEnum(str, Enum):
    tile = "tile"
    wood_strip_or_parquet = "wood strip or parquet"
    carpet = "carpet"
    rug = "rug"
    laminate_wood = "laminate wood"
    lineoleum = "lineoleum"
    vinyl_composition_tile = "vinyl composition tile"
    sheet_vinyl = "sheet vinyl"
    stone = "stone"
    bamboo = "bamboo"
    cork = "cork"
    terrazo = "terrazo"
    concrete = "concrete"
    none = "none"
    sealed = "sealed"
    clear_finish = "clear finish"
    paint = "paint"
    none_or_unfinished = "none or unfinished"


class FloorStrucEnum(str, Enum):
    balcony = "balcony"
    floating_floor = "floating floor"
    glass_floor = "glass floor"
    raised_floor = "raised floor"
    sprung_floor = "sprung floor"
    wood_framed = "wood-framed"
    concrete = "concrete"


class FloorWaterMoldEnum(str, Enum):
    mold_odor = "mold odor"
    wet_floor = "wet floor"
    water_stains = "water stains"
    wall_discoloration = "wall discoloration"
    floor_discoloration = "floor discoloration"
    ceiling_discoloration = "ceiling discoloration"
    peeling_paint_or_wallpaper = "peeling paint or wallpaper"
    bulging_walls = "bulging walls"
    condensation = "condensation"


class FreqCleanEnum(str, Enum):
    Daily = "Daily"
    Weekly = "Weekly"
    Monthly = "Monthly"
    Quarterly = "Quarterly"
    Annually = "Annually"
    other = "other"


class FurnitureEnum(str, Enum):
    cabinet = "cabinet"
    chair = "chair"
    desks = "desks"


class GenderRestroomEnum(str, Enum):
    all_gender = "all gender"
    female = "female"
    gender_neurtral = "gender neurtral"
    male = "male"
    male_and_female = "male and female"
    unisex = "unisex"


class GrowthHabitEnum(str, Enum):
    erect = "erect"
    semi_erect = "semi-erect"
    spreading = "spreading"
    prostrate = "prostrate"


class HandidnessEnum(str, Enum):
    ambidexterity = "ambidexterity"
    left_handedness = "left handedness"
    mixed_handedness = "mixed-handedness"
    right_handedness = "right handedness"


class HcProducedEnum(str, Enum):
    Oil = "Oil"
    Gas_Condensate = "Gas-Condensate"
    Gas = "Gas"
    Bitumen = "Bitumen"
    Coalbed_Methane = "Coalbed Methane"
    other = "other"


class HcrEnum(str, Enum):
    Oil_Reservoir = "Oil Reservoir"
    Gas_Reservoir = "Gas Reservoir"
    Oil_Sand = "Oil Sand"
    Coalbed = "Coalbed"
    Shale = "Shale"
    Tight_Oil_Reservoir = "Tight Oil Reservoir"
    Tight_Gas_Reservoir = "Tight Gas Reservoir"
    other = "other"


class HcrGeolAgeEnum(str, Enum):
    Archean = "Archean"
    Cambrian = "Cambrian"
    Carboniferous = "Carboniferous"
    Cenozoic = "Cenozoic"
    Cretaceous = "Cretaceous"
    Devonian = "Devonian"
    Jurassic = "Jurassic"
    Mesozoic = "Mesozoic"
    Neogene = "Neogene"
    Ordovician = "Ordovician"
    Paleogene = "Paleogene"
    Paleozoic = "Paleozoic"
    Permian = "Permian"
    Precambrian = "Precambrian"
    Proterozoic = "Proterozoic"
    Silurian = "Silurian"
    Triassic = "Triassic"
    other = "other"


class HeatCoolTypeEnum(str, Enum):
    radiant_system = "radiant system"
    heat_pump = "heat pump"
    forced_air_system = "forced air system"
    steam_forced_heat = "steam forced heat"
    wood_stove = "wood stove"


class HeatDelivLocEnum(str, Enum):
    north = "north"
    south = "south"
    east = "east"
    west = "west"


class HostSexEnum(str, Enum):
    female = "female"
    hermaphrodite = "hermaphrodite"
    non_binary = "non-binary"
    male = "male"
    transgender = "transgender"
    transgender_LEFT_PARENTHESISfemale_to_maleRIGHT_PARENTHESIS = "transgender (female to male)"
    transgender_LEFT_PARENTHESISmale_to_femaleRIGHT_PARENTHESIS = "transgender (male to female)"
    undeclared = "undeclared"


class IndoorSpaceEnum(str, Enum):
    bedroom = "bedroom"
    office = "office"
    bathroom = "bathroom"
    foyer = "foyer"
    kitchen = "kitchen"
    locker_room = "locker room"
    hallway = "hallway"
    elevator = "elevator"


class IndoorSurfEnum(str, Enum):
    cabinet = "cabinet"
    ceiling = "ceiling"
    counter_top = "counter top"
    door = "door"
    shelving = "shelving"
    vent_cover = "vent cover"
    window = "window"
    wall = "wall"


class IntWallCondEnum(str, Enum):
    new = "new"
    visible_wear = "visible wear"
    needs_repair = "needs repair"
    damaged = "damaged"
    rupture = "rupture"


class LightTypeEnum(str, Enum):
    natural_light = "natural light"
    electric_light = "electric light"
    desk_lamp = "desk lamp"
    flourescent_lights = "flourescent lights"
    none = "none"


class LithologyEnum(str, Enum):
    Basement = "Basement"
    Chalk = "Chalk"
    Chert = "Chert"
    Coal = "Coal"
    Conglomerate = "Conglomerate"
    Diatomite = "Diatomite"
    Dolomite = "Dolomite"
    Limestone = "Limestone"
    Sandstone = "Sandstone"
    Shale = "Shale"
    Siltstone = "Siltstone"
    Volcanic = "Volcanic"
    other = "other"


class MechStrucEnum(str, Enum):
    subway = "subway"
    coach = "coach"
    carriage = "carriage"
    elevator = "elevator"
    escalator = "escalator"
    boat = "boat"
    train = "train"
    car = "car"
    bus = "bus"


class OccupDocumentEnum(str, Enum):
    automated_count = "automated count"
    estimate = "estimate"
    manual_count = "manual count"
    videos = "videos"


class OrganismCountEnum(str, Enum):
    ATP = "ATP"
    MPN = "MPN"
    other = "other"


class OxyStatSampEnum(str, Enum):
    aerobic = "aerobic"
    anaerobic = "anaerobic"
    other = "other"


class PlantGrowthMedEnum(str, Enum):
    other_artificial_liquid_medium = "other artificial liquid medium"
    other_artificial_solid_medium = "other artificial solid medium"
    peat_moss = "peat moss"
    perlite = "perlite"
    pumice = "pumice"
    sand = "sand"
    soil = "soil"
    vermiculite = "vermiculite"
    water = "water"


class PlantSexEnum(str, Enum):
    Androdioecious = "Androdioecious"
    Androecious = "Androecious"
    Androgynous = "Androgynous"
    Androgynomonoecious = "Androgynomonoecious"
    Andromonoecious = "Andromonoecious"
    Bisexual = "Bisexual"
    Dichogamous = "Dichogamous"
    Diclinous = "Diclinous"
    Dioecious = "Dioecious"
    Gynodioecious = "Gynodioecious"
    Gynoecious = "Gynoecious"
    Gynomonoecious = "Gynomonoecious"
    Hermaphroditic = "Hermaphroditic"
    Imperfect = "Imperfect"
    Monoclinous = "Monoclinous"
    Monoecious = "Monoecious"
    Perfect = "Perfect"
    Polygamodioecious = "Polygamodioecious"
    Polygamomonoecious = "Polygamomonoecious"
    Polygamous = "Polygamous"
    Protandrous = "Protandrous"
    Protogynous = "Protogynous"
    Subandroecious = "Subandroecious"
    Subdioecious = "Subdioecious"
    Subgynoecious = "Subgynoecious"
    Synoecious = "Synoecious"
    Trimonoecious = "Trimonoecious"
    Trioecious = "Trioecious"
    Unisexual = "Unisexual"


class ProfilePositionEnum(str, Enum):
    summit = "summit"
    shoulder = "shoulder"
    backslope = "backslope"
    footslope = "footslope"
    toeslope = "toeslope"


class QuadPosEnum(str, Enum):
    North_side = "North side"
    West_side = "West side"
    South_side = "South side"
    East_side = "East side"


class RelSampLocEnum(str, Enum):
    edge_of_car = "edge of car"
    center_of_car = "center of car"
    under_a_seat = "under a seat"


class RoomCondtEnum(str, Enum):
    new = "new"
    visible_wear = "visible wear"
    needs_repair = "needs repair"
    damaged = "damaged"
    rupture = "rupture"
    visible_signs_of_moldSOLIDUSmildew = "visible signs of mold/mildew"


class RoomConnectedEnum(str, Enum):
    attic = "attic"
    bathroom = "bathroom"
    closet = "closet"
    conference_room = "conference room"
    elevator = "elevator"
    examining_room = "examining room"
    hallway = "hallway"
    kitchen = "kitchen"
    mail_room = "mail room"
    office = "office"
    stairwell = "stairwell"


class RoomLocEnum(str, Enum):
    corner_room = "corner room"
    interior_room = "interior room"
    exterior_wall = "exterior wall"


class RoomSampPosEnum(str, Enum):
    north_corner = "north corner"
    south_corner = "south corner"
    west_corner = "west corner"
    east_corner = "east corner"
    northeast_corner = "northeast corner"
    northwest_corner = "northwest corner"
    southeast_corner = "southeast corner"
    southwest_corner = "southwest corner"
    center = "center"


class RoomTypeEnum(str, Enum):
    attic = "attic"
    bathroom = "bathroom"
    closet = "closet"
    conference_room = "conference room"
    elevator = "elevator"
    examining_room = "examining room"
    hallway = "hallway"
    kitchen = "kitchen"
    mail_room = "mail room"
    private_office = "private office"
    open_office = "open office"
    stairwell = "stairwell"
    restroom = ",restroom"
    lobby = "lobby"
    vestibule = "vestibule"
    mechanical_or_electrical_room = "mechanical or electrical room"
    data_center = "data center"
    laboratory_wet = "laboratory_wet"
    laboratory_dry = "laboratory_dry"
    gymnasium = "gymnasium"
    natatorium = "natatorium"
    auditorium = "auditorium"
    lockers = "lockers"
    cafe = "cafe"
    warehouse = "warehouse"


class SampCaptStatusEnum(str, Enum):
    active_surveillance_in_response_to_an_outbreak = "active surveillance in response to an outbreak"
    active_surveillance_not_initiated_by_an_outbreak = "active surveillance not initiated by an outbreak"
    farm_sample = "farm sample"
    market_sample = "market sample"
    other = "other"


class SampCollectPointEnum(str, Enum):
    well = "well"
    test_well = "test well"
    drilling_rig = "drilling rig"
    wellhead = "wellhead"
    separator = "separator"
    storage_tank = "storage tank"
    other = "other"


class SampDisStageEnum(str, Enum):
    dissemination = "dissemination"
    growth_and_reproduction = "growth and reproduction"
    infection = "infection"
    inoculation = "inoculation"
    penetration = "penetration"
    other = "other"


class SampFloorEnum(str, Enum):
    number_1st_floor = "1st floor"
    number_2nd_floor = "2nd floor"
    basement = "basement"
    lobby = "lobby"


class SampMdEnum(str, Enum):
    DF = "DF"
    RT = "RT"
    KB = "KB"
    MSL = "MSL"
    other = "other"


class SampSubtypeEnum(str, Enum):
    oil_phase = "oil phase"
    water_phase = "water phase"
    biofilm = "biofilm"
    not_applicable = "not applicable"
    other = "other"


class SampWeatherEnum(str, Enum):
    clear_sky = "clear sky"
    cloudy = "cloudy"
    foggy = "foggy"
    hail = "hail"
    rain = "rain"
    snow = "snow"
    sleet = "sleet"
    sunny = "sunny"
    windy = "windy"


class SeasonUseEnum(str, Enum):
    Spring = "Spring"
    Summer = "Summer"
    Fall = "Fall"
    Winter = "Winter"


class SedimentTypeEnum(str, Enum):
    biogenous = "biogenous"
    cosmogenous = "cosmogenous"
    hydrogenous = "hydrogenous"
    lithogenous = "lithogenous"


class ShadingDeviceCondEnum(str, Enum):
    damaged = "damaged"
    needs_repair = "needs repair"
    new = "new"
    rupture = "rupture"
    visible_wear = "visible wear"


class ShadingDeviceTypeEnum(str, Enum):
    bahama_shutters = "bahama shutters"
    exterior_roll_blind = "exterior roll blind"
    gambrel_awning = "gambrel awning"
    hood_awning = "hood awning"
    porchroller_awning = "porchroller awning"
    sarasota_shutters = "sarasota shutters"
    slatted_aluminum = "slatted aluminum"
    solid_aluminum_awning = "solid aluminum awning"
    sun_screen = "sun screen"
    tree = "tree"
    trellis = "trellis"
    venetian_awning = "venetian awning"


class SoilHorizonEnum(str, Enum):
    O_horizon = "O horizon"
    A_horizon = "A horizon"
    E_horizon = "E horizon"
    B_horizon = "B horizon"
    C_horizon = "C horizon"
    R_layer = "R layer"
    Permafrost = "Permafrost"
    M_horizon = "M horizon"


class SpecificEnum(str, Enum):
    operation = "operation"
    as_built = "as built"
    construction = "construction"
    bid = "bid"
    design = "design"
    photos = "photos"


class SrDepEnvEnum(str, Enum):
    Lacustine = "Lacustine"
    Fluvioldeltaic = "Fluvioldeltaic"
    Fluviomarine = "Fluviomarine"
    Marine = "Marine"
    other = "other"


class SrGeolAgeEnum(str, Enum):
    Archean = "Archean"
    Cambrian = "Cambrian"
    Carboniferous = "Carboniferous"
    Cenozoic = "Cenozoic"
    Cretaceous = "Cretaceous"
    Devonian = "Devonian"
    Jurassic = "Jurassic"
    Mesozoic = "Mesozoic"
    Neogene = "Neogene"
    Ordovician = "Ordovician"
    Paleogene = "Paleogene"
    Paleozoic = "Paleozoic"
    Permian = "Permian"
    Precambrian = "Precambrian"
    Proterozoic = "Proterozoic"
    Silurian = "Silurian"
    Triassic = "Triassic"
    other = "other"


class SrKerogTypeEnum(str, Enum):
    Type_I = "Type I"
    Type_II = "Type II"
    Type_III = "Type III"
    Type_IV = "Type IV"
    other = "other"


class SrLithologyEnum(str, Enum):
    Clastic = "Clastic"
    Carbonate = "Carbonate"
    Coal = "Coal"
    Biosilicieous = "Biosilicieous"
    other = "other"


class SubstructureTypeEnum(str, Enum):
    crawlspace = "crawlspace"
    slab_on_grade = "slab on grade"
    basement = "basement"


class SurfAirContEnum(str, Enum):
    dust = "dust"
    organic_matter = "organic matter"
    particulate_matter = "particulate matter"
    volatile_organic_compounds = "volatile organic compounds"
    biological_contaminants = "biological contaminants"
    radon = "radon"
    nutrients = "nutrients"
    biocides = "biocides"


class SurfMaterialEnum(str, Enum):
    adobe = "adobe"
    carpet = "carpet"
    cinder_blocks = "cinder blocks"
    concrete = "concrete"
    hay_bales = "hay bales"
    glass = "glass"
    metal = "metal"
    paint = "paint"
    plastic = "plastic"
    stainless_steel = "stainless steel"
    stone = "stone"
    stucco = "stucco"
    tile = "tile"
    vinyl = "vinyl"
    wood = "wood"


class TidalStageEnum(str, Enum):
    low_tide = "low tide"
    ebb_tide = "ebb tide"
    flood_tide = "flood tide"
    high_tide = "high tide"


class TillageEnum(str, Enum):
    drill = "drill"
    cutting_disc = "cutting disc"
    ridge_till = "ridge till"
    strip_tillage = "strip tillage"
    zonal_tillage = "zonal tillage"
    chisel = "chisel"
    tined = "tined"
    mouldboard = "mouldboard"
    disc_plough = "disc plough"


class TrainLineEnum(str, Enum):
    red = "red"
    green = "green"
    orange = "orange"


class TrainStatLocEnum(str, Enum):
    south_station_above_ground = "south station above ground"
    south_station_underground = "south station underground"
    south_station_amtrak = "south station amtrak"
    forest_hills = "forest hills"
    riverside = "riverside"


class TrainStopLocEnum(str, Enum):
    end = "end"
    mid = "mid"
    downtown = "downtown"


class VisMediaEnum(str, Enum):
    photos = "photos"
    videos = "videos"
    commonly_of_the_building = "commonly of the building"
    site_context_LEFT_PARENTHESISadjacent_buildings_vegetation_terrain_streetsRIGHT_PARENTHESIS = "site context (adjacent buildings, vegetation, terrain, streets)"
    interiors = "interiors"
    equipment = "equipment"
    number_3D_scans = "3D scans"


class WallConstTypeEnum(str, Enum):
    frame_construction = "frame construction"
    joisted_masonry = "joisted masonry"
    light_noncombustible = "light noncombustible"
    masonry_noncombustible = "masonry noncombustible"
    modified_fire_resistive = "modified fire resistive"
    fire_resistive = "fire resistive"


class WallFinishMatEnum(str, Enum):
    plaster = "plaster"
    gypsum_plaster = "gypsum plaster"
    veneer_plaster = "veneer plaster"
    gypsum_board = "gypsum board"
    tile = "tile"
    terrazzo = "terrazzo"
    stone_facing = "stone facing"
    acoustical_treatment = "acoustical treatment"
    wood = "wood"
    metal = "metal"
    masonry = "masonry"


class WallLocEnum(str, Enum):
    north = "north"
    south = "south"
    east = "east"
    west = "west"


class WallSurfTreatmentEnum(str, Enum):
    painted = "painted"
    wall_paper = "wall paper"
    no_treatment = "no treatment"
    paneling = "paneling"
    stucco = "stucco"
    fabric = "fabric"


class WallTextureEnum(str, Enum):
    crows_feet = "crows feet"
    crows_foot_stomp = "crows-foot stomp"
     = ""
    double_skip = "double skip"
    hawk_and_trowel = "hawk and trowel"
    knockdown = "knockdown"
    popcorn = "popcorn"
    orange_peel = "orange peel"
    rosebud_stomp = "rosebud stomp"
    Santa_Fe_texture = "Santa-Fe texture"
    skip_trowel = "skip trowel"
    smooth = "smooth"
    stomp_knockdown = "stomp knockdown"
    swirl = "swirl"


class WaterFeatTypeEnum(str, Enum):
    fountain = "fountain"
    pool = "pool"
    standing_feature = "standing feature"
    stream = "stream"
    waterfall = "waterfall"


class WeekdayEnum(str, Enum):
    Monday = "Monday"
    Tuesday = "Tuesday"
    Wednesday = "Wednesday"
    Thursday = "Thursday"
    Friday = "Friday"
    Saturday = "Saturday"
    Sunday = "Sunday"


class WindowCondEnum(str, Enum):
    damaged = "damaged"
    needs_repair = "needs repair"
    new = "new"
    rupture = "rupture"
    visible_wear = "visible wear"


class WindowCoverEnum(str, Enum):
    blinds = "blinds"
    curtains = "curtains"
    none = "none"


class WindowHorizPosEnum(str, Enum):
    left = "left"
    middle = "middle"
    right = "right"


class WindowLocEnum(str, Enum):
    north = "north"
    south = "south"
    east = "east"
    west = "west"


class WindowMatEnum(str, Enum):
    clad = "clad"
    fiberglass = "fiberglass"
    metal = "metal"
    vinyl = "vinyl"
    wood = "wood"


class WindowTypeEnum(str, Enum):
    single_hung_sash_window = "single-hung sash window"
    horizontal_sash_window = "horizontal sash window"
    fixed_window = "fixed window"


class WindowVertPosEnum(str, Enum):
    bottom = "bottom"
    middle = "middle"
    top = "top"
    low = "low"
    high = "high"


class TargetGeneEnum(str, Enum):
    number_16S_rRNA = "16S_rRNA"
    """
    the small subunit of the bacterial/archean ribosome
    """
    number_23S_rRNA = "23S_rRNA"
    """
    the large subunit  of the bacterial/archean ribosome
    """
    number_18S_rRNA = "18S_rRNA"
    """
    the small subunit of the eukaryotic ribosome
    """
    number_28S_rRNA = "28S_rRNA"
    """
    the large subunit of the eukaryotic ribosome
    """


class CreditEnum(str, Enum):
    Conceptualization = "Conceptualization"
    """
    Conceptualization
    """
    Data_curation = "Data curation"
    """
    Data curation
    """
    Formal_Analysis = "Formal Analysis"
    """
    Formal Analysis
    """
    Funding_acquisition = "Funding acquisition"
    """
    Funding acquisition
    """
    Investigation = "Investigation"
    """
    Investigation
    """
    Methodology = "Methodology"
    """
    Methodology
    """
    Project_administration = "Project administration"
    """
    Project administration
    """
    Resources = "Resources"
    """
    Resources
    """
    Software = "Software"
    """
    Software
    """
    Supervision = "Supervision"
    """
    Supervision
    """
    Validation = "Validation"
    """
    Validation
    """
    Visualization = "Visualization"
    """
    Visualization
    """
    Writing_original_draft = "Writing original draft"
    """
    Writing – original draft
    """
    Writing_review_and_editing = "Writing review and editing"
    """
    Writing – review & editing
    """
    Principal_Investigator = "Principal Investigator"
    """
    principal investigator role
    """
    Submitter = "Submitter"
    """
    the person(s) who enter study and biosample metadata into the NMDC submission portal
    """


class StudyCategoryEnum(str, Enum):
    research_study = "research_study"
    """
    A detailed examination, analysis, or critical inspection of a hypothesis-driven experiment.
    """
    consortium = "consortium"
    """
    A group formed to undertake a venture that is beyond the capabilities of the individual members. Each member of the consortium brings a high level of expertise in a specific area to ensure the successful completion of the project.
    """


class ManifestCategoryEnum(str, Enum):
    """
    A list of contexts in which some DataObjects can be analyzed together.
    """
    instrument_run = "instrument_run"
    """
    A collection of data objects from a single run of an instrument.
    """
    poolable_replicates = "poolable_replicates"
    """
    A collection of data objects that can be pooled for downstream analyses.
    """
    fractions = "fractions"
    """
    A collection of data objects that represent fractions of a single sample.
    """


class InstrumentModelEnum(str, Enum):
    exploris_21T = "exploris_21T"
    exploris_240 = "exploris_240"
    exploris_480 = "exploris_480"
    ltq_orbitrap_velos = "ltq_orbitrap_velos"
    orbitrap_fusion_lumos = "orbitrap_fusion_lumos"
    orbitrap_eclipse_tribid = "orbitrap_eclipse_tribid"
    orbitrap_q_exactive = "orbitrap_q_exactive"
    solarix_7T = "solarix_7T"
    solarix_12T = "solarix_12T"
    solarix_15T = "solarix_15T"
    agilent_8890A = "agilent_8890A"
    agilent_7980A = "agilent_7980A"
    vortex_genie_2 = "vortex_genie_2"
    novaseq = "novaseq"
    novaseq_6000 = "novaseq_6000"
    novaseq_x = "novaseq_x"
    hiseq = "hiseq"
    hiseq_1000 = "hiseq_1000"
    hiseq_1500 = "hiseq_1500"
    hiseq_2000 = "hiseq_2000"
    hiseq_2500 = "hiseq_2500"
    hiseq_3000 = "hiseq_3000"
    hiseq_4000 = "hiseq_4000"
    hiseq_x_ten = "hiseq_x_ten"
    miniseq = "miniseq"
    miseq = "miseq"
    nextseq_1000 = "nextseq_1000"
    nextseq = "nextseq"
    nextseq_500 = "nextseq_500"
    nextseq_550 = "nextseq_550"
    gridion = "gridion"
    minion = "minion"
    promethion = "promethion"
    rs_II = "rs_II"
    sequel = "sequel"
    sequel_II = "sequel_II"
    revio = "revio"


class InstrumentVendorEnum(str, Enum):
    waters = "waters"
    agilent = "agilent"
    bruker = "bruker"
    thermo_fisher = "thermo_fisher"
    vwr = "vwr"
    perkin_elmer = "perkin_elmer"
    gilson = "gilson"
    scientific_industries = "scientific_industries"
    illumina = "illumina"
    pacbio = "pacbio"
    oxford_nanopore = "oxford_nanopore"
    pnnl = "pnnl"


class FailureWhatEnum(str, Enum):
    """
    The permitted values for describing where a failure occurred during processing in the lab during analysis workflows.
    """
    low_read_count = "low_read_count"
    """
    Number of output reads is not sufficient to continue to the next analysis step.
    """
    malformed_data = "malformed_data"
    """
    Workflow failure reading input or writing the output file(s).
    """
    assembly_size_too_small = "assembly_size_too_small"
    """
    The size of the metagenome or metatranscriptome assembly is too small to proceed to the next analysis workflow.
    """
    no_valid_data_generated = "no_valid_data_generated"
    """
    A process ran but did not produce any output. Ie binning ran but did not produce any medium or high quality bins.
    """
    other = "other"
    """
    A lab process or analysis workflow has failed in a way that has not been captured by the available values yet. Please use slot 'qc_comment' to specify details.
    """


class FailureWhereEnum(str, Enum):
    """
    The permitted values for describing where in the process, either a lab or analysis workflow step, the failure occurred.
    """
    NucleotideSequencing = "NucleotideSequencing"
    """
    A failure has occurred during nucleotide sequencing, a data generation process.
    """
    MassSpectrometry = "MassSpectrometry"
    """
    A failure has occurred during mass spectrometry, a data generation process.
    """
    Pooling = "Pooling"
    """
    A failure has occurred in pooling, a lab process.
    """
    Extraction = "Extraction"
    """
    A failure has occurred in extraction, a lab process.
    """
    LibraryPreparation = "LibraryPreparation"
    """
    A failure has occurred in library preparation, a lab process.
    """
    MetagenomeAssembly = "MetagenomeAssembly"
    """
    A failure has occurred in metagenome assembly, a workflow process.
    """
    MetatranscriptomeExpressionAnalysis = "MetatranscriptomeExpressionAnalysis"
    """
    A failure has occurred in metatranscriptome expression analysis, a workflow process.
    """
    MagsAnalysis = "MagsAnalysis"
    """
    A failure has occurred in binning, a workflow process to generate metagenome-assembled genomes (MAGS).
    """
    ReadQcAnalysis = "ReadQcAnalysis"
    """
    A failure has occurred in read qc, a workflow process.
    """
    ReadBasedTaxonomyAnalysis = "ReadBasedTaxonomyAnalysis"
    """
    A failure has occurred in reads based taxonomy, a workflow process.
    """
    MetagenomeAnnotation = "MetagenomeAnnotation"
    """
    A failure has occurred in annotation, a workflow process.
    """
    MetatranscriptomeAssembly = "MetatranscriptomeAssembly"
    """
    A failure has occurred in assembly, a workflow process.
    """
    MetatranscriptomeAnnotation = "MetatranscriptomeAnnotation"
    """
    A failure has occurred in annotation, a workflow process.
    """
    MetabolomicsAnalysis = "MetabolomicsAnalysis"
    """
    A failure has occurred in analyzing metabolomics data.
    """
    MetaproteomicsAnalysis = "MetaproteomicsAnalysis"
    """
    A failure has occurred in analyzing metaproteomics data.
    """
    NomAnalysis = "NomAnalysis"
    """
    A failure has occurred in analyzing NOM data.
    """


class ExecutionResourceEnum(str, Enum):
    NERSC_Cori = "NERSC-Cori"
    """
    NERSC Cori supercomputer
    """
    NERSC_Perlmutter = "NERSC-Perlmutter"
    """
    NERSC Perlmutter supercomputer
    """
    EMSL = "EMSL"
    """
    Environmental Molecular Sciences Laboratory
    """
    EMSL_RZR = "EMSL-RZR"
    """
    Environmental Molecular Sciences Laboratory RZR cluster
    """
    JGI = "JGI"
    """
    Joint Genome Institute
    """
    LANL_B_div = "LANL-B-div"
    """
    LANL Bioscience Division
    """


class FileTypeEnum(str, Enum):
    Virus_Summary = "Virus Summary"
    """
    Tab separated file listing the viruses found by geNomad.
    """
    Plasmid_Summary = "Plasmid Summary"
    """
    Tab separated file listing the plasmids found by geNomad.
    """
    GeNomad_Aggregated_Classification = "GeNomad Aggregated Classification"
    """
    Tab separated file which combines the results from neural network-based classification and marker-based classification for virus and plasmid detection with geNomad.
    """
    Reference_Calibration_File = "Reference Calibration File"
    """
    A file that contains data used to calibrate a natural organic matter or metabolomics analysis.
    """
    Metagenome_Raw_Reads = "Metagenome Raw Reads"
    """
    Interleaved paired-end raw metagenome sequencing data
    """
    Metagenome_Raw_Read_1 = "Metagenome Raw Read 1"
    """
    Read 1 raw metagenome sequencing data, aka forward reads
    """
    Metagenome_Raw_Read_2 = "Metagenome Raw Read 2"
    """
    Read 2 raw metagenome sequencing data, aka reverse reads
    """
    Metatranscriptome_Raw_Reads = "Metatranscriptome Raw Reads"
    """
    Interleaved paired-end raw metatranscriptome sequencing data
    """
    Metatranscriptome_Raw_Read_1 = "Metatranscriptome Raw Read 1"
    """
    Read 1 raw metatranscriptome sequencing data, aka forward reads
    """
    Metatranscriptome_Raw_Read_2 = "Metatranscriptome Raw Read 2"
    """
    Read 2 raw metatranscriptome sequencing data, aka reverse reads
    """
    Direct_Infusion_FT_ICR_MS_Analysis_Results = "Direct Infusion FT-ICR MS Analysis Results"
    """
    FT-ICR MS based molecular formula assignment results table
    """
    Direct_Infusion_FT_ICR_MS_QC_Plots = "Direct Infusion FT-ICR MS QC Plots"
    """
    Quality control plots for FT-ICR MS raw data acquired by direct infusion
    """
    LC_FT_ICR_MS_Analysis_Results = "LC FT-ICR MS Analysis Results"
    """
    LC FT-ICR MS-based molecular formula assignment results tables
    """
    LC_FT_ICR_MS_QC_Plots = "LC FT-ICR MS QC Plots"
    """
    Quality control plots for FT-ICR MS raw data acquired with liquid chromatography
    """
    GC_MS_Metabolomics_Results = "GC-MS Metabolomics Results"
    """
    GC-MS-based metabolite assignment results table
    """
    LC_MS_Metabolomics_Results = "LC-MS Metabolomics Results"
    """
    LC-MS-based metabolite assignment results table
    """
    Mass_Spectrometry_Reference_Spectral_Library = "Mass Spectrometry Reference Spectral Library"
    """
    Spectral library used for mass spectrometry based metabolite identification
    """
    Metaproteomics_Workflow_Statistics = "Metaproteomics Workflow Statistics"
    """
    Aggregate workflow statistics file
    """
    Protein_Report = "Protein Report"
    """
    Filtered protein report file
    """
    Peptide_Report = "Peptide Report"
    """
    Filtered peptide report file
    """
    Unfiltered_Metaproteomics_Results = "Unfiltered Metaproteomics Results"
    """
    MSGFjobs and MASIC output file
    """
    Read_Count_and_RPKM = "Read Count and RPKM"
    """
    Annotation read count and RPKM per feature JSON
    """
    QC_non_rRNA_R2 = "QC non-rRNA R2"
    """
    QC removed rRNA reads (R2) fastq
    """
    QC_non_rRNA_R1 = "QC non-rRNA R1"
    """
    QC removed rRNA reads (R1) fastq
    """
    Metagenome_Bins = "Metagenome Bins"
    """
    Metagenome bin contigs fasta
    """
    Metagenome_HQMQ_Bins_Compression_File = "Metagenome HQMQ Bins Compression File"
    """
    Compressed file containing high quality and medium quality metagenome bins and associated files
    """
    Metagenome_LQ_Bins_Compression_File = "Metagenome LQ Bins Compression File"
    """
    Compressed file containing low quality metagenome bins and associated files
    """
    Metagenome_Bins_Info_File = "Metagenome Bins Info File"
    """
    File containing version information on the binning workflow
    """
    CheckM_Statistics = "CheckM Statistics"
    """
    CheckM statistics report
    """
    Metagenome_Bins_Heatmap = "Metagenome Bins Heatmap"
    """
    The Heatmap presents the pdf file containing the KO analysis results for metagenome bins
    """
    Metagenome_Bins_Barplot = "Metagenome Bins Barplot"
    """
    The Bar chart presents the pdf file containing the KO analysis results for metagenome bins
    """
    Metagenome_Bins_Krona_Plot = "Metagenome Bins Krona Plot"
    """
    The Krona plot presents the HTML file containing the KO analysis results for metagenome bins
    """
    Read_Based_Analysis_Info_File = "Read Based Analysis Info File"
    """
    File containing reads based analysis information
    """
    GTDBTK_Bacterial_Summary = "GTDBTK Bacterial Summary"
    """
    GTDBTK bacterial summary
    """
    GTDBTK_Archaeal_Summary = "GTDBTK Archaeal Summary"
    """
    GTDBTK archaeal summary
    """
    GOTTCHA2_Krona_Plot = "GOTTCHA2 Krona Plot"
    """
    GOTTCHA2 krona plot HTML file
    """
    GOTTCHA2_Classification_Report = "GOTTCHA2 Classification Report"
    """
    GOTTCHA2 classification report file
    """
    GOTTCHA2_Report_Full = "GOTTCHA2 Report Full"
    """
    GOTTCHA2 report file
    """
    Kraken2_Krona_Plot = "Kraken2 Krona Plot"
    """
    Kraken2 krona plot HTML file
    """
    Centrifuge_Krona_Plot = "Centrifuge Krona Plot"
    """
    Centrifuge krona plot HTML file
    """
    Centrifuge_output_report_file = "Centrifuge output report file"
    """
    Centrifuge output report file
    """
    Kraken2_Classification_Report = "Kraken2 Classification Report"
    """
    Kraken2 output report file
    """
    Kraken2_Taxonomic_Classification = "Kraken2 Taxonomic Classification"
    """
    Kraken2 output read classification file
    """
    Centrifuge_Classification_Report = "Centrifuge Classification Report"
    """
    Centrifuge output report file
    """
    Centrifuge_Taxonomic_Classification = "Centrifuge Taxonomic Classification"
    """
    Centrifuge output read classification file
    """
    Structural_Annotation_GFF = "Structural Annotation GFF"
    """
    GFF3 format file with structural annotations
    """
    Structural_Annotation_Stats_Json = "Structural Annotation Stats Json"
    """
    Structural annotations stats json
    """
    Functional_Annotation_GFF = "Functional Annotation GFF"
    """
    GFF3 format file with functional annotations
    """
    Annotation_Info_File = "Annotation Info File"
    """
    File containing annotation info
    """
    Annotation_Amino_Acid_FASTA = "Annotation Amino Acid FASTA"
    """
    FASTA amino acid file for annotated proteins
    """
    Annotation_Enzyme_Commission = "Annotation Enzyme Commission"
    """
    Tab delimited file for EC annotation
    """
    Annotation_KEGG_Orthology = "Annotation KEGG Orthology"
    """
    Tab delimited file for KO annotation
    """
    Assembly_Info_File = "Assembly Info File"
    """
    File containing assembly info
    """
    Assembly_Coverage_BAM = "Assembly Coverage BAM"
    """
    Sorted bam file of reads mapping back to the final assembly
    """
    Assembly_AGP = "Assembly AGP"
    """
    An AGP format file that describes the assembly
    """
    Assembly_Scaffolds = "Assembly Scaffolds"
    """
    Final assembly scaffolds fasta
    """
    Assembly_Contigs = "Assembly Contigs"
    """
    Final assembly contigs fasta
    """
    Assembly_Coverage_Stats = "Assembly Coverage Stats"
    """
    Assembled contigs coverage information
    """
    Contig_Mapping_File = "Contig Mapping File"
    """
    Contig mappings between contigs and scaffolds
    """
    Error_Corrected_Reads = "Error Corrected Reads"
    """
    Error corrected reads fastq
    """
    Filtered_Sequencing_Reads = "Filtered Sequencing Reads"
    """
    Reads QC result fastq (clean data)
    """
    Read_Filtering_Info_File = "Read Filtering Info File"
    """
    File containing read filtering information
    """
    QC_Statistics_Extended = "QC Statistics Extended"
    """
    Extended report including methods and results for read filtering
    """
    QC_Statistics = "QC Statistics"
    """
    Reads QC summary statistics
    """
    TIGRFam_Annotation_GFF = "TIGRFam Annotation GFF"
    """
    GFF3 format file with TIGRfam
    """
    CRT_Annotation_GFF = "CRT Annotation GFF"
    """
    GFF3 format file with CRT
    """
    Genemark_Annotation_GFF = "Genemark Annotation GFF"
    """
    GFF3 format file with Genemark
    """
    Prodigal_Annotation_GFF = "Prodigal Annotation GFF"
    """
    GFF3 format file with Prodigal
    """
    TRNA_Annotation_GFF = "TRNA Annotation GFF"
    """
    GFF3 format file with TRNA
    """
    Misc_Annotation_GFF = "Misc Annotation GFF"
    """
    GFF3 format file with Misc
    """
    RFAM_Annotation_GFF = "RFAM Annotation GFF"
    """
    GFF3 format file with RFAM
    """
    TMRNA_Annotation_GFF = "TMRNA Annotation GFF"
    """
    GFF3 format file with TMRNA
    """
    Crispr_Terms = "Crispr Terms"
    """
    Crispr Terms
    """
    Product_Names = "Product Names"
    """
    Product names file
    """
    Gene_Phylogeny_tsv = "Gene Phylogeny tsv"
    """
    Gene Phylogeny tsv
    """
    Scaffold_Lineage_tsv = "Scaffold Lineage tsv"
    """
    phylogeny at the scaffold level
    """
    Clusters_of_Orthologous_Groups_LEFT_PARENTHESISCOGRIGHT_PARENTHESIS_Annotation_GFF = "Clusters of Orthologous Groups (COG) Annotation GFF"
    """
    GFF3 format file with COGs
    """
    KO_EC_Annotation_GFF = "KO_EC Annotation GFF"
    """
    GFF3 format file with KO_EC
    """
    CATH_FunFams_LEFT_PARENTHESISFunctional_FamiliesRIGHT_PARENTHESIS_Annotation_GFF = "CATH FunFams (Functional Families) Annotation GFF"
    """
    GFF3 format file with CATH FunFams
    """
    SUPERFam_Annotation_GFF = "SUPERFam Annotation GFF"
    """
    GFF3 format file with SUPERFam
    """
    SMART_Annotation_GFF = "SMART Annotation GFF"
    """
    GFF3 format file with SMART
    """
    Pfam_Annotation_GFF = "Pfam Annotation GFF"
    """
    GFF3 format file with Pfam
    """
    Annotation_Statistics = "Annotation Statistics"
    """
    Annotation statistics report
    """
    Direct_Infusion_FT_ICR_MS_Raw_Data = "Direct Infusion FT ICR-MS Raw Data"
    """
    Direct infusion Fourier transform ion cyclotron resonance mass spectrometry raw data
    """
    LC_FT_ICR_MS_Raw_Data = "LC FT-ICR MS Raw Data"
    """
    Fourier transform ion cyclotron resonance mass spectrometry raw data acquired with liquid chromatography
    """
    LC_DDA_MSSOLIDUSMS_Raw_Data = "LC-DDA-MS/MS Raw Data"
    """
    Liquid chromatographically separated MS1 and Data-Dependent MS2 binary instrument file
    """
    GC_MS_Raw_Data = "GC-MS Raw Data"
    """
    Gas chromatography-mass spectrometry raw data, full scan mode.
    """
    Configuration_toml = "Configuration toml"
    """
    A configuration toml file used by various programs to store settings that are specific to their respective software.
    """
    LC_MS_Lipidomics_Results = "LC-MS Lipidomics Results"
    """
    LC-MS-based lipidomics analysis results table
    """
    LC_MS_Lipidomics_Processed_Data = "LC-MS Lipidomics Processed Data"
    """
    Processed data for the LC-MS-based lipidomics analysis in hdf5 format
    """
    LC_MS_Metabolomics_Processed_Data = "LC-MS Metabolomics Processed Data"
    """
    Processed data for the LC-MS-based metabolomics analysis in hdf5 format
    """
    Contaminants_Amino_Acid_FASTA = "Contaminants Amino Acid FASTA"
    """
    FASTA amino acid file for contaminant proteins commonly observed in proteomics data.
    """
    Analysis_Tool_Parameter_File = "Analysis Tool Parameter File"
    """
    A configuration file used by a single computational software tool that stores settings that are specific to that tool.
    """
    Workflow_Operation_Summary = "Workflow Operation Summary"
    """
    A human readable record of analysis steps applied during an instance of a workflow operation.
    """
    Metatranscriptome_Expression = "Metatranscriptome Expression"
    """
    Metatranscriptome expression values and read counts for gene features predicted on contigs
    """
    Metatranscriptome_Expression_Intergenic = "Metatranscriptome Expression Intergenic"
    """
    Metatranscriptome expression values and read counts for intergenic regions.
    """
    Metatranscriptome_Expression_Info_File = "Metatranscriptome Expression Info File"
    """
    File containing version information on the expression workflow
    """
    rRNA_Filtered_Sequencing_Reads = "rRNA Filtered Sequencing Reads"
    """
    File containing ribosomal reads from the read qc filtering step.
    """
    BAI_File = "BAI File"
    """
    An index file found in the same directory as the binary alignment map (BAM) file, a compressed binary version of a sequence alignment/map (SAM) file.
    """


class DoiProviderEnum(str, Enum):
    EMSL = "emsl"
    JGI = "jgi"
    KBase = "kbase"
    OSTI = "osti"
    ESS_DIVE = "ess_dive"
    MassIVE = "massive"
    GSC = "gsc"
    Zenodo = "zenodo"
    EDI = "edi"
    Figshare = "figshare"


class DoiCategoryEnum(str, Enum):
    award_doi = "award_doi"
    """
    A type of DOI that resolves to a funding authority.
    """
    dataset_doi = "dataset_doi"
    """
    A type of DOI that resolves to generated data.
    """
    publication_doi = "publication_doi"
    """
    A type of DOI that resolves to a publication.
    """
    data_management_plan_doi = "data_management_plan_doi"
    """
    A type of DOI that resolves to a data management plan.
    """


class StatusEnum(str, Enum):
    pass = "pass"
    fail = "fail"


class NucleotideSequencingEnum(str, Enum):
    Metagenome = "metagenome"
    Metatranscriptome = "metatranscriptome"
    Amplicon = "amplicon_sequencing_assay"


class MassSpectrometryEnum(str, Enum):
    Metaproteome = "metaproteome"
    Metabolome = "metabolome"
    Lipidome = "lipidome"
    Natural_Organic_Matter = "nom"


class ExtractionTargetEnum(str, Enum):
    DNA = "DNA"
    RNA = "RNA"
    metabolite = "metabolite"
    protein = "protein"
    lipid = "lipid"
    natural_organic_matter = "natural organic matter"


class ProcessingInstitutionEnum(str, Enum):
    University_of_California_San_Diego = "UCSD"
    Joint_Genome_Institute = "JGI"
    Environmental_Molecular_Sciences_Laboratory = "EMSL"
    Battelle_Memorial_Institute = "Battelle"
    Argonne_National_Laboratory = "ANL"
    University_of_California_Davis_Genome_Center = "UCD_Genome_Center"
    Azenta_Life_Sciences = "Azenta"


class DataCategoryEnum(str, Enum):
    instrument_data = "instrument_data"
    """
    Data generated by a DataGeneration PlannedProcess
    """
    processed_data = "processed_data"
    """
    Data generated by a WorkflowExecution PlannedProcess
    """
    workflow_parameter_data = "workflow_parameter_data"
    """
    Data used as input into a workflow providing workflow specification.
    """


class SampleTypeEnum(str, Enum):
    soil = "soil"
    soil___water_extract = "soil - water extract"
    plant_associated = "plant associated"
    sediment = "sediment"
    water = "water"


class DNASampleFormatEnum(str, Enum):
    number_10_mM_Tris_HCl = "10 mM Tris-HCl"
    DNAStable = "DNAStable"
    Ethanol = "Ethanol"
    Low_EDTA_TE = "Low EDTA TE"
    MDA_reaction_buffer = "MDA reaction buffer"
    PBS = "PBS"
    Pellet = "Pellet"
    RNAStable = "RNAStable"
    TE = "TE"
    Water = "Water"
    Gentegra_DNA = "Gentegra-DNA"
    Gentegra_RNA = "Gentegra-RNA"


class JgiContTypeEnum(str, Enum):
    plate = "plate"
    tube = "tube"


class YesNoEnum(str, Enum):
    """
    replaces DnaDnaseEnum and DnaseRnaEnum
    """
    no = "no"
    yes = "yes"


class RNASampleFormatEnum(str, Enum):
    number_10_mM_Tris_HCl = "10 mM Tris-HCl"
    DNAStable = "DNAStable"
    Ethanol = "Ethanol"
    Low_EDTA_TE = "Low EDTA TE"
    MDA_reaction_buffer = "MDA reaction buffer"
    PBS = "PBS"
    Pellet = "Pellet"
    RNAStable = "RNAStable"
    TE = "TE"
    Water = "Water"
    Gentegra_DNA = "Gentegra-DNA"
    Gentegra_RNA = "Gentegra-RNA"


class AnalysisTypeEnum(str, Enum):
    metabolomics = "metabolomics"
    lipidomics = "lipidomics"
    Metagenomics = "metagenomics"
    """
    Standard short-read metagenomic sequencing
    """
    Metagenomics_LEFT_PARENTHESISlong_readRIGHT_PARENTHESIS = "metagenomics_long_read"
    """
    Long-read metagenomic sequencing
    """
    metaproteomics = "metaproteomics"
    metatranscriptomics = "metatranscriptomics"
    natural_organic_matter = "natural organic matter"
    bulk_chemistry = "bulk chemistry"
    Amplicon_sequencing_assay = "amplicon sequencing assay"


class SubmissionStatusEnum(str, Enum):
    In_Progress = "InProgress"
    """
    The submitter is currently working on the submission.
    """
    Submitted___Pending_Review = "SubmittedPendingReview"
    """
    Submission is ready for NMDC review, the submitter cannot edit.
    """
    Resubmitted___Pending_review = "ResubmittedPendingReview"
    """
    Submission has been resubmitted after updates. It is now ready for NMDC review. The submitter cannot edit.
    """
    Approved___Held = "ApprovedHeld"
    """
    Submission has been reviewed and approved. Information is complete, but not yet shared on the data portal. The submitter cannot edit.
    """
    Pending___Sent_to_User_Facility = "PendingUserFacility"
    """
    Submission has been reviewed and approved. Information is complete, but not yet shared on the data portal. Sample information shared with designated user facility and pending approvals. The submitter cannot edit.
    """
    Updates_Required = "UpdatesRequired"
    """
    Submission has been reviewed and submitter edits are required for approval. The submitter can reopen and edit the submission.
    """
    In_Progress___UpdateSOLIDUSAddition = "InProgressUpdate"
    """
    NMDC reviewer has reopened submission on behalf of submitter. The submitter is currently editing the submission.
    """
    Denied = "Denied"
    """
    Submission has been reviewed and denied. The submitter cannot edit.
    """
    Released = "Released"
    """
    Submission has been reviewed and approved and data is released on the data portal. The submitter cannot edit.
    """


class MetaproteomicsAnalysisCategoryEnum(str, Enum):
    """
    The category of metaproteomics analysis being performed.
    """
    matched_metagenome = "matched_metagenome"
    """
    A metaproteomics analysis that is matched to a metagenome derived from the same biosample.
    """
    in_silico_metagenome = "in_silico_metagenome"
    """
    A metaproteomics analysis that is matched to an in silico generated metagenome.
    """


class MetabolomicsAnalysisCategoryEnum(str, Enum):
    """
    The category of metabolomics analysis being performed.
    """
    gc_ms_metabolomics = "gc_ms_metabolomics"
    """
    A metabolomics analysis that is performed on gas chromatography mass spectrometry data.
    """
    lc_ms_lipidomics = "lc_ms_lipidomics"
    """
    A metabolomics analysis that is performed on liquid chromatography mass spectrometry data for lipidomics annotation.
    """
    lc_ms_metabolomics = "lc_ms_metabolomics"
    """
    A metabolomics analysis that is performed on liquid chromatography mass spectrometry data.
    """



class EukEval(ConfiguredBaseModel):
    """
    This class contains information pertaining to evaluating if a Metagenome-Assembled Genome (MAG) is eukaryotic.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:EukEval',
         'comments': ['A tool like eukCC (https://doi.org/10.1186/s13059-020-02155-4) '
                      'would generate information for this class.'],
         'from_schema': 'https://w3id.org/nmdc/nmdc'})

    type: Literal["https://w3id.org/nmdc/EukEval","nmdc:EukEval"] = Field(default="nmdc:EukEval", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })
    completeness: Optional[float] = Field(default=None, description="""Estimate of the completeness of the metagenome-assembled genome, estimated by a tool like CheckM.""", ge=0, json_schema_extra = { "linkml_meta": {'alias': 'completeness', 'domain_of': ['EukEval', 'MagBin']} })
    contamination: Optional[float] = Field(default=None, description="""Estimate of the completeness of the metagenome-assembled genome, estimated by a tool like CheckM.""", ge=0, json_schema_extra = { "linkml_meta": {'alias': 'contamination', 'domain_of': ['EukEval', 'MagBin']} })
    ncbi_lineage_tax_ids: Optional[str] = Field(default=None, description="""Dash-delimited ordered list of NCBI taxonomy identifiers (TaxId)""", json_schema_extra = { "linkml_meta": {'alias': 'ncbi_lineage_tax_ids',
         'domain_of': ['EukEval'],
         'examples': [{'value': '1-131567-2759-2611352-33682-191814-2603949'}]} })
    ncbi_lineage: Optional[str] = Field(default=None, description="""Comma delimited ordered list of NCBI taxonomy names.""", json_schema_extra = { "linkml_meta": {'alias': 'ncbi_lineage',
         'domain_of': ['EukEval'],
         'examples': [{'value': 'root,cellular '
                                'organisms,Eukaryota,Discoba,Euglenozoa,Diplonemea,Diplonemidae'}]} })

    @field_validator('ncbi_lineage_tax_ids')
    def pattern_ncbi_lineage_tax_ids(cls, v):
        pattern=re.compile(r"^\d+(-\d+)*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid ncbi_lineage_tax_ids format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid ncbi_lineage_tax_ids format: {v}"
            raise ValueError(err_msg)
        return v


class FunctionalAnnotationAggMember(ConfiguredBaseModel):
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:FunctionalAnnotationAggMember',
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'slot_usage': {'count': {'description': 'The number of sequences (for a '
                                                 'metagenome or metatranscriptome) or '
                                                 'spectra (for metaproteomics) '
                                                 'associated with the specified '
                                                 'function.',
                                  'name': 'count'},
                        'was_generated_by': {'name': 'was_generated_by',
                                             'pattern': '^(nmdc):(wfmgan|wfmp|wfmtan)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})(\\.[0-9]{1,})$',
                                             'range': 'AnnotatingWorkflow',
                                             'required': True,
                                             'structured_pattern': {'interpolated': True,
                                                                    'syntax': '{id_nmdc_prefix}:(wfmgan|wfmp|wfmtan)-{id_shoulder}-{id_blade}{id_version}$'}}}})

    was_generated_by: str = Field(default=..., json_schema_extra = { "linkml_meta": {'alias': 'was_generated_by',
         'domain_of': ['FunctionalAnnotationAggMember',
                       'FunctionalAnnotation',
                       'DataObject'],
         'mappings': ['prov:wasGeneratedBy'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(wfmgan|wfmp|wfmtan)-{id_shoulder}-{id_blade}{id_version}$'}} })
    gene_function_id: str = Field(default=..., description="""The identifier for the gene function.""", json_schema_extra = { "linkml_meta": {'alias': 'gene_function_id',
         'domain_of': ['FunctionalAnnotationAggMember'],
         'examples': [{'value': 'KEGG.ORTHOLOGY:K00627'}]} })
    count: int = Field(default=..., description="""The number of sequences (for a metagenome or metatranscriptome) or spectra (for metaproteomics) associated with the specified function.""", json_schema_extra = { "linkml_meta": {'alias': 'count', 'domain_of': ['FunctionalAnnotationAggMember']} })
    type: Literal["https://w3id.org/nmdc/FunctionalAnnotationAggMember","nmdc:FunctionalAnnotationAggMember"] = Field(default="nmdc:FunctionalAnnotationAggMember", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('was_generated_by')
    def pattern_was_generated_by(cls, v):
        pattern=re.compile(r"^(nmdc):(wfmgan|wfmp|wfmtan)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})(\.[0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid was_generated_by format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid was_generated_by format: {v}"
            raise ValueError(err_msg)
        return v


class Database(ConfiguredBaseModel):
    """
    An abstract holder for any set of metadata and data. It does not need to correspond to an actual managed database top level holder class. When translated to JSON-Schema this is the 'root' object. It should contain pointers to other objects of interest. For MongoDB, the lists of objects that Database slots point to correspond to **collections**.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'aliases': ['NMDC metadata object'],
         'class_uri': 'nmdc:Database',
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'tree_root': True})

    biosample_set: Optional[list[Biosample]] = Field(default=None, description="""This property links a database object to the set of samples within it.""", json_schema_extra = { "linkml_meta": {'alias': 'biosample_set', 'domain_of': ['Database'], 'mixins': ['object_set']} })
    calibration_set: Optional[list[CalibrationInformation]] = Field(default=None, description="""This property links a database object to the set of calibrations within it.""", json_schema_extra = { "linkml_meta": {'alias': 'calibration_set',
         'domain_of': ['Database'],
         'mixins': ['object_set']} })
    collecting_biosamples_from_site_set: Optional[list[CollectingBiosamplesFromSite]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'collecting_biosamples_from_site_set',
         'domain_of': ['Database'],
         'mixins': ['object_set']} })
    configuration_set: Optional[list[Union[Configuration,MassSpectrometryConfiguration,ChromatographyConfiguration]]] = Field(default=None, description="""This property links a database object to the set of configurations within it.""", json_schema_extra = { "linkml_meta": {'alias': 'configuration_set',
         'domain_of': ['Database'],
         'mixins': ['object_set']} })
    data_generation_set: Optional[list[Union[DataGeneration,NucleotideSequencing,MassSpectrometry]]] = Field(default=None, description="""This property links a database object to the set of data generations within it.""", json_schema_extra = { "linkml_meta": {'alias': 'data_generation_set',
         'domain_of': ['Database'],
         'mixins': ['object_set']} })
    data_object_set: Optional[list[DataObject]] = Field(default=None, description="""This property links a database object to the set of data objects within it.""", json_schema_extra = { "linkml_meta": {'alias': 'data_object_set',
         'domain_of': ['Database'],
         'mixins': ['object_set']} })
    field_research_site_set: Optional[list[FieldResearchSite]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'field_research_site_set',
         'domain_of': ['Database'],
         'mixins': ['object_set']} })
    functional_annotation_agg: Optional[list[FunctionalAnnotationAggMember]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'functional_annotation_agg', 'domain_of': ['Database']} })
    functional_annotation_set: Optional[list[FunctionalAnnotation]] = Field(default=None, description="""This property links a database object to the set of all functional annotations""", json_schema_extra = { "linkml_meta": {'alias': 'functional_annotation_set',
         'domain_of': ['Database'],
         'mixins': ['object_set']} })
    genome_feature_set: Optional[list[GenomeFeature]] = Field(default=None, description="""This property links a database object to the set of all features""", json_schema_extra = { "linkml_meta": {'alias': 'genome_feature_set',
         'domain_of': ['Database'],
         'mixins': ['object_set']} })
    instrument_set: Optional[list[Instrument]] = Field(default=None, description="""This property links a database object to the set of instruments within it.""", json_schema_extra = { "linkml_meta": {'alias': 'instrument_set', 'domain_of': ['Database'], 'mixins': ['object_set']} })
    manifest_set: Optional[list[Manifest]] = Field(default=None, description="""This property links a database object to the set of manifests within it.""", json_schema_extra = { "linkml_meta": {'alias': 'manifest_set', 'domain_of': ['Database'], 'mixins': ['object_set']} })
    material_processing_set: Optional[list[Union[MaterialProcessing,Pooling,Extraction,LibraryPreparation,SubSamplingProcess,MixingProcess,FiltrationProcess,ChromatographicSeparationProcess,DissolvingProcess,ChemicalConversionProcess]]] = Field(default=None, description="""This property links a database object to the set of material processing within it.""", json_schema_extra = { "linkml_meta": {'alias': 'material_processing_set',
         'domain_of': ['Database'],
         'mixins': ['object_set']} })
    processed_sample_set: Optional[list[ProcessedSample]] = Field(default=None, description="""This property links a database object to the set of processed samples within it.""", json_schema_extra = { "linkml_meta": {'alias': 'processed_sample_set',
         'domain_of': ['Database'],
         'mixins': ['object_set']} })
    storage_process_set: Optional[list[StorageProcess]] = Field(default=None, description="""This property links a database object to the set of storage processes within it.""", json_schema_extra = { "linkml_meta": {'alias': 'storage_process_set',
         'domain_of': ['Database'],
         'mixins': ['object_set']} })
    study_set: Optional[list[Study]] = Field(default=None, description="""This property links a database object to the set of studies within it.""", json_schema_extra = { "linkml_meta": {'alias': 'study_set', 'domain_of': ['Database'], 'mixins': ['object_set']} })
    workflow_execution_set: Optional[list[Union[WorkflowExecution,AnnotatingWorkflow,MetagenomeAssembly,MetatranscriptomeAssembly,MetatranscriptomeExpressionAnalysis,MagsAnalysis,MetagenomeSequencing,ReadQcAnalysis,ReadBasedTaxonomyAnalysis,MetabolomicsAnalysis,NomAnalysis,MetagenomeAnnotation,MetatranscriptomeAnnotation,MetaproteomicsAnalysis]]] = Field(default=None, description="""This property links a database object to the set of workflow executions.""", json_schema_extra = { "linkml_meta": {'alias': 'workflow_execution_set',
         'domain_of': ['Database'],
         'mixins': ['object_set']} })


class MobilePhaseSegment(ConfiguredBaseModel):
    """
    A fluid mixture of substances that flow though a chromatographic stationary phase.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:MobilePhaseSegment',
         'comments': ['Can be assembled into a list to account for a gradient.',
                      'Works best for separations that can be accounted for by a '
                      'series of essentially linear changes in the mobile phase '
                      'composition.',
                      "For gradients, each segment should declare it's duration.",
                      'For step-wise separations that might be implemented with a '
                      'syringe and a cartridge, it will probably be more important to '
                      'assert volumes.'],
         'from_schema': 'https://w3id.org/nmdc/nmdc'})

    duration: Optional[QuantityValue] = Field(default=None, description="""The elapsed time of an activity.""", json_schema_extra = { "linkml_meta": {'alias': 'duration',
         'domain_of': ['MixingProcess',
                       'DissolvingProcess',
                       'ChemicalConversionProcess',
                       'MobilePhaseSegment'],
         'examples': [{'value': "JsonObj(has_numeric_value=2, has_unit='hours')"}]} })
    substances_used: Optional[list[PortionOfSubstance]] = Field(default=None, description="""The substances that are combined to enable a ChemicalConversionProcess.""", json_schema_extra = { "linkml_meta": {'alias': 'substances_used',
         'domain_of': ['Extraction',
                       'StorageProcess',
                       'DissolvingProcess',
                       'ChemicalConversionProcess',
                       'MobilePhaseSegment']} })
    type: Literal["https://w3id.org/nmdc/MobilePhaseSegment","nmdc:MobilePhaseSegment"] = Field(default="nmdc:MobilePhaseSegment", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })
    volume: Optional[QuantityValue] = Field(default=None, description="""The volume of a substance.""", json_schema_extra = { "linkml_meta": {'alias': 'volume',
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-8683-0050'],
         'domain_of': ['Extraction',
                       'SubSamplingProcess',
                       'FiltrationProcess',
                       'MobilePhaseSegment',
                       'PortionOfSubstance']} })


class PortionOfSubstance(ConfiguredBaseModel):
    """
    A portion of any matter of defined composition that has discrete existence, whose origin may be biological, mineral or chemical.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:PortionOfSubstance',
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'mappings': ['schema:Substance'],
         'title': 'Portion of a Substance'})

    final_concentration: Optional[QuantityValue] = Field(default=None, description="""When solutions A (containing substance X) and B are combined together, this slot captures the concentration of X in the combination""", json_schema_extra = { "linkml_meta": {'alias': 'final_concentration',
         'domain_of': ['PortionOfSubstance'],
         'is_a': 'concentration'} })
    mass: Optional[QuantityValue] = Field(default=None, title="mass", description="""A physical quality that inheres in a bearer by virtue of the proportion of the bearer's amount of matter.""", json_schema_extra = { "linkml_meta": {'alias': 'mass',
         'domain_of': ['SubSamplingProcess', 'PortionOfSubstance'],
         'exact_mappings': ['PATO:0000125']} })
    source_concentration: Optional[QuantityValue] = Field(default=None, description="""When solutions A (containing substance X) and B are combined together, this slot captures the concentration of X in solution A""", json_schema_extra = { "linkml_meta": {'alias': 'source_concentration',
         'domain_of': ['PortionOfSubstance'],
         'is_a': 'concentration'} })
    known_as: Optional[ChemicalEntityEnum] = Field(default=None, description="""The substance from which a portion was taken.""", json_schema_extra = { "linkml_meta": {'alias': 'known_as', 'domain_of': ['PortionOfSubstance']} })
    substance_role: Optional[SubstanceRoleEnum] = Field(default=None, description="""The role of a substance in a process""", json_schema_extra = { "linkml_meta": {'alias': 'substance_role', 'domain_of': ['PortionOfSubstance']} })
    type: Literal["https://w3id.org/nmdc/PortionOfSubstance","nmdc:PortionOfSubstance"] = Field(default="nmdc:PortionOfSubstance", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })
    volume: Optional[QuantityValue] = Field(default=None, description="""The volume of a substance.""", json_schema_extra = { "linkml_meta": {'alias': 'volume',
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-8683-0050'],
         'domain_of': ['Extraction',
                       'SubSamplingProcess',
                       'FiltrationProcess',
                       'MobilePhaseSegment',
                       'PortionOfSubstance']} })


class MagBin(ConfiguredBaseModel):
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:MagBin', 'from_schema': 'https://w3id.org/nmdc/nmdc'})

    bin_name: str = Field(default=..., description="""Name of the metagenome-assembled genome.""", json_schema_extra = { "linkml_meta": {'alias': 'bin_name', 'domain_of': ['MagBin']} })
    bin_quality: Optional[BinQualityEnum] = Field(default=None, description="""The quality of the metagenome-assembled genome based on MIMAG standards (https://doi.org/10.1038/nbt.3893).""", json_schema_extra = { "linkml_meta": {'alias': 'bin_quality', 'domain_of': ['MagBin']} })
    completeness: Optional[float] = Field(default=None, description="""Estimate of the completeness of the metagenome-assembled genome, estimated by a tool like CheckM.""", ge=0, json_schema_extra = { "linkml_meta": {'alias': 'completeness', 'domain_of': ['EukEval', 'MagBin']} })
    contamination: Optional[float] = Field(default=None, description="""Estimate of the completeness of the metagenome-assembled genome, estimated by a tool like CheckM.""", ge=0, json_schema_extra = { "linkml_meta": {'alias': 'contamination', 'domain_of': ['EukEval', 'MagBin']} })
    gene_count: Optional[int] = Field(default=None, description="""Number of genes.""", ge=0, json_schema_extra = { "linkml_meta": {'alias': 'gene_count', 'domain_of': ['MagBin']} })
    gtdbtk_class: Optional[str] = Field(default=None, description="""Taxonomic class assigned by GTDB-Tk.""", json_schema_extra = { "linkml_meta": {'alias': 'gtdbtk_class',
         'domain_of': ['MagBin'],
         'exact_mappings': ['TAXRANK:0000002'],
         'see_also': ['doi:10.1093/bioinformatics/btz848']} })
    gtdbtk_domain: Optional[str] = Field(default=None, description="""Taxonomic domain assigned by GTDB-Tk.""", json_schema_extra = { "linkml_meta": {'alias': 'gtdbtk_domain',
         'domain_of': ['MagBin'],
         'exact_mappings': ['TAXRANK:0000022'],
         'see_also': ['doi:10.1093/bioinformatics/btz848']} })
    gtdbtk_family: Optional[str] = Field(default=None, description="""Taxonomic family assigned by GTDB-Tk.""", json_schema_extra = { "linkml_meta": {'alias': 'gtdbtk_family',
         'domain_of': ['MagBin'],
         'exact_mappings': ['TAXRANK:0000004'],
         'see_also': ['doi:10.1093/bioinformatics/btz848']} })
    gtdbtk_genus: Optional[str] = Field(default=None, description="""Taxonomic genus assigned by GTDB-Tk.""", json_schema_extra = { "linkml_meta": {'alias': 'gtdbtk_genus',
         'domain_of': ['MagBin'],
         'exact_mappings': ['TAXRANK:0000005'],
         'see_also': ['doi:10.1093/bioinformatics/btz848']} })
    gtdbtk_order: Optional[str] = Field(default=None, description="""Taxonomic order assigned by GTDB-Tk.""", json_schema_extra = { "linkml_meta": {'alias': 'gtdbtk_order',
         'domain_of': ['MagBin'],
         'exact_mappings': ['TAXRANK:0000003'],
         'see_also': ['doi:10.1093/bioinformatics/btz848']} })
    gtdbtk_phylum: Optional[str] = Field(default=None, description="""Taxonomic phylum assigned by GTDB-Tk.""", json_schema_extra = { "linkml_meta": {'alias': 'gtdbtk_phylum',
         'domain_of': ['MagBin'],
         'exact_mappings': ['TAXRANK:0000001'],
         'see_also': ['doi:10.1093/bioinformatics/btz848']} })
    gtdbtk_species: Optional[str] = Field(default=None, description="""Taxonomic genus assigned by GTDB-Tk.""", json_schema_extra = { "linkml_meta": {'alias': 'gtdbtk_species',
         'domain_of': ['MagBin'],
         'exact_mappings': ['TAXRANK:0000006'],
         'see_also': ['doi:10.1093/bioinformatics/btz848']} })
    members_id: Optional[list[str]] = Field(default=None, description="""Names of the contigs that make up a metagenome-assembled genome.""", json_schema_extra = { "linkml_meta": {'alias': 'members_id',
         'close_mappings': ['GENEPIO:0100596'],
         'domain_of': ['MagBin']} })
    num_16s: Optional[int] = Field(default=None, description="""Number of 16s sequences detected, a subunit of prokaryotic ribosomes.""", ge=0, json_schema_extra = { "linkml_meta": {'alias': 'num_16s',
         'broad_mappings': ['OMIT:0013243'],
         'domain_of': ['MagBin']} })
    num_23s: Optional[int] = Field(default=None, description="""Number of 23 seqeuences detected, a subunit of ribosomes.""", ge=0, json_schema_extra = { "linkml_meta": {'alias': 'num_23s',
         'broad_mappings': ['OMIT:0013245'],
         'domain_of': ['MagBin']} })
    num_5s: Optional[int] = Field(default=None, description="""Number of 5s seqeuences detected, a subunit of ribosomes.""", ge=0, json_schema_extra = { "linkml_meta": {'alias': 'num_5s', 'broad_mappings': ['OMIT:0013248'], 'domain_of': ['MagBin']} })
    num_t_rna: Optional[int] = Field(default=None, description="""Number of transfer transfer RNAs.""", ge=0, json_schema_extra = { "linkml_meta": {'alias': 'num_t_rna',
         'broad_mappings': ['OMIT:0013250'],
         'domain_of': ['MagBin']} })
    number_of_contig: Optional[int] = Field(default=None, description="""Number of contigs""", ge=0, json_schema_extra = { "linkml_meta": {'alias': 'number_of_contig',
         'domain_of': ['MagBin'],
         'exact_mappings': ['GENEPIO:0000093']} })
    total_bases: Optional[int] = Field(default=None, description="""Total number of basepairs.""", ge=0, json_schema_extra = { "linkml_meta": {'alias': 'total_bases', 'domain_of': ['MagBin']} })
    type: Literal["https://w3id.org/nmdc/MagBin","nmdc:MagBin"] = Field(default="nmdc:MagBin", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })
    eukaryotic_evaluation: Optional[EukEval] = Field(default=None, description="""Contains results from evaluating if a Metagenome-Assembled Genome is of eukaryotic lineage.""", json_schema_extra = { "linkml_meta": {'alias': 'eukaryotic_evaluation', 'domain_of': ['MagBin']} })


class MetaboliteIdentification(ConfiguredBaseModel):
    """
    This is used to link a metabolomics analysis workflow to a specific metabolite
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:MetaboliteIdentification',
         'from_schema': 'https://w3id.org/nmdc/nmdc'})

    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    highest_similarity_score: Optional[float] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'highest_similarity_score',
         'domain_of': ['MetaboliteIdentification'],
         'todos': ['Yuri to fill in description']} })
    metabolite_identified: Optional[str] = Field(default=None, description="""the specific metabolite identifier""", json_schema_extra = { "linkml_meta": {'alias': 'metabolite_identified', 'domain_of': ['MetaboliteIdentification']} })
    type: Literal["https://w3id.org/nmdc/MetaboliteIdentification","nmdc:MetaboliteIdentification"] = Field(default="nmdc:MetaboliteIdentification", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class GenomeFeature(ConfiguredBaseModel):
    """
    A feature localized to an interval along a genome
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:GenomeFeature',
         'comments': ['corresponds to an entry in GFF3'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'see_also': ['https://github.com/The-Sequence-Ontology/Specifications/blob/master/gff3.md'],
         'slot_usage': {'end': {'name': 'end', 'required': True},
                        'seqid': {'name': 'seqid', 'required': True},
                        'start': {'name': 'start', 'required': True}}})

    encodes: Optional[str] = Field(default=None, description="""The gene product encoded by this feature. Typically this is used for a CDS feature or gene feature which will encode a protein. It can also be used by a nc transcript ot gene feature that encoded a ncRNA""", json_schema_extra = { "linkml_meta": {'alias': 'encodes',
         'domain_of': ['GenomeFeature'],
         'todos': ['If we revert Reaction back into the schema, that would be a '
                   'reasonable domain for this slot']} })
    end: int = Field(default=..., description="""The end of the feature in positive 1-based integer coordinates""", ge=1, json_schema_extra = { "linkml_meta": {'alias': 'end',
         'close_mappings': ['biolink:end_interbase_coordinate'],
         'comments': ['- "unenforced constraint: end > start" - "For features that '
                      'cross the origin of a circular feature,  end = the position of '
                      'the end + the length of the landmark feature."'],
         'domain_of': ['GenomeFeature'],
         'is_a': 'gff_coordinate'} })
    feature_type: Optional[str] = Field(default=None, description="""TODO: Yuri to write""", json_schema_extra = { "linkml_meta": {'alias': 'feature_type', 'domain_of': ['GenomeFeature']} })
    phase: Optional[int] = Field(default=None, description="""The phase for a coding sequence entity. For example, phase of a CDS as represented in a GFF3 with a value of 0, 1 or 2.""", ge=0, le=2, json_schema_extra = { "linkml_meta": {'alias': 'phase',
         'domain_of': ['GenomeFeature'],
         'exact_mappings': ['biolink:phase']} })
    seqid: str = Field(default=..., description="""The ID of the landmark used to establish the coordinate system for the current feature.""", json_schema_extra = { "linkml_meta": {'alias': 'seqid',
         'domain_of': ['GenomeFeature'],
         'todos': ['change range from string to a object like uriorcurie?']} })
    start: int = Field(default=..., description="""The start of the feature in positive 1-based integer coordinates""", ge=1, json_schema_extra = { "linkml_meta": {'alias': 'start',
         'close_mappings': ['biolink:start_interbase_coordinate'],
         'domain_of': ['GenomeFeature'],
         'is_a': 'gff_coordinate'} })
    strand: Optional[str] = Field(default=None, description="""The strand on which a feature is located. Has a value of '+' (sense strand or forward strand) or  '-' (anti-sense strand or reverse strand).""", json_schema_extra = { "linkml_meta": {'alias': 'strand',
         'domain_of': ['GenomeFeature'],
         'exact_mappings': ['biolink:strand'],
         'todos': ['set the range to an enum?']} })
    type: Literal["https://w3id.org/nmdc/GenomeFeature","nmdc:GenomeFeature"] = Field(default="nmdc:GenomeFeature", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })
    feature_category: Optional[ControlledIdentifiedTermValue] = Field(default=None, description="""A Sequence Ontology term that describes the category of a feature""", json_schema_extra = { "linkml_meta": {'alias': 'feature_category',
         'domain_of': ['GenomeFeature', 'FunctionalAnnotation']} })


class FunctionalAnnotation(ConfiguredBaseModel):
    """
    An assignment of a function term (e.g. reaction or pathway) that is executed by a gene product,  or which the gene product plays an active role in. Functional annotations can be assigned manually by curators, or automatically in workflows.  In the context of NMDC, all function annotation is performed automatically, typically using HMM or Blast type methods
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:FunctionalAnnotation',
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'narrow_mappings': ['biolink:GeneToGoTermAssociation'],
         'see_also': ['https://img.jgi.doe.gov/docs/functional-annotation.pdf',
                      'https://github.com/microbiomedata/mg_annotation/blob/master/functional-annotation.wdl'],
         'slot_usage': {'has_function': {'name': 'has_function',
                                         'notes': ['Still missing patterns for COG and '
                                                   'RetroRules',
                                                   'These patterns are not yet tied to '
                                                   'the listed prefixes.',
                                                   'Are these lists are intended to be '
                                                   'open examples or closed?']},
                        'was_generated_by': {'description': 'provenance for the '
                                                            'annotation.',
                                             'name': 'was_generated_by',
                                             'notes': ['To be consistent with the rest '
                                                       'of the NMDC schema we use the '
                                                       'PROV annotation model, rather '
                                                       'than GPAD'],
                                             'pattern': '^(nmdc):(wfmgan)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})(\\.[0-9]{1,})$',
                                             'range': 'MetagenomeAnnotation',
                                             'structured_pattern': {'interpolated': True,
                                                                    'syntax': '{id_nmdc_prefix}:(wfmgan)-{id_shoulder}-{id_blade}{id_version}$'}}}})

    has_function: Optional[str] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'has_function',
         'domain_of': ['FunctionalAnnotation'],
         'notes': ['Still missing patterns for COG and RetroRules',
                   'These patterns are not yet tied to the listed prefixes.',
                   'Are these lists are intended to be open examples or closed?']} })
    subject: Optional[str] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'subject', 'domain_of': ['FunctionalAnnotation', 'OntologyRelation']} })
    was_generated_by: Optional[str] = Field(default=None, description="""provenance for the annotation.""", json_schema_extra = { "linkml_meta": {'alias': 'was_generated_by',
         'domain_of': ['FunctionalAnnotationAggMember',
                       'FunctionalAnnotation',
                       'DataObject'],
         'mappings': ['prov:wasGeneratedBy'],
         'notes': ['To be consistent with the rest of the NMDC schema we use the PROV '
                   'annotation model, rather than GPAD'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(wfmgan)-{id_shoulder}-{id_blade}{id_version}$'}} })
    type: Literal["https://w3id.org/nmdc/FunctionalAnnotation","nmdc:FunctionalAnnotation"] = Field(default="nmdc:FunctionalAnnotation", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })
    feature_category: Optional[ControlledIdentifiedTermValue] = Field(default=None, description="""A Sequence Ontology term that describes the category of a feature""", json_schema_extra = { "linkml_meta": {'alias': 'feature_category',
         'domain_of': ['GenomeFeature', 'FunctionalAnnotation']} })

    @field_validator('has_function')
    def pattern_has_function(cls, v):
        pattern=re.compile(r"^(KEGG_PATHWAY:\w{2,4}\d{5}|KEGG.REACTION:R\d+|RHEA:\d{5}|MetaCyc:[A-Za-z0-9+_.%-:]+|EC:\d{1,2}(\.\d{0,3}){0,3}|GO:\d{7}|MetaNetX:(MNXR\d+|EMPTY)|SEED:\w+|KEGG\.ORTHOLOGY:K\d+|EGGNOG:\w+|PFAM:PF\d{5}|TIGRFAM:TIGR\d+|SUPFAM:\w+|CATH:[1-6]\.[0-9]+\.[0-9]+\.[0-9]+|PANTHER.FAMILY:PTHR\d{5}(\:SF\d{1,3})?)$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_function format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_function format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('was_generated_by')
    def pattern_was_generated_by(cls, v):
        pattern=re.compile(r"^(nmdc):(wfmgan)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})(\.[0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid was_generated_by format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid was_generated_by format: {v}"
            raise ValueError(err_msg)
        return v


class AttributeValue(ConfiguredBaseModel):
    """
    The value for any value of a attribute for a sample. This object can hold both the un-normalized atomic value and the structured value
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'abstract': True,
         'class_uri': 'nmdc:AttributeValue',
         'from_schema': 'https://w3id.org/nmdc/nmdc'})

    has_raw_value: Optional[str] = Field(default=None, description="""The value that was specified for an annotation in raw form, i.e. a string. E.g. \"2 cm\" or \"2-4 cm\"""", json_schema_extra = { "linkml_meta": {'alias': 'has_raw_value', 'domain_of': ['AttributeValue']} })
    type: Literal["https://w3id.org/nmdc/AttributeValue","nmdc:AttributeValue"] = Field(default="nmdc:AttributeValue", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })


class QuantityValue(AttributeValue):
    """
    A simple quantity, e.g. 2cm
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:QuantityValue',
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'mappings': ['schema:QuantityValue'],
         'slot_usage': {'has_numeric_value': {'description': 'The number part of the '
                                                             'quantity',
                                              'name': 'has_numeric_value'},
                        'has_raw_value': {'description': 'Unnormalized atomic string '
                                                         'representation, should in '
                                                         'syntax {number} {unit}',
                                          'name': 'has_raw_value'},
                        'has_unit': {'description': 'The unit of the quantity',
                                     'name': 'has_unit'}}})

    has_maximum_numeric_value: Optional[Decimal] = Field(default=None, description="""The maximum value part, expressed as number, of the quantity value when the value covers a range.""", json_schema_extra = { "linkml_meta": {'alias': 'has_maximum_numeric_value',
         'domain_of': ['QuantityValue'],
         'is_a': 'has_numeric_value'} })
    has_minimum_numeric_value: Optional[Decimal] = Field(default=None, description="""The minimum value part, expressed as number, of the quantity value when the value covers a range.""", json_schema_extra = { "linkml_meta": {'alias': 'has_minimum_numeric_value',
         'domain_of': ['QuantityValue'],
         'is_a': 'has_numeric_value'} })
    has_numeric_value: Optional[Decimal] = Field(default=None, description="""The number part of the quantity""", json_schema_extra = { "linkml_meta": {'alias': 'has_numeric_value',
         'domain_of': ['QuantityValue'],
         'mappings': ['qud:quantityValue', 'schema:value']} })
    has_unit: Optional[str] = Field(default=None, description="""The unit of the quantity""", json_schema_extra = { "linkml_meta": {'alias': 'has_unit',
         'aliases': ['scale'],
         'domain_of': ['QuantityValue'],
         'mappings': ['qud:unit', 'schema:unitCode']} })
    has_raw_value: Optional[str] = Field(default=None, description="""Unnormalized atomic string representation, should in syntax {number} {unit}""", json_schema_extra = { "linkml_meta": {'alias': 'has_raw_value', 'domain_of': ['AttributeValue']} })
    type: Literal["https://w3id.org/nmdc/QuantityValue","nmdc:QuantityValue"] = Field(default="nmdc:QuantityValue", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })


class ImageValue(AttributeValue):
    """
    An attribute value representing an image.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:ImageValue', 'from_schema': 'https://w3id.org/nmdc/nmdc'})

    url: Optional[str] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'url',
         'domain_of': ['ImageValue', 'Protocol', 'DataObject'],
         'notes': ['See issue 207 - this clashes with the mixs field']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    display_order: Optional[int] = Field(default=None, description="""When rendering information, this attribute to specify the order in which the information should be rendered.""", json_schema_extra = { "linkml_meta": {'alias': 'display_order', 'domain_of': ['ImageValue']} })
    has_raw_value: Optional[str] = Field(default=None, description="""The value that was specified for an annotation in raw form, i.e. a string. E.g. \"2 cm\" or \"2-4 cm\"""", json_schema_extra = { "linkml_meta": {'alias': 'has_raw_value', 'domain_of': ['AttributeValue']} })
    type: Literal["https://w3id.org/nmdc/ImageValue","nmdc:ImageValue"] = Field(default="nmdc:ImageValue", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })


class PersonValue(AttributeValue):
    """
    An attribute value representing a person
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:PersonValue',
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'slot_usage': {'email': {'annotations': {'tooltip': {'tag': 'tooltip',
                                                              'value': 'Email address '
                                                                       'for this '
                                                                       'person.'}},
                                  'name': 'email'},
                        'has_raw_value': {'description': 'The full name of the '
                                                         'Investigator in format FIRST '
                                                         'LAST.',
                                          'name': 'has_raw_value',
                                          'notes': ['May eventually be deprecated in '
                                                    'favor of "name".']},
                        'name': {'annotations': {'tooltip': {'tag': 'tooltip',
                                                             'value': 'First name, '
                                                                      'middle initial, '
                                                                      'and last name '
                                                                      'of this '
                                                                      'person.'}},
                                 'description': 'The full name of the Investigator. It '
                                                'should follow the format FIRST '
                                                '[MIDDLE NAME| MIDDLE INITIAL] LAST, '
                                                'where MIDDLE NAME| MIDDLE INITIAL is '
                                                'optional.',
                                 'name': 'name'},
                        'orcid': {'annotations': {'tooltip': {'tag': 'tooltip',
                                                              'value': 'Open '
                                                                       'Researcher and '
                                                                       'Contributor ID '
                                                                       'for this '
                                                                       'person. See '
                                                                       'https://orcid.org'}},
                                  'name': 'orcid'}},
         'todos': ['add additional fields e.g for institution',
                   'deprecate "has_raw_value" in favor of "name"']})

    email: Optional[str] = Field(default=None, description="""An email address for an entity such as a person. This should be the primary email address used.""", json_schema_extra = { "linkml_meta": {'alias': 'email',
         'annotations': {'tooltip': {'tag': 'tooltip',
                                     'value': 'Email address for this person.'}},
         'domain_of': ['PersonValue'],
         'slot_uri': 'schema:email'} })
    name: Optional[str] = Field(default=None, description="""The full name of the Investigator. It should follow the format FIRST [MIDDLE NAME| MIDDLE INITIAL] LAST, where MIDDLE NAME| MIDDLE INITIAL is optional.""", json_schema_extra = { "linkml_meta": {'alias': 'name',
         'annotations': {'tooltip': {'tag': 'tooltip',
                                     'value': 'First name, middle initial, and last '
                                              'name of this person.'}},
         'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    orcid: Optional[str] = Field(default=None, description="""The ORCID of a person.""", json_schema_extra = { "linkml_meta": {'alias': 'orcid',
         'annotations': {'tooltip': {'tag': 'tooltip',
                                     'value': 'Open Researcher and Contributor ID for '
                                              'this person. See https://orcid.org'}},
         'domain_of': ['PersonValue']} })
    profile_image_url: Optional[str] = Field(default=None, description="""A url that points to an image of a person.""", json_schema_extra = { "linkml_meta": {'alias': 'profile_image_url', 'domain_of': ['PersonValue']} })
    websites: Optional[list[str]] = Field(default=None, description="""A list of websites that are associated with the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'websites',
         'comments': ['DOIs should not be included as websites. Instead, use the '
                      'associated_dois slot.',
                      "A consortium's homepage website should be included in the "
                      'homepage_website slot, not in websites.',
                      'consortium is a convenience term for a Study whose '
                      'study_category value is consortium',
                      'the website slot and its subproperties are virtually identical '
                      'to the url slot, except that they are multivalued and url is '
                      'single-valued.'],
         'domain_of': ['PersonValue', 'Study'],
         'see_also': ['nmdc:url']} })
    has_raw_value: Optional[str] = Field(default=None, description="""The full name of the Investigator in format FIRST LAST.""", json_schema_extra = { "linkml_meta": {'alias': 'has_raw_value',
         'domain_of': ['AttributeValue'],
         'notes': ['May eventually be deprecated in favor of "name".']} })
    type: Literal["https://w3id.org/nmdc/PersonValue","nmdc:PersonValue"] = Field(default="nmdc:PersonValue", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('websites')
    def pattern_websites(cls, v):
        pattern=re.compile(r"^[Hh][Tt][Tt][Pp][Ss]?:\/\/(?!.*[Dd][Oo][Ii]\.[Oo][Rr][Gg]).*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid websites format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid websites format: {v}"
            raise ValueError(err_msg)
        return v


class TextValue(AttributeValue):
    """
    A basic string value
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:TextValue', 'from_schema': 'https://w3id.org/nmdc/nmdc'})

    language: Optional[str] = Field(default=None, description="""Should use ISO 639-1 code e.g. \"en\", \"fr\"""", json_schema_extra = { "linkml_meta": {'alias': 'language', 'domain_of': ['TextValue']} })
    has_raw_value: Optional[str] = Field(default=None, description="""The value that was specified for an annotation in raw form, i.e. a string. E.g. \"2 cm\" or \"2-4 cm\"""", json_schema_extra = { "linkml_meta": {'alias': 'has_raw_value', 'domain_of': ['AttributeValue']} })
    type: Literal["https://w3id.org/nmdc/TextValue","nmdc:TextValue"] = Field(default="nmdc:TextValue", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })


class TimestampValue(AttributeValue):
    """
    A value that is a timestamp. The range should be ISO-8601
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:TimestampValue',
         'from_schema': 'https://w3id.org/nmdc/nmdc'})

    has_raw_value: Optional[str] = Field(default=None, description="""The value that was specified for an annotation in raw form, i.e. a string. E.g. \"2 cm\" or \"2-4 cm\"""", json_schema_extra = { "linkml_meta": {'alias': 'has_raw_value', 'domain_of': ['AttributeValue']} })
    type: Literal["https://w3id.org/nmdc/TimestampValue","nmdc:TimestampValue"] = Field(default="nmdc:TimestampValue", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })


class ControlledTermValue(AttributeValue):
    """
    A controlled term or class from an ontology
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:ControlledTermValue',
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'todos': ['add fields for ontology, branch']})

    term: Optional[Union[OntologyClass,EnvironmentalMaterialTerm,ChemicalEntity,FunctionalAnnotationTerm,Pathway,OrthologyGroup]] = Field(default=None, description="""pointer to an ontology class""", json_schema_extra = { "linkml_meta": {'alias': 'term', 'domain_of': ['ControlledTermValue']} })
    has_raw_value: Optional[str] = Field(default=None, description="""The value that was specified for an annotation in raw form, i.e. a string. E.g. \"2 cm\" or \"2-4 cm\"""", json_schema_extra = { "linkml_meta": {'alias': 'has_raw_value', 'domain_of': ['AttributeValue']} })
    type: Literal["https://w3id.org/nmdc/ControlledTermValue","nmdc:ControlledTermValue"] = Field(default="nmdc:ControlledTermValue", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })


class ControlledIdentifiedTermValue(ControlledTermValue):
    """
    A controlled term or class from an ontology, requiring the presence of term with an id
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:ControlledIdentifiedTermValue',
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'slot_usage': {'term': {'name': 'term', 'required': True}}})

    term: Union[OntologyClass,EnvironmentalMaterialTerm,ChemicalEntity,FunctionalAnnotationTerm,Pathway,OrthologyGroup] = Field(default=..., description="""pointer to an ontology class""", json_schema_extra = { "linkml_meta": {'alias': 'term', 'domain_of': ['ControlledTermValue']} })
    has_raw_value: Optional[str] = Field(default=None, description="""The value that was specified for an annotation in raw form, i.e. a string. E.g. \"2 cm\" or \"2-4 cm\"""", json_schema_extra = { "linkml_meta": {'alias': 'has_raw_value', 'domain_of': ['AttributeValue']} })
    type: Literal["https://w3id.org/nmdc/ControlledIdentifiedTermValue","nmdc:ControlledIdentifiedTermValue"] = Field(default="nmdc:ControlledIdentifiedTermValue", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })


class GeolocationValue(AttributeValue):
    """
    A normalized value for a location on the earth's surface
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:GeolocationValue',
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'mappings': ['schema:GeoCoordinates'],
         'slot_usage': {'has_raw_value': {'description': 'The raw value for a '
                                                         'geolocation should follow '
                                                         '{latitude} {longitude}',
                                          'name': 'has_raw_value'},
                        'latitude': {'name': 'latitude', 'required': True},
                        'longitude': {'name': 'longitude', 'required': True}}})

    latitude: float = Field(default=..., description="""The latitude of a location.""", json_schema_extra = { "linkml_meta": {'alias': 'latitude',
         'alt_descriptions': {'wikipedia': {'description': 'A geographic coordinate '
                                                           'that specifies the '
                                                           'north-south position of a '
                                                           'point on the surface of '
                                                           'the Earth or another '
                                                           'celestial body.',
                                            'source': 'wikipedia'}},
         'domain_of': ['GeolocationValue'],
         'examples': [{'value': '-33.460524'}],
         'mappings': ['schema:latitude'],
         'slot_uri': 'wgs84:lat'} })
    longitude: float = Field(default=..., description="""The longitude of a location.""", json_schema_extra = { "linkml_meta": {'alias': 'longitude',
         'alt_descriptions': {'wikipedia': {'description': 'A geographic coordinate '
                                                           'that specifies the '
                                                           'east-west position of a '
                                                           'point on the surface of '
                                                           'the Earth, or another '
                                                           'celestial body.',
                                            'source': 'wikipedia'}},
         'domain_of': ['GeolocationValue'],
         'examples': [{'value': '150.168149'}],
         'mappings': ['schema:longitude'],
         'slot_uri': 'wgs84:long'} })
    has_raw_value: Optional[str] = Field(default=None, description="""The raw value for a geolocation should follow {latitude} {longitude}""", json_schema_extra = { "linkml_meta": {'alias': 'has_raw_value', 'domain_of': ['AttributeValue']} })
    type: Literal["https://w3id.org/nmdc/GeolocationValue","nmdc:GeolocationValue"] = Field(default="nmdc:GeolocationValue", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })


class NamedThing(ConfiguredBaseModel):
    """
    a databased entity or concept/class
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'abstract': True,
         'class_uri': 'nmdc:NamedThing',
         'from_schema': 'https://w3id.org/nmdc/nmdc'})

    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/NamedThing","nmdc:NamedThing"] = Field(default="nmdc:NamedThing", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class GeneProduct(NamedThing):
    """
    A molecule encoded by a gene that has an evolved function
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:GeneProduct',
         'exact_mappings': ['biolink:GeneProduct'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'id_prefixes': ['PR', 'UniProtKB', 'gtpo'],
         'notes': ['we may include a more general gene product class in future to '
                   'allow for ncRNA annotation']})

    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/GeneProduct","nmdc:GeneProduct"] = Field(default="nmdc:GeneProduct", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class OntologyClass(NamedThing):
    """
    A representation of class defined in an external ontology.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:OntologyClass',
         'exact_mappings': ['biolink:OntologyClass', 'owl:Class', 'schema:Class'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'slot_usage': {'id': {'name': 'id',
                               'notes': ['The identifiers for terms from external '
                                         "ontologies can't have their ids constrained "
                                         'to the nmdc namespace'],
                               'pattern': '^[a-zA-Z0-9][a-zA-Z0-9_\\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\\-\\/\\.,]*$'}}})

    alternative_names: Optional[list[str]] = Field(default=None, description="""A list of alternative names used to refer to the entity. The distinction between name and alternative names is application-specific.  This should not be used for identifers which have their own slots (e.g., bioproject:PRJNA406974)""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_names',
         'domain_of': ['Biosample', 'OntologyClass', 'Study'],
         'exact_mappings': ['dcterms:alternative', 'skos:altLabel']} })
    relations: Optional[list[OntologyRelation]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'relations', 'domain_of': ['OntologyClass']} })
    definition: Optional[str] = Field(default=None, description="""The definition of the ontology term as provided by the ontology.""", json_schema_extra = { "linkml_meta": {'alias': 'definition', 'domain_of': ['OntologyClass']} })
    is_obsolete: Optional[bool] = Field(default=None, description="""A boolean value indicating whether the ontology term is obsolete.""", json_schema_extra = { "linkml_meta": {'alias': 'is_obsolete',
         'comments': ['If true (the ontology term is declared obsolete via the '
                      'ontology source itself),  the term is no longer considered a '
                      'valid term to use in an annotation at NMDC, and it no longer '
                      'has ontology_relation_set records.'],
         'domain_of': ['OntologyClass']} })
    is_root: Optional[bool] = Field(default=None, description="""A boolean value indicating whether the ontology term is a root term; it is not a subclass of  any other term.""", json_schema_extra = { "linkml_meta": {'alias': 'is_root', 'domain_of': ['OntologyClass']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ["The identifiers for terms from external ontologies can't have "
                   'their ids constrained to the nmdc namespace'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/OntologyClass","nmdc:OntologyClass"] = Field(default="nmdc:OntologyClass", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class EnvironmentalMaterialTerm(OntologyClass):
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:EnvironmentalMaterialTerm',
         'deprecated': 'not used. 2024-07-10 '
                       'https://github.com/microbiomedata/nmdc-schema/issues/1881',
         'from_schema': 'https://w3id.org/nmdc/nmdc'})

    alternative_names: Optional[list[str]] = Field(default=None, description="""A list of alternative names used to refer to the entity. The distinction between name and alternative names is application-specific.  This should not be used for identifers which have their own slots (e.g., bioproject:PRJNA406974)""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_names',
         'domain_of': ['Biosample', 'OntologyClass', 'Study'],
         'exact_mappings': ['dcterms:alternative', 'skos:altLabel']} })
    relations: Optional[list[OntologyRelation]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'relations', 'domain_of': ['OntologyClass']} })
    definition: Optional[str] = Field(default=None, description="""The definition of the ontology term as provided by the ontology.""", json_schema_extra = { "linkml_meta": {'alias': 'definition', 'domain_of': ['OntologyClass']} })
    is_obsolete: Optional[bool] = Field(default=None, description="""A boolean value indicating whether the ontology term is obsolete.""", json_schema_extra = { "linkml_meta": {'alias': 'is_obsolete',
         'comments': ['If true (the ontology term is declared obsolete via the '
                      'ontology source itself),  the term is no longer considered a '
                      'valid term to use in an annotation at NMDC, and it no longer '
                      'has ontology_relation_set records.'],
         'domain_of': ['OntologyClass']} })
    is_root: Optional[bool] = Field(default=None, description="""A boolean value indicating whether the ontology term is a root term; it is not a subclass of  any other term.""", json_schema_extra = { "linkml_meta": {'alias': 'is_root', 'domain_of': ['OntologyClass']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ["The identifiers for terms from external ontologies can't have "
                   'their ids constrained to the nmdc namespace'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/EnvironmentalMaterialTerm","nmdc:EnvironmentalMaterialTerm"] = Field(default="nmdc:EnvironmentalMaterialTerm", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class ChemicalEntity(OntologyClass):
    """
    An atom or molecule that can be represented with a chemical formula. Include lipids, glycans, natural products, drugs. There may be different terms for distinct acid-base forms, protonation states. A chemical entity is a  physical entity that pertains to chemistry or biochemistry.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'aliases': ['metabolite',
                     'chemical substance',
                     'chemical compound',
                     'chemical'],
         'class_uri': 'nmdc:ChemicalEntity',
         'comments': ['As with the parent OntologyClass, we will not assign an nmdc id '
                      'pattern or typecode to this class.'],
         'deprecated': 'true; as of Jan 2025, NMDC only needs a handful of chemicals '
                       'and its use cases can be served via an enumeration rather than '
                       'supporting a full class.',
         'exact_mappings': ['biolink:ChemicalSubstance'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'id_prefixes': ['CHEBI', 'MS'],
         'see_also': ['https://bioconductor.org/packages/devel/data/annotation/vignettes/metaboliteIDmapping/inst/doc/metaboliteIDmapping.html']})

    chemical_formula: Optional[str] = Field(default=None, description="""A generic grouping for molecular formulae and empirical formulae""", json_schema_extra = { "linkml_meta": {'alias': 'chemical_formula', 'domain_of': ['ChemicalEntity']} })
    alternative_names: Optional[list[str]] = Field(default=None, description="""A list of alternative names used to refer to the entity. The distinction between name and alternative names is application-specific.  This should not be used for identifers which have their own slots (e.g., bioproject:PRJNA406974)""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_names',
         'domain_of': ['Biosample', 'OntologyClass', 'Study'],
         'exact_mappings': ['dcterms:alternative', 'skos:altLabel']} })
    relations: Optional[list[OntologyRelation]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'relations', 'domain_of': ['OntologyClass']} })
    definition: Optional[str] = Field(default=None, description="""The definition of the ontology term as provided by the ontology.""", json_schema_extra = { "linkml_meta": {'alias': 'definition', 'domain_of': ['OntologyClass']} })
    is_obsolete: Optional[bool] = Field(default=None, description="""A boolean value indicating whether the ontology term is obsolete.""", json_schema_extra = { "linkml_meta": {'alias': 'is_obsolete',
         'comments': ['If true (the ontology term is declared obsolete via the '
                      'ontology source itself),  the term is no longer considered a '
                      'valid term to use in an annotation at NMDC, and it no longer '
                      'has ontology_relation_set records.'],
         'domain_of': ['OntologyClass']} })
    is_root: Optional[bool] = Field(default=None, description="""A boolean value indicating whether the ontology term is a root term; it is not a subclass of  any other term.""", json_schema_extra = { "linkml_meta": {'alias': 'is_root', 'domain_of': ['OntologyClass']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ["The identifiers for terms from external ontologies can't have "
                   'their ids constrained to the nmdc namespace'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/ChemicalEntity","nmdc:ChemicalEntity"] = Field(default="nmdc:ChemicalEntity", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class FunctionalAnnotationTerm(OntologyClass):
    """
    Abstract grouping class for any term/descriptor that can be applied to a functional unit of a genome (protein, ncRNA, complex).
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'abstract': True,
         'aliases': ['function', 'FunctionalAnnotation'],
         'class_uri': 'nmdc:FunctionalAnnotationTerm',
         'from_schema': 'https://w3id.org/nmdc/nmdc'})

    alternative_names: Optional[list[str]] = Field(default=None, description="""A list of alternative names used to refer to the entity. The distinction between name and alternative names is application-specific.  This should not be used for identifers which have their own slots (e.g., bioproject:PRJNA406974)""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_names',
         'domain_of': ['Biosample', 'OntologyClass', 'Study'],
         'exact_mappings': ['dcterms:alternative', 'skos:altLabel']} })
    relations: Optional[list[OntologyRelation]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'relations', 'domain_of': ['OntologyClass']} })
    definition: Optional[str] = Field(default=None, description="""The definition of the ontology term as provided by the ontology.""", json_schema_extra = { "linkml_meta": {'alias': 'definition', 'domain_of': ['OntologyClass']} })
    is_obsolete: Optional[bool] = Field(default=None, description="""A boolean value indicating whether the ontology term is obsolete.""", json_schema_extra = { "linkml_meta": {'alias': 'is_obsolete',
         'comments': ['If true (the ontology term is declared obsolete via the '
                      'ontology source itself),  the term is no longer considered a '
                      'valid term to use in an annotation at NMDC, and it no longer '
                      'has ontology_relation_set records.'],
         'domain_of': ['OntologyClass']} })
    is_root: Optional[bool] = Field(default=None, description="""A boolean value indicating whether the ontology term is a root term; it is not a subclass of  any other term.""", json_schema_extra = { "linkml_meta": {'alias': 'is_root', 'domain_of': ['OntologyClass']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ["The identifiers for terms from external ontologies can't have "
                   'their ids constrained to the nmdc namespace'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/FunctionalAnnotationTerm","nmdc:FunctionalAnnotationTerm"] = Field(default="nmdc:FunctionalAnnotationTerm", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class Pathway(FunctionalAnnotationTerm):
    """
    A pathway is a sequence of steps/reactions carried out by an organism or community of organisms
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'aliases': ['biological process', 'metabolic pathway', 'signaling pathway'],
         'class_uri': 'nmdc:Pathway',
         'deprecated': 'not used. 2024-07-10 '
                       'https://github.com/microbiomedata/nmdc-schema/issues/1881',
         'exact_mappings': ['biolink:Pathway'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'id_prefixes': ['KEGG_PATHWAY', 'COG'],
         'todos': ['If we reverted to including Reaction in the schema, then a '
                   'Reaction would be a reasonable part_of a Pathway',
                   "is Pathway instantiated in an MongoDB collection? Aren't Pathways "
                   'searchable in the Data Portal?']})

    alternative_names: Optional[list[str]] = Field(default=None, description="""A list of alternative names used to refer to the entity. The distinction between name and alternative names is application-specific.  This should not be used for identifers which have their own slots (e.g., bioproject:PRJNA406974)""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_names',
         'domain_of': ['Biosample', 'OntologyClass', 'Study'],
         'exact_mappings': ['dcterms:alternative', 'skos:altLabel']} })
    relations: Optional[list[OntologyRelation]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'relations', 'domain_of': ['OntologyClass']} })
    definition: Optional[str] = Field(default=None, description="""The definition of the ontology term as provided by the ontology.""", json_schema_extra = { "linkml_meta": {'alias': 'definition', 'domain_of': ['OntologyClass']} })
    is_obsolete: Optional[bool] = Field(default=None, description="""A boolean value indicating whether the ontology term is obsolete.""", json_schema_extra = { "linkml_meta": {'alias': 'is_obsolete',
         'comments': ['If true (the ontology term is declared obsolete via the '
                      'ontology source itself),  the term is no longer considered a '
                      'valid term to use in an annotation at NMDC, and it no longer '
                      'has ontology_relation_set records.'],
         'domain_of': ['OntologyClass']} })
    is_root: Optional[bool] = Field(default=None, description="""A boolean value indicating whether the ontology term is a root term; it is not a subclass of  any other term.""", json_schema_extra = { "linkml_meta": {'alias': 'is_root', 'domain_of': ['OntologyClass']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ["The identifiers for terms from external ontologies can't have "
                   'their ids constrained to the nmdc namespace'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/Pathway","nmdc:Pathway"] = Field(default="nmdc:Pathway", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class OrthologyGroup(FunctionalAnnotationTerm):
    """
    A set of genes or gene products in which all members are orthologous
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:OrthologyGroup',
         'exact_mappings': ['biolink:GeneFamily'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'id_prefixes': ['CATH',
                         'EGGNOG',
                         'KEGG.ORTHOLOGY',
                         'PANTHER.FAMILY',
                         'PFAM',
                         'SUPFAM',
                         'TIGRFAM'],
         'notes': ['KEGG.ORTHOLOGY prefix is used for KO numbers'],
         'todos': ["is OrthologyGroup instantiated in an MongoDB collection? Aren't "
                   'Pathways searchable in the Data Portal?']})

    alternative_names: Optional[list[str]] = Field(default=None, description="""A list of alternative names used to refer to the entity. The distinction between name and alternative names is application-specific.  This should not be used for identifers which have their own slots (e.g., bioproject:PRJNA406974)""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_names',
         'domain_of': ['Biosample', 'OntologyClass', 'Study'],
         'exact_mappings': ['dcterms:alternative', 'skos:altLabel']} })
    relations: Optional[list[OntologyRelation]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'relations', 'domain_of': ['OntologyClass']} })
    definition: Optional[str] = Field(default=None, description="""The definition of the ontology term as provided by the ontology.""", json_schema_extra = { "linkml_meta": {'alias': 'definition', 'domain_of': ['OntologyClass']} })
    is_obsolete: Optional[bool] = Field(default=None, description="""A boolean value indicating whether the ontology term is obsolete.""", json_schema_extra = { "linkml_meta": {'alias': 'is_obsolete',
         'comments': ['If true (the ontology term is declared obsolete via the '
                      'ontology source itself),  the term is no longer considered a '
                      'valid term to use in an annotation at NMDC, and it no longer '
                      'has ontology_relation_set records.'],
         'domain_of': ['OntologyClass']} })
    is_root: Optional[bool] = Field(default=None, description="""A boolean value indicating whether the ontology term is a root term; it is not a subclass of  any other term.""", json_schema_extra = { "linkml_meta": {'alias': 'is_root', 'domain_of': ['OntologyClass']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ["The identifiers for terms from external ontologies can't have "
                   'their ids constrained to the nmdc namespace'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/OrthologyGroup","nmdc:OrthologyGroup"] = Field(default="nmdc:OrthologyGroup", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class OntologyRelation(ConfiguredBaseModel):
    """
    A relationship between two ontology classes as specified either directly in the ontology in the form of axioms (statements or assertions that defines rules or constraints in an ontology) or inferred via reasoning.  The association object is defined by two terms (the subject and the object) and the relationship between them (the predicate). Because ontologies often have a plethora of relationships/axiom types and can have additional metadata on the relationship itself, these kinds of relationships are structured as a class instead of a simple set of slots on OntologyClass itself.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'aliases': ['OntologyAssociation'],
         'class_uri': 'nmdc:OntologyRelation',
         'comments': ["For example, the relationship between 'soil' and 'enriched "
                      "soil' in the Environmental Ontology (ENVO) is defined by the "
                      "following axioms: 'enriched soil' subclass_of 'soil', and "
                      "'enriched soil'  has_increased_levels_of (some) 'material "
                      "entity.'  Converting these stat∂ements to OntologyAssociations  "
                      'so they can be used in the NMDC data stores, the subject of the '
                      "first axiom or statement,  would be 'soil', the predicate would "
                      "be 'subclass_of', and the object would be 'enriched soil'.  For "
                      "the second axiom, the subject would be 'enriched soil', the "
                      "predicate would be 'has_increased_levels_of', and the object "
                      "would be 'material entity.' (Note that text labels are used in "
                      'this example for ease of understanding and  the literal values '
                      'of subject, predicate, and object in this class will be the id '
                      '(curie) of the  `OntologyClass` as defined below).  Not all '
                      'ontology axioms (associations) between terms need to be  '
                      'ingested into the NMDC data stores.  In general, subclass_of '
                      'and part_of relationships/axioms are often  good default '
                      'relations/associations to support ontology browsing in user '
                      'interfaces.'],
         'from_schema': 'https://w3id.org/nmdc/nmdc'})

    type: Literal["https://w3id.org/nmdc/OntologyRelation","nmdc:OntologyRelation"] = Field(default="nmdc:OntologyRelation", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })
    subject: str = Field(default=..., json_schema_extra = { "linkml_meta": {'alias': 'subject', 'domain_of': ['FunctionalAnnotation', 'OntologyRelation']} })
    predicate: str = Field(default=..., json_schema_extra = { "linkml_meta": {'alias': 'predicate', 'domain_of': ['OntologyRelation']} })
    object: str = Field(default=..., json_schema_extra = { "linkml_meta": {'alias': 'object', 'domain_of': ['OntologyRelation']} })


class FailureCategorization(ConfiguredBaseModel):
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:FailureCategorization',
         'from_schema': 'https://w3id.org/nmdc/nmdc'})

    qc_failure_what: Optional[FailureWhatEnum] = Field(default=None, description="""Provides a summary about what caused a lab or workflow process to fail""", json_schema_extra = { "linkml_meta": {'alias': 'qc_failure_what',
         'comments': ['For example, low read count from a sequencer, malformed fastq '
                      'files, etc.'],
         'domain_of': ['FailureCategorization']} })
    qc_failure_where: Optional[FailureWhereEnum] = Field(default=None, description="""Describes the nmdc schema class that corresonds to where the failure occurred. Most commonly this would be the same as Class that generated the results.""", json_schema_extra = { "linkml_meta": {'alias': 'qc_failure_where',
         'comments': ['If the assembly size was too small to proceed to annotation '
                      'failure_where would be MetagenomeAssembly.'],
         'domain_of': ['FailureCategorization']} })
    type: Literal["https://w3id.org/nmdc/FailureCategorization","nmdc:FailureCategorization"] = Field(default="nmdc:FailureCategorization", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })


class MaterialEntity(NamedThing):
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'abstract': True,
         'aliases': ['Material', 'Physical entity'],
         'class_uri': 'nmdc:MaterialEntity',
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'title': 'Material Entity'})

    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/MaterialEntity","nmdc:MaterialEntity"] = Field(default="nmdc:MaterialEntity", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class Sample(MaterialEntity):
    """
    A sample is a material entity that can be characterized by an experiment.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'abstract': True,
         'class_uri': 'nmdc:Sample',
         'from_schema': 'https://w3id.org/nmdc/nmdc'})

    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/Sample","nmdc:Sample"] = Field(default="nmdc:Sample", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class Biosample(Sample):
    """
    Biological source material which can be characterized by an experiment.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'aliases': ['sample', 'material sample', 'specimen', 'biospecimen'],
         'alt_descriptions': {'embl.ena': {'description': 'A sample contains '
                                                          'information about the '
                                                          'sequenced source material. '
                                                          'Samples are associated with '
                                                          'checklists, which define '
                                                          'the fields used to annotate '
                                                          'the samples. Samples are '
                                                          'always associated with a '
                                                          'taxon.',
                                           'source': 'embl.ena'}},
         'class_uri': 'nmdc:Biosample',
         'exact_mappings': ['OBI:0000747',
                            'NCIT:C43412',
                            'http://purl.obolibrary.org/obo/FBcv_0003024'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'notes': ["could add GOLD and EBI's biosample definitions to the "
                   'alt_descriptions?'],
         'rules': [{'description': 'DNA samples shipped to JGI for metagenomic '
                                   "analysis in tubes can't have any value for their "
                                   'plate position.',
                    'postconditions': {'slot_conditions': {'dna_cont_type': {'equals_string': 'plate',
                                                                             'name': 'dna_cont_type'}}},
                    'preconditions': {'slot_conditions': {'dna_cont_well': {'name': 'dna_cont_well',
                                                                            'pattern': '.+'}}},
                    'title': 'dna_well_requires_plate'},
                   {'description': 'DNA samples in plates must have a plate position '
                                   'that matches the regex. Note the requirement for '
                                   'an empty string in the tube case. Waiting for '
                                   'value_present validation to be added to runtime',
                    'postconditions': {'slot_conditions': {'dna_cont_well': {'name': 'dna_cont_well',
                                                                             'pattern': '^(?!A1$|A12$|H1$|H12$)(([A-H][1-9])|([A-H]1[0-2]))$'}}},
                    'preconditions': {'slot_conditions': {'dna_cont_type': {'equals_string': 'plate',
                                                                            'name': 'dna_cont_type'}}},
                    'title': 'dna_plate_requires_well'},
                   {'description': 'RNA samples shipped to JGI for metagenomic '
                                   "analysis in tubes can't have any value for their "
                                   'plate position.',
                    'postconditions': {'slot_conditions': {'rna_cont_type': {'equals_string': 'plate',
                                                                             'name': 'rna_cont_type'}}},
                    'preconditions': {'slot_conditions': {'rna_cont_well': {'name': 'rna_cont_well',
                                                                            'pattern': '.+'}}},
                    'title': 'rna_well_requires_plate'},
                   {'description': 'RNA samples in plates must have a plate position '
                                   'that matches the regex. Note the requirement for '
                                   'an empty string in the tube case. Waiting for '
                                   'value_present validation to be added to runtime',
                    'postconditions': {'slot_conditions': {'rna_cont_well': {'name': 'rna_cont_well',
                                                                             'pattern': '^(?!A1$|A12$|H1$|H12$)(([A-H][1-9])|([A-H]1[0-2]))$'}}},
                    'preconditions': {'slot_conditions': {'rna_cont_type': {'equals_string': 'plate',
                                                                            'name': 'rna_cont_type'}}},
                    'title': 'rna_plate_requires_well'}],
         'slot_usage': {'al_sat': {'description': 'The relative abundance of aluminum '
                                                  'in the sample',
                                   'examples': [{'value': '27%'}],
                                   'name': 'al_sat',
                                   'notes': ['Aluminum saturation is the percentage of '
                                             'the CEC occupies by aluminum. Like all '
                                             'cations, aluminum held by the cation '
                                             'exchange complex is in equilibrium with '
                                             'aluminum in the soil solution.'],
                                   'title': 'aluminum saturation/ extreme unusual '
                                            'properties',
                                   'todos': ['Example & validation. Can we configure '
                                             'things so that 27% & 27 % & 0.27 will '
                                             'validate?',
                                             "I think it's weird the way GSC writes "
                                             'the title. I recommend this change. '
                                             "Thoughts? I would argue this isn't an "
                                             "extreme unusual property. It's just a "
                                             'biogeochemical measurement.']},
                        'al_sat_meth': {'comments': ['Required when aluminum '
                                                     'saturation is provided.'],
                                        'description': 'Reference or method used in '
                                                       'determining Aluminum '
                                                       'saturation',
                                        'examples': [{'value': 'https://doi.org/10.1371/journal.pone.0176357'}],
                                        'name': 'al_sat_meth',
                                        'title': 'aluminum saturation method/ extreme '
                                                 'unusual properties',
                                        'todos': ["I think it's weird the way GSC "
                                                  'writes the title. I recommend this '
                                                  'change. Thoughts?']},
                        'alternative_identifiers': {'description': 'Unique identifier '
                                                                   'for a biosample '
                                                                   'submitted to '
                                                                   'additional '
                                                                   'resources. Matches '
                                                                   'the entity that '
                                                                   'has been submitted '
                                                                   'to NMDC',
                                                    'name': 'alternative_identifiers'},
                        'annual_precpt': {'examples': [{'value': '8.94 inch'}],
                                          'name': 'annual_precpt'},
                        'associated_studies': {'name': 'associated_studies',
                                               'pattern': '^(nmdc):sty-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                               'range': 'Study',
                                               'required': True,
                                               'structured_pattern': {'interpolated': True,
                                                                      'syntax': '{id_nmdc_prefix}:sty-{id_shoulder}-{id_blade}$'}},
                        'climate_environment': {'name': 'climate_environment',
                                                'todos': ['description says "can '
                                                          'include multiple climates" '
                                                          'but multivalued is set to '
                                                          'false',
                                                          'add examples, i need to see '
                                                          'some examples to add '
                                                          'correctly formatted '
                                                          'example.']},
                        'collected_from': {'name': 'collected_from',
                                           'pattern': '^(nmdc):frsite-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                           'structured_pattern': {'interpolated': True,
                                                                  'syntax': '{id_nmdc_prefix}:frsite-{id_shoulder}-{id_blade}$'}},
                        'cur_vegetation': {'comments': ['Values provided here can be '
                                                        'specific species of '
                                                        'vegetation or vegetation '
                                                        'regions',
                                                        'See for vegetation regions- '
                                                        'https://education.nationalgeographic.org/resource/vegetation-region'],
                                           'description': 'Vegetation classification '
                                                          'from one or more standard '
                                                          'classification systems, or '
                                                          'agricultural crop',
                                           'examples': [{'value': 'deciduous forest'},
                                                        {'value': 'forest'},
                                                        {'value': 'Bauhinia '
                                                                  'variegata'}],
                                           'name': 'cur_vegetation',
                                           'todos': ['Recommend changing this from '
                                                     'text value to some king of '
                                                     'ontology?']},
                        'cur_vegetation_meth': {'comments': ['Required when current '
                                                             'vegetation is provided.'],
                                                'examples': [{'value': 'https://doi.org/10.1111/j.1654-109X.2011.01154.x'}],
                                                'name': 'cur_vegetation_meth',
                                                'todos': ["I'm not sure this is a DOI, "
                                                          'PMID, or URI. Should pool '
                                                          'the community and find out '
                                                          'how they accomplish this if '
                                                          'provided.']},
                        'elev': {'comments': ['All elevations must be reported in '
                                              'meters. Provide the numerical portion '
                                              'only.',
                                              'Please use '
                                              'https://www.advancedconverter.com/map-tools/find-altitude-by-coordinates, '
                                              'if needed, to help estimate the '
                                              'elevation based on latitude and '
                                              'longitude coordinates.'],
                                 'examples': [{'value': '100'}],
                                 'name': 'elev',
                                 'range': 'float',
                                 'title': 'elevation, meters'},
                        'env_broad_scale': {'name': 'env_broad_scale',
                                            'required': True},
                        'env_local_scale': {'name': 'env_local_scale',
                                            'required': True},
                        'env_medium': {'name': 'env_medium', 'required': True},
                        'extreme_event': {'annotations': {'expected_value': {'tag': 'expected_value',
                                                                             'value': 'date, '
                                                                                      'string'}},
                                          'examples': [{'value': '1980-05-18, volcanic '
                                                                 'eruption'}],
                                          'name': 'extreme_event',
                                          'range': 'string'},
                        'fire': {'annotations': {'expected_value': {'tag': 'expected_value',
                                                                    'value': 'date '
                                                                             'string'}},
                                 'comments': ['Provide the date the fire occurred. If '
                                              'extended burning occurred provide the '
                                              'date range.'],
                                 'examples': [{'value': '1871-10-10'},
                                              {'value': '1871-10-01 to 1871-10-31'}],
                                 'name': 'fire',
                                 'pattern': '^[12]\\d{3}(?:(?:-(?:0[1-9]|1[0-2]))(?:-(?:0[1-9]|[12]\\d|3[01]))?)?(\\s+to\\s+[12]\\d{3}(?:(?:-(?:0[1-9]|1[0-2]))(?:-(?:0[1-9]|[12]\\d|3[01]))?)?)?$',
                                 'range': 'string',
                                 'todos': ['is "to" acceptable? Is there a better way '
                                           'to request that be written?']},
                        'flooding': {'annotations': {'expected_value': {'tag': 'expected_value',
                                                                        'value': 'date '
                                                                                 'string'}},
                                     'comments': ['Provide the date the flood '
                                                  'occurred. If extended flooding '
                                                  'occurred provide the date range.'],
                                     'examples': [{'value': '1927-04-15'},
                                                  {'value': '1927-04 to 1927-05'}],
                                     'name': 'flooding',
                                     'range': 'string',
                                     'todos': ['is "to" acceptable? Is there a better '
                                               'way to request that be written?',
                                               'What about if the "day" isn\'t known? '
                                               'Is this ok?']},
                        'gaseous_environment': {'examples': [{'value': 'CO2; 500ppm '
                                                                       'above ambient; '
                                                                       'constant'},
                                                             {'value': 'nitric '
                                                                       'oxide;0.5 '
                                                                       'micromole per '
                                                                       'liter;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
                                                'name': 'gaseous_environment',
                                                'todos': ['would like to see usage '
                                                          'examples for this slot. '
                                                          'Requiring micromole/L seems '
                                                          "too limiting and doesn't "
                                                          'match expected_value value',
                                                          'did I do this right? keep '
                                                          "the example that's provided "
                                                          'and add another? so as to '
                                                          'not override']},
                        'gold_biosample_identifiers': {'annotations': {'tooltip': {'tag': 'tooltip',
                                                                                   'value': 'Provide '
                                                                                            'the '
                                                                                            'GOLD '
                                                                                            'biosample '
                                                                                            'IDs '
                                                                                            'associated '
                                                                                            'with '
                                                                                            'this '
                                                                                            'biosample.'}},
                                                       'comments': ['This is the ID '
                                                                    'provided by GOLD '
                                                                    'that starts with '
                                                                    "'Gb'"],
                                                       'description': 'Unique '
                                                                      'identifier for '
                                                                      'a biosample '
                                                                      'submitted to '
                                                                      'GOLD that '
                                                                      'matches the '
                                                                      'NMDC submitted '
                                                                      'biosample',
                                                       'name': 'gold_biosample_identifiers'},
                        'heavy_metals': {'comments': ['For multiple heavy metals and '
                                                      'concentrations, separate by ;'],
                                         'description': 'Heavy metals present in the '
                                                        'sample and their '
                                                        'concentrations.',
                                         'examples': [{'value': 'mercury 0.09 '
                                                                'micrograms per gram'},
                                                      {'value': 'mercury 0.09 ug/g; '
                                                                'chromium 0.03 ug/g'}],
                                         'name': 'heavy_metals',
                                         'title': 'heavy metals/ extreme unusual '
                                                  'properties',
                                         'todos': ['Example & validation. Can we '
                                                   'configure things so that 27% & 27 '
                                                   '% & 0.27 will validate?',
                                                   "I think it's weird the way GSC "
                                                   'writes the title. I recommend this '
                                                   'change. Thoughts? I would argue '
                                                   "this isn't an extreme unusual "
                                                   "property. It's just a "
                                                   'biogeochemical measurement.']},
                        'heavy_metals_meth': {'comments': ['Required when heavy metals '
                                                           'are provided',
                                                           'If different methods are '
                                                           'used for multiple metals, '
                                                           'indicate the metal and '
                                                           'method. Separate metals by '
                                                           ';'],
                                              'examples': [{'value': 'https://doi.org/10.3390/ijms9040434'},
                                                           {'value': 'mercury '
                                                                     'https://doi.org/10.1007/BF01056090; '
                                                                     'chromium '
                                                                     'https://doi.org/10.1007/s00216-006-0322-8'}],
                                              'multivalued': True,
                                              'name': 'heavy_metals_meth',
                                              'title': 'heavy metals method/ extreme '
                                                       'unusual properties'},
                        'id': {'description': 'An NMDC assigned unique identifier for '
                                              'a biosample submitted to NMDC.',
                               'name': 'id',
                               'pattern': '^(nmdc):bsm-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                               'required': True,
                               'structured_pattern': {'interpolated': True,
                                                      'syntax': '{id_nmdc_prefix}:bsm-{id_shoulder}-{id_blade}$'}},
                        'lat_lon': {'name': 'lat_lon',
                                    'notes': ['This is currently a required field but '
                                              "it's not clear if this should be "
                                              'required for human hosts']},
                        'ph_meth': {'comments': ['This can include a link to the '
                                                 'instrument used or a citation for '
                                                 'the method.'],
                                    'examples': [{'value': 'https://www.southernlabware.com/pc9500-benchtop-ph-conductivity-meter-kit-ph-accuracy-2000mv-ph-range-2-000-to-20-000.html?gclid=Cj0KCQiAwJWdBhCYARIsAJc4idCO5vtvbVMf545fcvdROFqa6zjzNSoywNx6K4k9Coo9cCc2pybtvGsaAiR0EALw_wcB'},
                                                 {'value': 'https://doi.org/10.2136/sssabookser5.3.c16'}],
                                    'name': 'ph_meth'},
                        'salinity_meth': {'examples': [{'value': 'https://doi.org/10.1007/978-1-61779-986-0_28'}],
                                          'name': 'salinity_meth'},
                        'season_precpt': {'comments': ['Seasons are defined as spring '
                                                       '(March, April, May), summer '
                                                       '(June, July, August), autumn '
                                                       '(September, October, November) '
                                                       'and winter (December, January, '
                                                       'February).'],
                                          'examples': [{'value': '0.4 inch'},
                                                       {'value': '10.16 mm'}],
                                          'name': 'season_precpt',
                                          'notes': ['mean and average are the same '
                                                    'thing, but it seems like bad '
                                                    'practice to not be consistent. '
                                                    'Changed mean to average'],
                                          'title': 'average seasonal precipitation',
                                          'todos': ['check validation & examples. '
                                                    'always mm? so value only? Or '
                                                    'value + unit']},
                        'sieving': {'comments': ['Describe how samples were composited '
                                                 'or sieved.',
                                                 "Use 'sample link' to indicate which "
                                                 'samples were combined.'],
                                    'examples': [{'value': 'combined 2 cores | 4mm '
                                                           'sieved'},
                                                 {'value': '4 mm sieved and '
                                                           'homogenized'},
                                                 {'value': '50 g | 5 cores | 2 mm '
                                                           'sieved'}],
                                    'name': 'sieving',
                                    'todos': ['check validation and examples']},
                        'slope_aspect': {'comments': ['Aspect is the orientation of '
                                                      'slope, measured clockwise in '
                                                      'degrees from 0 to 360, where 0 '
                                                      'is north-facing, 90 is '
                                                      'east-facing, 180 is '
                                                      'south-facing, and 270 is '
                                                      'west-facing.'],
                                         'description': 'The direction a slope faces. '
                                                        'While looking down a slope '
                                                        'use a compass to record the '
                                                        'direction you are facing '
                                                        '(direction or degrees). This '
                                                        'measure provides an '
                                                        'indication of sun and wind '
                                                        'exposure that will influence '
                                                        'soil temperature and '
                                                        'evapotranspiration.',
                                         'examples': [{'value': '35'}],
                                         'name': 'slope_aspect'},
                        'slope_gradient': {'examples': [{'value': '10%'},
                                                        {'value': '10 %'},
                                                        {'value': '0.10'}],
                                           'name': 'slope_gradient',
                                           'todos': ['Slope is a percent. How does the '
                                                     'validation work? Check to '
                                                     'correct examples']},
                        'source_mat_id': {'comments': ['Identifiers must be prefixed. '
                                                       'Possible FAIR prefixes are '
                                                       'IGSNs '
                                                       '(http://www.geosamples.org/getigsn), '
                                                       'NCBI biosample accession '
                                                       'numbers, ARK identifiers '
                                                       '(https://arks.org/). These IDs '
                                                       'enable linking to derived '
                                                       'analytes and subsamples. If '
                                                       'you have not assigned FAIR '
                                                       'identifiers to your samples, '
                                                       'you can generate UUIDs '
                                                       '(https://www.uuidgenerator.net/).'],
                                          'description': 'A globally unique identifier '
                                                         'assigned to the biological '
                                                         'sample.',
                                          'examples': [{'value': 'IGSN:AU1243'},
                                                       {'value': 'UUID:24f1467a-40f4-11ed-b878-0242ac120002'}],
                                          'name': 'source_mat_id',
                                          'title': 'source material identifier',
                                          'todos': ['Currently, the comments say to '
                                                    'use UUIDs. However, if we '
                                                    'implement assigning NMDC '
                                                    'identifiers with the minter we '
                                                    'dont need to require a GUID. It '
                                                    'can be an optional field to fill '
                                                    'out only if they already have a '
                                                    'resolvable ID.']},
                        'tot_carb': {'examples': [{'value': '1 ug/L'}],
                                     'name': 'tot_carb',
                                     'todos': ['is this inorganic and organic? both? '
                                               'could use some clarification.',
                                               "ug/L doesn't seem like the right "
                                               'units. Should check this slots usage '
                                               'in databases and re-evaluate. I '
                                               "couldn't find any references that "
                                               'provided this data in this format']},
                        'tot_nitro_cont_meth': {'examples': [{'value': 'https://doi.org/10.2134/agronmonogr9.2.c32'},
                                                             {'value': 'https://acsess.onlinelibrary.wiley.com/doi/full/10.2136/sssaj2009.0389?casa_token=bm0pYIUdNMgAAAAA%3AOWVRR0STHaOe-afTcTdxn5m1hM8n2ltM0wY-b1iYpYdD9dhwppk5j3LvC2IO5yhOIvyLVeQz4NZRCZo'}],
                                                'name': 'tot_nitro_cont_meth'},
                        'tot_nitro_content': {'examples': [{'value': '5 mg N/ L'}],
                                              'name': 'tot_nitro_content'},
                        'tot_org_c_meth': {'examples': [{'value': 'https://doi.org/10.1080/07352680902776556'}],
                                           'name': 'tot_org_c_meth'},
                        'tot_org_carb': {'examples': [{'value': '5 mg N/ L'}],
                                         'name': 'tot_org_carb',
                                         'todos': ['check description. How are they '
                                                   'different?']},
                        'water_cont_soil_meth': {'comments': ['Required if providing '
                                                              'water content'],
                                                 'examples': [{'value': 'J. Nat. Prod. '
                                                                        'Plant '
                                                                        'Resour., '
                                                                        '2012, 2 '
                                                                        '(4):500-503'},
                                                              {'value': 'https://dec.alaska.gov/applications/spar/webcalc/definitions.htm'}],
                                                 'name': 'water_cont_soil_meth',
                                                 'todos': ['Why is it soil water '
                                                           'content method in the name '
                                                           'but not the title? Is this '
                                                           'slot used in other '
                                                           'samples?',
                                                           'Soil water content can be '
                                                           'measure MANY ways and '
                                                           'often, multiple ways are '
                                                           'used in one experiment '
                                                           '(gravimetric water content '
                                                           'and water holding capacity '
                                                           'and water filled pore '
                                                           'space, to name a few).',
                                                           'Should this be multi '
                                                           'valued? How to we manage '
                                                           'and validate this?']},
                        'water_content': {'annotations': {'expected_value': {'tag': 'expected_value',
                                                                             'value': 'string'},
                                                          'preferred_unit': {'tag': 'preferred_unit',
                                                                             'value': 'gram '
                                                                                      'per '
                                                                                      'gram '
                                                                                      'or '
                                                                                      'cubic '
                                                                                      'centimeter '
                                                                                      'per '
                                                                                      'cubic '
                                                                                      'centimeter'}},
                                          'examples': [{'value': '0.75 g water/g dry '
                                                                 'soil'},
                                                       {'value': '75% water holding '
                                                                 'capacity'},
                                                       {'value': '1.1 g fresh weight/ '
                                                                 'dry weight'},
                                                       {'value': '10% water filled '
                                                                 'pore space'}],
                                          'multivalued': True,
                                          'name': 'water_content',
                                          'range': 'string',
                                          'todos': ['value in preferred unit is too '
                                                    'limiting. need to change this',
                                                    'check and correct validation so '
                                                    'examples are accepted',
                                                    'how to manage multiple water '
                                                    'content methods?']},
                        'watering_regm': {'examples': [{'value': '1 '
                                                                 'liter;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'},
                                                       {'value': '75% water holding '
                                                                 'capacity; constant'}],
                                          'name': 'watering_regm'}},
         'unique_keys': {'samp_name_unique_key': {'unique_key_name': 'samp_name_unique_key',
                                                  'unique_key_slots': ['samp_name']}}})

    associated_studies: list[str] = Field(default=..., description="""The study associated with a resource.""", json_schema_extra = { "linkml_meta": {'alias': 'associated_studies',
         'domain_of': ['Biosample', 'DataGeneration'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:sty-{id_shoulder}-{id_blade}$'}} })
    biosample_categories: Optional[list[BiosampleCategoryEnum]] = Field(default=None, title="Categories the biosample belongs to", json_schema_extra = { "linkml_meta": {'alias': 'biosample_categories', 'domain_of': ['Biosample']} })
    collected_from: Optional[str] = Field(default=None, description="""The Site from which a Biosample was collected""", json_schema_extra = { "linkml_meta": {'alias': 'collected_from',
         'comments': ['this illustrates implementing a Biosample relation with a '
                      '(binary) slot'],
         'domain_of': ['Biosample'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:frsite-{id_shoulder}-{id_blade}$'},
         'todos': ['add an OBO slot_uri ?']} })
    embargoed: Optional[bool] = Field(default=None, description="""If true, the data are embargoed and not available for public access.""", json_schema_extra = { "linkml_meta": {'alias': 'embargoed',
         'domain_of': ['Biosample'],
         'recommended': True,
         'todos': ['make this required?',
                   'first apply to Biosample',
                   'try to apply to all Biosamples in a particular nmdc-server '
                   'SubmissionMetadata?',
                   'applying to a Study may not be granular enough']} })
    host_disease_stat: Optional[TextValue] = Field(default=None, title="host disease status", description="""List of diseases with which the host has been diagnosed; can include multiple diagnoses. The value of the field depends on host; for humans the terms should be chosen from the DO (Human Disease Ontology) at https://www.disease-ontology.org, non-human host diseases are free text""", json_schema_extra = { "linkml_meta": {'alias': 'host_disease_stat',
         'aliases': ['host disease status'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'disease name or Disease Ontology '
                                                     'term'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'rabies [DOID:11260]'}],
         'is_a': 'nucleic acid sequence source field',
         'slot_uri': 'MIXS:0000031',
         'string_serialization': '{termLabel} {[termID]}|{text}'} })
    img_identifiers: Optional[list[str]] = Field(default=None, title="IMG Identifiers", description="""A list of identifiers that relate the biosample to records in the IMG database.""", json_schema_extra = { "linkml_meta": {'alias': 'img_identifiers',
         'domain_of': ['MetagenomeAnnotation',
                       'Biosample',
                       'MetatranscriptomeAnnotation',
                       'MetatranscriptomeExpressionAnalysis',
                       'MagsAnalysis'],
         'is_a': 'external_database_identifiers',
         'todos': ['add is_a or mixin modeling, like other '
                   'external_database_identifiers',
                   'what class would IMG records belong to?! Are they Studies, '
                   'Biosamples, or something else?']} })
    neon_biosample_identifiers: Optional[list[str]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'neon_biosample_identifiers',
         'domain_of': ['Biosample'],
         'is_a': 'biosample_identifiers',
         'mixins': ['neon_identifiers']} })
    alternative_names: Optional[list[str]] = Field(default=None, description="""A list of alternative names used to refer to the entity. The distinction between name and alternative names is application-specific.  This should not be used for identifers which have their own slots (e.g., bioproject:PRJNA406974)""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_names',
         'domain_of': ['Biosample', 'OntologyClass', 'Study'],
         'exact_mappings': ['dcterms:alternative', 'skos:altLabel']} })
    gold_biosample_identifiers: Optional[list[str]] = Field(default=None, description="""Unique identifier for a biosample submitted to GOLD that matches the NMDC submitted biosample""", json_schema_extra = { "linkml_meta": {'alias': 'gold_biosample_identifiers',
         'annotations': {'tooltip': {'tag': 'tooltip',
                                     'value': 'Provide the GOLD biosample IDs '
                                              'associated with this biosample.'}},
         'comments': ["This is the ID provided by GOLD that starts with 'Gb'"],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'gold:Gb0312930'}],
         'is_a': 'biosample_identifiers',
         'mixins': ['gold_identifiers']} })
    insdc_biosample_identifiers: Optional[list[str]] = Field(default=None, description="""identifiers for corresponding sample in INSDC""", json_schema_extra = { "linkml_meta": {'alias': 'insdc_biosample_identifiers',
         'aliases': ['EBI biosample identifiers',
                     'NCBI biosample identifiers',
                     'DDBJ biosample identifiers'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'biosample:SAMEA5989477'},
                      {'description': 'I13_N_5-10 sample from Soil fungal diversity '
                                      'along elevational gradients',
                       'value': 'biosample:SAMD00212331'}],
         'is_a': 'biosample_identifiers',
         'mixins': ['insdc_identifiers'],
         'see_also': ['https://github.com/bioregistry/bioregistry/issues/108',
                      'https://www.ebi.ac.uk/biosamples/',
                      'https://www.ncbi.nlm.nih.gov/biosample',
                      'https://www.ddbj.nig.ac.jp/biosample/index-e.html']} })
    emsl_biosample_identifiers: Optional[list[str]] = Field(default=None, title="EMSL Biosample Identifiers", description="""A list of identifiers for the biosample from the EMSL database.  This is used to link the biosample, as modeled by NMDC, to the biosample in the planned EMSL NEXUS database.""", json_schema_extra = { "linkml_meta": {'alias': 'emsl_biosample_identifiers',
         'domain_of': ['Biosample'],
         'is_a': 'biosample_identifiers',
         'mixins': ['emsl_identifiers'],
         'todos': ['removed "planned" once NEXUS is online',
                   'determine real expansion for emsl prefix']} })
    igsn_biosample_identifiers: Optional[list[str]] = Field(default=None, title="IGSN Biosample Identifiers", description="""A list of identifiers for the biosample from the IGSN database.""", json_schema_extra = { "linkml_meta": {'alias': 'igsn_biosample_identifiers',
         'domain_of': ['Biosample'],
         'is_a': 'biosample_identifiers',
         'mixins': ['igsn_identifiers']} })
    abs_air_humidity: Optional[QuantityValue] = Field(default=None, title="absolute air humidity", description="""Actual mass of water vapor - mh20 - present in the air water vapor mixture""", json_schema_extra = { "linkml_meta": {'alias': 'abs_air_humidity',
         'aliases': ['absolute air humidity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram per gram, kilogram per '
                                                     'kilogram, kilogram, pound'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '9 gram per gram'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000122'} })
    add_recov_method: Optional[TextValue] = Field(default=None, title="secondary and tertiary recovery methods and start date", description="""Additional (i.e. Secondary, tertiary, etc.) recovery methods deployed for increase of hydrocarbon recovery from resource and start date for each one of them. If \"other\" is specified, please propose entry in \"additional info\" field""", json_schema_extra = { "linkml_meta": {'alias': 'add_recov_method',
         'aliases': ['secondary and tertiary recovery methods and start date'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration;timestamp'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Polymer Addition;2018-06-21T14:30Z'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001009'} })
    additional_info: Optional[TextValue] = Field(default=None, title="additional info", description="""Information that doesn't fit anywhere else. Can also be used to propose new entries for fields with controlled vocabulary""", json_schema_extra = { "linkml_meta": {'alias': 'additional_info',
         'aliases': ['additional info'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'text'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000300',
         'string_serialization': '{text}'} })
    address: Optional[TextValue] = Field(default=None, title="address", description="""The street name and building number where the sampling occurred.""", json_schema_extra = { "linkml_meta": {'alias': 'address',
         'aliases': ['address'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000218',
         'string_serialization': '{integer}{text}'} })
    adj_room: Optional[TextValue] = Field(default=None, title="adjacent rooms", description="""List of rooms (room number, room name) immediately adjacent to the sampling room""", json_schema_extra = { "linkml_meta": {'alias': 'adj_room',
         'aliases': ['adjacent rooms'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'room name;room number'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000219',
         'string_serialization': '{text};{integer}'} })
    aero_struc: Optional[TextValue] = Field(default=None, title="aerospace structure", description="""Aerospace structures typically consist of thin plates with stiffeners for the external surfaces, bulkheads and frames to support the shape and fasteners such as welds, rivets, screws and bolts to hold the components together""", json_schema_extra = { "linkml_meta": {'alias': 'aero_struc',
         'aliases': ['aerospace structure'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'plane'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000773',
         'string_serialization': '[plane|glider]'} })
    agrochem_addition: Optional[list[TextValue]] = Field(default=None, title="history/agrochemical additions", description="""Addition of fertilizers, pesticides, etc. - amount and time of applications""", json_schema_extra = { "linkml_meta": {'alias': 'agrochem_addition',
         'aliases': ['history/agrochemical additions'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'agrochemical name;agrochemical '
                                                     'amount;timestamp'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram, mole per liter, milligram '
                                                     'per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'roundup;5 milligram per liter;2018-06-21'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000639',
         'string_serialization': '{text};{float} {unit};{timestamp}'} })
    air_PM_concen: Optional[list[TextValue]] = Field(default=None, title="air particulate matter concentration", description="""Concentration of substances that remain suspended in the air, and comprise mixtures of organic and inorganic substances (PM10 and PM2.5); can report multiple PM's by entering numeric values preceded by name of PM""", json_schema_extra = { "linkml_meta": {'alias': 'air_PM_concen',
         'aliases': ['air particulate matter concentration'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'particulate matter '
                                                     'name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micrograms per cubic meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'PM2.5;10 microgram per cubic meter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000108',
         'string_serialization': '{text};{float} {unit}'} })
    air_temp: Optional[QuantityValue] = Field(default=None, title="air temperature", description="""Temperature of the air at the time of sampling""", json_schema_extra = { "linkml_meta": {'alias': 'air_temp',
         'aliases': ['air temperature'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '20 degree Celsius'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000124'} })
    air_temp_regm: Optional[list[TextValue]] = Field(default=None, title="air temperature regimen", description="""Information about treatment involving an exposure to varying temperatures; should include the temperature, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include different temperature regimens""", json_schema_extra = { "linkml_meta": {'alias': 'air_temp_regm',
         'aliases': ['air temperature regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'temperature value;treatment '
                                                     'interval and duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '25 degree '
                                'Celsius;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000551',
         'string_serialization': '{float} {unit};{Rn/start_time/end_time/duration}'} })
    al_sat: Optional[QuantityValue] = Field(default=None, title="aluminum saturation/ extreme unusual properties", description="""The relative abundance of aluminum in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'al_sat',
         'aliases': ['extreme_unusual_properties/Al saturation'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'percentage'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '27%'}],
         'is_a': 'core field',
         'notes': ['Aluminum saturation is the percentage of the CEC occupies by '
                   'aluminum. Like all cations, aluminum held by the cation exchange '
                   'complex is in equilibrium with aluminum in the soil solution.'],
         'slot_uri': 'MIXS:0000607',
         'todos': ['Example & validation. Can we configure things so that 27% & 27 % & '
                   '0.27 will validate?',
                   "I think it's weird the way GSC writes the title. I recommend this "
                   "change. Thoughts? I would argue this isn't an extreme unusual "
                   "property. It's just a biogeochemical measurement."]} })
    al_sat_meth: Optional[TextValue] = Field(default=None, title="aluminum saturation method/ extreme unusual properties", description="""Reference or method used in determining Aluminum saturation""", json_schema_extra = { "linkml_meta": {'alias': 'al_sat_meth',
         'aliases': ['extreme_unusual_properties/Al saturation method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or URL'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'comments': ['Required when aluminum saturation is provided.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'https://doi.org/10.1371/journal.pone.0176357'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000324',
         'string_serialization': '{PMID}|{DOI}|{URL}',
         'todos': ["I think it's weird the way GSC writes the title. I recommend this "
                   'change. Thoughts?']} })
    alkalinity: Optional[QuantityValue] = Field(default=None, title="alkalinity", description="""Alkalinity, the ability of a solution to neutralize acids to the equivalence point of carbonate or bicarbonate""", json_schema_extra = { "linkml_meta": {'alias': 'alkalinity',
         'aliases': ['alkalinity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milliequivalent per liter, '
                                                     'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '50 milligram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000421'} })
    alkalinity_method: Optional[TextValue] = Field(default=None, title="alkalinity method", description="""Method used for alkalinity measurement""", json_schema_extra = { "linkml_meta": {'alias': 'alkalinity_method',
         'aliases': ['alkalinity method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'description of method'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'titration'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000298',
         'string_serialization': '{text}'} })
    alkyl_diethers: Optional[QuantityValue] = Field(default=None, title="alkyl diethers", description="""Concentration of alkyl diethers""", json_schema_extra = { "linkml_meta": {'alias': 'alkyl_diethers',
         'aliases': ['alkyl diethers'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'mole per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.005 mole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000490'} })
    alt: Optional[QuantityValue] = Field(default=None, title="altitude", description="""Altitude is a term used to identify heights of objects such as airplanes, space shuttles, rockets, atmospheric balloons and heights of places such as atmospheric layers and clouds. It is used to measure the height of an object which is above the earth's surface. In this context, the altitude measurement is the vertical distance between the earth's surface above sea level and the sampled position in the air""", json_schema_extra = { "linkml_meta": {'alias': 'alt',
         'aliases': ['altitude'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '100 meter'}],
         'is_a': 'environment field',
         'slot_uri': 'MIXS:0000094'} })
    aminopept_act: Optional[QuantityValue] = Field(default=None, title="aminopeptidase activity", description="""Measurement of aminopeptidase activity""", json_schema_extra = { "linkml_meta": {'alias': 'aminopept_act',
         'aliases': ['aminopeptidase activity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'mole per liter per hour'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.269 mole per liter per hour'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000172'} })
    ammonium: Optional[QuantityValue] = Field(default=None, title="ammonium", description="""Concentration of ammonium in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'ammonium',
         'aliases': ['ammonium'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter, milligram '
                                                     'per liter, parts per million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '1.5 milligram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000427'} })
    amount_light: Optional[QuantityValue] = Field(default=None, title="amount of light", description="""The unit of illuminance and luminous emittance, measuring luminous flux per unit area""", json_schema_extra = { "linkml_meta": {'alias': 'amount_light',
         'aliases': ['amount of light'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'lux, lumens per square meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000140'} })
    ances_data: Optional[TextValue] = Field(default=None, title="ancestral data", description="""Information about either pedigree or other ancestral information description (e.g. parental variety in case of mutant or selection), e.g. A/3*B (meaning [(A x B) x B] x B)""", json_schema_extra = { "linkml_meta": {'alias': 'ances_data',
         'aliases': ['ancestral data'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'free text'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'A/3*B'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000247',
         'string_serialization': '{text}'} })
    annual_precpt: Optional[QuantityValue] = Field(default=None, title="mean annual precipitation", description="""The average of all annual precipitation values known, or an estimated equivalent value derived by such methods as regional indexes or Isohyetal maps.""", json_schema_extra = { "linkml_meta": {'alias': 'annual_precpt',
         'aliases': ['mean annual precipitation'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'millimeter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '8.94 inch'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000644'} })
    annual_temp: Optional[QuantityValue] = Field(default=None, title="mean annual temperature", description="""Mean annual temperature""", json_schema_extra = { "linkml_meta": {'alias': 'annual_temp',
         'aliases': ['mean annual temperature'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '12.5 degree Celsius'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000642'} })
    antibiotic_regm: Optional[list[TextValue]] = Field(default=None, title="antibiotic regimen", description="""Information about treatment involving antibiotic administration; should include the name of antibiotic, amount administered, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple antibiotic regimens""", json_schema_extra = { "linkml_meta": {'alias': 'antibiotic_regm',
         'aliases': ['antibiotic regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'antibiotic name;antibiotic '
                                                     'amount;treatment interval and '
                                                     'duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'penicillin;5 '
                                'milligram;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000553',
         'string_serialization': '{text};{float} '
                                 '{unit};{Rn/start_time/end_time/duration}'} })
    api: Optional[QuantityValue] = Field(default=None, title="API gravity", description="""API gravity is a measure of how heavy or light a petroleum liquid is compared to water (source: https://en.wikipedia.org/wiki/API_gravity) (e.g. 31.1¬∞ API)""", json_schema_extra = { "linkml_meta": {'alias': 'api',
         'aliases': ['API gravity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'degrees API'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000157'} })
    arch_struc: Optional[ArchStrucEnum] = Field(default=None, title="architectural structure", description="""An architectural structure is a human-made, free-standing, immobile outdoor construction""", json_schema_extra = { "linkml_meta": {'alias': 'arch_struc',
         'aliases': ['architectural structure'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'shed'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000774'} })
    aromatics_pc: Optional[TextValue] = Field(default=None, title="aromatics wt%", description="""Saturate, Aromatic, Resin and Asphaltene¬†(SARA) is an analysis method that divides¬†crude oil¬†components according to their polarizability and polarity. There are three main methods to obtain SARA results. The most popular one is known as the Iatroscan TLC-FID and is referred to as IP-143 (source: https://en.wikipedia.org/wiki/Saturate,_aromatic,_resin_and_asphaltene)""", json_schema_extra = { "linkml_meta": {'alias': 'aromatics_pc',
         'aliases': ['aromatics wt%'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'percent'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000133',
         'string_serialization': '{text};{float} {unit}'} })
    asphaltenes_pc: Optional[TextValue] = Field(default=None, title="asphaltenes wt%", description="""Saturate, Aromatic, Resin and Asphaltene¬†(SARA) is an analysis method that divides¬†crude oil¬†components according to their polarizability and polarity. There are three main methods to obtain SARA results. The most popular one is known as the Iatroscan TLC-FID and is referred to as IP-143 (source: https://en.wikipedia.org/wiki/Saturate,_aromatic,_resin_and_asphaltene)""", json_schema_extra = { "linkml_meta": {'alias': 'asphaltenes_pc',
         'aliases': ['asphaltenes wt%'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'percent'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000135',
         'string_serialization': '{text};{float} {unit}'} })
    atmospheric_data: Optional[list[TextValue]] = Field(default=None, title="atmospheric data", description="""Measurement of atmospheric data; can include multiple data""", json_schema_extra = { "linkml_meta": {'alias': 'atmospheric_data',
         'aliases': ['atmospheric data'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'atmospheric data '
                                                     'name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'wind speed;9 knots'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001097',
         'string_serialization': '{text};{float} {unit}'} })
    avg_dew_point: Optional[QuantityValue] = Field(default=None, title="average dew point", description="""The average of dew point measures taken at the beginning of every hour over a 24 hour period on the sampling day""", json_schema_extra = { "linkml_meta": {'alias': 'avg_dew_point',
         'aliases': ['average dew point'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '25.5 degree Celsius'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000141'} })
    avg_occup: Optional[TextValue] = Field(default=None, title="average daily occupancy", description="""Daily average occupancy of room. Indicate the number of person(s) daily occupying the sampling room.""", json_schema_extra = { "linkml_meta": {'alias': 'avg_occup',
         'aliases': ['average daily occupancy'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000775'} })
    avg_temp: Optional[QuantityValue] = Field(default=None, title="average temperature", description="""The average of temperatures taken at the beginning of every hour over a 24 hour period on the sampling day""", json_schema_extra = { "linkml_meta": {'alias': 'avg_temp',
         'aliases': ['average temperature'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '12.5 degree Celsius'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000142'} })
    bac_prod: Optional[QuantityValue] = Field(default=None, title="bacterial production", description="""Bacterial production in the water column measured by isotope uptake""", json_schema_extra = { "linkml_meta": {'alias': 'bac_prod',
         'aliases': ['bacterial production'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per cubic meter per '
                                                     'day'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '5 milligram per cubic meter per day'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000683'} })
    bac_resp: Optional[QuantityValue] = Field(default=None, title="bacterial respiration", description="""Measurement of bacterial respiration in the water column""", json_schema_extra = { "linkml_meta": {'alias': 'bac_resp',
         'aliases': ['bacterial respiration'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per cubic meter per '
                                                     'day, micromole oxygen per liter '
                                                     'per hour'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '300 micromole oxygen per liter per hour'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000684'} })
    bacteria_carb_prod: Optional[QuantityValue] = Field(default=None, title="bacterial carbon production", description="""Measurement of bacterial carbon production""", json_schema_extra = { "linkml_meta": {'alias': 'bacteria_carb_prod',
         'aliases': ['bacterial carbon production'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'nanogram per hour'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2.53 microgram per liter per hour'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000173'} })
    barometric_press: Optional[QuantityValue] = Field(default=None, title="barometric pressure", description="""Force per unit area exerted against a surface by the weight of air above that surface""", json_schema_extra = { "linkml_meta": {'alias': 'barometric_press',
         'aliases': ['barometric pressure'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'millibar'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '5 millibar'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000096'} })
    basin: Optional[TextValue] = Field(default=None, title="basin name", description="""Name of the basin (e.g. Campos)""", json_schema_extra = { "linkml_meta": {'alias': 'basin',
         'aliases': ['basin name'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000290',
         'string_serialization': '{text}'} })
    bathroom_count: Optional[TextValue] = Field(default=None, title="bathroom count", description="""The number of bathrooms in the building""", json_schema_extra = { "linkml_meta": {'alias': 'bathroom_count',
         'aliases': ['bathroom count'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '1'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000776'} })
    bedroom_count: Optional[TextValue] = Field(default=None, title="bedroom count", description="""The number of bedrooms in the building""", json_schema_extra = { "linkml_meta": {'alias': 'bedroom_count',
         'aliases': ['bedroom count'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000777'} })
    benzene: Optional[QuantityValue] = Field(default=None, title="benzene", description="""Concentration of benzene in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'benzene',
         'aliases': ['benzene'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter, parts per '
                                                     'million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000153'} })
    biochem_oxygen_dem: Optional[QuantityValue] = Field(default=None, title="biochemical oxygen demand", description="""Amount of dissolved oxygen needed by aerobic biological organisms in a body of water to break down organic material present in a given water sample at certain temperature over a specific time period""", json_schema_extra = { "linkml_meta": {'alias': 'biochem_oxygen_dem',
         'aliases': ['biochemical oxygen demand'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000653'} })
    biocide: Optional[TextValue] = Field(default=None, title="biocide administration", description="""List of biocides (commercial name of product and supplier) and date of administration""", json_schema_extra = { "linkml_meta": {'alias': 'biocide',
         'aliases': ['biocide administration'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'name;name;timestamp'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'ALPHA 1427;Baker Hughes;2008-01-23'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001011',
         'string_serialization': '{text};{text};{timestamp}'} })
    biocide_admin_method: Optional[TextValue] = Field(default=None, title="biocide administration method", description="""Method of biocide administration (dose, frequency, duration, time elapsed between last biociding and sampling) (e.g. 150 mg/l; weekly; 4 hr; 3 days)""", json_schema_extra = { "linkml_meta": {'alias': 'biocide_admin_method',
         'aliases': ['biocide administration method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement '
                                                     'value;frequency;duration;duration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000456',
         'string_serialization': '{float} '
                                 '{unit};{Rn/start_time/end_time/duration};{duration}'} })
    biol_stat: Optional[BiolStatEnum] = Field(default=None, title="biological status", description="""The level of genome modification.""", json_schema_extra = { "linkml_meta": {'alias': 'biol_stat',
         'aliases': ['biological status'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'natural'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000858'} })
    biomass: Optional[list[TextValue]] = Field(default=None, title="biomass", description="""Amount of biomass; should include the name for the part of biomass measured, e.g. Microbial, total. Can include multiple measurements""", json_schema_extra = { "linkml_meta": {'alias': 'biomass',
         'aliases': ['biomass'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'biomass type;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'ton, kilogram, gram'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'total;20 gram'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000174',
         'string_serialization': '{text};{float} {unit}'} })
    biotic_regm: Optional[TextValue] = Field(default=None, title="biotic regimen", description="""Information about treatment(s) involving use of biotic factors, such as bacteria, viruses or fungi.""", json_schema_extra = { "linkml_meta": {'alias': 'biotic_regm',
         'aliases': ['biotic regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'free text'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'sample inoculated with Rhizobium spp. Culture'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001038',
         'string_serialization': '{text}'} })
    biotic_relationship: Optional[BioticRelationshipEnum] = Field(default=None, title="observed biotic relationship", description="""Description of relationship(s) between the subject organism and other organism(s) it is associated with. E.g., parasite on species X; mutualist with species Y. The target organism is the subject of the relationship, and the other organism(s) is the object""", json_schema_extra = { "linkml_meta": {'alias': 'biotic_relationship',
         'aliases': ['observed biotic relationship'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'free living'}],
         'is_a': 'nucleic acid sequence source field',
         'slot_uri': 'MIXS:0000028'} })
    bishomohopanol: Optional[QuantityValue] = Field(default=None, title="bishomohopanol", description="""Concentration of bishomohopanol""", json_schema_extra = { "linkml_meta": {'alias': 'bishomohopanol',
         'aliases': ['bishomohopanol'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per liter, microgram '
                                                     'per gram'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '14 microgram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000175'} })
    blood_press_diast: Optional[QuantityValue] = Field(default=None, title="host blood pressure diastolic", description="""Resting diastolic blood pressure, measured as mm mercury""", json_schema_extra = { "linkml_meta": {'alias': 'blood_press_diast',
         'aliases': ['host blood pressure diastolic'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'millimeter mercury'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000258'} })
    blood_press_syst: Optional[QuantityValue] = Field(default=None, title="host blood pressure systolic", description="""Resting systolic blood pressure, measured as mm mercury""", json_schema_extra = { "linkml_meta": {'alias': 'blood_press_syst',
         'aliases': ['host blood pressure systolic'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'millimeter mercury'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000259'} })
    bromide: Optional[QuantityValue] = Field(default=None, title="bromide", description="""Concentration of bromide""", json_schema_extra = { "linkml_meta": {'alias': 'bromide',
         'aliases': ['bromide'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'parts per million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.05 parts per million'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000176'} })
    build_docs: Optional[BuildDocsEnum] = Field(default=None, title="design, construction, and operation documents", description="""The building design, construction and operation documents""", json_schema_extra = { "linkml_meta": {'alias': 'build_docs',
         'aliases': ['design, construction, and operation documents'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'maintenance plans'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000787'} })
    build_occup_type: Optional[list[BuildOccupTypeEnum]] = Field(default=None, title="building occupancy type", description="""The primary function for which a building or discrete part of a building is intended to be used""", json_schema_extra = { "linkml_meta": {'alias': 'build_occup_type',
         'aliases': ['building occupancy type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'market'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000761'} })
    building_setting: Optional[BuildingSettingEnum] = Field(default=None, title="building setting", description="""A location (geography) where a building is set""", json_schema_extra = { "linkml_meta": {'alias': 'building_setting',
         'aliases': ['building setting'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'rural'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000768'} })
    built_struc_age: Optional[QuantityValue] = Field(default=None, title="built structure age", description="""The age of the built structure since construction""", json_schema_extra = { "linkml_meta": {'alias': 'built_struc_age',
         'aliases': ['built structure age'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'year'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '15'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000145'} })
    built_struc_set: Optional[TextValue] = Field(default=None, title="built structure setting", description="""The characterization of the location of the built structure as high or low human density""", json_schema_extra = { "linkml_meta": {'alias': 'built_struc_set',
         'aliases': ['built structure setting'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'rural'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000778',
         'string_serialization': '[urban|rural]'} })
    built_struc_type: Optional[TextValue] = Field(default=None, title="built structure type", description="""A physical structure that is a body or assemblage of bodies in space to form a system capable of supporting loads""", json_schema_extra = { "linkml_meta": {'alias': 'built_struc_type',
         'aliases': ['built structure type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'free text'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000721',
         'string_serialization': '{text}'} })
    calcium: Optional[QuantityValue] = Field(default=None, title="calcium", description="""Concentration of calcium in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'calcium',
         'aliases': ['calcium'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter, micromole '
                                                     'per liter, parts per million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.2 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000432'} })
    carb_dioxide: Optional[QuantityValue] = Field(default=None, title="carbon dioxide", description="""Carbon dioxide (gas) amount or concentration at the time of sampling""", json_schema_extra = { "linkml_meta": {'alias': 'carb_dioxide',
         'aliases': ['carbon dioxide'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter, parts per '
                                                     'million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '410 parts per million'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000097'} })
    carb_monoxide: Optional[QuantityValue] = Field(default=None, title="carbon monoxide", description="""Carbon monoxide (gas) amount or concentration at the time of sampling""", json_schema_extra = { "linkml_meta": {'alias': 'carb_monoxide',
         'aliases': ['carbon monoxide'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter, parts per '
                                                     'million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.1 parts per million'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000098'} })
    carb_nitro_ratio: Optional[QuantityValue] = Field(default=None, title="carbon/nitrogen ratio", description="""Ratio of amount or concentrations of carbon to nitrogen""", json_schema_extra = { "linkml_meta": {'alias': 'carb_nitro_ratio',
         'aliases': ['carbon/nitrogen ratio'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.417361111'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000310'} })
    ceil_area: Optional[QuantityValue] = Field(default=None, title="ceiling area", description="""The area of the ceiling space within the room""", json_schema_extra = { "linkml_meta": {'alias': 'ceil_area',
         'aliases': ['ceiling area'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'square meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '25 square meter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000148'} })
    ceil_cond: Optional[CeilCondEnum] = Field(default=None, title="ceiling condition", description="""The physical condition of the ceiling at the time of sampling; photos or video preferred; use drawings to indicate location of damaged areas""", json_schema_extra = { "linkml_meta": {'alias': 'ceil_cond',
         'aliases': ['ceiling condition'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'damaged'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000779'} })
    ceil_finish_mat: Optional[CeilFinishMatEnum] = Field(default=None, title="ceiling finish material", description="""The type of material used to finish a ceiling""", json_schema_extra = { "linkml_meta": {'alias': 'ceil_finish_mat',
         'aliases': ['ceiling finish material'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'stucco'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000780'} })
    ceil_struc: Optional[TextValue] = Field(default=None, title="ceiling structure", description="""The construction format of the ceiling""", json_schema_extra = { "linkml_meta": {'alias': 'ceil_struc',
         'aliases': ['ceiling structure'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'concrete'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000782',
         'string_serialization': '[wood frame|concrete]'} })
    ceil_texture: Optional[CeilTextureEnum] = Field(default=None, title="ceiling texture", description="""The feel, appearance, or consistency of a ceiling surface""", json_schema_extra = { "linkml_meta": {'alias': 'ceil_texture',
         'aliases': ['ceiling texture'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'popcorn'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000783'} })
    ceil_thermal_mass: Optional[QuantityValue] = Field(default=None, title="ceiling thermal mass", description="""The ability of the ceiling to provide inertia against temperature fluctuations. Generally this means concrete that is exposed. A metal deck that supports a concrete slab will act thermally as long as it is exposed to room air flow""", json_schema_extra = { "linkml_meta": {'alias': 'ceil_thermal_mass',
         'aliases': ['ceiling thermal mass'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'joule per degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000143'} })
    ceil_type: Optional[CeilTypeEnum] = Field(default=None, title="ceiling type", description="""The type of ceiling according to the ceiling's appearance or construction""", json_schema_extra = { "linkml_meta": {'alias': 'ceil_type',
         'aliases': ['ceiling type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'coffered'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000784'} })
    ceil_water_mold: Optional[TextValue] = Field(default=None, title="ceiling signs of water/mold", description="""Signs of the presence of mold or mildew on the ceiling""", json_schema_extra = { "linkml_meta": {'alias': 'ceil_water_mold',
         'aliases': ['ceiling signs of water/mold'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'presence of mold visible'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000781',
         'string_serialization': '[presence of mold visible|no presence of mold '
                                 'visible]'} })
    chem_administration: Optional[list[Union[ControlledTermValue,ControlledIdentifiedTermValue]]] = Field(default=None, title="chemical administration", description="""List of chemical compounds administered to the host or site where sampling occurred, and when (e.g. Antibiotics, n fertilizer, air filter); can include multiple compounds. For chemical entities of biological interest ontology (chebi) (v 163), http://purl.bioontology.org/ontology/chebi""", json_schema_extra = { "linkml_meta": {'alias': 'chem_administration',
         'aliases': ['chemical administration'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'CHEBI;timestamp'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'agar [CHEBI:2509];2018-05-11T20:00Z'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000751',
         'string_serialization': '{termLabel} {[termID]};{timestamp}'} })
    chem_mutagen: Optional[list[TextValue]] = Field(default=None, title="chemical mutagen", description="""Treatment involving use of mutagens; should include the name of mutagen, amount administered, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple mutagen regimens""", json_schema_extra = { "linkml_meta": {'alias': 'chem_mutagen',
         'aliases': ['chemical mutagen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'mutagen name;mutagen '
                                                     'amount;treatment interval and '
                                                     'duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'nitrous acid;0.5 milligram per '
                                'liter;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000555',
         'string_serialization': '{text};{float} '
                                 '{unit};{Rn/start_time/end_time/duration}'} })
    chem_oxygen_dem: Optional[QuantityValue] = Field(default=None, title="chemical oxygen demand", description="""A measure of the capacity of water to consume oxygen during the decomposition of organic matter and the oxidation of inorganic chemicals such as ammonia and nitrite""", json_schema_extra = { "linkml_meta": {'alias': 'chem_oxygen_dem',
         'aliases': ['chemical oxygen demand'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000656'} })
    chem_treat_method: Optional[str] = Field(default=None, title="chemical treatment method", description="""Method of chemical administration(dose, frequency, duration, time elapsed between administration and sampling) (e.g. 50 mg/l; twice a week; 1 hr; 0 days)""", json_schema_extra = { "linkml_meta": {'alias': 'chem_treat_method',
         'aliases': ['chemical treatment method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement '
                                                     'value;frequency;duration;duration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000457',
         'string_serialization': '{float} '
                                 '{unit};{Rn/start_time/end_time/duration};{duration};{duration}'} })
    chem_treatment: Optional[TextValue] = Field(default=None, title="chemical treatment", description="""List of chemical compounds administered upstream the sampling location where sampling occurred (e.g. Glycols, H2S scavenger, corrosion and scale inhibitors, demulsifiers, and other production chemicals etc.). The commercial name of the product and name of the supplier should be provided. The date of administration should also be included""", json_schema_extra = { "linkml_meta": {'alias': 'chem_treatment',
         'aliases': ['chemical treatment'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'name;name;timestamp'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'ACCENT 1125;DOW;2010-11-17'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001012',
         'string_serialization': '{text};{text};{timestamp}'} })
    chloride: Optional[QuantityValue] = Field(default=None, title="chloride", description="""Concentration of chloride in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'chloride',
         'aliases': ['chloride'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter, parts per '
                                                     'million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '5000 milligram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000429'} })
    chlorophyll: Optional[QuantityValue] = Field(default=None, title="chlorophyll", description="""Concentration of chlorophyll""", json_schema_extra = { "linkml_meta": {'alias': 'chlorophyll',
         'aliases': ['chlorophyll'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per cubic meter, '
                                                     'microgram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '5 milligram per cubic meter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000177'} })
    climate_environment: Optional[list[TextValue]] = Field(default=None, title="climate environment", description="""Treatment involving an exposure to a particular climate; treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple climates""", json_schema_extra = { "linkml_meta": {'alias': 'climate_environment',
         'aliases': ['climate environment'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'climate name;treatment interval '
                                                     'and duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'tropical '
                                'climate;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001040',
         'string_serialization': '{text};{Rn/start_time/end_time/duration}',
         'todos': ['description says "can include multiple climates" but multivalued '
                   'is set to false',
                   'add examples, i need to see some examples to add correctly '
                   'formatted example.']} })
    collection_date: Optional[TimestampValue] = Field(default=None, title="collection date", description="""The time of sampling, either as an instance (single point in time) or interval. In case no exact time is available, the date/time can be right truncated i.e. all of these are valid times: 2008-01-23T19:23:10+00:00; 2008-01-23T19:23:10; 2008-01-23; 2008-01; 2008; Except: 2008-01; 2008 all are ISO8601 compliant""", json_schema_extra = { "linkml_meta": {'alias': 'collection_date',
         'aliases': ['collection date'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'date and time'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2018-05-11T10:00:00+01:00; 2018-05-11'}],
         'is_a': 'environment field',
         'slot_uri': 'MIXS:0000011'} })
    conduc: Optional[QuantityValue] = Field(default=None, title="conductivity", description="""Electrical conductivity of water""", json_schema_extra = { "linkml_meta": {'alias': 'conduc',
         'aliases': ['conductivity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milliSiemens per centimeter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '10 milliSiemens per centimeter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000692'} })
    cool_syst_id: Optional[TextValue] = Field(default=None, title="cooling system identifier", description="""The cooling system identifier""", json_schema_extra = { "linkml_meta": {'alias': 'cool_syst_id',
         'aliases': ['cooling system identifier'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'unique identifier'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '12345'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000785'} })
    crop_rotation: Optional[TextValue] = Field(default=None, title="history/crop rotation", description="""Whether or not crop is rotated, and if yes, rotation schedule""", json_schema_extra = { "linkml_meta": {'alias': 'crop_rotation',
         'aliases': ['history/crop rotation'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'crop rotation status;schedule'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'yes;R2/2017-01-01/2018-12-31/P6M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000318',
         'string_serialization': '{boolean};{Rn/start_time/end_time/duration}'} })
    cult_root_med: Optional[TextValue] = Field(default=None, title="culture rooting medium", description="""Name or reference for the hydroponic or in vitro culture rooting medium; can be the name of a commonly used medium or reference to a specific medium, e.g. Murashige and Skoog medium. If the medium has not been formally published, use the rooting medium descriptors.""", json_schema_extra = { "linkml_meta": {'alias': 'cult_root_med',
         'aliases': ['culture rooting medium'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'name, PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'http://himedialabs.com/TD/PT158.pdf'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001041',
         'string_serialization': '{text}|{PMID}|{DOI}|{URL}'} })
    cur_land_use: Optional[CurLandUseEnum] = Field(default=None, title="current land use", description="""Present state of sample site""", json_schema_extra = { "linkml_meta": {'alias': 'cur_land_use',
         'aliases': ['current land use'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'conifers'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001080'} })
    cur_vegetation: Optional[TextValue] = Field(default=None, title="current vegetation", description="""Vegetation classification from one or more standard classification systems, or agricultural crop""", json_schema_extra = { "linkml_meta": {'alias': 'cur_vegetation',
         'aliases': ['current vegetation'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'current vegetation type'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'comments': ['Values provided here can be specific species of vegetation or '
                      'vegetation regions',
                      'See for vegetation regions- '
                      'https://education.nationalgeographic.org/resource/vegetation-region'],
         'domain_of': ['FieldResearchSite', 'Biosample'],
         'examples': [{'value': 'deciduous forest'},
                      {'value': 'forest'},
                      {'value': 'Bauhinia variegata'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000312',
         'string_serialization': '{text}',
         'todos': ['Recommend changing this from text value to some king of ontology?']} })
    cur_vegetation_meth: Optional[TextValue] = Field(default=None, title="current vegetation method", description="""Reference or method used in vegetation classification""", json_schema_extra = { "linkml_meta": {'alias': 'cur_vegetation_meth',
         'aliases': ['current vegetation method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'comments': ['Required when current vegetation is provided.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'https://doi.org/10.1111/j.1654-109X.2011.01154.x'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000314',
         'string_serialization': '{PMID}|{DOI}|{URL}',
         'todos': ["I'm not sure this is a DOI, PMID, or URI. Should pool the "
                   'community and find out how they accomplish this if provided.']} })
    date_last_rain: Optional[TimestampValue] = Field(default=None, title="date last rain", description="""The date of the last time it rained""", json_schema_extra = { "linkml_meta": {'alias': 'date_last_rain',
         'aliases': ['date last rain'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'timestamp'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2018-05-11:T14:30Z'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000786'} })
    density: Optional[QuantityValue] = Field(default=None, title="density", description="""Density of the sample, which is its mass per unit volume (aka volumetric mass density)""", json_schema_extra = { "linkml_meta": {'alias': 'density',
         'aliases': ['density'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram per cubic meter, gram per '
                                                     'cubic centimeter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '1000 kilogram per cubic meter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000435'} })
    depos_env: Optional[DeposEnvEnum] = Field(default=None, title="depositional environment", description="""Main depositional environment (https://en.wikipedia.org/wiki/Depositional_environment). If \"other\" is specified, please propose entry in \"additional info\" field""", json_schema_extra = { "linkml_meta": {'alias': 'depos_env',
         'aliases': ['depositional environment'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Continental - Alluvial'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000992'} })
    depth: Optional[QuantityValue] = Field(default=None, title="depth", description="""The vertical distance below local surface, e.g. for sediment or soil samples depth is measured from sediment or soil surface, respectively. Depth can be reported as an interval for subsurface samples.""", json_schema_extra = { "linkml_meta": {'alias': 'depth',
         'aliases': ['depth'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '10 meter'}],
         'is_a': 'environment field',
         'slot_uri': 'MIXS:0000018'} })
    dew_point: Optional[QuantityValue] = Field(default=None, title="dew point", description="""The temperature to which a given parcel of humid air must be cooled, at constant barometric pressure, for water vapor to condense into water.""", json_schema_extra = { "linkml_meta": {'alias': 'dew_point',
         'aliases': ['dew point'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '22 degree Celsius'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000129'} })
    diether_lipids: Optional[list[TextValue]] = Field(default=None, title="diether lipids", description="""Concentration of diether lipids; can include multiple types of diether lipids""", json_schema_extra = { "linkml_meta": {'alias': 'diether_lipids',
         'aliases': ['diether lipids'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'diether lipid name;measurement '
                                                     'value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'nanogram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.2 nanogram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000178',
         'string_serialization': '{text};{float} {unit}'} })
    diss_carb_dioxide: Optional[QuantityValue] = Field(default=None, title="dissolved carbon dioxide", description="""Concentration of dissolved carbon dioxide in the sample or liquid portion of the sample""", json_schema_extra = { "linkml_meta": {'alias': 'diss_carb_dioxide',
         'aliases': ['dissolved carbon dioxide'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter, milligram '
                                                     'per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '5 milligram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000436'} })
    diss_hydrogen: Optional[QuantityValue] = Field(default=None, title="dissolved hydrogen", description="""Concentration of dissolved hydrogen""", json_schema_extra = { "linkml_meta": {'alias': 'diss_hydrogen',
         'aliases': ['dissolved hydrogen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.3 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000179'} })
    diss_inorg_carb: Optional[QuantityValue] = Field(default=None, title="dissolved inorganic carbon", description="""Dissolved inorganic carbon concentration in the sample, typically measured after filtering the sample using a 0.45 micrometer filter""", json_schema_extra = { "linkml_meta": {'alias': 'diss_inorg_carb',
         'aliases': ['dissolved inorganic carbon'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per liter, milligram '
                                                     'per liter, parts per million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2059 micromole per kilogram'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000434'} })
    diss_inorg_nitro: Optional[QuantityValue] = Field(default=None, title="dissolved inorganic nitrogen", description="""Concentration of dissolved inorganic nitrogen""", json_schema_extra = { "linkml_meta": {'alias': 'diss_inorg_nitro',
         'aliases': ['dissolved inorganic nitrogen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per liter, micromole '
                                                     'per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '761 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000698'} })
    diss_inorg_phosp: Optional[QuantityValue] = Field(default=None, title="dissolved inorganic phosphorus", description="""Concentration of dissolved inorganic phosphorus in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'diss_inorg_phosp',
         'aliases': ['dissolved inorganic phosphorus'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per liter, milligram '
                                                     'per liter, parts per million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '56.5 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000106'} })
    diss_iron: Optional[QuantityValue] = Field(default=None, title="dissolved iron", description="""Concentration of dissolved iron in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'diss_iron',
         'aliases': ['dissolved iron'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000139'} })
    diss_org_carb: Optional[QuantityValue] = Field(default=None, title="dissolved organic carbon", description="""Concentration of dissolved organic carbon in the sample, liquid portion of the sample, or aqueous phase of the fluid""", json_schema_extra = { "linkml_meta": {'alias': 'diss_org_carb',
         'aliases': ['dissolved organic carbon'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter, milligram '
                                                     'per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '197 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000433'} })
    diss_org_nitro: Optional[QuantityValue] = Field(default=None, title="dissolved organic nitrogen", description="""Dissolved organic nitrogen concentration measured as; total dissolved nitrogen - NH4 - NO3 - NO2""", json_schema_extra = { "linkml_meta": {'alias': 'diss_org_nitro',
         'aliases': ['dissolved organic nitrogen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per liter, milligram '
                                                     'per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.05 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000162'} })
    diss_oxygen: Optional[QuantityValue] = Field(default=None, title="dissolved oxygen", description="""Concentration of dissolved oxygen""", json_schema_extra = { "linkml_meta": {'alias': 'diss_oxygen',
         'aliases': ['dissolved oxygen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per kilogram, '
                                                     'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '175 micromole per kilogram'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000119'} })
    diss_oxygen_fluid: Optional[QuantityValue] = Field(default=None, title="dissolved oxygen in fluids", description="""Concentration of dissolved oxygen in the oil field produced fluids as it contributes to oxgen-corrosion and microbial activity (e.g. Mic).""", json_schema_extra = { "linkml_meta": {'alias': 'diss_oxygen_fluid',
         'aliases': ['dissolved oxygen in fluids'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per kilogram, '
                                                     'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000438'} })
    door_comp_type: Optional[DoorCompTypeEnum] = Field(default=None, title="door type, composite", description="""The composite type of the door""", json_schema_extra = { "linkml_meta": {'alias': 'door_comp_type',
         'aliases': ['door type, composite'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'revolving'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000795'} })
    door_cond: Optional[DoorCondEnum] = Field(default=None, title="door condition", description="""The phsical condition of the door""", json_schema_extra = { "linkml_meta": {'alias': 'door_cond',
         'aliases': ['door condition'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'new'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000788'} })
    door_direct: Optional[DoorDirectEnum] = Field(default=None, title="door direction of opening", description="""The direction the door opens""", json_schema_extra = { "linkml_meta": {'alias': 'door_direct',
         'aliases': ['door direction of opening'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'inward'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000789'} })
    door_loc: Optional[DoorLocEnum] = Field(default=None, title="door location", description="""The relative location of the door in the room""", json_schema_extra = { "linkml_meta": {'alias': 'door_loc',
         'aliases': ['door location'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'north'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000790'} })
    door_mat: Optional[DoorMatEnum] = Field(default=None, title="door material", description="""The material the door is composed of""", json_schema_extra = { "linkml_meta": {'alias': 'door_mat',
         'aliases': ['door material'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'wood'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000791'} })
    door_move: Optional[DoorMoveEnum] = Field(default=None, title="door movement", description="""The type of movement of the door""", json_schema_extra = { "linkml_meta": {'alias': 'door_move',
         'aliases': ['door movement'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'swinging'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000792'} })
    door_size: Optional[QuantityValue] = Field(default=None, title="door area or size", description="""The size of the door""", json_schema_extra = { "linkml_meta": {'alias': 'door_size',
         'aliases': ['door area or size'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'square meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2.5 square meter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000158'} })
    door_type: Optional[DoorTypeEnum] = Field(default=None, title="door type", description="""The type of door material""", json_schema_extra = { "linkml_meta": {'alias': 'door_type',
         'aliases': ['door type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'wooden'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000794'} })
    door_type_metal: Optional[DoorTypeMetalEnum] = Field(default=None, title="door type, metal", description="""The type of metal door""", json_schema_extra = { "linkml_meta": {'alias': 'door_type_metal',
         'aliases': ['door type, metal'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'hollow'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000796'} })
    door_type_wood: Optional[DoorTypeWoodEnum] = Field(default=None, title="door type, wood", description="""The type of wood door""", json_schema_extra = { "linkml_meta": {'alias': 'door_type_wood',
         'aliases': ['door type, wood'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'battened'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000797'} })
    door_water_mold: Optional[TextValue] = Field(default=None, title="door signs of water/mold", description="""Signs of the presence of mold or mildew on a door""", json_schema_extra = { "linkml_meta": {'alias': 'door_water_mold',
         'aliases': ['door signs of water/mold'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'presence of mold visible'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000793',
         'string_serialization': '[presence of mold visible|no presence of mold '
                                 'visible]'} })
    down_par: Optional[QuantityValue] = Field(default=None, title="downward PAR", description="""Visible waveband radiance and irradiance measurements in the water column""", json_schema_extra = { "linkml_meta": {'alias': 'down_par',
         'aliases': ['downward PAR'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microEinstein per square meter '
                                                     'per second, microEinstein per '
                                                     'square centimeter per second'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '28.71 microEinstein per square meter per second'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000703'} })
    drainage_class: Optional[DrainageClassEnum] = Field(default=None, title="drainage classification", description="""Drainage classification from a standard system such as the USDA system""", json_schema_extra = { "linkml_meta": {'alias': 'drainage_class',
         'aliases': ['drainage classification'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'well'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001085'} })
    drawings: Optional[DrawingsEnum] = Field(default=None, title="drawings", description="""The buildings architectural drawings; if design is chosen, indicate phase-conceptual, schematic, design development, and construction documents""", json_schema_extra = { "linkml_meta": {'alias': 'drawings',
         'aliases': ['drawings'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'sketch'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000798'} })
    efficiency_percent: Optional[QuantityValue] = Field(default=None, title="efficiency percent", description="""Percentage of volatile solids removed from the anaerobic digestor""", json_schema_extra = { "linkml_meta": {'alias': 'efficiency_percent',
         'aliases': ['efficiency percent'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000657'} })
    elev: Optional[float] = Field(default=None, title="elevation, meters", description="""Elevation of the sampling site is its height above a fixed reference point, most commonly the mean sea level. Elevation is mainly used when referring to points on the earth's surface, while altitude is used for points above the surface, such as an aircraft in flight or a spacecraft in orbit.""", json_schema_extra = { "linkml_meta": {'alias': 'elev',
         'aliases': ['elevation'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'}},
         'comments': ['All elevations must be reported in meters. Provide the '
                      'numerical portion only.',
                      'Please use '
                      'https://www.advancedconverter.com/map-tools/find-altitude-by-coordinates, '
                      'if needed, to help estimate the elevation based on latitude and '
                      'longitude coordinates.'],
         'domain_of': ['FieldResearchSite', 'Biosample'],
         'examples': [{'value': '100'}],
         'is_a': 'environment field',
         'slot_uri': 'MIXS:0000093'} })
    elevator: Optional[TextValue] = Field(default=None, title="elevator count", description="""The number of elevators within the built structure""", json_schema_extra = { "linkml_meta": {'alias': 'elevator',
         'aliases': ['elevator count'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000799'} })
    emulsions: Optional[list[TextValue]] = Field(default=None, title="emulsions", description="""Amount or concentration of substances such as paints, adhesives, mayonnaise, hair colorants, emulsified oils, etc.; can include multiple emulsion types""", json_schema_extra = { "linkml_meta": {'alias': 'emulsions',
         'aliases': ['emulsions'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'emulsion name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000660',
         'string_serialization': '{text};{float} {unit}'} })
    env_broad_scale: ControlledIdentifiedTermValue = Field(default=..., title="broad-scale environmental context", description="""Report the major environmental system the sample or specimen came from. The system(s) identified should have a coarse spatial grain, to provide the general environmental context of where the sampling was done (e.g. in the desert or a rainforest). We recommend using subclasses of EnvO’s biome class:  http://purl.obolibrary.org/obo/ENVO_00000428. EnvO documentation about how to use the field: https://github.com/EnvironmentOntology/envo/wiki/Using-ENVO-with-MIxS""", json_schema_extra = { "linkml_meta": {'alias': 'env_broad_scale',
         'aliases': ['broad-scale environmental context'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'The major environment type(s) '
                                                     'where the sample was collected. '
                                                     'Recommend subclasses of biome '
                                                     '[ENVO:00000428]. Multiple terms '
                                                     'can be separated by one or more '
                                                     'pipes.'},
                         'tooltip': {'tag': 'tooltip',
                                     'value': 'The biome or major environmental system '
                                              'where the sample or specimen '
                                              'originated. Choose values from '
                                              "subclasses of the 'biome' class "
                                              '[ENVO:00000428] in the Environment '
                                              'Ontology (ENVO). For host-associated or '
                                              'plant-associated samples, use terms '
                                              'from the UBERON or Plant Ontology to '
                                              'describe the broad anatomical or '
                                              'morphological context'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'oceanic epipelagic zone biome [ENVO:01000035]'}],
         'is_a': 'mixs_env_triad_field',
         'slot_uri': 'MIXS:0000012',
         'string_serialization': '{termLabel} {[termID]}'} })
    env_local_scale: ControlledIdentifiedTermValue = Field(default=..., title="local environmental context", description="""Report the entity or entities which are in the sample or specimen’s local vicinity and which you believe have significant causal influences on your sample or specimen. We recommend using EnvO terms which are of smaller spatial grain than your entry for env_broad_scale. Terms, such as anatomical sites, from other OBO Library ontologies which interoperate with EnvO (e.g. UBERON) are accepted in this field. EnvO documentation about how to use the field: https://github.com/EnvironmentOntology/envo/wiki/Using-ENVO-with-MIxS.""", json_schema_extra = { "linkml_meta": {'alias': 'env_local_scale',
         'aliases': ['local environmental context'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'Environmental entities having '
                                                     'causal influences upon the '
                                                     'entity at time of sampling.'},
                         'tooltip': {'tag': 'tooltip',
                                     'value': 'The specific environmental entities or '
                                              'features near the sample or specimen '
                                              'that significantly influence its '
                                              'characteristics or composition. These '
                                              'entities are typically smaller in scale '
                                              'than the broad environmental context. '
                                              'Values for this field should be '
                                              'countable, material nouns and must be '
                                              'chosen from subclasses of BFO:0000040 '
                                              '(material entity) that appear in the '
                                              'Environment Ontology (ENVO). For '
                                              'host-associated or plant-associated '
                                              'samples, use terms from the UBERON or '
                                              'Plant Ontology to describe specific '
                                              'anatomical structures or plant parts.'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'litter layer [ENVO:01000338]'}],
         'is_a': 'mixs_env_triad_field',
         'slot_uri': 'MIXS:0000013',
         'string_serialization': '{termLabel} {[termID]}'} })
    env_medium: ControlledIdentifiedTermValue = Field(default=..., title="environmental medium", description="""Report the environmental material(s) immediately surrounding the sample or specimen at the time of sampling. We recommend using subclasses of 'environmental material' (http://purl.obolibrary.org/obo/ENVO_00010483). EnvO documentation about how to use the field: https://github.com/EnvironmentOntology/envo/wiki/Using-ENVO-with-MIxS . Terms from other OBO ontologies are permissible as long as they reference mass/volume nouns (e.g. air, water, blood) and not discrete, countable entities (e.g. a tree, a leaf, a table top).""", json_schema_extra = { "linkml_meta": {'alias': 'env_medium',
         'aliases': ['environmental medium'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'The material displaced by the '
                                                     'entity at time of sampling. '
                                                     'Recommend subclasses of '
                                                     'environmental material '
                                                     '[ENVO:00010483].'},
                         'tooltip': {'tag': 'tooltip',
                                     'value': 'The predominant environmental material '
                                              'or substrate that directly surrounds or '
                                              'hosts the sample or specimen at the '
                                              'time of sampling. Choose values from '
                                              "subclasses of the 'environmental "
                                              "material' class [ENVO:00010483] in the "
                                              'Environment Ontology (ENVO). Values for '
                                              'this field should be measurable or mass '
                                              'material nouns, representing continuous '
                                              'environmental materials. For '
                                              'host-associated or plant-associated '
                                              'samples, use terms from the UBERON or '
                                              'Plant Ontology to indicate a tissue, '
                                              'organ, or plant structure'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'soil [ENVO:00001998]'}],
         'is_a': 'mixs_env_triad_field',
         'slot_uri': 'MIXS:0000014',
         'string_serialization': '{termLabel} {[termID]}'} })
    env_package: Optional[TextValue] = Field(default=None, description="""MIxS extension for reporting of measurements and observations obtained from one or more of the environments where the sample was obtained. All environmental packages listed here are further defined in separate subtables. By giving the name of the environmental package, a selection of fields can be made from the subtables and can be reported""", json_schema_extra = { "linkml_meta": {'alias': 'env_package',
         'aliases': ['environmental package'],
         'domain_of': ['Biosample'],
         'notes': ['no longer in MIxS as of 6.0?']} })
    escalator: Optional[TextValue] = Field(default=None, title="escalator count", description="""The number of escalators within the built structure""", json_schema_extra = { "linkml_meta": {'alias': 'escalator',
         'aliases': ['escalator count'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '4'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000800'} })
    ethylbenzene: Optional[QuantityValue] = Field(default=None, title="ethylbenzene", description="""Concentration of ethylbenzene in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'ethylbenzene',
         'aliases': ['ethylbenzene'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter, parts per '
                                                     'million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000155'} })
    exp_duct: Optional[QuantityValue] = Field(default=None, title="exposed ductwork", description="""The amount of exposed ductwork in the room""", json_schema_extra = { "linkml_meta": {'alias': 'exp_duct',
         'aliases': ['exposed ductwork'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'square meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000144'} })
    exp_pipe: Optional[QuantityValue] = Field(default=None, title="exposed pipes", description="""The number of exposed pipes in the room""", json_schema_extra = { "linkml_meta": {'alias': 'exp_pipe',
         'aliases': ['exposed pipes'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000220'} })
    experimental_factor: Optional[Union[ControlledTermValue,ControlledIdentifiedTermValue]] = Field(default=None, title="experimental factor", description="""Experimental factors are essentially the variable aspects of an experiment design which can be used to describe an experiment, or set of experiments, in an increasingly detailed manner. This field accepts ontology terms from Experimental Factor Ontology (EFO) and/or Ontology for Biomedical Investigations (OBI). For a browser of EFO (v 2.95) terms, please see http://purl.bioontology.org/ontology/EFO; for a browser of OBI (v 2018-02-12) terms please see http://purl.bioontology.org/ontology/OBI""", json_schema_extra = { "linkml_meta": {'alias': 'experimental_factor',
         'aliases': ['experimental factor'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'text or EFO and/or OBI'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'time series design [EFO:EFO_0001779]'}],
         'is_a': 'investigation field',
         'slot_uri': 'MIXS:0000008',
         'string_serialization': '{termLabel} {[termID]}|{text}'} })
    ext_door: Optional[TextValue] = Field(default=None, title="exterior door count", description="""The number of exterior doors in the built structure""", json_schema_extra = { "linkml_meta": {'alias': 'ext_door',
         'aliases': ['exterior door count'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000170'} })
    ext_wall_orient: Optional[ExtWallOrientEnum] = Field(default=None, title="orientations of exterior wall", description="""The orientation of the exterior wall""", json_schema_extra = { "linkml_meta": {'alias': 'ext_wall_orient',
         'aliases': ['orientations of exterior wall'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'northwest'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000817'} })
    ext_window_orient: Optional[ExtWindowOrientEnum] = Field(default=None, title="orientations of exterior window", description="""The compass direction the exterior window of the room is facing""", json_schema_extra = { "linkml_meta": {'alias': 'ext_window_orient',
         'aliases': ['orientations of exterior window'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'southwest'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000818'} })
    extreme_event: Optional[str] = Field(default=None, title="history/extreme events", description="""Unusual physical events that may have affected microbial populations""", json_schema_extra = { "linkml_meta": {'alias': 'extreme_event',
         'aliases': ['history/extreme events'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'date, string'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '1980-05-18, volcanic eruption'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000320'} })
    fao_class: Optional[FaoClassEnum] = Field(default=None, title="soil_taxonomic/FAO classification", description="""Soil classification from the FAO World Reference Database for Soil Resources. The list can be found at http://www.fao.org/nr/land/sols/soil/wrb-soil-maps/reference-groups""", json_schema_extra = { "linkml_meta": {'alias': 'fao_class',
         'aliases': ['soil_taxonomic/FAO classification'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Luvisols'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001083'} })
    fertilizer_regm: Optional[list[TextValue]] = Field(default=None, title="fertilizer regimen", description="""Information about treatment involving the use of fertilizers; should include the name of fertilizer, amount administered, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple fertilizer regimens""", json_schema_extra = { "linkml_meta": {'alias': 'fertilizer_regm',
         'aliases': ['fertilizer regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'fertilizer name;fertilizer '
                                                     'amount;treatment interval and '
                                                     'duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram, mole per liter, milligram '
                                                     'per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'urea;0.6 milligram per '
                                'liter;R2/2018-05-11:T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000556',
         'string_serialization': '{text};{float} '
                                 '{unit};{Rn/start_time/end_time/duration}'} })
    field: Optional[TextValue] = Field(default=None, title="field name", description="""Name of the hydrocarbon field (e.g. Albacora)""", json_schema_extra = { "linkml_meta": {'alias': 'field',
         'aliases': ['field name'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000291',
         'string_serialization': '{text}'} })
    filter_type: Optional[list[FilterTypeEnum]] = Field(default=None, title="filter type", description="""A device which removes solid particulates or airborne molecular contaminants""", json_schema_extra = { "linkml_meta": {'alias': 'filter_type',
         'aliases': ['filter type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'HEPA'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000765'} })
    fire: Optional[str] = Field(default=None, title="history/fire", description="""Historical and/or physical evidence of fire""", json_schema_extra = { "linkml_meta": {'alias': 'fire',
         'aliases': ['history/fire'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'date string'}},
         'comments': ['Provide the date the fire occurred. If extended burning '
                      'occurred provide the date range.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '1871-10-10'}, {'value': '1871-10-01 to 1871-10-31'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001086',
         'todos': ['is "to" acceptable? Is there a better way to request that be '
                   'written?']} })
    fireplace_type: Optional[TextValue] = Field(default=None, title="fireplace type", description="""A firebox with chimney""", json_schema_extra = { "linkml_meta": {'alias': 'fireplace_type',
         'aliases': ['fireplace type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'wood burning'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000802',
         'string_serialization': '[gas burning|wood burning]'} })
    flooding: Optional[str] = Field(default=None, title="history/flooding", description="""Historical and/or physical evidence of flooding""", json_schema_extra = { "linkml_meta": {'alias': 'flooding',
         'aliases': ['history/flooding'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'date string'}},
         'comments': ['Provide the date the flood occurred. If extended flooding '
                      'occurred provide the date range.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '1927-04-15'}, {'value': '1927-04 to 1927-05'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000319',
         'todos': ['is "to" acceptable? Is there a better way to request that be '
                   'written?',
                   'What about if the "day" isn\'t known? Is this ok?']} })
    floor_age: Optional[QuantityValue] = Field(default=None, title="floor age", description="""The time period since installment of the carpet or flooring""", json_schema_extra = { "linkml_meta": {'alias': 'floor_age',
         'aliases': ['floor age'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'years, weeks, days'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000164'} })
    floor_area: Optional[QuantityValue] = Field(default=None, title="floor area", description="""The area of the floor space within the room""", json_schema_extra = { "linkml_meta": {'alias': 'floor_area',
         'aliases': ['floor area'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'square meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000165'} })
    floor_cond: Optional[FloorCondEnum] = Field(default=None, title="floor condition", description="""The physical condition of the floor at the time of sampling; photos or video preferred; use drawings to indicate location of damaged areas""", json_schema_extra = { "linkml_meta": {'alias': 'floor_cond',
         'aliases': ['floor condition'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'new'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000803'} })
    floor_count: Optional[TextValue] = Field(default=None, title="floor count", description="""The number of floors in the building, including basements and mechanical penthouse""", json_schema_extra = { "linkml_meta": {'alias': 'floor_count',
         'aliases': ['floor count'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000225'} })
    floor_finish_mat: Optional[FloorFinishMatEnum] = Field(default=None, title="floor finish material", description="""The floor covering type; the finished surface that is walked on""", json_schema_extra = { "linkml_meta": {'alias': 'floor_finish_mat',
         'aliases': ['floor finish material'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'carpet'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000804'} })
    floor_struc: Optional[FloorStrucEnum] = Field(default=None, title="floor structure", description="""Refers to the structural elements and subfloor upon which the finish flooring is installed""", json_schema_extra = { "linkml_meta": {'alias': 'floor_struc',
         'aliases': ['floor structure'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'concrete'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000806'} })
    floor_thermal_mass: Optional[QuantityValue] = Field(default=None, title="floor thermal mass", description="""The ability of the floor to provide inertia against temperature fluctuations""", json_schema_extra = { "linkml_meta": {'alias': 'floor_thermal_mass',
         'aliases': ['floor thermal mass'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'joule per degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000166'} })
    floor_water_mold: Optional[FloorWaterMoldEnum] = Field(default=None, title="floor signs of water/mold", description="""Signs of the presence of mold or mildew in a room""", json_schema_extra = { "linkml_meta": {'alias': 'floor_water_mold',
         'aliases': ['floor signs of water/mold'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'ceiling discoloration'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000805'} })
    fluor: Optional[QuantityValue] = Field(default=None, title="fluorescence", description="""Raw or converted fluorescence of water""", json_schema_extra = { "linkml_meta": {'alias': 'fluor',
         'aliases': ['fluorescence'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram chlorophyll a per '
                                                     'cubic meter, volts'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2.5 volts'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000704'} })
    freq_clean: Optional[QuantityValue] = Field(default=None, title="frequency of cleaning", description="""The number of times the sample location is cleaned. Frequency of cleaning might be on a Daily basis, Weekly, Monthly, Quarterly or Annually.""", json_schema_extra = { "linkml_meta": {'alias': 'freq_clean',
         'aliases': ['frequency of cleaning'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration or {text}'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000226'} })
    freq_cook: Optional[QuantityValue] = Field(default=None, title="frequency of cooking", description="""The number of times a meal is cooked per week""", json_schema_extra = { "linkml_meta": {'alias': 'freq_cook',
         'aliases': ['frequency of cooking'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000227'} })
    fungicide_regm: Optional[list[TextValue]] = Field(default=None, title="fungicide regimen", description="""Information about treatment involving use of fungicides; should include the name of fungicide, amount administered, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple fungicide regimens""", json_schema_extra = { "linkml_meta": {'alias': 'fungicide_regm',
         'aliases': ['fungicide regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'fungicide name;fungicide '
                                                     'amount;treatment interval and '
                                                     'duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram, mole per liter, milligram '
                                                     'per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'bifonazole;1 mole per '
                                'liter;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000557',
         'string_serialization': '{text};{float} '
                                 '{unit};{Rn/start_time/end_time/duration}'} })
    furniture: Optional[FurnitureEnum] = Field(default=None, title="furniture", description="""The types of furniture present in the sampled room""", json_schema_extra = { "linkml_meta": {'alias': 'furniture',
         'aliases': ['furniture'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'chair'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000807'} })
    gaseous_environment: Optional[list[TextValue]] = Field(default=None, title="gaseous environment", description="""Use of conditions with differing gaseous environments; should include the name of gaseous compound, amount administered, treatment duration, interval and total experimental duration; can include multiple gaseous environment regimens""", json_schema_extra = { "linkml_meta": {'alias': 'gaseous_environment',
         'aliases': ['gaseous environment'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'gaseous compound name;gaseous '
                                                     'compound amount;treatment '
                                                     'interval and duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'CO2; 500ppm above ambient; constant'},
                      {'value': 'nitric oxide;0.5 micromole per '
                                'liter;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000558',
         'string_serialization': '{text};{float} '
                                 '{unit};{Rn/start_time/end_time/duration}',
         'todos': ['would like to see usage examples for this slot. Requiring '
                   "micromole/L seems too limiting and doesn't match expected_value "
                   'value',
                   "did I do this right? keep the example that's provided and add "
                   'another? so as to not override']} })
    gaseous_substances: Optional[list[TextValue]] = Field(default=None, title="gaseous substances", description="""Amount or concentration of substances such as hydrogen sulfide, carbon dioxide, methane, etc.; can include multiple substances""", json_schema_extra = { "linkml_meta": {'alias': 'gaseous_substances',
         'aliases': ['gaseous substances'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'gaseous substance '
                                                     'name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000661',
         'string_serialization': '{text};{float} {unit}'} })
    gender_restroom: Optional[GenderRestroomEnum] = Field(default=None, title="gender of restroom", description="""The gender type of the restroom""", json_schema_extra = { "linkml_meta": {'alias': 'gender_restroom',
         'aliases': ['gender of restroom'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'male'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000808'} })
    genetic_mod: Optional[TextValue] = Field(default=None, title="genetic modification", description="""Genetic modifications of the genome of an organism, which may occur naturally by spontaneous mutation, or be introduced by some experimental means, e.g. specification of a transgene or the gene knocked-out or details of transient transfection""", json_schema_extra = { "linkml_meta": {'alias': 'genetic_mod',
         'aliases': ['genetic modification'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI,url or free text'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'aox1A transgenic'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000859',
         'string_serialization': '{PMID}|{DOI}|{URL}|{text}'} })
    geo_loc_name: Optional[TextValue] = Field(default=None, title="geographic location (country and/or sea,region)", description="""The geographical origin of the sample as defined by the country or sea name followed by specific region name. Country or sea names should be chosen from the INSDC country list (http://insdc.org/country.html), or the GAZ ontology (http://purl.bioontology.org/ontology/GAZ)""", json_schema_extra = { "linkml_meta": {'alias': 'geo_loc_name',
         'aliases': ['geographic location (country and/or sea,region)'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'country or sea name (INSDC or '
                                                     'GAZ): region(GAZ), specific '
                                                     'location name'}},
         'domain_of': ['FieldResearchSite', 'Biosample'],
         'examples': [{'value': 'USA: Maryland, Bethesda'}],
         'is_a': 'environment field',
         'slot_uri': 'MIXS:0000010',
         'string_serialization': '{term}: {term}, {text}'} })
    glucosidase_act: Optional[QuantityValue] = Field(default=None, title="glucosidase activity", description="""Measurement of glucosidase activity""", json_schema_extra = { "linkml_meta": {'alias': 'glucosidase_act',
         'aliases': ['glucosidase activity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'mol per liter per hour'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '5 mol per liter per hour'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000137'} })
    gravidity: Optional[TextValue] = Field(default=None, title="gravidity", description="""Whether or not subject is gravid, and if yes date due or date post-conception, specifying which is used""", json_schema_extra = { "linkml_meta": {'alias': 'gravidity',
         'aliases': ['gravidity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'gravidity status;timestamp'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'yes;due date:2018-05-11'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000875',
         'string_serialization': '{boolean};{timestamp}'} })
    gravity: Optional[list[TextValue]] = Field(default=None, title="gravity", description="""Information about treatment involving use of gravity factor to study various types of responses in presence, absence or modified levels of gravity; treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple treatments""", json_schema_extra = { "linkml_meta": {'alias': 'gravity',
         'aliases': ['gravity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'gravity factor value;treatment '
                                                     'interval and duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'meter per square second, g'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '12 g;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000559',
         'string_serialization': '{float} {unit};{Rn/start_time/end_time/duration}'} })
    growth_facil: Optional[Union[ControlledTermValue,ControlledIdentifiedTermValue]] = Field(default=None, title="growth facility", description="""Type of facility where the sampled plant was grown; controlled vocabulary: growth chamber, open top chamber, glasshouse, experimental garden, field. Alternatively use Crop Ontology (CO) terms, see http://www.cropontology.org/ontology/CO_715/Crop%20Research""", json_schema_extra = { "linkml_meta": {'alias': 'growth_facil',
         'aliases': ['growth facility'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'free text or CO'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Growth chamber [CO_715:0000189]'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001043',
         'string_serialization': '{text}|{termLabel} {[termID]}'} })
    growth_habit: Optional[GrowthHabitEnum] = Field(default=None, title="growth habit", description="""Characteristic shape, appearance or growth form of a plant species""", json_schema_extra = { "linkml_meta": {'alias': 'growth_habit',
         'aliases': ['growth habit'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'spreading'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001044'} })
    growth_hormone_regm: Optional[list[TextValue]] = Field(default=None, title="growth hormone regimen", description="""Information about treatment involving use of growth hormones; should include the name of growth hormone, amount administered, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple growth hormone regimens""", json_schema_extra = { "linkml_meta": {'alias': 'growth_hormone_regm',
         'aliases': ['growth hormone regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'growth hormone name;growth '
                                                     'hormone amount;treatment '
                                                     'interval and duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram, mole per liter, milligram '
                                                     'per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'abscisic acid;0.5 milligram per '
                                'liter;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000560',
         'string_serialization': '{text};{float} '
                                 '{unit};{Rn/start_time/end_time/duration}'} })
    hall_count: Optional[TextValue] = Field(default=None, title="hallway/corridor count", description="""The total count of hallways and cooridors in the built structure""", json_schema_extra = { "linkml_meta": {'alias': 'hall_count',
         'aliases': ['hallway/corridor count'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000228'} })
    handidness: Optional[HandidnessEnum] = Field(default=None, title="handidness", description="""The handidness of the individual sampled""", json_schema_extra = { "linkml_meta": {'alias': 'handidness',
         'aliases': ['handidness'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'right handedness'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000809'} })
    hc_produced: Optional[HcProducedEnum] = Field(default=None, title="hydrocarbon type produced", description="""Main hydrocarbon type produced from resource (i.e. Oil, gas, condensate, etc). If \"other\" is specified, please propose entry in \"additional info\" field""", json_schema_extra = { "linkml_meta": {'alias': 'hc_produced',
         'aliases': ['hydrocarbon type produced'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Gas'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000989'} })
    hcr: Optional[HcrEnum] = Field(default=None, title="hydrocarbon resource type", description="""Main Hydrocarbon Resource type. The term \"Hydrocarbon Resource\" HCR defined as a natural environmental feature containing large amounts of hydrocarbons at high concentrations potentially suitable for commercial exploitation. This term should not be confused with the Hydrocarbon Occurrence term which also includes hydrocarbon-rich environments with currently limited commercial interest such as seeps, outcrops, gas hydrates etc. If \"other\" is specified, please propose entry in \"additional info\" field""", json_schema_extra = { "linkml_meta": {'alias': 'hcr',
         'aliases': ['hydrocarbon resource type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Oil Sand'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000988'} })
    hcr_fw_salinity: Optional[QuantityValue] = Field(default=None, title="formation water salinity", description="""Original formation water salinity (prior to secondary recovery e.g. Waterflooding) expressed as TDS""", json_schema_extra = { "linkml_meta": {'alias': 'hcr_fw_salinity',
         'aliases': ['formation water salinity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000406'} })
    hcr_geol_age: Optional[HcrGeolAgeEnum] = Field(default=None, title="hydrocarbon resource geological age", description="""Geological age of hydrocarbon resource (Additional info: https://en.wikipedia.org/wiki/Period_(geology)). If \"other\" is specified, please propose entry in \"additional info\" field""", json_schema_extra = { "linkml_meta": {'alias': 'hcr_geol_age',
         'aliases': ['hydrocarbon resource geological age'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Silurian'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000993'} })
    hcr_pressure: Optional[TextValue] = Field(default=None, title="hydrocarbon resource original pressure", description="""Original pressure of the hydrocarbon resource""", json_schema_extra = { "linkml_meta": {'alias': 'hcr_pressure',
         'aliases': ['hydrocarbon resource original pressure'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value range'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'atmosphere, kilopascal'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000395',
         'string_serialization': '{float} - {float} {unit}'} })
    hcr_temp: Optional[TextValue] = Field(default=None, title="hydrocarbon resource original temperature", description="""Original temperature of the hydrocarbon resource""", json_schema_extra = { "linkml_meta": {'alias': 'hcr_temp',
         'aliases': ['hydrocarbon resource original temperature'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value range'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '150-295 degree Celsius'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000393',
         'string_serialization': '{float} - {float} {unit}'} })
    heat_cool_type: Optional[list[HeatCoolTypeEnum]] = Field(default=None, title="heating and cooling system type", description="""Methods of conditioning or heating a room or building""", json_schema_extra = { "linkml_meta": {'alias': 'heat_cool_type',
         'aliases': ['heating and cooling system type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'heat pump'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000766'} })
    heat_deliv_loc: Optional[HeatDelivLocEnum] = Field(default=None, title="heating delivery locations", description="""The location of heat delivery within the room""", json_schema_extra = { "linkml_meta": {'alias': 'heat_deliv_loc',
         'aliases': ['heating delivery locations'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'north'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000810'} })
    heat_sys_deliv_meth: Optional[str] = Field(default=None, title="heating system delivery method", description="""The method by which the heat is delivered through the system""", json_schema_extra = { "linkml_meta": {'alias': 'heat_sys_deliv_meth',
         'aliases': ['heating system delivery method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'radiant'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000812',
         'string_serialization': '[conductive|radiant]'} })
    heat_system_id: Optional[TextValue] = Field(default=None, title="heating system identifier", description="""The heating system identifier""", json_schema_extra = { "linkml_meta": {'alias': 'heat_system_id',
         'aliases': ['heating system identifier'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'unique identifier'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000833'} })
    heavy_metals: Optional[list[TextValue]] = Field(default=None, title="heavy metals/ extreme unusual properties", description="""Heavy metals present in the sample and their concentrations.""", json_schema_extra = { "linkml_meta": {'alias': 'heavy_metals',
         'aliases': ['extreme_unusual_properties/heavy metals'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'heavy metal name;measurement '
                                                     'value unit'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per gram'}},
         'comments': ['For multiple heavy metals and concentrations, separate by ;'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'mercury 0.09 micrograms per gram'},
                      {'value': 'mercury 0.09 ug/g; chromium 0.03 ug/g'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000652',
         'string_serialization': '{text};{float} {unit}',
         'todos': ['Example & validation. Can we configure things so that 27% & 27 % & '
                   '0.27 will validate?',
                   "I think it's weird the way GSC writes the title. I recommend this "
                   "change. Thoughts? I would argue this isn't an extreme unusual "
                   "property. It's just a biogeochemical measurement."]} })
    heavy_metals_meth: Optional[list[TextValue]] = Field(default=None, title="heavy metals method/ extreme unusual properties", description="""Reference or method used in determining heavy metals""", json_schema_extra = { "linkml_meta": {'alias': 'heavy_metals_meth',
         'aliases': ['extreme_unusual_properties/heavy metals method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'comments': ['Required when heavy metals are provided',
                      'If different methods are used for multiple metals, indicate the '
                      'metal and method. Separate metals by ;'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'https://doi.org/10.3390/ijms9040434'},
                      {'value': 'mercury https://doi.org/10.1007/BF01056090; chromium '
                                'https://doi.org/10.1007/s00216-006-0322-8'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000343',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    height_carper_fiber: Optional[QuantityValue] = Field(default=None, title="height carpet fiber mat", description="""The average carpet fiber height in the indoor environment""", json_schema_extra = { "linkml_meta": {'alias': 'height_carper_fiber',
         'aliases': ['height carpet fiber mat'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'centimeter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000167'} })
    herbicide_regm: Optional[list[TextValue]] = Field(default=None, title="herbicide regimen", description="""Information about treatment involving use of herbicides; information about treatment involving use of growth hormones; should include the name of herbicide, amount administered, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple regimens""", json_schema_extra = { "linkml_meta": {'alias': 'herbicide_regm',
         'aliases': ['herbicide regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'herbicide name;herbicide '
                                                     'amount;treatment interval and '
                                                     'duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram, mole per liter, milligram '
                                                     'per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'atrazine;10 milligram per '
                                'liter;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000561',
         'string_serialization': '{text};{float} '
                                 '{unit};{Rn/start_time/end_time/duration}'} })
    horizon_meth: Optional[TextValue] = Field(default=None, title="soil horizon method", description="""Reference or method used in determining the horizon""", json_schema_extra = { "linkml_meta": {'alias': 'horizon_meth',
         'aliases': ['soil horizon method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000321',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    host_age: Optional[QuantityValue] = Field(default=None, title="host age", description="""Age of host at the time of sampling; relevant scale depends on species and study, e.g. Could be seconds for amoebae or centuries for trees""", json_schema_extra = { "linkml_meta": {'alias': 'host_age',
         'aliases': ['host age'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'year, day, hour'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '10 days'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000255'} })
    host_body_habitat: Optional[TextValue] = Field(default=None, title="host body habitat", description="""Original body habitat where the sample was obtained from""", json_schema_extra = { "linkml_meta": {'alias': 'host_body_habitat',
         'aliases': ['host body habitat'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'free text'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'nasopharynx'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000866',
         'string_serialization': '{text}'} })
    host_body_product: Optional[Union[ControlledTermValue,ControlledIdentifiedTermValue]] = Field(default=None, title="host body product", description="""Substance produced by the body, e.g. Stool, mucus, where the sample was obtained from. For foundational model of anatomy ontology (fma) or Uber-anatomy ontology (UBERON) terms, please see https://www.ebi.ac.uk/ols/ontologies/fma or https://www.ebi.ac.uk/ols/ontologies/uberon""", json_schema_extra = { "linkml_meta": {'alias': 'host_body_product',
         'aliases': ['host body product'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'FMA or UBERON'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'mucus [UBERON:0000912]'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000888',
         'string_serialization': '{termLabel} {[termID]}'} })
    host_body_site: Optional[Union[ControlledTermValue,ControlledIdentifiedTermValue]] = Field(default=None, title="host body site", description="""Name of body site where the sample was obtained from, such as a specific organ or tissue (tongue, lung etc...). For foundational model of anatomy ontology (fma) (v 4.11.0) or Uber-anatomy ontology (UBERON) (v releases/2014-06-15) terms, please see http://purl.bioontology.org/ontology/FMA or http://purl.bioontology.org/ontology/UBERON""", json_schema_extra = { "linkml_meta": {'alias': 'host_body_site',
         'aliases': ['host body site'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'FMA or UBERON'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'gill [UBERON:0002535]'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000867',
         'string_serialization': '{termLabel} {[termID]}'} })
    host_body_temp: Optional[QuantityValue] = Field(default=None, title="host body temperature", description="""Core body temperature of the host when sample was collected""", json_schema_extra = { "linkml_meta": {'alias': 'host_body_temp',
         'aliases': ['host body temperature'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '15 degree Celsius'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000274'} })
    host_color: Optional[TextValue] = Field(default=None, title="host color", description="""The color of host""", json_schema_extra = { "linkml_meta": {'alias': 'host_color',
         'aliases': ['host color'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'color'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000260',
         'string_serialization': '{text}'} })
    host_common_name: Optional[TextValue] = Field(default=None, title="host common name", description="""Common name of the host.""", json_schema_extra = { "linkml_meta": {'alias': 'host_common_name',
         'aliases': ['host common name'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'common name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'human'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000248',
         'string_serialization': '{text}'} })
    host_diet: Optional[list[TextValue]] = Field(default=None, title="host diet", description="""Type of diet depending on the host, for animals omnivore, herbivore etc., for humans high-fat, meditteranean etc.; can include multiple diet types""", json_schema_extra = { "linkml_meta": {'alias': 'host_diet',
         'aliases': ['host diet'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'diet type'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'herbivore'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000869',
         'string_serialization': '{text}'} })
    host_dry_mass: Optional[QuantityValue] = Field(default=None, title="host dry mass", description="""Measurement of dry mass""", json_schema_extra = { "linkml_meta": {'alias': 'host_dry_mass',
         'aliases': ['host dry mass'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'kilogram, gram'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '500 gram'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000257'} })
    host_family_relation: Optional[list[str]] = Field(default=None, title="host family relationship", description="""Familial relationships to other hosts in the same study; can include multiple relationships""", json_schema_extra = { "linkml_meta": {'alias': 'host_family_relation',
         'aliases': ['host family relationship'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'relationship type;arbitrary '
                                                     'identifier'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'offspring;Mussel25'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000872',
         'string_serialization': '{text};{text}'} })
    host_genotype: Optional[TextValue] = Field(default=None, title="host genotype", description="""Observed genotype""", json_schema_extra = { "linkml_meta": {'alias': 'host_genotype',
         'aliases': ['host genotype'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'genotype'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'C57BL/6'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000365',
         'string_serialization': '{text}'} })
    host_growth_cond: Optional[TextValue] = Field(default=None, title="host growth conditions", description="""Literature reference giving growth conditions of the host""", json_schema_extra = { "linkml_meta": {'alias': 'host_growth_cond',
         'aliases': ['host growth conditions'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI,url or free text'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'https://academic.oup.com/icesjms/article/68/2/349/617247'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000871',
         'string_serialization': '{PMID}|{DOI}|{URL}|{text}'} })
    host_height: Optional[QuantityValue] = Field(default=None, title="host height", description="""The height of subject""", json_schema_extra = { "linkml_meta": {'alias': 'host_height',
         'aliases': ['host height'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'centimeter, millimeter, meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.1 meter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000264'} })
    host_last_meal: Optional[list[TextValue]] = Field(default=None, title="host last meal", description="""Content of last meal and time since feeding; can include multiple values""", json_schema_extra = { "linkml_meta": {'alias': 'host_last_meal',
         'aliases': ['host last meal'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'content;duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'corn feed;P2H'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000870',
         'string_serialization': '{text};{duration}'} })
    host_length: Optional[QuantityValue] = Field(default=None, title="host length", description="""The length of subject""", json_schema_extra = { "linkml_meta": {'alias': 'host_length',
         'aliases': ['host length'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'centimeter, millimeter, meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '1 meter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000256'} })
    host_life_stage: Optional[TextValue] = Field(default=None, title="host life stage", description="""Description of life stage of host""", json_schema_extra = { "linkml_meta": {'alias': 'host_life_stage',
         'aliases': ['host life stage'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'stage'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'adult'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000251',
         'string_serialization': '{text}'} })
    host_phenotype: Optional[Union[ControlledTermValue,ControlledIdentifiedTermValue]] = Field(default=None, title="host phenotype", description="""Phenotype of human or other host. For phenotypic quality ontology (pato) (v 2018-03-27) terms, please see http://purl.bioontology.org/ontology/pato. For Human Phenotype Ontology (HP) (v 2018-06-13) please see http://purl.bioontology.org/ontology/HP""", json_schema_extra = { "linkml_meta": {'alias': 'host_phenotype',
         'aliases': ['host phenotype'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PATO or HP'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'elongated [PATO:0001154]'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000874',
         'string_serialization': '{termLabel} {[termID]}'} })
    host_sex: Optional[HostSexEnum] = Field(default=None, title="host sex", description="""Gender or physical sex of the host.""", json_schema_extra = { "linkml_meta": {'alias': 'host_sex',
         'aliases': ['host sex'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'non-binary'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000811'} })
    host_shape: Optional[TextValue] = Field(default=None, title="host shape", description="""Morphological shape of host""", json_schema_extra = { "linkml_meta": {'alias': 'host_shape',
         'aliases': ['host shape'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'shape'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'round'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000261',
         'string_serialization': '{text}'} })
    host_subject_id: Optional[TextValue] = Field(default=None, title="host subject id", description="""A unique identifier by which each subject can be referred to, de-identified.""", json_schema_extra = { "linkml_meta": {'alias': 'host_subject_id',
         'aliases': ['host subject id'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'unique identifier'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'MPI123'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000861',
         'string_serialization': '{text}'} })
    host_subspecf_genlin: Optional[list[str]] = Field(default=None, title="host subspecific genetic lineage", description="""Information about the genetic distinctness of the host organism below the subspecies level e.g., serovar, serotype, biotype, ecotype, variety, cultivar, or any relevant genetic typing schemes like Group I plasmid. Subspecies should not be recorded in this term, but in the NCBI taxonomy. Supply both the lineage name and the lineage rank separated by a colon, e.g., biovar:abc123.""", json_schema_extra = { "linkml_meta": {'alias': 'host_subspecf_genlin',
         'aliases': ['host subspecific genetic lineage'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'Genetic lineage below lowest '
                                                     'rank of NCBI taxonomy, which is '
                                                     'subspecies, e.g. serovar, '
                                                     'biotype, ecotype, variety, '
                                                     'cultivar.'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'serovar:Newport, variety:glabrum, cultivar: Red '
                                'Delicious'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001318',
         'string_serialization': '{rank name}:{text}'} })
    host_substrate: Optional[TextValue] = Field(default=None, title="host substrate", description="""The growth substrate of the host.""", json_schema_extra = { "linkml_meta": {'alias': 'host_substrate',
         'aliases': ['host substrate'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'substrate name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'rock'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000252',
         'string_serialization': '{text}'} })
    host_symbiont: Optional[list[str]] = Field(default=None, title="observed host symbionts", description="""The taxonomic name of the organism(s) found living in mutualistic, commensalistic, or parasitic symbiosis with the specific host.""", json_schema_extra = { "linkml_meta": {'alias': 'host_symbiont',
         'aliases': ['observed host symbionts'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'species name or common name'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'flukeworms'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001298',
         'string_serialization': '{text}'} })
    host_taxid: Optional[ControlledIdentifiedTermValue] = Field(default=None, title="host taxid", description="""NCBI taxon id of the host, e.g. 9606""", json_schema_extra = { "linkml_meta": {'alias': 'host_taxid',
         'aliases': ['host taxid'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'NCBI taxon identifier'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'comments': ['Homo sapiens [NCBITaxon:9606] would be a reasonable '
                      'has_raw_value'],
         'domain_of': ['Biosample'],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000250'} })
    host_tot_mass: Optional[QuantityValue] = Field(default=None, title="host total mass", description="""Total mass of the host at collection, the unit depends on host""", json_schema_extra = { "linkml_meta": {'alias': 'host_tot_mass',
         'aliases': ['host total mass'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'kilogram, gram'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2500 gram'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000263'} })
    host_wet_mass: Optional[QuantityValue] = Field(default=None, title="host wet mass", description="""Measurement of wet mass""", json_schema_extra = { "linkml_meta": {'alias': 'host_wet_mass',
         'aliases': ['host wet mass'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'kilogram, gram'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '1500 gram'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000567'} })
    humidity: Optional[QuantityValue] = Field(default=None, title="humidity", description="""Amount of water vapour in the air, at the time of sampling""", json_schema_extra = { "linkml_meta": {'alias': 'humidity',
         'aliases': ['humidity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram per cubic meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '25 gram per cubic meter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000100'} })
    humidity_regm: Optional[list[TextValue]] = Field(default=None, title="humidity regimen", description="""Information about treatment involving an exposure to varying degree of humidity; information about treatment involving use of growth hormones; should include amount of humidity administered, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple regimens""", json_schema_extra = { "linkml_meta": {'alias': 'humidity_regm',
         'aliases': ['humidity regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'humidity value;treatment '
                                                     'interval and duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram per cubic meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '25 gram per cubic '
                                'meter;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000568',
         'string_serialization': '{float} {unit};{Rn/start_time/end_time/duration}'} })
    indoor_space: Optional[IndoorSpaceEnum] = Field(default=None, title="indoor space", description="""A distinguishable space within a structure, the purpose for which discrete areas of a building is used""", json_schema_extra = { "linkml_meta": {'alias': 'indoor_space',
         'aliases': ['indoor space'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'foyer'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000763'} })
    indoor_surf: Optional[IndoorSurfEnum] = Field(default=None, title="indoor surface", description="""Type of indoor surface""", json_schema_extra = { "linkml_meta": {'alias': 'indoor_surf',
         'aliases': ['indoor surface'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'wall'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000764'} })
    indust_eff_percent: Optional[QuantityValue] = Field(default=None, title="industrial effluent percent", description="""Percentage of industrial effluents received by wastewater treatment plant""", json_schema_extra = { "linkml_meta": {'alias': 'indust_eff_percent',
         'aliases': ['industrial effluent percent'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'percentage'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000662'} })
    inorg_particles: Optional[list[TextValue]] = Field(default=None, title="inorganic particles", description="""Concentration of particles such as sand, grit, metal particles, ceramics, etc.; can include multiple particles""", json_schema_extra = { "linkml_meta": {'alias': 'inorg_particles',
         'aliases': ['inorganic particles'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'inorganic particle '
                                                     'name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'mole per liter, milligram per '
                                                     'liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000664',
         'string_serialization': '{text};{float} {unit}'} })
    inside_lux: Optional[QuantityValue] = Field(default=None, title="inside lux light", description="""The recorded value at sampling time (power density)""", json_schema_extra = { "linkml_meta": {'alias': 'inside_lux',
         'aliases': ['inside lux light'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'kilowatt per square metre'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000168'} })
    int_wall_cond: Optional[IntWallCondEnum] = Field(default=None, title="interior wall condition", description="""The physical condition of the wall at the time of sampling; photos or video preferred; use drawings to indicate location of damaged areas""", json_schema_extra = { "linkml_meta": {'alias': 'int_wall_cond',
         'aliases': ['interior wall condition'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'damaged'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000813'} })
    iw_bt_date_well: Optional[TimestampValue] = Field(default=None, title="injection water breakthrough date of specific well", description="""Injection water breakthrough date per well following a secondary and/or tertiary recovery""", json_schema_extra = { "linkml_meta": {'alias': 'iw_bt_date_well',
         'aliases': ['injection water breakthrough date of specific well'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'timestamp'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2018-05-11'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001010'} })
    iwf: Optional[QuantityValue] = Field(default=None, title="injection water fraction", description="""Proportion of the produced fluids derived from injected water at the time of sampling. (e.g. 87%)""", json_schema_extra = { "linkml_meta": {'alias': 'iwf',
         'aliases': ['injection water fraction'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'percent'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000455'} })
    last_clean: Optional[TimestampValue] = Field(default=None, title="last time swept/mopped/vacuumed", description="""The last time the floor was cleaned (swept, mopped, vacuumed)""", json_schema_extra = { "linkml_meta": {'alias': 'last_clean',
         'aliases': ['last time swept/mopped/vacuumed'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'timestamp'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2018-05-11:T14:30Z'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000814'} })
    lat_lon: Optional[GeolocationValue] = Field(default=None, title="geographic location (latitude and longitude)", description="""The geographical origin of the sample as defined by latitude and longitude. The values should be reported in decimal degrees and in WGS84 system""", json_schema_extra = { "linkml_meta": {'alias': 'lat_lon',
         'aliases': ['geographic location (latitude and longitude)'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'decimal degrees,  limit to 8 '
                                                     'decimal points'}},
         'domain_of': ['FieldResearchSite', 'Biosample'],
         'examples': [{'value': '50.586825 6.408977'}],
         'is_a': 'environment field',
         'notes': ["This is currently a required field but it's not clear if this "
                   'should be required for human hosts'],
         'slot_uri': 'MIXS:0000009',
         'string_serialization': '{float} {float}'} })
    light_intensity: Optional[QuantityValue] = Field(default=None, title="light intensity", description="""Measurement of light intensity""", json_schema_extra = { "linkml_meta": {'alias': 'light_intensity',
         'aliases': ['light intensity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'lux'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.3 lux'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000706'} })
    light_regm: Optional[TextValue] = Field(default=None, title="light regimen", description="""Information about treatment(s) involving exposure to light, including both light intensity and quality.""", json_schema_extra = { "linkml_meta": {'alias': 'light_regm',
         'aliases': ['light regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'exposure type;light '
                                                     'intensity;light quality'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'lux; micrometer, nanometer, '
                                                     'angstrom'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'incandescant light;10 lux;450 nanometer'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000569',
         'string_serialization': '{text};{float} {unit};{float} {unit}'} })
    light_type: Optional[list[LightTypeEnum]] = Field(default=None, title="light type", description="""Application of light to achieve some practical or aesthetic effect. Lighting includes the use of both artificial light sources such as lamps and light fixtures, as well as natural illumination by capturing daylight. Can also include absence of light""", json_schema_extra = { "linkml_meta": {'alias': 'light_type',
         'aliases': ['light type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'desk lamp'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000769'} })
    link_addit_analys: Optional[TextValue] = Field(default=None, title="links to additional analysis", description="""Link to additional analysis results performed on the sample""", json_schema_extra = { "linkml_meta": {'alias': 'link_addit_analys',
         'aliases': ['links to additional analysis'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000340',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    link_class_info: Optional[TextValue] = Field(default=None, title="link to classification information", description="""Link to digitized soil maps or other soil classification information""", json_schema_extra = { "linkml_meta": {'alias': 'link_class_info',
         'aliases': ['link to classification information'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000329',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    link_climate_info: Optional[TextValue] = Field(default=None, title="link to climate information", description="""Link to climate resource""", json_schema_extra = { "linkml_meta": {'alias': 'link_climate_info',
         'aliases': ['link to climate information'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000328',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    lithology: Optional[LithologyEnum] = Field(default=None, title="lithology", description="""Hydrocarbon resource main lithology (Additional information: http://petrowiki.org/Lithology_and_rock_type_determination). If \"other\" is specified, please propose entry in \"additional info\" field""", json_schema_extra = { "linkml_meta": {'alias': 'lithology',
         'aliases': ['lithology'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Volcanic'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000990'} })
    local_class: Optional[TextValue] = Field(default=None, title="soil_taxonomic/local classification", description="""Soil classification based on local soil classification system""", json_schema_extra = { "linkml_meta": {'alias': 'local_class',
         'aliases': ['soil_taxonomic/local classification'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'local classification name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['FieldResearchSite', 'Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000330',
         'string_serialization': '{text}'} })
    local_class_meth: Optional[TextValue] = Field(default=None, title="soil_taxonomic/local classification method", description="""Reference or method used in determining the local soil classification""", json_schema_extra = { "linkml_meta": {'alias': 'local_class_meth',
         'aliases': ['soil_taxonomic/local classification method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000331',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    magnesium: Optional[QuantityValue] = Field(default=None, title="magnesium", description="""Concentration of magnesium in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'magnesium',
         'aliases': ['magnesium'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'mole per liter, milligram per '
                                                     'liter, parts per million, '
                                                     'micromole per kilogram'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '52.8 micromole per kilogram'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000431'} })
    max_occup: Optional[QuantityValue] = Field(default=None, title="maximum occupancy", description="""The maximum amount of people allowed in the indoor environment""", json_schema_extra = { "linkml_meta": {'alias': 'max_occup',
         'aliases': ['maximum occupancy'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000229'} })
    mean_frict_vel: Optional[QuantityValue] = Field(default=None, title="mean friction velocity", description="""Measurement of mean friction velocity""", json_schema_extra = { "linkml_meta": {'alias': 'mean_frict_vel',
         'aliases': ['mean friction velocity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'meter per second'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.5 meter per second'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000498'} })
    mean_peak_frict_vel: Optional[QuantityValue] = Field(default=None, title="mean peak friction velocity", description="""Measurement of mean peak friction velocity""", json_schema_extra = { "linkml_meta": {'alias': 'mean_peak_frict_vel',
         'aliases': ['mean peak friction velocity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'meter per second'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '1 meter per second'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000502'} })
    mech_struc: Optional[MechStrucEnum] = Field(default=None, title="mechanical structure", description="""mechanical structure: a moving structure""", json_schema_extra = { "linkml_meta": {'alias': 'mech_struc',
         'aliases': ['mechanical structure'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'elevator'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000815'} })
    mechanical_damage: Optional[list[TextValue]] = Field(default=None, title="mechanical damage", description="""Information about any mechanical damage exerted on the plant; can include multiple damages and sites""", json_schema_extra = { "linkml_meta": {'alias': 'mechanical_damage',
         'aliases': ['mechanical damage'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'damage type;body site'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'pruning;bark'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001052',
         'string_serialization': '{text};{text}'} })
    methane: Optional[QuantityValue] = Field(default=None, title="methane", description="""Methane (gas) amount or concentration at the time of sampling""", json_schema_extra = { "linkml_meta": {'alias': 'methane',
         'aliases': ['methane'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter, parts per '
                                                     'billion, parts per million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '1800 parts per billion'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000101'} })
    micro_biomass_meth: Optional[str] = Field(default=None, title="microbial biomass method", description="""Reference or method used in determining microbial biomass""", json_schema_extra = { "linkml_meta": {'alias': 'micro_biomass_meth',
         'aliases': ['microbial biomass method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000339',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    microbial_biomass: Optional[QuantityValue] = Field(default=None, title="microbial biomass", description="""The part of the organic matter in the soil that constitutes living microorganisms smaller than 5-10 micrometer. If you keep this, you would need to have correction factors used for conversion to the final units""", json_schema_extra = { "linkml_meta": {'alias': 'microbial_biomass',
         'aliases': ['microbial biomass'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'ton, kilogram, gram per kilogram '
                                                     'soil'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000650'} })
    mineral_nutr_regm: Optional[list[TextValue]] = Field(default=None, title="mineral nutrient regimen", description="""Information about treatment involving the use of mineral supplements; should include the name of mineral nutrient, amount administered, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple mineral nutrient regimens""", json_schema_extra = { "linkml_meta": {'alias': 'mineral_nutr_regm',
         'aliases': ['mineral nutrient regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'mineral nutrient name;mineral '
                                                     'nutrient amount;treatment '
                                                     'interval and duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram, mole per liter, milligram '
                                                     'per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'potassium;15 '
                                'gram;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000570',
         'string_serialization': '{text};{float} '
                                 '{unit};{Rn/start_time/end_time/duration}'} })
    misc_param: Optional[list[TextValue]] = Field(default=None, title="miscellaneous parameter", description="""Any other measurement performed or parameter collected, that is not listed here""", json_schema_extra = { "linkml_meta": {'alias': 'misc_param',
         'aliases': ['miscellaneous parameter'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'parameter name;measurement '
                                                     'value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Bicarbonate ion concentration;2075 micromole per '
                                'kilogram'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000752',
         'string_serialization': '{text};{float} {unit}'} })
    n_alkanes: Optional[list[TextValue]] = Field(default=None, title="n-alkanes", description="""Concentration of n-alkanes; can include multiple n-alkanes""", json_schema_extra = { "linkml_meta": {'alias': 'n_alkanes',
         'aliases': ['n-alkanes'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'n-alkane name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'n-hexadecane;100 milligram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000503',
         'string_serialization': '{text};{float} {unit}'} })
    nitrate: Optional[QuantityValue] = Field(default=None, title="nitrate", description="""Concentration of nitrate in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'nitrate',
         'aliases': ['nitrate'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter, milligram '
                                                     'per liter, parts per million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '65 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000425'} })
    nitrite: Optional[QuantityValue] = Field(default=None, title="nitrite", description="""Concentration of nitrite in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'nitrite',
         'aliases': ['nitrite'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter, milligram '
                                                     'per liter, parts per million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.5 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000426'} })
    nitro: Optional[QuantityValue] = Field(default=None, title="nitrogen", description="""Concentration of nitrogen (total)""", json_schema_extra = { "linkml_meta": {'alias': 'nitro',
         'aliases': ['nitrogen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '4.2 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000504'} })
    non_min_nutr_regm: Optional[list[str]] = Field(default=None, title="non-mineral nutrient regimen", description="""Information about treatment involving the exposure of plant to non-mineral nutrient such as oxygen, hydrogen or carbon; should include the name of non-mineral nutrient, amount administered, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple non-mineral nutrient regimens""", json_schema_extra = { "linkml_meta": {'alias': 'non_min_nutr_regm',
         'aliases': ['non-mineral nutrient regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'non-mineral nutrient '
                                                     'name;non-mineral nutrient '
                                                     'amount;treatment interval and '
                                                     'duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram, mole per liter, milligram '
                                                     'per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'carbon dioxide;10 mole per '
                                'liter;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000571',
         'string_serialization': '{text};{float} '
                                 '{unit};{Rn/start_time/end_time/duration}'} })
    number_pets: Optional[QuantityValue] = Field(default=None, title="number of pets", description="""The number of pets residing in the sampled space""", json_schema_extra = { "linkml_meta": {'alias': 'number_pets',
         'aliases': ['number of pets'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000231'} })
    number_plants: Optional[QuantityValue] = Field(default=None, title="number of houseplants", description="""The number of plant(s) in the sampling space""", json_schema_extra = { "linkml_meta": {'alias': 'number_plants',
         'aliases': ['number of houseplants'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000230'} })
    number_resident: Optional[QuantityValue] = Field(default=None, title="number of residents", description="""The number of individuals currently occupying in the sampling location""", json_schema_extra = { "linkml_meta": {'alias': 'number_resident',
         'aliases': ['number of residents'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000232'} })
    occup_density_samp: Optional[QuantityValue] = Field(default=None, title="occupant density at sampling", description="""Average number of occupants at time of sampling per square footage""", json_schema_extra = { "linkml_meta": {'alias': 'occup_density_samp',
         'aliases': ['occupant density at sampling'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000217'} })
    occup_document: Optional[OccupDocumentEnum] = Field(default=None, title="occupancy documentation", description="""The type of documentation of occupancy""", json_schema_extra = { "linkml_meta": {'alias': 'occup_document',
         'aliases': ['occupancy documentation'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'estimate'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000816'} })
    occup_samp: Optional[QuantityValue] = Field(default=None, title="occupancy at sampling", description="""Number of occupants present at time of sample within the given space""", json_schema_extra = { "linkml_meta": {'alias': 'occup_samp',
         'aliases': ['occupancy at sampling'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '10'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000772'} })
    org_carb: Optional[QuantityValue] = Field(default=None, title="organic carbon", description="""Concentration of organic carbon""", json_schema_extra = { "linkml_meta": {'alias': 'org_carb',
         'aliases': ['organic carbon'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '1.5 microgram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000508'} })
    org_count_qpcr_info: Optional[str] = Field(default=None, title="organism count qPCR information", description="""If qpcr was used for the cell count, the target gene name, the primer sequence and the cycling conditions should also be provided. (Example: 16S rrna; FWD:ACGTAGCTATGACGT REV:GTGCTAGTCGAGTAC; initial denaturation:90C_5min; denaturation:90C_2min; annealing:52C_30 sec; elongation:72C_30 sec; 90 C for 1 min; final elongation:72C_5min; 30 cycles)""", json_schema_extra = { "linkml_meta": {'alias': 'org_count_qpcr_info',
         'aliases': ['organism count qPCR information'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'gene name;FWD:forward primer '
                                                     'sequence;REV:reverse primer '
                                                     'sequence;initial '
                                                     'denaturation:degrees_minutes;denaturation:degrees_minutes;annealing:degrees_minutes;elongation:degrees_minutes;final '
                                                     'elongation:degrees_minutes; '
                                                     'total cycles'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'number of cells per gram (or ml '
                                                     'or cm^2)'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000099',
         'string_serialization': '{text};FWD:{dna};REV:{dna};initial '
                                 'denaturation:degrees_minutes;denaturation:degrees_minutes;annealing:degrees_minutes;elongation:degrees_minutes;final '
                                 'elongation:degrees_minutes; total cycles'} })
    org_matter: Optional[QuantityValue] = Field(default=None, title="organic matter", description="""Concentration of organic matter""", json_schema_extra = { "linkml_meta": {'alias': 'org_matter',
         'aliases': ['organic matter'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '1.75 milligram per cubic meter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000204'} })
    org_nitro: Optional[QuantityValue] = Field(default=None, title="organic nitrogen", description="""Concentration of organic nitrogen""", json_schema_extra = { "linkml_meta": {'alias': 'org_nitro',
         'aliases': ['organic nitrogen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '4 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000205'} })
    org_particles: Optional[list[TextValue]] = Field(default=None, title="organic particles", description="""Concentration of particles such as faeces, hairs, food, vomit, paper fibers, plant material, humus, etc.""", json_schema_extra = { "linkml_meta": {'alias': 'org_particles',
         'aliases': ['organic particles'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'particle name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000665',
         'string_serialization': '{text};{float} {unit}'} })
    organism_count: Optional[list[QuantityValue]] = Field(default=None, title="organism count", description="""Total cell count of any organism (or group of organisms) per gram, volume or area of sample, should include name of organism followed by count. The method that was used for the enumeration (e.g. qPCR, atp, mpn, etc.) Should also be provided. (example: total prokaryotes; 3.5e7 cells per ml; qpcr)""", json_schema_extra = { "linkml_meta": {'alias': 'organism_count',
         'aliases': ['organism count'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'organism name;measurement '
                                                     'value;enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'number of cells per cubic meter, '
                                                     'number of cells per milliliter, '
                                                     'number of cells per cubic '
                                                     'centimeter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'total prokaryotes;3.5e7 cells per milliliter;qPCR'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000103'} })
    owc_tvdss: Optional[QuantityValue] = Field(default=None, title="oil water contact depth", description="""Depth of the original oil water contact (OWC) zone (average) (m TVDSS)""", json_schema_extra = { "linkml_meta": {'alias': 'owc_tvdss',
         'aliases': ['oil water contact depth'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000405'} })
    oxy_stat_samp: Optional[OxyStatSampEnum] = Field(default=None, title="oxygenation status of sample", description="""Oxygenation status of sample""", json_schema_extra = { "linkml_meta": {'alias': 'oxy_stat_samp',
         'aliases': ['oxygenation status of sample'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'aerobic'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000753'} })
    oxygen: Optional[QuantityValue] = Field(default=None, title="oxygen", description="""Oxygen (gas) amount or concentration at the time of sampling""", json_schema_extra = { "linkml_meta": {'alias': 'oxygen',
         'aliases': ['oxygen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter, parts per '
                                                     'million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '600 parts per million'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000104'} })
    part_org_carb: Optional[QuantityValue] = Field(default=None, title="particulate organic carbon", description="""Concentration of particulate organic carbon""", json_schema_extra = { "linkml_meta": {'alias': 'part_org_carb',
         'aliases': ['particulate organic carbon'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '1.92 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000515'} })
    part_org_nitro: Optional[QuantityValue] = Field(default=None, title="particulate organic nitrogen", description="""Concentration of particulate organic nitrogen""", json_schema_extra = { "linkml_meta": {'alias': 'part_org_nitro',
         'aliases': ['particulate organic nitrogen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per liter, micromole '
                                                     'per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.3 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000719'} })
    particle_class: Optional[list[TextValue]] = Field(default=None, title="particle classification", description="""Particles are classified, based on their size, into six general categories:clay, silt, sand, gravel, cobbles, and boulders; should include amount of particle preceded by the name of the particle type; can include multiple values""", json_schema_extra = { "linkml_meta": {'alias': 'particle_class',
         'aliases': ['particle classification'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'particle name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micrometer'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000206',
         'string_serialization': '{text};{float} {unit}'} })
    permeability: Optional[TextValue] = Field(default=None, title="permeability", description="""Measure of the ability of a hydrocarbon resource to allow fluids to pass through it. (Additional information: https://en.wikipedia.org/wiki/Permeability_(earth_sciences))""", json_schema_extra = { "linkml_meta": {'alias': 'permeability',
         'aliases': ['permeability'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value range'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'mD'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000404',
         'string_serialization': '{integer} - {integer} {unit}'} })
    perturbation: Optional[list[TextValue]] = Field(default=None, title="perturbation", description="""Type of perturbation, e.g. chemical administration, physical disturbance, etc., coupled with perturbation regimen including how many times the perturbation was repeated, how long each perturbation lasted, and the start and end time of the entire perturbation period; can include multiple perturbation types""", json_schema_extra = { "linkml_meta": {'alias': 'perturbation',
         'aliases': ['perturbation'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'perturbation type '
                                                     'name;perturbation interval and '
                                                     'duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'antibiotic '
                                'addition;R2/2018-05-11T14:30Z/2018-05-11T19:30Z/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000754',
         'string_serialization': '{text};{Rn/start_time/end_time/duration}'} })
    pesticide_regm: Optional[list[TextValue]] = Field(default=None, title="pesticide regimen", description="""Information about treatment involving use of insecticides; should include the name of pesticide, amount administered, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple pesticide regimens""", json_schema_extra = { "linkml_meta": {'alias': 'pesticide_regm',
         'aliases': ['pesticide regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'pesticide name;pesticide '
                                                     'amount;treatment interval and '
                                                     'duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram, mole per liter, milligram '
                                                     'per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'pyrethrum;0.6 milligram per '
                                'liter;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000573',
         'string_serialization': '{text};{float} '
                                 '{unit};{Rn/start_time/end_time/duration}'} })
    petroleum_hydrocarb: Optional[QuantityValue] = Field(default=None, title="petroleum hydrocarbon", description="""Concentration of petroleum hydrocarbon""", json_schema_extra = { "linkml_meta": {'alias': 'petroleum_hydrocarb',
         'aliases': ['petroleum hydrocarbon'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.05 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000516'} })
    ph: Optional[float] = Field(default=None, title="pH", description="""Ph measurement of the sample, or liquid portion of sample, or aqueous phase of the fluid""", json_schema_extra = { "linkml_meta": {'alias': 'ph',
         'aliases': ['pH'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '7.2'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001001'} })
    ph_meth: Optional[TextValue] = Field(default=None, title="pH method", description="""Reference or method used in determining ph""", json_schema_extra = { "linkml_meta": {'alias': 'ph_meth',
         'aliases': ['pH method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'comments': ['This can include a link to the instrument used or a citation '
                      'for the method.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'https://www.southernlabware.com/pc9500-benchtop-ph-conductivity-meter-kit-ph-accuracy-2000mv-ph-range-2-000-to-20-000.html?gclid=Cj0KCQiAwJWdBhCYARIsAJc4idCO5vtvbVMf545fcvdROFqa6zjzNSoywNx6K4k9Coo9cCc2pybtvGsaAiR0EALw_wcB'},
                      {'value': 'https://doi.org/10.2136/sssabookser5.3.c16'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001106',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    ph_regm: Optional[list[TextValue]] = Field(default=None, title="pH regimen", description="""Information about treatment involving exposure of plants to varying levels of ph of the growth media, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple regimen""", json_schema_extra = { "linkml_meta": {'alias': 'ph_regm',
         'aliases': ['pH regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value;treatment '
                                                     'interval and duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '7.6;R2/2018-05-11:T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001056',
         'string_serialization': '{float};{Rn/start_time/end_time/duration}'} })
    phaeopigments: Optional[list[TextValue]] = Field(default=None, title="phaeopigments", description="""Concentration of phaeopigments; can include multiple phaeopigments""", json_schema_extra = { "linkml_meta": {'alias': 'phaeopigments',
         'aliases': ['phaeopigments'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'phaeopigment name;measurement '
                                                     'value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per cubic meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2.5 milligram per cubic meter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000180',
         'string_serialization': '{text};{float} {unit}'} })
    phosphate: Optional[QuantityValue] = Field(default=None, title="phosphate", description="""Concentration of phosphate""", json_schema_extra = { "linkml_meta": {'alias': 'phosphate',
         'aliases': ['phosphate'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.7 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000505'} })
    phosplipid_fatt_acid: Optional[list[TextValue]] = Field(default=None, title="phospholipid fatty acid", description="""Concentration of phospholipid fatty acids; can include multiple values""", json_schema_extra = { "linkml_meta": {'alias': 'phosplipid_fatt_acid',
         'aliases': ['phospholipid fatty acid'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'phospholipid fatty acid '
                                                     'name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'mole per gram, mole per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2.98 milligram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000181',
         'string_serialization': '{text};{float} {unit}'} })
    photon_flux: Optional[QuantityValue] = Field(default=None, title="photon flux", description="""Measurement of photon flux""", json_schema_extra = { "linkml_meta": {'alias': 'photon_flux',
         'aliases': ['photon flux'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'number of photons per second per '
                                                     'unit area'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '3.926 micromole photons per second per square meter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000725'} })
    plant_growth_med: Optional[Union[ControlledTermValue,ControlledIdentifiedTermValue]] = Field(default=None, title="plant growth medium", description="""Specification of the media for growing the plants or tissue cultured samples, e.g. soil, aeroponic, hydroponic, in vitro solid culture medium, in vitro liquid culture medium. Recommended value is a specific value from EO:plant growth medium (follow this link for terms http://purl.obolibrary.org/obo/EO_0007147) or other controlled vocabulary""", json_schema_extra = { "linkml_meta": {'alias': 'plant_growth_med',
         'aliases': ['plant growth medium'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'EO or enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'hydroponic plant culture media [EO:0007067]'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001057'} })
    plant_product: Optional[TextValue] = Field(default=None, title="plant product", description="""Substance produced by the plant, where the sample was obtained from""", json_schema_extra = { "linkml_meta": {'alias': 'plant_product',
         'aliases': ['plant product'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'product name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'xylem sap [PO:0025539]'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001058',
         'string_serialization': '{text}'} })
    plant_sex: Optional[PlantSexEnum] = Field(default=None, title="plant sex", description="""Sex of the reproductive parts on the whole plant, e.g. pistillate, staminate, monoecieous, hermaphrodite.""", json_schema_extra = { "linkml_meta": {'alias': 'plant_sex',
         'aliases': ['plant sex'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Hermaphroditic'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001059'} })
    plant_struc: Optional[Union[ControlledTermValue,ControlledIdentifiedTermValue]] = Field(default=None, title="plant structure", description="""Name of plant structure the sample was obtained from; for Plant Ontology (PO) (v releases/2017-12-14) terms, see http://purl.bioontology.org/ontology/PO, e.g. petiole epidermis (PO_0000051). If an individual flower is sampled, the sex of it can be recorded here.""", json_schema_extra = { "linkml_meta": {'alias': 'plant_struc',
         'aliases': ['plant structure'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'PO'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'epidermis [PO:0005679]'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001060',
         'string_serialization': '{termLabel} {[termID]}'} })
    pollutants: Optional[list[TextValue]] = Field(default=None, title="pollutants", description="""Pollutant types and, amount or concentrations measured at the time of sampling; can report multiple pollutants by entering numeric values preceded by name of pollutant""", json_schema_extra = { "linkml_meta": {'alias': 'pollutants',
         'aliases': ['pollutants'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'pollutant name;measurement '
                                                     'value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram, mole per liter, milligram '
                                                     'per liter, microgram per cubic '
                                                     'meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'lead;0.15 microgram per cubic meter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000107',
         'string_serialization': '{text};{float} {unit}'} })
    porosity: Optional[TextValue] = Field(default=None, title="porosity", description="""Porosity of deposited sediment is volume of voids divided by the total volume of sample""", json_schema_extra = { "linkml_meta": {'alias': 'porosity',
         'aliases': ['porosity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value or range'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'percentage'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000211',
         'string_serialization': '{float} - {float} {unit}'} })
    potassium: Optional[QuantityValue] = Field(default=None, title="potassium", description="""Concentration of potassium in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'potassium',
         'aliases': ['potassium'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter, parts per '
                                                     'million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '463 milligram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000430'} })
    pour_point: Optional[QuantityValue] = Field(default=None, title="pour point", description="""Temperature at which a liquid becomes semi solid and loses its flow characteristics. In crude oil a high¬†pour point¬†is generally associated with a high paraffin content, typically found in crude deriving from a larger proportion of plant material. (soure: https://en.wikipedia.org/wiki/pour_point)""", json_schema_extra = { "linkml_meta": {'alias': 'pour_point',
         'aliases': ['pour point'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000127'} })
    pre_treatment: Optional[TextValue] = Field(default=None, title="pre-treatment", description="""The process of pre-treatment removes materials that can be easily collected from the raw wastewater""", json_schema_extra = { "linkml_meta": {'alias': 'pre_treatment',
         'aliases': ['pre-treatment'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'pre-treatment type'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000348',
         'string_serialization': '{text}'} })
    pres_animal_insect: Optional[str] = Field(default=None, title="presence of pets, animals, or insects", description="""The type and number of animals or insects present in the sampling space.""", json_schema_extra = { "linkml_meta": {'alias': 'pres_animal_insect',
         'aliases': ['presence of pets, animals, or insects'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration;count'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'cat;5'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000819'} })
    pressure: Optional[QuantityValue] = Field(default=None, title="pressure", description="""Pressure to which the sample is subject to, in atmospheres""", json_schema_extra = { "linkml_meta": {'alias': 'pressure',
         'aliases': ['pressure'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'atmosphere'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '50 atmosphere'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000412'} })
    prev_land_use_meth: Optional[str] = Field(default=None, title="history/previous land use method", description="""Reference or method used in determining previous land use and dates""", json_schema_extra = { "linkml_meta": {'alias': 'prev_land_use_meth',
         'aliases': ['history/previous land use method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000316',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    previous_land_use: Optional[TextValue] = Field(default=None, title="history/previous land use", description="""Previous land use and dates""", json_schema_extra = { "linkml_meta": {'alias': 'previous_land_use',
         'aliases': ['history/previous land use'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'land use name;date'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000315',
         'string_serialization': '{text};{timestamp}'} })
    primary_prod: Optional[QuantityValue] = Field(default=None, title="primary production", description="""Measurement of primary production, generally measured as isotope uptake""", json_schema_extra = { "linkml_meta": {'alias': 'primary_prod',
         'aliases': ['primary production'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per cubic meter per '
                                                     'day, gram per square meter per '
                                                     'day'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '100 milligram per cubic meter per day'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000728'} })
    primary_treatment: Optional[TextValue] = Field(default=None, title="primary treatment", description="""The process to produce both a generally homogeneous liquid capable of being treated biologically and a sludge that can be separately treated or processed""", json_schema_extra = { "linkml_meta": {'alias': 'primary_treatment',
         'aliases': ['primary treatment'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'primary treatment type'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000349',
         'string_serialization': '{text}'} })
    prod_rate: Optional[QuantityValue] = Field(default=None, title="production rate", description="""Oil and/or gas production rates per well (e.g. 524 m3 / day)""", json_schema_extra = { "linkml_meta": {'alias': 'prod_rate',
         'aliases': ['production rate'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'cubic meter per day'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000452'} })
    prod_start_date: Optional[TimestampValue] = Field(default=None, title="production start date", description="""Date of field's first production""", json_schema_extra = { "linkml_meta": {'alias': 'prod_start_date',
         'aliases': ['production start date'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'timestamp'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2018-05-11'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001008'} })
    profile_position: Optional[ProfilePositionEnum] = Field(default=None, title="profile position", description="""Cross-sectional position in the hillslope where sample was collected.sample area position in relation to surrounding areas""", json_schema_extra = { "linkml_meta": {'alias': 'profile_position',
         'aliases': ['profile position'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'summit'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001084'} })
    quad_pos: Optional[QuadPosEnum] = Field(default=None, title="quadrant position", description="""The quadrant position of the sampling room within the building""", json_schema_extra = { "linkml_meta": {'alias': 'quad_pos',
         'aliases': ['quadrant position'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'West side'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000820'} })
    radiation_regm: Optional[list[TextValue]] = Field(default=None, title="radiation regimen", description="""Information about treatment involving exposure of plant or a plant part to a particular radiation regimen; should include the radiation type, amount or intensity administered, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple radiation regimens""", json_schema_extra = { "linkml_meta": {'alias': 'radiation_regm',
         'aliases': ['radiation regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'radiation type name;radiation '
                                                     'amount;treatment interval and '
                                                     'duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'rad, gray'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'gamma radiation;60 '
                                'gray;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000575',
         'string_serialization': '{text};{float} '
                                 '{unit};{Rn/start_time/end_time/duration}'} })
    rainfall_regm: Optional[list[TextValue]] = Field(default=None, title="rainfall regimen", description="""Information about treatment involving an exposure to a given amount of rainfall, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple regimens""", json_schema_extra = { "linkml_meta": {'alias': 'rainfall_regm',
         'aliases': ['rainfall regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value;treatment '
                                                     'interval and duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'millimeter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '15 '
                                'millimeter;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000576',
         'string_serialization': '{float} {unit};{Rn/start_time/end_time/duration}'} })
    reactor_type: Optional[TextValue] = Field(default=None, title="reactor type", description="""Anaerobic digesters can be designed and engineered to operate using a number of different process configurations, as batch or continuous, mesophilic, high solid or low solid, and single stage or multistage""", json_schema_extra = { "linkml_meta": {'alias': 'reactor_type',
         'aliases': ['reactor type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'reactor type name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000350',
         'string_serialization': '{text}'} })
    redox_potential: Optional[QuantityValue] = Field(default=None, title="redox potential", description="""Redox potential, measured relative to a hydrogen cell, indicating oxidation or reduction potential""", json_schema_extra = { "linkml_meta": {'alias': 'redox_potential',
         'aliases': ['redox potential'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'millivolt'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '300 millivolt'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000182'} })
    rel_air_humidity: Optional[QuantityValue] = Field(default=None, title="relative air humidity", description="""Partial vapor and air pressure, density of the vapor and air, or by the actual mass of the vapor and air""", json_schema_extra = { "linkml_meta": {'alias': 'rel_air_humidity',
         'aliases': ['relative air humidity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'percentage'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '80%'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000121'} })
    rel_humidity_out: Optional[QuantityValue] = Field(default=None, title="outside relative humidity", description="""The recorded outside relative humidity value at the time of sampling""", json_schema_extra = { "linkml_meta": {'alias': 'rel_humidity_out',
         'aliases': ['outside relative humidity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram of air, kilogram of air'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '12 per kilogram of air'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000188'} })
    rel_samp_loc: Optional[RelSampLocEnum] = Field(default=None, title="relative sampling location", description="""The sampling location within the train car""", json_schema_extra = { "linkml_meta": {'alias': 'rel_samp_loc',
         'aliases': ['relative sampling location'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'center of car'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000821'} })
    reservoir: Optional[TextValue] = Field(default=None, title="reservoir name", description="""Name of the reservoir (e.g. Carapebus)""", json_schema_extra = { "linkml_meta": {'alias': 'reservoir',
         'aliases': ['reservoir name'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000303',
         'string_serialization': '{text}'} })
    resins_pc: Optional[TextValue] = Field(default=None, title="resins wt%", description="""Saturate, Aromatic, Resin and Asphaltene¬†(SARA) is an analysis method that divides¬†crude oil¬†components according to their polarizability and polarity. There are three main methods to obtain SARA results. The most popular one is known as the Iatroscan TLC-FID and is referred to as IP-143 (source: https://en.wikipedia.org/wiki/Saturate,_aromatic,_resin_and_asphaltene)""", json_schema_extra = { "linkml_meta": {'alias': 'resins_pc',
         'aliases': ['resins wt%'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'percent'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000134',
         'string_serialization': '{text};{float} {unit}'} })
    room_air_exch_rate: Optional[QuantityValue] = Field(default=None, title="room air exchange rate", description="""The rate at which outside air replaces indoor air in a given space""", json_schema_extra = { "linkml_meta": {'alias': 'room_air_exch_rate',
         'aliases': ['room air exchange rate'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'liter per hour'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000169'} })
    room_architec_elem: Optional[str] = Field(default=None, title="room architectural elements", description="""The unique details and component parts that, together, form the architecture of a distinguisahable space within a built structure""", json_schema_extra = { "linkml_meta": {'alias': 'room_architec_elem',
         'aliases': ['room architectural elements'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'free text'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000233',
         'string_serialization': '{text}'} })
    room_condt: Optional[RoomCondtEnum] = Field(default=None, title="room condition", description="""The condition of the room at the time of sampling""", json_schema_extra = { "linkml_meta": {'alias': 'room_condt',
         'aliases': ['room condition'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'new'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000822'} })
    room_connected: Optional[RoomConnectedEnum] = Field(default=None, title="rooms connected by a doorway", description="""List of rooms connected to the sampling room by a doorway""", json_schema_extra = { "linkml_meta": {'alias': 'room_connected',
         'aliases': ['rooms connected by a doorway'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'office'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000826'} })
    room_count: Optional[TextValue] = Field(default=None, title="room count", description="""The total count of rooms in the built structure including all room types""", json_schema_extra = { "linkml_meta": {'alias': 'room_count',
         'aliases': ['room count'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000234'} })
    room_dim: Optional[TextValue] = Field(default=None, title="room dimensions", description="""The length, width and height of sampling room""", json_schema_extra = { "linkml_meta": {'alias': 'room_dim',
         'aliases': ['room dimensions'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000192',
         'string_serialization': '{integer} {unit} x {integer} {unit} x {integer} '
                                 '{unit}'} })
    room_door_dist: Optional[TextValue] = Field(default=None, title="room door distance", description="""Distance between doors (meters) in the hallway between the sampling room and adjacent rooms""", json_schema_extra = { "linkml_meta": {'alias': 'room_door_dist',
         'aliases': ['room door distance'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000193',
         'string_serialization': '{integer} {unit}'} })
    room_door_share: Optional[TextValue] = Field(default=None, title="rooms that share a door with sampling room", description="""List of room(s) (room number, room name) sharing a door with the sampling room""", json_schema_extra = { "linkml_meta": {'alias': 'room_door_share',
         'aliases': ['rooms that share a door with sampling room'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'room name;room number'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000242',
         'string_serialization': '{text};{integer}'} })
    room_hallway: Optional[TextValue] = Field(default=None, title="rooms that are on the same hallway", description="""List of room(s) (room number, room name) located in the same hallway as sampling room""", json_schema_extra = { "linkml_meta": {'alias': 'room_hallway',
         'aliases': ['rooms that are on the same hallway'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'room name;room number'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000238',
         'string_serialization': '{text};{integer}'} })
    room_loc: Optional[RoomLocEnum] = Field(default=None, title="room location in building", description="""The position of the room within the building""", json_schema_extra = { "linkml_meta": {'alias': 'room_loc',
         'aliases': ['room location in building'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'interior room'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000823'} })
    room_moist_dam_hist: Optional[int] = Field(default=None, title="room moisture damage or mold history", description="""The history of moisture damage or mold in the past 12 months. Number of events of moisture damage or mold observed""", json_schema_extra = { "linkml_meta": {'alias': 'room_moist_dam_hist',
         'aliases': ['room moisture damage or mold history'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000235'} })
    room_net_area: Optional[TextValue] = Field(default=None, title="room net area", description="""The net floor area of sampling room. Net area excludes wall thicknesses""", json_schema_extra = { "linkml_meta": {'alias': 'room_net_area',
         'aliases': ['room net area'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'square feet, square meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000194',
         'string_serialization': '{integer} {unit}'} })
    room_occup: Optional[QuantityValue] = Field(default=None, title="room occupancy", description="""Count of room occupancy at time of sampling""", json_schema_extra = { "linkml_meta": {'alias': 'room_occup',
         'aliases': ['room occupancy'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000236'} })
    room_samp_pos: Optional[RoomSampPosEnum] = Field(default=None, title="room sampling position", description="""The horizontal sampling position in the room relative to architectural elements""", json_schema_extra = { "linkml_meta": {'alias': 'room_samp_pos',
         'aliases': ['room sampling position'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'south corner'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000824'} })
    room_type: Optional[RoomTypeEnum] = Field(default=None, title="room type", description="""The main purpose or activity of the sampling room. A room is any distinguishable space within a structure""", json_schema_extra = { "linkml_meta": {'alias': 'room_type',
         'aliases': ['room type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'bathroom'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000825'} })
    room_vol: Optional[TextValue] = Field(default=None, title="room volume", description="""Volume of sampling room""", json_schema_extra = { "linkml_meta": {'alias': 'room_vol',
         'aliases': ['room volume'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'cubic feet, cubic meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000195',
         'string_serialization': '{integer} {unit}'} })
    room_wall_share: Optional[TextValue] = Field(default=None, title="rooms that share a wall with sampling room", description="""List of room(s) (room number, room name) sharing a wall with the sampling room""", json_schema_extra = { "linkml_meta": {'alias': 'room_wall_share',
         'aliases': ['rooms that share a wall with sampling room'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'room name;room number'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000243',
         'string_serialization': '{text};{integer}'} })
    room_window_count: Optional[int] = Field(default=None, title="room window count", description="""Number of windows in the room""", json_schema_extra = { "linkml_meta": {'alias': 'room_window_count',
         'aliases': ['room window count'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000237'} })
    root_cond: Optional[TextValue] = Field(default=None, title="rooting conditions", description="""Relevant rooting conditions such as field plot size, sowing density, container dimensions, number of plants per container.""", json_schema_extra = { "linkml_meta": {'alias': 'root_cond',
         'aliases': ['rooting conditions'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI,url or free text'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'http://himedialabs.com/TD/PT158.pdf'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001061',
         'string_serialization': '{PMID}|{DOI}|{URL}|{text}'} })
    root_med_carbon: Optional[TextValue] = Field(default=None, title="rooting medium carbon", description="""Source of organic carbon in the culture rooting medium; e.g. sucrose.""", json_schema_extra = { "linkml_meta": {'alias': 'root_med_carbon',
         'aliases': ['rooting medium carbon'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'carbon source name;measurement '
                                                     'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'sucrose'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000577',
         'string_serialization': '{text};{float} {unit}'} })
    root_med_macronutr: Optional[TextValue] = Field(default=None, title="rooting medium macronutrients", description="""Measurement of the culture rooting medium macronutrients (N,P, K, Ca, Mg, S); e.g. KH2PO4 (170¬†mg/L).""", json_schema_extra = { "linkml_meta": {'alias': 'root_med_macronutr',
         'aliases': ['rooting medium macronutrients'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'macronutrient name;measurement '
                                                     'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'KH2PO4;170¬†milligram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000578',
         'string_serialization': '{text};{float} {unit}'} })
    root_med_micronutr: Optional[TextValue] = Field(default=None, title="rooting medium micronutrients", description="""Measurement of the culture rooting medium micronutrients (Fe, Mn, Zn, B, Cu, Mo); e.g. H3BO3 (6.2¬†mg/L).""", json_schema_extra = { "linkml_meta": {'alias': 'root_med_micronutr',
         'aliases': ['rooting medium micronutrients'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'micronutrient name;measurement '
                                                     'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'H3BO3;6.2¬†milligram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000579',
         'string_serialization': '{text};{float} {unit}'} })
    root_med_ph: Optional[QuantityValue] = Field(default=None, title="rooting medium pH", description="""pH measurement of the culture rooting medium; e.g. 5.5.""", json_schema_extra = { "linkml_meta": {'alias': 'root_med_ph',
         'aliases': ['rooting medium pH'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '7.5'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001062'} })
    root_med_regl: Optional[TextValue] = Field(default=None, title="rooting medium regulators", description="""Growth regulators in the culture rooting medium such as cytokinins, auxins, gybberellins, abscisic acid; e.g. 0.5¬†mg/L NAA.""", json_schema_extra = { "linkml_meta": {'alias': 'root_med_regl',
         'aliases': ['rooting medium regulators'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'regulator name;measurement '
                                                     'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'abscisic acid;0.75 milligram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000581',
         'string_serialization': '{text};{float} {unit}'} })
    root_med_solid: Optional[TextValue] = Field(default=None, title="rooting medium solidifier", description="""Specification of the solidifying agent in the culture rooting medium; e.g. agar.""", json_schema_extra = { "linkml_meta": {'alias': 'root_med_solid',
         'aliases': ['rooting medium solidifier'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'agar'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001063',
         'string_serialization': '{text}'} })
    root_med_suppl: Optional[TextValue] = Field(default=None, title="rooting medium organic supplements", description="""Organic supplements of the culture rooting medium, such as vitamins, amino acids, organic acids, antibiotics activated charcoal; e.g. nicotinic acid (0.5¬†mg/L).""", json_schema_extra = { "linkml_meta": {'alias': 'root_med_suppl',
         'aliases': ['rooting medium organic supplements'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'supplement name;measurement '
                                                     'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'nicotinic acid;0.5 milligram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000580',
         'string_serialization': '{text};{float} {unit}'} })
    salinity: Optional[QuantityValue] = Field(default=None, title="salinity", description="""The total concentration of all dissolved salts in a liquid or solid sample. While salinity can be measured by a complete chemical analysis, this method is difficult and time consuming. More often, it is instead derived from the conductivity measurement. This is known as practical salinity. These derivations compare the specific conductance of the sample to a salinity standard such as seawater.""", json_schema_extra = { "linkml_meta": {'alias': 'salinity',
         'aliases': ['salinity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'practical salinity unit, '
                                                     'percentage'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '25 practical salinity unit'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000183'} })
    salinity_meth: Optional[TextValue] = Field(default=None, title="salinity method", description="""Reference or method used in determining salinity""", json_schema_extra = { "linkml_meta": {'alias': 'salinity_meth',
         'aliases': ['salinity method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'https://doi.org/10.1007/978-1-61779-986-0_28'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000341',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    salt_regm: Optional[list[TextValue]] = Field(default=None, title="salt regimen", description="""Information about treatment involving use of salts as supplement to liquid and soil growth media; should include the name of salt, amount administered, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple salt regimens""", json_schema_extra = { "linkml_meta": {'alias': 'salt_regm',
         'aliases': ['salt regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'salt name;salt amount;treatment '
                                                     'interval and duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram, microgram, mole per liter, '
                                                     'gram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'NaCl;5 gram per '
                                'liter;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000582',
         'string_serialization': '{text};{float} '
                                 '{unit};{Rn/start_time/end_time/duration}'} })
    samp_capt_status: Optional[SampCaptStatusEnum] = Field(default=None, title="sample capture status", description="""Reason for the sample""", json_schema_extra = { "linkml_meta": {'alias': 'samp_capt_status',
         'aliases': ['sample capture status'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'farm sample'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000860'} })
    samp_collec_device: Optional[str] = Field(default=None, title="sample collection device", description="""The device used to collect an environmental sample. This field accepts terms listed under environmental sampling device (http://purl.obolibrary.org/obo/ENVO). This field also accepts terms listed under specimen collection device (http://purl.obolibrary.org/obo/GENEPIO_0002094).""", json_schema_extra = { "linkml_meta": {'alias': 'samp_collec_device',
         'aliases': ['sample collection device'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'device name'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'swab, biopsy, niskin bottle, push core, drag swab '
                                '[GENEPIO:0002713]'}],
         'is_a': 'nucleic acid sequence source field',
         'slot_uri': 'MIXS:0000002',
         'string_serialization': '{termLabel} {[termID]}|{text}'} })
    samp_collec_method: Optional[str] = Field(default=None, title="sample collection method", description="""The method employed for collecting the sample.""", json_schema_extra = { "linkml_meta": {'alias': 'samp_collec_method',
         'aliases': ['sample collection method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI,url , or text'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'swabbing'}],
         'is_a': 'nucleic acid sequence source field',
         'slot_uri': 'MIXS:0001225',
         'string_serialization': '{PMID}|{DOI}|{URL}|{text}'} })
    samp_collect_point: Optional[SampCollectPointEnum] = Field(default=None, title="sample collection point", description="""Sampling point on the asset were sample was collected (e.g. Wellhead, storage tank, separator, etc). If \"other\" is specified, please propose entry in \"additional info\" field""", json_schema_extra = { "linkml_meta": {'alias': 'samp_collect_point',
         'aliases': ['sample collection point'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'well'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001015'} })
    samp_dis_stage: Optional[SampDisStageEnum] = Field(default=None, title="sample disease stage", description="""Stage of the disease at the time of sample collection, e.g. inoculation, penetration, infection, growth and reproduction, dissemination of pathogen.""", json_schema_extra = { "linkml_meta": {'alias': 'samp_dis_stage',
         'aliases': ['sample disease stage'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'infection'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000249'} })
    samp_floor: Optional[SampFloorEnum] = Field(default=None, title="sampling floor", description="""The floor of the building, where the sampling room is located""", json_schema_extra = { "linkml_meta": {'alias': 'samp_floor',
         'aliases': ['sampling floor'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '4th floor'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000828'} })
    samp_loc_corr_rate: Optional[TextValue] = Field(default=None, title="corrosion rate at sample location", description="""Metal corrosion rate is the speed of metal deterioration due to environmental conditions. As environmental conditions change corrosion rates change accordingly. Therefore, long term corrosion rates are generally more informative than short term rates and for that reason they are preferred during reporting. In the case of suspected MIC, corrosion rate measurements at the time of sampling might provide insights into the involvement of certain microbial community members in MIC as well as potential microbial interplays""", json_schema_extra = { "linkml_meta": {'alias': 'samp_loc_corr_rate',
         'aliases': ['corrosion rate at sample location'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value range'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'millimeter per year'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000136',
         'string_serialization': '{float} - {float} {unit}'} })
    samp_mat_process: Optional[Union[ControlledTermValue,ControlledIdentifiedTermValue]] = Field(default=None, title="sample material processing", description="""A brief description of any processing applied to the sample during or after retrieving the sample from environment, or a link to the relevant protocol(s) performed.""", json_schema_extra = { "linkml_meta": {'alias': 'samp_mat_process',
         'aliases': ['sample material processing'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'text'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'filtering of seawater, storing samples in ethanol'}],
         'is_a': 'nucleic acid sequence source field',
         'slot_uri': 'MIXS:0000016',
         'string_serialization': '{text}'} })
    samp_md: Optional[QuantityValue] = Field(default=None, title="sample measured depth", description="""In non deviated well, measured depth is equal to the true vertical depth, TVD (TVD=TVDSS plus the reference or datum it refers to). In deviated wells, the MD is the length of trajectory of the borehole measured from the same reference or datum. Common datums used are ground level (GL), drilling rig floor (DF), rotary table (RT), kelly bushing (KB) and mean sea level (MSL). If \"other\" is specified, please propose entry in \"additional info\" field""", json_schema_extra = { "linkml_meta": {'alias': 'samp_md',
         'aliases': ['sample measured depth'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value;enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '1534 meter;MSL'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000413'} })
    samp_name: Optional[str] = Field(default=None, title="sample name", description="""A local identifier or name that for the material sample used for extracting nucleic acids, and subsequent sequencing. It can refer either to the original material collected or to any derived sub-samples. It can have any format, but we suggest that you make it concise, unique and consistent within your lab, and as informative as possible. INSDC requires every sample name from a single Submitter to be unique. Use of a globally unique identifier for the field source_mat_id is recommended in addition to sample_name.""", json_schema_extra = { "linkml_meta": {'alias': 'samp_name',
         'aliases': ['sample name'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'text'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'ISDsoil1'}],
         'is_a': 'investigation field',
         'slot_uri': 'MIXS:0001107',
         'string_serialization': '{text}'} })
    samp_preserv: Optional[TextValue] = Field(default=None, title="preservative added to sample", description="""Preservative added to the sample (e.g. Rnalater, alcohol, formaldehyde, etc.). Where appropriate include volume added (e.g. Rnalater; 2 ml)""", json_schema_extra = { "linkml_meta": {'alias': 'samp_preserv',
         'aliases': ['preservative added to sample'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milliliter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000463',
         'string_serialization': '{text};{float} {unit}'} })
    samp_room_id: Optional[TextValue] = Field(default=None, title="sampling room ID or name", description="""Sampling room number. This ID should be consistent with the designations on the building floor plans""", json_schema_extra = { "linkml_meta": {'alias': 'samp_room_id',
         'aliases': ['sampling room ID or name'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000244'} })
    samp_size: Optional[QuantityValue] = Field(default=None, title="amount or size of sample collected", description="""The total amount or size (volume (ml), mass (g) or area (m2) ) of sample collected.""", json_schema_extra = { "linkml_meta": {'alias': 'samp_size',
         'aliases': ['amount or size of sample collected'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'millliter, gram, milligram, '
                                                     'liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '5 liter'}],
         'is_a': 'nucleic acid sequence source field',
         'slot_uri': 'MIXS:0000001'} })
    samp_sort_meth: Optional[list[TextValue]] = Field(default=None, title="sample size sorting method", description="""Method by which samples are sorted; open face filter collecting total suspended particles, prefilter to remove particles larger than X micrometers in diameter, where common values of X would be 10 and 2.5 full size sorting in a cascade impactor.""", json_schema_extra = { "linkml_meta": {'alias': 'samp_sort_meth',
         'aliases': ['sample size sorting method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'description of method'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000216',
         'string_serialization': '{text}'} })
    samp_store_dur: Optional[TextValue] = Field(default=None, title="sample storage duration", description="""Duration for which the sample was stored""", json_schema_extra = { "linkml_meta": {'alias': 'samp_store_dur',
         'aliases': ['sample storage duration'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'duration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'P1Y6M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000116',
         'string_serialization': '{duration}'} })
    samp_store_loc: Optional[TextValue] = Field(default=None, title="sample storage location", description="""Location at which sample was stored, usually name of a specific freezer/room""", json_schema_extra = { "linkml_meta": {'alias': 'samp_store_loc',
         'aliases': ['sample storage location'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'location name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Freezer no:5'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000755',
         'string_serialization': '{text}'} })
    samp_store_temp: Optional[QuantityValue] = Field(default=None, title="sample storage temperature", description="""Temperature at which sample was stored, e.g. -80 degree Celsius""", json_schema_extra = { "linkml_meta": {'alias': 'samp_store_temp',
         'aliases': ['sample storage temperature'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '-80 degree Celsius'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000110'} })
    samp_subtype: Optional[SampSubtypeEnum] = Field(default=None, title="sample subtype", description="""Name of sample sub-type. For example if \"sample type\" is \"Produced Water\" then subtype could be \"Oil Phase\" or \"Water Phase\". If \"other\" is specified, please propose entry in \"additional info\" field""", json_schema_extra = { "linkml_meta": {'alias': 'samp_subtype',
         'aliases': ['sample subtype'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'biofilm'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000999'} })
    samp_taxon_id: Optional[ControlledIdentifiedTermValue] = Field(default=None, title="Taxonomy ID of DNA sample", description="""NCBI taxon id of the sample.  Maybe be a single taxon or mixed taxa sample. Use 'synthetic metagenome’ for mock community/positive controls, or 'blank sample' for negative controls.""", json_schema_extra = { "linkml_meta": {'alias': 'samp_taxon_id',
         'aliases': ['Taxonomy ID of DNA sample'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'Taxonomy ID'}},
         'comments': ['coal metagenome [NCBITaxon:1260732] would be a reasonable '
                      'has_raw_value'],
         'domain_of': ['Biosample'],
         'is_a': 'investigation field',
         'slot_uri': 'MIXS:0001320'} })
    samp_time_out: Optional[TextValue] = Field(default=None, title="sampling time outside", description="""The recent and long term history of outside sampling""", json_schema_extra = { "linkml_meta": {'alias': 'samp_time_out',
         'aliases': ['sampling time outside'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'time'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'hour'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000196'} })
    samp_transport_cond: Optional[TextValue] = Field(default=None, title="sample transport conditions", description="""Sample transport duration (in days or hrs) and temperature the sample was exposed to (e.g. 5.5 days; 20 ¬∞C)""", json_schema_extra = { "linkml_meta": {'alias': 'samp_transport_cond',
         'aliases': ['sample transport conditions'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value;measurement '
                                                     'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'days;degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '5 days;-20 degree Celsius'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000410',
         'string_serialization': '{float} {unit};{float} {unit}'} })
    samp_tvdss: Optional[TextValue] = Field(default=None, title="sample true vertical depth subsea", description="""Depth of the sample i.e. The vertical distance between the sea level and the sampled position in the subsurface. Depth can be reported as an interval for subsurface samples e.g. 1325.75-1362.25 m""", json_schema_extra = { "linkml_meta": {'alias': 'samp_tvdss',
         'aliases': ['sample true vertical depth subsea'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value or measurement '
                                                     'value range'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000409',
         'string_serialization': '{float}-{float} {unit}'} })
    samp_type: Optional[TextValue] = Field(default=None, title="sample type", description="""The type of material from which the sample was obtained. For the Hydrocarbon package, samples include types like core, rock trimmings, drill cuttings, piping section, coupon, pigging debris, solid deposit, produced fluid, produced water, injected water, swabs, etc. For the Food Package, samples are usually categorized as food, body products or tissues, or environmental material. This field accepts terms listed under environmental specimen (http://purl.obolibrary.org/obo/GENEPIO_0001246).""", json_schema_extra = { "linkml_meta": {'alias': 'samp_type',
         'aliases': ['sample type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'GENEPIO:0001246'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'built environment sample [GENEPIO:0001248]'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000998',
         'string_serialization': '{termLabel} {[termID]}'} })
    samp_weather: Optional[SampWeatherEnum] = Field(default=None, title="sampling day weather", description="""The weather on the sampling day""", json_schema_extra = { "linkml_meta": {'alias': 'samp_weather',
         'aliases': ['sampling day weather'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'foggy'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000827'} })
    samp_well_name: Optional[TextValue] = Field(default=None, title="sample well name", description="""Name of the well (e.g. BXA1123) where sample was taken""", json_schema_extra = { "linkml_meta": {'alias': 'samp_well_name',
         'aliases': ['sample well name'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000296',
         'string_serialization': '{text}'} })
    saturates_pc: Optional[TextValue] = Field(default=None, title="saturates wt%", description="""Saturate, Aromatic, Resin and Asphaltene¬†(SARA) is an analysis method that divides¬†crude oil¬†components according to their polarizability and polarity. There are three main methods to obtain SARA results. The most popular one is known as the Iatroscan TLC-FID and is referred to as IP-143 (source: https://en.wikipedia.org/wiki/Saturate,_aromatic,_resin_and_asphaltene)""", json_schema_extra = { "linkml_meta": {'alias': 'saturates_pc',
         'aliases': ['saturates wt%'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'percent'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000131',
         'string_serialization': '{text};{float} {unit}'} })
    season: Optional[TextValue] = Field(default=None, title="season", description="""The season when sampling occurred. Any of the four periods into which the year is divided by the equinoxes and solstices. This field accepts terms listed under season (http://purl.obolibrary.org/obo/NCIT_C94729).""", json_schema_extra = { "linkml_meta": {'alias': 'season',
         'aliases': ['season'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'NCIT:C94729'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'autumn [NCIT:C94733]'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000829',
         'string_serialization': '{termLabel} {[termID]}'} })
    season_environment: Optional[list[TextValue]] = Field(default=None, title="seasonal environment", description="""Treatment involving an exposure to a particular season (e.g. Winter, summer, rabi, rainy etc.), treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment""", json_schema_extra = { "linkml_meta": {'alias': 'season_environment',
         'aliases': ['seasonal environment'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'seasonal environment '
                                                     'name;treatment interval and '
                                                     'duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'rainy;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001068',
         'string_serialization': '{text};{Rn/start_time/end_time/duration}'} })
    season_precpt: Optional[QuantityValue] = Field(default=None, title="average seasonal precipitation", description="""The average of all seasonal precipitation values known, or an estimated equivalent value derived by such methods as regional indexes or Isohyetal maps.""", json_schema_extra = { "linkml_meta": {'alias': 'season_precpt',
         'aliases': ['mean seasonal precipitation'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'millimeter'}},
         'comments': ['Seasons are defined as spring (March, April, May), summer '
                      '(June, July, August), autumn (September, October, November) and '
                      'winter (December, January, February).'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.4 inch'}, {'value': '10.16 mm'}],
         'is_a': 'core field',
         'notes': ['mean and average are the same thing, but it seems like bad '
                   'practice to not be consistent. Changed mean to average'],
         'slot_uri': 'MIXS:0000645',
         'todos': ['check validation & examples. always mm? so value only? Or value + '
                   'unit']} })
    season_temp: Optional[QuantityValue] = Field(default=None, title="mean seasonal temperature", description="""Mean seasonal temperature""", json_schema_extra = { "linkml_meta": {'alias': 'season_temp',
         'aliases': ['mean seasonal temperature'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '18 degree Celsius'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000643'} })
    season_use: Optional[SeasonUseEnum] = Field(default=None, title="seasonal use", description="""The seasons the space is occupied""", json_schema_extra = { "linkml_meta": {'alias': 'season_use',
         'aliases': ['seasonal use'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Winter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000830'} })
    secondary_treatment: Optional[TextValue] = Field(default=None, title="secondary treatment", description="""The process for substantially degrading the biological content of the sewage""", json_schema_extra = { "linkml_meta": {'alias': 'secondary_treatment',
         'aliases': ['secondary treatment'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'secondary treatment type'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000351',
         'string_serialization': '{text}'} })
    sediment_type: Optional[SedimentTypeEnum] = Field(default=None, title="sediment type", description="""Information about the sediment type based on major constituents""", json_schema_extra = { "linkml_meta": {'alias': 'sediment_type',
         'aliases': ['sediment type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'biogenous'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001078'} })
    sewage_type: Optional[TextValue] = Field(default=None, title="sewage type", description="""Type of wastewater treatment plant as municipial or industrial""", json_schema_extra = { "linkml_meta": {'alias': 'sewage_type',
         'aliases': ['sewage type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'sewage type name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000215',
         'string_serialization': '{text}'} })
    shad_dev_water_mold: Optional[str] = Field(default=None, title="shading device signs of water/mold", description="""Signs of the presence of mold or mildew on the shading device""", json_schema_extra = { "linkml_meta": {'alias': 'shad_dev_water_mold',
         'aliases': ['shading device signs of water/mold'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'no presence of mold visible'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000834',
         'string_serialization': '[presence of mold visible|no presence of mold '
                                 'visible]'} })
    shading_device_cond: Optional[ShadingDeviceCondEnum] = Field(default=None, title="shading device condition", description="""The physical condition of the shading device at the time of sampling""", json_schema_extra = { "linkml_meta": {'alias': 'shading_device_cond',
         'aliases': ['shading device condition'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'new'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000831'} })
    shading_device_loc: Optional[TextValue] = Field(default=None, title="shading device location", description="""The location of the shading device in relation to the built structure""", json_schema_extra = { "linkml_meta": {'alias': 'shading_device_loc',
         'aliases': ['shading device location'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'exterior'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000832',
         'string_serialization': '[exterior|interior]'} })
    shading_device_mat: Optional[TextValue] = Field(default=None, title="shading device material", description="""The material the shading device is composed of""", json_schema_extra = { "linkml_meta": {'alias': 'shading_device_mat',
         'aliases': ['shading device material'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'material name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000245',
         'string_serialization': '{text}'} })
    shading_device_type: Optional[ShadingDeviceTypeEnum] = Field(default=None, title="shading device type", description="""The type of shading device""", json_schema_extra = { "linkml_meta": {'alias': 'shading_device_type',
         'aliases': ['shading device type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'slatted aluminum awning'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000835'} })
    sieving: Optional[TextValue] = Field(default=None, title="composite design/sieving", description="""Collection design of pooled samples and/or sieve size and amount of sample sieved""", json_schema_extra = { "linkml_meta": {'alias': 'sieving',
         'aliases': ['composite design/sieving'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'design name and/or size;amount'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'comments': ['Describe how samples were composited or sieved.',
                      "Use 'sample link' to indicate which samples were combined."],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'combined 2 cores | 4mm sieved'},
                      {'value': '4 mm sieved and homogenized'},
                      {'value': '50 g | 5 cores | 2 mm sieved'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000322',
         'string_serialization': '{{text}|{float} {unit}};{float} {unit}',
         'todos': ['check validation and examples']} })
    silicate: Optional[QuantityValue] = Field(default=None, title="silicate", description="""Concentration of silicate""", json_schema_extra = { "linkml_meta": {'alias': 'silicate',
         'aliases': ['silicate'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.05 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000184'} })
    size_frac: Optional[TextValue] = Field(default=None, title="size fraction selected", description="""Filtering pore size used in sample preparation""", json_schema_extra = { "linkml_meta": {'alias': 'size_frac',
         'aliases': ['size fraction selected'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'filter size value range'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0-0.22 micrometer'}],
         'is_a': 'nucleic acid sequence source field',
         'slot_uri': 'MIXS:0000017',
         'string_serialization': '{float}-{float} {unit}'} })
    size_frac_low: Optional[QuantityValue] = Field(default=None, title="size-fraction lower threshold", description="""Refers to the mesh/pore size used to pre-filter/pre-sort the sample. Materials larger than the size threshold are excluded from the sample""", json_schema_extra = { "linkml_meta": {'alias': 'size_frac_low',
         'aliases': ['size-fraction lower threshold'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micrometer'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.2 micrometer'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000735'} })
    size_frac_up: Optional[QuantityValue] = Field(default=None, title="size-fraction upper threshold", description="""Refers to the mesh/pore size used to retain the sample. Materials smaller than the size threshold are excluded from the sample""", json_schema_extra = { "linkml_meta": {'alias': 'size_frac_up',
         'aliases': ['size-fraction upper threshold'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micrometer'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '20 micrometer'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000736'} })
    slope_aspect: Optional[QuantityValue] = Field(default=None, title="slope aspect", description="""The direction a slope faces. While looking down a slope use a compass to record the direction you are facing (direction or degrees). This measure provides an indication of sun and wind exposure that will influence soil temperature and evapotranspiration.""", json_schema_extra = { "linkml_meta": {'alias': 'slope_aspect',
         'aliases': ['slope aspect'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'degree'}},
         'comments': ['Aspect is the orientation of slope, measured clockwise in '
                      'degrees from 0 to 360, where 0 is north-facing, 90 is '
                      'east-facing, 180 is south-facing, and 270 is west-facing.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '35'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000647'} })
    slope_gradient: Optional[QuantityValue] = Field(default=None, title="slope gradient", description="""Commonly called 'slope'. The angle between ground surface and a horizontal line (in percent). This is the direction that overland water would flow. This measure is usually taken with a hand level meter or clinometer""", json_schema_extra = { "linkml_meta": {'alias': 'slope_gradient',
         'aliases': ['slope gradient'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'percentage'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '10%'}, {'value': '10 %'}, {'value': '0.10'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000646',
         'todos': ['Slope is a percent. How does the validation work? Check to correct '
                   'examples']} })
    sludge_retent_time: Optional[QuantityValue] = Field(default=None, title="sludge retention time", description="""The time activated sludge remains in reactor""", json_schema_extra = { "linkml_meta": {'alias': 'sludge_retent_time',
         'aliases': ['sludge retention time'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'hours'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000669'} })
    sodium: Optional[QuantityValue] = Field(default=None, title="sodium", description="""Sodium concentration in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'sodium',
         'aliases': ['sodium'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter, parts per '
                                                     'million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '10.5 milligram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000428'} })
    soil_horizon: Optional[SoilHorizonEnum] = Field(default=None, title="soil horizon", description="""Specific layer in the land area which measures parallel to the soil surface and possesses physical characteristics which differ from the layers above and beneath""", json_schema_extra = { "linkml_meta": {'alias': 'soil_horizon',
         'aliases': ['soil horizon'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'A horizon'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001082'} })
    soil_text_measure: Optional[QuantityValue] = Field(default=None, title="soil texture measurement", description="""The relative proportion of different grain sizes of mineral particles in a soil, as described using a standard system; express as % sand (50 um to 2 mm), silt (2 um to 50 um), and clay (<2 um) with textural name (e.g., silty clay loam) optional.""", json_schema_extra = { "linkml_meta": {'alias': 'soil_text_measure',
         'aliases': ['soil texture measurement'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000335'} })
    soil_texture_meth: Optional[str] = Field(default=None, title="soil texture method", description="""Reference or method used in determining soil texture""", json_schema_extra = { "linkml_meta": {'alias': 'soil_texture_meth',
         'aliases': ['soil texture method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000336',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    soil_type: Optional[TextValue] = Field(default=None, title="soil type", description="""Description of the soil type or classification. This field accepts terms under soil (http://purl.obolibrary.org/obo/ENVO_00001998).  Multiple terms can be separated by pipes.""", json_schema_extra = { "linkml_meta": {'alias': 'soil_type',
         'aliases': ['soil type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'ENVO_00001998'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['FieldResearchSite', 'Biosample'],
         'examples': [{'value': 'plinthosol [ENVO:00002250]'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000332',
         'string_serialization': '{termLabel} {[termID]}'} })
    soil_type_meth: Optional[TextValue] = Field(default=None, title="soil type method", description="""Reference or method used in determining soil series name or other lower-level classification""", json_schema_extra = { "linkml_meta": {'alias': 'soil_type_meth',
         'aliases': ['soil type method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000334',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    solar_irradiance: Optional[QuantityValue] = Field(default=None, title="solar irradiance", description="""The amount of solar energy that arrives at a specific area of a surface during a specific time interval""", json_schema_extra = { "linkml_meta": {'alias': 'solar_irradiance',
         'aliases': ['solar irradiance'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'kilowatts per square meter per '
                                                     'day, ergs per square centimeter '
                                                     'per second'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '1.36 kilowatts per square meter per day'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000112'} })
    soluble_inorg_mat: Optional[list[TextValue]] = Field(default=None, title="soluble inorganic material", description="""Concentration of substances such as ammonia, road-salt, sea-salt, cyanide, hydrogen sulfide, thiocyanates, thiosulfates, etc.""", json_schema_extra = { "linkml_meta": {'alias': 'soluble_inorg_mat',
         'aliases': ['soluble inorganic material'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'soluble inorganic material '
                                                     'name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram, microgram, mole per liter, '
                                                     'gram per liter, parts per '
                                                     'million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000672',
         'string_serialization': '{text};{float} {unit}'} })
    soluble_org_mat: Optional[list[TextValue]] = Field(default=None, title="soluble organic material", description="""Concentration of substances such as urea, fruit sugars, soluble proteins, drugs, pharmaceuticals, etc.""", json_schema_extra = { "linkml_meta": {'alias': 'soluble_org_mat',
         'aliases': ['soluble organic material'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'soluble organic material '
                                                     'name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram, microgram, mole per liter, '
                                                     'gram per liter, parts per '
                                                     'million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000673',
         'string_serialization': '{text};{float} {unit}'} })
    soluble_react_phosp: Optional[QuantityValue] = Field(default=None, title="soluble reactive phosphorus", description="""Concentration of soluble reactive phosphorus""", json_schema_extra = { "linkml_meta": {'alias': 'soluble_react_phosp',
         'aliases': ['soluble reactive phosphorus'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter, milligram '
                                                     'per liter, parts per million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.1 milligram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000738'} })
    source_mat_id: Optional[TextValue] = Field(default=None, title="source material identifier", description="""A globally unique identifier assigned to the biological sample.""", json_schema_extra = { "linkml_meta": {'alias': 'source_mat_id',
         'aliases': ['source material identifiers'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'for cultures of microorganisms: '
                                                     'identifiers for two culture '
                                                     'collections; for other material '
                                                     'a unique arbitrary identifer'}},
         'comments': ['Identifiers must be prefixed. Possible FAIR prefixes are IGSNs '
                      '(http://www.geosamples.org/getigsn), NCBI biosample accession '
                      'numbers, ARK identifiers (https://arks.org/). These IDs enable '
                      'linking to derived analytes and subsamples. If you have not '
                      'assigned FAIR identifiers to your samples, you can generate '
                      'UUIDs (https://www.uuidgenerator.net/).'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'IGSN:AU1243'},
                      {'value': 'UUID:24f1467a-40f4-11ed-b878-0242ac120002'}],
         'is_a': 'nucleic acid sequence source field',
         'slot_uri': 'MIXS:0000026',
         'string_serialization': '{text}',
         'todos': ['Currently, the comments say to use UUIDs. However, if we implement '
                   'assigning NMDC identifiers with the minter we dont need to require '
                   'a GUID. It can be an optional field to fill out only if they '
                   'already have a resolvable ID.']} })
    space_typ_state: Optional[TextValue] = Field(default=None, title="space typical state", description="""Customary or normal state of the space""", json_schema_extra = { "linkml_meta": {'alias': 'space_typ_state',
         'aliases': ['space typical state'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'typically occupied'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000770',
         'string_serialization': '[typically occupied|typically unoccupied]'} })
    specific: Optional[SpecificEnum] = Field(default=None, title="specifications", description="""The building specifications. If design is chosen, indicate phase: conceptual, schematic, design development, construction documents""", json_schema_extra = { "linkml_meta": {'alias': 'specific',
         'aliases': ['specifications'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'construction'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000836'} })
    specific_humidity: Optional[QuantityValue] = Field(default=None, title="specific humidity", description="""The mass of water vapour in a unit mass of moist air, usually expressed as grams of vapour per kilogram of air, or, in air conditioning, as grains per pound.""", json_schema_extra = { "linkml_meta": {'alias': 'specific_humidity',
         'aliases': ['specific humidity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram of air, kilogram of air'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '15 per kilogram of air'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000214'} })
    sr_dep_env: Optional[SrDepEnvEnum] = Field(default=None, title="source rock depositional environment", description="""Source rock depositional environment (https://en.wikipedia.org/wiki/Source_rock). If \"other\" is specified, please propose entry in \"additional info\" field""", json_schema_extra = { "linkml_meta": {'alias': 'sr_dep_env',
         'aliases': ['source rock depositional environment'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Marine'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000996'} })
    sr_geol_age: Optional[SrGeolAgeEnum] = Field(default=None, title="source rock geological age", description="""Geological age of source rock (Additional info: https://en.wikipedia.org/wiki/Period_(geology)). If \"other\" is specified, please propose entry in \"additional info\" field""", json_schema_extra = { "linkml_meta": {'alias': 'sr_geol_age',
         'aliases': ['source rock geological age'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Silurian'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000997'} })
    sr_kerog_type: Optional[SrKerogTypeEnum] = Field(default=None, title="source rock kerogen type", description="""Origin of kerogen. Type I: Algal (aquatic), Type II: planktonic and soft plant material (aquatic or terrestrial), Type III: terrestrial woody/ fibrous plant material (terrestrial), Type IV: oxidized recycled woody debris (terrestrial) (additional information: https://en.wikipedia.org/wiki/Kerogen). If \"other\" is specified, please propose entry in \"additional info\" field""", json_schema_extra = { "linkml_meta": {'alias': 'sr_kerog_type',
         'aliases': ['source rock kerogen type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Type IV'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000994'} })
    sr_lithology: Optional[SrLithologyEnum] = Field(default=None, title="source rock lithology", description="""Lithology of source rock (https://en.wikipedia.org/wiki/Source_rock). If \"other\" is specified, please propose entry in \"additional info\" field""", json_schema_extra = { "linkml_meta": {'alias': 'sr_lithology',
         'aliases': ['source rock lithology'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Coal'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000995'} })
    standing_water_regm: Optional[list[TextValue]] = Field(default=None, title="standing water regimen", description="""Treatment involving an exposure to standing water during a plant's life span, types can be flood water or standing water, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple regimens""", json_schema_extra = { "linkml_meta": {'alias': 'standing_water_regm',
         'aliases': ['standing water regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'standing water type;treatment '
                                                     'interval and duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'standing '
                                'water;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001069',
         'string_serialization': '{text};{Rn/start_time/end_time/duration}'} })
    store_cond: Optional[TextValue] = Field(default=None, title="storage conditions", description="""Explain how and for how long the soil sample was stored before DNA extraction (fresh/frozen/other).""", json_schema_extra = { "linkml_meta": {'alias': 'store_cond',
         'aliases': ['storage conditions'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'storage condition type;duration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '-20 degree Celsius freezer;P2Y10D'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000327',
         'string_serialization': '{text};{duration}'} })
    substructure_type: Optional[list[SubstructureTypeEnum]] = Field(default=None, title="substructure type", description="""The substructure or under building is that largely hidden section of the building which is built off the foundations to the ground floor level""", json_schema_extra = { "linkml_meta": {'alias': 'substructure_type',
         'aliases': ['substructure type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'basement'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000767'} })
    sulfate: Optional[QuantityValue] = Field(default=None, title="sulfate", description="""Concentration of sulfate in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'sulfate',
         'aliases': ['sulfate'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter, milligram '
                                                     'per liter, parts per million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '5 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000423'} })
    sulfate_fw: Optional[QuantityValue] = Field(default=None, title="sulfate in formation water", description="""Original sulfate concentration in the hydrocarbon resource""", json_schema_extra = { "linkml_meta": {'alias': 'sulfate_fw',
         'aliases': ['sulfate in formation water'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000407'} })
    sulfide: Optional[QuantityValue] = Field(default=None, title="sulfide", description="""Concentration of sulfide in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'sulfide',
         'aliases': ['sulfide'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter, milligram '
                                                     'per liter, parts per million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000424'} })
    surf_air_cont: Optional[list[SurfAirContEnum]] = Field(default=None, title="surface-air contaminant", description="""Contaminant identified on surface""", json_schema_extra = { "linkml_meta": {'alias': 'surf_air_cont',
         'aliases': ['surface-air contaminant'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'radon'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000759'} })
    surf_humidity: Optional[QuantityValue] = Field(default=None, title="surface humidity", description="""Surfaces: water activity as a function of air and material moisture""", json_schema_extra = { "linkml_meta": {'alias': 'surf_humidity',
         'aliases': ['surface humidity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'percentage'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '10%'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000123'} })
    surf_material: Optional[SurfMaterialEnum] = Field(default=None, title="surface material", description="""Surface materials at the point of sampling""", json_schema_extra = { "linkml_meta": {'alias': 'surf_material',
         'aliases': ['surface material'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'wood'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000758'} })
    surf_moisture: Optional[QuantityValue] = Field(default=None, title="surface moisture", description="""Water held on a surface""", json_schema_extra = { "linkml_meta": {'alias': 'surf_moisture',
         'aliases': ['surface moisture'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'parts per million, gram per '
                                                     'cubic meter, gram per square '
                                                     'meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.01 gram per square meter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000128'} })
    surf_moisture_ph: Optional[float] = Field(default=None, title="surface moisture pH", description="""ph measurement of surface""", json_schema_extra = { "linkml_meta": {'alias': 'surf_moisture_ph',
         'aliases': ['surface moisture pH'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '7'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000760'} })
    surf_temp: Optional[QuantityValue] = Field(default=None, title="surface temperature", description="""Temperature of the surface at the time of sampling""", json_schema_extra = { "linkml_meta": {'alias': 'surf_temp',
         'aliases': ['surface temperature'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '15 degree Celsius'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000125'} })
    suspend_part_matter: Optional[QuantityValue] = Field(default=None, title="suspended particulate matter", description="""Concentration of suspended particulate matter""", json_schema_extra = { "linkml_meta": {'alias': 'suspend_part_matter',
         'aliases': ['suspended particulate matter'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.5 milligram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000741'} })
    suspend_solids: Optional[list[TextValue]] = Field(default=None, title="suspended solids", description="""Concentration of substances including a wide variety of material, such as silt, decaying plant and animal matter; can include multiple substances""", json_schema_extra = { "linkml_meta": {'alias': 'suspend_solids',
         'aliases': ['suspended solids'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'suspended solid name;measurement '
                                                     'value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram, microgram, milligram per '
                                                     'liter, mole per liter, gram per '
                                                     'liter, part per million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000150',
         'string_serialization': '{text};{float} {unit}'} })
    tan: Optional[QuantityValue] = Field(default=None, title="total acid number", description="""Total Acid Number¬†(TAN) is a measurement of acidity that is determined by the amount of¬†potassium hydroxide¬†in milligrams that is needed to neutralize the acids in one gram of oil.¬†It is an important quality measurement of¬†crude oil. (source: https://en.wikipedia.org/wiki/Total_acid_number)""", json_schema_extra = { "linkml_meta": {'alias': 'tan',
         'aliases': ['total acid number'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000120'} })
    temp: Optional[QuantityValue] = Field(default=None, title="temperature", description="""Temperature of the sample at the time of sampling.""", json_schema_extra = { "linkml_meta": {'alias': 'temp',
         'aliases': ['temperature'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '25 degree Celsius'}],
         'is_a': 'environment field',
         'slot_uri': 'MIXS:0000113'} })
    temp_out: Optional[QuantityValue] = Field(default=None, title="temperature outside house", description="""The recorded temperature value at sampling time outside""", json_schema_extra = { "linkml_meta": {'alias': 'temp_out',
         'aliases': ['temperature outside house'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '5 degree Celsius'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000197'} })
    tertiary_treatment: Optional[TextValue] = Field(default=None, title="tertiary treatment", description="""The process providing a final treatment stage to raise the effluent quality before it is discharged to the receiving environment""", json_schema_extra = { "linkml_meta": {'alias': 'tertiary_treatment',
         'aliases': ['tertiary treatment'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'tertiary treatment type'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000352',
         'string_serialization': '{text}'} })
    tidal_stage: Optional[TidalStageEnum] = Field(default=None, title="tidal stage", description="""Stage of tide""", json_schema_extra = { "linkml_meta": {'alias': 'tidal_stage',
         'aliases': ['tidal stage'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'high tide'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000750'} })
    tillage: Optional[list[TillageEnum]] = Field(default=None, title="history/tillage", description="""Note method(s) used for tilling""", json_schema_extra = { "linkml_meta": {'alias': 'tillage',
         'aliases': ['history/tillage'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'chisel'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001081'} })
    tiss_cult_growth_med: Optional[TextValue] = Field(default=None, title="tissue culture growth media", description="""Description of plant tissue culture growth media used""", json_schema_extra = { "linkml_meta": {'alias': 'tiss_cult_growth_med',
         'aliases': ['tissue culture growth media'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI,url or free text'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'https://link.springer.com/content/pdf/10.1007/BF02796489.pdf'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0001070',
         'string_serialization': '{PMID}|{DOI}|{URL}|{text}'} })
    toluene: Optional[QuantityValue] = Field(default=None, title="toluene", description="""Concentration of toluene in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'toluene',
         'aliases': ['toluene'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter, parts per '
                                                     'million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000154'} })
    tot_carb: Optional[QuantityValue] = Field(default=None, title="total carbon", description="""Total carbon content""", json_schema_extra = { "linkml_meta": {'alias': 'tot_carb',
         'aliases': ['total carbon'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '1 ug/L'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000525',
         'todos': ['is this inorganic and organic? both? could use some clarification.',
                   "ug/L doesn't seem like the right units. Should check this slots "
                   "usage in databases and re-evaluate. I couldn't find any references "
                   'that provided this data in this format']} })
    tot_depth_water_col: Optional[QuantityValue] = Field(default=None, title="total depth of water column", description="""Measurement of total depth of water column""", json_schema_extra = { "linkml_meta": {'alias': 'tot_depth_water_col',
         'aliases': ['total depth of water column'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '500 meter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000634'} })
    tot_diss_nitro: Optional[QuantityValue] = Field(default=None, title="total dissolved nitrogen", description="""Total dissolved nitrogen concentration, reported as nitrogen, measured by: total dissolved nitrogen = NH4 + NO3NO2 + dissolved organic nitrogen""", json_schema_extra = { "linkml_meta": {'alias': 'tot_diss_nitro',
         'aliases': ['total dissolved nitrogen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '40 microgram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000744'} })
    tot_inorg_nitro: Optional[QuantityValue] = Field(default=None, title="total inorganic nitrogen", description="""Total inorganic nitrogen content""", json_schema_extra = { "linkml_meta": {'alias': 'tot_inorg_nitro',
         'aliases': ['total inorganic nitrogen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '40 microgram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000745'} })
    tot_iron: Optional[QuantityValue] = Field(default=None, title="total iron", description="""Concentration of total iron in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'tot_iron',
         'aliases': ['total iron'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter, milligram '
                                                     'per kilogram'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000105'} })
    tot_nitro: Optional[QuantityValue] = Field(default=None, title="total nitrogen concentration", description="""Total nitrogen concentration of water samples, calculated by: total nitrogen = total dissolved nitrogen + particulate nitrogen. Can also be measured without filtering, reported as nitrogen""", json_schema_extra = { "linkml_meta": {'alias': 'tot_nitro',
         'aliases': ['total nitrogen concentration'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per liter, micromole '
                                                     'per liter, milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '50 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000102'} })
    tot_nitro_cont_meth: Optional[str] = Field(default=None, title="total nitrogen content method", description="""Reference or method used in determining the total nitrogen""", json_schema_extra = { "linkml_meta": {'alias': 'tot_nitro_cont_meth',
         'aliases': ['total nitrogen content method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'https://doi.org/10.2134/agronmonogr9.2.c32'},
                      {'value': 'https://acsess.onlinelibrary.wiley.com/doi/full/10.2136/sssaj2009.0389?casa_token=bm0pYIUdNMgAAAAA%3AOWVRR0STHaOe-afTcTdxn5m1hM8n2ltM0wY-b1iYpYdD9dhwppk5j3LvC2IO5yhOIvyLVeQz4NZRCZo'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000338',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    tot_nitro_content: Optional[QuantityValue] = Field(default=None, title="total nitrogen content", description="""Total nitrogen content of the sample""", json_schema_extra = { "linkml_meta": {'alias': 'tot_nitro_content',
         'aliases': ['total nitrogen content'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per liter, micromole '
                                                     'per liter, milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '5 mg N/ L'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000530'} })
    tot_org_c_meth: Optional[TextValue] = Field(default=None, title="total organic carbon method", description="""Reference or method used in determining total organic carbon""", json_schema_extra = { "linkml_meta": {'alias': 'tot_org_c_meth',
         'aliases': ['total organic carbon method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'https://doi.org/10.1080/07352680902776556'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000337',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    tot_org_carb: Optional[QuantityValue] = Field(default=None, title="total organic carbon", description="""Definition for soil: total organic carbon content of the soil, definition otherwise: total organic carbon content""", json_schema_extra = { "linkml_meta": {'alias': 'tot_org_carb',
         'aliases': ['total organic carbon'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram Carbon per kilogram sample '
                                                     'material'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '5 mg N/ L'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000533',
         'todos': ['check description. How are they different?']} })
    tot_part_carb: Optional[QuantityValue] = Field(default=None, title="total particulate carbon", description="""Total particulate carbon content""", json_schema_extra = { "linkml_meta": {'alias': 'tot_part_carb',
         'aliases': ['total particulate carbon'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per liter, micromole '
                                                     'per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '35 micromole per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000747'} })
    tot_phosp: Optional[QuantityValue] = Field(default=None, title="total phosphorus", description="""Total phosphorus concentration in the sample, calculated by: total phosphorus = total dissolved phosphorus + particulate phosphorus""", json_schema_extra = { "linkml_meta": {'alias': 'tot_phosp',
         'aliases': ['total phosphorus'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'micromole per liter, milligram '
                                                     'per liter, parts per million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.03 milligram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000117'} })
    tot_phosphate: Optional[QuantityValue] = Field(default=None, title="total phosphate", description="""Total amount or concentration of phosphate""", json_schema_extra = { "linkml_meta": {'alias': 'tot_phosphate',
         'aliases': ['total phosphate'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per liter, micromole '
                                                     'per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000689'} })
    tot_sulfur: Optional[QuantityValue] = Field(default=None, title="total sulfur", description="""Concentration of total sulfur in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'tot_sulfur',
         'aliases': ['total sulfur'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter, parts per '
                                                     'million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000419'} })
    train_line: Optional[TrainLineEnum] = Field(default=None, title="train line", description="""The subway line name""", json_schema_extra = { "linkml_meta": {'alias': 'train_line',
         'aliases': ['train line'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'red'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000837'} })
    train_stat_loc: Optional[TrainStatLocEnum] = Field(default=None, title="train station collection location", description="""The train station collection location""", json_schema_extra = { "linkml_meta": {'alias': 'train_stat_loc',
         'aliases': ['train station collection location'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'forest hills'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000838'} })
    train_stop_loc: Optional[TrainStopLocEnum] = Field(default=None, title="train stop collection location", description="""The train stop collection location""", json_schema_extra = { "linkml_meta": {'alias': 'train_stop_loc',
         'aliases': ['train stop collection location'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'end'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000839'} })
    turbidity: Optional[QuantityValue] = Field(default=None, title="turbidity", description="""Measure of the amount of cloudiness or haziness in water caused by individual particles""", json_schema_extra = { "linkml_meta": {'alias': 'turbidity',
         'aliases': ['turbidity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'formazin turbidity unit, '
                                                     'formazin nephelometric units'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.3 nephelometric turbidity units'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000191'} })
    tvdss_of_hcr_press: Optional[QuantityValue] = Field(default=None, title="depth (TVDSS) of hydrocarbon resource pressure", description="""True vertical depth subsea (TVDSS) of the hydrocarbon resource where the original pressure was measured (e.g. 1578 m).""", json_schema_extra = { "linkml_meta": {'alias': 'tvdss_of_hcr_press',
         'aliases': ['depth (TVDSS) of hydrocarbon resource pressure'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000397'} })
    tvdss_of_hcr_temp: Optional[QuantityValue] = Field(default=None, title="depth (TVDSS) of hydrocarbon resource temperature", description="""True vertical depth subsea (TVDSS) of the hydrocarbon resource where the original temperature was measured (e.g. 1345 m).""", json_schema_extra = { "linkml_meta": {'alias': 'tvdss_of_hcr_temp',
         'aliases': ['depth (TVDSS) of hydrocarbon resource temperature'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000394'} })
    typ_occup_density: Optional[float] = Field(default=None, title="typical occupant density", description="""Customary or normal density of occupants""", json_schema_extra = { "linkml_meta": {'alias': 'typ_occup_density',
         'aliases': ['typical occupant density'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '25'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000771'} })
    ventilation_rate: Optional[QuantityValue] = Field(default=None, title="ventilation rate", description="""Ventilation rate of the system in the sampled premises""", json_schema_extra = { "linkml_meta": {'alias': 'ventilation_rate',
         'aliases': ['ventilation rate'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'cubic meter per minute, liters '
                                                     'per second'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '750 cubic meter per minute'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000114'} })
    ventilation_type: Optional[TextValue] = Field(default=None, title="ventilation type", description="""Ventilation system used in the sampled premises""", json_schema_extra = { "linkml_meta": {'alias': 'ventilation_type',
         'aliases': ['ventilation type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'ventilation type name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Operable windows'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000756',
         'string_serialization': '{text}'} })
    vfa: Optional[QuantityValue] = Field(default=None, title="volatile fatty acids", description="""Concentration of Volatile Fatty Acids in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'vfa',
         'aliases': ['volatile fatty acids'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter, parts per '
                                                     'million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000152'} })
    vfa_fw: Optional[QuantityValue] = Field(default=None, title="vfa in formation water", description="""Original volatile fatty acid concentration in the hydrocarbon resource""", json_schema_extra = { "linkml_meta": {'alias': 'vfa_fw',
         'aliases': ['vfa in formation water'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000408'} })
    vis_media: Optional[VisMediaEnum] = Field(default=None, title="visual media", description="""The building visual media""", json_schema_extra = { "linkml_meta": {'alias': 'vis_media',
         'aliases': ['visual media'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '3D scans'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000840'} })
    viscosity: Optional[TextValue] = Field(default=None, title="viscosity", description="""A measure of oil's resistance¬†to gradual deformation by¬†shear stress¬†or¬†tensile stress (e.g. 3.5 cp; 100 ¬∞C)""", json_schema_extra = { "linkml_meta": {'alias': 'viscosity',
         'aliases': ['viscosity'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value;measurement '
                                                     'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'cP at degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000126',
         'string_serialization': '{float} {unit};{float} {unit}'} })
    volatile_org_comp: Optional[list[TextValue]] = Field(default=None, title="volatile organic compounds", description="""Concentration of carbon-based chemicals that easily evaporate at room temperature; can report multiple volatile organic compounds by entering numeric values preceded by name of compound""", json_schema_extra = { "linkml_meta": {'alias': 'volatile_org_comp',
         'aliases': ['volatile organic compounds'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'volatile organic compound '
                                                     'name;measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'microgram per cubic meter, parts '
                                                     'per million, nanogram per '
                                                     'liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'formaldehyde;500 nanogram per liter'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000115',
         'string_serialization': '{text};{float} {unit}'} })
    wall_area: Optional[QuantityValue] = Field(default=None, title="wall area", description="""The total area of the sampled room's walls""", json_schema_extra = { "linkml_meta": {'alias': 'wall_area',
         'aliases': ['wall area'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'square meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000198'} })
    wall_const_type: Optional[WallConstTypeEnum] = Field(default=None, title="wall construction type", description="""The building class of the wall defined by the composition of the building elements and fire-resistance rating.""", json_schema_extra = { "linkml_meta": {'alias': 'wall_const_type',
         'aliases': ['wall construction type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'fire resistive'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000841'} })
    wall_finish_mat: Optional[WallFinishMatEnum] = Field(default=None, title="wall finish material", description="""The material utilized to finish the outer most layer of the wall""", json_schema_extra = { "linkml_meta": {'alias': 'wall_finish_mat',
         'aliases': ['wall finish material'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'wood'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000842'} })
    wall_height: Optional[QuantityValue] = Field(default=None, title="wall height", description="""The average height of the walls in the sampled room""", json_schema_extra = { "linkml_meta": {'alias': 'wall_height',
         'aliases': ['wall height'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'centimeter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000221'} })
    wall_loc: Optional[WallLocEnum] = Field(default=None, title="wall location", description="""The relative location of the wall within the room""", json_schema_extra = { "linkml_meta": {'alias': 'wall_loc',
         'aliases': ['wall location'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'north'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000843'} })
    wall_surf_treatment: Optional[WallSurfTreatmentEnum] = Field(default=None, title="wall surface treatment", description="""The surface treatment of interior wall""", json_schema_extra = { "linkml_meta": {'alias': 'wall_surf_treatment',
         'aliases': ['wall surface treatment'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'paneling'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000845'} })
    wall_texture: Optional[WallTextureEnum] = Field(default=None, title="wall texture", description="""The feel, appearance, or consistency of a wall surface""", json_schema_extra = { "linkml_meta": {'alias': 'wall_texture',
         'aliases': ['wall texture'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'popcorn'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000846'} })
    wall_thermal_mass: Optional[QuantityValue] = Field(default=None, title="wall thermal mass", description="""The ability of the wall to provide inertia against temperature fluctuations. Generally this means concrete or concrete block that is either exposed or covered only with paint""", json_schema_extra = { "linkml_meta": {'alias': 'wall_thermal_mass',
         'aliases': ['wall thermal mass'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'joule per degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000222'} })
    wall_water_mold: Optional[TextValue] = Field(default=None, title="wall signs of water/mold", description="""Signs of the presence of mold or mildew on a wall""", json_schema_extra = { "linkml_meta": {'alias': 'wall_water_mold',
         'aliases': ['wall signs of water/mold'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'no presence of mold visible'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000844',
         'string_serialization': '[presence of mold visible|no presence of mold '
                                 'visible]'} })
    wastewater_type: Optional[TextValue] = Field(default=None, title="wastewater type", description="""The origin of wastewater such as human waste, rainfall, storm drains, etc.""", json_schema_extra = { "linkml_meta": {'alias': 'wastewater_type',
         'aliases': ['wastewater type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'wastewater type name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000353',
         'string_serialization': '{text}'} })
    water_cont_soil_meth: Optional[str] = Field(default=None, title="water content method", description="""Reference or method used in determining the water content of soil""", json_schema_extra = { "linkml_meta": {'alias': 'water_cont_soil_meth',
         'aliases': ['water content method'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID,DOI or url'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'comments': ['Required if providing water content'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'J. Nat. Prod. Plant Resour., 2012, 2 (4):500-503'},
                      {'value': 'https://dec.alaska.gov/applications/spar/webcalc/definitions.htm'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000323',
         'string_serialization': '{PMID}|{DOI}|{URL}',
         'todos': ['Why is it soil water content method in the name but not the title? '
                   'Is this slot used in other samples?',
                   'Soil water content can be measure MANY ways and often, multiple '
                   'ways are used in one experiment (gravimetric water content and '
                   'water holding capacity and water filled pore space, to name a '
                   'few).',
                   'Should this be multi valued? How to we manage and validate this?']} })
    water_content: Optional[list[str]] = Field(default=None, title="water content", description="""Water content measurement""", json_schema_extra = { "linkml_meta": {'alias': 'water_content',
         'aliases': ['water content'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'string'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'gram per gram or cubic '
                                                     'centimeter per cubic '
                                                     'centimeter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.75 g water/g dry soil'},
                      {'value': '75% water holding capacity'},
                      {'value': '1.1 g fresh weight/ dry weight'},
                      {'value': '10% water filled pore space'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000185',
         'todos': ['value in preferred unit is too limiting. need to change this',
                   'check and correct validation so examples are accepted',
                   'how to manage multiple water content methods?']} })
    water_current: Optional[QuantityValue] = Field(default=None, title="water current", description="""Measurement of magnitude and direction of flow within a fluid""", json_schema_extra = { "linkml_meta": {'alias': 'water_current',
         'aliases': ['water current'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'cubic meter per second, knots'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '10 cubic meter per second'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000203'} })
    water_cut: Optional[QuantityValue] = Field(default=None, title="water cut", description="""Current amount of water (%) in a produced fluid stream; or the average of the combined streams""", json_schema_extra = { "linkml_meta": {'alias': 'water_cut',
         'aliases': ['water cut'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'percent'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000454'} })
    water_feat_size: Optional[QuantityValue] = Field(default=None, title="water feature size", description="""The size of the water feature""", json_schema_extra = { "linkml_meta": {'alias': 'water_feat_size',
         'aliases': ['water feature size'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'square meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000223'} })
    water_feat_type: Optional[WaterFeatTypeEnum] = Field(default=None, title="water feature type", description="""The type of water feature present within the building being sampled""", json_schema_extra = { "linkml_meta": {'alias': 'water_feat_type',
         'aliases': ['water feature type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'stream'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000847'} })
    water_prod_rate: Optional[QuantityValue] = Field(default=None, title="water production rate", description="""Water production rates per well (e.g. 987 m3 / day)""", json_schema_extra = { "linkml_meta": {'alias': 'water_prod_rate',
         'aliases': ['water production rate'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'cubic meter per day'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000453'} })
    water_temp_regm: Optional[list[TextValue]] = Field(default=None, title="water temperature regimen", description="""Information about treatment involving an exposure to water with varying degree of temperature, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple regimens""", json_schema_extra = { "linkml_meta": {'alias': 'water_temp_regm',
         'aliases': ['water temperature regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value;treatment '
                                                     'interval and duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'degree Celsius'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '15 degree '
                                'Celsius;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000590',
         'string_serialization': '{float} {unit};{Rn/start_time/end_time/duration}'} })
    watering_regm: Optional[list[TextValue]] = Field(default=None, title="watering regimen", description="""Information about treatment involving an exposure to watering frequencies, treatment regimen including how many times the treatment was repeated, how long each treatment lasted, and the start and end time of the entire treatment; can include multiple regimens""", json_schema_extra = { "linkml_meta": {'alias': 'watering_regm',
         'aliases': ['watering regimen'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value;treatment '
                                                     'interval and duration'},
                         'occurrence': {'tag': 'occurrence', 'value': 'm'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milliliter, liter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '1 liter;R2/2018-05-11T14:30/2018-05-11T19:30/P1H30M'},
                      {'value': '75% water holding capacity; constant'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000591',
         'string_serialization': '{float} {unit};{Rn/start_time/end_time/duration}'} })
    weekday: Optional[WeekdayEnum] = Field(default=None, title="weekday", description="""The day of the week when sampling occurred""", json_schema_extra = { "linkml_meta": {'alias': 'weekday',
         'aliases': ['weekday'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Sunday'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000848'} })
    win: Optional[TextValue] = Field(default=None, title="well identification number", description="""A unique identifier of a well or wellbore. This is part of the Global Framework for Well Identification initiative which is compiled by the Professional Petroleum Data Management Association (PPDM) in an effort to improve well identification systems. (Supporting information: https://ppdm.org/ and http://dl.ppdm.org/dl/690)""", json_schema_extra = { "linkml_meta": {'alias': 'win',
         'aliases': ['well identification number'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'text'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000297',
         'string_serialization': '{text}'} })
    wind_direction: Optional[TextValue] = Field(default=None, title="wind direction", description="""Wind direction is the direction from which a wind originates""", json_schema_extra = { "linkml_meta": {'alias': 'wind_direction',
         'aliases': ['wind direction'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'wind direction name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Northwest'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000757',
         'string_serialization': '{text}'} })
    wind_speed: Optional[QuantityValue] = Field(default=None, title="wind speed", description="""Speed of wind measured at the time of sampling""", json_schema_extra = { "linkml_meta": {'alias': 'wind_speed',
         'aliases': ['wind speed'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'meter per second, kilometer per '
                                                     'hour'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '21 kilometer per hour'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000118'} })
    window_cond: Optional[WindowCondEnum] = Field(default=None, title="window condition", description="""The physical condition of the window at the time of sampling""", json_schema_extra = { "linkml_meta": {'alias': 'window_cond',
         'aliases': ['window condition'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'rupture'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000849'} })
    window_cover: Optional[WindowCoverEnum] = Field(default=None, title="window covering", description="""The type of window covering""", json_schema_extra = { "linkml_meta": {'alias': 'window_cover',
         'aliases': ['window covering'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'curtains'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000850'} })
    window_horiz_pos: Optional[WindowHorizPosEnum] = Field(default=None, title="window horizontal position", description="""The horizontal position of the window on the wall""", json_schema_extra = { "linkml_meta": {'alias': 'window_horiz_pos',
         'aliases': ['window horizontal position'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'middle'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000851'} })
    window_loc: Optional[WindowLocEnum] = Field(default=None, title="window location", description="""The relative location of the window within the room""", json_schema_extra = { "linkml_meta": {'alias': 'window_loc',
         'aliases': ['window location'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'west'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000852'} })
    window_mat: Optional[WindowMatEnum] = Field(default=None, title="window material", description="""The type of material used to finish a window""", json_schema_extra = { "linkml_meta": {'alias': 'window_mat',
         'aliases': ['window material'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'wood'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000853'} })
    window_open_freq: Optional[TextValue] = Field(default=None, title="window open frequency", description="""The number of times windows are opened per week""", json_schema_extra = { "linkml_meta": {'alias': 'window_open_freq',
         'aliases': ['window open frequency'],
         'annotations': {'expected_value': {'tag': 'expected_value', 'value': 'value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000246'} })
    window_size: Optional[TextValue] = Field(default=None, title="window area/size", description="""The window's length and width""", json_schema_extra = { "linkml_meta": {'alias': 'window_size',
         'aliases': ['window area/size'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'inch, meter'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000224',
         'string_serialization': '{float} {unit} x {float} {unit}'} })
    window_status: Optional[TextValue] = Field(default=None, title="window status", description="""Defines whether the windows were open or closed during environmental testing""", json_schema_extra = { "linkml_meta": {'alias': 'window_status',
         'aliases': ['window status'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'open'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000855',
         'string_serialization': '[closed|open]'} })
    window_type: Optional[WindowTypeEnum] = Field(default=None, title="window type", description="""The type of windows""", json_schema_extra = { "linkml_meta": {'alias': 'window_type',
         'aliases': ['window type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'fixed window'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000856'} })
    window_vert_pos: Optional[WindowVertPosEnum] = Field(default=None, title="window vertical position", description="""The vertical position of the window on the wall""", json_schema_extra = { "linkml_meta": {'alias': 'window_vert_pos',
         'aliases': ['window vertical position'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'middle'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000857'} })
    window_water_mold: Optional[TextValue] = Field(default=None, title="window signs of water/mold", description="""Signs of the presence of mold or mildew on the window.""", json_schema_extra = { "linkml_meta": {'alias': 'window_water_mold',
         'aliases': ['window signs of water/mold'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'enumeration'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': 'no presence of mold visible'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000854',
         'string_serialization': '[presence of mold visible|no presence of mold '
                                 'visible]'} })
    xylene: Optional[QuantityValue] = Field(default=None, title="xylene", description="""Concentration of xylene in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'xylene',
         'aliases': ['xylene'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'milligram per liter, parts per '
                                                     'million'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000156'} })
    ecosystem: Optional[str] = Field(default=None, description="""An ecosystem is a combination of a physical environment (abiotic factors) and all the organisms (biotic factors) that interact with this environment. Ecosystem is in position 1/5 in a GOLD path.""", json_schema_extra = { "linkml_meta": {'alias': 'ecosystem',
         'comments': ['The abiotic factors play a profound role on the type and '
                      'composition of organisms in a given environment. The GOLD '
                      'Ecosystem at the top of the five-level classification system is '
                      'aimed at capturing the broader environment from which an '
                      'organism or environmental sample is collected. The three broad '
                      'groups under Ecosystem are Environmental, Host-associated, and '
                      'Engineered. They represent samples collected from a natural '
                      'environment or from another organism or from engineered '
                      'environments like bioreactors respectively.'],
         'domain_of': ['Biosample', 'Study'],
         'is_a': 'gold_path_field',
         'see_also': ['https://gold.jgi.doe.gov/help']} })
    ecosystem_category: Optional[str] = Field(default=None, description="""Ecosystem categories represent divisions within the ecosystem based on specific characteristics of the environment from where an organism or sample is isolated. Ecosystem category is in position 2/5 in a GOLD path.""", json_schema_extra = { "linkml_meta": {'alias': 'ecosystem_category',
         'comments': ['The Environmental ecosystem (for example) is divided into Air, '
                      'Aquatic and Terrestrial. Ecosystem categories for '
                      'Host-associated samples can be individual hosts or phyla and '
                      'for engineered samples it may be manipulated environments like '
                      'bioreactors, solid waste etc.'],
         'domain_of': ['Biosample', 'Study'],
         'is_a': 'gold_path_field',
         'see_also': ['https://gold.jgi.doe.gov/help']} })
    ecosystem_type: Optional[str] = Field(default=None, description="""Ecosystem types represent things having common characteristics within the Ecosystem Category. These common characteristics based grouping is still broad but specific to the characteristics of a given environment. Ecosystem type is in position 3/5 in a GOLD path.""", json_schema_extra = { "linkml_meta": {'alias': 'ecosystem_type',
         'comments': ['The Aquatic ecosystem category (for example) may have ecosystem '
                      'types like Marine or Thermal springs etc. Ecosystem category '
                      'Air may have Indoor air or Outdoor air as different Ecosystem '
                      'Types. In the case of Host-associated samples, ecosystem type '
                      'can represent Respiratory system, Digestive system, Roots etc.'],
         'domain_of': ['Biosample', 'Study'],
         'is_a': 'gold_path_field',
         'see_also': ['https://gold.jgi.doe.gov/help']} })
    ecosystem_subtype: Optional[str] = Field(default=None, description="""Ecosystem subtypes represent further subdivision of Ecosystem types into more distinct subtypes. Ecosystem subtype is in position 4/5 in a GOLD path.""", json_schema_extra = { "linkml_meta": {'alias': 'ecosystem_subtype',
         'comments': ['Ecosystem Type Marine (Environmental -> Aquatic -> Marine) is '
                      'further divided (for example) into Intertidal zone, Coastal, '
                      'Pelagic, Intertidal zone etc. in the Ecosystem subtype '
                      'category.'],
         'domain_of': ['Biosample', 'Study'],
         'is_a': 'gold_path_field',
         'see_also': ['https://gold.jgi.doe.gov/help']} })
    specific_ecosystem: Optional[str] = Field(default=None, description="""Specific ecosystems represent specific features of the environment like aphotic zone in an ocean or gastric mucosa within a host digestive system. Specific ecosystem is in position 5/5 in a GOLD path.""", json_schema_extra = { "linkml_meta": {'alias': 'specific_ecosystem',
         'comments': ['Specific ecosystems help to define samples based on very '
                      'specific characteristics of an environment under the five-level '
                      'classification system.'],
         'domain_of': ['Biosample', 'Study'],
         'is_a': 'gold_path_field',
         'see_also': ['https://gold.jgi.doe.gov/help']} })
    add_date: Optional[str] = Field(default=None, description="""The date on which the information was added to the database.""", json_schema_extra = { "linkml_meta": {'alias': 'add_date', 'domain_of': ['Biosample', 'DataGeneration']} })
    community: Optional[str] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'community', 'domain_of': ['Biosample']} })
    habitat: Optional[str] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'habitat', 'domain_of': ['FieldResearchSite', 'Biosample']} })
    host_name: Optional[str] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'host_name', 'domain_of': ['Biosample']} })
    location: Optional[str] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'location', 'domain_of': ['Biosample']} })
    mod_date: Optional[str] = Field(default=None, description="""The last date on which the database information was modified.""", json_schema_extra = { "linkml_meta": {'alias': 'mod_date', 'domain_of': ['Biosample', 'DataGeneration']} })
    ncbi_taxonomy_name: Optional[str] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'ncbi_taxonomy_name', 'domain_of': ['Biosample']} })
    proport_woa_temperature: Optional[str] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'proport_woa_temperature', 'domain_of': ['Biosample']} })
    salinity_category: Optional[str] = Field(default=None, description="""Categorical description of the sample's salinity. Examples: halophile, halotolerant, hypersaline, huryhaline""", json_schema_extra = { "linkml_meta": {'alias': 'salinity_category',
         'domain_of': ['Biosample'],
         'notes': ['maps to gold:salinity'],
         'see_also': ['https://github.com/microbiomedata/nmdc-metadata/pull/297']} })
    sample_collection_site: Optional[str] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'sample_collection_site', 'domain_of': ['Biosample']} })
    soluble_iron_micromol: Optional[str] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'soluble_iron_micromol', 'domain_of': ['Biosample']} })
    subsurface_depth: Optional[QuantityValue] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'subsurface_depth', 'domain_of': ['Biosample']} })
    dna_absorb1: Optional[float] = Field(default=None, title="DNA absorbance 260/280", description="""260/280 measurement of DNA sample purity""", json_schema_extra = { "linkml_meta": {'alias': 'dna_absorb1',
         'comments': ['Recommended value is between 1 and 3.'],
         'domain_of': ['Biosample', 'ProcessedSample'],
         'examples': [{'value': '2.02'}],
         'is_a': 'biomaterial_purity',
         'rank': 7,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics'} })
    dna_absorb2: Optional[float] = Field(default=None, title="DNA absorbance 260/230", description="""260/230 measurement of DNA sample purity""", json_schema_extra = { "linkml_meta": {'alias': 'dna_absorb2',
         'comments': ['Recommended value is between 1 and 3.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '2.02'}],
         'is_a': 'biomaterial_purity',
         'rank': 8,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics'} })
    dna_collect_site: Optional[str] = Field(default=None, title="DNA collection site", description="""Provide information on the site your DNA sample was collected from""", json_schema_extra = { "linkml_meta": {'alias': 'dna_collect_site',
         'domain_of': ['Biosample'],
         'examples': [{'value': 'untreated pond water'}],
         'rank': 15,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics',
         'string_serialization': '{text}'} })
    dna_concentration: Optional[float] = Field(default=None, title="DNA concentration in ng/ul", ge=0, le=2000, json_schema_extra = { "linkml_meta": {'alias': 'dna_concentration',
         'comments': ['Units must be in ng/uL. Enter the numerical part only. Must be '
                      'calculated using a fluorometric method. Acceptable values are '
                      '0-2000.'],
         'domain_of': ['Biosample', 'ProcessedSample'],
         'examples': [{'value': '100'}],
         'rank': 5,
         'recommended': True,
         'see_also': ['nmdc:nucleic_acid_concentration'],
         'slot_group': 'JGI-Metagenomics'} })
    dna_cont_type: Optional[JgiContTypeEnum] = Field(default=None, title="DNA container type", description="""Tube or plate (96-well)""", json_schema_extra = { "linkml_meta": {'alias': 'dna_cont_type',
         'domain_of': ['Biosample'],
         'examples': [{'value': 'plate'}],
         'rank': 10,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics'} })
    dna_cont_well: Optional[str] = Field(default=None, title="DNA plate position", json_schema_extra = { "linkml_meta": {'alias': 'dna_cont_well',
         'comments': ["Required when 'plate' is selected for container type.",
                      'Leave blank if the sample will be shipped in a tube.',
                      'JGI will not process samples in corner wells, so A1, A12, H1 '
                      'and H12 will not pass validation.',
                      'For partial plates, fill by columns, like B1-G1,A2-H2,A3-D3 '
                      '(NOT A2-A11,B1-B8).'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'B2'}],
         'rank': 11,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics',
         'string_serialization': '{96 well plate pos}'} })
    dna_container_id: Optional[str] = Field(default=None, title="DNA container label", json_schema_extra = { "linkml_meta": {'alias': 'dna_container_id',
         'comments': ['Must be unique across all tubes and plates, and <20 characters. '
                      'All samples in a plate should have the same plate label.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Pond_MT_041618'}],
         'rank': 9,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics',
         'string_serialization': '{text < 20 characters}'} })
    dna_dnase: Optional[YesNoEnum] = Field(default=None, title="DNase treatment DNA", json_schema_extra = { "linkml_meta": {'alias': 'dna_dnase',
         'comments': ['Note DNase treatment is required for all RNA samples.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'no'}],
         'rank': 13,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics'} })
    dna_isolate_meth: Optional[str] = Field(default=None, title="DNA isolation method", description="""Describe the method/protocol/kit used to extract DNA/RNA.""", json_schema_extra = { "linkml_meta": {'alias': 'dna_isolate_meth',
         'aliases': ['Sample Isolation Method'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'phenol/chloroform extraction'}],
         'rank': 16,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics',
         'string_serialization': '{text}'} })
    dna_organisms: Optional[str] = Field(default=None, title="DNA expected organisms", description="""List any organisms known or suspected to grow in co-culture, as well as estimated % of the organism in that culture.""", json_schema_extra = { "linkml_meta": {'alias': 'dna_organisms',
         'domain_of': ['Biosample'],
         'examples': [{'value': 'expected to contain microbes (59%) fungi (30%), '
                                'viruses (10%), tadpoles (1%)'}],
         'rank': 14,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics',
         'string_serialization': '{text}'} })
    dna_project_contact: Optional[str] = Field(default=None, title="DNA seq project contact", json_schema_extra = { "linkml_meta": {'alias': 'dna_project_contact',
         'comments': ['Do not edit these values. A template will be provided by NMDC '
                      'in which these values have been pre-filled.'],
         'deprecated': 'This slot is not always provided by JGI and does not need to '
                       'be in the file that we send back to JGI. Not needed in UI',
         'domain_of': ['Biosample'],
         'examples': [{'value': 'John Jones'}],
         'last_updated_on': '2025-07-02T00:00:00',
         'modified_by': 'ORCID:0000-0002-8683-0050',
         'rank': 18,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics',
         'string_serialization': '{text}'} })
    dna_samp_id: Optional[str] = Field(default=None, title="DNA sample ID", json_schema_extra = { "linkml_meta": {'alias': 'dna_samp_id',
         'comments': ['Do not edit these values. A template will be provided by NMDC '
                      'in which these values have been pre-filled.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '187654'}],
         'rank': 3,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics',
         'string_serialization': '{text}',
         'todos': ['Removed identifier = TRUE from dna_samp_ID in JGI_sample_slots, as '
                   "a class can't have two identifiers. How to force uniqueness? Moot "
                   'because that column will be prefilled?']} })
    dna_sample_format: Optional[DNASampleFormatEnum] = Field(default=None, title="DNA sample format", description="""Solution in which the DNA sample has been suspended""", json_schema_extra = { "linkml_meta": {'alias': 'dna_sample_format',
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Water'}],
         'rank': 12,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics'} })
    dna_sample_name: Optional[str] = Field(default=None, title="DNA sample name", description="""Give the DNA sample a name that is meaningful to you. Sample names must be unique across all JGI projects and contain a-z, A-Z, 0-9, - and _ only.""", json_schema_extra = { "linkml_meta": {'alias': 'dna_sample_name',
         'domain_of': ['Biosample'],
         'examples': [{'value': 'JGI_pond_041618'}],
         'rank': 4,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics',
         'string_serialization': '{text}'} })
    dna_seq_project: Optional[str] = Field(default=None, title="DNA seq project ID", json_schema_extra = { "linkml_meta": {'alias': 'dna_seq_project',
         'aliases': ['Seq Project ID'],
         'comments': ['Do not edit these values. A template will be provided by NMDC '
                      'in which these values have been pre-filled.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '1191234'}],
         'rank': 1,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics',
         'string_serialization': '{text}'} })
    dna_seq_project_pi: Optional[str] = Field(default=None, title="DNA seq project PI", json_schema_extra = { "linkml_meta": {'alias': 'dna_seq_project_pi',
         'comments': ['Do not edit these values. A template will be provided by NMDC '
                      'in which these values have been pre-filled.'],
         'deprecated': 'This slot is not always provided by JGI and does not need to '
                       'be in the file that we send back to JGI. Not needed in UI',
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Jane Johnson'}],
         'last_updated_on': '2025-07-02T00:00:00',
         'modified_by': 'ORCID:0000-0002-8683-0050',
         'rank': 17,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics',
         'string_serialization': '{text}'} })
    dna_seq_project_name: Optional[str] = Field(default=None, title="DNA seq project name", json_schema_extra = { "linkml_meta": {'alias': 'dna_seq_project_name',
         'comments': ['Do not edit these values. A template will be provided by NMDC '
                      'in which these values have been pre-filled.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'JGI Pond metagenomics'}],
         'rank': 2,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics',
         'string_serialization': '{text}'} })
    dna_volume: Optional[float] = Field(default=None, title="DNA volume in ul", ge=0, le=1000, json_schema_extra = { "linkml_meta": {'alias': 'dna_volume',
         'comments': ['Units must be in uL. Enter the numerical part only. Value must '
                      'be 0-1000. This form accepts values < 25, but JGI may refuse to '
                      'process them unless permission has been granted by a project '
                      'manager'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '25'}],
         'rank': 6,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics',
         'string_serialization': '{float}'} })
    proposal_dna: Optional[str] = Field(default=None, title="DNA proposal ID", json_schema_extra = { "linkml_meta": {'alias': 'proposal_dna',
         'comments': ['Do not edit these values. A template will be provided by NMDC '
                      'in which these values have been pre-filled.'],
         'deprecated': 'This slot is not always provided by JGI and does not need to '
                       'be in the file that we send back to JGI. Not needed in UI',
         'domain_of': ['Biosample'],
         'examples': [{'value': '504000'}],
         'last_updated_on': '2025-07-02T00:00:00',
         'modified_by': 'ORCID:0000-0002-8683-0050',
         'rank': 19,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics',
         'string_serialization': '{text}'} })
    dnase_rna: Optional[YesNoEnum] = Field(default=None, title="DNase treated", json_schema_extra = { "linkml_meta": {'alias': 'dnase_rna',
         'aliases': ['Was Sample DNAse treated?'],
         'comments': ['Note DNase treatment is required for all RNA samples.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'no'}],
         'rank': 13,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics'} })
    proposal_rna: Optional[str] = Field(default=None, title="RNA proposal ID", json_schema_extra = { "linkml_meta": {'alias': 'proposal_rna',
         'comments': ['Do not edit these values. A template will be provided by NMDC '
                      'in which these values have been pre-filled.'],
         'deprecated': 'This slot is not always provided by JGI and does not need to '
                       'be in the file that we send back to JGI. Not needed in UI',
         'domain_of': ['Biosample'],
         'examples': [{'value': '504000'}],
         'last_updated_on': '2025-07-02T00:00:00',
         'modified_by': 'ORCID:0000-0002-8683-0050',
         'rank': 19,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics',
         'string_serialization': '{text}'} })
    rna_absorb1: Optional[float] = Field(default=None, title="RNA absorbance 260/280", description="""260/280 measurement of RNA sample purity""", json_schema_extra = { "linkml_meta": {'alias': 'rna_absorb1',
         'comments': ['Recommended value is between 1 and 3.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '2.02'}],
         'rank': 7,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics',
         'string_serialization': '{float}'} })
    rna_absorb2: Optional[float] = Field(default=None, title="RNA absorbance 260/230", description="""260/230 measurement of RNA sample purity""", json_schema_extra = { "linkml_meta": {'alias': 'rna_absorb2',
         'comments': ['Recommended value is between 1 and 3.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '2.02'}],
         'rank': 8,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics',
         'string_serialization': '{float}'} })
    rna_collect_site: Optional[str] = Field(default=None, title="RNA collection site", description="""Provide information on the site your RNA sample was collected from""", json_schema_extra = { "linkml_meta": {'alias': 'rna_collect_site',
         'domain_of': ['Biosample'],
         'examples': [{'value': 'untreated pond water'}],
         'rank': 15,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics',
         'string_serialization': '{text}'} })
    rna_concentration: Optional[float] = Field(default=None, title="RNA concentration in ng/ul", ge=0, le=1000, json_schema_extra = { "linkml_meta": {'alias': 'rna_concentration',
         'comments': ['Units must be in ng/uL. Enter the numerical part only. Must be '
                      'calculated using a fluorometric method. Acceptable values are '
                      '0-2000.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '100'}],
         'rank': 5,
         'recommended': True,
         'see_also': ['nmdc:nucleic_acid_concentration'],
         'slot_group': 'JGI-Metatranscriptomics',
         'string_serialization': '{float}'} })
    rna_cont_type: Optional[JgiContTypeEnum] = Field(default=None, title="RNA container type", description="""Tube or plate (96-well)""", json_schema_extra = { "linkml_meta": {'alias': 'rna_cont_type',
         'domain_of': ['Biosample'],
         'examples': [{'value': 'plate'}],
         'rank': 10,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics'} })
    rna_cont_well: Optional[str] = Field(default=None, title="RNA plate position", json_schema_extra = { "linkml_meta": {'alias': 'rna_cont_well',
         'comments': ["Required when 'plate' is selected for container type.",
                      'Leave blank if the sample will be shipped in a tube.',
                      'JGI will not process samples in corner wells, so A1, A12, H1 '
                      'and H12 will not pass validation.',
                      'For partial plates, fill by columns, like B1-G1,A2-H2,A3-D3 '
                      '(NOT A2-A11,B1-B8).'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'B2'}],
         'rank': 11,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics',
         'string_serialization': '{96 well plate pos}'} })
    rna_container_id: Optional[str] = Field(default=None, title="RNA container label", json_schema_extra = { "linkml_meta": {'alias': 'rna_container_id',
         'comments': ['Must be unique across all tubes and plates, and <20 characters. '
                      'All samples in a plate should have the same plate label.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Pond_MT_041618'}],
         'rank': 9,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics',
         'string_serialization': '{text < 20 characters}'} })
    rna_isolate_meth: Optional[str] = Field(default=None, title="RNA isolation method", description="""Describe the method/protocol/kit used to extract DNA/RNA.""", json_schema_extra = { "linkml_meta": {'alias': 'rna_isolate_meth',
         'aliases': ['Sample Isolation Method'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'phenol/chloroform extraction'}],
         'rank': 16,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics',
         'string_serialization': '{text}'} })
    rna_organisms: Optional[str] = Field(default=None, title="RNA expected organisms", description="""List any organisms known or suspected to grow in co-culture, as well as estimated % of the organism in that culture.""", json_schema_extra = { "linkml_meta": {'alias': 'rna_organisms',
         'domain_of': ['Biosample'],
         'examples': [{'value': 'expected to contain microbes (59%) fungi (30%), '
                                'viruses (10%), tadpoles (1%)'}],
         'rank': 14,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics',
         'string_serialization': '{text}'} })
    rna_project_contact: Optional[str] = Field(default=None, title="RNA seq project contact", json_schema_extra = { "linkml_meta": {'alias': 'rna_project_contact',
         'comments': ['Do not edit these values. A template will be provided by NMDC '
                      'in which these values have been pre-filled.'],
         'deprecated': 'This slot is not always provided by JGI and does not need to '
                       'be in the file that we send back to JGI. Not needed in UI',
         'domain_of': ['Biosample'],
         'examples': [{'value': 'John Jones'}],
         'last_updated_on': '2025-07-02T00:00:00',
         'modified_by': 'ORCID:0000-0002-8683-0050',
         'rank': 18,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics',
         'string_serialization': '{text}'} })
    rna_samp_id: Optional[str] = Field(default=None, title="RNA sample ID", json_schema_extra = { "linkml_meta": {'alias': 'rna_samp_id',
         'comments': ['Do not edit these values. A template will be provided by NMDC '
                      'in which these values have been pre-filled.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '187654'}],
         'rank': 3,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics',
         'string_serialization': '{text}'} })
    rna_sample_format: Optional[RNASampleFormatEnum] = Field(default=None, title="RNA sample format", description="""Solution in which the RNA sample has been suspended""", json_schema_extra = { "linkml_meta": {'alias': 'rna_sample_format',
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Water'}],
         'rank': 12,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics'} })
    rna_sample_name: Optional[str] = Field(default=None, title="RNA sample name", description="""Give the RNA sample a name that is meaningful to you. Sample names must be unique across all JGI projects and contain a-z, A-Z, 0-9, - and _ only.""", ge=0, le=2000, json_schema_extra = { "linkml_meta": {'alias': 'rna_sample_name',
         'domain_of': ['Biosample'],
         'examples': [{'value': 'JGI_pond_041618'}],
         'rank': 4,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics',
         'string_serialization': '{text}'} })
    rna_seq_project: Optional[str] = Field(default=None, title="RNA seq project ID", json_schema_extra = { "linkml_meta": {'alias': 'rna_seq_project',
         'aliases': ['Seq Project ID'],
         'comments': ['Do not edit these values. A template will be provided by NMDC '
                      'in which these values have been pre-filled.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '1191234'}],
         'rank': 1,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics',
         'string_serialization': '{text}'} })
    rna_seq_project_pi: Optional[str] = Field(default=None, title="RNA seq project PI", json_schema_extra = { "linkml_meta": {'alias': 'rna_seq_project_pi',
         'comments': ['Do not edit these values. A template will be provided by NMDC '
                      'in which these values have been pre-filled.'],
         'deprecated': 'This slot is not always provided by JGI and does not need to '
                       'be in the file that we send back to JGI. Not needed in UI',
         'domain_of': ['Biosample'],
         'examples': [{'value': 'Jane Johnson'}],
         'last_updated_on': '2025-07-02T00:00:00',
         'modified_by': 'ORCID:0000-0002-8683-0050',
         'rank': 17,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics',
         'string_serialization': '{text}'} })
    rna_seq_project_name: Optional[str] = Field(default=None, title="RNA seq project name", json_schema_extra = { "linkml_meta": {'alias': 'rna_seq_project_name',
         'comments': ['Do not edit these values. A template will be provided by NMDC '
                      'in which these values have been pre-filled.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'JGI Pond metatranscriptomics'}],
         'rank': 2,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics',
         'string_serialization': '{text}'} })
    rna_volume: Optional[float] = Field(default=None, title="RNA volume in ul", ge=0, le=1000, json_schema_extra = { "linkml_meta": {'alias': 'rna_volume',
         'comments': ['Units must be in uL. Enter the numerical part only. Value must '
                      'be 0-1000. This form accepts values < 25, but JGI may refuse to '
                      'process them unless permission has been granted by a project '
                      'manager'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '25'}],
         'rank': 6,
         'recommended': True,
         'slot_group': 'JGI-Metatranscriptomics',
         'string_serialization': '{float}'} })
    collection_date_inc: Optional[str] = Field(default=None, title="incubation collection date", description="""Date the incubation was harvested/collected/ended. Only relevant for incubation samples.""", json_schema_extra = { "linkml_meta": {'alias': 'collection_date_inc',
         'comments': ['Date should be formatted as YYYY(-MM(-DD)). Ie, 2021-04-15, '
                      '2021-04 and 2021 are all acceptable.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '2021-04-15'}, {'value': '2021-04'}, {'value': '2021'}],
         'notes': ['MIxS collection_date accepts (truncated) ISO8601. DH taking '
                   'arbitrary precision date only'],
         'rank': 2,
         'recommended': True,
         'see_also': ['MIXS:0000011'],
         'slot_group': 'MIxS Inspired',
         'string_serialization': '{date, arbitrary precision}'} })
    collection_time: Optional[str] = Field(default=None, title="collection time, GMT", description="""The time of sampling, either as an instance (single point) or interval.""", json_schema_extra = { "linkml_meta": {'alias': 'collection_time',
         'comments': ['Time should be entered as HH:MM(:SS) in GMT. See here for a '
                      'converter: https://www.worldtimebuddy.com/pst-to-gmt-converter'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '13:33'}, {'value': '13:33:55'}],
         'notes': ['MIxS collection_date accepts (truncated) ISO8601. DH taking '
                   'seconds optional time only'],
         'rank': 1,
         'recommended': True,
         'see_also': ['MIXS:0000011'],
         'slot_group': 'MIxS Inspired',
         'string_serialization': '{time, seconds optional}'} })
    collection_time_inc: Optional[str] = Field(default=None, title="incubation collection time, GMT", description="""Time the incubation was harvested/collected/ended. Only relevant for incubation samples.""", json_schema_extra = { "linkml_meta": {'alias': 'collection_time_inc',
         'comments': ['Time should be entered as HH:MM(:SS) in GMT. See here for a '
                      'converter: https://www.worldtimebuddy.com/pst-to-gmt-converter'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '13:33'}, {'value': '13:33:55'}],
         'notes': ['MIxS collection_date accepts (truncated) ISO8601. DH taking '
                   'seconds optional time only'],
         'rank': 3,
         'recommended': True,
         'see_also': ['MIXS:0000011'],
         'slot_group': 'MIxS Inspired',
         'string_serialization': '{time, seconds optional}'} })
    experimental_factor_other: Optional[str] = Field(default=None, title="experimental factor- other", description="""Other details about your sample that you feel can't be accurately represented in the available columns.""", json_schema_extra = { "linkml_meta": {'alias': 'experimental_factor_other',
         'comments': ['This slot accepts open-ended text about your sample.',
                      'We recommend using key:value pairs.',
                      'Provided pairs will be considered for inclusion as future '
                      'slots/terms in this data collection template.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'experimental treatment: value'}],
         'rank': 7,
         'recommended': True,
         'see_also': ['MIXS:0000008', 'MIXS:0000300'],
         'slot_group': 'MIxS Inspired',
         'string_serialization': '{text}'} })
    filter_method: Optional[str] = Field(default=None, title="filter method", description="""Type of filter used or how the sample was filtered""", json_schema_extra = { "linkml_meta": {'alias': 'filter_method',
         'comments': ['describe the filter or provide a catalog number and '
                      'manufacturer'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'C18'}, {'value': 'Basix PES, 13-100-106 FisherSci'}],
         'rank': 6,
         'recommended': True,
         'see_also': ['MIXS:0000765'],
         'slot_group': 'MIxS Inspired',
         'string_serialization': '{text}'} })
    isotope_exposure: Optional[str] = Field(default=None, title="isotope exposure/addition", description="""List isotope exposure or addition applied to your sample.""", json_schema_extra = { "linkml_meta": {'alias': 'isotope_exposure',
         'comments': ['This is required when your experimental design includes the use '
                      'of isotopically labeled compounds'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '13C glucose'}, {'value': '18O water'}],
         'rank': 16,
         'recommended': True,
         'see_also': ['MIXS:0000751'],
         'slot_group': 'MIxS Inspired',
         'todos': ['Can we make the H218O correctly super and subscripted?']} })
    micro_biomass_c_meth: Optional[str] = Field(default=None, title="microbial biomass carbon method", description="""Reference or method used in determining microbial biomass carbon""", json_schema_extra = { "linkml_meta": {'alias': 'micro_biomass_c_meth',
         'comments': ['required if "microbial_biomass_c" is provided'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'https://doi.org/10.1016/0038-0717(87)90052-6'},
                      {'value': 'https://doi.org/10.1016/0038-0717(87)90052-6 | '
                                'https://www.sciencedirect.com/science/article/abs/pii/0038071787900526'}],
         'rank': 11,
         'recommended': True,
         'see_also': ['MIXS:0000339'],
         'slot_group': 'MIxS Inspired',
         'string_serialization': '{PMID}|{DOI}|{URL}',
         'todos': ['How should we separate values? | or ;? lets be consistent']} })
    micro_biomass_n_meth: Optional[str] = Field(default=None, title="microbial biomass nitrogen method", description="""Reference or method used in determining microbial biomass nitrogen""", json_schema_extra = { "linkml_meta": {'alias': 'micro_biomass_n_meth',
         'comments': ['required if "microbial_biomass_n" is provided'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'https://doi.org/10.1016/0038-0717(87)90052-6'},
                      {'value': 'https://doi.org/10.1016/0038-0717(87)90052-6 | '
                                'https://www.sciencedirect.com/science/article/abs/pii/0038071787900526'}],
         'rank': 13,
         'see_also': ['MIXS:0000339'],
         'slot_group': 'MIxS Inspired',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    microbial_biomass_c: Optional[str] = Field(default=None, title="microbial biomass carbon", description="""The part of the organic matter in the soil that constitutes living microorganisms smaller than 5-10 micrometer.""", json_schema_extra = { "linkml_meta": {'alias': 'microbial_biomass_c',
         'comments': ['If you provide this, correction factors used for conversion to '
                      'the final units and method are required'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.05 ug C/g dry soil'}],
         'rank': 10,
         'see_also': ['MIXS:0000650'],
         'slot_group': 'MIxS Inspired',
         'string_serialization': '{float} {unit}'} })
    microbial_biomass_n: Optional[str] = Field(default=None, title="microbial biomass nitrogen", description="""The part of the organic matter in the soil that constitutes living microorganisms smaller than 5-10 micrometer.""", json_schema_extra = { "linkml_meta": {'alias': 'microbial_biomass_n',
         'comments': ['If you provide this, correction factors used for conversion to '
                      'the final units and method are required'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.05 ug N/g dry soil'}],
         'rank': 12,
         'see_also': ['MIXS:0000650'],
         'slot_group': 'MIxS Inspired',
         'string_serialization': '{float} {unit}'} })
    non_microb_biomass: Optional[str] = Field(default=None, title="non-microbial biomass", description="""Amount of biomass; should include the name for the part of biomass measured, e.g.insect, plant, total. Can include multiple measurements separated by ;""", json_schema_extra = { "linkml_meta": {'alias': 'non_microb_biomass',
         'domain_of': ['Biosample'],
         'examples': [{'value': 'insect 0.23 ug; plant 1g'}],
         'rank': 8,
         'see_also': ['MIXS:0000174', 'MIXS:0000650'],
         'slot_group': 'MIxS Inspired',
         'string_serialization': '{text};{float} {unit}'} })
    non_microb_biomass_method: Optional[str] = Field(default=None, title="non-microbial biomass method", description="""Reference or method used in determining biomass""", json_schema_extra = { "linkml_meta": {'alias': 'non_microb_biomass_method',
         'comments': ['required if "non-microbial biomass" is provided'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'https://doi.org/10.1038/s41467-021-26181-3'}],
         'rank': 9,
         'see_also': ['MIXS:0000650'],
         'slot_group': 'MIxS Inspired',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    org_nitro_method: Optional[str] = Field(default=None, title="organic nitrogen method", description="""Method used for obtaining organic nitrogen""", json_schema_extra = { "linkml_meta": {'alias': 'org_nitro_method',
         'comments': ['required if "org_nitro" is provided'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'https://doi.org/10.1016/0038-0717(85)90144-0'}],
         'rank': 14,
         'see_also': ['MIXS:0000338', 'MIXS:0000205'],
         'slot_group': 'MIxS Inspired',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    other_treatment: Optional[str] = Field(default=None, title="other treatments", description="""Other treatments applied to your samples that are not applicable to the provided fields""", json_schema_extra = { "linkml_meta": {'alias': 'other_treatment',
         'comments': ['This is an open text field to provide any treatments that '
                      'cannot be captured in the provided slots.'],
         'domain_of': ['Biosample'],
         'notes': ['Values entered here will be used to determine potential new '
                   'slots.'],
         'rank': 15,
         'recommended': True,
         'see_also': ['MIXS:0000300'],
         'slot_group': 'MIxS Inspired',
         'string_serialization': '{text}'} })
    start_date_inc: Optional[str] = Field(default=None, title="incubation start date", description="""Date the incubation was started. Only relevant for incubation samples.""", json_schema_extra = { "linkml_meta": {'alias': 'start_date_inc',
         'comments': ['Date should be formatted as YYYY(-MM(-DD)). Ie, 2021-04-15, '
                      '2021-04 and 2021 are all acceptable.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '2021-04-15'}, {'value': '2021-04'}, {'value': '2021'}],
         'notes': ['MIxS collection_date accepts (truncated) ISO8601. DH taking '
                   'arbitrary precision date only'],
         'rank': 4,
         'recommended': True,
         'see_also': ['MIXS:0000011'],
         'slot_group': 'MIxS Inspired',
         'string_serialization': '{date, arbitrary precision}'} })
    start_time_inc: Optional[str] = Field(default=None, title="incubation start time, GMT", description="""Time the incubation was started. Only relevant for incubation samples.""", json_schema_extra = { "linkml_meta": {'alias': 'start_time_inc',
         'comments': ['Time should be entered as HH:MM(:SS) in GMT. See here for a '
                      'converter: https://www.worldtimebuddy.com/pst-to-gmt-converter'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '13:33'}, {'value': '13:33:55'}],
         'notes': ['MIxS collection_date accepts (truncated) ISO8601. DH taking '
                   'seconds optional time only'],
         'rank': 5,
         'recommended': True,
         'see_also': ['MIXS:0000011'],
         'slot_group': 'MIxS Inspired',
         'string_serialization': '{time, seconds optional}'} })
    project_id: Optional[str] = Field(default=None, title="project ID", description="""Proposal IDs or names associated with dataset""", json_schema_extra = { "linkml_meta": {'alias': 'project_id',
         'domain_of': ['Biosample'],
         'rank': 1,
         'recommended': True,
         'slot_group': 'EMSL',
         'string_serialization': '{text}'} })
    replicate_number: Optional[str] = Field(default=None, title="replicate number", description="""If sending biological replicates, indicate the rep number here.""", json_schema_extra = { "linkml_meta": {'alias': 'replicate_number',
         'comments': ['This will guide staff in ensuring your samples are blocked & '
                      'randomized correctly'],
         'domain_of': ['Biosample'],
         'rank': 6,
         'recommended': True,
         'slot_group': 'EMSL',
         'string_serialization': '{integer}'} })
    sample_shipped: Optional[str] = Field(default=None, title="sample shipped amount", description="""The total amount or size (volume (ml), mass (g) or area (m2) ) of sample sent to EMSL.""", json_schema_extra = { "linkml_meta": {'alias': 'sample_shipped',
         'comments': ['This field is only required when completing metadata for '
                      'samples being submitted to EMSL for analyses.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '15 g'}, {'value': '100 uL'}, {'value': '5 mL'}],
         'rank': 3,
         'recommended': True,
         'slot_group': 'EMSL',
         'string_serialization': '{float} {unit}'} })
    sample_type: Optional[SampleTypeEnum] = Field(default=None, title="sample type", description="""Type of sample being submitted""", json_schema_extra = { "linkml_meta": {'alias': 'sample_type',
         'comments': ["This can vary from 'environmental package' if the sample is an "
                      'extraction.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'water extracted soil'}],
         'rank': 2,
         'recommended': True,
         'slot_group': 'EMSL'} })
    technical_reps: Optional[str] = Field(default=None, title="number technical replicate", description="""If sending technical replicates of the same sample, indicate the replicate count.""", json_schema_extra = { "linkml_meta": {'alias': 'technical_reps',
         'comments': ['This field is only required when completing metadata for '
                      'samples being submitted to EMSL for analyses.'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '2'}],
         'rank': 5,
         'recommended': True,
         'slot_group': 'EMSL',
         'string_serialization': '{integer}'} })
    analysis_type: Optional[list[AnalysisTypeEnum]] = Field(default=None, title="analysis/data type", description="""Select all the data types associated or available for this biosample""", json_schema_extra = { "linkml_meta": {'alias': 'analysis_type',
         'comments': ['MIxS:investigation_type was included as a `see_also` but that '
                      "term doesn't resolve any more"],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'metagenomics; metabolomics; metaproteomics'}],
         'rank': 3,
         'recommended': True,
         'slot_group': 'Sample ID'} })
    sample_link: Optional[list[str]] = Field(default=None, title="sample linkage", description="""A unique identifier to assign parent-child, subsample, or sibling samples. This is relevant when a sample or other material was used to generate the new sample.""", json_schema_extra = { "linkml_meta": {'alias': 'sample_link',
         'comments': ['This field allows multiple entries separated by ; (Examples: '
                      'Soil collected from the field will link with the soil used in '
                      'an incubation. The soil a plant was grown in links to the plant '
                      'sample. An original culture sample was transferred to a new '
                      'vial and generated a new sample)'],
         'domain_of': ['Biosample'],
         'examples': [{'value': 'IGSN:DSJ0284'}],
         'rank': 5,
         'recommended': True,
         'slot_group': 'Sample ID',
         'string_serialization': '{text}:{text}'} })
    bulk_elect_conductivity: Optional[QuantityValue] = Field(default=None, title="bulk electrical conductivity", description="""Electrical conductivity is a measure of the ability to carry electric current, which is mostly dictated by the chemistry of and amount of water.""", json_schema_extra = { "linkml_meta": {'alias': 'bulk_elect_conductivity',
         'comments': ['Provide the value output of the field instrument.'],
         'domain_of': ['Biosample'],
         'examples': [{'description': 'The conductivity measurement was 0.017 '
                                      'millisiemens per centimeter.',
                       'value': "JsonObj(has_raw_value='0.017 mS/cm', "
                                "has_numeric_value=0.017, has_unit='mS/cm')"}]} })
    infiltrations: Optional[list[str]] = Field(default=None, description="""The amount of time it takes to complete each infiltration activity""", json_schema_extra = { "linkml_meta": {'alias': 'infiltrations',
         'aliases': ['infiltration_1', 'infiltration_2'],
         'domain_of': ['Biosample'],
         'examples': [{'value': "['00:01:32', '00:00:53']"}],
         'list_elements_ordered': True,
         'see_also': ['https://www.protocols.io/view/field-sampling-protocol-kqdg3962pg25/v1']} })
    zinc: Optional[QuantityValue] = Field(default=None, title="zinc", description="""Concentration of zinc in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'zinc',
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'mg/kg (ppm)'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2.5 mg/kg'}],
         'see_also': ['https://www.ornl.gov/content/bio-scales-0']} })
    manganese: Optional[QuantityValue] = Field(default=None, title="manganese", description="""Concentration of manganese in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'manganese',
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'mg/kg (ppm)'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '24.7 mg/kg'}],
         'see_also': ['https://www.ornl.gov/content/bio-scales-0']} })
    ammonium_nitrogen: Optional[QuantityValue] = Field(default=None, title="ammonium nitrogen", description="""Concentration of ammonium nitrogen in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'ammonium_nitrogen',
         'aliases': ['NH4-N'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'mg/kg'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '2.3 mg/kg'}],
         'see_also': ['https://www.ornl.gov/content/bio-scales-0']} })
    nitrate_nitrogen: Optional[QuantityValue] = Field(default=None, title="nitrate_nitrogen", description="""Concentration of nitrate nitrogen in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'nitrate_nitrogen',
         'aliases': ['NO3-N'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'mg/kg'}},
         'comments': ['often below some specified limit of detection'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '0.29 mg/kg'}],
         'see_also': ['https://www.ornl.gov/content/bio-scales-0']} })
    nitrite_nitrogen: Optional[QuantityValue] = Field(default=None, title="nitrite_nitrogen", description="""Concentration of nitrite nitrogen in the sample""", json_schema_extra = { "linkml_meta": {'alias': 'nitrite_nitrogen',
         'aliases': ['NO2-N'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit', 'value': 'mg/kg'}},
         'domain_of': ['Biosample'],
         'examples': [{'value': '1.2 mg/kg'}],
         'see_also': ['https://www.ornl.gov/content/bio-scales-0']} })
    lbc_thirty: Optional[QuantityValue] = Field(default=None, title="lime buffer capacity (at 30 minutes)", description="""lime buffer capacity, determined after 30 minute incubation""", json_schema_extra = { "linkml_meta": {'alias': 'lbc_thirty',
         'aliases': ['lbc30', 'lime buffer capacity (at 30 minutes)'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'ppm CaCO3/pH'}},
         'comments': ['This is the mass of lime, in mg, needed to raise the pH of one '
                      'kg of soil by one pH unit'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '543 mg/kg'}],
         'see_also': ['https://www.ornl.gov/content/bio-scales-0',
                      'https://secure.caes.uga.edu/extension/publications/files/pdf/C%20874_5.PDF']} })
    lbceq: Optional[QuantityValue] = Field(default=None, title="lime buffer capacity (after 5 day incubation)", description="""lime buffer capacity, determined at equilibrium after 5 day incubation""", json_schema_extra = { "linkml_meta": {'alias': 'lbceq',
         'aliases': ['lime buffer capacity (at 5-day equilibrium)'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'},
                         'preferred_unit': {'tag': 'preferred_unit',
                                            'value': 'ppm CaCO3/pH'}},
         'comments': ['This is the mass of lime, in mg, needed to raise the pH of one '
                      'kg of soil by one pH unit'],
         'domain_of': ['Biosample'],
         'examples': [{'value': '1575 mg/kg'}],
         'see_also': ['https://www.ornl.gov/content/bio-scales-0']} })
    id: str = Field(default=..., description="""An NMDC assigned unique identifier for a biosample submitted to NMDC.""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}},
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:bsm-{id_shoulder}-{id_blade}$'}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""Unique identifier for a biosample submitted to additional resources. Matches the entity that has been submitted to NMDC""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/Biosample","nmdc:Biosample"] = Field(default="nmdc:Biosample", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('associated_studies')
    def pattern_associated_studies(cls, v):
        pattern=re.compile(r"^(nmdc):sty-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid associated_studies format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid associated_studies format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('collected_from')
    def pattern_collected_from(cls, v):
        pattern=re.compile(r"^(nmdc):frsite-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid collected_from format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid collected_from format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('img_identifiers')
    def pattern_img_identifiers(cls, v):
        pattern=re.compile(r"^img\.taxon:[a-zA-Z0-9_][a-zA-Z0-9_\/\.]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid img_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid img_identifiers format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('neon_biosample_identifiers')
    def pattern_neon_biosample_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid neon_biosample_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid neon_biosample_identifiers format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('gold_biosample_identifiers')
    def pattern_gold_biosample_identifiers(cls, v):
        pattern=re.compile(r"^gold:Gb[0-9]+$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid gold_biosample_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid gold_biosample_identifiers format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('insdc_biosample_identifiers')
    def pattern_insdc_biosample_identifiers(cls, v):
        pattern=re.compile(r"^biosample:SAM[NED]([A-Z])?[0-9]+$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid insdc_biosample_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid insdc_biosample_identifiers format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('emsl_biosample_identifiers')
    def pattern_emsl_biosample_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid emsl_biosample_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid emsl_biosample_identifiers format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('igsn_biosample_identifiers')
    def pattern_igsn_biosample_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid igsn_biosample_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid igsn_biosample_identifiers format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('fire')
    def pattern_fire(cls, v):
        pattern=re.compile(r"^[12]\d{3}(?:(?:-(?:0[1-9]|1[0-2]))(?:-(?:0[1-9]|[12]\d|3[01]))?)?(\s+to\s+[12]\d{3}(?:(?:-(?:0[1-9]|1[0-2]))(?:-(?:0[1-9]|[12]\d|3[01]))?)?)?$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid fire format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid fire format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('pres_animal_insect')
    def pattern_pres_animal_insect(cls, v):
        pattern=re.compile(r"^(cat|dog|rodent|snake|other);\d+$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid pres_animal_insect format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid pres_animal_insect format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('dna_cont_well')
    def pattern_dna_cont_well(cls, v):
        pattern=re.compile(r"^(?!A1$|A12$|H1$|H12$)(([A-H][1-9])|([A-H]1[0-2]))$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid dna_cont_well format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid dna_cont_well format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('rna_cont_well')
    def pattern_rna_cont_well(cls, v):
        pattern=re.compile(r"^(?!A1$|A12$|H1$|H12$)(([A-H][1-9])|([A-H]1[0-2]))$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid rna_cont_well format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid rna_cont_well format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('infiltrations')
    def pattern_infiltrations(cls, v):
        pattern=re.compile(r"^(?:[0-9]|[1-9][0-9]|9[0-9]|0[0-9]|0[0-5][0-9]):[0-5][0-9]:[0-5][0-9]$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid infiltrations format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid infiltrations format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^(nmdc):bsm-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class ProcessedSample(Sample):
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:ProcessedSample',
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'slot_usage': {'id': {'name': 'id',
                               'pattern': '^(nmdc):procsm-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                               'required': True,
                               'structured_pattern': {'interpolated': True,
                                                      'syntax': '{id_nmdc_prefix}:procsm-{id_shoulder}-{id_blade}$'}}},
         'title': 'Processed Sample'})

    biomaterial_purity: Optional[QuantityValue] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'biomaterial_purity', 'domain_of': ['ProcessedSample']} })
    dna_absorb1: Optional[float] = Field(default=None, title="DNA absorbance 260/280", description="""260/280 measurement of DNA sample purity""", json_schema_extra = { "linkml_meta": {'alias': 'dna_absorb1',
         'comments': ['Recommended value is between 1 and 3.'],
         'domain_of': ['Biosample', 'ProcessedSample'],
         'examples': [{'value': '2.02'}],
         'is_a': 'biomaterial_purity',
         'rank': 7,
         'recommended': True,
         'slot_group': 'JGI-Metagenomics'} })
    dna_concentration: Optional[float] = Field(default=None, title="DNA concentration in ng/ul", ge=0, le=2000, json_schema_extra = { "linkml_meta": {'alias': 'dna_concentration',
         'comments': ['Units must be in ng/uL. Enter the numerical part only. Must be '
                      'calculated using a fluorometric method. Acceptable values are '
                      '0-2000.'],
         'domain_of': ['Biosample', 'ProcessedSample'],
         'examples': [{'value': '100'}],
         'rank': 5,
         'recommended': True,
         'see_also': ['nmdc:nucleic_acid_concentration'],
         'slot_group': 'JGI-Metagenomics'} })
    external_database_identifiers: Optional[list[str]] = Field(default=None, description="""Link to corresponding identifier in external database""", json_schema_extra = { "linkml_meta": {'abstract': True,
         'alias': 'external_database_identifiers',
         'close_mappings': ['skos:closeMatch'],
         'comments': ['The value of this field is always a registered CURIE'],
         'domain_of': ['ProcessedSample'],
         'is_a': 'alternative_identifiers'} })
    sampled_portion: Optional[list[SamplePortionEnum]] = Field(default=None, description="""The portion of the sample that is taken for downstream activity.""", json_schema_extra = { "linkml_meta": {'alias': 'sampled_portion',
         'domain_of': ['SubSamplingProcess', 'ProcessedSample']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}},
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:procsm-{id_shoulder}-{id_blade}$'}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/ProcessedSample","nmdc:ProcessedSample"] = Field(default="nmdc:ProcessedSample", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('external_database_identifiers')
    def pattern_external_database_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid external_database_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid external_database_identifiers format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^(nmdc):procsm-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class Site(MaterialEntity):
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'abstract': True,
         'class_uri': 'nmdc:Site',
         'comments': ['BCO sample collection site ?'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'title': 'Site'})

    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/Site","nmdc:Site"] = Field(default="nmdc:Site", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class FieldResearchSite(Site):
    """
    A site, outside of a laboratory, from which biosamples may be collected.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'aliases': ['research plot'],
         'class_uri': 'nmdc:FieldResearchSite',
         'comments': ["Provides grouping of biosamples at level that's more specific "
                      'than belonging to the same study',
                      'be very clear that this could be part of a larger site',
                      'In Bioscales, one might say that rhizosphere soil and bulk soil '
                      'from around the same plant were obtained at the same site. In '
                      'this case, the site would correspond to one tree.',
                      "might correspond to GOLD's identifier on a page like "
                      'https://gold.jgi.doe.gov/biosample?id=Gb0305833'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'slot_usage': {'id': {'name': 'id',
                               'pattern': '^(nmdc):frsite-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                               'required': True,
                               'structured_pattern': {'interpolated': True,
                                                      'syntax': '{id_nmdc_prefix}:frsite-{id_shoulder}-{id_blade}$'}},
                        'part_of': {'name': 'part_of',
                                    'pattern': '^(nmdc):frsite-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                    'range': 'FieldResearchSite',
                                    'structured_pattern': {'interpolated': True,
                                                           'syntax': '{id_nmdc_prefix}:frsite-{id_shoulder}-{id_blade}$'}}},
         'title': 'Field Research Site'})

    cur_vegetation: Optional[TextValue] = Field(default=None, title="current vegetation", description="""Vegetation classification from one or more standard classification systems, or agricultural crop""", json_schema_extra = { "linkml_meta": {'alias': 'cur_vegetation',
         'aliases': ['current vegetation'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'current vegetation type'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['FieldResearchSite', 'Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000312',
         'string_serialization': '{text}'} })
    elev: Optional[float] = Field(default=None, title="elevation", description="""Elevation of the sampling site is its height above a fixed reference point, most commonly the mean sea level. Elevation is mainly used when referring to points on the earth's surface, while altitude is used for points above the surface, such as an aircraft in flight or a spacecraft in orbit.""", json_schema_extra = { "linkml_meta": {'alias': 'elev',
         'aliases': ['elevation'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'measurement value'}},
         'domain_of': ['FieldResearchSite', 'Biosample'],
         'examples': [{'value': '100 meter'}],
         'is_a': 'environment field',
         'slot_uri': 'MIXS:0000093'} })
    geo_loc_name: Optional[TextValue] = Field(default=None, title="geographic location (country and/or sea,region)", description="""The geographical origin of the sample as defined by the country or sea name followed by specific region name. Country or sea names should be chosen from the INSDC country list (http://insdc.org/country.html), or the GAZ ontology (http://purl.bioontology.org/ontology/GAZ)""", json_schema_extra = { "linkml_meta": {'alias': 'geo_loc_name',
         'aliases': ['geographic location (country and/or sea,region)'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'country or sea name (INSDC or '
                                                     'GAZ): region(GAZ), specific '
                                                     'location name'}},
         'domain_of': ['FieldResearchSite', 'Biosample'],
         'examples': [{'value': 'USA: Maryland, Bethesda'}],
         'is_a': 'environment field',
         'slot_uri': 'MIXS:0000010',
         'string_serialization': '{term}: {term}, {text}'} })
    habitat: Optional[str] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'habitat', 'domain_of': ['FieldResearchSite', 'Biosample']} })
    lat_lon: Optional[GeolocationValue] = Field(default=None, title="geographic location (latitude and longitude)", description="""The geographical origin of the sample as defined by latitude and longitude. The values should be reported in decimal degrees and in WGS84 system""", json_schema_extra = { "linkml_meta": {'alias': 'lat_lon',
         'aliases': ['geographic location (latitude and longitude)'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'decimal degrees,  limit to 8 '
                                                     'decimal points'}},
         'domain_of': ['FieldResearchSite', 'Biosample'],
         'examples': [{'value': '50.586825 6.408977'}],
         'is_a': 'environment field',
         'slot_uri': 'MIXS:0000009',
         'string_serialization': '{float} {float}'} })
    local_class: Optional[TextValue] = Field(default=None, title="soil_taxonomic/local classification", description="""Soil classification based on local soil classification system""", json_schema_extra = { "linkml_meta": {'alias': 'local_class',
         'aliases': ['soil_taxonomic/local classification'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'local classification name'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['FieldResearchSite', 'Biosample'],
         'examples': [{'value': ''}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000330',
         'string_serialization': '{text}'} })
    part_of: Optional[list[str]] = Field(default=None, description="""Links a resource to another resource that either logically or physically includes it.""", json_schema_extra = { "linkml_meta": {'alias': 'part_of',
         'aliases': ['is part of'],
         'domain_of': ['FieldResearchSite', 'Study'],
         'slot_uri': 'dcterms:isPartOf',
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:frsite-{id_shoulder}-{id_blade}$'}} })
    soil_type: Optional[TextValue] = Field(default=None, title="soil type", description="""Description of the soil type or classification. This field accepts terms under soil (http://purl.obolibrary.org/obo/ENVO_00001998).  Multiple terms can be separated by pipes.""", json_schema_extra = { "linkml_meta": {'alias': 'soil_type',
         'aliases': ['soil type'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'ENVO_00001998'},
                         'occurrence': {'tag': 'occurrence', 'value': '1'}},
         'domain_of': ['FieldResearchSite', 'Biosample'],
         'examples': [{'value': 'plinthosol [ENVO:00002250]'}],
         'is_a': 'core field',
         'slot_uri': 'MIXS:0000332',
         'string_serialization': '{termLabel} {[termID]}'} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}},
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:frsite-{id_shoulder}-{id_blade}$'}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/FieldResearchSite","nmdc:FieldResearchSite"] = Field(default="nmdc:FieldResearchSite", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('part_of')
    def pattern_part_of(cls, v):
        pattern=re.compile(r"^(nmdc):frsite-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid part_of format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid part_of format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^(nmdc):frsite-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class Instrument(MaterialEntity):
    """
    A material entity that is designed to perform a function in a scientific investigation, but is not a reagent.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'aliases': ['device'],
         'class_uri': 'nmdc:Instrument',
         'exact_mappings': ['OBI:0000968'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'slot_usage': {'id': {'name': 'id',
                               'pattern': '^(nmdc):inst-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                               'structured_pattern': {'interpolated': True,
                                                      'syntax': '{id_nmdc_prefix}:inst-{id_shoulder}-{id_blade}$'}}}})

    vendor: Optional[InstrumentVendorEnum] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'vendor', 'domain_of': ['Instrument']} })
    model: Optional[InstrumentModelEnum] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'model', 'domain_of': ['Instrument']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}},
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:inst-{id_shoulder}-{id_blade}$'}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/Instrument","nmdc:Instrument"] = Field(default="nmdc:Instrument", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^(nmdc):inst-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class PlannedProcess(NamedThing):
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'abstract': True,
         'class_uri': 'OBI:0000011',
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'title': 'Planned Process'})

    has_input: Optional[list[str]] = Field(default=None, description="""An input to a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_input', 'aliases': ['input'], 'domain_of': ['PlannedProcess']} })
    has_output: Optional[list[str]] = Field(default=None, description="""An output from a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_output', 'aliases': ['output'], 'domain_of': ['PlannedProcess']} })
    processing_institution: Optional[ProcessingInstitutionEnum] = Field(default=None, description="""The organization that processed the sample.""", json_schema_extra = { "linkml_meta": {'alias': 'processing_institution', 'domain_of': ['PlannedProcess']} })
    protocol_link: Optional[Protocol] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'protocol_link',
         'domain_of': ['Configuration', 'PlannedProcess', 'Study']} })
    start_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was started""", json_schema_extra = { "linkml_meta": {'alias': 'start_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    end_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was ended""", json_schema_extra = { "linkml_meta": {'alias': 'end_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    qc_status: Optional[StatusEnum] = Field(default=None, description="""Stores information about the result of a process (ie the process of sequencing a library may have for qc_status of 'fail' if not enough data was generated)""", json_schema_extra = { "linkml_meta": {'alias': 'qc_status', 'domain_of': ['PlannedProcess']} })
    qc_comment: Optional[str] = Field(default=None, description="""Slot to store additional comments about laboratory or workflow output. For workflow output it may describe the particular workflow stage that failed. (ie Failed at call-stage due to a malformed fastq file).""", json_schema_extra = { "linkml_meta": {'alias': 'qc_comment', 'domain_of': ['PlannedProcess']} })
    has_failure_categorization: Optional[list[FailureCategorization]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'has_failure_categorization', 'domain_of': ['PlannedProcess']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/PlannedProcess","http://purl.obolibrary.org/obo/OBI_0000011","nmdc:PlannedProcess","OBI:0000011"] = Field(default="OBI:0000011", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class CollectingBiosamplesFromSite(PlannedProcess):
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:CollectingBiosamplesFromSite',
         'close_mappings': ['OBI:0000744'],
         'comments': ['this illustrates implementing a Biosample relation with a '
                      'process class'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'slot_usage': {'has_input': {'name': 'has_input',
                                      'pattern': '^(nmdc):(frsite|site)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                      'range': 'Site',
                                      'required': True,
                                      'structured_pattern': {'interpolated': True,
                                                             'syntax': '{id_nmdc_prefix}:(frsite|site)-{id_shoulder}-{id_blade}$'}},
                        'has_output': {'name': 'has_output',
                                       'pattern': '^(nmdc):bsm-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                       'range': 'Biosample',
                                       'required': True,
                                       'structured_pattern': {'interpolated': True,
                                                              'syntax': '{id_nmdc_prefix}:bsm-{id_shoulder}-{id_blade}$'}},
                        'id': {'name': 'id',
                               'pattern': '^(nmdc):clsite-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                               'required': True,
                               'structured_pattern': {'interpolated': True,
                                                      'syntax': '{id_nmdc_prefix}:clsite-{id_shoulder}-{id_blade}$'}}},
         'title': 'Collecting Biosamples From Site'})

    has_input: list[str] = Field(default=..., description="""An input to a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_input',
         'aliases': ['input'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(frsite|site)-{id_shoulder}-{id_blade}$'}} })
    has_output: list[str] = Field(default=..., description="""An output from a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_output',
         'aliases': ['output'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:bsm-{id_shoulder}-{id_blade}$'}} })
    processing_institution: Optional[ProcessingInstitutionEnum] = Field(default=None, description="""The organization that processed the sample.""", json_schema_extra = { "linkml_meta": {'alias': 'processing_institution', 'domain_of': ['PlannedProcess']} })
    protocol_link: Optional[Protocol] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'protocol_link',
         'domain_of': ['Configuration', 'PlannedProcess', 'Study']} })
    start_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was started""", json_schema_extra = { "linkml_meta": {'alias': 'start_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    end_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was ended""", json_schema_extra = { "linkml_meta": {'alias': 'end_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    qc_status: Optional[StatusEnum] = Field(default=None, description="""Stores information about the result of a process (ie the process of sequencing a library may have for qc_status of 'fail' if not enough data was generated)""", json_schema_extra = { "linkml_meta": {'alias': 'qc_status', 'domain_of': ['PlannedProcess']} })
    qc_comment: Optional[str] = Field(default=None, description="""Slot to store additional comments about laboratory or workflow output. For workflow output it may describe the particular workflow stage that failed. (ie Failed at call-stage due to a malformed fastq file).""", json_schema_extra = { "linkml_meta": {'alias': 'qc_comment', 'domain_of': ['PlannedProcess']} })
    has_failure_categorization: Optional[list[FailureCategorization]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'has_failure_categorization', 'domain_of': ['PlannedProcess']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}},
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:clsite-{id_shoulder}-{id_blade}$'}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/CollectingBiosamplesFromSite","nmdc:CollectingBiosamplesFromSite"] = Field(default="nmdc:CollectingBiosamplesFromSite", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('has_input')
    def pattern_has_input(cls, v):
        pattern=re.compile(r"^(nmdc):(frsite|site)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_input format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_input format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_output')
    def pattern_has_output(cls, v):
        pattern=re.compile(r"^(nmdc):bsm-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_output format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_output format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^(nmdc):clsite-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class StorageProcess(PlannedProcess):
    """
    A planned process with the objective to preserve and protect material entities by placing them in an identified  location which may have a controlled environment.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:StorageProcess',
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'related_mappings': ['OBI:0302893'],
         'slot_usage': {'has_input': {'name': 'has_input',
                                      'pattern': '^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                      'range': 'Sample',
                                      'structured_pattern': {'interpolated': True,
                                                             'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}},
                        'has_output': {'name': 'has_output',
                                       'pattern': '^(nmdc):procsm-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                       'range': 'ProcessedSample',
                                       'structured_pattern': {'interpolated': True,
                                                              'syntax': '{id_nmdc_prefix}:procsm-{id_shoulder}-{id_blade}$'}},
                        'id': {'name': 'id',
                               'pattern': '^(nmdc):storpr-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                               'required': True,
                               'structured_pattern': {'interpolated': True,
                                                      'syntax': '{id_nmdc_prefix}:storpr-{id_shoulder}-{id_blade}$'}},
                        'substances_used': {'description': 'The substance(s) that a '
                                                           'processed sample is stored '
                                                           'in.',
                                            'name': 'substances_used'}}})

    substances_used: Optional[list[PortionOfSubstance]] = Field(default=None, description="""The substance(s) that a processed sample is stored in.""", json_schema_extra = { "linkml_meta": {'alias': 'substances_used',
         'domain_of': ['Extraction',
                       'StorageProcess',
                       'DissolvingProcess',
                       'ChemicalConversionProcess',
                       'MobilePhaseSegment']} })
    contained_in: Optional[ContainerCategoryEnum] = Field(default=None, description="""A type of container.""", json_schema_extra = { "linkml_meta": {'alias': 'contained_in',
         'domain_of': ['SubSamplingProcess', 'StorageProcess'],
         'examples': [{'value': 'test tube'},
                      {'value': 'falcon tube'},
                      {'value': 'whirlpak'}]} })
    temperature: Optional[QuantityValue] = Field(default=None, description="""The value of a temperature measurement or temperature used in a process.""", json_schema_extra = { "linkml_meta": {'alias': 'temperature',
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-8683-0050'],
         'domain_of': ['ChromatographyConfiguration',
                       'Extraction',
                       'SubSamplingProcess',
                       'StorageProcess',
                       'ChromatographicSeparationProcess',
                       'DissolvingProcess',
                       'ChemicalConversionProcess'],
         'notes': ['Not to be confused with the MIXS:0000113']} })
    has_input: Optional[list[str]] = Field(default=None, description="""An input to a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_input',
         'aliases': ['input'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}} })
    has_output: Optional[list[str]] = Field(default=None, description="""An output from a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_output',
         'aliases': ['output'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:procsm-{id_shoulder}-{id_blade}$'}} })
    processing_institution: Optional[ProcessingInstitutionEnum] = Field(default=None, description="""The organization that processed the sample.""", json_schema_extra = { "linkml_meta": {'alias': 'processing_institution', 'domain_of': ['PlannedProcess']} })
    protocol_link: Optional[Protocol] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'protocol_link',
         'domain_of': ['Configuration', 'PlannedProcess', 'Study']} })
    start_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was started""", json_schema_extra = { "linkml_meta": {'alias': 'start_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    end_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was ended""", json_schema_extra = { "linkml_meta": {'alias': 'end_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    qc_status: Optional[StatusEnum] = Field(default=None, description="""Stores information about the result of a process (ie the process of sequencing a library may have for qc_status of 'fail' if not enough data was generated)""", json_schema_extra = { "linkml_meta": {'alias': 'qc_status', 'domain_of': ['PlannedProcess']} })
    qc_comment: Optional[str] = Field(default=None, description="""Slot to store additional comments about laboratory or workflow output. For workflow output it may describe the particular workflow stage that failed. (ie Failed at call-stage due to a malformed fastq file).""", json_schema_extra = { "linkml_meta": {'alias': 'qc_comment', 'domain_of': ['PlannedProcess']} })
    has_failure_categorization: Optional[list[FailureCategorization]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'has_failure_categorization', 'domain_of': ['PlannedProcess']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}},
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:storpr-{id_shoulder}-{id_blade}$'}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/StorageProcess","nmdc:StorageProcess"] = Field(default="nmdc:StorageProcess", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('has_input')
    def pattern_has_input(cls, v):
        pattern=re.compile(r"^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_input format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_input format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_output')
    def pattern_has_output(cls, v):
        pattern=re.compile(r"^(nmdc):procsm-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_output format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_output format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^(nmdc):storpr-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class MaterialProcessing(PlannedProcess):
    """
    A process that takes one or more samples as inputs and generates one or more samples as outputs.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'abstract': True,
         'broad_mappings': ['OBI:0000094'],
         'class_uri': 'nmdc:MaterialProcessing',
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'notes': ['This class is a replacement for BiosampleProcessing.'],
         'slot_usage': {'has_input': {'name': 'has_input',
                                      'pattern': '^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                      'range': 'Sample',
                                      'structured_pattern': {'interpolated': True,
                                                             'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}},
                        'has_output': {'name': 'has_output',
                                       'pattern': '^(nmdc):(procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                       'range': 'ProcessedSample',
                                       'structured_pattern': {'interpolated': True,
                                                              'syntax': '{id_nmdc_prefix}:(procsm)-{id_shoulder}-{id_blade}$'}}}})

    instrument_used: Optional[list[str]] = Field(default=None, description="""What instrument was used during DataGeneration or MaterialProcessing.""", json_schema_extra = { "linkml_meta": {'alias': 'instrument_used',
         'domain_of': ['MaterialProcessing', 'DataGeneration'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:inst-{id_shoulder}-{id_blade}$'}} })
    has_input: Optional[list[str]] = Field(default=None, description="""An input to a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_input',
         'aliases': ['input'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}} })
    has_output: Optional[list[str]] = Field(default=None, description="""An output from a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_output',
         'aliases': ['output'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(procsm)-{id_shoulder}-{id_blade}$'}} })
    processing_institution: Optional[ProcessingInstitutionEnum] = Field(default=None, description="""The organization that processed the sample.""", json_schema_extra = { "linkml_meta": {'alias': 'processing_institution', 'domain_of': ['PlannedProcess']} })
    protocol_link: Optional[Protocol] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'protocol_link',
         'domain_of': ['Configuration', 'PlannedProcess', 'Study']} })
    start_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was started""", json_schema_extra = { "linkml_meta": {'alias': 'start_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    end_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was ended""", json_schema_extra = { "linkml_meta": {'alias': 'end_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    qc_status: Optional[StatusEnum] = Field(default=None, description="""Stores information about the result of a process (ie the process of sequencing a library may have for qc_status of 'fail' if not enough data was generated)""", json_schema_extra = { "linkml_meta": {'alias': 'qc_status', 'domain_of': ['PlannedProcess']} })
    qc_comment: Optional[str] = Field(default=None, description="""Slot to store additional comments about laboratory or workflow output. For workflow output it may describe the particular workflow stage that failed. (ie Failed at call-stage due to a malformed fastq file).""", json_schema_extra = { "linkml_meta": {'alias': 'qc_comment', 'domain_of': ['PlannedProcess']} })
    has_failure_categorization: Optional[list[FailureCategorization]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'has_failure_categorization', 'domain_of': ['PlannedProcess']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/MaterialProcessing","nmdc:MaterialProcessing"] = Field(default="nmdc:MaterialProcessing", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('instrument_used')
    def pattern_instrument_used(cls, v):
        pattern=re.compile(r"^(nmdc):inst-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid instrument_used format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid instrument_used format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_input')
    def pattern_has_input(cls, v):
        pattern=re.compile(r"^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_input format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_input format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_output')
    def pattern_has_output(cls, v):
        pattern=re.compile(r"^(nmdc):(procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_output format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_output format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class Pooling(MaterialProcessing):
    """
    physical combination of several instances of like material.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:Pooling',
         'exact_mappings': ['OBI:0600016'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'slot_usage': {'has_input': {'minimum_cardinality': 2,
                                      'name': 'has_input',
                                      'pattern': '^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                      'required': True,
                                      'structured_pattern': {'interpolated': True,
                                                             'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}},
                        'has_output': {'maximum_cardinality': 1,
                                       'minimum_cardinality': 1,
                                       'name': 'has_output',
                                       'pattern': '^(nmdc):procsm-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                       'required': True,
                                       'structured_pattern': {'interpolated': True,
                                                              'syntax': '{id_nmdc_prefix}:procsm-{id_shoulder}-{id_blade}$'}},
                        'id': {'name': 'id',
                               'pattern': '^(nmdc):poolp-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                               'required': True,
                               'structured_pattern': {'interpolated': True,
                                                      'syntax': '{id_nmdc_prefix}:poolp-{id_shoulder}-{id_blade}$'}}}})

    instrument_used: Optional[list[str]] = Field(default=None, description="""What instrument was used during DataGeneration or MaterialProcessing.""", json_schema_extra = { "linkml_meta": {'alias': 'instrument_used',
         'domain_of': ['MaterialProcessing', 'DataGeneration'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:inst-{id_shoulder}-{id_blade}$'}} })
    has_input: list[str] = Field(default=..., description="""An input to a process.""", min_length=2, json_schema_extra = { "linkml_meta": {'alias': 'has_input',
         'aliases': ['input'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}} })
    has_output: list[str] = Field(default=..., description="""An output from a process.""", min_length=1, max_length=1, json_schema_extra = { "linkml_meta": {'alias': 'has_output',
         'aliases': ['output'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:procsm-{id_shoulder}-{id_blade}$'}} })
    processing_institution: Optional[ProcessingInstitutionEnum] = Field(default=None, description="""The organization that processed the sample.""", json_schema_extra = { "linkml_meta": {'alias': 'processing_institution', 'domain_of': ['PlannedProcess']} })
    protocol_link: Optional[Protocol] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'protocol_link',
         'domain_of': ['Configuration', 'PlannedProcess', 'Study']} })
    start_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was started""", json_schema_extra = { "linkml_meta": {'alias': 'start_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    end_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was ended""", json_schema_extra = { "linkml_meta": {'alias': 'end_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    qc_status: Optional[StatusEnum] = Field(default=None, description="""Stores information about the result of a process (ie the process of sequencing a library may have for qc_status of 'fail' if not enough data was generated)""", json_schema_extra = { "linkml_meta": {'alias': 'qc_status', 'domain_of': ['PlannedProcess']} })
    qc_comment: Optional[str] = Field(default=None, description="""Slot to store additional comments about laboratory or workflow output. For workflow output it may describe the particular workflow stage that failed. (ie Failed at call-stage due to a malformed fastq file).""", json_schema_extra = { "linkml_meta": {'alias': 'qc_comment', 'domain_of': ['PlannedProcess']} })
    has_failure_categorization: Optional[list[FailureCategorization]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'has_failure_categorization', 'domain_of': ['PlannedProcess']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}},
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:poolp-{id_shoulder}-{id_blade}$'}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/Pooling","nmdc:Pooling"] = Field(default="nmdc:Pooling", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('instrument_used')
    def pattern_instrument_used(cls, v):
        pattern=re.compile(r"^(nmdc):inst-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid instrument_used format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid instrument_used format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_input')
    def pattern_has_input(cls, v):
        pattern=re.compile(r"^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_input format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_input format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_output')
    def pattern_has_output(cls, v):
        pattern=re.compile(r"^(nmdc):procsm-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_output format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_output format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^(nmdc):poolp-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class Extraction(MaterialProcessing):
    """
    A material separation in which a desired component of an input material is separated from the remainder.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:Extraction',
         'exact_mappings': ['OBI:0302884'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'slot_usage': {'has_input': {'name': 'has_input',
                                      'pattern': '^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                      'range': 'Sample',
                                      'required': True,
                                      'structured_pattern': {'interpolated': True,
                                                             'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}},
                        'has_output': {'name': 'has_output',
                                       'pattern': '^(nmdc):(procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                       'required': True,
                                       'structured_pattern': {'interpolated': True,
                                                              'syntax': '{id_nmdc_prefix}:(procsm)-{id_shoulder}-{id_blade}$'}},
                        'id': {'name': 'id',
                               'pattern': '^(nmdc):extrp-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                               'required': True,
                               'structured_pattern': {'interpolated': True,
                                                      'syntax': '{id_nmdc_prefix}:extrp-{id_shoulder}-{id_blade}$'}},
                        'volume': {'description': 'The volume of the solvent/solute '
                                                  'being used, not the input.',
                                   'name': 'volume'}}})

    substances_used: Optional[list[PortionOfSubstance]] = Field(default=None, description="""The substances that are combined to enable a ChemicalConversionProcess.""", json_schema_extra = { "linkml_meta": {'alias': 'substances_used',
         'domain_of': ['Extraction',
                       'StorageProcess',
                       'DissolvingProcess',
                       'ChemicalConversionProcess',
                       'MobilePhaseSegment']} })
    extraction_targets: Optional[list[ExtractionTargetEnum]] = Field(default=None, description="""Provides the target biomolecule that has been separated from a sample during an extraction process.""", json_schema_extra = { "linkml_meta": {'alias': 'extraction_targets',
         'domain_of': ['Extraction'],
         'narrow_mappings': ['NCIT:C177560', 'MIXS:0000037'],
         'notes': ['todos, remove nucl_acid_ext from OmicsProcessing (DataGeneration)'],
         'rank': 1000} })
    input_mass: Optional[QuantityValue] = Field(default=None, title="sample mass used", description="""Total mass of sample used in activity.""", json_schema_extra = { "linkml_meta": {'alias': 'input_mass',
         'aliases': ['sample mass', 'sample weight'],
         'domain_of': ['Extraction'],
         'exact_mappings': ['MS:1000004'],
         'narrow_mappings': ['MIXS:0000111']} })
    volume: Optional[QuantityValue] = Field(default=None, description="""The volume of the solvent/solute being used, not the input.""", json_schema_extra = { "linkml_meta": {'alias': 'volume',
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-8683-0050'],
         'domain_of': ['Extraction',
                       'SubSamplingProcess',
                       'FiltrationProcess',
                       'MobilePhaseSegment',
                       'PortionOfSubstance']} })
    temperature: Optional[QuantityValue] = Field(default=None, description="""The value of a temperature measurement or temperature used in a process.""", json_schema_extra = { "linkml_meta": {'alias': 'temperature',
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-8683-0050'],
         'domain_of': ['ChromatographyConfiguration',
                       'Extraction',
                       'SubSamplingProcess',
                       'StorageProcess',
                       'ChromatographicSeparationProcess',
                       'DissolvingProcess',
                       'ChemicalConversionProcess'],
         'notes': ['Not to be confused with the MIXS:0000113']} })
    instrument_used: Optional[list[str]] = Field(default=None, description="""What instrument was used during DataGeneration or MaterialProcessing.""", json_schema_extra = { "linkml_meta": {'alias': 'instrument_used',
         'domain_of': ['MaterialProcessing', 'DataGeneration'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:inst-{id_shoulder}-{id_blade}$'}} })
    has_input: list[str] = Field(default=..., description="""An input to a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_input',
         'aliases': ['input'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}} })
    has_output: list[str] = Field(default=..., description="""An output from a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_output',
         'aliases': ['output'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(procsm)-{id_shoulder}-{id_blade}$'}} })
    processing_institution: Optional[ProcessingInstitutionEnum] = Field(default=None, description="""The organization that processed the sample.""", json_schema_extra = { "linkml_meta": {'alias': 'processing_institution', 'domain_of': ['PlannedProcess']} })
    protocol_link: Optional[Protocol] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'protocol_link',
         'domain_of': ['Configuration', 'PlannedProcess', 'Study']} })
    start_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was started""", json_schema_extra = { "linkml_meta": {'alias': 'start_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    end_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was ended""", json_schema_extra = { "linkml_meta": {'alias': 'end_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    qc_status: Optional[StatusEnum] = Field(default=None, description="""Stores information about the result of a process (ie the process of sequencing a library may have for qc_status of 'fail' if not enough data was generated)""", json_schema_extra = { "linkml_meta": {'alias': 'qc_status', 'domain_of': ['PlannedProcess']} })
    qc_comment: Optional[str] = Field(default=None, description="""Slot to store additional comments about laboratory or workflow output. For workflow output it may describe the particular workflow stage that failed. (ie Failed at call-stage due to a malformed fastq file).""", json_schema_extra = { "linkml_meta": {'alias': 'qc_comment', 'domain_of': ['PlannedProcess']} })
    has_failure_categorization: Optional[list[FailureCategorization]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'has_failure_categorization', 'domain_of': ['PlannedProcess']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}},
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:extrp-{id_shoulder}-{id_blade}$'}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/Extraction","nmdc:Extraction"] = Field(default="nmdc:Extraction", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('instrument_used')
    def pattern_instrument_used(cls, v):
        pattern=re.compile(r"^(nmdc):inst-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid instrument_used format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid instrument_used format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_input')
    def pattern_has_input(cls, v):
        pattern=re.compile(r"^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_input format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_input format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_output')
    def pattern_has_output(cls, v):
        pattern=re.compile(r"^(nmdc):(procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_output format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_output format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^(nmdc):extrp-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class LibraryPreparation(MaterialProcessing):
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'aliases': ['LibraryConstruction'],
         'class_uri': 'nmdc:LibraryPreparation',
         'close_mappings': ['OBI:0000711'],
         'comments': ['OBI:0000711 specifies a DNA input (but not ONLY a DNA input)'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'slot_usage': {'has_input': {'name': 'has_input',
                                      'pattern': '^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                      'required': True,
                                      'structured_pattern': {'interpolated': True,
                                                             'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}},
                        'has_output': {'name': 'has_output',
                                       'pattern': '^(nmdc):(procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                       'required': True,
                                       'structured_pattern': {'interpolated': True,
                                                              'syntax': '{id_nmdc_prefix}:(procsm)-{id_shoulder}-{id_blade}$'}},
                        'id': {'name': 'id',
                               'pattern': '^(nmdc):libprp-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                               'required': True,
                               'structured_pattern': {'interpolated': True,
                                                      'syntax': '{id_nmdc_prefix}:libprp-{id_shoulder}-{id_blade}$'}},
                        'pcr_cond': {'description': 'Description of reaction '
                                                    'conditions and components of '
                                                    'polymerase chain reaction '
                                                    'performed during library '
                                                    'preparation',
                                     'name': 'pcr_cond'}}})

    is_stranded: Optional[bool] = Field(default=None, description="""Is the (RNA) library stranded or non-stranded (unstranded).""", json_schema_extra = { "linkml_meta": {'alias': 'is_stranded',
         'comments': ['A value of true means the library is stranded, flase means '
                      'non-stranded.'],
         'domain_of': ['LibraryPreparation']} })
    library_preparation_kit: Optional[str] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'library_preparation_kit',
         'domain_of': ['LibraryPreparation'],
         'exact_mappings': ['GENEPIO:0001450']} })
    library_type: Optional[LibraryTypeEnum] = Field(default=None, title="library type", json_schema_extra = { "linkml_meta": {'alias': 'library_type',
         'domain_of': ['LibraryPreparation'],
         'examples': [{'value': 'DNA'}]} })
    nucl_acid_amp: Optional[TextValue] = Field(default=None, title="nucleic acid amplification", description="""A link to a literature reference, electronic resource or a standard operating procedure (SOP), that describes the enzymatic amplification (PCR, TMA, NASBA) of specific nucleic acids""", json_schema_extra = { "linkml_meta": {'alias': 'nucl_acid_amp',
         'aliases': ['nucleic acid amplification'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'PMID, DOI or URL'}},
         'domain_of': ['LibraryPreparation'],
         'examples': [{'value': 'https://phylogenomics.me/protocols/16s-pcr-protocol/'}],
         'is_a': 'sequencing field',
         'slot_uri': 'MIXS:0000038',
         'string_serialization': '{PMID}|{DOI}|{URL}'} })
    pcr_cond: Optional[TextValue] = Field(default=None, title="pcr conditions", description="""Description of reaction conditions and components of polymerase chain reaction performed during library preparation""", json_schema_extra = { "linkml_meta": {'alias': 'pcr_cond',
         'aliases': ['pcr conditions'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'initial '
                                                     'denaturation:degrees_minutes;annealing:degrees_minutes;elongation:degrees_minutes;final '
                                                     'elongation:degrees_minutes;total '
                                                     'cycles'}},
         'domain_of': ['LibraryPreparation'],
         'examples': [{'value': 'initial '
                                'denaturation:94_3;annealing:50_1;elongation:72_1.5;final '
                                'elongation:72_10;35'}],
         'is_a': 'sequencing field',
         'slot_uri': 'MIXS:0000049',
         'string_serialization': 'initial '
                                 'denaturation:degrees_minutes;annealing:degrees_minutes;elongation:degrees_minutes;final '
                                 'elongation:degrees_minutes;total cycles'} })
    pcr_cycles: Optional[int] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'pcr_cycles',
         'domain_of': ['LibraryPreparation'],
         'exact_mappings': ['OBI:0002475']} })
    pcr_primers: Optional[TextValue] = Field(default=None, title="pcr primers", description="""PCR primers that were used to amplify the sequence of the targeted gene, locus or subfragment. This field should contain all the primers used for a single PCR reaction if multiple forward or reverse primers are present in a single PCR reaction. The primer sequence should be reported in uppercase letters""", json_schema_extra = { "linkml_meta": {'alias': 'pcr_primers',
         'aliases': ['pcr primers'],
         'annotations': {'expected_value': {'tag': 'expected_value',
                                            'value': 'FWD: forward primer '
                                                     'sequence;REV:reverse primer '
                                                     'sequence'}},
         'domain_of': ['LibraryPreparation'],
         'examples': [{'value': 'FWD:GTGCCAGCMGCCGCGGTAA;REV:GGACTACHVGGGTWTCTAAT'}],
         'is_a': 'sequencing field',
         'slot_uri': 'MIXS:0000046',
         'string_serialization': 'FWD:{dna};REV:{dna}'} })
    stranded_orientation: Optional[StrandedOrientationEnum] = Field(default=None, description="""Lists the strand orientiation for a stranded RNA library preparation.""", json_schema_extra = { "linkml_meta": {'alias': 'stranded_orientation', 'domain_of': ['LibraryPreparation']} })
    instrument_used: Optional[list[str]] = Field(default=None, description="""What instrument was used during DataGeneration or MaterialProcessing.""", json_schema_extra = { "linkml_meta": {'alias': 'instrument_used',
         'domain_of': ['MaterialProcessing', 'DataGeneration'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:inst-{id_shoulder}-{id_blade}$'}} })
    has_input: list[str] = Field(default=..., description="""An input to a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_input',
         'aliases': ['input'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}} })
    has_output: list[str] = Field(default=..., description="""An output from a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_output',
         'aliases': ['output'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(procsm)-{id_shoulder}-{id_blade}$'}} })
    processing_institution: Optional[ProcessingInstitutionEnum] = Field(default=None, description="""The organization that processed the sample.""", json_schema_extra = { "linkml_meta": {'alias': 'processing_institution', 'domain_of': ['PlannedProcess']} })
    protocol_link: Optional[Protocol] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'protocol_link',
         'domain_of': ['Configuration', 'PlannedProcess', 'Study']} })
    start_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was started""", json_schema_extra = { "linkml_meta": {'alias': 'start_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    end_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was ended""", json_schema_extra = { "linkml_meta": {'alias': 'end_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    qc_status: Optional[StatusEnum] = Field(default=None, description="""Stores information about the result of a process (ie the process of sequencing a library may have for qc_status of 'fail' if not enough data was generated)""", json_schema_extra = { "linkml_meta": {'alias': 'qc_status', 'domain_of': ['PlannedProcess']} })
    qc_comment: Optional[str] = Field(default=None, description="""Slot to store additional comments about laboratory or workflow output. For workflow output it may describe the particular workflow stage that failed. (ie Failed at call-stage due to a malformed fastq file).""", json_schema_extra = { "linkml_meta": {'alias': 'qc_comment', 'domain_of': ['PlannedProcess']} })
    has_failure_categorization: Optional[list[FailureCategorization]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'has_failure_categorization', 'domain_of': ['PlannedProcess']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}},
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:libprp-{id_shoulder}-{id_blade}$'}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/LibraryPreparation","nmdc:LibraryPreparation"] = Field(default="nmdc:LibraryPreparation", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('instrument_used')
    def pattern_instrument_used(cls, v):
        pattern=re.compile(r"^(nmdc):inst-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid instrument_used format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid instrument_used format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_input')
    def pattern_has_input(cls, v):
        pattern=re.compile(r"^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_input format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_input format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_output')
    def pattern_has_output(cls, v):
        pattern=re.compile(r"^(nmdc):(procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_output format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_output format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^(nmdc):libprp-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class SubSamplingProcess(MaterialProcessing):
    """
    Separating a sample aliquot from the starting material for downstream activity.

    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:SubSamplingProcess',
         'contributors': ['ORCID:0009-0001-1555-1601',
                          'ORCID:0000-0002-8683-0050',
                          'ORCID:0000-0001-9076-6066',
                          'ORCID:0009-0008-4013-7737'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'notes': ['A subsample may be (a) a portion of the sample obtained by '
                   'selection or division; (b) an individual unit of the lot taken as '
                   'part of the sample; (c) the final unit of multistage sampling. The '
                   "term 'subsample' is used either in the sense of a 'sample of a "
                   "sample' or as a synonym for 'unit'. In practice, the meaning is "
                   'usually apparent from the context or is defined.',
                   'TODO - Montana to visit slot descriptions'],
         'related_mappings': ['OBI:0000744'],
         'slot_usage': {'has_input': {'name': 'has_input',
                                      'pattern': '^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                      'range': 'Sample',
                                      'structured_pattern': {'interpolated': True,
                                                             'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}},
                        'has_output': {'description': 'The subsample.',
                                       'name': 'has_output',
                                       'pattern': '^(nmdc):(procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                       'range': 'ProcessedSample',
                                       'structured_pattern': {'interpolated': True,
                                                              'syntax': '{id_nmdc_prefix}:(procsm)-{id_shoulder}-{id_blade}$'}},
                        'id': {'name': 'id',
                               'pattern': '^(nmdc):subspr-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                               'required': True,
                               'structured_pattern': {'interpolated': True,
                                                      'syntax': '{id_nmdc_prefix}:subspr-{id_shoulder}-{id_blade}$'}},
                        'mass': {'description': 'The output mass of the SubSampling '
                                                'Process.',
                                 'name': 'mass'},
                        'volume': {'description': 'The output volume of the '
                                                  'SubSampling Process.',
                                   'name': 'volume'}}})

    container_size: Optional[QuantityValue] = Field(default=None, description="""The volume of the container an analyte is stored in or an activity takes place in""", json_schema_extra = { "linkml_meta": {'alias': 'container_size',
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-8683-0050'],
         'domain_of': ['SubSamplingProcess', 'FiltrationProcess']} })
    contained_in: Optional[ContainerCategoryEnum] = Field(default=None, description="""A type of container.""", json_schema_extra = { "linkml_meta": {'alias': 'contained_in',
         'domain_of': ['SubSamplingProcess', 'StorageProcess'],
         'examples': [{'value': 'test tube'},
                      {'value': 'falcon tube'},
                      {'value': 'whirlpak'}]} })
    temperature: Optional[QuantityValue] = Field(default=None, description="""The value of a temperature measurement or temperature used in a process.""", json_schema_extra = { "linkml_meta": {'alias': 'temperature',
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-8683-0050'],
         'domain_of': ['ChromatographyConfiguration',
                       'Extraction',
                       'SubSamplingProcess',
                       'StorageProcess',
                       'ChromatographicSeparationProcess',
                       'DissolvingProcess',
                       'ChemicalConversionProcess'],
         'notes': ['Not to be confused with the MIXS:0000113']} })
    volume: Optional[QuantityValue] = Field(default=None, description="""The output volume of the SubSampling Process.""", json_schema_extra = { "linkml_meta": {'alias': 'volume',
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-8683-0050'],
         'domain_of': ['Extraction',
                       'SubSamplingProcess',
                       'FiltrationProcess',
                       'MobilePhaseSegment',
                       'PortionOfSubstance']} })
    mass: Optional[QuantityValue] = Field(default=None, title="mass", description="""The output mass of the SubSampling Process.""", json_schema_extra = { "linkml_meta": {'alias': 'mass',
         'domain_of': ['SubSamplingProcess', 'PortionOfSubstance'],
         'exact_mappings': ['PATO:0000125']} })
    sampled_portion: Optional[list[SamplePortionEnum]] = Field(default=None, description="""The portion of the sample that is taken for downstream activity.""", json_schema_extra = { "linkml_meta": {'alias': 'sampled_portion',
         'domain_of': ['SubSamplingProcess', 'ProcessedSample']} })
    instrument_used: Optional[list[str]] = Field(default=None, description="""What instrument was used during DataGeneration or MaterialProcessing.""", json_schema_extra = { "linkml_meta": {'alias': 'instrument_used',
         'domain_of': ['MaterialProcessing', 'DataGeneration'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:inst-{id_shoulder}-{id_blade}$'}} })
    has_input: Optional[list[str]] = Field(default=None, description="""An input to a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_input',
         'aliases': ['input'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}} })
    has_output: Optional[list[str]] = Field(default=None, description="""The subsample.""", json_schema_extra = { "linkml_meta": {'alias': 'has_output',
         'aliases': ['output'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(procsm)-{id_shoulder}-{id_blade}$'}} })
    processing_institution: Optional[ProcessingInstitutionEnum] = Field(default=None, description="""The organization that processed the sample.""", json_schema_extra = { "linkml_meta": {'alias': 'processing_institution', 'domain_of': ['PlannedProcess']} })
    protocol_link: Optional[Protocol] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'protocol_link',
         'domain_of': ['Configuration', 'PlannedProcess', 'Study']} })
    start_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was started""", json_schema_extra = { "linkml_meta": {'alias': 'start_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    end_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was ended""", json_schema_extra = { "linkml_meta": {'alias': 'end_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    qc_status: Optional[StatusEnum] = Field(default=None, description="""Stores information about the result of a process (ie the process of sequencing a library may have for qc_status of 'fail' if not enough data was generated)""", json_schema_extra = { "linkml_meta": {'alias': 'qc_status', 'domain_of': ['PlannedProcess']} })
    qc_comment: Optional[str] = Field(default=None, description="""Slot to store additional comments about laboratory or workflow output. For workflow output it may describe the particular workflow stage that failed. (ie Failed at call-stage due to a malformed fastq file).""", json_schema_extra = { "linkml_meta": {'alias': 'qc_comment', 'domain_of': ['PlannedProcess']} })
    has_failure_categorization: Optional[list[FailureCategorization]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'has_failure_categorization', 'domain_of': ['PlannedProcess']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}},
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:subspr-{id_shoulder}-{id_blade}$'}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/SubSamplingProcess","nmdc:SubSamplingProcess"] = Field(default="nmdc:SubSamplingProcess", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('instrument_used')
    def pattern_instrument_used(cls, v):
        pattern=re.compile(r"^(nmdc):inst-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid instrument_used format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid instrument_used format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_input')
    def pattern_has_input(cls, v):
        pattern=re.compile(r"^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_input format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_input format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_output')
    def pattern_has_output(cls, v):
        pattern=re.compile(r"^(nmdc):(procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_output format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_output format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^(nmdc):subspr-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class MixingProcess(MaterialProcessing):
    """
    The combining of components, particles or layers into a more homogeneous state.

    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:MixingProcess',
         'comments': ['The mixing may be achieved manually or mechanically by shifting '
                      'the material with stirrers or pumps or by revolving or shaking '
                      'the container.',
                      'The process must not permit segregation of particles of '
                      'different size or properties.',
                      'Homogeneity may be considered to have been achieved in a '
                      'practical sense when the sampling error of the processed '
                      'portion is negligible compared to the total error of the '
                      'measurement system.'],
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-8683-0050'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'slot_usage': {'has_input': {'name': 'has_input',
                                      'pattern': '^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                      'range': 'Sample',
                                      'structured_pattern': {'interpolated': True,
                                                             'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}},
                        'has_output': {'description': 'The mixed sample.',
                                       'name': 'has_output',
                                       'pattern': '^(nmdc):procsm-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                       'range': 'ProcessedSample',
                                       'structured_pattern': {'interpolated': True,
                                                              'syntax': '{id_nmdc_prefix}:procsm-{id_shoulder}-{id_blade}$'}},
                        'id': {'name': 'id',
                               'pattern': '^(nmdc):mixpro-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                               'required': True,
                               'structured_pattern': {'syntax': '{id_nmdc_prefix}:mixpro-{id_shoulder}-{id_blade}$'}}}})

    duration: Optional[QuantityValue] = Field(default=None, description="""The elapsed time of an activity.""", json_schema_extra = { "linkml_meta": {'alias': 'duration',
         'domain_of': ['MixingProcess',
                       'DissolvingProcess',
                       'ChemicalConversionProcess',
                       'MobilePhaseSegment'],
         'examples': [{'value': "JsonObj(has_numeric_value=2, has_unit='hours')"}]} })
    instrument_used: Optional[list[str]] = Field(default=None, description="""What instrument was used during DataGeneration or MaterialProcessing.""", json_schema_extra = { "linkml_meta": {'alias': 'instrument_used',
         'domain_of': ['MaterialProcessing', 'DataGeneration'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:inst-{id_shoulder}-{id_blade}$'}} })
    has_input: Optional[list[str]] = Field(default=None, description="""An input to a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_input',
         'aliases': ['input'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}} })
    has_output: Optional[list[str]] = Field(default=None, description="""The mixed sample.""", json_schema_extra = { "linkml_meta": {'alias': 'has_output',
         'aliases': ['output'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:procsm-{id_shoulder}-{id_blade}$'}} })
    processing_institution: Optional[ProcessingInstitutionEnum] = Field(default=None, description="""The organization that processed the sample.""", json_schema_extra = { "linkml_meta": {'alias': 'processing_institution', 'domain_of': ['PlannedProcess']} })
    protocol_link: Optional[Protocol] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'protocol_link',
         'domain_of': ['Configuration', 'PlannedProcess', 'Study']} })
    start_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was started""", json_schema_extra = { "linkml_meta": {'alias': 'start_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    end_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was ended""", json_schema_extra = { "linkml_meta": {'alias': 'end_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    qc_status: Optional[StatusEnum] = Field(default=None, description="""Stores information about the result of a process (ie the process of sequencing a library may have for qc_status of 'fail' if not enough data was generated)""", json_schema_extra = { "linkml_meta": {'alias': 'qc_status', 'domain_of': ['PlannedProcess']} })
    qc_comment: Optional[str] = Field(default=None, description="""Slot to store additional comments about laboratory or workflow output. For workflow output it may describe the particular workflow stage that failed. (ie Failed at call-stage due to a malformed fastq file).""", json_schema_extra = { "linkml_meta": {'alias': 'qc_comment', 'domain_of': ['PlannedProcess']} })
    has_failure_categorization: Optional[list[FailureCategorization]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'has_failure_categorization', 'domain_of': ['PlannedProcess']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}},
         'structured_pattern': {'syntax': '{id_nmdc_prefix}:mixpro-{id_shoulder}-{id_blade}$'}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/MixingProcess","nmdc:MixingProcess"] = Field(default="nmdc:MixingProcess", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('instrument_used')
    def pattern_instrument_used(cls, v):
        pattern=re.compile(r"^(nmdc):inst-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid instrument_used format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid instrument_used format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_input')
    def pattern_has_input(cls, v):
        pattern=re.compile(r"^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_input format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_input format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_output')
    def pattern_has_output(cls, v):
        pattern=re.compile(r"^(nmdc):procsm-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_output format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_output format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^(nmdc):mixpro-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class FiltrationProcess(MaterialProcessing):
    """
    The process of segregation of phases; e.g. the separation of suspended solids from a liquid or gas, usually by forcing a carrier gas or liquid through a porous medium.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:FiltrationProcess',
         'contributors': ['ORCID:0009-0001-1555-1601',
                          'ORCID:0000-0002-8683-0050',
                          'ORCID:0000-0001-9076-6066',
                          'ORCID:0009-0008-4013-7737'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'related_mappings': ['CHMO:0001640'],
         'slot_usage': {'has_input': {'name': 'has_input',
                                      'pattern': '^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                      'range': 'Sample',
                                      'structured_pattern': {'interpolated': True,
                                                             'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}},
                        'has_output': {'name': 'has_output',
                                       'pattern': '^(nmdc):procsm-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                       'range': 'ProcessedSample',
                                       'structured_pattern': {'interpolated': True,
                                                              'syntax': '{id_nmdc_prefix}:procsm-{id_shoulder}-{id_blade}$'}},
                        'id': {'name': 'id',
                               'pattern': '^(nmdc):filtpr-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                               'required': True,
                               'structured_pattern': {'interpolated': True,
                                                      'syntax': '{id_nmdc_prefix}:filtpr-{id_shoulder}-{id_blade}$'}},
                        'volume': {'description': 'The volume of sample filtered.',
                                   'name': 'volume'}}})

    conditionings: Optional[list[str]] = Field(default=None, description="""Preliminary treatment of either phase with a suitable solution of the other phase (in the absence of main extractable solute(s)) so that when the subsequent equilibration is carried out changes in the (volume) phase ratio or in the concentrations of other components are minimized.""", json_schema_extra = { "linkml_meta": {'alias': 'conditionings',
         'domain_of': ['FiltrationProcess'],
         'list_elements_ordered': True} })
    container_size: Optional[QuantityValue] = Field(default=None, description="""The volume of the container an analyte is stored in or an activity takes place in""", json_schema_extra = { "linkml_meta": {'alias': 'container_size',
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-8683-0050'],
         'domain_of': ['SubSamplingProcess', 'FiltrationProcess']} })
    filter_material: Optional[str] = Field(default=None, description="""A porous material on which solid particles present in air or other fluid which flows through it are largely caught and retained.""", json_schema_extra = { "linkml_meta": {'alias': 'filter_material',
         'comments': ['Filters are made with a variety of materials: cellulose and '
                      'derivatives, glass fibre, ceramic, synthetic plastics and '
                      'fibres. Filters may be naturally porous or be made so by '
                      'mechanical or other means. Membrane/ceramic filters are '
                      'prepared with highly controlled pore size in a sheet of '
                      'suitable material such as polyfluoroethylene, polycarbonate or '
                      'cellulose esters. Nylon mesh is sometimes used for '
                      'reinforcement. The pores constitute 80–85% of the filter volume '
                      'commonly and several pore sizes are available for air sampling '
                      '(0.45−0.8 μm are commonly employed).'],
         'domain_of': ['FiltrationProcess']} })
    filter_pore_size: Optional[QuantityValue] = Field(default=None, description="""A quantitative or qualitative measurement of the physical dimensions of the pores in a material.""", json_schema_extra = { "linkml_meta": {'alias': 'filter_pore_size', 'domain_of': ['FiltrationProcess']} })
    filtration_category: Optional[str] = Field(default=None, description="""The type of conditioning applied to a filter, device, etc.""", json_schema_extra = { "linkml_meta": {'alias': 'filtration_category', 'domain_of': ['FiltrationProcess']} })
    is_pressurized: Optional[bool] = Field(default=None, description="""Whether or not pressure was applied to a thing or process.""", json_schema_extra = { "linkml_meta": {'alias': 'is_pressurized', 'domain_of': ['FiltrationProcess']} })
    separation_method: Optional[SeparationMethodEnum] = Field(default=None, description="""The method that was used to separate a substance from a solution or mixture.""", json_schema_extra = { "linkml_meta": {'alias': 'separation_method', 'domain_of': ['FiltrationProcess']} })
    volume: Optional[QuantityValue] = Field(default=None, description="""The volume of sample filtered.""", json_schema_extra = { "linkml_meta": {'alias': 'volume',
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-8683-0050'],
         'domain_of': ['Extraction',
                       'SubSamplingProcess',
                       'FiltrationProcess',
                       'MobilePhaseSegment',
                       'PortionOfSubstance']} })
    instrument_used: Optional[list[str]] = Field(default=None, description="""What instrument was used during DataGeneration or MaterialProcessing.""", json_schema_extra = { "linkml_meta": {'alias': 'instrument_used',
         'domain_of': ['MaterialProcessing', 'DataGeneration'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:inst-{id_shoulder}-{id_blade}$'}} })
    has_input: Optional[list[str]] = Field(default=None, description="""An input to a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_input',
         'aliases': ['input'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}} })
    has_output: Optional[list[str]] = Field(default=None, description="""An output from a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_output',
         'aliases': ['output'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:procsm-{id_shoulder}-{id_blade}$'}} })
    processing_institution: Optional[ProcessingInstitutionEnum] = Field(default=None, description="""The organization that processed the sample.""", json_schema_extra = { "linkml_meta": {'alias': 'processing_institution', 'domain_of': ['PlannedProcess']} })
    protocol_link: Optional[Protocol] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'protocol_link',
         'domain_of': ['Configuration', 'PlannedProcess', 'Study']} })
    start_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was started""", json_schema_extra = { "linkml_meta": {'alias': 'start_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    end_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was ended""", json_schema_extra = { "linkml_meta": {'alias': 'end_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    qc_status: Optional[StatusEnum] = Field(default=None, description="""Stores information about the result of a process (ie the process of sequencing a library may have for qc_status of 'fail' if not enough data was generated)""", json_schema_extra = { "linkml_meta": {'alias': 'qc_status', 'domain_of': ['PlannedProcess']} })
    qc_comment: Optional[str] = Field(default=None, description="""Slot to store additional comments about laboratory or workflow output. For workflow output it may describe the particular workflow stage that failed. (ie Failed at call-stage due to a malformed fastq file).""", json_schema_extra = { "linkml_meta": {'alias': 'qc_comment', 'domain_of': ['PlannedProcess']} })
    has_failure_categorization: Optional[list[FailureCategorization]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'has_failure_categorization', 'domain_of': ['PlannedProcess']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}},
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:filtpr-{id_shoulder}-{id_blade}$'}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/FiltrationProcess","nmdc:FiltrationProcess"] = Field(default="nmdc:FiltrationProcess", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('instrument_used')
    def pattern_instrument_used(cls, v):
        pattern=re.compile(r"^(nmdc):inst-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid instrument_used format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid instrument_used format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_input')
    def pattern_has_input(cls, v):
        pattern=re.compile(r"^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_input format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_input format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_output')
    def pattern_has_output(cls, v):
        pattern=re.compile(r"^(nmdc):procsm-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_output format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_output format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^(nmdc):filtpr-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class ChromatographicSeparationProcess(MaterialProcessing):
    """
    The process of using a selective partitioning of the analyte or interferent between two immiscible phases.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:ChromatographicSeparationProcess',
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-1368-8217'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'slot_usage': {'has_input': {'name': 'has_input',
                                      'pattern': '^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                      'range': 'Sample',
                                      'structured_pattern': {'interpolated': True,
                                                             'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}},
                        'has_output': {'name': 'has_output',
                                       'pattern': '^(nmdc):procsm-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                                       'range': 'ProcessedSample',
                                       'structured_pattern': {'interpolated': True,
                                                              'syntax': '{id_nmdc_prefix}:procsm-{id_shoulder}-{id_blade}$'}},
                        'id': {'name': 'id',
                               'pattern': '^(nmdc):cspro-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                               'required': True,
                               'structured_pattern': {'syntax': '{id_nmdc_prefix}:cspro-{id_shoulder}-{id_blade}$'}}}})

    chromatographic_category: Optional[ChromatographicCategoryEnum] = Field(default=None, description="""The type of chromatography used in a process.""", json_schema_extra = { "linkml_meta": {'alias': 'chromatographic_category',
         'domain_of': ['ChromatographyConfiguration',
                       'ChromatographicSeparationProcess']} })
    ordered_mobile_phases: Optional[list[MobilePhaseSegment]] = Field(default=None, description="""The solution(s) that moves through a chromatography column.""", json_schema_extra = { "linkml_meta": {'alias': 'ordered_mobile_phases',
         'domain_of': ['ChromatographyConfiguration',
                       'ChromatographicSeparationProcess'],
         'list_elements_ordered': True} })
    stationary_phase: Optional[StationaryPhaseEnum] = Field(default=None, description="""The material the stationary phase is comprised of used in chromatography.""", json_schema_extra = { "linkml_meta": {'alias': 'stationary_phase',
         'domain_of': ['ChromatographyConfiguration',
                       'ChromatographicSeparationProcess']} })
    temperature: Optional[QuantityValue] = Field(default=None, description="""The value of a temperature measurement or temperature used in a process.""", json_schema_extra = { "linkml_meta": {'alias': 'temperature',
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-8683-0050'],
         'domain_of': ['ChromatographyConfiguration',
                       'Extraction',
                       'SubSamplingProcess',
                       'StorageProcess',
                       'ChromatographicSeparationProcess',
                       'DissolvingProcess',
                       'ChemicalConversionProcess'],
         'notes': ['Not to be confused with the MIXS:0000113']} })
    instrument_used: Optional[list[str]] = Field(default=None, description="""What instrument was used during DataGeneration or MaterialProcessing.""", json_schema_extra = { "linkml_meta": {'alias': 'instrument_used',
         'domain_of': ['MaterialProcessing', 'DataGeneration'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:inst-{id_shoulder}-{id_blade}$'}} })
    has_input: Optional[list[str]] = Field(default=None, description="""An input to a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_input',
         'aliases': ['input'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}} })
    has_output: Optional[list[str]] = Field(default=None, description="""An output from a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_output',
         'aliases': ['output'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:procsm-{id_shoulder}-{id_blade}$'}} })
    processing_institution: Optional[ProcessingInstitutionEnum] = Field(default=None, description="""The organization that processed the sample.""", json_schema_extra = { "linkml_meta": {'alias': 'processing_institution', 'domain_of': ['PlannedProcess']} })
    protocol_link: Optional[Protocol] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'protocol_link',
         'domain_of': ['Configuration', 'PlannedProcess', 'Study']} })
    start_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was started""", json_schema_extra = { "linkml_meta": {'alias': 'start_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    end_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was ended""", json_schema_extra = { "linkml_meta": {'alias': 'end_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    qc_status: Optional[StatusEnum] = Field(default=None, description="""Stores information about the result of a process (ie the process of sequencing a library may have for qc_status of 'fail' if not enough data was generated)""", json_schema_extra = { "linkml_meta": {'alias': 'qc_status', 'domain_of': ['PlannedProcess']} })
    qc_comment: Optional[str] = Field(default=None, description="""Slot to store additional comments about laboratory or workflow output. For workflow output it may describe the particular workflow stage that failed. (ie Failed at call-stage due to a malformed fastq file).""", json_schema_extra = { "linkml_meta": {'alias': 'qc_comment', 'domain_of': ['PlannedProcess']} })
    has_failure_categorization: Optional[list[FailureCategorization]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'has_failure_categorization', 'domain_of': ['PlannedProcess']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}},
         'structured_pattern': {'syntax': '{id_nmdc_prefix}:cspro-{id_shoulder}-{id_blade}$'}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/ChromatographicSeparationProcess","nmdc:ChromatographicSeparationProcess"] = Field(default="nmdc:ChromatographicSeparationProcess", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('instrument_used')
    def pattern_instrument_used(cls, v):
        pattern=re.compile(r"^(nmdc):inst-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid instrument_used format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid instrument_used format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_input')
    def pattern_has_input(cls, v):
        pattern=re.compile(r"^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_input format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_input format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_output')
    def pattern_has_output(cls, v):
        pattern=re.compile(r"^(nmdc):procsm-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_output format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_output format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^(nmdc):cspro-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class DissolvingProcess(MaterialProcessing):
    """
    A mixing step where a soluble component is mixed with a liquid component.

    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'aliases': ['Solubilization'],
         'class_uri': 'nmdc:DissolvingProcess',
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-1368-8217'],
         'exact_mappings': ['CHMO:0002773'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'slot_usage': {'id': {'name': 'id',
                               'pattern': '^(nmdc):dispro-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                               'required': True,
                               'structured_pattern': {'interpolated': True,
                                                      'syntax': '{id_nmdc_prefix}:dispro-{id_shoulder}-{id_blade}$'}}}})

    duration: Optional[QuantityValue] = Field(default=None, description="""The elapsed time of an activity.""", json_schema_extra = { "linkml_meta": {'alias': 'duration',
         'domain_of': ['MixingProcess',
                       'DissolvingProcess',
                       'ChemicalConversionProcess',
                       'MobilePhaseSegment'],
         'examples': [{'value': "JsonObj(has_numeric_value=2, has_unit='hours')"}]} })
    temperature: Optional[QuantityValue] = Field(default=None, description="""The value of a temperature measurement or temperature used in a process.""", json_schema_extra = { "linkml_meta": {'alias': 'temperature',
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-8683-0050'],
         'domain_of': ['ChromatographyConfiguration',
                       'Extraction',
                       'SubSamplingProcess',
                       'StorageProcess',
                       'ChromatographicSeparationProcess',
                       'DissolvingProcess',
                       'ChemicalConversionProcess'],
         'notes': ['Not to be confused with the MIXS:0000113']} })
    substances_used: Optional[list[PortionOfSubstance]] = Field(default=None, description="""The substances that are combined to enable a ChemicalConversionProcess.""", json_schema_extra = { "linkml_meta": {'alias': 'substances_used',
         'domain_of': ['Extraction',
                       'StorageProcess',
                       'DissolvingProcess',
                       'ChemicalConversionProcess',
                       'MobilePhaseSegment']} })
    instrument_used: Optional[list[str]] = Field(default=None, description="""What instrument was used during DataGeneration or MaterialProcessing.""", json_schema_extra = { "linkml_meta": {'alias': 'instrument_used',
         'domain_of': ['MaterialProcessing', 'DataGeneration'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:inst-{id_shoulder}-{id_blade}$'}} })
    has_input: Optional[list[str]] = Field(default=None, description="""An input to a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_input',
         'aliases': ['input'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}} })
    has_output: Optional[list[str]] = Field(default=None, description="""An output from a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_output',
         'aliases': ['output'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(procsm)-{id_shoulder}-{id_blade}$'}} })
    processing_institution: Optional[ProcessingInstitutionEnum] = Field(default=None, description="""The organization that processed the sample.""", json_schema_extra = { "linkml_meta": {'alias': 'processing_institution', 'domain_of': ['PlannedProcess']} })
    protocol_link: Optional[Protocol] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'protocol_link',
         'domain_of': ['Configuration', 'PlannedProcess', 'Study']} })
    start_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was started""", json_schema_extra = { "linkml_meta": {'alias': 'start_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    end_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was ended""", json_schema_extra = { "linkml_meta": {'alias': 'end_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    qc_status: Optional[StatusEnum] = Field(default=None, description="""Stores information about the result of a process (ie the process of sequencing a library may have for qc_status of 'fail' if not enough data was generated)""", json_schema_extra = { "linkml_meta": {'alias': 'qc_status', 'domain_of': ['PlannedProcess']} })
    qc_comment: Optional[str] = Field(default=None, description="""Slot to store additional comments about laboratory or workflow output. For workflow output it may describe the particular workflow stage that failed. (ie Failed at call-stage due to a malformed fastq file).""", json_schema_extra = { "linkml_meta": {'alias': 'qc_comment', 'domain_of': ['PlannedProcess']} })
    has_failure_categorization: Optional[list[FailureCategorization]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'has_failure_categorization', 'domain_of': ['PlannedProcess']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}},
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:dispro-{id_shoulder}-{id_blade}$'}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/DissolvingProcess","nmdc:DissolvingProcess"] = Field(default="nmdc:DissolvingProcess", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('instrument_used')
    def pattern_instrument_used(cls, v):
        pattern=re.compile(r"^(nmdc):inst-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid instrument_used format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid instrument_used format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_input')
    def pattern_has_input(cls, v):
        pattern=re.compile(r"^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_input format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_input format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_output')
    def pattern_has_output(cls, v):
        pattern=re.compile(r"^(nmdc):(procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_output format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_output format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^(nmdc):dispro-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class ChemicalConversionProcess(MaterialProcessing):
    """
    A process that results in the interconversion of chemical species by a reaction to transform the reagents into products.

    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'aliases': ['chemical reaction'],
         'class_uri': 'nmdc:ChemicalConversionProcess',
         'comments': ['The values of both has_reagents slot and has_input slot are '
                      'considered the reagents of a chemical process.'],
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-1368-8217'],
         'exact_mappings': ['MISO:0000001'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'slot_usage': {'id': {'name': 'id',
                               'pattern': '^(nmdc):chcpr-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$',
                               'required': True,
                               'structured_pattern': {'interpolated': True,
                                                      'syntax': '{id_nmdc_prefix}:chcpr-{id_shoulder}-{id_blade}$'}}}})

    chemical_conversion_category: Optional[ChemicalConversionCategoryEnum] = Field(default=None, description="""The type of chemical conversion process.""", json_schema_extra = { "linkml_meta": {'alias': 'chemical_conversion_category',
         'domain_of': ['ChemicalConversionProcess']} })
    duration: Optional[QuantityValue] = Field(default=None, description="""The elapsed time of an activity.""", json_schema_extra = { "linkml_meta": {'alias': 'duration',
         'domain_of': ['MixingProcess',
                       'DissolvingProcess',
                       'ChemicalConversionProcess',
                       'MobilePhaseSegment'],
         'examples': [{'value': "JsonObj(has_numeric_value=2, has_unit='hours')"}]} })
    temperature: Optional[QuantityValue] = Field(default=None, description="""The value of a temperature measurement or temperature used in a process.""", json_schema_extra = { "linkml_meta": {'alias': 'temperature',
         'contributors': ['ORCID:0009-0001-1555-1601', 'ORCID:0000-0002-8683-0050'],
         'domain_of': ['ChromatographyConfiguration',
                       'Extraction',
                       'SubSamplingProcess',
                       'StorageProcess',
                       'ChromatographicSeparationProcess',
                       'DissolvingProcess',
                       'ChemicalConversionProcess'],
         'notes': ['Not to be confused with the MIXS:0000113']} })
    substances_used: Optional[list[PortionOfSubstance]] = Field(default=None, description="""The substances that are combined to enable a ChemicalConversionProcess.""", json_schema_extra = { "linkml_meta": {'alias': 'substances_used',
         'domain_of': ['Extraction',
                       'StorageProcess',
                       'DissolvingProcess',
                       'ChemicalConversionProcess',
                       'MobilePhaseSegment']} })
    substances_volume: Optional[QuantityValue] = Field(default=None, description="""The volume of the combined substances that was included in a ChemicalConversionProcess.""", json_schema_extra = { "linkml_meta": {'alias': 'substances_volume', 'domain_of': ['ChemicalConversionProcess']} })
    instrument_used: Optional[list[str]] = Field(default=None, description="""What instrument was used during DataGeneration or MaterialProcessing.""", json_schema_extra = { "linkml_meta": {'alias': 'instrument_used',
         'domain_of': ['MaterialProcessing', 'DataGeneration'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:inst-{id_shoulder}-{id_blade}$'}} })
    has_input: Optional[list[str]] = Field(default=None, description="""An input to a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_input',
         'aliases': ['input'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(bsm|procsm)-{id_shoulder}-{id_blade}$'}} })
    has_output: Optional[list[str]] = Field(default=None, description="""An output from a process.""", json_schema_extra = { "linkml_meta": {'alias': 'has_output',
         'aliases': ['output'],
         'domain_of': ['PlannedProcess'],
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:(procsm)-{id_shoulder}-{id_blade}$'}} })
    processing_institution: Optional[ProcessingInstitutionEnum] = Field(default=None, description="""The organization that processed the sample.""", json_schema_extra = { "linkml_meta": {'alias': 'processing_institution', 'domain_of': ['PlannedProcess']} })
    protocol_link: Optional[Protocol] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'protocol_link',
         'domain_of': ['Configuration', 'PlannedProcess', 'Study']} })
    start_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was started""", json_schema_extra = { "linkml_meta": {'alias': 'start_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    end_date: Optional[str] = Field(default=None, description="""The date on which any process or activity was ended""", json_schema_extra = { "linkml_meta": {'alias': 'end_date',
         'comments': ['We are using string representations of dates until all '
                      'components of our ecosystem can handle ISO 8610 dates',
                      'The date should be formatted as YYYY-MM-DD'],
         'domain_of': ['PlannedProcess'],
         'todos': ['add date string validation pattern']} })
    qc_status: Optional[StatusEnum] = Field(default=None, description="""Stores information about the result of a process (ie the process of sequencing a library may have for qc_status of 'fail' if not enough data was generated)""", json_schema_extra = { "linkml_meta": {'alias': 'qc_status', 'domain_of': ['PlannedProcess']} })
    qc_comment: Optional[str] = Field(default=None, description="""Slot to store additional comments about laboratory or workflow output. For workflow output it may describe the particular workflow stage that failed. (ie Failed at call-stage due to a malformed fastq file).""", json_schema_extra = { "linkml_meta": {'alias': 'qc_comment', 'domain_of': ['PlannedProcess']} })
    has_failure_categorization: Optional[list[FailureCategorization]] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'has_failure_categorization', 'domain_of': ['PlannedProcess']} })
    id: str = Field(default=..., description="""A unique identifier for a thing. Must be either a CURIE shorthand for a URI or a complete URI""", json_schema_extra = { "linkml_meta": {'alias': 'id',
         'domain_of': ['NamedThing'],
         'examples': [{'description': 'https://github.com/microbiomedata/nmdc-schema/pull/499#discussion_r1018499248',
                       'value': 'nmdc:mgmag-00-x012.1_7_c1'}],
         'notes': ['abstracted pattern: '
                   'prefix:typecode-authshoulder-blade(.version)?(_seqsuffix)?',
                   'a minimum length of 3 characters is suggested for typecodes, but 1 '
                   'or 2 characters will be accepted',
                   'typecodes must correspond 1:1 to a class in the NMDC schema. this '
                   'will be checked via per-class id slot usage assertions',
                   'minting authority shoulders should probably be enumerated and '
                   'checked in the pattern'],
         'structured_aliases': {'data_object_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                   'literal_form': 'data_object_id',
                                                   'predicate': 'NARROW_SYNONYM'},
                                'workflow_execution_id': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                          'literal_form': 'workflow_execution_id',
                                                          'predicate': 'NARROW_SYNONYM'}},
         'structured_pattern': {'interpolated': True,
                                'syntax': '{id_nmdc_prefix}:chcpr-{id_shoulder}-{id_blade}$'}} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    description: Optional[str] = Field(default=None, description="""a human-readable description of a thing""", json_schema_extra = { "linkml_meta": {'alias': 'description',
         'domain_of': ['ImageValue', 'NamedThing'],
         'slot_uri': 'dcterms:description'} })
    alternative_identifiers: Optional[list[str]] = Field(default=None, description="""A list of alternative identifiers for the entity.""", json_schema_extra = { "linkml_meta": {'alias': 'alternative_identifiers',
         'domain_of': ['MetaboliteIdentification', 'NamedThing']} })
    type: Literal["https://w3id.org/nmdc/ChemicalConversionProcess","nmdc:ChemicalConversionProcess"] = Field(default="nmdc:ChemicalConversionProcess", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('instrument_used')
    def pattern_instrument_used(cls, v):
        pattern=re.compile(r"^(nmdc):inst-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid instrument_used format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid instrument_used format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_input')
    def pattern_has_input(cls, v):
        pattern=re.compile(r"^(nmdc):(bsm|procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_input format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_input format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('has_output')
    def pattern_has_output(cls, v):
        pattern=re.compile(r"^(nmdc):(procsm)-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid has_output format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid has_output format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('id')
    def pattern_id(cls, v):
        pattern=re.compile(r"^(nmdc):chcpr-([0-9][a-z]{0,6}[0-9])-([A-Za-z0-9]{1,})$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid id format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid id format: {v}"
            raise ValueError(err_msg)
        return v

    @field_validator('alternative_identifiers')
    def pattern_alternative_identifiers(cls, v):
        pattern=re.compile(r"^[a-zA-Z0-9][a-zA-Z0-9_\.]+:[a-zA-Z0-9_][a-zA-Z0-9_\-\/\.,\(\)\=\#]*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid alternative_identifiers format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
            err_msg = f"Invalid alternative_identifiers format: {v}"
            raise ValueError(err_msg)
        return v


class Protocol(ConfiguredBaseModel):
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'class_uri': 'nmdc:Protocol', 'from_schema': 'https://w3id.org/nmdc/nmdc'})

    url: Optional[str] = Field(default=None, json_schema_extra = { "linkml_meta": {'alias': 'url',
         'domain_of': ['ImageValue', 'Protocol', 'DataObject'],
         'notes': ['See issue 207 - this clashes with the mixs field']} })
    name: Optional[str] = Field(default=None, description="""A human readable label for an entity""", json_schema_extra = { "linkml_meta": {'alias': 'name', 'domain_of': ['PersonValue', 'NamedThing', 'Protocol']} })
    type: Literal["https://w3id.org/nmdc/Protocol","nmdc:Protocol"] = Field(default="nmdc:Protocol", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })


class CreditAssociation(ConfiguredBaseModel):
    """
    This class supports binding associated researchers to studies. There will be at least a slot for a CRediT Contributor Role and for a person value. Specifically see the associated researchers tab on the NMDC_SampleMetadata-V4_CommentsForUpdates at https://docs.google.com/spreadsheets/d/1INlBo5eoqn2efn4H2P2i8rwRBtnbDVTqXrochJEAPko/edit#gid=0
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'aliases': ['study role', 'credit table', 'associated researchers'],
         'class_uri': 'prov:Association',
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'see_also': ['https://casrai.org/credit/']})

    applies_to_person: PersonValue = Field(default=..., json_schema_extra = { "linkml_meta": {'alias': 'applies_to_person', 'domain_of': ['CreditAssociation']} })
    applied_roles: list[CreditEnum] = Field(default=..., json_schema_extra = { "linkml_meta": {'alias': 'applied_roles',
         'annotations': {'tooltip': {'tag': 'tooltip',
                                     'value': 'Identify all CRediT roles associated '
                                              'with this contributor. CRediT '
                                              'Information: '
                                              'https://info.orcid.org/credit-for-research-contribution '
                                              '; CRediT: https://credit.niso.org/'}},
         'domain_of': ['CreditAssociation']} })
    type: Literal["https://w3id.org/nmdc/CreditAssociation","http://www.w3.org/ns/prov#Association","nmdc:CreditAssociation","prov:Association"] = Field(default="prov:Association", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })


class Doi(ConfiguredBaseModel):
    """
    A centrally registered identifier symbol used to uniquely identify objects given by the International DOI Foundation. The DOI system is particularly used for electronic documents.
    """
    linkml_meta: ClassVar[LinkMLMeta] = LinkMLMeta({'aliases': ['DOIs', 'digital object identifiers'],
         'class_uri': 'nmdc:Doi',
         'exact_mappings': ['OBI:0002110'],
         'from_schema': 'https://w3id.org/nmdc/nmdc',
         'rules': [{'description': 'If doi_category is a publication_doi, then '
                                   'doi_provider is not required. Otherwise, '
                                   'doi_provider is required.',
                    'postconditions': {'slot_conditions': {'doi_provider': {'name': 'doi_provider',
                                                                            'required': True}}},
                    'preconditions': {'slot_conditions': {'doi_category': {'equals_string': 'dataset_doi',
                                                                           'name': 'doi_category'}}},
                    'title': 'dataset_dois_require_provider'},
                   {'description': 'If doi_category is a publication_doi, then '
                                   'doi_provider is not required. Otherwise, '
                                   'doi_provider is required.',
                    'postconditions': {'slot_conditions': {'doi_provider': {'name': 'doi_provider',
                                                                            'required': True}}},
                    'preconditions': {'slot_conditions': {'doi_category': {'equals_string': 'award_doi',
                                                                           'name': 'doi_category'}}},
                    'title': 'award_dois_require_provider'}]})

    doi_value: str = Field(default=..., description="""A digital object identifier, which is intended to persistantly identify some resource on the web.""", json_schema_extra = { "linkml_meta": {'alias': 'doi_value',
         'aliases': ['DOI', 'digital object identifier'],
         'domain_of': ['Doi'],
         'exact_mappings': ['OBI:0002110'],
         'examples': [{'description': 'The DOI links to an electronic document.',
                       'value': 'doi:10.46936/10.25585/60000880'}],
         'narrow_mappings': ['edam.data:1188']} })
    doi_provider: Optional[DoiProviderEnum] = Field(default=None, description="""The authority, or organization, the DOI is associated with.""", json_schema_extra = { "linkml_meta": {'alias': 'doi_provider',
         'close_mappings': ['NCIT:C74932'],
         'domain_of': ['Doi'],
         'examples': [{'description': 'The corresponding DOI is associated with '
                                      'ESS-DIVE.',
                       'value': 'ess_dive'}]} })
    doi_category: DoiCategoryEnum = Field(default=..., description="""The resource type the corresponding doi resolves to.""", json_schema_extra = { "linkml_meta": {'alias': 'doi_category',
         'domain_of': ['Doi'],
         'examples': [{'description': 'The corresponding DOI is a dataset resource '
                                      'type.',
                       'value': 'dataset_doi'}]} })
    type: Literal["https://w3id.org/nmdc/Doi","nmdc:Doi"] = Field(default="nmdc:Doi", description="""the class_uri of the class that has been instantiated""", json_schema_extra = { "linkml_meta": {'alias': 'type',
         'designates_type': True,
         'domain_of': ['EukEval',
                       'FunctionalAnnotationAggMember',
                       'MobilePhaseSegment',
                       'PortionOfSubstance',
                       'MagBin',
                       'MetaboliteIdentification',
                       'GenomeFeature',
                       'FunctionalAnnotation',
                       'AttributeValue',
                       'NamedThing',
                       'OntologyRelation',
                       'FailureCategorization',
                       'Protocol',
                       'CreditAssociation',
                       'Doi'],
         'examples': [{'value': 'nmdc:Biosample'}, {'value': 'nmdc:Study'}],
         'notes': ['makes it easier to read example data files',
                   'required for polymorphic MongoDB collections'],
         'see_also': ['https://github.com/microbiomedata/nmdc-schema/issues/1048',
                      'https://github.com/microbiomedata/nmdc-schema/issues/1233',
                      'https://github.com/microbiomedata/nmdc-schema/issues/248'],
         'slot_uri': 'rdf:type',
         'structured_aliases': {'workflow_execution_class': {'contexts': ['https://bitbucket.org/berkeleylab/jgi-jat/macros/nmdc_metadata.yaml'],
                                                             'literal_form': 'workflow_execution_class',
                                                             'predicate': 'NARROW_SYNONYM'}}} })

    @field_validator('doi_value')
    def pattern_doi_value(cls, v):
        pattern=re.compile(r"^doi:10.\d{2,9}/.*$")
        if isinstance(v, list):
            for element in v:
                if isinstance(element, str) and not pattern.match(element):
                    err_msg = f"Invalid doi_value format: {element}"
                    raise ValueError(err_msg)
        elif isinstance(v, str) and not pattern.match(v):
      