"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchApiGateway = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const metrics_1 = require("./monitoring/aws/api-gateway/metrics");
/**
 * @stability stable
 */
class WatchApiGateway extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.api = props.restApi.node.findChild('Resource');
        this.apiName = this.api.name;
        this.stage = props.restApi.deploymentStage.stageName;
        this.watchful = props.watchful;
        this.metrics = new metrics_1.ApiGatewayMetricFactory();
        const alarmThreshold = props.serverErrorThreshold == null ? 1 : props.serverErrorThreshold;
        if (alarmThreshold) {
            const count5xxMetric = this.metrics.metricErrors(this.apiName, this.stage).count5XX.with({
                statistic: 'sum',
                period: aws_cdk_lib_1.Duration.minutes(5),
            });
            this.watchful.addAlarm(count5xxMetric.createAlarm(this, '5XXErrorAlarm', {
                alarmDescription: `at ${alarmThreshold}`,
                threshold: alarmThreshold,
                comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: 1,
            }));
        }
        this.watchful.addSection(props.title, {
            links: [{ title: 'Amazon API Gateway Console', url: linkForApiGateway(props.restApi) }],
        });
        [undefined, ...props.watchedOperations || []].forEach(operation => this.watchful.addWidgets(this.createCallGraphWidget(operation, alarmThreshold), ...props.cacheGraph ? [this.createCacheGraphWidget(operation)] : [], this.createLatencyGraphWidget(operation), this.createIntegrationLatencyGraphWidget(operation)));
    }
    createCallGraphWidget(opts, alarmThreshold) {
        const leftAnnotations = alarmThreshold
            ? [{ value: alarmThreshold, color: '#ff0000', label: '5XX Errors Alarm' }]
            : [];
        return new aws_cloudwatch_1.GraphWidget({
            title: `${opts ? `${opts.httpMethod} ${opts.resourcePath}` : 'Overall'} Calls/min`,
            width: 12,
            stacked: false,
            left: [
                this.metrics.metricCalls(this.apiName, this.stage, opts),
                this.metrics.metricErrors(this.apiName, this.stage, opts).count4XX,
                this.metrics.metricErrors(this.apiName, this.stage, opts).count5XX,
            ],
            leftAnnotations,
        });
    }
    createCacheGraphWidget(opts) {
        return new aws_cloudwatch_1.GraphWidget({
            title: `${opts ? `${opts.httpMethod} ${opts.resourcePath}` : 'Overall'} Cache/min`,
            width: 12,
            stacked: false,
            left: [
                this.metrics.metricCalls(this.apiName, this.stage, opts),
                this.metrics.metricCache(this.apiName, this.stage, opts).hits,
                this.metrics.metricCache(this.apiName, this.stage, opts).misses,
            ],
        });
    }
    createLatencyGraphWidget(opts) {
        return new aws_cloudwatch_1.GraphWidget({
            title: `${opts ? `${opts.httpMethod} ${opts.resourcePath}` : 'Overall'} (1-minute periods)`,
            width: 12,
            stacked: false,
            left: Object.values(this.metrics.metricLatency(this.apiName, this.stage, opts)),
        });
    }
    createIntegrationLatencyGraphWidget(opts) {
        return new aws_cloudwatch_1.GraphWidget({
            title: `${opts ? `${opts.httpMethod} ${opts.resourcePath}` : 'Overall'} Integration (1-minute periods)`,
            width: 12,
            stacked: false,
            left: Object.values(this.metrics.metricIntegrationLatency(this.apiName, this.stage, opts)),
        });
    }
}
exports.WatchApiGateway = WatchApiGateway;
_a = JSII_RTTI_SYMBOL_1;
WatchApiGateway[_a] = { fqn: "cdk-watchful.WatchApiGateway", version: "0.6.29" };
function linkForApiGateway(api) {
    return `https://console.aws.amazon.com/apigateway/home?region=${api.stack.region}#/apis/${api.restApiId}`;
}
//# sourceMappingURL=data:application/json;base64,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