"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchDynamoTable = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const constructs_1 = require("constructs");
const metrics_1 = require("./monitoring/aws/dynamodb/metrics");
const DEFAULT_PERCENT = 80;
/**
 * @stability stable
 */
class WatchDynamoTable extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        const table = props.table;
        this.watchful = props.watchful;
        this.metrics = new metrics_1.DynamoDbMetricFactory();
        const cfnTable = table.node.defaultChild;
        const billingMode = (_b = cfnTable.billingMode) !== null && _b !== void 0 ? _b : dynamodb.BillingMode.PROVISIONED;
        switch (billingMode) {
            case dynamodb.BillingMode.PAY_PER_REQUEST:
                this.createWidgetsForPayPerRequestTable(props.title, table);
                break;
            case dynamodb.BillingMode.PROVISIONED:
                this.createWidgetsForProvisionedTable(props.title, table, props.readCapacityThresholdPercent, props.writeCapacityThresholdPercent);
                break;
        }
    }
    /**
     * Create widgets for tables with billingMode=PROVISIONED
     * Include alarms when capacity is over 80% of the provisioned value
     */
    createWidgetsForProvisionedTable(title, table, readCapacityThresholdPercent, writeCapacityThresholdPercent) {
        const cfnTable = table.node.defaultChild;
        const metrics = this.metrics.metricConsumedCapacityUnits(table.tableName);
        const readCapacityMetric = metrics.read;
        const writeCapacityMetric = metrics.write;
        const throughput = cfnTable.provisionedThroughput;
        this.watchful.addAlarm(this.createDynamoCapacityAlarm('read', readCapacityMetric, throughput.readCapacityUnits, readCapacityThresholdPercent));
        this.watchful.addAlarm(this.createDynamoCapacityAlarm('write', writeCapacityMetric, throughput.writeCapacityUnits, writeCapacityThresholdPercent));
        this.watchful.addSection(title, {
            links: [{ title: 'Amazon DynamoDB Console', url: linkForDynamoTable(table) }],
        });
        this.watchful.addWidgets(this.createDynamoCapacityGraph('Read', readCapacityMetric, throughput.readCapacityUnits, readCapacityThresholdPercent), this.createDynamoCapacityGraph('Write', writeCapacityMetric, throughput.writeCapacityUnits, writeCapacityThresholdPercent));
    }
    /**
     * Create widgets for tables with billingMode=PAY_PER_REQUEST
     * Include consumed capacity metrics
     */
    createWidgetsForPayPerRequestTable(title, table) {
        const metrics = this.metrics.metricConsumedCapacityUnits(table.tableName);
        const readCapacityMetric = metrics.read;
        const writeCapacityMetric = metrics.write;
        this.watchful.addSection(title, {
            links: [{ title: 'Amazon DynamoDB Console', url: linkForDynamoTable(table) }],
        });
        this.watchful.addWidgets(this.createDynamoPPRGraph('Read', readCapacityMetric), this.createDynamoPPRGraph('Write', writeCapacityMetric));
    }
    createDynamoCapacityGraph(type, metric, provisioned, percent = DEFAULT_PERCENT) {
        return new cloudwatch.GraphWidget({
            title: `${type} Capacity Units/${metric.period.toMinutes()}min`,
            width: 12,
            stacked: true,
            left: [metric],
            leftAnnotations: [
                {
                    label: 'Provisioned',
                    value: provisioned * metric.period.toSeconds(),
                    color: '#58D68D',
                },
                {
                    color: '#FF3333',
                    label: `Alarm on ${percent}%`,
                    value: calculateUnits(provisioned, percent, metric.period),
                },
            ],
        });
    }
    createDynamoPPRGraph(type, metric) {
        return new cloudwatch.GraphWidget({
            title: `${type} Capacity Units/${metric.period.toMinutes()}min`,
            width: 12,
            stacked: true,
            left: [metric],
        });
    }
    createDynamoCapacityAlarm(type, metric, provisioned, percent = DEFAULT_PERCENT) {
        const periodMinutes = 5;
        const threshold = calculateUnits(provisioned, percent, aws_cdk_lib_1.Duration.minutes(periodMinutes));
        const metricWithPeriod = metric.with({
            statistic: 'sum',
            period: aws_cdk_lib_1.Duration.minutes(periodMinutes),
        });
        const alarm = metricWithPeriod.createAlarm(this, `CapacityAlarm:${type}`, {
            alarmDescription: `at ${threshold}% of ${type} capacity`,
            threshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
        });
        return alarm;
    }
}
exports.WatchDynamoTable = WatchDynamoTable;
_a = JSII_RTTI_SYMBOL_1;
WatchDynamoTable[_a] = { fqn: "cdk-watchful.WatchDynamoTable", version: "0.6.29" };
function linkForDynamoTable(table, tab = 'overview') {
    return `https://console.aws.amazon.com/dynamodb/home?region=${table.stack.region}#tables:selected=${table.tableName};tab=${tab}`;
}
function calculateUnits(provisioned, percent, period) {
    return provisioned * ((percent === undefined ? DEFAULT_PERCENT : percent) / 100) * period.toSeconds();
}
//# sourceMappingURL=data:application/json;base64,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