"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchEcsService = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const metrics_1 = require("./monitoring/aws/ecs/metrics");
/**
 * @stability stable
 */
class WatchEcsService extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.watchful = props.watchful;
        if (props.ec2Service) {
            this.ecsService = props.ec2Service;
            this.serviceName = props.ec2Service.serviceName;
            this.clusterName = props.ec2Service.cluster.clusterName;
        }
        else if (props.fargateService) {
            this.ecsService = props.fargateService;
            this.serviceName = props.fargateService.serviceName;
            this.clusterName = props.fargateService.cluster.clusterName;
        }
        else {
            throw new Error('No service provided to monitor.');
        }
        this.targetGroup = props.targetGroup;
        this.targetGroupName = this.targetGroup.targetGroupFullName;
        this.loadBalancerName = this.targetGroup.firstLoadBalancerFullName;
        this.metrics = new metrics_1.EcsMetricFactory();
        this.watchful.addSection(props.title, {
            links: [
                { title: 'ECS Service', url: linkForEcsService(this.ecsService) },
            ],
        });
        const { cpuUtilizationMetric, cpuUtilizationAlarm } = this.createCpuUtilizationMonitor(props.cpuMaximumThresholdPercent);
        const { memoryUtilizationMetric, memoryUtilizationAlarm } = this.createMemoryUtilizationMonitor(props.memoryMaximumThresholdPercent);
        const { targetResponseTimeMetric, targetResponseTimeAlarm } = this.createTargetResponseTimeMonitor(props.targetResponseTimeThreshold);
        const { healthyHostsMetric, unhealthyHostsMetric } = this.createHostCountMetrics();
        const { requestsMetric, requestsAlarm } = this.createRequestsMonitor(props.requestsThreshold);
        const { http2xxMetric, http3xxMetric, http4xxMetric, http5xxMetric } = this.createHttpRequestsMetrics();
        const { requestsErrorRateMetric, requestsErrorRateAlarm } = this.requestsErrorRate(props.requestsErrorRateThreshold);
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: `CPUUtilization/${cpuUtilizationMetric.period.toMinutes()}min`,
            width: 12,
            left: [cpuUtilizationMetric],
            leftAnnotations: [cpuUtilizationAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `MemoryUtilization/${memoryUtilizationMetric.period.toMinutes()}min`,
            width: 12,
            left: [memoryUtilizationMetric],
            leftAnnotations: [memoryUtilizationAlarm.toAnnotation()],
        }));
        this.watchful.addWidgets(new cloudwatch.SingleValueWidget({
            title: 'Healthy Hosts',
            height: 6,
            width: 6,
            metrics: [healthyHostsMetric],
        }), new cloudwatch.SingleValueWidget({
            title: 'UnHealthy Hosts',
            height: 6,
            width: 6,
            metrics: [unhealthyHostsMetric],
        }), new cloudwatch.GraphWidget({
            title: `TargetResponseTime/${targetResponseTimeMetric.period.toMinutes()}min`,
            width: 6,
            left: [targetResponseTimeMetric],
            leftAnnotations: [targetResponseTimeAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `Requests/${requestsMetric.period.toMinutes()}min`,
            width: 6,
            left: [requestsMetric],
            leftAnnotations: [requestsAlarm.toAnnotation()],
        }));
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: 'HTTP Requests Overview',
            width: 12,
            left: [http2xxMetric, http3xxMetric, http4xxMetric, http5xxMetric],
        }), new cloudwatch.GraphWidget({
            title: `HTTP Requests Error rate/${requestsErrorRateMetric.period.toMinutes()}min`,
            width: 12,
            left: [requestsErrorRateMetric],
            leftAnnotations: [requestsErrorRateAlarm.toAnnotation()],
        }));
    }
    createCpuUtilizationMonitor(cpuMaximumThresholdPercent = 0) {
        const cpuUtilizationMetric = this.metrics.metricCpuUtilizationAverage(this.clusterName, this.serviceName);
        const cpuUtilizationAlarm = cpuUtilizationMetric.createAlarm(this, 'cpuUtilizationAlarm', {
            alarmDescription: 'cpuUtilizationAlarm',
            threshold: cpuMaximumThresholdPercent,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(cpuUtilizationAlarm);
        return { cpuUtilizationMetric, cpuUtilizationAlarm };
    }
    createMemoryUtilizationMonitor(memoryMaximumThresholdPercent = 0) {
        const memoryUtilizationMetric = this.metrics.metricMemoryUtilizationAverage(this.clusterName, this.serviceName);
        const memoryUtilizationAlarm = memoryUtilizationMetric.createAlarm(this, 'memoryUtilizationAlarm', {
            alarmDescription: 'memoryUtilizationAlarm',
            threshold: memoryMaximumThresholdPercent,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(memoryUtilizationAlarm);
        return { memoryUtilizationMetric, memoryUtilizationAlarm };
    }
    createTargetResponseTimeMonitor(targetResponseTimeThreshold = 0) {
        const targetResponseTimeMetric = this.metrics.metricTargetResponseTime(this.targetGroupName, this.loadBalancerName).avg;
        const targetResponseTimeAlarm = targetResponseTimeMetric.createAlarm(this, 'targetResponseTimeAlarm', {
            alarmDescription: 'targetResponseTimeAlarm',
            threshold: targetResponseTimeThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(targetResponseTimeAlarm);
        return { targetResponseTimeMetric, targetResponseTimeAlarm };
    }
    createRequestsMonitor(requestsThreshold = 0) {
        const requestsMetric = this.metrics.metricRequestCount(this.targetGroupName, this.loadBalancerName);
        const requestsAlarm = requestsMetric.createAlarm(this, 'requestsAlarm', {
            alarmDescription: 'requestsAlarm',
            threshold: requestsThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(requestsAlarm);
        return { requestsMetric, requestsAlarm };
    }
    createHttpRequestsMetrics() {
        const metrics = this.metrics.metricHttpStatusCodeCount(this.targetGroupName, this.loadBalancerName);
        const http2xxMetric = metrics.count2XX;
        const http3xxMetric = metrics.count3XX;
        const http4xxMetric = metrics.count4XX;
        const http5xxMetric = metrics.count5XX;
        return { http2xxMetric, http3xxMetric, http4xxMetric, http5xxMetric };
    }
    createHostCountMetrics() {
        const healthyHostsMetric = this.metrics.metricMinHealthyHostCount(this.targetGroupName, this.loadBalancerName);
        const unhealthyHostsMetric = this.metrics.metricMaxUnhealthyHostCount(this.targetGroupName, this.loadBalancerName);
        return { healthyHostsMetric, unhealthyHostsMetric };
    }
    requestsErrorRate(requestsErrorRateThreshold = 0) {
        const requestsErrorRateMetric = this.metrics.metricHttpErrorStatusCodeRate(this.targetGroupName, this.loadBalancerName);
        const requestsErrorRateAlarm = requestsErrorRateMetric.createAlarm(this, 'requestsErrorRateAlarm', {
            alarmDescription: 'requestsErrorRateAlarm',
            threshold: requestsErrorRateThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(requestsErrorRateAlarm);
        return { requestsErrorRateMetric, requestsErrorRateAlarm };
    }
}
exports.WatchEcsService = WatchEcsService;
_a = JSII_RTTI_SYMBOL_1;
WatchEcsService[_a] = { fqn: "cdk-watchful.WatchEcsService", version: "0.6.29" };
function linkForEcsService(ecsService) {
    return `https://console.aws.amazon.com/ecs/home?region=${ecsService.stack.region}#/clusters/${ecsService.cluster.clusterName}/services/${ecsService.serviceName}/details`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZWNzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL2Vjcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLHlEQUF5RDtBQUd6RCwyQ0FBdUM7QUFFdkMsMERBQWdFOzs7O0FBNEJoRSxNQUFhLGVBQWdCLFNBQVEsc0JBQVM7Ozs7SUFXNUMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUEyQjtRQUNuRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQztRQUMvQixJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUU7WUFDcEIsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDO1lBQ25DLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUM7WUFDaEQsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUM7U0FDekQ7YUFBTSxJQUFJLEtBQUssQ0FBQyxjQUFjLEVBQUU7WUFDL0IsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDO1lBQ3ZDLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUM7WUFDcEQsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUM7U0FDN0Q7YUFBTTtZQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsaUNBQWlDLENBQUMsQ0FBQztTQUNwRDtRQUVELElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUNyQyxJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsbUJBQW1CLENBQUM7UUFDNUQsSUFBSSxDQUFDLGdCQUFnQixHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMseUJBQXlCLENBQUM7UUFDbkUsSUFBSSxDQUFDLE9BQU8sR0FBRyxJQUFJLDBCQUFnQixFQUFFLENBQUM7UUFFdEMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRTtZQUNwQyxLQUFLLEVBQUU7Z0JBQ0wsRUFBRSxLQUFLLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRSxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEVBQUU7YUFDbEU7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLEVBQUUsb0JBQW9CLEVBQUUsbUJBQW1CLEVBQUUsR0FBRyxJQUFJLENBQUMsMkJBQTJCLENBQUMsS0FBSyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDekgsTUFBTSxFQUFFLHVCQUF1QixFQUFFLHNCQUFzQixFQUFFLEdBQUcsSUFBSSxDQUFDLDhCQUE4QixDQUFDLEtBQUssQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBRXJJLE1BQU0sRUFBRSx3QkFBd0IsRUFBRSx1QkFBdUIsRUFBRSxHQUFHLElBQUksQ0FBQywrQkFBK0IsQ0FBQyxLQUFLLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUN0SSxNQUFNLEVBQUUsa0JBQWtCLEVBQUUsb0JBQW9CLEVBQUUsR0FBRyxJQUFJLENBQUMsc0JBQXNCLEVBQUUsQ0FBQztRQUVuRixNQUFNLEVBQUUsY0FBYyxFQUFFLGFBQWEsRUFBRSxHQUFHLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUM5RixNQUFNLEVBQUUsYUFBYSxFQUFFLGFBQWEsRUFBRSxhQUFhLEVBQUUsYUFBYSxFQUFFLEdBQUcsSUFBSSxDQUFDLHlCQUF5QixFQUFFLENBQUM7UUFDeEcsTUFBTSxFQUFFLHVCQUF1QixFQUFFLHNCQUFzQixFQUFFLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBR3JILElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUN0QixJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUM7WUFDekIsS0FBSyxFQUFFLGtCQUFrQixvQkFBb0IsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLEtBQUs7WUFDckUsS0FBSyxFQUFFLEVBQUU7WUFDVCxJQUFJLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQztZQUM1QixlQUFlLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxZQUFZLEVBQUUsQ0FBQztTQUN0RCxDQUFDLEVBQ0YsSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDO1lBQ3pCLEtBQUssRUFBRSxxQkFBcUIsdUJBQXVCLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxLQUFLO1lBQzNFLEtBQUssRUFBRSxFQUFFO1lBQ1QsSUFBSSxFQUFFLENBQUMsdUJBQXVCLENBQUM7WUFDL0IsZUFBZSxFQUFFLENBQUMsc0JBQXNCLENBQUMsWUFBWSxFQUFFLENBQUM7U0FDekQsQ0FBQyxDQUNILENBQUM7UUFDRixJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FDdEIsSUFBSSxVQUFVLENBQUMsaUJBQWlCLENBQUM7WUFDL0IsS0FBSyxFQUFFLGVBQWU7WUFDdEIsTUFBTSxFQUFFLENBQUM7WUFDVCxLQUFLLEVBQUUsQ0FBQztZQUNSLE9BQU8sRUFBRSxDQUFDLGtCQUFrQixDQUFDO1NBQzlCLENBQUMsRUFDRixJQUFJLFVBQVUsQ0FBQyxpQkFBaUIsQ0FBQztZQUMvQixLQUFLLEVBQUUsaUJBQWlCO1lBQ3hCLE1BQU0sRUFBRSxDQUFDO1lBQ1QsS0FBSyxFQUFFLENBQUM7WUFDUixPQUFPLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQztTQUNoQyxDQUFDLEVBQ0YsSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDO1lBQ3pCLEtBQUssRUFBRSxzQkFBc0Isd0JBQXdCLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxLQUFLO1lBQzdFLEtBQUssRUFBRSxDQUFDO1lBQ1IsSUFBSSxFQUFFLENBQUMsd0JBQXdCLENBQUM7WUFDaEMsZUFBZSxFQUFFLENBQUMsdUJBQXVCLENBQUMsWUFBWSxFQUFFLENBQUM7U0FDMUQsQ0FBQyxFQUNGLElBQUksVUFBVSxDQUFDLFdBQVcsQ0FBQztZQUN6QixLQUFLLEVBQUUsWUFBWSxjQUFjLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxLQUFLO1lBQ3pELEtBQUssRUFBRSxDQUFDO1lBQ1IsSUFBSSxFQUFFLENBQUMsY0FBYyxDQUFDO1lBQ3RCLGVBQWUsRUFBRSxDQUFDLGFBQWEsQ0FBQyxZQUFZLEVBQUUsQ0FBQztTQUNoRCxDQUFDLENBQ0gsQ0FBQztRQUNGLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUN0QixJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUM7WUFDekIsS0FBSyxFQUFFLHdCQUF3QjtZQUMvQixLQUFLLEVBQUUsRUFBRTtZQUNULElBQUksRUFBRSxDQUFDLGFBQWEsRUFBRSxhQUFhLEVBQUUsYUFBYSxFQUFFLGFBQWEsQ0FBQztTQUNuRSxDQUFDLEVBQ0YsSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDO1lBQ3pCLEtBQUssRUFBRSw0QkFBNEIsdUJBQXVCLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxLQUFLO1lBQ2xGLEtBQUssRUFBRSxFQUFFO1lBQ1QsSUFBSSxFQUFFLENBQUMsdUJBQXVCLENBQUM7WUFDL0IsZUFBZSxFQUFFLENBQUMsc0JBQXNCLENBQUMsWUFBWSxFQUFFLENBQUM7U0FDekQsQ0FBQyxDQUNILENBQUM7SUFDSixDQUFDO0lBRU8sMkJBQTJCLENBQUMsMEJBQTBCLEdBQUcsQ0FBQztRQUNoRSxNQUFNLG9CQUFvQixHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsMkJBQTJCLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDMUcsTUFBTSxtQkFBbUIsR0FBRyxvQkFBb0IsQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLHFCQUFxQixFQUFFO1lBQ3hGLGdCQUFnQixFQUFFLHFCQUFxQjtZQUN2QyxTQUFTLEVBQUUsMEJBQTBCO1lBQ3JDLGtCQUFrQixFQUFFLFVBQVUsQ0FBQyxrQkFBa0IsQ0FBQyxzQkFBc0I7WUFDeEUsaUJBQWlCLEVBQUUsQ0FBQztTQUNyQixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQzVDLE9BQU8sRUFBRSxvQkFBb0IsRUFBRSxtQkFBbUIsRUFBRSxDQUFDO0lBQ3ZELENBQUM7SUFFTyw4QkFBOEIsQ0FBQyw2QkFBNkIsR0FBRyxDQUFDO1FBQ3RFLE1BQU0sdUJBQXVCLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyw4QkFBOEIsQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUNoSCxNQUFNLHNCQUFzQixHQUFHLHVCQUF1QixDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsd0JBQXdCLEVBQUU7WUFDakcsZ0JBQWdCLEVBQUUsd0JBQXdCO1lBQzFDLFNBQVMsRUFBRSw2QkFBNkI7WUFDeEMsa0JBQWtCLEVBQUUsVUFBVSxDQUFDLGtCQUFrQixDQUFDLHNCQUFzQjtZQUN4RSxpQkFBaUIsRUFBRSxDQUFDO1NBQ3JCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDL0MsT0FBTyxFQUFFLHVCQUF1QixFQUFFLHNCQUFzQixFQUFFLENBQUM7SUFDN0QsQ0FBQztJQUVPLCtCQUErQixDQUFDLDJCQUEyQixHQUFHLENBQUM7UUFDckUsTUFBTSx3QkFBd0IsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLHdCQUF3QixDQUFDLElBQUksQ0FBQyxlQUFlLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUMsR0FBRyxDQUFDO1FBQ3hILE1BQU0sdUJBQXVCLEdBQUcsd0JBQXdCLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSx5QkFBeUIsRUFBRTtZQUNwRyxnQkFBZ0IsRUFBRSx5QkFBeUI7WUFDM0MsU0FBUyxFQUFFLDJCQUEyQjtZQUN0QyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsc0JBQXNCO1lBQ3hFLGlCQUFpQixFQUFFLENBQUM7U0FDckIsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNoRCxPQUFPLEVBQUUsd0JBQXdCLEVBQUUsdUJBQXVCLEVBQUUsQ0FBQztJQUMvRCxDQUFDO0lBRU8scUJBQXFCLENBQUMsaUJBQWlCLEdBQUcsQ0FBQztRQUNqRCxNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxlQUFlLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDcEcsTUFBTSxhQUFhLEdBQUcsY0FBYyxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO1lBQ3RFLGdCQUFnQixFQUFFLGVBQWU7WUFDakMsU0FBUyxFQUFFLGlCQUFpQjtZQUM1QixrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsc0JBQXNCO1lBQ3hFLGlCQUFpQixFQUFFLENBQUM7U0FDckIsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDdEMsT0FBTyxFQUFFLGNBQWMsRUFBRSxhQUFhLEVBQUUsQ0FBQztJQUMzQyxDQUFDO0lBR08seUJBQXlCO1FBQy9CLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMseUJBQXlCLENBQUMsSUFBSSxDQUFDLGVBQWUsRUFBRSxJQUFJLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUNwRyxNQUFNLGFBQWEsR0FBRyxPQUFPLENBQUMsUUFBUSxDQUFDO1FBQ3ZDLE1BQU0sYUFBYSxHQUFHLE9BQU8sQ0FBQyxRQUFRLENBQUM7UUFDdkMsTUFBTSxhQUFhLEdBQUcsT0FBTyxDQUFDLFFBQVEsQ0FBQztRQUN2QyxNQUFNLGFBQWEsR0FBRyxPQUFPLENBQUMsUUFBUSxDQUFDO1FBQ3ZDLE9BQU8sRUFBRSxhQUFhLEVBQUUsYUFBYSxFQUFFLGFBQWEsRUFBRSxhQUFhLEVBQUUsQ0FBQztJQUN4RSxDQUFDO0lBRU8sc0JBQXNCO1FBQzVCLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJLENBQUMsZUFBZSxFQUFFLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQy9HLE1BQU0sb0JBQW9CLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQywyQkFBMkIsQ0FBQyxJQUFJLENBQUMsZUFBZSxFQUFFLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ25ILE9BQU8sRUFBRSxrQkFBa0IsRUFBRSxvQkFBb0IsRUFBRSxDQUFDO0lBQ3RELENBQUM7SUFFTyxpQkFBaUIsQ0FBQywwQkFBMEIsR0FBRyxDQUFDO1FBQ3RELE1BQU0sdUJBQXVCLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyw2QkFBNkIsQ0FBQyxJQUFJLENBQUMsZUFBZSxFQUFFLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ3hILE1BQU0sc0JBQXNCLEdBQUcsdUJBQXVCLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSx3QkFBd0IsRUFBRTtZQUNqRyxnQkFBZ0IsRUFBRSx3QkFBd0I7WUFDMUMsU0FBUyxFQUFFLDBCQUEwQjtZQUNyQyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsc0JBQXNCO1lBQ3hFLGlCQUFpQixFQUFFLENBQUM7U0FDckIsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUMvQyxPQUFPLEVBQUUsdUJBQXVCLEVBQUUsc0JBQXNCLEVBQUUsQ0FBQztJQUM3RCxDQUFDOztBQWxMSCwwQ0FvTEM7OztBQUdELFNBQVMsaUJBQWlCLENBQUMsVUFBZTtJQUN4QyxPQUFPLGtEQUFrRCxVQUFVLENBQUMsS0FBSyxDQUFDLE1BQU0sY0FBYyxVQUFVLENBQUMsT0FBTyxDQUFDLFdBQVcsYUFBYSxVQUFVLENBQUMsV0FBVyxVQUFVLENBQUM7QUFDNUssQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNsb3Vkd2F0Y2ggZnJvbSAnYXdzLWNkay1saWIvYXdzLWNsb3Vkd2F0Y2gnO1xuaW1wb3J0ICogYXMgZWNzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1lY3MnO1xuaW1wb3J0IHsgQXBwbGljYXRpb25UYXJnZXRHcm91cCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1lbGFzdGljbG9hZGJhbGFuY2luZ3YyJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgSVdhdGNoZnVsIH0gZnJvbSAnLi9hcGknO1xuaW1wb3J0IHsgRWNzTWV0cmljRmFjdG9yeSB9IGZyb20gJy4vbW9uaXRvcmluZy9hd3MvZWNzL21ldHJpY3MnO1xuXG5cbmV4cG9ydCBpbnRlcmZhY2UgV2F0Y2hFY3NTZXJ2aWNlT3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNwdU1heGltdW1UaHJlc2hvbGRQZXJjZW50PzogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtZW1vcnlNYXhpbXVtVGhyZXNob2xkUGVyY2VudD86IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdGFyZ2V0UmVzcG9uc2VUaW1lVGhyZXNob2xkPzogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVxdWVzdHNUaHJlc2hvbGQ/OiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByZXF1ZXN0c0Vycm9yUmF0ZVRocmVzaG9sZD86IG51bWJlcjtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBXYXRjaEVjc1NlcnZpY2VQcm9wcyBleHRlbmRzIFdhdGNoRWNzU2VydmljZU9wdGlvbnMge1xuICByZWFkb25seSB0aXRsZTogc3RyaW5nO1xuICByZWFkb25seSB3YXRjaGZ1bDogSVdhdGNoZnVsO1xuICByZWFkb25seSBmYXJnYXRlU2VydmljZT86IGVjcy5GYXJnYXRlU2VydmljZTtcbiAgcmVhZG9ubHkgZWMyU2VydmljZT86IGVjcy5FYzJTZXJ2aWNlO1xuICByZWFkb25seSB0YXJnZXRHcm91cDogQXBwbGljYXRpb25UYXJnZXRHcm91cDtcbn1cblxuZXhwb3J0IGNsYXNzIFdhdGNoRWNzU2VydmljZSBleHRlbmRzIENvbnN0cnVjdCB7XG5cbiAgcHJpdmF0ZSByZWFkb25seSB3YXRjaGZ1bDogSVdhdGNoZnVsO1xuICBwcml2YXRlIHJlYWRvbmx5IGVjc1NlcnZpY2U6IGFueTtcbiAgcHJpdmF0ZSByZWFkb25seSB0YXJnZXRHcm91cDogQXBwbGljYXRpb25UYXJnZXRHcm91cDtcbiAgcHJpdmF0ZSByZWFkb25seSBzZXJ2aWNlTmFtZTogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IGNsdXN0ZXJOYW1lOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgdGFyZ2V0R3JvdXBOYW1lOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgbG9hZEJhbGFuY2VyTmFtZTogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IG1ldHJpY3M6IEVjc01ldHJpY0ZhY3Rvcnk7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFdhdGNoRWNzU2VydmljZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMud2F0Y2hmdWwgPSBwcm9wcy53YXRjaGZ1bDtcbiAgICBpZiAocHJvcHMuZWMyU2VydmljZSkge1xuICAgICAgdGhpcy5lY3NTZXJ2aWNlID0gcHJvcHMuZWMyU2VydmljZTtcbiAgICAgIHRoaXMuc2VydmljZU5hbWUgPSBwcm9wcy5lYzJTZXJ2aWNlLnNlcnZpY2VOYW1lO1xuICAgICAgdGhpcy5jbHVzdGVyTmFtZSA9IHByb3BzLmVjMlNlcnZpY2UuY2x1c3Rlci5jbHVzdGVyTmFtZTtcbiAgICB9IGVsc2UgaWYgKHByb3BzLmZhcmdhdGVTZXJ2aWNlKSB7XG4gICAgICB0aGlzLmVjc1NlcnZpY2UgPSBwcm9wcy5mYXJnYXRlU2VydmljZTtcbiAgICAgIHRoaXMuc2VydmljZU5hbWUgPSBwcm9wcy5mYXJnYXRlU2VydmljZS5zZXJ2aWNlTmFtZTtcbiAgICAgIHRoaXMuY2x1c3Rlck5hbWUgPSBwcm9wcy5mYXJnYXRlU2VydmljZS5jbHVzdGVyLmNsdXN0ZXJOYW1lO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ05vIHNlcnZpY2UgcHJvdmlkZWQgdG8gbW9uaXRvci4nKTtcbiAgICB9XG5cbiAgICB0aGlzLnRhcmdldEdyb3VwID0gcHJvcHMudGFyZ2V0R3JvdXA7XG4gICAgdGhpcy50YXJnZXRHcm91cE5hbWUgPSB0aGlzLnRhcmdldEdyb3VwLnRhcmdldEdyb3VwRnVsbE5hbWU7XG4gICAgdGhpcy5sb2FkQmFsYW5jZXJOYW1lID0gdGhpcy50YXJnZXRHcm91cC5maXJzdExvYWRCYWxhbmNlckZ1bGxOYW1lO1xuICAgIHRoaXMubWV0cmljcyA9IG5ldyBFY3NNZXRyaWNGYWN0b3J5KCk7XG5cbiAgICB0aGlzLndhdGNoZnVsLmFkZFNlY3Rpb24ocHJvcHMudGl0bGUsIHtcbiAgICAgIGxpbmtzOiBbXG4gICAgICAgIHsgdGl0bGU6ICdFQ1MgU2VydmljZScsIHVybDogbGlua0ZvckVjc1NlcnZpY2UodGhpcy5lY3NTZXJ2aWNlKSB9LFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHsgY3B1VXRpbGl6YXRpb25NZXRyaWMsIGNwdVV0aWxpemF0aW9uQWxhcm0gfSA9IHRoaXMuY3JlYXRlQ3B1VXRpbGl6YXRpb25Nb25pdG9yKHByb3BzLmNwdU1heGltdW1UaHJlc2hvbGRQZXJjZW50KTtcbiAgICBjb25zdCB7IG1lbW9yeVV0aWxpemF0aW9uTWV0cmljLCBtZW1vcnlVdGlsaXphdGlvbkFsYXJtIH0gPSB0aGlzLmNyZWF0ZU1lbW9yeVV0aWxpemF0aW9uTW9uaXRvcihwcm9wcy5tZW1vcnlNYXhpbXVtVGhyZXNob2xkUGVyY2VudCk7XG5cbiAgICBjb25zdCB7IHRhcmdldFJlc3BvbnNlVGltZU1ldHJpYywgdGFyZ2V0UmVzcG9uc2VUaW1lQWxhcm0gfSA9IHRoaXMuY3JlYXRlVGFyZ2V0UmVzcG9uc2VUaW1lTW9uaXRvcihwcm9wcy50YXJnZXRSZXNwb25zZVRpbWVUaHJlc2hvbGQpO1xuICAgIGNvbnN0IHsgaGVhbHRoeUhvc3RzTWV0cmljLCB1bmhlYWx0aHlIb3N0c01ldHJpYyB9ID0gdGhpcy5jcmVhdGVIb3N0Q291bnRNZXRyaWNzKCk7XG5cbiAgICBjb25zdCB7IHJlcXVlc3RzTWV0cmljLCByZXF1ZXN0c0FsYXJtIH0gPSB0aGlzLmNyZWF0ZVJlcXVlc3RzTW9uaXRvcihwcm9wcy5yZXF1ZXN0c1RocmVzaG9sZCk7XG4gICAgY29uc3QgeyBodHRwMnh4TWV0cmljLCBodHRwM3h4TWV0cmljLCBodHRwNHh4TWV0cmljLCBodHRwNXh4TWV0cmljIH0gPSB0aGlzLmNyZWF0ZUh0dHBSZXF1ZXN0c01ldHJpY3MoKTtcbiAgICBjb25zdCB7IHJlcXVlc3RzRXJyb3JSYXRlTWV0cmljLCByZXF1ZXN0c0Vycm9yUmF0ZUFsYXJtIH0gPSB0aGlzLnJlcXVlc3RzRXJyb3JSYXRlKHByb3BzLnJlcXVlc3RzRXJyb3JSYXRlVGhyZXNob2xkKTtcblxuXG4gICAgdGhpcy53YXRjaGZ1bC5hZGRXaWRnZXRzKFxuICAgICAgbmV3IGNsb3Vkd2F0Y2guR3JhcGhXaWRnZXQoe1xuICAgICAgICB0aXRsZTogYENQVVV0aWxpemF0aW9uLyR7Y3B1VXRpbGl6YXRpb25NZXRyaWMucGVyaW9kLnRvTWludXRlcygpfW1pbmAsXG4gICAgICAgIHdpZHRoOiAxMixcbiAgICAgICAgbGVmdDogW2NwdVV0aWxpemF0aW9uTWV0cmljXSxcbiAgICAgICAgbGVmdEFubm90YXRpb25zOiBbY3B1VXRpbGl6YXRpb25BbGFybS50b0Fubm90YXRpb24oKV0sXG4gICAgICB9KSxcbiAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHtcbiAgICAgICAgdGl0bGU6IGBNZW1vcnlVdGlsaXphdGlvbi8ke21lbW9yeVV0aWxpemF0aW9uTWV0cmljLnBlcmlvZC50b01pbnV0ZXMoKX1taW5gLFxuICAgICAgICB3aWR0aDogMTIsXG4gICAgICAgIGxlZnQ6IFttZW1vcnlVdGlsaXphdGlvbk1ldHJpY10sXG4gICAgICAgIGxlZnRBbm5vdGF0aW9uczogW21lbW9yeVV0aWxpemF0aW9uQWxhcm0udG9Bbm5vdGF0aW9uKCldLFxuICAgICAgfSksXG4gICAgKTtcbiAgICB0aGlzLndhdGNoZnVsLmFkZFdpZGdldHMoXG4gICAgICBuZXcgY2xvdWR3YXRjaC5TaW5nbGVWYWx1ZVdpZGdldCh7XG4gICAgICAgIHRpdGxlOiAnSGVhbHRoeSBIb3N0cycsXG4gICAgICAgIGhlaWdodDogNixcbiAgICAgICAgd2lkdGg6IDYsXG4gICAgICAgIG1ldHJpY3M6IFtoZWFsdGh5SG9zdHNNZXRyaWNdLFxuICAgICAgfSksXG4gICAgICBuZXcgY2xvdWR3YXRjaC5TaW5nbGVWYWx1ZVdpZGdldCh7XG4gICAgICAgIHRpdGxlOiAnVW5IZWFsdGh5IEhvc3RzJyxcbiAgICAgICAgaGVpZ2h0OiA2LFxuICAgICAgICB3aWR0aDogNixcbiAgICAgICAgbWV0cmljczogW3VuaGVhbHRoeUhvc3RzTWV0cmljXSxcbiAgICAgIH0pLFxuICAgICAgbmV3IGNsb3Vkd2F0Y2guR3JhcGhXaWRnZXQoe1xuICAgICAgICB0aXRsZTogYFRhcmdldFJlc3BvbnNlVGltZS8ke3RhcmdldFJlc3BvbnNlVGltZU1ldHJpYy5wZXJpb2QudG9NaW51dGVzKCl9bWluYCxcbiAgICAgICAgd2lkdGg6IDYsXG4gICAgICAgIGxlZnQ6IFt0YXJnZXRSZXNwb25zZVRpbWVNZXRyaWNdLFxuICAgICAgICBsZWZ0QW5ub3RhdGlvbnM6IFt0YXJnZXRSZXNwb25zZVRpbWVBbGFybS50b0Fubm90YXRpb24oKV0sXG4gICAgICB9KSxcbiAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHtcbiAgICAgICAgdGl0bGU6IGBSZXF1ZXN0cy8ke3JlcXVlc3RzTWV0cmljLnBlcmlvZC50b01pbnV0ZXMoKX1taW5gLFxuICAgICAgICB3aWR0aDogNixcbiAgICAgICAgbGVmdDogW3JlcXVlc3RzTWV0cmljXSxcbiAgICAgICAgbGVmdEFubm90YXRpb25zOiBbcmVxdWVzdHNBbGFybS50b0Fubm90YXRpb24oKV0sXG4gICAgICB9KSxcbiAgICApO1xuICAgIHRoaXMud2F0Y2hmdWwuYWRkV2lkZ2V0cyhcbiAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHtcbiAgICAgICAgdGl0bGU6ICdIVFRQIFJlcXVlc3RzIE92ZXJ2aWV3JyxcbiAgICAgICAgd2lkdGg6IDEyLFxuICAgICAgICBsZWZ0OiBbaHR0cDJ4eE1ldHJpYywgaHR0cDN4eE1ldHJpYywgaHR0cDR4eE1ldHJpYywgaHR0cDV4eE1ldHJpY10sXG4gICAgICB9KSxcbiAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHtcbiAgICAgICAgdGl0bGU6IGBIVFRQIFJlcXVlc3RzIEVycm9yIHJhdGUvJHtyZXF1ZXN0c0Vycm9yUmF0ZU1ldHJpYy5wZXJpb2QudG9NaW51dGVzKCl9bWluYCxcbiAgICAgICAgd2lkdGg6IDEyLFxuICAgICAgICBsZWZ0OiBbcmVxdWVzdHNFcnJvclJhdGVNZXRyaWNdLFxuICAgICAgICBsZWZ0QW5ub3RhdGlvbnM6IFtyZXF1ZXN0c0Vycm9yUmF0ZUFsYXJtLnRvQW5ub3RhdGlvbigpXSxcbiAgICAgIH0pLFxuICAgICk7XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZUNwdVV0aWxpemF0aW9uTW9uaXRvcihjcHVNYXhpbXVtVGhyZXNob2xkUGVyY2VudCA9IDApIHtcbiAgICBjb25zdCBjcHVVdGlsaXphdGlvbk1ldHJpYyA9IHRoaXMubWV0cmljcy5tZXRyaWNDcHVVdGlsaXphdGlvbkF2ZXJhZ2UodGhpcy5jbHVzdGVyTmFtZSwgdGhpcy5zZXJ2aWNlTmFtZSk7XG4gICAgY29uc3QgY3B1VXRpbGl6YXRpb25BbGFybSA9IGNwdVV0aWxpemF0aW9uTWV0cmljLmNyZWF0ZUFsYXJtKHRoaXMsICdjcHVVdGlsaXphdGlvbkFsYXJtJywge1xuICAgICAgYWxhcm1EZXNjcmlwdGlvbjogJ2NwdVV0aWxpemF0aW9uQWxhcm0nLFxuICAgICAgdGhyZXNob2xkOiBjcHVNYXhpbXVtVGhyZXNob2xkUGVyY2VudCxcbiAgICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY2xvdWR3YXRjaC5Db21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX1RIUkVTSE9MRCxcbiAgICAgIGV2YWx1YXRpb25QZXJpb2RzOiAzLFxuICAgIH0pO1xuICAgIHRoaXMud2F0Y2hmdWwuYWRkQWxhcm0oY3B1VXRpbGl6YXRpb25BbGFybSk7XG4gICAgcmV0dXJuIHsgY3B1VXRpbGl6YXRpb25NZXRyaWMsIGNwdVV0aWxpemF0aW9uQWxhcm0gfTtcbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlTWVtb3J5VXRpbGl6YXRpb25Nb25pdG9yKG1lbW9yeU1heGltdW1UaHJlc2hvbGRQZXJjZW50ID0gMCkge1xuICAgIGNvbnN0IG1lbW9yeVV0aWxpemF0aW9uTWV0cmljID0gdGhpcy5tZXRyaWNzLm1ldHJpY01lbW9yeVV0aWxpemF0aW9uQXZlcmFnZSh0aGlzLmNsdXN0ZXJOYW1lLCB0aGlzLnNlcnZpY2VOYW1lKTtcbiAgICBjb25zdCBtZW1vcnlVdGlsaXphdGlvbkFsYXJtID0gbWVtb3J5VXRpbGl6YXRpb25NZXRyaWMuY3JlYXRlQWxhcm0odGhpcywgJ21lbW9yeVV0aWxpemF0aW9uQWxhcm0nLCB7XG4gICAgICBhbGFybURlc2NyaXB0aW9uOiAnbWVtb3J5VXRpbGl6YXRpb25BbGFybScsXG4gICAgICB0aHJlc2hvbGQ6IG1lbW9yeU1heGltdW1UaHJlc2hvbGRQZXJjZW50LFxuICAgICAgY29tcGFyaXNvbk9wZXJhdG9yOiBjbG91ZHdhdGNoLkNvbXBhcmlzb25PcGVyYXRvci5HUkVBVEVSX1RIQU5fVEhSRVNIT0xELFxuICAgICAgZXZhbHVhdGlvblBlcmlvZHM6IDMsXG4gICAgfSk7XG4gICAgdGhpcy53YXRjaGZ1bC5hZGRBbGFybShtZW1vcnlVdGlsaXphdGlvbkFsYXJtKTtcbiAgICByZXR1cm4geyBtZW1vcnlVdGlsaXphdGlvbk1ldHJpYywgbWVtb3J5VXRpbGl6YXRpb25BbGFybSB9O1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVUYXJnZXRSZXNwb25zZVRpbWVNb25pdG9yKHRhcmdldFJlc3BvbnNlVGltZVRocmVzaG9sZCA9IDApIHtcbiAgICBjb25zdCB0YXJnZXRSZXNwb25zZVRpbWVNZXRyaWMgPSB0aGlzLm1ldHJpY3MubWV0cmljVGFyZ2V0UmVzcG9uc2VUaW1lKHRoaXMudGFyZ2V0R3JvdXBOYW1lLCB0aGlzLmxvYWRCYWxhbmNlck5hbWUpLmF2ZztcbiAgICBjb25zdCB0YXJnZXRSZXNwb25zZVRpbWVBbGFybSA9IHRhcmdldFJlc3BvbnNlVGltZU1ldHJpYy5jcmVhdGVBbGFybSh0aGlzLCAndGFyZ2V0UmVzcG9uc2VUaW1lQWxhcm0nLCB7XG4gICAgICBhbGFybURlc2NyaXB0aW9uOiAndGFyZ2V0UmVzcG9uc2VUaW1lQWxhcm0nLFxuICAgICAgdGhyZXNob2xkOiB0YXJnZXRSZXNwb25zZVRpbWVUaHJlc2hvbGQsXG4gICAgICBjb21wYXJpc29uT3BlcmF0b3I6IGNsb3Vkd2F0Y2guQ29tcGFyaXNvbk9wZXJhdG9yLkdSRUFURVJfVEhBTl9USFJFU0hPTEQsXG4gICAgICBldmFsdWF0aW9uUGVyaW9kczogMyxcbiAgICB9KTtcbiAgICB0aGlzLndhdGNoZnVsLmFkZEFsYXJtKHRhcmdldFJlc3BvbnNlVGltZUFsYXJtKTtcbiAgICByZXR1cm4geyB0YXJnZXRSZXNwb25zZVRpbWVNZXRyaWMsIHRhcmdldFJlc3BvbnNlVGltZUFsYXJtIH07XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZVJlcXVlc3RzTW9uaXRvcihyZXF1ZXN0c1RocmVzaG9sZCA9IDApIHtcbiAgICBjb25zdCByZXF1ZXN0c01ldHJpYyA9IHRoaXMubWV0cmljcy5tZXRyaWNSZXF1ZXN0Q291bnQodGhpcy50YXJnZXRHcm91cE5hbWUsIHRoaXMubG9hZEJhbGFuY2VyTmFtZSk7XG4gICAgY29uc3QgcmVxdWVzdHNBbGFybSA9IHJlcXVlc3RzTWV0cmljLmNyZWF0ZUFsYXJtKHRoaXMsICdyZXF1ZXN0c0FsYXJtJywge1xuICAgICAgYWxhcm1EZXNjcmlwdGlvbjogJ3JlcXVlc3RzQWxhcm0nLFxuICAgICAgdGhyZXNob2xkOiByZXF1ZXN0c1RocmVzaG9sZCxcbiAgICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY2xvdWR3YXRjaC5Db21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX1RIUkVTSE9MRCxcbiAgICAgIGV2YWx1YXRpb25QZXJpb2RzOiAzLFxuICAgIH0pO1xuICAgIHRoaXMud2F0Y2hmdWwuYWRkQWxhcm0ocmVxdWVzdHNBbGFybSk7XG4gICAgcmV0dXJuIHsgcmVxdWVzdHNNZXRyaWMsIHJlcXVlc3RzQWxhcm0gfTtcbiAgfVxuXG5cbiAgcHJpdmF0ZSBjcmVhdGVIdHRwUmVxdWVzdHNNZXRyaWNzKCkge1xuICAgIGNvbnN0IG1ldHJpY3MgPSB0aGlzLm1ldHJpY3MubWV0cmljSHR0cFN0YXR1c0NvZGVDb3VudCh0aGlzLnRhcmdldEdyb3VwTmFtZSwgdGhpcy5sb2FkQmFsYW5jZXJOYW1lKTtcbiAgICBjb25zdCBodHRwMnh4TWV0cmljID0gbWV0cmljcy5jb3VudDJYWDtcbiAgICBjb25zdCBodHRwM3h4TWV0cmljID0gbWV0cmljcy5jb3VudDNYWDtcbiAgICBjb25zdCBodHRwNHh4TWV0cmljID0gbWV0cmljcy5jb3VudDRYWDtcbiAgICBjb25zdCBodHRwNXh4TWV0cmljID0gbWV0cmljcy5jb3VudDVYWDtcbiAgICByZXR1cm4geyBodHRwMnh4TWV0cmljLCBodHRwM3h4TWV0cmljLCBodHRwNHh4TWV0cmljLCBodHRwNXh4TWV0cmljIH07XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZUhvc3RDb3VudE1ldHJpY3MoKSB7XG4gICAgY29uc3QgaGVhbHRoeUhvc3RzTWV0cmljID0gdGhpcy5tZXRyaWNzLm1ldHJpY01pbkhlYWx0aHlIb3N0Q291bnQodGhpcy50YXJnZXRHcm91cE5hbWUsIHRoaXMubG9hZEJhbGFuY2VyTmFtZSk7XG4gICAgY29uc3QgdW5oZWFsdGh5SG9zdHNNZXRyaWMgPSB0aGlzLm1ldHJpY3MubWV0cmljTWF4VW5oZWFsdGh5SG9zdENvdW50KHRoaXMudGFyZ2V0R3JvdXBOYW1lLCB0aGlzLmxvYWRCYWxhbmNlck5hbWUpO1xuICAgIHJldHVybiB7IGhlYWx0aHlIb3N0c01ldHJpYywgdW5oZWFsdGh5SG9zdHNNZXRyaWMgfTtcbiAgfVxuXG4gIHByaXZhdGUgcmVxdWVzdHNFcnJvclJhdGUocmVxdWVzdHNFcnJvclJhdGVUaHJlc2hvbGQgPSAwKSB7XG4gICAgY29uc3QgcmVxdWVzdHNFcnJvclJhdGVNZXRyaWMgPSB0aGlzLm1ldHJpY3MubWV0cmljSHR0cEVycm9yU3RhdHVzQ29kZVJhdGUodGhpcy50YXJnZXRHcm91cE5hbWUsIHRoaXMubG9hZEJhbGFuY2VyTmFtZSk7XG4gICAgY29uc3QgcmVxdWVzdHNFcnJvclJhdGVBbGFybSA9IHJlcXVlc3RzRXJyb3JSYXRlTWV0cmljLmNyZWF0ZUFsYXJtKHRoaXMsICdyZXF1ZXN0c0Vycm9yUmF0ZUFsYXJtJywge1xuICAgICAgYWxhcm1EZXNjcmlwdGlvbjogJ3JlcXVlc3RzRXJyb3JSYXRlQWxhcm0nLFxuICAgICAgdGhyZXNob2xkOiByZXF1ZXN0c0Vycm9yUmF0ZVRocmVzaG9sZCxcbiAgICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY2xvdWR3YXRjaC5Db21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX1RIUkVTSE9MRCxcbiAgICAgIGV2YWx1YXRpb25QZXJpb2RzOiAzLFxuICAgIH0pO1xuICAgIHRoaXMud2F0Y2hmdWwuYWRkQWxhcm0ocmVxdWVzdHNFcnJvclJhdGVBbGFybSk7XG4gICAgcmV0dXJuIHsgcmVxdWVzdHNFcnJvclJhdGVNZXRyaWMsIHJlcXVlc3RzRXJyb3JSYXRlQWxhcm0gfTtcbiAgfVxuXG59XG5cblxuZnVuY3Rpb24gbGlua0ZvckVjc1NlcnZpY2UoZWNzU2VydmljZTogYW55KSB7XG4gIHJldHVybiBgaHR0cHM6Ly9jb25zb2xlLmF3cy5hbWF6b24uY29tL2Vjcy9ob21lP3JlZ2lvbj0ke2Vjc1NlcnZpY2Uuc3RhY2sucmVnaW9ufSMvY2x1c3RlcnMvJHtlY3NTZXJ2aWNlLmNsdXN0ZXIuY2x1c3Rlck5hbWV9L3NlcnZpY2VzLyR7ZWNzU2VydmljZS5zZXJ2aWNlTmFtZX0vZGV0YWlsc2A7XG59XG5cblxuIl19