"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchLambdaFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const metrics_1 = require("./monitoring/aws/lambda/metrics");
const DEFAULT_DURATION_THRESHOLD_PERCENT = 80;
/**
 * @stability stable
 */
class WatchLambdaFunction extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const cfnFunction = props.fn.node.defaultChild;
        const timeoutSec = cfnFunction.timeout || 3;
        this.watchful = props.watchful;
        this.fn = props.fn;
        this.metrics = new metrics_1.LambdaMetricFactory();
        this.watchful.addSection(props.title, {
            links: [
                { title: 'AWS Lambda Console', url: linkForLambdaFunction(this.fn) },
                { title: 'CloudWatch Logs', url: linkForLambdaLogs(this.fn) },
            ],
        });
        const { errorsMetric, errorsAlarm } = this.createErrorsMonitor(props.errorsPerMinuteThreshold);
        const { throttlesMetric, throttlesAlarm } = this.createThrottlesMonitor(props.throttlesPerMinuteThreshold);
        const { durationMetric, durationAlarm } = this.createDurationMonitor(timeoutSec, props.durationThresholdPercent);
        const invocationsMetric = this.metrics.metricInvocations(this.fn.functionName);
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: `Invocations/${invocationsMetric.period.toMinutes()}min`,
            width: 6,
            left: [invocationsMetric],
        }), new cloudwatch.GraphWidget({
            title: `Errors/${errorsMetric.period.toMinutes()}min`,
            width: 6,
            left: [errorsMetric],
            leftAnnotations: [errorsAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `Throttles/${throttlesMetric.period.toMinutes()}min`,
            width: 6,
            left: [throttlesMetric],
            leftAnnotations: [throttlesAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `Duration/${durationMetric.period.toMinutes()}min`,
            width: 6,
            left: [durationMetric],
            leftAnnotations: [durationAlarm.toAnnotation()],
        }));
    }
    createErrorsMonitor(errorsPerMinuteThreshold = 0) {
        const fn = this.fn;
        const errorsMetric = this.metrics.metricErrors(fn.functionName);
        const errorsAlarm = errorsMetric.createAlarm(this, 'ErrorsAlarm', {
            alarmDescription: `Over ${errorsPerMinuteThreshold} errors per minute`,
            threshold: errorsPerMinuteThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(errorsAlarm);
        return { errorsMetric, errorsAlarm };
    }
    createThrottlesMonitor(throttlesPerMinuteThreshold = 0) {
        const fn = this.fn;
        const throttlesMetric = this.metrics.metricThrottles(fn.functionName);
        const throttlesAlarm = throttlesMetric.createAlarm(this, 'ThrottlesAlarm', {
            alarmDescription: `Over ${throttlesPerMinuteThreshold} throttles per minute`,
            threshold: throttlesPerMinuteThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(throttlesAlarm);
        return { throttlesMetric, throttlesAlarm };
    }
    createDurationMonitor(timeoutSec, durationPercentThreshold = DEFAULT_DURATION_THRESHOLD_PERCENT) {
        const fn = this.fn;
        const durationMetric = this.metrics.metricDuration(fn.functionName).p99;
        const durationThresholdSec = Math.floor(durationPercentThreshold / 100 * timeoutSec);
        const durationAlarm = durationMetric.createAlarm(this, 'DurationAlarm', {
            alarmDescription: `p99 latency >= ${durationThresholdSec}s (${durationPercentThreshold}%)`,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            threshold: durationThresholdSec * 1000,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(durationAlarm);
        return { durationMetric, durationAlarm };
    }
}
exports.WatchLambdaFunction = WatchLambdaFunction;
_a = JSII_RTTI_SYMBOL_1;
WatchLambdaFunction[_a] = { fqn: "cdk-watchful.WatchLambdaFunction", version: "0.6.29" };
function linkForLambdaFunction(fn, tab = 'graph') {
    return `https://console.aws.amazon.com/lambda/home?region=${fn.stack.region}#/functions/${fn.functionName}?tab=${tab}`;
}
function linkForLambdaLogs(fn) {
    return `https://console.aws.amazon.com/cloudwatch/home?region=${fn.stack.region}#logEventViewer:group=/aws/lambda/${fn.functionName}`;
}
//# sourceMappingURL=data:application/json;base64,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